#predefined procedures to be used in special fields (buttonfunction, whenreadvar,...) of xml defined widgets (e.g. preferences window) 
#e.g <entry variable="CosSmoothedElems" whenreadvar="FromCosToDegree" 
#--> xmlprograms::FromCosToDegree var
#the namespace xmlprograms will be automatically prepended to the xml field, and extra arguments could be appended (e.g. to provide widget, variable name,...)


namespace eval xmlprograms {
}

#--------------validate entries functions------------
proc xmlprograms::IsFloatingPoint { item } {
    return [string is double -strict $item]    
}

proc xmlprograms::IsFloatingPointPositive { item } {
    if { [xmlprograms::IsFloatingPoint $item] && $item>=0 } {
        return 1
    } 
    return 0  
}

proc xmlprograms::IsFloatingPointMinMax { min max item } {
    if { [xmlprograms::IsFloatingPoint $item] && $item>=$min && $item<=$max} {
        return 1
    } 
    return 0  
}

proc xmlprograms::IsInteger { item } {   
    return [string is integer -strict $item]
}

proc xmlprograms::IsIntegerMinMax { min max item } {   
    if { [xmlprograms::IsInteger $item] && $item>=$min && $item<=$max} {
        return 1
    } 
    return 0  
}

proc xmlprograms::IsNatural { item } {
    if { [xmlprograms::IsInteger $item] && $item>=0 } {
        return 1
    }
    return 0
}

proc xmlprograms::IsFile { item } {
    if {$item == ""} {
        return 0
    }
    return 1
}

proc xmlprograms::IsRealFormat { item } {
    if {$item == ""} {  
        return 0
    }
    if { [ regexp {^%[^eEfFgG]*[eEfFgG]$} $item] } {
        set err [ catch { format $item 1.2345678912345567890}]
        if { $err } { 
            return 0
        }
    } else {
        return 0
    }
    return 1
}

proc xmlprograms::IsNotEmpty { item } {
    if {$item == ""} {
        return 0
    }
    return 1
}

#--------------end validate entries functions------------

  
proc xmlprograms::FormatG { var } {
    return [format %g $var]
}   

proc xmlprograms::TclColour { var } {
    set var {*}$var
    if { $var == "0" } {
        set var "#ff0000"
    } else {
        set rgb [split $var #]
        if { [llength $rgb] == "3" } {      
            #accepting color format 22#200#100
            set var [GidUtils::GiDColorToTkColor $var]
        }
    }
    return $var
}

proc xmlprograms::FromCosToDegree { var } {
    if { [expr $var>1.0] || [expr -1.0>$var] } {
        WarnWinText "$var not inside -1 to 1, imposible to calculate cosinus"
    }
    return [format %g [expr 57.295779513082323*acos($var)]]
}

proc xmlprograms::FromDegreeToCos { var } {
    return [format %g [expr cos($var*0.017453292519943295)]]
}

proc xmlprograms::SetWarnLineHeight { var } {
    #GiD_Set WarnLineHeight $var
    set w .gid.comm.fwarn.list
    if { [winfo exists $w] } {        
        $w configure -height $var
    }
    return $var
}

proc xmlprograms::FontFromGiDToTcl { fontgid } {
    # GiD give 'PGF(DefaultFont)'={{C:\Windows\Fonts\cour.ttf} 0 14 {{Courier New} {Regular} {Fixed width}}}
    # On preprocess variable its transform to: Input format {{Courier New (Fixed width)} {Regular} 14}
    lassign [lindex $fontgid 0] fontgid_filename fontgid_type_id fontgid_size fontgid_types
    lassign $fontgid_types font_name font_type font_size
    if { $font_size == "Fixed width" } {
        set font_name "$font_name (Fixed width)"
    }
    return [list $font_name $font_type $fontgid_size]
}

proc xmlprograms::FontFromTclToGiD { fonttcl } {
    # tcl format {{Courier New} {Regular} 14}
    #'On postrocess variable its transform to: {C:\Windows\Fonts\cour.ttf} 0 14
    lassign [lindex $fonttcl 0] font_name font_type font_size
    set font_key [list $font_name $font_type]
    if { [info exists ::PGFFonts_new($font_key)] } {
        lassign $::PGFFonts_new($font_key) fontgid_filename fontgid_type_id
    } else {
        #trick to try to found if exists the same font name but not exactly for this type (e.g exists Magneto Bold but not Magneto Regular)
        set alternative_font_key [lindex [array names ::PGFFonts_new  [list $font_name *]] 0]
        if { $alternative_font_key != "" } {
            lassign $::PGFFonts_new($alternative_font_key) fontgid_filename fontgid_type_id        
        } else {
            WarnWin [_ "Font '%s' not found. \nTry Utilities-->Preferences-->Fonts \n and click on 'Rebuild font list'." $fonttcl]
            set fontgid_filename ""
            set fontgid_type_id ""
        }
    }
    return [list $fontgid_filename $fontgid_type_id $font_size $font_key]
}

proc xmlprograms::GetFonts {} {
    set lst_valores_combo ""
    foreach ifont [GiD_PGF_Fonts get] {   
        set font_key [lindex $ifont 2]
        lassign $font_key font_name font_type
        if { [llength $font_key]==3 && [lindex $font_key 2] == "Fixed width" } {
            set font_name "$font_name (Fixed width)"
            set font_key [list $font_name $font_type]
        }
        lappend lst_valores_combo $font_name
        set ::PGFFonts_new($font_key) [lrange $ifont 0 1]
    }
    set lst_valores_combo [lsort -dictionary -unique -nocase $lst_valores_combo]
    return [list $lst_valores_combo]
}

proc xmlprograms::GetTkFonts {} {
    set lst_fonts [lsort -unique [ font families]]
    return [list $lst_fonts $lst_fonts]
}

proc xmlprograms::GetTkFixedFonts {} {
    if { $::tcl_platform(os) == "Linux" } {
        # using GetTkFonts instead of GetTkFixedFonts, because on linux, it causes an exit 1 afterwards
        return [ GetTkFonts]
    } else {
        set lst_fixed_fonts {}
        font create _____tmp -size 10
        foreach fnt [ lsort -unique [ font families]] {
            font configure _____tmp -family $fnt
            if { [ font measure _____tmp M] == [ font measure _____tmp i]} {
                lappend lst_fixed_fonts $fnt
            }
        }
        font delete _____tmp
        set lst_fonts [ lsort $lst_fixed_fonts]
        return [ list $lst_fonts $lst_fonts]
    }
}

proc xmlprograms::FontFromPrefToTk { fontgid } {
    W "FontFromPrefToTk $fontgid"
}

proc xmlprograms::FontFromTkToPref { fontgid } {
    W "FontFromTkToPref $fontgid"
}

proc xmlprograms::RebuildFontList { baseframe } {
    set w {}
    foreach w [ winfo children $baseframe] {
        if { [ winfo class $w] == "TButton"} {
            break
        }
    }
    GidUtils::WaitState $baseframe
    if { $w != {}} {
        set old_label [ $w cget -text]
        $w configure -text [_ "Rebuilding font list ..."]
        update
    }
    GiD_PGF_Fonts resetfontlist
    if { $w != {}} {
        $w configure -text $old_label
    }
    GidUtils::EndWaitState $baseframe
    # needs to be done the cache of tcl fonts, which can be appened to
    # TODO
    # GiD_PGF_Fonts fontlistfilename
    # need to refill fonts comboboxed:
    # GetFonts
    # GetPmfonts

    # InfoWin [_ "Font list rebuild. \nNew fonts will appear the next time the preferences window is opened."] $baseframe
    InfoWin [_ "Font list rebuild. \nPreferences window will be reopened to show the changes."] $baseframe
    # reopen preferences window to reflect the changes
    if { [ winfo exists $baseframe]} {
        destroy [ winfo toplevel $baseframe]
        after 500 PreferencesWindow
    }
}

proc xmlprograms::FontFromPmGiDToTcl { fontgid } {
    set fonttcl [lindex [lindex {*}$fontgid 3] 0]
    return $fonttcl
}
 
proc xmlprograms::FontFromTclToPmGiD { fonttcl } {
    return $fonttcl
} 
  
proc xmlprograms::GetPmfonts {} {
    set fonts [ GiD_PGF_Fonts get -pmfont]
    return [list $fonts $fonts ]
}

proc xmlprograms::GetLanguages {} {
    set langs [::GidUtils::GetAllLanguages]
    #reorder langs by translated strings, not by langs
    set langsandlabels [list]
    foreach i $langs {
        lappend langsandlabels [list [_ "%s" [::GIDi18n::language $i]] $i]
    }
    set langsandlabels [lsort -index 0 $langsandlabels]
    set langlabels ""
    set langs ""
    foreach item $langsandlabels {
        lappend langlabels [lindex $item 0]
        lappend langs [lindex $item 1]
    }
    return [list $langs $langlabels ]
}

proc xmlprograms::GetThemesAvailable {} {
    # re-read the themes folder, may be there are new themes which have been downloaded
    gid_themes::ReadThemes
    set themenames [gid_themes::GetThemesAndLabels]
    set translatedthemenames ""
    set themes ""
    foreach {theme name} $themenames {
        lappend themes $theme
        lappend translatedthemenames [_ $name]                
    }
    return [list $themes $translatedthemenames]
}

proc xmlprograms::GetThemeSizesCurrent {} {      
    set tranlatedsizes [list]
    set final_sizes [list]
    set original_sizes $::gid_themes::Themes([gid_themes::GetCurrentTheme],ThemeSizes)
    foreach size $original_sizes {
        #translations applied to all themes
        switch $size {
            -2 {
                lappend final_sizes -2
                lappend tranlatedsizes [_ "Super small"]
            }
            -1 {
                lappend final_sizes -1
                lappend tranlatedsizes [_ "Small"]
            }
            0 {
                lappend final_sizes 0
                lappend tranlatedsizes [_ "Medium"]
            }            
            1 {
                lappend final_sizes 1
                lappend tranlatedsizes [_ "Large"]
            }            
            2 {
                lappend final_sizes 2
                lappend tranlatedsizes [_ "Super large"]
            }
            default {
                lappend final_sizes $size
                lappend tranlatedsizes [_ "Size %s" $size]
            }
        }
    }
    return [list $final_sizes $tranlatedsizes]
}

proc xmlprograms::OpenBrowserForBatch { baseframe variable} {      
    set types [list \
                   [list [_ "Batch file"] [list .bch]] \
                   [list [_ "All files"] .*]]
    set $variable [MessageBoxGetFilename file write [_ "write batch file"] \
                       [set $variable] $types .bch 0]      
    return variable
}

proc xmlprograms::OpenBrowserForDirectory { baseframe variable} {      
    set $variable [MessageBoxGetFilename directory write [_ "Select a directory"]]
    return variable
}

proc xmlprograms::RedrawBoundaryMesh {baseframe validpath} {
    canvas $validpath.canvas#1 -height 7 -width 300
    $validpath.canvas#1 create line 2 5 250 5 -fill red -width 2
    $validpath.canvas#1 create line 255 5 260 5 -fill red -width 2
    $validpath.canvas#1 create line 265 5 270 5 -fill red -width 2
    $validpath.canvas#1 create line 275 5 277 5 -fill red -width 2
    $validpath.canvas#1 create line 284 5 286 5 -fill red -width 2
    $validpath.canvas#1 create line 2 0 2 8 -fill black -width 1
    grid $validpath.canvas#1 -in $baseframe -sticky w    
    #RedrawDivisions_blm $canvas
}

proc xmlprograms::RedrawDivisions_blm { baseframe validpath BoundaryLayerGrowLaw BoundaryLayerGrowFactor } {
    set canvas $validpath.canvas#1
    if { ![winfo exists $canvas] } {
        return
    }      
    if { $BoundaryLayerGrowFactor <=0.0 } {
        #in case of 0 or negative next while become a infinite loop!!
        return
    }
    $canvas delete temp
    set maxwidth 300
    
    #First distance = 4 pixel
    set DistanceFirst 4
    
    set layerX [expr 2+$DistanceFirst]
    switch $BoundaryLayerGrowLaw {
        0 {
            #geometric
            set denominator [expr {1.0-$BoundaryLayerGrowFactor}]
            set i 1
            while { $maxwidth > $layerX && $i < 1000 } {
                $canvas create line $layerX 0 $layerX 8 -fill black -width 1  -tags temp
                if { [expr {abs($denominator)}] < 1.0e-10 } {
                    set layerX [expr 2+round($DistanceFirst*($i+1)*$BoundaryLayerGrowFactor)]
                } else {
                    set layerX [expr 2+round($DistanceFirst*(1.0-pow($BoundaryLayerGrowFactor,$i+1))/$denominator)]
                }
                incr i
            }
        }
        1 {
            #exponential
            set i 1
            while { $maxwidth > $layerX && $i < 1000 } {
                $canvas create line $layerX 0 $layerX 8 -fill black -width 1  -tags temp
                set layerX [expr 2+round(exp(log($DistanceFirst)+($BoundaryLayerGrowFactor*$i)))]
                incr i
            }
        }
        2 {
            #geometric_mod
            set i 1
            while { $maxwidth > $layerX && $i < 1000 } {
                $canvas create line $layerX 0 $layerX 8 -fill black -width 1  -tags temp
                set layerX [expr 2+round($DistanceFirst*(1+$i*(1+$BoundaryLayerGrowFactor*(1+$BoundaryLayerGrowFactor)*$i)))]
                incr i
            }
        }
    }
}

proc xmlprograms::IsMac {} {
    return [expr {$::tcl_platform(os) == "Darwin"}]
}
proc xmlprograms::IsMacOSMojaveOrGreater {} {
    return [ isMacOSVersionEqualOrGreater Mojave]
}
proc xmlprograms::IsLinux {} {
    return [expr {$::tcl_platform(os) == "Linux"}]
}

proc xmlprograms::IsWindows {} {
    return [expr {$::tcl_platform(platform) == "windows"}]
}

proc xmlprograms::GetCurrentPrePostMode {} {
    return [ ::GetCurrentPrePostMode]
}

proc xmlprograms::CloudUserPreferences {} {    
    return $::GID_USER_PREFERENCES_CLOUD
}

proc xmlprograms::CloudEnableDatamanager {} {    
    return $::GID_ENABLE_DATAMANAGER
}

proc xmlprograms::ShowCloudPreferences {} {
    return [ expr { [ CloudUserPreferences ] || [ CloudEnableDatamanager ] } ]
}

#e.g. to invoke a custom procedure that doesn't belong to ::xmlprograms namespace and could add extra arguments like widget and its inner variable
#     <entrywithbutton buttonfunction='CustomProc MyProcedure' ...
#     proc ::MyProcedure { widget varname } { set $varname hello }  
#can be used on attibutes: condition whenwritevar whenreadvar fillfunction buttonfunction function updatefunction validation  
proc xmlprograms::CustomProc { procname args } {
    $procname {*}$args
}

proc xmlprograms::GetMainDrawAreaMaximumHeight {} {
    lassign [ GidUtils::GetMainDrawAreaSize] w h
    return $h
}

proc xmlprograms::RestartGiDAfterSettingPreferences { original_value} {
    # whenwritevar does not what you think it does, it allows you to change the value selected by the user
    # and is not an event called after the preference has been writen.
    # it's more a validation event !!!
    # give some time for preferences window to save preferences
    # otherwise preferences file will be truncated
    after 1000 ::RestartGiD
    return $original_value
}

proc xmlprograms::GenericCheckGraphicsFromXml { w } {
    ShowGraphicsInformation $w
}
