#        Este fichero implementa el codigo Tcl para leer y mantener un conjunto
#        de magnitudes y sus unidades.
#
# Copyright (c) 2000 CIMNE
#

namespace eval Units {
    namespace export Init AppendFile
    
    variable  exp_space     {[ ]*}
    variable  exp_dig       \[0-9\]
    variable  exp_letra     \[a-z_\]
    variable  exp_magname   "$exp_letra+\[a-z_ \]*"
    variable  exp_int       $exp_dig+
    variable  exp_real      \[+-\]?$exp_int\\.?$exp_dig*\(e\[+-\]?$exp_int\)?
    variable  exp_unittype  \{$exp_space\(Reference|Basic\)($exp_space,$exp_space\(Reference|Basic\))?$exp_space\}
    variable  exp_strunit   "\[ a-z0-9^\\*/\-\u00B0\u00BA]*"
    variable  exp_unit      "($exp_real)?$exp_space\($exp_strunit\)($exp_unittype)?"
    variable  exp_unitlist  ($exp_unit)($exp_space\,(.+))?
    variable  exp_magnitude "^($exp_magname)$exp_space\:$exp_space\(.+)"
    variable  exp_sysunit   ^($exp_magname)$exp_space\:\($exp_strunit)$
    variable  exp_bstruct   "^BEGIN$exp_space\(SYSTEM$exp_space\\($exp_space\(.+)$exp_space\\)|TABLE)$"
    variable  exp_estruct    ^END$
    variable  exp_modelunit  ^(MODEL|PROBLEM)$exp_space\:(.*)
    variable  exp_udsystem    ^BEGIN$exp_space\SYSTEM
    variable  exp_udstate    "^USER DEFINED$exp_space\:(.*)"
    variable  exp_bsstate    "^USE BASE SYSTEMS$exp_space\:(.*)"
    variable  exp_maghidden  "^HIDDEN(?:\\s*):(.+)"
    variable  FileName       ""
    variable  FileId         ""
    variable  CurrentLine    ""
    variable  LineCounter    0
    variable  bInModelFile   0
    variable  TableIdx       0
    
    variable  UserDefinedState 1
    variable  UseBaseSystemsState 1
    
    variable  MagNames {}
    variable  Magnitudes
    variable  Hidden {}
    variable  MagHidden {}
    
    variable  SysNames {}
    variable  Systems
    
    variable  ModelUnit    ""
    variable  ModelSysName ""
    variable  OwnerSystem  ""
    variable  FactorCache
    variable disallowed_units_sorted ""
}

proc Units::SetUnitsDisallowed { simple_units } {
    variable disallowed_units_sorted 
    set disallowed_units_sorted [lsort $simple_units]
}

proc Units::GetUnitsDisallowed { } {
    variable disallowed_units_sorted 
    return $disallowed_units_sorted
}

proc Units::ResetUnitsDisallowed { } {
    variable disallowed_units_sorted 
    set disallowed_units_sorted ""
}

proc Units::IsUnitDisallowed { unitinfo } {
    variable disallowed_units_sorted
    set disallowed 0
    set numerator [lindex $unitinfo 3]
    set denominator [lindex $unitinfo 4]
    foreach item [concat $numerator $denominator] {
        if { [lsearch -sorted $disallowed_units_sorted $item]!=-1 } {
            set disallowed 1
            break
        }
    }
    return $disallowed
}

#  Units::Init --
#
#        Inicializa el estado de Unit
#
#  Argumentos:
#
#  Resultados:
#

proc Units::Init { } {
    variable  exp_space     {[ ]*}
    variable  exp_dig       \[0-9\]
    variable  exp_letra     \[a-z_\]
    variable  exp_magname   "$exp_letra+\[a-z_ \]*"
    variable  exp_int       $exp_dig+
    variable  exp_real      \[+-\]?$exp_int\\.?$exp_dig*\(e\[+-\]?$exp_int\)?
    variable  exp_unittype  \{$exp_space\(Reference|Basic\)($exp_space,$exp_space\(Reference|Basic\))?$exp_space\}
    variable  exp_strunit   {[ \\a-z0-9^\*/\-\u00B0\u00BA]+}
    variable  exp_unit      "($exp_real)?$exp_space\($exp_strunit\)($exp_unittype)?"
    variable  exp_unitlist  ($exp_unit)($exp_space\,(.+))?
    variable  exp_magnitude "^($exp_magname)$exp_space\:$exp_space\(.+)"
    variable  exp_sysunit   ^($exp_magname)$exp_space\:\($exp_strunit)$
    variable  exp_bstruct   "^BEGIN$exp_space\(SYSTEM$exp_space\\($exp_space\(.+)$exp_space\\)|TABLE)$"
    variable  exp_estruct   ^END$
    variable  exp_modelunit  ^(MODEL|PROBLEM)$exp_space\:(.*)
    variable  exp_udsystem    ^BEGIN$exp_space\SYSTEM
    variable  exp_udstate    "^USER DEFINED$exp_space\:(.*)"
    variable  exp_bsstate    "^USE BASE SYSTEMS$exp_space\:(.*)"
    variable  exp_maghidden  "^HIDDEN(?:\\s*):(.+)"
    
    variable  FileId        ""
    variable  CurrentLine   ""
    variable  LineCounter    0
    variable  bInModelFile   0
    variable  bInGidFile     0
    variable  TableIdx       0
    
    variable  UserDefinedState 1
    variable  UseBaseSystemsState 1
    
    variable  MagNames {}
    variable  Magnitudes
    variable  Hidden {}
    variable  MagHidden {}
    variable  SysNames {}
    variable  Systems
    
    variable  ModelUnit    ""
    variable  ModelSysName ""
    variable  OwnerSystem  ""
    variable  FactorCache
    variable disallowed_units_sorted ""
    
    unset -nocomplain Magnitudes
    unset -nocomplain Systems
    array unset FactorCache
}

#  Units::ParserTableLine --
#
#        Analiza la linea line y determina si es una definicion de magnitud valida
#        en caso positivo adiciona las unidades definidas a la tabla de magnitudes.
#
#  Argumentos:
#        line    linea que contiene la deinicion.
#
#  Resultados:
#        La tabla de magnitudes se incrementa en la nuevas unidades para la magnitud
#        o error.

proc Units::ParserTableLine {} {
    variable CurrentLine
    variable exp_magnitude
    variable Magnitudes
    
    # ----- Verifico la terminacion del bloque table ------
    
    if ![string compare [string toupper $CurrentLine] "END"] {
        return [list 1 ok]
    }
    if [regexp -nocase $exp_magnitude $CurrentLine {} magname units] {
        set magname [string trim [string toupper $magname]]
        if { ![info exists Magnitudes($magname)] } {
            set Magnitudes($magname) {}
            set Magnitudes($magname,BASIC) -1
            set Magnitudes($magname,REFERENCE) -1
        }
        return [ParserUnitsList $magname $units]
    } else {
        return [list 0 [_ "Invalid magnitud name or units list ill-formed"]]
    }
}

proc ParserSystemLine {} {
    variable CurrentLine
}

#  Units::ParserUnitsList --
#
#        Analiza la lista de unidades y determina si es una lista de unidades valida
#        en caso positivo adiciona las unidades definidas a la tabla de magnitudes.
#
#  Argumentos:
#        magname   nombre de la magnitud de las unidades
#        lista     cadena que contiene las unidades
#
#  Resultados:
#        La tabla de magnitudes se incrementa en la nuevas unidades para la magnitud
#        o error.

proc Units::ParserUnitsList { magname lista } {
    variable exp_unitlist
    variable  exp_unit
    variable MagNames
    
    set listatmp $lista
    while { $listatmp != "" } {
        set listatmp [string trim $listatmp]
        set llista [string length $listatmp]
        if { [regexp -nocase $exp_unitlist $listatmp Matched {} factor {} unidad {} tipo1 {} tipo2 {} listatmp] } {
            if { $llista == [string length $Matched] } {
                AppendUnit $magname $factor $unidad $tipo1 $tipo2
            } else {
                return [list 0 [_ "Could not read unit set %1\$s. Only read %2\$s" $lista $Matched]]
            }
        }
    }
    lappend MagNames $magname
    return [list 1 ok]
}

#  Units::AppendUnit --
#
#        Adiciona una unidad a la tabla de magnitudes. Verifica si ya existe.
#        En caso de existir sustituye la definicion anterior.
#
#  Argumentos:
#
#        magname    nombre de la magnitud a la que pertenece la unidad.
#        factor     factor de conversion respecto a la unidad basica.
#        unidad     expresion que define la unidad.
#        tipo1      Tipo de la unidad ( Reference o Basic ).
#        tipo2      Tipo de la unidad ( Reference o Basic ).
#
#  Resultados:
#

proc Units::AppendUnit { magname factor unidad tipo1 tipo2 } {
    variable ShortFileName
    variable Magnitudes
    
    
    if { $factor == "" } {
        set factor 1.0
    }
    
    # Verifico si la expresion de la unidad es correcta
    
    ParserUnit::Init $unidad
    ParserUnit::Start
    if { $ParserUnit::Status != "END" } {
        W [concat $ParserUnit::Status $magname $unidad ==> [lindex $ParserUnit::Status end]]
        return
    }
    # Verifico si ya existe esta unidad
    
    set SearchInfo [FindUnit $ParserUnit::Numerator $ParserUnit::Denominator]
    if { $SearchInfo != {} } {
        set UnitsList $Magnitudes([lindex $SearchInfo 0])
        set u [lindex $UnitsList [lindex $SearchInfo 1]]
        if { $factor == [lindex $u 0] } {
            W [_ "In %1\$s, processing magnitude %2\$s: %3\$s is already defined as %4\$s" \
                    $ShortFileName $magname $unidad [lindex $SearchInfo 0]]
        } else {
            W [_ "Unit :%1\$s with factor %2\$g is already defined with %3\$g" $unidad $factor [lindex $u 0]]
        }
    } else {
        if {$magname eq "TEMPERATURE"} {
            if {![catch "GiDTeXSymbols::Get $ParserUnit::Numerator" NumeratorTextList]} {
                set DenominatorTextList ""
            } else {
                W [_ "Invalid temperature symbol %1\$s. Should be Cel, Kel, or Far" $ParserUnit::Numerator]
                return
            }
        } else {
            CreateExpTextList $ParserUnit::Numerator NumeratorTextList
            CreateExpTextList $ParserUnit::Denominator DenominatorTextList
        }
        regsub -all { } $unidad {} unidad
        lappend Magnitudes($magname) [list $factor $ParserUnit::Numerator $ParserUnit::Denominator \
                $NumeratorTextList $DenominatorTextList [string trim $unidad]]
        set lastidx [llength $Magnitudes($magname)]
        incr lastidx -1
        if { $tipo1 != "" } {
            set tipo1 [string toupper $tipo1]
            if { $Magnitudes($magname,$tipo1) != -1 } {
                W [_ "Type: %1\$s in magnitude %2\$s is already defined" $tipo1 $magname]
            } else {
                set Magnitudes($magname,$tipo1) $lastidx
            }
        }
        if { $tipo2 != "" } {
            set tipo2 [string toupper $tipo2]
            if { $Magnitudes($magname,$tipo2) != -1 } {
                W [_ "Type: %1\$s in magnitude %2\$s is already defined" $tipo2 $magname]
            } else {
                set Magnitudes($magname,$tipo2) $lastidx
            }
        }
    }
}

#  Units::AppendFile --
#
#        Incrementa el contenido de la tabla de magnitudes a partir del
#        contenido de un fichero.
#
#  Argumentos:
#
#        filename   nombre del fichero
#
#  Resultados:
#

proc Units::AppendFile { fname {gfile 0} } {
    variable FileName $fname
    variable ShortFileName [file tail $fname]
    variable FileId
    variable CurrentLine
    variable LineCounter  0
    variable bInModelFile 0
    variable bInGidFile   $gfile
    
    variable UserDefinedState 1
    variable UseBaseSystemsState 1
    
    variable exp_bstruct
    variable exp_udstate
    variable  exp_bsstate
    variable  exp_maghidden
    
    variable Hidden
    
    set fail 0
    
    if { [catch {open $fname} FileId] } {
        set fail 1
        W $FileId
    } else {
        if { $gfile } {
            # in order to read the special characters contained in
            # units_defaults.uni in the definition of Temperature magnitude.
            fconfigure $FileId -encoding "iso8859-1"
        }
        while { ![eof $FileId] } {
            gets $FileId CurrentLine
            incr LineCounter
            # ---- Elimino los espacios al inicio y final de la linea ----
            
            set CurrentLine [string trim $CurrentLine]
            if { [string match \#* $CurrentLine] } {
                if { [regexp -nocase {#\s?encoding\s+(.*)} $CurrentLine dummy encoding] } {
                    fconfigure $FileId -encoding $encoding
                }
                continue
            }
            if { [string length $CurrentLine] } {
                if [regexp -nocase $exp_bstruct $CurrentLine {} struct argopt] {
                    if { ![string compare [string toupper $struct] "TABLE"] } {
                        if ![ParserTable] {
                            set fail 1
                            break
                        }
                    } else {
                        set argopt [string toupper $argopt]
                        if ![ParserSystem $argopt] {
                            set fail 1
                            break
                        }
                    }
                } elseif [regexp -nocase $exp_bsstate $CurrentLine {} bsstate] {
                    set bsstate [string trim [string toupper $bsstate]]
                    if { ![string compare $bsstate "ENABLED"] } {
                        set UseBaseSystemsState 1
                    } elseif { ![string compare $bsstate "DISABLED"] } {
                        set UseBaseSystemsState 0
                    } else {
                        W [_ "Expected ENABLED | DISABLED at line %1\$i in %2\$s" $LineCounter $ShortFileName]
                        set fail 1
                        break
                    }
                } elseif [regexp -nocase $exp_maghidden $CurrentLine -> lstmag] {
                    set lstmag [regexp -inline -all {[^,]+} $lstmag]
                    foreach _mag $lstmag {
                        # quizas deberiamos comprobar que las magnitudes
                        # estan definidas.
                        lappend Hidden [string toupper [string trim $_mag]]
                    }
                } elseif [regexp -nocase $exp_udstate $CurrentLine {} udstate] {
                    set udstate [string trim [string toupper $udstate]]
                    if { ![string compare $udstate "ENABLED"] } {
                        set UserDefinedState 1
                    } elseif { ![string compare $udstate "DISABLED"] } {
                        set UserDefinedState 0
                    } else {
                        W [_ "Expected ENABLED | DISABLED at line %1\$i in %2\$s" $LineCounter $ShortFileName]
                        set fail 1
                        break
                    }
                } else {
                    if {[string equal -nocase [string range $CurrentLine 0 11] "DefineSymbol"]} {
                        eval ::GiDTeXSymbols::DefineSymbol [string range $CurrentLine 12 end]
                    } else {
                        W [_ "Expected (BEGIN TABLE | SYSTEM) | USER DEFINED | USE BASE SYSTEMS | DefineSymbol at line %1\$i in %2\$s" $LineCounter $ShortFileName]
                        set fail 1
                        break
                    }
                }
            }
        }
        close $FileId
    }
    return $fail
}

#  Units::AppendModelFile --
#
#        Lee un archivo de definicion de unidades de trabajo de un modelo.
#
#  Argumentos:
#
#        filename   nombre del fichero
#
#  Resultados:
#
#       Se tendra la unidad del modelo geometrico, el sistema de unidades de
#       salida al calculo y posiblemente un sistema de unidades propio del modelo.
#       El sistema del modelo estara en una tabla aparte.
#

proc Units::AppendModelFile { fname } {
    variable FileName $fname
    variable FileId
    variable CurrentLine
    variable LineCounter  0
    variable bInModelFile 1
    variable bInGidFile   0
    variable ModelUnit
    variable ModelSysName
    variable SysNames
    variable UseBaseSystemsState
    
    variable exp_modelunit
    variable exp_udsystem
    variable exp_bsstate
    
    set result 1
    if { [catch {open $fname} FileId] } {
        set result 0
        W $FileId
    } else {
        set prb  0
        set mdl  0
        set usys 0
        while { ![eof $FileId] } {
            gets $FileId CurrentLine
            incr LineCounter
            # ---- Elimino los espacios al inicio y final de la linea ----
            
            set CurrentLine [string trim $CurrentLine]
            if { [string match #* $CurrentLine] } { continue }
            if { [string length $CurrentLine] } {
                if [regexp -nocase $exp_modelunit $CurrentLine {} name value] {
                    set name [string toupper $name]
                    if [string compare $name "MODEL"] {
                        set lsys [concat [GiD_Info unitssystems gid] [GiD_Info unitssystems prj]]                        
                        set msn [string trim [string toupper $value]]
                        if { [string compare $msn "USER-DEFINED"] && [lsearch  $lsys $msn] == -1 } {
                            W [_ "Error in line %s at %s: system %s is not defined" $LineCounter $FileName $value]
                            set result 0
                            break                            
                        }
                        set ModelSysName [string trim [string toupper $value]]
                        set prb 1
                    } else {
                        set mustr [string trim $value]
                        set mdl 1
                    }
                } elseif [regexp -nocase $exp_bsstate $CurrentLine {} bsstate] {
                    set bsstate [string trim [string toupper $bsstate]]
                    if { ![string compare $bsstate "ENABLED"] } {
                        set UseBaseSystemsState 1
                    } elseif { ![string compare $bsstate "DISABLED"] } {
                        set UseBaseSystemsState 0
                    } else {
                        W [_ "Expected ENABLED | DISABLED at line %s in %s" $LineCounter $FileName]
                        set result 0
                        break
                    }
                } else {
                    if [regexp -nocase $exp_udsystem $CurrentLine] {
                        if [ParserSystem USER-DEFINED] {
                            set usys 1
                        } else {
                            set result 0
                            break
                        }
                    } else {
                        W [_ "Unknown structure at line: %s in file: %s" $LineCounter $FileName]
                        set result 0
                        break
                    }
                }
                if { $prb && $mdl && $usys } break
            }
        }
        close $FileId
        if { !$result } {
            return 1
        }
        if { $mdl } {
            set found [FindUnitGidAndPrjStr $mustr ::Units::ModelUnit]
            if { !$found } {
                W [_ "Unit %s not found" $mustr]
            }            
        }
        if { !$mdl || $ModelUnit == {} } {
            W [_ "There is not model unit defined in %s" $fname]
            return 1
        }
        if { [lindex $ModelUnit 0] != "LENGTH" } {
            set ModelUnit {}
            W [_ "Model unit %s must be defined within LENGTH magnitude in %s" $mustr $fname]
            return 1
        }
        if { !$prb } {
            W [_ "There must be a unit system referenced in %s" $fname]
            return 1
        }
    }
    return 0
}

proc Units::ParserTable {} {
    variable FileName
    variable FileId
    variable CurrentLine
    variable LineCounter
    
    set isok 1
    while { ![eof $FileId] } {
        gets $FileId CurrentLine
        incr LineCounter
        
        # ---- Elimino los espacios al inicio y final de la linea ----
        
        set CurrentLine [string trim $CurrentLine]
        if { [string length $CurrentLine] } {
            if { ![string compare [string toupper $CurrentLine] "END"] } {
                break
            }
            set r [ParserTableLine]
            if { ![lindex $r 0] } {
                set isok 0
                W [_ "Error at line %s in file %s : %s" $LineCounter $FileName [lindex $r 1]]
            }
        }
    }
    return $isok
}

# busca la magnitud m en la lista l

proc Units::FindMagnitud { l m } {
    set i 0
    foreach mi $l {
        if ![string compare [lindex $mi 0] $m] {
            return $i
        }
        incr i
    }
    return -1
}

proc Units::ParserSystem { system_name } {
    variable FileName
    variable FileId
    variable CurrentLine
    variable LineCounter
    variable SysNames
    variable Systems
    variable OwnerSystem
    
    variable bInModelFile
    variable bInGidFile
    
    variable exp_sysunit
    
    set system_name [string trim $system_name]
    if { [lsearch -exact $SysNames $system_name] != -1 } {
        W [_ "Error in %s, line %s : system %s is already defined" $FileName $LineCounter $system_name]
        return 0
    }
    set efound 0
    set blendefined 0
    while { ![eof $FileId] } {
        gets $FileId CurrentLine
        incr LineCounter
        
        # ---- Elimino los espacios al inicio y final de la linea ----
        
        set CurrentLine [string trim $CurrentLine]
        if { [string length $CurrentLine] } {
            if { ![string compare [string toupper $CurrentLine] "END"] } {
                set efound 1
                break
            }
            if [regexp -nocase $exp_sysunit $CurrentLine {} mag unit] {
                ParserUnit::Init $unit
                ParserUnit::Start
                if { $ParserUnit::Status != "END" } {
                    W [lindex $ParserUnit::Status end]
                    return 0
                }
                set SearchInfo [::Units::FindUnit $ParserUnit::Numerator $ParserUnit::Denominator]
                if { ![llength $SearchInfo] } {
                    set SearchInfo [FindUnitGidAndPrj $ParserUnit::Numerator $ParserUnit::Denominator]
                }
                if { [llength $SearchInfo] } {
                    set where [lindex $SearchInfo 2]
                    set magdef [lindex $SearchInfo 0]
                    set mag [string trim [string toupper $mag]]
                    if ![string compare $mag "LENGTH"] {
                        set blendefined 1
                    }
                    if ![string compare $mag $magdef] {
                        if $bInModelFile {
                            if { [FindMagnitud $OwnerSystem $mag] == -1 } {
                                lappend OwnerSystem [list $mag [lindex $SearchInfo 1] $where]
                            } else {
                                W [_ "%s already referenced in line %s at file %s" $mag $LineCounter $FileName]
                            }
                        } else {
                            lappend Systems($system_name) [list $mag [lindex $SearchInfo 1] $where]
                        }
                    } else {
                        W [_ "Error at line %s: %s should be referenced only in magnitude %s, not in %s." $LineCounter $unit $magdef $mag]
                        return 0
                    }
                } else {
                    W [_ "Error at line %s in file %s: unit %s not defined." $LineCounter $FileName [string trim $unit]]
                    return 0
                }
            } else {
                W [_ "Syntax error at line %s in file %s." $LineCounter $FileName]
                return 0
            }
        }
    }
    
    if $bInModelFile {
        if !$blendefined {
            set OwnerSystem ""
            W [_ "Error: LENGTH not referenced in the unit system defined in file %s." $FileName]
            return 0
        }
    } elseif !$blendefined {
        set sysdefined [GiD_Info unitssystems gid]
        if { [lsearch $sysdefined $system_name] == -1 } {
            set Systems($system_name) ""
            W [_ "Error:  LENGTH not referenced in the unit system %s, defined in file %s." $system_name $FileName]
            return 0
        }
    }
    
    if !$efound {
        W [_ "Error: END expected at line %s in file %s." $LineCounter $FileName]
        return 0
    }
    if !$bInModelFile {
        lappend SysNames $system_name
    }
    return 1
}

#  FindUnit --
#
#        Dada una unidad ( Num/Den ) determina la magnitud y el indice
#        dentro de la magnitud de dicha unidad.
#
#  Argumentos:
#
#        Numerator       numerador de la unidad
#        Denominator     denominador de la unidad
#
#  Resultados:
#        Retorna una lista con el nombre de la magnitud como primer elemento
#        y el indice de la unidad como ultimo elemento
#

proc Units::FindUnit { Numerator Denominator } {
    variable MagNames
    variable Magnitudes
    variable bInGidFile
    
    set IdxFound -1
    set N [lsort $Numerator]
    set D [lsort $Denominator]
    foreach m $MagNames {
        set Idx 0
        set UnitsList $Units::Magnitudes($m)
        foreach u $UnitsList {
            if { ![string compare [lsort [lindex $u 1]] $N] && \
                ![string compare [lsort [lindex $u 2]] $D] } {
                set IdxFound $Idx
                break
            }
            incr Idx
        }
        if { $IdxFound != -1 } {
            if $bInGidFile {
                return [list $m $IdxFound 0]
            } else {
                return [list $m $IdxFound 1]
            }
        }
    }
    return {}
}

#  Busca la unidad en la tabla global

proc FindUnitOnlyGid { Numerator Denominator } {
    set IdxFound -1
    set N [lsort $Numerator]
    set D [lsort $Denominator]
    set MagNames [GiD_Info magnitudes 0]
    foreach m $MagNames {
        set Idx 0
        set UnitsList [GiD_Info magnitudes 0 $m UNITS]
        foreach u $UnitsList {
            if { ![string compare [lsort [lindex $u 1]] $N] && \
                ![string compare [lsort [lindex $u 2]] $D] } {
                set IdxFound $Idx
                break
            }
            incr Idx
        }
        if { $IdxFound != -1 } {
            break
        }
    }
    if { $IdxFound == -1 } {
        return {}
    } else {
        return [list $m $IdxFound 0]
    }
}

# Busca la unidad en la tabla del proyecto y luego en la de GiD

proc FindUnitGidAndPrj { Numerator Denominator } {
    set IdxFound -1
    set N [lsort $Numerator]
    set D [lsort $Denominator]
    foreach where {1 0} {
        set MagNames [GiD_Info magnitudes $where]
        foreach m $MagNames {
            set Idx 0
            set UnitsList [GiD_Info magnitudes $where $m UNITS]
            foreach u $UnitsList {
                if { ![string compare [lsort [lindex $u 1]] $N] && ![string compare [lsort [lindex $u 2]] $D] } {
                    set IdxFound $Idx
                    break
                }
                incr Idx
            }
            if { $IdxFound != -1 } {
                return [list $m $IdxFound $where]
            }
        }
    }
    return {}
}

proc FindUnitGidAndPrjStr { ustr uiname } {
    upvar \#0 $uiname v
    
    ParserUnit::Init $ustr
    ParserUnit::Start
    if { $ParserUnit::Status != "END" } {
        #W [lindex $ParserUnit::Status end]
        return 0
    }
    
    set v [FindUnitGidAndPrj $ParserUnit::Numerator $ParserUnit::Denominator]
    if [llength $v] {
        return 1
    } else {
        #W [_ "Unit %s not found" $ustr]
        return 0
    }
}

proc FindUnitInList { Numerator Denominator UnitsList} {
    set N [lsort $Numerator]
    set D [lsort $Denominator]
    set Idx 0
    foreach u $UnitsList {
        if { ![string compare [lsort [lindex $u 1]] $N] && ![string compare [lsort [lindex $u 2]] $D] } {
            return $Idx
        }
        incr Idx
    }
    return {}
}

namespace eval GiDTeXSymbols {
    namespace export Get ForceLoadFont
    
    #         \\C      {iso8859-7 "\xB0\C"}
    #         \\F      {iso8859-7 "\xB0\F"}
    #         \\K      {iso8859-7 "\xB0\K"}
    #         �C      {iso8859-7 "\xB0\C"}
    #         �F      {iso8859-7 "\xB0\F"}
    #         �K      {iso8859-7 "\xB0\K"}
    
    variable SymbolTable
    
    set SymbolTable(\\mult) [encoding convertfrom iso8859-7 \u00B7]
    set SymbolTable(\\degree) [encoding convertfrom iso8859-7 \u00B0]
    set SymbolTable(\\ohm) [encoding convertfrom iso8859-7 \u00D9]
    set SymbolTable(Cel) "[encoding convertfrom iso8859-7 \u00B0]C"
    set SymbolTable(Kel) "K"
    set SymbolTable(Far) "[encoding convertfrom iso8859-7 \u00B0]F"
    
    variable SymbolNames [list \\mult \\degree \\ohm Cel Kel Far]
    
    variable fontloaded 0
    
    proc ForceLoadFont {} {
        variable fontloaded
        
        if {!$fontloaded} {
            font measure SmallFont [encoding convertfrom iso8859-7 "\u00D9"]
            catch {
                label .__label__GID -text [encoding convertfrom iso8859-7 "\u00D9"]
            }
            set fontloaded 1
        }
    }
    
    proc DefineSymbol {txt utxt args} {
        variable SymbolTable
        
        array set options {
            -encoding iso8859-1
            -replace 0
        }
        array set options $args
        if {!$options(-replace) && [info exists SymbolTable($txt)]} {
            return
        }
        set SymbolTable($txt) [encoding convertfrom $options(-encoding) $utxt]
        #eval encoding convertfrom $enc $utxt
    }
    
    proc Get { name } {
        variable SymbolTable
        
        return $SymbolTable($name)
    }
}

proc CreateExpTextList { Lista rTxtLista } {
    upvar $rTxtLista TxtLista
    
    set LastStr ""
    set adddot  0
    set TxtLista {}
    foreach e $Lista {
        if { $adddot } {
            append LastStr [GiDTeXSymbols::Get "\\mult"]
        }
        set adddot 1
        if { [llength $e] == 2 } {
            set txt [lindex $e 0]
            if {![catch "GiDTeXSymbols::Get $txt" _txt]} {
                set txt $_txt
            }
            append LastStr $txt
            lappend TxtLista $LastStr [lindex $e 1]
            set LastStr ""
        } else {
            if {[catch "GiDTeXSymbols::Get $e" txt]} {
                set txt $e
            }
            append LastStr $txt
        }
    }
    if { $LastStr != "" } {
        lappend TxtLista $LastStr
    }
}

# Retorna la lista de sistemas definidos

proc Units::GetUnitsSystems {} {
    variable SysNames
    variable UseBaseSystemsState
    set PtSys [GiD_Info unitssystems prj]
    if { $UseBaseSystemsState } {
        foreach s [GiD_Info unitssystems gid] {
            if { [lsearch -exact $PtSys $s] == -1 } {
                lappend PtSys $s
            }
        }
    }
    return $PtSys
}

# Retorna la informacion de la unidad

proc Units::GetUnit { uinfo } {
    lassign $uinfo magnitude units_basic_or_reference index_gid_or_project
    return [GiD_Info magnitudes $index_gid_or_project $magnitude $units_basic_or_reference]    
}

# Retorna las magnitudes dentro del sistema system_name

proc Units::GetSystem { system_name } {
    variable Systems
    if { $system_name == "" } {
        set sys [GiD_Info unitssystems usersys]
    } else {
        set sys [GiD_Info unitssystems $system_name prj]
        
        foreach uinfo [GiD_Info unitssystems $system_name gid] {
            if { [FindMagnitud $sys [lindex $uinfo 0]] == -1 } {
                lappend sys $uinfo
            }
        }
    }
    return $sys
}

proc Units::ConvertTemperature { value ufrom uto } {
    GidConvertTemperature $value [TemperatureFactor $ufrom $uto]
}

#split string in value and unit
#e.g. "5.3mm" to {5.3 mm} 
proc Units::SplitValueUnit { str } {
    if { [string is double -strict $str] } {
        return [list $str ""]
    }
    set value ""
    set unit ""
    variable exp_real
    variable exp_space
    variable exp_strunit
    set eunit "^($exp_real)$exp_space\($exp_strunit\)$"
    regexp -nocase $eunit $str {} value {} unit
    return [list $value $unit]
}

proc Units::ConvertValueUnit { str system_name } {    
    variable uinfovar
    variable FactorCache
    set value ""
    set unitcvt ""    
    set str [string trim $str]
    lassign [Units::SplitValueUnit $str] value unit
    if { $unit != "" } {
        # look for unit first in cache
        set key [list $system_name $unit] ;#take into account the system name, non only the unit name
        if { [info exists FactorCache($key)] } {
            lassign $FactorCache($key) factor unit
            return [list [format %g [expr $value*$factor]] $unit]
        } else {
            set unitcvt $unit
            if { [FindUnitGidAndPrjStr $unit ::Units::uinfovar] } {
                set syslist [GetSystem $system_name]
                if { [llength $syslist] } {
                    set mag [lindex $uinfovar 0]
                    if { [set idx [FindMagnitud $syslist $mag]] != -1 } {
                        set uiout [lindex $syslist $idx]
                        set u0 [GiD_Info magnitudes [lindex $uinfovar 2] $mag [lindex $uinfovar 1]]
                        set u1 [GiD_Info magnitudes [lindex $uiout 2] $mag [lindex $uiout 1]]                   
                        if { [string compare $mag "TEMPERATURE"] } {
                            set factor [expr double([lindex $u1 0]) / [lindex $u0 0]]
                            set value [format "%g" [expr $value*$factor]]
                        } else {
                            set value [ConvertTemperature $value [lindex $u0 end] [lindex $u1 end]]
                        }
                        set unitcvt [lindex $u1 end]
                    }
                } else {
                    W [_ "Unit system not found: %s" $system_name]
                }
            } else {
                W [_ "Error in " $str]
            }
        }
        set FactorCache($key) [list $factor $unitcvt]        
    } else {
        W [_ "Invalid unit expresion: %s" $str]
    }    
    return [list $value $unitcvt]
}

proc GidConvertValueUnit { str } {
    set system_name [GiD_Info unitssystems prbsys]
    set res [::Units::ConvertValueUnit $str $system_name]
    if { $res == {} } {
        error [_ "Error converting '%s' within units system '%s'" $str $system_name]
    }
    return $res
}

proc Units::ConvertLength { value uinfo system_name } {
    variable Magnitudes
    variable FactorCache
    set u0 [GiD_Info magnitudes [lindex $uinfo 2] LENGTH [lindex $uinfo 1]]    
    set unit [lindex $u0 end]
    set key [list $system_name $unit] ;#take into account the system name, non only the unit name
    if { [info exists FactorCache($key)] } {
        lassign $FactorCache($key) factor unit
        return [list [format "%g" [expr $value*$factor]] $unit]
    }
    set unicvt $unit
    set syslist [GetSystem $system_name]
    if [llength $syslist] {
        if { [set idx [FindMagnitud $syslist LENGTH]] != -1 } {
            set uiout [lindex $syslist $idx]
            set u1 [GiD_Info magnitudes [lindex $uiout 2] LENGTH [lindex $uiout 1]]
            set factor [expr double([lindex $u1 0]) / [lindex $u0 0]]
            set value [format "%g" [expr $value * $factor]]
            set unitcvt [lindex $u1 end]
        }
    }
    
    set FactorCache($key) [list $factor $unitcvt]
    return [list $value $unitcvt]
}

proc GidConvertLength { value } {
    set system_name [GiD_Info unitssystems prbsys]
    set uinfo  [GiD_Info unitssystems modunit]    
    return [::Units::ConvertLength $value $uinfo $system_name]
}

# -----------------------------------------------------------------
# TemperatureFactor, GidGetUnitFactor are GidGetLengthFactor
# are new procs used from C to do catching of units conversion factors
# -----------------------------------------------------------------

proc TemperatureFactor { ufrom uto } {
    if { [string equal $ufrom C] || [string equal $ufrom {\C}]
        || [string equal $ufrom "\u00BAC"]
        || [string equal $ufrom "\u00B0C"]
        || [string equal $ufrom "Cel"]} {
        if { [string equal $uto F] || [string equal $uto {\F}]
            || [string equal $uto "\u00BAF"]
            || [string equal $uto "\u00B0F"]
            || [string equal $uto "Far"]} {
            # Convierte de C a F
            return -1
        } elseif { [string equal $uto K] || [string equal $uto {\K}]
            || [string equal $uto "\u00BAK"]
            || [string equal $uto "\u00B0K"]
            || [string equal $uto "Kel"]} {
            # Convierte de C a K
            return -2
        }
    } elseif { [string equal $ufrom F] || [string equal $ufrom {\F}]
        || [string equal $ufrom "\u00BAF"]
        || [string equal $ufrom "\u00B0F"]
        || [string equal $ufrom "Far"]} {
        if { [string equal $uto C] || [string equal $uto {\C}]
            || [string equal $uto "\u00BAC"]
            || [string equal $uto "\u00B0C"]
            || [string equal $uto "Cel"]} {
            # Convierte de F a C
            return -3
        } elseif { [string equal $uto K] || [string equal $uto {\K}]
            || [string equal $uto "\u00BAK"]
            || [string equal $uto "\u00B0K"]
            || [string equal $uto "Kel"]} {
            # Convierte de F a K
            return -4
        }
    } elseif { [string equal $ufrom K] || [string equal $ufrom {\K}]
        || [string equal $ufrom "\u00BAK"]
        || [string equal $ufrom "\u00B0K"]
        || [string equal $ufrom "Kel"]} {
        if { [string equal $uto C] || [string equal $uto {\C}]
            || [string equal $uto  "\u00BAC"]
            || [string equal $uto  "\u00B0C"]
            || [string equal $uto "Cel"]} {
            # Convierte de K a C
            return -5
        } elseif { [string equal $uto F] || [string equal $uto {\F}]
            || [string equal $uto "\u00BAF"]
            || [string equal $uto "\u00B0F"]
            || [string equal $uto "Far"]} {
            # Convierte de K a F
            return -6
        }
    }
    return -7
}

proc GidTclGetUnitFactor { unit } {
    global uinfovar
    set factor 0
    set system_name [GiD_Info unitssystems prbsys]
    if { [FindUnitGidAndPrjStr $unit uinfovar] } {
        set syslist [::Units::GetSystem $system_name]
        if [llength $syslist] {
            set mag [lindex $uinfovar 0]
            if { [set idx [::Units::FindMagnitud $syslist $mag]] != -1 } {
                set uiout [lindex $syslist $idx]
                set u0 [GiD_Info magnitudes [lindex $uinfovar 2] $mag [lindex $uinfovar 1]]
                set u1 [GiD_Info magnitudes [lindex $uiout 2] $mag [lindex $uiout 1]]
                if { [string equal $mag "TEMPERATURE"] } {
                    set factor [TemperatureFactor [lindex $u0 end] [lindex $u1 end]]
                } else {
                    set factor [expr double([lindex $u1 0]) / [lindex $u0 0]]
                }
            }
        } else {
            W [_ "Unit system not found: %s" $system_name]
        }
    } else {
        W [_ "Unit %s not found" $unit]
    }
    unset uinfovar
    return $factor
}

proc GidGetLengthFactor { } {
    set system_name [GiD_Info unitssystems prbsys]
    set uinfo  [GiD_Info unitssystems modunit]
    set u0 [GiD_Info magnitudes [lindex $uinfo 2] LENGTH [lindex $uinfo 1]]
    set syslist [::Units::GetSystem $system_name]
    set factor 1.0
    if [llength $syslist] {
        if { [set idx [::Units::FindMagnitud $syslist LENGTH]] != -1 } {
            set uiout [lindex $syslist $idx]
            set u1 [GiD_Info magnitudes [lindex $uiout 2] LENGTH [lindex $uiout 1]]
            set factor [expr double([lindex $u1 0]) / [lindex $u0 0]]
        }
    }
    return $factor
}

# ------------------------------------------------------------------
# END de rutinas de catching.
# ------------------------------------------------------------------

proc Units::GetUnitStr { smag system_name } {
    variable Magnitudes
    set syslist [GetSystem $system_name]
    if { [set idx [FindMagnitud $syslist $smag]] != -1 } {
        set uinfo [lindex $syslist $idx]
        set u [GiD_Info magnitudes [lindex $uinfo 2] $smag [lindex $uinfo 1]]        
        return [lindex $u end]
    }
    W [_ "Error: magnitude %s is not defined within system %s" $smag $system_name]
    return ""
}

proc GidGetUnitStr { smag } {
    set system_name [GiD_Info unitssystems prbsys]
    return [::Units::GetUnitStr [string toupper [string trim $smag]] $system_name]
}

proc LoadGidUnits {} {
    global GIDDEFAULTTCL
    Units::Init
    set unipath [file join $::GIDDEFAULTTCL units_defaults.uni]
    Units::AppendFile $unipath 1
}

proc GetUserDefSysState {} {
    return $::Units::UserDefinedState
}

proc GetUnitsList { MagName } {
    return $::Units::Magnitudes($MagName)
}

proc GetBasicUnit { MagName } {
    return $::Units::Magnitudes($MagName,BASIC)
}

proc GetReferenceUnit { MagName } {
    return $::Units::Magnitudes($MagName,REFERENCE)
}

proc GetUnitsSystem { system_name } {
    return $::Units::Systems($system_name)
}

proc LoadProblemTypeUnits { fname } {
    # esta variable se inicializa aqui para que Units::Init
    # no la sobreescriba. De hecho la lista de magnitudes
    # ocultas solo tiene significado el en .uni del pt.
    namespace eval Units {
        variable MagHidden {}
    }
    Units::Init
    set fail [Units::AppendFile $fname]
    if { !$fail } {
        namespace eval ::Units {
            variable Hidden
            variable MagHidden $Hidden
        }
    }
    return $fail
}

proc LoadModelUnits { fname } {
    Units::Init
    return [Units::AppendModelFile $fname]
}

proc TestUnitsSystems {} {
    W [set l [GiD_Info unitssystems gid]]
    foreach i $l {
        W [GiD_Info unitssystems $i gid]
    }
}

proc Units::QuestionHasUnits { question } {
    set has_units 0
    if { [string range $question end-6 end] == "#UNITS#" } {
        set has_units 1
    }
    return $has_units
}

#check that values are valid for current units, 
#e.g. applied values of conditions where the condition has been redefined with other magnitude, 
#     or same magnitude but the unit is not in the list of current units for this magnitude (because .uni has changed)
proc Units::FixWrongValues { } {
    set numchanges 0
    set messages ""
    lassign [Units::FixWrongValuesConditions] numchanges_local messages_local
    incr numchanges $numchanges_local
    append messages $messages_local
    
    lassign [Units::FixWrongValuesMaterials] numchanges_local messages_local
    incr numchanges $numchanges_local
    append messages $messages_local

    lassign [Units::FixWrongValuesGeneralData] numchanges_local messages_local
    incr numchanges $numchanges_local
    append messages $messages_local
    
    lassign [Units::FixWrongValuesIntervalData] numchanges_local messages_local
    incr numchanges $numchanges_local
    append messages $messages_local
    
    return [list $numchanges $messages]
}

proc Units::FixWrongValuesConditions { } {
    set numchanges 0
    set messages ""
    set num_intervals [lindex [GiD_Info intvdata num] 1]
    if { $num_intervals } {
        set intervals [objarray new_from_to intarray 1 $num_intervals]
    } else {
        set intervals [list 0]
    }
    foreach type {point line surface volume layer group} {
        set over over_$type
        foreach condition [GiD_Info conditions $over] {
            array unset allowed_field_units
            set questions_values_default [lrange [GiD_Info conditions $condition] 2 end]
            set i_field 0
            foreach {question value_original} $questions_values_default {
                if { [Units::QuestionHasUnits $question] } {
                    set unit ""
                    lassign [Units::SplitValueUnit $value_original] value unit
                    if { $unit == "" } {
                        set value_unit_default [GiD_AccessValue get -default intvdata $interval $question]
                        lassign [Units::SplitValueUnit $value_unit_default] value_default unit_default                
                        if { $unit_default != "" } {
                            set magnitude [GiD_Units get magnitude_from_unit $unit_default]
                            set magnitude_units [GiD_Units get magnitude_units $magnitude]
                        } else {
                            set magnitude_units ""
                        }
                    } else {
                        set magnitude [GiD_Units get magnitude_from_unit $unit]
                        set magnitude_units [GiD_Units get magnitude_units $magnitude]
                    }
                    set allowed_field_units($i_field) $magnitude_units
                }
                incr i_field
            }
            foreach mode {geometry mesh} {
                foreach interval $intervals {
                    if { $interval == 0 } {
                        set items [GiD_Info conditions $condition $mode]
                    } else {
                        set items [GiD_Info conditions -interval $interval $condition $mode]
                    }
                    foreach item $items {
                        set fields_and_values_to_fix [list]
                        set values [lrange $item 3 end]
                        set i_field 0
                        foreach value_original $values {
                            if { [info exists allowed_field_units($i_field)] } {
                                lassign [Units::SplitValueUnit $value_original] value unit
                                if { $unit != "" } {
                                    if { [lsearch $allowed_field_units($i_field) $unit] == -1 } {
                                        set new_value "${value}[lindex $allowed_field_units($i_field) 0]"
                                        lappend fields_and_values_to_fix $i_field $new_value
                                    }
                                }
                            }
                            incr i_field
                        }
                        if { [llength $fields_and_values_to_fix] } {
                            set selection [list ]
                            set item_0 [lindex $item 0]
                            if { $mode == "geometry" } {
                                set entities_type ${type}s
                                set figure_num [lindex $item 1]
                                set selection [list $figure_num]                                
                            } else {
                                 #"mesh"
                                if { $item_0 == "N" } {
                                    set entities_type "nodes"
                                    set figure_num [lindex $item 1]
                                    set selection [list $figure_num]
                                } elseif { $item_0 == "E" } {
                                    set entities_type "elements"
                                    set figure_num [lindex $item 1]
                                    set selection [list $figure_num]
                                } else {
                                    set entities_type "face_elements"
                                    set figure_num $item_0
                                    set face_num [lindex $item 1]
                                    set selection [list $figure_num $face_num]                                    
                                }                                
                            }                        
                            set num_changes_local [GiD_AccessValueAssignedCondition -field_index set $condition $entities_type $fields_and_values_to_fix $selection]                            
                            incr numchanges $num_changes_local
                            append messages [_ "Fix invalid units: condition '%s' new values '%s' on %s %s" $condition $fields_and_values_to_fix $entities_type $selection]\n
                        }
                    }
                }
            }
        }
    }
    return [list $numchanges $messages]
}

proc Units::FixWrongValuesMaterials { } {
    set numchanges 0
    set messages ""
    foreach material [GiD_Info materials] {
        set questions_and_values_to_fix [list]
        set questions_values [lrange [GiD_Info materials $material] 1 end]        
        foreach {question value_original} $questions_values {
            if { [Units::QuestionHasUnits $question] } {
                set unit ""
                lassign [Units::SplitValueUnit $value_original] value unit
                if { $unit == "" } {
                    set value_unit_default [GiD_AccessValue get -default material $material $question]
                    lassign [Units::SplitValueUnit $value_unit_default] value_default unit_default                
                    if { $unit_default != "" } {
                        #better set default value, maybe current value is in other units!!
                        set new_value "${value_default}$unit_default"     
                        lappend questions_and_values_to_fix $question $new_value
                    }
                } else {
                    set magnitude [GiD_Units get magnitude_from_unit $unit]
                    set magnitude_units [GiD_Units get magnitude_units $magnitude]
                    if { [lsearch $magnitude_units $unit] == -1 } {                                
                        set new_value "${value}[lindex $magnitude_units 0]"     
                        lappend questions_and_values_to_fix $question $new_value                                           
                    }                
                }                                
            }            
        }
        if { [llength $questions_and_values_to_fix] } {
            foreach {question new_value} $questions_and_values_to_fix {
                GiD_AccessValue set material $material $question $new_value
            }            
            incr numchanges
            append messages [_ "Fix invalid units: material '%s' new values '%s'" $material $questions_and_values_to_fix]\n
        }
    }
    return [list $numchanges $messages]
}

proc Units::FixWrongValuesGeneralData { } {
    set numchanges 0
    set messages ""
    
    set questions_and_values_to_fix [list]
    set questions_values [lrange [GiD_Info gendata] 1 end]        
    foreach {question value_original} $questions_values {
        if { [Units::QuestionHasUnits $question] } {
            set unit ""
            lassign [Units::SplitValueUnit $value_original] value unit
            if { $unit == "" } {
                set value_unit_default [GiD_AccessValue get -default gendata $question]
                lassign [Units::SplitValueUnit $value_unit_default] value_default unit_default                
                if { $unit_default != "" } {
                    #better set default value, maybe current value is in other units!!
                    set new_value "${value_default}$unit_default"     
                    lappend questions_and_values_to_fix $question $new_value
                }
            } else {
                set magnitude [GiD_Units get magnitude_from_unit $unit]
                set magnitude_units [GiD_Units get magnitude_units $magnitude]
                if { [lsearch $magnitude_units $unit] == -1 } {                                
                    set new_value "${value}[lindex $magnitude_units 0]"     
                    lappend questions_and_values_to_fix $question $new_value                                           
                }                
            }                                
        }            
    }
    if { [llength $questions_and_values_to_fix] } {
        foreach {question new_value} $questions_and_values_to_fix {
            GiD_AccessValue set gendata $question $new_value
        }
        incr numchanges
        append messages [_ "Fix invalid units: general data new values '%s'" $questions_and_values_to_fix]\n
    }    
    return [list $numchanges $messages]
}

proc Units::FixWrongValuesIntervalData { } {
    set numchanges 0
    set messages ""
    set num_intervals [lindex [GiD_Info intvdata num] 1]
    set intervals [objarray new_from_to intarray 1 $num_intervals]
    foreach interval $intervals {
        set questions_and_values_to_fix [list]
        set questions_values [lrange [GiD_Info intvdata -interval $interval] 1 end]        
        foreach {question value_original} $questions_values {
            if { [Units::QuestionHasUnits $question] } {
                set unit ""
                lassign [Units::SplitValueUnit $value_original] value unit
                if { $unit == "" } {
                    set value_unit_default [GiD_AccessValue get -default intvdata $interval $question]
                    lassign [Units::SplitValueUnit $value_unit_default] value_default unit_default                
                    if { $unit_default != "" } {
                        #better set default value, maybe current value is in other units!!
                        set new_value "${value_default}$unit_default"     
                        lappend questions_and_values_to_fix $question $new_value
                    }
                } else {
                    set magnitude [GiD_Units get magnitude_from_unit $unit]
                    set magnitude_units [GiD_Units get magnitude_units $magnitude]
                    if { [lsearch $magnitude_units $unit] == -1 } {                        
                        set new_value "${value}[lindex $magnitude_units 0]"
                        lappend questions_and_values_to_fix $question $new_value                       
                    }
                }
            }
        }
        if { [llength $questions_and_values_to_fix] } {
            foreach {question new_value} $questions_and_values_to_fix {
                GiD_AccessValue set intvdata $interval $question $new_value
            }
            incr numchanges
            append messages [_ "Fix invalid units: interval data %s new values '%s'" $interval $questions_and_values_to_fix]\n        
        }
    }    
    return [list $numchanges $messages]
}
