# -*- mode: Tcl; tcl-indent-level :  4 -*-

# some times gid still hasn't defined this variable, needed in gid_filesystem.tcl
if { ![ info exists ::gid_start_mode]} {
    set ::gid_start_mode normal
}

foreach filename {gid_minimal_init.tcl gid_filesystem.tcl } {
    source [file join $::GIDDEFAULT scripts $filename]
}

if { $::gid_start_mode == "client" || $::gid_start_mode == "normal" } {
    #do not initialize in gid_start_mode == "server"
    #specially to avoid that calling the proc that appear in tclIndex the file gid_client.tcl is sourced!!
    gid_client_initialize_cursors
}

if { $::gid_start_mode == "client" } {
    if { [catch {gid_client_init} msg] } {
        set txt [_ "Cannot connect with GiD server at %s port %s.\nError %s" $::gid_start_server_ip $::gid_start_port $msg]
        #append txt \n[_ "Continue starting in normal (local) mode instead of client mode"]
        tk_messageBox -message $txt -type ok {*}[ GidUtils::GiDDefaultParentIfPresent]
        #set ::gid_start_mode normal
        exit 1
    }
}

wm withdraw .
if { $::tcl_platform(os) == "Darwin" } {
    catch {
        console hide
    }
}

proc DefineMouseButtons { } {
    if { $::tcl_platform(os) == "Darwin" } {
        # http://wiki.tcl.tk/12987 FAQ 7.7:
        # For historical reasons, MacMice buttons 2 and 3 refer to the right and middle buttons
        # respectively, which is indeed the opposite way round from Windows and *nix systems.
        set ::gid_central_button 3
        set ::gid_middle_button 3
        set ::gid_right_button 2
    } else {
        # use both aliases for middle mouse button
        set ::gid_central_button 2
        set ::gid_middle_button 2
        set ::gid_right_button 3
    }
}

proc DefineAcceleratorKey {} {
    if { [tk windowingsystem] eq "aqua"} {
        set ::acceleratorKey Command
        set ::acceleratorText Command
    } else {
        set ::acceleratorKey Control
        set ::acceleratorText Ctrl
    }
}

DefineMouseButtons
DefineAcceleratorKey

proc WarnWinDirect { text {parent ""} { icn warning}} {
    set options [list -message $text -title Warning -icon $icn -type ok]
    if { ( $parent != "") && [ winfo exists $parent]} {
        lappend options -parent $parent
    } else {
        # try with main window .gid
        if { [ winfo exists .gid] && [ winfo ismapped .gid]} {
            lappend options -parent .gid
        }
    }
    tk_messageBox {*}$options
}

# so that it can be called from other interpreters, like lognoter or ramdebugger
proc InitializeGiDThemes {} {
    #must read themes before splash, because find images is needed
    set err [ catch {
        package require gid_themes
    } err_txt]
    if { $err} {
        # also in stdout for the case when WarnWinDirect is not available
        puts $err_txt
        WarnWinDirect $err_txt
    }
    set err [ catch {
        # this is needed so that splash.png is read with img::png package which supports the
        # 'png -gamma' option
        # otherwise, tk internal png's readers are used instead"
        package require img::png
    } err_txt]
    if { $err} {
        # also in stdout for the case when WarnWinDirect is not available
        puts $err_txt
        WarnWinDirect $err_txt
    }

    set err [ catch {
        gid_themes::ReadThemes
    } err_txt]
    if { $err} {
        # also in stdout for the case when WarnWinDirect is not available
        puts $err_txt
        WarnWinDirect $err_txt
    }
    set theme ""
    if { $::GidPriv(ThemeSelection) != "1" } {
        set theme $::GidPriv(ThemeSelection)
    } else {
        set theme [gid_themes::GetDefaultTheme]
        if { $theme == "" } {
            error "It doesn't exists any theme. Check content of '[ gid_themes::GetThemesBaseFolder]' folder"
        }
    }
    gid_themes::SetCurrentTheme $theme 0
}

InitializeGiDThemes

proc GiDSplash { } {
    global GidPriv
    set mainversion [GiD_Info GiDVersion]
    set txt [concat [_ "Version"] $mainversion]

    set w .splash
    if { [winfo exists $w] } { destroy $w }
    toplevel $w
    # wm attributes $w -topmost 1 ;#splash can hide the error message or window to select opengl by software
    if { $::tcl_platform(platform) == "windows" } {
        wm attributes $w -toolwindow 1
    } else {
        wm withdraw $w
    }
    #wm attributes $w -alpha 0.8
    set im [gid_themes::GetImageCommon $::GidPriv(SplashWindow) 1.0]
    set x [expr [winfo screenwidth .splash]/2-[image width $im]/2]
    set y [expr [winfo screenheight .splash]/2-[image height $im]/2]

    # wm geometry $w +$x+$y
    wm overrideredirect .splash 1
    set base $w.l
    tk::label $base -image $im -relief ridge -borderwidth 2
    #draw version txt over the image
    set font_face "Helvetica 10 bold"
    if { $::tcl_platform(os) == "Darwin"} {
        # in Mac OS X it looks better without the bold face
        set font_face "Helvetica 14"
    }
    tk::label $base.lv -text $txt -font $font_face \
        -foreground #9d9d9d -background #010101 -borderwidth 0 -highlightthickness 0
    set label_pos_x 385
    set label_pos_y 75
    lassign [ gid_themes::_linux_ApplyScreenScaleFactorToInts [ list $label_pos_x $label_pos_y]] label_pos_x label_pos_y
    place $base.lv -x $label_pos_x -y $label_pos_y -in $base
    
    pack $base
    wm geometry $w +$x+$y
    GidUtils::WindowAboveGid $w
    update
    if { $::tcl_platform(platform) != "windows" } {
      wm deiconify $w
    }
    bind $w <1> [list destroy $w]

    after 3000 "if { [ winfo exists $w] } { destroy $w }"
}

proc GiDSplashMessage { txt} {
    set w .splash
    if { $::GidPriv(ShowPackages) } {
        packageOriginal require Tk
    } else {
        package require Tk
    }
    if { ![ winfo exists $w]} {
        # WarnWinText "$txt"
        return
    }
    #wm attributes $w -topmost 1 ;#splash can hide the error message or window to select opengl by software
    wm overrideredirect $w 1
    set lbl $w.show_packages_label
    set max_len 70
    if { ![ winfo exists $lbl]} {
        tk::label $lbl -text [ string repeat " " $max_len] -font "Helvetica 8" \
            -foreground #fefefe -background #010101
        # -x 2 because the image is drawn with a border 2 ( ridge)
        set label_pos_x 2
        set label_pos_y 233
        lassign [ gid_themes::_linux_ApplyScreenScaleFactorToInts [ list $label_pos_x $label_pos_y]] label_pos_x label_pos_y
        place $lbl -x $label_pos_x -y $label_pos_y -in $w.l
    }
    set len_spc [ expr $max_len - [ string length $txt]]
    if { $len_spc > 0} {
        append txt [ string repeat " " $len_spc]
    }
    $lbl configure -text $txt -font "Helvetica 10"
}

if { $::GidPriv(ShowPackages) } {
    proc MyPackage { cmd args} {
        switch $cmd {
            require {
                set len_args [ llength $args]
                if { $len_args >= 1} {
                    set pkg_name [ lindex $args 0]
                    if { $len_args > 1} {
                        # skip -exact flag
                        if { $pkg_name == "-exact"} {
                            set pkg_name [ lindex $args 1]
                        }
                    }
                    lappend ::MPLstPackages $pkg_name
                    incr ::MPNumPackages
                    GiDSplashMessage "Loading $pkg_name package ( $::MPNumPackages ) ..."
                }
            }
            showlist {
                set num_uniq [ llength [ lsort -unique $::MPLstPackages]]
                WarnWinDirect "Num = [ llength $::MPLstPackages] ( $num_uniq)\n$::MPLstPackages"
                return
            }
            default {
            }
        }
        return [ eval packageOriginal $cmd $args]
    }

    rename package packageOriginal
    rename MyPackage package

    set ::MPLstPackages {}
    set ::MPNumPackages 0
}

#ramdebugger
if { [catch {
    package require commR ;#modification (commR) of tcllib comm package
    comm::register gid 1
} err_txt] } {
    set GidPriv(DevelopMenuState) 0 ;#to hide it in menu
    #bgerror "ramdebugger not available: $err_txt"
} else {
    # if this shortcut is changed, remember to also change "-accelerator F12" in tclfileP.tcl
    bind all <F12> {
        package require RamDebugger
    }
}

#test to speech texts using Microsoft Windows Speech com object
set ::SPEECH_WARNLINE 0
if { $::SPEECH_WARNLINE == 1 && $::tcl_platform(platform) == "windows" } {
    set ::GiD_Voice ""
    catch {
        package require tcom
        set ::GiD_Voice [::tcom::ref createobject Sapi.SpVoice]
    }
} else {
    set ::GiD_Voice ""
}

GiDMenu::DisableUpdateMenus

proc InitialGiDRestart { { first_time 1}} {
    set w .iniconf
    if { [info exists ::UseLocalOpenGL] } {
        if { [GiD_Set SoftwareOpenGL] != $::UseLocalOpenGL} {
            GiD_Set SoftwareOpenGL $::UseLocalOpenGL
            after 1000 GiD_Set OpenGL(EmulateFrontBuffer) $::UseLocalOpenGL
        }
    }
    if { [info exists ::UseLocalTheme] } {
        #always apply begining color options (example change from black to black)
        #gid_themes::SetCurrentTheme $::UseLocalTheme 0
        GiD_Set Theme(Current) $::UseLocalTheme
        # this is done after reading user preferences (readdefaults $gid_defaults $cloud;#kernel values)
    }
    destroy $w
    if { !$first_time} {
        RestartGiD
    }
}

proc ShowGraphicsInformation { w_parent } {
    package require gid_cross_platform
    set lst_txt [gid_cross_platform::getGraphicsInformation]
    lappend lst_txt ""
    WarnWin [ join $lst_txt "\n"] $w_parent
}

proc InitialConfiguration { { first_time 1} {configogl 1} {configtheme 1} } {
    if { $first_time && [winfo exists .gid] } {
        wm withdraw .gid
    }
    set w .iniconf
    toplevel $w
    set sx [winfo screenwidth .]
    set sy [winfo screenheight .]
    set txt_width 500    ;# win width is about 529 / 556 / 573 pixels ( padx's, decorations, frames, ...)
    set win_height 350   ;# win height is about 311 / 450 / 488 pixels ( pady's, decorations, frames, ...)
    set width [ expr ( $sx - $txt_width) / 2]
    wm geometry $w +$width+[ expr ( $sy - $win_height) / 2]
    set width $txt_width

    wm resizable $w 0 0

    if { $configogl && $configtheme } {
        wm title $w [_ "Initial themes and graphics configuration"]
    } elseif { $configogl } {
        wm title $w [_ "Initial graphics configuration (OpenGL)"]
    } elseif { $configtheme } {
        wm title $w [_ "Initial themes configuration"]
    }

    if { $configogl } {
        set ::UseLocalOpenGL [GiD_Set SoftwareOpenGL]

        if { $::tcl_platform(os) == "Darwin"} {
            package require gid_cross_platform
            
            set res {}
            set glinfo_exe [ gid_cross_platform::getGraphicsInformationExecutable]
            if { [ file executable $glinfo_exe]} {
                set err  [ catch {
                    set res [ split [ exec $glinfo_exe | grep -i "hardware renderers"] \n]
                } err_txt]
            } else {
                WarnWin "Helper program not found: $glinfo_exe . Asuming no graphics hardware acceleration present."
            }
            
            # if { $err} { WarnWin "Error executing glInfo.exe\n$err_txt" }
            set count 0
            foreach line $res {
                if { [ regexp {([0-9]+) Hardware renderers} $line dummy count]} {
                    if { $count > 0} {
                        break
                    }
                }
            }
            if { $count == 0} {
                # no hardware accelerated hardware found or configured in macOS, so selecting software mode
                set ::UseLocalOpenGL 1
                GiD_Set SoftwareOpenGL 1
                WarnWin "No hardware accelerated renderer found.\nUsing software renderer instead."
            }
        }

        set base_ogl $w.ogl
        ttk::labelframe $base_ogl -text [_ "Initial graphics configuration ( OpenGL)"]:
        ttk::radiobutton $base_ogl.rd1 -text [_ "Fast visualization mode."] -value 0 -variable ::UseLocalOpenGL
        ttk::label $base_ogl.rd1txt -wraplength $width \
            -text [_ "(Models will be drawn quicker but some issues could appear. I'm confident about my Graphic Card and I want to use its capabilities)."] \
            -state disabled
        ttk::radiobutton $base_ogl.rd2 -text [_ "Safe visualization mode."] -value 1 -variable ::UseLocalOpenGL
        ttk::label $base_ogl.rd2txt -wraplength $width \
            -text [_ "(Models will be drawn slower but the artifacts would disappear. I'm experiencing troubles with my Graphic Card and I want to use OpenGL by software)."] \
            -state disabled

        grid $base_ogl.rd1 -sticky w
        grid $base_ogl.rd1txt -sticky new
        grid $base_ogl.rd2 -sticky w
        grid $base_ogl.rd2txt -sticky new

        # if { $::tcl_platform(platform) != "windows"} {
            ttk::button $base_ogl.b_graphics -text [_ "Check graphics"] \
                -command [ list ShowGraphicsInformation $w]
            grid $base_ogl.b_graphics -padx 3 -pady 3
        # }

        grid $base_ogl -sticky nsew  -padx 4 -pady 4
        grid columnconfigure $base_ogl 0 -weight 1
        grid rowconfigure $base_ogl {1 3} -weight 1
    }

    if { $configtheme } {
        set themes_and_labels [gid_themes::GetThemesAndLabels]
        if { [llength $themes_and_labels] >= 2 } {
            set base_theme $w.theme
            ttk::labelframe $base_theme -text [_ "Initial themes configuration"]:
            set listvalues [list]
            set listlabels [list]
            foreach {theme label} $themes_and_labels {
                lappend listvalues $theme
                lappend listlabels [_ $label]
            }
            ttk::label $base_theme.ltheme -text [_ "Theme"]:
            TTKComboBox $base_theme.cbtheme -labels $listlabels -values $listvalues -textvariable ::UseLocalTheme -state readonly
            grid $base_theme.ltheme $base_theme.cbtheme -sticky w -padx 2 -pady 2
            grid $base_theme -sticky news -padx 4 -pady 4
            grid columnconfigure $base_theme 1 -weight 1
        }

        set ::UseLocalTheme [gid_themes::GetDefaultTheme]

        if { $::tcl_platform(os) == "Darwin"} {
            set base_menu_type $w.mt
            ttk::labelframe $base_menu_type -text [_ "GiD menu bar configuration"]:
            ttk::radiobutton $base_menu_type.rd1 -text [_ "Embedded in graphical window"] \
                -value generic -variable ::InitialWindow(MenuType) -command [list GiD_Set Theme(MenuType) generic]
            ttk::radiobutton $base_menu_type.rd2 -text [_ "Mac OS X top menu bar"] \
                -value native -variable ::InitialWindow(MenuType) -command [list GiD_Set Theme(MenuType) native]
            set ::InitialWindow(MenuType) [GiD_Set Theme(MenuType)]
            grid $base_menu_type.rd1 -sticky w
            grid $base_menu_type.rd2 -sticky w
            grid $base_menu_type -sticky news -padx 4 -pady 4
            grid columnconfigure $base_menu_type 0 -weight 1
        }

        ttk::label $w.txt -wraplength $width \
            -text [_ "(This preference, and others related with the graphics configuration, can always be accessed through 'Graphical' panel in the 'Utilities-->Preferences' window.)"] \
            -state disabled
        grid $w.txt -sticky ew -padx 4 -pady 4
    }

    set base_buttons $w.frmbuttons
    ttk::frame $base_buttons -style BottomFrame.TFrame
    ttk::button $base_buttons.b1 -text [_ "Ok"] -command [list InitialGiDRestart $first_time] -style BottomFrame.TButton
    if { [ winfo exists [ GidUtils::GetMainDrawAreaWidget]]} {
        # if main window area exists, don't quit, but close window
        ttk::button $base_buttons.b2 -text [_ "Close"] -command [ list destroy $w ] -style BottomFrame.TButton
    } else {
        ttk::button $base_buttons.b2 -text [_ "Quit"] -command [ list exit 0 ] -style BottomFrame.TButton
    }
    grid $base_buttons.b1 $base_buttons.b2 -pady 4
    grid $base_buttons -sticky news -pady 4 -row 4
    grid columnconfigure $base_buttons {0 1} -weight 1

    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 3 -weight 1

    tkwait window $w
}


proc CreateIcon {} {
    set logos {}
    set lst_sizes [ list 512 256 128 64 48 32 16]
    set full_filename [file join $::GIDDEFAULT resources images scalable gid.svg]
    foreach size $lst_sizes {
        lappend logos [gid_themes::ImageCreatePhoto $full_filename -svg_width $size]
    }
    wm iconphoto .gid -default {*}$logos
}

gid_themes::SetFontsDefault 1024

#set ::tk_strictMotif 0
# precision in floating point to string conversion
#set tcl_precision 17 ;#left the default value=0 from Tcl 8.5 with a special meaning

package require BWidget
#Widget::theme 1
package require dialogwin
package require tdom
package require gidscale

proc bindThemeChanged { } {
    #global GidCache
    set gid_ttk_theme [gid_themes::GetTtkTheme]
    #if { [info exists ::GidCache(ttk_theme)] && $::GidCache(ttk_theme) == $gid_ttk_theme } {
    #    return 0
    #}
    #set ::GidCache(ttk_theme) $gid_ttk_theme
    if { [ttk::style theme use] != $gid_ttk_theme } {
        # trick:
        # on Windows 8 and 10 (not in 7) when user log out and log in again a live GiD will be automatically changed
        # its "ttk::style theme use" to xpnative and then Tcl errors will be raised for example swapping pre/post
        # this proc is invoked by the virtual event <<ThemeChanged>> of . and we try to
        # automatically restore the previous GiD theme (e.g. clammod_light or clammod)
        if { [lsearch [ttk::style theme names] $gid_ttk_theme] != -1 } {
            ttk::style theme use $gid_ttk_theme
        } else {
            #the fist starting, without any .ini the theme 'clammod' seems that already doesn't exists
        }
    }
    #calculating apropiate color of style BottomFrame.TFrame
    set colordefaultframe [ttk::style lookup $ttk::currentTheme -background]
    set framebuttoncolor $colordefaultframe
    set valueerror ""
    if { [ info exists ::bindThemeChanged(inside)] && $::bindThemeChanged(inside) } {
        # WarnWinText "bindThemeChanged - already inside"
        return
    }
    set ::bindThemeChanged(inside) 1
    if [catch {
        set n [ scan $colordefaultframe #%2x%2x%2x r g b]
        if { $n != 3} {
            set rgb [ winfo rgb . $colordefaultframe]
            set r [ expr int( [ lindex $rgb 0]/256.0)]
            set g [ expr int( [ lindex $rgb 1]/256.0)]
            set b [ expr int( [ lindex $rgb 2]/256.0)]
        }
        set factor 17
        set r [ expr $r + $factor]
        if { $r > 255} { set r 255}
        set g [ expr $g + $factor]
        if { $g > 255} { set g 255}
        set b [ expr $b + $factor]
        if { $b > 255} { set b 255}
        set framebuttoncolor [ format #%02x%02x%02x $r $g $b]
    } valueerror ] {
        #if error
        set framebuttoncolor #fdfae9
    }
    #to let testcolors work only change color if background color is default
    if { [ttk::style configure BottomFrame.TFrame -background] == "#fdfae9" } {
        ttk::style configure "BottomFrame.TButton" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.TButton" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.TFrame" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.TFrame" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.TLabel" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.TLabel" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.TEntry" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.TEntry" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.TCheckbutton" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.TCheckbutton" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.TCombobox" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.TCombobox" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.TLabelframe" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.TLabelframe" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.TMenubutton" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.TMenubutton" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.TNotebook" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.TNotebook" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.TPanedwindow" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.TPanedwindow" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.Horizontal.TProgressbar" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.Horizontal.TProgressbar" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.Vertical.TProgressbar" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.Vertical.TProgressbar" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.TRadiobutton" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.TRadiobutton" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.Horizontal.TScrollbar" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.Horizontal.TScrollbar" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.Vertical.TScrollbar" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.Vertical.TScrollbar" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.TSeparator" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.TSeparator" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.TSizegrip" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.TSizegrip" -background [list disabled $framebuttoncolor active $framebuttoncolor]
        ttk::style configure "BottomFrame.Treeview" -background $framebuttoncolor -focuscolor $framebuttoncolor
        ttk::style map "BottomFrame.Treeview" -background [list disabled $framebuttoncolor active $framebuttoncolor]
    }

    # WarnWinText "[ ttk::style theme use] <----> [ gid_themes::GetCurrentTheme]"
    #CHANGE TK COLORS HERE

    set ::bindThemeChanged(inside) 0
    return 0
}

proc GiDSetBindingsTogl { w } {
    bind $w <Button-1> { Togl_Button-1 %W %x %y}
    bind $w <ButtonRelease-1> { Togl_ButtonRelease-1 %W %x %y}
    # %k = The keycode field from the event ( KeyPress & KeyRelease)
    # %N = The keysym corresponding to the event, substituted as a decimal number  ( KeyPress & KeyRelease)
    # %s = The state field from the event
    bind $w <KeyPress> { Togl_KeyPress %k %N %s}
    bind $w <$::gid_central_button> {%W button 2 %x %y}
    ##### to handle mouse wheel
    catch {
        bind $w <Button-4> {
            %W gidmakecurrent
            %W button 4 %x %y
        }
        bind $w <Button-5> {
            %W gidmakecurrent
            %W button 5 %x %y
        }
    }
    bind $w <MouseWheel> {MouseWheelEventToButtonEvent %W %D}
    bind $w <Motion> { Togl_Motion %W %x %y }
    bind $w <Leave> { %W leave %x %y }
    bind $w <Enter> { %W enter %x %y }
    bind $w <Button-$::gid_right_button> { Togl_Button-Right %W %X %Y}
    bind $w <Alt-Motion> { Togl_Alt-Motion %W %x %y }
    bind $w <Alt-1> { Togl_Alt-Button-1 %W %x %y}
    bind $w <Alt-ButtonRelease-1> {Togl_Alt-ButtonRelease-1 %W %x %y}
    bind $w <${::acceleratorKey}-ButtonPress-1> "ContextualEntity::Activate %W %x %y; break"
    bind $w <Configure> [list ::GiD_RaiseEvent GiD_Event_ResizeTogl %W %w %h %x %y]
    return 0
}

proc SetBindingsForMouse { w ibutton cmd x y } {
    GiD_Process {*}$cmd
    $w button 1 $x $y
    set ::GidPriv(OldButtonRelease-$ibutton) [bind $w <ButtonRelease-$ibutton>]
    bind $w <ButtonRelease-$ibutton> "%W button -r 1 %x %y ;\
        GiD_Process escape;\
        bind %W <ButtonRelease-$ibutton> \$::GidPriv(OldButtonRelease-$ibutton) ;\
        break"
}

proc SetButtonMeaning { w } {
    global GidPriv
    if { $GidPriv(ChangeButtonMeaning) == "0" } {
        foreach "mod cmd" [list Alt "'DynamicPan" Shift "'Zoom Dynamic" Alt-Shift "'Rotate Trackball"] {
            bind $w <$mod-ButtonPress-1> "[list SetBindingsForMouse $w 1 $cmd %x %y] ; break"
        }
    } else {

        set ::gid_central_button 2
        set ::gid_right_button 3

        if { $::tcl_platform(os) == "Darwin" } {
            # http://wiki.tcl.tk/12987 FAQ 7.7:
            # For historical reasons, MacMice buttons 2 and 3 refer to the right and middle buttons
            # respectively, which is indeed the opposite way round from Windows and *nix systems.
            set ::gid_central_button 3
            set ::gid_right_button 2
        }

        #new bindings
        bind $w <Shift-1> "[list SetBindingsForMouse $w 1 {'Rotate Trackball} %x %y] ; break"
        bind $w <Shift-$::gid_central_button> "[list SetBindingsForMouse $w 2 {'Zoom Dynamic} %x %y] ; break"
        bind $w <Shift-$::gid_right_button> "[list SetBindingsForMouse $w 3 'DynamicPan %x %y] ; break"
    }
}


# now only read one of both: .ini or default best file in \scripts
# and in this second case read also the 'kernel' C++ interpreted part, calling readdefaults
set gid_defaults [GiD_GetUserSettingsFilename];#gid.ini
if { ![file exists $gid_defaults] } {
    #do not use gid.ini, instead use [file tail $gid_defaults], other problemtypes could use its alternative name
    set similar_gid_defaults [GetSimilarDefaultsFile [file tail $gid_defaults]]
    if { [file exists $similar_gid_defaults] } {
        set gid_defaults $similar_gid_defaults
    }
}
if { ![file exists $gid_defaults] } {
    set gid_defaults [file join $::GIDDEFAULT scripts gidDefaults]
}
set cloud 0
ReadDefaultValues $gid_defaults $cloud;#tcl values

if { [info exists ::ForceOpenSelectOpenGLWindow] } {
    #if exists 1 force open, 0 force not to open
    #do not exists for default behavior: open only first start
    if { $::ForceOpenSelectOpenGLWindow  } {
        #starting with -openglconfig flag
        GiD_Set OpenGL(Configured) 0 ;#replace the possible value read from gid_defaults
    } else {
        #starting with -??? flag
        GiD_Set OpenGL(Configured) 1 ;#replace the possible value read from gid_defaults
        GiD_Set Theme_configured 1 ;#replace the possible value read from gid_defaults
    }
}

set initial_configuration_used 0
set user_selected_menu_type [GiD_Set Theme(MenuType)]
set ogl_configured [GiD_Set OpenGL(Configured)]
if { ![string is boolean -strict $ogl_configured] } {
    #?? try to continue without tcl error...
    set ogl_configured 1
}
set theme_configured [GiD_Set Theme_configured]
if { ![string is boolean -strict $theme_configured] } {
    #?? try to continue without tcl error...
    set theme_configured 1
}

set save_gid_defaults [GiD_Set SaveGidDefaults]
if { (!$ogl_configured || !$theme_configured) && $save_gid_defaults } {
    #if SaveGidDefaults==0 (flag -c) then won't save defaults, and don't show windowd asking for opengl or theme
    InitialConfiguration 1 [expr !$ogl_configured] [expr !$theme_configured]
    #careful theme background and color entities aplied after readdefaults
    if { $::tcl_platform(os) == "Darwin"} {
        set initial_configuration_used 1
        set user_selected_menu_type [GiD_Set Theme(MenuType)]
    }
} else {
    if { [GiD_Set SplashWindow] } {
        if { ![info exists ::GID_DISABLE_SPLASH_WINDOW] || !$::GID_DISABLE_SPLASH_WINDOW } {
            GiDSplash
            GiDSplashMessage "Initializing $::GidPriv(ProgName,Default) ..."
        }        
    }
}

#apply apropiated colors to window
set theme [gid_themes::GetCurrentTheme]
# gid_themes::SetCurrentTheme $theme 0
OnSetCurrentTheme $theme
gid_themes::SetOptions

# Respect user selected menutype. SetOptions modifies it.
if { $initial_configuration_used && ( $::tcl_platform(os) == "Darwin" ) } {
    GiD_Set Theme(MenuType) $user_selected_menu_type
}

# readdefaults $gid_defaults $cloud;#kernel values.
#WARNING, its too early to read the kernel values, gid_togle is not initialized, call 'readdefaults' later

if { ![ info exists ::MainWindowGeom]} {
    # something bad happened with the user's defaults
    ReadDefaultValues [file join $::GIDDEFAULT scripts gidDefaults] $cloud;#tcl values
    # then complement with user's defaults
}

proc GiDCreateMainWindow { } {
    set w .gid
    toplevel $w -class GiD

    # this is to avoid send commands
    tk appname gid
    #catch { rename send "" }

    wm protocol $w WM_DELETE_WINDOW QuitFullScreen
    ChangeWindowTitle UNNAMED UNKNOWN
    wm minsize $w 1 1

    # on Mac OS X listen to user set preference
    if { [GiD_Set Theme(MenuType)] == "native" &&  ($::tcl_platform(platform) != "windows" && $::tcl_platform(os) != "Darwin") } {
        #native menus not available for this platform
        GiD_Set Theme(MenuType) "generic"
    }

    #set menus after read defaults, to set the appropiated language
    SetAllMenusVariables

    menu $w.display3dbutton -tearoff no
    Create3rdButtonMenu $w.display3dbutton

    menu $w.display3dbutton_graph -tearoff no
    Create3rdButtonMenuGraph $w.display3dbutton_graph

    # update idletasks is needed so that gid's main window can be positioned in the second monitor.
    wm withdraw $w
    update idletasks
    WmGidGeom $w $::MainWindowGeom
    if { [GiD_Set Theme(MenuType)] == "generic" } {
        ttk::frame $w.topm
        set ::GiDTBLayout($w.topm) [list]
    }

    foreach _f_ {top left right rightb bottom bottomc} {
        set ::GiDTBLayout([ttk::frame $w.$_f_]) [list]
    }

    ttk::frame $w.bottominfo -style ForcedFrame
    ttk::separator $w.bottominfo.separator -orient horizontal -style ForcedSeparator
    grid $w.bottominfo.separator -columnspan 100 -sticky "news" -pady {2 0}
    grid columnconfigure $w.bottominfo 0 -weight 1

    if { $::tcl_platform(platform) == "windows" } {
        ttk::frame $w.central -style sunken.TFrame
    } else {
        ttk::frame $w.central
    }

    if { [ info exists ::env(GLX_ERROR)]} {
        WarnWinDirect [_ "ERROR: GLX configuration is not valid.\nCheck if the graphics driver installation is correct.\nUsing software rendering mode!"]
        GiD_Set SoftwareOpenGL 1
    }
}

GiDCreateMainWindow
CreateIcon
DWInitUserDataOptions

GidUtils::Disable windows
gid_themes::SetOptions ;#this call GUI_UpdateMenus
GidUtils::Enable windows

#name is the widget name, its parent must exists
#ident GID, sharelist must be 1 for more than one gid_togls, else axes are shown wrongly
proc CreateNewToglGID { name sharelist } {
    set ident GID
    if { [GiD_Set SoftwareOpenGL] } {
        set extra_flags [list -softrender true]
        # set alphaFlag false
        # enabling alpha also in software mode to take PNG snapshots with alpha channel.
        set alphaFlag true
    } else {
        set extra_flags [list ]
        set alphaFlag true
    }
    # in Mac using stereo = true causes trembling on some hardware, so disable it
    # in fact, disable it for everything, as 'nobody' seems to use it
    # lappend extra_flags -stereo true
    if { $sharelist } {
        lappend extra_flags -sharelist $ident
        # -sharecontext $ident
    }

    #-alpha true -> aunque sea lo correcto en windows con algunos drivers "no certificados" da problemas al hacer el c.fill
    set bpp [winfo screendepth .]
    if { ( $bpp == 24) ||  ( $bpp == 32)} {
        # windows WSL with Xming does not like alpha channel
        if { [ string first "WSL" $::tcl_platform(osVersion)] == -1 } {
            # std linux i guess
            set rgbFlags [list -redsize 8 -greensize 8 -bluesize 8 -alphasize 8 -alpha $alphaFlag]
        } else {
            # it's something like '5.4.72-microsoft-standard-WSL2'
            set rgbFlags [list -redsize 8 -greensize 8 -bluesize 8]
        }
    } elseif { $bpp == 16} {
        set rgbFlags [list -redsize 5 -greensize 6 -bluesize 5 -alphasize 8]
        # } elseif { $bpp == 15} {
        #     set rgbFlags [list -redsize 5 -greensize 5 -bluesize 5 -alphasize 8]
        # } elseif { $bpp == 8} {
        #     set rgbFlags [list -redsize 3 -greensize 3 -bluesize 2]
    } else {
        set rgbFlags ""
    }
    set depthBits [winfo depth .]
    lappend rgbFlags -depthsize $depthBits -stencil true -stencilsize 8
    # deprecated:  -overlay $overlay_flag
    #set rgbFlags [list -redsize 8 -greensize 8 -bluesize 8 -alphasize 8 -depthsize 24 -stencilsize 0]
    # W "Creating Togl with  -rgba true -double true -depth true -ident $ident -privatecmap false {*}$rgbFlags {*}$extra_flags"
    if { [catch { gid_togl $name -rgba true -double true -depth true -ident $ident -privatecmap false {*}$rgbFlags {*}$extra_flags} err_msg] } {
        if { ![GiD_Set SoftwareOpenGL] } {
            # Trying now with SoftwareOpenGL
            # WarnWin "Couldn't create a graphical window with '$rgbFlags' and '$extra_flags': $err_msg\n\nNow trying with SoftwareOpenGL"
            set text "Couldn't create a graphical window with '$rgbFlags' and '$extra_flags': $err_msg\n\nNow trying with SoftwareOpenGL. \nYou can change this setting in the Preferences window --> Graphical --> System --> OpenGL options"
            WarnWin $text
            # tk_messageBox in macOS is not fully modal...
            # set answer [ tk_messageBox -message  $text -icon warning -type yesno -detail "Select 'Yes' to try with SoftwareOpenGL."]
            # tkwait variable answer
            destroy $name
            GiD_Set SoftwareOpenGL 1
            lappend extra_flags -softrender true
            if { [catch { gid_togl $name -rgba true -double true -depth true -ident $ident -privatecmap false {*}$rgbFlags {*}$extra_flags } err_msg] } {
                WarnWinDirect "Couldn't create a graphical window with '$rgbFlags' and '$extra_flags': $err_msg\n\nCheck your graphical configuration."
                # leave it as it was
                GiD_Set SoftwareOpenGL 0
            }
        }
        #error "Couldn't create a graphical window with '$rgbFlags' and '$extra_flags': $err_msg\n\n$soft_msg"
    }
    return $name
}

CreateNewToglGID .gid.central.s 0

set prev_mesh_type [GiD_Set MeshType]
#ReadDefaultValues $gid_defaults $cloud;#tcl values
# read defaults from c kernel
readdefaults $gid_defaults $cloud;#kernel values

# No need anymore, as GiD_Classic already uses the clam theme !
if { $::tcl_platform(os) == "Linux" && [gid_themes::GetCurrentTheme] == "GiD_native" } {
    # GiD_classic too ugly
    ttk::style theme use clam
    ttk::style configure TopMenu.IconButton -font TkDefaultFont -padding {6 0}
    ttk::style configure TopMenu.Horizontal.IconButton -font TkDefaultFont -padding {6 0}
    ttk::style configure TopMenu.Vertical.IconButton -font TkDefaultFont -padding {0 6}
    ttk::style configure RightButtons.IconButton -anchor w -padding {4 2}
    ttk::style configure RightButtons.IconButton -anchor w -padding {4 2}
}

if { [GiD_Project set disable_windows] } {
    wm withdraw .gid
} else {
    wm deiconify .gid
}

if { $prev_mesh_type != [GiD_Set MeshType] } {
    SetAllMenusVariables
}

proc MouseWheelEventToButtonEvent { w increment } {
    if { [grab current $w] != "" } {
        #avoid this event when other window was grab, do nothing
        #maybe could ignore only for the known problem if the file dialog window [grab current $w] == gid.__tk_filedialog
    } else {
        $w gidmakecurrent
        if { $increment > 0 } {
            $w button 4 $increment 0
        } else {
            $w button 5 $increment 0
        }
    }
}

proc Togl_Button-1 { w x y } {
    set ::DynamicEightDirections($w,LastPostition,x) $x
    set ::DynamicEightDirections($w,LastPostition,y) $y
    $w gidmakecurrent
    focus -force $w
    set ::GidPriv(PressingButton1) 1
    if { [info exists ::zoom_box_state(in)] } {
        if { $::zoom_box_state(in) == 1 } {
            set ::zoom_box_state($w) [list $x $y]
            GiD_RaiseEvent GiD_Event_DynamicBox $w $x $y BEGIN
            set ::zoom_box_state(in) 2
        } elseif { $::zoom_box_state(in) == 2 } {
            GiD_RaiseEvent GiD_Event_DynamicBox $w $x $y END
            array unset ::zoom_box_state
        }
    }
    $w button 1 $x $y
    if { $::ActionsMenuFunction != " MoveLegends" } {
        #e.g. do not do this rotation when $::ActionsMenuFunction == " MoveLegends"
        set axis_id [GiD_Togl pick_one $w axis $x $y]
        if { $axis_id > 0 } {
            if { $axis_id == 1 } {
                # ID_SELEC_AXIS_X
                GidRotateView::PlaneYZ
            } elseif { $axis_id == 2 } {
                # ID_SELEC_AXIS_Y
                GidRotateView::PlaneXZ
            } elseif { $axis_id == 3 } {
                # ID_SELEC_AXIS_Z
                GidRotateView::PlaneXY
            } else {
                W "clicked unexpected axis $axis_id"
            }
        }
    }
    after 300 [list catch [list unset -nocomplain ::GidPriv(PressingButton1)] ]
}

proc Togl_ButtonRelease-1 { w x y } {
    if { [info exists ::zoom_box_state($w)] } {
        if { $::zoom_box_state($w) != [list $x $y] } {
            GiD_RaiseEvent GiD_Event_DynamicBox $w $x $y END
            array unset ::zoom_box_state
        }
    }
    if { [info exists ::GidPriv(PressingButton1)] } {
        $w button -r -f 1 $x $y
    } else {
        $w button -r 1 $x $y
    }
}


set ::active_id 0
set ::active_type ""

proc ShowActiveFigureUnderCursor { w x y } {
    set must_redraw 0
    set view_mode [GiD_Info Project ViewMode]
    if { $view_mode == "GEOMETRYUSE" } {
        set types {point line surface volume dimension}
        #lappend types legend  ;#test, but to work well the objects of legend (comments, axis or scale) must be Figure and must be a plex with them...
        #lappend types axis
        set command GiD_Geometry
    } elseif { $view_mode == "MESHUSE" } {
        set types {node element dimension}
        #lappend types legend
        #lappend types axis
        set command GiD_Mesh
    } else {
        set types {node element}
        set command ""
    }
    foreach type $types {
        set id [GiD_Togl pick_one $w $type $x $y]
        if { $::active_id != $id } {
            if { $::active_id } {
                set must_redraw 1
                if { $command != "" } {
                    if { $::active_type == "dimension" } {
                        GiD_Dimension edit $::active_id selected 0
                    } elseif { $::active_type == "axis" } {
                        #W "axis edit $::active_type $::active_id selected 0"
                    } elseif { $::active_type == "legend" } {
                        #W "legend edit $::active_type $::active_id selected 0"
                    } else {
                        $command edit $::active_type $::active_id selected 0
                    }
                } else {
                    #post do not exists selected
                }
            }
            set ::active_id $id
            set ::active_type $type
            if { $::active_id } {
                set must_redraw 1
                if { $command != "" } {
                    if { $::active_type == "dimension" } {
                        GiD_Dimension edit $::active_id selected 1
                    } elseif { $::active_type == "axis" } {
                        #W "axis edit $::active_type $::active_id selected 1"
                    } elseif { $::active_type == "legend" } {
                        #W "legend edit $::active_type $::active_id selected 1"
                    } else {
                        $command edit $::active_type $::active_id selected 1
                    }
                } else {
                    #post do not exists selected
                }
                break
            }
        }
    }
    if { $must_redraw } {
        GiD_Redraw
    }
}

proc ShowActiveAxisUnderCursor { w x y } {    
    set must_redraw 0
    set id [GiD_Togl pick_one $w axis $x $y]    
    if { $::active_id != $id } {
        if { $::active_id } {
            set must_redraw 1
            GiD_Axis edit $::active_id selected 0
        }
        set ::active_id $id
        if { $::active_id } {
            set must_redraw 1
            GiD_Axis edit $::active_id selected 1
        }
    }
    if { $must_redraw } {
        GiD_Redraw
    }
}

proc Togl_Motion { w x y } {
    if { [info exists ::zoom_box_state($w)] } {
        GiD_RaiseEvent GiD_Event_DynamicBox $w $x $y CONTINUE
    }
    $w motion $x $y
    #ShowActiveFigureUnderCursor $w $x $y
    ShowActiveAxisUnderCursor $w $x $y
}

proc Togl_Button-Right { w x y } {
    $w gidmakecurrent
    # set wtop [ winfo toplevel $w]
    # raise $wtop
    # focus $wtop
    # when changing this, remember that it is used in:
    #   GiDSetBindingsTogl    and    PostGraphs::PickOneGraph
    GiDShowContextualMenu $x $y
}

proc GetDynamicEightDirectionsPosition { w new_x new_y} {
    if { [ info exists ::DynamicEightDirections($w,LastPostition,x)] &&
        [ info exists ::DynamicEightDirections($w,LastPostition,y)]} {
        set old_x $::DynamicEightDirections($w,LastPostition,x)
        set old_y $::DynamicEightDirections($w,LastPostition,y)
        set dif_x [ expr abs( $new_x - $old_x)]
        set dif_y [ expr abs( $new_y - $old_y)]
        # four directions: axis aligned
        # if { $dif_x > $dif_y} {
        #     # make y axis aligned
        #     set new_y $old_y
        # } else {
        #     # make x axis aligned
        #     set new_x $old_x
        # }
        # eight directions
        if { $dif_x > $dif_y} {
            # make y axis aligned
            if { [catch { set div [expr double($dif_y)/double($dif_x)] } err_txt] } {
                #WarnWinText "GetDynamicEightDirectionsPosition: $err_txt\ndoing double( $dif_y) / double( $dif_x)"
            } else {
                if { $div < 0.4142135623731} {
                    set new_y $old_y
                } else {
                    # at 45 degrees
                    #scale bigger dif_x
                    set avg_dif [ expr 0.5 * ( $dif_x + $dif_y)]
                    set dif_y [ expr $new_y - $old_y]
                    if { $dif_y < 0} {
                        set dif_y [ expr -$avg_dif]
                    } else {
                        set dif_y $avg_dif
                    }
                    set new_y [ expr $old_y + $dif_y]
                    set dif_x [ expr $new_x - $old_x]
                    if { $dif_x < 0} {
                        set dif_x [ expr -$avg_dif]
                    } else {
                        set dif_x $avg_dif
                    }
                    set new_x [ expr $old_x + $dif_x]
                }
            }
        } else {
            # make x axis aligned
            if { [catch {set div [expr double($dif_x)/double($dif_y)]} err_txt] } {
                #WarnWinText "GetDynamicEightDirectionsPosition: $err_txt\ndoing double( $dif_x) / double( $dif_y)"
            } else {
                if { $div < 0.4142135623731} {
                    set new_x $old_x
                } else {
                    # at 45 degrees
                    set avg_dif [ expr 0.5 * ( $dif_x + $dif_y)]
                    set dif_x [ expr $new_x - $old_x]
                    if { $dif_x < 0} {
                        set dif_x [ expr -$avg_dif]
                    } else {
                        set dif_x $avg_dif
                    }
                    set new_x [ expr $old_x + $dif_x]
                    set dif_y [ expr $new_y - $old_y]
                    if { $dif_y < 0} {
                        set dif_y [ expr -$avg_dif]
                    } else {
                        set dif_y $avg_dif
                    }
                    set new_y [ expr $old_y + $dif_y]
                }
            }
        }
    }
    return [ list $new_x $new_y]
}

proc Togl_Alt-Motion { w x y } {
    lassign [GetDynamicEightDirectionsPosition $w $x $y] new_x new_y
    .gid.central.s motion $new_x $new_y
}

proc Togl_Alt-Button-1 { w x y } {
    # W "Togl_Alt-Button-1 $w $x $y"
    lassign [GetDynamicEightDirectionsPosition $w $x $y] new_x new_y
    Togl_Button-1 $w $new_x $new_y
}

proc Togl_Alt-ButtonRelease-1 { w x y } {
    # W "Togl_Alt-ButtonRelease-1 $w $x $y"
    lassign [GetDynamicEightDirectionsPosition $w $x $y] new_x new_y
    Togl_ButtonRelease-1 $w $new_x $new_y
}


# %k = The keycode field from the event ( KeyPress & KeyRelease)
# %N = The keysym corresponding to the event, substituted as a decimal number  ( KeyPress & KeyRelease)
# %s = The state field from the event
proc Togl_KeyPress { k N s} {
    #in theory by the Tk manual until 122 are lower letters
    #but in my machine lowercase and uppercase provide the same k !!
    #and then 90 is the last letter
    #(if set 122 letters like F5 ->k=116 and is raised twice the command associated!!
    # above is incorrect:
    # --> %k is keycode ( code of key taste in keyboard, nor the ascii letter !!!
    # 123 = F12
    # 145 = ScrollLock
    if { $N >=32 && $N <=122 && $s==0 } {
        #only do this if s is zero to avoid for example send the 'a' when pressing <Ctrl-a>
        if { [winfo exists .gid.comm.e] } {
            focus -force .gid.comm.e
            #event generate .gid.comm.e <KeyPress> -keysym $K -when tail
            event generate .gid.comm.e <KeyPress> -keycode $k -when tail
        }
    }
}

proc Togl_Escape {} {
    if { [info exists ::zoom_box_state(in)] } {
        foreach w [array names ::zoom_box_state] {
            if { $w != "in" && [winfo exists $w] } {
                GiD_RaiseEvent GiD_Event_DynamicBox $w {*}$::zoom_box_state($w) END
            }
        }
        array unset ::zoom_box_state
    }
    if { [winfo exists .gid.comm.e] } {
        .gid.comm.e delete 0 end
    }
    GiD_Process escape
}

proc GiDShowContextualMenu { x y } {
    if { [GiD_Info project ViewMode] == "GRAPHUSE" } {
        GiD_PopupMenu .gid.display3dbutton_graph $x $y
    } else {
        GiD_PopupMenu .gid.display3dbutton $x $y
    }
}

proc OnDestroyWindowGiD { W w } {
    if { $W != $w } return
    #reenter multiple times, one by toplevel child, only interest w
    # ensure lognoter, ramdebugger interpreters are closed also...
    GiDCloseLognoter
    if { [interp exists ramdebugger] } {
        EndLoadRamDebugger
    }
}

proc handle_open_multiple_files { files} {
    set do_multiple_post 0
    if { [ llength $files] > 1} {
        set num_post_files 0
        foreach filename $files {
            if { [ IsPostFilename $filename]} {
                incr num_post_files
            }
        }
        if { $num_post_files > 1} {
            set do_multiple_post 1
        }
    }
    if { $do_multiple_post} {
        ExecuteFileCommand ReadMultiple POST "" $files
    }
    return $do_multiple_post
}

#drag and drop files
proc register_dnd_event { win_dnd } {
    # OLE commands in windows
    if { [info commands olednd] != "" } {
        #olednd is a deprecated internal GiD command, removed from 7.5.10b version
        olednd bindtarget $win_dnd FileName <Drop> "LoadFileInGid"
    } else {
        #if {$::tcl_platform(platform) eq "windows"} {
        catch { package require tkdnd 2.6 } ;# only if it is compiled
        if { [info commands dnd] != "" } {
            proc DropMultipleFiles { files } {
                set processed_as_multiple_files [ handle_open_multiple_files $files]
                if { !$processed_as_multiple_files } {
                    foreach i $files {
                        LoadFileInGid $i
                    }
                }
            }
            ##dnd bindtarget .gid FileName <Drop> "LoadFileInGid %D"
            #dnd bindtarget .gid text/uri-list <Drop> [ list after idle "DropMultipleFiles %D"]
            #updated syntax tkdnd 2.6 (but deprecated syntax also works, at least on Windows)
            tkdnd::drop_target register $win_dnd DND_Files
            bind $win_dnd <<Drop:DND_Files>> [list after idle "DropMultipleFiles %D"]
        }
        #}
    }
}



proc GiDSetBindings { } {
    #.gid is a toplevel, toplevel bindings are raised not only once, but for all children %W, be careful!!!
    bind .gid <Unmap> {UnMapAllWindows %W}
    bind .gid <Map> {MapAllWindows %W}
    bind .gid <${::acceleratorKey}-a> SelectJoinOrNoJoin
    bind .gid <${::acceleratorKey}-A> SelectJoinOrNoJoin
    #bind .gid <${::acceleratorKey}-h> SelectionOnlyVisibleOrAll
    # if view accelerator changes please update in GrDataMenus.tcl
    # MenuAcceler(1,1)  for the menu bar entries
    # 3dButtonAcceler(0,3)   for the right button menu entries
    bind .gid <${::acceleratorKey}-Key-1> {GidRotateView::PlaneXY}
    bind .gid <${::acceleratorKey}-Key-2> {GidRotateView::PlaneXZ}
    bind .gid <${::acceleratorKey}-Key-3> {GidRotateView::PlaneYZ}
    bind .gid <${::acceleratorKey}-Key-4> {GidRotateView::IsometricZ}
    
    bind .gid <${::acceleratorKey}-t> {focus .gid.comm.e}
    bind .gid <Escape> Togl_Escape
    if { [GiD_Info Project Debug] == "debug" } {
        bind .gid <${::acceleratorKey}-x><${::acceleratorKey}-d> DebugMesherPost
        bind .gid <${::acceleratorKey}-x><${::acceleratorKey}-l> { auto_reset }
    }
    #bind all <Triple-Button-1> {grab release [grab current .gid] }
    if { $::tcl_platform(os) == "Darwin"} {
        # in Darwin ( Mac OS X) Drag and Drop events are not inherited and are assigned to final widget's
        register_dnd_event .gid.central.s
    } elseif { $::tcl_platform(os) == "Linux"} {
        register_dnd_event .gid.central
    } else {
        register_dnd_event .gid
    }

    bind . <Destroy> [list +OnDestroyWindowGiD %W .] ;# + to add to previous script
    bind . <<ThemeChanged>> bindThemeChanged

    if {[tk windowingsystem] eq "aqua"} {
        #left default
    } else {
        #change 4 to 1
        bind Listbox <MouseWheel> {
            %W yview scroll [expr {- (%D / 120) * 1}] units
        }
        bind Listbox <Shift-MouseWheel> {
            %W xview scroll [expr {- (%D / 120) * 1}] units
        }
    }

    if {"x11" eq [tk windowingsystem]} {
        bind Listbox <4> {
            %W yview scroll -3 units
            break
        }

        bind Listbox <5> {
            %W yview scroll 3 units
            break
        }
    }

    bind Text <4> {
        %W yview scroll -3 units
        break
    }

    bind Text <5> {
        %W yview scroll 3 units
        break
    }
    return 0
}

proc GiDGridMainWindow { } {
    # top
    if { [GiD_Set Theme(MenuType)] == "native" } {
        #reserve the row 0 to handle swapping at runtime between native and generic like menus
        set row 1
    } else {
        grid .gid.topm -row 0 -column 0 -columnspan 4 -sticky "snew"
        grid columnconfigure .gid.topm 0 -weight 1
        set row 1
    }
    grid .gid.top -row $row -column 0 -columnspan 4 -sticky "snew"
    grid columnconfigure .gid.top 0 -weight 1

    incr row

    set crow $row
    # left
    grid .gid.left -row $row -column 0 -sticky "ns"
    grid rowconfigure .gid.left 0 -weight 1
    # central
    grid .gid.central -row $row -column 1 -sticky "snew"
    if { [winfo exists .gid.central.wins.graphwin1] } {
        #was called before PrepareToglForSeveralWindows
        grid .gid.central.s -in .gid.central.wins.graphwin1 -sticky nsew
    } else {
        grid .gid.central.s -sticky nsew
    }
    
    grid rowconfigure .gid.central 0 -weight 1
    grid columnconfigure .gid.central 0 -weight 1
    # right toolbars
    grid .gid.right -row $row -column 2 -sticky "ns"
    grid rowconfigure .gid.right 0 -weight 1
    # right buttons
    grid .gid.rightb -row $row -column 3 -sticky "ns"
    grid rowconfigure .gid.rightb 0 -weight 1
    incr row
    # bottom
    grid .gid.bottom -row $row -column 0 -columnspan 4 -sticky "snew"
    grid columnconfigure .gid.bottom 0 -weight 1
    incr row
    grid .gid.bottomc -row $row -column 0 -columnspan 4 -sticky "snew"
    grid columnconfigure .gid.bottomc 0 -weight 1
    #lowerbottom
    incr row
    grid .gid.bottominfo -row $row -column 0 -columnspan 4 -sticky "snew"


    grid rowconfigure .gid $crow -weight 1
    grid columnconfigure .gid 1 -weight 1

    grid remove .gid.bottominfo
    #configuration is stored, real grid on proc BottomStatusFrame
}

GiDSetBindingsTogl .gid.central.s
GiDSetBindings
GiDGridMainWindow


set ::WarnLineVar ""
set ::ActionsMenuVar ""
set ::ActionsMenuVarType INTERP
set ::EntryVar ""
set ::EntryVarHilit ""
set ::ActionsMenuLabels ""

proc ChangeWarnLineVar { val } {
    update idletasks
    if { ![winfo exists .gid.comm.fwarn.list] } {
        return
    }
    global WarnLineVar uparrow vecToUseInVectors

    if { [string index $val 0] == "\r" } {
        set ::WarnLineVar "[.gid.comm.fwarn.list get end] [string range $val 1 end]"
        .gid.comm.fwarn.list delete end
    } else {
        set ::WarnLineVar $val
    }
    .gid.comm.fwarn.list insert end $::WarnLineVar
    if { $::GiD_Voice != "" } {
        $::GiD_Voice Speak $val 1
    }
    set uparrow [.gid.comm.fwarn.list size]
    .gid.comm.fwarn.list yview [expr $uparrow-2]
    if { [info exists vecToUseInVectors] } {
        GetVarVecs $val
    }
}

proc procTraceVar { name elm op } {
    global uparrow vecToUseInVectors
    upvar $name x

    if { $name ==  "::ActionsMenuVar" } {
        SetButtons
    }
    if { $name ==  "::WarnLineVar" } {
        .gid.comm.fwarn.list insert end $x
        set uparrow [.gid.comm.fwarn.list size]
        .gid.comm.fwarn.list yview [expr $uparrow-2]
        if { [info exists vecToUseInVectors] } {
            GetVarVecs $x
        }
    }
    if { $name ==  "::EntryVar"} { .gid.comm.e delete 0 end ; .gid.comm.e insert 0 $x }
    if { $name ==  "::EntryVarHilit" } { .gid.comm.e select range 0 end }
}
trace add variable ::ActionsMenuVar write procTraceVar
trace add variable ::EntryVar write procTraceVar
trace add variable ::EntryVarHilit write procTraceVar

SetButtonMeaning .gid.central.s

#LoadGidAddons must be called after load Tk, to modify some bindings, etc
LoadGidAddons
# LoadMenuVerifImages needs Tk and Img and, in Mac, also gid_themes
LoadMenuVerifImages

after idle ChangeTopMenusAndWindows

# to load problemtype icons ... needed in linux for the file browser
after idle [list FastCacheFromProblemtypeMenuDir [gid_filesystem::get_folder_standard problemtypes]]


if { $::GidPriv(ShowPackages) } {
    proc restoreOriginalPackage {} {
        # rename it back
        set ::GidPriv(ShowPackages) 0
        if { [ info proc MyPackage] == ""} {
            rename package MyPackage
            rename packageOriginal package
        }
    }
    # still problem with automatic loading of models
    after idle restoreOriginalPackage
}

# gid private and public events:
source [file join $::GIDDEFAULT scripts gid_implemented_events.tcl]

GiD_RaiseEvent GiD_Event_AfterReadPreferences $gid_defaults
# to avoid printing anything
return
