# parserdep.tcl --
#
# code to parse sequence associated to a dependency of question of material, general data, etc
#


namespace eval ParserDep {
    namespace export Init Start

    variable StrInput ""
    variable pchr 0
    variable lstr 0
    variable DepList {}
    variable ErrFound ""
    variable ActionName { "SET" "HIDE" "DISABLE" "RESTORE" "-" "CONDITION" "TITLESTATE" }
}

proc ParserDep::DumpInput { } {
    variable StrInput
    variable pchr

    puts [string range $StrInput $pchr end]
}

proc ParserDep::BackChar { } {
    variable pchr

    incr pchr -1
}

proc ParserDep::SkipSpaces { } {
    while { [InputAvailable] } {
        set chr [GetChar]
        if { ![string is space $chr] } {
            BackChar
            break
        }
    }
}

proc ParserDep::GetChar { } {
    variable StrInput
    variable pchr

    set chr [string index $StrInput $pchr]
    incr pchr
    return $chr
}

proc ParserDep::SeeChar { } {
    variable StrInput
    variable pchr

    return [string index $StrInput $pchr]
}

proc ParserDep::InputAvailable { } {
    variable pchr
    variable lstr

    return [expr $pchr < $lstr]
}

proc ParserDep::Error { {Msg ""} } {
    variable StrInput
    variable pchr
    variable ErrFound

    if { $Msg != "" } {
        set MsgErr "Expected $Msg"
    } else {
        set MsgErr "Unexpected error"
    }

    set ErrFound "$MsgErr near position $pchr of $StrInput"
    return 0
}

proc ParserDep::Init { Input } {
    variable StrInput $Input
    variable pchr 0
    variable lstr [string length $Input]
    variable DepList {}
    variable ErrFound ""
}

proc DumpState { name } {
  WarnWinText "level [info level] $name "  
}


#################

proc ParserDep::Start { } {    
    set fail [ParseAll]
    if { $fail } { 
        return 0
    } else {
        return 1
    }
}

proc ParserDep::FixValue { Value } {
    set Value [string trim $Value]
    if { ![string compare [string toupper $Value] "\#DEFAULT\#"] } {
        set Value "default"
    }    
    return $Value
}

proc ParserDep::FixAction { Action } {
    variable ActionName
    set Action [string trim $Action]
    set Action [string toupper $Action]
    if { [lsearch -exact $ActionName $Action] == -1 } {
        Error "$Action must be some of $ActionName"
        set Action ""
    }
    return $Action
}

proc ParserDep::FixLabel { Label } {
    set Label [string trim $Label]
    return $Label
}

proc ParserDep::FixArgument { Action Argument } {
    set Argument [string trim $Argument]
    if [string equal $Action "TITLESTATE"] {
        set Argument [string tolower $Argument]
    }
    return $Argument
}

proc ParserDep::ParseAll { } {
    variable CurrentMode   
    variable DepList
    set DepList ""
    set CurrentMode "START"
    set res 0    
    while { [InputAvailable] } {
        if { $CurrentMode == "START" } {
            ParseStart            
            set ActionList ""
        } elseif { $CurrentMode == "VALUE" } {
            set Value [ParseValue]
            set Value [FixValue $Value]
            if { $Value == "" } {
                set CurrentMode "ERROR"
            }
        } elseif { $CurrentMode == "ACTION" } {            
            set Action [ParseAction]
            if { $Action == "-" } {
                lappend DepList [list $Value "-"]
                set CurrentMode "START"                
            } else {
                set Action [FixAction $Action]
                if { $Action == "" } {
                    set CurrentMode "ERROR"
                }
            }
        } elseif { $CurrentMode == "LABEL" } {
            set Label [ParseLabel]
            set Label [FixLabel $Label]
            if { $Label == "" } {
                set CurrentMode "ERROR"
            }
        } elseif { $CurrentMode == "ARGUMENT" } {
            set Argument [ParseArgument]
            set Argument [FixArgument $Action $Argument]
            if { $CurrentMode == "APPEND_ACTION" } {
                lappend ActionList [list $Action $Label $Argument]
                set CurrentMode "ACTION"
            } elseif { $CurrentMode == "APPEND_DEP" } {
                lappend ActionList [list $Action $Label $Argument]
                lappend DepList [list $Value $ActionList]
                set CurrentMode "START"          
            } else {
                set CurrentMode "ERROR"
            }
        } elseif { $CurrentMode == "END" } {
            set res 0
            break
        } elseif { $CurrentMode == "ERROR" } {
            set res 1
            break
        } else {
            set res 1
            break
        }
    }
    return $res
}

proc ParserDep::ParseStart { } {
    variable CurrentMode
    
    SkipSpaces
    if { [InputAvailable] } {
        set chr [GetChar]
        if { $chr == "(" } {
            set CurrentMode "VALUE"
            return 0
        } else {
            set CurrentMode "ERROR"
            Error "("
            return 1
        }
    } else {
        set CurrentMode "END"
        return 0
    }    
}

proc ParserDep::ParseValue { } {
    variable CurrentMode
  
    SkipSpaces
    set Value ""
    while { [InputAvailable] } {
        set chr [GetChar]
        switch -- $chr {
            "\"" {
                while [InputAvailable] {
                    set chr [GetChar]
                    if { $chr == "\"" } {
                        break
                    }
                    append Value $chr
                }
                if { $chr != "\"" } {
                    set CurrentMode "ERROR"
                    Error "\""
                    return 1
                }
            }
            "," {
                set CurrentMode "ACTION"
                return $Value
            }
            ")" {
                set CurrentMode "ERROR"
                Error ","
                return 1
            }
            default {
                append Value $chr
            }
        }
    }
    CurrentMode= "ERROR"
    Error ","
    return  1
}

proc ParserDep::ParseAction { } {
    variable CurrentMode
    
    SkipSpaces
    set Action ""
    while { [InputAvailable] } {
        set chr [GetChar]
        switch -- $chr {
            "," {
                set CurrentMode "LABEL"
                return $Action
            }
            ")" {
                set Action [string trim $Action]
                if { $Action == "-" } {
                    set CurrentMode "START"
                    return $Action
                } else {
                    set CurrentMode "ERROR"
                    Error ","
                    return 1
                }
            }           
            default {
                append Action $chr
            }
        }
    }
    set CurrentMode "ERROR"
    Error ","
    return 1
}

proc ParserDep::ParseLabel { } {
    variable CurrentMode   
    
    SkipSpaces
    set Label ""
    while { [InputAvailable] } {
        set chr [GetChar]
        switch -- $chr {
            "," {
                set CurrentMode "ARGUMENT"
                return $Label
            }
            ")" {
                set CurrentMode "ERROR"
                Error ","
                return 1
            }            
            default {
                append Label $chr
            }
        }
    }
    set CurrentMode "ERROR"
    Error ","
    return 1
}

proc ParserDep::ParseArgument { } {
    variable CurrentMode       
    SkipSpaces
    set Argument ""
    while { [InputAvailable] } {
        set chr [GetChar]
        switch -- $chr {
            ")" {
                set CurrentMode "APPEND_DEP"
                return $Argument                
            }
            "," {
                set CurrentMode "APPEND_ACTION"
                return $Argument
            }
            "\"" {
                while [InputAvailable] {
                    set chr [GetChar]
                    if { $chr == "\"" } {
                        break
                    }
                    append Argument $chr
                }
                if { $chr != "\"" } {
                    set CurrentMode "ERROR"
                    Error "\""
                    return 1
                }
            }
            default {
                append Argument $chr
            }
        }
    }
    set CurrentMode "ERROR"
    Error ")"
    return 1
}
