package require gid_quaternion

namespace eval GidRotateView {
    variable m_xy {1 0 0 0 1 0 0 0 1}
    variable m_xz {1 0 0 0 0 -1 0 1 0}
    variable m_yz {0 0 1 1 0 0 0 1 0 }
    variable m_minus_xy {-1 0 0 0 1 0 0 0 -1}
    variable m_minus_xz {-1 0 0 0 0 1 0 1 0}
    variable m_minus_yz {0 0 -1 -1 0 0 0 1 0}
    variable m_isometric_x {0 0.7071067690849304 0.7071067690849304 -0.7071067690849304 -0.5 0.5 0.7071067690849304 -0.5 0.5}
    variable m_isometric_y {0.7071067690849304 -0.5000001192092896 0.5 0 0.7071067094802856 0.70710688829422 -0.7071067690849304 -0.5 0.5}
    variable m_isometric_z {-0.7071068286895752 -0.5 0.5 0.7071067690849304 -0.5 0.5 0 0.7071068286895752 0.7071067690849304}

    variable _dt_last_redraw 0
    variable _redraw_time_treshold 250 ; # ms (0.25 sec, 4 fps)
}


#extract the 3x3 rotation submatrix
proc GidRotateView::GetRotationMatrix { m4 } {
    lassign $m4 m11 m12 m13 m14 m21 m22 m23 m24 m31 m32 m33 m34 m41 m42 m43 m44
    return [list $m11 $m12 $m13 $m21 $m22 $m23 $m31 $m32 $m33]
}

#verify that is a valid axis system, vectors are orthonormals and preserve orientation
proc GidRotateView::CheckRotationMatrix { r3 } {
    lassign $r3 r11 r12 r13 r21 r22 r23 r31 r32 r33
    set col1 [list $r11 $r21 $r31]
    set col2 [list $r12 $r22 $r32]
    set col3 [list $r13 $r23 $r33]
    set col1_modulus [MathUtils::VectorModulus $col1]
    set col2_modulus [MathUtils::VectorModulus $col2]
    set col3_modulus [MathUtils::VectorModulus $col3]
    if { [expr abs($col1_modulus-1.0)] > 1e-7 } {
        W "not unitary col1 $col1, mod $col1_modulus"
    }
    if { [expr abs($col2_modulus-1.0)] > 1e-7 } {
        W "not unitary col2 $col2 , mod $col2_modulus"
    }
    if { [expr abs($col3_modulus-1.0)] > 1e-7 } {
        W "not unitary col3 $col3 , mod $col3_modulus"
    }
    set vector_prod [MathUtils::VectorVectorialProd $col1 $col2]
    if { [expr abs([MathUtils::VectorModulus $vector_prod]-1.0)] > 1e-7 } {
        W "not unitary vector prod $vector_prod, area [MathUtils::VectorModulus $vector_prod]"
    }
    set scalar_prod [MathUtils::VectorDotProd $vector_prod $col3]
    if { [expr abs($scalar_prod-1.0)] > 1e-7 } {
        W "not unitary scalar prod $scalar_prod"
    }
}

proc GidRotateView::AnimateRotationView { m_end_3 {num_steps 15 }} {
    variable _dt_last_redraw
    variable _redraw_time_treshold
    if { $_dt_last_redraw>$_redraw_time_treshold } {
        #if redraw is slow redraw only the final view
        set num_steps 1
    }
    set m_start_3 [GidRotateView::GetRotationMatrix [GiD_Project view rotation_matrix]]
    set quaternion_start [Quaternion::FromRotationMatrix3 $m_start_3]
    set quaternion_end [Quaternion::FromRotationMatrix3 $m_end_3]
    for {set i 1} {$i<=$num_steps} {incr i} {
        set t [expr double($i)/$num_steps]
        set quaternion_i [Quaternion::Slerp $quaternion_start $quaternion_end $t]
        set mi_4 [Quaternion::ToRotationMatrix4 $quaternion_i]
        GiD_Project view rotation_matrix $mi_4
        set t0 [clock milliseconds]
        GiD_Redraw
        set _dt_last_redraw [expr {[clock milliseconds]-$t0}]
        if { $_dt_last_redraw>$_redraw_time_treshold && $i<$num_steps } {
            # too slow, do only the final view
            set i [expr $num_steps-1]
        }
    }
}

proc GidRotateView::PlaneXY { } {
    variable m_xy
    #GiD_Process 'Rotate Angle 270 90
    GidRotateView::AnimateRotationView $m_xy
}

proc GidRotateView::PlaneXZ { } {
    variable m_xz
    #GiD_Process 'Rotate Angle 270 0
    GidRotateView::AnimateRotationView $m_xz    
}

proc GidRotateView::PlaneYZ { } {
    variable m_yz
    #GiD_Process 'Rotate Angle 0 0
    GidRotateView::AnimateRotationView $m_yz
}

proc GidRotateView::PlaneMinusXY { } {
    variable m_minus_xy
    #GiD_Process 'Rotate Angle 90 -90
    GidRotateView::AnimateRotationView $m_minus_xy
}

proc GidRotateView::PlaneMinusYZ { } {
    variable m_minus_yz
    #GiD_Process 'Rotate Angle 180 0
    GidRotateView::AnimateRotationView $m_minus_yz
}

proc GidRotateView::PlaneMinusXZ {} {
    variable m_minus_xz
    #GiD_Process 'Rotate Angle 90 0
    GidRotateView::AnimateRotationView $m_minus_xz
}

proc GidRotateView::IsometricX {} {
    variable m_isometric_x
    #GiD_Process 'Rotate Angle 180 90 'Rotate ScrAxes y 45 'Rotate ScrAxes x 45 escape
    GidRotateView::AnimateRotationView $m_isometric_x
}

proc GidRotateView::IsometricY {} {
    variable m_isometric_y
    #GiD_Process 'Rotate Angle 270 90 'Rotate ScrAxes y -45 'Rotate ScrAxes x 45 escape
    GidRotateView::AnimateRotationView $m_isometric_y
}

proc GidRotateView::IsometricZ {} {
    variable m_isometric_z
    #GiD_Process 'Rotate Angle 0 0 'Rotate ScrAxes y -45 'Rotate ScrAxes x 45 escape
    GidRotateView::AnimateRotationView $m_isometric_z
}
