namespace eval Register {   
    variable initial_dir_search
}

proc Register::IpSplitItems { txt } {
    set items [list]
    set port ""
    if { [regexp {([0-9]+)[.]([0-9]+)[.]([0-9]+)[.]([0-9]+)(:[0-9]+)?} $txt dummy n1 n2 n3 n4 port] } {
        set items [list $n1 $n2 $n3 $n4 $port]
    }
    return $items
}

proc Register::IsIp { txt } {
    set is_ip 0
    if { [llength [Register::IpSplitItems $txt]] } {
        set is_ip 1
    }
    return $is_ip
}

proc Register::GetLocalMachineDevice { } {
    package require verifp
    set device [lindex [vp_getsysinfo] 0]
    return $device
}

proc Register::GetDevicesUsb { } {
    package require verifp
    set devices [lrange [vp_getsysinfo] 1 end] ;#first device is local machine    
    return $devices
}

#is_module 0 == gid, 1 == problemtype
proc Register::GiD_Local { is_module {default ""}} {    
    lassign [Register::GetLocalMachineDevice] id_machinename id_operating_system id_sysinfo
    set password [Register::OpenPasswordEntryWindow local $is_module $id_machinename $id_operating_system $id_sysinfo $default]
    if { $password != "" } {        
        set fail 0
        if { $is_module } {
            set module [lindex $password 1] ;#the problemtype
            set password [lindex $password 0]
        } else {
            set module gid            
        }        
        lassign [GiD_Password $module register $password $id_machinename] password_ok message
        if { !$password_ok } {
            WarnWin [_ "The local password is not valid: %s" $message].
            Register::GiD_Local $is_module
        }  
    }
}

proc Register::GiD_Usb { is_module {default ""} } {
    set devices [Register::GetDevicesUsb]
    set usb_selected ""
    if { [llength $devices] == 0 } {
        WarnWin [_ "Any valid usb memory stick is detected. Plug your memory stick"]
    } elseif { [llength $devices]>1 || [isMacOSVersionEqualOrGreater Catalina] } {
        # macOS Catalina or greater needs tk_chooseDirectory to get access to the USB device
        # i.e. needs the user to select the usb device, despite being the only one present
        set usb_selected [GiDSelectRowWindow::SelectDeviceWindow]
    } else {
        set usb_selected [lindex $devices 0]
    }    
    if { $usb_selected != "" } {
        lassign $usb_selected id_machinename id_description id_sysinfo id_unit
        #id_operating_system doesn't matter for usb
        set password [Register::OpenPasswordEntryWindow usb $is_module $id_machinename "" $id_sysinfo $default]
        if { $password != "" } {
            set fail 0
            if { $is_module } {
                set module [lindex $password 1] ;#the problemtype
                set password [lindex $password 0]                
            } else {
                set module gid
            }            
            lassign [GiD_Password $module register $password $id_machinename] password_ok message
            if { !$password_ok } {
                WarnWin [_ "The USB password is not valid: %s" $message].
                Register::GiD_Usb $is_module
            }
        }
    }    
}

proc Register::GiD_Floating { is_module {default ""} } {
    set server_ip [Register::OpenPasswordEntryWindow floating $is_module "" "" "" $default]
    if { $server_ip != "" } {
        set fail 0
        if { $is_module } {
            set module [lindex $server_ip 1] ;#the problemtype
            set server_ip [lindex $server_ip 0]            
        } else {
            set module gid
        }
        lassign [GiD_Password $module register $server_ip] password_ok message
        if { !$password_ok } {
            set message [_ "The server IP=%s is not valid. %s" $server_ip $message]\n
            append message [_ "Check if passerver is properly configured on the server"]
            WarnWin $message
            after idle Register::GiD_Floating $is_module
        }
    }
}

#used by command GiD_Password $program save
proc Register::SavePassword { filename machine password date module } {
    if { [llength [split $password .]] == 4 } {
        #for an IP save only a word: the IP, avoiding the machine
        set txt "$password # $date Password for module '$module'"
    } else {
        set txt "$machine $password # $date Password for module '$module'"
    }
    set found 0    
    set old_file_content [GidUtils::ReadFile $filename "" 0]        
    set txt_without_comment [string trim [lindex [split $txt #] 0]]
    foreach line [split $old_file_content \n] {            
        set line_without_comment [string trim [lindex [split $line #] 0]]
        if { $line_without_comment == $txt_without_comment } {
            set found 1
            break
        }
    }    
    if { !$found } {
        set fp ""
        #catch necessary to avoid raise possible Tcl error "couldn't open "C:/Program Files/GiD/GiD 13.1.10d/scripts/TemporalVariables": permission denied"
        catch {        set fp [open $filename a] } msg
        if { $fp != "" } {
            puts $fp $txt
            close $fp
        } else {
            package require gid_cross_platform            
            gid_cross_platform::run_as_administrator [GidUtils::GetFullPathGiDExe] -tclsh [file join $::GIDDEFAULT scripts write_text.tcl] $filename $txt
        }       
    }
}

proc Register::RegisterFromFile { filename } {
    # sample of file contents:
    # mediador 33137efc1jqvyvy0 # 2006 01 10 Password for Problem type 'C:/gid_project/problemtypes/tdyn4.10/tdyn3d.gid'
    # mediador a6b0c6561jpp4v70 # 2005 05 03 Password for module 'C:/gid_project/problemtypes/Tdyn5/tdyn2d.gid'
    # mediador bq9akuu1p851cub9vnmmv2kc1g # V8 Registration date: Mar 16 2006; Expiration date: Jan  1 2035.
    # usb_17 badakuu1p851cub9vnmmv2kc1g # V7 Registration date: Mar 16 2006; Expiration date: Jan  1 2035.
    # 142.183.133.108:1625
    set res ""
    set fp [open $filename r]
    if { $fp != "" } {
        while { [gets $fp line] >= 0} {
            set line [string trim $line]
            if { [string index $line 0] == "#" } continue
            #si el comentario contiene llaves abiertas o cosas asi no seria una lista, y llength daria error!!!
            if { [llength $line] == 1 } {
                set type GID_PASSERVER
                set server_ip [lindex $line 0]       
                set module gid
                lassign [GiD_Password $module register $server_ip] password_ok message
                if { $password_ok } {
                    set fail 0
                } else {
                    set fail 1
                }
            } elseif { [llength $line] > 1 } {
                set machine [lindex $line 0]
                set password [lindex $line 1]
                set comments [lrange $line 2 end]
                if { [regexp -nocase {(?:Password for module|Password for Problem type)\s+'(.*)'} $comments dummy pt] } {
                    set type MODULE
                    set ptsplit [file split $pt]
                    set pos [lsearch $ptsplit problemtypes]
                    if { $pos != -1 } {
                        set ptrelative [eval file join [lrange $ptsplit [expr $pos+1] end]]
                    } else {
                        set ptrelative $pt
                    }
                    set ptfull [file join [gid_filesystem::get_folder_standard problemtypes] $ptrelative]
                    set existspt 0
                    if { [file exists $ptfull] && [gid_filesystem::file isdirectory $ptfull] } {
                        set existspt 1
                    } elseif { [file exists $ptfull.lnk] && [gid_filesystem::is_link $ptfull.lnk] == 2 } {
                        set existspt 1
                        set ptfull [gid_filesystem::read_link $ptfull.lnk 1]
                    } elseif { [file exists [file dirname $ptfull].lnk] && \
                                   [gid_filesystem::is_link [file dirname $ptfull].lnk] == 2 } {
                        set existspt 1
                        set ptname [file tail $ptfull]
                        set ptfull [file join [gid_filesystem::read_link [file dirname $ptfull].lnk 1] $ptname]
                    }
                    if { $existspt } {
                        set module $ptrelative
                        lassign [GiD_Password $module register $password $machine] password_ok message
                        if { $password_ok } {
                            set fail 0
                        } else {
                            set fail 1
                        }
                    } else {
                        #molule not found
                        set fail 1
                    }
                } else {
                    #try for GiD
                    set type GID
                    set module gid
                    lassign [GiD_Password $module register $password $machine] password_ok message
                    if { $password_ok } {
                        set fail 0
                    } else {
                        set fail 1
                    }
                }
            } else {
                set type UNKNOWN
                set fail 1
            }
            
            if { $type == "GID" } {
                lappend res [list $fail $type $machine $password]
            } elseif { $type == "GID_PASSERVER" } {
                lappend res [list $fail $type $ip]
            } elseif { $type == "MODULE" } {
                lappend res [list $fail $type $machine $password $ptrelative]
            } else { ;#UNKNOWN
                lappend res [list $fail $type $line]
            }
            
        }
        close $fp
    }
    return $res
}

proc Register::RegisterFromFileWindow { } {
    global GidPriv
    if { ![info exists GidPriv(filename,RegisterFromFileWindow)] } {
        set GidPriv(filename,RegisterFromFileWindow) ""
    }
    set types [list [list [_ "Passwords file"] ".pwd"] [list [_ "All files"] ".*"]]
    set filename [MessageBoxGetFilename file read [_ "read passwords file"] $GidPriv(filename,RegisterFromFileWindow) $types .pwd 0]
    if { $filename != "" } {
        set GidPriv(filename,RegisterFromFileWindow) [file tail $filename]
        set msg [_ "Register status"]\n
        foreach item [Register::RegisterFromFile $filename] {
            if { [lindex $item 0] == 0 } {
                set res [_ "ok"]
            } elseif { [lindex $item 0] == 1 } {
                set res [_ "fail"]
            } elseif { [lindex $item 0] == -1 } {
                #unverified
                set res "??"
            } else {
                #unexpected
                set res "!!"
            }
            set type [lindex $item 1]
            if { $type == "GID" } {
                set msgi "$GidPriv(ProgName) for [lindex $item 2] ([lindex $item 3])"
            } elseif { $type == "GID_PASSERVER" } {
                set msgi "$GidPriv(ProgName) from server [lindex $item 2]"
            } elseif { $type == "MODULE" } {
                set msgi "[lindex $item 4] for [lindex $item 2] ([lindex $item 3])"
            } else { ;#UNKNOWN
                set msgi "unknown $line"
            }
            append msg "$res ----- $msgi\n"
        }
        WarnWinText $msg
    }
}

#maybe could avoid this proc and directly use Register::AskPasswordWindow
#is_module 1 for problemtypes 0 for GiD
proc Register::OpenPasswordEntryWindow { password_type is_module id_machinename id_operating_system id_sysinfo default } {
    global GidPriv
    set result ""
    if { $is_module } {
        #ask password for a problemtype
        set password [Register::AskPasswordWindow $password_type $is_module $id_machinename $id_operating_system $id_sysinfo $default]
        set GidPriv(pass_info) $password
        if { $password != ""} {
            if {$password != "--CANCEL--" && ( $GidPriv(RegisterModuletext) == "" || $GidPriv(RegisterModuletext) == "..." ) } {
                WarnWin [_ "Before entering the password, it is necessary to select one or more Problem types"]
            }
            if { $password == "--CANCEL--" } {
                set result ""
            } else {
                set result $password
            }
        }
    } else {
        #ask password for GiD
        set fin 0
        while { !$fin} {
            # GetLocalMachineDevice returns Windows, Linux or darwin
            # and the password page wants Win, Lin or Mac
            if { $id_operating_system == "darwin"} {
                set id_operating_system "macOS / Mac OS X (Darwin)"
            }
            set password [Register::AskPasswordWindow  $password_type $is_module $id_machinename $id_operating_system $id_sysinfo $default]
            if { $password == "--CANCEL--" || $password == "" } {
                set result ""
                set fin 1
            } else {
                # longitud password v7 = 24 chars, v6 y anteriores = 16
                set is_wrong 0
                if { $password_type == "floating" } {
                    if { ![Register::IsIp $password] } {
                        lassign [Register::GetLocalMachineDevice] id_machinename id_operating_system id_sysinfo
                        lassign [GiD_Password gid verify $password $id_machinename] is_ok message
                        if { $is_ok } {
                            GidUtils::SetWarnLine [_ "The password is not a server ip but a local password, use Register GiD->Local"]
                            after 1000 Register::GiD_Local $is_module $password
                            return ""
                        } else {
                            set is_wrong 1
                        }
                    }                                               
                } elseif { $password_type == "local" || $password_type == "usb"} {
                    if { [Register::IsIp $password] } {
                        GidUtils::SetWarnLine [_ "The password seems a server ip, use Register GiD->Floating"]
                        after 1000 Register::GiD_Floating $is_module $password
                        return ""
                    }
                    if { [string length $password] < 26 } {
                        set is_wrong 1
                    }
                } elseif { $password_type == "unknown" } {
                    if { [string length $password] <= 26 && ![Register::IsIp $password] } {
                        set is_wrong 1
                    }
                } else {
                   W "Unexpected password_type=$password_type"
                }                
                if { $is_wrong } {
                    set msg [_ "Expected password type %s for %s %s %s" $password_type $id_machinename $id_operating_system $id_sysinfo]\n
                    append msg [_ "This password is not valid for version %s" [lindex [split [GiD_Info GiDVersion] .] 0]]\n
                    append msg [_ "Passwords and licences from older version are incompatible with this version."]\n
                    append msg [_ "Get a temporal password at %s" $GidPriv(PassWeb)]\n
                    append msg [_ "Please, contact %s to get a new licence." $GidPriv(ContactMail)]
                    WarnWin $msg
                } else {
                    set result $password
                    set fin 1
                }
            }
        }
    }
    return $result
}


#return fail==1 if some password module fail, -1 if can't be verified or 0 else
proc Register::GiDValidateModulePassword {def_pass_file id_name usb_pass_file password modules disablewindows } {
    global GidPriv
    #package require tdom
    #set pass_info $GidPriv(pass_info)
    #if {[llength $pass_info] != 2} {
    #  return 0
    #}
    #set pass [string trim [lindex $pass_info 0]]
    #set pt_list [lindex $pass_info 1]
    set pass [string trim $password]
    if {$pass eq ""} {
        # empty password is invalid
        return 0
    }
    set pt_list $modules    
    set date [clock format [clock second] -format "%Y %m %d"]
    set invalid_list [list]
    set valid_list [list]
    set unverified_list [list]  
    proc trunc_fname {name} {
        set items [file split $name]
        if {[llength $items]>3} {
            return [eval file join ... [lrange $items end-2 end]]
        }
        return $name
    }
    foreach dir $pt_list {
        if { [file exists $dir.lnk] && [gid_filesystem::is_link $dir.lnk] == 2 } {
            set dir [gid_filesystem::read_link $dir.lnk 1]
        }
        set pass_file $def_pass_file
        # load validation routine
        set xmlfile [file join $dir [file root [file tail $dir]]].xml
        set possible_msg ""
        if {[file exists $xmlfile]} {
            set xml [tDOM::xmlReadFile $xmlfile]
            if {[info exists xmldoc]} {
                $xmldoc delete
            }
            set xmldoc [dom parse $xml]
            set root [$xmldoc documentElement]
            set vp_node [$root select Program/ValidatePassword]
            if {$vp_node ne ""} {
                set vp_script [$vp_node text]
                if {$vp_script eq ""} {
                    lappend unverified_list [list $dir [_ "Node ValidatePassword doesn't have script"]]
                    continue
                }
                # now try to validate the password
                namespace eval GiD_TmpNS set dir [list $dir]
                namespace eval GiD_TmpNS set key [list $pass]
                namespace eval GiD_TmpNS set computername $id_name
                set tcl_code [catch {
                    namespace eval GiD_TmpNS $vp_script
                } catch_res]
                namespace delete GiD_TmpNS
                if {$tcl_code == 1} {
                    lappend unverified_list \
                        [list $dir [_ "tcl error: %s" $catch_res]]
                    continue
                }
                if {[llength $catch_res] == 2} {
                    set it0 [lindex $catch_res 0]
                    if {[string is integer $it0]} {
                        if {!$it0} {
                            lappend invalid_list \
                                [list $dir [lindex $catch_res 1]]
                            continue
                        }
                        if {$it0 == 2 && [lindex $catch_res 1] ne ""} {
                            lappend valid_list \
                                [list $dir [lindex $catch_res 1]]
                            continue
                        }
                        set possible_msg [lindex $catch_res 1]
                    }
                } elseif {$catch_res eq "0"} {
                    lappend invalid_list [list $dir "password rejected"]
                } elseif {$catch_res eq "2"} {
                    # 2 means password has being written.
                    continue
                }
            } else {
                #lappend unverified_list [list $dir [_ "Node ValidatePassword does not exist"]]
            }
            set passdir_node [$root select Program/PasswordPath]
            if {$passdir_node ne ""} {
                set passdir_txt [$passdir_node text]
                if {$passdir_txt eq ""} {
                    lappend invalid_list \
                        [list $dir [_ "Node PasswordPath is empty"]]
                    continue
                }
                if {[file pathtype $passdir_txt] eq "absolute"} {
                    set dir $passdir_txt
                } else {
                    set dir [file join $dir $passdir_txt]
                }
            }
        } else {
            lappend unverified_list [list $dir [_ "The password has not been verified now. It will be verified when running the analysis code"]]
        }

        # write the password in the default way
        # now write the password info.
        set pass_file [file join $dir $pass_file]
        set exists_file [file exists $pass_file]
        if {!$exists_file} {
            if {![file writable [file dir $pass_file]]} {
                lappend invalid_list \
                    [list $dir \
                         [_ "couldn't write into %s" [trunc_fname $pass_file]]]
                continue
            }
        } elseif {![file writable $pass_file]} {
            lappend invalid_list \
                [list $dir \
                     [_ "file %s isn't writable" [trunc_fname $pass_file]]]
            continue
        }

        set dir [file normalize $dir]
        if { $usb_pass_file != {} } {
            set lst_mod [ file split $dir]
            set lst_gid [ file split $::GIDDEFAULT]
            lappend lst_gid problemtypes
            set idx 0
            set fin [ llength $lst_gid]
            set fin2 [ llength $lst_mod]
            if { $fin2 < $fin} { set fin $fin2}
            for { set idx 0 } { ( $idx < $fin) && \
                                    ( [ string tolower [ lindex $lst_mod $idx]] == [ string tolower [ lindex $lst_gid $idx]])} { incr idx} {
            }
            set module_name [ eval file join [ lrange $lst_mod $idx end]]
            regsub -all { } $module_name {_} tmp_mod
            regsub -all {\\} $tmp_mod {/} tmp_mod2
            set module [ string tolower $tmp_mod2]

            Register::SavePassword $usb_pass_file $id_name $pass $date $dir
            # y en el archivo normal tambien:
            Register::SavePassword $pass_file $id_name $pass $date $dir
        } else {
            Register::SavePassword $pass_file $id_name $pass $date $dir
        }
        if {$possible_msg ne ""} {
            lappend valid_list [list $dir $possible_msg]
        }
    }

    if { !$disablewindows } {
        set title [_ "Password Validation"]
        set rows [list]

        set GidPriv(RegisterModuletext) ""
        set GidPriv(RegisterModule) ""
        foreach i $invalid_list {
            set pt_dir [lindex $i 0]
            set pt_name [file root [file tail $pt_dir]]
            lappend GidPriv(RegisterModuletext) $pt_name
            lappend GidPriv(RegisterModule) $pt_dir
            lappend rows [list $pt_name [_ "fail"] [lindex $i 1]]
        }
        foreach i $unverified_list {
            set pt_dir [lindex $i 0]
            set pt_name [file root [file tail $pt_dir]]
            lappend rows [list $pt_name [_ "unverified"] [lindex $i 1]]
        }
        foreach i $valid_list {
            set pt_dir [lindex $i 0]
            set pt_name [file root [file tail $pt_dir]]
            lappend rows [list $pt_name [_ "ok"] [lindex $i 1]]
        }

        if {[llength $rows]} {
            if {[winfo exists .gid.tempwineps]} {
                set ivpw .gid.tempwineps.ivpw
            } else {
                set ivpw .gid.ivpw
            }
            if {[llength $rows]==1} {
                set i [lindex $rows 0]
                set pt_name [lindex $i 0]
                MessageDlg $ivpw -title $title -type ok \
                    -message [lindex $i 2]
            } else {

                package require tablelist_tile

                dialogwin_snit $ivpw -title $title -cancelname -
                set f [$ivpw giveframe]
                tablelist::tablelist $f.tb -width 60 \
                    -columns [list 10 [_ "Problem type"] 10 [_ "Status"] 40 [_ "Reason"]]
                grid $f.tb -sticky snew
                grid rowconfigure $f 0 -weight 1
                grid columnconfigure $f 0 -weight 1
                eval $f.tb insert end $rows
                $ivpw createwindow
                destroy $ivpw
            }
        }
    }

    set has_invalid [llength $invalid_list]
    set has_valid [llength $valid_list]
    set has_unverified [llength $unverified_list]
    if { $has_invalid > 0 } {
        set fail 1
    } else {
        if { $has_unverified > 0 } {
            set fail -1
        } else {
            set fail 0
        }
    }
    return $fail
}

#related procs of old file treeforpass.tcl moved here and added to Register namespace

proc Register::PressingButOne { tree what node } {
    global GidPriv

    if { $what == "Double-1" && [$tree nodes $node] != "" } {
        if { [$tree itemcget $node -open] == 0 } {
            $tree itemconfigure $node -open 1
            Register::TreeModifyDir 1 $tree $node
        } else {
            $tree itemconfigure $node -open 0
            Register::TreeModifyDir 0 $tree $node
        }
        return
    }
    if { [$tree nodes $node] != "" } { return }
    switch -exact $what {
        1 {
            $tree selection set $node
        }
        Control-1 {
            if { [lsearch -exact [$tree selection get] $node] != -1 } {
                $tree selection remove $node
            } else {
                $tree selection add $node
            }
        }
    }
}

#BWidget Tree automatically substitute special characters
proc Register::TreeNodeName { node } {
    return  [string map {& _ | _ ^ _ ! _} $node]
}

proc Register::RetFreeNode { tree node } {
    set node [Register::TreeNodeName $node]
    if { ![$tree exists $node] } { return $node }
    set inum 1
    while { [$tree exists $node$inum] } { incr inum }
    return $node$inum
}

proc Register::SetProblemTypeTreeForDir { dir tree treeparent } {
    global GidPriv
    global GiDVisitedDir ;#to avoid infinite loops with links
    if { [info exists GiDVisitedDir($dir)] } {
        return 0
    }
    set GiDVisitedDir($dir) 1
    set inum 0
    set dirs [glob -nocomplain -tails -directory $dir -types d *]
    foreach link [glob -nocomplain -tails -directory $dir -types f *.lnk] {
        if { [gid_filesystem::is_link [file join $dir $link]] == 2 } {
            lappend dirs $link
        }
    }
    foreach i [lsort -dictionary $dirs] {
        set ifull [file join $dir $i]
        if { [file extension $i] == ".lnk" && [gid_filesystem::is_link $ifull] == 2 } {
            set i [file root $i] ;#to remove .lnk
            set ifull [gid_filesystem::read_link $ifull 1]
        }
        if { [file extension $i] == ".gid" } {
            set ishort [file root $i]
            set node [Register::RetFreeNode $tree $ishort]
            $tree insert end $treeparent $node -text $ishort -image [gid_themes::GetImage $::GidPriv(FolderGiDBitmapName) small_icons]           
            incr inum
        } elseif { [file isdirectory $ifull] } {
            #set ishort [file root $i]
            set ishort $i
            set node [Register::RetFreeNode $tree $ishort]
            $tree insert end $treeparent $node -text $ishort -image [gid_themes::GetImage folder.png small_icons]
            set inumin [Register::SetProblemTypeTreeForDir $ifull $tree [Register::TreeNodeName $ishort]]
            if { $inumin == 0 } {
                $tree delete $node
            } else { incr inum $inumin }
        }
    }
    return $inum
}

proc Register::TreeModifyDir { idx tree node } {
    global GidPriv
    $tree itemconfigure $node -image [gid_themes::GetImage folder.png small_icons]           
}

proc Register::SetProblemtypeForTree { w } {
    global GidPriv GIDDEFAULT

    set sw [ScrolledWindow $w.sw \
            -relief sunken -borderwidth 2]
    set tree [Tree $w.sw.tree \
            -relief flat -borderwidth 0 -width 15 -highlightthickness 0 \
            -highlightcolor red -selectforeground black -selectbackground green -highlightbackground yellow -width 30 \
            -redraw 1 -dropenabled 1 -dragenabled 1 \
            -dragevent 3 \
            -droptypes {
        TREE_NODE    {copy {} move {} link {}}
        LISTBOX_ITEM {copy {} move {} link {}}
    } \
            -opencmd   "Register::TreeModifyDir 1 $w.sw.tree" \
            -closecmd  "Register::TreeModifyDir 0 $w.sw.tree"]
        
    $sw setwidget $tree
        
    $tree configure -selectbackground yellow
        
    foreach i "Double-1 1 Control-1" {
        $tree bindImage <$i> "Register::PressingButOne $tree $i"
        $tree bindText  <$i> "Register::PressingButOne $tree $i"
    }
    pack $sw -side top -expand yes -fill both

    #now only accept modules inside problemtypes directory
    global GiDVisitedDir ;#to avoid infinite loops with links
    unset -nocomplain GiDVisitedDir
    Register::SetProblemTypeTreeForDir [gid_filesystem::get_folder_standard problemtypes] $tree root
    return $tree
}

proc Register::ChooseProblemtype { w } {
    global GidPriv GIDDEFAULT
    set t [toplevel $w.t]
    if { $::tcl_platform(platform) == "windows" } {
        wm transient $t [winfo toplevel [winfo parent $t]]
        wm attributes $t -toolwindow 1
    }
    wm title $t [_ "Choose Problem type"]

    wm geometry $t +[expr [winfo x [winfo toplevel $w]]+20]+[expr [winfo y [winfo toplevel $w]]+20]

    set tree [Register::SetProblemtypeForTree $t]
    pack [ label $t.l -text [_ "Press Control-mouse1 to select several Problem types"] \
            -font SmallFont]
    pack [ttk::frame $t.buts -style BottomFrame.TFrame] -fill x    
    ttk::button $t.buts.ok -text [_ "OK"] -style BottomFrame.TButton -command { set GidPriv(selectmodules) 1 }
    ttk::button $t.buts.cancel -text [_ "Cancel"] -style BottomFrame.TButton -command { set GidPriv(selectmodules) 0 }
    pack $t.buts.ok $t.buts.cancel -side left -padx 3 -pady 3 -expand 1
    set GidPriv(selectmodules) 0
    grab $t
    tkwait variable GidPriv(selectmodules)
    set retval "--CANCEL--"
    if { $GidPriv(selectmodules) == 1 } {
        set retval ""
        foreach i [$tree selection get] {
            set node $i
            set nodename [$tree itemcget $i -text]
            while { [$tree parent $node] != "root" } {
                set node [$tree parent $node]
                set nodename [file join [$tree itemcget $node -text] $nodename]
            }            
            set dir [GidUtils::GiveProblemTypeFullname $nodename].gid
            if { [file isdirectory $dir] } {
                set val $dir
            } else {
                WarnWin [_ "error -%s-" $nodename]
            }
            lappend retval $val
        }
    }
    destroy $t
    return $retval
}

proc Register::CallChooseProblemtype { button } {
    global GidPriv
    
    set retval [Register::ChooseProblemtype $button]
    if { $retval != "--CANCEL--" } {
        set GidPriv(RegisterModule) $retval
        set GidPriv(RegisterModuletext) ""
        foreach i $GidPriv(RegisterModule) {
            lappend GidPriv(RegisterModuletext) [file tail [file root $i]]
        }
        if { $GidPriv(RegisterModuletext) == "" } {
            set GidPriv(RegisterModuletext) ...
        }
        GidHelp $GidPriv(RegisterModulewin) [join $GidPriv(RegisterModuletext) \n]
    }
}

proc Register::GetInitialSearchDir { } {
    set dir ""
    if { $::tcl_platform(platform) == "windows"} {
        if { $::tcl_platform(pointerSize) == 8 } {
            #GiD 64 bits
            set varname ProgramFiles     
        } else {
            #GiD 32 bits
            set varname ProgramFiles(x86)
        }
        if { [info exists ::env($varname)] } {
            set dir [file join $::env($varname) GiD]
            if { ![file exists $dir] || ![file isdirectory $dir]} {
                set dir $::GIDDEFAULT
            }
        } else {
            set dir $::GIDDEFAULT
        }            
    } else {
        set dir $::GIDDEFAULT
    }
    return $dir
}

#split password in blocks separed by '-', to be like the one returned by the web
proc Register::DecoratePassword { password } {
    set password_decorated ""
    set i 0
    while 1 {
        append password_decorated [string range $password $i [expr {$i+3}]]-
        incr i 4
        if { $i >=[string length $password]  } { break }
    }
    set password_decorated [string trimright $password_decorated -]
    return $password_decorated
}

proc Register::SearchInFileFirstValidPasswordLocal { filename id_machinename } {
    set content [GidUtils::ReadFile $filename]
    
    set rex "^$id_machinename\\s+(\\S+)" 
    set password ""
    foreach line [split $content \n] {
        set line [string trim $line]
        if { [string index $line 0] == "#" } {
            continue
        }
        if { [regexp $rex $line dummy pass] } {
            if { $pass eq "IDIDIDIDIDIDIDID" } { 
                continue 
            }
            lassign [GiD_Password gid verify $pass $id_machinename] is_ok message
            if { $is_ok } {
                set password [Register::DecoratePassword $pass]
                break
            }
        }
    }
    return $password
}

proc Register::SearchInFileFirstValidPasswordFloating { filename } {    
    set content [GidUtils::ReadFile $filename]
    set server_ip ""
    foreach line [split $content \n] {
        set line [string trim $line]
        if { [string index $line 0] == "#" } {
            continue
        }
        if { [Register::IsIp $line] } {            
            lassign [Register::IpSplitItems $line] n1 n2 n3 n4 port
            set ip $n1.$n2.$n3.$n4
            if { $port != "" } {
                append ip :$port
            }
            lassign [GiD_Password gid verify $ip]  is_ok message
            if { $is_ok } {
                set server_ip $ip
                break
            }            
        }
        
    }
    return $server_ip    
}

#is_module  must be 0 for GID 1 for a module
proc Register::SearchOtherPasswords { w password_type is_module id_machinename entry } {
    variable initial_dir_search
    if { ![info exists initial_dir_search] } {
        set initial_dir_search [Register::GetInitialSearchDir]
    }
    set password ""    
    if { $is_module } {
        set title [_ "Select %s problemtype.gid directory where the password is installed" $::GidPriv(ProgName)]
    } else {
        set title [_ "Select %s installation directory where the password is installed" $::GidPriv(ProgName)]
    }
    set dir ""
    if { $::tcl_platform(os) == "Darwin"} {
        # in Mac tk_chooseDirectory does not work in /Applications folder, i.e. applications appear disable and can not be selcted
        # AppName.app/Contents/MacOS --> so removing Contents/MacOS
        regsub {/Contents/MacOS} $initial_dir_search {} ini_dir
        set dir [tk_getOpenFile -initialdir $ini_dir -parent $w -title $title]
        #set dir [MessageBoxGetFilename file read $title $ini_dir]
        if { $dir != "" } { 
            # Must check if it exists
            set dir [file join $dir Contents MacOS]
        }
    } else {
        set dir [MessageBoxGetFilename directory read $title $initial_dir_search]
    }
    if { $dir == "" } { 
        return 
    }
    set initial_dir_search $dir
    
    if { $is_module } {
        if { ![string equal -nocase [file extension $dir] ".gid"] } {
            WarnWin [_ "Error: You must select the directory containing the problemtype.gid"]
            return
        }
        set filename [file join $dir password.txt]
        if { ![file exists $filename] } {
            WarnWin [_ "Error: This problemtype does not contain passwords"]
            return
        }
        set fin [open $filename r]
        set txt [read $fin]
        close $fin
        
        set rex "(?n)^$id_machinename\\s+(\\S+)"
        set olderpass ""
        foreach "- pass" [regexp -inline -all $rex $txt] {
            set olderpass $pass
        }
        set ipserver ""
        if { $olderpass == "" } {            
            if { [Register::IsIp $txt] } {
                lassign [Register::IpSplitItems $txt] n1 n2 n3 n4 port
                set olderpass $n1.$n2.$n3.$n4
                if { $port != "" } {
                    append olderpass :$port
                }
            }
        }
        set password $olderpass
    } else {
        set filename [file join $dir scripts TemporalVariables]
        if { ![file exists $filename] } {
            WarnWin [_ "Error: You must select the directory containing one installed %s" $::GidPriv(ProgName)]
            return
        }        
        if { $password_type == "floating" } {
            set ip [Register::SearchInFileFirstValidPasswordFloating $filename]
            if { $ip == "" } {                
                lassign [Register::GetLocalMachineDevice] id_machinename id_operating_system id_sysinfo
                set password [Register::SearchInFileFirstValidPasswordLocal $filename $id_machinename]
                if { $password != "" } {
                    #destroy $w
                    set ::tkPriv(button) "cancel" ;#dark trick to close the previous window
                    GidUtils::SetWarnLine [_ "Floating server ip not found, but a local password %s is valid, use Register GiD->" $password]
                    after 1000 Register::GiD_Local $is_module $password
                    return
                }
            } else {
                set password $ip
            }
        } elseif { $password_type == "local" || $password_type == "usb"} { 
            set password [Register::SearchInFileFirstValidPasswordLocal $filename $id_machinename]
            if { $password == "" } {
                set ip [Register::SearchInFileFirstValidPasswordFloating $filename]
                if { $ip != "" } {
                    #destroy $w
                    set ::tkPriv(button) "cancel" ;#dark trick to close the previous window
                    GidUtils::SetWarnLine [_ "Local password not found, but a server ip %s is valid, use Register GiD->Floating" $ip]
                    after 1000 Register::GiD_Floating $is_module $ip
                    return
                }
            }
        } else {
            W "Unexpected password_type=$password_type"
        }
    }
    if { $password == "" } {
        if { $is_module } {
            set message [_ "Could not find a valid password in this problemtype"]
        } else {
            set message [_ "Could not find a valid password in this %s" $::GidPriv(ProgName)]
        }
        WarnWin $message
    } else {
        $entry delete 0 end
        $entry insert end $password
    }
}

proc Register::EnterLink {w} {
    $w tag configure link -underline on -foreground #336699
    $w configure -cursor hand2
}

proc Register::LeaveLink {w} {
    $w tag configure link -underline off -foreground #336699
    $w configure -cursor arrow
}

#if selection is void copy all text contents
proc Register::CopyToClipboard { t txt} {
    if {  $txt != "" } {
        clipboard clear -displayof $t
        clipboard append -displayof $t $txt
    }
}

proc Register::CopyTextSelectionOrAll { t } {
    set txt ""
    if {[string match $t [selection own -displayof $t]]} {
        catch {
            set txt [selection get -displayof $t]
        }
    }
    if { $txt == "" } {
        set txt [$t get 1.0 end-1c]
    }
    Register::CopyToClipboard $t $txt
}

proc Register::GetSelection { w } {
    if {
        ![catch {selection get -displayof $w -selection CLIPBOARD} txt]
    } {
        return $txt
    }
    return ""
}

proc Register::ClipboardCopy { w } {
    if {[string match $w [selection own -displayof $w]]} {
        clipboard clear -displayof $w
        catch {
            set txt [selection get -displayof $w]
            clipboard append -displayof $w $txt
        }
    }
}

proc Register::ClipboardPaste { w } {
    if {![catch {Register::GetSelection $w} txt]} {
        $w insert insert $txt
    }
}

proc Register::MenuContextualEntryPASS { lb x y X Y} {
    set menu $lb.menu
    destroy $menu
    menu $menu -tearoff 0    
    $menu add command -label [_ "Copy"] -command [list Register::ClipboardCopy $lb]
    $menu add command -label [_ "Paste"] -command [list Register::ClipboardPaste $lb]    
    tk_popup $menu $X $Y
}

proc Register::MenuContextualTextPASS { lb x y X Y { url ""}} {
    set menu $lb.menu
    if { ![ winfo exists $lb.menu]} {
        destroy $menu
        menu $menu -tearoff 0
        #$menu add command -label [_ "Copy"] -command [list Register::ClipboardCopy $lb]
        $menu add command -label [_ "Copy"] -command [list Register::CopyTextSelectionOrAll $lb]
        if { $url != ""} {
            $menu add command -label [_ "Copy URL"] -command [ list Register::CopyToClipboard $lb $url]
        }
    }
    tk_popup $menu $X $Y
}

proc Register::GetURL { web gid_main_version password_type id_machinename id_operating_system id_sysinfo } {
    package require http
    package require gid_cross_platform
    if { $password_type == "local" } {
        set Licencetype 1
    } elseif { $password_type == "usb" } {
        set Licencetype 2
    } elseif { $password_type == "floating" } {
        set Licencetype 3
    } else {
        set Licencetype 1
    }
    #Periodtime 1 (month), 2 permanent
    set OS [string range [string totitle $id_operating_system] 0 2]
    set query [eval http::formatQuery [list dis 1 Gidversion $gid_main_version Licencetype $Licencetype Machine $id_machinename OS $OS Number $id_sysinfo]]       
    set url $web?$query
    return $url
}

proc Register::VisitWeb { web gid_main_version password_type id_machinename id_operating_system id_sysinfo } {
    package require gid_cross_platform
    set url [ Register::GetURL $web $gid_main_version $password_type $id_machinename $id_operating_system $id_sysinfo]
    gid_cross_platform::open_by_extension $url    
}

#is_module 0 for GID, 1 for a module
#password_type: "unknown", "local", "usb" "floating"
proc Register::AskPasswordWindow {password_type is_module id_machinename id_operating_system id_sysinfo default} {
    global tkPriv GidPriv
    set result ""

    set w .gid.tempwineps

    if { $is_module } {
        set title [_ "Register Problem type"] 
        set web $GidPriv(ProblemtypeWebPassword)
        set email $GidPriv(ProblemtypeContactMail)
    } else {
        set title [_ "Enter password window"]
        set web $GidPriv(PassWeb)
        set email $GidPriv(ContactMail)
    }
    set gid_main_version [GidUtils::GetGiDVersionMain]

    if { $password_type == "local" || $password_type == "unknown" }  {
        if { $password_type == "local" && [string range $id_machinename 0 3] == "usb_" } {
            W "Warning: it seems that the machine is an usb one"
        }
        set text [_ "Contact %s to obtain the password for this host" $email]
        append text ":\n\n"
        append text [_ "Name"] ": $id_machinename\n"
        append text [_ "Operating System"] ": $id_operating_system\n"
        append text [_ "Sysinfo"] ": $id_sysinfo\n"
        if { !$is_module } {
            append text [_ "Version"] ": $gid_main_version\n"
        }
    } elseif { $password_type == "usb" } {
        if { [string range $id_machinename 0 3] != "usb_" } {
            W "Warning: it seems that the machine is not an usb one"         
        }
        set text [_ "Contact %s to obtain the password for this host" $email]
        append text ":\n\n"
        append text [_ "Name"] ": $id_machinename\n"
        append text [_ "Sysinfo"] ": $id_sysinfo\n"
        if { !$is_module } {
            append text [_ "Version"] ": $gid_main_version\n"
        }
    } elseif { $password_type == "floating" } {
        set text [_ "Contact %s to obtain the floating password for passerver" $email]
        append text ":\n\n"
        append text [_ "Enter the IP of the server where passerverd is running"]
    } else {
        W "Unexpected password_type=$password_type"
    }
    GidUtils::SetWarnLine [ regsub -all {\n} $text { }]

    # 1. Create the top-level window and divide it into top, middle and bottom parts.
    
    while { [winfo exists $w] } { append w a }
    catch {destroy $w}
    toplevel $w
    wm transient $w [winfo toplevel [winfo parent $w]]
    if { $::tcl_platform(platform) == "windows" } {
        # wm transient $w [winfo toplevel [winfo parent $w]]
        wm attributes $w -toolwindow 1
    }
    
    set GidPriv(PassW) $w

    wm title $w $title
    wm iconname $w Dialog
    wm protocol $w WM_DELETE_WINDOW [list event generate $w <Escape>]
    
    ttk::frame $w.top -style flat.TFrame    
    text $w.top.msg -wrap word -font NormalFont -borderwidth 1 -height 8 -width 30
    
    set gid_url [ Register::GetURL $web $gid_main_version $password_type $id_machinename $id_operating_system $id_sysinfo]
    $w.top.msg tag bind link <1> [list Register::VisitWeb $web $gid_main_version $password_type $id_machinename $id_operating_system $id_sysinfo]
    $w.top.msg tag configure link -underline off -foreground #336699
    $w.top.msg tag bind link <Any-Enter> [list Register::EnterLink $w.top.msg]    
    $w.top.msg tag bind link <Any-Leave> [list Register::LeaveLink $w.top.msg]
    
    $w.top.msg insert 0.0 $text    
    $w.top.msg insert end "\n"
    $w.top.msg insert end "[_ "or get it from"]: "
    $w.top.msg insert end $web link    
        
    $w.top.msg configure -state disabled
    bind $w.top.msg <1> [list focus $w.top.msg]
    bind $w.top.msg <$::gid_right_button> [list Register::MenuContextualTextPASS $w.top.msg %x %y %X %Y $gid_url]
    
    grid $w.top.msg -sticky nsew
    grid rowconfigure $w.top 0 -weight 1
    grid columnconfigure $w.top 0 -weight 1    
    
    ttk::frame $w.middle -style raised.TFrame -borderwidth 1    
    ttk::entry $w.middle.e -textvariable ::GidPriv(entry) -width 50
    if { $password_type == "floating" } {
        setTooltip $w.middle.e [_ "Enter the IP address of the licence server (passerver)"]
    } else {
        setTooltip $w.middle.e [_ "Enter a password for GiD.\nYou can get a one-month free password from %s" $::GidPriv(PassWeb)]
    }
    ttk::button $w.middle.search -image [gid_themes::GetImage find_folder.png small_icons] \
        -command [list Register::SearchOtherPasswords $w $password_type $is_module $id_machinename $w.middle.e]
    GidHelp $w.middle.search [_ "Use this button to copy the password from previously installed programs"]
    
    $w.middle.e delete 0 end
    $w.middle.e insert end $default
    $w.middle.e selection range 0 end
    
    bind $w.middle.e <$::gid_right_button> [list Register::MenuContextualEntryPASS $w.middle.e %x %y %X %Y]
    
    if { $is_module } {
        ttk::label $w.middle.l1 -text [_ "Choose Problem type"]:
        if { $password_type == "floating" } {
            set text [_ "Enter the server IP address"]:
        } else {
            set text [_ "Enter the password"]:           
        }
        tk::label $w.middle.l2 -text $text
        
        ttk::button $w.middle.om -textvariable GidPriv(RegisterModuletext)
        set GidPriv(RegisterModulewin) $w.middle.om
        if {![info exists GidPriv(RegisterModule)] || ![llength $GidPriv(RegisterModule)]} {
            if {[catch "GiD_Info problemtypepath" GidPriv(RegisterModule)]} {
                set GidPriv(RegisterModuletext) "..."
                set GidPriv(RegisterModule) ""
                GidHelp $w.middle.om ...
            } else {
                set GidPriv(RegisterModule) [list $GidPriv(RegisterModule)]
                set GidPriv(RegisterModuletext) [GiD_Info Project ProblemType]
                GidHelp $w.middle.om $GidPriv(RegisterModuletext)
            }
        }
        
        $w.middle.om configure -command [list Register::CallChooseProblemtype $w.middle.om]
        
        text $w.middle.msg -wrap word -height 3 -width 30
        
        $w.middle.msg insert end [_ "This password is used to register the selected Problem types.\nTo register %s, please select 'Register' in the 'Help' menu." $::GidPriv(ProgName)]
        $w.middle.msg configure -state disabled
        
        grid $w.middle.l1 $w.middle.om $w.middle.search -sticky ew        
        grid $w.middle.l2 $w.middle.e -sticky ew 
        grid configure $w.middle.e -columnspan 2
        grid $w.middle.msg -sticky news -columnspan 3 -pady 2 -padx 2
        grid rowconfigure $w.middle {0 1 2} -weight 1
        grid columnconfigure $w.middle 1 -weight 1                        
    } else {                
        if { $password_type == "floating" } {
            set text [_ "Enter the server IP address"]:
        } else {
            set text [_ "Enter the password"]:           
        }
        ttk::label $w.middle.l2 -text $text
        grid $w.middle.l2 $w.middle.e $w.middle.search -sticky ew
        grid columnconfigure $w.middle 1 -weight 1
    }

    # 3. Create a row of buttons at the bottom of the dialog.    
    ttk::frame $w.bot -style BottomFrame.TFrame  

    set values [list ok cancel]
    set labels [list [_ "Ok"] [_ "Cancel"]]
    set helps [list [_ "Validates entered password and register GiD."] \
                [_ "Close this window."]]    
    set i 0
    foreach value $values label $labels hh $helps {
	ttk::button $w.bot.button$i -text $label -command [list set tkPriv(button) $value] -style BottomFrame.TButton
	setTooltip $w.bot.button$i $hh
	lappend buttonslist $w.bot.button$i
	# pack $w.button$i -in $w.bot -side left -expand 1 -padx 3 -pady 2 -fill x
	grid $w.bot.button$i -sticky ew -padx 20 -pady 10 -row 0 -column $i
	grid columnconfigure $w.bot $i -weight 1
	## bind $w.bot.button$i <Return> "
	##     update idletasks
	##     after 100
	##     set tkPriv(button) $value
	##     break
	##     "
	incr i
    }    
    grid rowconfigure $w.bot 0 -weight 1
    
    bind $w <Escape> "
        update idletasks
        after 100
        set tkPriv(button) cancel
        "                 
    bind $w <Return> "
        update idletasks
        after 100
        set tkPriv(button) ok
        "
    grid $w.top -sticky nsew -padx 4 -pady 4
    grid $w.middle -sticky ew -padx 4 -pady 4
    grid $w.bot -sticky ew -padx 4 -pady 4
    grid rowconfigure $w 0 -weight 1
    grid rowconfigure $w 2 -weight 1
    grid columnconfigure $w 0 -weight 1
    
    # 5. Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display and de-iconify it.
    wm withdraw $w
    update idletasks        
    # extrange errors with updates
    if { ![winfo exists $w] } { 
        return ""
    }
    
    set parent [winfo toplevel [winfo parent $w]]
    set x [expr [winfo x $parent]+[winfo width $parent ]/2- [winfo reqwidth $w]/2]
    set y [expr [winfo y $parent]+[winfo height $parent ]/2- [winfo reqheight $w]/2]
    if { $x < 0 } { set x 0 }
    if { $y < 0 } { set y 0 }

    WmGidGeom $w +$x+$y
    update
    if { ![winfo exists $w] } { 
        return ""
    }
    wm deiconify $w
    
    # 6. Set a grab and claim the focus too.
    
    set oldFocus [focus]
    set oldGrab [grab current $w]
    if {$oldGrab != ""} {
        set grabStatus [grab status $oldGrab]
    }

    # Make it topmost to avoid putting the window behind gid's.
    # wm attributes $w -topmost 1
    # raise $w
    # GidUtils::WindowAboveGid $w

    raise $w
    grab $w
    
    focus $w.middle.e
    after 100 [list catch [list focus -force $w.middle.e]]
    
    # 7. Wait for the user to respond, then restore the focus and
    # return the index of the selected button.  Restore the focus
    # before deleting the window, since otherwise the window manager
    # may take the focus away so we can't redirect it.  Finally,
    # restore any grab that was in effect.
    while 1 {
        tkwait variable tkPriv(button)
        set result [$w.middle.e get]
        break
    }
    
    catch {focus $oldFocus}
    destroy $w
    if {$oldGrab != ""} {
        if {$grabStatus == "global"} {
            grab -global $oldGrab
        } else {
            grab $oldGrab
        }
    }
        
    if { $is_module } {
        if { $tkPriv(button) == "ok" } {
            set result [list $::GidPriv(entry)]
            if {$GidPriv(RegisterModule) ne ""} {
                lappend result $GidPriv(RegisterModule)
            }
        } elseif { $tkPriv(button) == "cancel" } {
            set result --CANCEL--
        }
    } else {
        if { $tkPriv(button) == "ok"} {
            if { ![info exists ::GidPriv(entry)]} {
                set ::GidPriv(entry) ""
            }
            regsub -all -- {-} $::GidPriv(entry) {} retval
            regsub -all -- {\n} $retval {} retval2
            regsub -all -- { } $retval2 {} retval
            set result $retval
        } elseif { $tkPriv(button) == "cancel"} {
            set result --CANCEL--
        }
    }
    return $result
}

#to select an UBS from a list
namespace eval GiDSelectRowWindow {
    variable _w
    variable _list
    variable _seleccion
}

proc GiDSelectRowWindow::GetSelection { } {
    variable _w
    variable _list
    variable _seleccion
    set _seleccion [$_list curselection]
    if { $_seleccion != ""} {
        set _seleccion [$_list get $_seleccion]
        destroy $_w
    } else {
        WarnWin [_ "Please, select first one sysinfo"]
    }
}

proc GiDSelectRowWindow::SelectDeviceWindowCatalina {} {
    set lst_usb_devices [Register::GetDevicesUsb]
    set selected ""
    if { [ llength $lst_usb_devices] != 0} {
        set usb_selected_path [ tk_chooseDirectory \
            -initialdir "/Volumes" -mustexist 1 -parent .gid \
            -title [_ "Select one USB device to register"]]
        # May be the user selected a file inside the USB device
        if { [ regexp {^(/Volumes/[^/]+)/.*$} $usb_selected_path dummy volume_name] != 0} {
            set usb_selected_path $volume_name
        }
        if { $usb_selected_path != ""} {
            # from the usb device list, get the one with the selected path
            foreach item [Register::GetDevicesUsb] {
                lassign $item name vendor_product sysinfo gid_password_filename
                set item_mount_path [ file dirname $gid_password_filename]
                if { $usb_selected_path == $item_mount_path} {
                    set selected $item
                    break
                }
            }
        } else {
            # User selected nothing, so quitting
            return ""
        }
        if { $selected == ""} {
            ErrorWin "'$usb_selected_path' can not be detected as USB device."
            return ""
        }
    }
    return $selected
}

proc GiDSelectRowWindow::SelectDeviceWindow { } {
    variable _w
    variable _list
    variable _seleccion

    # macOS version 10.13.x is Darwin version 17.x
    # macOS version 10.14.x is Darwin version 18.x
    # macOS version 10.15.x is Darwin version 19.x <-- restricted access to USB devices, same as folders Documents, Download, ...
    if { [ isMacOSVersionEqualOrGreater Catalina ]} {
        set _seleccion [ GiDSelectRowWindow::SelectDeviceWindowCatalina]
        return $_seleccion
    }
    
    if { [GidUtils::IsTkDisabled] } {
        #e.g. batch mode without windows
        return
    }

    package require tablelist_tile
    
    set title [_ "Sysinfo selection"]
    set msg [_ "Select one of the following sysinfos to register the program"]
    set tabla_datos [list]
    foreach item [Register::GetDevicesUsb] {
        lassign $item name os_or_description sysinfo unit_filename
        set usb_path [file rootname $unit_filename]
        lappend tabla_datos [list $name $os_or_description $sysinfo $usb_path]
    }
    
    set _w .gid.row_window
    set w $_w
    
    catch { destroy $w }
    
    InitWindow2 $w -title $title \
        -geometryvariable PrePostGiDSelectRowWindowGeom -ontop
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0
    
    ttk::frame $w.f1
    
    set fila_tbl 0
    if { $msg != ""} {
        label $w.f1.l1 -text $msg -font BoldFont -justify left -anchor w -height 1 
        incr fila_tbl
    }
    
    set sw [ScrolledWindow $w.f1.lf -auto both -relief groove -borderwidth 2]
    
    set cols {}
    #
    set cab [list [list 10 [_ "Name"] left] [list 16 [_ "Description"] left] [list 16 [_ "Sysinfo"] left] [list 10 [_ "Path"] left]]
    foreach c $cab {
        foreach e $c {
            lappend cols $e
        }
    }
    
    set _list [tablelist::tablelist $sw.lb -exportselection 0 -columns $cols -labelcommand tablelist::sortByColumn -stretch end]
    #-selectmode extended
    $sw setwidget $_list
    
    if { $msg != ""} {
        grid $w.f1.l1   -sticky ew
    }
    grid $sw       -sticky nsew
    grid columnconfigure $w.f1 0 -weight 1
    grid rowconfigure $w.f1 $fila_tbl -weight 1
    
    ttk::frame $w.buts -style BottomFrame.TFrame
    
    ttk::button $w.buts.b1 -text [_ "Select"] -command [list GiDSelectRowWindow::GetSelection] -style BottomFrame.TButton
    ttk::button $w.buts.b2 -text [_ "Cancel"] -command [list destroy $w] -style BottomFrame.TButton
    
    grid $w.buts.b1 $w.buts.b2 -padx 4 -pady 10
    
    grid $w.f1 -sticky nsew
    grid $w.buts -sticky ew
    grid anchor $w.buts center
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1
    
    
    bind $w <Double-1> GiDSelectRowWindow::GetSelection
    
    update
    
    # a rellenar datos
    $_list delete 0 end
    foreach item $tabla_datos {
        set f $item
        $_list insert end $f
    }
    
    set _seleccion ""
    tkwait window $_w
    return $_seleccion   
}

#related procs of gid_zmesher.tcl moved here
namespace eval Zmesher {
}

proc Zmesher::GetURL { web gid_main_version password_type id_machinename id_operating_system id_sysinfo } {
    package require http
    if { $password_type == "local" } {
        set LicenceType 1
    } else {
        #by now allow only local licence
        set LicenceType 1
    }
    set OS [string range [string totitle $id_operating_system] 0 2]
    set query [eval http::formatQuery [list dis 1 Gidversion $gid_main_version LicenceType $LicenceType Machine $id_machinename OS $OS Number $id_sysinfo]]       
    set url $web?$query
    return $url
}

proc Zmesher::VisitWeb { web gid_main_version password_type id_machinename id_operating_system id_sysinfo } {
    package require gid_cross_platform
    set url [ Zmesher::GetURL $web $gid_main_version $password_type $id_machinename $id_operating_system $id_sysinfo ]
    gid_cross_platform::open_by_extension $url    
}

#password_type: local
proc Zmesher::AskPasswordWindow { password_type id_machinename id_operating_system id_sysinfo {default ""}} {   
    if { [GidUtils::IsTkDisabled] } {
        return 1
    }   
    set web https://www.gidsimulation.com/zmesher
    set gid_main_version [GidUtils::GetGiDVersionMain]
    set text  [_ "Must get a password in order to use Zmesher cartesian mesher"]\n
    append text [_ "To obtain the Zmesher password for this host"]:
    append text "\n\n"
    append text [_ "Name"] ": $id_machinename\n"
    append text [_ "Operating System"] ": $id_operating_system\n"
    append text [_ "Sysinfo"] ": $id_sysinfo\n"
    append text [_ "Version"] ": $gid_main_version\n"
    append text "\n"
    append text [_ "Fill in this form"]:
    
    set w .gid.zmesheraskpassword

    InitWindow2 $w -title [_ "Display info"] \
        -geometryvariable PreRetrieveProblemtypeDisplayInfoWindowGeom -ontop
    ttk::frame $w.top -style flat.TFrame 
    
    text $w.top.msg -wrap word -font NormalFont -borderwidth 1 -height 10 -width 50
    set zmesher_url [Zmesher::GetURL $web $gid_main_version $password_type $id_machinename $id_operating_system $id_sysinfo]
    $w.top.msg tag bind link <1> [ list Zmesher::VisitWeb $web $gid_main_version $password_type $id_machinename $id_operating_system $id_sysinfo]
    $w.top.msg tag configure link -underline off -foreground #336699
    $w.top.msg tag bind link <Any-Enter> [list Register::EnterLink $w.top.msg]    
    $w.top.msg tag bind link <Any-Leave> [list Register::LeaveLink $w.top.msg]
    
    $w.top.msg insert 0.0 $text
    $w.top.msg insert end $web link
               
    $w.top.msg configure -state disabled
    bind $w.top.msg <1> [list focus $w.top.msg]
    bind $w.top.msg <$::gid_right_button> [list Register::MenuContextualTextPASS $w.top.msg %x %y %X %Y $zmesher_url]

    ttk::frame $w.middle -style raised.TFrame -borderwidth 1    
    ttk::entry $w.middle.e -width 50
    setTooltip $w.middle.e [_ "Enter a password for zmesher"]       
    $w.middle.e delete 0 end
    $w.middle.e insert end $default
    $w.middle.e selection range 0 end    
    bind $w.middle.e <$::gid_right_button> [list Register::MenuContextualEntryPASS $w.middle.e %x %y %X %Y ]

    ttk::label $w.middle.l2 -text [_ "Enter the password"]:
    grid $w.middle.l2 $w.middle.e -sticky ew
    grid columnconfigure $w.middle 1 -weight 1
    
    ttk::frame $w.bot -style BottomFrame.TFrame  
    ttk::button $w.bot.ok -text [_ "Ok"] -width 10 -command [list Zmesher::OnOk $w $id_machinename] -style BottomFrame.TButton 
    ttk::button $w.bot.cancel -text [_ "Cancel"] -width 10 -command [list destroy $w] -style BottomFrame.TButton   

    grid $w.top.msg -sticky nsew
    grid rowconfigure $w.top 0 -weight 1
    grid columnconfigure $w.top 0 -weight 1
    

    grid $w.bot.ok $w.bot.cancel -padx 2 -pady 3
    grid anchor $w.bot center

    grid $w.top -sticky nsew -padx 4 -pady 4
    grid $w.middle -sticky ew -padx 4 -pady 4
    grid $w.bot -sticky ew -padx 4 -pady 4
    grid rowconfigure $w 0 -weight 1
    grid columnconfigure $w 0 -weight 1
    # this grab may fail "grab failed: another application has grab"
    catch {
	grab $w
    }
    focus $w.middle.e
    return 0
}

proc Zmesher::OnOk { w id_machinename } {
    set text_widget $w.middle.e
    if { [winfo exists $text_widget] } {
        set password [string trim [$text_widget get]]
        if { $password != "" } {
            lassign [GiD_Password zmesher register $password $id_machinename] password_ok msg
            if { $password_ok } {
                destroy $w
                GidUtils::SetWarnLine [_ "Password ok, now zmesher can generate cartesian meshes"]
            } else {
                WarnWin $msg
            }
        }
    }
}

#used by command GiD_Password $program save
proc Zmesher::SavePassword { id_machinename password } {
    set filename [file join $::GIDDEFAULT scripts TemporalVariables]
    set txt "$id_machinename $password # Password for module 'zmesher'"     
    set found 0
    set old_file_content [GidUtils::ReadFile $filename "" 0]
    foreach line [split $old_file_content \n] {
        if { [string index $line 0] == "#" } {
            continue
        }
        if { [lindex $line 0] == $id_machinename && [lindex $line 1] == $password } {
            set found 1
            break
        }
    }
    if { !$found } {    
        set fp ""
        #catch necessary to avoid raise possible Tcl error "couldn't open "C:/Program Files/GiD/GiD 13.1.10d/scripts/TemporalVariables": permission denied"
        catch {        set fp [open $filename a] } msg
        if { $fp != "" } {
            puts $fp $txt
            close $fp
        } else {
            package require gid_cross_platform
            gid_cross_platform::run_as_administrator [GidUtils::GetFullPathGiDExe] -tclsh [file join $::GIDDEFAULT scripts write_text.tcl] $filename $txt
        }
    }
    return 0
}
