namespace eval Quaternion {} {
    variable QUATERNION_EPS 1e-4

}

proc Quaternion::Indentity {} {
    return [list 1.0 0.0 0.0 0.0]
}

# double axis[3], double angle radians
# return Quaternion output
proc Quaternion::FromAxisAngle { axis angle } {
    # Formula from http://www.euclideanspace.com/maths/geometry/rotations/conversions/angleToQuaternion/
    lassign $axis axis_x axis_y axis_z
    set w [expr cos($angle / 2.0)]
    set c [expr sin($angle / 2.0)]
    set vx [expr $c * $axis_x]
    set vy [expr $c * $axis_y]
    set vz [expr $c * $axis_z]
    return [list $w $vx $vy $vz]
}

#Quaternion* q, double output[3]
proc Quaternion::ToAxisAngle { q } {
    lassign $q qw qx qy qz
    # Formula from http://www.euclideanspace.com/maths/geometry/rotations/conversions/quaternionToAngle/
    set angle [expr 2.0 * acos($qw)]
    set divider [expr sqrt(1.0 - $qw * $qw)]
    if { $divider != 0.0 } {
        # Calculate the axis
        set axis_x [expr $qx / $divider]
        set axis_y [expr $qy / $divider]
        set axis_z [expr $qz / $divider]
    } else {
        # Arbitrary normalized axis
        set axis_x  1
        set axis_y 0
        set axis_z 0
    }
    return [list [list $axis_x $axis_y $axis_z] $angle]
}

#double angle, Quaternion* output
proc Quaternion::FromXRotation { angle } {
    return [Quaternion::FromAxisAngle {1.0 0 0} $angle]
}

#double angle, Quaternion* output
proc Quaternion::FromYRotation { angle } {
    return [Quaternion::FromAxisAngle {0 1.0 0} $angle]
}

#double angle, Quaternion* output
proc Quaternion::FromZRotation { angle } {
    return [Quaternion::FromAxisAngle {0 0 1.0} $angle]
}

#double eulerZYX[3], Quaternion* output
proc Quaternion::FromEulerZYX { eulerZYX }  {
    # Based on https://en.wikipedia.org/wiki/Conversion_between_quaternions_and_Euler_angles
    lassign $eulerZYX eulerZYX_x eulerZYX_y eulerZYX_z
    set cy [expr cos($eulerZYX_z * 0.5)]
    set sy [expr sin($eulerZYX_z * 0.5)]
    set cr [expr cos($eulerZYX_x * 0.5)]
    set sr [expr sin($eulerZYX_x * 0.5)]
    set cp [expr cos($eulerZYX_y * 0.5)]
    set sp [expr sin($eulerZYX_y * 0.5)]

    set w [expr $cy * $cr * $cp + $sy * $sr * $sp]
    set vx [expr $cy * $sr * $cp - $sy * $cr * $sp]
    set vy [expr $cy * $cr * $sp + $sy * $sr * $cp]
    set vz [expr $sy * $cr * $cp - $cy * $sr * $sp]
    return [list $w $vx $vy $vz]
}

#Quaternion* q, double output[3]
proc Quaternion::ToEulerZYX { q } {
    lassign $q qw qx qy qz
    # Roll (x-axis rotation)
    set sinr_cosp [expr 2.0 * ($qw *$qx +$qy *$qz)]
    set cosr_cosp [expr 1.0 - 2.0 * ($qx *$qx +$qy *$qy)]
    set eulerZYX_x [expr atan2($sinr_cosp, $cosr_cosp)]
    # Pitch (y-axis rotation)
    set sinp [expr 2.0 * ($qw *$qy -$qz *$qx)]
    if { [expr abs($sinp)] >= 1} {
        # use 90 degrees if out of range
        set M_PI 3.1415926535897932384626433832795
        if { $sinp>=0.0} {
            set eulerZYX_y [expr abs($M_PI / 2)]
        } else {
            set eulerZYX_y [expr -abs($M_PI / 2)]
        }
    } else {
        set eulerZYX_y [expr asin($sinp)]
    }
    # Yaw (z-axis rotation)
    set siny_cosp [expr 2.0 * ($qw *$qz +$qx *$qy)]
    set cosy_cosp [expr 1.0 - 2.0 * ($qy *$qy +$qz *$qz)]
    set eulerZYX_z [expr atan2($siny_cosp, $cosy_cosp)]
    return [list $eulerZYX_x $eulerZYX_y $eulerZYX_z]
}

#Quaternion* q, Quaternion* output
proc Quaternion::Conjugate { q } {
    lassign $q qw qx qy qz
    set w $qw
    set vx [expr -$qx]
    set vy [expr -$qy]
    set vz [expr -$qz]
    return [list $w $vx $vy $vz]
}

#Quaternion* q
proc Quaternion::Norm { q } {
    lassign $q qw qx qy qz
    return [expr sqrt($qw*$qw + $qx*$qx + $qy*$qy + $qz*$qz)]
}

proc Quaternion::FromRotationMatrix3 { mat } {
    lassign $mat m00 m01 m02 m10 m11 m12 m20 m21 m22
    set tr [expr $m00 + $m11 + $m22]
    if { $tr > 0} { 
        # S=4*qw 
        set S [expr sqrt( $tr + 1.0) * 2]
        if { $S == 0} {
            set iS 0
        } else {
            set iS [expr 1.0 / $S]
        }
        set qw [expr 0.25 * $S]
        set qx [expr ( $m21 - $m12) * $iS]
        set qy [expr ( $m02 - $m20) * $iS] 
        set qz [expr ( $m10 - $m01) * $iS] 
    } elseif { ( $m00 > $m11) && ( $m00 > $m22)} {
        # S=4*qx 
        set S [expr sqrt(1.0 + $m00 - $m11 - $m22) * 2]
        if { $S == 0} {
            set iS 0
        } else {
            set iS [expr 1.0 / $S]
        }
        set qw [expr ( $m21 - $m12) * $iS]
        set qx [expr 0.25 * $S]
        set qy [expr ( $m01 + $m10) * $iS]
        set qz [expr ( $m02 + $m20) * $iS]
    } elseif { $m11 > $m22} { 
        # S=4*qy
        set S [expr sqrt(1.0 + $m11 - $m00 - $m22) * 2]
        if { $S == 0} {
            set iS 0
        } else {
            set iS [expr 1.0 / $S]
        }
        set qw [expr ( $m02 - $m20) * $iS]
        set qx [expr ( $m01 + $m10) * $iS]
        set qy [expr 0.25 * $S]
        set qz [expr ( $m12 + $m21) * $iS]
    } else { 
         # S=4*qz
        set S [expr sqrt(1.0 + $m22 - $m00 - $m11) * 2]
        if { $S == 0} {
            set iS 0
        } else {
            set iS [expr 1.0 / $S]
        }
        set qw [expr ( $m10 - $m01) * $iS]
        set qx [expr ( $m02 + $m20) * $iS]
        set qy [expr ( $m12 + $m21) * $iS]
        set qz [expr 0.25 * $S]
    }
    return [list $qw $qx $qy $qz]
}

proc Quaternion::FromRotationMatrix4 { m4 } {
    lassign $m4 m00 m01 m02 m03 m10 m11 m12 m13 m20 m21 m22 m23 m30 m31 m32 m33 
    return [Quaternion::FromRotationMatrix3 [list $m00 $m01 $m02 $m10 $m11 $m12 $m20 $m21 $m22]]
}

#Quaternion* q, Quaternion* output
proc Quaternion::Normalize { q } {
    lassign $q qw qx qy qz
    set len [Quaternion::Norm $q]
    set w [expr $qw/$len]
    set vx [expr $qx/$len]
    set vy [expr $qy/$len]
    set vz [expr $qz/$len]
    return [list $w $vx $vy $vz]
}

# Formula from http://www.euclideanspace.com/maths/algebra/realNormedAlgebra/quaternions/arithmetic/index.htm
#             a*e - b*f - c*g - d*h
#        + i (b*e + a*f + c*h- d*g)
#        + j (a*g - b*h + c*e + d*f)
#        + k (a*h + b*g - c*f + d*e)
proc Quaternion::Multiply { q1 q2 } {
    lassign $q1 pw px py pz
    lassign $q2 qw qx qy qz
    set w [expr $pw   *$qw    - $px*$qx - $py*$qy - $pz*$qz]
    set vx [expr $px*$qw    + $pw   *$qx + $py*$qz - $pz*$qy]
    set vy [expr $pw   *$qy - $px*$qz + $py*$qw    + $pz*$qx]
    set vz [expr $pw   *$qz + $px*$qy - $py*$qx + $pz*$qw]
    return [list $w $vx $vy $vz]
}

proc Quaternion::ToRotationMatrix3 { quat } {
    # does not depend if quaternion is normalized
    lassign $quat qw qx qy qz
    set sqw [expr $qw * $qw];
    set sqx [expr $qx * $qx];
    set sqy [expr $qy * $qy];
    set sqz [expr $qz * $qz];
    
    # invs (inverse square length) is only required if quaternion is not already normalised
    set invs [expr 1.0 / ( $sqx + $sqy + $sqz + $sqw)]
    set m00 [expr ( $sqx - $sqy - $sqz + $sqw) * $invs]; # since sqw + sqx + sqy + sqz =1/invs*invs
    set m11 [expr (- $sqx + $sqy - $sqz + $sqw) * $invs];
    set m22 [expr (- $sqx - $sqy + $sqz + $sqw) * $invs];
    
    set tmp1 [expr $qx * $qy];
    set tmp2 [expr $qz * $qw];
    set m10 [expr 2.0 * ( $tmp1 + $tmp2) * $invs];
    set m01 [expr 2.0 * ( $tmp1 - $tmp2) * $invs];
    
    set tmp1 [expr $qx * $qz];
    set tmp2 [expr $qy * $qw];
    set m20 [expr 2.0 * ( $tmp1 - $tmp2) * $invs];
    set m02 [expr 2.0 * ( $tmp1 + $tmp2) * $invs];
    set tmp1 [expr $qy * $qz];
    set tmp2 [expr $qx * $qw];
    set m21 [expr 2.0 * ( $tmp1 + $tmp2) * $invs];
    set m12 [expr 2.0 * ( $tmp1 - $tmp2) * $invs]; 
    
    return [list $m00 $m01 $m02 $m10 $m11 $m12 $m20 $m21 $m22]
}

proc Quaternion::ToRotationMatrix4 { quat } {
    lassign [Quaternion::ToRotationMatrix3 $quat] m00 m01 m02 m10 m11 m12 m20 m21 m22
    return [list $m00 $m01 $m02 0.0 $m10 $m11 $m12 0.0 $m20 $m21 $m22 0.0 0.0 0.0 0.0 1.0]
}

proc Quaternion::Add { q1 q2 } {
    lassign $q1 a1 b1 c1 d1
    lassign $q2 a2 b2 c2 d2
    return [list [expr {$a1 + $a2}] [expr {$b1 + $b2}] [expr {$c1 + $c2}] [expr {$d1 + $d2}]]
}

#Quaternion* q, double v[3], double output[3]
proc Quaternion::Rotate { q  v } {
    lassign $q qw qx qy qz
    lassign $v vx vy vz
    set ww [expr $qw * $qw]
    set xx [expr $qx * $qx]
    set yy [expr $qy * $qy]
    set zz [expr  $qz * $qz]
    set wx [expr  $qw * $qx]
    set wy [expr  $qw * $qy]
    set wz [expr  $qw * $qz]
    set xy [expr  $qx * $qy]
    set xz [expr  $qx * $qz]
    set yz [expr  $qy * $qz]

    # Formula from http://www.euclideanspace.com/maths/algebra/realNormedAlgebra/quaternions/transforms/index.htm
    # p2.x = w*w*p1.x + 2*y*w*p1.z - 2*z*w*p1.y + x*x*p1.x + 2*y*x*p1.y + 2*z*x*p1.z - z*z*p1.x - y*y*p1.x;
    # p2.y = 2*x*y*p1.x + y*y*p1.y + 2*z*y*p1.z + 2*w*z*p1.x - z*z*p1.y + w*w*p1.y - 2*x*w*p1.z - x*x*p1.y;
    # p2.z = 2*x*z*p1.x + 2*y*z*p1.y + z*z*p1.z - 2*w*y*p1.x - y*y*p1.z + 2*w*x*p1.y - x*x*p1.z + w*w*p1.z;

    set result_x [expr $ww*$vx + 2*$wy*$vz - 2*$wz*$vy + $xx*$vx + 2*$xy*$vy + 2*$xz*$vz - $zz*$vx - $yy*$vx]
    set result_y [expr 2*$xy*$vx + $yy*$vy + 2*$yz*$vz + 2*$wz*$vx - $zz*$vy + $ww*$vy - 2*$wx*$vz - $xx*$vy]
    set result_z [expr 2*$xz*$vx + 2*$yz*$vy + $zz*$vz - 2*$wy*$vx - $yy*$vz + 2*$wx*$vy - $xx*$vz + $ww*$vz]

    return [list $result_x $result_y $result_z]
}

# Interpolates between two quaternions.
# Quaternion q1, Quaternion q2, double t
# t [0, 1].
# 0 is equal with q1, 1 is equal with q2, 0.5 is the middle between q1 and q2.
# return Quaternion interpolated
proc Quaternion::Slerp { q1 q2 t }  {
    variable QUATERNION_EPS
    # converted from C code of https://github.com/MartinWeigel/Quaternion/tree/master
    # Based on http://www.euclideanspace.com/maths/algebra/realNormedAlgebra/quaternions/slerp/index.htm
    lassign $q1 pw px py pz
    lassign $q2 qw qx qy qz

    set cosHalfTheta [expr $pw*$qw + $px*$qx + $py*$qy + $pz*$qz]

    # if q1=q2 or qa=-q2 then theta = 0 and we can return qa
    if { [expr abs($cosHalfTheta)] >= 1.0 } {
        return $q1
    }

    set halfTheta [expr acos($cosHalfTheta)]
    set sinHalfTheta [expr sqrt(1.0-$cosHalfTheta*$cosHalfTheta)]
    # If theta = 180 degrees then result is not fully defined
    # We could rotate around any axis normal to q1 or q2
    if { [expr abs($sinHalfTheta)] < $QUATERNION_EPS } {
        set w [expr ($pw * 0.5 + $qw * 0.5)]
        set vx [expr ($px * 0.5 + $qx * 0.5)]
        set vy [expr  ($py * 0.5 + $qy * 0.5)]
        set vz [expr  ($pz * 0.5 + $qz * 0.5)]
    } else {
        # Default quaternion calculation
        set ratioA [expr sin((1.0-$t) * $halfTheta) / $sinHalfTheta]
        set ratioB [expr sin($t * $halfTheta) / $sinHalfTheta]
        set w [expr ($pw * $ratioA + $qw * $ratioB)]
        set vx [expr ($px * $ratioA + $qx * $ratioB)]
        set vy [expr ($py * $ratioA + $qy * $ratioB)]
        set vz [expr  ($pz * $ratioA + $qz * $ratioB)]
    }
    return [list $w $vx $vy $vz]
}