#this file must be small to be sourced when starting with Tcl only or with Tcl and Tk
#the invoked procedures must be self-contained, avoiding source other big files based on tclIndex

package require gid_cross_platform

#defined here to avoid to source dev_kit.tcl
proc GidUtils_IsTkDisabled { } {
    return [expr {[info commands wm] == ""}]
}

#
proc WarnWin { text {parent .} {id 0}} {
    #define initially empty, to avoid error trying to invoke in batch without window
    return
}

#### 
proc MsgcatGetLocales {} {
    # folder to locate .msg message catalog files for translation
    set dir [file join $::GIDDEFAULT resources msgs]
    # get names of file == locale. force 'en' to be in the list
    set locales {en}
    foreach filename [glob -nocomplain -directory $dir *.msg] {
        set name [string range [lindex [file split $filename] end] 0 end-4]
        set locales [linsert $locales end $name]
    }
    set locales [lsort -unique $locales]
    return $locales
}

proc MsgcatSetLocale {locale} {
    # folder to locate .msg message catalog files for translation
    set dir [file join $::GIDDEFAULT resources msgs]
    ::msgcat::mclocale $locale
    ::msgcat::mcload $dir
    set problemtype_path [GiD_Info problemtypepath]
    if { $problemtype_path != "" } {
        # load the problemtype's message catalog
        ::GIDi18nProblemType::load $problemtype_path
    }
}

proc MsgcatMcInitialize {} {
    set gid_defaults [file nativename [GiD_GetUserSettingsFilename]]
    set found -1
    if {[file exists $gid_defaults]} {
        set fileid [open $gid_defaults r]
        fconfigure $fileid -encoding "utf-8"
        set data [read $fileid]
        close $fileid
        foreach line [split $data "\n"] {
            if { [string match "Language*" $line] } {
                set found 1
                set language [string range $line 9 end]
            }
        }
    }   
    if { $found == -1} {
        set preferences [::msgcat::mcpreferences]
        set preferences [linsert $preferences 0 en] ;#until the message catalog is fully translated
        set locales [MsgcatGetLocales]
        # find firsts language available for the preferences
        set found -1
        foreach preference $preferences {
            set found [lsearch -exact $locales $preference]
            if { $found != -1 } {
                set language $preference
                break
            }
        }
        if { $found == -1 } {
            # if not found use english as default
            set language en
        }
    }    
    MsgcatSetLocale $language
}

#########
#to remove possible global :: of event names

#category: GENERAL, PROBLEMTYPE, PLUGIN
#propietary: gid, problemtype_name, plugin_name (according with category)
proc GiD_RegisterEvent { event_name procedure {category GENERAL} {propietary gid} } {
    set fail 0
    if { [lsearch -sorted -dictionary [GiD_Info events] $event_name] == -1 } {
        set fail 1
        error "Invalid event $event_name"
    }
    if { [info procs $procedure] == "" } {
        set fail 1
        error "proc $procedure doesn't exists"
    }
    if { [llength [info args $procedure]] != [llength [GiD_Info events -args $event_name]] } {
        set fail 1
        error "Wrong arguments of $procedure, must be [GiD_Info events -args $event_name]"
    }
    set categories [list GENERAL PROBLEMTYPE PLUGIN]
    if { [lsearch $categories $category] == -1 } {
        error "Wrong category $category. Must be $categories"
    }
    if { !$fail } {
        lappend ::GidPrivEvents($event_name) [list $procedure $category $propietary]
    }
    return $fail
}

proc GiD_UnRegisterEvent { event_name procedure {category GENERAL} {propietary gid} } {
    set fail 0
    if { [lsearch -sorted -dictionary [GiD_Info events] $event_name] == -1 } {
        set fail 1
        error "Invalid event $event_name"
    }
    if { [info exists ::GidPrivEvents($event_name)] } {
        set index [lsearch -index 0 $::GidPrivEvents($event_name) $procedure]
        if { $index != -1 } {
            set item [lindex $::GidPrivEvents($event_name) $index]
            lassign $item item_procedure item_category item_propietary
            if { $item_category == $category && $item_propietary == $propietary } {
                set ::GidPrivEvents($event_name) [lreplace $::GidPrivEvents($event_name) $index $index]
                if { [llength $::GidPrivEvents($event_name)] == 0 } {
                    unset ::GidPrivEvents($event_name)
                }
            } else {
                error "event $event_name proc $procedure is not registered for category $category propietary $propietary"
            }
        } else {
            set fail 1
            error "proc $procedure is not registered for event $event_name"
        }
    } else {
        set fail 1
        error "proc $procedure is not registered for event $event_name"
    }
    return $fail
}

proc GiD_GetRegisteredEventProcs { event_name {category GENERAL} {propietary gid} } {    
    set procs [list]
    if { [info exists ::GidPrivEvents($event_name)] } {
        foreach item $::GidPrivEvents($event_name) {
            lassign $item item_procedure item_category item_propietary
            if { $item_category == $category && $item_propietary == $propietary } {
                lappend procs $item_procedure
            }
        }
    }
    return $procs
}

#to be used to raise the events
proc GiD_GetRegisteredEventProcsAll { event_name } {
    set procs [list]
    if { [info exists ::GidPrivEvents($event_name)] } {
        foreach item $::GidPrivEvents($event_name) {
            lappend procs [lindex $item 0]
        }
    }
    return $procs
}

proc GiD_GetIsRegisteredEvent { event_name } {
    set is_registered 0
    if { [info exists ::GidPrivEvents($event_name)] } {
        set is_registered 1
    }
    return $is_registered
}

proc GiD_GetIsRegisteredEventProc { event_name procedure {category GENERAL} {propietary gid} } {    
    set ret_val 0
    if { [lsearch [GiD_GetRegisteredEventProcs $event_name $category $propietary] $procedure] != -1 } {
        set ret_val 1
    }
    return $ret_val
}

proc GiD_GetRegisteredEventsAndProcs { {category GENERAL} {propietary gid} } {
    set events_and_procs [list]
    foreach event_name [lsort [array names ::GidPrivEvents]] {
        set procs [GiD_GetRegisteredEventProcs $event_name $category $propietary]
        if { [llength $procs] } {
            lappend events_and_procs [list $event_name $procs]
        }
    }
    return $events_and_procs
}

#to unregister all events unloading a problemtype
#e.g. GiD_UnRegisterEvents problemtype Examples/cmas2d_customlib
proc GiD_UnRegisterEvents { {category GENERAL} {propietary gid} } {
    foreach item [GiD_GetRegisteredEventsAndProcs $category $propietary] {
        lassign $item event_name procs
        foreach procedure $procs {
            GiD_UnRegisterEvent $event_name $procedure $category $propietary
        }
    }
    return 0
}

proc RaiseEvent_Registered { event_name args } {
    set result 0  
    foreach proc [GiD_GetRegisteredEventProcsAll $event_name] {
        if { [catch { set result [$proc {*}$args] } msg options] } {
            if { ![GidUtils_IsTkDisabled] } {
                error $msg $::errorInfo
                #return -options $options $msg
            }
        }
    }
    return $result
}

# evaluate the proc only if it exists
#like GiD_RaiseEvent but evaluate a procedure that is not considered as event
#it also could evaluate procedures hidden to info procs
proc Eval_GenericProc { procedure args } {
    set result ""
    if { [info procs $procedure] != "" } {
        if { [catch { set result [$procedure {*}$args] } msg] } {
            if { ![GidUtils_IsTkDisabled] } {
                error "$procedure $args -> $msg"
            }
        }
    }
    return $result
}

#find other versions
proc GetSimilarDefaultsFile { tail_name } {
    set gid_defaults_similar ""    
    set gid_defaults [GiD_GetUserSettingsFilename]
    set base_dir [file dirname [file dirname $gid_defaults]]
    #find in all folders like 14.1.0d, 14.1.1d
    set folders [glob -nocomplain -tails -directory $base_dir -type d *]
    set folders_developer [list]
    set folders_official [list]
    foreach folder $folders {
        if { [string index $folder end] == "d" } {
            lappend folders_developer $folder
        } else {
            lappend folders_official $folder
        }
    }
    set best_folder ""
    if { [string index [GiD_Info GiDVersion] end] == "d"} {
        set folders_same_category $folders_developer
        set folders_other_category $folders_official 
    } else {
        set folders_same_category $folders_official
        set folders_other_category $folders_developer 
    }
    foreach folder $folders_same_category {
        set filename [file join $base_dir $folder $tail_name]
        if { [file exists $filename] } {            
            if { $best_folder == "" || [file mtime [file join $base_dir $best_folder $tail_name]] < [file mtime $filename] } {
                set best_folder $folder
                set gid_defaults_similar $filename
            }
        }
    }
    if { $gid_defaults_similar == "" } {
        foreach folder $folders_other_category {
            set filename [file join $base_dir $folder $tail_name]
            if { [file exists $filename] } {
                if { $best_folder == "" || [file mtime [file join $base_dir $best_folder $tail_name]] < [file mtime $filename] } {
                    #consider better the highest version number, maybe it is better the more recent by date
                    set best_folder $folder
                    set gid_defaults_similar $filename
                }
            }
        }
    }
    if { $gid_defaults_similar == "" } {
        #find old classical shared location
        if { $::tcl_platform(platform) != "windows" && $tail_name == "gid.ini" } {
            set filename [file join $base_dir .gidDefaults]
        } elseif { $tail_name == "gid_macros.tcl" } {
            set filename [file join $base_dir Macros.tcl]
        } else {
            set filename [file join $base_dir $tail_name]
        }
        if { [file exists $filename] } {
            set gid_defaults_similar $filename
        }
    }
    return $gid_defaults_similar
}

#return the alternative preferences file to be used forced by -c or -c2 command line
#trick, there is not any Tcl command to directly get alternative_filename
#       compare with -ignore_alternative_configuration_file
#       and if different assume that is because AlternativeConfigurationFile is set by -c2 or -c
#       and AlternativeConfigurationFile is [GiD_GetUserSettingsFilename] 
#       return "" if not AlternativeConfigurationFile is set
proc GetAlternativeUserSettingsFilename { } {
    set alternative_filename  ""
    if { [GiD_GetUserSettingsFilename] != [GiD_GetUserSettingsFilename -ignore_alternative_configuration_file] } {
        set alternative_filename [GiD_GetUserSettingsFilename]
    }
    return $alternative_filename
}

#tcl values
proc ReadDefaultValues { filename cloud } {
    set fail 0
    set fp [open $filename r]
    if { $fp != "" } {
        fconfigure $fp -encoding "utf-8"
        set content [read -nonewline $fp]
        close $fp   
        set fail [ReadDefaultValuesFromString $content $filename $cloud]
    }
    return $fail
}

#filename could be "" if is not from a file, only used for ::GiDTBLayout variables
proc ReadDefaultValuesFromString { data filename cloud } {
    set fail 0
    #don't forget to check Main.cc which uses GidUnknownDefaults GidKnownDefaults

    global GidPriv GidUnknownDefaults GidKnownDefaults
    global CalcAngleUnits GidProcWin  
        
    if { !$cloud } {
        global GiDTBLayout
        global MainWindowGeom DefaultSearchDirectoryP RunProcInfo

        #InitWindow variables that changed its name, if they are not transformed to the new name 
        #the open window proc could be called by both names, and a window like the 'sets of post' 
        #could be opened again and again restarting GiD or swapping pre<->post
        array set NewNameWindowGeom {
            LayersWindowGeom PreLayersWindowGeom
            PostDisplayWindowGeom PostDisplayWindowGeom
            TMacrosWindowGeom PrePostMacrosWindowGeom
        }

        if { ![info exists ::GiDTBLayout] } {
            array set ::GiDTBLayout {
                INSIDE,CURRENT ""
                INSIDE,INI     ""
                INSIDE,PRE     ""
                INSIDE,POST    ""
            }
        }
        lappend ::GiDTBLayout(inifile) $filename
        set ::GiDTBLayout($filename,INSIDE,INI) [list]

        if { ![info exists GidProcWin(remoteserver)] } {
            set GidProcWin(remoteserver) ""
        }
        if { ![info exists GidProcWin(remotegroup)] } {
            set GidProcWin(remotegroup) ""
        }
        if { ![info exists GidProcWin(remoteuser)] } {
            set GidProcWin(remoteuser) ""
        }

    }
    
    
  
    foreach line [split $data \n] {
        set aa [string trim $line]
        if { $aa == "" } continue
        if { [catch { set varname [lindex $aa 0] } ] } {
            continue
        }
        set unknow 0
        switch -glob -- $varname {
            CalcAngleUnits {
                set CalcAngleUnits [lindex $aa 1]
            }
            DefaultSearchDirectory {
                if { !$cloud } {
                    # old compatibility
                    # set DefaultSearchDirectory [lrange $aa 1 end]
                    set ::DefaultDirectories(file) [lrange $aa 1 end]
                }
            }
            DefaultSearchDirectoryP {
                if { !$cloud } {
                    set DefaultSearchDirectoryP [lrange $aa 1 end]
                }
            }
            DefaultDirectories {
                if { !$cloud } {
                    array set ::DefaultDirectories [ lrange $aa 1 end]
                }
            }
            DuplicateEntities {
                set GidPriv(DuplicateEntities) [lindex $aa 1]
            }
            MainWindowGeom {
                #this is an unfortunate exception to the other windows, is not in GidPriv array and doesn't has 'PrePost' prefix
                if { !$cloud } {
                    set MainWindowGeom [lrange $aa 1 end]
                }
            }
            NormalFont {
                if { !$cloud } {
                    if { !$::GidPriv(OffScreen) && ![GidUtils_IsTkDisabled] } {
                        lassign $aa fn family size weight slant
                        font configure $fn -family $family -size $size -weight $weight -slant $slant
                        font configure BoldFont -family $family -size $size -weight bold -slant $slant
                        set small_size [expr $size-2]
                        # this is ok for screen with 96dpi but not for 75dpi screens
                        catch {
                            set dpi [winfo fpixels . 1i]
                            if { ( $dpi < 95.0) && ( $dpi > 85.0) } {
                                set small_size [expr $size-1]
                            } elseif { ( $dpi <= 85.0) } {
                                set small_size $size
                            }
                        }
                        font configure SmallFont -family $family \
                            -size [ expr int( 0.5 + $small_size)] \
                            -weight $weight -slant $slant
                    }
                }
            } 
            RunProcInfo {
                if { !$cloud } {
                    set RunProcInfo [lrange $aa 1 end]
                    # 3 seconds to permmit the GiD window to appear
                    after 3000 ActualizeProcInfoBegin
                }
            }
            remoteserver {
                if { !$cloud } {
                    set GidProcWin(remoteserver) [lindex $aa 1]
                }
            }
            remotegroup {
                if { !$cloud } {
                    set GidProcWin(remotegroup) [lindex $aa 1]
                }
            }
            remoteuser {
                if { !$cloud } {
                   set GidProcWin(remoteuser) [lindex $aa 1]
                }
            }
            remoteauto {
                set GidProcWin(remoteauto) [lindex $aa 1]
            }
            remotetime {
                set GidProcWin(remotetime) [lindex $aa 1]
            }
            remoteport  {
                set GidProcWin(remoteport) [lindex $aa 1]
            }
            remotetimeout {
                set GidProcWin(remotetimeout) [lindex $aa 1]
            }
            remoteresultsdownload {
                set GidProcWin(remoteresultsdownload) [lindex $aa 1]
            }
            remoteresultsdelete {
                set GidProcWin(remoteresultsdelete) [lindex $aa 1]
            }     
            WarnWinTextFont {
                if { !$cloud } {
                    if { !$::GidPriv(OffScreen) && ![GidUtils_IsTkDisabled]} {
                        lassign $aa fn family size weight slant
                        if { [ lsearch [ font names] WarnWinTextFont] == -1} {
                            font create WarnWinTextFont
                        }
                        font configure $fn -family $family -size $size -weight $weight -slant $slant
                    }
                }
            }
            TkDefaultFont -
            FixedFont -
            BigFont -
            SmallFont {
                if { !$cloud } {
                    if { !$::GidPriv(OffScreen) && ![GidUtils_IsTkDisabled]} {
                        lassign $aa fn family size weight slant
                        font configure $fn -family $family -size $size -weight $weight -slant $slant
                    }
                }
            }
            WarnWinTextFontUsed {
                if { !$cloud } {
                    set ::GidPriv(WarnWinTextFontUsed) [lindex $aa 1]
                }
            }
            LayersBackOpposite {
                set GidPriv(LayersBackOpposite) [lindex $aa 1]
            }            
            GiDTkFBoxPreview {
                set GidPriv(GiDTkFBoxPreview) [lindex $aa 1]
            }
            LayersAlphabetic {
                set GidPriv(LayersAlphabetic) [lindex $aa 1]
            }
            OpenGL(Configured) {
                if { !$cloud } {
                    GiD_Set OpenGL(Configured) [lindex $aa 1]
                }
            }
            Theme_configured {
                if { !$cloud } {
                    GiD_Set Theme_configured [lindex $aa 1]
                }
            }
            LayerLayout {
                set GidPriv(LayerLayout) [lindex $aa 1]
            }
            PostDisplayLocation {
                set GidPriv(PostDisplayLocation) [lindex $aa 1]
            }
            ToglWinMode {
                set GidPriv(ToglWinMode) [lindex $aa 1]
            }
            ProgressInMeshingMore {
                set ::GidPriv(ProgressInMeshing,More) [lindex $aa 1]
                if { ![string is boolean $::GidPriv(ProgressInMeshing,More)] } {
                    # to avoid problems with wrong gid.ini file that has 
                    # ProgressInMeshingMore 1SoftwareOpenGLSoftwareOpenGLSoftwareOpenGLSoftwar
                    # it happened in some Linux, and e.g. gid 17.0.2
                    # maybe written multiple threads of process in same gid.ini ??
                    set ::GidPriv(ProgressInMeshing,More) 1
                }
            }
            MeshCartesianSameSizes {
                set ::MeshCartesianDataPriv(samesizes) [lindex $aa 1]
            }
            BitmapsGeom {
                #some window variables renamed, the most important toolbars, not external toplevels (to import old values)
                #Window variable renamed
                set varname PrePostBitmapsWindowGeom
                if { [lindex $aa 1] != "NONE" } {
                    set GidPriv($varname) [list OPEN [lindex $aa 1] 1 CreateBitmaps]
                } else {
                    set GidPriv($varname) [list NONE {} 1 CreateBitmaps]
                }
            }
            BottomEntryGeom {
                #Window variable renamed
                set varname PrePostEntryWindowGeom
                if { [lindex $aa 1] != "NONE" } {
                    set GidPriv($varname) [list OPEN [lindex $aa 1] 1 BottomEntryFrame]
                } else {
                    set GidPriv($varname) [list NONE {} 1 BottomEntryFrame]
                }
            }
            ListEntitiesGeom {
                if { !$cloud } {
                    #Window variable renamed
                    set varname PreListEntitiesWindowGeom
                    if { [lindex $aa 1] != "NONE" } {
                        set GidPriv($varname) [list OPEN [lindex $aa 1]]
                    } else {
                        set GidPriv($varname) [list NONE +0+176]
                    }
                }
            }
            RightButtonsGeom {
                if { !$cloud } {
                    #Window variable renamed
                    set varname PrePostRightButWindowGeom
                    if { [lindex $aa 1] != "NONE" } {
                        set GidPriv($varname) [list OPEN [lindex $aa 1] 1 RightButtons]
                    } else {
                        set GidPriv($varname) [list NONE {} 1 RightButtons]
                    }
                }
            }
            *WindowGeom {
                if { !$cloud } {
                    if { [info exists NewNameWindowGeom($varname)] } {
                        set varname $NewNameWindowGeom($varname)
                        if { [info exists GidPriv($varname)] } {
                            #the same window state is specified twice in gid.ini
                            #once with the old name, and once with the new name
                            #ignore this that is using the old name (inherit or shared gid.ini)
                            continue
                        }
                    }
                    if { [string range $varname 0 2] == "Pre" ||  [string range $varname 0 3] == "Post" } {
                        #set name [lindex $aa 0]
                        set GidPriv($varname) [lrange $aa 1 end]
                        if { [lindex $GidPriv($varname) 2] != "0" && [lindex $GidPriv($varname) 2] != "1" } {
                            set GidPriv($varname) [linsert $GidPriv($varname) 2 1]
                        }
                    } else {
                        #if gid.ini is shared between versions (or allow variables of problemtypes) GidUnknownDefaults variables must be saved again 
                        #if it is not shared (like nowadays, but maybe is changed again...) it is better to avoid the old garbage
                        set unknow 1
                    }
                }
            }
            *Geom {
                if { !$cloud } {
                    if { [info exists NewNameWindowGeom($varname)] } {
                        set varname $NewNameWindowGeom($varname)
                        if { [info exists GidPriv($varname)] } {
                            #the same window state is specified twice in gid.ini
                            #once with the old name, and once with the new name
                            #ignore this that is using the old name (inherit or shared gid.ini)
                            continue
                        }
                        set GidPriv($varname) [lrange $aa 1 end]
                    }
                }
            }
            #do this here because when all variables are read is too late and the splasw use the devault value and was showed               
            SplashWindow {
                GiD_Set SplashWindow [lindex $aa 1]
            }
            SoftwareOpenGL {
                if { !$cloud } {
                    GiD_Set SoftwareOpenGL [lindex $aa 1]
                }
            }
            default {
                set unknow 1
            }
        }
        if { $unknow } {
            #don't forget to check Main.cc and Post/utils_post.cc which uses GidUnknownDefaults GidKnownDefaults
            #to preserve unknown variables (probably used by a new version), because the file is shared by all versions
            set GidUnknownDefaults($varname) [lrange $aa 1 end]
        } else {
            #don't forget to check Main.cc and Post/utils_post.cc which uses GidUnknownDefaults GidKnownDefaults
            set GidKnownDefaults($varname) [lrange $aa 1 end]
        }
        if {[regexp ^INSIDE [lindex $aa 1]]} {
            if { !$cloud } {
                if { [lsearch $::GiDTBLayout($filename,INSIDE,INI) $varname] == -1 } {
                    lappend ::GiDTBLayout($filename,INSIDE,INI) $varname
                }
            }
        }
    }
    if { !$cloud } {
        if {[llength $::GiDTBLayout($filename,INSIDE,INI)]} {
            set ::GiDTBLayout(INSIDE,INI) $::GiDTBLayout($filename,INSIDE,INI)
        }
    }
    return $fail
}

proc CheckFileOwner { file} {
    if { $::tcl_platform(platform) == "windows" } { 
        return
    }
    set owner_file [ file attributes $file -owner]
    set owner_dir [ file attributes [ file dirname $file] -owner]
    set group_file [ file attributes $file -group]
    set group_dir [ file attributes [ file dirname $file] -group]
    if { "$owner_file" != "$owner_dir"} {
        # let's change owner of the file to owner of the directory where the file is
        # to avoid problems such that the owner of the preferences file is root but we're in a user directory
        set err [ catch {
            file attributes $file -owner $owner_dir -group $group_dir
        } err_txt ]
        if { $err} {
            WarnWinDirect "Sorry but could not change owner of \nfile '$file' to '$owner_dir': \n$err_txt"
        }
    }
}

proc WriteTCLDefaultsInFile { filename cloud } {
    global GidPriv
    # avoid corruptions, write on a different file
    # rename it afterwards
    #not to do backup here because first the C function has renamed gid.ini to .bak and has written half (C content) of new gid.ini
    set data [WriteTCLDefaultsInString $cloud]
    #mode "a" to append, because part has been written before from C++
    set fail [GidUtils::WriteFile $filename $data "a" "utf-8"]
    CheckFileOwner $filename
    #look for it in tk.tcl
    if { [info exists ::GidPriv(EventFile)] } {
        close $::GidPriv(EventFile)
    }
    #if all was ok writing gid.ini delete the gid.ini.bak file
    file delete ${filename}.bak
    return 0
}

proc WriteTCLDefaultsInString { cloud } {
    set res ""
    #don't forget to check Main.cc and Post/utils_post.cc which uses GidUnknownDefaults GidKnownDefaults
    
    global GidPriv GiDPrivToolBar GidUnknownDefaults
    global MainWindowGeom DefaultSearchDirectoryP
    global CalcAngleUnits RunProcInfo GidProcWin
    
    if { $::GidPriv(OffScreen) || [GidUtils_IsTkDisabled] } {
        return $res
    }
    
    if { ![winfo exists .gid] } {
        set MainWindowGeom "800x600+100+100"
    } else {
        if { $::tcl_platform(platform) eq "windows" && [wm state .gid] eq "zoomed" } {
            set zoomed 1
        } elseif { ![catch { wm attributes .gid -zoomed } ret] && $ret } {
            set zoomed 1
        } else {
            set zoomed 0
        }
        if { $zoomed } {
            if { ![info exists MainWindowGeom] } {
                set MainWindowGeom "800x600+100+100"
            }
            set geom $MainWindowGeom
            if { [lindex $geom 0] ne "ZOOMED" && [regexp {([^x]*)x([^+-]*)([+-]*[0-9]*)([+-]*[0-9]*)} $geom] } {
                set MainWindowGeom "ZOOMED $geom"
            } elseif { [regexp {([^x]*)x([^+-]*)([+-]*[0-9]*)([+-]*[0-9]*)} [lindex $geom 1]] } {
                set MainWindowGeom $geom
            } else {
                set MainWindowGeom "ZOOMED 800x600+100+100"
            }
        } else {
            set MainWindowGeom [wm geometry .gid]
            regexp {([^x]*)x([^+-]*)([+-]*[0-9]*)([+-]*[0-9]*)} $MainWindowGeom trash w h x y
            if { [.gid cget -menu] != "" } {
                if { $::tcl_platform(platform) eq "windows" } {
                    set menuheight [ GiD_Info menuheight]
                } else {
                    set menuheight 0
                }
                incr h $menuheight
            }
            if { $::tcl_platform(platform) eq "windows" } {
                set windowcaptionheight [GiD_Info windowcaptionheight]
            } else {
                set windowcaptionheight 0
            }
            if { $x <= 0 } { set x +-4 }
            if { $y <= 0 } { set y +-4 }
            if { $w >= [ winfo screenwidth .gid] && [expr $h+$windowcaptionheight] >= [ winfo screenheight .gid] } {
                set w [ winfo screenwidth .gid]
                set h [ winfo screenheight .gid]
                set x +-4
                set y +-4
            }
            set MainWindowGeom ${w}x${h}${x}${y}
        }
    }
    
    
    if { ![info exists CalcAngleUnits] } {
        set CalcAngleUnits "rad"
    }
    
    if { !$cloud } {
        set varslist { MainWindowGeom $::DefaultDirectories(file) DefaultSearchDirectoryP}
        foreach i $varslist {
            if { [info exists $i] } {
                append res "$i [set $i]\n"
            }
        }
        set x [lsort -stride 2 -dictionary [array get ::DefaultDirectories]]
        if { [llength $x] } {
            append res "DefaultDirectories $x\n"
        }        
    }
    if { [info exists CalcAngleUnits] } {
        append res "CalcAngleUnits $CalcAngleUnits\n"
    }
    
    if { [info exists GidPriv(LayersBackOpposite)] } {
        append res "LayersBackOpposite $GidPriv(LayersBackOpposite)\n"
    }
    if { [info exists GidPriv(DuplicateEntities)] } {
        append res "DuplicateEntities $GidPriv(DuplicateEntities)\n"
    }
    if { [info exists GidPriv(GiDTkFBoxPreview)] } {
        append res "GiDTkFBoxPreview $GidPriv(GiDTkFBoxPreview)\n"
    }
    if { [info exists GidPriv(LayersAlphabetic)] } {
        append res "LayersAlphabetic $GidPriv(LayersAlphabetic)\n"
    }    
    
    if { !$cloud } {
        foreach i "NormalFont FixedFont BigFont SmallFont TkDefaultFont WarnWinTextFont" {
            if { [lsearch [font names] $i] != -1} {
                set family [font configure $i -family]
                set size [font configure $i -size]
                if { [info exists ::FontScaleFactor]} {
                    # save the size of fonts with scale factor 1
                    set org_size [expr int(0.5+double($size)/double($::FontScaleFactor))]
                    set size $org_size
                }
                set weight [font configure $i -weight]
                set slant [font configure $i -slant]
                append res "[list $i $family $size $weight $slant]\n"
            }
        }
        if { [info exists ::GidPriv(WarnWinTextFontUsed)]} {
            append res "WarnWinTextFontUsed $::GidPriv(WarnWinTextFontUsed)\n"
        }
    
        catch {
            # write toolbars in order
            set visited [list]
            foreach f {top left right bottom} {
                foreach tb $::GiDTBLayout(.gid.$f) {
                    lappend visited [set name $::GiDTBLayout(tb,$tb)]
                    append res "$name $GidPriv($name)\n"
                    # why not unset GidPriv($name)?
                }
            }
            foreach {name val} [array get GidPriv *WindowGeom] {
                if {[lsearch $visited $name] != -1} continue
                regsub {Geom$} $name {} window_name
                if { [info exists GidPriv($window_name)] } {
                    set w $GidPriv($window_name)
                    if { [winfo exists $w] && [winfo class $w] == "Toplevel" } {
                        set InitComm $GidPriv($window_name,InitComm)
                        set win_state OPEN
                        # if onlygeometry == 1 then we want to save only geometry of window
                        # to avoid opening windows when gid starts, which need some model information, etc.
                        if { [info exists ::GidPriv($window_name,onlygeometry)] && ( $::GidPriv($window_name,onlygeometry))} {
                            set win_state NONE
                        }
                        WriteYourGeomToVar $w $win_state $name $InitComm
                        set val $GidPriv($name)
                    }
                } elseif { [info exists GiDPrivToolBar($name)] } {
                    set w $GiDPrivToolBar($name)
                    set type [lindex $GidPriv($name) 0]
                    if { [regexp {(INSIDE[a-zA-Z]+)[0-9]*} $type {} type] } {
                        set vv $GidPriv($name)
                        array set  grid_info [grid info $w]
                        if {$grid_info(-sticky) eq "ns" ||
                            $grid_info(-sticky) eq "sn" } {
                            set type $type$grid_info(-column)
                        } else {
                            set type $type$grid_info(-row)
                        }
                        set GidPriv($name) "$type [lrange $vv 1 end]"
                        set val $GidPriv($name)
                    }
                }
                append res "$name $val\n"
            }
        } msg
    
        if { [info exists ::RunProcInfo] } {
            if { $::RunProcInfo != "" } {
                append res "RunProcInfo $::RunProcInfo\n"
            }
        }

        foreach i {remoteserver remotegroup remoteuser} {
            if { [info exists ::GidProcWin($i)] } {
                append res "$i $::GidProcWin($i)\n"
            }
        }

    }

    foreach i {remoteauto remotetime remoteport remotetimeout remoteresultsdownload remoteresultsdelete} {
        if { [info exists ::GidProcWin($i)] } {
            append res "$i $::GidProcWin($i)\n"
        }
    }
    
    if {[info exists GidPriv(LayerLayout)]} {
        append res "LayerLayout $::GidPriv(LayerLayout)\n"
    }
    
    if {[info exists GidPriv(PostDisplayLocation)]} {
        append res "PostDisplayLocation $::GidPriv(PostDisplayLocation)\n"
    }
    
    if {[info exists ::GidPriv(ToglWinMode)]} {
        append res "ToglWinMode $GidPriv(ToglWinMode)\n"
    }
    
    if { [info exists ::GidPriv(ProgressInMeshing,More)]} {
        append res "ProgressInMeshingMore $::GidPriv(ProgressInMeshing,More)\n"
    }   
    
    if { [info exists ::MeshCartesianDataPriv(samesizes)]} {
        append res "MeshCartesianSameSizes $::MeshCartesianDataPriv(samesizes)\n"
    }
    
    foreach category {Layers Groups Sets} {
        if { $category == "Layers" } {
            set column_items {name touse color visibility state transparency back}
        } elseif { $category == "Groups" } {
            set column_items {name color visibility transparency}
        } elseif { $category == "Sets" } {
            set column_items {color name visibility style transparency interior edges back elements}       
        } else {
            set column_items {}
        }
        foreach item $column_items {
            set varname ColumnVisible($category,$item)
            if { $cloud || ([GiD_Set $varname] != [GiD_Set -default $varname]) } {
                append res "$varname [GiD_Set $varname]\n"
            }
        }
    }
    
    #to preserve unknown variables (probably used by a new version), because the file is shared by all versions
    #don't forget to check Main.cc and Post/utils_post.cc which uses GidUnknownDefaults GidKnownDefaults
    foreach varname [lsort [array names GidUnknownDefaults]] {
        append res "$varname $GidUnknownDefaults($varname)\n"
    }
    
    if { [string index $res end] == "\n" } {
        #remove last endline
        set res [string range $res 0 end-1]
    }
    return $res
}

#save both parts, kernel and tcl, to a file named <gid_defaults>
proc SavePreferences { gid_defaults cloud raise_events } {
    writedefaults $gid_defaults $cloud; #kernel values
    WriteTCLDefaultsInFile $gid_defaults $cloud; #tcl values
    if { $raise_events } {
        GiD_RaiseEvent GiD_Event_AfterSavePreferences $gid_defaults
    }
}

#read both parts, kernel and tcl, from a file named <gid_defaults>
proc ReadPreferences { gid_defaults cloud raise_events } {
    ReadDefaultValues $gid_defaults $cloud ; #tcl values
    readdefaults $gid_defaults $cloud; #kernel values
    if { $raise_events } {
        GiD_RaiseEvent GiD_Event_AfterReadPreferences $gid_defaults
    }
}

proc SavePreferencesToData { cloud } {
    #BY NOW IT DOESN'T EXISTS A writedefaults -to_string option, write a temporary file to allow use SavePreferences!!                
    #set data [writedefaults -to_string $cloud] ; #kernel values
    #append data [WriteTCLDefaultsInString $cloud] ; #tcl values
    set tmp_file_preferences [gid_cross_platform::get_unused_tmp_filename]
    set raise_events 0
    SavePreferences $tmp_file_preferences $cloud $raise_events
    set data [GidUtils::ReadFile $tmp_file_preferences "utf-8" 0]
    file delete $tmp_file_preferences
    return $data
}

proc ReadPreferencesFromData { data cloud } {
    set fail 0
    #BY NOW IT DOESN'T EXISTS A readdefaults -from_string option, write a temporary file to disk to allow use ReadPreferences!!                
    #ReadDefaultValuesFromString $data "" $cloud; #tcl values
    #readdefaults -from_string $data $cloud; #kernel values
    set tmp_file_preferences [gid_cross_platform::get_unused_tmp_filename]
    set fail [GidUtils::WriteFile $tmp_file_preferences $data "w" "utf-8"]   
    set raise_events 0
    ReadPreferences $tmp_file_preferences $cloud $raise_events
    file delete $tmp_file_preferences
    return $fail
}
