# gid raised a lot of emergent dialogs (from process) to ask questions to users
# this is a bad design (specially for a server-client application)
# it is much better to collect all data (with GUI client dialogs) and the do all the work (server without user dialogs)
#
# this file contain procedures to port the old code avoiding these 'server dialogs' (forecast the question and collect answers before process)

set ::show_gid_mesh_remote 0

set ::avoid_open_window_mesh_from_process 1

#auxiliary proc to be provided as callback to MessageBoxOptionsButtonsModeless
proc DoNothingCallback { args } {}

#image: error, info, question or warning
proc MessageBoxOk { title question image {wraplength 0} {parent .gid}} {
    # MessageBoxOptionsButtons { title question options labels image option_do_it_for_all_text {wraplength 0} {parent ""} {cancel_value ""}}
    MessageBoxOptionsButtons $title $question [list ok] [list [_ "Ok"]] $image "" $wraplength $parent ""
    return 0
}

# with prefix GID_ to replace tk_messageBox with same exact syntax but using themed widgets
proc GID_tk_messageBox { args } {
    set result ""
    lassign {ok "" "" "" "" "" 0 "" ""} type title question options labels image wraplength parent cancel_value
    array set flags $args
    foreach flag {-type -parent -message -icon -title} varname {type parent question image title} {
        if { [info exists flags($flag)] } {
            set $varname $flags($flag)
        }
    }
    switch $type {
        abortretryignore {
            set options [list abort retry ignore]
            set labels  [list [_ "Abort"] [_ "Retry"] [_ "Ignore"]]
            set cancel_value abort
        }
        ok {
            set options [list ok]
            set labels  [list [_ "Ok"]]
        }
        okcancel {
            set options [list ok cancel]
            set labels  [list [_ "Ok"] [_ "Cancel"]]
        }
        retrycancel {
            set options [list retry cancel]
            set labels  [list [_ "Retry"] [_ "Cancel"]]
        }
        yesno {
            set options [list yes no]
            set labels  [list [_ "Yes"] [_ "No"]]
        }
        yesnocancel {
            set options [list yes no cancel]
            set labels  [list [_ "Yes"] [_ "No"] [_ "Cancel"]]
        }
    }
    set result [MessageBoxOptionsButtons $title $question $options $labels $image "" $wraplength $parent $cancel_value]
    return $result
}

proc MessageBoxOkModeless { title question image {button_callback DoNothingCallback} {cancel_callback DoNothingCallback} {wraplength 0}} {
    MessageBoxOptionsButtonsModeless $title $question [list ok] [list [_ "Ok"]] $image $button_callback $cancel_callback $wraplength
    return 0
}

#button_callback is a proc, it will be called when selecting a button, adding as extra argument the option keyword of the button. Can use DoNothingCallback to just do nothing
#cancel_callback is a proc, it will be called if the window is closed. Can use DoNothingCallback to just do nothing
proc MessageBoxOptionsButtonsModeless { title question options labels image {button_callback GiD_Process} {cancel_callback {GiD_Process escape}} {wraplength 0}} {
    set w .gid.messagebox_options_buttons
    InitWindow2 $w -title $title -geometryvariable PrePostMessageBoxCommonWindowGeom -onlyposition -ontop
    #toplevel $w
    #wm title $w $title
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0
    wm protocol $w WM_DELETE_WINDOW "destroy $w ; $cancel_callback"
    bind $w <Escape> "destroy $w ; $cancel_callback"
    ttk::frame $w.frm_body
    ttk::label $w.frm_body.image -image [gid_themes::GetImage [ReplaceImageName $image] large_icons]
    ttk::label $w.frm_body.lbl -text $question -justify left -font BigFont -wraplength $wraplength
    ttk::frame $w.frm_buttons -style BottomFrame.TFrame
    set underline_list {}
    set i 0
    foreach option $options label $labels {
        set underline_index [FindUnderChar label underline_list]
        set underline_character [string index $label $underline_index]
        ttk::button $w.frm_buttons.b$i -text $label -command "destroy $w ; $button_callback [list $option]" -style BottomFrame.TButton
        bind $w.frm_buttons.b$i <Return> [list $w.frm_buttons.b$i invoke]
        if { [string is ascii -strict $underline_character] } {
            #to avoid problems for example with Korean characters
            $w.frm_buttons.b$i configure -underline $underline_index
            bind $w <KeyPress-[string tolower $underline_character]> [list $w.frm_buttons.b$i invoke]
            bind $w <KeyPress-[string toupper $underline_character]> [list $w.frm_buttons.b$i invoke]
        }
        grid $w.frm_buttons.b$i -row 0 -column $i -sticky ew -padx 2 -pady 2
        grid columnconfigure $w.frm_buttons $i -weight 1
        incr i
    }
    if { [llength $options] == 1 } {
        grid configure $w.frm_buttons.b0 -sticky ""
    }
    grid $w.frm_body.image $w.frm_body.lbl -sticky w -padx 5 -pady 5
    #grid configure $w.frm_body.lbl -sticky ew
    grid $w.frm_body -sticky nsew
    grid columnconfigure $w.frm_body 1 -weight 1
    grid rowconfigure $w.frm_body 0 -weight 1
    grid $w.frm_buttons -sticky ew -padx 2 -pady 2
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1
    focus $w.frm_buttons.b0
    return 0
}

#options: is the list of keywords, one of them is the return value
#labels: are the list of strings associated to options (could be translated)
#option_do_it_for_all_text, "" to not show checkbow, otherwise show this text and return a list of two values {result do_it_for_all}
#wraplength to avoid too long text lines. e.g. wraplength 150m for 150 mm (without m mean pixels)
#parent is the parent widget (.gid by default) to avoid window hidden under main window
#cancel_value allow to set value to returned if the user close the window with the upper-rigt cross or press <Escape>, by defult will try to find a "Cancel" or "No" button and invoke it.
proc MessageBoxOptionsButtons { title question options labels image option_do_it_for_all_text {wraplength 0} {parent ""} {cancel_value ""}} {
    if { [llength $options] != [llength $labels] } {
        error "options and labels must match ('options=$options' labels=$labels')"
    }
    if { $parent == ""} {
        set parent .gid
    }
    if { $parent == "."} {
        set w .__messagebox_options_buttons
    } else {
        set w $parent.__messagebox_options_buttons
    }
    InitWindow2 $w -title $title -geometryvariable PrePostMessageBoxCommonWindowGeom -onlyposition -ontop
    if { ![winfo exists $w] } {
        # windows disabled || UseMoreWindows == 0
        return
    }
    set close_window_icon_command [list DoNothingCallback]    
    if { $cancel_value != "" } {
        set close_window_icon_command [list set ::GidPriv(messagebox,selected_option) $cancel_value]
    } else {
        #when click the close button do no always invoke the first button, but return a selection is compulsory in this widget
        set cancel_index -1
        if { [llength $labels] == 1 } {
            #if there is only one option then invoke it
            set cancel_index 0
        } else {
            set cancel_index [lsearch $labels [_ "Cancel"]]
            if { $cancel_index == -1 } {
                set cancel_index [lsearch $labels [_ "cancel"]]
                if { $cancel_index == -1 } {
                    set cancel_index [lsearch $labels [_ "No"]]
                    if { $cancel_index == -1 } {
                        set cancel_index [lsearch $labels [_ "no"]]
                    }
                }
            }
        }
        if { $cancel_index != -1 } {
            #if there is some button labeled cancel then invoke its action
            set close_window_icon_command [list set ::GidPriv(messagebox,selected_option) [lindex $options $cancel_index]]
        }
    }
    
    wm protocol $w WM_DELETE_WINDOW $close_window_icon_command
    bind $w <Escape> $close_window_icon_command

    ttk::frame $w.frm_body
    ttk::label $w.frm_body.image -image [gid_themes::GetImage [ReplaceImageName $image] large_icons]
    ttk::label $w.frm_body.lbl -text $question -justify left -font BigFont -wraplength $wraplength
    ttk::frame $w.frm_buttons -style BottomFrame.TFrame
    set underline_list {}
    set i 0
    foreach option $options label $labels {
        set underline_index [FindUnderChar label underline_list]
        set underline_character [string index $label $underline_index]
        ttk::button $w.frm_buttons.b$i -text $label -command [list set ::GidPriv(messagebox,selected_option) $option] -style BottomFrame.TButton
        bind $w.frm_buttons.b$i <Return> [list $w.frm_buttons.b$i invoke]
        if { [string is ascii -strict $underline_character] } {
            #to avoid problems for example with Korean characters
            $w.frm_buttons.b$i configure -underline $underline_index
            bind $w <KeyPress-[string tolower $underline_character]> [list $w.frm_buttons.b$i invoke]
            bind $w <KeyPress-[string toupper $underline_character]> [list $w.frm_buttons.b$i invoke]
        }
        grid $w.frm_buttons.b$i -row 0 -column $i -sticky ew -padx 5 -pady 5
        grid columnconfigure $w.frm_buttons $i -weight 0
        incr i
    }
    if { [llength $options] == 1 } {
        grid configure $w.frm_buttons.b0 -sticky ""
    }
    grid anchor $w.frm_buttons center

    grid $w.frm_body.image $w.frm_body.lbl -sticky w -padx 5 -pady 5
    #grid configure $w.frm_body.lbl -sticky ew
    grid $w.frm_body -sticky nsew
    grid columnconfigure $w.frm_body 1 -weight 1
    grid rowconfigure $w.frm_body 0 -weight 1
    if { $option_do_it_for_all_text != "" } {
        if { ! [info exists ::GidPriv(messagebox,do_it_for_all)] } {
            set ::GidPriv(messagebox,do_it_for_all) 0
        }
        ttk::frame $w.frm_do_it_for_all -style BottomFrame.TFrame
        ttk::checkbutton $w.frm_do_it_for_all.chk -text $option_do_it_for_all_text -variable ::GidPriv(messagebox,do_it_for_all) -style BottomFrame.TCheckbutton
        grid $w.frm_do_it_for_all.chk -sticky w
        grid $w.frm_do_it_for_all -sticky ew -columnspan 2
    }
    grid $w.frm_buttons -sticky ew -padx 2 -pady 2
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1


    set is_iconic [GidUtils::IsStateIconic]
    set focus $w.frm_buttons.b0
    if { $is_iconic } {
        focus $focus
        #grab $w ;#if grab when iconic then is not possible to deiconify GiD in Windows !!!
    } else {
        ::tk::SetFocusGrab $w $focus
    }
    vwait ::GidPriv(messagebox,selected_option)
    if { $is_iconic } {
        destroy $w
    } else {
        ::tk::RestoreFocusGrab $w $focus
    }
    set result $::GidPriv(messagebox,selected_option)
    if { $option_do_it_for_all_text != "" } {
        set result [list $result $::GidPriv(messagebox,do_it_for_all)]
    }
    return $result
}

#type: any, word, real, real+, int, int+, text, textro, string
#default_answer: initial value
#button_assign: 0 (assign/close) or 1 (ok/cancel)
#fill_extra_frame_proc is the name of a proc that expects a frame as argument and will fill it with the widgets he want
proc MessageBoxEntry { title question type default_answer button_assign image {fill_extra_frame_proc ""} {parent .gid}} {
    if { $button_assign } {
        set button_labels [list [_ "Assign"] [_ "Close"]]
    } else {
        set button_labels [list [_ "Ok"] [_ "Cancel"]]
    }
    set w ${parent}.__msg_box_entry
    if { $parent == "."} {
        set w .__msg_box_entry
    }
    set answer [tk_dialogEntryRAM $w $title $question $image $type $default_answer $button_labels $fill_extra_frame_proc]
    if { $answer == "--CANCEL--" } {
        set answer ""
    }
    return $answer
}

#options:
#state: normal, readonly
proc MessageBoxCombobox { title question options state image {parent .gid}} {
    if { $state == "normal" } {
        set allow_new 1
    } elseif { $state == "readonly" } {
        set allow_new 0
    } else {
        set allow_new 1
    }
    set w ${parent}.__msg_box_combobox
    if { $parent == "."} {
        set w .__msg_box_combobox
    }
    set answer [tk_dialogComboRAM $w $title $question $image $allow_new $options]
    if { $answer == "--CANCEL--" } {
        set answer ""
    }
    return $answer
}

proc MessageBoxGetNormal { title  {parent .gid}} {
    set w ${parent}.__msg_box_get_normal
    if { $parent == "."} {
        set w .__msg_box_get_normal
    }
    set normal [split [AskForNormal $w $title 0] ,]
    return [MathUtils::VectorNormalized $normal]
}

#cathegory: file directory or project (project is for a GiD model), other values are considered as file
#mode: read write
proc MessageBoxGetFilename { cathegory {mode read} {title ""} {initial_filename ""} {file_types ""} {default_extension ""} {multiple 0} {extra_options ""} } {
    return [Browser-ramR $cathegory $mode .gid $title $initial_filename $file_types $default_extension $multiple $extra_options]
}

proc GetOrDefault { variable_name default_value } {
    if { [info exists $variable_name] } {
        set result [set $variable_name]
    } else {
        set result $default_value
    }
    return $result
}

########
proc DoUtilitiesRepair { } {
    set question [_ "Sure, repair geometry?"]
    set answer [MessageBoxOptionsButtons [_ "Repair"] $question [list Yes Cancel] [list [_ "Ok"] [_ "Cancel"]] question ""]
    if { $answer == "Cancel" } {
        return 1
    }
    GiD_Process Mescape Utilities Repair Yes escape
}

#entity:Points Lines Surfaces Volumes Nodes Elements
proc DoList { type } {
    set ids [GidUtils::PickEntities $type multiple [_ "Enter %s to assign to list" $type]]
    if { $ids != "" } {
        ListEntities [list $type {*}$ids]
    }
}

proc DoListMass { type } {
    set ids [GidUtils::PickEntities $type multiple [_ "Enter %s to assign to list" $type]]
    if { $ids != "" } {
        ShowMassProperties [list $type {*}$ids]
    }
}

proc DoGeometryDelete { type } {
    set cmd [list Mescape Geometry Delete $type]
    if { [GiD_Set DeleteAlsoLower] } {
        lappend cmd LowerEntities
    } else {
        lappend cmd NoLowerEntities
    }
    GiD_Process {*}$cmd
}

proc DoMeshDelete { type } {
    if { $type == "Nodes" } {
        set cmd [list Mescape Meshing EditMesh DeleteNodes]
    } elseif { $type == "Elements" } {
        set cmd [list Mescape Meshing EditMesh DeleteElems]
        if { [GiD_Set DeleteAlsoLower] } {
            lappend cmd LowerEntities
        } else {
            lappend cmd NoLowerEntities
        }
    } else {
        error "Unexpected type=$type"
    }
    GiD_Process {*}$cmd
}

proc DoGeometryCreateNURBSSurfaceAutomatic { } {
    set cmd [list Mescape Geometry Create NurbsSurface Automatic]
    #I could only ask the number of lines once, next times with increasing amount if sides is raised from GiD server (C++)
    #to allow the client open these emergent dialog it is implemented the event GiD_Event_MessageBoxEntry
    #(in fact with this event it is not necessay to ask here the number of lines, I am doing to work locally when possible)
    set default [GetOrDefault ::GiD_Dialog(DoGeometryCreateNURBSSurfaceAutomatic,number_sides) 4]
    set number_sides [MessageBoxEntry [_ "Enter value window"] [_ "Enter number of lines"] int+ $default 0 question]
    if { $number_sides != "" } {
        set ::GiD_Dialog(DoGeometryCreateNURBSSurfaceAutomatic,number_sides) $number_sides
        lappend cmd $number_sides
        GiD_Process {*}$cmd
    }
}

#I am not using it because then lines u=0 v=0 are not drawn to aid selecting the direction to cut
proc DoDivideSurfaceNumDivisions_not_used { } {
    set cmd [list Mescape Geometry Edit DivideSurf NumDivisions]
    set surface_id [GidUtils::PickEntities Surfaces single [_ "Select surface to divide"]]
    if { $surface_id != "" } {
        lappend cmd $surface_id
        set question [_ "Choose sense"]
        set answer_sense [MessageBoxOptionsButtons [_ "Divide surface"] $question {USense VSense Cancel} [list [_ "U Sense"] [_ "V Sense"] [_ "Cancel"]] question ""]
        if { $answer_sense != "Cancel" } {
            lappend cmd $answer_sense
            set default [GetOrDefault ::GiD_Dialog(DoDivideSurfaceNumDivisions,number_sides) 2]
            set number_sides [MessageBoxEntry [_ "Enter value window"] [_ "Enter number of divisions"] int+ $default 0 question]
            if { $number_sides != "" } {
                set ::GiD_Dialog(DoDivideSurfaceNumDivisions,number_sides) $number_sides
                lappend cmd $number_sides
                GiD_Process {*}$cmd
            }
        }
    }
}

proc DoDivideSurfaceNumDivisions { } {
    set cmd [list Mescape Geometry Edit DivideSurf NumDivisions]
    GiD_Process {*}$cmd
}

proc DoCreateObjectRectangle { } {
    set cmd [list Mescape Geometry Create Object Rectangle]
    GiD_Process {*}$cmd
}

proc DoCreateObjectPolygon { } {
    set cmd [list Mescape Geometry Create Object PolygonPNR]
    set number_sides [MessageBoxEntry [_ "Enter value window"] [_ "Enter number of sides"] int+ 4 0 question]
    if { $number_sides != "" } {
        lappend cmd $number_sides
        set center [GidUtils::GetCoordinates [_ "Enter a center for the polygon"] FNoJoin GEOMETRYUSE]
        if { $center != "" } {
            if { $::GidPriv(selcoord,id) > 0 } {
                #it is an existent point
                lappend cmd FJoin $::GidPriv(selcoord,id)
            } else {
                lappend cmd FNoJoin [join $center ,]
            }
            set normal [MessageBoxGetNormal [_ "Enter a normal for the polygon"]]
            if { $normal != "" } {
                lappend cmd [join $normal ,]
                GiD_Process {*}$cmd
            }
        }
    }
}

proc DoCreateObjectCircle { } {
    set cmd [list Mescape Geometry Create Object CirclePNR]
    #allowing local recopilation of data
    set center [GidUtils::GetCoordinates [_ "Enter a center for the circle"] FNoJoin GEOMETRYUSE]
    if { $center != "" } {
        if { $::GidPriv(selcoord,id) > 0 } {
            #it is an existent point
            lappend cmd FJoin $::GidPriv(selcoord,id)
        } else {
            lappend cmd FNoJoin [join $center ,]
        }
        set normal [MessageBoxGetNormal [_ "Enter a normal for the circle"]]
        if { $normal != "" } {
            lappend cmd [join $normal ,]
            GiD_Process {*}$cmd
        }
    }
}

proc DoCreateObjectSphere { } {
    set cmd [list Mescape Geometry Create Object Sphere]
    GiD_Process {*}$cmd
}

proc DoCreateObjectCylinder { } {
    set cmd [list Mescape Geometry Create Object Cylinder]
    set center [GidUtils::GetCoordinates [_ "Enter a center for the base"] FNoJoin GEOMETRYUSE]
    if { $center != "" } {
        if { $::GidPriv(selcoord,id) > 0 } {
            #it is an existent point
            lappend cmd FJoin $::GidPriv(selcoord,id)
        } else {
            lappend cmd FNoJoin [join $center ,]
        }
        set normal [MessageBoxGetNormal [_ "Enter a normal for the cylinder's base"]]
        if { $normal != "" } {
            lappend cmd [join $normal ,]
            GiD_Process {*}$cmd
        }
    }
}

proc DoCreateObjectCone { } {
    set cmd [list Mescape Geometry Create Object Cone]
    set center [GidUtils::GetCoordinates [_ "Enter a center for the base"] FNoJoin GEOMETRYUSE]
    if { $center != "" } {
        if { $::GidPriv(selcoord,id) > 0 } {
            #it is an existent point
            lappend cmd FJoin $::GidPriv(selcoord,id)
        } else {
            lappend cmd FNoJoin [join $center ,]
        }
        set normal [MessageBoxGetNormal [_ "Enter a normal for the cone's base"]]
        if { $normal != "" } {
            lappend cmd [join $normal ,]
            GiD_Process {*}$cmd
        }
    }
}

proc DoCreateObjectPrism { } {
    set cmd [list Mescape Geometry Create Object Prism]
    set number_sides [MessageBoxEntry [_ "Enter value window"] [_ "Enter number of sides"] int+ 4 0 question]
    if { $number_sides != "" } {
        lappend cmd $number_sides
        set center [GidUtils::GetCoordinates [_ "Enter a center for the polygon"] FNoJoin GEOMETRYUSE]
        if { $center != "" } {
            if { $::GidPriv(selcoord,id) > 0 } {
                #it is an existent point
                lappend cmd FJoin $::GidPriv(selcoord,id)
            } else {
                lappend cmd FNoJoin [join $center ,]
            }
            set normal [MessageBoxGetNormal [_ "Enter a normal for the polygon's base"]]
            if { $normal != "" } {
                lappend cmd [join $normal ,]
                GiD_Process {*}$cmd
            }
        }
    }
}

proc DoCreateObjectTorus { } {
    set cmd [list Mescape Geometry Create Object Torus]
    set center [GidUtils::GetCoordinates [_ "Enter a center for the torus"] FNoJoin GEOMETRYUSE]
    if { $center != "" } {
        if { $::GidPriv(selcoord,id) > 0 } {
            #it is an existent point
            lappend cmd FJoin $::GidPriv(selcoord,id)
        } else {
            lappend cmd FNoJoin [join $center ,]
        }
        set normal [MessageBoxGetNormal [_ "Enter a normal for the torus's base"]]
        if { $normal != "" } {
            lappend cmd [join $normal ,]
            GiD_Process {*}$cmd
        }
    }
}


#can provide answer_save_changes "Yes" to save or "No" to discard an do not ask the question
proc DoFilesNew { {answer_save_changes ""} } {
    set cmd [list MEscape Files New]
    if { [GiD_Info Project AreChanges] } {
        if { $answer_save_changes == "" } {
            set question [_ "Save changes to this project?"]
            set answer_save_changes [MessageBoxOptionsButtons [_ "New project"] $question [list Yes No Cancel] [list [_ "Yes"] [_ "No"] [_ "Cancel"]] question ""]
        }
        if { $answer_save_changes == "Cancel" } {
            GidUtils::SetWarnLine [_ "Leaving function"]
            return 1
        } elseif { $answer_save_changes == "Yes" } {
            set fail [DoProjectSave]
            if { $fail } {
                return 1
            }
        } else {
            lappend cmd $answer_save_changes
        }
    }
    # The preferences windows leaves us inside the Utilities Var menu, making the Mescape unusable as it's handles as a variable ...
    GiD_Process {*}$cmd
    return 0
}

proc DoQuit {} {
    if { [ info exists ::GidPriv(FullScreen)] && $::GidPriv(FullScreen)} {
        SwitchFullScreen
    }
    set cmd [list escape escape escape escape Quit]
    if { [GiD_Info Project AreChanges] } {
        set question [_ "Save changes before quitting?"]
        set answer_save_changes [MessageBoxOptionsButtons [_ "New project"] $question [list Yes No Cancel] [list [_ "Yes"] [_ "No"] [_ "Cancel"]] question ""]
        if { $answer_save_changes == "Cancel" } {
            GidUtils::SetWarnLine [_ "Leaving function"]
            set ::DoingQuitFullScreen 0
            return 1
        } elseif { $answer_save_changes == "Yes" } {
            set fail [DoProjectSave]
            if { $fail } {
                set ::DoingQuitFullScreen 0
                return 1
            }
        } else {
            lappend cmd $answer_save_changes
        }
    }
    if { $::gid_start_mode == "client" } {
        #not to send quit commands to the server if we don't want to quit the server!!
        gid_client_exit
    } else {
        GiD_Process {*}$cmd
    }
    set ::DoingQuitFullScreen 0
}

proc DoPostprocess { } {
    GiD_Process Mescape Postprocess
}

proc DoPreprocess { {answer_leave_postprocess ""} } {
    if { $answer_leave_postprocess == "" } {
        set question [_ "Sure, leaving postprocess?"]
        set answer_leave_postprocess [MessageBoxOptionsButtons [_ "New project"] $question [list Yes Cancel] [list [_ "Ok"] [_ "Cancel"]] question ""]
    }
    if { $answer_leave_postprocess == "Cancel" } {
        return 1
    }
    GiD_Process Mescape Preprocess Yes
}

proc DoFilesRead { } {
    set cmd [list Mescape Files Read]
    set answer_save_changes ""
    if { [GiD_Info Project AreChanges] } {
        set question [_ "Save changes to this project?"]
        set answer_save_changes [MessageBoxOptionsButtons [_ "Read project"] $question [list Yes No Cancel] [list [_ "Yes"] [_ "No"] [_ "Cancel"]] question ""]
        if { $answer_save_changes == "Cancel" } {
            GidUtils::SetWarnLine [_ "Leaving function"]
            return 1
        } elseif { $answer_save_changes == "Yes" } {
            set fail [DoProjectSave]
            if { $fail } {
                return 1
            }
        } else {
            lappend cmd $answer_save_changes
        }
    }
    GidUtils::SetWarnLine  [_ "Enter name of the project"]
    set filename [MessageBoxGetFilename project read [_ "Read project"] "" "" "" 0 [list]]
    if  { $filename != "" } {
        if { [GiD_Set AutomaticAnswers] } {
            #don't need to see the file, try to read it in any case
            lappend cmd $filename escape
        } else {
            set filename_geo [file join $filename [file rootname [file tail $filename]].geo]
            set filename_geo_version [lindex [gid_filesystem::read_info_geo_file $filename_geo] 1]
            if { $filename_geo_version > [GiD_Info Project GeoVersion] } {
                set question [_ "The file version format is too modern for this program version.\nDo you want to read it anyway?\n(maybe the program will crash)"]
                set answer [MessageBoxOptionsButtons [_ "Read project"] $question [list Continue Cancel] [list [_ "Try it"] [_ "Cancel"]] question ""]
                if { $answer == "Cancel" } {
                    GidUtils::SetWarnLine [_ "Leaving function"]
                    return 1
                }
                lappend cmd $filename Continue escape
            } else {
                lappend cmd $filename escape
            }
        }
        GiD_Process {*}$cmd
    } else {
        GidUtils::SetWarnLine [_ "Leaving function"]
        return 1
    }
    return 0
}

proc DoProjectSave { } {
    set fail 0
    if { [GidUtils::ModelHasName] } {
        GiD_Process Mescape Files Save escape
    } else {
        set fail [DoProjectSaveAs]
    }
    return $fail
}

proc DoProjectSaveAs { } {
    GidUtils::SetWarnLine  [_ "Enter name of the project"]
    set title [_ "Save project"]
    set cmd [list Mescape Files SaveAs]
    set extra_options [list [_ "Save options"] BrowserExtraCreateSaveAs BrowserExtraGetSaveAs]
    set initial_filename ""
    if { [GidUtils::ModelHasName] } {
        set initial_filename [file tail [GiD_Info project modelname]]
    }
    set filename [MessageBoxGetFilename project write $title $initial_filename "" "" 0 $extra_options]
    if  { $filename != "" } {
        set get_extra_options_procedure [lindex $extra_options 2]
        if { $get_extra_options_procedure != "" } {
            lappend cmd {*}[$get_extra_options_procedure]
        }
        lappend cmd $filename
        GiD_Process {*}$cmd
    } else {
        GidUtils::SetWarnLine [_ "Leaving function"]
        return 1
    }
    return 0
}

#DoFilesImport IGES IGES {.igs .iges} 0 0 {{Import options} BrowserExtraCreateOptionsImportGeom BrowserExtraGetOptionsImportGeom}
# multiple: 0 or 1 to allow selection of multiple files
# create_layer_filename: 0 or 1 to create a layer by filename and try to create inside the entities
proc DoFilesImport { format format_label extensions multiple create_layer_filename extra_options } {
    set cmd [list Mescape Files ${format}Read] ;##regular syntax, all others are irregular
    set category file
    if { $format == "XYZ_ToPoints" } {
        set cmd [list Mescape Files XYZRead ToPoints]
    } elseif { $format == "XYZ_ToNodes" } {
        set cmd [list Mescape Files XYZRead ToNodes]
    } elseif { $format == "SurfMeshes" } {
        set cmd [list Mescape Files ReadSurfMeshes ReadFromFile]
    } elseif { $format == "VTKVoxels" } {
        set cmd [list Mescape Files LoadVTKVoxels]
    } elseif { $format == "BatchFile" } {
        set cmd [list Mescape Files BatchFile]
    } elseif { $format == "InsertProject" } {
        set cmd [list Mescape Files InsertGeom]
        set category project
    }
    set title [_ "%s read" $format_label]
    set types [list [list $format_label $extensions] [list [_ "All files"] ".*"]]
    #set default_extension [lindex $extensions 0]
    set default_extension ""
    GidUtils::SetWarnLine  [_ "Enter name of the file"]
    set filenames [MessageBoxGetFilename $category read $title "" $types $default_extension $multiple $extra_options]
    if  { [llength $filenames] } {
        set get_extra_options_procedure [lindex $extra_options 2]
        if { $get_extra_options_procedure != "" } {
            lappend cmd {*}[$get_extra_options_procedure]
        }
        if { $multiple } {
            foreach filename $filenames {
                if { $create_layer_filename } {
                    set layer [file rootname [file tail $filename]]
                    if { ![GiD_Layers exists $layer] } {
                        GiD_Layers create $layer
                    }
                    GiD_Layers edit to_use $layer
                }
                GiD_Process {*}$cmd $filename escape
            }
        } else {
            set filename $filenames
            GiD_Process {*}$cmd $filename escape
        }
    } else {
        GidUtils::SetWarnLine [_ "Leaving function"]
        return 1
    }
    return 0
}

#format_cmd: IGES STEP DXF ACIS Rhino GID_MESH RASTER GID_REPORT
proc DoFilesExport { format extensions extra_options } {
    set format_cmd ${format}Write
    if { $format == "DXF" } {
        set format_label AutoCAD
    } elseif { $format == "GID_MESH" } {
        set format_label [_ "%s mesh ASCII#C#menu" $::GidPriv(ProgName)]
        set format_cmd WriteMesh ;#irregular
    } elseif { $format == "RASTER" } {
        set format_label "Arc/Info ASCII"
    } elseif { $format == "GID_REPORT" } {
        set format_label [_ "Data report"]
        set format_cmd WriteAscii ;#irregular
    } else {
        set format_label [_ $format]
    }
    set title [_ "%s write" $format_label]
    set types [list [list $format_label $extensions] [list [_ "All files"] ".*"]]
    set default_extension [lindex $extensions 0]

    GidUtils::SetWarnLine  [_ "Enter name of the file"]
    set filename [MessageBoxGetFilename file write $title "" $types $default_extension 0 $extra_options]
    if  { $filename != "" } {
        if { $format == "RASTER" } {
            if { [file exists $filename] } {
                file delete $filename
            }
            set far_points_set_nodata 0
            if { [info exists ::ExportRASTEROptions(var,far_points_set_nodata)] } {
                set far_points_set_nodata $::ExportRASTEROptions(var,far_points_set_nodata)
            }
            set far_points_distance 0.0
            if { [info exists ::ExportRASTEROptions(var,far_points_distance)] } {
                set far_points_distance $::ExportRASTEROptions(var,far_points_distance)
            }
            DoNodesToRasterArcInfoGridASCII $filename $far_points_set_nodata $far_points_distance
        } else {
            set cmd [list Mescape Files $format_cmd]
            set get_extra_options_procedure [lindex $extra_options 2]
            if { $get_extra_options_procedure != "" } {
                lappend cmd {*}[$get_extra_options_procedure]
            }
            lappend cmd $filename
            if { ![GiD_Set DialogWrowser] } {
                if { [file exists $filename] } {
                    set answer_overwrite Yes ;#don't ask again, already asked by MessageBoxGetFilename
                    lappend cmd $answer_overwrite
                }
            }
            lappend cmd escape
            GiD_Process {*}$cmd
        }
    } else {
        GidUtils::SetWarnLine [_ "Leaving function"]
        return 1
    }
    return 0
}

proc DoFilesExportCalculationFile { } {
    if { ![GiD_Set CalcWithoutMesh] && ![GidUtils::ExistsMesh] } {
        set msg [_ "Don't forget to generate the mesh"]
        GidUtils::SetWarnLine $msg
        WarnWin $msg
        return 1
    }
    set format_cmd WriteCalcFile
    if { [GiD_Set DefaultFileNameInCalcFile] && [GidUtils::ModelHasName] } {
        #will get automatic name based on model name
        set cmd [list Mescape Files $format_cmd]
    } else {
        set format_label [_ "calculation data"]
        set title [_ "%s write" $format_label]
        if { [info exists ::GidPriv(CalculationFileExtension)] } {
            set extensions [list $::GidPriv(CalculationFileExtension)]
        } else {
            set extensions .dat
        }
        set types [list [list $format_label $extensions] [list [_ "All files"] ".*"]]
        set default_extension [lindex $extensions 0]
        GidUtils::SetWarnLine [_ "Enter name of the file"]
        set filename [MessageBoxGetFilename file write $title "" $types $default_extension 0 [list]]
        if  { $filename != "" } {
            set cmd [list Mescape Files $format_cmd $filename escape]
        } else {
            GidUtils::SetWarnLine [_ "Leaving function"]
            return 1
        }
    }
    GiD_Process {*}$cmd
    return 0
}

#format: BMP GIF JPEG PNG TGA TIFF VRML SVG PGF PS EPS PDF
proc DoPrintToFile { format extensions {vectorial ""}} {
    set format_label [_ $format]
    set title [_ "%s write" $format_label]
    set types [list [list $format_label $extensions] [list [_ "All files"] ".*"]]
    set default_extension [lindex $extensions 0]
    GidUtils::SetWarnLine  [_ "Enter name of the file"]
    if { [info exists ::default_filename($format)] } {
        set ::default_filename($format) [GidUtils::AddNumberToFile $::default_filename($format)]
    } else {
        set ::default_filename($format) ""
    }
    set filename [MessageBoxGetFilename file write $title $::default_filename($format) $types $default_extension 0 [list]]
    if  { $filename != "" } {
        set is_vectorial 0
        if { $vectorial != "" } {
            #could be vectorial or pixmap, select depending on the variable
            if { $vectorial } {
                set answer_vectorial Yes
                set is_vectorial 1
            } else {
                set answer_vectorial No
            }
            GiD_Process 'Hardcopy Options VectorialPS $answer_vectorial Escape
        }
        if { $format == "SVG" || $format == "PGF" } {
            #could be only vectorial, not pixmap
            set is_vectorial 1
        }
        set cmd [list 'Hardcopy $format]
        if { $is_vectorial && [GiD_Set OpenGL(ContourFillTexture)] } {
            set answer_allow_temporary_mode_change Yes ;#do not ask user
            lappend cmd $answer_allow_temporary_mode_change
        }
        lappend cmd $filename
        if { ![GiD_Set DialogWrowser] } {
            if { [file exists $filename] } {
                set answer_overwrite Yes ;#don't ask again, already asked by MessageBoxGetFilename
                lappend cmd $answer_overwrite
            }
        }
        lappend cmd escape
        GiD_Process {*}$cmd
        set ::default_filename($format) $filename
    } else {
        GidUtils::SetWarnLine [_ "Leaving function"]
        return 1
    }
    return 0
}

proc DoZoom { action } {
    if { $action == "In" || $action == "Out" } {
        set ::zoom_box_state(in) 1 ;#variable to indicate the action to be done on mouse events (open a selection rectangle)
    }
    GiD_Process 'Zoom $action
}


proc DoMeshingGenerate { } {
    if { $::avoid_open_window_mesh_from_process == 0 } {
        GiD_Process Mescape Meshing Generate
    } else {
        if { [GidUtils::ExistsMesh] } {
            if { [GiD_Set AutomaticAnswers] } {
                set delete_old_mesh Yes
            } else {
                #here it cannot be used  [GiD_Set -meshing_parameters_model MaintainOldMesh] because they are not "loaded" yet
                if { [GiD_Set MaintainOldMesh] } {
                    set question [_ "Erase old mesh?"]
                    set delete_old_mesh [MessageBoxOptionsButtons [_ "Generate mesh"] $question [list Yes No Cancel] [list [_ "Yes"] [_ "No"] [_ "Cancel"]] question ""]
                } else {
                    set question [_ "The old mesh will be erased. Continue?"]
                    set delete_old_mesh [MessageBoxOptionsButtons [_ "Generate mesh"] $question [list Yes Cancel] [list [_ "Yes"] [_ "Cancel"]] question ""]
                }
                if { $delete_old_mesh == "Cancel" } {
                    GidUtils::SetWarnLine [_ "Leaving function"]
                    return  1
                }
            }
            set ::ChangeVarInfo(DeleteOldMesh) $delete_old_mesh
        } else {
            set ::ChangeVarInfo(DeleteOldMesh) ""
        }
        MeshGenerationWindowDefaultValues
    }
    return 0
}

proc MeshGenerationWindowDefaultValues { } {
    set recommended_size [GiD_Info Project RecommendedMeshSize]
    set last_generation_size [GiD_Info Project LastElementSize]
    if { $last_generation_size == "NONE" } {
        set last_generation_size $recommended_size
    }
    set get_meshing_parameters_from_model [GiD_Set LoadMeshingPrefOfModel]
    MeshGenerationWindow $last_generation_size $recommended_size $get_meshing_parameters_from_model
}

proc MeshGenerationWindow { LastGenSize RecommendedSize LoadMeshingPrefOfModel } {
    if { [GidUtils::IsTkDisabled] } {
        #e.g. batch mode without windows
        return
    }
    set w .gid.meshgeneration
    if { $LastGenSize<0.0 } {
        set combovalues [format %g $RecommendedSize]
    } else {
        set combovalues [format "%g %g" $LastGenSize $RecommendedSize]
    }
    
    InitWindow2 $w -title [_ "Mesh generation"] -geometryvariable PreMeshGenerationWindowGeom -onlygeometry -ontop
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0
    ttk::frame $w.frmSize
    ttk::label $w.frmSize.sizeText -text [_ "Enter size of elements to be generated"]
    ttk::combobox $w.frmSize.comboSizes -values $combovalues
    $w.frmSize.comboSizes current 0
    GidHelp "$w.frmSize $w.frmSize.sizeText $w.frmSize.comboSizes" \
        [_ "Enter the general size to be assigned to the geometrical\
        \n entities which have no size assigned."]
    ttk::frame $w.frmMeshingParams
    set ::ChangeVarInfo(LoadMeshingPrefOfModel) $LoadMeshingPrefOfModel
    ttk::checkbutton $w.frmMeshingParams.mParamFromModel -text [_ "Get meshing parameters from model"] \
        -variable ::ChangeVarInfo(LoadMeshingPrefOfModel)
    GidHelp "$w.frmMeshingParams.mParamFromModel" \
        [_ "Choose the meshing parameters to be used for the mesh generation:\
        \n from the model (you can see them from Utilities->status window),\
        \n or from the meshing preferences (you can see them from Utilities->Preferences->Meshing."]


    ttk::frame $w.frmButtons -style BottomFrame.TFrame
    ttk::button $w.frmButtons.btnOK -text [_ "Ok"] -command [list MeshGenerationOK $w] -style BottomFrame.TButton
    ttk::button $w.frmButtons.btnclose -text [_ "Cancel"] -command [list MeshGenerationClose $w] -style BottomFrame.TButton
    grid $w.frmSize.sizeText -sticky ew
    grid $w.frmSize.comboSizes -sticky w
    grid $w.frmMeshingParams.mParamFromModel -sticky sew

    if { $::show_gid_mesh_remote } {
        if { ![info exists ::ChangeVarInfo(MeshRemote)] } {
            set ::ChangeVarInfo(MeshRemote) 0
        }
        ttk::checkbutton $w.frmMeshingParams.mMeshRemote -text [_ "Cloud mesh generation (beta)"] -variable ::ChangeVarInfo(MeshRemote)
        GidHelp "$w.frmMeshingParams.mMeshRemote" [_ "To generate the mesh in the GiD Cloud mesh service."]
        grid $w.frmMeshingParams.mMeshRemote -sticky sew
        if { ![GiD_Login is_logged] } {
            set ::ChangeVarInfo(MeshRemote) 0
            $w.frmMeshingParams.mMeshRemote configure -state disabled
            GidHelp "$w.frmMeshingParams.mMeshRemote" [_ "To generate the mesh in the GiD Cloud mesh service. Please log in with your GiD account."]
        }
    }

    grid $w.frmSize -sticky new -padx 5
    grid $w.frmMeshingParams -sticky sew -padx 5 -pady 5
    grid columnconfigure $w.frmSize 0 -weight 1
    grid columnconfigure $w.frmMeshingParams 0 -weight 1
    grid rowconfigure $w 2 -weight 1
    grid columnconfigure $w 0 -weight 1
    grid $w.frmButtons -sticky ews -row 3
    grid anchor $w.frmButtons center
    grid $w.frmButtons.btnOK $w.frmButtons.btnclose -padx 5 -pady 6
    $w.frmSize.comboSizes selection range 0 end
    focus $w.frmSize.comboSizes
    wm minsize $w 250 120
    wm maxsize $w 350 250
    bind $w <Alt-c> [list $w.frmButtons.btnclose invoke]
    bind $w <Escape> [list $w.frmButtons.btnclose invoke]
    bind $w <Return> [list $w.frmButtons.btnOK invoke]
}

proc MeshGenerationOK { w  } {
    set cb $w.frmSize.comboSizes
    if { ![winfo exists $cb] } {
        return 1
    }
    set mesh_size [$cb get]
    if { ![string is double -strict $mesh_size] || $mesh_size<1e-15} {
        WarnWin [_ "Size must be greater than zero. Try again."]
        return 1
    }
    if { [info exists ::ChangeVarInfo(LoadMeshingPrefOfModel)] && $::ChangeVarInfo(LoadMeshingPrefOfModel) } {
        set meshing_parameters_from Model
    } else {
        set meshing_parameters_from Preferences
    }
    set mesh_remote 0
    if { [info exists ::ChangeVarInfo(MeshRemote)] } {
        if { [GiD_Login is_logged] } {
            set mesh_remote $::ChangeVarInfo(MeshRemote)
        } else {
            set mesh_remote 0
        }
    }
    destroy $w
    MeshGenerationOKDo $mesh_size $meshing_parameters_from $mesh_remote
}

#meshing_parameters_from Model or Preferences
proc MeshGenerationOKDo { mesh_size {meshing_parameters_from ""} {mesh_remote 0}} {
    if { $meshing_parameters_from == "" } {
        #to not change proc arguments do meshing_parameters_from optional and if not set get it from global variable
        if { [info exists ::ChangeVarInfo(LoadMeshingPrefOfModel)] && $::ChangeVarInfo(LoadMeshingPrefOfModel) } {
            set meshing_parameters_from Model
        } else {
            set meshing_parameters_from Preferences
        }
    }
    if { $mesh_remote } {
        GidUtils::EvalAndEnterInBatch MeshGenerationRemoteDo $mesh_size $meshing_parameters_from
    } else {
        MeshGenerationLocalDo $mesh_size $meshing_parameters_from
    }
}

proc MeshGenerationLocalDo { mesh_size meshing_parameters_from } {
    set fail 0
    if { $::avoid_open_window_mesh_from_process == 0 } {
        GiD_Process $mesh_size MeshingParametersFrom=$meshing_parameters_from escape
    } else {
        set cmd [list Mescape Meshing Generate]
        if { [info exists ::ChangeVarInfo(DeleteOldMesh)] && $::ChangeVarInfo(DeleteOldMesh) != "" } {
            lappend cmd $::ChangeVarInfo(DeleteOldMesh)
        }
        lappend cmd $mesh_size MeshingParametersFrom=$meshing_parameters_from escape
        GiD_EnterInBatch "*****MARK START_BLOCK GenerateMesh $mesh_size\n"
        set result [GidUtils::EvalAndEnterInBatch GiD_RaiseEvent GiD_Event_BeforeMeshGeneration $mesh_size]
        if { $result != "-cancel-" } {
            GiD_Process {*}$cmd
        }
        GiD_EnterInBatch "*****MARK END_BLOCK GenerateMesh $mesh_size\n"
    }
    return $fail
}

proc MeshGenerationClose { w } {
    destroy $w
    if { $::avoid_open_window_mesh_from_process == 0 } {
        GiD_Process escape
    }
}

proc GetDefaultMeshToRasterSize { } {
    set mesh_bounding_box [lindex [GiD_Info layers -bbox -use mesh] 0]
    set p0 [lrange $mesh_bounding_box 0 2]
    set p1 [lrange $mesh_bounding_box 3 5]
    set diagonal [MathUtils::VectorDiff $p1 $p0]
    set diagonal2d_modulus [MathUtils::VectorModulus [concat [lrange $diagonal 0 1] 0.0]]
    set default_raster_size [expr $diagonal2d_modulus*0.01]
    return [format %.2g $default_raster_size]
}

proc DoNodesRasterSize { action node_ids raster_size far_points_set_nodata far_points_distance } {
    set ids_uncompacted [GidUtils::UnCompactNumberList $node_ids]
    set raster [GIS::GetRasterFromNodes $ids_uncompacted $raster_size $far_points_set_nodata $far_points_distance]
    set increment 1 ;#1 to not subsample
    set show_advance_bar 0
    set value_smoothed_to_nodes 1
    set fail [GIS::ImportRaster $raster $action $increment $show_advance_bar $value_smoothed_to_nodes]
    return $fail
}

#action can be: geometry mesh
proc DoGeometryCreateGeometryFromMeshNodesToGridSurfaces { action } {
    set fail 0
    if { ![info exists ::default_mesh_to_raster_size] } {
        set ::default_mesh_to_raster_size [GetDefaultMeshToRasterSize]
    }
    set raster_size [MessageBoxEntry [_ "Enter value window"] [_ "Enter raster size"] real+ $::default_mesh_to_raster_size 0 question BrowserExtraCreateExportRASTER]
    if { $raster_size != "" && [string is double -strict $raster_size] } {
        MeshView 1
        set node_ids [GidUtils::PickEntities Nodes multiple [_ "Select nodes to create raster"]]
        if { $node_ids != "" } {
            set far_points_set_nodata 0
            if { [info exists ::ExportRASTEROptions(var,far_points_set_nodata)] } {
                set far_points_set_nodata $::ExportRASTEROptions(var,far_points_set_nodata)
            }
            set far_points_distance 0.0
            if { [info exists ::ExportRASTEROptions(var,far_points_distance)] } {
                set far_points_distance $::ExportRASTEROptions(var,far_points_distance)
            }
            set fail [GidUtils::EvalAndEnterInBatch DoNodesRasterSize $action $node_ids $raster_size $far_points_set_nodata $far_points_distance]
        }
        set ::default_mesh_to_raster_size $raster_size
    } else {
        set fail 1
    }
    return $fail
}

proc DoTrianglesToRasterSize { action triangle_ids raster_size far_points_set_nodata far_points_distance } {
    set ids_uncompacted [GidUtils::UnCompactNumberList $triangle_ids]
    set raster [GIS::GetRasterFromTriangles $ids_uncompacted $raster_size $far_points_set_nodata $far_points_distance]
    if {$raster == ""} {
        set fail 1
    } else {
        set increment 1 ;#1 to not subsample
        set show_advance_bar 0
        set value_smoothed_to_nodes 1
        set fail [GIS::ImportRaster $raster $action $increment $show_advance_bar $value_smoothed_to_nodes]
    }
    return $fail
}

#action can be: geometry mesh
proc DoGeometryCreateGeometryFromMeshTrianglesToGridSurfaces { action } {
    set fail 0
    if { ![info exists ::default_mesh_to_raster_size] } {
        set ::default_mesh_to_raster_size [GetDefaultMeshToRasterSize]
    }
    set raster_size [MessageBoxEntry [_ "Enter value window"] [_ "Enter raster size"] real+ $::default_mesh_to_raster_size 0 question BrowserExtraCreateExportRASTER]
    if { $raster_size != "" && [string is double -strict $raster_size] } {
        MeshView 1
        set triangle_ids [GidUtils::PickEntities "Elements ElementType Triangle" multiple [_ "Select triangles to create raster"]]
        if { $triangle_ids != "" } {
            set far_points_set_nodata 0
            if { [info exists ::ExportRASTEROptions(var,far_points_set_nodata)] } {
                set far_points_set_nodata $::ExportRASTEROptions(var,far_points_set_nodata)
            }
            set far_points_distance 0.0
            if { [info exists ::ExportRASTEROptions(var,far_points_distance)] } {
                set far_points_distance $::ExportRASTEROptions(var,far_points_distance)
            }
            set fail [GidUtils::EvalAndEnterInBatch DoTrianglesToRasterSize $action $triangle_ids $raster_size $far_points_set_nodata $far_points_distance]
        }
        set ::default_mesh_to_raster_size $raster_size
    } else {
        set fail 1
    }
    return $fail
}

proc DoNodesToRasterArcInfoGridASCII { filename far_points_set_nodata far_points_distance } {
    set fail 0
    set action convert_file
    if { $filename != "" } {
        if { ![info exists ::default_mesh_to_grid_size] } {
            set ::default_mesh_to_grid_size [GetDefaultMeshToRasterSize]
        }
        set raster_size [MessageBoxEntry [_ "Enter value window"] [_ "Enter raster size"] real+ $::default_mesh_to_grid_size 0 question BrowserExtraCreateExportRASTER]
        if { $raster_size != "" && [string is double -strict $raster_size] } {
            set node_ids [GidUtils::PickEntities Nodes multiple [_ "Select nodes to create raster"]]
            if { $node_ids != "" } {
                set ids_uncompacted [GidUtils::UnCompactNumberList $node_ids]
                set raster [GIS::GetRasterFromNodes $ids_uncompacted $raster_size $far_points_set_nodata $far_points_distance]
                set increment 1 ;#1 to not subsample
                set show_advance_bar 1
                set value_smoothed_to_nodes 0
                set fail [GIS::ImportRaster $raster $action $increment $show_advance_bar $filename $value_smoothed_to_nodes]
            }
            set ::default_mesh_to_grid_size $raster_size
        } else {
            set fail 1
        }
    }
    return $fail
}

proc DoMeshEditMeshMoveNodesToZRaster {} {
    set fail 0
    set extensions {.adf .asc .bmp .dat .hgt .jpg .jpeg .N1 .ppm .png .sid .tga .tif .tiff .txt}
    set types [list [list [_ "Image"] $extensions] [list [_ "All files"] ".*"]]
    set filenames_raster [MessageBoxGetFilename file read [_ "Read raster file"] "" $types "" 1 [list]]
    if  { [llength $filenames_raster] } {
        set node_ids [GidUtils::PickEntities Nodes multiple [_ "Select nodes to create raster"]]
        if { $node_ids != "" } {
            foreach filename_raster $filenames_raster {
              incr fail [GidUtils::EvalAndEnterInBatch GIS::MoveNodesToZFilenameRaster $filename_raster $node_ids $::GidPriv(MoveNodesToZRasterPreserveZ)]
            }
            GiD_Redraw
        }
    }
    return $fail
}

#return a flat list of keys values, only the ones without default value (but default values relative to this GiD version, could be different in other versions or not exists)
proc GetMeshVariablesAndValuesNotDefault { } {
    set key_values [list]
    foreach key [GiD_Info variables -mesh -expand_array_names] {
        set value [GiD_Set $key]
        if { $value != [GiD_Set -default $key] } {
            lappend key_values $key $value
        }
    }
    return $key_values
}


proc MeshGenerationRemoteDo { mesh_size meshing_parameters_from } {
    set fail 0
    set result [GidUtils::EvalAndEnterInBatch GiD_RaiseEvent GiD_Event_BeforeMeshGeneration $mesh_size]
    if { $result == "-cancel-" } {
        GiD_RaiseEvent GiD_Event_AfterMeshGeneration $fail
        return 0
    }
    set t0 [clock seconds]
    #set storage nextcloud ;# 3 times slower compared with S3 (e.g. trivial case immediate local mesh, S3 11 seconds, nextcloud 53 seconds)
    set storage S3
    set upload_zipped 1
    package require gid_cross_platform
    package require json
    GidMeshCloudClient::package_require_http

    #these are not the real amount of entities to be meshed, must consider meshing data force to mesh/no mesh and others but by now...
    set num_other 0
    set num_lines [GiD_Geometry list -count -higherentity 0 line]
    set num_surfaces [GiD_Geometry list -count -higherentity 0 surface]
    set num_volumes [GiD_Geometry list  -count -higherentity 0 volume]
    GiD_RaiseEvent GiD_Event_BeforeMeshProgress $num_other $num_lines $num_surfaces $num_volumes

    GiD_Project set last_general_mesh_size $mesh_size
    if { [GidUtils::ExistsMesh] } {
        #by now don't support preserve old mesh, expect read the remote generated without previous mesh
        #GiD_Process Mescape Meshing CancelMesh PreserveFrozen Yes escape
        #to preserver frozen (or inner cached) in remote mesh case must send to server this mesh (a .msh file partial?), to append the new one, 
        #and must be able to read back this .msh with existing old entities (now functions expects a clean mesh)
        GiD_Process Mescape Meshing CancelMesh Yes escape
    }

    set result_location "" ;#local or download
    if { $storage == "S3" } {
        set result_location download
    } elseif { $storage == "nextcloud" }  {
        set result_location local ;#or download
    } else {
        error "unexpected storage $storage"
    }

    set tmp_modelname [gid_cross_platform::get_unused_tmp_filename remote_mesh_input .gid]
    if { 0 } {
        #save because will Read again because was changed by SaveAs
        GiD_Process Mescape Files Save
        set modelname [GiD_Info Project modelname].gid
        #save model, will be better not to use save as, to not modify the current modelname and then must be restored the old name or unsaved name
        #GiD_Project backup save $tmp_modelname
        GiD_Process Mescape Files SaveAs $tmp_modelname
        #restore the original model because was changed by SaveAs
        GiD_Process Mescape Files Read $modelname
    } else {
        #avoid require save_as and then read again, with a lot of secondary visual effects and expensive
        file mkdir $tmp_modelname
        set fail [GidMeshCloudClient::SaveInputFiles $tmp_modelname]
    }
    GidUtils::SetWarnLine [_ "Generate mesh uploading data to cloud"]

    set remote_path "" ;#upload to nextcloud
    set push_id "" ;#upload to S3
    set task_id ""

    set user_token [GiD_Login logged_user_token]
    if { $storage == "nextcloud" } {
        #set remote_path [GidMeshCloudClient::UploadFolderZippedNextcloud $tmp_modelname]
        set remote_path [GidMeshCloudClient::UploadFolderNextcloud $tmp_modelname]
    } elseif { $storage == "S3" } {
        if { $upload_zipped } {
            set upload_json_data [GidMeshCloudClient::UploadFolderZippedS3 $user_token $tmp_modelname]
        } else {
            set upload_json_data [GidMeshCloudClient::UploadFolderS3 $user_token $tmp_modelname]
        }
        set err [catch {
            set upload_dict [json::json2dict $upload_json_data]
        } msg]
        if { $err } {
            W "Cannot upload data. Check server status and Internet connection. $upload_json_data"
            set fail 1
        } else {
            if { [dict exists $upload_dict push_id] } {
                set push_id [dict get $upload_dict push_id]
            } else {
                set push_id ""
                set fail 1
                W "Cannot upload data. $upload_json_data"
            }
        }
    } else {
        error "unexpected storage $storage"
    }

    file delete -force $tmp_modelname


    if { !$fail } {
        set remote_results ""
        set meshing_parameters_from [string tolower $meshing_parameters_from]
        set meshing_parameters [list]
        if { $meshing_parameters_from == "preferences" } {
            set meshing_parameters [GetMeshVariablesAndValuesNotDefault]
        }
        if { $storage == "nextcloud" } {
            set mesh_json_data [GidMeshCloudClient::GenerateMeshNextCloud $user_token $remote_path $result_location $mesh_size $meshing_parameters_from $meshing_parameters]
        } else {
            set mesh_json_data [GidMeshCloudClient::GenerateMeshS3 $user_token $push_id $result_location $mesh_size $meshing_parameters_from $meshing_parameters]
        }
        set err [catch {
            set mesh_dict [json::json2dict $mesh_json_data]
        } msg]
        if { $err } {
            W "Cannot start remote mesh task. $mesh_json_data"
            set fail 1
        }
        if { !$fail } {
            if { [dict exists $mesh_dict task_id] } {
                set task_id [dict get $mesh_dict task_id]
                #set start_time [dict get $mesh_dict start_time] ;#e.g. "2023-01-02 19:23:40"
                GidUtils::SetWarnLine $task_id
            } else {
                set fail 1
                set message ""
                if { [dict exists $mesh_dict message] } {
                    set message [dict get $mesh_dict message]
                } else {
                    set message $mesh_dict
                }
                W "Error: $message"
            }
        }
        if { !$fail } {
            set percent_prev ""
            #set i_check_status 0
            #set i_loop 0
            set check_status_milliseconds 500
            set t_last_check 0
            while { 1 } {
                #incr i_loop
                update
                after 100 ;#do loops to react each 0.1 seconds, but check less frequenly, initially each 0.5 second to each 10 seconds if meshing is slow
                set t_now [clock milliseconds]
                if { [expr {$t_now-$t_last_check}] < $check_status_milliseconds} {
                    continue
                }
                #incr i_check_status
                if { $check_status_milliseconds <10000 } {
                    #initially check with more frequency, for fast cases
                    set check_status_milliseconds [expr int($check_status_milliseconds*1.1)]
                }
                set t_last_check $t_now
                set status_json_data [GidMeshCloudClient::GetStatus $user_token $task_id]
                set status_dict [json::json2dict $status_json_data]
                if { [dict exists $status_dict status] } {
                    set status [dict get $status_dict status]
                } else {
                    set status ""
                    set fail 1
                }
                #set log [dict get $status_dict log]
                if { $status == 0 } {
                    #running
                    set percent_total 0
                    if { [dict exists $status_dict percent percent_total] } {
                        set percent_total [dict get $status_dict percent percent_total]
                        if { $percent_total != $percent_prev } {
                            set percent_prev $percent_total
                            foreach key {percent_other percent_lines percent_surfaces percent_volumes num_nodes num_elements} {
                                set $key [dict get $status_dict percent $key]
                            }
                            #W "percent $percent_total"
                            set result [GiD_RaiseEvent GiD_Event_MeshProgress $percent_total $percent_other $percent_lines $percent_surfaces $percent_volumes $num_nodes $num_elements]
                            if { $result == "-cancel-" } {
                                GidMeshCloudClient::UserStop $user_token $task_id
                                MessageBoxOk [_ "Warning"] [_ "User stop"] question
                                set fail 2
                                break
                            }
                        }
                    }
                } elseif { $status == 1 } {
                    #completed
                    set fail 0
                    if { $storage == "nextcloud" } {
                        set remote_results [file join $remote_path [file rootname [file tail $remote_path]]]
                    } elseif { $storage == "S3" } {
                        if { [dict exists $status_dict file] } {
                            #can be located at $remote_results
                            #e.g. "/v1/mesh_model/result/model_mesh_8a588ed6-7fc3-49e2-bcb4-33dc96ba4c15"
                            #in fact this 'file' field is unneded because the extracted id is the same task_id returned by the generation step
                            set remote_results [dict get $status_dict file]
                            #set task_id [file tail $remote_results]
                        } else {
                            set fail 1
                            W "JSON result without 'file' item"
                        }
                    } else {
                        set fail 1
                        error "unexpected storage $storage"
                    }
                    break
                } elseif { $status == -1 } {
                    #error
                    set fail 1
                    set log 0
                    # "" -> read error messages from memory Tcl variable using MeshErrors::GetMessages
                    set error_message ""
                    if { [dict exists $status_dict error_message] } {
                        set error_message [dict get $status_dict error_message]
                    }
                    MeshErrors::SetMessages $error_message
                    GiD_RaiseEvent GiD_Event_BeforeMeshErrors $error_message
                    if { ![GidUtils::AreWindowsDisabled] } {
                        GidUtils::OpenWindow MESHERRORS
                    }
                    break
                } else {
                    #unexpected
                    set fail 1
                    error "unexpected status $status"
                    break
                }
            }
            #GidUtils::SetWarnLine [_ "Generate mesh cloud num times status checked %s, loops %s" $i_check_status $i_loop]
        }
    }

    if { !$fail } {
        set tmp_folder_out [gid_cross_platform::get_unused_tmp_filename remote_mesh_output .gid]
        file mkdir $tmp_folder_out
        if { $storage == "nextcloud" } {
            set fail [GidMeshCloudClient::DownloadExistentFilesNextcloud $remote_results $tmp_folder_out]
        } elseif { $storage == "S3" } {
            set fail [GidMeshCloudClient::DownloadExistentFilesS3 $user_token $task_id $tmp_folder_out]
        } else {
            error "unexpected storage $storage"
        }
        if { $fail } {
            W "Download files fail. Folder '$tmp_folder_out' not deleted'"
        } else {
            set fail [GidMeshCloudClient::ReadOutputFiles $tmp_folder_out]
            if { $fail } {
                W "Result mesh cannot be read. Folder '$tmp_folder_out' not deleted'"
            } else {
                GiD_MustRemeshFlag set 0
                file delete -force $tmp_folder_out
            }
        }
    }
    if { $storage == "nextcloud" } {
        GidMeshCloudClient::DeleteFolderNextcloud $remote_path
    } elseif { $storage == "S3" } {
        if { $push_id != "" } {
            #delete the input files
            GidMeshCloudClient::DeleteFolderS3 $user_token $push_id 0
        }
        if { $task_id != "" } {
            #delete the output result files
            GidMeshCloudClient::DeleteFolderS3 $user_token $task_id 1
        }
    }
    GiD_RaiseEvent GiD_Event_AfterMeshProgress
    GiD_RaiseEvent GiD_Event_AfterMeshGeneration $fail
    set t1 [clock seconds]
    GidUtils::SetWarnLine [_ "Generate mesh cloud (storage %s)  time %s seconds" $storage [expr ($t1-$t0)]]
    return $fail
}
