# dialog-ram.tcl --
#
# This file defines the procedure tk_dialog, which creates a dialog
# box containing an image, a message, and one or more buttons.
#
# SCCS: @(#) dialog.tcl 1.25 96/04/10 15:43:33
#
# Copyright (c) 1992-1993 The Regents of the University of California.
# Copyright (c) 1994-1996 Sun Microsystems, Inc.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#

proc ConfigureListScrollbars { listbox sx sy } {
    foreach i "x y" {
        if { ![info exists s${i}] || ![winfo exists [set s${i}]] } { continue }
        lassign [$listbox ${i}view] ${i}1 ${i}2
        if { [set ${i}1] == 0 && [set ${i}2] == 1 } {
            after idle grid remove [set s${i}]
        } else {
            after idle grid [set s${i}]
        }
    }
}

proc ReplaceImageName { image } {
    if { $image == "gidquestionhead" || $image == "questhead" || $image == "question"  } {
        set image question.png
    } elseif { $image == "info" } {
        set image info.png
    } elseif { $image == "error" } {
        set image error.png
    } elseif { $image == "warning" } {
        set image warning.png
    }
    return $image
}

proc DestroyWinInDialogs { w geomname } {
    global GidPriv

    bind $w <Destroy> ""
    regexp {[-+]+[0-9]+[-+]+[0-9]+} [wm geometry $w] GidPriv($geomname)
    #     set GidPriv($geomname) [wm geometry $w]
}

#
# This procedure displays a dialog box, with an entry, waits for a button in
# the dialog to be invoked, then returns the entry.
# If cancel is pressed, it returns: --CANCEL--
#
# Arguments:
# w -                Window to use for dialog top-level.
# title -        Title to display in dialog's decorative frame.
# text -        Message to display in dialog.
# image -       image to display in dialog (empty string means none).
# type -        Can be:
#                        word:    word without spaces
#                        any:     anything
#                        real:    real
#                        real+:   positive real
#                        int:     int
#                        int+:    positive int
#                        text:    text with spaces and '\n'
#                        textro:  text with spaces and '\n' read-only
#                        string:  text with spaces but without '\n'
# default -    default value for entry
# args -        One or more strings to display in buttons across the
#                bottom of the dialog box.
#
# fill_extra_frame_proc is the name of a proc that expects a frame as argument and will fill it with the widgets he want

proc tk_dialogEntryRAM {w title text image type default {btnstxt ""} {fill_extra_frame_proc ""}} {
    global tkPriv GidPriv tcl_platform

    set image [ReplaceImageName $image]

    if { $btnstxt == "" } {
        set args [list [_ "Ok"] [_ "Cancel"]]
    } else {
        set args $btnstxt
    }

    # 1. Create the top-level window and divide it into top, middle
    # and bottom parts.

    while { [winfo exists $w] } { append w a }
    catch {destroy $w}
    toplevel $w
    if { $::tcl_platform(platform) == "windows" } {
        # does not work sometimes, i.e. the window does not appear with wm deiconify
        # and the process is blocked at tkwait visibility $w
        # wm transient $w [winfo toplevel [winfo parent $w]]
        wm attributes $w -toolwindow 1
    }

    wm title $w $title
    wm iconname $w Dialog
    wm protocol $w WM_DELETE_WINDOW [list event generate $w <Escape>]

    ttk::frame $w.bot -style BottomFrame.TFrame
    pack $w.bot -side bottom -fill both
    ttk::frame $w.top -style raised.TFrame -borderwidth 1

    if { $type != "text" && $type != "textro"} {
        pack $w.top -side top -fill both -expand 1
    } else {
        pack $w.top -side top -fill both
    }

    # ttk::frame $w.middle -style raised.TFrame -borderwidth 1
    ttk::frame $w.middle -borderwidth 1

    set is_number 0
    if { ($type == "int") || ($type == "int+") || ($type == "real") || ($type == "real+")} {
        set is_number 1
    }
    if { ($type != "text") && ($type != "textro")} {
        # int, int+, real, real+
        # word, any
        ttk::entry $w.middle.e -xscroll [list $w.middle.xscroll set] -textvariable ::GidPriv(entry)
        if { $type == "password" } {
            $w.middle.e configure -show *
        }

        $w.middle.e delete 0 end
        if { $default != "" } {
            #fis, if $default == "" then dont take care of return key
            $w.middle.e insert end $default
            $w.middle.e selection range 0 end
        }

        set T $w.middle.e
        set hsb [ttk::scrollbar $w.middle.xscroll -orient horizontal -command [list $w.middle.e xview]]
        #     pack $w.middle.xscroll -side bottom -fill x
        #     pack $w.middle.e -side top

        grid $w.middle.e -sticky ew -padx 2
        grid $w.middle.xscroll -sticky ew
        grid columnconfigure $w.middle 0 -weight 1

        if { $is_number} {
            pack $w.middle
        } else {
            pack $w.middle -expand 1 -fill both
        }

        grid remove $hsb
        bind $T <Configure> [list ConfigureListScrollbars $T $hsb ""]

    } else {
        # $type == "text" || $type == "textro"
        set ::GidPriv(entry) ""

        text $w.middle.t -relief sunken -width 40 -height 8 -yscroll [list $w.middle.yscroll set] -borderwidth 1
        $w.middle.t delete 0.0 end
        $w.middle.t insert end $default
        #$w.middle.t tag add selection 0.0 end
        if { $type == "textro"} {
            $w.middle.t configure -state disabled
            bind $w.middle.t <1> "focus $w.middle.t"
        }

        ttk::scrollbar $w.middle.yscroll -orient vertical -command [list $w.middle.t yview]

        # grid $w.middle.t $w.middle.yscroll -sticky ewns -ipady 3
        #
        # grid rowconfigure $w.middle 0 -weight 1
        # grid rowconfigure $w.middle 1 -weight 1
        # grid columnconfigure $w.middle 0 -weight 1
        # grid columnconfigure $w.middle 1 -weight 0
        pack $w.middle.t -side left -expand 1 -fill both
        pack $w.middle.yscroll -expand 1 -fill y

        pack $w.middle -expand 1 -fill both
    }

    # 2. Fill the top part with image and message (use the option
    # database for -wraplength so that it can be overridden by
    # the caller).

    #    option add *Dialog.msg.wrapLength 3i widgetDefault
    ttk::label $w.msg -justify left -text $text -wraplength 75m -font BigFont

    pack $w.msg -in $w.top -side right -expand 1 -fill both -padx 12 -pady 12
    if {$image != ""} {
        ttk::label $w.bitmap -image [gid_themes::GetImage $image large_icons]
        pack $w.bitmap -in $w.top -side left -padx 12 -pady 12
    }

    if { $type != "text" && $type != "textro"} {
        bind $w <Return> "
        update idletasks
        after 100
        set tkPriv(button) 0
        "
    }

    if { $fill_extra_frame_proc != "" } {                
        ttk::frame $w.f_more_options
        $fill_extra_frame_proc $w.f_more_options
        pack $w.f_more_options -fill x -padx 4 -pady 4
    }

    # 3. Create a row of buttons at the bottom of the dialog.

    set buttonslist ""
    set UnderLineList ""
    set i 0
    set default 0
    foreach but $args {
        regsub -all {([a-z0-9])([A-Z])} $but {\1 \2} but
    }
    foreach but $args {
        ttk::button $w.button$i -text $but -command [list set tkPriv(button) $i] -style BottomFrame.TButton
        lappend buttonslist $w.button$i
        pack $w.button$i -in $w.bot -side left -expand 1 -padx 12 -pady 8

        bind $w.button$i <Return> "
        update idletasks
        after 100
        set tkPriv(button) $i
        break
        "
        incr i
    }

    if { [llength $buttonslist] > 0 } {
        #set escape binding to right button
        bind $w <Escape> "
        update idletasks
        after 100
        set tkPriv(button) [expr [llength $buttonslist]-1]
        "

        #set buttons size acordind to current text
    }


    # 5. Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display and de-iconify it.

    if { [winfo parent $w] == ".gid" } {
        bind $w <Destroy> [list +DestroyWinInDialogs $w GeomDialog]
    }

    if { [info exists GidPriv(GeomDialog)] && [winfo parent $w] == ".gid"} {
        WmGidGeom  $w $GidPriv(GeomDialog)
    } else {
        wm withdraw $w
        update idletasks

        # extrange errors with updates
        if { ![winfo exists $w] } { return 0}

        set pare [winfo toplevel [winfo parent $w]]
        set x [expr [winfo x $pare]+[winfo width $pare ]/2- [winfo reqwidth $w]/2]
        set y [expr [winfo y $pare]+[winfo height $pare ]/2- [winfo reqheight $w]]
        if { $x < 0 } { set x 0 }
        if { $y < 0 } { set y 0 }
        WmGidGeom $w +$x+$y
        update
        # extrange errors with updates
        if { ![winfo exists $w] } { return 0}
        wm deiconify $w
    }

    # 6. Set a grab and claim the focus too.

    set oldFocus [focus]
    if { $::tcl_platform(os) != "Darwin"} {
        # with transient windows sometimes they do not appear and
        # the command below blocks the execution ...
        # tkwait visibility $w
    }
    set oldGrab [grab current $w]
    if {$oldGrab != ""} {
        set grabStatus [grab status $oldGrab]
    }

    # Make it topmost to avoid putting the window behind gid's.
    # wm attributes $w -topmost 1
    # raise $w
    # topmost forbidden
    # GidUtils::WindowAboveGid $w

    raise $w
    grab $w

    if { $type != "text" && $type != "textro" } {
        focus $w.middle.e
        $w.middle.e selection range 0 end
        after 100 [list catch [list focus -force $w.middle.e]]
    } else {
        focus $w.button0
        after 100 [list catch [list focus -force $w.button0]]
    }


    # 7. Wait for the user to respond, then restore the focus and
    # return the index of the selected button.  Restore the focus
    # before deleting the window, since otherwise the window manager
    # may take the focus away so we can't redirect it.  Finally,
    # restore any grab that was in effect.

    while 1 {
        tkwait variable tkPriv(button)
        if { $tkPriv(button) == 1} { break }
        if { $type != "text" && $type != "textro" } {
            set result [$w.middle.e get]
        } else {
            set result [$w.middle.t get 0.0 end]
        }
        switch $type {
            word {
                set result [string trim $result]
                if { [string first { } $result] != -1 } {
                    set errorRes [_ "A single word must be entered"]
                }
            }
            string {
                set result [string trim $result]
            }
            real {
                if { ![regexp {^[  ]*([+-]?[0-9]*\.?[0-9]*([eE][+-]?[0-9]+)?)[  ]*$} $result] } {
                    set errorRes [_ "One number must be entered"]
                }
            }
            real+ {
                if { ![regexp {^[  ]*([+]?[0-9]*\.?[0-9]*([eE][+-]?[0-9]+)?)[  ]*$} $result] } {
                    set errorRes [_ "One positive number must be entered"]
                }
            }
            int {
                if { ![regexp {^[  ]*[+-]?[0-9]+[  ]*$} $result] } {
                    set errorRes [_ "One integer number must be entered"]
                }
            }
            int+ {
                if { ![regexp {^[  ]*[+]?[0-9]*[1-9]+[0-9]*[  ]*$} $result] } {
                    set errorRes [_ "One positive integer number must be entered"]
                }
            }
            text {
                set len [string length $result]
                # para quitar el \n que anyade el get de antes al final.
                if { $len >= 2} {
                    set result [string range $result 0 [expr $len - 2]]
                } else {
                    set result "--NULL--"
                }
                if { ($result == "\"\"") } {
                    set result "--NULL--"
                }
                set ::GidPriv(entry) $result
            }
        }
        if { [info exists errorRes] } {
            WarnWin $errorRes
            unset errorRes
        } else {
            break
        }
    }

    if { $oldFocus == "" || [catch {focus $oldFocus}] } {
        if { [winfo exists .gid] } {
            focus [focus -lastfor .gid]
        }
    }
    destroy $w
    if {$oldGrab != ""} {
        if {$grabStatus == "global"} {
            grab -global $oldGrab
        } else {
            grab $oldGrab
        }
    }
    if { $tkPriv(button) == 1} { return --CANCEL-- }
    if { ![info exists ::GidPriv(entry)] } {
        set ::GidPriv(entry) ""
    }
    return $::GidPriv(entry)
}

# tk_dialogQueryReplaceRAM
#
# This procedure displays a dialog box, with two entries,
# one for the query and another for the replace string
# string: text with spaces but without '\n'
# and waits for a button in
# the dialog to be invoked,
# then returns a list with the two values of the entries
#              and a third optional parameter 'nocase'
#              for word/any/text/string if user checked the option
# If cancel is pressed, it returns: --CANCEL--
#
# Arguments:
# w -                Window to use for dialog top-level.
# title -        Title to display in dialog's decorative frame.
# text -        Message to display in dialog.
# image -       image to display in dialog (empty string means none).
#                 may be one of question, info, message, warning or error
# type -        Can be:
#                        word:    word without spaces
#                        any:     anything
#                        real:    real
#                        real+:   positive real
#                        int:     int
#                        int+:    positive int
#                        text:    text with spaces and '\n'
#                        textro:  text with spaces and '\n' read-only
#                        string:  text with spaces but without '\n'
# lst_defaults - list of two default values for the entries
# lst_label_entries - list of labels that prepend each entry
# args -         One or more strings to display in buttons across the
#                bottom of the dialog box.
# -np- W [tk_dialogQueryReplaceRAM .gid.tt try-1 "Enter query and replace stings" question string [list {} {}] [list queryString replaceString]]
proc tk_dialogQueryReplaceRAM {w title text image type lst_defaults lst_label_entries {btnstxt ""} } {
    global tkPriv GidPriv tcl_platform

    set image [ReplaceImageName $image]

    if { $btnstxt == "" } {
        set args [list [_ "Ok"] [_ "Cancel"]]
    } else {
        set args $btnstxt
    }

    # 1. Create the top-level window and divide it into top, middle
    # and bottom parts.

    while { [winfo exists $w] } { append w a }
    catch {destroy $w}
    toplevel $w
    if { $::tcl_platform(platform) == "windows" } {
        # does not work sometimes, i.e. the window does not appear with wm deiconify
        # and the process is blocked at tkwait visibility $w
        # wm transient $w [winfo toplevel [winfo parent $w]]
        wm attributes $w -toolwindow 1
    }

    wm title $w $title
    wm iconname $w Dialog
    wm protocol $w WM_DELETE_WINDOW [list event generate $w <Escape>]

    ttk::frame $w.bot -style BottomFrame.TFrame
    pack $w.bot -side bottom -fill both
    ttk::frame $w.top -style raised.TFrame -borderwidth 1

    if { $type != "text" && $type != "textro"} {
        pack $w.top -side top -fill both -expand 1
    } else {
        pack $w.top -side top -fill both
    }
    
    ttk::frame $w.middle -borderwidth 1

    set is_number 0
    if { ($type == "int") || ($type == "int+") || ($type == "real") || ($type == "real+")} {
        set is_number 1
    }
    set lst_suffixes [list _query _replace]
    set lst_labels $lst_label_entries
    if { [llength $lst_label_entries] != 2} {
        set lst_labels [list [_ "Query %s" $type] [_ "Replace %s" $type] ]
    }
    set first_suffix [lindex $lst_suffixes 0]
    set default([lindex $lst_suffixes 0]) [lindex $lst_defaults 0]
    set default([lindex $lst_suffixes 1]) [lindex $lst_defaults 1]

    set case_checkbox_placed 0
    foreach suffix $lst_suffixes lbl $lst_labels {
        if { ($type != "text") && ($type != "textro")} {
            # int, int+, real, real+, word, any
            ttk::label $w.middle.l${suffix} -text ${lbl}:
            ttk::entry $w.middle.e${suffix} -xscroll [list $w.middle.xscroll${suffix} set] -textvariable ::GidPriv(entry${suffix})
            if { $type == "password" } {
                $w.middle.e${suffix} configure -show *
            }

            $w.middle.e${suffix} delete 0 end
            if { $default($suffix) != "" } {
                #fix if $default == "" don't take care of return keys
                $w.middle.e${suffix} insert end $default(${suffix})
                $w.middle.e${suffix} selection range 0 end
            }

            set T $w.middle.e${suffix}
            set hsb [ttk::scrollbar $w.middle.xscroll${suffix} -orient horizontal -command [list $w.middle.e${suffix} xview]]

            grid $w.middle.l${suffix} $w.middle.e${suffix} -sticky ew -padx 2
            grid configure $w.middle.l${suffix} -sticky e
            grid $w.middle.xscroll${suffix} -sticky ew -column 1
            if { !$is_number && !$case_checkbox_placed } {
                if { ![info exists ::GidPriv(entry_nocase)] } {
                    set ::GidPriv(entry_nocase) 0
                }
                ttk::checkbutton $w.middle.l_nocase -text [_ "Ignore case"] -variable ::GidPriv(entry_nocase)
                grid $w.middle.l_nocase -sticky w -column 1
                set case_checkbox_placed 1
            }
            grid columnconfigure $w.middle 1 -weight 1
            grid remove $hsb
            bind $T <Configure> [list ConfigureListScrollbars $T $hsb ""]
        } else {
            # $type == "text" || $type == "textro"
            set ::GidPriv(entry${suffix}) ""
            ttk::label $w.middle.l${suffix} -text ${lbl}:
            text $w.middle.t${suffix} -relief sunken -width 40 -height 8 -yscroll [list $w.middle.yscroll${suffix} set] -borderwidth 1
            $w.middle.t${suffix} delete 0.0 end
            $w.middle.t${suffix} insert end $default(${suffix})
            if { $type == "textro"} {
                $w.middle.t${suffix} configure -state disabled
                bind $w.middle.t${suffix} <1> "focus $w.middle.t${suffix}"
            }
            ttk::scrollbar $w.middle.yscroll${suffix} -orient vertical -command [list $w.middle.t${suffix} yview]
            grid $w.middle.l${suffix} -sticky nw
            grid $w.middle.t${suffix} $w.middle.yscroll${suffix} -sticky news
            grid configure $w.middle.yscroll${suffix} -sticky nse
            if { !$is_number && !$case_checkbox_placed} {
                if { ![info exists ::GidPriv(entry_nocase)]} {
                    set ::GidPriv(entry_nocase) 0
                }
                ttk::checkbutton $w.middle.l_nocase -text [_ "Ignore case"] -variable ::GidPriv(entry_nocase)
                grid $w.middle.l_nocase -sticky w
                set case_checkbox_placed 1
            }

            grid columnconfigure $w.middle 0 -weight 1
        }
        # end for suffix
    }

    if { $is_number} {
        pack $w.middle
    } else {
        pack $w.middle -expand 1 -fill both
    }

    # 2. Fill the top part with image and message (use the option
    # database for -wraplength so that it can be overridden by
    # the caller).

    #    option add *Dialog.msg.wrapLength 3i widgetDefault
    ttk::label $w.msg -justify left -text $text -wraplength 75m -font BigFont

    pack $w.msg -in $w.top -side right -expand 1 -fill both -padx 12 -pady 12
    if { $image != "" } {
        ttk::label $w.bitmap -image [gid_themes::GetImage $image large_icons]
        pack $w.bitmap -in $w.top -side left -padx 12 -pady 12
    }

    if { $type != "text" && $type != "textro"} {
        bind $w <Return> "
        update idletasks
        after 100
        set tkPriv(button) 0
        "
    }

    # 3. Create a row of buttons at the bottom of the dialog.

    set buttonslist ""
    set UnderLineList ""
    set i 0
    # set default 0
    foreach but $args {
        regsub -all {([a-z0-9])([A-Z])} $but {\1 \2} but
    }
    foreach but $args {
        ttk::button $w.button$i -text $but -command [list set tkPriv(button) $i] -style BottomFrame.TButton
        lappend buttonslist $w.button$i
        pack $w.button$i -in $w.bot -side left -expand 1 -padx 12 -pady 8

        bind $w.button$i <Return> "
        update idletasks
        after 100
        set tkPriv(button) $i
        break
        "
        incr i
    }

    if { [llength $buttonslist] > 0 } {
        #set escape binding to right button
        bind $w <Escape> "
        update idletasks
        after 100
        set tkPriv(button) [expr [llength $buttonslist]-1]
        "
        #set buttons size acordind to current text
    }


    # 5. Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display and de-iconify it.

    wm withdraw $w
    update idletasks

    # extrange errors with updates
    if { ![winfo exists $w] } {
        return 0
    }

    set pare [winfo toplevel [winfo parent $w]]
    set x [expr [winfo x $pare]+[winfo width $pare ]/2- [winfo reqwidth $w]/2]
    set y [expr [winfo y $pare]+[winfo height $pare ]/2- [winfo reqheight $w]]
    if { $x < 0 } {
        set x 0
    }
    if { $y < 0 } {
        set y 0
    }
    WmGidGeom $w +$x+$y
    update
    # extrange errors with updates
    if { ![winfo exists $w] } { return 0}
    wm deiconify $w

    # 6. Set a grab and claim the focus too.

    set oldFocus [focus]
    if { $::tcl_platform(os) != "Darwin"} {
        # with transient windows sometimes they do not appear and
        # the command below blocks the execution ...
        # tkwait visibility $w
    }
    set oldGrab [grab current $w]
    if {$oldGrab != ""} {
        set grabStatus [grab status $oldGrab]
    }

    # Make it topmost to avoid putting the window behind gid's.
    # wm attributes $w -topmost 1
    # raise $w
    # topmost forbidden
    # GidUtils::WindowAboveGid $w

    raise $w
    grab $w

    if { $type != "text" && $type != "textro" } {
        focus $w.middle.e${first_suffix}
        $w.middle.e${first_suffix} selection range 0 end
        after 100 [list catch [list focus -force $w.middle.e${first_suffix}]]
    } else {
        focus $w.button0
        after 100 [list catch [list focus -force $w.button0]]
    }

    # 7. Wait for the user to respond, then restore the focus and
    # return the index of the selected button.  Restore the focus
    # before deleting the window, since otherwise the window manager
    # may take the focus away so we can't redirect it.  Finally,
    # restore any grab that was in effect.

    while 1 {
        tkwait variable tkPriv(button)
        if { $tkPriv(button) == 1} { break }
        foreach suffix $lst_suffixes {
            if { $type != "text" && $type != "textro" } {
                set result($suffix) [$w.middle.e$suffix get]
            } else {
                set result($suffix) [$w.middle.t$suffix get 0.0 end]
            }
            switch $type {
                word {
                    set result($suffix) [string trim $result($suffix)]
                    if { [string first { } $result($suffix)] != -1 } {
                        set errorRes [_ "A single word must be entered"]
                    }
                }
                string {
                    set result($suffix) [string trim $result($suffix)]
                }
                real {
                    if { ![regexp {^[  ]*([+-]?[0-9]*\.?[0-9]*([eE][+-]?[0-9]+)?)[  ]*$} $result($suffix)] } {
                        set errorRes [_ "One number must be entered"]
                    }
                }
                real+ {
                    if { ![regexp {^[  ]*([+]?[0-9]*\.?[0-9]*([eE][+-]?[0-9]+)?)[  ]*$} $result($suffix)] } {
                        set errorRes [_ "One positive number must be entered"]
                    }
                }
                int {
                    if { ![regexp {^[  ]*[+-]?[0-9]+[  ]*$} $result($suffix)] } {
                        set errorRes [_ "One integer number must be entered"]
                    }
                }
                int+ {
                    if { ![regexp {^[  ]*[+]?[0-9]*[1-9]+[0-9]*[  ]*$} $result($suffix)] } {
                        set errorRes [_ "One positive integer number must be entered"]
                    }
                }
                text {
                    set len [string length $result($suffix)]
                    # para quitar el \n que anyade el get de antes al final.
                    if { $len >= 2} {
                        set result($suffix) [string range $result($suffix) 0 [expr $len - 2]]
                    } else {
                        set result($suffix) "--NULL--"
                    }
                    if { ($result($suffix) == "\"\"") } {
                        set result($suffix) "--NULL--"
                    }
                    set ::GidPriv(entry$suffix) $result($suffix)
                }
            }
        }
        if { [info exists errorRes] } {
            WarnWin $errorRes
            unset errorRes
        } else {
            break
        }
    }

    if { $oldFocus == "" || [catch {focus $oldFocus}] } {
        ## if { [winfo exists .gid] } {
        ##     focus [focus -lastfor .gid]
        ## }
    }
    destroy $w
    if {$oldGrab != ""} {
        if {$grabStatus == "global"} {
            grab -global $oldGrab
        } else {
            grab $oldGrab
        }
    }
    if { $tkPriv(button) == 1} { return --CANCEL-- }

    set ret_value [list]
    foreach suffix $lst_suffixes {
        if { ![info exists ::GidPriv(entry$suffix)] } {
            set ::GidPriv(entry$suffix) ""
        }
        lappend ret_value $::GidPriv(entry$suffix)
    }
    if { $case_checkbox_placed && $::GidPriv(entry_nocase)} {
        lappend ret_value nocase
    }
    return $ret_value
}

#
# This procedure displays a dialog box, with a combobox, waits for a button in
# the dialog to be invoked, then returns the entry.
# If cancel is pressed, it returns: --CANCEL--
#
# Arguments:
# w -                Window to use for dialog top-level.
# title -        Title to display in dialog's decorative frame.
# text -        Message to display in dialog.
# image -       Image to display in dialog (empty string means none).
# AcceptNew     Can be: TCL_BOOLEAN or 2 (then, word must begin with a letter)
#                       3 (then, one positive number must be entered)
#                       4 any word (including spaces)
#
# args some words to add to the combo
#

proc tk_dialogComboRAM {w title text image AcceptNew args} {
    global tkPriv GidPriv tcl_platform

    set image [ReplaceImageName $image]

    # 1. Create the top-level window and divide it into top, middle
    # and bottom parts.

    while { [winfo exists $w] } { append w a }
    catch {destroy $w}
    toplevel $w
    if { $::tcl_platform(platform) == "windows" } {
        wm transient $w [winfo toplevel [winfo parent $w]]
        wm attributes $w -toolwindow 1
    }

    wm title $w $title
    wm iconname $w Dialog
    wm protocol $w WM_DELETE_WINDOW [list event generate $w <Escape>]

    ttk::frame $w.bot -style BottomFrame.TFrame
    pack $w.bot -side bottom -fill both
    ttk::frame $w.top -style raised.TFrame
    pack $w.top -side top -fill both -expand 1

    ttk::frame $w.middle -style raised.TFrame

    set ::GidPriv(entry) ""
    set iseditable $AcceptNew
    if { $iseditable > 1 } { set  iseditable 1 }
    if { ![info exists ::GidPriv(entry)] } {
        set ::GidPriv(entry) ""
    }
    ComboBox $w.middle.e -editable $iseditable -textvariable ::GidPriv(entry)

    set last ""
    set values ""
    foreach i [lindex $args 0] {
        if { $i == $last } { continue }
        lappend values $i
        set last $i
    }
    $w.middle.e configure -values $values

    if { $AcceptNew == 0 || $AcceptNew == 3 } {
        set ::GidPriv(entry) [lindex [lindex $args 0] 0]
    }

    pack $w.middle.e -side top -padx 30 -fill x -expand 1 -pady 5
    pack $w.middle -fill x -expand 1 -pady 5


    # 2. Fill the top part with image and message (use the option
    # database for -wraplength so that it can be overridden by
    # the caller).

    option add *Dialog.msg.wrapLength 3i widgetDefault
    ttk::label $w.msg -justify left -text $text -wraplength 75m -font BigFont

    pack $w.msg -in $w.top -side right -expand 1 -fill both -padx 12 -pady 12
    if {$image != ""} {
        ttk::label $w.bitmap -image [gid_themes::GetImage $image large_icons]
        pack $w.bitmap -in $w.top -side left -padx 12 -pady 12
    }

    bind $w <Return> "
    update idletasks
    after 100
    set tkPriv(button) 0
    "

    # 3. Create a row of buttons at the bottom of the dialog.

    set buttonslist ""
    set UnderLineList ""
    set i 0

    set default 0
    set args [list [_ "Ok"] [_ "Cancel"]]
    foreach but $args {
        regsub -all {([a-z0-9])([A-Z])} $but {\1 \2} but
    }
    foreach but $args {
        ttk::button $w.button$i -text $but -command [list set tkPriv(button) $i] -style BottomFrame.TButton
        lappend buttonslist $w.button$i
        pack $w.button$i -in $w.bot -side left -expand 1 -padx 12 -pady 8
        bind $w.button$i <Return> "
        update idletasks
        after 100
        set tkPriv(button) $i
        break
        "
        incr i
    }

    if { [llength $buttonslist] > 0 } {
        #set escape binding to right button
        bind $w <Escape> "
        update idletasks
        after 100
        set tkPriv(button) [expr [llength $buttonslist]-1]
        "

        #set buttons size acordind to current text
    }


    # 5. Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display and de-iconify it.

    if { [winfo parent $w] == ".gid" } {
        bind $w <Destroy> [list +DestroyWinInDialogs $w GeomDialog]
    }
    if { [info exists GidPriv(GeomDialog)] && [winfo parent $w] == ".gid" } {
        WmGidGeom $w $GidPriv(GeomDialog)
    } else {
        wm withdraw $w
        update idletasks
        # extrange errors with updates
        if { ![winfo exists $w] } { return 0}

        set pare [winfo toplevel [winfo parent $w]]
        set x [expr [winfo x $pare]+[winfo width $pare ]/2- [winfo reqwidth $w]/2]
        set y [expr [winfo y $pare]+[winfo height $pare ]/2- [winfo reqheight $w]]
        if { $x < 0 } { set x 0 }
        if { $y < 0 } { set y 0 }
        WmGidGeom $w +$x+$y
        update
        # extrange errors with updates
        if { ![winfo exists $w] } { return 0}
        wm deiconify $w
    }

    # 6. Set a grab and claim the focus too.

    set oldFocus [focus]
    if { $::tcl_platform(os) != "Darwin"} {
        tkwait visibility $w
    } else {
        update
    }
    set oldGrab [grab current $w]
    if {$oldGrab != ""} {
        set grabStatus [grab status $oldGrab]
    }

    # Make it topmost to avoid putting the window behind gid's.
    # wm attributes $w -topmost 1
    # raise $w
    # topmost forbidden
    # GidUtils::WindowAboveGid $w

    raise $w
    grab $w

    # ramsan 6/02
    focus $w.middle.e
    update idletasks
    event generate $w.middle.e <<TraverseIn>>

    # 7. Wait for the user to respond, then restore the focus and
    # return the index of the selected button.  Restore the focus
    # before deleting the window, since otherwise the window manager
    # may take the focus away so we can't redirect it.  Finally,
    # restore any grab that was in effect.

    while 1 {
        tkwait variable tkPriv(button)
        if { $tkPriv(button) == 1} {
            break
        }
        set result $::GidPriv(entry)
        if { $AcceptNew != 0 && $AcceptNew != 4 } {
            #if not editable not check is word is single
            if { ![regexp {^[          ]*[^ ]+[  ]*$} $result] } {
                set errorRes [_ "A single word must be entered"]
            }
        }
        if { $AcceptNew == 2 && ![regexp {^[a-zA-Z]} $result] } {
            set errorRes [_ "Word must begin with a letter"]
        } elseif { $AcceptNew == 3 && ![regexp {^[  ]*([+]?[0-9]*\.?[0-9]*([eE][+-]?[0-9]+)?)[  ]*$} $result] } {
            set errorRes [_ "One positive number must be entered"]
        }
        if { [info exists errorRes] } {
            WarnWin $errorRes
            unset errorRes
        } else {
            break
        }
    }

    catch {focus $oldFocus}
    destroy $w
    if {$oldGrab != ""} {
        if {$grabStatus == "global"} {
            grab -global $oldGrab
        } else {
            grab $oldGrab
        }
    }
    if { $tkPriv(button) == 1} { return --CANCEL-- }
    if { ![info exists ::GidPriv(entry)] } {
        set ::GidPriv(entry) ""
    }
    return $::GidPriv(entry)
}

proc WarnWinTextWidgetConfigureFont { wt { new_fnt ""}} {
    set def_fnt SmallFont
    if { [info exists ::GidPriv(WarnWinTextFontUsed)] } {
        set def_fnt $::GidPriv(WarnWinTextFontUsed)
    }
    if { $new_fnt == ""} {
        set new_fnt $def_fnt
    }
    set ::GidPriv(WarnWinTextFontUsed) $new_fnt
    $wt configure -font $new_fnt
}

proc WarnWinTextFontSizeIncrease { wt increment} {
    set WWFontName WarnWinTextFont
    set act_font [$wt cget -font]
    # by default it uses SmallFont
    # set now as Custom font
    if { $act_font != $WWFontName} {
        # recreate WarnWinTextFont
        if { [lsearch [font names] $WWFontName] != -1} {
            font delete $WWFontName
        }
    }
    if { [lsearch [font names] $WWFontName] == -1} {
        catch {
            font create $WWFontName
        }
        font configure $WWFontName {*}[font configure [$wt cget -font]]
    }
    set old_size [font configure $WWFontName -size]
    set err [catch  {
        font configure $WWFontName -size [expr int( 0.5 + $old_size + $increment)]
    } err_txt]
    if { $act_font != $WWFontName} {
        $wt configure -font $WWFontName
        WarnWinTextWidgetConfigureFont $wt $WWFontName
    }
}

proc WarnWinTextGetDefaultFont { } {
    # return SmallFont or user configure font
    set def_fnt ""
    foreach ff [list SmallFont NormalFont TkDefaultFont] {
        if { [lsearch [font names] $ff] != -1} {
            set def_fnt $ff
            break
        }
    }
    return $def_fnt
}

proc WarnWinTextGetWarnWinTextFont { } {
    # return SmallFont or user configure font
    set def_fnt [WarnWinTextGetDefaultFont]
    set WWFontName WarnWinTextFont
    if { [lsearch [font names] $WWFontName] != -1} {
        set def_fnt $WWFontName
    }
    if { [info exists ::GidPriv(WarnWinTextFontUsed)] } {
        set def_fnt $::GidPriv(WarnWinTextFontUsed)
    }
    return $def_fnt
}

proc WarnWinTextFixedFont { wt varName} {
    set fixed [set $varName]
    if { $fixed} {
        WarnWinTextWidgetConfigureFont $wt FixedFont
    } else {
        WarnWinTextWidgetConfigureFont $wt [WarnWinTextGetWarnWinTextFont]
    }
}

proc WarnWinTextResetFont { wt} {
    WarnWinTextWidgetConfigureFont $wt [WarnWinTextGetDefaultFont]
}

proc WWTGetLinuxEditorList {} {
    if { ![info exists ::WWTGetLinuxEditorList_edt_lst]} {
        set ::WWTGetLinuxEditorList_edt_lst {}
        foreach block [list {code "VS Code" ""} {gedit gedit ""} {kwrite kwrite ""} {kate Kate ""} {mousepad mousepad ""} {xedit xedit ""} {emacs emacs ""}] {
            lassign $block editor name fmt
            set full_editor ""
            set err [catch { set full_editor [exec which $editor]}]
            if { $full_editor != ""} {
                lappend ::WWTGetLinuxEditorList_edt_lst [list $editor $name $fmt]
            }
        }
    }
    return $::WWTGetLinuxEditorList_edt_lst
}

proc WWTGetWindowsEditorList {} {
    if { ![info exists ::WWTGetWindowsEditorList_edt_lst]} {
        set ::WWTGetWindowsEditorList_edt_lst {}
        foreach block [list {code "VS Code" ""} {notepad.exe Notepad ""} {emacs Emacs ""} {write.exe "Write" ""} {scitex Scitex ""} ] {
            lassign $block editor name fmt
            set full_editor ""
            set err [catch { set full_editor [exec where $editor]} err_txt]
            if { !$err} {
                set list_of_paths [split $full_editor \n]
                set full_editor [lindex $list_of_paths 0]
            }
            if { $full_editor != ""} {
                lappend ::WWTGetWindowsEditorList_edt_lst [list $editor $name $fmt]
            }
        }
    }
    return $::WWTGetWindowsEditorList_edt_lst
}

proc WWTGetEditorList {} {
    if { $::tcl_platform(platform) == "windows" } {
        return [WWTGetWindowsEditorList]
    } else {
        return [WWTGetLinuxEditorList]
    }
}

proc WarnWinTextOpenInEditor { wt editor {extension ""}} {
    if { [winfo class $wt] == "Shtmlview"} {
        $wt sourceView
        set txt [[$wt getTextWidget] get 0.0 end]
        $wt render $txt
    } else {
        set txt [$wt get 0.0 end]
    }
    if {$extension == ""} {
        set extension [GuessFormatFromContent $txt]
    }
    set W_text_file [gid_cross_platform::get_unused_tmp_filename gid-dialog $extension]

    set fo [open $W_text_file w]
    puts $fo $txt
    close $fo


    if { $editor == "--INTERNAL--"} {
        package require texteditor
        TextEditor::Create $W_text_file
    } else {
        package require gid_cross_platform
        if { [catch { gid_cross_platform::execute_program_in_path $editor $W_text_file } error]} {
            after 5000 [list file delete $W_text_file]
            return -code error "couldn't execute '$editor': $error"
        }
    }
    if { $editor != "code"} {
        # code does an automatic refresh, so when the file is deleted, it disappears from code too!!!
        after 5000 [list file delete  $W_text_file]
    }
}

proc GuessFormatFromContent { txt } {
    # Empty for .TMP
    set fmt ""
    #simple check, the same of  fileutil::fileType (tcllib fileutil package)
    if { [string match -nocase "*\<\?xml*" $txt] } {
        set fmt .xml
    } else {
        #check if it is a valid xml
        package require tdom
        if { ![catch { set document [dom parse -simple $txt] } err] } {
            $document delete
            set fmt .xml
        } else {

        }
    }
    return $fmt
}

proc WarnWinTextOpenInBrowser { wt {which ""}} {
    set W_text_file ""
    set fo [file tempfile W_text_file WtmpContents]
    close $fo
    file delete $W_text_file
    append W_text_file .html
    set fo [open $W_text_file w]
    puts $fo [$wt get 0.0 end]
    close $fo

    package require gid_cross_platform
    if { $which == ""} {
        gid_cross_platform::open_by_extension $W_text_file
    } else {
        if { [catch { gid_cross_platform::execute_program_in_path $which $W_text_file } error]} {
            after 5000 [list file delete $W_text_file]
            return -code error "couldn't execute '$editor': $error"
        }
    }
    if { $which != "code"} {
        # code does an automatic refresh, so when the file is deleted, it disappears from code too!!!
        after 5000 [list file delete $W_text_file]
    }
}

proc WarnWinTextRightButtonMenu { w } {
    # Just a variable name
    set new_w_id $w.new_id

    set mfont [menu $w.m -tearoff 0]
    # $mfont add label -text "Font options:"
    $mfont add command -label [_ "Font size ++"] -command [list ::WarnWinTextFontSizeIncrease $w 2]
    $mfont add command -label [_ "Font size --"] -command [list ::WarnWinTextFontSizeIncrease $w -2]
    $mfont add checkbutton -label [_ "Fixed size"] \
        -variable ::WarnWinText($new_w_id) \
        -command [list ::WarnWinTextFixedFont $w ::WarnWinText($new_w_id)]
    $mfont add command -label [_ "Font reset"] -command [list ::WarnWinTextResetFont $w]

    set meditor [menu $w.m.me -tearoff 0]
    foreach block [WWTGetEditorList] {
        lassign $block ed name fmt
        $meditor add command -label $name -command [list ::WarnWinTextOpenInEditor $w $ed $fmt]
    }
    $meditor add command -label "GiD's editor" -command [list ::WarnWinTextOpenInEditor $w --INTERNAL--]
    $meditor add separator
    $meditor add command -label "Browser" -command [list ::WarnWinTextOpenInBrowser $w]
    $mfont add cascade -label [_ "Open in "] -menu $meditor

    if { [info exists ::gid_right_button] } {
        bind $w <$::gid_right_button> [list GiD_PopupMenu $mfont %X %Y]
    }
    return $mfont
}

proc WarnWinText { text { title "Warning"} { translate 0} { no_new_line 0} { new_w_id .warning}} {
    if { [GidUtils::IsTkDisabled] } {
        #e.g. batch mode without windows
        return
    }
    # translate the default value
    if { ($title == "Warning") && $translate } {
        set title [_ "Warning"]
    }

    if { [string length $new_w_id] == 0} {
        set new_w_id .warning
    }
    if { [string index $new_w_id 0] != "."} {
        set new_w_id .$new_w_id
    }

    if { [winfo exists .gid] } {
        set w .gid$new_w_id
    } else {
        set w $new_w_id
    }
    if { ! [winfo exists $w] } {
        if { 0 && [info procs InitWindow] != "" } {
            #not using InitWindow because disappear when changing from pre to post
            #but maybe now is fixed, the problem wat the must start by PrePost*
            InitWindow2 $w -title $title -geometryvariable PrePostWarnWinTextWindowGeom
            # -ontop ;# eventually
            if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0
        } else {
            toplevel $w
            set wpar [winfo parent $w]
            if { $wpar != "."} {
                wm transient $w [winfo toplevel $wpar]
            }
            # if { $::tcl_platform(platform) == "windows" } {
            #     wm attributes $w -toolwindow 1
            # }
            wm title $w $title
        }
        #text with scrolls only if required
        ttk::frame $w.fr
        ttk::scrollbar $w.fr.scrolly -command [list $w.fr.t yview] -orient vertical
        ttk::scrollbar $w.fr.scrollx -command [list $w.fr.t xview] -orient horizontal
        text $w.fr.t -yscrollcommand [list $w.fr.scrolly set] -xscrollcommand [list $w.fr.scrollx set] -wrap none
        # Configure with last options set for font:
        WarnWinTextWidgetConfigureFont $w.fr.t [WarnWinTextGetWarnWinTextFont]

        grid $w.fr.t -row 1 -column 1 -sticky nsew
        grid $w.fr.scrolly -row 1 -column 2 -sticky ns
        grid $w.fr.scrollx -row 2 -column 1 -sticky ew
        grid rowconfigure $w.fr 1 -weight 1
        grid columnconfigure $w.fr 1 -weight 1
        grid $w.fr -sticky nsew -padx 5 -pady 5

        grid remove $w.fr.scrolly
        grid remove $w.fr.scrollx
        bind $w.fr.t <Configure> "ConfigureListScrollbars $w.fr.t $w.fr.scrollx $w.fr.scrolly"

        #lower buttons
        ttk::frame $w.frmButtons -style BottomFrame.TFrame
        set close_txt "Close"
        if { $translate } {
            set close_txt [_ "Close"]
        }
        ttk::button $w.frmButtons.btnclose -text $close_txt \
            -command "destroy $w" \
            -underline 0 -style BottomFrame.TButton

        grid $w.frmButtons -sticky ews -columnspan 7
        grid anchor $w.frmButtons center
        grid $w.frmButtons.btnclose -padx 5 -pady 6
        grid columnconfigure $w.frmButtons 0 -weight 1
        grid rowconfigure $w.frmButtons 0 -weight 1
        grid columnconfigure $w 0 -weight 1
        grid rowconfigure $w 0 -weight 1



        # right mouse button menu on text widget
        set mfont [WarnWinTextRightButtonMenu $w.fr.t]
        focus $w.frmButtons.btnclose
        bind $w <Alt-c> [list $w.frmButtons.btnclose invoke]
        bind $w <Escape> [list $w.frmButtons.btnclose invoke]

        # Other global options
        if { [info exists ::WarnWinTextOptions] && ($::WarnWinTextOptions != "")} {
            $w.fr.t configure {*}$::WarnWinTextOptions
        }
    }

    set c "\n"
    if { $no_new_line} {
        set c ""
    }
    $w.fr.t insert end "$text$c"
    $w.fr.t see end
    ConfigureListScrollbars $w.fr.t $w.fr.scrollx $w.fr.scrolly
    # commented updated to avoid error when doing
    # Utilities --> Tools --> Develop -> Console Python...
    # import tohil; tohil.call( 'W', 'hola')
    # update
    # do not call directly update or update idletasks
    # update idletasks forces one loop over pending events
    # that causes GLWidget-OpenGL.cc : WWT0D_OutputPendingMessages()
    # to be blocked
    after idle "update idletasks; raise $w"
}

proc CloseWarnWinText { } {
    set w .gid.warning
    if { [winfo exists $w] } {
        destroy $w
    } else {
        set w .warning
        if { [winfo exists $w] } {
            destroy $w
        }
    }
}

proc W { text { title "Warning"} } {
    WarnWinText $text $title
}

proc W2 { text { title "Warning"} } {
    # default values
    set translate 0
    set no_new_line 0
    set new_w_id .warning2
    WarnWinText $text "$title (2)" $translate $no_new_line $new_w_id
}

proc WV { varname_list { indent_txt ""} } {
    set text [list]
    set prefix $indent_txt
    # $indent_txt will be added on each variable entry so that it can be used as indentation
    # if it does not end with a space, it will be added
    if { $indent_txt != ""} {
        set prefix $indent_txt
        if { [string index $indent_txt end] != " "} {
            set prefix "$indent_txt "
        }
    }
    foreach varname $varname_list {
        upvar $varname contents
        set err [catch {
            set len [llength $contents]
            if { $len == 1} {
                set text_var "$prefix$varname = $contents"
            } else {
                set text_var "$prefix$varname ([llength $contents]) = $contents"
            }
            if { $contents == ""} {
                set text_var "$prefix$varname is empty"
            }
        } err_txt]
        if { $err} {
            set text_var "can't read '$varname': no such variable"
        }
        lappend text $text_var
    }
    WarnWinText [join $text \n]
}

proc Warray { arrayname} {
    upvar $arrayname tbl

    foreach {key val} [array get tbl] {
        W "${arrayname}($key) = $val"
    }
}

proc Wmatrix { varname_list { msg ""} } {
    foreach varname $varname_list {
        upvar $varname contents
        set err [catch {
            set len [llength $contents]
            if { $len == 1} {
                set text_var "$varname = $contents"
            } else {
                # matrix
                set len_2 [expr sqrt($len)]
                set text_var [list "$varname ([llength $contents] = ${len_2}x${len_2} matrix) ="]
                set idx 0
                for { set f 0} { $f < $len_2} { incr f} {
                    set mat_file ""
                    for { set c 0} { $c < $len_2} { incr c} {
                        append mat_file "[lindex $contents $idx] \t"
                        incr idx
                    }
                    lappend text_var $mat_file
                }
            }
            if { $contents == ""} {
                set text_var "'$varname' is empty"
            }
        } err_txt]
        if { $err} {
            set text_var "can't read '$varname': no such variable"
        }
        if { $msg == ""} {
            WarnWinText [join $text_var \n]
        } else {
            WarnWinText "$msg [join $text_var \n]"
        }
    }
}

proc W_MarkDownRightButtonMenu { w } {
    # Just a variable name
    set new_w_id $w.new_id
    set m [menu $w.m -tearoff 0]
    if { 1 } {
        set mfont $m
        # $mfont add label -text "Font options:"
        $mfont add command -label [_ "Font size ++"] -command [list $w setSize +1]
        $mfont add command -label [_ "Font size --"] -command [list $w setSize -1]
        if { 0 } {
            $mfont add checkbutton -label [_ "Fixed size"] \
                -variable ::WarnWinText($new_w_id) \
                -command [list ::WarnWinTextFixedFont $w ::WarnWinText($new_w_id)]
            $mfont add command -label [_ "Font reset"] -command [list ::WarnWinTextResetFont $w]
        }
        $mfont add command -label [_ "View source"] -command [list $w sourceView]
        $mfont add command -label [_ "Edit source"] -command [list $w editView]
    }
    set meditor [menu $m.me -tearoff 0]
    foreach block [WWTGetEditorList] {
        lassign $block ed name fmt
        $meditor add command -label $name -command [list ::WarnWinTextOpenInEditor $w $ed $fmt]
    }
    $meditor add command -label "GiD's editor" -command [list ::WarnWinTextOpenInEditor $w --INTERNAL--]
    $meditor add separator
    $meditor add command -label "Browser" -command [list ::WarnWinTextOpenInBrowser $w]
    $m add cascade -label [_ "Open in "] -menu $meditor

    if { [info exists ::gid_right_button] } {
        bind [$w getTextWidget] <$::gid_right_button> [list GiD_PopupMenu $m %X %Y]
    }
    return $m
}

# can provide the text (MarkDown or html format) or the path to the file with the text
# e.g.
# -np- W_MarkDown "# objarray\n## Tcl package for arrays\n"
# -np- W_MarkDown "" -filename {C:\gid_project_more\gid 16 branch\scripts\objarray\objarray.md}
# but the text cannot be edited or add more text like the proc W !!
proc W_MarkDown { text args } {
    if { [GidUtils::IsTkDisabled] } {
        #e.g. batch mode without windows
        return
    }
    package require parse_args
    parse_args::parse_args $args {
        -title    {-default "-automatic-"}
        -filename {-default ""}
    }
    package require shtmlview::shtmlview
    if { $title=="-automatic-" } {
        set title [_ "Information"]
    }
    if { [winfo exists .gid] } {
        set w .gid.markdown
    } else {
        set w .markdown
    }
    if { ! [winfo exists $w] } {
        if { 1 && [info procs InitWindow2] != "" } {
            #not using InitWindow because disappear when changing from pre to post
            #but maybe now is fixed, the problem wat the must start by PrePost*
            InitWindow2 $w -title [_ $title] -geometryvariable PrePostMarkDownWindowGeom
            # -ontop ;# eventually
            if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0
        } else {
            toplevel $w
            set wpar [winfo parent $w]
            if { $wpar != "."} {
                wm transient $w [winfo toplevel $wpar]
            }
            # if { $::tcl_platform(platform) == "windows" } {
            #     wm attributes $w -toolwindow 1
            # }
            wm title $w $title
        }

        ttk::frame $w.fr
        ::shtmlview::shtmlview $w.fr.md -toolbar 0 -tablesupport 1
        grid $w.fr.md -sticky nsew
        grid $w.fr -sticky nsew -padx 5 -pady 5
        grid rowconfigure $w.fr 0 -weight 1
        grid columnconfigure $w.fr 0 -weight 1
        grid rowconfigure $w 0 -weight 1
        grid columnconfigure $w 0 -weight 1

        #lower buttons
        ttk::frame $w.frmButtons -style BottomFrame.TFrame
        ttk::button $w.frmButtons.btnclose -text [_ "Close"] \
            -command "destroy $w" \
            -underline 0 -style BottomFrame.TButton

        grid $w.frmButtons -sticky ews -columnspan 7
        grid anchor $w.frmButtons center
        grid $w.frmButtons.btnclose -padx 5 -pady 6
        grid columnconfigure $w.frmButtons 0 -weight 1
        grid rowconfigure $w.frmButtons 0 -weight 1
        grid columnconfigure $w 0 -weight 1
        grid rowconfigure $w 0 -weight 1


        # right mouse button menu on text widget
        W_MarkDownRightButtonMenu $w.fr.md
        focus $w.frmButtons.btnclose
        bind $w <Alt-c> [list $w.frmButtons.btnclose invoke]
        bind $w <Escape> [list $w.frmButtons.btnclose invoke]
    }

    if { $filename != "" } {
        $w.fr.md browse $filename 
    } else {
        $w.fr.md render $text
    }
    
    # commented updated to avoid error when doing
    # Utilities --> Tools --> Develop -> Console Python...
    # import tohil; tohil.call( 'W', 'hola')
    # update
    # do not call directly update or update idletasks
    # update idletasks forces one loop over pending events
    # that causes GLWidget-OpenGL.cc : WWT0D_OutputPendingMessages()
    # to be blocked

    after idle "update idletasks; raise $w"
}

namespace eval TipWin {
    variable _mostrar
    variable _wid
    variable _id        1
    variable _imagen
}

proc TipWin::Close { title_org} {
    variable _wid

    regsub -all { } $title_org {_} pp
    regsub -all {\.} $pp {_} title

    if { ![info exists _wid($title)] || ![winfo exists $_wid($title)]} {
        return
    }

    destroy $_wid($title)

}

proc TipWin::CambiaMostrar { title_org w} {
    variable _mostrar

    regsub -all { } $title_org {_} pp
    regsub -all {\.} $pp {_} title

    if { $_mostrar($title)} {
        set _mostrar($title) 0
    } else {
        set _mostrar($title) 1
    }
    $w toggle
}

proc TipWin::CreateWindow { par show_cb title_org txt} {
    variable _mostrar
    variable _wid
    variable _id

    regsub -all { } $title_org {_} pp
    regsub -all {\.} $pp {_} title

    if { ![info exists _mostrar($title)] } {
        set _mostrar($title) 1
    }
    if { !$_mostrar($title)} {
        return
    } else {
        if { [info exists _wid($title)] && [winfo exists $_wid($title)]} {
            raise $_wid($title)
            return
        }
    }

    if { $par == "."} {
        set w .__Tip_Win$_id
    } else {
        set w $par.__Tip_Win$_id
    }
    incr _id

    set _wid($title) $w

    toplevel $w
    if { $::tcl_platform(platform) == "windows" } {
        #must not set transient if parent was . instead .gid!!
        wm transient $w [winfo toplevel [winfo parent $w]]
        wm attributes $w -toolwindow 1
    }

    wm title $w $title_org
    ttk::frame $w.f

    set max_w 0
    set txt_w 0
    set txt_h 1
    set n_txt [string length $txt]
    for { set i 0} { $i < $n_txt} { incr i} {
        incr txt_w
        if { "[string index $txt $i]" == "\n"} {
            incr txt_h
            if { $txt_w > $max_w} { set max_w $txt_w}
            set txt_w 0
        }
    }
    if { $txt_w > $max_w} { set max_w $txt_w}

    text $w.f.t -borderwidth 4  -wrap word -relief flat -width $max_w -height $txt_h

    $w.f.t insert end $txt
    $w.f.t configure -state disabled
    grid $w.f.t -sticky nwes -padx 4 -pady 8

    if { $show_cb} {
        if { ![info exists TipWin::_mostrar($title)] } { set TipWin::_mostrar($title) 0 }
        ttk::checkbutton $w.f.cb -text [_ "Show this message again"] -variable TipWin::_mostrar($title)
        grid $w.f.cb -sticky we -padx 4 -pady 8
    }

    # marco de los botones
    ttk::frame $w.but -style BottomFrame.TFrame
    ttk::button $w.but.close -text [_ "Close"] -command [list TipWin::Close $title] -takefocus 0 -style BottomFrame.TButton

    grid $w.but.close -sticky ews -padx 4 -pady 10

    #enpaquetamos marcos
    grid $w.f -sticky news -padx 0 -pady 0
    grid $w.but -sticky wes
    grid anchor $w.but center

    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1

    AdjustPosition $w $par
}

proc TipWin::CreateHelpButton { w title txt} {
    ttk::button $w -image [gid_themes::GetImage help.png large_icons] -command [list TipWin::CreateWindow .gid 0 $title $txt]
}

# This file defines the procedure tk_optionMenuRAM, similar to
# the standard tk_optionMenu, but separating labels and commands
# to be used in multilingual codes.
# Creates an option button and its associated menu.
#
# ::tk_optionMenuRAM --
# This procedure creates an option button named $w and an associated
# menu.  Together they provide the functionality of Motif option menus:
# they can be used to select one of many values, and the current value
# appears in the global variable varName, as well as in the text of
# the option menubutton.  The name of the menu is returned as the
# procedure's result, so that the caller can use it to change configuration
# options on the menu or otherwise manipulate it.
#
# Arguments:
# w -                        The name to use for the menubutton.
# varname -                Global variable to hold the currently selected value.
# labels -                list of labels to show as menu options.
# values -                list of values (internal commands untranslated).

proc ::tk_optionMenuRAM {w varname labels values} {
    if { ![info exists ::$varname] } {
        set ::$varname [lindex $values 0]
    }

    if { [set idx [lsearch $values [set ::$varname]]] == -1 } {
        set idx 0
    }
    set ::GidPriv(tkopt,$w,text) [lindex $labels $idx]
    if { ![winfo exists $w] } {
        ttk::menubutton $w -text [lindex $labels $idx] -textvariable ::GidPriv(tkopt,$w,text) -menu $w.menu direction flush -takefocus 1
        bind $w <Destroy> "unset ::GidPriv(tkopt,$w,text)"
    } else {
        $w configure -text [lindex $labels $idx] -textvariable ::GidPriv(tkopt,$w,text) -menu $w.menu -direction flush -takefocus ""
    }
    menu $w.menu -tearoff 0
    foreach i $labels j $values {
        $w.menu add radiobutton -label $i -value $j -variable ::$varname -command [list set ::GidPriv(tkopt,$w,text) $i]
    }

    set cmd [list tk_optionMenuRAMselectlabel $labels $values ::GidPriv(tkopt,$w,text) ::$varname]
    trace add variable ::$varname write $cmd
    bind $w <Destroy> [list +trace remove variable ::$varname write $cmd]

    return $w.menu
}


proc tk_optionMenuRAMselectlabel { labels values menutextvariable fullvarname varname field args } {
    if { [info exists ::$fullvarname] } {
        set idx [lsearch $values [set ::$fullvarname]]
    } else {
        set idx -1
    }
    if { $idx != -1 } {
        set $menutextvariable [lindex $labels $idx]
    } else {
        #set $menutextvariable ""
        set $menutextvariable [set ::$fullvarname]
    }
}
