set ::plot_graph 1 ;#to show graph in the graph window instead of the main window
set ::widget_graphs "combobox"

namespace eval PostGraphs {
    variable data
    variable cmds

    array set data {
        cmds,Point_Evolution PointEvolution
        cmds,Point_Evolution,X,require 0
        cmds,Point_Evolution,Y,require 1
        cmds,Point_Complex_Evolution PointComplexEvolution
        cmds,Point_Complex_Evolution,X,require 1
        cmds,Point_Complex_Evolution,Y,require 0
        cmds,Point_Graph PointGraph
        cmds,Point_Graph,X,require 2
        cmds,Point_Graph,Y,require 2
        cmds,Border_Graph BorderGraph
        cmds,Border_Graph,X,require 1
        cmds,Border_Graph,Y,require 1
        cmds,Line_Graph LineGraph
        cmds,Line_Graph,X,require 1
        cmds,Line_Graph,Y,require 2
        cmds,Integrate_Scalar Integrate
        cmds,Integrate_Scalar,X,require 0
        cmds,Integrate_Scalar,Y,require 1
        cmds,Integrate_Vector_Normal Integrate
        cmds,Integrate_Vector_Normal,X,require 0
        cmds,Integrate_Vector_Normal,Y,require 1
        cmds,Integrate_Vector_Tangential Integrate
        cmds,Integrate_Vector_Tangential,X,require 0
        cmds,Integrate_Vector_Tangential,Y,require 1
        tmp,applying 0
    }
}

proc PostGraphs::GetGraphSetAutomaticName { } {
    set basename [_ "graphset"]
    set i 1
    set graph_set $basename-$i
    while { [GiD_GraphSet exists $graph_set] } {
        incr i
        set graph_set $basename-$i
    }
    return $graph_set
}

proc PostGraphs::NewGraphSet { cb } {
    variable data
    set graph_set [PostGraphs::GetGraphSetAutomaticName]
    #GiD_GraphSet create $graph_set
    #GiD_GraphSet current $graph_set
    GiD_Process Mescape Results Graphs GraphSet Create $graph_set
    GiD_Process Mescape Results Graphs GraphSet Current $graph_set
    if { [winfo exists $cb] } {
        $cb configure -values [GiD_GraphSet list]
    }
    $data(tree) insert end [list $graph_set] root
    set data(current_graphset) $graph_set

    set data(presel) "setgraph"
    set currentset [GiD_GraphSet current]
    set numitems [$data(tree) item count]
    foreach id [$data(tree) item range 0 end] {
        set name [lindex [$data(tree) item text $id] 0]
        if { $name == $currentset} {
            $data(tree) selection clear
            $data(tree) selection add $id
            if { ![ info exists data(itemselected)] } {
                set data(itemselected) $id
            }
        }
    }
}

proc PostGraphs::DeleteGraphSet { cb } {
    variable data
    set graph_set [GiD_GraphSet current]
    set ret [MessageBoxOptionsButtons [_ "Confirm"] [_ "Are you sure to delete graphset '%s'" $graph_set] \
        {0 1} [list [_ "Yes"] [_ "No#C#I don't want to do that"]] question ""]
    if { $ret == 0 } {
        #GiD_GraphSet delete $graph_set
        set numitems [$data(tree) item count]
        if { $numitems > 1} {
            GiD_Process Mescape Results Graphs GraphSet Delete $graph_set
            if { [winfo exists $cb] } {
                $cb configure -values [GiD_GraphSet list]
            }

            set numitems [$data(tree) item count]
            foreach id [$data(tree) item range 0 end] {
                set name [lindex [$data(tree) item text $id] 0]
                if { $name == $graph_set} {
                    set itemdel $id
                }
            }
            $data(tree) item delete $itemdel

            set data(current_graphset) [GiD_GraphSet current]
        }
    }
}

proc PostGraphs::RenameGraphSet { cb } {
    variable data
    set graph_set [GiD_GraphSet current]
    set new_name [MessageBoxEntry [_ "Rename %s" $graph_set] [_ "Enter new name of %s '%s'" [_ "graphset"] $graph_set] any "" 0 question.png]
    if { $new_name != "" } {
        if {[lsearch [GiD_GraphSet list] $new_name] == -1} {
            GiD_Process Mescape Results Graphs GraphSet Edit $graph_set Name $new_name escape
            if { [winfo exists $cb] } {
                $cb configure -values [GiD_GraphSet list]
                #PostGraphs::RefreshFromGiD
                PostGraphs::FillGeneralOptions
            }
            set numitems [$data(tree) item count]
            foreach id [$data(tree) item range 0 end] {
                set name [lindex [$data(tree) item text $id] 0]
                if { $name == $graph_set} {
                    $data(tree) item text $id 0 $new_name
                }
            }
        }
        set data(current_graphset) [GiD_GraphSet current]
    }
}

proc PostGraphs::SetCurrentGraphSet { cb } {
    variable data

    if { $data(current_graphset) != [GiD_GraphSet current] } {
        GiD_GraphSet current $data(current_graphset)

        if { [winfo exists $cb] } {
            PostGraphs::RefreshFromGiD
        }
    }

}

proc PostGraphs::SetCurrentGraphSetCombo { cb } {
    variable data

    if { $data(current_graphset) != [GiD_GraphSet current] } {
        set numitems [$data(tree) item count]
        foreach id [$data(tree) item range 0 end] {
            set name [lindex [$data(tree) item text $id] 0]
            if { $name == $data(current_graphset)} {
                PostGraphs::ControlTree select $data(tree) $id
            }
        }
        GiD_GraphSet current $data(current_graphset)

        if { [winfo exists $cb] } {
            PostGraphs::RefreshFromGiD
        }
    }

}

proc PostGraphs::Create {{w .gid.graphs}} {
    variable data

    InitWindow2 $w -title [_ "Graphs Window"] \
        -geometryvariable PostResultsGraphsWindowGeom \
        -initcommand PostGraphs::Create
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0
    set data(wid,main) $w

    ttk::frame $w.fsets

    set data(wid,graphsets) [ttk::combobox $w.fsets.cb1 -textvariable PostGraphs::data(current_graphset) -values [GiD_GraphSet list] -state readonly]
    bind $w.fsets.cb1 <<ComboboxSelected>> [list PostGraphs::SetCurrentGraphSetCombo %W]
    #to avoid mousewheel of $w.fsets.cb1 Lisbox be propagated to gid_togl

    set data(current_graphset) [GiD_GraphSet current]

    ttk::button $w.fsets.bnew -image [gid_themes::GetImage new.png small_icons] -command [list PostGraphs::NewGraphSet $w.fsets.cb1]
    GidHelp $w.fsets.bnew  [_ "Create a new set of graphs"]
    ttk::button $w.fsets.bdel -image [gid_themes::GetImage delete.png small_icons] -command [list PostGraphs::DeleteGraphSet $w.fsets.cb1]
    GidHelp $w.fsets.bdel  [_ "Delete a set of graphs"]
    ttk::button $w.fsets.bren -image [gid_themes::GetImage rename.png small_icons] -command [list PostGraphs::RenameGraphSet $w.fsets.cb1]
    GidHelp $w.fsets.bren  [_ "Rename a set of graphs"]
    ttk::frame $w.fcurrentset
    grid $w.fsets.cb1 $w.fsets.bnew $w.fsets.bdel $w.fsets.bren -sticky w
    #grid configure $w.fsets.cb1 -sticky ew
    #grid columnconfigure $w.fsets 0 -weight 1
    grid rowconfigure $w.fcurrentset 0 -weight 1
    grid columnconfigure $w.fcurrentset 0 -weight 1
    grid $w.fsets -sticky new
    grid $w.fcurrentset -sticky nsew

    set nb [NoteBook $w.fcurrentset.nb -internalborderwidth 1]
    set data(wid,nb) $nb

    if { $::plot_graph } {
        #page to draw graphs
        set fplot [$nb insert end plot -text [_ "Plot graph"]]
    }
    set fcreate [$nb insert end create -text [_ "Create"]]
    set foptions [$nb insert end options -text [_ "Options"]]

    if { $::plot_graph } {
        CreatePlot $fplot
    }
    CreateGraphs $fcreate
    CreateOptions $foptions

    set fcmd [ttk::frame $w.fcmd -style BottomFrame.TFrame]
    set data(wid,fcmd) $fcmd
    ttk::button $fcmd.bapply -text [_ "Apply"] -command PostGraphs::OnApply -takefocus 0 -style BottomFrame.TButton
    ttk::button $fcmd.breset -text [_ "Reset"] -command PostGraphs::OnReset -takefocus 0 -style BottomFrame.TButton
    set data(wid,breset) $fcmd.breset
    ttk::button $fcmd.bclose -text [_ "Close"] -command [list destroy $w] -takefocus 0 -style BottomFrame.TButton

    set data(wid,apply) $fcmd.bapply
    set data(wid,reset) $fcmd.breset
    set data(wid,close) $fcmd.bclose

    if { !$::plot_graph } {
        ttk::button $fcmd.toggle -image [gid_themes::GetImage "postgraf.png" medium_icons] \
            -command PostGraphs::ToggleView -takefocus 0 -style BottomFrame.TButton
        GidHelpT $fcmd.toggle [_ "Toggle mesh and graphs view"]
    }

    grid $fcmd.bapply -row 0 -column 1 -padx 5 -pady 6
    grid $fcmd.breset -row 0 -column 2 -padx 5 -pady 6
    grid $fcmd.bclose -row 0 -column 3 -padx 5 -pady 6
    if { !$::plot_graph } {
        grid $fcmd.toggle -row 0 -column 5 -sticky "e"
    }
    grid columnconfigure $fcmd 0 -weight 1
    grid columnconfigure $fcmd 4 -weight 1

    #$nb itemconfigure plot -raisecmd "PostGraphs::UpdatePlotGraph ; grid remove $fcmd.bapply $fcmd.breset ; grid $fcmd.bclose"
    $nb itemconfigure plot -raisecmd "grid remove $fcmd.bapply $fcmd.breset ; grid $fcmd.bclose"
    $nb itemconfigure create -raisecmd "grid remove $fcmd.breset ; grid $fcmd.bapply $fcmd.bclose"
    $nb itemconfigure options -raisecmd "grid $fcmd.bapply $fcmd.breset $fcmd.bclose"

    grid $nb -sticky snew -columnspan 2
    grid $fcmd -sticky ew

    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 1 -weight 1
    if { $::plot_graph } {
        $data(wid,nb) raise plot
    } else {
        $data(wid,nb) raise create
    }
    focus $fcmd.bapply
    bind $w <Return> [list $fcmd.bapply invoke]
    bind $w <Escape> Togl_Escape

    bind $w <${::acceleratorKey}-a> [bind .gid <${::acceleratorKey}-a>]
    bind $w <Destroy> [list +PostGraphs::Destroy %W $w] ;# + to add to previous script

}

proc PostGraphs::Destroy { W w } {
    if { $W != $w } return
    #reenter multiple times, one by toplevel child, only interest w

    variable data
    array unset -nocomplain data ;#remove also traces associated to the variable

    trace remove variable PostGraphs::data(opt,coord) write "PostGraphs::OnChangeCoord"
    trace remove variable PostGraphs::data(opt,origcolor,value) write "PostGraphs::UpdateOrigColor"
    trace remove variable PostGraphs::data(opt,radcolor,value) write "PostGraphs::UpdateRadColor"

    trace remove variable PostGraphs::data(opt,color,value) write "PostGraphs::UpdateColor"
    trace remove variable PostGraphs::data(opt,colorascfill,value) write "PostGraphs::OnChangeCFill"
}

proc PostGraphs::FillGraphSets {} {
    variable data
    if { [info exists data(wid,graphsets)] } {
        set cb $data(wid,graphsets)
        if { [winfo exists $cb] } {
            $cb configure -values [GiD_GraphSet list]
        }
    }
    set data(current_graphset) [GiD_GraphSet current]
}

proc PostGraphs::PickOneGraph { w x y action } {
    #variable data
    #data(wid,togl) is equal to $w

    GidUtils::Disable warnline
    set selwhat Graphs
    #set selwhat All
    #$w gidmakecurrent ;#to draw well
    GiD_Process 'SelectEntities $selwhat
    $w button 1 [expr $x-5] [expr $y-5]
    $w button 1 [expr $x+5] [expr $y+5]
    after 0 GiD_Process Escape
    vwait ::GidPriv(selection)
    if { [info exists ::GidPriv(selection)] && ( $::GidPriv(selection) != "")} {
        set selected_graph [lindex {*}$::GidPriv(selection) 1]
        #could be something a range like 33:35, then get arbitrarily the first one 33
        set selected_graph [lindex [split $selected_graph :] 0]
    } else {
        set selected_graph ""
    }
    unset -nocomplain ::GidPriv(selection)
    GidUtils::Enable warnline
    if { $action == "show_contextual_menu" } {
        if { [llength $selected_graph] == 1 } {
            #WarnWinText "selec: $selected_graph"
            set ::ActionsMenuVarType TMP
            set ::ActionsMenuVar ""
            set ::ActionsMenuLabels ""
            set graph_num [lindex $selected_graph 0]
            set graph [GiD_Graph get_name $graph_num]
            if { $graph != "" } {
                lappend ::ActionsMenuVar [list -np- PostGraphs::ClipboardCopyGraph $graph]
                lappend ::ActionsMenuLabels [_ "Clipboard copy graph '%s'" $graph]
                lappend ::ActionsMenuVar [list -np- PostGraphs::DeleteGraphAsk $graph]
                lappend ::ActionsMenuLabels [_ "Delete graph '%s'" $graph]
                lappend ::ActionsMenuVar [list -np- PostGraphs::ShowEditGraph $graph]
                lappend ::ActionsMenuLabels [_ "Edit graph '%s'" $graph]
            }
        }
        # Togl_Button-Right expects %X %Y and not %x %y
        # i.e. x_root and y_root and not x_window y_window
        set X [ expr [ winfo rootx $w ] + $x]
        set Y [ expr [ winfo rooty $w ] + $y]
        Togl_Button-Right $w $X $Y ; # this calls GiDShowContextualMenu
    } elseif {$action == "show_edit"} {
        if { [llength $selected_graph] == 1 } {
            set graph_num [lindex $selected_graph 0]
            set graph [GiD_Graph get_name $graph_num]
            if { $graph != "" } {
                PostGraphs::ShowEditGraph $graph
            }
        }
    } elseif {$action == "swap_selection"} {
        if { $selected_graph != "" } {
            foreach graph_num [GidUtils::UnCompactNumberList $selected_graph] {
                set graph [GiD_Graph get_name $graph_num]
                if { $graph != "" } {
                    GiD_Graph selection swap $graph
                }
            }
            PostGraphs::UpdatePlotGraph
        }
    } else {
        WarnWinText "PostGraphs::PickOneGraph unexpected action $action"
    }
}

proc PostGraphs::ShowEditGraph { graph } {
    variable data
    if { $graph != "" } {
        set numitems [$data(tree) item count]
        foreach id [$data(tree) item range 0 end] {
            set name [lindex [$data(tree) item text $id] 0]
            if { $name == $graph} {
                PostGraphs::ControlTree select $data(tree) $id
            }
        }
        set data(presel) "graph"
        $data(wid,nb) raise options
    }
}

proc PostGraphs::CreatePlot { f } {
    variable data
    set data(wid,togl) [GiD_Project set windows_layout GRAPH $f].s

    #bind $data(wid,togl) <${::acceleratorKey}-Button-1> [list PostGraphs::PickOneGraph %W %x %y]
    bind $data(wid,togl) <$::gid_right_button> [list PostGraphs::PickOneGraph %W %x %y show_contextual_menu]
    bind $data(wid,togl) <Double-Button-1> [list PostGraphs::PickOneGraph %W %x %y show_edit]

    #bind $data(wid,togl) <Delete> [list PostGraphs::DeleteSelectionOnGraphs]
    #bind $data(wid,togl) <Button-1> [list PostGraphs::PickOneGraph %W %x %y swap_selection]
    #bind $data(wid,togl) <ButtonPress-1> [list PostGraphs::PickOneGraph %W %x %y swap_selection]
    #bind $data(wid,togl) <ButtonRelease-1> [list PostGraphs::PickOneGraph %W %x %y swap_selection]
}

proc PostGraphs::ZoomFrame { } {
    variable data
    if { [info exists data(wid,togl)] && [winfo exists $data(wid,togl)] && [winfo class $data(wid,togl)] == "Gid_togl" } {
        set prev_togl_current [GiD_Togl current]
        if { $data(wid,togl) != $prev_togl_current } {
            $data(wid,togl) gidmakecurrent ;#to send zoom to the graph, not the model
        }
        GiD_Process 'Zoom Frame Escape
        if { $data(wid,togl) != $prev_togl_current } {
            GiD_Togl current $prev_togl_current
        }

    }
}


proc PostGraphs::UpdatePlotGraph { } {
    variable data
    if { [info exists data(wid,togl)] && [winfo exists $data(wid,togl)] && [winfo class $data(wid,togl)] == "Gid_togl" } {
        if { [$data(wid,nb) raise] == "plot" } {
            #$data(wid,nb) raise plot
            PostGraphs::ZoomFrame
            PostGraphs::ZoomFrame
        }
    }
}

proc PostGraphs::CreateGraphs { f } {

    set c [ CreateScrolledCanvas $f.sc]
    $f.sc configure -relief flat
    set base [ ttk::frame $c.base]

    # From result
    ttk::labelframe $base.result -text [_ "From result"]
    set fresult $base.result
    set fview [ttk::frame $fresult.fview]
    CreateView $fview
    set faxis [ttk::frame $fresult.faxis]
    CreateAxis $faxis
    PostGraphs::UpdateResults
    grid $fview -sticky "nsew" -pady 3
    grid $faxis -sticky "nsew" -pady 3
    grid columnconfigure $fresult 0 -weight 1
    grid rowconfigure $fresult 1 -weight 1


    # From graphs
    ttk::labelframe $base.graphs -text [_ "From graphs"]
    set fgraphs $base.graphs
    ttk::label $fgraphs.test -text [_ "Test"]
    grid $fgraphs.test -sticky "nsew"
    grid columnconfigure $fgraphs 0 -weight 1
    grid rowconfigure $fgraphs 0 -weight 1

    # From Formula

    # From values


    grid $fresult -sticky "nsew"
    #grid $fgraphs -sticky "nsew"
    #grid $fvalues -sticky "nsew"

    grid rowconfigure $base 0 -weight 1
    grid columnconfigure $base 0 -weight 1

    AddToScrolledCanvas $f.sc $c.base
    grid $f.sc -sticky news

    grid rowconfigure $f 0 -weight 1
    grid columnconfigure $f 0 -weight 1

    # to show the correct selected option
    # like the checkbox if Point_Graph or Line_Graph are selected
    PostGraphs::OnChangeView
}

proc PostGraphs::OnTitleVisible { } {
    variable data
    if { $data(opt,title,visible) != [GiD_Info postprocess get graphs_option TitleVisible] } {
        set graph_set [GiD_GraphSet current]
        GiD_GraphSet edit $graph_set title_visible $PostGraphs::data(opt,title,visible)
        PostGraphs::UpdatePlotGraph
    }
}

proc PostGraphs::OnLegendLocation { } {
    #GiD_Process Mescape Results Graphs OptionsGraph Legend2Title
    variable data
    if { $data(opt,legend,location) != [GiD_Info postprocess get graphs_option LegendLocation] } {
        set graph_set [GiD_GraphSet current]
        GiD_GraphSet edit $graph_set legend_location $PostGraphs::data(opt,legend,location)
        PostGraphs::UpdatePlotGraph
    }
}

#return 0 1 instead of words
proc PostGraphs::IsLegendOpaque { } {
    if { [lindex [GiD_Info postprocess get legends_state] 1] == "Transparent" } {
        set opaque 0
    } elseif { [lindex [GiD_Info postprocess get legends_state] 1] == "Opaque" } {
        set opaque 1
    } else {
        #unexpected
        set opaque 0
    }
    return $opaque
}

proc PostGraphs::OnLegendOpaque { } {
    variable data
    if { $data(opt,legend,opaque) != [PostGraphs::IsLegendOpaque] } {
        #set graph_set [GiD_GraphSet current]
        #GiD_GraphSet edit $graph_set legend_opaque $PostGraphs::data(opt,legend,opaque)
        GiD_Process 'Hardcopy Options OpaqueLegends [expr {$data(opt,legend,opaque)?Yes:No}]
        PostGraphs::UpdatePlotGraph
    }
}

proc PostGraphs::CreateOptions { f { sg 1 }} {
    variable data

    set ftree [ ttk::frame $f.t]
    CreateTree $ftree

    set data(frametree) $ftree

    set fgopt [ ttk::frame $f.go]
    set flopt [ ttk::frame $f.lo]

    set data(fGoptions) $fgopt
    set data(fLoptions) $flopt

    CreateGOptions $fgopt
    CreateLOptions $flopt

    if { $sg } {
        grid $ftree $fgopt -sticky "news"
    } else {
        grid $ftree $flopt -sticky "news"
    }

    grid rowconfigure $f 0 -weight 1
    grid columnconfigure $f 1 -weight 1

}

proc PostGraphs::CreateGOptions { f } {
    variable data

    set c [ CreateScrolledCanvas $f.sc]
    $f.sc configure -relief flat
    set base [ ttk::frame $c.base]


    # Legend options
    ttk::labelframe $base.legend -text [_ "Legend"]
    set flegend $base.legend
    ttk::label $flegend.llegendlocation -text [_ "Legend location"]
    set data(opt,legend,location) [GiD_Info postprocess get graphs_option LegendLocation]
    TTKComboBox $flegend.cblegendlocation -labels [list [_ "Upper-right"] [_ "Top"] [_ "None"]] -values {1 2 0} -textvariable PostGraphs::data(opt,legend,location) \
        -state readonly -modifycmd PostGraphs::OnLegendLocation

    ttk::label $flegend.llegendopaque -text [_ "Background"]
    set data(opt,legend,opaque) [PostGraphs::IsLegendOpaque]
    TTKComboBox $flegend.cblegendopaque -labels [list [_ "Opaque"] [_ "Transparent"]] -values {1 0} -textvariable PostGraphs::data(opt,legend,opaque) \
        -state readonly -modifycmd PostGraphs::OnLegendOpaque

    set data(opt,title,visible)  [GiD_Info postprocess get graphs_option TitleVisible]
    ttk::checkbutton $flegend.chkvisible -variable PostGraphs::data(opt,title,visible) -text [_ "Show graphset title"] -command [list PostGraphs::OnTitleVisible]

    grid $flegend.llegendlocation $flegend.cblegendlocation -sticky w
    grid $flegend.llegendopaque $flegend.cblegendopaque -sticky w
    grid $flegend.chkvisible -sticky w

    ttk::frame $base.axis
    # X-axis options
    ttk::labelframe $base.axis.x -text [_ "X axis"]
    set faxis $base.axis.x
    PostGraphs::CreateAxisOptions $faxis "X"
    set data(wid,xaxisframe) $base.axis.x

    # Y-axis options
    ttk::labelframe $base.axis.y -text [_ "Y axis"]
    set faxis $base.axis.y
    PostGraphs::CreateAxisOptions $faxis "Y"
    set data(wid,yaxisframe) $base.axis.y

    grid $base.axis.x $base.axis.y
    grid columnconfigure $base.axis 2 -weight 1

    # Coordinates options
    ttk::labelframe $base.coord -text [_ "Coordinates type"]
    set fcoord $base.coord
    set coord_info [GiD_Info postprocess get graphs_option -allowed_values CoordType]
    set data(opt,coord,all) $coord_info
    ttk::label $fcoord.lcoord -text [_ "Coordinates type"]:
    ttk::radiobutton $fcoord.rcart -text [_ "Cartesian"] -value Cartesian -variable PostGraphs::data(opt,coord)
    ttk::radiobutton $fcoord.rpolar -text [_ "Polar"] -value Polar -variable PostGraphs::data(opt,coord)

    ttk::label $fcoord.luse -text [_ "Use"]
    TTKComboBox $fcoord.cbangleaxis -values [list x y] \
        -textvariable PostGraphs::data(opt,angleaxis) -state readonly


    ttk::label $fcoord.laxis -text [_ "axis values as angle"]
    ttk::label $fcoord.luse2 -text [_ "Use"]
    TTKComboBox $fcoord.cbangleunit -values [list [_ "Degrees"] [_ "Radians"]] \
        -textvariable PostGraphs::data(opt,angleunit) -state readonly

    ttk::label $fcoord.lunit -text [_ "as angle unit"]
    ttk::label $fcoord.laxes -text [_ "Show axes in (0,0) point"]:
    if { ![info exists PostGraphs::data(opt,origaxes)] } { set PostGraphs::data(opt,origaxes) "No" }
    ttk::checkbutton $fcoord.ckaxes -variable PostGraphs::data(opt,origaxes)  \
        -onvalue "Yes" -offvalue "No"

    ttk::label $fcoord.lorigaxes -text [_ "Show radial marks"]:
    if { ![info exists PostGraphs::data(opt,radmarks)] } { set PostGraphs::data(opt,radmarks) "No" }
    ttk::checkbutton $fcoord.ckrad -variable PostGraphs::data(opt,radmarks)  \
        -onvalue "Yes" -offvalue "No"

    TTKComboBox $fcoord.cbradpat \
        -textvariable PostGraphs::data(opt,patradmarks) -width 10 \
        -state readonly -values [ list "Continous" "Point" "Slash"] \
        -labels [ list [_ "Continous"] [_ "Point"] [_ "Slash"]]

    ttk::label $fcoord.lminpol -text [_ "Minimum value"]:
    ttk::entry $fcoord.eminpol -textvariable PostGraphs::data(minpolar)
    ttk::label $fcoord.lmaxpol -text [_ "Maximum value"]:
    ttk::entry $fcoord.emaxpol -textvariable PostGraphs::data(maxpolar)

    ttk::label $fcoord.origcolor -width 4 -borderwidth 1 -relief solid
    bind $fcoord.origcolor <1> PostGraphs::SelectOrigColor
    set data(wid,origcolor) $fcoord.origcolor

    ttk::label $fcoord.radcolor -width 4 -borderwidth 1 -relief solid
    bind $fcoord.radcolor <1> PostGraphs::SelectRadColor
    set data(wid,radcolor) $fcoord.radcolor

    set data(opt,nocolor) [$fcoord.origcolor cget -background]

    set data(wid,angleaxis) $fcoord.cbangleaxis
    set data(wid,angleunit) $fcoord.cbangleunit
    set data(wid,origaxes) $fcoord.ckaxes
    set data(wid,radmarks) $fcoord.ckrad
    set data(wid,radpattern) $fcoord.cbradpat
    set data(wid,lminpol) $fcoord.lminpol
    set data(wid,eminpol) $fcoord.eminpol
    set data(wid,lmaxpol) $fcoord.lmaxpol
    set data(wid,emaxpol) $fcoord.emaxpol

    grid $fcoord.rcart $fcoord.rpolar -sticky "nw"
    grid $fcoord.luse $fcoord.cbangleaxis $fcoord.laxis -sticky "nw"
    grid $fcoord.luse2 $fcoord.cbangleunit $fcoord.lunit -sticky "nw"
    grid $fcoord.origcolor $fcoord.laxes $fcoord.ckaxes -sticky "nw"
    grid $fcoord.radcolor $fcoord.lorigaxes $fcoord.ckrad $fcoord.cbradpat -sticky "nw"
    grid $fcoord.lminpol $fcoord.eminpol -sticky "nw"
    grid $fcoord.lmaxpol $fcoord.emaxpol -sticky "nw"
    grid rowconfigure $fcoord 1 -weight 1
    grid columnconfigure $fcoord 3 -weight 1


    trace add variable PostGraphs::data(opt,coord) write "PostGraphs::OnChangeCoord"
    trace add variable PostGraphs::data(opt,radcolor,value) write "PostGraphs::UpdateRadColor"
    trace add variable PostGraphs::data(opt,origcolor,value) write "PostGraphs::UpdateOrigColor"


    # Style options
    ttk::labelframe $base.style -text [_ "Style"]
    set fstyle $base.style
    ttk::label $fstyle.loutline -text [_ "Outline on model"]:
    if { ![info exists PostGraphs::data(opt,outline)] } { set PostGraphs::data(opt,outline) "No" }
    ttk::checkbutton $fstyle.ckoutline -variable PostGraphs::data(opt,outline) \
        -onvalue "Yes" -offvalue "No"
    set data(wid,ckoutline) $fstyle.cboutline

    ttk::label $fstyle.lgrids -text [_ "Grids"]:
    if { ![info exists PostGraphs::data(opt,grids)] } { set PostGraphs::data(opt,grids) "No" }
    ttk::checkbutton $fstyle.ckgrids -variable PostGraphs::data(opt,grids) \
        -onvalue "Yes" -offvalue "No"
    set data(wid,ckgrids) $fstyle.ckgrids

    grid $fstyle.loutline $fstyle.ckoutline -sticky "nw"
    grid $fstyle.lgrids $fstyle.ckgrids -sticky "nw"


    grid $base.legend -sticky "news"
    grid $base.coord -sticky "news"
    grid $base.axis -sticky "news"
    grid $base.style -sticky "news"

    grid rowconfigure $base 4 -weight 1
    grid columnconfigure $base 0 -weight 1

    AddToScrolledCanvas $f.sc $c.base
    grid $f.sc -sticky news

    grid rowconfigure $f 0 -weight 1
    grid columnconfigure $f 0 -weight 1

    PostGraphs::FillGeneralOptions
    PostGraphs::FillCoordinatesOptions
}

proc PostGraphs::CreateAxisOptions {f axis}  {
    variable data

    ttk::label $f.lmin -text [_ "Minimum"]:
    ttk::entry $f.emin -textvariable PostGraphs::data(opt,axis,$axis,min)
    ttk::label $f.lmax -text [_ "Maximum"]:
    ttk::entry $f.emax -textvariable PostGraphs::data(opt,axis,$axis,max)
    ttk::label $f.ldiv -text [_ "Divisions"]:
    ttk::entry $f.ediv -textvariable PostGraphs::data(opt,axis,$axis,div)
    ttk::label $f.llabel -text [_ "Label"]:
    ttk::entry $f.elabel -textvariable PostGraphs::data(opt,axis,$axis,label)
    ttk::label $f.lunit -text [_ "Unit"]:
    ttk::entry $f.eunit -textvariable PostGraphs::data(opt,axis,$axis,unit)
    ttk::label $f.llog -text [_ "Logarithmic scale"]:
    if { ![info exists PostGraphs::data(opt,axis,$axis,log)] } { set PostGraphs::data(opt,axis,$axis,log) "No" }
    ttk::checkbutton $f.elog -variable PostGraphs::data(opt,axis,$axis,log)  -onvalue "Yes" -offvalue "No"

    if { $axis eq "Y"} {
        set data(wid,lymin) $f.lmin
        set data(wid,ymin) $f.emin
        set data(wid,lymax) $f.lmax
        set data(wid,ymax) $f.emax
        set data(wid,lydiv) $f.ldiv
        set data(wid,ydiv) $f.ediv
        set data(wid,lylab) $f.llabel
        set data(wid,ylab) $f.elabel
        set data(wid,lyunit) $f.lunit
        set data(wid,yunit) $f.eunit
        set data(wid,lylog) $f.llog
        set data(wid,ylog) $f.elog
    }

    if { $axis eq "X"} {
        set data(wid,lxmin) $f.lmin
        set data(wid,xmin) $f.emin
        set data(wid,lxmax) $f.lmax
        set data(wid,xmax) $f.emax
        set data(wid,lxdiv) $f.ldiv
        set data(wid,xdiv) $f.ediv
        set data(wid,lxlab) $f.llabel
        set data(wid,xlab) $f.elabel
        set data(wid,lxunit) $f.lunit
        set data(wid,xunit) $f.eunit
        set data(wid,lxlog) $f.llog
        set data(wid,xlog) $f.elog
    }

    grid $f.llog $f.elog  -sticky "nw"
    grid $f.lmin $f.emin  -sticky "nw"
    grid $f.lmax $f.emax  -sticky "nw"
    grid $f.ldiv $f.ediv  -sticky "nw"
    grid $f.llabel $f.elabel  -sticky "nw"
    grid $f.lunit $f.eunit  -sticky "nw"

}

proc PostGraphs::FillGeneralOptions { } {
    variable data

    # Outline on model
    set outline_info [GiD_Info postprocess get graphs_option OutlineOnModel]
    set data(opt,outline) $outline_info

    # Grids
    set grids_info [GiD_Info postprocess get graphs_option Grids]
    set data(opt,grids) $grids_info

    # Legend options

    # Axis options
    foreach axis {X Y} {
        foreach {opt opt_arg} {log Log min Min max Max div Divisions label Label unit Unit} {
            if { $opt eq "min"} {
                set is_log [GiD_Info postprocess get graphs_option ${axis}_axis log]
                if {$is_log} {
                    set user_min [GiD_Info postprocess get graphs_option ${axis}_axis UserMin]
                    set log_min [GiD_Info postprocess get graphs_option ${axis}_axis LogMin]
                    if { !$user_min} {
                        if { $log_min} {
                            set is_user_min [GiD_Info postprocess get graphs_option ${axis}_axis Min]
                            set data(opt,axis,$axis,$opt) $is_user_min
                        }
                    }
                } else {
                    set axis_info [GiD_Info postprocess get graphs_option ${axis}_axis $opt_arg]
                    set data(opt,axis,$axis,$opt) $axis_info
                }
            } elseif { $opt eq "max"} {
                set is_log [GiD_Info postprocess get graphs_option ${axis}_axis log]
                if {$is_log} {
                    set user_max [GiD_Info postprocess get graphs_option ${axis}_axis UserMax]
                    set log_max [GiD_Info postprocess get graphs_option ${axis}_axis LogMax]
                    if { !$user_max} {
                        if { $log_max} {
                            set is_user_max [GiD_Info postprocess get graphs_option ${axis}_axis Max]
                            set data(opt,axis,$axis,$opt) $is_user_max
                        }
                    }
                } else {
                    set axis_info [GiD_Info postprocess get graphs_option ${axis}_axis $opt_arg]
                    set data(opt,axis,$axis,$opt) $axis_info
                }
            } else {
                set axis_info [GiD_Info postprocess get graphs_option ${axis}_axis $opt_arg]
                set data(opt,axis,$axis,$opt) $axis_info
            }
        }
    }
}

proc PostGraphs::Alive { } {
    variable data
    expr {[info exists data(wid,main)] && [winfo exists $data(wid,main)]}
}

proc PostGraphs::UpdateColor {args} {
    variable data

    set color $data(opt,color,value)
    if {$color eq ""} {
        $data(wid,lcolorsel) configure -background $data(opt,nocolor)
    } else {
        if { [$data(wid,lcolorsel) cget -background] != $color } {
            $data(wid,lcolorsel) configure -background $color
        }
    }
}

proc PostGraphs::UpdateOrigColor {args} {
    variable data

    set color $data(opt,origcolor,value)
    if {$color eq ""} {
        $data(wid,origcolor) configure -background $data(opt,nocolor)
    } else {
        $data(wid,origcolor) configure -background $color
    }
}

proc PostGraphs::UpdateRadColor {args} {
    variable data

    set color $data(opt,radcolor,value)
    if {$color eq ""} {
        $data(wid,radcolor) configure -background $data(opt,nocolor)
    } else {
        $data(wid,radcolor) configure -background $color
    }
}

proc PostGraphs::SelectColor {} {
    variable data

    if {[$data(wid,lcolorsel) cget -state] ne "disabled"} {
        if {$data(opt,color,value) eq ""} {
            set color [ GIDChooseColor $data(wid,main).selcolor \
                -title [_ "Choose graph color"]]
        } else {
            set color [ GIDChooseColor $data(wid,main).selcolor \
                -color $data(opt,color,value) \
                -title [_ "Choose graph color"]]
        }

        if {$color ne ""} {
            set data(opt,color,value) $color
        }
    }
}

proc PostGraphs::SelectOrigColor {} {
    variable data

    if {[$data(wid,origcolor) cget -state] ne "disabled"} {
        if {$data(opt,origcolor,value) eq ""} {
            set color [ GIDChooseColor $data(wid,main).selcolor \
                -title [_ "Choose origin axis color"]]
        } else {
            set color [ GIDChooseColor $data(wid,main).selcolor \
                -color $data(opt,origcolor,value) \
                -title [_ "Choose origin axis color"]]
        }

        if {$color ne ""} {
            set data(opt,origcolor,value) $color
        }
    }
}

proc PostGraphs::SelectRadColor {} {
    variable data

    if {[$data(wid,radcolor) cget -state] ne "disabled"} {
        if {$data(opt,radcolor,value) eq ""} {
            set color [ GIDChooseColor $data(wid,main).selcolor \
                -title [_ "Choose radial marks color"]]
        } else {
            set color [ GIDChooseColor $data(wid,main).selcolor \
                -color $data(opt,radcolor,value) \
                -title [_ "Choose radial marks color"]]
        }

        if {$color ne ""} {
            set data(opt,radcolor,value) $color
        }
    }
}

proc PostGraphs::CBUpdateCurrent {cb {cur ""}} {
    if {$cur eq ""} {
        $cb set ""
    } else {
        $cb set $cur
    }
}

proc PostGraphs::EntrySetValue { e v } {
    variable data

    $data(wid,$e) configure -validate none
    uplevel \#0 set [$data(wid,$e) cget -textvariable] $v
    $data(wid,$e) configure -validate all
}

proc PostGraphs::FillCoordinatesOptions { } {
    variable data

    # coordinates type
    set coord_info [GiD_Info postprocess get graphs_option CoordType]
    set cur_coord $coord_info
    set data(opt,coord) $cur_coord

    # Polar coordinates options
    set allowed_angaxis [GiD_Info postprocess get graphs_option -allowed_values AngleAxis]
    set cur_angaxis [GiD_Info postprocess get graphs_option AngleAxis]
    set cur_angaxis_idx [lsearch $allowed_angaxis $cur_angaxis]
    if { [PostGraphs::Alive]} {
        if {$cur_angaxis_idx == -1} {
            #$data(wid,angleaxis) configure -text {}
            $data(wid,angleaxis) set ""
        } else {
            #$data(wid,angleaxis) setvalue "@$cur_angaxis_idx"
            $data(wid,angleaxis) \
                set [ lindex [ $data(wid,angleaxis) cget -values ] \
                $cur_angaxis_idx ]
        }
    }

    set allowed_angunit [GiD_Info postprocess get graphs_option -allowed_values AngleUnit]
    set cur_angunit [GiD_Info postprocess get graphs_option AngleUnit]
    set cur_angunit_idx [lsearch $allowed_angunit $cur_angunit]
    if { [PostGraphs::Alive]} {
        if {$cur_angunit_idx == -1} {
            #$data(wid,angleunit) configure -text {}
            $data(wid,angleunit) set ""
        } else {
            #$data(wid,angleunit) setvalue "@$cur_angunit_idx"
            $data(wid,angleunit) \
                set [ lindex [ $data(wid,angleunit) cget -values ] \
                $cur_angunit_idx ]
        }
    }

    set origaxes_info [GiD_Info postprocess get graphs_option ShowOrigAxes]
    set data(opt,origaxes) $origaxes_info

    set radmarks_info [GiD_Info postprocess get graphs_option ShowRadMarks]
    set data(opt,radmarks) $radmarks_info

    set origcolor_info [GiD_Info postprocess get graphs_option ColorOrig]
    set data(opt,origcolor,value) $origcolor_info

    set radcolor_info [GiD_Info postprocess get graphs_option ColorRad]
    set data(opt,radcolor,value) $radcolor_info

    set radpat_info [GiD_Info postprocess get graphs_option PatRad]
    set data(opt,patradmarks) $radpat_info

    set polar_min [GiD_Info postprocess get graphs_option PolarMin]
    set data(minpolar) $polar_min

    set polar_max [GiD_Info postprocess get graphs_option PolarMax]
    set data(maxpolar) $polar_max

    OnChangeCoord
}

proc PostGraphs::OnChangeGraphSelection { graphs } {
    variable data

    if { [llength $graphs] == 0} {
        # Style
        #$data(wid,style) configure -text {}
        $data(wid,style) set ""

        # Color
        set data(opt,color,value) ""

        # disable widgets related
        foreach _w {chkvisible lstyle style lcolor ecolor lcolorsel cbcfill \
            llwidth sclwidth lpat cbpat lpatfact epatfact \
                lpsize scpsize lwindow b1window} {
                $data(wid,$_w) configure -state disabled
        }
    } else {
        set g [lindex $graphs 0]
        if { $g == "" } {
            return
        }
        # enable widgets related
        foreach _w {chkvisible lstyle style lcolor ecolor lcolorsel cbcfill \
            llwidth sclwidth lpat cbpat lpatfact epatfact \
                lpsize scpsize lwindow b1window} {
                $data(wid,$_w) configure -state normal
        }

        # Visible
        set visible [GiD_Info postprocess get graphs_option visible $g]
        set data(opt,visible,value) $visible

        # Style
        set style [GiD_Info postprocess get graphs_option style $g]
        CBUpdateCurrent $data(wid,style) $style

        # ColorAsCfill
        set colorascfill [GiD_Info postprocess get graphs_option colorascfill $g]
        set data(opt,colorascfill,value) $colorascfill

        # Color
        if { $data(opt,colorascfill,value) eq "No" } {
            $data(wid,lcolor) configure -state normal
            $data(wid,ecolor) configure -state normal
            $data(wid,lcolorsel) configure -state normal
            set color [GiD_Info postprocess get graphs_option color $g]
        } else {
            $data(wid,lcolor) configure -state disabled
            $data(wid,ecolor) configure -state disabled
            $data(wid,lcolorsel) configure -state disabled
            set color ""
        }
        set data(opt,color,value) $color

        # LineWidth
        set data(opt,linewidth,value) [GiD_Info postprocess get graphs_option linewidth $g]

        # Pattern
        set cur_pat_val [GiD_Info postprocess get graphs_option pattern $g]
        set cur_pat_val [string range $cur_pat_val 1 end]
        set all [list ]
        set cur_pat_str ""
        foreach pat $data(opt,pattern,all,names) {
            set str $data(opt,pattern,$pat,str)
            lappend all $str
            if {$data(opt,pattern,$pat,val) == $cur_pat_val} {
                set cur_pat_str $str
            }
        }
        if {$cur_pat_str eq ""} {
            set cur_pat_str $cur_pat_val
        }
        CBUpdateCurrent $data(wid,cbpat) $cur_pat_str

        # PatternFactor
        set patternfactor [GiD_Info postprocess get graphs_option patternfactor $g]
        EntrySetValue epatfact $patternfactor

        # PointSize
        set data(opt,pointsize,value) [GiD_Info postprocess get graphs_option pointsize $g]

    }
}

proc PostGraphs::LO_OnModifyCB { name } {
    variable data
}

proc PostGraphs::NewGraph { cb } {

}

proc PostGraphs::DeleteGraphs { graphs } {
    variable data
    set data(tmp,applying) 1
    foreach graph $graphs {
        GiD_Process Mescape Results Graphs OptionsGraph DeleteGraph $graph
    }
    if { [winfo exists $data(tree)] } {
        foreach graph $graphs {
            set numitems [$data(tree) item count]
            foreach id [$data(tree) item range 0 end] {
                set name [lindex [$data(tree) item text $id] 0]
                if { $name == $graph} {
                    $data(tree) item delete $id
                    break
                }
            }
        }
    }
    set data(tmp,applying) 0
}

#graph is the graph name
proc PostGraphs::ClipboardCopyGraph { graph_name } {
    if { $graph_name != "" } {
        set graphset_name [GiD_GraphSet current]
        set data [GetGraphDataClipboardFormat $graphset_name $graph_name]
        clipboard clear
        clipboard append $data
        GidUtils::SetWarnLine [_ "Graph data copied to clipboard"]
    }
}

proc PostGraphs::DeleteGraphAsk { graph } {
    if { $graph != "" } {
        set ret [MessageBoxOptionsButtons [_ "Confirm"] [_ "Are you sure to delete graph '%s'" $graph] \
            {0 1} [list [_ "Yes"] [_ "No#C#I don't want to do that"]] question ""]
        if { $ret == 0 } {
            PostGraphs::DeleteGraphs [list $graph]
        }
    }
}

proc PostGraphs::DeleteSelectedGraphs {} {
    variable data

    set graph [lindex [$data(tree) item text $data(itemselected)] 0]
    PostGraphs::DeleteGraphAsk $graph
    #foreach graph $graphs {
    #    PostGraphs::DeleteGraphAsk $graph
    #}
}

proc PostGraphs::DeleteSelectionOnGraphs { } {
    set graphs [list]
    foreach graph [GiD_Graph list] {
        if { [GiD_Graph selection get $graph] } {
            lappend graphs $graph
        }
    }
    foreach graph $graphs {
        PostGraphs::DeleteGraphAsk $graph
    }
}

proc PostGraphs::RenameGraph {} {
    variable data

    set graph [lindex [$data(tree) item text $data(itemselected)] 0]
    set new_name [MessageBoxEntry [_ "Rename %s" $graph] [_ "Enter new name of %s '%s'" [_ "graph"] $graph] any "" 0 question.png]
    if { $new_name != "" } {
        GiD_Process Mescape Results Graphs OptionsGraph TitleGraphs $graph $new_name
        $data(tree) item text $data(itemselected) 0 $new_name
    }
}

proc PostGraphs::CreateLOptions { f } {
    variable data

    set c [ CreateScrolledCanvas $f.sc]
    $f.sc configure -relief flat
    set base [ ttk::frame $c.base]

    panedwindow $base.pn -orient horizontal
    ttk::labelframe $base.pn.o -text [_ "Options"]:
    $base.pn add $base.pn.o -sticky snew

    # Graphs options frame
    set fo $base.pn.o

    # Visible delete rename
    ttk::label $fo.lvisible -text ""
    if { ![info exists PostGraphs::data(opt,visible,value)] } { set PostGraphs::data(opt,visible,value) 1 }
    ttk::checkbutton $fo.chkvisible -variable PostGraphs::data(opt,visible,value) -text [_ "Visible"]
    set data(wid,chkvisible) $fo.chkvisible

    ttk::button $fo.bdel -image [gid_themes::GetImage delete.png medium_icons] -text [_ "Delete"] \
        -command [list PostGraphs::DeleteSelectedGraphs]

    ttk::button $fo.bren -image [gid_themes::GetImage rename.png medium_icons] -command [list PostGraphs::RenameGraph]


    # Style
    set data(opt,style,all,names) {None Dot Line DotLine}
    set values [list [_ "None"] [_ "Dot"] [_ "Line"] [_ "Dot Line"]]
    ttk::label $fo.lstyle -text [_ "Style"]:
    TTKComboBox $fo.cbstyle -state readonly \
        -values $values -textvariable PostGraphs::data(opt,style,value) \
        -modifycmd "PostGraphs::LO_OnModifyCB style"
    set data(wid,lstyle) $fo.lstyle
    set data(wid,style) $fo.cbstyle

    # Color
    ttk::label $fo.lcolor -text [_ "Color"]:
    ttk::frame $fo.fc
    ttk::entry $fo.fc.ecolor -textvariable PostGraphs::data(opt,color,value)
    ttk::label $fo.fc.lcolorsel -width 4 -borderwidth 1 -relief solid
    bind $fo.fc.lcolorsel <1> PostGraphs::SelectColor
    set data(wid,lcolor) $fo.lcolor
    set data(wid,ecolor) $fo.fc.ecolor
    set data(wid,lcolorsel) $fo.fc.lcolorsel
    set data(opt,nocolor) [$fo.fc.lcolorsel cget -background]
    grid $fo.fc.ecolor $fo.fc.lcolorsel -sticky "ew"
    grid columnconfigure $fo.fc 0 -weight 1


    # Color as contour fill
    ttk::label $fo.lcfill -text ""

    if { ![info exists PostGraphs::data(opt,colorascfill,value)] } { set PostGraphs::data(opt,colorascfill,value) "No" }
    ttk::checkbutton $fo.cbcfill \
        -variable PostGraphs::data(opt,colorascfill,value) -text [_ "Color as contour fill"] -onvalue "Yes" -offvalue "No"

    set data(wid,cbcfill) $fo.cbcfill
    set data(opt,nocolor) [$fo.fc.lcolorsel cget -background]

    # LineWidth
    ttk::label $fo.llwidth -text [_ "Line width"]:
    set lw_info [GiD_Info postprocess get graphs_option -allowed_values linewidth]
    lassign $lw_info lw_from lw_to lw_step
    if { ![info exists PostGraphs::data(opt,linewidth,value)] } {
        set PostGraphs::data(opt,linewidth,value) 0
    }
    gidscale $fo.sclwidth -showvalue 1 \
        -orient horizontal -variable PostGraphs::data(opt,linewidth,value) -from $lw_from -to $lw_to -resolution $lw_step \
        -command "PostGraphs::OnChangeScale linewidth"
    set data(wid,llwidth) $fo.llwidth
    set data(wid,sclwidth) $fo.sclwidth

    # Pattern
    set dot [ GiDTeXSymbols::Get {\mult}]
    array set data [list \
        opt,pattern,Continuous,str "--------" \
        opt,pattern,Continuous,val ffff \
        opt,pattern,Slash,str "--  --  " \
        opt,pattern,Slash,val ff00 \
        opt,pattern,SlashDot,str "- $dot - $dot " \
        opt,pattern,SlashDot,val ff18 \
        opt,pattern,LittleSlash,str "- - - - " \
        opt,pattern,LittleSlash,val f0f0 \
        opt,pattern,Dots,str "$dot $dot $dot $dot " \
        opt,pattern,Dots,val f0f0 \
        opt,pattern,BigSlash,str "--- --- " \
        opt,pattern,BigSlash,val cccc \
        opt,pattern,SlashSpace,str "-   -   " \
        opt,pattern,SlashSpace,val f000 \
        opt,pattern,DotSlashDot,str "$dot-$dot$dot-$dot" \
        opt,pattern,DotSlashDot,val cfcc]
    set data(opt,pattern,all,names) \
        {Continuous Slash SlashDot LittleSlash Dots BigSlash SlashSpace DotSlashDot}
    set all [list ]
    foreach pattern $data(opt,pattern,all,names) {
        lappend all $data(opt,pattern,$pattern,str)
    }
    ttk::label $fo.lpat -text [_ "Pattern"]:
    TTKComboBox $fo.cbpat -state readonly \
        -values $all -textvariable PostGraphs::data(opt,pattern,value) \
        -modifycmd "PostGraphs::LO_OnModifyCB pattern"
    set data(wid,lpat) $fo.lpat
    set data(wid,cbpat) $fo.cbpat

    # PatternFactor
    ttk::label $fo.lpatfact -text [_ "Pattern factor"]:
    ttk::entry $fo.epatfact \
        -textvariable PostGraphs::data(opt,patternfactor,value) \
        -validate all \
        -validatecommand "PostGraphs::OnEditLPatFact %V %P %s"
    set data(wid,lpatfact) $fo.lpatfact
    set data(wid,epatfact) $fo.epatfact

    # PointSize
    ttk::label $fo.lpsize -text [_ "Point size"]:
    set ps_info [GiD_Info postprocess get graphs_option -allowed_values pointsize]
    lassign $ps_info ps_from ps_to ps_step
    if { ![info exists PostGraphs::data(opt,pointsize,value)] } {
        set PostGraphs::data(opt,pointsize,value) 0
    }
    gidscale $fo.scpsize -showvalue 1 \
        -orient horizontal -variable PostGraphs::data(opt,pointsize,value) -from $ps_from -to $ps_to -resolution $ps_step \
        -command "PostGraphs::OnChangeScale pointsize"
    set data(wid,lpsize) $fo.lpsize
    set data(wid,scpsize) $fo.scpsize

    # InvertLineGraph

    # Results window
    ttk::label $fo.lwindow -text [_ "Values"]:
    ttk::button $fo.b1window -text [_ "Show table"] \
        -command [list PostGraphs::GraphWindow [ winfo toplevel $f]] \
        -takefocus 0 -style BottomFrame.TButton
    GidHelp [list $fo.lwindow $fo.b1window]   [_ "Allow know or modify the values of the graph (copy/paste can be used)"]
    
    set data(wid,lwindow) $fo.lwindow
    set data(wid,b1window) $fo.b1window

    grid $fo.lvisible $fo.chkvisible $fo.bdel $fo.bren -sticky ew -pady 2

    foreach {w2 w3 i} {lstyle cbstyle 1 \
        lcolor fc 2 lcfill cbcfill 3 \
        llwidth sclwidth 4 lpat cbpat 5 \
        lpatfact epatfact 6 lpsize scpsize 7 \
        lwindow b1window 8 } {
            grid $fo.$w2 -column 0 -row $i -sticky ew -pady 2
        grid $fo.$w3 -column 1 -row $i -sticky ew -pady 2 -columnspan 3
    }
    grid configure $fo.b1window -sticky w -columnspan 1
    grid columnconfigure $fo 1 -weight 1

    grid $base.pn -sticky "snew"
    grid rowconfigure $base 0 -weight 1
    grid columnconfigure $base 0 -weight 1

    AddToScrolledCanvas $f.sc $c.base
    grid $f.sc -sticky news

    grid rowconfigure $f 0 -weight 1
    grid columnconfigure $f 0 -weight 1


    trace add variable PostGraphs::data(opt,color,value) write "PostGraphs::UpdateColor"
    trace add variable PostGraphs::data(opt,colorascfill,value) write "PostGraphs::OnChangeCFill"
}

proc PostGraphs::ControlTree { args } {
    variable data

    if { [lindex $args 0] == "select" } {
        update idletasks
        set tree [lindex $args 1]
        set id [lindex $args 2]
        if { $id == "" } {
            return
        }
        $tree selection clear
        $tree selection add $id

        set name [$tree item text $id]
        set data(itemselected) $id

        set sog [$tree item parent $id]

        if { $sog == 0 } { #es un set
        set data(current_graphset) [lindex $name 0]
        PostGraphs::SetCurrentGraphSet %W
        if { $data(presel) != "setgraph"} { #el anterior era un graph
        grid remove $data(fLoptions)
        grid $data(fGoptions) -row 0 -column 1 -sticky news
        grid $data(wid,apply) $data(wid,reset) $data(wid,close)
    }
    PostGraphs::FillGeneralOptions
    PostGraphs::FillCoordinatesOptions
    set data(presel) "setgraph"

    } else { #es un graph
        set setname [lindex [$tree item text $sog] 0]
        set data(current_graphset) $setname
        PostGraphs::SetCurrentGraphSet %W
        if { $data(presel) != "graph"} { #el anterior era un set
        grid remove $data(fGoptions)
        grid $data(fLoptions) -row 0 -column 1 -sticky news
        grid remove $data(wid,reset)
        grid $data(wid,apply) $data(wid,close)
    }
    PostGraphs::OnChangeGraphSelection $name
    set data(presel) "graph"

}
}
}
proc PostGraphs::FillTree { T } {
variable data

set setsName [GiD_GraphSet list]

set itemsList [$T item range 0 end]
foreach id $itemsList {
    foreach sn $setsName {
        set test [lsearch $itemsList $id]
        if { $test != -1 } {
            set name [ lindex [$T item text $id] 0]
            if { $name == $sn } {
                $T item delete $id
                set itemsList [$T item range 0 end]
            }
        }
    }
}

foreach i $setsName {
    set itemset [$T insert end [ list $i ] root]
    set graphsName [GiD_Graph list $i]
    foreach j $graphsName {
        set item [$T insert end [ list $j ] $itemset]
    }
}

set data(presel) "setgraph"
set currentset [GiD_GraphSet current]
set numitems [$data(tree) item count]
foreach id [$data(tree) item range 0 end] {
    set name [lindex [$data(tree) item text $id] 0]
    if { $name == $currentset} {
        $T selection clear
        $T selection add $id
        if { ![ info exists data(itemselected)] } {
            set data(itemselected) $id
        }
    }
}
update idletasks
}


proc PostGraphs::CreateTree { f } {
    package require fulltktree

    variable data

    set base [ ttk::frame $f.base]

    # width name justify type is_editable
    set cols {
        { 50 Graphs left item 1 }
        { 3 "" right image 1 }
    }
    set T [fulltktree $f.t -columns $cols -width 180]
    set data(tree) $T
    $T configure \
        -selecthandler "PostGraphs::ControlTree select" \
        -item_image "" \
        -folder_image "" \
        -arrow_image_expand 0 \
        -showlines 1 -indent 15 -showheader 0

    $T element create eWindow window
    $T element create eRect rect -fill #ff0000
    $T element create eText2 text -wrap none

    set S [$T style create s2 -orient vertical]
    $T style elements $S {eText2 eRect eWindow}
    $T style layout $S eRect -union {eText2 eWindow} -ipadx 8 -ipady 8 -padx 4 -pady {0 4}
    $T style layout $S eText2 -pady {0 6} -squeeze x
    $T style layout $S eWindow -iexpand x -squeeze x


    $T notify bind MyTag <Expand-before> {
        %T item image %I 1 [gid_themes::GetImage blank.png small_icons]
        %T item element configure %I 0 e_text_sel -fill ""
    }
    $T notify bind MyTag <Collapse-before> {
        %T item image %I 1 [gid_themes::GetImage blank.png small_icons]
        %T item element configure %I 0 e_text_sel -fill ""
        set listchilds [%T item children %I]
        foreach itemid $listchilds {
            if { [%T item image $itemid 1] == [gid_themes::GetImage ArrowLeft.png small_icons] } {
                %T item image %I 1 [gid_themes::GetImage ArrowLeft.png small_icons]
                %T item element configure %I 0 e_text_sel -fill #ff0000
                break;
            }
        }
    }
    $T dragimage configure -visible 0

    $T column configure all -button 0
    $T column configure all -minwidth {} -squeeze 0 -weight 1 ; #trick to modify fulltktree settings to have more equispaced columns

    grid $T -sticky nsew

    grid columnconfigure $f 0 -weight 1
    grid rowconfigure $f 0 -weight 1

    set setsName [GiD_GraphSet list]

    foreach i $setsName {
        set itemset [$T insert end [ list $i ] root]
        set graphsName [GiD_Graph list $i]
        foreach j $graphsName {
            set item [$T insert end [ list $j ] $itemset]
        }
    }

    set data(presel) "setgraph"
    set currentset [GiD_GraphSet current]
    set numitems [$data(tree) item count]
    foreach id [$data(tree) item range 0 end] {
        set name [lindex [$data(tree) item text $id] 0]
        if { $name == $currentset} {
            $T selection clear
            $T selection add $id
            if { ![ info exists data(itemselected)] } {
                set data(itemselected) $id
            }
        }
    }

    update idletasks

    bind $T <Key-Delete> [list PostGraphs::DeleteSelectedGraphs]
    #     bind $T <Double-Button-1> [list PostGraphs::BeginEditLayers $T]

    return $T
}

# proc PostGraphs::BeginEditLayers { T } {
#     set I [$T item id active]
#     set C 0
#     set E eRect
#     ::TreeCtrl::FileListEdit $T $I $C $E
# }

proc PostGraphs::OnChangeScale { name v } {

}

proc PostGraphs::OnChangeCFill {args} {
    variable data
    if { $data(opt,colorascfill,value) eq "No" } {
        $data(wid,lcolor) configure -state normal
        $data(wid,ecolor) configure -state normal
        $data(wid,lcolorsel) configure -state normal
        set g [lindex [$data(tree) item text $data(itemselected)] 0]
        if { $g != "" } {
            set color [GiD_Info postprocess get graphs_option color $g]
            set data(opt,color,value) $color
        }
    } else {
        $data(wid,lcolor) configure -state disabled
        $data(wid,ecolor) configure -state disabled
        $data(wid,lcolorsel) configure -state disabled
        set data(opt,color,value) ""
    }
}

proc PostGraphs::OnChangeCoord { args } {
    variable data

    if { $PostGraphs::data(opt,coord) eq "Polar"} {

        #         set f $data(wid,xaxisframe)
        #         foreach item [list {*}[grid slaves $f] $f] {
        #             $item configure -state disabled
        #         }
        #
        #         set f $data(wid,yaxisframe)
        #         foreach item [list {*}[grid slaves $f] $f] {
        #             $item configure -state disabled
        #         }
        #
        $data(wid,lymin) configure -state disabled
        $data(wid,ymin)  configure -state disabled
        $data(wid,lymax) configure -state disabled
        $data(wid,ymax)  configure -state disabled
        $data(wid,lydiv) configure -state disabled
        $data(wid,ydiv)  configure -state disabled
        $data(wid,lylab)  configure -state disabled
        $data(wid,ylab)  configure -state disabled
        $data(wid,lyunit)  configure -state disabled
        $data(wid,yunit)  configure -state disabled
        $data(wid,lylog)  configure -state disabled
        $data(wid,ylog)  configure -state disabled

        $data(wid,lxmin)  configure -state disabled
        $data(wid,xmin) configure -state disabled
        $data(wid,lxmax)  configure -state disabled
        $data(wid,xmax) configure -state disabled
        $data(wid,lxdiv) configure -state disabled
        $data(wid,xdiv)  configure -state disabled
        $data(wid,lxlab)  configure -state disabled
        $data(wid,xlab)  configure -state disabled
        $data(wid,lxunit)  configure -state disabled
        $data(wid,xunit)  configure -state disabled
        $data(wid,lxlog) configure -state disabled
        $data(wid,xlog) configure -state disabled

        #         $data(wid,xaxisframe) configure -state disabled
        #         $data(wid,yaxisframe) configure -state disabled

        $data(wid,angleaxis) configure -state normal
        $data(wid,angleunit) configure -state normal
        $data(wid,origaxes) configure -state normal
        $data(wid,radmarks) configure -state normal
        $data(wid,radpattern) configure -state normal

        $data(wid,origcolor) configure -state normal
        $data(wid,radcolor) configure -state normal

        $data(wid,lminpol) configure -state normal
        $data(wid,eminpol) configure -state normal
        $data(wid,lmaxpol) configure -state normal
        $data(wid,emaxpol) configure -state normal

        set origcolor_info [GiD_Info postprocess get graphs_option ColorOrig]
        set data(opt,origcolor,value) $origcolor_info

        set radcolor_info [GiD_Info postprocess get graphs_option ColorRad]
        set data(opt,radcolor,value) $radcolor_info

        set polar_min [GiD_Info postprocess get graphs_option PolarMin]
        set data(minpolar) $polar_min

        set polar_max [GiD_Info postprocess get graphs_option PolarMax]
        set data(maxpolar) $polar_max

        UpdateOrigColor
        UpdateRadColor
    }

    if { $PostGraphs::data(opt,coord) eq "Cartesian"} {
        $data(wid,lymin) configure -state normal
        $data(wid,ymin)  configure -state normal
        $data(wid,lymax)  configure -state normal
        $data(wid,ymax)  configure -state normal
        $data(wid,lydiv) configure -state normal
        $data(wid,ydiv)  configure -state normal
        $data(wid,lylab)  configure -state normal
        $data(wid,ylab) configure -state normal
        $data(wid,lyunit)  configure -state normal
        $data(wid,yunit)  configure -state normal
        $data(wid,lylog)  configure -state normal
        $data(wid,ylog)  configure -state normal

        $data(wid,lxmin) configure -state normal
        $data(wid,xmin)  configure -state normal
        $data(wid,lxmax)  configure -state normal
        $data(wid,xmax)  configure -state normal
        $data(wid,lxdiv) configure -state normal
        $data(wid,xdiv)  configure -state normal
        $data(wid,lxlab)  configure -state normal
        $data(wid,xlab)  configure -state normal
        $data(wid,lxunit)  configure -state normal
        $data(wid,xunit)  configure -state normal
        $data(wid,lxlog) configure -state normal
        $data(wid,xlog) configure -state normal

        #         $data(wid,xaxisframe) configure -state normal
        #         $data(wid,yaxisframe) configure -state normal

        #
        #         set f $data(wid,xaxisframe)
        #         foreach item [list {*}[grid slaves $f] $f] {
        #             $item configure -state normal
        #         }
        #
        #         set f $data(wid,yaxisframe)
        #         foreach item [list {*}[grid slaves $f] $f] {
        #             $item configure -state normal
        #         }

        $data(wid,angleaxis) configure -state disabled
        $data(wid,angleunit) configure -state disabled
        $data(wid,origaxes) configure -state disabled
        $data(wid,radmarks) configure -state disabled
        $data(wid,radpattern) configure -state disabled

        $data(wid,origcolor) configure -state disabled
        $data(wid,radcolor) configure -state disabled

        $data(wid,lminpol) configure -state disabled
        $data(wid,eminpol) configure -state disabled
        $data(wid,lmaxpol) configure -state disabled
        $data(wid,emaxpol) configure -state disabled

        set data(opt,origcolor,value) ""
        set data(opt,radcolor,value) ""
        UpdateOrigColor
        UpdateRadColor
    }
}

proc PostGraphs::OnEditLPatFact { action new old } {
    variable data

    if {$action eq "key" } {

    } elseif {$action eq "focusin"} {
        set data(opt,patternfactor,value,save) $old
    } elseif {$action eq "focusout"}  {
        if {(![string is double $new] || $new<0)} {
            EntrySetValue epatfact $data(opt,patternfactor,value,save)
        }
    }
    after idle $data(wid,epatfact) configure -validate all
    return 1
}

proc PostGraphs::OnReset { } {
    variable data
    # Reset Options
    # Coordinates type
    GiD_Process Mescape Results Graphs OptionsGraph Coordtype Cartesian escape
    OnChangeCoord

    # Outline on model

    # Grids

    # Style

    # MainTitle

    # Axis
    GiD_Process Mescape Results Graphs OptionsGraph X_Axis Reset Y_Axis Reset escape

    PostGraphs::FillGeneralOptions
    PostGraphs::FillCoordinatesOptions
    PostGraphs::ZoomFrame
}

proc PostGraphs::OnApply { } {
    variable data
    switch [$data(wid,nb) raise] {
        create {
            PostGraphs::ApplyCreate
        }
        options {
            if { $data(presel) == "graph"} {
                PostGraphs::ApplyGraphOptions
            } else {
                PostGraphs::ApplyGeneralOptions
                PostGraphs::ApplyCoordinatesOptions
                PostGraphs::ZoomFrame
            }
            $data(wid,nb) raise plot
            PostGraphs::FillGeneralOptions
            PostGraphs::ZoomFrame
        }
    }
}

proc PostGraphs::ApplyGeneralOptions { } {
    variable data

    set data(tmp,applying) 1

    # Outline on model
    set cur [GiD_Info postprocess get graphs_option OutlineOnModel]
    if {$cur ne $data(opt,outline)} {
        GiD_Process MEscape Results Graphs OptionsGraph GraphOutLineOnModel [expr {$data(opt,outline)?Yes:No}]
    }

    # Grids
    set cur [GiD_Info postprocess get graphs_option Grids]
    if {$cur ne $data(opt,grids)} {
        GiD_Process MEscape Results Graphs OptionsGraph Grids [expr {$data(opt,grids)?Yes:No}]
    }

    # Legend

    # Axis options
    set coordtype [GiD_Info postprocess get graphs_option CoordType]

    # apply changes after checking what's the user changed
    # if we do it in the middle, strange things will happen, as internal values change during the process
    set gid_cmd_lst [ list]
    foreach axis {X Y} {
        foreach {opt opt_arg} {log Log min Min max Max div Divisions label Label unit Unit} {
            set axis_info [GiD_Info postprocess get graphs_option ${axis}_axis $opt_arg]
            if { ($data(opt,axis,$axis,$opt) ne $axis_info) || $coordtype eq "Polar"} {
                if { $opt == "min"} {
                    if { $data(opt,axis,$axis,$opt) >= $data(opt,axis,$axis,max) && $coordtype eq "Polar"} {
                        WarnWin [_ "Minimum can not be greater than the maximum"]
                    } else {
                        lappend gid_cmd_lst [ list \
                            Mescape Results Graphs OptionsGraph ${axis}_axis \
                            Set${opt_arg} $data(opt,axis,$axis,$opt) \
                        ]
                }
            } elseif { $opt == "max"} {
                if { $data(opt,axis,$axis,$opt) <= $data(opt,axis,$axis,min) && $coordtype eq "Polar"} {
                    WarnWin [_ "Maximum can not be minor than the minimum"]
                } else {
                    lappend gid_cmd_lst [ list \
                        Mescape Results Graphs OptionsGraph ${axis}_axis \
                        Set${opt_arg} $data(opt,axis,$axis,$opt) \
                    ]
            }
        } else {
            lappend gid_cmd_lst [ list \
                Mescape Results Graphs OptionsGraph ${axis}_axis \
                Set${opt_arg} $data(opt,axis,$axis,$opt) \
            ]
    }
}
}
}
foreach gid_cmd $gid_cmd_lst {
    GiD_Process {*}$gid_cmd
}

set data(tmp,applying) 0
PostGraphs::FillGeneralOptions
}

proc PostGraphs::ApplyCoordinatesOptions { } {
    variable data

    set data(tmp,applying) 1

    # Polar coordiantes options
    set cur [GiD_Info postprocess get graphs_option CoordType]
    if {$cur ne $data(opt,coord)} {
        GiD_Process Mescape Results Graphs OptionsGraph CoordType $data(opt,coord) escape
    }
    #set idx [$data(wid,angleaxis) getvalue]
    set idx [ $data(wid,angleaxis) current ]
    if {$idx != -1} {
        set all_axis [GiD_Info postprocess get graphs_option -allowed_values AngleAxis]
        set angleaxis_info [GiD_Info postprocess get graphs_option AngleAxis]
        set cur_angleaxis_idx [lsearch $all_axis $angleaxis_info]
        if {$cur_angleaxis_idx != $idx} {
            GiD_Process Mescape Results Graphs OptionsGraph CoordType Options AngleAxis [lindex $all_axis $idx] escape
        }
    }

    #set idx [$data(wid,angleunit) getvalue]
    set idx [ $data(wid,angleunit) current ]
    if {$idx != -1} {
        set all_unit [GiD_Info postprocess get graphs_option -allowed_values AngleUnit]
        set angleunit_info [GiD_Info postprocess get graphs_option AngleUnit]
        set cur_angleunit_idx [lsearch $all_unit $angleunit_info]
        if {$cur_angleunit_idx != $idx} {
            GiD_Process Mescape Results Graphs OptionsGraph CoordType Options AngleUnit [lindex $all_unit $idx] escape
        }
    }

    set cur [GiD_Info postprocess get graphs_option ShowOrigAxes]
    if {$cur ne $data(opt,origaxes)} {
        GiD_Process Mescape Results Graphs OptionsGraph CoordType Options ShowOrigAxes [expr {$data(opt,origaxes)?Yes:No}]
    }

    set cur [GiD_Info postprocess get graphs_option ShowRadMarks]
    if {$cur ne $data(opt,radmarks)} {
        GiD_Process Mescape Results Graphs OptionsGraph CoordType Options ShowRadMarks [expr {$data(opt,radmarks)?Yes:No}]
    }

    set origcolor [GiD_Info postprocess get graphs_option ColorOrig]
    if {$data(opt,origcolor,value) != "" && $origcolor ne $data(opt,origcolor,value)} {
        GiD_Process Mescape Results Graphs OptionsGraph CoordType Options ColorOrigAxes $data(opt,origcolor,value)
    }

    set radcolor [GiD_Info postprocess get graphs_option ColorRad]
    if {$data(opt,radcolor,value) != "" && $radcolor ne $data(opt,radcolor,value)} {
        GiD_Process Mescape Results Graphs OptionsGraph CoordType Options ColorRadMarks $data(opt,radcolor,value)
    }

    set idx [$data(wid,radpattern) current]
    if {$idx != -1} {
        set all_radpat [GiD_Info postprocess get graphs_option -allowed_values PatRad]
        set radpat [GiD_Info postprocess get graphs_option PatRad]
        set cur_radpat_idx [lsearch $all_radpat $radpat]
        if {$cur_radpat_idx != $idx} {
            GiD_Process Mescape Results Graphs OptionsGraph CoordType Options PatternRadMarks [lindex $all_radpat $idx]
        }
    }

    # may be both are changing making:
    set cur [GiD_Info postprocess get graphs_option PolarMin]
    if {$cur ne $data(minpolar)} {
        # set max [GiD_Info postprocess get graphs_option PolarMax]
        set max $data(maxpolar)
        if {$data(minpolar) >= $max} {
            WarnWin [_ "Minimum can not be greater than the maximum"]
        } else {
            GiD_Process Mescape Results Graphs OptionsGraph CoordType Options PolarMin $data(minpolar)
        }
    }

    set cur [GiD_Info postprocess get graphs_option PolarMax]
    if {$cur ne $data(maxpolar)} {
        # set min [GiD_Info postprocess get graphs_option PolarMin]
        set min $data(minpolar)
        if {$data(maxpolar) <= $min} {
            WarnWin [_ "Maximum can not be minor than the minimum"]
        } else {
            GiD_Process Mescape Results Graphs OptionsGraph CoordType Options PolarMax $data(maxpolar)
        }
    }


    set data(tmp,applying) 0
    PostGraphs::FillCoordinatesOptions
    PostGraphs::FillGeneralOptions
    PostGraphs::UpdateOrigColor
    PostGraphs::UpdateRadColor
}

proc PostGraphs::ApplyGraphOptions { } {
    variable data

    set hay_cfill [GiD_Info postprocess get graphs_option HayCFill]

    set data(tmp,applying) 1
    set need_update 0

    set graphs [ list [lindex [$data(tree) item text $data(itemselected)] 0]]
    # Style
    # Color
    # ColorAsCfill
    # LineWidth
    # Pattern
    # PatternFactor
    # PointSize
    foreach graph $graphs {
        foreach {opt name} {Visible visible StyleGraphs style ColorGraphs color \
            colorascfill colorascfill linewidthgraphs linewidth patterngraphs pattern \
            patternfactor patternfactor pointsize pointsize} {
                if { $name eq "pattern" } {
                    set v [lindex $data(opt,pattern,all,names) [$data(wid,cbpat) current]]
            } elseif {$name eq "style"} {
                set v [lindex $data(opt,style,all,names) [$data(wid,style) current]]
            } else {
                set v $data(opt,$name,value)
            }
            if { $v != "" } {
                if { $name == "pattern" } {
                    set prev_v [string range [GiD_Info postprocess get graphs_option $name $graph] 1 end]
                    #convert true value #ffff into process keyname
                    foreach pat $data(opt,pattern,all,names) {
                        if { $data(opt,pattern,$pat,val) == $prev_v } {
                            set prev_v $pat
                            break
                        }
                    }
                } else {
                    set prev_v [GiD_Info postprocess get graphs_option $name $graph]
                }

                if { $v != $prev_v } {
                    set need_update 1
                    GiD_Process MEscape Results Graphs OptionsGraph $opt $graph $v
                }
            }

            if { ( $name eq "colorascfill") && ( $PostGraphs::data(opt,colorascfill,value) eq "Yes") && ( $hay_cfill eq "No" )} {
                WarnWin [_ "Please try to load contour fill results first"]
            }
        }
    }
    set data(tmp,applying) 0
}

proc PostGraphs::ApplyCreate { {node ""} } {
    variable data

    set view [GetCurrentView]
    if {$view eq "No_Graph"} {
        GiD_Process MEscape Results Graphs OptionsGraph ClearGraphs
    } else {
        if {[AllArgumentsOk]} {
            PostGraphs::ChangeAnalysis
            GiD_Process Mescape {*}$data(cmds,args)
            FinishButton [winfo toplevel $data(wid,fcmd)] $data(wid,fcmd) \
                [_ "Press 'Finish' to end selection"] \
                PostGraphs::AfterPressFinish disableall [GiD_Set SmallWinSelecting]
        }
    }
}

proc PostGraphs::AllArgumentsOk { } {
    variable data

    set view [GetCurrentView]

    if { ![info exists data(cmds,$view)] } {
        return no
    }

    set data(cmds,args) [list Results Graphs $data(cmds,$view)]

    if { $view == "Integrate_Scalar" } {
        lappend data(cmds,args) $::GidPriv(PostSpaceDimension) AllSteps ScalarResult
    } elseif { $view == "Integrate_Vector_Normal" } {
        lappend data(cmds,args) $::GidPriv(PostSpaceDimension) AllSteps NormalIntegration VectorResult
    } elseif { $view == "Integrate_Vector_Tangential" } {
        lappend data(cmds,args) $::GidPriv(PostSpaceDimension) AllSteps TangentialIntegration VectorResult
    }

    set lst_axis [ list X Y]
    if { $view == "Line_Graph"} {
        set lst_axis [ list Y X]
    }

    foreach a $lst_axis {
        set tree $data(wid,tree,$a)
        set sel [$tree selection get]
        if {!$data(cmds,$view,$a,require)} {
            continue
        }
        if {![llength $sel]} {
            return no
        }
        set node [lindex $sel 0]
        set data_node [$tree itemcget $node -data]
        if { $data(cmds,$view,$a,require)==1 } {
            lappend data(cmds,args) [lindex $data_node 0]
            if { [llength $data_node]==2 && [lindex $data_node 1] != "" } {
                #require 1 is only the minimum, if there are two values (result + component) then provide both
                lappend data(cmds,args) [lindex [lindex $data_node 1] 0] ;#first item of list of components
            }
        } elseif { $data(cmds,$view,$a,require)==2 } {
            lappend data(cmds,args) [lindex $data_node 0]
            if { [llength $data_node]==2 } {
                lappend data(cmds,args) [lindex [lindex $data_node 1] 0] ;#first item of list of components
            } elseif { [llength $data_node]==1 } {
                lappend data(cmds,args) [lindex $data_node 0]
            } else {
                #unexpected
                lappend data(cmds,args) [lindex $data_node 0]
            }
        } else {
            #unexpected
            WarnWinText "PostGraphs::AllArgumentsOk, unexpecte require $data(cmds,$view,$a,require)"
        }
    }

    if { $view == "Point_Graph" && $::ViewMenuPriv(pointgraphallsteps) } {
        lappend data(cmds,args) AllSteps
    }

    if { $view == "Line_Graph" && $::ViewMenuPriv(LineGraph2DConstraint)} {
        lappend data(cmds,args) do2Dconstraint
    }

    return yes
}

proc PostGraphs::ChangeAnalysis { } {
    variable data
    set data(analysis,save) [GiD_Info postprocess get cur_analysis]
    set data(analysis,step,save) [GiD_Info postprocess get cur_step]
    GiD_Process MEscape Results AnalysisSel [GetCurrentAnalysis] [GetCurrentStep]
}

proc PostGraphs::RestoreAnalysis { } {
    variable data
    GiD_Process MEscape Results AnalysisSel $data(analysis,save) $data(analysis,step,save)
}

proc PostGraphs::AfterPressFinish { } {
    variable data
    PostGraphs::RestoreAnalysis

    #to update state of graphs window:
    #  FinishButton set all widgets to disable storing its previous state (e.g. disabled if no graphs)
    #  PostGraphs::RefreshFromGiD updated some state to enabled (e.g. is a graph is created)
    #  FinishButton restore to previous state, (e.g set again to disable overwritting the enable!!)
}

proc PostGraphs::CreateAxis { f } {
    variable data

    panedwindow $f.pn -orient horizontal
    ttk::labelframe $f.pn.x -text [_ "X axis"]:
    set data(wid,lf,X) $f.pn.x
    CreateResultTree $f.pn.x X
    ttk::labelframe $f.pn.y -text [_ "Y axis"]:
    set data(wid,lf,Y) $f.pn.y
    CreateResultTree $f.pn.y Y
    $f.pn add $f.pn.x $f.pn.y -sticky "snew"
    grid $f.pn -sticky "snew"
    grid rowconfigure $f 0 -weight 1
    grid columnconfigure $f 0 -weight 1
}

proc PostGraphs::UpdateResults { } {
    variable data

    set view [GetCurrentView]
    # 2Dconstraint checkbox is only available for Line_Graph
    if { $view == "Line_Graph"} {
        # enable 2Dconstraint checkbox
    } else {
        # disable 2Dconstraint checkbox
    }
    switch $view {
        Point_Evolution {
            set msg [_ "All step at %s" [ $data(wid,analysis) get ] ]
            PostGraphs::ClearTree X $msg
            $data(wid,lf,X) configure -text [_ "X axis: All steps"]
            PostGraphs::FillAxis Y
        }
        Point_Complex_Evolution {
            # set msg [_ "All step at %s" [ $data(wid,analysis) get ] ]
            # PostGraphs::ClearTree X $msg
            # $data(wid,lf,X) configure -text [_ "X axis: All steps"]
            # PostGraphs::FillAxis Y
            PostGraphs::FillAxis X
            set msg [_ "Imaginary part of selected result"]
            PostGraphs::ClearTree Y $msg
            $data(wid,lf,Y) configure -text [_ "Y axis: imaginary part"]
        }
        Point_Graph {
            PostGraphs::FillAxis X
            PostGraphs::FillAxis Y
        }
        Border_Graph {
            PostGraphs::FillBorderGraph
        }
        Line_Graph {
            # PostGraphs::ClearTree X [_ "line length"]
            # $data(wid,lf,X) configure -text [_ "X axis: line length"]
            # PostGraphs::FillAxis Y
            PostGraphs::FillLineGraph
        }
        Integrate_Scalar {
            set msg [_ "All step at %s" [ $data(wid,analysis) get ] ]
            PostGraphs::ClearTree X $msg
            $data(wid,lf,X) configure -text [_ "X axis: All steps"]
            PostGraphs::FillAxis Y
        }
        Integrate_Vector_Normal {
            set msg [_ "All step at %s" [ $data(wid,analysis) get ] ]
            PostGraphs::ClearTree X $msg
            $data(wid,lf,X) configure -text [_ "X axis: All steps"]
            PostGraphs::FillAxis Y
        }
        Integrate_Vector_Tangential {
            set msg [_ "All step at %s" [ $data(wid,analysis) get ] ]
            PostGraphs::ClearTree X $msg
            $data(wid,lf,X) configure -text [_ "X axis: All steps"]
            PostGraphs::FillAxis Y
        }
        default {
            PostGraphs::FillResults $data(wid,tree,X) "" 0
            PostGraphs::FillResults $data(wid,tree,Y) "" 0
        }
    }
}

proc PostGraphs::OnSelectNode { Axis node } {
    variable data

    set tree $data(wid,tree,$Axis)
    if {[llength [set info [$tree itemcget $node -data]]]}  {
        $tree selection set $node
        set txt [$tree itemcget $node -text]
        $data(wid,lf,$Axis) configure -text [_ "%s axis: %s" $Axis $txt]
    } else {
        if {[$tree itemcget $node -open]} {
            $tree closetree $node
        } else {
            $tree opentree $node
        }
    }
}

proc PostGraphs::CreateResultTree {f Axis}  {
    variable data

    set c [ CreateScrolledCanvas $f.sw]
    $f.sw configure -relief flat

    set t [Tree $f.sw.t -borderwidth 0]
    $t bindImage <${::acceleratorKey}-1> {}
    $t bindText <${::acceleratorKey}-1> {}
    $t bindImage <1> [list PostGraphs::OnSelectNode $Axis]
    $t bindText <1> [list PostGraphs::OnSelectNode $Axis]
    $t bindImage <Double-Button-1> PostGraphs::ApplyCreate
    $t bindText <Double-Button-1> PostGraphs::ApplyCreate

    set data(wid,tree,$Axis) $t

    AddToScrolledCanvas $f.sw $t
    grid $f.sw -sticky news

    grid rowconfigure $f 0 -weight 1
    grid columnconfigure $f 0 -weight 1
}

proc PostGraphs::ClearTree { axis msg } {
    variable data

    set tree $data(wid,tree,$axis)
    $tree delete [$tree nodes root]
    if {$msg ne ""} {
        $tree insert end root "" -text $msg
    }
}

#to split a result name like "a//b1 b2//c" in its parts a {b1 b2} c
#be careful that a singlE / don't split, like "Velocity (m/s)" not splitted into {Velocity (m} s
#http://wiki.tcl.tk/16249, use \uFFFF as impossible_char in a string
proc PostGraphs::Split { name } {
    return [split [string map [list // \uFFFF] $name] \uFFFF]
}

proc PostGraphs::Join { list } {
    return [join $list //]
}

proc PostGraphs::FillResults {tree results also_components } {
    variable data

    $tree delete [$tree nodes root]
    set view [GetCurrentView]
    set analysis [GetCurrentAnalysis]
    set step [GetCurrentStep]

    $tree delete [$tree nodes root]
    if {![llength $results]} {
        set txt [_ "<No usable results found>"]
        $tree insert end root "" -text $txt
        return
    }

    foreach r $results {
        set levels [PostGraphs::Split $r]
        set nodep root
        foreach l [lrange $levels 0 end-1] {
            set nodeid [PostGraphs::Join [list $nodep [PRToNode $l]]]
            if {![$tree exists $nodeid]} {
                set txt [TranslateResultName $l]
                $tree insert end $nodep $nodeid -text $txt -image [gid_themes::GetImage folder.png small_icons]
            }
            set nodep $nodeid
        }
        if { $also_components } {
            set components [ GiD_Info postprocess get components_list $view $r $analysis $step]
        } else {
            set components ""
        }
        set txt [TranslateResultName [lindex $levels end]]
        if {[llength $components] > 1} {
            $tree insert end $nodep [PRToNode $r] -text $txt -image [gid_themes::GetImage prof1.png]
            set nodep [PRToNode $r]
            foreach c $components {
                set nodeid [PostGraphs::Join [list $nodep [PRToNode $c]]]
                set txt [TranslateResultComponentName $c]
                $tree insert end $nodep $nodeid \
                    -text $txt -image [gid_themes::GetImage point.png small_icons] \
                    -data [list $r $c]
            }
        } else {
            $tree insert end $nodep [PRToNode $r] -text $txt \
                -image [gid_themes::GetImage point.png small_icons] -data [list $r $components]
        }
    }
}

proc PostGraphs::FillAxis { axis } {
    variable data

    set tree $data(wid,tree,$axis)
    set view [GetCurrentView]
    set analysis [GetCurrentAnalysis]
    set step [GetCurrentStep]

    set results {}
    catch {
        set results [ GiD_Info postprocess get results_list $view $analysis $step]
    }
    $data(wid,lf,$axis) configure -text [_ "%s axis" $axis]:

    if { $view == "Integrate_Vector_Normal" || $view == "Integrate_Vector_Tangential" } {
        set also_components 0
    } else {
        set also_components 1
    }
    PostGraphs::FillResults $tree $results $also_components
}

proc PostGraphs::OnChangeView { {force 0} } {
    variable data

    set cur_view [GetCurrentView]
    if {$force || ![info exists data(view,last)] || $data(view,last) ne $cur_view} {
        set data(view,last) $cur_view
        PostGraphs::UpdateResults
    }
    if { $cur_view == "Point_Graph" } {
        grid $data(wid,chkallsteps)
    } else {
        grid remove $data(wid,chkallsteps)
    }
    if { $cur_view == "Line_Graph" } {
        grid $data(wid,cb2Dconstraint)
    } else {
        grid remove $data(wid,cb2Dconstraint)
    }
}

proc PostGraphs::FillBorderGraph { } {
    variable data

    $data(wid,lf,X) configure -text [_ "X axis"]:
    set tree $data(wid,tree,X)
    $tree delete [$tree nodes root]
    set options [list]
    lappend options X_Variation [_ "X Variation"]
    lappend options Y_Variation [_ "Y Variation"]
    lappend options Z_Variation [_ "Z Variation"]
    lappend options LineVariation [_ "Line Variation"]
    foreach {x lx} $options {
        $tree insert end root $x -text $lx -image [gid_themes::GetImage point.png small_icons] -data [list $x]
    }
    PostGraphs::FillAxis Y
}

proc PostGraphs::FillLineGraph { } {
    variable data

    $data(wid,lf,X) configure -text [_ "X axis"]:
    set tree $data(wid,tree,X)
    $tree delete [$tree nodes root]
    set options [list]
    lappend options X_Variation [_ "X Variation"]
    lappend options Y_Variation [_ "Y Variation"]
    lappend options Z_Variation [_ "Z Variation"]
    lappend options LineVariation [_ "Line Variation"]
    lappend options LineProjection [_ "Line Projection"]
    foreach {x lx} $options {
        $tree insert end root $x -text $lx -image [gid_themes::GetImage point.png small_icons] -data [list $x]
    }
    PostGraphs::FillAxis Y
}

proc PostGraphs::OnChangeAnalysis { } {
    variable data
    set cur_ana [GetCurrentAnalysis]
    if {![info exists data(analysis,last)] || \
        $data(analysis,last) ne $cur_ana} {
        set data(analysis,last) $cur_ana
        PostGraphs::UpdateResults
        PostGraphs::FillSteps
    }
}

proc PostGraphs::OnChangeStep { } {
    variable data

    if {![info exists data(step,last)] || $data(step,last) ne $data(step)} {
        set data(step,last) $data(step)
        PostGraphs::UpdateResults
    }
}

proc PostGraphs::CreateView { f } {
    variable data

    ttk::label $f.lview -text [_ "View"]:
    #ComboBox $f.cbview -textvariable PostGraphs::data(view) -editable 0 -borderwidth 1 \
    #    -modifycmd PostGraphs::OnChangeView
    TTKComboBox $f.cbview -textvariable PostGraphs::data(view) \
        -state readonly
    set data(wid,view) $f.cbview
    # no entiendo por que TTKComboBox lo invoca cuando se cambia la propiedad.
    after idle "$f.cbview configure -modifycmd PostGraphs::OnChangeView"

    ttk::label $f.lana -text [_ "Analysis"]:
    #ComboBox $f.cbanalysis -textvariable PostGraphs::data(analysis) -editable 0  \
    #    -borderwidth 1 -modifycmd PostGraphs::OnChangeAnalysis
    TTKComboBox $f.cbanalysis -textvariable PostGraphs::data(analysis) \
        -state readonly -modifycmd PostGraphs::OnChangeAnalysis
    set data(wid,analysis) $f.cbanalysis

    ttk::label $f.lstep -text [_ "Step"]:
    TTKComboBox $f.cbstep -textvariable PostGraphs::data(step) \
        -state readonly -modifycmd PostGraphs::OnChangeStep
    #ComboBox $f.cbstep -textvariable PostGraphs::data(step) -editable 0 -borderwidth 1 \
    #    -modifycmd PostGraphs::OnChangeStep
    set data(wid,cbstep) $f.cbstep


    if { ![info exists ::ViewMenuPriv(pointgraphallsteps)] } {
        set ::ViewMenuPriv(pointgraphallsteps) 1
    }
    ttk::checkbutton $f.chkallsteps -text [_ "All steps"] -variable ::ViewMenuPriv(pointgraphallsteps)
    set data(wid,chkallsteps) $f.chkallsteps

    # Add here the 2Dconstraint checkbox for the Line_Graph
    if { ![info exists ::ViewMenuPriv(LineGraph2DConstraint)] } {
        set ::ViewMenuPriv(LineGraph2DConstraint) 0
    }
    ttk::checkbutton $f.cb2Dconstraint -text [_ "2D constraint"] -variable ::ViewMenuPriv(LineGraph2DConstraint)
    set data(wid,cb2Dconstraint) $f.cb2Dconstraint

    FillViewFrame

    grid $f.lview $f.cbview $f.lana $f.cbanalysis $f.lstep $f.cbstep $f.chkallsteps $f.cb2Dconstraint -sticky ew
    grid columnconfigure $f {1 3 5} -weight 1
    grid remove $f.chkallsteps $f.cb2Dconstraint
}

proc PostGraphs::FillViewFrame { } {
    variable data
    if {[PostGraphs::Alive]} {
        FillAnalysis
        FillSteps
        FillViews
    }
}

proc PostGraphs::FillAnalysis { } {
    variable data

    set all [lsort -dictionary [ GiD_Info postprocess get all_analysis]]
    # set all [ GiD_Info postprocess get all_analysis]
    set curr [ GiD_Info postprocess get cur_analysis]
    set data(analysis,all) $all

    if {![llength $all]} {
        ClearView
    } else {
        # translate the analysis
        set i 0
        set icur 0
        foreach a $all {
            if {$a eq $curr} {
                set icur $i
            }
            lappend all_trans [TranslateResultName $a]
            incr i
        }
        $data(wid,analysis) configure -values $all -labels $all_trans
        #TTKMenubuttonConfigure $data(wid,analysis) -values $all_trans
        #$data(wid,analysis) setvalue "@$icur"
        $data(wid,analysis) set $curr
    }
}

proc PostGraphs::ClearTTKCombo { cb } {
    $cb set ""
    $cb configure -values "" -labels ""
}

proc PostGraphs::ClearView { } {
    variable data
    ClearTTKCombo $data(wid,analysis)
    ClearTTKCombo $data(wid,cbstep)
    ClearTTKCombo $data(wid,view)
    array set data {
        view ""
        view,last ""
        analysis ""
        analysis,last ""
        step ""
        step,last ""
    }
}

proc PostGraphs::GetCurrentAnalysis { } {
    variable data
    return $data(analysis)
}

proc PostGraphs::GetCurrentView { } {
    variable data
    return $data(view)
}

proc PostGraphs::GetCurrentStep { } {
    variable data
    return $data(step)
}

proc PostGraphs::FillSteps { } {
    variable data
    set cur_ana [GetCurrentAnalysis]
    if {$cur_ana ne ""} {
        set curr_st [GiD_Info postprocess get cur_step $cur_ana]
        set all_st [GiD_Info postprocess get all_steps $cur_ana]
        if { $curr_st ni $all_st } {
            set curr_st [lindex $all_st 0]
        }
        $data(wid,cbstep) configure -values $all_st
        $data(wid,cbstep) set $curr_st
    } else {
        ClearView
    }
}

proc PostGraphs::FillViews { } {
    variable data

    set cur_ana [GetCurrentAnalysis]
    if {$cur_ana ne ""} {
        set all [ GiD_Info postprocess get all_graphs_views]
        set data(view,all) $all
        #bloque de traduccion: solo para que RamTranslator los encuentre#
        list [_ "No Graph"] [_ "Point Evolution"] [_ "Point Graph"] \
            [_ "Border Graph"] [_ "Line Graph"] [_ "Integrate Scalar"] \
            [_ "Integrate Vector Normal"] [_ "Integrate Vector Tangential"] \
            [_ "Point Complex Evolution"]
        #fin bloque de traduccion#
        foreach v $all {
            lappend all_trans [_ [regsub -all _ $v " "]]
        }
        $data(wid,view) configure -values $all -labels $all_trans
        # TTKMenubuttonConfigure $data(wid,view) -values $all_trans
        if {[info exists data(view,last)] && $data(view,last) ne ""} {
            #$data(wid,view) setvalue @[lsearch $all $data(view,last)]
            $data(wid,view) set $data(view,last)
        } else {
            set data(view,last) ""
            #$data(wid,view) setvalue first
            $data(wid,view) set [ lindex $all 0 ]
        }
    } else {
        set data(view,all) ""
        ClearView
    }
}

proc PostGraphs::RefreshFromGiD { } {
    # creo que if { [GidUtils::IsTkDisabled] } esta al reves !!!
    # y de hecho llamar a esta funcion no hace nada
    # se esta actualizando todo porque se recrea la ventana cada vez
    # y no hace falta la llamad adicional en C++ a PostGraphs::RefreshFromGiD
    # y ademas no hace falta otra funcion PostGraphs::RefreshFromGiD_after
    # porque no hay after idle
    if { [GidUtils::IsTkDisabled] } {
        #after idle PostGraphs::RefreshFromGiD_after
        #avoid use of after idle if it is not compulsory!!
        PostGraphs::RefreshFromGiD_after
    }
}

proc PostGraphs::RefreshFromGiD_after { } {
    variable data
    if {$data(tmp,applying)} {
        return
    } else {
        if { [PostGraphs::Alive] } {
            if { $::plot_graph } {
                PostGraphs::UpdatePlotGraph
            }
            PostGraphs::FillGraphSets
            PostGraphs::FillGeneralOptions
            PostGraphs::FillCoordinatesOptions
            PostGraphs::FillTree $data(tree)
        }
    }
}

proc PostGraphs::ToggleView { } {
    ::TogglePostGraph
}

proc dFT_Graph { graph_name} {
    # args: which_part graph_name
    # dft( real) --> complex( r, i)
    package require math::fourier
    package require math::complexnumbers

    lassign [ GiD_Graph get $graph_name] x_label y_label lst_x_values lst_y_values x_unit y_unit

    if { [ llength $lst_x_values] == 0} {
        WarnWin [_ "Graph %s does not have any value" $graph_name]
        return
    }

    set dft_complex_values [ math::fourier::dft $lst_y_values]
    set new_lst_x_values {}
    set new_lst_y_values {}
    set x_idx 1
    # switch $which_part {
    #         real {
    #             foreach complex $dft_complex_values {
    #                 lappend new_lst_x_values $x_idx
    #                 lappend new_lst_y_values [ math::complexnumbers::real $complex]
    #                 incr x_idx
    #             }
    #         }
    #         imag {
    #             foreach complex $dft_complex_values {
    #                 lappend new_lst_x_values $x_idx
    #                 lappend new_lst_y_values [ math::complexnumbers::imag $complex]
    #                 incr x_idx
    #             }
    #         }
    #         mod {
    #             foreach complex $dft_complex_values {
    #                 lappend new_lst_x_values $x_idx
    #                 lappend new_lst_y_values [ math::complexnumbers::mod $complex]
    #                 incr x_idx
    #             }
    #         }
    # }
    foreach complex $dft_complex_values {
        lappend new_lst_x_values [ math::complexnumbers::real $complex]
        lappend new_lst_y_values [ math::complexnumbers::imag $complex]
        incr x_idx
    }

    # set new_name "dft.${which_part}( $graph_name)"
    #set label_x "index"
    #set label_y "dft.${which_part}"
    set new_name "dFT( $graph_name)"
    set label_x "dFT.real"
    set label_y "dFT.imag"
    set new_x_unit ""
    set new_y_unit ""
    GiD_Graph create $new_name $label_x $label_y $new_lst_x_values $new_lst_y_values $new_x_unit $new_y_unit
    GiD_Graph show
    # WarnWin "get DFTGraph.${which_part}($graph_name)"
}

proc inverse_dFT_Graph { graph_name } {
    # args: which_part graph_name
    # inv-dFT( r, i) --> ( idx, r)
    package require math::fourier
    package require math::complexnumbers

    lassign [ GiD_Graph get $graph_name] x_label y_label lst_x_values lst_y_values x_unit y_unit

    if { [ llength $lst_x_values] == 0} {
        WarnWin [_ "Graph %s does not have any value" $graph_name]
        return
    }

    set lst_complex_values {}
    foreach r $lst_x_values i $lst_y_values {
        lappend lst_complex_values [ math::complexnumbers::complex $r $i]
    }

    # set dft_complex_values [ math::fourier::dft $lst_complex_values]
    set dft_complex_values [ math::fourier::inverse_dft $lst_complex_values]
    set new_lst_x_values {}
    set new_lst_y_values {}
    set x_idx 1
    # set label_x "index"
    # set label_y "dft.${which_part}"
    # switch $which_part {
    #         real {
    #             set label_x "index"
    #             set label_y "dft.${which_part}"
    #             foreach complex $dft_complex_values {
    #                 lappend new_lst_x_values $x_idx
    #                 lappend new_lst_y_values [ math::complexnumbers::real $complex]
    #                 incr x_idx
    #             }
    #         }
    #         imag {
    #             set label_x "index"
    #             set label_y "dft.${which_part}"
    #             foreach complex $dft_complex_values {
    #                 lappend new_lst_x_values $x_idx
    #                 lappend new_lst_y_values [ math::complexnumbers::imag $complex]
    #                 incr x_idx
    #             }
    #         }
    #         mod {
    #             set label_x "index"
    #             set label_y "dft.${which_part}"
    #             foreach complex $dft_complex_values {
    #                 lappend new_lst_x_values $x_idx
    #                 lappend new_lst_y_values [ math::complexnumbers::mod $complex]
    #                 incr x_idx
    #             }
    #         }
    #         complex {
    #             set label_x "real"
    #             set label_y "imag"
    #             foreach complex $dft_complex_values {
    #                 lappend new_lst_x_values [ math::complexnumbers::real $complex]
    #                 lappend new_lst_y_values [ math::complexnumbers::imag $complex]
    #             }
    #         }
    # }

    set label_x "index"
    set label_y "idFT"
    foreach complex $dft_complex_values {
        lappend new_lst_x_values $x_idx
        lappend new_lst_y_values [ math::complexnumbers::real $complex]
        incr x_idx
    }

    # set new_name "dft-complex.${which_part}( $graph_name)"
    set new_name "idFT( $graph_name)"
    set new_x_unit ""
    set new_y_unit ""
    GiD_Graph create $new_name $label_x $label_y $new_lst_x_values $new_lst_y_values $new_x_unit $new_y_unit
    GiD_Graph show
}

proc PostGraphs::GraphWindow { toplevel_window} {
    variable data

    set graphs [ list [lindex [$data(tree) item text $data(itemselected)] 0]]
    foreach graph $graphs {
        set i 0
        if { ( $toplevel_window == "") || ( $toplevel_window == ".") } {
            set toplevel_window .gid
        }
        set w ${toplevel_window}.gresults-$i
        while { [winfo exists $w] } {
            incr i
            set w .gid.gresults-$i
        }
        ::GraphResults::Create $graph $w
    }
}
