
package require math::linearalgebra

namespace eval Optimization {
    #array to store the history of the optimizer tested points
    variable Evolution
    
    #to write or not this files
    variable WriteLogFile 0
    variable WritePlotFile 1
    
    #
    variable Method CG ;#CG==Conjugate gradient
    variable CG_Tolerance 1e-5
    variable CG_MaxIterations 100
    variable Function
    variable Gradient

    #for GUI
    variable w
    variable xyplot
    variable DesignVariables

    #to use a metamodel instead the true function/gradient
    variable ExperimentResult
    variable NExperiments
    variable ResponseSurfaceDegree

    #for constrained optimization
    variable ActiveConstraint
    variable Weigth ;#penalty coefficients
    variable ActiveConstraint

    #for control of running processes
    variable MaxRunSimultaneous 1
    variable TimeToCheck 5000
    variable _PrevUseMoreWindows   
    variable _state {continue} 
}

#line search: unidimensional minimization from a point x along a fixed direction d
#used as auxiliary procedure by the gradient conjugate optimizer
#input:
#function: objective function to minimize (single objective with x1 ...xn parameters)
#it expect a parameter with a list of the variable values proc f { x } 
#gradient: gradient of the objective function  (df/dx1, df/dx2,...,df/dxn)
#it expect a parameter with a list of the variable values proc g { x } 
#x: values of the variables at the current point
#xl: lower limits of the variables
#xu: upper limits of the variables
#d: direction where to do the unidimensional line search
#f: value of the objective function at x (to avoid recalculate it) 
#tol: stop tolerance for the line search
#maxiter: maximum of iterations for the line search 
#output:
#the scalar value alpha of the minimum point x+alpha*d
proc Optimization::LineSearch { function gradient x xl xu d f tol maxiter } {
    set p 0.1 ;#optimistically expected 10% reduction
    set dmodulus 0.0
    set iiter 0
    set dmodulus [::math::linearalgebra::norm $d]
    set d [::math::linearalgebra::unitLengthVector $d]

    Optimization::PutsLogFile "***Begin LineSearch dir=$d f=$f"
    
    set alpha 0
    set a1 0.0
#     set a2 1.0
    #set f1 [eval [list $function $x]] ;avoid this initial analysis, it's calculated before
    #set wa [eval [list $gradient $x]] ;avoid this initial analysis, it's calculated before
    set f1 $f
    set wa [::math::linearalgebra::scale -1.0 $d]
    set g1 [::math::linearalgebra::dotproduct $wa $d]
    
    set a2 [expr {-$p*abs($f1)/$g1}] ;#fist estimation, instead a2=1.0
    set wa [::math::linearalgebra::axpy $a2 $d $x]
    
    set row 0
    foreach xli $xl xui $xu {
        if { [lindex $wa $row] < $xli } {
            set a2 [expr {($xli-[lindex $x $row])/[lindex $d $row]}]
            set wa [::math::linearalgebra::axpy $a2 $d $x]
            Optimization::PutsLogFile "***LineSearch: set a2 to $a2: x=$wa"
        } elseif { [lindex $wa $row] > $xui } {
            set a2 [expr {($xui-[lindex $x $row])/[lindex $d $row]}]
            set wa [::math::linearalgebra::axpy $a2 $d $x]
            Optimization::PutsLogFile "***LineSearch: set a2 to $a2: x=$wa"
        }
        incr row
    }

    set f2 [eval [list $function $wa]]
    # Optimization::PutsLogFile "***LineSearch: Calculated Function $wa"
    set wa [eval [list $gradient $wa]]
    # Optimization::PutsLogFile "***LineSearch: Calculated Gradient $wa"

    set g2 [::math::linearalgebra::dotproduct $wa $d]    
    
    for {set iiter 1} {$iiter < $maxiter} {incr iiter} {
        set df [expr {$f2-$f1}]
        set dg [expr {$g2-$g1}]
        set dl [expr {$a2-$a1}]
        if { [expr {abs($dl)}] < $tol }  break
        set dl2 [expr {$dl*$dl}]
        set dl3 [expr {$dl2*$dl}]
        set a [expr {(-2.0*$df+($g1+$g2)*$dl)/$dl3}]
        set b [expr {(3.0*$df-(2.0*$g1+$g2)*$dl)/$dl2}]
        set c [expr {$b*$b-3.0*$a*$g1}]
        if { [expr {abs($a)}] > 1e-15 && $c >= 0.0 } {   
            #cubic interpolation
            set alpha [expr {$a1+(-$b+sqrt($c))/(3.0*$a)}]  
        } else {
            if { [expr {abs($a)}] > 1e-15 } {
                #quadratic interpolation
                set alpha [expr {$a1-$g1/(2.0*$b)}]
            } else {
                #linear function
            }
        }    

        set wa [::math::linearalgebra::axpy $alpha $d $x]

        set row 0
        foreach xli $xl xui $xu {
            if { [lindex $wa $row] < $xli } {
                set alpha [expr {($xli-[lindex $x $row])/[lindex $d $row]}]
                set wa [::math::linearalgebra::axpy $alpha $d $x]
                Optimization::PutsLogFile "***LineSearch: set alpha to $alpha: x=$wa"
            } elseif { [lindex $wa $row] > $xui } {
                set alpha [expr {($xui-[lindex $x $row])/[lindex $d $row]}]
                set wa [::math::linearalgebra::axpy $alpha $d $x]
                Optimization::PutsLogFile "***LineSearch: set alpha to $alpha: x=$wa"
            }
            incr row
        }

        set fk [eval [list $function $wa]]
        Optimization::PutsLogFile "***LineSearch $iiter x=$wa f=$fk ($a1 $a2)"
# Optimization::PutsLogFile "***LineSearch: Calculated Function $wa"
        set wa [eval [list $gradient $wa]]
# Optimization::PutsLogFile "***LineSearch: Calculated Gradient $wa"
        set gk [::math::linearalgebra::dotproduct $wa $d]
    
        #must select the appropiate interval??
        if {$gk > 0.0 } {
            if { [expr {abs($alpha-$a2)}] < 1e-15 }  break
            set f2 $fk
            set g2 $gk
            set a2 $alpha
        } else {   
            if { [expr {abs($alpha-$a1)}] < 1e-15 }  break
            set f1 $fk
            set g1 $gk
            set a1 $alpha
        }
        
        #test convergence
        if { [expr {abs($gk)}] < $tol }  break
        if { [expr {abs($a2-$a1)}] < $tol }  break
        if { [expr {abs($f2-$f1)}] < $tol }  break
        if { $iiter >= $maxiter } break    
    }
    if { [expr {abs($dmodulus-1.0)}] > 1e-15 } {
        #set d [::math::linearalgebra::scale $dmodulus $d]
        set alpha [expr {$alpha/$dmodulus}]
    }
    Optimization::PutsLogFile "***End LineSearch $iiter iter"
    return $alpha
}

#minimization of an objective function (single objective with x1 ...xn parameters)
#input:
#function: objective function to minimize (single objective with x1 ...xn parameters)
#it expect a parameter with a list of the variable values proc f { x } 
#gradient: gradient of the objective function  (df/dx1, df/dx2,...,df/dxn)
#it expect a parameter with a list of the variable values proc g { x } 
#x: values of the variables at the initial seed point
#xl: lower limits of the variables
#xu: upper limits of the variables 
#tol: stop tolerance for the line search
#maxiter: maximum of iterations for the line search 
#method: some variants: "FLETCHER-REEVES", "POLAK-RIBIERE", "HESTENES-STIEFEL"
#output:
#the location x of the minimum
proc Optimization::MinimizeCG { function gradient x xl xu {tol 1e-5} {maxiter 1000} {method "FLETCHER-REEVES"} } {
    variable _Stop
    
    set fail 0
    set iiter 0
    set n [llength $x]
    set f [eval [list $function $x]]
    set r [::math::linearalgebra::scale -1.0 [eval [list $gradient $x]]]
    set d $r
    set r2 [::math::linearalgebra::dotproduct $r $r]

    Optimization::PutsLogFile "MinimizeCG iiter=$iiter x=$x f=$f r2=$r2 d=$d"
    Optimization::ResetResults
    set firstiter 1
    Optimization::AddToResults $x $f
    Optimization::PutsPlotFile [list $x $f]
    
    for {set iiter $firstiter} {$iiter < $maxiter} {incr iiter} {
        if { $Optimization::_Stop } {
            break
        }
        if { $method == "POLAK-RIBIERE" } {
            set aux $r ;#-grad
        }
        set alpha [LineSearch $function $gradient $x $xl $xu $d $f $tol $maxiter]
        set x [::math::linearalgebra::axpy $alpha $d $x]
        #force variable limits changing direction
        set row 0
        foreach xi $x xli $xl xui $xu {
            if { $xi < $xli } {
                Optimization::PutsLogFile "***MinimizeCG set x($row) lower limit to $xli"
                ::math::linearalgebra::setelem x $row $xli
            } elseif { $xi > $xui } {
                Optimization::PutsLogFile "***MinimizeCG set x($row) upper limit to $xui"
                ::math::linearalgebra::setelem x $row $xui
            }
            incr row
        }
        set fprev $f
        set f [eval [list $function $x]]
        set tol1 [expr {$tol*(1.0+abs($f))}]
        if { $r2 < $tol1 } break
        #         controlar mas condiciones de parada, por ejemplo
        #         fprev-f<eps*(1+abs(f))
        #         xprev*x<sqrt(eps)*(1+x*x)
        #         g*g<pow(eps,1.0/3.0)*(1+abs(f))
        set r [::math::linearalgebra::scale -1.0 [eval [list $gradient $x]]]
        set r2prev $r2
        set r2 [::math::linearalgebra::dotproduct $r $r]

        Optimization::PutsLogFile "MinimizeCG iiter=$iiter x=$x f=$f r2=$r2 d=$d $alpha"
        Optimization::AddToResults $x $f
        Optimization::PutsPlotFile [list $x $f]
        if { [expr {$iiter%$n}] == 0 } {
            #restart conjugate gradient each n iter
            Optimization::PutsLogFile "***restart conjugate gradient"
            set d $r
        } else {
            if { $method == "FLETCHER-REEVES" } {
                #Fletcher-Reeves:(G*G)/(Gprev*Gprev)
                set beta [expr {$r2/$r2prev}]
            } elseif { $method == "POLAK-RIBIERE" } {
                #Polak-Ribiere:(G*(G-Gprev))/(Gprev*Gprev)
                set c [::math::linearalgebra::sub $aux $r]
                set a [::math::linearalgebra::dotproduct $r $c]
                set beta [expr {-$a/$r2prev}]
                if { $beta < 0.0 } {
                    set beta 0.0
                }
            } elseif { $method == "HESTENES-STIEFEL" } {
                #Hestenes-Stiefel:(G*(G-Gprev))/(d*(G-Gprev))
                set c [::math::linearalgebra::sub $aux $r]
                set a [::math::linearalgebra::dotproduct $r $c]
                set b [::math::linearalgebra::dotproduct $d $c]
                set beta [expr {-$a/$b}]
                if { $beta < 0.0 } {
                    set beta 0.0
                }
            }
            set d [::math::linearalgebra::axpy $beta $d $r]
        }
        #maximum slope test
        #set d $r
    } 
    return $x
}

#to calculate approximated gradient by finite difference centered stencil
#to be used for calculate gradiend if analytical derivatives are not available
proc Optimization::GradientFiniteDifferences { function x } {
    set incre 1e-5
    set grad {}
    foreach xi $x {
        set fnext [eval [list $function [expr {$xi+$incre}]]]
        set fprev [eval [list $function [expr {$xi-$incre}]]]
        lappend grad [expr {($fnext-$fprev)/(2.0*$incre)}]
    }
    return $grad
}

#classical Rosenbrocks parabolic valley ("banana") function used for testing
proc Optimization::FunctionRosenbrocks { x } {
    set a [lindex $x 0]
    set b [lindex $x 1]
    set a2 [expr {$a*$a}]
    return [expr {100.0*($b-$a2)*($b-$a2)+((1.0-$a)*(1.0-$a))}]
}

proc Optimization::GradientRosenbrocks { x } {
    set a [lindex $x 0]
    set b [lindex $x 1]
    set a2 [expr {$a*$a}]
    set grad [list [expr {-2.0*(1.0-$a)-400.0*$a*($b-$a2)}] [expr {200.0*($b-$a2)}]]
    return $grad
}

#log file for debug use
proc Optimization::OpenLogFile { } {
    variable logfile
    set filename [gid_cross_platform::get_unused_tmp_filename optimization .log]
    #set filename {C:\tmp\tunel_calsef.gid\opt.log}
    set logfile [open $filename w]
}

proc Optimization::CloseLogFile { } {
    variable logfile
    if { [info exists logfile] } {
        close $logfile
        unset logfile
    }
}

proc Optimization::PutsLogFile { txt } {
    variable logfile
    if { [info exists logfile] } {
        puts $logfile $txt
        flush $logfile
    }
}

#to save to disk the history of tested points 
#interesting if the program crash or is stopped, and to restart from the last point
proc Optimization::OpenPlotFile { } {
    variable plotfile
    set filename [gid_cross_platform::get_unused_tmp_filename optimization .plt]
    #set filename {C:\tmp\tunel_calsef.gid\opt.plt}
    set plotfile [open $filename w]
}

proc Optimization::ClosePlotFile { } {
    variable plotfile
    if { [info exists plotfile] } {
        close $plotfile
        unset plotfile
    }
}

proc Optimization::PutsPlotFile { txt } {
    variable plotfile
    if { [info exists plotfile] } {
        puts $plotfile $txt
        flush $plotfile
    }
}

proc Optimization::RestartFromPlotFile { filename } {
    variable Evolution
    array unset Evolution
    if { [file exists $filename] } {
        set fp [open $filename r]
        set a [read $fp]
        foreach ln [split $a \n] {
            if { [llength $ln] != 2 } break
            lappend Evolution(x) [lindex $ln 0]
            lappend Evolution(f) [lindex $ln 1]
        }
        close $fp
    }
}

#to reset the array to store the history of the optimizer tested points
proc Optimization::ResetResults { } {
    variable Evolution
    array unset Evolution
}

#to add a point and value to the array to store the history of the optimizer tested points
proc Optimization::AddToResults { x f } {
    variable Evolution
    lappend Evolution(x) $x
    lappend Evolution(f) $f
}

#open a graphical window to set parameters and see evolution plot
#the optimization procedures could be also run without any window
proc Optimization::OpenGUI { { parent .gid } } {
    package require tablelist_tile
    package require Plotchart
    package require BWidget

    variable w
    variable xyplot
    variable Evolution
    variable WriteLogFile
    variable WritePlotFile
    variable Method
    variable CG_Tolerance
    variable CG_MaxIterations
    variable Function
    variable Gradient

    set w $parent.optimization
    if { [winfo exists $w] } { destroy $w } 
    InitWindow2 $w -title [_ "Optimization"] \
        -geometryvariable PreOptimizationWindowGeom \
        -initcommand Optimization::OpenGUI  
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0

    set def_back [$w cget -background]
    NoteBook $w.nb -internalborderwidth 1 -activebackground [CCColorActivo $def_back]
    
    set fvariables [$w.nb insert end variables -text [_ "Design variables"]]
    set fgraph [$w.nb insert end graph -text [_ "Graph"]]
    if { ![info exists Optimization::WritePlotFile] } { set Optimization::WritePlotFile 0 }
    ttk::checkbutton $fvariables.cbplotfile -variable Optimization::WritePlotFile -text [_ "Plot file"]
    if { ![info exists Optimization::WriteLogFile] } { set Optimization::WriteLogFile 0 }
    ttk::checkbutton $fvariables.cblogfile -variable Optimization::WriteLogFile -text [_ "Log file"]
    ttk::combobox $fvariables.method -textvariable Optimization::Method -values {CG}
    ttk::frame $fvariables.fcg
    ttk::entry $fvariables.fcg.maxiterations -textvariable Optimization::CG_MaxIterations
    ttk::entry $fvariables.fcg.tolerance -textvariable Optimization::CG_Tolerance
    ttk::entry $fvariables.fcg.f -textvariable Optimization::Function
    ttk::entry $fvariables.fcg.g -textvariable Optimization::Gradient
    set sw [ScrolledWindow $fvariables.lf \
            -auto both -relief sunken -borderwidth 2]
    
    set tl [tablelist::tablelist $sw.lb \
            -exportselection 0 \
            -columns [list \
            15 [_ "Name"] center \
            15 [_ "Lower bound"] left \
            15 [_ "Upper bound"] left \
            15 [_ "Initial value"] left \
            ] \
            -labelcommand tablelist::sortByColumn -selectmode extended \
            -stretch end -relief flat -labelrelief flat -showseparators 1 \
            -borderwidth 0 ]              
            
    $tl columnconfigure 0 -editable 1
    $tl columnconfigure 1 -editable 1 
    $tl columnconfigure 2 -editable 1 
    $tl columnconfigure 3 -editable 1 

    $sw setwidget $tl

    grid $fvariables.cbplotfile -sticky w
    grid $fvariables.cblogfile -sticky w
    grid $fvariables.method -sticky ew
    grid $fvariables.fcg.maxiterations -sticky ew
    grid $fvariables.fcg.tolerance -sticky ew
    grid $fvariables.fcg.f -sticky ew
    grid $fvariables.fcg.g -sticky ew
    grid $fvariables.fcg -sticky new -columnspan 2 
    grid columnconfigure $fvariables.fcg 0 -weight 1
    
    grid $sw -sticky nsew
    grid columnconfigure $fvariables 0 -weight 1
    grid rowconfigure $fvariables 4 -weight 1

    if { ![info exists ::Optimization::Evolution(currentplot)] } {
        set ::Optimization::Evolution(currentplot) "f"
    }
    ttk::frame $fgraph.frb
    ttk::radiobutton $fgraph.frb.rb1 -variable ::Optimization::Evolution(currentplot) -value "f" -text "f" 
    ttk::radiobutton $fgraph.frb.rb2 -variable ::Optimization::Evolution(currentplot) -value "dx" -text "|dx|"
    ttk::radiobutton $fgraph.frb.rb3 -variable ::Optimization::Evolution(currentplot) -value "x1_vs_x2" -text "x1 vs x2"

    canvas $fgraph.c -background white

    grid $fgraph.frb.rb1 $fgraph.frb.rb2 $fgraph.frb.rb3 -sticky w
    grid $fgraph.frb -sticky new
    grid $fgraph.c -sticky nsew
    grid columnconfigure $fgraph 0 -weight 1
    grid rowconfigure $fgraph 1 -weight 1

    ttk::frame $w.buts -style BottomFrame.TFrame
    ttk::button $w.buts.b1 -text [_ "Start"] -command [list Optimization::Start $w] -style BottomFrame.TButton
    ttk::button $w.buts.b2 -text [_ "Close"] -command [list destroy $w] -style BottomFrame.TButton    
    grid $w.buts.b1 $w.buts.b2 -padx 2 -pady 3

    grid $w.nb -sticky nsew
    grid $w.buts -sticky ew
    grid anchor $w.buts center
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1
    
    $w.nb raise variables
    bind $fgraph.c <Configure> [list Optimization::Plot $fgraph.c]
    trace add variable ::Optimization::Evolution(currentplot) write "Optimization::Plot $fgraph.c ;#"
}

proc Optimization::DestroyGUI { W w } {
    variable Evolution
    if { $W != $w } return
    set fgraph "$w.nb.fgraph"
    trace remove variable ::Optimization::Evolution(currentplot) write "Optimization::Plot $fgraph.c ;#"
}


proc Optimization::Start { w } {
    variable Function
    variable Gradient
    variable DesignVariables
    variable WriteLogFile
    variable WritePlotFile
    variable Method
    variable CG_Tolerance
    variable CG_MaxIterations
    
    variable _Stop
    $w.buts.b1 configure -text [_ "Stop"] -command [list Optimization::Stop $w]
    set Optimization::_Stop 0

    Optimization::FillVariablesFromGUI

    if { $WriteLogFile} {
        Optimization::OpenLogFile
    }
    if { $WritePlotFile} {
        Optimization::OpenPlotFile
    }

    if { [info exists ::GIDDEFAULT] } {
        GidUtils::DisableGraphics
    }   
   
    set x {}
    set xl {}
    set xu {}
    foreach varname [array names DesignVariables] {
        lappend xl [lindex $DesignVariables($varname) 0]
        lappend xu [lindex $DesignVariables($varname) 1]
        lappend x [lindex $DesignVariables($varname) 2]
    }

    if { $Method == "CG" } {
        set result [MinimizeCG $Function $Gradient $x $xl $xu $CG_Tolerance $CG_MaxIterations "FLETCHER-REEVES"]    
    } else {
        set result ""
        WarnWinText "This method is not implemented"        
    }

    if { [info exists ::GIDDEFAULT] } {
        GiD_Set UseMoreWindows 1 ;#por si acaso lo vuelvo a activar
        GidUtils::EnableGraphics 
    }
    
    if { $WritePlotFile} {
        Optimization::ClosePlotFile
    }
    if { $WriteLogFile} {
        Optimization::CloseLogFile
    }
    if { $_Stop == 0 } {
        $w.buts.b1 configure -text [_ "Start"] -command [list Optimization::Start $w]
    }
    WarnWinText $result
}

proc Optimization::Stop { w } {
    variable _Stop      
    $w.buts.b1 configure -text [_ "Start"] -command [list Optimization::Start $w]
    set Optimization::_Stop 1
}

proc Optimization::Plot { c } {
    variable Evolution
    variable xyplot

    if { ![info exists Evolution(f)] } {        
        return
    }
   
    if { ![info exists Evolution(currentplot)] } {       
        set Evolution(currentplot) "f"
    }
    set what $Evolution(currentplot)

    set graphpoints ""
    if { $what == "f" } {
        set title [_ "Optimization evolution"] 
        set xtext [_ "Iteration"]
        set ytext [_ "Objective"]    
        set i 0
        foreach y $Evolution(f) {
            lappend graphpoints [list $i $y]
            incr i
        }    
        set xformat "%.0f" ;# -ticklines true -ticklength 3 -scale $xaxis
        set yformat "%.3f" ;# -ticklines true -ticklength 3
    } elseif { $what == "dx" } {
        set title [_ "Optimization evolution"] 
        set xtext [_ "Iteration"]
        set ytext [_ "|dx|"]
        set i 1
        set ptprev ""
        foreach pt $Evolution(x) {
            if { $ptprev != "" } {
                set y [::math::linearalgebra::norm [::math::linearalgebra::sub $pt $ptprev]]
                lappend graphpoints [list $i $y]
                incr i
            }
            set ptprev $pt
        }
        set xformat "%.0f" ;# -ticklines true -ticklength 3 -scale $xaxis
        set yformat "%.3f" ;# -ticklines true -ticklength 3
    } elseif { $what == "x1_vs_x2" } {
        set title [_ "Optimization evolution"] 
        set xtext x1
        set ytext x2
        set graphpoints $Evolution(x)
        set xformat "%.3f" ;# -ticklines true -ticklength 3 -scale $xaxis
        set yformat "%.3f" ;# -ticklines true -ticklength 3
    }

    if { $graphpoints == "" } {
        return
    }

    $c delete all
    set xmin 1e20
    set xmax -1e20
    set ymin 1e20
    set ymax -1e20
    foreach pt $graphpoints {
        set x [lindex $pt 0]
        set y [lindex $pt 1]
        if { $xmax < $x } {
            set xmax $x
        }
        if { $xmin > $x } {
            set xmin $x
        }
        if { $ymax < $y } {
            set ymax $y
        }
        if { $ymin > $y } {
            set ymin $y
        }
    }
    set nincre [llength $graphpoints]
    incr nincre -1
    if { $nincre > 15 } {
        set nincre 15
    } elseif { $nincre < 1 } {
        set nincre 1
    }
    
    set xstep [expr {($xmax-$xmin)/double($nincre)}]
    set ystep [expr {($ymax-$ymin)/double($nincre)}]
    
    set xaxis [list $xmin $xmax $xstep]
    set yaxis [list $ymin $ymax $ystep]
    
    set margin 40
    set pxmin $margin
    set pymin $margin
    set pxmax [expr {[$c cget -width]-$margin}] ;#no cambian width height !!!
    set pymax [expr {[$c cget -height]-$margin}]
    
    set xyplot [::Plotchart::createXYPlot $c [list $xmin $xmax $xstep] [list $ymin $ymax $ystep]]
    #::Plotchart::viewPort $c $pxmin $pymin $pxmax $pymax 
    
    $xyplot title $title
    $xyplot xtext $xtext
    $xyplot ytext $ytext
    $xyplot xconfig -format $xformat ;# -ticklines true -ticklength 3 -scale $xaxis
    $xyplot yconfig -format $yformat ;# -ticklines true -ticklength 3
    #$xyplot grid {{1 2 3} {1 2 3}}  {{20 20 20} {21 21 21}}
    #type: line, symbol, or both.
    #symbol: plus, cross, circle, up, down, dot, upfilled or downfilled 
    $xyplot dataconfig $what -colour red -type both -symbol plus    
    #$xyplot xconfig -scale $xaxis
    #$xyplot yconfig -scale $yaxis
    foreach pt $graphpoints {
        $xyplot plot $what [lindex $pt 0] [lindex $pt 1]
    }
}

proc Optimization::PopulateTestRosenbrocks { } {
    variable Function
    variable Gradient
    variable DesignVariables 
  
    variable Weigth
    variable ResponseSurfaceDegree
    variable ActiveConstraint
    
    
    set ::Optimization::WritePlotFile 1
    set ::Optimization::WriteLogFile 0
    set ::Optimization::Function Optimization::FunctionRosenbrocks
    set ::Optimization::Gradient Optimization::GradientRosenbrocks 
    set ::Optimization::Method CG
    set ::Optimization::CG_Tolerance 1e-5
    set ::Optimization::CG_MaxIterations 100
    
    
    if { [info exists DesignVariables] } {
        array unset DesignVariables
    }
    set DesignVariables(x1) {0.2 1.5 0.5}
    set DesignVariables(x2) {0.2 1.5 0.5}
    
    
    set Weigth(0) 1e4
    set Weigth(1) 1e-5
    set Weigth(2) 1.0

    set ResponseSurfaceDegree 1 ;#for two variables: a*x+b*y+c
    #set ResponseSurfaceDegree 2 ;#for two variables: a*x^2+b*y^2+c*xy+d*x+e*y+f
    set ActiveConstraint(0) 1
    set ActiveConstraint(1) 1
    set ActiveConstraint(2) 1

    FillGUIFromVariables    
}

proc Optimization::FillVariablesFromGUI { } {
    variable Function
    variable Gradient
    variable DesignVariables
    variable CurrentDesign
    variable w
    variable WriteLogFile
    variable WritePlotFile

    set fvariables $w.nb.fvariables
    
    array unset DesignVariables
    set tl $w.nb.fvariables.lf.lb
    set values [$tl get 0 end]
    foreach line $values {
        set varname [lindex $line 0]
        set DesignVariables($varname) [lrange $line 1 end]
        set CurrentDesign($varname) [lindex $DesignVariables($varname) 2]
    }
}

proc Optimization::FillGUIFromVariables { } {
    variable Function
    variable Gradient
    variable DesignVariables 
    variable CurrentDesign
    variable w
    variable WriteLogFile
    variable WritePlotFile

    set fvariables $w.nb.fvariables
    #must set the combo from Function and Gradient...
    #$fvariables.fcg.f insert end $Function
    #$fvariables.fcg.g insert end $Gradient
    #$fvariables.fcg.g insert end "Finite differences"

    set tl $w.nb.fvariables.lf.lb
    $tl delete 0 end
    foreach varname [lsort -dictionary [array names DesignVariables]] {
        set CurrentDesign($varname) [lindex $DesignVariables($varname) 2]
        $tl insert end [concat $varname $DesignVariables($varname)]
    }
}

#To create a metamodel (polynomial) to be used to evaluate f and grad fastly 
#with an approximated model instead the real one

proc Optimization::GetResponseSurfaceCoefficients { x } {
    variable ResponseSurfaceCoefficients
    Optimization::CalcResponseSurface $x
    vwait ::Optimization::ResponseSurfaceCoefficients
# Optimization::PutsLogFile "GetResponseSurfaceCoefficients After vwait"
    return $ResponseSurfaceCoefficients
}

proc Optimization::GetResponseSurfaceValue { coeff x } {
    variable ResponseSurfaceDegree
    set res ""
    if { $ResponseSurfaceDegree == 1 } {
        #for two variables: a*x+b*y+c
        set v [concat $x 1.0]
        set res [::math::linearalgebra::dotproduct $coeff $v]
    } elseif { $ResponseSurfaceDegree == 2 } {
        #for two variables: a*x^2+b*y^2+c*xy+d*x+e*y+f
        foreach {x0 x1} $x break
        set v [list [expr {$x0*$x0}] [expr {$x1*$x1}] [expr {$x0*$x1}] $x0 $x1 1.0]
        set res [::math::linearalgebra::dotproduct $coeff $v]
    }
    return $res
}

proc Optimization::GetResponseSurfaceGradient { coeff x } {
    variable ResponseSurfaceDegree
    set res ""
    if { $ResponseSurfaceDegree == 1 } {
        #for two variables: a*x+b*y+c --> {a b}
        set res [lrange $coeff 0 1]
    } elseif { $ResponseSurfaceDegree == 2 } {
        #for two variables: a*x^2+b*y^2+c*xy+d*x+e*y+f --> {2*a*x+c*y+d  2*b*xy+c*x+e}
        foreach {x0 x1} $x break
        foreach {a b c d e f} $coeff break
        set res [list [expr {2*$a*$x0+$c*$x1+$d}] [expr {2*$ba*$x1+$c*$x0+$e}]]
    }
    return $res
}

proc Optimization::CalcResponseSurface { x } {
    variable DesignVariables
    variable CurrentDesign
    variable ExperimentResult
    variable NExperiments
    variable CalculationQueue
    variable _StateMultipleCalculation
    variable _PrevUseMoreWindows
    
    set _PrevUseMoreWindows [GiD_Set UseMoreWindows]
    GiD_Set UseMoreWindows 0 ;#to hide end analysis window

    set CalculationQueue ""
    catch { array unset ExperimentResult }

    set _StateMultipleCalculation "calculating"

    
    GiD_RegisterEvent GiD_Event_AfterRunCalculation Optimization::AfterRunCalculation
       
    set ndesignvars [llength [array names DesignVariables]]
    set degree 1
    set exploration FULL_FACTORIAL
    if { $exploration == "FULL_FACTORIAL" } {
        #set NExperiments [expr {int(pow(2,$ndesignvars))}]
        set NExperiments [expr {1<<$ndesignvars}]
        set stack 0
        while { [llength $stack] > 0 } {
            if { [llength $stack] <$ndesignvars } {
                lappend stack 0
            } else {
                #apply this factorial stack combination
                set ivar 0
                foreach varname [array names DesignVariables] i $stack {   
                    #set CurrentDesign($varname) [lindex $DesignVariables($varname) $i]  
                    set incre [expr {0.1*([lindex $DesignVariables($varname) 1]-[lindex $DesignVariables($varname) 0])}]
                    set CurrentDesign($varname) [expr {[lindex $x $ivar]+$incre*(2*$i-1)}]
                    if { $CurrentDesign($varname) < [lindex $DesignVariables($varname) 0] } {
                        set CurrentDesign($varname) [lindex $DesignVariables($varname) 0]  
                    } elseif { $CurrentDesign($varname) > [lindex $DesignVariables($varname) 1] } {
                        set CurrentDesign($varname) [lindex $DesignVariables($varname) 1]  
                    }
                    #WarnWinText "$varname $CurrentDesign($varname)"   
                    incr ivar
                }
                Optimization::UpdateModel
                set tmpname [Optimization::SaveTmpModel]
                Optimization::AddToCalculationQueue $tmpname
                
                #
                set lastitem [lindex $stack end]
                incr lastitem
                lset stack end $lastitem
                while { $lastitem > 1 } {
                    set stack [lrange $stack 0 end-1]
                    if { [llength $stack] == 0 } break
                    set lastitem [lindex $stack end]
                    incr lastitem
                    lset stack end $lastitem              
                }
            }
        }              
    }   

    
    GiD_UnRegisterEvent GiD_Event_AfterRunCalculation Optimization::AfterRunCalculation
    while { $_StateMultipleCalculation == "calculating" } {
        vwait ::Optimization::_StateMultipleCalculation
    }
    
    Optimization::FinishCalcResponseSurface
}

proc Optimization::FinishCalcResponseSurface { } {
    variable ExperimentResult
    variable DesignVariables
    variable NExperiments
    variable _PrevUseMoreWindows
    variable ResponseSurfaceCoefficients
# Optimization::PutsLogFile "***FinishCalcResponseSurface"
   
    GiD_Set UseMoreWindows $_PrevUseMoreWindows
#     WarnWinText "Experiments Finished"
#     foreach name [array names ExperimentResult] {
#         WarnWinText "$name $ExperimentResult($name)"
#     }
    
    set nrows $NExperiments
    set ncols [llength [array names DesignVariables]]
    incr ncols ;#three variables a,b,c (a*x+b*y+c)
    set a [::math::linearalgebra::mkMatrix $nrows $ncols 1.0]
    set row 0
    foreach name [array names ExperimentResult] {
        set col 0
        foreach newvalue [split $name _] {
            ::math::linearalgebra::setelem a $row $col $newvalue
            incr col
        }
        incr row
    }
    
    set at [::math::linearalgebra::transpose $a]

    set coeff {}
    for {set i 0} {$i < 3} {incr i} {  
        set b [::math::linearalgebra::mkVector $nrows]
        set row 0
        foreach name [array names ExperimentResult] {
            set newvalue [lindex $ExperimentResult($name) $i]
            ::math::linearalgebra::setelem b $row $newvalue
            incr row
        }
        set atb [::math::linearalgebra::matmul $at $b]
        set ata [::math::linearalgebra::matmul $at $a]
        lappend coeff [::math::linearalgebra::solveGauss $ata $atb]
    }
    

#     WarnWinText "coeff $coeff"
#     Optimization::PutsLogFile "***set ResponseSurfaceCoefficients $coeff"
    #update ;#vwait sometimes not work, try to aid
    after idle [list set ::Optimization::ResponseSurfaceCoefficients $coeff]
    return $coeff
}




#procedures that can be used only running inside the GiD Tcl interpreter 
proc Optimization::ReadModel { name  } {
    if { [GiD_ModifiedFileFlag get] } {
        GiD_Process Mescape Files Read No $name escape
    } else {
        GiD_Process Mescape Files Read $name escape
    }    
}

proc Optimization::UpdateModel { } {
    #regenerate the model and mesh with the current design variables
    variable CurrentDesign
    if { [GetCurrentPrePostMode] != "PRE" }  {
        DoPreprocess Yes
    }
    Optimization::ReadModel "C:/tmp/tunel_calsef.gid"
    set OldCreateAlwaysNewPoint [GiD_Set CreateAlwaysNewPoint]
    GiD_Set CreateAlwaysNewPoint 1
    set newpos "0,[expr 7+$CurrentDesign(x1)],0"
    GiD_Process Mescape Geometry Edit MovePoint Fjoin 9 FnoJoin $newpos escape
    set newpos "[expr 10+$CurrentDesign(x2)],-1,0"
    GiD_Process Mescape Geometry Edit MovePoint Fjoin 5 FnoJoin $newpos escape
    GiD_Set CreateAlwaysNewPoint $OldCreateAlwaysNewPoint    

    set size DefaultSize
    if { [GidUtils::ExistsMesh] } {
        GiD_Process Mescape Meshing Generate Yes $size
    } else {
        GiD_Process Mescape Meshing Generate $size
    }
}

proc Optimization::SaveTmpModel { } {
    variable CurrentDesign
    if { [GetCurrentPrePostMode] != "PRE" }  {
        DoPreprocess Yes
    }
    set values {}
    foreach var [array names CurrentDesign] {
        lappend values $CurrentDesign($var)
    }
    set name [join $values _].gid
    set tmpname [file join "C:/tmp/tunel_calsef.gid" $name]
    GiD_Process Mescape Files SaveAs $tmpname escape
    return $tmpname
}

proc Optimization::AddToCalculationQueue { name } {
    variable CalculationQueue
#     Optimization::PutsLogFile "***AddToCalculationQueue $name"
    lappend CalculationQueue $name
    Optimization::Calculate
}

proc Optimization::Calculate { } {
    variable MaxRunSimultaneous
    variable TimeToCheck
    variable CalculationQueue
    variable _state
     
    if { [info exists ::RunProcInfo] && [llength $::RunProcInfo] >= $MaxRunSimultaneous && \
        [llength $CalculationQueue] > 0} {
        after $TimeToCheck [list Optimization::Calculate]
        return
    }

    if { [info exists _state] && $_state == "wait" } {
        return
    }
    set _state "wait"
    set name [lindex $CalculationQueue 0]
    if { $name == "" } {
        set _state "continue"
        return
    }
    set CalculationQueue [lrange $CalculationQueue 1 end]

    if { [GetCurrentPrePostMode] !="PRE" }  {
        DoPreprocess Yes
    }
    
    Optimization::ReadModel $name   

    # Optimization::PutsLogFile "***Calculate $name"
    PWStartProc 0
    set _state "continue"
}

proc Optimization::AfterRunCalculation { basename dir problemtypedir where error errorfilename} {
    variable ExperimentResult
    variable NExperiments
    variable CalculationQueue
    variable _StateMultipleCalculation
    # Optimization::PutsLogFile "***AfterRunCalculation $dir"


    set dir [lindex $dir 0] ;#parece que sobran llaves por error
    set name [file rootname $dir]
    set ExperimentResult($basename) [Optimization::GetSomeResults $name]
    if { [file extension $dir] == ".gid" } { 
        #miro extension para estar mas seguro de no borrar nada indebido
        file delete -force $dir
    }
    if { [llength [array names ExperimentResult]] == $NExperiments } {        
        set _StateMultipleCalculation "Finish"
    } elseif { [llength $CalculationQueue] > 0} {            
        Optimization::Calculate 
    }    
}

proc Optimization::GetSomeResults { name } { 
    set maxdisplacement 0
    set maxstress 0
    set area 0
    if { [GetCurrentPrePostMode] !="PRE" }  {
        DoPreprocess Yes
    } 

    Optimization::ReadModel $name

    set num 1
    set infosurf [GiD_Info list_entities -more surfaces $num]
    regexp {Area=([^\n]*)\n} $infosurf dummy area

    if { [GetCurrentPrePostMode] !="POST" }  {
        DoPostprocess
    }

    #set analysis "Static"
    # set analysis "Analisis_de_carga"
    set analysis "Load_analysis"
    set step 1

    set cont 0 
    while { [GetCurrentPrePostMode] =="PRE" }  { }  {
        incr cont
        if { $cont > 1000 } break ;#to avoid infinte loops
    }

    if { $cont > 0 } {
        WarnWinText "Optimization::GetSomeResults cont=$cont"
    }


    if { 1 } {
#         set result "Displacements_(m)"
        set result "DESPLAZAMIENTOS"
        set maxdisplacement [lindex [GiD_Result get -max [list $result $analysis $step]] 3]
        
#         set result "Axial_Force_(N/m)"
#         set maxstress [lindex [GiD_Result get -max [list $result $analysis $step]] 3]
        set result "Tensiones_SET1"
        set maxstress [lindex [GiD_Result get -max [list $result $analysis $step]] 6]
    }
    
    if { 0 } {
        set result "Displacements_(m)"
        set component "|Displacements|"
        GiD_Process Mescape Results AnalysisSel $analysis $step ContourFill $result $component
        set minmax [GiD_Info Postprocess Get cur_contour_limits]   
        set maxdisplacement [lindex $minmax 1]  

        set result "Axial_Force_(N/m)"
        set component "|Axial_Force|"
        GiD_Process Mescape Results AnalysisSel $analysis $step ContourFill $result $component
        set minmax [GiD_Info Postprocess Get cur_contour_limits]   
        set maxstress [lindex $minmax 1]                       
        #WarnWinText "Min displ=[lindex $minmax 0] Max displ=[lindex $minmax 1]"
        
    }
    if { 0 } {
        set result "Displacements_(m)"
        set components [GiD_Info postprocess get cur_components_list Displacements_(m)]
        set component [lindex $components 3] ;#"|Displacements|"
        
        set maxdisplacement [lindex [GiD_Result get -max [list $result $analysis $step]] 3]

        set result_view Display_Vectors
        if { [lsearch [GiD_Info Postprocess Get all_results_views] $result_view] == -1 } {
            WarnWinText "Bad result_view $result_view"
            return
        }    
        foreach analysis [GiD_Info Postprocess Get all_analysis] {
            foreach step [GiD_Info Postprocess Get all_steps $analysis] {            
                set results [GiD_Info Postprocess Get results_list $result_view $analysis $step]
                if { [lsearch $results $result] == -1 } {
                    WarnWinText "Bad result $result"
                    return
                }
            }
        }
        #-np- WarnWinText [lindex [GiD_Result get -max {Displacements_(m) Static 1}] 3]        
    }


    if { [GetCurrentPrePostMode] != "PRE" }  {
        DoPreprocess Yes
    }

    return "$area $maxdisplacement $maxstress"
}

proc Optimization::Constraint_0 { Displacement } {
#     set MaxDisplacement 0.01
    set MaxDisplacement 0.005
    return [expr {$Displacement-$MaxDisplacement}]
}

proc Optimization::Constraint_1 { Stress } {
#     set MaxStress 3.0e6
    set MaxStress 1.5e6
    return [expr {$Stress-$MaxStress}]
}

proc Optimization::Constraint_2 { x0 } {
    set Minx0 0.2
    return [expr {$Minx0-$x0}]
}

proc Optimization::GradientConstraint_2 { } {
    return  {-1 0}
}

proc Optimization::FunctionPenalized { x } {
    variable Weigth
    variable ActiveConstraint
    #Optimization::PutsLogFile "***begin FunctionPenalized $x"
    set coeff [Optimization::GetAnalysisCoefficients $x]
    #Optimization::PutsLogFile "***continue FunctionPenalized $x"
    foreach {Area Displacement Stress} $coeff break

    set Constraint(0) [Optimization::Constraint_0 $Displacement]
    set Constraint(1) [Optimization::Constraint_1 $Stress]
    set Constraint(2) [Optimization::Constraint_2 [lindex $x 0]]
    set f $Area
    for {set i 0} {$i < 3} {incr i} {
        set ActiveConstraint($i) [expr {$Constraint($i)>=0}]
        if { $ActiveConstraint($i) } {
            set f [expr {$f+$Weigth($i)*$Constraint($i)}]
        }
    }
    Optimization::PutsLogFile "***FunctionPenalized $f ($Area [expr {$Weigth(0)*$Constraint(0)}] \
        [expr {$Weigth(1)*$Constraint(1)}] [expr {$Weigth(2)*$Constraint(2)}])"
    return $f
}

proc Optimization::GradientPenalized { x } {
    variable Weigth
    variable ActiveConstraint
    #Optimization::PutsLogFile "***begin GradientPenalized $x"
    set coeff [Optimization::GetResponseSurfaceCoefficients $x]
    #Optimization::PutsLogFile "***continue GradientPenalized $x"
    set gradArea [lrange [lindex $coeff 0] 0 1]
    set gradConstraint(0) [lrange [lindex $coeff 1] 0 1]
    set gradConstraint(1) [lrange [lindex $coeff 2] 0 1]
    set gradConstraint(2) [Optimization::GradientConstraint_2]
    set grad $gradArea
    for {set i 0} {$i < 3} {incr i} {
        if { $ActiveConstraint($i) } {
            set grad [::math::linearalgebra::axpy $Weigth($i) $gradConstraint($i) $grad]
        }
    }
#     Optimization::PutsLogFile "***GradientPenalized $grad"
#     Optimization::PutsLogFile "***GradientPenalized $gradArea \
#         [::math::linearalgebra::scale $Weigth(0) $gradConstraint(0)] \
#         [::math::linearalgebra::scale $Weigth(1) $gradConstraint(1)] \
#         [::math::linearalgebra::scale $Weigth(2) $gradConstraint(2)]"
    return $grad
}


proc Optimization::AfterRunCalculationSetState { basename dir problemtypedir where error errorfilename } {
    after idle [list set ::Optimization::_StateSingleCalculation "finished"]
}

proc Optimization::GetAnalysisCoefficients { x } {
    variable CurrentDesign 
    variable DesignVariables
    variable _StateSingleCalculation

    set  _StateSingleCalculation "calculating"
    GiD_RegisterEvent GiD_Event_AfterRunCalculation Optimization::AfterRunCalculationSetState

    foreach varname [array names DesignVariables] value $x {                
        set CurrentDesign($varname) $value
    }      
    Optimization::UpdateModel 
    set tmpname [Optimization::SaveTmpModel]
    Optimization::AddToCalculationQueue $tmpname
    if { $_StateSingleCalculation == "calculating" } {
        vwait ::Optimization::_StateSingleCalculation
    }

    GiD_UnRegisterEvent GiD_Event_AfterRunCalculation Optimization::AfterRunCalculationSetState
    
    set name [join $x _]
    set ExperimentResult($name) [Optimization::GetSomeResults $tmpname]
    if { [file extension $tmpname] == ".gid" } { 
        #miro extension para estar mas seguro de no borrar nada indebido
        file delete -force $tmpname
    }
#     set function [lindex $ExperimentResult($name) 0]
#     return $function
    return $ExperimentResult($name)
}








#tests to use GiD as external process, and talk with comm
proc Optimization::StartComm { } {
    variable OptimizerPort
    package require commR
    set OptimizerPort [comm::register Optimizer 1]    
}

proc Optimization::StartGiD { } {
    set pid [exec {c:\gid_project\gid.exe} &]
    return $pid    
}

proc Optimization::GetPortFromPID { pid } {    
    variable OptimizerPort
    for { set i 12350 } { $i < 12360 } { incr i } {
        if { $i == $OptimizerPort } { continue }
        catch { set remotepid [::comm::comm send $i pid] }
        if { $remotepid == $pid } {
            return $i
        }
    }
    return 0
}



if { [info exists ::GIDDEFAULT] } {
    #running inside GiD
    set parent .gid
    Optimization::OpenGUI $parent
    #Optimization::PopulateTestRosenbrocks
} else {
    #define some procs to allow work outside GiD
    source "WithoutGiD.tcl"
    source "dev_kit.tcl"
    source "dialog-ram.tcl"
    set parent ""
    Optimization::OpenGUI $parent
    #Optimization::PopulateTestRosenbrocks
}


#example to find u,v of the surface number $::id_surface 
#with  x=$::px y=$::py
#$::pz could be used a posteriory to check the distance along z to the surface
proc FunctionUVSurfacePoint { x } {          
    lassign [GiD_Info parametric surface $::id_surface coord {*}$x] sx sy sz
    return [expr {($sx-$::px)*($sx-$::px)+($sy-$::py)*($sy-$::py)}]
}

proc GradientUVSurfacePoint { x } {
    lassign [GiD_Info parametric surface $::id_surface coord {*}$x] sx sy sz
    lassign [GiD_Info parametric surface $::id_surface deriv_u {*}$x] sxu syu szu
    lassign [GiD_Info parametric surface $::id_surface deriv_v {*}$x] sxv syv szv
    set gx [expr {2.0*(($sx-$::px)*$sxu+($sy-$::py)*$syu)}]
    set gy [expr {2.0*(($sx-$::px)*$sxv+($sy-$::py)*$syv)}]
    return [list $gx $gy]
}

proc PopulateUVSurfacePoint { } {       
    set ::Optimization::WritePlotFile 0
    set ::Optimization::WriteLogFile 0
    set ::Optimization::Function FunctionUVSurfacePoint
    set ::Optimization::Gradient GradientUVSurfacePoint
    set ::Optimization::Method CG
    set ::Optimization::CG_Tolerance 1e-5
    set ::Optimization::CG_MaxIterations 100
    
    
    set ::id_surface 1
    set ::px 0.4799
    set ::py 0.50584
    set ::pz 0.0
    #set as u,v starting value the ones of the closest point
    #but can also set arbitrary values e.g. 0.5 0.5
    lassign [GiD_Info parametric surface $::id_surface uv_fromcoord $::px $::py $::pz] u v

    if { [info exists DesignVariables] } {
        array unset DesignVariables
    }
    set  ::Optimization::DesignVariables(x1) [list 0.0 1.0 $u]
    set  ::Optimization::DesignVariables(x2) [list 0.0 1.0 $v]
           
    ::Optimization::FillGUIFromVariables        
}

proc CalculateUVSurfacePoint { id x y z } {
    set ::id_surface $id
    set ::px $x
    set ::py $y
    set ::pz $z
    lassign [GiD_Info parametric surface $::id_surface uv_fromcoord $::px $::py $::pz] u0 v0
    return [::Optimization::MinimizeCG FunctionUVSurfacePoint GradientUVSurfacePoint [list $u0 $v0] {0.0 0.0} {1.0 1.0} 1.e-8 100 "FLETCHER-REEVES"]    
}

proc ShowPointUVSurfacePoint { id x y z } {
    set uv [CalculateUVSurfacePoint $id $x $y $z]
    set pt [GiD_Info parametric surface $id coord {*}$uv]
    GiD_Process Mescape Utilities SignalEntities Points NoJoin {*}$pt
}
