proc MQDrawGraphCurrent { w } {
    upvar #0 [winfo name $w] data

    if { [GiD_Info Mesh] != $data(infomesh) } {
        MeshQuality $w
    } else {
        MQDrawGraph $w $data(frame) $data(etype)
    }
}


proc MQDrawGraphCanvas { c iniang endang numdivisions listb distributiontype labelx labely } {
    #$c delete curve
    #$c delete axestext
    #$c delete lineselect
    $c delete all
    
    #update
    #set xmax [$c cget -width]
    set xmax [winfo width $c]
    #    if { $xmax == 1 } { set xmax [winfo reqwidth $c] }
    set ymax [winfo height $c]
    
    set list ""
    set total 0
    for {set i 0 } { $i < $numdivisions } { incr i } {
        if { [ string match $distributiontype "Accumulated"]} {
            incr total [lindex $listb $i]
            lappend list $total
        } else {
            lappend list [lindex $listb $i]
        }
    }

    set ynummax 0
    for {set i 0 } { $i < $numdivisions } { incr i } {
        set yval [lindex $list $i]
        if { $yval > $ynummax } {
            set ynummax $yval
        }
        if { $i == 0 || $yval < $ynummin } {
            set ynummin $yval
        }
    }
    if { $ynummax == $ynummin } { incr ynummax }
    set textwidth [font measure NormalFont $ynummax]
    set xm [expr $textwidth+15]
    set textheight [font metrics NormalFont -linespace]
    set ym [expr int($textheight+25)]    
    $c create line $xm $ym $xm [expr $ymax-$ym] [expr $xmax-$xm] \
            [expr $ymax-$ym]  -tags axestext
    set inumtics 8
    for {set i 0 } { $i < $inumtics } { incr i } {
        set angle [format "%3.3g" \
                [expr $iniang+$i/double($inumtics-1)*($endang-$iniang)]]
        set xval [expr $xm+$i/double($inumtics-1)*($xmax-2*$xm)]
        $c create line $xval [expr $ymax-$ym+2] $xval [expr $ymax-$ym] \
                -tags axestext
        $c create text $xval [expr $ymax-$ym+2] -anchor n -justify center \
                -text $angle -font NormalFont -tags axestext

        set numelms [expr int($ynummin+$i/double($inumtics-1)*($ynummax-$ynummin))]
        set yval [expr $ymax-$ym-$i/double($inumtics-1)*($ymax-2*$ym)]
        $c create line [expr $xm-2] $yval $xm $yval -tags axestext
        $c create text [expr $xm-3] $yval -anchor e -justify right \
                -text $numelms -font NormalFont -tags axestext
    }

    set xfact [expr ($xmax-2.0*$xm)/double($numdivisions)]
    set yfact [expr ($ymax-2.0*$ym)/double($ynummax-$ynummin)]

    set lastyval [expr $ymax-$ym]
    for {set i 0 } { $i < $numdivisions } { incr i } {
        set yval [expr $ymax-$ym-([lindex $list $i]-$ynummin)*$yfact]
        #$c create line [expr $i*$xfact+$xm] $lastyval \
        #        [expr $i*$xfact+$xm] $yval\
        #        [expr ($i+1)*$xfact+$xm] $yval -tags curve -fill blue
        
        #$c create line [expr $i*$xfact+$xm] $lastyval \
        #        [expr ($i+1)*$xfact+$xm] $yval -tags curve -fill blue
        $c create rectangle [expr $i*$xfact+$xm] $lastyval \
                [expr ($i+1)*$xfact+$xm] $yval -tags curve -fill #f0f0f0 -outline steelblue
        # $c create rectangle [expr $i*$xfact+$xm] $lastyval \
        #         [expr ($i+1)*$xfact+$xm] $yval -tags curve -outline steelblue
        #set lastyval $yval
    }

    $c create text 5 5 -anchor nw -justify left \
            -text $labely -font BoldFont -tags axestext
    $c create text [expr $xmax-5] [expr $ymax-5] \
            -anchor se -justify left \
            -text $labelx -font BoldFont -tags axestext
    return [list $xm $ym]
}
    
proc MQDrawGraph { w frame etype } {
    if { $etype == "Point" } return
    global MeshQualityPriv
    upvar #0 [winfo name $w] data
    set data(frame) $frame
    if { ![winfo exists $frame.f1.e1] } return
    if { ![info exists data(etype)] } {
        set data(etype) unset_etype
    }
    if { $data(etype) != $etype } {
        set data(etype) $etype
        MQChangeQualityCriteria $w
    }

    set values [lindex [AvailableMeshCriteria $etype] 1]
    if { [lsearch $values $MeshQualityPriv(qualitytype)] == -1 } {
        set MeshQualityPriv(qualitytype) [lindex $values 0]
    }
    

    set iniang [$frame.f1.e1 get]
    set endang [$frame.f1.e2 get]
    set numdivisions [$frame.f2.e1 get]
        

    if { [string match Minimum*angle $MeshQualityPriv(qualitytype)] } {
        set listb [GiD_Info MeshQuality MinAngle $etype $iniang $endang \
                $numdivisions]
    } elseif { [string match Maximum*angle $MeshQualityPriv(qualitytype)] } {
        set listb [GiD_Info MeshQuality MaxAngle $etype $iniang $endang \
                $numdivisions]
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Element vol"] } {
        set listb [GiD_Info MeshQuality ElemSize $etype $iniang $endang \
                $numdivisions]
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Minimum edge"] } {
        set listb [GiD_Info MeshQuality ElemMinEdge $etype $iniang $endang \
                $numdivisions]
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Maximum edge"] } {
        set listb [GiD_Info MeshQuality ElemMaxEdge $etype $iniang $endang \
                $numdivisions]
    } elseif {[string equal $MeshQualityPriv(qualitytype) "Shape quality"] } {
        set listb [GiD_Info MeshQuality ElemShapeQuality $etype $iniang $endang \
                $numdivisions]
    } elseif {[string equal $MeshQualityPriv(qualitytype) "Minimum Jacobian"] } {
        set listb [GiD_Info MeshQuality ElemMinJacobian $etype $iniang $endang \
                $numdivisions]
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Space filling"] } {
        set listb [GiD_Info MeshQuality SpaceFilling $etype $iniang $endang \
                $numdivisions]
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Num neighbors"] } {
        set listb [GiD_Info MeshQuality NumNeighbors $etype $iniang $endang \
                $numdivisions]
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Radius"] } {
        set listb [GiD_Info MeshQuality ElemRadius $etype $iniang $endang \
                $numdivisions]
    }
    
    set iniangout [lindex $listb $numdivisions]
    if { [expr abs($iniangout-$iniang)>1e-30] } {
      set iniang [format "%3.3g" $iniangout]
      $frame.f1.e1 delete 0 end
      $frame.f1.e1 insert end $iniang
    }

    set endangout [lindex $listb [expr $numdivisions+1]]
    if { [expr abs($endangout-$endang)>1e-30] } {
      set endang [format "%3.3g" $endangout]
      $frame.f1.e2 delete 0 end
      $frame.f1.e2 insert end $endang
    }
    
    set labely [_ "NumElems"]
    
    set margins [MQDrawGraphCanvas $frame.c $iniang $endang $numdivisions \
            $listb $MeshQualityPriv(distributiontype) $data(TextX) $labely]
    lassign $margins MeshQualityPriv(xmargin) MeshQualityPriv(ymargin)
    
}

proc MQChangeQualityCriteria { w } {
    global MeshQualityPriv
    upvar #0 [winfo name $w] data
    set frame $data(frame)
    if { ![winfo exists $frame.f1.e1] } return
    if { [string equal $MeshQualityPriv(qualitytype) "Element vol"] } {
      set data(TextX) [_ "Volume"]
      set data(Units) ""
      $frame.f1.l1 configure -text [_ "%s: From" $data(TextX)]
      $frame.f1.e1 delete 0 end
      $frame.f1.e1 insert end 0.0
      $frame.f1.l2 configure -text [_ "%s To" $data(Units)]
      $frame.f1.e2 delete 0 end
      $frame.f1.e2 insert end 0.0
      $frame.f1.l3 configure -text $data(Units)
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Minimum edge"] } {
      set data(TextX) [_ "Min Size"]
      set data(Units) ""
      $frame.f1.l1 configure -text [_ "%s: From" $data(TextX)]
      $frame.f1.e1 delete 0 end
      $frame.f1.e1 insert end 0.0
      $frame.f1.l2 configure -text [_ "%s To" $data(Units)]
      $frame.f1.e2 delete 0 end
      $frame.f1.e2 insert end 0.0
      $frame.f1.l3 configure -text $data(Units)
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Maximum edge"] } {
      set data(TextX) [_ "Max Size"]
      set data(Units) ""
      $frame.f1.l1 configure -text [_ "%s: From" $data(TextX)]
      $frame.f1.e1 delete 0 end
      $frame.f1.e1 insert end 0.0
      $frame.f1.l2 configure -text [_ "%s To" $data(Units)]
      $frame.f1.e2 delete 0 end
      $frame.f1.e2 insert end 0.0
      $frame.f1.l3 configure -text $data(Units)
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Shape quality"] } {
        set data(TextX) [_ "Quality"]
        set data(Units) ""
        $frame.f1.l1 configure -text [_ "%s: From" $data(TextX)]
        $frame.f1.e1 delete 0 end
        $frame.f1.e1 insert end 0.0
        $frame.f1.l2 configure -text [_ "%s To" $data(Units)]
        $frame.f1.e2 delete 0 end
        $frame.f1.e2 insert end 0.0
        $frame.f1.l3 configure -text $data(Units)
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Minimum Jacobian"] } {
        set data(TextX) [_ "Min Jacobian"]
        set data(Units) ""
        $frame.f1.l1 configure -text [_ "%s: From" $data(TextX)]
        $frame.f1.e1 delete 0 end
        $frame.f1.e1 insert end 0.0
        $frame.f1.l2 configure -text [_ "%s To" $data(Units)]
        $frame.f1.e2 delete 0 end
        $frame.f1.e2 insert end 0.0
        $frame.f1.l3 configure -text $data(Units)
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Space filling"] } {
        set data(TextX) [_ "Space filling"]
        set data(Units) ""
        $frame.f1.l1 configure -text [_ "%s: From" $data(TextX)]
        $frame.f1.e1 delete 0 end
        $frame.f1.e1 insert end 0.0
        $frame.f1.l2 configure -text [_ "%s To" $data(Units)]
        $frame.f1.e2 delete 0 end
        $frame.f1.e2 insert end 0.0
        $frame.f1.l3 configure -text $data(Units)
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Num neighbors"] } {
        set data(TextX) [_ "Num neighbors"]
        set data(Units) ""
        $frame.f1.l1 configure -text [_ "%s: From" $data(TextX)]
        $frame.f1.e1 delete 0 end
        $frame.f1.e1 insert end 0.0
        $frame.f1.l2 configure -text [_ "%s To" $data(Units)]
        $frame.f1.e2 delete 0 end
        $frame.f1.e2 insert end 0.0
        $frame.f1.l3 configure -text $data(Units)
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Radius"] } {
        set data(TextX) [_ "Radius"]
        set data(Units) ""
        $frame.f1.l1 configure -text [_ "%s: From" $data(TextX)]
        $frame.f1.e1 delete 0 end
        $frame.f1.e1 insert end 0.0
        $frame.f1.l2 configure -text [_ "%s To" $data(Units)]
        $frame.f1.e2 delete 0 end
        $frame.f1.e2 insert end 0.0
        $frame.f1.l3 configure -text $data(Units)
    } else {
        set data(TextX) [_ "Angle"]
        set data(Units) [_ "Degrees"]
        $frame.f1.l1 configure -text [_ "%s: From" $data(TextX)]
        $frame.f1.e1 delete 0 end
        $frame.f1.e1 insert end 0.0
        $frame.f1.l2 configure -text [_ "%s To" $data(Units)]
        $frame.f1.e2 delete 0 end
        # $frame.f1.e2 insert end 180.0
        # set from == to for automatic min and max limits
        $frame.f1.e2 insert end 0.0
        $frame.f1.l3 configure -text $data(Units)
    }
    MQDrawGraph $w $data(frame) $data(etype)
}

proc FillMenuWithLayers { m } {
    global GidPriv
    set ::GidPriv(LayersAlsoLower) [GiD_Set LayersAlsoLower]
    set ::GidPriv(LayersAlsoHigher) [GiD_Set LayersAlsoHigher]
    $m delete 0 end
    $m add checkbutton -label [_ "Also lower entities"] -variable ::GidPriv(LayersAlsoLower) -command {GiD_Set LayersAlsoLower $::GidPriv(LayersAlsoLower)}
    $m add checkbutton -label [_ "Also higher entities"] -variable ::GidPriv(LayersAlsoHigher) -command {GiD_Set LayersAlsoHigher $::GidPriv(LayersAlsoHigher)}
    $m add command -label [_ "New layer"] -command [list SendSelectionToNewLayerAutomaticName]    
    set names [GiD_Layers list]
    if { [llength $names] } {
        $m add separator
        foreach name $names {
            $m add command -label $name -command [list SendSelectionToOldLayer $name]
        }
    }
}

proc MQCreate { w fr etype } {
    global MeshQualityPriv
    upvar #0 [winfo name $w] data

    global GidPriv
    set GidPriv(selection) ""

    canvas $fr.c -background white -relief ridge -borderwidth 2
    grid $fr.c -sticky nsew -row 1 -column 1
    GidHelp "$fr.c"  [_ "Double click to see the elements filtered with a x-value."]

    set MeshQualityPriv(qualitytype) "Minimum angle"
    set data(TextX) [_ "Angle"]

    ttk::frame $fr.f1
    ttk::label $fr.f1.l1 -text [_ "Angle: From"]
    ttk::entry $fr.f1.e1 -width 5   
    $fr.f1.e1 insert end 0.0
    ttk::label $fr.f1.l2 -text [_ "Degrees. To"]
    ttk::entry $fr.f1.e2 -width 5 
    #$fr.f1.e2 insert end 180.0
    $fr.f1.e2 insert end 0.0
    ttk::label $fr.f1.l3 -text [_ "Degrees"]
    grid $fr.f1.l1 -row 1 -column 1
    grid $fr.f1.e1 -sticky ew -row 1 -column 2
    grid $fr.f1.l2  -row 1 -column 3
    grid $fr.f1.e2 -sticky ew -row 1 -column 4
    grid $fr.f1.l3 -row 1 -column 5
    grid columnconf $fr.f1 2 -weight 1
    grid columnconf $fr.f1 4 -weight 1

    ttk::frame $fr.f2
    ttk::label $fr.f2.l1 -text [_ "Num divisions"]:
    ttk::entry $fr.f2.e1 -width 5  
    $fr.f2.e1 insert end 100

    ttk::label $fr.f2.l2 -text [_ "Distribution"]:    
    if { ![info exists ::MeshQualityPriv(distributiontype) ] } {
        set ::MeshQualityPriv(distributiontype) Accumulated
    }
    [TTKCB_CreateOrConfigure $fr.f2.m] $fr.f2.m -textvariable ::MeshQualityPriv(distributiontype) \
            -labels [list [_ "Accumulated"] [_  "Normal"]] \
            -values {Accumulated Normal}            
            
    set MeshQualityPriv(distributiontype) "Accumulated"
    grid $fr.f2.l1 -row 1 -column 1
    grid $fr.f2.e1 -sticky ew -row 1 -column 2
    grid $fr.f2.l2 -sticky e -row 1 -column 3
    grid $fr.f2.m -sticky w -row 1 -column 4
    grid columnconf $fr.f2 2 -weight 1
    grid columnconf $fr.f2 4 -weight 1

    ttk::frame $fr.type
    ttk::label $fr.type.l -anchor w -text [_ "Quality criteria"]:

    foreach {labels values} [AvailableMeshCriteria $etype] break
    if { ![info exists ::MeshQualityPriv(qualitytype) ] } {
        set ::MeshQualityPriv(qualitytype) [lindex $values 0]
    }
    [ TTKCB_CreateOrConfigure $fr.type.m] $fr.type.m -textvariable ::MeshQualityPriv(qualitytype) \
            -labels $labels \
            -values $values
    set MeshQualityPriv(qualitytype) [lindex $values 0]

    ttk::menubutton $fr.sendlaytouse -text [_ "Send to"] -underline 0 \
            -menu $fr.sendlaytouse.menu
    menu $fr.sendlaytouse.menu -borderwidth 1 -activeborderwidth 1

    FillMenuWithLayers $fr.sendlaytouse.menu

    grid $fr.type.l -row 1 -column 1
    grid $fr.type.m -row 1 -column 2

    grid $fr.f1 -sticky ew -row 2 -column 1
    grid $fr.f2 -sticky ew -row 3 -column 1
    grid $fr.type -sticky w  -row 4 -column 1
    grid $fr.sendlaytouse -sticky e  -row 4 -column 1 ;#same row and col as $fr.type
    grid columnconf $fr 1 -weight 1
    grid rowconf $fr 1 -weight 1

    bind $fr.f1.e1 <Return> [list MQDrawGraphCurrent $w]
    bind $fr.f1.e2 <Return> [list MQDrawGraphCurrent $w]
    bind $fr.f2.e1 <Return> [list MQDrawGraphCurrent $w]
    bind $fr.c <Double-Button-1> [list SelectElements $w $fr %x %y]
    bind $fr.c <Configure> [list MQDrawGraph $w $fr $etype]
    bind $fr.sendlaytouse <Button-1> [list FillMenuWithLayers $fr.sendlaytouse.menu]
}

proc SendSelectionToGetSelection { } {
    global GidPriv
    set selection ""
    #escape to update GidPriv(Selection) to the final value
    GiD_Process escape    
    if { $GidPriv(selection) == "" } {
        WarnWin [_ "Double-click first on the graph to select filtered elements"]
    }
    #remove extra quotes
    set selection [string trim $GidPriv(selection) \"]    
    return $selection
}

proc SendSelectionToNewLayerAutomaticName { } {
    global GidPriv
    set selection [SendSelectionToGetSelection]
    if { [llength $selection] } {
        set name [Layers::GetAutomaticLayerName ""]
        GiD_Process 'Layers New $name escape
        set command [list 'Layers Entities $name]
        if { [GiD_Set LayersAlsoLower] } {
            lappend command LowerEntities
        }
        if { [GiD_Set LayersAlsoHigher] } {
            lappend command HigherEntities
        }
        lappend command {*}$selection escape
        GiD_Process {*}$command
    }
}

proc SendSelectionToOldLayer { name } {
    global GidPriv
    set selection [SendSelectionToGetSelection]
    if { [llength $selection] } {
        set command [list 'Layers Entities $name]
        if { [GiD_Set LayersAlsoLower] } {
            lappend command LowerEntities
        }
        if { [GiD_Set LayersAlsoHigher] } {
            lappend command HigherEntities
        }
        lappend command {*}$selection escape
        GiD_Process {*}$command
    }      
}

#return two lists, labels and keynames
proc AvailableMeshCriteria { etype } {
    set res ""
    switch $etype {
	Tetrahedron -       
        Tetrahedra -
	Hexahedron -
        Hexahedra {
            set res [list [list [_ "Minimum dihedral angle"] [_ "Maximum dihedral angle"] \
                    [_ "Element vol"] [_ "Minimum edge"] [_ "Maximum edge"] [_ "Shape quality"] \
                    [_ "Minimum Jacobian"]] \
                { "Minimum dihedral angle"  "Maximum dihedral angle"  "Element vol" \
                "Minimum edge" "Maximum edge" "Shape quality" "Minimum Jacobian" }]
        }
        Sphere {
            set res [list [list [_ "Radius"] [_ "Element vol"] [_ "Space filling"] [_ "Num neighbors"]] \
                    { "Radius" "Element vol" "Space filling" "Num neighbors"}]
        }
        Circle {
            set res [list [list [_ "Radius"] [_ "Element vol"] [_ "Space filling"] [_ "Num neighbors"]] \
                    { "Radius" "Element vol" "Space filling" "Num neighbors"}]
        }
	Line -
        Linear {
            set res [list [list [_ "Element vol"]] { "Element vol"}]
        }
        Prism {
            set res [list [list [_ "Minimum dihedral angle"] [_ "Maximum dihedral angle"] \
                    [_ "Element vol"] [_ "Minimum edge"] [_ "Maximum edge"] [_ "Minimum Jacobian"]] \
                    { "Minimum dihedral angle"  "Maximum dihedral angle" "Element vol" \
                    "Minimum edge" "Maximum edge" "Minimum Jacobian" }]
        }
        Pyramid {
            set res [list [list [_ "Minimum dihedral angle"] [_ "Maximum dihedral angle"] \
                    [_ "Element vol"] [_ "Minimum edge"] [_ "Maximum edge"] [_ "Minimum Jacobian"]] \
                    { "Minimum dihedral angle"  "Maximum dihedral angle" "Element vol" \
                    "Minimum edge" "Maximum edge" "Minimum Jacobian" }]
        }
        Triangle -
        Quadrilateral {
            set res [list [list [_ "Minimum angle"] [_ "Maximum angle"] [_ "Element vol"] \
                    [_ "Minimum edge"] [_ "Maximum edge"] [_ "Shape quality"] [_ "Minimum Jacobian"]] \
                    { "Minimum angle" "Maximum angle" "Element vol" "Minimum edge" \
                    "Maximum edge" "Shape quality" "Minimum Jacobian" }]
        }
        default {
            set res [list [list ] { }]
        }
    }
    return $res
}


proc MeshQuality { { w .gid.meshq } } {
    global MeshQualityPriv
    set mesh [GiD_Info Mesh]
    if { [lindex $mesh 0] == 0 } {
        catch { destroy $w }
        WarnWin [_ "There are no meshes to evaluate their quality"]
        return
    }
    set meshtypes [lrange $mesh 1 end]   
    if { $meshtypes == "" } {
        catch { destroy $w }
        WarnWin [_ "There are no meshes of triangles, quadrilaterals, tetrahedrals, hexahedrals or prisms"]
        return
    }


    InitWindow2 $w -title [_ "Mesh quality"] \
        -geometryvariable PreMeshQualityWindowGeom \
        -initcommand MeshQuality -onlygeometry -ontop
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0

    wm withdraw $w

    set def_back [$w cget -background]
    ttk::frame $w.buts  -style BottomFrame.TFrame
    ttk::button $w.buts.cn -text [_ "Close"] -style BottomFrame.TButton -command [list destroy $w]

    grid $w.buts.cn -row 1 -column 1 -padx 5 -pady 6

    ttk::frame $w.f
    grid $w.buts -row 1 -column 0 -sticky ews -columnspan 7
    grid anchor $w.buts center
    grid $w.f -row 0 -column 0 -sticky nsew
    grid rowconfigure $w.f 0 -weight 1
    grid columnconfigure $w.f 0 -weight 1
    grid rowconfigure $w 0 -weight 1
    grid columnconfigure $w 0 -weight 1

    NoteBook $w.f.nb -internalborderwidth 1 -activebackground [CCColorActivo $def_back]
    foreach name $meshtypes {
        MQCreate $w [$w.f.nb insert end $name -text [_ $name] -raisecmd "ChangeCurrentPage $w $name"] $name
    }
    $w.f.nb raise [lindex $meshtypes 0]
    pack $w.f.nb -fill both -expand yes


    update idletasks
    wm deicon $w


    upvar #0 [winfo name $w] data
    set data(infomesh) $mesh   
    bind $w <Destroy> [list +DestroyMeshQuality %W $w]

    trace add variable MeshQualityPriv(qualitytype) write "MQChangeQualityCriteria $w ;#"
    trace add variable MeshQualityPriv(distributiontype) write "MQDrawGraphCurrent $w ;#"

    #to set the current first criteria labels
    foreach {labels values} [AvailableMeshCriteria [lindex $meshtypes 0]] break
    set MeshQualityPriv(qualitytype) [lindex $values 0]
}

proc DestroyMeshQuality { W w } {
    global MeshQualityPriv
    if { $W != $w } return
    trace remove variable MeshQualityPriv(qualitytype) write "MQChangeQualityCriteria $w ;#"
    trace remove variable MeshQualityPriv(distributiontype) write "MQDrawGraphCurrent $w ;#"
}

proc ChangeCurrentPage { w etype } {
    upvar #0 [winfo name $w] data
    set data(frame) [$w.f.nb getframe $etype]
    set data(etype) $etype
}

proc SelectElements { w frame x y } {
    upvar #0 [winfo name $w] data
    global MeshQualityPriv


    set c $frame.c
    $c delete lineselect
    set iniang [$frame.f1.e1 get]
    set endang [$frame.f1.e2 get]
    #set xmax [$c cget -width]
    #set ymax [$c cget -height]
    set xmax [winfo width $c]
    #if { $xmax == 1 } {
#        set xmax [winfo reqwidth $c]
#    }
    set ymax [winfo height $c]
#    if { $ymax == 1 } {
#        set ymax [winfo reqheight $c]
#    }

    set xm $MeshQualityPriv(xmargin)
    set ym $MeshQualityPriv(ymargin)
    set value [expr ($x-$xm)/($xmax-2.0*$xm)*($endang-$iniang)+$iniang]
    if { $value < $iniang } {
        set value $iniang
        set x [expr ($value-$iniang)/($endang-$iniang)*($xmax-2.0*$xm)+$xm]
    } elseif { $value > $endang } {
        set value $endang
        set x [expr ($value-$iniang)/($endang-$iniang)*($xmax-2.0*$xm)+$xm]
    }    
    $c create line $x $ym $x [expr $ymax-$ym] -tags lineselect -fill red

    MeshView 1
    GiD_Process 'SelectEntities Elements
    if { [string match Minimum*angle $MeshQualityPriv(qualitytype)] } {
        GiD_Process [list filter:badminangle=$value and elementtype=$data(etype)]
    } elseif { [string match Maximum*angle $MeshQualityPriv(qualitytype)] } {
        GiD_Process [list filter:badmaxangle=$value and elementtype=$data(etype)]
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Element vol"] } {
        GiD_Process [list filter:minvolume=$value and elementtype=$data(etype)]
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Minimum edge"] } {
        GiD_Process [list filter:minlength=$value and elementtype=$data(etype)]
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Maximum edge"] } {
        GiD_Process [list filter:maxlength=$value and elementtype=$data(etype)]
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Shape quality"] } {
        GiD_Process [list filter:minquality=$value and elementtype=$data(etype)]
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Minimum Jacobian"] } {
        GiD_Process [list filter:minjacob=$value and elementtype=$data(etype)]
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Space filling"] } {
        GiD_Process [list filter:spacefilling=$value and elementtype=$data(etype)]
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Num neighbors"] } {
        GiD_Process [list filter:numneighbors=$value and elementtype=$data(etype)]
    } elseif { [string equal $MeshQualityPriv(qualitytype) "Radius"] } {
        GiD_Process [list filter:elemradius=$value and elementtype=$data(etype)]
    }
    GiD_Process 1:end
}

#will return an averaged value of the distribution graph
#prop can be "SpaceFilling" or "NumNeighbors" , "ElemSize", "ElemRadius"
#only elements in on layers are computed
proc GetSphereElementsProperties { prop {ndiv 100} } {
    set type $prop
    set info [GiD_Info MeshQuality $type Sphere 0 0 $ndiv]
    set min [lindex $info end-1]
    set max [lindex $info end]
    set dx [expr {($max-$min)/double($ndiv)}]
    set numerator 0
    set denominator 0
    for {set i 0} {$i <$ndiv} {incr i} {
        set f [lindex $info $i]
        set x [expr {$min+($i+0.5)*$dx}]
        set numerator [expr {$numerator+$f*$x}]
        set denominator [expr {$denominator+$f}]
    }
    set res [expr {double($numerator)/$denominator}]
    return $res
}
