
namespace eval Groups {
    variable allow_select_lower_entities 0 ;#better not show to allow select lowerentities for groups
    variable also_lower_entities 0 ;#by default not select also lowerentities, it is dangerous to select unwanted entities
    variable SelectColumn ;#local temporary value
    variable _fullnames ;#internal array to map full group name by tree_id
    variable _tree_isopen ;#to save/restore the state open/collapse of the tree
    variable _T
}

#proc Groups::Test { args } {
#    W "Groups::Test $args"
#}

# to open and show the tab of LAYER and GROUPS windows
proc Groups::OpenTabIndex { index } {
    if { $index==0 || $index==1 } {
        set LayersOrGroupsCurrentTab $index
    } else {
        error "Invalid index $index"
    }
    set ::GidPriv(LayersOrGroupsCurrentTab) $index
    #GidUtils::OpenWindow GROUPS
    GidUtils::OpenWindow LAYER
}

proc Groups::OpenOutside {  { w .gid.central.wgroup } } {
    if { [winfo exists $w] } {
        destroy $w
    }
    InitWindow2 $w -title [_ "Groups"] -geometryvariable PreGroupsWindowGeom \
        -initcommand Groups::OpenOutside -ontop
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0
    Groups::Win $w
    #add lower buttons
    ttk::frame $w.buts -style BottomFrame.TFrame
    ttk::button $w.buts.close -text [_ "Close"] -command [list destroy $w] -style BottomFrame.TButton
    grid $w.buts.close -sticky ew -padx 5 -pady 6
    grid $w.buts -sticky ews -pady {2 0}
    grid anchor $w.buts center
}

#-button1handler of fulltktree works bad, not allow multiple selection and other bugs...
proc Groups::Button1_trick { T x y } {
    set sel [$T selection get]
    set identify [$T identify $x $y]
    if { $sel == "" } {
        #I don't know why when clicking the element rect of the color the selection is cleared!!
        if { [lindex $identify 0] == "item" } {
            set id [lindex $identify 1]
            $T selection add $id
            set sel [$T selection get]
            set sel $id
        }
    }
    return [Groups::Button1 $T $sel $identify $x $y]
}

proc Groups::CreateListWin { w } {
    package require fulltktree

    #trick: set the color column as 'text' but this is falsem after set its style to 's_color_text'
    #the name and style must have the word 'text' to avoid an error selecting the range to be ordered!!
    set columns [list \
        [list 10 [_ "Name"] left text 1 1] \
        [list 5 [_ "C#C#Color"] center text 1 0] \
        [list 5 [_ "I/O#C#On/Off"] center image 1 0] \
        [list 5 [_ "Tr#C#transparent"] center image 1 0] \
    ]
#set sort_columns {dictionary dictionary integer {command Groups::ImageCmp}}
set sort_columns {dictionary integer integer integer}

set T [fulltktree $w.t \
    -returnhandler Groups::CreateNewAux \
    -contextualhandler Groups::ContextualMenu \
    -editbeginhandler Groups::RenameStart \
    -editaccepthandler Groups::RenameEnd \
    -deletehandler Groups::Delete \
    -draghandler Groups::DragAndDrop \
    -columns $columns -expand 1 -compass_background 0 \
    -selectmode extended -showheader 1 \
    -showroot 0 -showbuttons 1 -showrootbutton 0 \
    -width 240 -height 240 -borderwidth 0 \
    -sort_type_cols $sort_columns \
    -item_image "" -folder_image "" -arrow_image_expand 0 \
    -showlines 1 -indent 15]
#-button1handler Groups::Button1
#-selecthandler2 {Groups::Test press2}

set t1 [$T givetreectrl]
gid_themes::ConfigureHeaders $t1
gid_themes::ApplyHeadersConfig $t1

#-button1handler of fulltktree works bad, not allow multiple selection and other bugs...
#and also works bad <ButtonRelease-1> because the selection is lost in the special 's_color_text' column
#bind $T <ButtonRelease-1> "Groups::Button1_trick %W %x %y"
bind $T <ButtonPress-1> "Groups::Button1_trick %W %x %y"

$T element create e_col_rect rect -fill "" -outline black -width 17 -height 17 -outlinewidth 1
#trick: set the color column as 'text' but this is falsem after set its style to 's_color_text'
#the name and style must have the word 'text' to avoid an error selecting the range to be ordered!!
set S [$T style create s_color_text]
$T style elements $S [list e_hidden e_rect e_col_rect e_text_sel e_selmarker_up e_selmarker_down]

#$T style layout $S e_hidden -padx 0 -squeeze x -expand nswe
$T style layout $S e_hidden -visible 0
$T style layout $S e_col_rect -expand nsew
#$T style layout $S e_text_sel -squeeze x -sticky w
$T style layout $S e_rect -union [list e_text_sel] -iexpand nswe

$T style layout $S e_selmarker_up -detach 1
$T style layout $S e_selmarker_up -height 2 -sticky nwe -iexpand x
$T style layout $S e_selmarker_down -detach 1 -expand n
$T style layout $S e_selmarker_down -height 2 -sticky swe -iexpand x

#modify also the layout of fulltktree 'image' style to hide the element 'e_hidden'
#because I am using this 'e_hidden' field as a trick to store a text to sort the column
$T style layout image e_hidden -visible 0

set sensitive [TreeCtrl::SetSensitive $t1]
lappend sensitive [list 1 $S e_text_sel e_col_rect]
TreeCtrl::SetSensitive $t1 $sensitive
#what about TreeCtrl::SetDragImage and TreeCtrl::SetEditable ???

set column_items {name color visibility transparency}
TreeSelectColumnsVisibility $T Groups $column_items

$T notify bind $T <Expand-after> {Groups::TreeExpandAfter %I}
$T notify bind $T <Collapse-after> {Groups::TreeCollapseAfter %I}
return $T
}

proc Groups::TreeExpandAfter { tree_id  } {
    variable _tree_isopen
    set fullname [Groups::GetGroupNameFromTreeId $tree_id]
    set _tree_isopen($fullname) 1
}

proc Groups::TreeCollapseAfter { tree_id } {
    variable _tree_isopen    
    set fullname [Groups::GetGroupNameFromTreeId $tree_id]
    unset -nocomplain _tree_isopen($fullname)
}

proc Groups::ResetTreeStatus { } {
    variable _tree_isopen
    array unset _tree_isopen
}

proc GroupsAndLayersResetTreeStatus { } {
    Groups::ResetTreeStatus
    Layers::ResetTreeStatus
}

proc GroupsAndLayersSaveTreeStatus { filename } {
    set fail 0
    file delete $filename    
    set fp [open $filename w]
    if { $fp == "" } {
        set fail 1
    } else {
        fconfigure $fp -encoding utf-8
        puts $fp {<?xml version="1.0" encoding="utf-8"?>}
        puts $fp "<gid>"
        puts $fp "  <pre>"
        puts $fp "    <groups_open>"
        foreach fullname [lsort -dictionary [array names Groups::_tree_isopen]] {
            puts $fp "      <group fullname='$fullname' open='1'/>"
        }
        puts $fp "    </groups_open>"
        puts $fp "    <layers_open>"
        foreach fullname [lsort -dictionary [array names Layers::_tree_isopen]] {
            puts $fp "      <layer fullname='$fullname' open='1'/>"
        }
        puts $fp "    </layers_open>"
        puts $fp "  </pre>"
        puts $fp "</gid>"
        #close $fp
        GidUtils::CatchClose $fp $filename
    }
    return $fail
}

proc GroupsAndLayersReadTreeStatus { filename } {
    GroupsAndLayersResetTreeStatus
    package require tdom
    if { [file exists $filename] } {
        set xml [GidUtils::ReadFile $filename utf-8]
        if { $xml != "" } {
            if { [catch { set document [dom parse $xml] } err] } {
                error [_ "Error reading file '%s': %s" $filename $err]
            }
            set root [$document documentElement]
            if { [$root nodeName] == "gid"} {
                foreach group_node [$root selectNodes /gid/pre/groups_open/group] {
                    set fullname [$group_node getAttribute fullname]
                    #assume that the attribute open==1 without read it
                    set Groups::_tree_isopen($fullname) 1
                }
                foreach layer_node [$root selectNodes /gid/pre/layers_open/layer] {
                    set fullname [$layer_node getAttribute fullname]
                    #assume that the attribute open==1 without read it
                    set Layers::_tree_isopen($fullname) 1
                }
            }
            $document delete
        }
    }
    return 0
}

proc Groups::Win { w } {
    variable _T

    if { ![winfo exists $w.middle] } {
        ttk::frame $w.middle
        set T [Groups::CreateListWin $w.middle]
        set _T $T

        grid $T -sticky nsew -padx 1 -pady 1
        grid configure $T -pady "2 0"

        ttk::frame $w.top -style ForcedFrame
        ttk::button $w.top.new -image [gid_themes::GetImage group_new.png small_icons] -command [list Groups::CreateNew $T] -style IconButton
        GidHelp $w.top.new [_ "Create a new group at root level."]
        ttk::button $w.top.newfolder -image [gid_themes::GetImage group_newfolder.png small_icons] -command [list Groups::CreateNewFolder $T] -style IconButton
        GidHelp $w.top.newfolder [_ "Create a new child group with the selected parent."]
        ttk::button $w.top.delete -image [gid_themes::GetImage delete.png small_icons] -command [list Groups::Delete $T] -style IconButton
        GidHelp $w.top.delete [_ "Delete a group."]
        ttk::menubutton $w.top.assign -image [gid_themes::GetImage group_assign.png small_icons] -menu $w.top.assign.m -takefocus 1 -style IconButton
        GidHelp $w.top.assign [_ "Assign the selected entities to a group."]
        Groups::CreateMenuAssign $w.top.assign.m $T
        ttk::menubutton $w.top.unassign -image [gid_themes::GetImage group_unassign.png small_icons] -menu $w.top.unassign.m -takefocus 1 -style IconButton
        GidHelp $w.top.unassign [_ "Unassign the selected entities from a group."]
        Groups::CreateMenuUnassign $w.top.unassign.m $T
        ttk::button $w.top.draw -image [gid_themes::GetImage group_draw.png small_icons] -command [list Groups::Draw $T] -style IconButton
        GidHelp $w.top.draw [_ "Draw groups by color"]
        ttk::button $w.top.list_entities -image [gid_themes::GetImage list.png small_icons] -command [list Groups::ListEntitiesSelection $T] -style IconButton
        GidHelp $w.top.list_entities [_ "List group entities"]
        #ttk::menubutton $w.top.allowed -menu $w.top.allowed.m -takefocus 1 -image [gid_themes::GetImage group_allowed.png small_icons] -style IconButton
        #GidHelp $w.top.allowed [_ "Type of entities allowed to belong to the group"]
        #Groups::CreateMenuAllowed $w.top.allowed.m $T

        if { [llength [GiD_Info conditions over_group]] } {
            ttk::button $w.top.list_conditions -image [gid_themes::GetImage condition.png small_icons] -command [list Groups::ListConditions $T] -style IconButton
            GidHelp $w.top.list_conditions [_ "List group conditions"]
            grid $w.top.new $w.top.newfolder $w.top.delete $w.top.assign $w.top.unassign $w.top.draw $w.top.list_entities $w.top.list_conditions -sticky w
            #$w.top.allowed
        } else {
            grid $w.top.new $w.top.newfolder $w.top.delete $w.top.assign $w.top.unassign $w.top.draw $w.top.list_entities -sticky w
            #$w.top.allowed
        }

        grid $w.top -sticky ew
        grid $w.middle -sticky nsew

        grid columnconfigure $w.middle 0 -weight 1
        grid rowconfigure $w.middle 0 -weight 1

        grid columnconfigure $w 0 -weight 1
        grid rowconfigure $w 1 -weight 1
    } else {
        Groups::ClearTree $T
    }
    Groups::FillInfoTree $T "" 0
    focus $T
}

proc Groups::SelectColumnsWindow { } {
    variable _T
    variable SelectColumn ;#local temporary value
    set category Groups
    set column_items {name color visibility transparency}
    set column_labels [list [_ "Name"] [_ "Color"] [_ "Visibility"] [_ "Transparency"]]
    set tree $_T
    set w .gid.selectcolumns
    InitWindow2 $w -title [_ "Select %s columns" [_ $category]] \
        -geometryvariable PreSelect${category}ColumnsWindowGeom \
        -initcommand ${category}::SelectColumnsWindow -ontop
    set c [CreateScrolledCanvas $w.body]
    grid $w.body -sticky nsew
    set f [ttk::frame $c.f]
    foreach item $column_items text $column_labels {
        if { ![info exists ::GidPriv(ColumnVisible,$category,$item)] } {
            set ::GidPriv(ColumnVisible,$category,$item) 1
        }
        set SelectColumn($item) $::GidPriv(ColumnVisible,$category,$item)
        ttk::checkbutton $f.$item -text $text -variable ${category}::SelectColumn($item)
        grid $f.$item -sticky w
    }
    AddToScrolledCanvas $w.body $f

    grid rowconfigure $w 0 -weight 1
    grid columnconfigure $w 0 -weight 1
    CreateLowerButtonsApplyClose $w [list ${category}::SelectColumnsWindowOnApply $_T $category $column_items]
}

proc Groups::SelectColumnsWindowOnApply { tree category column_items } {
    variable SelectColumn
    set changes 0
    foreach item $column_items {
        if { $::GidPriv(ColumnVisible,$category,$item) != $SelectColumn($item) } {
            set ::GidPriv(ColumnVisible,$category,$item) $SelectColumn($item)
            set changes 1
        }
    }
    if { $changes } {
        TreeSelectColumnsVisibility $tree $category $column_items
    }
}

proc Groups::AllowType { group type_to_allow } {
    set group_allowed_types [GiD_Groups get allowed_types $group]
    if { [lsearch $group_allowed_types $type_to_allow] == -1 } {
        lappend group_allowed_types $type_to_allow
        #GiD_Groups edit allowed_types $group $group_allowed_types
        GiD_Process 'Groups Edit Allowed_types $group {*}$group_allowed_types escape
    }
}

proc Groups::DisallowType { group type_to_disallow } {
    set group_allowed_types [GiD_Groups get allowed_types $group]
    set change 0
    set new_group_allowed_types [list]
    foreach type $group_allowed_types {
        if { $type != $type_to_disallow } {
            lappend new_group_allowed_types $type
        } else {
            set change 1
        }
    }
    if { $change } {
        #GiD_Groups edit allowed_types $group $new_group_allowed_types
        GiD_Process 'Groups Edit Allowed_types $group {*}$new_group_allowed_types escape
    }
}

proc Groups::GetDisallowedTypes { group } {
    set group_disallowed_types [list]
    set group_allowed_types [GiD_Groups get allowed_types $group]
    foreach type {points lines surfaces volumes nodes elements faces} {
        if { [lsearch $group_allowed_types $type] == -1 } {
            lappend group_disallowed_types $type
        }

    }
    return $group_disallowed_types
}

proc Groups::AllowElementType { group element_type_to_allow } {
    set group_allowed_element_types [GiD_Groups get allowed_element_types $group]
    if { [lsearch $group_allowed_element_types $element_type_to_allow] == -1 } {
        lappend group_allowed_element_types $element_type_to_allow
        #GiD_Groups edit allowed_element_types $group $group_allowed_element_types
        GiD_Process 'Groups Edit Allowed_element_types $group {*}$group_allowed_element_types escape
    }
}

proc Groups::DisallowElementType { group element_type_to_disallow } {
    set group_allowed_element_types [GiD_Groups get allowed_element_types $group]
    set change 0
    set new_group_group_allowed_element_types [list]
    foreach element_type $group_allowed_element_types {
        if { $element_type != $element_type_to_disallow } {
            lappend new_group_group_allowed_element_types $element_type
        } else {
            set change 1
        }
    }
    if { $change } {
        #GiD_Groups edit allowed_element_types $group $new_group_group_allowed_element_types
        GiD_Process 'Groups Edit Allowed_element_types $group {*}$new_group_group_allowed_element_types escape
    }
}

proc Groups::GetDisallowedElementTypes { group } {
    set group_disallowed_element_types [list]
    set group_allowed_element_types [GiD_Groups get allowed_element_types $group]
    foreach element_type [GidUtils::GetElementTypes all] {
        if { [lsearch $group_allowed_element_types $element_type] == -1 } {
            lappend group_disallowed_element_types $element_type
        }

    }
    return $group_disallowed_element_types
}

proc Groups::RemoveNotAllowedTypesGeometry { allowed_types commands labels images} {
    set num_geometry_allowed_types 0
    foreach type {points lines surfaces volumes} {
        if { [lsearch $allowed_types $type] != -1 } {
            incr num_geometry_allowed_types
        }
    }
    set commands_final [list]
    set labels_final [list]
    set images_final [list]
    foreach command $commands label $labels image $images {
        if { ($command == "---" && [llength $commands_final]) || ($command == "All_Geom_Types" && $num_geometry_allowed_types > 1) || [lsearch $allowed_types [string tolower $command]] != -1 } {
            lappend commands_final $command
            lappend labels_final $label
            lappend images_final $image
        }
    }
    if { [lindex $commands_final end] == "---" } {
        set commands_final [lrange $commands_final 0 end-1]
        set labels_final [lrange $labels_final 0 end-1]
        set images_final [lrange $images_final 0 end-1]
    }
    return [list $commands_final $labels_final $images_final]
}

proc Groups::RemoveNotAllowedTypesMesh { allowed_types commands labels images} {
    set num_mesh_allowed_types 0
    foreach type {nodes elements faces} {
        if { [lsearch $allowed_types $type] != -1 } {
            incr num_mesh_allowed_types
        }
    }
    set commands_final [list]
    set labels_final [list]
    set images_final [list]
    foreach command $commands label $labels image $images {
        if { ($command == "---" && [llength $commands_final]) || ($command == "All_Mesh_Types" && $num_mesh_allowed_types > 1) || [lsearch $allowed_types [string tolower $command]] != -1 } {
            lappend commands_final $command
            lappend labels_final $label
            lappend images_final $image
        }
    }
    if { [lindex $commands_final end] == "---" } {
        set commands_final [lrange $commands_final 0 end-1]
        set labels_final [lrange $labels_final 0 end-1]
        set images_final [lrange $images_final 0 end-1]
    }
    return [list $commands_final $labels_final $images_final]
}

proc Groups::CreateMenuAssign { m T } {
    variable allow_select_lower_entities
    variable also_lower_entities
    menu $m -postcommand [list Groups::CreateMenuAssign2 $m $T]
    if { $allow_select_lower_entities } {
        $m add checkbutton -label [_ "Also lower entities"] -variable ::Groups::also_lower_entities
    }
    $m configure -activeborderwidth 1
}

proc Groups::CreateMenuAssign2 { m T } {
    set group_names [Groups::GetSelectedGroupnames $T]
    if { [llength $group_names] ==1 } {
        set group [lindex $group_names 0]
        set geometry_commands [GetGeometryCommands 1]
        lset geometry_commands end All_Geom_Types ;#replace All by All_Geom_Types
        set geometry_labels [GetGeometryLabels 1]
        set geometry_images [GetGeometryImageFiles 1]

        set mesh_commands [GetMeshCommands 1]
        set mesh_labels [GetMeshLabels 1]
        lset mesh_commands end All_Mesh_Types ;#replace All by All_Mesh_Types
        set mesh_images [GetMeshImageFiles 1]
        set mesh_commands [linsert $mesh_commands 2 Faces]
        set mesh_labels [linsert $mesh_labels 2 [_ "Faces"]]
        set mesh_images [linsert $mesh_images 2 "face.png"]

        #filter based on group specific properties
        set group_allowed_types [GiD_Groups get allowed_types $group]
        lassign [Groups::RemoveNotAllowedTypesGeometry $group_allowed_types $geometry_commands $geometry_labels $geometry_images] geometry_commands geometry_labels geometry_images
        lassign [Groups::RemoveNotAllowedTypesMesh $group_allowed_types $mesh_commands $mesh_labels $mesh_images] mesh_commands mesh_labels mesh_images

        AddEntitiesToMenuXXX $m [list Groups::Assign $T] \
            [list $geometry_commands $geometry_labels $geometry_images] \
            [list $mesh_commands $mesh_labels $mesh_images]
    }
}

proc Groups::Assign { T over } {
    variable also_lower_entities
    set group_names [Groups::GetSelectedGroupnames $T]
    if { [llength $group_names] !=1 } {
        WarnWin [_ "before applying 'Entities' it is necessary to select 1 group"]
        return
    }
    set fullname [lindex $group_names 0]
    if { $also_lower_entities } {
        GiD_Process escape escape escape Utilities EntitiesGroups Assign $fullname LowerEntities $over
    } else {
        GiD_Process escape escape escape Utilities EntitiesGroups Assign $fullname $over
    }
    #set parent [winfo parent [winfo parent $T]]
    #FinishButton $parent $parent.top [_ "Press 'Finish' to end selection"] "" disableall [GiD_Set SmallWinSelecting]
    # $w is same parent as $::wChangeLayButtonFrame
    set w [ winfo parent $::wChangeLayFrame]
    FinishButton $w $::wChangeLayButtonFrame [_ "Press 'Finish' to end selection"] "" disableall [GiD_Set SmallWinSelecting]
}

proc Groups::CreateMenuUnassign { m T } {
    variable allow_select_lower_entities
    variable also_lower_entities
    menu $m -postcommand [list Groups::CreateMenuUnassign2 $m $T]
    if { $allow_select_lower_entities } {
        $m add checkbutton -label [_ "Also lower entities"] -variable ::Groups::also_lower_entities
    }
    $m configure -activeborderwidth 1
}

proc Groups::CreateMenuUnassign2 { m T } {
    set group_names [Groups::GetSelectedGroupnames $T]
    if { [llength $group_names] ==1 } {
        set group [lindex $group_names 0]

        set geometry_commands [GetGeometryCommands 1]
        lset geometry_commands end All_Geom_Types ;#replace All by All_Geom_Types
        set geometry_labels [GetGeometryLabels 1]
        set geometry_images [GetGeometryImageFiles 1]

        set mesh_commands [GetMeshCommands 1]
        lset mesh_commands end All_Mesh_Types ;#replace All by All_Mesh_Types
        set mesh_labels [GetMeshLabels 1]
        set mesh_images [GetMeshImageFiles 1]
        set mesh_commands [linsert $mesh_commands 2 Faces]
        set mesh_labels [linsert $mesh_labels 2 [_ "Faces"]]
        set mesh_images [linsert $mesh_images 2 "face.png"]

        #filter based on group specific properties
        set group_allowed_types [GiD_Groups get allowed_types $group]
        lassign [Groups::RemoveNotAllowedTypesGeometry $group_allowed_types $geometry_commands $geometry_labels $geometry_images] geometry_commands geometry_labels geometry_images
        lassign [Groups::RemoveNotAllowedTypesMesh $group_allowed_types $mesh_commands $mesh_labels $mesh_images] mesh_commands mesh_labels mesh_images
        AddEntitiesToMenuXXX $m [list Groups::Unassign $T] \
            [list $geometry_commands $geometry_labels $geometry_images] \
            [list $mesh_commands $mesh_labels $mesh_images]
    }
}

proc Groups::Unassign { T over } {
    variable also_lower_entities
    set group_names [Groups::GetSelectedGroupnames $T]
    if { [llength $group_names] !=1 } {
        WarnWin [_ "before applying 'Entities' it is necessary to select 1 group"]
        return
    }
    set fullname [lindex $group_names 0]
    if { $also_lower_entities } {
        GiD_Process escape escape escape Utilities EntitiesGroups Unassign $fullname LowerEntities $over
    } else {
        GiD_Process escape escape escape Utilities EntitiesGroups Unassign $fullname $over
    }
    #set parent [winfo parent [winfo parent $T]]
    #FinishButton $parent $parent.top [_ "Press 'Finish' to end selection"] "" disableall [GiD_Set SmallWinSelecting]
    set w [ winfo parent $::wChangeLayFrame]
    FinishButton $w $::wChangeLayButtonFrame [_ "Press 'Finish' to end selection"] "" disableall [GiD_Set SmallWinSelecting]
}


proc Groups::CreateMenuAllowed { m T } {
    menu $m -postcommand [list Groups::CreateMenuAllowed2 $m $T]
    $m configure -activeborderwidth 1
}

proc Groups::CreateMenuAllowed2 { m T } {
    set group_names [Groups::GetSelectedGroupnames $T]
    if { [llength $group_names] ==1 } {
        set group [lindex $group_names 0]
        set group_disallowed_types [Groups::GetDisallowedTypes $group]
        set view_mode [GiD_Info Project ViewMode]
        if { $view_mode == "GEOMETRYUSE" } {
            set entity_types {points lines surfaces volumes}
        } else {
            set entity_types {nodes elements faces}
            #groups over edges are really not implemented
        }
        if { [llength $group_disallowed_types] } {
            menu $m.allow_type
            $m add cascade -label [_ "Allow type"] -menu $m.allow_type -state normal -compound left
            foreach type $group_disallowed_types {
                if { [lsearch $entity_types $type] == -1 } {
                    #show in menus only current entity_types
                    continue
                }
                set label [_ $type]
                set image_name [string range $type 0 end-1].png
                $m.allow_type add command -label $label -command [list Groups::AllowType $group $type] -image [gid_themes::GetImage $image_name menu] -compound left
            }
        }
        set group_allowed_types [GiD_Groups get allowed_types $group]
        if { [llength $group_allowed_types] } {
            menu $m.disallow_type
            $m add cascade -label [_ "Disallow type"] -menu $m.disallow_type -state normal -compound left
            foreach type $group_allowed_types {
                if { [lsearch $entity_types $type] == -1 } {
                    #show in menus only current entity_types
                    continue
                }
                set label [_ $type]
                set image_name [string range $type 0 end-1].png
                $m.disallow_type add command -label $label -command [list Groups::DisallowType $group $type] -image [gid_themes::GetImage $image_name menu] -compound left
            }
        }

        if { $view_mode != "GEOMETRYUSE" } {
            set group_allowed_element_types [GiD_Groups get allowed_element_types $group]
            set group_disallowed_element_types [Groups::GetDisallowedElementTypes $group]
            $m add separator
            if { [llength $group_disallowed_element_types] } {
                menu $m.allow_element_type
                $m add cascade -label [_ "Allow element type"] -menu $m.allow_element_type -state normal -compound left
                foreach element_type $group_disallowed_element_types {
                    set label [_ $element_type]
                    $m.allow_element_type add command -label $label -command [list Groups::AllowElementType $group $element_type] -compound left
                }
            }
            if { [llength $group_allowed_element_types] } {
                menu $m.disallow_element_type
                $m add cascade -label [_ "Disallow element type"] -menu $m.disallow_element_type -state normal -compound left
                foreach element_type $group_allowed_element_types {
                    set label [_ $element_type]
                    $m.disallow_element_type add command -label $label -command [list Groups::DisallowElementType $group $element_type] -compound left
                }
            }
        }
    }
}

proc Groups::NameJoin { args } {
    set parts [list]
    foreach item $args {
        if { $item != "" } {
            lappend parts $item
        }
    }
    if { [llength $parts] > 1 } {
        set fullname [join $parts //]
    } else {
        #join to remove {} when item has spaces
        set fullname [join $parts]        
    }
    return $fullname
}

proc Groups::NameSplit { name } {
    return [GidUtils::Split $name //]
}

proc Groups::NameTail { name } {
    return [file tail $name]
}

proc Groups::FillInfo { } {
    variable _T
    #     variable _filling
    #     if { [info exists _filling]} {
    #         return
    #     }
    #     set _filling 1
    if { [GidUtils::IsTkDisabled] } {
        #e.g. batch mode without windows
        return
    }
    if { [info exists _T] && [winfo exists $_T] } {
        Groups::ClearTree $_T
        Groups::FillInfoTree $_T
    }

    #update also the external window
    set nb .gid.central.wgroup.middle.t
    if { [winfo exists $nb] } {
        Groups::ClearTree $nb
        Groups::FillInfoTree $nb
    }
    #     unset _filling

    DWSetCurrentGroupsToAssign
}

#state: normal, disabled or hidden, use * to list all without take into account its state
#condition_disallowed_types is a list of not allowed types: points lines surfaces volumes elements faces
#condition_disallowed_element_types is a list of not allowed element types: linear triangle ...
#warning: not add nodes to condition_disallowed_types list, because now when meshing the nodes are always
#         added to the group of the source entity!!
proc Groups::ListGroups { {state normal} {condition_disallowed_types ""} {condition_disallowed_element_types ""} } {
    set groups {}
    foreach group [GiD_Groups list] {
        if { $state == "*" || [GiD_Groups get state $group] == $state } {
            set valid 1
            foreach over $condition_disallowed_types {
                if { [GiD_EntitiesGroups get $group $over -count] } {
                    set valid 0
                    break
                }
            }
            if { $valid } {
                foreach element_type $condition_disallowed_element_types {
                    if { [GiD_EntitiesGroups get $group elements -count -element_type [list $element_type]] } {
                        set valid 0
                        break
                    }
                }
            }
            if { $valid } {
                lappend groups $group
            }
        }
    }
    return [lsort -unique -dictionary $groups]
}

#list tailnames of all childs of a group
proc Groups::ListChildGroups { group {state normal} } {
    set groups {}
    foreach child_group [GiD_Groups list $group] {
        set fullname [Groups::NameJoin $group $child_group]
        if { [GiD_Groups get state $fullname] == $state } {
            lappend groups $child_group
        }
    }
    return [lsort -unique -dictionary $groups]
}

#list fullnames of all descendants of a group
proc Groups::ListDescendantGroups { group {state normal} } {
    set groups {}
    foreach child_group [GiD_Groups list $group] {
        set fullname [Groups::NameJoin $group $child_group]
        if { [GiD_Groups get state $fullname] == $state } {
            lappend groups $fullname
        }
        lappend groups {*}[Groups::ListDescendantGroups $fullname $state]
    }
    return [lsort -unique -dictionary $groups]
}

proc Groups::ClearTree { T } {
    Groups::UnSetAllTreeIds
    $T item delete all
}

proc Groups::FillInfoTree { T {parent ""} {tree_id_parent 0}} {
    set groups [Groups::ListChildGroups $parent]
    foreach group $groups {
        set fullname [Groups::NameJoin $parent $group]
        set tree_id [Groups::Create $T $fullname $tree_id_parent]
        Groups::FillInfoTree $T $fullname $tree_id
    }
}

proc Groups::GetAutomaticName { {prefix ""} } {
    set names [GiD_Groups list] ;#take into account also hidden groups
    set fullname ""
    for {set i 0} {$i<10000} {incr i} {
        set fullname [Groups::NameJoin $prefix Group${i}]
        if { [lsearch -exact $names $fullname] == -1 } {
            break
        }
    }
    return $fullname
}

proc Groups::GetAutomaticCloneTailName { fullname_to_clone } {
    set names [GiD_Groups list] ;#take into account also hidden groups
    set parent [GiD_Groups get parent $fullname_to_clone]
    set part [Groups::NameTail $fullname_to_clone]
    set tailname ""
    for {set i 1} {$i<10000} {incr i} {
        set tailname ${part}-clone${i}
        set fullname [Groups::NameJoin $parent $tailname]
        if { [lsearch -exact $names $fullname] == -1 } {
            break
        }
    }
    return $tailname
}


proc Groups::CreateNewAux { T args } {
    #parent is a Fulltktree, T is TreeCtrl
    return [Groups::CreateNew [winfo parent $T]]
}

proc Groups::CreateNew { T } {
    set tree_id_parent 0
    set prefix ""
    set fullname [Groups::GetAutomaticName $prefix]
    #GiD_Groups create $fullname
    GiD_Process 'Groups Create $fullname escape
    set tree_id [Groups::Create $T $fullname $tree_id_parent]
    Groups::StartRename $T $tree_id
    return $fullname
}

proc Groups::CreateNewFolder { T } {
    set prefix ""
    if { [$T selection count] == 1 } {
        set tree_id_parent [$T selection get 0]
        set prefix [Groups::GetGroupNameFromTreeId $tree_id_parent]
    } else {
        WarnWin [_ "A parent group must be selected before"]
        return ""
    }
    set fullname [Groups::GetAutomaticName $prefix]
    #GiD_Groups create $fullname
    GiD_Process 'Groups Create $fullname escape
    set tree_id [Groups::Create $T $fullname $tree_id_parent]
    Groups::StartRename $T $tree_id
    return $fullname
}

proc Groups::Create { T fullname tree_id_parent } {
    variable _tree_isopen
    set tailname [Groups::NameTail $fullname];#GiD_Groups get name $fullname
    set rgba [GiD_Groups get color $fullname]
    set color [string range $rgba 0 end-2]
    set opaque [GiD_Groups get opaque $fullname]
    set visible [GiD_Groups get visible $fullname]
    if { $visible } {
        set img_visible [gid_themes::GetImage layer_on.png small_icons]
    } else {
        set img_visible [gid_themes::GetImage layer_off.png small_icons]
    }
    if { $opaque } {
        set img_opaque [gid_themes::GetImage transparent_off.png small_icons]
    } else {
        set img_opaque [gid_themes::GetImage transparent_on.png small_icons]
    }
    set tree_id [$T insert end [list $tailname "" $img_visible $img_opaque] $tree_id_parent]
    $T item style set $tree_id 1 s_color_text
    $T item element configure $tree_id 1 e_col_rect -fill $color + e_hidden -text 0x[string range $color 1 end]
    $T item element configure $tree_id 2 e_hidden -text $visible
    $T item element configure $tree_id 3 e_hidden -text $opaque
    Groups::SetTreeIdGroupName $tree_id $fullname
    if { [info exists _tree_isopen($fullname)] } {
        $T item expand $tree_id
    } else {
        $T item collapse $tree_id
    }
    return $tree_id
}

#return true if itself or some child has entities assigned
proc Groups::NumEntitiesRecursive { parent } {
    set count 0
    if { $parent != "" } {
        set count [GiD_Groups get num_entities $parent]
    }
    foreach child [GiD_Groups list $parent] {
        incr count [Groups::NumEntitiesRecursive [Groups::NameJoin $parent $child]]
    }
    return $count
}

#return true if itself or some child has conditions assigned
proc Groups::NumConditionsRecursive { parent } {
    set count 0
    if { $parent != "" } {
        set  count [GiD_Groups get num_conditions $parent]
    }
    foreach child [GiD_Groups list $parent] {
        incr count [Groups::NumConditionsRecursive [Groups::NameJoin $parent $child]]
    }
    return $count
}

#warning: don't remove argument sel, it is added automatically as a trick by the -deletehandler !!!
proc Groups::Delete { T {sel ""} } {
    variable _tree_isopen
    if { $sel == "" } {
        set sel [$T selection get]
    }
    set groups_to_delete [list]
    set delete_only_empty_groups 1
    set ask 1
    foreach tree_id $sel {
        #set fullname [$T item tag names $tree_id]
        set fullname [Groups::GetGroupNameFromTreeId $tree_id]
        if { ![GiD_Groups exists $fullname] } {
            #e.g. if its parent was also in the list, deleting them childs are deleted
            continue
        }
        if { $ask } {
            set num_entities [Groups::NumEntitiesRecursive $fullname]
            set num_conditions [Groups::NumConditionsRecursive $fullname]
            if { $num_entities || $num_conditions } {
                if { $num_entities && $num_conditions } {
                    set message [_ "Group '%s' has %s entities. Delete it? (entities won't be deleted but applied conditions will be deleted also)" $fullname $num_entities]
                } elseif { $num_entities } {
                    set message [_ "Group '%s' has %s entities. Delete it? (entities won't be deleted)" $fullname $num_entities]
                } else {
                    set message [_ "Group '%s' has %s conditions. Delete it? (applied conditions will be deleted also)" $fullname $num_conditions]
                }
                set answer [MessageBoxOptionsButtons [_ "Groups"] $message \
                    {0 1 2} [list [_ "Yes"] [_ "No"] [_ "Cancel"]] question [_ "Do it for all"]]
                set do_it_for_all 0
                lassign $answer ret do_it_for_all
                if { $do_it_for_all } {
                    set ask 0
                }
                if { $ret == 0 } {
                    if { $ask } {
                        #yes: do it asking each time
                    } else {
                        #yes to all: yes and not ask more
                        set delete_only_empty_groups 0
                    }
                } elseif { $ret == 1 } {
                    if { $ask } {
                        #no: not to do but continue with the rest
                        continue
                    } else {
                        #no to all: no and not ask more (delete only empty groups)
                        set delete_only_empty_groups 1
                        continue
                    }
                } elseif { $ret == 2 } {
                    #cancel
                    break
                } else {
                    #unexpected
                    break
                }
            }
        } else {
            if { $delete_only_empty_groups } {
                set num_entities [Groups::NumEntitiesRecursive $fullname]
                if { $num_entities} {
                    #delete only empty groups
                    continue
                }
            }
        }

        Groups::UnSetTreeId $tree_id
        $T item delete $tree_id
        lappend groups_to_delete $fullname
        unset -nocomplain _tree_isopen($fullname)
    }
    if { [llength $groups_to_delete] } {
        #foreach fullname $groups_to_delete {
        #    GiD_Groups delete $fullname
        #}
        GiD_Process 'Groups Delete {*}$groups_to_delete escape
    }
}

proc Groups::Draw { T } {
    set group_names [Groups::GetSelectedGroupnames $T]
    if { [llength $group_names] } {
        #GiD_Groups draw $fullnames
        GiD_Process 'Groups -Show_Legend 0 Draw {*}$group_names escape
        set parent [winfo parent [winfo parent $T]]
        ##FinishButton $parent $parent.top [_ "Press 'Finish' to end selection"] {GiD_Groups end_draw} disableall [GiD_Set SmallWinSelecting]
        #FinishButton $parent $parent.top [_ "Press 'Finish' to end selection"] "" disableall [GiD_Set SmallWinSelecting]
        # $w is same parent as $::wChangeLayButtonFrame
        set w [ winfo parent $::wChangeLayFrame]
        FinishButton $w $::wChangeLayButtonFrame [_ "Press 'Finish' to end selection"] "" disableall [GiD_Set SmallWinSelecting]
    }
}

#limit 43000 is a trick to avoid long lines >=43684 characters are not visible in current Tk text widget!!
#if limit==0 then consider as no limit
proc Groups::ListEntities { group_names {limit 43000} } {
    set text ""
    set view_mode [GiD_Info Project ViewMode]
    if { $view_mode == "GEOMETRYUSE"}  {
        set over all_geometry
        set types [list [_ "Points"] [_ "Lines"] [_ "Surfaces"] [_ "volumes"]]
    } elseif { $view_mode == "MESHUSE"}  {
        set over all_mesh
        set types [list [_ "Nodes"] [_ "Elements"] [_ "Faces"]]
    } else {
        error "Unexpected mode $view_mode"
    }
    foreach fullname $group_names {
        set data [GiD_EntitiesGroups get $fullname $over]
        set has_entities 0
        foreach type $types type_data $data {
            if { [llength $type_data] > 0 } {
                if { $type != [_ "Faces"] || [llength [lindex $type_data 0]] > 0} {
                    set has_entities 1
                    break
                }
            }
        }
        append text "$fullname:\n"
        if { $has_entities } {
            foreach type $types type_data $data {
                if { [llength $type_data] > 0 } {
                    if { $type != [_ "Faces"] || [llength [lindex $type_data 0]] > 0} {
                        if { $type == [_ "Faces"] } {
                            set txt ""
                            foreach element [lindex $type_data 0] face [lindex $type_data 1] {
                                lappend txt $element:$face
                            }
                            #set count [llength [lindex $type_data 0]]
                            set count [llength $txt]
                            if { $limit && [string length $txt] > $limit} {
                                #long lines >=43684 characters are not visible in current Tk text widget!!
                                set type_data [string range $txt 0 $limit]...
                            }
                            append text "  $count $type: $txt\n"
                        } else {
                            if { $type == [_ "Elements"] } {
                                #show also the number of types of the elements
                                unset -nocomplain num_element_type
                                foreach element_id $type_data {
                                    incr num_element_type([lindex [GiD_Mesh get element $element_id] 1])
                                }
                                set element_names ""
                                foreach element_type [lsort -dictionary [array names num_element_type]] {
                                    lappend element_names [list $num_element_type($element_type) [_ $element_type]]
                                }
                                set count [llength $type_data]
                                if { $limit && [string length $type_data] > $limit} {
                                    #long lines >=43684 characters are not visible in current Tk text widget!!
                                    set type_data [string range $type_data 0 $limit]...
                                }
                                append text "  $count $type ([join $element_names {, }]): $type_data\n"
                            } else {
                                set count [llength $type_data]
                                if { $limit && [string length $type_data] > $limit} {
                                    #long lines >=43684 characters are not visible in current Tk text widget!!
                                    set type_data [string range $type_data 0 $limit]...
                                }
                                append text "  $count $type: $type_data\n"
                            }
                        }
                    }
                }
            }
            append text "\n"
        } else {
            append text "\n"
        }
    }
    return $text
}

proc Groups::ListEntitiesSelection { T } {
    #without select nothing list all
    set group_names [Groups::GetSelectedGroupnames $T]
    set text [Groups::ListEntities $group_names]
    W $text [_ "Group's entities"]
}

#return a list with items containing the condition name and its  values
proc Groups::GetConditionValues { fullname } {
    set values [list]
    foreach condition_name [GiD_Info conditions over_group] {
        set condition_values [list]
        foreach data [GiD_Info conditions $condition_name geometry] {
            if { [lindex $data 1] == $fullname } {
                lappend condition_values [lrange $data 3 end]
            }
        }
        if { [llength $condition_values] } {
            lappend values [list $condition_name $condition_values]
        }
    }
    return $values
}

proc Groups::ListConditions { T } {
    set group_names [Groups::GetSelectedGroupnames $T]
    W "" [_ "Group's conditions"]
    foreach fullname [lsort -dictionary $group_names] {
        set group_data [Groups::GetConditionValues $fullname]
        if { [llength $group_data]} {
            W "Group: $fullname"
            foreach item $group_data {
                lassign $item condition_name condition_values
                WarnWinText "[llength $condition_values] $condition_name"
                foreach data $condition_values {
                    W $data
                }
                W ""
            }
        }
    }
}

proc Groups::RenameStart { T item col } {
    if { $col == 0 } {
        #return [list combo 1 $values ""]
        return [list entry]
    }
}

proc Groups::RenameEnd { T tree_id col newname } {
    if { $col == 0 } {
        set newname [string trim $newname]
        set fullname [Groups::GetGroupNameFromTreeId $tree_id]
        set fullparentname [GiD_Groups get parent $fullname]
        set newfullname [Groups::NameJoin $fullparentname $newname]
        if { $fullname != $newfullname } {
            if { [GiD_Groups is_forbidden_name $newfullname] } {
                WarnWin [_ "Group %s has forbidden syntax" $newfullname]
            } else {
                if { [GiD_Groups exists $newfullname] } {
                    WarnWin [_ "Group %s already exists" $newfullname]
                } else {
                    #GiD_Groups edit name $fullname $newfullname
                    GiD_Process 'Groups Edit Name $fullname $newfullname escape
                    $T item element configure $tree_id 0 e_text_sel -text $newname
                    Groups::SetTreeIdGroupName $tree_id $newfullname
                    #must update also childs if any
                    set prefix $fullname//
                    set len [string length $prefix]
                    foreach i [Groups::GetAllGroupNames] {
                        if { [string range [Groups::GetGroupNameFromTreeId $i]  0 $len-1] == $prefix } {
                            Groups::SetTreeIdGroupName $i [Groups::NameJoin $newfullname [string range [Groups::GetGroupNameFromTreeId $i] $len end]]
                        }
                    }
                }
            }
        }
    }
}

proc Groups::StartRename { T {tree_id ""} } {
    #to start rename
    if { $tree_id == "" } {
        set sel [$T selection get]
        if { [llength $sel] == 1 } {
            set tree_id [lindex $sel 0]
        }
    }
    if { $tree_id == "" } {
        return
    }

    $T selection clear all
    $T selection add $tree_id
    $T activate $tree_id
    $T see $tree_id
    focus [$T givetreectrl]
    update
    event generate [$T givetreectrl] <F2>
}

proc Groups::Clone { T } {
    set sel [$T selection get]
    foreach tree_id $sel {
        set fullname_to_clone [Groups::GetGroupNameFromTreeId $tree_id]
        set parent [GiD_Groups get parent $fullname_to_clone]
        set tailname [Groups::GetAutomaticCloneTailName $fullname_to_clone]
        set fullname [Groups::NameJoin $parent $tailname]
        #GiD_Groups clone $fullname_to_clone $tailname
        GiD_Process 'Groups Clone $fullname_to_clone $tailname escape
        if { $parent == "" } {
            set tree_id_parent 0
        } else {
            set tree_id_parent [Groups::GetTreeIdFromGroupName $T $parent]
        }
        set tree_id [Groups::Create $T $fullname $tree_id_parent]
    }
    GidUtils::UpdateWindow GROUPS
    return 0
}

proc Groups::GetGroupNameFromTreeId { tree_id } {
    variable _fullnames
    return $_fullnames($tree_id)
}

proc Groups::SetTreeIdGroupName { tree_id newfullname} {
    variable _fullnames
    set _fullnames($tree_id) $newfullname
}

proc Groups::UnSetTreeId { tree_id } {
    variable _fullnames
    unset _fullnames($tree_id)
}

proc Groups::UnSetAllTreeIds { } {
    variable _fullnames
    unset -nocomplain _fullnames
}

proc Groups::GetAllGroupNames { } {
    variable _fullnames
    return [array names _fullnames]
}

proc Groups::GetTreeIdFromGroupName { T group_name } {
    set tree_id -1
    foreach item [$T item children 0] {
        if { [Groups::GetGroupNameFromTreeId $item] == $group_name } {
            set tree_id $item
        }
    }
    return $tree_id
}

proc Groups::SelectGroup { group_name } {
    variable _T
    set T $_T
    if { ![winfo exists $T] } {
        return
    }
    set tree_id [Groups::GetTreeIdFromGroupName $T $group_name]
    if { $tree_id != -1 } {
        $T selection clear all
        $T selection add $tree_id
        $T activate $tree_id
        $T see $tree_id
        focus [$T givetreectrl]
    }
}

proc Groups::Button1 { T sel identify x y } {
    if { [llength $sel] == 1 } {
        #trick: when click on/off, etc of a row different of currently selected
        #user want to set the clicked row, not the previously selected one
        set item [lindex $identify 1]
        set sel $item
    }
    set column [lindex $identify 3]
    if { $column == 1 } {
        #color
        Groups::ChangeColor $T $sel
    } elseif { $column == 2 } {
        #on/off
        Groups::ChangeVisible $T $sel
    } elseif { $column == 3 } {
        #opaque
        Groups::ChangeOpaque $T $sel
    } else {
    }
}

proc Groups::ChangeColor { T {sel ""} {rgba ""} } {
    if { $sel == "" } {
        set sel [$T selection get]
    }
    set tree_id [lindex $sel 0]
    if { $tree_id == "" } {
        return
    }
    if { $rgba == "" } {
        set fullname [Groups::GetGroupNameFromTreeId $tree_id]
        set rgba [GiD_Groups get color $fullname]
        set rgba [GIDChooseColor .gid.selectcolor -title [_ "Select color"] -color $rgba]
        if { $rgba == "" } {
            return
        }
    }
    set color [string range $rgba 0 end-2]
    foreach tree_id $sel {
        set fullname [Groups::GetGroupNameFromTreeId $tree_id]
        #GiD_Groups edit color $fullname $rgba
        GiD_Process 'Groups Edit Color $fullname $rgba escape

        $T item element configure $tree_id 1 e_col_rect -fill $color + e_hidden -text 0x[string range $color 1 end]
    }
    GiD_Redraw
}

proc Groups::ChangeVisible { T {sel ""} {visible ""}} {
    if { $sel == "" } {
        set sel [$T selection get]
    }
    set tree_id [lindex $sel 0]
    if { $tree_id == "" } {
        return
    }

    if { $visible == "" } {
        set fullname [Groups::GetGroupNameFromTreeId $tree_id]
        set visible [GiD_Groups get visible $fullname]
        if { $visible } {
            set visible 0
        } else {
            set visible 1
        }
    }
    if { $visible } {
        set img_visible [gid_themes::GetImage layer_on.png small_icons]
    } else {
        set img_visible [gid_themes::GetImage layer_off.png small_icons]
    }
    foreach tree_id $sel {
        set fullname [Groups::GetGroupNameFromTreeId $tree_id]
        #GiD_Groups edit visible $fullname $visible
        GiD_Process 'Groups Edit Visible $fullname $visible escape
        $T item element configure $tree_id 2 e_image -image $img_visible + e_hidden -text $visible
    }
    #to update also on/off icon of descendants
    foreach tree_id $sel {
        foreach item_descendant [$T item descendants $tree_id] {
            $T item element configure $item_descendant 2 e_image -image $img_visible + e_hidden -text $visible
        }
    }
    GiD_Redraw
}

proc Groups::ChangeOpaque { T {sel ""} {opaque ""} } {
    if { $sel == "" } {
        set sel [$T selection get]
    }
    set tree_id [lindex $sel 0]
    if { $tree_id == "" } {
        return
    }
    if { $opaque == "" } {
        #swap current opaque value of first selected item
        set fullname [Groups::GetGroupNameFromTreeId $tree_id]
        if { [GiD_Groups get opaque $fullname] } {
            set opaque 0
        } else {
            set opaque 1
        }
    }
    if { $opaque } {
        set img_opaque [gid_themes::GetImage transparent_off.png small_icons]
    } else {
        set img_opaque [gid_themes::GetImage transparent_on.png small_icons]
    }
    foreach tree_id $sel {
        set fullname [Groups::GetGroupNameFromTreeId $tree_id]
        #GiD_Groups edit opaque $fullname $opaque
        GiD_Process 'Groups Edit Opaque $fullname $opaque escape
        $T item element configure $tree_id 3 e_image -image $img_opaque + e_hidden -text $opaque
    }
    GiD_Redraw
}

proc Groups::DragAndDrop { T tree_id_new_parent sel place } {
    set change_parent 0
    if { $place == "center" } {
        set change_parent 1
        set fullparentname [Groups::GetGroupNameFromTreeId $tree_id_new_parent]
    } elseif { $place == "prev" } {
        set change_parent 1
        set tree_id_new_parent 0
        set fullparentname ""
    } elseif { $place == "next" } {
        set change_parent 1
        set tree_id_new_parent 0
        set fullparentname ""
    }

    if { $change_parent } {
        foreach tree_id $sel {
            set tree_id_old_parent [$T item parent $tree_id]
            set fullname [Groups::GetGroupNameFromTreeId $tree_id]
            #GiD_Groups edit parent $fullname $fullparentname
            GiD_Process 'Groups Edit Parent $fullname $fullparentname escape
            set tailname [Groups::NameTail $fullname] ;#GiD_Groups get name $fullname
            set newfullname [Groups::NameJoin $fullparentname $tailname]
            Groups::SetTreeIdGroupName $tree_id $newfullname
            $T item lastchild $tree_id_new_parent $tree_id
            if { $tree_id_new_parent != 0 } {
                #$T convert_to_folder $tree_id_parent
                $T item style map $tree_id_new_parent 0 folder [list e_text_sel e_text_sel]
                $T item configure $tree_id_new_parent -button 1
            }

            if { ![$T item numchildren $tree_id_old_parent] } {
                #now old parent doesn't has any child
                $T item style map $tree_id_old_parent 0 text [list e_text_sel e_text_sel]
                $T item configure $tree_id_old_parent -button 0
            }
        }
    }
}

proc Groups::ContextualMenu { T tree_id x y } {
    set category Groups
    set w $T.menucontextual
    if { [winfo exists $w] } {
        destroy $w
    }
    menu $w

    set group_names [Groups::GetSelectedGroupnames $T]
    set n_selected [llength $group_names]

    $w add command -label [_ "New group#C#group"] -command [list Groups::CreateNew $T] \
        -image [gid_themes::GetImage group_new.png small_icons] -compound left
    $w add command -label [_ "New child group#C#group"] -command [list Groups::CreateNewFolder $T] \
        -image [gid_themes::GetImage group_newfolder.png small_icons] -compound left

    if { $n_selected == 1 } {
        $w add command -label [_ "Rename"] -command [list Groups::StartRename $T] -state normal \
            -image [gid_themes::GetImage rename.png small_icons] -compound left
    } else {
        $w add command -label [_ "Rename"] -state disabled \
            -image [gid_themes::GetImage rename.png small_icons] -compound left
    }
    if { $n_selected > 0 } {
        $w add command -label [_ "Delete"] -command [list Groups::Delete $T] -state normal \
            -image [gid_themes::GetImage delete.png small_icons] -compound left
        $w add command -label [_ "Clone"] -command [list Groups::Clone $T] -state normal \
            -image [gid_themes::GetImage clone.png small_icons] -compound left
    } else {
        $w add command -label [_ "Delete"] -state disabled \
            -image [gid_themes::GetImage delete.png small_icons] -compound left
        $w add command -label [_ "Clone"] -state disabled \
            -image [gid_themes::GetImage clone.png small_icons] -compound left
    }

    $w add separator

    if { $n_selected > 0 } {
        set state normal
    } else {
        set state disabled
    }
    set item color
    if { ![info exists ::GidPriv(ColumnVisible,$category,$item)] || $::GidPriv(ColumnVisible,$category,$item) } {
        $w add command -label [_ "Color"]... -command [list Groups::ChangeColor $T] -state $state \
            -image [gid_themes::GetImage color.png small_icons] -compound left
    }
    if { $n_selected == 1 } {
        set fullname [lindex $group_names 0]
        set item visibility
        if { ![info exists ::GidPriv(ColumnVisible,$category,$item)] || $::GidPriv(ColumnVisible,$category,$item) } {
            set visible [GiD_Groups get visible $fullname]
            if { $visible } {
                $w add command -label [_ "Off#C#verb"] -command [list Groups::ChangeVisible $T "" 0] -state $state \
                    -image [gid_themes::GetImage layer_off.png small_icons] -compound left
            } else {
                $w add command -label [_ "On#C#verb"] -command [list Groups::ChangeVisible $T "" 1] -state $state \
                    -image [gid_themes::GetImage layer_on.png small_icons] -compound left
            }
        }
        set item transparency
        if { ![info exists ::GidPriv(ColumnVisible,$category,$item)] || $::GidPriv(ColumnVisible,$category,$item) } {
            set opaque [GiD_Groups get opaque $fullname]
            if { $opaque } {
                $w add command -label [_ "Transparent#C#verb"] -command [list Groups::ChangeOpaque $T "" 0] -state $state \
                    -image [gid_themes::GetImage transparent_on.png small_icons] -compound left
            } else {
                $w add command -label [_ "Opaque#C#verb"] -command [list Groups::ChangeOpaque $T "" 1] -state $state \
                    -image [gid_themes::GetImage transparent_off.png small_icons] -compound left
            }
        }
    } else {
        set item visibility
        if { ![info exists ::GidPriv(ColumnVisible,$category,$item)] || $::GidPriv(ColumnVisible,$category,$item) } {
            $w add command -label [_ "On#C#verb"] -command [list Groups::ChangeVisible $T "" 1] -state $state \
                -image [gid_themes::GetImage layer_on.png small_icons] -compound left
            $w add command -label [_ "Off#C#verb"] -command [list Groups::ChangeVisible $T "" 0] -state $state \
                -image [gid_themes::GetImage layer_off.png small_icons] -compound left
        }
        set item transparency
        if { ![info exists ::GidPriv(ColumnVisible,$category,$item)] || $::GidPriv(ColumnVisible,$category,$item) } {
            $w add command -label [_ "Opaque#C#verb"] -command [list Groups::ChangeOpaque $T "" 1] -state $state \
                -image [gid_themes::GetImage transparent_off.png small_icons] -compound left
            $w add command -label [_ "Transparent#C#verb"] -command [list Groups::ChangeOpaque $T "" 0] -state $state \
                -image [gid_themes::GetImage transparent_on.png small_icons] -compound left
        }
    }
    $w add separator
    $w add command -label [_ "Select all"] -command [list $T selection add all]
    $w add separator
    if { $n_selected == 1 } {
        $w add cascade -label [_ "Assign"] -menu $w.assign -state normal \
            -image [gid_themes::GetImage group_assign.png small_icons] -compound left
    } else {
        $w add cascade -label [_ "Assign"] -menu $w.assign -state disabled \
            -image [gid_themes::GetImage group_assign.png small_icons] -compound left
    }
    Groups::CreateMenuAssign $w.assign $T

    if { $n_selected == 1 } {
        $w add cascade -label [_ "Unassign"] -menu $w.unassign -state normal \
            -image [gid_themes::GetImage group_unassign.png small_icons] -compound left
    } else {
        $w add cascade -label [_ "Unassign"] -menu $w.unassign -state disabled \
            -image [gid_themes::GetImage group_unassign.png small_icons] -compound left
    }
    Groups::CreateMenuUnassign $w.unassign $T

    $w add command -label [_ "Draw groups by color"] -command [list Groups::Draw $T] -state $state \
        -image [gid_themes::GetImage group_draw.png small_icons] -compound left
    $w add command -label [_ "List group entities"] -command [list Groups::ListEntitiesSelection $T] -state normal \
        -image [gid_themes::GetImage list.png small_icons] -compound left
    if { [llength [GiD_Info conditions over_group]] } {
        $w add command -label [_ "List group conditions"] -command [list Groups::ListConditions $T] -state $state \
            -image [gid_themes::GetImage condition.png small_icons] -compound left
    }

    $w add separator
    if { $n_selected == 1 } {
        $w add cascade -label [_ "Allowed"] -menu $w.allowed -state normal -compound left
    } else {
        $w add cascade -label [_ "Allowed"] -menu $w.assign -state disabled -compound left
    }
    # -image [gid_themes::GetImage group_allowed.png small_icons]
    Groups::CreateMenuAllowed $w.allowed $T

    $w add separator
    if { $n_selected > 0 } {
        set state normal
    } else {
        set state disabled
    }
    $w add command -label [_ "Groups to layers"] -command [list Groups::GroupsToLayersAsk $T] -state $state \
        -image [gid_themes::GetImage groups.png small_icons] -compound left

    $w add command -label [_ "Select columns"]... -command [list Groups::SelectColumnsWindow]

    #     $w add separator
    #     $w add command -label [_ "Filter"] -command [list Layers::SelectLayersByRegexp $T] \
    #         -image [gid_themes::GetImage filter.png small_icons] -compound left

    #set x [expr [winfo rootx $T]+$x+2]
    #set y [expr [winfo rooty $T]+$y]
    GiD_PopupMenu $w $x $y
}

#test of sorting with a tcl procedure
proc Groups::ImageCmp { T column tree_id1 tree_id2 } {
    set img_1 [$T item element cget $tree_id1 $column e_image -image]
    set img_2 [$T item element cget $tree_id2 $column e_image -image]
    return [string compare $img_1 $img_2]
}

proc Groups::LayersToGroups { layer_names } {
    foreach layer_name $layer_names {
        set group_name $layer_name
        if { ![GiD_Groups exists $group_name] } {
            GiD_Groups create $group_name
        }
        set group_color [GiD_Layers get color $layer_name]
        #GiD_Groups edit color $group_name $group_color
        GiD_Process 'Groups Edit Color $group_name $group_color escape
        if { ![GiD_Layers get visible $layer_name] } {
            #GiD_Groups edit visible $group_name 0
            GiD_Process 'Groups Edit Visible $group_name 0 escape
        }
        foreach over {points lines surfaces volumes nodes elements} {
            set selection [GiD_Info Layers -entities $over $layer_name]
            if { [llength $selection] } {
                GiD_EntitiesGroups assign $group_name $over $selection
            }
        }
    }
    GidUtils::UpdateWindow GROUPS
}

proc Groups::IfNothingSelectedSelectAll { T } {
    set sel [$T selection get]
    if { $sel == "" } {
        $T selection add all
        set sel [$T selection get]
    }
    return $sel
}

proc Groups::GetSelectedGroupnames { T } {
    if { ![winfo exists $T] } {
        return
    }
    set sel [Groups::IfNothingSelectedSelectAll $T]
    set group_names [list]
    foreach tree_id $sel {
        lappend group_names [Groups::GetGroupNameFromTreeId $tree_id]
    }
    #set group_names [lsort -unique -dictionary $group_names]
    return $group_names
}

proc Groups::GroupsToLayersAsk { T } {
    set group_names [Groups::GetSelectedGroupnames $T]
    if { [llength $group_names] } {
        set retval [MessageBoxOptionsButtons [_ "Groups to layers"] \
            [_ "Are you sure to create layers from the selected groups?"] \
            {0 1} [list [_ "Yes"] [_ "No#C#I don't want to do that"]] question ""]
        if { $retval == 0 } {
            Groups::GroupsToLayers $group_names
        }
    }
}

proc Groups::GroupsToLayers { group_names } {
    GidUtils::DisableGraphics
    set view_mode_start [GiD_Info Project ViewMode]
    set layer_names [GiD_Layers list]
    set layer_names_start $layer_names
    foreach group_name $group_names {
        set layer_name $group_name
        if { [lsearch $layer_names $layer_name] == -1 } {
            GiD_Process 'Layers New $layer_name escape
            lappend $layer_names $layer_name
        }
        set group_color [string range [GiD_Groups get color $group_name] 1 end]
        set group_visible [GiD_Groups get visible $group_name]
        scan $group_color %2x%2x%2x%2x r g b a
        set layer_color [format %03i%03i%03i $r $g $b]
        set layer_alpha [format %03i $a]
        GiD_Process 'Layers Color $layer_name $layer_color Transparent $layer_name $layer_alpha escape
        if { !$group_visible } {
            GiD_Process 'Layers Off $layer_name escape
        }
        foreach over {points lines surfaces volumes nodes elements} {
            set selection [GiD_EntitiesGroups get $group_name $over]
            if { [llength $selection] } {
                if { $over == "nodes" || $over == "elements" } {
                    MeshView 0
                } else {
                    GeometryView
                }
                GiD_Process 'Layers Entities $layer_name $over {*}$selection escape escape
            }
        }
    }
    set all_group_names [GiD_Groups list]
    foreach layer_name $layer_names_start {
        if { [lsearch $all_group_names $layer_name] == -1 } {
            #try to delete old layers (could be deleted if they become empty)
            GiD_Process 'Layers Delete $layer_name escape
        }
    }
    if { $view_mode_start != [GiD_Info Project ViewMode] } {
        if { $view_mode_start == "GEOMETRYUSE" } {
            GeometryView
        } else {
            MeshView 0
        }
    }
    GidUtils::EnableGraphics
    GidUtils::UpdateWindow LAYER
}

proc Groups::GetAssignedTypesDisallowed { group } {
    set disallowed_types [list]
    foreach type [Groups::GetDisallowedTypes $group] {
        if { [GiD_EntitiesGroups get $group $type -count] } {
            lappend disallowed_types $type
        }
    }
    return $disallowed_types
}

proc Groups::GetAssignedElementTypesDisallowed { group } {
    set disallowed_element_types [list]
    foreach element_type [Groups::GetDisallowedElementTypes $group] {
        if { [GiD_EntitiesGroups get $group elements -count -element_type [list $element_type]] } {
            lappend disallowed_element_types $element_type
        }
    }
    return $disallowed_element_types
}

proc Groups::RemoveAssignedTypesDisallowed { group } {
    set num_unassigned 0
    foreach type [Groups::GetAssignedTypesDisallowed $group] {
        incr num_unassigned [GiD_EntitiesGroups unassign $group $type]
    }
    return $num_unassigned
}

proc Groups::RemoveAssignedElementTypesDisallowed { group } {
    set element_types [Groups::GetAssignedElementTypesDisallowed $group]
    set num_unassigned [GiD_EntitiesGroups unassign $group -element_type $element_types elements]
    return $num_unassigned
}

proc Groups::RemoveAssignedTypesDisallowedAllGroups { } {
    set num_changed 0
    set message ""
    set num_unassigned 0
    foreach group [Groups::ListGroups * "" ""] {
        set num_unassigned_local [Groups::RemoveAssignedTypesDisallowed $group]
        incr num_unassigned_local [Groups::RemoveAssignedElementTypesDisallowed $group]
        if { $num_unassigned_local } {
            incr num_changed
            incr num_unassigned $num_unassigned_local
        }
    }
    if { $num_changed } {
        set message [_ "Unassigned %s disallowed entities from %s groups" $num_unassigned $num_changed]
    }
    return [list $num_changed $message]
}
