#GiD post read plugin

if { [GidUtils::VersionCmp 10.1.7d] < 0 } {  
    WarnWinText [_ "This plugin requires GiD %s or later" 10.1.7d]
}

namespace eval GiDPost {
    variable plugin_version 1.0 ;#plugin version  
    variable attribute_type string ;#how attributes are encoded  (integer for deprecated versions)
    variable gidpost_version ;#version of the gidpost file
}

proc GiDPost::GetFreeHDF5Id { } {  
    set prefix hd
    for {set i 1} {$i<1000} {incr i} {
        set id ${prefix}${i}
        if { [info commands $id] == "" } {
            return $id 
        }
    }
    return ""
}

proc GiDPost::IsGiDPost { filename } {  
    variable gidpost_version 
    variable attribute_type

    package require hdf5
    
    set h [GiDPost::GetFreeHDF5Id]
    if { [catch { hdf5 -readonly $h $filename } err] } {
        #doesn't has HDF5 format
        return 0
    }                    
    set attribute_type string
    set gidpost_version [$h get_attribute -default "" / "GiD Post Results File"]
    if { $gidpost_version != "" } {
        if { [llength $gidpost_version] > 1 } {
            #49, 46, 48
            #initial tclhdf5 package versions used as trick array of integers representing utf-8 instead of hdf5 string
            set attribute_type integer
            set gidpost_version [$h get_attribute -type $attribute_type -default "" / "GiD Post Results File"]
        }
        set is_hdf5_gid_post 1       
    } else {
        set is_hdf5_gid_post 0        
    }
    $h close   
    return $is_hdf5_gid_post
}

# -np- W [ GiDPost::IsIsogeometricResultsFile {D:\public\miguel\GID-3320-IBER-gidpost-iberwoop-hdf5-corrupt\Troncos331.gid\Troncos331.post.res}]
proc GiDPost::IsIsogeometricResultsFile { filename } {
    package require hdf5
    if { ![gid_filesystem::file exists $filename] } {
        return 0
    }
    set h [GiDPost::GetFreeHDF5Id]
    if { [catch {hdf5 -readonly $h $filename} err] } {
        WarnWin [_ "File %s is not an HDF5 file" $filename]
        return 0
    }
    if { ![IsGiDPost $filename] } {
        return 0
    }
    set is_isogeometric_results_file 0
    set results_path /Results  
    if { [$h is_group $results_path] } {
        set result_paths [$h glob -directory $results_path -types group *] 
        foreach result_path $result_paths {
            set location [$h get_attribute $result_path ResultLocation]
            if { $location == "OnNurbsLine" || $location == "OnNurbsSurface" || $location == "OnNurbsVolume" } {
                set is_isogeometric_results_file 1
                break
            }
        }
    }
    $h close
    return $is_isogeometric_results_file
}


proc GiDPost::ReadPostUnstructuredMesh { h meshpath } {
    variable attribute_type      
    
    set num_meshes 0

    set meshname [$h get_attribute -type $attribute_type $meshpath Name]
    set dimension [$h get_attribute -type $attribute_type -default 3 $meshpath Dimension]
    set element_type [$h get_attribute -type $attribute_type $meshpath ElemType]
    set element_nnode [$h get_attribute -type $attribute_type $meshpath Nnode]   
     
    if { [ $h is_dataset $meshpath/Coordinates/1 ] } {  
        set node_ids [lindex [$h get $meshpath/Coordinates/1] 0]
        for {set idim 0} {$idim < $dimension} {incr idim} {
            set coordinates($idim) [lindex [$h get $meshpath/Coordinates/[expr {$idim+2}]] 0]
        }
    } else {
        #nodes are stored in other mesh, the first one
        set node_ids {}
    }
        
    set element_ids [lindex [$h get $meshpath/Elements/1] 0]
    for {set idim 0} {$idim < $element_nnode} {incr idim} {
        set connectivities($idim) [lindex [$h get $meshpath/Elements/[expr {$idim+2}]] 0]
    }

    set nodes {}       
    set i 0   
    foreach node_id $node_ids {
        for {set idim 0} {$idim < $dimension} {incr idim} {
            lappend nodes [lindex $coordinates($idim) $i]
        }
        for {set idim $dimension} {$idim < 3} {incr idim} {
            lappend nodes 0
        }
        incr i
    }
       
    set elements {}    
    set i 0
    foreach element_id $element_ids {
        for {set idim 0} {$idim < $element_nnode} {incr idim} {          
            lappend elements [lindex $connectivities($idim) $i]
        }
        incr i
    }

    GiD_MeshPost create $meshname $element_type $element_nnode $node_ids $nodes $element_ids $elements
    incr num_meshes
    return $num_meshes
}


proc GiDPost::ReadPostMeshes { filename } {
    package require hdf5
    #not remove package require because could be directly called without be invoked by GiDPost::ReadPost
    if { ![gid_filesystem::file exists $filename] } {        
        return 0      
    }
    
    set h [GiDPost::GetFreeHDF5Id]
    if { [catch {hdf5 -readonly $h $filename} err] } {
        #WarnWin [_ "File %s is not an HDF5 file" $filename]
        return 0
    }
           
    set num_meshes 0

    set meshpath /Meshes
    if { [ $h is_group $meshpath ] } {   
        set groups [$h glob -directory $meshpath -types group *]
        foreach group $groups {            
            incr num_meshes [GiDPost::ReadPostUnstructuredMesh $h $group]            
        }      
    }
    $h close
    return $num_meshes
}

proc GiDPost::GetElementDimension { elemtype } {    
    if { $elemtype == "Linear" } {
        set dim 1
    } elseif { $elemtype == "Triangle" || $elemtype == "Quadrilateral"} {
        set dim 2
    } elseif { $elemtype == "Tetrahedra" || $elemtype == "Pyramid" || $elemtype == "Prism" || $elemtype == "Hexahedra"} {
        set dim 3
    } else {
        set dim 0
    }  
    return $dim
}

proc GiDPost::ReadPostGaussPoints { h } {
    variable attribute_type 
    set num_gauss_points 0 
    
    set previous_gauss_points_names [GiD_Result gauss_point names]
    set path /GaussPoints
    if { [$h is_group $path] } {
        set gauss_points_paths [$h glob -directory $path -types group *]   
        foreach gauss_points_path $gauss_points_paths {
            #set mesh_name [lindex [split $gauss_points_path /] end] ;#to define only for this mesh name
            set name [$h get_attribute -type $attribute_type $gauss_points_path Name]
            if { [lsearch $previous_gauss_points_names $name] == -1 } {            
                set elemtype [$h get_attribute -type $attribute_type $gauss_points_path ElemType]
                set npoint [$h get_attribute -type $attribute_type $gauss_points_path GP_number]                                
                if { $elemtype == "Linear" } {
                    set nodes_included [$h get_attribute -type $attribute_type $gauss_points_path NodesIncluded]
                    set coordinates {}                    
                    #now it is not implemented line gauss points setting its location
                } else {
                    set nodes_included 0
                    set internal_coord [$h get_attribute -type $attribute_type $gauss_points_path InternalCoord]
                    if { $internal_coord } {
                        set coordinates ""
                    } else {
                        set ndim [GiDPost::GetElementDimension $elemtype]
                        for {set idim 1} {$idim<=$ndim} {incr idim} {
                            set v($idim) [lindex [$h get $gauss_points_path/$idim] 0]
                        }
                        set coordinates [list ]
                        for {set ipoint 0} {$ipoint<$npoint} {incr ipoint} {
                            for {set idim 1} {$idim<=$ndim} {incr idim} {
                                lappend coordinates [lindex $v($idim) $ipoint]
                            }
                        }                                                
                    }
                }
                if { $nodes_included } {
                    GiD_Result gauss_point create $name $elemtype $npoint -nodes_included $coordinates
                } else {
                    GiD_Result gauss_point create $name $elemtype $npoint $coordinates
                }        
                incr num_gauss_points
            } else  {
                #gauss points already exists
            }
        }
    }
    return $num_gauss_points    
}

proc GiDPost::ReadPostResultRangeTables { h } {
    variable attribute_type 
    set num_result_ranges_tables 0 
    
    set previous_result_ranges_tables_names [GiD_Result result_ranges_table names]
    set path /ResultRangesTable
    if { [$h is_group $path] } {
        set result_ranges_tables_paths [$h glob -directory $path -types group *]
        foreach result_ranges_tables_path $result_ranges_tables_paths {
            set name [$h get_attribute -type $attribute_type $result_ranges_tables_path Name]
            if { [lsearch $previous_result_ranges_tables_names $name] == -1 } { 
                set ranges_paths [$h glob -directory $result_ranges_tables_path -types group *] 
                set values {}
                foreach ranges_path $ranges_paths {
                    set range_label [$h get_attribute -type $attribute_type $ranges_path Name]
                    set range_min [$h get_attribute -type $attribute_type -default "" $ranges_path Min]
                    set range_max [$h get_attribute -type $attribute_type -default "" $ranges_path Max]
                    lappend values $range_min $range_max $range_label
                }
                GiD_Result result_ranges_table create $name $values
                incr num_result_ranges_tables            
            } else {
                #result ranges table name already exists
            }
        }
    }
    return $num_result_ranges_tables    
}

#deprecated
proc GiDPost::ReadPostResultV10 { h resultpath analysis_name result_name time_step } {
    variable attribute_type   
   
    set num_results 0 
    
    set type [$h get_attribute -type $attribute_type $resultpath ResultType]
    set over [$h get_attribute -type $attribute_type $resultpath ResultLocation]
    set gauss_point_name [$h get_attribute -type $attribute_type -default "" $resultpath GaussPointsName]
    set num_components [$h get_attribute -type $attribute_type -default "" $resultpath NumComponents]

    set component_paths [$h glob -directory $resultpath -types dataset *]

    #GiD_Result use index starting from 0
    set ids {}
    if { $over == "OnGaussPoints" } {
        set gauss_data [GiD_Result gauss_point get $gauss_point_name]
        set gauss_point_nnode [lindex $gauss_data 1]
        #element number is repeated for each gauss point, but GiD_Result expect only once
        #and NewElementNum (used for GiD_MeshPost) start from 0, but GiD_Result expects starting from 1
        set repeated_ids [lindex [$h get $resultpath/1] 0]
        set n [llength $repeated_ids]
        for {set i 0} {$i<$n} {incr i $gauss_point_nnode} {           
            lappend ids [lindex $repeated_ids $i]
        }
    } else {
        set ids [lindex [$h get $resultpath/1] 0]
    }
    set component_names ComponentNames
    set nums {}
    foreach component_path $component_paths {
        lappend nums [lindex [split $component_path /] end]
    }
    foreach i [lrange [lsort -integer $nums] 1 end] {
        set default_name ${result_name}-[expr {$i-1}]
        lappend component_names [$h get_attribute -type $attribute_type -default $default_name $resultpath "Component [expr {$i-1}]"]
        lappend values [lindex [$h get $resultpath/$i] 0]
    }
    set result [list Result $result_name $analysis_name $time_step $type $over]
    if { $over == "OnGaussPoints" } {
        lappend result $gauss_point_name
    }
    GiD_Result create -array $result $component_names [list $ids $values]
    return $num_results     
}

proc GiDPost::ReadPostResultV11 { h result_path } {
    variable attribute_type   
   
    set num_results 0 
    
    set result_name [$h get_attribute $result_path Name]
    set analysis_name [$h get_attribute $result_path Analysis]
    set time_step [$h get_attribute $result_path Step]
    set type [$h get_attribute $result_path ResultType]
    set over [$h get_attribute $result_path ResultLocation]
    set gauss_point_name [$h get_attribute -default "" $result_path GaussPointsName]
    set range_table [$h get_attribute -default "" $result_path RangeTable]
    set num_components [$h get_attribute -default 0 $result_path NumComponents]
    set component_names {}
    for {set i 1} {$i<=$num_components} {incr i} {
        lappend component_names [$h get_attribute $result_path "Component $i"]
    }
    set unit_name [$h get_attribute -default "" $result_path UnitName]
    # unused now by standard GiD
    #set components_local [$h get_attribute -default "" $result_path ComponentsLocal]
    #set is_deformation_vector [$h get_attribute -default 0 $result_path IsDeformationVector]
    #set local_axes_result [$h get_attribute -default 0 $result_path "LocalAxes Result"]
    #set local_axes_step [$h get_attribute -default 0 $result_path "LocalAxes step"]
    #set local_axes_vector [$h get_attribute -default 0 $result_path "LocalAxes vector"]
        

    set ids {}
    if { $over == "OnGaussPoints" } {
        set gauss_data [GiD_Result gauss_point get $gauss_point_name]
        set gauss_point_nnode [lindex $gauss_data 1]
        #element number is repeated for each gauss point, but GiD_Result expect only once
        #and NewElementNum (used for GiD_MeshPost) start from 0, but GiD_Result expects starting from 1
        set repeated_ids [lindex [$h get $result_path/1] 0]
        set n [llength $repeated_ids]
        for {set i 0} {$i<$n} {incr i $gauss_point_nnode} {           
            lappend ids [lindex $repeated_ids $i]
        }
    } elseif { $over == "OnNodes" }  {
        set ids [lindex [$h get $result_path/1] 0]
    } elseif { $over == "OnNurbsLine" || $over == "OnNurbsSurface" || $over == "OnNurbsVolume" } {
        #entity ids repeated num_control_points for each NURBS  
        set ids [lindex [$h get $result_path/1] 0]          
    } else {
        WarnWinText "Unexpected result defined $over"
        return 0
    }
    set component_paths [$h glob -directory $result_path -types dataset *]
    foreach component_path [lrange $component_paths 1 end] {
        lappend values [lindex [$h get $component_path] 0]
    }
    set result [list Result $result_name $analysis_name $time_step $type $over]
    if { $over == "OnGaussPoints" } {
        lappend result $gauss_point_name
    }
    #check that values are not "" to avoid raise warnings when reading in GiD
    if { $range_table != "" } {
        if { $unit_name != "" } {
            if { $component_names != "" } {
                GiD_Result create -array $result [list ComponentNames {*}$component_names] [list Unit $unit_name] [list ResultRangesTable $range_table] [list $ids $values]
            } else {
                GiD_Result create -array $result [list Unit $unit_name] [list ResultRangesTable $range_table] [list $ids $values]
            }
        } else {
            if { $component_names != "" } {
                GiD_Result create -array $result [list ComponentNames {*}$component_names] [list ResultRangesTable $range_table] [list $ids $values]
            } else {
                GiD_Result create -array $result [list ResultRangesTable $range_table] [list $ids $values]
            }
        }
    } else {
        if { $unit_name != "" } {
            if { $component_names != "" } {
                GiD_Result create -array $result [list ComponentNames {*}$component_names] [list Unit $unit_name] [list $ids $values]
            } else {
                GiD_Result create -array $result [list Unit $unit_name] [list $ids $values]
            }
        } else {
            if { $component_names != "" } {
                GiD_Result create -array $result [list ComponentNames {*}$component_names] [list $ids $values]
            } else {
                GiD_Result create -array $result [list $ids $values]
            }
        }
    }
    return $num_results     
}


proc GiDPost::ReadPostResults { filename } { 
    variable gidpost_version
    package require hdf5
    #not remove package require because could be directly called without be invoked by GiDPost::ReadPost       
    if { ![gid_filesystem::file exists $filename] } {
        return 0
    }
    
    set h [GiDPost::GetFreeHDF5Id]
    if { [catch {hdf5 -readonly $h $filename} err] } {
        #WarnWin [_ "File %s is not an HDF5 file" $filename]
        return [list 0 0]
    }          

    if { ![IsGiDPost $filename ] } {               
        return [list 0 0]
    }
  
    set num_graphs 0
    set num_results 0   
    set num_gauss_points [GiDPost::ReadPostGaussPoints $h]
    set num_result_ranges_tables [GiDPost::ReadPostResultRangeTables $h]
    
    set results_path /Results  
    if { [$h is_group $results_path] } {
        if { $gidpost_version >=1.1 } {
            set result_paths [$h glob -directory $results_path -types group *] 
            foreach result_path $result_paths {               
                GiDPost::ReadPostResultV11 $h $result_path                
                incr num_results
            }            
        } else {
            set analysis_paths [$h glob -directory $results_path -types group *]   
            foreach analysis_path $analysis_paths {
                set analysis_name [lindex [split $analysis_path /] end]            
                set result_paths [$h glob -directory $analysis_path -types group *] 
                foreach result_path $result_paths {
                    set result_name [lindex [split $result_path /] end]
                    set time_steps_paths [$h glob -directory $result_path -types group *] 
                    foreach time_step_path $time_steps_paths {
                        set time_step [lindex [split $time_step_path /] end]
                        GiDPost::ReadPostResultV10 $h $time_step_path $analysis_name $result_name $time_step
                    }
                    incr num_results 
                }
            }
        }
    }
        
    $h close   
    return [list $num_results $num_graphs]
}

#to check if the file has meshes, and then delete current ones
proc GiDPost::HasPostMeshes { h } {
    set num_meshes 0
    set meshpath /Meshes
    if { [ $h is_group $meshpath ] } {
        set groups [$h glob -directory $meshpath -types group *]
        set num_meshes [llength $groups]
    }
    return $num_meshes
}

proc GiDPost::ReadPost { filename } { 
    package require hdf5
    
    if { ![gid_filesystem::file exists $filename] } {
        WarnWin [_ "File %s not found" $filename]
        return [list 0 0 0]
    }

    set h [GiDPost::GetFreeHDF5Id]
    if { [catch {hdf5 -readonly $h $filename} err] } {
       WarnWin [_ "File %s is not an HDF5 file" $filename]
       return [list 0 0 0]
    }
    if { ![IsGiDPost $filename] } {
        WarnWin [_ "File %s is not a GiD post HDF5 file" $filename]
        return [list 0 0 0]
    }

    if { [llength [GiD_Info postprocess get all_analysis]] || [GiDPost::HasPostMeshes $h] } {
        #to avoid previous results
        #also avoid previous mesh if there is other mesh in the file to be read
        DoFilesNew Yes
    }
    
    $h close 

    set num_meshes [GiDPost::ReadPostMeshes $filename]
    lassign [GiDPost::ReadPostResults $filename] num_results num_graphs  
    return [list $num_meshes $num_results $num_graphs]
}



proc GiDPost::ReadPostW { } {
    set filename [MessageBoxGetFilename file read [_ "Read post results file"] \
            {} {{{GiD post} {.h5 }} {{All files} {.*}}} ]
    if {$filename == ""} {
        return
    }
    
    set res [GiDPost::ReadPost $filename]
    lassign $res num_meshes num_results num_graphs

    if { $num_results > 0 } {
        #set the first analysis and step to use
        set analysis_to_use [lindex [GiD_Info postprocess get all_analysis] 0]
        set step_to_use [lindex [GiD_Info postprocess get all_steps $analysis_to_use] 0]
        GiD_Process Mescape Results AnalysisSel $analysis_to_use $step_to_use escape
    }
    GidUtils::SetWarnLine [_ "Read %d meshes %d results and %d graphs" $num_meshes $num_results $num_graphs]
}


proc GiDPost::AddToMenuPOST { } {
    if { [GidUtils::AreWindowsDisabled] } {
        return
    }
    if { [GiDMenu::GetOptionIndex Files [list Import [_ "GiD Post"]...] POST] != -1 } {
        return
    }
    #try to insert this menu after the word "Files->Import->XYZ ascii points"
    set position [GiDMenu::GetOptionIndex Files [list Import "XYZ ascii points"] POST]
    if { $position == -1 } {
        set position end
    }
    GiDMenu::InsertOption Files [list Import [_ "GiD Post"]...] $position POST GiDPost::ReadPostW "" "" insertafter _   
}

proc GiDPost::AddToMenu { } {
    GiDPost::AddToMenuPOST
}

#invoke this menus changes
#GiDPost::AddToMenu
#GiDMenu::UpdateMenus

#register the proc to be automatically called when re-creating all menus (e.g. when doing files new)
#GiD_RegisterPluginAddedMenuProc GiDPost::AddToMenu
 
##register the proc to be automatically called when dropping a file
#GiD_RegisterExtensionProc ".h5" PRE GiDPost::ReadPre
