namespace eval BasicTest {
    variable m_messages_in_stdout     0
    variable m_write_in_debug_file    0
}

proc BasicTest::printingInDebugFile {} {
    variable m_write_in_debug_file
    return $m_write_in_debug_file
}

proc BasicTest::printOutput { msg  { title "Warning"} { translate 0} { no_new_line 0} } {
    variable m_write_in_debug_file
    
    if { $m_write_in_debug_file} {
        GidUtils::DebugOpenPrintClose "$msg"
    }
    if { ![ GidUtils::IsTkDisabled] && !$BasicTest::m_messages_in_stdout } {
        WarnWinText $msg $title $translate $no_new_line
        # WarnWinText does not an update anymore,
        # doing it here so that we can see each package require
        update
    } else {
        if { $no_new_line} {
            puts -nonewline "$msg"
        } else {
            puts "$msg"
        }
    }
}

proc BasicTest::printDynamicLibraryPaths {} {
    if { $::tcl_platform(platform) != "windows" } {
        if { $::tcl_platform(os) != "Darwin" } {
            # linux
            if { [ info exists ::env(LD_LIBRARY_PATH)] } {
                BasicTest::printOutput "LD_LIBRARY_PATH = $::env(LD_LIBRARY_PATH)"
            } else {
                BasicTest::printOutput "::env(LD_LIBRARY_PATH) does not exist"
            }
        } else {
            # mac os
            if { [ info exists ::env(DYLIB_LIBRARY_PATH)] } {
                BasicTest::printOutput "DYLIB_LIBRARY_PATH = $::env(DYLIB_LIBRARY_PATH)"
            } else {
                BasicTest::printOutput "::env(DYLIB_LIBRARY_PATH) does not exist"
            }
            if { [ info exists ::env(DYLD_FALLBACK_LIBRARY_PATH)] } {
                BasicTest::printOutput "DYLD_FALLBACK_LIBRARY_PATH = $::env(DYLD_FALLBACK_LIBRARY_PATH)"
            } else {
                BasicTest::printOutput "::env(DYLD_FALLBACK_LIBRARY_PATH) does not exist"
            }
        }
    }
}

# which may be : all, tcltk (default), tcl, tk, import (pre and post import dll)
proc select_pkgs_separated { { which_ones tcltk}} {
    set lst_pkgs_in_gid [ list ]
    # for tcl or all, test verifp
    if { ( $which_ones == "all") || ( $which_ones == "tcltk") || ( $which_ones == "tcl")} {
        set lst_pkgs_in_gid [ list verifp]
    }
    return $lst_pkgs_in_gid
}

# which may be : all, tcltk (default), tcl, tk, import (pre and post import dll)
proc select_pkgs_in_gid { { which_ones tcltk}} {
    set lst_pkgs_separated_tcl [ list Tcl customLib customlib_extras customlib_native_groups \
        customLib_utils customLib_utils::c customLib_utils::math \
        dbf gdal gdalconst \
        gid_bounding_box gid_cross_platform gid_map gid_monitoring gid_quaternion \
        hdf5 http \
        Itcl \
        map::geocode::nominatim map::slippy \
        netcdf \
        objarray ogr osr parse_args pdf4tcl percentencoding \
        rl_json rtree \
        shapetcl \
        sqlite3 \
        Tclx tdbc tdbc::sqlite3 tdom Thread \
        tls tohil vfs ]
    set lst_pkgs_separated_tk [ list Tk BWidget \
        customLib_utils::img customLib_utils::tk \
        dialogwin fulltktable fulltktree \
        gid_helpviewer gid_smart_wizard gid_themes gid_wizard \
        gidscale \
        hugecombo hugelist \
        img::jpeg img::png img::tiff \
        map::slippy::cache  map::slippy::fetcher \
        tkdnd Tkhtml tksvg Tktable treectrl \
        wcb]
    # set lst_pkgs_separated_all [ list Tcl Tk BWidget customLib customlib_extras customlib_native_groups \
    #     customLib_utils customLib_utils::c customLib_utils::img customLib_utils::math customLib_utils::tk \
    #     dbf dialogwin fulltktable fulltktree gdal gdalconst \
    #     gid_bounding_box gid_cross_platform gid_helpviewer gid_map gid_monitoring gid_quaternion gid_smart_wizard gid_themes gid_wizard \
    #     gidscale hdf5 http \
    #     hugecombo hugelist \
    #     img::jpeg img::png img::tiff \
    #     Itcl \
    #     map::geocode::nominatim map::slippy map::slippy::cache map::slippy::fetcher \
    #     netcdf \
    #     objarray ogr osr parse_args pdf4tcl percentencoding \
    #     rl_json rtree \
    #     shapetcl \
    #     sqlite3 \
    #     Tclx tdbc tdbc::sqlite3 tdom Thread \
    #     tkdnd Tkhtml tksvg Tktable tls tohil treectrl vfs \
    #     wcb]
    # BasicTest::printOutput "lst_pkgs_separated_tcl = [ llength $lst_pkgs_separated_tcl]"
    # BasicTest::printOutput "lst_pkgs_separated_tk  = [ llength $lst_pkgs_separated_tk]"
    # BasicTest::printOutput "sum = [ llength [ lsort -unique [ concat $lst_pkgs_separated_tcl $lst_pkgs_separated_tk]]]"
    # BasicTest::printOutput "lst_pkgs_separated_all = [ llength $lst_pkgs_separated_all]"
    set lst_pkgs_separated_all [ lsort -unique [ concat $lst_pkgs_separated_tcl $lst_pkgs_separated_tk]]

    if { $::tcl_platform(platform) == "windows"} {
        # former 'reg' now in Tcl 8.6.1 is registry
        lappend lst_pkgs_separated_tk dde gdi printer registry twapi
        lappend lst_pkgs_separated_all dde gdi printer registry twapi
    }
    if { $::tcl_platform(os) == "Darwin" } {
        lappend lst_pkgs_separated_tk cocoaprint
        lappend lst_pkgs_separated_all cocoaprint
    }
    if { $which_ones == "tcl"} {
        return $lst_pkgs_separated_tcl
    } elseif { $which_ones == "tk"} {
        return $lst_pkgs_separated_tk
    } elseif { ( $which_ones == "all") || ( $which_ones == "tcltk")} {
        return $lst_pkgs_separated_all
    } else {
        return [ list]
    }
}

# which may be : all, tcltk (default), tcl, tk, import (pre and post import dll)
proc select_pkgs_optional { { which_ones all}} {
    set lst_pkgs_optional [ list]
    # for tcl or all, test tdbc*
    if { ( $which_ones == "all") || ( $which_ones == "tcltk") || ( $which_ones == "tcl")} {
        set lst_pkgs_optional [ list tdbc::mysql tdbc::odbc tdbc::postgres]
    }
    return $lst_pkgs_optional
}

# which may be : all, tcltk (default), tcl, tk, import (pre and post import dll)
proc do_test_packages { { which_ones tcltk} { messages_in_stdout 0}} {
    set lst_selection [ list all tcltk tcl tk import]
    if { [ lsearch $lst_selection $which_ones] == -1} {
        BasicTest::printOutput "Wrong test selection '$which_ones'. Should be one of: [ join $lst_selection ", "]"
        return
    }
    interp create test_intp
    set BasicTest::m_messages_in_stdout $messages_in_stdout
    test_intp eval lappend auto_path $::auto_path
    test_intp eval set ::CUSTOM_LIB_USE_NATIVE_GROUPS $::CUSTOM_LIB_USE_NATIVE_GROUPS

    if { [ BasicTest::printingInDebugFile]} {
        GidUtils::DebugOpenPrintCloseWithTimestamp "do_test_packages $which_ones"
        BasicTest::printOutput "test_packages results in [ GidUtils::GetDebugFilename]"
        GidUtils::SetWarnLine "test_packages results in [ GidUtils::GetDebugFilename]"
    }

    BasicTest::printDynamicLibraryPaths
    BasicTest::printOutput "--> Starting some BASIC TESTS : selected test [ string toupper $which_ones]\n"
    # adding package verifp makes GiD crash in linux when deleting the interpreter ...
    # it has something to do with RaiseEvent_AfterChangeLicence(...)
    set lst_pkgs_in_gid [ select_pkgs_in_gid [ string tolower $which_ones]]
    set lst_pkgs_separated [ select_pkgs_separated [ string tolower $which_ones]]
    set lst_pkgs_optional [ select_pkgs_optional [ string tolower $which_ones]]

    set num_packages [ expr [ llength $lst_pkgs_separated] + [ llength $lst_pkgs_in_gid]]
    set num_packages_ok 0
    set num_packages_error 0
    set lst_failed_packages [ list ]
    foreach pkg $lst_pkgs_separated {
        set err 0
        set err_txt ""
        # no new-line: final 1
        BasicTest::printOutput "package require $pkg ... " "basic test" 0 1
        set err [ catch {
            test_intp eval package require $pkg
            #  eval package require $pkg
        } err_txt]
        if { $err} {
            incr num_packages_error
            lappend lst_failed_packages $pkg
        } else {
            incr num_packages_ok
        }
        BasicTest::printOutput " $err_txt"
    }
    if { [ llength $lst_pkgs_optional] != 0} {
        BasicTest::printOutput "--> optional packages, may be used in problem-types"
        foreach pkg $lst_pkgs_optional {
            set err 0
            set err_txt ""
            # no new-line: final 1
            BasicTest::printOutput "package require $pkg ... " "basic test" 0 1
            set err [ catch {
                test_intp eval package require $pkg
                #  eval package require $pkg
            } err_txt]
            BasicTest::printOutput " $err_txt"
        }
    }
    if { [ llength $lst_pkgs_in_gid] != 0} {
        BasicTest::printOutput "--> problematic packages, testing inside GiD's interpreter"
        foreach pkg $lst_pkgs_in_gid {
            set err 0
            set err_txt ""
            # no new-line: final 1
            BasicTest::printOutput "package require $pkg ... " "basic test" 0 1
            set err [ catch {
                eval package require $pkg
                #  eval package require $pkg
            } err_txt]
            if { $err} {
                incr num_packages_error
                lappend lst_failed_packages $pkg
            } else {
                incr num_packages_ok
            }
            BasicTest::printOutput " $err_txt"
        }

        ##############################
        ### package specific tests ###
        ##############################
        BasicTest::printOutput "--> package specific tests"
        foreach pkg [ list objarray verifp rtree tohil gid_json] \
            test_file [ list [ file join $::GIDDEFAULTTCL objarray test.tcl ] \
                            [ file join $::GIDDEFAULTTCL verifp test.tcl ]    \
                            [ file join $::GIDDEFAULTTCL rtree test.tcl ]     \
                            [ file join $::GIDDEFAULTTCL tohil test.tcl ]     \
                            [ file join $::GIDDEFAULTTCL gid_json test.tcl ]  \
                           ] \
            test_name [ list test_objarray test_verifp test_rtree test_tohil test_gid_json] {
                BasicTest::printOutput "testing package $pkg ... " "basic test" 0 1
            set failed -1
            set err [ catch {
                set failed [ test_intp eval package require $pkg ; source $test_file ; $test_name]
                #  eval package require $pkg
            } err_txt]
            if { !$err} {
                if { $failed == 0} {
                    set txt "OK passed"
                } else {
                    set txt "$failed FAILED tests ( failed = $failed)"
                }
            } else {
                set txt "Error loading tests: $err_txt"
            }
            BasicTest::printOutput " $txt"
        }
        BasicTest::printOutput "--> Done with some BASIC TESTS with [ string toupper $which_ones] packages"
        BasicTest::printOutput "--> [ string toupper $which_ones] packages ok: $num_packages_ok    error: $num_packages_error     total: $num_packages"
    }

    BasicTest::printDynamicLibraryPaths
    interp delete test_intp

    if { ( $which_ones == "all") || ( $which_ones == "import")} {
        if { $num_packages_error == 0} {
            # avoid creation of .vv and .png
            set old_ReadSaveViewAutomatic [ GiD_Set ReadSaveViewAutomatic ]
            set old_PostReadResultState [ GiD_Set PostReadResultState ]
            GiD_Set ReadSaveViewAutomatic 0
            GiD_Set PostReadResultState 0

            # only run import test if package loading was ok, 
            # may be the failed one(s) is the one we need to import
            test_import_create_check_table
            test_pre_import
            test_post_import

            # restore variable values
            GiD_Set ReadSaveViewAutomatic $old_ReadSaveViewAutomatic
            GiD_Set PostReadResultState $old_PostReadResultState

        } else {
            BasicTest::printOutput "Skipping import tests as some Tcl packages failed to load."
            BasicTest::printOutput "To run the import tests execute: -np- test_packages import        or -np- test_packages all"
        }
    } else {
        if { $which_ones != "import" } {
            BasicTest::printOutput "To run the import tests execute: -np- test_packages import        or -np- test_packages all"
        }
    }

    # returns 0 if ok
    if { $num_packages_error} {
        return -code error "List of FAILED packages: $lst_failed_packages"
    } else {
        return -code ok
    }
}



proc do_test_fps { { num_redraws 0} } {
    # set cmd [ list GiD_Process escape escape escape escape escape redraw]
    set cmd [ list GiD_Process 'redraw]
    if { $num_redraws == 0 } {
        # try to guess number of redraws in a second
        # minimum of 30 redraws
        # time returns: XXX microseconds per iteration
        # to 3 redraws to force and count, at least, one swap buffers
        set try_t [ lindex [ time $cmd 2] 0]
        set num_redraws [ expr { round( 1 / ( $try_t * 1e-6))}]
        if { $num_redraws < 30} {
            set num_redraws 30
        }
        if { $num_redraws > 300} {
            set num_redraws 300
        }
    }
    # BasicTest::printOutput text title translation no_new_line
    BasicTest::printOutput "Redrawing $num_redraws times ..." "Warning" 0 1
    # for { set i 0} { $i < 30} { incr i} {
    # 	BasicTest::printOutput "Redrawing $i"
    # 	GiD_Process escape escape escape escape escape redraw
    # }
    # time already returns the average time per iteration
    set total_t [ lindex [ time $cmd $num_redraws] 0]
    set fps [ expr round( 1.0 / ( $total_t * 1e-9)) / 1000.0]
    BasicTest::printOutput " done: [ format %.3g $fps] fps."
    return $fps
}

proc createGiDicons {} {
    set oldCreatePointState [ GiD_Set CreateAlwaysNewPoint]
    GiD_Set CreateAlwaysNewPoint 1

    # line # 1
    GiD_Process 'Layers New Line
    GiD_Process Mescape Geometry Create Line 1,0 0,1 escape escape escape escape

    # line # 2
    GiD_Process 'Layers New Arc
    GiD_Process Mescape Geometry Create Arc 1,0 0.707106781186547524,0.707106781186547524 0,1 escape escape escape escape

    # line # 3
    GiD_Process 'Layers New Spline
    GiD_Process Mescape Geometry Create NurbsLine   \
        0.00939113443219438,0.202071448463461280    \
        0.2629124589050134,0.00			    \
        0.6056341191565106,0.20725205262849360	    \
        0.36150392911588575,0.58031132456522280	    \
        0.1408445563706704,0.39378168859685820	    \
        0.0,0.50777047246490290			    \
        0.25352132447281905,1.00		    \
        0.97652637519056,0.65802968238310770	    \
        0.7089197528264648,0.24352247867865350	    \
        0.8450701457380534,0.072537753652852620	    \
        1.0,0.217616359406025140                    \
        escape escape escape escape

    GiD_Process 'Layers New Surface
    # line # 4
    GiD_Process Mescape Geometry Create Arc 0.67,0 0.42,0.42 0,0.67 escape escape escape escape
    # surface # 1
    GiD_Process Mescape Utilities Copy Lines DoExtrude Surfaces MaintainLayers Translation \
        FNoJoin 0.0,0.0,0.0 FNoJoin 0.33,0.33,0.0 \
        4  \
        escape escape escape escape

    # Mesh last surface as the ViewGeomMesh icon: one side with 3 quads, another with 4 quads
    GiD_Process 'Layers New SurfaceOnly
    GiD_Process 'Layers Entities SurfaceOnly Surfaces 1 escape \
        escape escape escape escape
    GiD_Process Mescape Meshing Structured Surfaces 1 escape \
        3 6 escape \
        4 5 escape \
        escape escape escape escape
    GiD_Process Mescape Meshing ElemType Quadrilateral 1 \
        escape escape escape escape
    GiD_Process Mescape Meshing Generate 0.01 MeshingParametersFrom=Model \
        escape escape escape escape

    # to turn back to geometry view
    # GiD_Process Mescape Geometry Create escape escape escape escape
    GiD_Process 'SetViewMode Geometry

    # volume # 1
    GiD_Process 'Layers New Volume
    # Prism #sides x_centre y_centre z_centre x_normal y_normal z_normal radius
    GiD_Process Mescape Geometry Create Object Prism 4 \
        0.5,0.5,0.0 \
        0.0,0.0,1.0 \
        0.707106781186547524 1 \
        escape escape escape escape
    GiD_Process 'Layers New VolumeOnly
    GiD_Process 'Layers Entities VolumeOnly Volumes 1 escape \
        escape escape escape escape

    # polyline
    GiD_Process 'Layers New Poly
    GiD_Process Mescape Geometry Create Line         \
        0.03007511571224468,0.0074074992476406280    \
        0.0,0.348147956118569630		     \
        0.3834587550030616,0.244444093781735760      \
        0.1203009204809529,0.82222227231689480       \
        0.9548877840636072,1.00		             \
        1.0,0.340740456870928960		     \
        0.6992488428775533,0.00                      \
        escape escape escape escape
    GiD_Process 'Layers Color Poly #3b913cff escape escape escape escape
    # in theory
    # GiD_Process Mescape Geometry Create Polyline \
    #	1 2 3 4 5 6 escape escape escape

    # Point-Node
    GiD_Process 'Layers New Point-Node
    # Sphere x_centre y_centre z_centre radius
    # prev radius = 0.707106781186547524
    GiD_Process Mescape Geometry Create Object Sphere 0.5 0.5 0.0 0.5 1 \
        escape escape escape escape
    # #003fff(ff)
    GiD_Process 'Layers Color Point-Node #003fffff escape escape escape escape


    set num_lines [ GiD_Info geometry NumLines]
    set num_surfaces [ GiD_Info geometry NumSurfaces]
    # polyline
    GiD_Process 'Layers New Element
    GiD_Process Mescape Geometry Create Line        \
        0.0,0.33333333333   \
        0.0,0.66666666667   \
        0.33333333333,1.0   \
        1.0,0.25            \
        0.75,0.0            \
        0.55,0.3            \
        close               \
        escape escape escape escape
    set last_line [ GiD_Info geometry NumLines]
    GiD_Process Mescape Geometry Create NurbsSurface \
        [expr $num_lines + 1]:$last_line \
        escape escape escape escape
    set surf_id [ GiD_Info geometry NumSurfaces]
    GiD_Process Meshing ElemenType Triangle $surf_id
    GiD_Process Mescape Meshing Generate 1 MeshingParametersFrom=Model \
        escape escape escape escape
    GiD_Process 'SetViewMode Geometry
    # GiD_Process 'Layers Color Element #3b913cff escape escape escape escape

    GiD_Set CreateAlwaysNewPoint $oldCreatePointState

    # To draw thick lines:
    GiD_OpenGL draw -linewidth 10.0
    GiD_OpenGL draw -pointsize 10.0
    # to create volume icon:
    # reset rotation:
    # GiD_Process 'Rotate Angle 270 90
    # now rotate
    # GiD_Process View Rotate Objaxes x 10 escape escape
    # GiD_Process View Rotate Objaxes y 10 escape escape

    # to get images
    GidUtils::SetMainDrawAreaSize 320 320
    GiD_Process View Hardcopy Options ShowAxes No PrintLogo No escape escape escape escape
}

proc pppp { } {
    #original set to create NurbsLine icon:
    set lst [ list                       \
        -4.61538,-2.62542,0    \
        -3.71237,-3.27759,0    \
        -2.49164,-2.6087,0     \
        -3.3612,-1.40468,0     \
        -4.14716,-2.00669,0    \
        -4.64883,-1.6388,0     \
        -3.74582,-0.0501672,0  \
        -1.17057,-1.15385,0    \
        -2.12375,-2.49164,0    \
        -1.6388,-3.04348,0     \
        -1.08696,-2.57525,0    \
    ]
    set lst [ list  \
        0.0750768,0.083561,0   \
        0.0487892,0.385869,0   \
        0.383957,0.293862,0    \
        0.15394,0.806472,0     \
        0.883423,0.964198,0    \
        0.922854,0.379297,0    \
        0.659978,0.076989,0    \
    ]
    set maxx -100.0
    set maxy -100.0
    set maxz -100.0
    set minx 100.0
    set miny 100.0
    set minz 100.0
    foreach pnt $lst {
        lassign [ split $pnt ,] x y z
        if { $x > $maxx} { set maxx $x}
        if { $y > $maxy} { set maxy $y}
        if { $z > $maxz} { set maxz $z}
        if { $x < $minx} { set minx $x}
        if { $y < $miny} { set miny $y}
        if { $z < $minz} { set minz $z}
    }
    set cx [ expr ( $maxx + $minx) * 0.5]
    set cy [ expr ( $maxy + $miny) * 0.5]
    set cz [ expr ( $maxz + $minz) * 0.5]
    set dx [ expr ( $maxx - $minx)]
    set dy [ expr ( $maxy - $miny)]
    set dz [ expr ( $maxz - $minz)]
    
    # scaled to (0,0 1,1)
    foreach pnt $lst {
        lassign [ split $pnt ,] x y z
        set nx [ expr ( $x - $minx) / $dx]
        set ny [ expr ( $y - $miny) / $dy]
        set nz [ expr ( $z - $minz) / ( $dz ? $dz : 1)]
        W $nx,$ny$nz
    }
}

proc test_new_project {} {
    set current_view_mode [GiD_Info Project ViewMode]
    if { $current_view_mode == "POSTUSE" } {
        GiD_Process Mescape Preprocess yes escape escape escape escape
    } elseif { $current_view_mode == "MESHUSE"} {
        GiD_Process Mescape View SetViewMode Geometry escape escape escape escape
    }
    DoFilesNew NO
}

proc test_import_create_check_table {} {
    # to create this table do following
    # in test_single_import
    # set build_table_file_checks 1
    # run on clean GiD
    # -np- test_packages import
    # -np- test_import_print_check_table
    # copy and paste the output here:
    set ::test_import_checks(POSTUSE,Examples/Import/plate.vtk) {numnodes 315}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/city.dbf) {numpoints 15784}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/box_exodus.e) {numnodes 2171}
    if { $::tcl_platform(platform) == "windows"} {
        # windows uses an old version of python (3.10) with an older version of meshIO
        # whose off import only handles triangles and tref.off contains quadrilaterals
        set ::test_import_checks(GEOMETRYUSE,Examples/Import/tref.off) {numnodes 0}
    } else {
        set ::test_import_checks(GEOMETRYUSE,Examples/Import/tref.off) {numnodes 1280}
    }
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/tref_triangs.off) {numnodes 1280}
    set ::test_import_checks(POSTUSE,Examples/Import/tref.off) {numnodes 1280}
    set ::test_import_checks(POSTUSE,Examples/Import/tref_triangs.off) {numnodes 1280}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/usa.vtk) {numnodes 3652}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/advanced_brep.ifc) {numpoints 8}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/Church.igs) {numpoints 47}
    set ::test_import_checks(POSTUSE,Examples/Post/mesa.post.res) {numnodes 19}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/queen.obj) {numnodes 15556}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/queen_meshio.obj) {numnodes 15556}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/Eco.nas) {numnodes 2952}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/teapot.ply) {numnodes 530}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/Cathedral.dxf) {numpoints 311}
    set ::test_import_checks(POSTUSE,Examples/Post/mensu.post.res) {numnodes 116}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/hinge.sat) {numpoints 48}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/plate.vtk) {numnodes 315}
    set ::test_import_checks(POSTUSE,Examples/Post/test_hdf5.post.h5) {numnodes 18}
    set ::test_import_checks(GEOMETRYUSE,resources/textures/GiDlogo32.png) {numpoints 1089}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/tube_box_union.3dm) {numpoints 24}
    set ::test_import_checks(POSTUSE,Examples/Post/temppnts.post.res) {numnodes 52}
    set ::test_import_checks(POSTUSE,Examples/Import/usa.vtk) {numnodes 3652}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/tossa.arc) {numpoints 26085}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/Mblade.igs) {numpoints 52}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/sample.stp) {numpoints 102}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/Head.sat) {numpoints 6}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/Spoon.vda) {numpoints 16}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/spaceship.3DS) {numnodes 259}
    set ::test_import_checks(POSTUSE,Examples/Post/manual-beam-shell.post.h5) {numnodes 5810}
    set ::test_import_checks(POSTUSE,Examples/Import/queen.obj) {numnodes 15556}
    set ::test_import_checks(POSTUSE,Examples/Import/queen_meshio.obj) {numnodes 15556}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/model.dae) {numnodes 9216}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/Hammer.x_b) {numpoints 77}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/Lug.x_t) {numpoints 10}
    set ::test_import_checks(POSTUSE,Examples/Import/teapot.ply) {numnodes 530}
    set ::test_import_checks(GEOMETRYUSE,Examples/Import/city.shp) {numpoints 15784}
}

proc test_import_get_relative_name { filename} {
    regsub $::GIDDEFAULT $filename {} relative_name
    # in windows $::GIDDEFAULT does not have a trailing '/'
    # so removing it from the begining of relative_name
    if { [ string index $relative_name 0] == "/"} {
        set relative_name [ string range $relative_name 1 end]
    }
    return $relative_name
}

proc test_import_add_entry_to_table { current_view_mode filename what how_many} {
    set relative_name [ test_import_get_relative_name $filename]
    set ::test_import_checks($current_view_mode,$relative_name) [ list $what $how_many]
}

proc test_import_print_check_table {} {
    foreach entry [ array names ::test_import_checks] {
        W "set ::test_import_checks($entry) \{$::test_import_checks($entry)\}"
    }
}

proc test_import_get_table_entry {current_view_mode filename} {
    set relative_name [ test_import_get_relative_name $filename]
    set what []
    set how_many []
    if { [ info exists ::test_import_checks($current_view_mode,$relative_name)]} {
        lassign $::test_import_checks($current_view_mode,$relative_name) what how_many
    }
    return [ list $what $how_many]
}

proc test_import_check_if_result_ok { current_view_mode filename what2check how_many2check} {
    set relative_name [ test_import_get_relative_name $filename]
    set ok true
    if { [ info exists ::test_import_checks($current_view_mode,$relative_name)]} {
        lassign $::test_import_checks($current_view_mode,$relative_name) what how_many
        if { ( $what != $what2check) || ( $how_many != $how_many2check)} {
            set ok false
        }
    } else {
        W "test_import_check_if_result_ok no information about $current_view_mode,$relative_name"
    }
    return $ok
}

proc test_single_import { category filename command} {
    set build_table_file_checks 0

    set current_view_mode [GiD_Info Project ViewMode]
    set oldAutomaticCollapse [ GiD_Set AutoCollapseAfterImport]
    # W "$category//[ file tail $filename]"
    # update
    # after 1000
    GiD_Set AutoCollapseAfterImport 1
    set err [
             catch {
                set current_view_mode [GiD_Info Project ViewMode]
                if { $current_view_mode != "POSTUSE"} {
                    set layer_name "$category//[ file tail $filename]"
                    GiD_Process 'Layers New $layer_name
                }
                {*}$command $filename
                update
             } err_txt
       ]
    GiD_Set AutoCollapseAfterImport $oldAutomaticCollapse
    lassign [ check_something_read] what how_many
    set result_ok true
    if { $build_table_file_checks} {
        test_import_add_entry_to_table $current_view_mode $filename $what $how_many
        # this is a guess and may not be true
        set result_ok [ expr [ llength $what] != 0]
    } else {
        # check if result is as expected
        if { $what == ""} {
            set what "numnodes"
            set how_many 0
        }
        set result_ok [ test_import_check_if_result_ok $current_view_mode $filename $what $how_many]
    }
    if { $result_ok} {
        set ret_val OK
    } else {
        set expected_result [ test_import_get_table_entry  $current_view_mode $filename]
        set ret_val "ERROR in '[ file tail $filename]' : expected '$expected_result' but read '$what $how_many'"
    }
    if { $err} {
        set ret_val "ERROR : $err_txt"
    }
    return $ret_val
}

proc read_shapefile_with_conditions { import_geom_options filename} {
    # we will get filename.dbf
    # change extension to .shp an enable -create_conditions
    set shapefile [ file rootname $filename].shp
    GiD_Process Mescape Files ShapefileRead -create_conditions:1 {*}$import_geom_options $shapefile
}

proc read_shapefile_without_conditions { import_geom_options filename} {
    # we will get filename.dbf
    # change extension to .shp an enable -create_conditions
    set shapefile [ file rootname $filename].shp
    GiD_Process Mescape Files ShapefileRead -create_conditions:0 {*}$import_geom_options $shapefile
}

proc check_something_read { } {
    set there_is_something 0
    set current_view_mode [GiD_Info Project ViewMode]
    set what {}
    set how_many {}
    if { $current_view_mode == "POSTUSE" } {
        set numpoints 0
        set numnodes [ GiD_Info Mesh -post numnodes]
    } else {
        set numpoints [ GiD_Info Geometry numpoints]
        set numnodes [ GiD_Info Mesh numnodes]
    }
    if { ( $numpoints > 0) || ( $numnodes > 0)}  {
        set there_is_something 1
    }

    # in theory should only be one of numpoints or numnodes
    if { $numpoints > 0} {
        lappend what numpoints
        lappend how_many $numpoints
    } elseif { $numnodes > 0} {
        lappend what numnodes
        lappend how_many $numnodes
    }
    # return $there_is_something
    return [ list $what $how_many]
} 

proc test_pre_import { } {
    # Clear everything
    test_new_project

    # List of Files to Import
    set base_folder [gid_filesystem::get_folder_standard Examples]
    set import_files [ glob -directory [ file join $base_folder Import] *]
    W "Testing [ llength $import_files] import files"
    # adding specific file for testing gdal:
    lappend import_files [ file join [gid_filesystem::get_folder_standard textures] GiDlogo32.png]
    
    set lst_ext [ list .dxf .igs .nas .x_t .x_b \
                      .vda .sat .3ds \
                      .obj .ply .dae .png \
                      .stp .step .STEP \
                      .shp .dbf \
                      .e .3dm .arc .ifc .vtk \
                     .off ]
    set lst_category [ list DXFRead IgesRead NASTRANRead ParasolidRead ParasolidRead \
                           VDARead AcisRead 3DStudioRead \
                           OBJRead PlyRead TclCollada TclGDAL \
                           STEPRead STEPRead STEPRead \
                           ShapefileRead ShapefileDbfRead \
                           TclExodus RhinoRead ArcInfoGDALRead IfcRead VtkRead \
                           OffRead
                          ]
    set import_geom_options [ list -collapse:1 -autotol:1 -ignorelayer:1 -collapse_allow_more_tasks:1 --]
    set lst_cmds [ list \
                       [ list GiD_Process Mescape Files DXFRead {*}$import_geom_options] \
                       [ list GiD_Process Mescape Files IgesRead {*}$import_geom_options] \
                       [ list GiD_Process Mescape Files NASTRANRead ] \
                       [ list GiD_Process Mescape Files ParasolidRead {*}$import_geom_options] \
                       [ list GiD_Process Mescape Files ParasolidRead {*}$import_geom_options] \
                       \
                       [ list GiD_Process Mescape Files VDARead {*}$import_geom_options] \
                       [ list GiD_Process Mescape Files AcisRead {*}$import_geom_options] \
                       [ list GiD_Process Mescape Files 3DStudioRead ] \
                       \
                       [ list GiD_Process Mescape Files OBJRead ] \
                       [ list GiD_Process Mescape Files PlyRead ] \
                       { Collada::ReadPreSingleFile} \
                       { GDAL::ReadPreRasterSingleFile} \
                       \
                       [ list GiD_Process Mescape Files STEPRead ] \
                       [ list GiD_Process Mescape Files STEPRead ] \
                       [ list GiD_Process Mescape Files STEPRead ] \
                       \
                       [ list read_shapefile_without_conditions $import_geom_options] \
                       [ list read_shapefile_with_conditions $import_geom_options] \
                       \
                       {Exodus::ReadPreSingleFile} \
                       [ list GiD_Process Mescape Files RhinoRead] \
                       {GDAL::ReadPreRasterSingleFile} \
                       [ list GiD_Process Mescape Files IfcRead] \
                       [ list GiD_Process Mescape Files VTKRead] \
                       {MeshIO::ReadPreSingleFile} \
                      ]

    # skip .vtk files in pre as they do not have VTKvoxels
    # lappend lst_ext [ list .vtk]
    # lappend lst_category [ list LoadVTKVoxels]
    # lappend lst_cmds [ list { GiD_Process Mescape Files LoadVTKVoxels }]
    
    ## set lst_ext [ list .png .arc]
    ## set lst_category [ list TclGDAL ArcInfoGDALRead]
    ## set lst_cmds [ list \
    ##     { GDAL::ReadPre} \
    ##     { GDAL::ReadPre} \
    ##     ]
        
    # this is in post:
    # Examples/Import/tref.off
    # same as .obj and .ply
    set err 0
    foreach ext $lst_ext cat $lst_category cmd $lst_cmds {
        if { ( $ext == "") || ( $cat == "") || ( $cmd == "")} {
            WarnWin "import files definition, categories and commands do not match in length"
            set err 1
            break
        }
        set ext2cmd($ext,Category) $cat
        set ext2cmd($ext,Command) $cmd
    }
    if { $err} {
        return
    }
    set missing_import [ list]
    set num_ok 0
    set num_error 0
    set num_files 0
    foreach import_file $import_files {
        if { [ file isdirectory $import_file]} {
            continue
        }
        incr num_files
        set str_idx [ format %03d $num_files]
        set ext [ string tolower [ file extension $import_file]]
        set just_name [ file tail $import_file]
        if { [ info exists ext2cmd($ext,Category)]} {
            # no new-line: final 1
            BasicTest::printOutput "$str_idx. testing $ext2cmd($ext,Category) ... " "basic test" 0 1
            test_new_project
            set result_txt [ test_single_import $ext2cmd($ext,Category) $import_file $ext2cmd($ext,Command)]
            BasicTest::printOutput $result_txt
            if { $result_txt == "OK"} {
                incr num_ok
            } else {
                incr num_error
            }
            # after 1000
        } else {
            if { [file extension $just_name] == ".mtl" } {
                BasicTest::printOutput  "- skipping known Obj Material library: $just_name"
                incr num_ok
            } else {
                BasicTest::printOutput  "- skipping $just_name"
                lappend missing_import $just_name
            }
        }
    }
    set num_missing [ llength $missing_import]
    BasicTest::printOutput "--> Files imported in PRE ok: $num_ok    error: $num_error    missing: $num_missing    total: $num_files"
    if { $num_missing != 0} {
        BasicTest::printOutput "    Following files could not be imported; $missing_import"
    }
    test_new_project
}

proc test_post_import { } {
    # Clear everything
    test_new_project
    GiD_Process Mescape Postprocess escape escape escape escape

    # List of Files to Import
    set base_folder [gid_filesystem::get_folder_standard Examples]
    set import_files [ glob -directory [ file join $base_folder Import] *obj *off *ply *vtk]
    # some more: hf5
    set more_files  [ glob -directory [ file join $base_folder Post] *.h5 *.res]
    set import_files [ concat $import_files $more_files]
    
    set dll_import_foder [ file join [gid_filesystem::get_folder_standard plugins] Import]
    set shared_ext [info sharedlibextension]
    set obj_dll [ file join $dll_import_foder obj_gid_import obj_gid_import$shared_ext]
    set off_dll [ file join $dll_import_foder off_gid_import off_gid_import$shared_ext]
    set ply_dll [ file join $dll_import_foder ply_gid_import ply_gid_import$shared_ext]
    foreach dyn_lib [ list $obj_dll $off_dll $ply_dll] {
        set err [ catch { 
            GiD_PostImportDynLib load $dyn_lib
        } err_txt]
        set lib_name [ file tail $dyn_lib]
        if { $err} {
            BasicTest::printOutput "Could not load post dynamic library '$lib_name'"
            BasicTest::printOutput $err_txt
            # return
        } else {
            BasicTest::printOutput "Post dynamic library '$lib_name' loaded OK"
        }
    }
    set lst_ext [ list .obj .off .ply .res .h5 .vtk]
    set lst_category [ list PostObjImport PostOffImport PostPlyImport \
        PostRead PostHdf5Read PostVtkImport]
    set lst_cmds [ list \
        { GiD_Process Mescape Files ImportDynamicLib CallDynamicLib WaveFrontObjectsimport } \
        { GiD_Process Mescape Files ImportDynamicLib CallDynamicLib ObjectFileFormatImport } \
        { GiD_Process Mescape Files ImportDynamicLib CallDynamicLib PolygonFileFormatImport } \
        { GiD_Process Files Read} \
        { GiD_Process Files Read} \
        { GiD_Process Files VTKRead} \
        ]
    set err 0
    foreach ext $lst_ext cat $lst_category cmd $lst_cmds {
        if { ( $ext == "") || ( $cat == "") || ( $cmd == "")} {
            WarnWin "import files definition, categories and commands do not match in length"
            set err 1
            break
        }
        set ext2cmd($ext,Category) $cat
        set ext2cmd($ext,Command) $cmd
    }
    if { $err} {
        return
    }
    set missing_import [ list]
    set num_ok 0
    set num_error 0
    set num_files 0
    foreach import_file $import_files {
        if { [ file isdirectory $import_file]} {
            continue
        }
        incr num_files
        set ext [ string tolower [ file extension $import_file]]
        set just_name [ file tail $import_file]
        if { [ info exists ext2cmd($ext,Category)]} {
            # no new-line: final 1
            BasicTest::printOutput "testing $ext2cmd($ext,Category) ... " "basic test" 0 1
            test_new_project
            GiD_Process Mescape Postprocess escape escape escape escape
            if { $just_name == "manual-beam-shell.post.h5"} {
                BasicTest::printOutput "( expected to have Result 'Displacements' repeated ) ... " "basic test" 0 1
            }
            set result_txt [ test_single_import $ext2cmd($ext,Category) $import_file $ext2cmd($ext,Command)]
            BasicTest::printOutput $result_txt
            if { $result_txt == "OK"} {
                incr num_ok
            } else {
                incr num_error
            }
            # after 1000
         } else {
            # BasicTest::printOutput  "could not import $just_name"
            lappend missing_import $just_name
        }
    }
    set num_missing [ llength $missing_import]
    BasicTest::printOutput "--> Files imported in POST ok: $num_ok    error: $num_error    missing: $num_missing    total: $num_files"
    if { $num_missing != 0} {
        BasicTest::printOutput "    Following files could not be imported; $missing_import"
    }

    test_post_read_and_export_hdf5

    test_new_project
}

proc test_post_read_and_export_hdf5 { } {
    # clean everything
    
    test_new_project
    GiD_Process Mescape Postprocess escape escape escape escape

    # a read and export to hdf5 test
    set base_folder [gid_filesystem::get_folder_standard Examples]
    set post_file [ file join $base_folder Post mensu.post.res]
    close [ file tempfile test_post test_post]
    set tmp_file ${test_post}.post.h5
    set file_size 0
    set err [ catch {
        GiD_Process Mescape Files Read $post_file
        GiD_Process Mescape Files Saveall HDF5GidPost $tmp_file
        set file_size [ file size $tmp_file]
        file delete $tmp_file
    } err_txt]
    if { !$err && ( $file_size != 0)} {
            BasicTest::printOutput "exporting HDF5 OK"
    } else {
        if { $err} {
            BasicTest::printOutput "exporting HDF5 error: $err_txt"
        }
        if { $file_size == 0} {
            BasicTest::printOutput "exporting HDF5 error: size of exported file is 0"
        }
    }
}
