namespace eval Login {    
    variable username ""
    variable password ""
    variable dict_error_messages [ dict create]
    variable user_programs  ;#cache to avoid repeat expensive function GiD_Login logged_user_programs_available
    variable _automatic_disconnect_cb
}

proc Login::initialize_error_messages { } {
    variable dict_error_messages
    set dict_error_messages [ dict create]
    set lst_codes [ list \
                        "CLOUDLICENCE_ERROR_UNKNOWN"                           \
                        "CLOUDLICENCE_ERROR_NO_INTERNET"                       \
                        "CLOUDLICENCE_ERROR_USERNAME_OR_PASSWORD_NOT_VALID"    \
                        "CLOUDLICENCE_ERROR_NO_LICENCES"                       \
                        "CLOUDLICENCE_ERROR_TOKEN_EXPIRED"                     \
                        "CLOUDLICENCE_ERROR_TOKEN_NOT_JWT"                     \
                        "CLOUDLICENCE_ERROR_TOKEN_UNSIGNED"                    \
                        "CLOUDLICENCE_ERROR_TOKEN_SIGNATURE_ALGORITHM_INVALID" \
                        "CLOUDLICENCE_ERROR_TOKEN_SIGNATURE_INVALID"           \
                        "CLOUDLICENCE_ERROR_TOKEN_HOST_INVALID"                \
                        "CLOUDLICENCE_ERROR_TOKEN_INVALID"                     \
                        "CLOUDLICENCE_ERROR_TOKEN_EXPIRED_NOT_REFRESHED"       \
                        "CLOUDLICENCE_ERROR_MAX_SESSIONS_REACHED"              \
                       ]
    # some messages has subcodes, like the expired token messages, so adding code to all of them
    # default subcode is 'general'
    set def_subcode general
    foreach code $lst_codes {
        # create automatic message from code:
        set len_prefix_to_remove [ string length "CLOUDLICENCE_ERROR_"]
        set msg "ERROR [ string tolower [ regsub -all {_} [ string range $code $len_prefix_to_remove end ] { } ] ]"
        dict set dict_error_messages $code $def_subcode $msg
    }

    # translated messages:
    # some messages has subcodes, like the expired token messages, so adding code to all of them
    dict set dict_error_messages CLOUDLICENCE_ERROR_NO_INTERNET $def_subcode [_ "No Internet connection."]
    dict set dict_error_messages CLOUDLICENCE_ERROR_USERNAME_OR_PASSWORD_NOT_VALID $def_subcode [_ "Username or password not valid."]
    dict set dict_error_messages CLOUDLICENCE_ERROR_NO_LICENCES $def_subcode [_ "Professional mode not allowed: no licence available for this user."]
    dict set dict_error_messages CLOUDLICENCE_ERROR_MAX_SESSIONS_REACHED $def_subcode [_ "Maximum of sessions reached for this user."]
    dict set dict_error_messages CLOUDLICENCE_ERROR_TOKEN_EXPIRED_NOT_REFRESHED $def_subcode  [_ "Sign in or Session has expired and cannot be automatically refreshed. Sign in again."]
    dict set dict_error_messages CLOUDLICENCE_ERROR_TOKEN_EXPIRED_NOT_REFRESHED sign-in [_ "Sign in has expired and cannot be automatically refreshed. Sign in again."]
    dict set dict_error_messages CLOUDLICENCE_ERROR_TOKEN_EXPIRED_NOT_REFRESHED session [_ "Session has expired and cannot be automatically refreshed. Sign in again."]
}

proc Login::get_error_message { code { subcode "general"} } {
    variable dict_error_messages
    set msg "(Login) Unknown error"
    if { [ dict size $dict_error_messages] == 0} {
        Login::initialize_error_messages
    }
    if { [ dict exists $dict_error_messages $code $subcode]} {
        set msg [ dict get $dict_error_messages $code $subcode]
    } else {
        if { $subcode == "general" || $subcode == "" } {
            set msg "(Login) Unknown error '$code'"
        } else {
            set msg "(Login) Unknown error code='$code' (subcode='$subcode')"
        }
        
    }
    return $msg
}

proc Login::show_error_message { program version code} {
    set msg [ Login::get_error_message $code]
    # gid_client_update_warnline "$program $version : $msg"
    gid_client_MessageWin "$program version $version : $msg"
}

proc Login::Window { } {
    variable username
    variable password

    set is_logged [GiD_Login is_logged]
    if { $is_logged } {
        set username [GiD_Login logged_username]
        set title [_ "Sign out"]
    } else {
        set title [_ "Sign in"]
    }
    set w .gid.login
    InitWindow2 $w -title $title -geometryvariable PrePostLoginWindowGeom \
        -ontop
    #ttk::frame $w.top -style flat.TFrame

    ttk::frame $w.middle -style raised.TFrame -borderwidth 1

    ttk::label $w.middle.lusername -text [_ "Email address"]:
    ttk::entry $w.middle.eusername -width 50 -textvariable Login::username
    setTooltip $w.middle.eusername [_ "The username is your e-mail address"]
    #$w.middle.eusername selection range 0 end

    ttk::label $w.middle.lpassword -text [_ "Password"]:
    ttk::entry $w.middle.epassword -width 50 -textvariable Login::password -show *
    setTooltip $w.middle.epassword [_ "Enter password"]
    bind $w.middle.eusername <Return> "focus $w.middle.epassword"
    bind $w.middle.eusername <KP_Enter> "focus $w.middle.epassword"

    ttk::frame $w.middle.flicence_manager
    ttk::label $w.middle.flicence_manager.llicence_manager -text [_ "Account settings"]...
    ttk::button $w.middle.flicence_manager.licence_manager -image [gid_themes::GetImage cloud_settings.png toolbar] \
        -command [list Login::GoToLicenceManager]
    GidHelp $w.middle.flicence_manager.licence_manager [_ "Licence manager"]
    grid $w.middle.lusername $w.middle.eusername -sticky ew -padx 4
    if { $is_logged } {
        $w.middle.eusername configure -state disabled
    } else {
        grid $w.middle.lpassword $w.middle.epassword -sticky ew -padx 4
    }
    grid $w.middle.flicence_manager -columnspan 2 -sticky e
    grid $w.middle.flicence_manager.llicence_manager $w.middle.flicence_manager.licence_manager -sticky e -padx 4

    grid columnconfigure $w.middle 1 -weight 1

    if { $::GID_ENABLE_DATAMANAGER != 0 } {
        ttk::frame $w.middle.fdata_manager
        
        GidDataManager::frameInWindow $w.middle.fdata_manager

        grid $w.middle.fdata_manager -columnspan 2 -sticky e
        grid columnconfigure $w.middle 2 -weight 1
    }

    ttk::frame $w.bot -style BottomFrame.TFrame
    set next_action ""
    if { $is_logged } {
        set command "if { \[winfo exists $w] } { destroy $w } ; Login::Logout"
        ttk::button $w.bot.logout -text [_ "Sign out"] -command $command -style BottomFrame.TButton
        set next_action $command
    } else {
        set command "Login::Login $w"
        ttk::button $w.bot.login -text [_ "Sign in"] -command $command -style BottomFrame.TButton
        set next_action $command
    }

    bind $w.middle.epassword <Return> $next_action
    bind $w.middle.epassword <KP_Enter> $next_action

    ttk::button $w.bot.close -text [_ "Close"] -command [list destroy $w] -style BottomFrame.TButton


    ##grid $w.top.msg -sticky nsew
    #grid rowconfigure $w.top 0 -weight 1
    #grid columnconfigure $w.top 0 -weight 1

    if { $is_logged } {
        grid $w.bot.logout $w.bot.close -padx 2 -pady 3
    } else {
        grid $w.bot.login $w.bot.close -padx 2 -pady 3
    }
    grid anchor $w.bot center

    #grid $w.top -sticky nsew -padx 4 -pady 4
    grid $w.middle -sticky nsew -padx 4 -pady 4
    grid $w.bot -sticky ew -padx 4 -pady 4
    grid rowconfigure $w 0 -weight 1
    grid columnconfigure $w 0 -weight 1
    #grab $w
    wm minsize $w 300 136

    focus $w.middle.eusername
    return 0
}

proc Login::MenuContextual { T x y } {
    set m $T.menucontextual
    if { [winfo exists $m] } {
        destroy $m
    }
    menu $m
    set is_logged [GiD_Login is_logged]
    if { $is_logged } {
        $m add command -label [_ "Sign out"] -command [list Login::Logout]
    } else {
        $m add command -label [_ "Sign in"] -command [list Login::Window]
    }
    set x [expr [winfo rootx $T]+$x+2]
    set y [expr [winfo rooty $T]+$y]
    GiD_PopupMenu $m $x $y
}

proc Login::Login { w } {
    variable username
    variable password
    set fail 0
    set username [string trim $username]
    set password [string trim $password]
    if { $username == "" } {
        WarnWin [_ "An username must be entered"]
        return
    }
    if { [string first @ $username] == -1 } {
        WarnWin [_ "wrong username: must be an e-mail"]
        return
    }
    if { $password == "" } { 
        WarnWin [_ "A password must be entered"]
        return
    }
    destroy $w
    GidUtils::WaitState
    if { [catch { GiD_Login login $username $password } error_code] } {
        # do not remember the password
        set password ""
        set fail 1
        set message [ Login::get_error_message $error_code]
        WarnWin $message
    } else {
        # do not remember the password
        set password ""
        set program gid
        set main_version [GidUtils::GetGiDVersionMain]
        if { $program == "gid" && (![info exists ::GidPriv(password,mode)] || $::GidPriv(password,mode) == "academic") } {        
            if { [catch { GiD_Login start_session $program $main_version } error_code] } {
                #set fail 1
                #do not consider as fail, only warning, maybe the user want to log for other 
                #features but doesn't has GiD cloud licence
                set message [ Login::get_error_message $error_code]
                GidUtils::SetWarnLine $message
            }
        }
    }
    GidUtils::EndWaitState
    if { $fail } {
        after idle [list Login::Window]
    }
}

proc Login::Logout { } {
    GidUtils::WaitState
    if { [catch { GiD_Login logout } error_code] } {
        if { $error_code == "CLOUDLICENCE_ERROR_NO_INTERNET" } {
            # the stanard message is "No Internet connection" but here we nee to add something more:
            set message [_ "%s Sign out locally only" [ Login::get_error_message $error_code]]
        } else {
            set message [ Login::get_error_message $error_code]
        }
        WarnWin $message
    }
    GidUtils::EndWaitState
}

proc Login::ReadTokenFromFile { } {    
    set login_folder [GiD_GetUserSettingsCommonDirectory]
    set sign_filename [file join $login_folder gid_common.sign]
    if { [file exists $sign_filename] } {
        set user_token ""
        set line [GidUtils::ReadFile $sign_filename]        
        if { [lindex $line 0] == "token" } {
            set user_token [lindex $line 1]
            if { $user_token != "" } {
                if { [catch { GiD_Login login_with_token $user_token } error_code] } {
                    if { $error_code == "CLOUDLICENCE_ERROR_TOKEN_EXPIRED_NOT_REFRESHED" } {
                        set message [ Login::get_error_message $error_code sign-in]
                    } elseif { $error_code == "CLOUDLICENCE_ERROR_NO_INTERNET" } {
                        set message [ Login::get_error_message $error_code]
                    } else {                        
                        set message [_ "Sign error '%s'" [ Login::get_error_message $error_code]]
                    }
                    W $message
                }
                set program gid
                set main_version [GidUtils::GetGiDVersionMain]
                if { $program == "gid" && (![info exists ::GidPriv(password,mode)] || $::GidPriv(password,mode) == "academic") } {                
                    set user_licence_token ""
                    set session_filename [file join $login_folder ${program}-${main_version}.session]
                    if { [file exists $session_filename] } {
                        set line [GidUtils::ReadFile $session_filename]
                        if { [lindex $line 0] == "token" } {
                            set user_licence_token [lindex $line 1]
                            if { $user_licence_token != "" } {
                                if { [catch { GiD_Login start_session_with_token $program $main_version $user_licence_token } error_code] } {
                                    if { $error_code == "CLOUDLICENCE_ERROR_TOKEN_EXPIRED_NOT_REFRESHED" } {
                                        set message [ Login::get_error_message $error_code session]
                                    } else {
                                        set message [_ "Session error '%s'" [ Login::get_error_message $error_code]]
                                    }
                                    W $message
                                }
                            }
                        } else {
                            W "file '$session_filename' unexpected line $line"
                        }
                    }
                }
            }
        } else {
            W "file '$sign_filename' unexpected line $line"
        }                 
    }
}

proc Login::GoToLicenceManager { } {
    set server_url [GiD_Login server_url]
    if { [GiD_Login is_logged] } {
        set user_token [GiD_Login logged_user_token]
        package require verifp
        lassign [lindex [vp_getsysinfo] 0] machine_name os fingerprint
        
        package require http
        set query [eval http::formatQuery [list user_token $user_token machine_name $machine_name]] 
        VisitWeb "https://$server_url/View/WelcomeFromGiD?$query"
    } else {  
        VisitWeb "https://$server_url/View"
    }
}

proc Login::UpdateLoginButton { status username } {

    if { $status == "logout" } {
        if { [info exists ::GidPriv(LoginGiDButtonName)] && [winfo exists $::GidPriv(LoginGiDButtonName)] } {
            $::GidPriv(LoginGiDButtonName) configure -image [gid_themes::GetImage login.png toolbar]
            GidHelp $::GidPriv(LoginGiDButtonName) ""
            bind $::GidPriv(LoginGiDButtonName) <Button-$::gid_right_button> "[list Login::MenuContextual %W %x %y] ; break"
        }

    } elseif { $status == "login" } {
        if { [info exists ::GidPriv(LoginGiDButtonName)] && [winfo exists $::GidPriv(LoginGiDButtonName)] } {
            set image ""
            #if exists user image 32x32 base64 png provided by the login, use it
            set image_data [GiD_Login logged_user_avatar_local]
            if { $image_data != "" } {
                set size [gid_themes::GetImageSize toolbar]
                if { [catch { set image [image create photo -data $image_data] } msg] } {
                    set image ""
                    GidUtils::SetWarnLine [concat [_ "Wrong user image. Use default one"] ": $msg"]
                } else {
                    set image [GidUtils::ResizeImage $image $size $size]
                    GidUtils::ImageSetTransparentCircle $image $size
                }
            }
            if { $image == "" } {
                set image [gid_themes::GetImage logout.png toolbar]
            }
            $::GidPriv(LoginGiDButtonName) configure -image $image
            GidHelp $::GidPriv(LoginGiDButtonName) $username
            #set again this bind because GidHelp is replacing it to show a help submenu
            bind $::GidPriv(LoginGiDButtonName) <Button-$::gid_right_button> "[list Login::MenuContextual %W %x %y] ; break"
        }
    } else {
        error "unexpected status"
    }
    return
}

#cache to avoid repeat expensive function GiD_Login logged_user_programs_available
proc Login::GetLoggedUserProgramsAvailable { username } {
    variable user_programs
    if { ![info exists user_programs($username)] } {
        set user_programs($username) [GiD_Login logged_user_programs_available]
    }
    return $user_programs($username)
}


proc Login::OkDownloadPreferencesData { http_state } {
    set httpcode [lindex [dict get $http_state http] 1]
    if { $httpcode == 200} {
        set data [dict get $http_state body]
        set cloud 1
        ReadPreferencesFromData $data $cloud
        GidUtils::SetWarnLine [_ "Preferences read from cloud"]
    } else {        
        GidUtils::SetWarnLine "Preferences from cloud: error $httpcode [GidDataManager::getHttpCodeMessage $httpcode]"
    }
}

proc Login::OkCreateFoldersAndUploadData { http_state } {
    #set httpcode [lindex [dict get $http_state http] 1]
    #201 {DATAMANAGER_FILE_OR_FOLDER_CREATED}
    #204 {DATAMANAGER_FILE_OR_FOLDER_OVERWRITTEN_OR_DELETED}
    #W "Preferences saved in cloud. http_state $http_state"
    GidUtils::SetWarnLine [_ "Preferences saved to cloud"]
}

#file_key is a string to be used as key to get/set cloud preferences, something like "gid.ini" , "IBER.ini" , ...
proc Login::ReadLoggedUserPreferences { file_key } {
    #must use the future API provided by Javi and then it won't be a file but a string value related to file_key from some database
    #then mtime cannot be used, maybe can be stored as a special comment of the string (or a special attribute stored in the database)
    #and a file cannot be read/print, must implement a version of ReadPreferences/SavePreferences based on the on memory data read from the file.
    #by now use a version that use a hardcoded file
    
    if { $::GID_ENABLE_DATAMANAGER == 1 } {
        set folder_defaults [file normalize [gid_filesystem::get_folder_nextcloud .]]        
        if { $folder_defaults != "" } {       
            #set gid_defaults [file join "R:" $file_key] ;#test with a local disk
            set gid_defaults [file join $folder_defaults $file_key]
            if { [file exists $gid_defaults] } {
                #must take into account the date of this information vs date of other local preferences
                if { [file mtime $gid_defaults] >= [file mtime [GiD_GetUserSettingsFilename]] } {
                    set data [GidUtils::ReadFile $gid_defaults "utf-8" 0]
                    set cloud 1
                    ReadPreferencesFromData $data $cloud             
                }
            }
            set ::GidPriv(user_preferences_from_cloud_were_read) 1
        }
    } elseif { $::GID_ENABLE_DATAMANAGER == 2 } {
        set gid_defaults [file join gid [GiD_Info GiDVersion] $file_key]       
        set prom [GidDataManager::pDownloadData $gid_defaults]
        $prom done Login::OkDownloadPreferencesData GidDataManager::ShowErrorPGetUrl
    }
    return 0   
}

proc Login::SaveLoggedUserPreferences { file_key } {
    #must use the future API provided by Javi...
    #by now use the nexcloud unit
    #set nextcloud_unit "R:";#test with a local disk
            
    if { $::GID_ENABLE_DATAMANAGER == 1 } {
        set nextcloud_unit [GidDataManager::getConnectedPath]
        if { $nextcloud_unit != "" } {
            if { [string index $nextcloud_unit 1] == ":" } {
                append nextcloud_unit /
            }
            set folder_defaults [file join $nextcloud_unit gid [GiD_Info GiDVersion]]
            if { ![file exists $folder_defaults] } {
                file mkdir $folder_defaults
            }
            if { [file exists $folder_defaults] && [file isdirectory $folder_defaults] } {
                set gid_defaults [file join $folder_defaults $file_key]
                #set file_key_mtime [clock seconds] ;#in case of a file is unneded, the file has its date, but for a database need to store also this date
                set cloud 1
                set data [SavePreferencesToData $cloud]
                set fail [GidUtils::WriteFile $gid_defaults $data "w" "utf-8"]
            }
        }
    } elseif { $::GID_ENABLE_DATAMANAGER == 2 } {
        set gid_defaults [file join gid [GiD_Info GiDVersion] $file_key]
        #set file_key_mtime [clock seconds] ;#in case of a file is unneded, the file has its date, but for a database need to store also this date
        set cloud 1
        set data [SavePreferencesToData $cloud]                
        #set prom [GidDataManager::pCreateFoldersAndUploadData $gid_defaults $data]
        #$prom done Login::OkCreateFoldersAndUploadData GidDataManager::ShowErrorPGetUrl
        set prom [GidDataManager::AsyncCreateFoldersAndUploadData $gid_defaults $data]        
        $prom done
    }
    return 0
}
