################
#debugging tools
################

# #trick to profile Tcl code if Tclx package is available
#   package require Tclx
#   profile on ;#to start profiling
#   profile off PR ;#to stop profiling PR is an arbitrary array name)
#   profrep PR real C:/temp/gid_profile.txt ;# to write a report of the profile in a file


proc my_source { filename status } {
    source $filename
    GidUtils::SetWarnLine [ _ "Auto source %s" $filename]
}

#to source easily a tcl file located in scripts
#and track it to automatically re-souce if change in disk
proc SRC { filename } {
    if  { [ string tolower [ file extension $filename]] != ".tcl" } {
        set filename $filename.tcl
    }
    if { [ file pathtype $filename] == "relative" } {
        set full_filename [ file join $::GIDDEFAULT scripts $filename]
    } else {
        set full_filename [ file join $filename]
    }
    if { [ file exists $full_filename] }  {
        source $full_filename
        package require gid_cross_platform
        #gid_cross_platform::track_file_cancel $full_filename
        gid_cross_platform::track_file $full_filename 1000 my_source
    } else {
        error "file '$full_filename' not found"
    }
    return
}

namespace eval GidUtils {
    variable accumulated_commands [ list]
    variable is_view_prev ""
}

#debugging tool: use it to know when is invoked a procedure
#W [ GidUtils::GetStackTrace]
proc GidUtils::GetStackTrace { } {
    set text ""
    for {set i [ expr [ info level]-1]} {$i>=1 } {incr i -1} {
        append text [ info level $i]\n
    }
    return $text 
}

#debugging tool: to get in a general way the name and values of the invoked procedure
# use: W [ GidUtils::GetProcedureAndValues [ info level 0]]
proc GidUtils::GetProcedureAndValues { info_level } {
    set proc_name [ lindex $info_level 0]
    set arg_values [ lrange $info_level 1 end]
    set text $proc_name
    set arg_names [ info args $proc_name]
    foreach arg_name $arg_names arg_value $arg_values {
        append text " $arg_name=$arg_value"
    }
    return $text
}

#to check the current global variable traces, interesting to detect unremoved or repeated traces
proc GidUtils::TestTraces { } {
    set ignorevars {EntryVar EntryVarHilit ActionsMenuVar}
    set ignorearrays {GidPriv(Coordinates)}
    set tracedvars {}
    set badtracedvars {}
    foreach var [ info globals] {
        global $var
        set items [ array names $var]
        if { $items != "" } {
            foreach item $items {
                set traceinfo [ trace info variable [ set var]($item)]
                if { $traceinfo != "" } {
                    if { [ lsearch $ignorearrays [ set var]($item)] == -1 } {
                        lappend tracedvars [ set var]($item)
                    }
                    if { [ llength [ lsort -unique $traceinfo] ] != [ llength $traceinfo] } {
                        lappend badtracedvars [ set var]($item)
                    }
                }
            }
        } else {
            set traceinfo [ trace info variable $var]
            if { $traceinfo != "" } {
                if { [ lsearch $ignorevars $var] == -1 } {
                    lappend tracedvars $var
                }
                if { [ llength [ lsort -unique $traceinfo] ] != [ llength $traceinfo] } {
                    lappend badtracedvars $var
                }
            }
        }
    }
    WarnWinText "vars with trace: $tracedvars"
    if { $badtracedvars != "" } {
        WarnWinText "repeating script: $badtracedvars"
    }
}

#debugging tool: to try to compare variables state
proc GidUtils::PrintVariables { filename also_default } {
  set fp [ open $filename w]
  foreach variable [ GiD_Info variables -pre] {
    if { [ catch {set array_names [ GiD_Set -array_names $variable]} msg] } { 
      puts $fp "$variable ERROR $msg"
      W "GiD_Set -array_names $variable ERROR $msg"
    } else {
      if { [ llength $array_names] } {
        foreach name $array_names {
          set full_name ${variable}(${name})
          if { [ catch {set value [ GiD_Set $full_name]} msg] } {
            puts $fp "$full_name ERROR $msg"
            W "GiD_Set $full_name ERROR $msg"
          } else {
            set default_value ""
            if { !$also_default && [ catch {set default_value [ GiD_Set -default $full_name] } msg] } {
                W "GiD_Set -default $full_name ERROR $msg"
            }
            if { $also_default || $value != $default_value } {
                puts $fp "$full_name $value"
            }
          }
        }
      } else {
        if { [ catch {set value [ GiD_Set $variable]} msg] } {
          puts $fp "$variable ERROR $msg"
          W "GiD_Set $variable ERROR $msg"
        } else {
            set default_value ""
            if { !$also_default &&  [ catch {set default_value [ GiD_Set -default $variable] } msg] } {
                W "GiD_Set -default $variable ERROR $msg"
            }
            if { $also_default || $value != $default_value } {
                puts $fp "$variable $value"
            }
        }
      }
    }
  }
  close $fp
}

#debugging tool: returns the content of array nicely
proc GidUtils::FormatArray {a {pattern *}} {    
    set result ""
    upvar 1 $a array  
    if {![ array exists array]} {
        error "\"$a\" isn't an array"
    }
    set maxl 0
    foreach name [ lsort [ array names array $pattern]] {
        if {[ string length $name] > $maxl} {
            set maxl [ string length $name]
        }
    }
    set maxl [ expr {$maxl + [ string length $a] + 2}]
    foreach name [ lsort [ array names array $pattern]] {
        set nameString [ format %s(%s) $a $name]
        append result [ format "%-*s = %s" $maxl $nameString $array($name)]\n
    }
    return $result
}

proc GidUtils::WriteArrayToFile { array_name file_name } {
    set fo [ open $file_name a]
    if { $fo != "" } {
        puts $fo [ GidUtils::FormatArray $array_name *]
        close $fo
    }
}

#debugging tool: to easily print something to a debug file
# e.g. a common use
# GidUtils::DebugOpenPrintClose [ GidUtils::GetStackTrace]
proc GidUtils::GetDebugFilename {} {
    return [ file join [ gid_cross_platform::get_tmp] gid_debug.txt]
}
proc GidUtils::DebugOpenPrintClose { text } {
    set file_name [ GidUtils::GetDebugFilename]
    set fo [ open $file_name a]
    puts $fo $text
    close $fo                   
}

proc GidUtils::GetFormatedTimestamp { { with_ms 1}} {
    set t_ms [ clock milliseconds]
    set t_s [ expr int( $t_ms / 1000)]
    if { $with_ms} {
        set rel_ms [ format %03d [ expr $t_ms - $t_s * 1000]]
        set timestamp [ clock format $t_s -format "%Y-%m-%d %H:%M:%S.$rel_ms"]
    } else {
        set timestamp [ clock format $t_s -format "%Y-%m-%d %H:%M:%S"]
    }
    return $timestamp
}

proc GidUtils::DebugOpenPrintCloseWithTimestamp { text } {
    set t_ms [ clock milliseconds]
    set t_s [ expr int( $t_ms / 1000)]
    set rel_ms [ expr $t_ms - $t_s * 1000]
    set timestamp [ GidUtils::GetFormatedTimestamp]
    set file_name [ GidUtils::GetDebugFilename]
    set fo [ open $file_name a]
    puts $fo "$timestamp $text"
    close $fo
}

# to have better performance do not open and close each print
proc GidUtils::DebugOpen { { filename "" } } {
    variable fp_debug
    if { [ info exists fp_debug] && $fp_debug != "" } {
        #error "file was not closed. close it"
        GidUtils::DebugClose
    }
    if { $filename == "" } {
        set filename [ GidUtils::GetDebugFilename]
    }
    set fp_debug [ open $filename w]
}

proc GidUtils::DebugAppend { { filename "" } } {
    variable fp_debug
    if { [ info exists fp_debug] && $fp_debug != "" } {
      #error "file was not closed"
      GidUtils::DebugClose
    }
    if { $filename == "" } {
        set filename [ GidUtils::GetDebugFilename]
    }
    set fp_debug [ open $filename a]
}

proc GidUtils::DebugClose { } {
    variable fp_debug
    if { $fp_debug != "" } {
      close $fp_debug
      set fp_debug ""
    }
}

#to be faster not check to open and not flush
proc GidUtils::DebugPrintFast { text } {
  variable fp_debug  
  puts $fp_debug $text
}

proc GidUtils::DebugPrint { text } {
  variable fp_debug
  if { ![ info exists fp_debug] || $fp_debug == "" } {
      GidUtils::DebugOpen
  }
  puts $fp_debug $text
  flush $fp_debug
}

proc GidUtils::DebugFlush { } {
  variable fp_debug
  flush $fp_debug
}

## tool to show the internal process commands
proc GidUtils::ProcessShow { words is_view } {
    variable accumulated_commands
    variable is_view_prev
    global GidPriv
    if { !$is_view || $::GidPriv(ShowProcessCommandsViewFuntions) } {
        if {$is_view != $is_view_prev } {
            if { $is_view_prev == "" } {
                set is_view_prev $is_view
            } else {
                GidUtils::ProcessShowFlush $is_view_prev
            }
        }
        lappend accumulated_commands $words
    }
}

proc GidUtils::ProcessShowFlush { is_view } {
    variable accumulated_commands
    variable is_view_prev
    if { [ llength $accumulated_commands] } {
        W $accumulated_commands [ _ "Show process commands"]
        set accumulated_commands [ list]
    }
    set is_view_prev ""
}

#can call this proc or set the variable ::GidPriv(ShowProcessCommands)
#value 0 or 1
proc GidUtils::ShowProcessCommands { value } {
    global GidPriv
    set event_after_process {GiD_Event_AfterProcess GidUtils::ProcessShow GENERAL gid}
    set event_after_end_command {GiD_Event_AfterEndCommand GidUtils::ProcessShowFlush GENERAL gid}
    if { $value } {
        if { ![ GiD_GetIsRegisteredEventProc {*}$event_after_process] } {
            GiD_RegisterEvent {*}$event_after_process
            GiD_RegisterEvent {*}$event_after_end_command
            GidUtils::SetWarnLine [ _ "Show process commands set to %s" $value]
        }
        if { $::GidPriv(ShowProcessCommands) != 1} {
            #to update also the variable, in case that the proc is called directly and not by the trace
            set ::GidPriv(ShowProcessCommands) 1
        }
    } else {
        if { [ GiD_GetIsRegisteredEventProc {*}$event_after_process] } {
            GiD_UnRegisterEvent {*}$event_after_process
            GiD_UnRegisterEvent {*}$event_after_end_command
            GidUtils::SetWarnLine [ _ "Show process commands set to %s" $value]
            GidUtils::ProcessShowFlush 0
        }
        if { $::GidPriv(ShowProcessCommands) != 0} {
            #to update also the variable, in case that the proc is called directly and not by the trace
            set ::GidPriv(ShowProcessCommands) 0
        }
    }
}

proc GidUtils::ShowProcessCommandsToggle { } {
    global GidPriv
    set ::GidPriv(ShowProcessCommands) [ expr !$::GidPriv(ShowProcessCommands)]
}

proc GidUtils::ShowProcessCommandsOnChangeVariable {} {
    global GidPriv
    GidUtils::ShowProcessCommands $::GidPriv(ShowProcessCommands)
}

proc GidUtils::ProfileTclOnChangeVariable {} {
    global GidPriv
    if { $::GidPriv(ProfileTcl) } {
        GidUtils::SetWarnLine [ _ "Tcl profiler started, click again the menu to stop and show report"]
        package require gid_cross_platform
        package require Tclx
        #profile ?-commands? ?-eval? on
        profile on ;#to start profiling
    } else {        
        profile off ProfileReportArray ;#to stop profiling PR is an arbitrary array name)
        # to write a report of the profile in a file
        set filename [ gid_cross_platform::get_unused_tmp_filename profile .txt]
        #sortKey: calls, cpu or real
        profrep ProfileReportArray real $filename
        unset ProfileReportArray
        set data [ GidUtils::ReadFile $filename "" 0]
        file delete $filename
        W $data
        GidUtils::SetWarnLine [ _ "Tcl profiler finished"]
    }
}

namespace eval GidUtils::Crono {
    variable _text ""
}

proc GidUtils::Crono::Start { } {
    variable _text
    set ::GidUtils::Crono::_start_time [ clock milliseconds]
    set ::GidUtils::Crono::_start_partial_time $::GidUtils::Crono::_start_time
    set _text ""
}

proc GidUtils::Crono::StartPartial { } {
    set ::GidUtils::Crono::_start_partial_time [ clock milliseconds]
    set _text ""
}

proc GidUtils::Crono::Get { } {
    if { ![ info exists ::GidUtils::Crono::_start_time] } {
        GidUtils::Crono::Start
    }
    set current_time [ clock milliseconds]
    return [ expr ( double( $current_time) - $::GidUtils::Crono::_start_time) / 1000.0]
}

proc GidUtils::Crono::GetPartial { } {
    if { ![ info exists ::GidUtils::Crono::_start_partial_time] } {
        GidUtils::Crono::StartPartial
    }
    set current_time [ clock milliseconds]
    set ret [ expr ( double( $current_time) - $::GidUtils::Crono::_start_partial_time) / 1000.0]
    GidUtils::Crono::StartPartial
    return $ret
}

proc GidUtils::Crono::End { } {
    set ret [ GidUtils::Crono::Get]
    GidUtils::Crono::Start
    return $ret
}

proc GidUtils::Crono::AddTimePoint { text} {
    variable _text
    set total_t [ format %.3f [ GidUtils::Crono::Get]]
    set partial_t [ format %.3f [ GidUtils::Crono::GetPartial]]
    append _text "${total_t} s ( ${partial_t} s) - $text\n"
}

proc GidUtils::Crono::GetTimePointText { } {
    variable _text
    return $_text
}
