
#  New menu funcions
#  Now menus are identified by UNTRANSLATED name (English is considered as reference untranslated string)
#  (positions of options in their menus start at 0)

namespace eval GiDMenu {
    variable do_UpdateMenus 1
}

proc GiDMenu::EnableUpdateMenus {} {
    variable do_UpdateMenus
    set do_UpdateMenus 1
}

proc GiDMenu::DisableUpdateMenus {} {
    variable do_UpdateMenus
    set do_UpdateMenus 0
}

# Updates changes
proc GiDMenu::UpdateMenus {} {
    variable do_UpdateMenus
    if { $do_UpdateMenus} {
        after 100 [list CreateTopMenus]
    }
}


# -----------------------------------------------------------------------------
#  GiDMenu::Delete
#
#  Delete a toplevel menu
#  menu_name_untranslated (untranslated english menu name, for example "Files#C#menu"
#  prepost: PRE POST or PREPOST
#  translationfunc can be _ (the default value, for GiD standard strings) or = (for module owned strings)
#  for example:
#    GiDMenu::Delete "Mesh#C#menu" PRE
#    it's also valid the old sintax, without #C#menu , but is more expensive to find, e.g:
#    GiDMenu::Delete "Mesh" PRE
#
#  Remember call GiDMenu::UpdateMenus to apply all changes.
# -----------------------------------------------------------------------------
proc GiDMenu::Delete { menu_name_untranslated prepost {translationfunc _} } {
    if { [GidUtils_IsTkDisabled] } {
        #e.g. batch mode without windows
        return 1
    }
    if {$prepost!="PRE" && $prepost!="POST" && $prepost!="PREPOST"} {
        return -code error [_ "Wrong arg. Must be PRE, POST or PREPOST."]
    }
    if { $prepost eq "PRE" } {
        set where [list "" "PRE"]
    } elseif { $prepost eq "POST" } {
        set where [list "P" "POST"]
    } else {
        set where [list "" "PRE" "P" "POST"]
    }
    set variable_names { MenuEntries MenuCommands MenuAcceler MenuIcons }
    foreach { c desc } $where {
        set pos [GiDMenu::_FindIndex $menu_name_untranslated $desc $translationfunc]
        if { $pos == -1 } {
            error [_ "ERROR: Menu %s doesn't exist in $desc." $menu_name_untranslated]
        }
        upvar \#0 MenuNames$c MenuNames
        foreach variable_name $variable_names {
            upvar \#0 ${variable_name}$c $variable_name
        }
        set MenuNames [lreplace $MenuNames $pos $pos]
        foreach variable_name $variable_names {
            array unset $variable_name $pos,*
            array unset $variable_name $pos
        }
        set len [llength $MenuNames]
        for {set i [expr $pos+1]} {$i <= $len} {incr i} {
            set i_prev [expr {$i-1}]
            foreach variable_name $variable_names {
                foreach item [concat [array names $variable_name $i,*] [array names $variable_name $i]] {
                    set indexes [split $item ,]
                    lset indexes 0 $i_prev
                    set new_item [join $indexes ,]
                    set ${variable_name}($new_item) [set ${variable_name}($item)]
                    unset ${variable_name}($item)
                }
            }
        }
        foreach variable_name $variable_names {
            array unset $variable_name $len,*
            array unset $variable_name $len
        }
    }
}

# -----------------------------------------------------------------------------
#  GiDMenu::Create
#
#  Creates a new menu. New menus are inserted between the 'Calculate' and 'Help' menu
#  menu_name_untranslated (untranslated english menu name, for example "Files#C#menu"
#  prepost: PRE POST or PREPOST
#  translationfunc can be _ (the default value,for GiD standard strings) or = (for module strings)
#  pos: index where the new menu will be inserted, begin from 0, "end" can be used
#       if set to -1 (default) then is inserted before the 'Help' menu
#  Remember call GiDMenu::UpdateMenus to apply all changes.
# -----------------------------------------------------------------------------
proc GiDMenu::Create { menu_name_untranslated prepost {pos0 -1} {translationfunc _}} {
    if { [GidUtils_IsTkDisabled] } {
        #e.g. batch mode without windows
        return 1
    }
    if { $prepost != "PRE" && $prepost != "POST" && $prepost != "PREPOST" } {
        error [_ "Wrong arg. Must be PRE, POST or PREPOST."]
    }
    if {$prepost eq "PRE"} {
        set where [list ""]
    } elseif {$prepost eq "POST"} {
        set where [list "P"]
    } else {
        set where [list "" "P"]
    }
    set variable_names { MenuEntries MenuCommands MenuAcceler MenuIcons }
    foreach c $where {
        upvar \#0 MenuNames$c MenuNames
        foreach variable_name $variable_names {
            upvar \#0 ${variable_name}$c $variable_name
        }
        set pos $pos0
        set length [llength $MenuNames]
        if { $pos == "end" } {
            set pos $length
        }
        if { $pos == -1 } {
            #find the position of the help menu
            if { $c == "" } {
                set pos [GiDMenu::_FindIndex "Help#C#menu" PRE _]
            } else {
                set pos [GiDMenu::_FindIndex "Help#C#menu" POST _]
            }
            if { $pos == -1 } {
                set pos [expr [llength $MenuNames]-1]
            }
        }
        if { $pos < 0 } {
            set pos 0
        } elseif { $pos > $length } {
            set pos $length
        }
        set MenuNames [linsert $MenuNames $pos [$translationfunc $menu_name_untranslated]]
        for { set i $length} { $i > $pos } { incr i -1 } {
            set i_prev [expr {$i-1}]
            foreach variable_name $variable_names {
                foreach item [concat [array names $variable_name $i_prev,*] [array names $variable_name $i_prev]] {
                    set indexes [split $item ,]
                    lset indexes 0 $i
                    set new_item [join $indexes ,]
                    set ${variable_name}($new_item) [set ${variable_name}($item)]
                }
            }
        }
        foreach variable_name $variable_names {
            foreach item [array names $variable_name $pos,*] {
                unset ${variable_name}($item)
            }
            set ${variable_name}($pos) [list ""]
        }
    }
    return 0
}


# -----------------------------------------------------------------------------
#  GiDMenu::InsertOption
#
#  Creates a new option for a given menu in a given  position.
#  menu_name_untranslated: the toplevel menu name untranslated
#  option_name_untranslated: the option name, must be a list, allowing sublevels
#  e.g [list "Create#C#menu" "Arc#C#menu"]
#  it is also accepted the old name without comments: [list "Create" "Arc"]
#  position: the new menu option is inserted after the given position
#  (positions start at 0), "end" can be used
#  prepost: PRE POST or PREPOST
#  Arg 'command' is the command called when the menu option is selected.
#  Arg acceler is an optional key accelerator, like "Control-s"
#  Arg icon is the name of an small icon (16x16 pixels)
#  ins_repl: "replace" (default): the new menu option will replace an old if exists in the given position,
#            "insert" : will be inserted before the position
#            "insertafter" : will be inserted after the position
#  translationfunc can be _ (the default value,for GiD standard strings) or = (for module strings)
#
#  Remember call GiDMenu::UpdateMenus to apply all changes.
# -----------------------------------------------------------------------------

proc GiDMenu::InsertOption { menu_name_untranslated option_name_untranslated position prepost \
    command {acceler ""} {icon ""} {ins_repl "replace"} {translationfunc _} } {
    if { [GidUtils_IsTkDisabled] } {
        #e.g. batch mode without windows
        return 1
    }
    if {$prepost!="PRE" && $prepost!="POST" && $prepost!="PREPOST"} {
        error [_ "Wrong arg. Must be PRE, POST or PREPOST."]
    }

    if {$prepost eq "PRE"} {
        set where [list "" "PRE"]
    } elseif {$prepost eq "POST"} {
        set where [list "P" "POST"]
    } else {
        set where [list "" "PRE" "P" "POST"]
    }
    set variable_names { MenuEntries MenuCommands MenuAcceler MenuIcons }
    foreach {c desc} $where {
        upvar \#0 MenuNames$c MenuNames
        foreach variable_name $variable_names {
            upvar \#0 ${variable_name}$c $variable_name
        }
        set i [GiDMenu::_FindIndex $menu_name_untranslated $desc $translationfunc]
        if { $i == -1 } {
            error [_ "ERROR: Menu %s doesn't exist in $desc." $menu_name_untranslated]
        }
        set i [GiDMenu::_FindOptionIndex $i [lrange $option_name_untranslated 0 end-1] $desc $translationfunc]
        if { $i == -1 } {
            return 1
        }
        if { ![info exists MenuEntries($i)] } {
            set MenuEntries($i) ""
        }
        set tail_opt_name [$translationfunc [lindex $option_name_untranslated end]]
        if { $position == "end" } {
            set position [expr {[llength $MenuEntries($i)]-1}]
        }
        if { $ins_repl == "insertafter" } {
            incr position
        }
        set is_runtime_menu 0
        if { [lindex $command 0] == "-np-" } {
            set command [lrange $command 1 end]
            set is_runtime_menu 1
        }
        if {[llength $MenuEntries($i)]<= $position} {
            lappend MenuEntries($i) $tail_opt_name
            lappend MenuCommands($i) "-np- $command"
            if {[info exists MenuAcceler($i)]} {
                lappend MenuAcceler($i) $acceler
            } else {
                if { $acceler != "" } {
                    set MenuAcceler($i) [lrepeat [expr [llength $MenuEntries($i)]-1] ""]
                    lappend MenuAcceler($i) $acceler
                }
            }
            if {[info exists MenuIcons($i)]} {
                lappend MenuIcons($i) $icon
            } else {
                if { $icon != "" } {
                    set MenuIcons($i) [lrepeat [expr [llength $MenuEntries($i)]-1] ""]
                    lappend MenuIcons($i) $icon
                }
            }
            if { $is_runtime_menu } {
                #trick used by CreateCascadedMenu to add to submenu
                set position [expr {[llength MenuCommands($i)]-1}]
                set MenuEntries($i,$position) ""
                #set MenuCommands($i,$position) ""
            }
        } else {
            if { $ins_repl == "replace" } {
                del_submenus $i $position $desc
                set MenuEntries($i) [lreplace $MenuEntries($i) $position $position $tail_opt_name]
                set MenuCommands($i) [lreplace $MenuCommands($i) $position $position "-np- $command"]
                if { [info exists MenuAcceler($i)] } {
                    if { [llength $MenuAcceler($i)] > $position} {
                        set MenuAcceler($i) [lreplace $MenuAcceler($i) $position $position $acceler]
                    }
                } else {
                    if { $acceler != "" } {
                        set MenuAcceler($i) [lrepeat [llength $MenuEntries($i)] ""]
                        set MenuAcceler($i) [lreplace $MenuAcceler($i) $position $position $acceler]
                    }
                }
                if { [info exists MenuIcons($i)] } {
                    if { [llength $MenuIcons($i)] > $position } {
                        set MenuIcons($i) [lreplace $MenuIcons($i) $position $position $icon]
                    }
                } else {
                    if { $icon != "" } {
                        set MenuIcons($i) [lrepeat [llength $MenuEntries($i)] ""]
                        set MenuIcons($i) [lreplace $MenuIcons($i) $position $position $icon]
                    }
                }
            } else {
                # "insert"
                set MenuEntries($i) [linsert $MenuEntries($i) $position $tail_opt_name]
                set MenuCommands($i) [linsert $MenuCommands($i) $position "-np- $command"]
                if {[info exists MenuAcceler($i)]} {
                    set MenuAcceler($i) [linsert $MenuAcceler($i) $position $acceler]
                } else {
                    if { $acceler != "" } {
                        set MenuAcceler($i) [lrepeat [expr [llength $MenuEntries($i)]-1] ""]
                        set MenuAcceler($i) [linsert $MenuAcceler($i) $position $acceler]
                    }
                }
                if {[info exists MenuIcons($i)]} {
                    set MenuIcons($i) [linsert $MenuIcons($i) $position $icon]
                } else {
                    if { $icon != "" } {
                        set MenuIcons($i) [lrepeat [expr [llength $MenuEntries($i)]-1] ""]
                        set MenuIcons($i) [linsert $MenuIcons($i) $position $icon]
                    }
                }
                move_submenus $i $position $desc "down"
            }
            if { $is_runtime_menu } {
                #trick used by CreateCascadedMenu to add to submenu
                set MenuEntries($i,$position) ""
                #set MenuCommands($i,$position) ""
            }
        }
    }
    return 0
}


# -----------------------------------------------------------------------------
#  GiDMenu::RemoveOption
#
#  Removes an option of a given menu.
#  menu_name_untranslated: the toplevel menu name untranslated
#  option_name_untranslated: the option name, can be a list for sublevels
#  e.g [list "Create#C#menu" "Arc#C#menu"]
#  it is also accepted the old name without comments: [list "Create" "Arc"]
#  prepost: PRE POST or PREPOST
#  translationfunc can be _ (the default value,for GiD standard strings) or = (for module strings)
#
#  Remember call GiDMenu::UpdateMenus to apply all changes.
# -----------------------------------------------------------------------------
proc GiDMenu::RemoveOption {menu_name_untranslated option_name_untranslated prepost {translationfunc _}} {
    if { [GidUtils_IsTkDisabled] } {
        #e.g. batch mode without windows
        return 1
    }
    if {$prepost!="PRE" && $prepost!="POST" && $prepost!="PREPOST"} {
        error [_ "Wrong arg. Must be PRE, POST or PREPOST."]
    }

    if {$prepost eq "PRE"} {
        set where [list "" "PRE"]
    } elseif {$prepost eq "POST"} {
        set where [list "P" "POST"]
    } else {
        set where [list "" "PRE" "P" "POST"]
    }
    set variable_names { MenuEntries MenuCommands MenuAcceler MenuIcons }
    foreach {c desc} $where {
        upvar \#0 MenuNames$c MenuNames
        foreach variable_name $variable_names {
            upvar \#0 ${variable_name}$c $variable_name
        }
        set i [GiDMenu::_FindIndex $menu_name_untranslated $desc $translationfunc]
        if { $i == -1 } {
            error [_ "ERROR: Menu %s doesn't exist in $desc." $menu_name_untranslated]
        }
        set index [GiDMenu::_FindOptionIndex $i $option_name_untranslated $desc $translationfunc]
        if { $index == -1 } {
            return 1
        }
        set i [join [lrange [split $index ,] 0 end-1] ,]
        set position [lindex [split $index ,] end]
        del_submenus $i $position $desc
        move_submenus $i $position $desc "up"
        set MenuEntries($i)  [lreplace $MenuEntries($i) $position $position]
        set MenuCommands($i) [lreplace $MenuCommands($i) $position $position]
        if {[info exists MenuAcceler($i)] && [llength $MenuAcceler($i)] > $position} {
            set MenuAcceler($i) [lreplace $MenuAcceler($i) $position $position]
        }
        if {[info exists MenuIcons($i)] && [llength $MenuIcons($i)] > $position} {
            set MenuIcons($i) [lreplace $MenuIcons($i) $position $position]
        }
    }
    return 0
}

# -----------------------------------------------------------------------------
#  GiDMenu::ModifyOption
#
#  Modify a previous menu identified by menu_name_untranslated option_name_untranslated and prepost
#  menu_name_untranslated: the toplevel menu name untranslated
#  option_name_untranslated: the option name, can be a list for sublevels
#  e.g [list "Create#C#menu" "Arc#C#menu"]
#  it is also accepted the old name without comments: [list "Create" "Arc"]
#  Arg prepost: PRE POST or PREPOST
#  Arg new_option_name new option name (-default- to keep the current)
#  Arg new_command is an optional new command (-default- to keep the current)
#  Arg new_acceler is an optional key accelerator, like "Control-s" (-default- to keep the current)
#  Arg new_icon is the name of an small icon (16x16 pixels) (-default- to keep the current)
#  Arg translationfunc can be _ (the default value,for GiD standard strings) or = (for module strings)
#
#  Remember call GiDMenu::UpdateMenus to apply all changes.
# -----------------------------------------------------------------------------
proc GiDMenu::ModifyOption { menu_name_untranslated option_name_untranslated prepost new_option_name \
    {new_command -default-} {new_acceler -default-} {new_icon -default-} {translationfunc _} } {

    if {$prepost!="PRE" && $prepost!="POST" && $prepost!="PREPOST"} {
        return -code error [_ "Wrong arg. Must be PRE, POST or PREPOST."]
    }

    if {$prepost eq "PRE"} {
        set where [list "" "PRE"]
    } elseif {$prepost eq "POST"} {
        set where [list "P" "POST"]
    } else {
        set where [list "" "PRE" "P" "POST"]
    }
    set variable_names { MenuEntries MenuCommands MenuAcceler MenuIcons }
    foreach {c desc} $where {
        upvar \#0 MenuNames$c MenuNames
        foreach variable_name $variable_names {
            upvar \#0 ${variable_name}$c $variable_name
        }
        set i [GiDMenu::_FindIndex $menu_name_untranslated $desc $translationfunc]
        if { $i == -1 } {
            GidUtils::SetWarnLine [_ "ERROR: Menu %s doesn't exist in $desc." $menu_name_untranslated]
            return 1
        }
        set index [GiDMenu::_FindOptionIndex $i $option_name_untranslated $desc $translationfunc]
        if { $index == -1 } {
            return 1
        }
        set i [join [lrange [split $index ,] 0 end-1] ,]
        set position [lindex [split $index ,] end]
        if { $new_option_name != "-default-" } {
            set MenuEntries($i) [lreplace $MenuEntries($i) $position $position $new_option_name]
        }
        if { $new_command != "-default-" } {
            set MenuCommands($i) [lreplace $MenuCommands($i) $position $position "-np- $new_command"]
        }
        if { $new_acceler != "-default-" } {
            if { [info exists MenuAcceler($i)] } {
                if { [llength $MenuAcceler($i)] > $position} {
                    set MenuAcceler($i) [lreplace $MenuAcceler($i) $position $position $new_acceler]
                }
            } else {
                if { $new_acceler != "" } {
                    set MenuAcceler($i) [lrepeat [llength $MenuEntries($i)] ""]
                    set MenuAcceler($i) [lreplace $MenuAcceler($i) $position $position $new_acceler]
                }
            }
        }
        if { $new_icon != "-default-" } {
            if { [info exists MenuIcons($i)] } {
                if { [llength $MenuIcons($i)] > $position } {
                    set MenuIcons($i) [lreplace $MenuIcons($i) $position $position $new_icon]
                }
            } else {
                if { $new_icon != "" } {
                    set MenuIcons($i) [lrepeat [llength $MenuEntries($i)] ""]
                    set MenuIcons($i) [lreplace $MenuIcons($i) $position $position $new_icon]
                }
            }
        }
    }
    return 0
}


# -----------------------------------------------------------------------------
#  GiDMenu::_FindIndex
#
#  return the menu index
#  menu_name_untranslated (untranslated english menu name, for example "Files#C#menu"
#  prepost: PRE or POST (where to find)
#  translationfunc can be _ (for GiD standard strings) or = (for additional module strings)
# -----------------------------------------------------------------------------
proc GiDMenu::_FindIndex { menu_name_untranslated prepost {translationfunc _} } {
    global MenuNames MenuNamesP
    if {$prepost eq "PRE" } {
        upvar \#0 MenuNames TranslatedNames
        } else { ;#POST
            upvar \#0 MenuNamesP TranslatedNames
        }
        set i [lsearch $TranslatedNames [$translationfunc $menu_name_untranslated]]
        if { $i!= -1 } {
        return $i
    }
    if {$prepost eq "PRE" } {
        #back compatibility: Meshing menu renamed to Mesh in 7.5.9b
        if { $menu_name_untranslated == "Meshing" } {
            set i [lsearch $TranslatedNames [$translationfunc "Mesh#C#menu"]]
            if { $i != -1 } {
                return $i
            }
        }
    }
    #back compatibility: menus renamed from oldname to oldname#C#menu in 8.0
    set postfix "#C#menu"
    set i [lsearch $TranslatedNames [$translationfunc $menu_name_untranslated$postfix]]
    if { $i != -1 } {
        return $i
    }
    return -1
}


#to find a option position by its name. returns -1 if not found
proc GiDMenu::GetOptionIndex { menu_name_untranslated option_name_untranslated prepost {translationfunc _} } {
    if { $prepost != "PRE" && $prepost != "POST" } {
        return -code error [_ "Wrong arg. Must be PRE or POST."]
    }
    set i [GiDMenu::_FindIndex $menu_name_untranslated $prepost $translationfunc]
    if { $i == -1 } {
        return -1
    }

    set index [GiDMenu::_FindOptionIndex $i $option_name_untranslated $prepost $translationfunc]
    if { $index != -1 } {
        set position [lindex [split $index ,] end]
    } else {
        set position -1
    }
    return $position
}


# -----------------------------------------------------------------------------
#  return the submenu option index
#  parentindex: the index (from 0) of the parent menu
#  option_name_untranslated (untranslated english menu name, for example "Export#C#menu>IGES#C#menu")
#  prepost: PRE or POST (where to find)
#  insert:
#  translationfunc can be _ (for GiD standard strings) or = (for additional module strings)
# to remove a separator must specify "---$i" where $i is is the index of separator, starting by 0
# -----------------------------------------------------------------------------
proc GiDMenu::_FindOptionIndex { parentindex nameslist prepost {translationfunc _}} {
    global MenuEntries MenuEntriesP
    if {$prepost eq "PRE" } {
        upvar \#0 MenuEntries TranslatedEntries
        } else { ;#POST
            upvar \#0 MenuEntriesP TranslatedEntries
        }

        set postfix "#C#menu"
        set index $parentindex
        foreach suboption $nameslist {
        set found 0
        set cont 0
        if { ![info exists TranslatedEntries($index)] } {
            return -1
        }
        if { [string range $suboption 0 2] == "---" } {
            set nseparator [string range $suboption 3 end]
            if { $nseparator == "" } {
                set nseparator 0
            }
            set suboption "---"
            set iseparator 0
        }
        if { $suboption == "---" } {
            foreach item $TranslatedEntries($index) {
                if { $item == "---" } {
                    if { $iseparator == $nseparator } {
                        append index ,$cont
                        set found 1
                        break
                    }
                    incr iseparator
                }
                incr cont
            }
        } else {
            #is not a separator
            #find first translating the original name
            set suboption_translated [$translationfunc $suboption]
            set cont 0
            foreach item $TranslatedEntries($index) {
                if { [cmpoption $item $suboption_translated] == 0 } {
                    if { $suboption == "---" } {
                        if { $iseparator == $nseparator} {
                            append index ,$cont
                            set found 1
                            break
                        }
                        incr iseparator
                    } else {
                        append index ,$cont
                        set found 1
                        break
                    }
                }
                incr cont
            }
            if { !$found } {
                #find translating the name adding #C#menu
                set suboption_postfix_translated [$translationfunc $suboption$postfix]
                set cont 0
                foreach item $TranslatedEntries($index) {
                    if { [cmpoption $item $suboption_postfix_translated] == 0 } {
                        append index ,$cont
                        set found 1
                        break
                    }
                    incr cont
                }
            }
            if { !$found } {
                #find translating the name adding #C#menu,verb or noun or nothing
                set cont 0
                foreach item $TranslatedEntries($index) {
                    foreach comment {verb noun nothing} {
                        if { [cmpoption $item [$translationfunc $suboption$postfix,$comment]] == 0 } {
                            append index ,$cont
                            set found 1
                            break
                        }
                    }
                    if { $found } {
                        break
                    }
                    incr cont
                }
            }
            if { !$found } {
                set pos [string first "->" $suboption]
                if { $pos != -1 } {
                    #find translating the name splitting a->b
                    set suboption_part0 [string range $suboption 0 $pos-1]
                    set suboption_part1 [string range $suboption $pos+2 end]
                    set cont 0
                    foreach item $TranslatedEntries($index) {
                        if { [cmpoption $item [$translationfunc $suboption_part0]->[$translationfunc $suboption_part1]] == 0 } {
                            append index ,$cont
                            set found 1
                            break
                        }
                        if { $found } {
                            break
                        }
                        incr cont
                    }
                }
            }
        }
        if { !$found } {
            return -1
        }
    }
    return $index
}

proc GiDMenu::cmpoption { a b } {
    if { [string match *... $a] } {
        set a [string range $a 0 end-3]
    }
    if { [string match *... $b] } {
        set b [string range $b 0 end-3]
    }
    return [string compare $a $b]
}


proc GiDMenu::del_submenus {i position pre} {
    global MenuNames MenuEntries MenuCommands MenuAcceler MenuIcons
    global MenuNamesP MenuEntriesP MenuCommandsP MenuAccelerP MenuIconsP

    if {$pre=="PRE"} {
        if {[info exists MenuEntries($i,$position)]} {
            append i ",$position"
            rec_del_menu $i MenuEntries MenuCommands MenuAcceler MenuIcons
        }
    } elseif {$pre=="POST"} {
        if {[info exists MenuEntriesP($i,$position)]} {
            append i ",$position"
            rec_del_menu $i MenuEntriesP MenuCommandsP MenuAccelerP MenuIconsP
        }
    }
}

proc GiDMenu::rec_del_menu {param ent com acc icons} {
    upvar $ent Entries
    upvar $com Commands
    upvar $acc Acceler
    upvar $icons Icons

    set m_length [llength $Entries($param)]

    set pos 0

    unset Entries($param)
    if {[info exists Commands($param)]} {unset Commands($param) }
    if {[info exists Acceler($param)]}  {unset Acceler($param) }
    if {[info exists Icons($param)]}  {unset Icons($param) }

    set base $param
    while {$pos<$m_length} {
        set param "$base,$pos"
        if {[info exists Entries($param)]} {
            rec_del_menu $param Entries Commands Acceler Icons
        }
        set pos [expr $pos +1]
    }
}

proc GiDMenu::move_submenus {i position pre inc} {
    global MenuNames MenuEntries MenuCommands MenuAcceler MenuIcons
    global MenuNamesP MenuEntriesP MenuCommandsP MenuAccelerP MenuIconsP

    set last_menu [llength [split $i ,]]
    incr last_menu

    if {$pre=="PRE"} {
        set j 1
        set m_length [llength $MenuEntries($i)]
        set base $i
        while {$position<$m_length} {
            # read from bottom to top when insert, and opposite when removing
            set param1 "$base,$position"
            set param2 "$base,[expr $m_length - $j]"
            if {[info exists MenuEntries($param1)]} {
                if {$inc=="up"} {
                    rec_move_menu $param1 $last_menu MenuEntries MenuCommands MenuAcceler MenuIcons $inc
                }
            }
            if {[info exists MenuEntries($param2)]} {
                if {$inc=="down"} {
                    rec_move_menu $param2 $last_menu MenuEntries MenuCommands MenuAcceler MenuIcons $inc
                }
            }
            set position [expr $position +1]
            set j [expr $j +1]
        }
    } elseif {$pre=="POST"} {
        set j 1
        set m_length [llength $MenuEntriesP($i)]
        set base $i
        while {$position<$m_length} {
            # read from bottom to top when insert, and opposite when removing
            set param1 "$base,$position"
            set param2 "$base,[expr $m_length - $j]"
            if {[info exists MenuEntriesP($param1)]} {
                if {$inc=="up"} {
                    rec_move_menu $param1 $last_menu MenuEntriesP MenuCommandsP MenuAccelerP MenuIconsP $inc
                }
            }
            if {[info exists MenuEntriesP($param2)]} {
                if {$inc=="down"} {
                    rec_move_menu $param2 $last_menu MenuEntriesP MenuCommandsP MenuAccelerP MenuIconsP $inc
                }
            }
            set position [expr $position +1]
            set j [expr $j +1]
        }
    }
}


proc GiDMenu::dec_inc_menu {param last_menu inc} {
    set up_menus   ""
    set down_menus ""
    set menu ""
    set j 0
    set num 0

    while {$num < [expr $last_menu-1] && $j<[string length $param]} {
        set char [string index $param $j]
        append up_menus $char
        if {$char == ","} {set num [expr $num +1]}
        set j [expr $j +1]
    }

    set char [string index $param $j]
    while {$char != "," && $j<[string length $param]}  {
        append menu $char
        set j [expr $j +1]
        set char [string index $param $j]
    }

    while {$j <= [string length $param]} {
        append down_menus $char
        set j [expr $j +1]
        set char [string index $param $j]
    }
    append down_menus $char

    if {$inc=="down"} {
        set menu [expr $menu + 1]
    } else {
        set menu [expr $menu - 1]
    }
    set new_menu $up_menus$menu$down_menus
    return $new_menu
}


proc GiDMenu::rec_move_menu {param last_menu ent com acc icons inc} {
    upvar $ent Entries
    upvar $com Commands
    upvar $acc Acceler
    upvar $icons Icons

    set m_length [llength $Entries($param)]
    set pos 0

    set new_menu [dec_inc_menu $param $last_menu $inc]

    set Entries($new_menu) $Entries($param)
    unset Entries($param)

    if {[info exists Commands($param)]} {
        set Commands($new_menu) $Commands($param)
        unset Commands($param)
    }
    if {[info exists Acceler($param)]}  {
        set Acceler($new_menu)  $Acceler($param)
        unset Acceler($param)
    }

    if {[info exists Icons($param)]}  {
        set Icons($new_menu)  $Icons($param)
        unset Icons($param)
    }
    set base $param
    while {$pos<$m_length} {
        set param "$base,$pos"
        if {[info exists Entries($param)]} {rec_move_menu $param $last_menu Entries Commands Acceler Icons $inc}
        set pos [expr $pos +1]
    }
}


#utilities not interacting with GiD

namespace eval MathUtils {
    variable EPSILON 1e-10
    variable Epsilon $EPSILON ;#old name for back compatibility
    variable PI 3.1415926535897932384626433832795
    variable Pi $PI ;#old name for back compatibility
    variable 2PI 6.283185307179586476925286766559
    variable RAD_TO_DEGREE 57.295779513082320876798154814105
    variable DEGREE_TO_RAD 0.01745329251994329576923690768489
}

#pi is a list with x y z? coordinates, return the diff vector p2-p1 as a list (the vector from p1 to p2)
proc MathUtils::VectorDiff { p1 p2 } {
    foreach i $p1 j $p2 {
        lappend res [expr {$j-$i}]
    }
    return $res
}

proc MathUtils::VectorSum { p1 p2 } {
    foreach i $p1 j $p2 {
        lappend res [expr {$j+$i}]
    }
    return $res
}

#pi is a list with x y z? coordinates, return the dot prod value
proc MathUtils::VectorDotProd { p1 p2 } {
    set res 0
    foreach i $p1 j $p2 {
        set res [expr $res+($i*$j)]
    }
    return $res
}

#pi is a list with x y z coordinates, return the vectorial prod vector as list
proc MathUtils::VectorVectorialProd { p1 p2 } {
    set ax [lindex $p1 0]
    set ay [lindex $p1 1]
    set az [lindex $p1 2]
    set bx [lindex $p2 0]
    set by [lindex $p2 1]
    set bz [lindex $p2 2]
    set vectorprod [expr $ay*$bz-$az*$by]
    lappend vectorprod [expr $az*$bx-$ax*$bz]
    lappend vectorprod [expr $ax*$by-$ay*$bx]
    return $vectorprod
}

#multiplication of a scalar 'a' by a vector 'p'
proc MathUtils::ScalarByVectorProd { a p } {
    foreach i $p {
        lappend res [expr {$a*$i}]
    }
    return $res
}

#typical operation a*x+y
proc MathUtils::AXPY { a x y } {
    foreach i $x j $y {
        lappend res [expr {$a*$i+$j}]
    }
    return $res
}

#p is a list with x y z? coordinates, return a unitary vector
proc MathUtils::VectorNormalized { p } {
    set m [MathUtils::VectorModulus $p]
    if { [expr abs($m)] < $MathUtils::EPSILON } {
        return $p
    }
    return [MathUtils::ScalarByVectorProd [expr 1.0/$m] $p]
}

#p is a list with x y z? coordinates, return the modulus value
proc MathUtils::VectorModulus { p } {
    return [expr sqrt([MathUtils::VectorDotProd $p $p])]
}

#pi is a list with x y z? coordinates, return the distance value
proc MathUtils::VectorDistance { p1 p2 } {
    return [MathUtils::VectorModulus [MathUtils::VectorDiff $p1 $p2]]
}

#pi is a list with x y z? coordinates, return the distance^2 value
#(for some uses must be faster to compare dist^2 avoiding calculate the square root)
proc MathUtils::VectorDistance2 { p1 p2 } {
    set p [MathUtils::VectorDiff $p1 $p2]
    return [MathUtils::VectorDotProd $p $p]
}

#midpoint between p1 p2
proc MathUtils::VectorMean { p1 p2 } {
    return [MathUtils::ScalarByVectorProd 0.5 [MathUtils::VectorSum $p1 $p2]]
}

#unitary vector from p1 to p2
proc MathUtils::VectorDirectionNormalized { p1 p2 } {
    return [MathUtils::VectorNormalized [MathUtils::VectorDiff $p1 $p2]]
}

#pi is a list with x y z? coordinates, return the angle value in radians
proc MathUtils::VectorAngle { pvertex p2 p3} {
    set v1 [MathUtils::VectorDiff $p2 $pvertex]
    set modul1 [MathUtils::VectorModulus $v1]
    if { $modul1<$MathUtils::EPSILON } return ""

    set v2 [MathUtils::VectorDiff $p3 $pvertex]
    set modul2 [MathUtils::VectorModulus $v2]
    if { $modul2<$MathUtils::EPSILON } return ""

    set cosangle [expr [MathUtils::VectorDotProd $v1 $v2]/($modul1*$modul2)]
    set angle [expr acos($cosangle)]
    return $angle
}

#return the center of the circle that pass by three 3D points
proc MathUtils::VectorCircumCenter { p1 p2 p3 } {
    set ac [MathUtils::VectorDiff $p1 $p3]
    set ab [MathUtils::VectorDiff $p1 $p2]
    set abXac [MathUtils::VectorVectorialProd $ab $ac]
    set ac_len2 [MathUtils::VectorDotProd $ac $ac]
    set s1 [MathUtils::ScalarByVectorProd $ac_len2 [MathUtils::VectorVectorialProd $abXac $ab]]
    set ab_len2 [MathUtils::VectorDotProd $ab $ab]
    set s2 [MathUtils::ScalarByVectorProd $ab_len2 [MathUtils::VectorVectorialProd $ac $abXac]]
    set factor [expr 1.0/(2.0*[MathUtils::VectorDotProd $abXac $abXac])]
    set toCircumsphereCenter [MathUtils::ScalarByVectorProd $factor [MathUtils::VectorSum $s1 $s2]]
    set circumsphereRadius [MathUtils::VectorModulus $toCircumsphereCenter]
    set center [MathUtils::VectorSum $p1 $toCircumsphereCenter]
    return $center
}

#x_axis must be a normalized vector
#return y_axis z_axis
#(0.015625=1/64 is an arbitrary value used historically by AutoCAD)
#if alternative ==0 z_axis is vertical, else y_axis is vertical
proc MathUtils::CalculateLocalAxisFromXAxis { x_axis {alternative 0 } } {
    if { [expr {abs([lindex $x_axis 0])<0.015625 && abs([lindex $x_axis 1]) < 0.015625}] } {
        set y_axis [MathUtils::VectorVectorialProd {0.0 1.0 0.0} $x_axis]
    } else {
        set y_axis [MathUtils::VectorVectorialProd {0.0 0.0 1.0} $x_axis]
    }
    set y_axis [MathUtils::VectorNormalized $y_axis]
    set z_axis [MathUtils::VectorVectorialProd $x_axis $y_axis]
    if { $alternative } {
        #change axis Z to point to Y and axis Y to point to -Z
        set tmp $z_axis
        set z_axis $y_axis
        set y_axis [MathUtils::ScalarByVectorProd -1.0 $tmp]
    }
    return [list $y_axis $z_axis]
}

# x_axis, y_axis, z_axis are normalized vectors
# return the 3 euler angles that define these axes in a shorter description
proc MathUtils::CalculateEulerAnglesFromLocalAxis { x_axis y_axis z_axis } {
    if { [lindex $z_axis 2]< [expr 1.0-$MathUtils::EPSILON] && [lindex $z_axis 2] > [expr -1.0+$MathUtils::EPSILON] } {
        set senb [expr sqrt(1.0-[lindex $z_axis 2]*[lindex $z_axis 2])]
        set angles_1 [expr acos([lindex $z_axis 2])]
        set angles_2 [expr acos([lindex $z_axis 1]/$senb)]
        if { [expr [lindex $z_axis 0]/$senb] < 0.0 } {
            set angles_2 [expr $MathUtils::2PI-$angles_2]
        }
        set angles_0 [expr acos(-1.0*[lindex $y_axis 2]/$senb)]
        if { [expr [lindex $x_axis 2]/$senb] < 0.0 } {
            set angles_0 [expr $MathUtils::2PI-$angles_0]
        }
    } else {
        set angles_0 0.0
        set angles_1 [expr acos([lindex $z_axis 2])]
        set angles_2 [expr acos([lindex $x_axis 0])]
        if { [lindex $x_axis 1] >= 0.0 } {
            set angles_2 [expr $MathUtils::2PI-$angles_2]
        }
    }
    return [list $angles_0 $angles_1 $angles_2]
}

proc MathUtils::FromRadToDegree { x } {
    return [expr $x*$MathUtils::RAD_TO_DEGREE]
}

proc MathUtils::FromDegreeToRad { x } {
    return [expr $x*$MathUtils::DEGREE_TO_RAD]
}

#integrate the xy curve defined by xs and ys. xs must be increasing
#method could be "linear" (piecewise linear segments)
#             or "romberg" Romberg's intgration method of a interpolating cubic spline
proc MathUtils::IntegrateXY { xs ys {method linear} } {
    set value ""
    if { $method == "linear" } {
        set value 0.0
        set x_prev [lindex $xs 0]
        set y_prev [lindex $ys 0]
        foreach x [lrange $xs 1 end] y [lrange $ys 1 end] {
            set dx [expr {$x-$x_prev}]
            set y_mid [expr {($y+$y_prev)*0.5}]
            set value [expr {$value+$dx*$y_mid}]
            set x_prev $x
            set y_prev $y
        }
    } elseif { $method == "romberg" } {
        package require math::interpolate
        package require math::calculus
        set ::f_cubic_splines_coeffs [math::interpolate::prepare-cubic-splines $xs $ys]
        proc f_cubic_splines { x } {
            math::interpolate::interp-cubic-splines $::f_cubic_splines_coeffs $x
        }
        set value [lindex [::math::calculus::romberg f_cubic_splines [lindex $xs 0] [lindex $xs end]] 0]
        unset ::f_cubic_splines_coeffs
        rename f_cubic_splines ""
    } else {
        set value ""
    }
    return $value
}

# return 1 if the number is a power of 2
# n must be an integer
proc MathUtils::IsPowerofTwo { n } {
    if { [string is integer -strict $n] } {
        error "MathUtils::IsPowerofTwo expects an integer. n=$n"
    }
    set is_power_of_two 0
    if {$n > 0 && ($n & (~($n-1))) == $n } {
        set is_power_of_two 1
    }
    return $is_power_of_two
}

#utilities interacting with GiD
namespace eval GidUtils {
}

#register a tcl command for the GiD Undo/Batch, and eval this command
proc GidUtils::EvalAndEnterInBatch_OLD { args } {
    GiD_EnterInBatch "*****MARK START_COMMAND_TCL\n"
    GiD_EnterInBatch "\{-tcl- $args\}\n"
    GiD_EnterInBatch "*****MARK END_COMMAND_TCL\n"
    set old_disable_writebatch [GiD_Project set disable_writebatch]
    if { $old_disable_writebatch != 1 } {
        #try to hide the use of process by this proc to batch and Undo
        GiD_Project set disable_writebatch 1
    }
    #eval $args
    #do not use eval, use uplevel to evaluate at global scope
    uplevel #0 $args
    if { $old_disable_writebatch != 1 } {
        GiD_Project set disable_writebatch $old_disable_writebatch
    }
}

proc GidUtils::EvalAndEnterInBatch { args } {
    GiD_Process "-tcl- $args"
}

# returns the distance (or null) between two selected coordinates
proc GidUtils::GetDistance { } {
    set viewmode [GiD_Info Project ViewMode]
    set num_p1 0
    set prev_selected 1
    set p1 [GidUtils::GetCoordinates [_ "Enter first point (ESC to leave)"]]
    if { $p1=="" } {
        return ""
    }
    if { $::GidPriv(selcoord,id) != -1 && [string is integer -strict $::GidPriv(selcoord,id)]} {
        set num_p1 $::GidPriv(selcoord,id)
        if { $viewmode == "GEOMETRYUSE" } {
            set prev_selected [GiD_Geometry get point $num_p1 selected]
            if { !$prev_selected } {
                GiD_Geometry edit point $num_p1 selected 1
                GiD_Redraw
            }
        } elseif { $viewmode == "MESHUSE" } {
            set prev_selected [GiD_Mesh get node $num_p1 selected]
            if { !$prev_selected } {
                GiD_Mesh edit node $num_p1 selected 1
                GiD_Redraw
            }
        }
    }
    set p2 [GidUtils::GetCoordinates [_ "Enter second point (ESC to leave)"]]
    if { $num_p1 } {
        if { !$prev_selected } {
            set viewmode [GiD_Info Project ViewMode]
            if { $viewmode == "GEOMETRYUSE" } {
                GiD_Geometry edit point $num_p1 selected $prev_selected
                GiD_Redraw
            } elseif { $viewmode == "MESHUSE" } {
                GiD_Mesh edit node $num_p1 selected $prev_selected
                GiD_Redraw
            }
        }
    }
    if { $p2=="" } {
        return ""
    }
    set d [MathUtils::VectorDistance $p1 $p2]
    GidUtils::SetWarnLine [_ "Distance=%s" $d]
    return $d
}

proc GidUtils::GetVector { } {
    set viewmode [GiD_Info Project ViewMode]
    set num_p1 0
    set prev_selected 1
    set p1 [GidUtils::GetCoordinates [_ "Enter origin point (ESC to leave)"]]
    if { $p1=="" } return ""
    if { $::GidPriv(selcoord,id) != -1 && [string is integer -strict $::GidPriv(selcoord,id)]} {
        set num_p1 $::GidPriv(selcoord,id)
        if { $viewmode == "GEOMETRYUSE" } {
            set prev_selected [GiD_Geometry get point $num_p1 selected]
            if { !$prev_selected } {
                GiD_Geometry edit point $num_p1 selected 1
                GiD_Redraw
            }
        } elseif { $viewmode == "MESHUSE" } {
            set prev_selected [GiD_Mesh get node $num_p1 selected]
            if { !$prev_selected } {
                GiD_Mesh edit node $num_p1 selected 1
                GiD_Redraw
            }
        }
    }
    set p2 [GidUtils::GetCoordinates [_ "Enter destination point (ESC to leave)"]]
    if { $num_p1 } {
        if { !$prev_selected } {
            if { $viewmode == "GEOMETRYUSE" } {
                GiD_Geometry edit point $num_p1 selected $prev_selected
                GiD_Redraw
            } elseif { $viewmode == "MESHUSE" } {
                GiD_Mesh edit node $num_p1 selected $prev_selected
                GiD_Redraw
            }
        }
    }
    if { $p2=="" } return ""
    set p [MathUtils::VectorDiff $p1 $p2]
    GidUtils::SetWarnLine [_ "Vector=%s" $p]
    return $p
}

#return the angle in degrees
proc GidUtils::GetAngle { } {
    set viewmode [GiD_Info Project ViewMode]
    set num_p(1) 0
    set prev_selected(1) 1
    set num_p(2) 0
    set prev_selected(2) 1
    set p1 [GidUtils::GetCoordinates [_ "Enter vertex point (ESC to leave)"]]
    if { $p1=="" } return ""
    if { $::GidPriv(selcoord,id) != -1 && [string is integer -strict $::GidPriv(selcoord,id)]} {
        set num_p(1) $::GidPriv(selcoord,id)
        if { $viewmode == "GEOMETRYUSE" } {
            set prev_selected(1) [GiD_Geometry get point $num_p(1) selected]
            if { !$prev_selected(1) } {
                GiD_Geometry edit point $num_p(1) selected 1
                GiD_Redraw
            }
        } elseif { $viewmode == "MESHUSE" } {
            set prev_selected(1) [GiD_Mesh get node $num_p(1) selected]
            if { !$prev_selected(1) } {
                GiD_Mesh edit node $num_p(1) selected 1
                GiD_Redraw
            }
        }
    }
    set p2 [GidUtils::GetCoordinates [_ "Enter second point (ESC to leave)"]]
    if { $p2=="" } {
        if { $num_p(1) } {
            if { $num_p(1) && !$prev_selected(1) } {
                if { $viewmode == "GEOMETRYUSE" } {
                    GiD_Geometry edit point $num_p(1) selected $prev_selected(1)
                    GiD_Redraw
                } elseif { $viewmode == "MESHUSE" } {
                    GiD_Mesh edit node $num_p(1) selected $prev_selected(1)
                    GiD_Redraw
                }
            }
        }
        return ""
    }
    if { $::GidPriv(selcoord,id) != -1 && [string is integer -strict $::GidPriv(selcoord,id)]} {
        set num_p(2) $::GidPriv(selcoord,id)
        if { $viewmode == "GEOMETRYUSE" } {
            set prev_selected(2) [GiD_Geometry get point $num_p(2) selected]
            if { !$prev_selected(2) } {
                GiD_Geometry edit point $num_p(2) selected 1
                GiD_Redraw
            }
        } elseif { $viewmode == "MESHUSE" } {
            set prev_selected(2) [GiD_Mesh get node $num_p(2) selected]
            if { !$prev_selected(2) } {
                GiD_Mesh edit node $num_p(2) selected 1
                GiD_Redraw
            }
        }
    }
    set p3 [GidUtils::GetCoordinates [_ "Enter third point (ESC to leave)"]]
    if { $num_p(1) || $num_p(2) } {
        if { $num_p(1) && !$prev_selected(1) } {
            if { $viewmode == "GEOMETRYUSE" } {
                GiD_Geometry edit point $num_p(1) selected $prev_selected(1)
            } elseif { $viewmode == "MESHUSE" } {
                GiD_Mesh edit node $num_p(1) selected $prev_selected(1)
            }
        }
        if { $num_p(2) && !$prev_selected(2) } {
            if { $viewmode == "GEOMETRYUSE" } {
                GiD_Geometry edit point $num_p(2) selected $prev_selected(2)
            } elseif { $viewmode == "MESHUSE" } {
                GiD_Mesh edit node $num_p(2) selected $prev_selected(2)
            }
        }
        GiD_Redraw
    }
    if { $p3=="" } return ""

    set angle [MathUtils::FromRadToDegree [MathUtils::VectorAngle $p1 $p2 $p3]]
    GidUtils::SetWarnLine [_ "Angle=%s" $angle]
    return $angle
}

#return the center of the arc that pass by three 3D points
proc GidUtils::GetArcCenter { } {
    set viewmode [GiD_Info Project ViewMode]
    set num_p(1) 0
    set prev_selected(1) 1
    set num_p(2) 0
    set prev_selected(2) 1
    set p1 [GidUtils::GetCoordinates [_ "Enter first arc point (ESC to leave)"]]
    if { $p1=="" } return ""
    if { $::GidPriv(selcoord,id) != -1 && [string is integer -strict $::GidPriv(selcoord,id)]} {
        set num_p(1) $::GidPriv(selcoord,id)
        if { $viewmode == "GEOMETRYUSE" } {
            set prev_selected(1) [GiD_Geometry get point $num_p(1) selected]
            if { !$prev_selected(1) } {
                GiD_Geometry edit point $num_p(1) selected 1
                GiD_Redraw
            }
        } elseif { $viewmode == "MESHUSE" } {
            set prev_selected(1) [GiD_Mesh get node $num_p(1) selected]
            if { !$prev_selected(1) } {
                GiD_Mesh edit node $num_p(1) selected 1
                GiD_Redraw
            }
        }
    }
    set p2 [GidUtils::GetCoordinates [_ "Enter second arc point (ESC to leave)"]]
    if { $p2=="" } {
        if { $num_p(1) } {
            if { $num_p(1) && !$prev_selected(1) } {
                if { $viewmode == "GEOMETRYUSE" } {
                    GiD_Geometry edit point $num_p(1) selected $prev_selected(1)
                    GiD_Redraw
                } elseif { $viewmode == "MESHUSE" } {
                    GiD_Mesh edit node $num_p(1) selected $prev_selected(1)
                    GiD_Redraw
                }
            }
        }
        return ""
    }
    if { $::GidPriv(selcoord,id) != -1 && [string is integer -strict $::GidPriv(selcoord,id)]} {
        set num_p(2) $::GidPriv(selcoord,id)
        if { $viewmode == "GEOMETRYUSE" } {
            set prev_selected(2) [GiD_Geometry get point $num_p(2) selected]
            if { !$prev_selected(2) } {
                GiD_Geometry edit point $num_p(2) selected 1
                GiD_Redraw
            }
        } elseif { $viewmode == "MESHUSE" } {
            set prev_selected(2) [GiD_Mesh get node $num_p(2) selected]
            if { !$prev_selected(2) } {
                GiD_Mesh edit node $num_p(2) selected 1
                GiD_Redraw
            }
        }
    }
    set p3 [GidUtils::GetCoordinates [_ "Enter third arc point (ESC to leave)"]]
    if { $num_p(1) || $num_p(2) } {
        if { $viewmode == "GEOMETRYUSE" } {
            if { $num_p(1) && !$prev_selected(1) } {
                GiD_Geometry edit point $num_p(1) selected $prev_selected(1)
            }
            if { $num_p(2) && !$prev_selected(2) } {
                GiD_Geometry edit point $num_p(2) selected $prev_selected(2)
            }
            GiD_Redraw
        } elseif { $viewmode == "MESHUSE" } {
            if { $num_p(1) && !$prev_selected(1) } {
                GiD_Mesh edit node $num_p(1) selected $prev_selected(1)
            }
            if { $num_p(2) && !$prev_selected(2) } {
                GiD_Mesh edit node $num_p(2) selected $prev_selected(2)
            }
            GiD_Redraw
        }
    }
    if { $p3=="" } return ""
    set center [MathUtils::VectorCircumCenter $p1 $p2 $p3]
    set radius [MathUtils::VectorModulus [MathUtils::VectorDiff $center $p1]]
    GidUtils::SetWarnLine "R=$radius, C=([join $center ,])"
    return $center
}

#return the curvature o the curve with id=num relative to the parameter t (from 0.0 to1.0)
proc GidUtils::GetCurveCurvature { num t } {
    set dert [GiD_Info parametric line $num deriv_t $t]
    set dertt [GiD_Info parametric line $num deriv_tt $t]
    set a [::MathUtils::VectorModulus [MathUtils::VectorVectorialProd $dert $dertt]]
    set b [::MathUtils::VectorModulus $dert]
    return [expr $a/($b*$b*$b)]
}


# It returns the enties identifier numbers (for example 1 3 5:8 represents 1 3 5 6 7 8)
# type can be "Points" "Lines" "Surfaces" "Volumes" "Dimensions" "AllTypes" "Nodes" "Elements" "Faces" or "Graphs"
# to select only a subtype could set type to something like this
#     "Elements ElementType Triangle Quadrilateral ..."
#     "Faces FaceParentType Triangle Quadrilateral"
#     "Lines LineType STLINE ARCLINE POLYLINE NURBLINE"
#     "Surfaces SurfaceType PLSURFACE COONSURFACE NURBSURFACE CONTACTSURFACE MESHSURFACE"
#     "Volumes VolumeType VOLUME CONTACTVOLUME"
# select_mode can be "single" or "multiple"
# for AllTypes selection returns some as
# Points 19 23 Lines 28:29 32:33 37 Surfaces 1 6 21:22 Volumes 1 Dimension 1:2
# or for mesh entities some as Nodes 236 244 Elements 4122 4130 4152 4165
#                              Faces 35,1 35,3 ... (element_id,face_index)
proc GidUtils::PickEntities { type select_mode {message {}}} {
    if { $message != "" } {
        GidUtils::SetWarnLine $message
    }
    set word_select_mode NoJustOne
    switch $select_mode {
        single {
            set word_select_mode JustOne
        }
        multiple {
            set word_select_mode NoJustOne
        }
        default {
            error "GidUtils::PickEntities. unexpected select_mode=$select_mode"
        }
    }
    set old_disabled [GiD_Project set disable_graphinterp]
    if { $old_disabled } {
        GiD_Project set disable_graphinterp 0
    }
    set old_disable_writebatch [GiD_Project set disable_writebatch]
    if { $old_disable_writebatch != 1 } {
        #try to hide the use of process by this proc to batch and Undo
        GiD_Project set disable_writebatch 1
    }
    GiD_Process 'SelectEntities $word_select_mode {*}$type
    vwait ::GidPriv(selection)
    if { $old_disable_writebatch != 1 } {
        GiD_Project set disable_writebatch $old_disable_writebatch
    }
    if { $old_disabled } {
        GiD_Project set disable_graphinterp $old_disabled
    }
    if { [info exists ::GidPriv(selection)] && [llength ::GidPriv(selection)] } {
        if { $select_mode == "single" } {
            set selection [lindex {*}$::GidPriv(selection) 1]
        } else {
            set selection [lrange {*}$::GidPriv(selection) 1 end]
        }
    } else {
        set selection ""
    }
    return $selection
}

# togl tk widget togl
# type can be "Points" "Lines" "Surfaces" "Volumes" "Dimensions" "AllTypes" "Nodes" "Elements" "Faces" or "Graphs"
# select_mode can be "single" or "multiple"
# x y : integer screen values of upper-left corner (upper instead lower because  y are mouse coordinates and start on top and go down, not from bottom to top as usual)
# width height: positive integer screen values of box size
proc GidUtils::PickRectangle {togl type select_mode x y width height} {
    GidUtils::Disable warnline
    set word_select_mode NoJustOne
    switch $select_mode {
        single {
            set word_select_mode JustOne
        }
        multiple {
            set word_select_mode NoJustOne
        }
        default {
            error " GidUtils::PickRectangle. unexpected select_mode=$select_mode"
        }
    }
    GiD_Process 'SelectEntities $word_select_mode {*}$type
    $togl button 1 $x $y
    $togl button 1 [expr $x+$width] [expr $y+$height]
    after 0 GiD_Process escape
    vwait ::GidPriv(selection)
    if { $type == "AllTypes" } {
        set entities [lrange [lindex $::GidPriv(selection) 0] 1 end]
    } else {
        set entities [lindex $::GidPriv(selection) 0]
    }
    unset ::GidPriv(selection)
    GidUtils::Enable warnline
    return $entities
}

#selecting only in a small square close to x y
proc GidUtils::Pick {togl type select_mode x y} {
    return [GidUtils::PickRectangle $togl $type $select_mode [expr $x-5] [expr $y-5] 10 10]
}



# returns the coordinates x y z Or null selecting from the screen
# GidPriv(selcoord,id) store the point id (-1 nothing selected, 0 new point, >0 point id)
# to get the number of a existent point, can also use PickEntities
# PickMode can be "FJoin" (default)    : to pick existing points
#                 "FNoJoin"            : to pick a new point generic
#                 (FJoin and FNoJoin are to do the same as Join NoJoin without complain if it is the current mode)
#                 "PointInLine"       : to pick a new point on a curve
#                 "PointInSurface"    : to pick a new point on a curve
#                 "TangentInLine"     : to pick a new unitary curve tangent vector
#                 "NormalInSurface"   : to pick a new unitary surface normal vector
#                 "ArcCenter"         : to pick a new point center of an arc
#                 "LineParameter"     : to pick a new point in a line, specifying the t parameter
#                 "SurfaceParameters" : to pick a new point in a surface, specifying the u,v parameters
#                 "MapPointSurface"   : to pick a new point in a surface, specifying a near initial point
#                 "PointInElement"    : to pick a new point on a mesh element
#                 "NormalInElement"    : to pick a new point on a mesh element
#use can be GEOMETRYUSE or MESHUSE (e.g. to force pick nodes instead geometric points)
#
#Note: just after the selection, for PointInElement or NormalInElement the element number is stored
#      in the variable $::GidPriv(selection)

proc GidUtils::GetCoordinates { { WarnLineMsg "" } { PickMode  "FJoin" } { use "" } } {
    if { $WarnLineMsg  == "" } { set WarnLineMsg [_ "Enter point (ESC to leave)"] }
    GidUtils::SetWarnLine $WarnLineMsg
    if { $use == "" } {
        set use [GiD_Info Project ViewMode]
    }
    if { $PickMode != "Join" && $PickMode != "FJoin" && $PickMode != "NoJoin" && $PickMode != "FNoJoin" } {
        #Join and NoJoin could be for both, geometry or mesh
        if { $PickMode == "PointInElement" || $PickMode == "NormalInElement" } {
            set use MESHUSE
        } else {
            set use GEOMETRYUSE
        }
    }
    SetUse $use
    set old_disabled [GiD_Project set disable_graphinterp]
    set old_focus_raise [focus]
    if { $old_focus_raise != "" } {
        set old_focus_raise [winfo toplevel $old_focus_raise]
    }
    if { $old_disabled } {
        GiD_Project set disable_graphinterp 0
    }
    set old_disable_writebatch [GiD_Project set disable_writebatch]
    if { $old_disable_writebatch != 1 } {
        #try to hide the use of process by this proc to batch and Undo
        GiD_Project set disable_writebatch 1
    }
    #GiD_Process Mescape Utilities Id Silent $PickMode
    GiD_Process 'GetPointCoord Silent $PickMode
    update idletasks
    vwait ::GidPriv(selcoord,id)
    GiD_Process escape
    if { $old_disable_writebatch != 1 } {
        GiD_Project set disable_writebatch $old_disable_writebatch
    }
    if { $old_disabled } {
        GiD_Project set disable_graphinterp $old_disabled
    }
    # if { $old_focus_raise != "" } { raise $old_focus_raise}
    if { $::GidPriv(selcoord,id) == -1 } {
        set xyz [list]
    } else {
        set xyz [list $::GidPriv(selcoord,x) $::GidPriv(selcoord,y) $::GidPriv(selcoord,z)]
    }
    return $xyz
}

proc GidUtils::GetGiDVersionMain { } {
    return [lindex [split [GiD_Info GiDVersion] .] 0]
}

#to compare if a GiD version is more recent of another
proc GidUtils::VersionNumber { VersionString } {
    set pattern {([0-9]+)[.]([0-9]+)[.]?([0-9]*)([a-z]?)(?:-beta([0-9]+))?}
    regexp -nocase $pattern $VersionString total v1 v2 v3 letter betanumber
    set res 0
    if { [info exists v1] } {
        if { $v1 != "" } { set res [expr $res+$v1*100000] }
    }
    if { [info exists v2] } {
        if { $v2 != "" } { set res [expr $res+$v2*1000] }
    }
    if { [info exists v3] } {
        if { $v3 != "" } { set res [expr $res+$v3] }
    }
    return $res
}

proc GidUtils::VersionSubchar { VersionString } {
    set pattern {([0-9]+)[.]([0-9]+)[.]?([0-9]*)([a-z]?)(?:-beta([0-9]+))?}
    regexp -nocase $pattern $VersionString total v1 v2 v3 letter betanumber
    set res ""
    if { [info exists letter] } { set res $letter }
    return $res
}

proc GidUtils::VersionBetaNumber { VersionString } {
    set pattern {([0-9]+)[.]([0-9]+)[.]?([0-9]*)([a-z]?)(?:-beta([0-9]+))?}
    regexp -nocase $pattern $VersionString total v1 v2 v3 letter betanumber
    set res ""
    if { [info exists betanumber] } { set res $betanumber }
    return $res
}

proc GidUtils::IsDeveloperVersion { } {
    set letter [GidUtils::VersionSubchar [GiD_Info GiDVersion]]
    if { $letter == "d" || $letter == "b" } {
        set is_developer 1
    } else {
        set is_developer 0
    }
    return $is_developer
}

proc GidUtils::IsImportantReleaseVersion { } {
    if { [string range [GiD_Info GiDVersion] end-2 end] == "-IR" } {
        set is_important_release 1
    } else {
        set is_important_release 0
    }
    return $is_important_release
}

# -1 - no OpenGL
#  0 - hardware accelerated OpenGL
#  1 - software OpenGL
proc GidUtils::OpenGLSoftwareRenderer { } {
    set err [catch { set renderer [GiD_OpenGL draw -getstring renderer] }]
    if { $err} {
        return -1
    }
    set software_renderer 0
    # Windows: safe mode or remote desktop
    if { $renderer == "GDI Generic"} {
        set software_renderer 1
    } else {
        # gid with offscreen has softpipe or OffScreen as renderer
        # virtual vmware has llvmpipe as ( cpu) renderer
        # - considering this as software renderer, despite using all cpu cores...
        foreach word [list softpipe OffScreen llvmpipe] {
            if { [lsearch $renderer $word] != -1} {
                set software_renderer 1
                break
            }
        }
    }
    return $software_renderer
}

proc GidUtils::OpenGLHasVBO { } {
    set err [catch { set extensions [GiD_OpenGL draw -getstring extensions] }]
    if { $err} {
        return 0
    }
    set hasVBO 0
    if { [lsearch $extensions "GL_ARB_vertex_buffer_object"] != -1} {
        set hasVBO 1
    }
    return $hasVBO
}

# return -1 0 or 1 if Version1 is less than, equal to, or greater than Version1
proc GidUtils::TwoVersionsCmp { Version1  Version2 } {
    set Number1 [GidUtils::VersionNumber $Version1]
    set Number2 [GidUtils::VersionNumber $Version2]
    if { $Number1 < $Number2 } { return -1 }
    if { $Number1 > $Number2 } { return 1 }
    set c1 [GidUtils::VersionSubchar $Version1]
    set c2 [GidUtils::VersionSubchar $Version2]
    if { $c1 < $c2 } { return -1 }
    if { $c1 > $c2 } { return 1 }
    #assumed more recent the non-beta version
    set BetaNumber1 [GidUtils::VersionBetaNumber $Version1]
    set BetaNumber2 [GidUtils::VersionBetaNumber $Version2]
    if { $BetaNumber1 != "" } {
        if { $BetaNumber2 != "" } {
            if { $BetaNumber1 < $BetaNumber2 } { return -1 }
            if { $BetaNumber1 > $BetaNumber2 } { return 1 }
        } else {
            return -1
        }
    } else {
        if { $BetaNumber2 != "" } { return 1 }
    }
    return 0
}

# this function always returns -1 if GiD is before to the
# 6.2.0 version, first including info GiDVersion
# return -1 if GidPriv(Version) < Version
# return  0 if GidPriv(Version) == Version
# return  1 if GidPriv(Version) > Version

#sample use for a probletype that need GiD7.4b or later new features:
#    set VersionRequired "7.4b"
#    set comp -1
#    catch { set comp [::GidUtils::VersionCmp $VersionRequired] }
#    if { $comp < 0 } {
#        error "This interface requires GiD $VersionRequired or later."
#    }

proc GidUtils::VersionCmp { Version } {
    if { [catch { set GidVersion [GiD_Info GiDVersion] }] }  {
        return -1
    }
    if { $GidVersion == "Unknown" } {
        return -1
    }
    return [GidUtils::TwoVersionsCmp $GidVersion $Version]
}

#check if current version satisfy the minimum required version
# Usually a GiD >= to a MinimumOfficialVersion will have all its features, but there are exceptions, 
# e.g. maybe a new GiD-Tcl command is added also to next official, but is not available in previous DEV
# to be sure must specify min version of both branchs: official and DEV(master)/IR
# if { [GidUtils::VersionSatisfy $MinimumOfficialVersion $MinimumDeveloperVersion] } ...
# it is not enough a single comparison if { [GidUtils::VersionCmp $MinimumVersion] >=0 }
proc GidUtils::VersionSatisfy { MinimumOfficialVersion MinimumDeveloperVersion } {
    if { [GidUtils::IsImportantReleaseVersion] || [GidUtils::IsDeveloperVersion] } {
        set cmp [GidUtils::VersionCmp $MinimumDeveloperVersion]
    } else {
        set cmp [GidUtils::VersionCmp $MinimumOfficialVersion]
    }
    return [expr {$cmp>=0}]
}

#to access to values by fieldname instead index (more easy to maintain code)
proc GidUtils::GetProblemDataValue { fieldname } {
    set foundname ""
    set foundvalue ""
    foreach "name value" [lrange [GiD_Info gendata] 1 end] {
        if { [string equal $fieldname $name] } { return $value }
        if { [string match $fieldname* $name] } {
            if { $foundname != "" } {
                error "error: '$fieldname' has more than one names: $foundname and $name"
            }
            set foundname $name
            set foundvalue $value
        }
    }
    if { $foundname == "" } {
        error "error: '$fieldname' is not found in Problem data"
    }
    return $foundvalue
}

#to access to values by fieldname instead index (more easy to maintain code)
proc GidUtils::GetCurrentIntervalDataValue { fieldname } {
    set foundname ""
    set foundvalue ""
    foreach "name value" [lrange [GiD_Info intvdata] 1 end] {
        if { [string equal $fieldname $name] } { return $value }
        if { [string match $fieldname* $name] } {
            if { $foundname != "" } {
                error "error: '$fieldname' has more than one names: $foundname and $name"
            }
            set foundname $name
            set foundvalue $value
        }
    }
    if { $foundname == "" } {
        error "error: '$fieldname' is not found in Problem data"
    }
    return $foundvalue
}

proc GidUtils::GetConditionQuestionIndex { condition_name condition_question } {
    set found_name ""
    set found_index -1
    set index 0
    foreach {name value} [lrange [GiD_Info conditions $condition_name] 2 end] {
        if { [string equal $condition_question $name] } {
            return $index
        }
        if { [string match $condition_question* $name] } {
            if { $found_name != "" } {
                error "error: '$condition_question' has more than one names: $found_name and $name"
            }
            set found_name $name
            set found_index $index
        }
        incr index
    }
    if { $found_name == "" } {
        error "error: '$condition_question' is not found in condition $condition_name"
    }
    return $found_index
}

proc GidUtils::GetConditionDefaultValues { condition_name } {
    set values [list]
    foreach {question value} [lrange [GiD_Info conditions $condition_name] 2 end] {
        lappend values $value
    }
    return $values
}

#change status of redraws, etc to speed or enhance scripts

#after use a pair: GidUtils::DisableGraphics - GidUtils::EnableGraphics
#the previous stated is automatically restored (using a stack)
proc GidUtils::DisableGraphics { } {
    if { [GidUtils_IsTkDisabled] } {
        #e.g. batch mode without windows
        return
    }
    foreach i [list windows graphinput graphics warnline] {
        GidUtils::Disable $i
    }
}

proc GidUtils::EnableGraphics { } {
    if { [GidUtils_IsTkDisabled] } {
        #e.g. batch mode without windows
        return
    }
    foreach i [list windows graphinput graphics warnline] {
        GidUtils::Enable $i
    }
    return
}

proc GidUtils::Push { i value } {
    variable stack
    lappend stack($i) $value
}

proc GidUtils::Pop { i } {
    variable stack
    if { [info exists stack($i)] && [llength $stack($i)] } {
        set value [lindex $stack($i) end]
        set stack($i) [lrange $stack($i) 0 end-1]
    } else {
        set value ""
    }
    return $value
}

# i must be: windows graphinput graphics warnline writebatch
proc GidUtils::Disable { i } {
    if { [lsearch {windows graphinput graphics warnline writebatch} $i] == -1 } {
        return 1
    }
    GidUtils::Push $i [GiD_Project set disable_$i]
    GiD_Project set disable_$i 1
    return 0
}

# i must be: windows graphinput graphics warnline writebatch
proc GidUtils::Enable { i } {
    if { [lsearch {windows graphinput graphics warnline writebatch} $i] == -1 } {
        return 1
    }
    set value [GidUtils::Pop $i]
    if { $value != "" } {
        GiD_Project set disable_$i $value
    } else {
        GiD_Project set disable_$i 0
    }
    return 0
}

proc GidUtils::DisableWarnLine {  } {
    return [GidUtils::Disable warnline]
}

proc GidUtils::EnableWarnLine {  } {
    return [GidUtils::Enable warnline]
}

proc GidUtils::WaitState { {w .gid.central.s} } {
    if { [GidUtils_IsTkDisabled] } {
        return
    }
    if { [winfo exists $w] } {
        GidUtils::Push cursor [$w cget -cursor]
        set cursor2set watch
        if { [info exists ::gid_cursor(WATCH)]} {
            # defined in gid_client_initialize_cursors (gid_client.tcl) adapted to platform and theme
            set cursor2set $::gid_cursor(WATCH)
        }
        $w configure -cursor $cursor2set
        update
    }
}

proc GidUtils::EndWaitState { {w .gid.central.s} } {
    if { [GidUtils_IsTkDisabled] } {
        return
    }
    if { [winfo exists $w] } {
        set gid_cursor [GidUtils::Pop cursor]
        $w configure -cursor $gid_cursor
        update
    }
}

proc GidUtils::SetWarnLine { text } {
    if { [GidUtils_IsTkDisabled] || [GiD_Project set disable_warnline] } {
        return
    }
    ChangeWarnLineVar $text
}

#Utility to show a progress bar.
#varname must be a global variable name with the current value to be showed automatically
#stopproc is an optional callback procedure name to be called when the user press stop
#when the current value is equal to max the advance bar is automatically deleted
#Note: only is possible to use one progress bar
#example:
#  global MyVarName
#  set MyVarName 0
#  ::GidUtils::CreateAdvanceBar [= "Loading file"] [= "Percentage"] ::MyVarName 100  [list WarnWin [= "User stop"]]
#  # ...
#  set MyVarName 20

proc GidUtils::CreateAdvanceBar { title text varname { max 100 } { stopproc "" } } {
    if { [GiD_Project set disable_progressbar] } {
        return
    }
    variable ab_maximum
    variable ab_varname
    variable ab_stopproc
    set ab_maximum $max
    set ab_varname $varname
    set ab_stopproc $stopproc
    set ::GidPriv(StopLoading) 0
    AdvanceBar 0 $ab_maximum 0 $title $text
    if { $ab_varname != "" } {
        trace add variable $ab_varname write "::GidUtils::UpdateAdvanceBar ;#"
    }
    if { $ab_stopproc != "" } {
        trace add variable ::GidPriv(StopLoading) write "::GidUtils::CallStopProc ;#"
    }
}

proc GidUtils::UpdateAdvanceBar { } {
    variable ab_maximum
    variable ab_varname
    if { ![info exists $ab_varname] } return
    if { [set $ab_varname] >= $ab_maximum } {
        ::GidUtils::DeleteAdvanceBar
    } else {
        AdvanceBar [set $ab_varname] $ab_maximum [set $ab_varname]
    }
}

proc GidUtils::CallStopProc { } {
    variable ab_stopproc
    if { $::GidPriv(StopLoading) == "0" } return
    ::GidUtils::DeleteAdvanceBar
    if { $ab_stopproc != "" } {
        eval $ab_stopproc
    }
}

# proc GidUtils::UpdateAdvanceBar { current } {
#     variable ab_maximum
#     AdvanceBar $current $ab_maximum $current
# }

proc GidUtils::DeleteAdvanceBar { } {
    if { [GidUtils::IsTkDisabled] } {
        return
    }
    variable ab_maximum
    variable ab_varname
    variable ab_stopproc
    if { $ab_varname != "" } {
        trace remove variable $ab_varname write "::GidUtils::UpdateAdvanceBar ;#"
    }
    if { $ab_stopproc != "" } {
        trace remove variable ::GidPriv(StopLoading) write "::GidUtils::CallStopProc ;#"
    }
    AdvanceBar $ab_maximum $ab_maximum $ab_maximum
}

# to show a splash image, for example:
# global GIDDEFAULT
# GidUtils::Splash [file join [gid_filesystem::get_folder_standard resources] images/GiDsplash.png] .splash 0
# VersionAndPos: list with three items: version_label posx posy
proc GidUtils::Splash { imagefile {w .splash} {selfclose 1} {VersionAndPos ""} {milliseconds_to_close 2000}} {
    if { [GidUtils::AreWindowsDisabled] } {
        return
    }
    #if selfclose==0 ignore the SplashWindow GiD variable and show the window
    #tipically selfclose==1 for splash initial windows (then use SplashWindow)
    #          selfclose==0 for help about windows (then show avoiding the SplashWindow state)
    #           milliseconds_to_close: milliseconds to auto close the splash window if selfclose == 1
    if { ![GiD_Set SplashWindow] && $selfclose==1 } {
        return
    }

    if { ![file exists $imagefile] } {
        W [_ "File '%s' does not exist" $imagefile]
        return
    }
    if { [winfo exists $w]} {
        destroy $w
        update
    }
    toplevel $w
    if { $::tcl_platform(platform) == "windows" } {
        wm attributes $w -toolwindow 1
    }
    set im [image create photo -file $imagefile]
    set x [expr [winfo screenwidth $w]/2-[image width $im]/2]
    set y [expr [winfo screenheight $w]/2-[image height $im]/2]
    wm geometry $w +$x+$y
    wm transient $w .gid
    wm overrideredirect $w 1
    pack [label $w.l -image $im -relief ridge -borderwidth 2]

    if { $VersionAndPos != "" } {
        if { [llength $VersionAndPos] == 3 } {
            set VersionNumber [lindex $VersionAndPos 0]
            set x [lindex $VersionAndPos 1]
            set y [lindex $VersionAndPos 2]
        } else {
            set VersionNumber [lindex $VersionAndPos 0]
            set x 62
            set y 62
        }
        set fnt "Sans 10"
        set fnt_small "Sans 8"
        if { $::tcl_platform(platform) == "windows" } {
            set fnt "verdana 10"
            set fnt_small "verdana 8"
        }
        label .splash.lv -text $VersionNumber -font $fnt \
            -background white -foreground black \
            -relief solid -borderwidth 1 -padx 12 -pady 3
        set label_width_and_place [expr $x + [winfo reqwidth .splash.lv]]
        set image_width [expr [image width $im] + 4]
        set toplevel_width [expr [winfo reqwidth $w] - 4]
        set max_width $toplevel_width
        if { $image_width > $max_width} {
            set max_width $image_width
        }
        if { $max_width - $label_width_and_place < 4} {
            # not centered or outside window
            .splash.lv configure -font $fnt_small
        }
        place .splash.lv -x $x -y $y
    }

    bind $w <1> "destroy $w"
    bind $w <KeyPress> "destroy $w"
    raise $w .gid
    grab $w
    focus $w
    update
    if { $selfclose } {
        after $milliseconds_to_close "if { [winfo exists $w] } { destroy $w }"
    }
}

proc GidUtils::GetCurrentLanguage { } {
    GiD_Set Language
}

proc GidUtils::SetCurrentLanguage { newlan {updatewidgets 1 } } {
    GiD_Set Language $newlan
}

proc GidUtils::GetAllLanguages { } {
    MsgcatGetLocales
}

#if an id appear twice in the list it like it was not selected
#e.g. "4 5 5 6 7 7 7 8" --> "4 6 7 8"
proc GidUtils::RemoveRepeatedTwiceNumberList { a } {
    set ret [list]
    set iprev ""
    foreach i [lsort -integer $a] {
        if {$i != $iprev} {
            set times 1
            lappend ret $i
            set iprev $i
        } else {
            set ret [lreplace $ret end end]
            set iprev ""
        }
    }
    return $ret
}

#GiD selection accept and returns entities idetifiers (integers > 0) as
#a "compacted" list, the sintax "a:a+n" is interpreted as "a a+1 a+2 ... a+n"

#instead "4 8 9 10 11 15" compact list --> "4 8:11 15"
#items must be positive integer numbers (can be unordered)
proc GidUtils::CompactNumberList { a } {
    set ret ""
    set istart 0
    if { [string match "*:*" $a] } {
        #imput list is compacted
        return $a
    }
    foreach i [GidUtils::RemoveRepeatedTwiceNumberList [lsort -integer $a]] {
        if { !$istart} {
            set istart $i
        } else {
            if {$i != [expr $iprev+1]} {
                if { $istart != $iprev } {
                    if { [expr $istart +1] == $iprev } {
                        append ret "$istart $iprev "
                    } else {
                        append ret "$istart:$iprev "
                    }
                } else {
                    append ret "$istart "
                }
                set istart $i
            }
        }
        set iprev $i
    }
    if { $istart } {
        if { $istart != $iprev } {
            if { [expr $istart +1] == $iprev } {
                append ret "$istart $iprev"
            } else {
                append ret "$istart:$iprev"
            }
        } else {
            append ret "$istart"
        }
    }
    return $ret
}

#instead "4 8:11 15"  uncompact list --> "4 8 9 10 11 15"
#items must be positive increasing integer numbers
#returns a objarray of integers
proc GidUtils::UnCompactNumberList { a } {
    set n [llength $a]
    set obj [objarray new intarray $n]
    set index 0
    foreach item $a {
        set item [string map {: { }} $item]
        if { [llength $item] == 2 } {
            lassign $item from to
            set obj2 [objarray new_from_to intarray $from $to]
            objarray replace $obj $index $index $obj2
            incr index [objarray length $obj2]
        } else {
            objarray set $obj $index $item
            incr index
        }
    }
    return $obj
}

#returns the count of uncompacted integers
proc GidUtils::CountUnCompactNumberList { a } {
    set count 0
    foreach item $a {
        set item [string map {: { }} $item]
        if { [llength $item] == 2 } {
            lassign $item from to
            incr count [expr {$to-$from+1}]
        } else {
            incr count
        }
    }
    return $count
}


#items must be face ids with syntax element_id,face_id
#returns a list of two objarray of integers, one for element ids and other for face ids
#e.g. faces  "4,1 25,3 8,1"  --> {"4 25 8" "1 3 1"}
proc GidUtils::SplitFaceListInTwoObjArrays { a } {
    set n [llength $a]
    set obj_element_ids [objarray new intarray $n]
    set obj_face_ids [objarray new chararray $n]
    set index 0
    foreach item $a {
        set item [string map {, { }} $item]
        if { [llength $item] == 2 } {
            lassign $item element_id face_id
            objarray set $obj_element_ids $index $element_id
            objarray set $obj_face_ids $index $face_id
        } else {
            error "expected face syntax element_id,face_id and item=$item"
        }
        incr index
    }
    return [list $obj_element_ids $obj_face_ids]
}

#procedure of GiD similar to split but separator can be a multicharacter, like //
#used for tree results, groups and layernames
proc GidUtils::Split { x separator } {
    set splitchar \uFFFF ;#forbidden unicode character, x must never contain it
    return [split [string map "$separator $splitchar" $x] $splitchar]
}

#return the lengh of the longest string
proc GidUtils::MaxStringLength { args } {
    set maxlen 0
    foreach msg $args {
        set length [string length $msg]
        if {$length > $maxlen} {
            set maxlen $length
        }
    }
    return $maxlen
}

# get the integer with (for button) maximum string lengh of a collection of strings,
# and in case of Chinese -width (by Tk error?) create a button smaller than needed to show the whole text, then set $width*2
proc GidUtils::GetButtonWidthFromStrings { args } {
    set width [GidUtils::MaxStringLength $args]
    if { [GiD_Set Language] == "zh" } {
        #in case of Chinese setting -width create a button smaller than needed to show the whole text
        set width [expr $width*2]
    }
    return $width
}

proc GidUtils::GetWindowNames { } {
    set names [list ADVANCEDVIEWSETTINGS ANIMATE ANIMATIONCONTROLS ANIMATIONSCRIPT AXISREFERENCE \
        CALCULATE CALCULATOR CLIPPLANES COMMENTS CONDITION CONSOLE COORDINATES COPY CUSTOMLIB \
        ENVIRONMENTREFLECTION HELP GRAPHS GRAPHSCREATE GROUPS INFO INTERVALDATA LAYER LIST \
        MACROS MATERIAL MESHCHORDALERROR MESHERRORS MESHGENERATION MESHPROGRESS MESHQUALITY MOVE \
        NOTES NURBSLINEEDIT NURBSSURFACEEDIT \
        PAGESETUP PARAMETRICLINE PARAMETRICSURFACE PERSPECTIVE POSTCONTOURLIMITS POSTCOPY PREFERENCES PROBLEMDATA PROCESSINFO \
        READBATCH REDO RENDERCUSTOM RESULTS RESULTSCREATE RESULTSCREATESTATISTICAL RESULTSDISTRIBUTION RESULTSMULTIPLE RESULTSRANGES \
        SELECTION SETS STATUS TOOLBAR_STATUSINFO TOOLBARS UNDO WARNING]
    return $names
}

proc GidUtils::ExistsWindow { window } {
    set exists 0
    if { [GidUtils_IsTkDisabled] } {
        return 0
    }
    #GROUPS for window outside
    set keyProg {
        ADVANCEDVIEWSETTINGS .gid.wStereoW
        ANIMATE .gid.wPostAnimate
        ANIMATIONCONTROLS .gid.wPostAnimateControls
        ANIMATIONSCRIPT .gid.wAutoAnimate
        AXISREFERENCE .gid.axis
        CALCULATE .gid.wrun
        CALCULATOR .gid.wcalc
        CLIPPLANES .gid.wClipPlanes
        COMMENTS .gid.wCommentsWW
        CONDITION .gid.conditions1
        CONSOLE .tkcon
        COORDINATES .gid.wSetCoord
        COPY .gid.wcopymove
        CUSTOMLIB .gid.central.boundaryconds
        ENVIRONMENTREFLECTION .gid.wEnvironmentReflection
        HELP .help
        GRAPHS .gid.graphs
        GRAPHSCREATE .gid.wPostCreateGraphs
        GROUPS .gid.central.wgroup
        INFO .gid.info_gid
        INTERVALDATA .gid.intvdata1
        LAYER .gid.central.wlay
        LIST .gid.listent
        MACROS .gid.tmacros
        MATERIAL .gid.materials1
        MESHCHORDALERROR .gid.chordalerror
        MESHERRORS .gid.mesherrors
        MESHGENERATION .gid.meshgeneration        
        MESHPROGRESS .gid.wprogress
        MESHQUALITY .gid.meshq
        MOVE .gid.wcopymove
        NOTES .gid.texteditor
        NURBSLINEEDIT .gid.nurbsurfmove
        NURBSSURFACEEDIT .gid.nurbsurfmove
        PAGESETUP .gid.wPageSetup
        PARAMETRICLINE .gid.parametricline
        PARAMETRICSURFACE .gid.parametricsurface
        PERSPECTIVE .gid.wPerspectiveW
        POSTCONTOURLIMITS .gid.wPostContLim
        POSTCOPY .gid.wtransformation
        PREFERENCES .gid.gid_preferences
        PROBLEMDATA .gid.prbdata1
        PROCESSINFO .gid.outputview1
        READBATCH .gid.wReadBatch
        REDO .gid.redo
        RENDERCUSTOM .gid.customizerender
        RESULTS .gid.wPostGeom
        RESULTSCREATE .gid.wCreateResult
        RESULTSCREATESTATISTICAL .gid.wStatisticalScalarResult
        RESULTSDISTRIBUTION .gid.resq
        RESULTSMULTIPLE .gid.wMultipleResults
        RESULTSRANGES .gid.postrangestable
        SELECTION .gid.filter_win
        SETS .gid.wPostDisplay
        STATUS .gid.listent
        TOOLBAR_STATUSINFO .gid.bottominfo
        TOOLBARS .gid.toolbars
        UNDO .gid.undo
        WARNING .gid.warning
    }
    if { [dict exists $keyProg $window] } {
        set w [dict get $keyProg $window]
        set exists [winfo exists $w]
    } else {
        error [_ "Unexpected window %s" $window]
    }
    return $exists
}

# function to close some window if it exists, window can be somoe or [GidUtils::GetWindowNames]
# or ALL (to close all window except the toolbars with prefix TOOLBAR_)
# for toolbars see also GidUtils::DisableToolbar
proc GidUtils::CloseWindow { window } {
    set key_closed [list]
    if { [GidUtils_IsTkDisabled] } {
        return $key_closed
    }
    #GROUPS for window outside
    set keyProg {
        ADVANCEDVIEWSETTINGS .gid.wStereoW
        ANIMATE .gid.wPostAnimate
        ANIMATIONCONTROLS .gid.wPostAnimateControls
        ANIMATIONSCRIPT .gid.wAutoAnimate
        AXISREFERENCE .gid.axis
        CALCULATE .gid.wrun
        CALCULATOR .gid.wcalc
        CLIPPLANES .gid.wClipPlanes
        COMMENTS .gid.wCommentsWW
        CONDITION .gid.conditions1
        CONSOLE .tkcon
        COORDINATES .gid.wSetCoord
        COPY .gid.wcopymove
        CUSTOMLIB .gid.central.boundaryconds
        ENVIRONMENTREFLECTION .gid.wEnvironmentReflection
        HELP .help
        GRAPHS .gid.graphs
        GRAPHSCREATE .gid.wPostCreateGraphs
        GROUPS .gid.central.wgroup
        INFO .gid.info_gid
        INTERVALDATA .gid.intvdata1
        LAYER .gid.central.wlay
        LIST .gid.listent
        MACROS .gid.tmacros
        MATERIAL .gid.materials1
        MESHCHORDALERROR .gid.chordalerror
        MESHERRORS .gid.mesherrors
        MESHGENERATION .gid.meshgeneration
        MESHPROGRESS .gid.wprogress
        MESHQUALITY .gid.meshq
        MOVE .gid.wcopymove
        NOTES .gid.texteditor
        NURBSLINEEDIT .gid.nurbsurfmove
        NURBSSURFACEEDIT .gid.nurbsurfmove
        PAGESETUP .gid.wPageSetup
        PARAMETRICLINE .gid.parametricline
        PARAMETRICSURFACE .gid.parametricsurface
        PERSPECTIVE .gid.wPerspectiveW
        POSTCONTOURLIMITS .gid.wPostContLim
        POSTCOPY .gid.wtransformation
        PREFERENCES .gid.gid_preferences
        PROBLEMDATA .gid.prbdata1
        PROCESSINFO .gid.outputview1
        READBATCH .gid.wReadBatch
        REDO .gid.redo
        RENDERCUSTOM .gid.customizerender
        RESULTS .gid.wPostGeom
        RESULTSCREATE .gid.wCreateResult
        RESULTSCREATESTATISTICAL .gid.wStatisticalScalarResult
        RESULTSDISTRIBUTION .gid.resq
        RESULTSMULTIPLE .gid.wMultipleResults
        RESULTSRANGES .gid.postrangestable
        SELECTION .gid.filter_win
        SETS .gid.wPostDisplay
        STATUS .gid.listent
        TOOLBAR_STATUSINFO .gid.bottominfo
        TOOLBARS .gid.toolbars
        UNDO .gid.undo
        WARNING .gid.warning
    }
    if {  $window == "ALL" || $window == "CUSTOMLIB" || $window == "LAYER" || $window == "SETS"} {
        if { $window == "ALL" } {
            set special_windows {CUSTOMLIB LAYER SETS}
        } else {
            set special_windows $window
        }
        foreach key $special_windows {
            set w [dict get $keyProg $key]
            if { [winfo exists $w] } {
                if { $key == "CUSTOMLIB" } {
                    if { [winfo exists .gid.central.boundaryconds] } {
                        gid_groups_conds::open_conditions_inside -update_preferences 0 .gid.central.boundaryconds
                    }
                } elseif { $key == "LAYER" } {
                    if { $::GidPriv(ShowGroupsTab) } {
                        #save the state of the current tab
                        set nb $w.body.nb
                        if { [winfo exists $nb] } {
                            set ::GidPriv(LayersOrGroupsCurrentTab) [$nb index [$nb select]]
                        }
                    }
                    if { [winfo class $w] == "Toplevel" } {
                        destroy $w
                    } else {
                        Layers::CloseWindowInside $w
                    }
                } elseif { $key == "SETS" } {
                    PostDisplayClose $w
                } else {
                    #unexpected case
                }
                lappend key_closed $key
            }
        }
        if { $window == "ALL" } {
            # do the all other windows
            # special case: toobars TOOLBAR_STATUSINFO -> "Status & Information"
            set toolbar_windows {TOOLBAR_STATUSINFO}
            foreach {key prog} $keyProg {
                if { $key != "LAYER" && [lsearch $toolbar_windows $key] == -1 } {
                    set w $prog
                    if { [winfo exists $w] } {
                        lappend key_closed $key
                        destroy $w
                    }
                }
            }
        }
    } elseif { [dict exists $keyProg $window] } {
        #$window != "LAYER" (previous if)
        set w [dict get $keyProg $window]
        if { [winfo exists $w] } {
            lappend key_closed $window
            destroy $w
        }
    } else {
        error [_ "Unexpected window %s" $window]
    }
    return $key_closed
}

#Added window option ALL

proc GidUtils::OpenWindow { window } {
    if { [GidUtils_IsTkDisabled] } {
        return 1
    }
    #WARNING some empty because are not interesting
    set keyProg [list \
        ADVANCEDVIEWSETTINGS { StereoW } \
        ANIMATE { PostAnimateWindow } \
        ANIMATIONCONTROLS { PostAnimateControlsWindow } \
        ANIMATIONSCRIPT { AutoAnimate::Create } \
        AXISREFERENCE { GlobalAxis::ChangeAxis } \
        CALCULATE { OpenProcWin } \
        CALCULATOR { CalculatorV } \
        CLIPPLANES { ClipPlanes } \
        COMMENTS { CommentsW } \
        CONDITION { DWCondWindow } \
        CONSOLE { OpenConsole } \
        CONSOLE_PYTHON { GiD_Python_Open_IDLE_Shell } \
        COORDINATES { SetCoord } \
        COPY { CopyMove Copy } \
        CUSTOMLIB { if { [GroupsXmlDocExists] } { gid_groups_conds::open_conditions menu -force_open 1 } } \
        ENVIRONMENTREFLECTION { EnvironmentReflectionWindow } \
        HELP { GiDHelpWindow } \
        LAYER { Layers::ChangeLayers } \
        GRAPHS { PostGraphs::Create } \
        GRAPHSCREATE { PostCreateGraphs } \
        GROUPS { Groups::OpenOutside } \
        INFO [list $::GidPriv(InfoGiDProc)] \
        INTERVALDATA { DWIntervalDataWindow } \
        LIST { ListEntities status } \
        MACROS { toolbarmacros::MacrosWindow } \
        MATERIAL { DWMatWindow } \
        MESHCHORDALERROR { CEBegin } \
        MESHERRORS { MeshErrors::CreateWindow } \
        MESHGENERATION { MeshGenerationWindowDefaultValues } \
        MESHQUALITY { MeshQuality } \
        MESHPROGRESS { } \
        MOVE { CopyMove Move } \
        NOTES { OpenNotes } \
        NURBSLINEEDIT { EditNURBS line } \
        NURBSSURFACEEDIT { EditNURBS surface } \
        PAGESETUP { PageSetup } \
        PARAMETRICLINE { ParametricLine } \
        PARAMETRICSURFACE { ParametricSurface } \
        PERSPECTIVE { PerspectiveW } \
        POSTCONTOURLIMITS { PostContLimitsWin } \
        POSTCOPY { PostTransformations } \
        PREFERENCES { PreferencesWindow } \
        PROBLEMDATA { DWProblemDataWindow } \
        PROCESSINFO { PWViewOutput } \
        READBATCH { ReadBatch } \
        REDO { Undo::RedoWindow } \
        RENDERCUSTOM { CustomizeRender } \
        RESULTS { PostGeom::Create } \
        RESULTSCREATE { PostCreateResult } \
        RESULTSCREATESTATISTICAL { PostStatisticalResult::StatisticalScalarResult } \
        RESULTSDISTRIBUTION { PostResultQuality } \
        RESULTSMULTIPLE { PostMultipleResults } \
        RESULTSRANGES { PostRangesTable::Create } \
        SELECTION { make_win_filter } \
        SETS { PostDisplay } \
        STATUS { ListEntities status } \
        TOOLBAR_STATUSINFO { BottomStatusFrame INSIDE } \
        TOOLBARS { TOBegin } \
        UNDO { Undo::UndoWindow } \
        WARNING { } ]
    if { [dict exists $keyProg $window] } {
        after idle [dict get $keyProg $window]
    } elseif { $window == "ALL" } {
        foreach {key prog} $keyProg {
            eval $prog
        }
    } else {
        error [_ "Unexpected window %s" $window]
    }
}

# to open and show the tab of LAYER and GROUPS windows
proc GidUtils::OpenTab { window } {
    if { [GidUtils_IsTkDisabled] } {
        return 1
    }
    #WARNING some empty because are not interesting
    set keyProg [list \
        LAYER { Groups::OpenTabIndex 0 } \
        GROUPS { Groups::OpenTabIndex 1 } ]
    if { [dict exists $keyProg $window] } {
        after idle [dict get $keyProg $window]
    } else {
        error [_ "Unexpected window %s" $window]
    }
}

proc GidUtils::ToggleWindow { window } {
    if { [GidUtils::ExistsWindow $window] } {
        GidUtils::CloseWindow $window
    } else {
        GidUtils::OpenWindow $window
    }
}

proc GidUtils::UpdateWindow { window } {
    if { [GidUtils::AreWindowsDisabled] } {
        return 1
    }
    set keyProg {
        COORDINATES { UpdateSetCoord }
        COPY { UpdateCopyMove }
        CUSTOMLIB { if { [GroupsXmlDocExists] } { gid_groups_conds::actualize_conditions_window } }
        DEFORMATION_VOLUME { DFRM::FillInfoDeformation }
        LAYER { Layers::FillInfo }
        GROUPS { Groups::FillInfo }
        MOVE { UpdateCopyMove }
        PERSPECTIVE { FillPerspectiveWInfo }
        PREFERENCES { CreateWidgetsFromXml::ChangeAllVariablesValues gid_preferences }
        SETS { FillPDInfo }
    }
    if { [dict exists $keyProg $window] } {
        set key $window
        if { ![info exists ::updating_window($key)] } {
            set ::updating_window($key) 1 ;#trick to protect from parallel accessing (events)
            eval [dict get $keyProg $key]
            unset ::updating_window($key)
        }
    } elseif { $window == "ALL" } {
        foreach {key prog} $keyProg {
            if { ![info exists ::updating_window($key)] } {
                set ::updating_window($key) 1 ;#trick to protect from parallel accessing (events)
                eval $prog
                unset ::updating_window($key)
            }
        }
    } else {
        #update with close and reopen (if exists)
        if { [GidUtils::ExistsWindow $window] } {
            GidUtils::CloseWindow $window
            GidUtils::OpenWindow $window
        }
        #error [_ "Unexpected window %s" $window]
    }
}


#return 0 if is not a link,  1 if the file is a link or  2 if it is Windows shell link
proc GidUtils::IsLink { lnk } {
    set islink 0
    global GidCache
    if { [info exists ::GidCache(IsLink,$lnk)] } {
        return $::GidCache(IsLink,$lnk)
    }

    if { ![file exists $lnk] } {
        return 0
    }
    if { [string match "windows" $::tcl_platform(platform)] } {
        set type [file type $lnk]
        if { $type == "file" && [string tolower [file extension $lnk]] == ".lnk" } {
            set fp [open $lnk]
            fconfigure $fp -encoding binary -translation binary -eofchar {}
            binary scan [read $fp 4] i tmp
            if { $tmp == "76" } {
                binary scan [read $fp 16] h32 tmp
                if { $tmp == "10412000000000000c00000000000064" } {
                    set islink 2 ;#windows shell link
                }
            }
            close $fp
        }
        if { !$islink } {
            if { ![catch  {set res [file readlink $lnk]}] } {
                set islink 1
            }
        }
    } else {
        if { ![catch  {set res [file readlink $lnk]}] } {
            set islink 1
        }
    }
    set ::GidCache(IsLink,$lnk) $islink
    return $islink
}

#return the name pointed by a link or Windows shell link
proc GidUtils::ReadLink { lnk show_ui } {
    set res ""
    set islink [GidUtils::IsLink $lnk]
    if { $islink == 1 } {
        catch {set res [file readlink $lnk]} err
    } elseif { $islink == 2 } {
        catch { set res [GiD_GetInfoLnkFile $lnk $show_ui] }
    }
    return $res
}

#return if is a gid geo, its version and the probletype used
proc GidUtils::GetInfoGeoFileData { data } {
    set is_geo 0
    set version 0
    set  problemtype ""
    set len 0
    set pos 0
    #assumed little endian
    if { [binary scan [string range $data 0 3] i len] == 1 } {
        incr pos 4
        set header [string range $data $pos [expr $pos+$len-2]]
        if { [string range $header 0 11] == "RAMSAN-gid-v" } {
            incr pos $len
            set version [string range $header 12 end]
            if { [binary scan [string range $data $pos [expr $pos+3]] i len] == 1 } {
                incr pos 4
                set problemtype [string range $data $pos [expr $pos+$len-2]]
                set is_geo 1
            }
        } else {
            #check if is ASCII version
            set lines [split $data \n]
            set header [lindex $lines 0]
            if { [string range $header 0 15] == "RAMSAN-ASCII-gid" } {
                set version [string range $header 18 end]
                set problemtype [lindex $lines 1] ;#line has problemtype and also an integer of is cuadratic
                set problemtype [string range $problemtype 0 [string last " " $problemtype]-1]
                set is_geo 1
            }
        }
    }
    set problemtype [file join $problemtype] ;#\ -> /
    return [list $is_geo $version $problemtype]
}

#close catching errors fp is the file handler to be closed
proc GidUtils::CatchClose { fp {filename ""} } {
    set fail 0
    if { [catch {close $fp} error_message options] } {
        set fail 1
        if { [lrange [dict get $options -errorcode] 0 1] == "POSIX ENOSPC" } {
            GiD_RaiseEvent GiD_Event_DiskFull $filename
        }
    }
    return $fail
}

proc GidUtils::CatchFileCopyForce { filename_src filename_dst } {
    set fail 0
    if { [catch {gid_filesystem::file copy -force $filename_src $filename_dst} error_message options] } {
        set fail 1
        set error_code [lrange [dict get $options -errorcode] 0 1]
        if { $error_code == "POSIX EINVAL" } {
            set is_cloud_path [GidDataManager::isCloudPath $filename_dst]
            if { $is_cloud_path } {
                #in case of nextcloud without disk space is returning {POSIX EINVAL {invalid argument}} !!
                set error_code "POSIX ENOSPC"
            }
        }
        if { $error_code == "POSIX ENOSPC" } {
            GiD_RaiseEvent GiD_Event_DiskFull $filename_dst
        }
    }
    return $fail
}

#trick to show in file browser the .gid folder with a .ico
proc GidUtils::SetFolderIconWindows { dirname ico_src infotip } {
    set fail 0
    if { [file exists $dirname] && [file isdirectory $dirname] && [file exists $ico_src]} {
        if { [file extension $ico_src] == ".exe" } {
            set iconfile $ico_src
        } else {
            set iconfile [file tail $ico_src]
            #copy locally the .ico
            set ico_dst [file join $dirname $iconfile]
            if { [file exists $ico_dst] } {
                file delete $ico_dst
            }
            set fail [GidUtils::CatchFileCopyForce $ico_src $ico_dst]
            if { !$fail } {
                file attributes $ico_dst -hidden 1
            }
        }
        set desktop_ini_template {
            [.ShellClassInfo]
            ConfirmFileOp=0
            NoSharing=0
            IconFile=$iconfile
            IconIndex=0
            InfoTip=$infotip
        }
        if { [catch { GidUtils::WriteFile [file join $dirname Desktop.ini] [subst -nobackslashes -nocommands $desktop_ini_template]} msg] } {
            set fail 1
        } else {
            file attributes $dirname -system 1
        }
    } else {
        set fail 1
    }
    return $fail
}

#in special case of a gid_project
proc GidUtils::SetFolderIconWindowsGiDProject { dirname } {
    set fail 0
    if { $::tcl_platform(platform) == "windows" } {
        set is_cloud_path [GidDataManager::isCloudPath $dirname]
        if { !$is_cloud_path } {
            set is_cloud_path [gid_cross_platform::is_google_drive_path $dirname]
        }
        if { $is_cloud_path } {
            # better do not try to create the file Desktop.ini and copy the icon
            # beacause become a problem to delete it (also from Windows Explorer)
        } else {
            if { 1 } {
                # do a permantent copy of the gid_project.ico inside the folder
                if { 1 } {
                    # find if there is a problemtype ico, otherwise use the gid_project.ico
                    set pt [file join [GiD_Info Project ProblemType]]
                    if { $pt == "UNKNOWN" || ![info exists ::GidPriv(problemtype,Icon,$pt)] } {
                        set ico_src [file join [gid_filesystem::get_folder_standard resources] images gid_project.ico]
                    } else {
                        set ico_src $::GidPriv(problemtype,Icon,$pt)
                    }
                } else {
                    # copy always the gid_project.ico file
                    set ico_src [file join [gid_filesystem::get_folder_standard resources] images gid_project.ico]
                }
            } else {
                #provide the path to gid.exe to get its internal ico
                set ico_src [GidUtils::GetFullPathGiDExe]
            }
            set info_tip [_ "This is a %s model folder" $::GidPriv(ProgName)]
            set fail [GidUtils::SetFolderIconWindows $dirname $ico_src $info_tip]
        }
    }
    return $fail
}

#read all file contents if exists
proc GidUtils::ReadFile { filename {encoding ""} {binary 0}} {
    set content ""
    if { [file exists $filename] && ![file isdirectory $filename]} {
        set mode r
        if { $binary } {
            append mode b
        }
        set fp ""
        if { [catch { set fp [open $filename $mode] } msg] } {
            set fp ""
        }
        if { $fp != "" } {
            if { $encoding != "" } {
                fconfigure $fp -encoding $encoding
            }
            set content [read -nonewline $fp]
            close $fp
        }
    }
    return $content
}

proc GidUtils::ReadFileLastLine { filename {encoding ""} {binary 0}} {
    set content ""
    if { [file exists $filename] && ![file isdirectory $filename]} {
        set mode r
        if { $binary } {
            append mode b
        }
        set fp ""
        if { [catch { set fp [open $filename $mode] } msg] } {
            set fp ""
        }
        if { $fp != "" } {
            if { $encoding != "" } {
                fconfigure $fp -encoding $encoding
            }
            seek $fp -1024 end
            set last_part [string trimright [read $fp 1024] \n]
            set content [lindex [split $last_part \n] end]
            close $fp
        }
    }
    return $content
}

#mode: w or a (append)
proc GidUtils::WriteFile { filename data {mode "w"} {encoding ""} {binary 0}} {
    set fail 0
    set fp [open $filename $mode]
    if { $fp != "" } {
        if { $encoding != "" } {
            fconfigure $fp -encoding $encoding
        }
        if { $binary } {
            fconfigure $fp -translation binary
        }
        puts -nonewline $fp $data
        #close $fp
        GidUtils::CatchClose $fp $filename
    } else {
        set fail 1
    }
    return $fail
}

#return the number of lines of a file
proc GidUtils::LineCount { filename {eofchar "\n"} } {
    set i 0
    set fid [open $filename]
    # Use a 512K buffer.
    fconfigure $fid -buffersize 524288 -translation binary
    while {![eof $fid]} {
        incr i [expr {[llength [split [read $fid 524288] $eofchar]]-1}]
    }
    close $fid
    return $i
}

#
proc GidUtils::GetRegistryKeyRecursive { key } {
    package require registry
    set data ""
    append data "\[$key]\n"
    set items [list]
    if { [catch { set items [registry values $key]} msg] } {
        append data "error:$msg"
        return $data
    }
    foreach item $items {
        set value [registry get $key $item]
        set type [registry type $key $item]
        switch -exact -- $type {
            binary {
                set value hex:[binary encode hex $value]
            }
            none {
                set value none:$value
            }
            sz {
                set value \"$value\"
            }
            expand_sz {
                set value expand_sz:\"$value\"
            }
            dword {
                set value dword:[format "%08x" $value]
            }
            dword_big_endian {
                set value dword_big_endian:[format "%08x" $value]
            }
            link {
                set value link:$value
            }
            multi_sz {
                set value multi_sz:\"$value\"
            }
            resource_list {
                set value resource_list:$value
            }
            default  {
                set value unexpected:$value
            }
        }
        append data "\"$item\"=$value\n"
    }
    foreach subkey [registry keys $key] {
        append data [GidUtils::GetRegistryKeyRecursive $key\\$subkey]
    }
    return $data
}

# proc to facilitate users with problems detecting the USB devices in GiD to send us a file with the registry data
# can be called for example writting in the lower command line -np- GidUtils::GetUsbDataToSendToGiD
proc GidUtils::GetUsbDataToSendToGiD { {filename ""} } {
    if { $filename == "" } {
        package require gid_cross_platform
        set filename [file join [gid_cross_platform::get_tmp] "usb_data.txt"]
    }
    set data1 [GetRegistryKeyRecursive {HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\USBSTOR\Enum}]
    set data2 [GetRegistryKeyRecursive {HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Enum\USBSTOR}]
    set data3 [GetRegistryKeyRecursive {HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Enum\USB}]
    set data4 [GetRegistryKeyRecursive {HKEY_LOCAL_MACHINE\SYSTEM\MountedDevices}]
    set data5 [GetRegistryKeyRecursive {HKEY_CURRENT_USER\Software\Microsoft\Windows\CurrentVersion\Explorer\MountPoints2\CPC\Volume}]
    GidUtils::WriteFile $filename $data1\n$data2\n$data3\n$data4\n$data5
    W "File $filename created with some registery data related to the USB devices. Please send it to the GiD support team in case of problems detecting the USB devices"
}

#this is an auxiliary procedure to facilitate the use of 'tester'
#request is a list {request_variables request_expr_procedures} that are also lists
proc GidUtils::WriteMonitoring { requests filename } {
    package require gid_monitoring
    lassing $requests request_variables request_expr_procedures
    gid_monitoring::set_request $request_variables
    gid_monitoring::set_expr_procedures $request_expr_procedures
    gid_monitoring::saveinfo $filename
}

proc GidUtils::GetMeanNormal_Geometry { } {
    set normal_sum [list 0 0 0]
    foreach surface_id [GiD_Geometry list surface] {
        set normal [GidUtils::GetSurfaceNormal $surface_id]
        set normal_sum [MathUtils::VectorSum $normal_sum $normal]
    }
    return [MathUtils::VectorNormalized $normal_sum]
}

proc GidUtils::GetMeanNormal_Mesh { } {
    set normal_sum [list 0 0 0]
    foreach element_id [GiD_Mesh list -element_type {triangle quadrilateral circle} element] {
        set normal [GiD_Mesh get element $element_id normal]
        set normal_sum [MathUtils::VectorSum $normal_sum $normal]
    }
    return [MathUtils::VectorNormalized $normal_sum]
}


#return the relative (to basedir) path of target
#useful to have the relative path of a problemtype (relative to the \problemtypes GiD directory)
#e.g: GidUtils::MakeRelativePath {C:\gid\problemtypes} {C:\gid\problemtypes\ansys55\ansys55_3d}
#will return "ansys55/ansys55_3d"
proc GidUtils::MakeRelativePath { basedir target } {
    set bparts [file split [file normalize $basedir]]
    set tparts [file split [file normalize $target]]

    if {[lindex $bparts 0] eq [lindex $tparts 0]} {
        # If the first part doesn't match - there is no good relative path
        set blen [llength $bparts]
        set tlen [llength $tparts]
        for {set i 1} {$i < $blen && $i < $tlen} {incr i} {
            if {[lindex $bparts $i] ne [lindex $tparts $i]} { break }
        }
        set path [lrange $tparts $i end]
        for {} {$i < $blen} {incr i} {
            set path [linsert $path 0 ..]
        }
        set target [join $path [file separator]]
    }
    #file join to change \ by / also for windows
    return [file join $target]
}

proc GidUtils::MakeAbsolutePath { filename } {
    set fullname ""
    set type [file pathtype $filename]
    if { $type == "absolute" } {
        set fullname $filename
    } elseif { $type == "relative" || $type == "volumerelative" } {
        set fullname [file join [pwd] $filename]
    } else {
        #unexpected
        set fullname $filename
    }
    return $fullname
}

#to append increasily -number to a filename, without lost its extension
proc GidUtils::AddNumberToFile { filename } {
    set dirname [file dirname $filename]
    set extension [file extension $filename]
    set file_tail_without_extension [file tail [file rootname $filename]]
    if { [regexp {^(.*?)-([0-9]+)$} $file_tail_without_extension dummy name num_string] } {
        set num [string trimleft $num_string 0] ;#to avoid that  num 09 is consided as octal and incr num raise error that is not an integer
        if { [string is integer -strict $num] } {
            incr num
        } else {
            set num 1
        }
        set prefix_left_zeros [string repeat 0 [expr {[string length $num_string]-[string length $num]}]] ;#to preserve zeros prefixing the number
        set num ${prefix_left_zeros}$num
    } else {
        set name $file_tail_without_extension
        set num 1
    }
    return [file join $dirname ${name}-$num${extension}]
}

proc GidUtils::GetFullPathGiDExe { } {
    if { $::tcl_platform(platform) == "windows"} {
        #set fullpath [file join [gid_filesystem::get_folder_gid] gid.exe]
        #this is not true runnig in develop, can be gid_64.exe, gid_32.exe
        set fullpath $::argv0
    } else {
        set fullpath [file join [gid_filesystem::get_folder_gid] gid]
    }
    return $fullpath
}

#e.g. from {D:\problemtypes\gid_project_old\problemtypes\Examples\cmas2d.gid} it returns Examples/cmas2d.gid
proc GidUtils::GetRelativeProblemTypePath { dir } {
    set pos [lindex [lsearch -all [file split $dir] problemtypes] end]
    if { $pos != "" } {
        set relative_dir [join [lrange [file split $dir] $pos+1 end] [file separator]]
    } else {
        set relative_dir $dir
    }
    return [file join $relative_dir]
}

#called from C++
proc GidUtils::GiveFileInsideProblemType { ptname ext {checkifexists 0} } {
    set filename ""
    if { [string tolower [file extension $ptname]] == ".gid" } {
        set ptname [file rootname $ptname]
    }
    set tailname [file tail $ptname]
    set ptname $ptname.gid
    set path_type [file pathtype $ptname]
    #try to find problemtypes folder in several locations
    set locations [list standard user_settings nextcloud appstream]
    foreach location $locations {
        if { $path_type == "absolute" } {
            #find only once, location doesn't matter
            set fullptname $ptname
        } else {
            set fullptname [gid_filesystem::get_folder_$location problemtypes]
        }
        if { $fullptname != "" } {
            foreach item [file split $ptname] {
                set fullptname [file join $fullptname $item]
                if { [string length $fullptname] == 3 && [string index $fullptname 1] == ":" } {
                    #avoid ask "file exists Z:/.lnk" that is slow in Windows with an existing unit and filename starting with .
                    continue
                }
                if { [GidUtils::IsLink $fullptname.lnk] == 2 } {
                    set fullptname [GidUtils::ReadLink $fullptname.lnk 1]
                }
            }
            set filename [file join $fullptname $tailname$ext]
            if { $checkifexists } {
                if { ![file exists $filename] } {
                    set filename ""
                }
            }
            if { $filename != "" } {
                #if found do not find in more locations
                break
            }
            if { $path_type == "absolute" } {
                #find only once, location doesn't matter
                break
            }
        }
    }
    return $filename
}

proc GidUtils::ModelHasName { } {
    set has_name 1
    if { [GiD_Info Project ModelName] == "UNNAMED" } {
        set has_name 0
    }
    return $has_name
}

proc GidUtils::GetDirectoryModel { } {
    if { [GidUtils::ModelHasName] } {
        set directory [GiD_Info Project ModelName].gid
        if { [file pathtype $directory] == "relative" } {
            set directory [file join [pwd] $directory]
        }
    } else {
        set directory [file join [GiD_Info project TmpDirectory] UNNAMED.gid]
    }
    return $directory
}

# canonical 'gallery' name inside a gid_project. change only once here.
proc GidUtils::GetGalleryDirectoryBaseName {} {
    return "gallery"
}

# returns "" if flag create_if... is true but the directory does not exist and can not be created
proc GidUtils::GetDirectoryModelGallery { { create_if_not_exists 0 } } {
    if { [GidUtils::ModelHasName] } {
        set directory [GiD_Info Project ModelName].gid
        if { [file pathtype $directory] == "relative" } {
            set directory [file join [pwd] $directory]
        }
    } else {
        set directory [file join [GiD_Info project TmpDirectory] UNNAMED.gid]
    }
    set gallery_dir [ file join $directory [ GidUtils::GetGalleryDirectoryBaseName]]
    if { $create_if_not_exists} {
        if { ![file exists $gallery_dir] && [ gid_filesystem::file_writable $directory]} {
            file mkdir $gallery_dir
        } else {
            set gallery_dir ""
        }
    }
    return $gallery_dir
}


#base_name is the name of the gid folder, without the .gid extension
#extension is .geo, .msh, etc
#e.g: GidUtils::GetFilenameInsideProject "C:/my dir/arc" .geo  --> "C:/my dir/arc.gid/arc.geo"
proc GidUtils::GetFilenameInsideProject { base_name extension } {
    set filename [file join ${base_name}.gid [file tail $base_name]$extension]
    return $filename
}

#filename could have links in some parts of the path, this procedure dereference them
proc GidUtils::GiveRealFilenameDereferencingLinks { filename {checkifexists 0} } {
    set outfilename ""
    foreach item [file split $filename] {
        set outfilename [file join $outfilename $item]
        if { [GidUtils::IsLink $outfilename.lnk] == 2 } {
            set outfilename [GidUtils::ReadLink $outfilename.lnk 1]
        }
    }
    if { $checkifexists } {
        if { ![file exists $outfilename] } {
            set outfilename ""
        }
    }
    return $outfilename
}

#return the full path to a relative or absolute problemtype name, e.g.
#return "" if is not a valid GiD problemtype (must have some problemtype file .bas, .cnd, etc)
#in:  nastran3/nastran            out: C:/GiD project/problemtypes/nastran3/nastran
#in:  C:/tmp/nastran3/nastran     out: C:/tmp/nastran3/nastran
#called from C++
proc GidUtils::GiveProblemTypeFullname { ptname } {
    global GidCache
    if { [info exists ::GidCache(ProblemTypeFullname,$ptname)] } {
        return $::GidCache(ProblemTypeFullname,$ptname)
    }

    set checkifexists 1
    if { [GidUtils::GiveFileInsideProblemType $ptname ".geo" $checkifexists] != "" } {
        #it is a model, not a problemtype
        return ""
    }
    set fullfilename ""
    foreach ext {.bas .cnd .prb .mat .uni .tcl .tbe .bat .spd .xml} {
        set fullfilename [GidUtils::GiveFileInsideProblemType $ptname $ext $checkifexists]
        if { $fullfilename != "" } {
            break
        }
    }
    if { $fullfilename != "" } {
        set fullpath [file root [file dirname $fullfilename]]
    } else {
        set fullpath ""
    }
    set ::GidCache(ProblemTypeFullname,$ptname) $fullpath
    return $fullpath
}

# GidUtils::GetListProblemTypes return the list of problemtypes
# internal proc GidUtils::_GetRecursiveProblemTypeDirs
proc GidUtils::_GetRecursiveProblemTypeDirs { dir current path } {
    set lst_dir [list]
    if { $dir == "" } {
        return $lst_dir
    }
    global GidPriv
    set inum 0
    if { ![info exists GidPriv(dirfiles,$dir)] } {
        #set GidPriv(dirfiles,$dir) and subdirectories
        FastCacheFromProblemtypeMenuDir $dir
        #Note: FastCacheFromProblemtypeMenuDir also call ReadProblemtypeConfigFile
    }
    foreach i $GidPriv(dirfiles,$dir) {
        set ifull [file join $dir $i]
        if { [file extension $i] == ".lnk" && [gid_filesystem::is_link $ifull] == 2 } {
            set i [file root $i] ;#to remove .lnk
        }
        if { [file extension $i] == ".gid" } {
            set pt [file root $i]
            set fullrelativept [file join $path $pt]
            lappend lst_dir $fullrelativept
        } else {
            # it is a folder
            if { $i == [lindex [file split $current] 0] } {
                set lst_pt [GidUtils::_GetRecursiveProblemTypeDirs $ifull [lrange [file split $current] 1 end] [file join $path $i]]
                set lst_dir [concat $lst_dir $lst_pt]
            } else {
                set lst_pt [GidUtils::_GetRecursiveProblemTypeDirs $ifull "" [file join $path $i]]
                set lst_dir [concat $lst_dir $lst_pt]
            }
        }
    }
    return $lst_dir
}

#location: standard, user_settings, nextcloud,appstream, all
proc GidUtils::GetListProblemTypes { {location "all"} } {
    set lst_problemtypes [list]
    set current_problemtype [GiD_Info Project ProblemType]
    set locations [list standard user_settings nextcloud appstream]
    if { [lsearch $locations $location] != -1 } {
        set dir_username_problemtypes [gid_filesystem::get_folder_$location problemtypes]
        set lst_user_pts [GidUtils::_GetRecursiveProblemTypeDirs $dir_username_problemtypes $current_problemtype ""]
        lappend lst_problemtypes {*}$lst_user_pts
    } elseif { $location == "all" } {
        foreach location $locations {
            set dir_username_problemtypes [gid_filesystem::get_folder_$location problemtypes]
            set lst_user_pts [GidUtils::_GetRecursiveProblemTypeDirs $dir_username_problemtypes $current_problemtype ""]
            lappend lst_problemtypes {*}$lst_user_pts
        }
    }
    return $lst_problemtypes
}

# GidUtils::GetProblemTypeCurrentOutputFilenames return list of OutputFile 'logs' of calculation program(s)
proc GidUtils::GetProblemTypeCurrentOutputFilenames { } {
    return [PWFIndOutputFilenameCurrent]
}

#to print a integer number over an image (create a new image if dst == "")
proc GidUtils::PrintNumberOverImage { number {dst ""} {xorig 0} {yorig 0} } {
    if { ![string is integer $number] } {
        return $dst
    }
    #gif black digits 0 to 9 with transparent background
    set src [image create photo -data { \
        R0lGODlhRgAIAJEAANnZ2QAAAP///////yH5BAEAAAAALAAAAABGAAgAAALM
        RBIEO84igmBExF0QjIiIC0oAQLEB4AAISAQABMExiDCjABAAAQQEACCCYMRZ
        XBDsuAABChAAEEAIAAIgCCF8C7Q4CgIAQfDj7uIoEARAAAEBCACgAABB8C0w
        AikIAEQEBYCICIIdZxFBcAzCwoACAATBh8ALowBB8CEsLghGHMQFwYkICwMK
        ABAEP+4sjGIT/LC4OAphx8VdEBCDMAuCHWdxQbDj4sIoAAQABMGOC4ujABAR
        BDsOKDYAHMGPg4ggGBFxEAQ/Ig6CEgAUADs=  }]

        #colorize $src #919191
        set image $src
        set color #919191
        # get image sizes
        set width  [image width  $image]
        set height [image height $image]
        # get color's R G B
        lassign [winfo rgb . $color] rr gg bb
        # compute new colors
        set trans [list]
        set colors [list]
        for {set y 0} {$y < $height} {incr y} {
        set row [list]
        for {set x 0} {$x < $width} {incr x} {
            # save transparency
            lappend trans $x $y [$image transparency get $x $y]
            # compute the new color
            foreach {r g b} [$image get $x $y] break
            set r [expr {round((256 - $r) * $rr / 256)}]
            set g [expr {round((256 - $g) * $gg / 256)}]
            set b [expr {round((256 - $b) * $bb / 256)}]
            # append to the current row
            lappend row [format #%4.4x%4.4x%4.4x $r $g $b]
        }
        # append the row
        lappend colors $row
    }
    # set new image
    $image put $colors
    # restore transparency
    foreach {x y t} $trans {
        $image transparency set $x $y $t
    }
    #end colorize

    set w 7
    set h 8
    set n [string length $number]
    if { $dst == "" } {
        set dst [image create photo -width [expr $n*$w] -height $h]
        $dst blank
    }
    for {set i 0} {$i < $n} {incr i} {
        set digit [string index $number $i]
        set x1 [expr $digit*$w]
        set x2 [expr $i*$w+$xorig]
        set y2 $yorig
        $dst copy $src -from $x1 0 [expr $x1+$w] $h -to $x2 $y2  [expr $x2+$w] [expr $y2+$h]
    }
    image delete $src
    return $dst
}

proc GidUtils::ResizeImage { img width height } {
    set new_img [image create photo -width $width -height $height]
    img_zoom $new_img $img Lanczos3
    return $new_img
}

proc GidUtils::ImageNewGrayScale { img } {
    set w [image width $img]
    set h [image height $img]
    set new_img [image create photo -width $w -height $w]
    for {set x 0} {$x<$w} {incr x} {
        for {set y 0} {$y<$h} {incr y} {
            lassign [$img get $x $y] r g b
            set l [expr {int(0.2126*$r + 0.7152*$g + 0.0722*$b)}]
            $new_img put [format "#%02x%02x%02x" $l $l $l] -to $x $y
        }
    }
    return $new_img
}

# e.g. to look disabled
proc GidUtils::ImageNewTransparent { img {alpha 0.3} } {
    return [image create photo -data [$img data -format png] -format [list png -alpha $alpha]]
}

#apply transparency outside the circle of diameter
#the image is expected a square
proc GidUtils::ImageSetTransparentCircle { image diameter } {
    set r [expr int($diameter/2)]
    set r2 [expr $r*$r]
    for {set x 0} {$x<$diameter} {incr x} {
        for {set y 0} {$y<$diameter} {incr y} {
            set d2 [expr {($x-$r)*($x-$r)+($y-$r)*($y-$r)}]
            if { $d2>$r2 } {
                #outside the circle
                $image transparency set $x $y 1
            } else {
                #inside the circle
                if { 0 && [$image transparency get $x $y] } {
                    #set transparent as opaque circle_background color
                    set circle_background #ffffff
                    $image transparency set $x $y 0
                    $image put $circle_background -to $x $y
                }
            }
        }
    }
    return
}

proc GidUtils::GetToobarNames {} {
    global ToolbarsPriv
    set names [list]
    foreach item $::ToolbarsPriv(toolbars) {
        lappend names [lindex $item 0]
    }
    return $names
}

# hide the toolbar and avoid to be showed again from the Toolbar window
# name is the toolbar name: "Right buttons" "Command line" "Up menu"
# "Geometry & View bar" "Standard bar" "View results bar" "MacrosToolbar"
# "Status & Information"
# Added option to disable all toolbars, name=AllToolbars
proc GidUtils::DisableToolbar { name } {
    set prev_toolbars_data [list]
    global ToolbarsPriv GidPriv ToolbarsInsideLocationPriv
    if { ![info exists ::ToolbarsPriv(toolbars)] } {
        return $prev_toolbars_data
    }
    foreach item $::ToolbarsPriv(toolbars) {
        lassign $item name_i geomvarname_i command_i title_i
        if { $name=="AllToolbars" || $name == $name_i } {
            if { [info exists GidPriv($geomvarname_i)] } {
                lappend prev_toolbars_data [list $geomvarname_i $GidPriv($geomvarname_i)]
                set location [lindex $GidPriv($geomvarname_i) 0]
                if { $location != "NONE" } {
                    #to try to preserve previous location if was not the default one, like compassfem
                    set ::ToolbarsInsideLocationPriv($geomvarname_i) $location
                }
            } else {
                set GidPriv($geomvarname_i) [list NONE {} 1 $command_i]
            }
            {*}$command_i NONE
            set ::ToolbarsPriv(Hide,$name_i) 1
            if { $name!="AllToolbars" } {
                break
            }
        }
    }
    return $prev_toolbars_data
}

proc GidUtils::EnableToolbar { name } {
    global ToolbarsPriv GidPriv ToolbarsInsideLocationPriv
    if { ![info exists ::ToolbarsPriv(toolbars)] } {
        return 1
    }
    foreach item $::ToolbarsPriv(toolbars) {
        lassign $item name_i geomvarname_i command_i title_i
        if { $name=="AllToolbars" || $name == $name_i } {
            if { [info exists ::ToolbarsInsideLocationPriv($geomvarname_i)] } {
                set location $::ToolbarsInsideLocationPriv($geomvarname_i)
            } else {
                set location INSIDE
                if { $name_i == "Geometry & View bar" || $name_i == "View results bar" || $name_i == "MacrosToolbar" } {
                    append location LEFT
                } elseif { $name_i == "Standard bar" } {
                    append location TOP
                }
            }
            set GidPriv($geomvarname_i) [list $location {} 1 $command_i]
            {*}$command_i [lindex $GidPriv($geomvarname_i) 0]
            set ::ToolbarsPriv(Hide,$name_i) 0
            if { $name!="AllToolbars" } {
                break
            }
        }
    }
    return 0
}

#hide all GUI widgets but the main togl drawing widget
proc GidUtils::RemoveGUI { } {
    global GUIPriv ToolbarsPriv
    set ::GUIPriv(prev_windows_closed) [GidUtils::CloseWindow ALL]
    set ::GUIPriv(prev_toolbars_data) [GidUtils::DisableToolbar AllToolbars]
    # array unset to not hide to allow to restore the toolbars from Toolbars window
    array unset ::ToolbarsPriv Hide,*
}

proc GidUtils::RestoreGUI { } {
    global GUIPriv
    global GidPriv
    if { [info exists ::GUIPriv(prev_toolbars_data)] } {
        foreach geomvarname_and_data $::GUIPriv(prev_toolbars_data) {
            lassign $geomvarname_and_data geomvarname_i data_i
            set ::GidPriv($geomvarname_i) $data_i
            set command_i [lindex $data_i 3]
            if { [llength $command_i] > 1 && [lindex $command_i 0] == "toolbarmacros::Create" } {
                # instead "toolbarmacros::Create" I have "toolbarmacros::Create NONE" !!!
                set command_i [lindex $command_i 0]
            }
            {*}$command_i [lindex $data_i 0]
        }
        unset ::GUIPriv(prev_toolbars_data)
    }
    if { [info exists ::GUIPriv(prev_windows_closed)] } {
        foreach key $::GUIPriv(prev_windows_closed)  {
            GidUtils::OpenWindow $key
        }
        unset ::GUIPriv(prev_windows_closed)
    }
    update idletasks
}

#to save a georeference file when saving an image
#if read this image as background real size, will be set again in its location.
#This procedure can be automatically invoked when saving an image by
#defining the GiD_Event_AfterSaveImage: event procedure like this:
#GiD_RegisterEvent GiD_Event_AfterSaveImage GidUtils:SaveGeoreferenceFile PROBLEMTYPE IBER
#
proc GidUtils:SaveGeoreferenceFile { filename type } {
    set fail 0
    switch $type {
        tif {
            set newext tfw
        }
        jpg {
            set newext jgw
        }
        png {
            set newext pgw
        }
        gif {
            set newext gfw
        }
        bmp {
            set newext bpw
        }
        default {
            set fail 1
        }
    }
    if { !$fail } {
        foreach item [GiD_Info view] {
            #check that the current view is in XY plane, else don't save the georeference file
            if { [lindex $item 0] == "m" } {
                for { set i 1 } { $i <= 12 } { incr i } {
                    set v [lindex $item $i]
                    switch $i {
                        1 -
                        6 -
                        11 {
                            if { [expr abs($v-1.0)] > 1e-6 } {
                                set fail 1
                                break
                            }
                        }
                        default {
                            if { [expr abs($v)] > 1e-6 } {
                                set fail 1
                                break
                            }
                        }
                    }
                }
                break
            }
        }
        if { $fail } {
            GidUtils::SetWarnLine [_ "Georeference file not saved because current view is not XY"]
        }
    }
    if { !$fail } {
        set newfilename [file rootname $filename].$newext
        set fp [open $newfilename w]
        if { $fp == "" } {
            set fail 1
        } else {
            lassign [GiD_Info ortholimits] L R B T N F E
            set Mxx [expr {double($R-$L)/[winfo width .gid.central.s]}]
            set Myx 0
            set Mxy 0
            set Myy [expr {double($B-$T)/[winfo height .gid.central.s]}]
            set Dx $L
            set Dy $T
            puts $fp $Mxx
            puts $fp $Myx
            puts $fp $Mxy
            puts $fp $Myy
            puts $fp $Dx
            puts $fp $Dy
            #close $fp
            GidUtils::CatchClose $fp $newfilename
            GidUtils::SetWarnLine [_ "Georeference file saved"]
        }
    }
    return $fail
}


#to be used by TKWIDGET to easily add a select file button
#e.g.
#QUESTION: your_question
#VALUE: your_filename
#TKWIDGET: GidUtils::TkwidgetGetFilenameButton

proc GidUtils::TkwidgetGetFilenameButton { event args } {
    global tkwidgedprivfilenamebutton
    switch $event {
        INIT {
            lassign $args PARENT current_row_variable GDN STRUCT QUESTION
            upvar $current_row_variable ROW
            #initialize variable to current field value
            set tkwidgedprivfilenamebutton($QUESTION,filename) [DWLocalGetValue $GDN $STRUCT $QUESTION]
            #set entry $PARENT.e$ROW
            set entry ""
            foreach item [grid slaves $PARENT -row [expr $ROW-1]] {
                if { [winfo class $item] == "Entry"  || [winfo class $item] == "TEntry" } {
                    #assumed that it is the only entry of this row
                    set entry $item
                    break
                }
            }
            #trick to fill in the values pressing transfer from an applied condition
            if { [lindex [info level 2] 0] == "DWUpdateConds" } {
                set values [lrange [lindex [info level 2] 2] 3 end]
                set index_field [DWGetQuestionIndex $GDN $STRUCT $QUESTION]
                if { $index_field != -1 } {
                    set value [lindex $values $index_field-1]
                    set tkwidgedprivfilenamebutton($QUESTION,filename) $value
                }
            }
            set w [ttk::frame $PARENT.cfilenamebutton$QUESTION] ;#use a name depending on $QUESTION to allow more than one row changed
            ttk::entry $w.e1 -textvariable tkwidgedprivfilenamebutton($QUESTION,filename)
            ttk::button $w.b1 -image [gid_themes::GetImage "folder.png"] \
                -command [list GidUtils::_GetFilenameCmd tkwidgedprivfilenamebutton($QUESTION,filename) $w.e1 0]
            set tkwidgedprivfilenamebutton($QUESTION,widget) $w
            grid $w.e1 $w.b1 -sticky ew
            grid columnconfigure $w {0} -weight 1
            grid $w -row [expr $ROW-1] -column 1 -sticky ew
            if { $entry != "" } {
                grid remove $entry
            } else {
                #assumed that entry is hidden and then hide the usurpating frame
                grid remove $w
            }
        }
        SYNC {
            lassign $args GDN STRUCT QUESTION
            if { [info exists tkwidgedprivfilenamebutton($QUESTION,filename)] } {
                DWLocalSetValue $GDN $STRUCT $QUESTION $tkwidgedprivfilenamebutton($QUESTION,filename)
            }
        }
        DEPEND {
            lassign $args GDN STRUCT QUESTION ACTION VALUE
            if { [info exists tkwidgedprivfilenamebutton($QUESTION,widget)] && [winfo exists $tkwidgedprivfilenamebutton($QUESTION,widget)] } {
                if { $ACTION == "HIDE" } {
                    grid remove $tkwidgedprivfilenamebutton($QUESTION,widget)
                } else {
                    #RESTORE
                    grid $tkwidgedprivfilenamebutton($QUESTION,widget)
                }
            } else {

            }
        }
        CLOSE {
            array unset tkwidgedprivfilenamebutton
        }
        default {
            return [list ERROR [_ "Unexpected tkwidget event"]]
        }
    }
    #a tkwidget procedure must return "" if Ok or [list ERROR $description] or [list WARNING $description]
    return ""
}


#axiliary procedure used by GidUtils::TkwidgetGetFilenameButton
proc GidUtils::_GetFilenameCmd { varname entry {tail 0}} {
    set types [list [list [_ "All files"] ".*"]]
    set defaultextension ""
    set title [_ "Select file"]
    set current_value [MessageBoxGetFilename file read $title [set ::$varname] $types $defaultextension 0]
    if { $tail } {
        set current_value [file tail $current_value]
    }
    if { $current_value != "" && $entry != "" && [winfo exists $entry] } {
        $entry delete 0 end
        $entry insert end $current_value
    }
    #set variable after change entry, else if variable is the own entry variable then delete 0 end will empty both
    set ::$varname $current_value
    return $current_value
}

#to be used by TKWIDGET to easily add a select directory button
#e.g.
#QUESTION: your_question
#VALUE: your_filename
#TKWIDGET: GidUtils::TkwidgetGetDirectoryButton

proc GidUtils::TkwidgetGetDirectoryButton { event args } {
    global tkwidgedprivfilenamebutton
    switch $event {
        INIT {
            lassign $args PARENT current_row_variable GDN STRUCT QUESTION
            upvar $current_row_variable ROW
            #initialize variable to current field value
            set tkwidgedprivfilenamebutton($QUESTION,filename) [DWLocalGetValue $GDN $STRUCT $QUESTION]
            #set entry $PARENT.e$ROW
            set entry ""
            foreach item [grid slaves $PARENT -row [expr $ROW-1]] {
                if { [winfo class $item] == "Entry"  || [winfo class $item] == "TEntry" } {
                    #assumed that it is the only entry of this row
                    set entry $item
                    break
                }
            }
            #trick to fill in the values pressing transfer from an applied condition
            if { [lindex [info level 2] 0] == "DWUpdateConds" } {
                set values [lrange [lindex [info level 2] 2] 3 end]
                set index_field [LabelField $GDN $STRUCT $QUESTION]
                set value [lindex $values $index_field-1]
                set tkwidgedprivfilenamebutton($QUESTION,filename) $value
            }
            set w [ttk::frame $PARENT.cfilenamebutton$QUESTION] ;#use a name depending on $QUESTION to allow more than one row changed
            ttk::entry $w.e1 -textvariable tkwidgedprivfilenamebutton($QUESTION,filename)
            ttk::button $w.b1 -image [gid_themes::GetImage "folder.png"] \
                -command [list GidUtils::_GetDirectoryCmd tkwidgedprivfilenamebutton($QUESTION,filename) $w.e1 0]
            set tkwidgedprivfilenamebutton($QUESTION,widget) $w
            grid $w.e1 $w.b1 -sticky ew
            grid columnconfigure $w {0} -weight 1
            grid $w -row [expr $ROW-1] -column 1 -sticky ew
            if { $entry != "" } {
                grid remove $entry
            } else {
                #assumed that entry is hidden and then hide the usurpating frame
                grid remove $w
            }
        }
        SYNC {
            lassign $args GDN STRUCT QUESTION
            if { [info exists tkwidgedprivfilenamebutton($QUESTION,filename)] } {
                DWLocalSetValue $GDN $STRUCT $QUESTION $tkwidgedprivfilenamebutton($QUESTION,filename)
            }
        }
        DEPEND {
            lassign $args GDN STRUCT QUESTION ACTION VALUE
            if { [info exists tkwidgedprivfilenamebutton($QUESTION,widget)] && \
                [winfo exists $tkwidgedprivfilenamebutton($QUESTION,widget)] } {
                if { $ACTION == "HIDE" } {
                    grid remove $tkwidgedprivfilenamebutton($QUESTION,widget)
                } else {
                    #RESTORE
                    grid $tkwidgedprivfilenamebutton($QUESTION,widget)
                }
            } else {

            }
        }
        CLOSE {
            array unset tkwidgedprivfilenamebutton
        }
        default {
            return [list ERROR [_ "Unexpected tkwidget event"]]
        }
    }
    #a tkwidget procedure must return "" if Ok or [list ERROR $description] or [list WARNING $description]
    return ""
}

#axiliary procedure used by GidUtils::TkwidgetGetDirectoryButton
proc GidUtils::_GetDirectoryCmd { varname entry {tail 0}} {
    set types [list [list [_ "All files"] ".*"]]
    set defaultextension ""
    set title [_ "Select directory"]
    set current_value [MessageBoxGetFilename directory read $title [set ::$varname] $types $defaultextension 0]
    if { $tail } {
        set current_value [file tail $current_value]
    }
    if { $current_value != "" && $entry != "" && [winfo exists $entry] } {
        $entry delete 0 end
        $entry insert end $current_value
    }
    #set variable after change entry, else if variable is the own entry variable then delete 0 end will empty both
    set ::$varname $current_value
    return $current_value
}

#dark trick: maybe the entry has been created but not gridded because is has been hidden by some dependency?
#try to find some ungridded entry with an associated label named .questionXX with label equal to question
proc GidUtils::DarkTrickTryFindUngriddedEntry { PARENT QUESTION } {
    set entry ""
    set question_translated [= [DWUnder2Space $QUESTION]]
    foreach item [winfo children $PARENT] {
        if { [winfo class $item] == "Entry"  || [winfo class $item] == "TEntry" } {
            if { [grid info $item] == "" } {
                set entry_tail [lindex [split $item .] end]
                if { [string index $entry_tail 0] == "e" } {
                    set question_label ${PARENT}.question[string range $entry_tail 1 end]
                    if { [winfo exists $question_label] && ([winfo class $question_label] == "TLabel" || [winfo class $question_label] == "Label") &&
                        [$question_label cget -text] == $question_translated } {
                        set entry $item
                        break
                    }
                }
            }
        }
    }
    return $entry
}

#to be used by TKWIDGET to easily add a pick node button
#e.g.
#QUESTION: your_question
#VALUE: your_node_id
#TKWIDGET: GidUtils::TkwidgetPickNode

proc GidUtils::TkwidgetPickNode { event args } {
    global tkwidgedprivpicknodebuttons
    switch $event {
        INIT {
            lassign $args PARENT current_row_variable GDN STRUCT QUESTION
            upvar $current_row_variable ROW
            set entry ""
            set entry_gridded 0
            foreach item [grid slaves $PARENT -row [expr $ROW-1]] {
                if { [winfo class $item] == "Entry"  || [winfo class $item] == "TEntry" } {
                    #assumed that it is the only entry of this row
                    set entry $item
                    set entry_gridded 1
                    break
                }
            }
            if { $entry == "" } {
                set entry [GidUtils::DarkTrickTryFindUngriddedEntry $PARENT $QUESTION]
            }
            if { $entry != "" } {
                set tkwidgedprivpicknodebuttons($QUESTION) [ttk::button $PARENT.bpicknode$QUESTION \
                    -image [gid_themes::GetImage "point.png" small_icons] -command [list GidUtils::_PickNodeCmd $entry]]
                grid $tkwidgedprivpicknodebuttons($QUESTION) -row [expr $ROW-1] -column 2 -sticky w
                grid configure $entry -sticky ew
                if { !$entry_gridded } {
                    grid remove $entry
                    grid remove $tkwidgedprivpicknodebuttons($QUESTION)
                }
            }
            return ""
        }
        SYNC {
            #lassign $args GDN STRUCT QUESTION
            #DWLocalSetValue $GDN $STRUCT $QUESTION $value
        }
        DEPEND {
            lassign $args GDN STRUCT QUESTION ACTION VALUE
            if { [info exists tkwidgedprivpicknodebuttons($QUESTION)] && \
                [winfo exists $tkwidgedprivpicknodebuttons($QUESTION)] } {
                if { $ACTION == "HIDE" } {
                    grid remove $tkwidgedprivpicknodebuttons($QUESTION)
                } else {
                    #RESTORE
                    grid $tkwidgedprivpicknodebuttons($QUESTION)
                }
            }
        }
        CLOSE {
            array unset tkwidgedprivpicknodebuttons
        }
        default {
            return [list ERROR [_ "Unexpected tkwidget event"]]
        }
    }
    #a tkwidget procedure must return "" if Ok or [list ERROR $description] or [list WARNING $description]
    return ""
}

#axiliary procedure used by GidUtils::TkwidgetPickNode
proc GidUtils::_PickNodeCmd { entry } {
    set id [GidUtils::PickEntities Nodes single]
    if { $id != "" } {
        $entry delete 0 end
        $entry insert end $id
    }
}


#to be used by TKWIDGET to easily add a pick node button
#e.g.
#QUESTION: your_question
#VALUE: your_node_id
#TKWIDGET: GidUtils::TkwidgetPickPointOrNode

proc GidUtils::TkwidgetPickPointOrNode { event args } {
    global tkwidgedprivpicknodebuttons
    switch $event {
        INIT {
            lassign $args PARENT current_row_variable GDN STRUCT QUESTION
            upvar $current_row_variable ROW
            set entry ""
            set entry_gridded 0
            foreach item [grid slaves $PARENT -row [expr $ROW-1]] {
                if { [winfo class $item] == "Entry"  || [winfo class $item] == "TEntry" } {
                    #assumed that it is the only entry of this row
                    set entry $item
                    set entry_gridded 1
                    break
                }
            }
            if { $entry == "" } {
                set entry [GidUtils::DarkTrickTryFindUngriddedEntry $PARENT $QUESTION]
            }
            if { $entry != "" } {
                set tkwidgedprivpicknodebuttons($QUESTION) [ttk::button $PARENT.bpicknode$QUESTION \
                    -image [gid_themes::GetImage "point.png" small_icons] \
                    -command [list GidUtils::_PickPointOrNodeCmd $entry]]
                grid $tkwidgedprivpicknodebuttons($QUESTION) -row [expr $ROW-1] -column 2 -sticky w
                grid configure $entry -sticky ew
                if { !$entry_gridded } {
                    grid remove $entry
                    grid remove $tkwidgedprivpicknodebuttons($QUESTION)
                }
            }
            return ""
        }
        SYNC {
            #lassign $args GDN STRUCT QUESTION
            #DWLocalSetValue $GDN $STRUCT $QUESTION $value
        }
        DEPEND {
            lassign $args GDN STRUCT QUESTION ACTION VALUE
            if { [info exists tkwidgedprivpicknodebuttons($QUESTION)] && \
                [winfo exists $tkwidgedprivpicknodebuttons($QUESTION)] } {
                if { $ACTION == "HIDE" } {
                    grid remove $tkwidgedprivpicknodebuttons($QUESTION)
                } else {
                    #RESTORE
                    grid $tkwidgedprivpicknodebuttons($QUESTION)
                }
            }
        }
        CLOSE {
            array unset tkwidgedprivpicknodebuttons
        }
        default {
            return [list ERROR [_ "Unexpected tkwidget event"]]
        }
    }
    #a tkwidget procedure must return "" if Ok or [list ERROR $description] or [list WARNING $description]
    return ""
}

#axiliary procedure used by GidUtils::TkwidgetPickPoint
proc GidUtils::_PickPointCmd { entry } {
    set id [GidUtils::PickEntities Points single]
    if { $id != "" } {
        $entry delete 0 end
        $entry insert end $id
    }
}

#axiliary procedure used by GidUtils::TkwidgetPickPointOrNode
proc GidUtils::_PickPointOrNodeCmd { entry } {
    set viewmode [GiD_Info Project ViewMode]
    if { $viewmode == "GEOMETRYUSE" } {
        return [GidUtils::_PickPointCmd $entry]
    } elseif { $viewmode == "MESHUSE" } {
        return [GidUtils::_PickNodeCmd $entry]
    } else {
        return ""
    }
}

#to be used by TKWIDGET to easily add a pick node button
#e.g.
#QUESTION: your_question
#VALUE: your_value
#TKWIDGET: GidUtils::TkwidgetPickRelativePointInElement
proc GidUtils::TkwidgetPickRelativePointInElement { event args } {
    global tkwidgedprivpicknodebuttons
    switch $event {
        INIT {
            set PARENT [lindex $args 0]
            upvar [lindex $args 1] ROW
            set GDN [lindex $args 2]
            set STRUCT [lindex $args 3]
            set QUESTION [lindex $args 4]
            set entry ""
            foreach item [grid slaves $PARENT -row [expr $ROW-1]] {
                if { [winfo class $item] == "Entry"  || [winfo class $item] == "TEntry" } {
                    #assumed that it is the only entry of this row
                    set entry $item
                    break
                }
            }
            if { [string match -nocase [string range $STRUCT 0 8] "CND,Line_"] } {
                set element_level line
            } elseif { [string match -nocase [string range $STRUCT 0 11] "CND,Surface_"] } {
                set element_level surface
            } elseif { [string match -nocase [string range $STRUCT 0 10] "CND,Volume_"] } {
                set element_level volume
            } else {
                set element_level ""
            }

            if { [GiD_Info project ViewMode] == "GEOMETRYUSE" }  {
                if { $element_level == "line" } {
                    set tkwidgedprivpicknodebuttons($QUESTION) [button $PARENT.bpicknode$QUESTION \
                        -image  [gid_themes::GetImage "point.png" small_icons] \
                        -command [list GidUtils::_PickRelativePointInLineCmd $entry]]
                } elseif { $element_level == "surface" } {
                    set tkwidgedprivpicknodebuttons($QUESTION) [button $PARENT.bpicknode$QUESTION \
                        -image  [gid_themes::GetImage "point.png" small_icons] \
                        -command [list GidUtils::_PickRelativePointInSurfaceCmd $entry]]
                } elseif { $element_level == "volume" } {
                    set tkwidgedprivpicknodebuttons($QUESTION) [button $PARENT.bpicknode$QUESTION \
                        -image  [gid_themes::GetImage "point.png" small_icons] \
                        -command [list GidUtils::_PickRelativePointInVolumeCmd $entry]]
                } else {
                    error "GidUtils::TkwidgetPickRelativePointInElement. Unexpected element_level=$element_level"
                }
            } elseif { [GiD_Info project ViewMode] == "MESHUSE" }  {
                set tkwidgedprivpicknodebuttons($QUESTION) [button $PARENT.bpicknode$QUESTION \
                    -image  [gid_themes::GetImage "point.png" small_icons] \
                    -command [list ::_PickRelativePointInElementCmd $element_level $entry]]
            } else {
                error "GidUtils::TkwidgetPickRelativePointInElement. Unexpected view mode=[GiD_Info project ViewMode]"
            }
            grid $tkwidgedprivpicknodebuttons($QUESTION) -row [expr $ROW-1] -column 2 -sticky w
            grid configure $entry -sticky ew
            return ""
        }
        SYNC {
            #set GDN [lindex $args 0]
            #set STRUCT [lindex $args 1]
            #set QUESTION [lindex $args 2]
            #DWLocalSetValue $GDN $STRUCT $QUESTION $value
        }
        DEPEND {
            #set GDN [lindex $args 0]
            #set STRUCT [lindex $args 1]
            set QUESTION [lindex $args 2]
            set ACTION [lindex $args 3]
            #set value [lindex $args 4]
            if { [info exists tkwidgedprivpicknodebuttons($QUESTION)] && \
                [winfo exists $tkwidgedprivpicknodebuttons($QUESTION)] } {
                if { $ACTION == "HIDE" } {
                    grid remove $tkwidgedprivpicknodebuttons($QUESTION)
                } else {
                    #RESTORE
                    grid $tkwidgedprivpicknodebuttons($QUESTION)
                }
            }
        }
        CLOSE {
            array unset tkwidgedprivpicknodebuttons
        }
        default {
            return [list ERROR [_ "Unexpected tkwidget event"]]
        }
    }
}

#axiliary procedure used by GidUtils::TkwidgetPickRelativePointInElement

proc GidUtils::_PickRelativePointInLineCmd { entry } {
    set xyz [GidUtils::GetCoordinates [_ "Pick point into an curve"] PointInLine GEOMETRYUSE]
    if { $xyz != "" } {
        set line_id $::GidPriv(selection)
        if { $line_id != "" } {
            set t [GiD_Info Parametric Line $line_id t_fromcoord {*}$xyz]
            set res [format %f $t]
            $entry delete 0 end
            $entry insert end $res
        }
    }
}

proc GidUtils::_PickRelativePointInSurfaceCmd { entry } {
    set xyz [GidUtils::GetCoordinates [_ "Pick point into an surface"] PointInSurface GEOMETRYUSE]
    if { $xyz != "" } {
        set surface_id $::GidPriv(selection)
        if { $surface_id != "" } {
            set uv [GiD_Info Parametric Surface $surface_id uv_fromcoord {*}$xyz]
            lassign $uv u v
            set res [list [format %f $u] [format %f $v]]
            $entry delete 0 end
            $entry insert end $res
        }
    }
}

# parametric volumes not exists like with surfaces or curves, in this case store global coordinates, not local coordinates
proc GidUtils::_PickRelativePointInVolumeCmd { entry } {
    set xyz [GidUtils::GetCoordinates [_ "Pick point"] NoJoin GEOMETRYUSE]
    if { $xyz != "" } {
        lassign $xyz x y z
        set res [list [format %f $x] [format %f $y] [format %f $z]]
        $entry delete 0 end
        $entry insert end $res
    }
}

proc GidUtils::_PickRelativePointInElementCmd { element_level entry } {
    set xyz [GidUtils::GetCoordinates [_ "Pick point into an element"] PointInElement MESHUSE]
    if { $xyz != "" } {
        set element_id $::GidPriv(selection)
        set res [GidUtils::GetElementRelativeCoordinatesFromCoord $element_level $xyz $element_id]
        $entry delete 0 end
        $entry insert end $res
    }
}

#to be used by TKWIDGET to easily add a layer combobox
#e.g.
#QUESTION: your_question
#VALUE: your_layername
#TKWIDGET: GidUtils::TkwidgetGetLayername

proc GidUtils::TkwidgetGetLayername { event args } {
    global tkwidgedprivlayer
    switch $event {
        INIT {
            lassign $args PARENT current_row_variable GDN STRUCT QUESTION
            upvar $current_row_variable ROW
            #initialize variable to current field value
            if { [DWLocalGetValue $GDN $STRUCT $QUESTION] != "" } {
                set tkwidgedprivlayer($QUESTION) [DWLocalGetValue $GDN $STRUCT $QUESTION]
            } else {
                set tkwidgedprivlayer($QUESTION) [lindex [GiD_Layers list] 0]
            }

            #set entry $PARENT.e$ROW
            set entry ""
            foreach item [grid slaves $PARENT -row [expr $ROW-1]] {
                if { [winfo class $item] == "Entry"  || [winfo class $item] == "TEntry" } {
                    #assumed that it is the only entry of this row
                    set entry $item
                    break
                }
            }
            if { $entry != "" } {
                #set width [$entry cget -width]
                set width 9
                set tkwidgedprivlayer($QUESTION) [ttk::combobox $PARENT.clayer$QUESTION \
                    -textvariable tkwidgedprivlayer($QUESTION,layername) \
                    -values [GiD_Layers list] \
                    -width $width]

                grid remove $entry
                grid $tkwidgedprivlayer($QUESTION) -row [expr $ROW-1] -column 1 -sticky ew -columnspan 2
            }
        }
        SYNC {
            lassign $args GDN STRUCT QUESTION
            if { [info exists tkwidgedprivlayer($QUESTION,layername)] } {
                DWLocalSetValue $GDN $STRUCT $QUESTION $tkwidgedprivlayer($QUESTION,layername)
            }
        }
        DEPEND {
            lassign $args GDN STRUCT QUESTION ACTION VALUE
            if { [info exists tkwidgedprivlayer($QUESTION)] && \
                [winfo exists $tkwidgedprivlayer($QUESTION)] } {
                if { $ACTION == "HIDE" } {
                    grid remove $tkwidgedprivlayer($QUESTION)
                } else {
                    #RESTORE
                    grid $tkwidgedprivlayer($QUESTION)
                }
            } else {

            }
        }
        CLOSE {
            #array unset tkwidgedprivlayer
        }
        default {
            return [list ERROR [_ "Unexpected tkwidget event"]]
        }
    }
    #a tkwidget procedure must return "" if Ok or [list ERROR $description] or [list WARNING $description]
    return ""
}

#to be used by TKWIDGET to easily add a group combobox
#e.g.
#QUESTION: your_question
#VALUE: your_groupname
#TKWIDGET: GidUtils::TkwidgetGetGroupname

proc GidUtils::TkwidgetGetGroupname { event args } {
    global tkwidgedprivgroup
    switch $event {
        INIT {
            lassign $args PARENT current_row_variable GDN STRUCT QUESTION
            upvar $current_row_variable ROW
            #initialize variable to current field value
            if { [DWLocalGetValue $GDN $STRUCT $QUESTION] != "" } {
                set tkwidgedprivgroup($QUESTION) [DWLocalGetValue $GDN $STRUCT $QUESTION]
            } else {
                set tkwidgedprivgroup($QUESTION) [lindex [GiD_Groups list] 0]
            }

            #set entry $PARENT.e$ROW
            set entry ""
            foreach item [grid slaves $PARENT -row [expr $ROW-1]] {
                if { [winfo class $item] == "Entry"  || [winfo class $item] == "TEntry" } {
                    #assumed that it is the only entry of this row
                    set entry $item
                    break
                }
            }
            if { $entry != "" } {
                #set width [$entry cget -width]
                set width 9
                set tkwidgedprivgroup($QUESTION) [ttk::combobox $PARENT.cgroup$QUESTION \
                    -textvariable tkwidgedprivgroup($QUESTION,groupname) \
                    -values [GiD_Groups list] \
                    -width $width]

                grid remove $entry
                grid $tkwidgedprivgroup($QUESTION) -row [expr $ROW-1] -column 1 -sticky ew -columnspan 2
            }
        }
        SYNC {
            lassign $args GDN STRUCT QUESTION
            if { [info exists tkwidgedprivgroup($QUESTION,groupname)] } {
                DWLocalSetValue $GDN $STRUCT $QUESTION $tkwidgedprivgroup($QUESTION,groupname)
            }
        }
        DEPEND {
            lassign $args GDN STRUCT QUESTION ACTION VALUE
            if { [info exists tkwidgedprivgroup($QUESTION)] && [winfo exists $tkwidgedprivgroup($QUESTION)] } {
                if { $ACTION == "HIDE" } {
                    grid remove $tkwidgedprivgroup($QUESTION)
                } else {
                    #RESTORE
                    grid $tkwidgedprivgroup($QUESTION)
                }
            } else {

            }
        }
        CLOSE {
            #array unset tkwidgedprivgroup
        }
        default {
            return [list ERROR [_ "Unexpected tkwidget event"]]
        }
    }
    #a tkwidget procedure must return "" if Ok or [list ERROR $description] or [list WARNING $description]
    return ""
}

#to be used by TKWIDGET to easily add a material combobox
#e.g.
#QUESTION: your_question
#VALUE: your_materialname
#TKWIDGET: GidUtils::TkwidgetGetMaterialname

proc GidUtils::TkwidgetGetMaterialname { event args } {
    global tkwidgedprivmaterial
    switch $event {
        INIT {
            lassign $args PARENT current_row_variable GDN STRUCT QUESTION
            upvar $current_row_variable ROW
            #initialize variable to current field value
            if { [DWLocalGetValue $GDN $STRUCT $QUESTION] != "" } {
                set tkwidgedprivmaterial($QUESTION) [DWLocalGetValue $GDN $STRUCT $QUESTION]
            } else {
                set tkwidgedprivmaterial($QUESTION) [lindex [GiD_Info Materials] 0]
            }

            #set entry $PARENT.e$ROW
            set entry ""
            foreach item [grid slaves $PARENT -row [expr $ROW-1]] {
                if { [winfo class $item] == "Entry"  || [winfo class $item] == "TEntry" } {
                    #assumed that it is the only entry of this row
                    set entry $item
                    break
                }
            }
            if { $entry != "" } {
                #set width [$entry cget -width]
                set width 9
                set tkwidgedprivmaterial($QUESTION) [ttk::combobox $PARENT.cmaterial$QUESTION \
                    -textvariable tkwidgedprivmaterial($QUESTION,materialname) \
                    -values [GiD_Info Materials] \
                    -width $width]

                grid remove $entry
                grid $tkwidgedprivmaterial($QUESTION) -row [expr $ROW-1] -column 1 -sticky ew -columnspan 2
            }
        }
        SYNC {
            lassign $args GDN STRUCT QUESTION
            if { [info exists tkwidgedprivmaterial($QUESTION,materialname)] } {
                DWLocalSetValue $GDN $STRUCT $QUESTION $tkwidgedprivmaterial($QUESTION,materialname)
            }
        }
        DEPEND {
            lassign $args GDN STRUCT QUESTION ACTION VALUE
            if { [info exists tkwidgedprivmaterial($QUESTION)] && [winfo exists $tkwidgedprivmaterial($QUESTION)] } {
                if { $ACTION == "HIDE" } {
                    grid remove $tkwidgedprivmaterial($QUESTION)
                } else {
                    #RESTORE
                    grid $tkwidgedprivmaterial($QUESTION)
                }
            } else {

            }
        }
        CLOSE {
            #array unset tkwidgedprivmaterial
        }
        default {
            return [list ERROR [_ "Unexpected tkwidget event"]]
        }
    }
    #a tkwidget procedure must return "" if Ok or [list ERROR $description] or [list WARNING $description]
    return ""
}

proc GidUtils::ValidateEntryDouble { value entry } {
    set res [string is double $value]
    if { !$res } {
        GidUtils::SetWarnLine [_ "Invalid value: %s, must be a real number" $value]
        if { 0 && [winfo exists $entry] } {
            $entry delete 0 end
            $entry insert end 0
            focus $entry
        }
    }
    return $res
}

#to be used by TKWIDGET to easily add a vector field with all coordinates in a line
#e.g.
#QUESTION: your_question
#VALUE: vx vy vz
#TKWIDGET: GidUtils::TkwidgetGetVector3D

proc GidUtils::TkwidgetGetVector3D { event args } {
    global tkwidgedprivvector3d
    switch $event {
        INIT {
            lassign $args PARENT current_row_variable GDN STRUCT QUESTION
            upvar $current_row_variable ROW
            #initialize variable to current field value
            set default_value [DWLocalGetValue $GDN $STRUCT $QUESTION]
            lassign $default_value tkwidgedprivvector3d($QUESTION,x) tkwidgedprivvector3d($QUESTION,y) tkwidgedprivvector3d($QUESTION,z)
            #set entry $PARENT.e$ROW
            set entry ""
            foreach item [grid slaves $PARENT -row [expr $ROW-1]] {
                if { [winfo class $item] == "Entry"  || [winfo class $item] == "TEntry" } {
                    #assumed that it is the only entry of this row
                    set entry $item
                    break
                }
            }
            #trick to fill in the values pressing transfer from an applied condition
            if { [lindex [info level 2] 0] == "DWUpdateConds" } {
                set values [lrange [lindex [info level 2] 2] 3 end]
                set index_field [LabelField $GDN $STRUCT $QUESTION]
                set value [lindex $values $index_field-1]
                lassign $value tkwidgedprivvector3d($QUESTION,x) tkwidgedprivvector3d($QUESTION,y) tkwidgedprivvector3d($QUESTION,z)
            }
            set w [ttk::frame $PARENT.cvector3d$QUESTION]
            foreach axe {x y z} {
                ttk::entry $w.$axe -textvariable tkwidgedprivvector3d($QUESTION,$axe) \
                    -validate focusout -validatecommand [list GidUtils::ValidateEntryDouble %P %W] -width 6
            }
            set tkwidgedprivvector3d($QUESTION,widget) $w

            grid $w.x $w.y $w.z -sticky ew
            grid columnconfigure $w {0 1 2} -weight 1
            grid $tkwidgedprivvector3d($QUESTION,widget) -row [expr $ROW-1] -column 1 -sticky ew -columnspan 2
            if { $entry != "" } {
                grid remove $entry
            } else {
                #assumed that entry is hidden
                grid remove $tkwidgedprivvector3d($QUESTION,widget)
            }
        }
        SYNC {
            lassign $args GDN STRUCT QUESTION
            if { [info exists tkwidgedprivvector3d($QUESTION,x)] && \
                [info exists tkwidgedprivvector3d($QUESTION,y)] && \
                    [info exists tkwidgedprivvector3d($QUESTION,z)] } {
                    set value [list $tkwidgedprivvector3d($QUESTION,x) $tkwidgedprivvector3d($QUESTION,y) $tkwidgedprivvector3d($QUESTION,z)]
                DWLocalSetValue $GDN $STRUCT $QUESTION $value
            }
        }
        DEPEND {
            lassign $args GDN STRUCT QUESTION ACTION VALUE
            if { [info exists tkwidgedprivvector3d($QUESTION,widget)] && [winfo exists $tkwidgedprivvector3d($QUESTION,widget)] } {
                if { $ACTION == "HIDE" } {
                    grid remove $tkwidgedprivvector3d($QUESTION,widget)
                } else {
                    #RESTORE
                    grid $tkwidgedprivvector3d($QUESTION,widget)
                }
            } else {

            }
        }
        CLOSE {
            array unset tkwidgedprivvector3d
        }
        default {
            return [list ERROR [_ "Unexpected tkwidget event"]]
        }
    }
    #a tkwidget procedure must return "" if Ok or [list ERROR $description] or [list WARNING $description]
    return ""
}

#to be used by TKWIDGET to use a text widget instead of an entry
#e.g.
#QUESTION: your_question
#VALUE:
#TKWIDGET: GidUtils::TkwidgetText
#
#store the text encoded as base64 to be saved/read to/from disk allowing multiple lines (not allowed in normal question data)
#to print the value for the calculation data must be decoded again!!
#instead of use *gendata(your_question) must do something like *tcl(binary decode base64 *gendata(your_question))
proc GidUtils::TkwidgetText { event args } {
    global tkwidgedprivtext
    switch $event {
        INIT {
            lassign $args PARENT current_row_variable GDN STRUCT QUESTION
            upvar $current_row_variable ROW
            #initialize variable to current field value
            set value [binary decode base64 [DWLocalGetValue $GDN $STRUCT $QUESTION]]
            set entry ""
            foreach item [grid slaves $PARENT -row [expr $ROW-1]] {
                if { [winfo class $item] == "Entry"  || [winfo class $item] == "TEntry" } {
                    #assumed that it is the only entry of this row
                    set entry $item
                    break
                }
            }
            #trick to fill in the values pressing transfer from an applied condition
            if { [lindex [info level 2] 0] == "DWUpdateConds" } {
                set values [lrange [lindex [info level 2] 2] 3 end]
                set index_field [LabelField $GDN $STRUCT $QUESTION]
                set value [lindex $values $index_field-1]
            }
            set w [tk::text $PARENT.ctext$QUESTION]
            $w insert end $value
            set tkwidgedprivtext($QUESTION,widget) $w
            grid $tkwidgedprivtext($QUESTION,widget) -row [expr $ROW-1] -column 1 -sticky ew -columnspan 2
            if { $entry != "" } {
                grid remove $entry
            } else {
                #assumed that entry is hidden
                grid remove $tkwidgedprivtext($QUESTION,widget)
            }
        }
        SYNC {
            lassign $args GDN STRUCT QUESTION
            if { [info exists tkwidgedprivtext($QUESTION,widget)] && [winfo exists $tkwidgedprivtext($QUESTION,widget)] } {
                set value [$tkwidgedprivtext($QUESTION,widget) get 1.0 end]
                DWLocalSetValue $GDN $STRUCT $QUESTION [binary encode base64 $value]
            }
        }
        DEPEND {
            lassign $args GDN STRUCT QUESTION ACTION VALUE
            if { [info exists tkwidgedprivtext($QUESTION,widget)] && [winfo exists $tkwidgedprivtext($QUESTION,widget)] } {
                if { $ACTION == "HIDE" } {
                    grid remove $tkwidgedprivtext($QUESTION,widget)
                } else {
                    #RESTORE
                    grid $tkwidgedprivtext($QUESTION,widget)
                }
            } else {

            }
        }
        CLOSE {
            array unset tkwidgedprivtext
        }
        default {
            return [list ERROR [_ "Unexpected tkwidget event"]]
        }
    }
    #a tkwidget procedure must return "" if Ok or [list ERROR $description] or [list WARNING $description]
    return ""
}

#to be used by TKWIDGET to use a button widget instead of the entry
#e.g.
#QUESTION: your_question
#VALUE:
#TKWIDGET: GidUtils::TkwidgetButton {-command {W "hello world"} -text "Press me"}
#
proc GidUtils::TkwidgetButton { configure_options event args } {
    global tkwidgedprivbutton
    switch $event {
        INIT {
            lassign $args PARENT current_row_variable GDN STRUCT QUESTION
            upvar $current_row_variable ROW
            set entry ""
            foreach item [grid slaves $PARENT -row [expr $ROW-1]] {
                if { [winfo class $item] == "Entry"  || [winfo class $item] == "TEntry" } {
                    #assumed that it is the only entry of this row
                    set entry $item
                    break
                }
            }
            set w [ttk::button $PARENT.ctext$QUESTION {*}$configure_options]
            set tkwidgedprivbutton($QUESTION,widget) $w
            grid $tkwidgedprivbutton($QUESTION,widget) -row [expr $ROW-1] -column 1 -sticky ew -columnspan 2
            if { $entry != "" } {
                grid remove $entry
            } else {
                #assumed that entry is hidden
                grid remove $tkwidgedprivbutton($QUESTION,widget)
            }
        }
        SYNC {
            # do nothing in this case
        }
        DEPEND {
            lassign $args GDN STRUCT QUESTION ACTION VALUE
            if { [info exists tkwidgedprivbutton($QUESTION,widget)] && [winfo exists $tkwidgedprivbutton($QUESTION,widget)] } {
                if { $ACTION == "HIDE" } {
                    grid remove $tkwidgedprivbutton($QUESTION,widget)
                } else {
                    #RESTORE
                    grid $tkwidgedprivbutton($QUESTION,widget)
                }
            } else {

            }
        }
        CLOSE {
            array unset tkwidgedprivbutton
        }
        default {
            return [list ERROR [_ "Unexpected tkwidget event"]]
        }
    }
    #a tkwidget procedure must return "" if Ok or [list ERROR $description] or [list WARNING $description]
    return ""
}

#to be used by TKWIDGET to set the entry configure options like width
#e.g.
#QUESTION: your_question
#VALUE:
#TKWIDGET: GidUtils::TkwidgetEntryConfigure {-width 20}
#Note: this procedure has en extra argument 'configure_options' before the standard ones 'event' and 'args' of all tkwidged-like procedures
proc GidUtils::TkwidgetEntryConfigure { configure_options event args } {
    switch $event {
        INIT {
            lassign $args PARENT current_row_variable GDN STRUCT QUESTION
            upvar $current_row_variable ROW
            #initialize variable to current field value
            #set value [DWLocalGetValue $GDN $STRUCT $QUESTION]
            set entry ""
            foreach item [grid slaves $PARENT -row [expr $ROW-1]] {
                if { [winfo class $item] == "Entry"  || [winfo class $item] == "TEntry" } {
                    #assumed that it is the only entry of this row
                    set entry $item
                    break
                } elseif { [winfo class $item] == "TFrame" } {
                    #in case of units there is a frame with a label and a canvas for the unit
                    foreach sub_item [winfo children $item] {
                        if { [winfo class $sub_item] == "Entry"  || [winfo class $sub_item] == "TEntry" } {
                            #assumed that it is the only entry of this row
                            set entry $sub_item
                            break
                        }
                    }
                    if { $entry != "" } {
                        break
                    }
                }
            }
            if { $entry != "" && [winfo exists $entry] && [info exists configure_options] } {
                $entry configure {*}$configure_options
            }
        }
        SYNC {
        }
        DEPEND {
        }
        CLOSE {
        }
        default {
            return [list ERROR [_ "Unexpected tkwidget event"]]
        }
    }
    #a tkwidget procedure must return "" if Ok or [list ERROR $description] or [list WARNING $description]
    return ""
}

proc GidUtils::wcb_checkEntryForIntGreaterZero {w idx str} {
    package require wcb
    set newText [wcb::postInsertEntryText $w $idx $str]
    if { ![string is integer $newText] || $newText<=0 } {
        wcb::cancel
    }
}

#to be used by TKWIDGET to validate the entry value
#e.g.
#QUESTION: your_question
#VALUE:
#TKWIDGET: GidUtils::TkwidgetEntryValidate {wcb::checkEntryLen 10} wcb::checkEntryForInt
#Note: this procedure has en extra argument 'check_callbacks' before the standard ones 'event' and 'args' of all tkwidged-like procedures
#predefined wcb package check callbacks:
# wcb::checkStrForRegExp <regular_expression>
# wcb::checkStrForAlpha
# wcb::checkStrForNum
# wcb::checkStrForAlnum
# wcb::convStrToUpper
# wcb::convStrToLower
# wcb::checkEntryForInt
# wcb::checkEntryForUInt <max> (* for no max limit)
# wcb::checkEntryForReal
# wcb::checkEntryForFixed <num_decimals> (* for no limit decimals)
# wcb::checkEntryLen <len>
#
#it is possible to define an use other validations, like
# GidUtils::wcb_checkEntryForIntGreaterZero

proc GidUtils::TkwidgetEntryValidate { check_callbacks event args } {
    switch $event {
        INIT {
            package require wcb
            lassign $args PARENT current_row_variable GDN STRUCT QUESTION
            upvar $current_row_variable ROW
            #initialize variable to current field value
            #set value [DWLocalGetValue $GDN $STRUCT $QUESTION]
            set entry ""
            foreach item [grid slaves $PARENT -row [expr $ROW-1]] {
                if { [winfo class $item] == "Entry"  || [winfo class $item] == "TEntry" } {
                    #assumed that it is the only entry of this row
                    set entry $item
                    break
                } elseif { [winfo class $item] == "TFrame" } {
                    #in case of units there is a frame with a label and a canvas for the unit
                    foreach sub_item [winfo children $item] {
                        if { [winfo class $sub_item] == "Entry"  || [winfo class $sub_item] == "TEntry" } {
                            #assumed that it is the only entry of this row
                            set entry $sub_item
                            break
                        }
                    }
                    if { $entry != "" } {
                        break
                    }
                }
            }
            if { $entry != "" && [winfo exists $entry] && [info exists check_callbacks] } {
                wcb::callback $entry before insert {*}$check_callbacks
            }
        }
        SYNC {
        }
        DEPEND {
        }
        CLOSE {
        }
        default {
            return [list ERROR [_ "Unexpected tkwidget event"]]
        }
    }
    #a tkwidget procedure must return "" if Ok or [list ERROR $description] or [list WARNING $description]
    return ""
}


proc GidUtils::IsTkDisabled { } {
    return [GidUtils_IsTkDisabled]
}

proc GidUtils::AreWindowsDisabled { } {
    if { [GidUtils_IsTkDisabled] } {
        return 1
    }
    if { [GiD_Project set disable_windows] } {
        return 1
    }
    if { ![winfo exists .gid] } {
        return 1
    }
    return 0
}

#to check if GiD is minimized
#to avoid that the program cannot be de-iconized if an event shows a window when minimized
proc GidUtils::IsStateIconic { } {
    set is_state_iconic 0
    if { [winfo exists .gid] && [wm state .gid] == "iconic" } {
        #states: normal, iconic, withdrawn, icon, or (Windows and Mac OS X only) zoomed
        set is_state_iconic 1
    }
    return $is_state_iconic
}

proc GidUtils::ExistsMesh { } {
    return [expr [GiD_Info Mesh MaxNumNodes]?1:0]
}

#expected tk color as
#   #RGB
#   #RRGGBB
#   #RRRGGGBBB
#   #RRRRGGGGBBBB
#with R G B a single hexadecimal digit
#allow also gid trick, with an extra alpha value with syntax
#   #RRGGBBAA
proc GidUtils::IsValidTkColor { color } {
    set is_valid 1
    if  { [string index $color 0] == "#" } {
        set length [string length $color]
        if { $length != 4 && $length != 7 && $length != 10 && $length != 13} {
            if { $length == 9 } {
                #maybe is a gid trick, with an extra alpha value with syntax #RRGGBBAA, allow it
            } else {
                set is_valid 0
            }
        }
        set n [scan $color #%2x%2x%2x r g b]
        if { $n == 3} {
            foreach color_component [list $r $g $g] {
                if { $color_component<0 || $color_component>255} {
                    set is_valid 0
                    break
                }
            }
        } else {
            set is_valid 0
        }
    } else {
        set is_valid 0
    }
    return $is_valid
}

#change from GiD color varible style rrr#ggg#bbb (xxx from 0 to 255)
#to Tk style #rrggbb hexadecimals
proc GidUtils::GiDColorToTkColor { colorgid } {
    set result ""
    set n_cols [scan $colorgid %03d#%03d#%03d r g b]
    if { $n_cols != 3} {
        set result $colorgid
        W "GidUtils::GiDColorToTkColor could not scan color $colorgid"
    } else {
        set result #[format %02x%02x%02x $r $g $b]
    }
    return $result
}

#change from Tk style color #rrggbb hexadecimals
#to GiD color varible style rrr#ggg#bbb (xxx from 0 to 255)
proc GidUtils::TkColorToGiDColor { tkcolor } {
    set n_cols [scan $tkcolor #%2x%2x%2x r g b]
    if { $n_cols != 3} {
        # default values for when the 'scan' fails
        set r 255
        set g 0
        set b 0
        W "GidUtils::TkColorToGiDColor could not scan color $tkcolor as #%2x%2x%2x . Using red."
    }
    return [format %03d#%03d#%03d $r $g $b]
}

# change the color of a widget asking the user
proc GidUtils::ChangeWidgetColor { w colvar} {
    upvar $colvar UserColor
    set w_par $w
    if { $w == "" } {
        set w_par .gid
    }
    set color_old $UserColor
    set color_new [tk_chooseColor -parent $w_par -initialcolor $color_old]
    if { $color_new != ""} {
        if { [GidUtils::IsValidTkColor $color_new] } {
            if { $w != ""} {
                $w configure -background $color_new
            }
            set UserColor $color_new
        }
    }
}


#get list of integers increasing from start to end
proc GidUtils::GetListIncreasingFromTo { start end } {
    return [objarray new_from_to intarray $start $end]
}

#get list of n equi-spaced real values from a to b
proc GidUtils::GetRange { a b n } {
    set increment [expr {double($b-$a)/($n-1)}]
    return [objarray new_n_from_increment doublearray $n $a $increment]
}

# combine the elements of several lists
# e.g. GidUtils::CartesianProduct {a b} {c} {d e}
#  -> {a c d} {a c e} {b c d} {b c e}
proc GidUtils::CartesianProduct { args } {
    set xs {{}}
    foreach ys $args {
        set result {}
        foreach x $xs {
            foreach y $ys {
                lappend result [list {*}$x $y]
            }
        }
        set xs $result
    }
    return $xs
}


#root_name is a name to check if not empty if it is an appropiated xml file: Infoproblemtype , DownloadProblemtype
#fields are the fieldnames we are interested in, if empty all Program fields are returned
#it is returned a list of field value
proc GidUtils::ReadProblemtypeXml { xmlfile root_name fields } {
    set res {}
    if { [file exists $xmlfile] && [file isfile $xmlfile]} {
        package require tdom
        set xml [tDOM::xmlReadFile $xmlfile]
        set doc [dom parse $xml]
        set root [$doc documentElement]
        if { $root_name == "" || [$root nodeName] == $root_name} {
            #set file_version [$root getAttribute version]
            if { $fields == "" } {
                foreach node [$root selectNodes Program/*] {
                    lappend fields [$node nodeName]
                }
            }
            foreach field $fields {
                foreach node [$root select Program/$field] {
                    if {$node ne ""} {
                        set value [$node text]
                    } else {
                        set value ""
                    }
                    lappend res $field $value
                }
            }
        }
        $doc delete
    }
    return $res
}

#trivial auxiliary file to store problemtype name and version with the model
proc GidUtils::SaveXmlWithNameAndVersion { filename problemtype_name problemtype_version } {
    set fp [open $filename w]
    if { $fp != "" } {
        puts $fp {<?xml version='1.0' encoding='utf-8'?><!-- -*- coding: utf-8;-*- -->}
        puts $fp "<$problemtype_name version='$problemtype_version'/>"
        #close $fp
        GidUtils::CatchClose $fp $filename
    }
}

#trivial auxiliary file to store problemtype name and version with the model
#return -1 if the xml is not an problemtype one or if the xml not exists
proc GidUtils::ReadXmlWithNameAndVersion { filename problemtype_name } {
    set model_problemtype_version_number -1
    if { [file exists $filename] } {
        set fp [open $filename r]
        if { $fp != "" } {
            set line ""
            gets $fp header
            gets $fp line ;#something like: <Nastran version='4.2'/>
            close $fp
            set line [string range $line 1 end-2]
            set model_problemtype [lindex $line 0]
            if { $model_problemtype == $problemtype_name } {
                set model_version [lindex $line 1]
                if { [lindex [split $model_version =] 0] == "version" } {
                    set model_problemtype_version_number [string range [lindex [split $model_version =] 1] 1 end-1]
                }
            } else {
                set model_problemtype_version_number -1
            }
        }
    }
    return $model_problemtype_version_number
}

#integrate the xy curve graph specified by its numeric index
#method could be "linear" (piecewise linear segments)
#             or "romberg" Romberg's intgration method of a interpolating cubic spline
proc GidUtils::IntegrateGraph { graphset_index graph_index {method linear} } {
    set integral 0
    set graphset_name [lindex [GiD_GraphSet list] $graphset_index]
    if { $graphset_name  != "" } {
        set graph_name [lindex [GiD_Graph list $graphset_name] $graph_index]
        if { $graph_name  != "" } {
            set graph_data [GiD_Graph get $graph_name $graphset_name]
            lassign $graph_data lx ly xs ys
            set integral [MathUtils::IntegrateXY $xs $ys $method]
            if { $integral == "" } {
                set integral 0
            }
        }
    }
    return $integral
}

# get the corners (z=0) of the bounding box of a graph, to be used by validation tests
# return an object BoundingBox, not a list of its pmin pmax values
# can get the values with the proc [BoundingBoxGet $bbox]
proc GidUtils::GiD_Graph_Get_Bounding_box { graphset_index graph_index } {
    package require gid_bounding_box
    set bbox [BoundingBox]
    set graphset_name [lindex [GiD_GraphSet list] $graphset_index]
    if { $graphset_name  != "" } {
        set graph_name [lindex [GiD_Graph list $graphset_name] $graph_index]
        if { $graph_name  != "" } {
            set graph_data [GiD_Graph get $graph_name $graphset_name]
            lassign $graph_data lx ly xs ys
            set min_x [objarray minimum $xs]
            set max_x [objarray maximum $xs]
            set min_y [objarray minimum $ys]
            set max_y [objarray maximum $ys]
            $bbox Set $min_x $min_y 0 $max_x $max_y 0
        }
    }
    return $bbox
}


# return an object BoundingBox, not a list of its pmin pmax values
# can get the values with the proc [BoundingBoxGet $bbox]
proc GidUtils::GiD_Sets_Get_Bounding_box { set_names {step_index ""} } {
    package require gid_bounding_box
    set bbox [BoundingBox]
    if { $step_index == "" } {
        set step_index [GiD_Info postprocess get cur_step_index]
    }
    lassign [GiD_Info Mesh -post -step_index $step_index Nodes -array2] node_ids_total node_coordinates_total
    foreach set_name $set_names {
        set node_ids_set [::GiD_EntitiesSets get $set_name nodes]
        # set index [expr ($node_id-1)] ;#do not require search assuming that node_ids_total is dense from 1 to n, but can have holes
        objarray foreach node_id $node_ids_set {
            set pos [expr {[objarray search -sorted $node_ids_total $node_id]*3}]
            $bbox Join {*}[objarray range $node_coordinates_total $pos [expr {$pos+2}]]
        }
    }
    return $bbox
}

# coordinates: objarray of x y z x y z ...
proc GidUtils::Coordinates_Get_Bounding_box { coordinates } {
    package require gid_bounding_box
    set bbox [BoundingBox]
    set n [objarray length $coordinates]
    for { set node_index 0 } { $node_index< $n } { incr node_index 3 } {
        $bbox Join {*}[objarray range $coordinates $node_index [expr $node_index+2]]
    }
    return $bbox
}

# for back compatibilty and to facilitate usage
# get the corners (z=0) of the bounding box of a graph, to be used by validation tests
proc GidUtils::BoundingBoxGraph { graphset_index graph_index } {
    return [BoundingBoxGet [GidUtils::GiD_Graph_Get_Bounding_box $graphset_index $graph_index]]
}

proc GidUtils::BoundingBoxCoordinates { coordinates } {
    return [BoundingBoxGet [GidUtils::Coordinates_Get_Bounding_box $coordinates]]
}

proc GidUtils::BoundingBoxSets { set_names {step_index ""} } {
    return [BoundingBoxGet [GidUtils::GiD_Sets_Get_Bounding_box $set_names $step_index]]
}

proc GidUtils::GetEntityCenter { entity_type entity_id } {
    return [GiD_Geometry get $entity_type $entity_id label_position]
}

#to get the 'apparent normal'
proc GidUtils::GetSurfaceNormal { surface_id } {
    set data [GiD_Info list_entities surfaces $surface_id]
    if { ![regexp {Normal: ([^\n]*)\n} $data dummy normal] } {
        error "GidUtils::GetSurfaceNormal $surface_id"
        set normal {0.0 0.0 0.0}
    }
    return $normal
}

#used for tester
proc GidUtils::GetMinMaxHigherEntities { entity_type } {
    lassign {10000 0} min_n max_n
    if { $entity_type == "node" } {
        set ids [GiD_Mesh list node]
        objarray foreach entity_id $ids {
            set num_higherentities [GiD_Mesh get node $entity_id higherentity]
            if {$min_n>$num_higherentities } {
                set min_n $num_higherentities
            }
            if {$max_n<$num_higherentities } {
                set max_n $num_higherentities
            }
        }
    } else {
        set ids [GiD_Geometry list $entity_type]
        objarray foreach entity_id $ids {
            set num_higherentities [GiD_Geometry get $entity_type $entity_id higherentity]
            if {$min_n>$num_higherentities } {
                set min_n $num_higherentities
            }
            if {$max_n<$num_higherentities } {
                set max_n $num_higherentities
            }
        }
    }
    return [list $min_n $max_n]
}

#return the id's list of higher entities
#entity_type: point line surface (a volume doesn't has higher entities)
#entity_id: integer that identify the entity
proc GidUtils::GetEntityHigherEntities { entity_type entity_id } {
    set data [GiD_Info list_entities -more ${entity_type}s $entity_id]
    if { ![regexp {Higher entities (.*): ([^\n]*)\n} $data dummy entity_type_parent higher_entities] } {
        set higher_entities ""
        #error  "GidUtils::GetEntityHigherEntities. Higher entities not found for entity_type $entity_type $entity_id"
    }
    return [objarray new intarray -values [string trim $higher_entities]]
}

#entity_type: points lines surfaces volumes
#key: Elemtype IsStructured Meshing size num_divisions weight tops SkipMesh BLMnlevels1 BLMnlevels2 BLMfirsth1 BLMfirsth2 Mesher
#   out codes
#   Elemtype:0 None 1 Linear 2 Triangle 3 Quadrilateral 4 Tetrahedra 5 Hexahedra 6 Prism 7 Only points 8 Pyramid 9 Sphere 10 Circle
#   Meshing: No Default Yes No,Duplicate Duplicate Default,Duplicate
#   SkipMesh: -1 No 0 Automatic 1 Yes
#   Mesher: 1 RFast 2 Rsurf 3 DelaunaySurf3 4 AdvancingFront4 5 MinElem 6 "Advancing front" 7 DelaunayVol7 8 Isosurface 9 Tetgen 10 Octree 11 PVolume11
#   weight: (w1,w2)
#   tops: semi-structured volumes codification (num_surface_1,num_surface_2),top surface global ids (if negative mean that is also set as master)
proc GidUtils::GetMeshData { entity_type entity_id {key ""}} {
    set result ""
    set data [GiD_Info list_entities $entity_type $entity_id]
    if { [regexp -line "Meshing Info: (.*)" $data dummy part_line] } {
        array set values [string map {= " "} [lrange $part_line 1 end]]
        if { $key != "" } {
            if { [info exists values($key)] } {
                set result $values($key)
            } else {
                set result ""
            }
        } else {
            set result [array get values]
        }
    }
    return $result
}

proc GidUtils::GetVolumeSurfaces { volume_id } {
    return [lindex [GiD_Geometry -v2 get volume $volume_id] 2]
}

proc GidUtils::GetSurfaceLines { surface_id } {
    return [lindex [GiD_Geometry -v2 get surface $surface_id] 3]
}

proc GidUtils::GetLinePoints { line_id } {
    return [objarray new intarray -values [lrange [GiD_Geometry -v2 get line $line_id] 2 3]]
}

# return the id's ordered list of unrepeated points of the lines
proc GidUtils::GetLinesPoints { line_ids } {
    set line_ids [GidUtils::UnCompactNumberList $line_ids]
    set num_lines [llength $line_ids]
    set points_ids [objarray new intarray [expr $num_lines*2]]
    set pos 0
    foreach line_id $line_ids {
        objarray set $points_ids $pos [GidUtils::GetLinePoints $line_id]
        incr pos 2
    }
    objarray sort -unique $points_ids
    return $points_ids
}

proc GidUtils::GetPointsCoordinates { point_ids } {
    set num_points [objarray length $point_ids]
    set num_coordinates [expr $num_points*3]
    set xyzs [objarray new doublearray $num_coordinates]
    set pos 0
    objarray foreach point_id $point_ids {
        objarray set $xyzs $pos [GiD_Geometry get point $point_id coordinates]
        incr pos 3
    }
    return $xyzs
}

proc GidUtils::GiD_Geometry_Edit_Point_Coordinates { point_id xyz } {
    GiD_Process MEscape Geometry Edit MovePoint $point_id [join $xyz ,] escape
}

#return the id's list of lower entities
#entity_type: line surface volume (a point doesn't has lower entities)
#entity_id: integer that identify the entity
proc GidUtils::GetEntityLowerEntities { entity_type entity_id } {
    if { $entity_type == "volume" } {
        set result [GidUtils::GetVolumeSurfaces $entity_id]
    } elseif { $entity_type == "surface" } {
        set result [GidUtils::GetSurfaceLines $entity_id]
    } elseif { $entity_type == "line" } {
        set result [GidUtils::GetLinePoints $entity_id]
    } else {
        error "GidUtils::GetEntityLowerEntities. Unexpected entity_type $entity_type $entity_id"
    }
    return $result
}

#return the amount of lower entities
#entity_type: line surface volume (a point doesn't has lower entities)
#entity_id: integer that identify the entity
proc GidUtils::GetEntityNumLowerEntities { entity_type entity_id } {
    if { $entity_type == "volume" } {
        set num_lower_entities_ids [objarray length [lindex [GiD_Geometry -v2 get $entity_type $entity_id] 2]]
    } elseif { $entity_type == "surface" } {
        set num_lower_entities_ids [objarray length [lindex [GiD_Geometry -v2 get $entity_type $entity_id] 3]]
    } elseif { $entity_type == "line" } {
        set num_lower_entities_ids 2
    } else {
        set num_lower_entities_ids 0
        error "GidUtils::GetEntityNumLowerEntities. Unexpected entity_type $entity_type $entity_id"
    }
    return $num_lower_entities_ids
}

#return the list of entities of higherentity==1 of the cathegory <entity_type>
#entity_type: point line surface (volume have always num_higherentity==0) | node
#num_higherentity: integer >=0
proc GidUtils::GetEntitiesFilterHigherentity { entity_type num_higherentity } {
    return [GidUtils::GetEntitiesFilterGeneric $entity_type HigherEntity $num_higherentity]
}

#return the list of entities of the cathegory <entity_type> filtered with filter=
#entity_type: point line surface volume | node element
#   filter_type filter_value
#   Higherentity <num_higherentity>
proc GidUtils::GetEntitiesFilterGeneric { entity_type filter_type filter_value } {
    GidUtils::DisableGraphics
    set old_disable_writebatch [GiD_Project set disable_writebatch]
    if { $old_disable_writebatch != 1 } {
        #try to hide the use of process by this proc to batch and Undo
        GiD_Project set disable_writebatch 1
    }
    GiD_Process 'SelectEntities ${entity_type}s Filter:$filter_type=$filter_value 1:end Escape
    if { $old_disable_writebatch != 1 } {
        GiD_Project set disable_writebatch $old_disable_writebatch
    }
    GidUtils::EnableGraphics
    return [GidUtils::UnCompactNumberList [lrange [string range $::GidPriv(selection) 1 end-1] 1 end]]
}

#filter surfaces with normal pointing to a direction (normal unitary) with tolerance
#surface_ids: list of ids of surfaces (e.g. [GiD_Geometry list surface])
proc GidUtils::GetSurfacesFilterNormal { surface_ids test_normal tolerance_deviation_angle_degrees } {
    set selected_surface_ids [list]
    set tolerance_cos_angle [expr cos([MathUtils::FromDegreeToRad $tolerance_deviation_angle_degrees])]
    foreach surface_id $surface_ids {
        set layer_name [lindex [GiD_Geometry -v2 get surface $surface_id] 1]
        if { ![GiD_Layers get frozen $layer_name] } {
            set surface_normal [GidUtils::GetSurfaceNormal $surface_id]
            set cos_angle [MathUtils::VectorDotProd $surface_normal $test_normal]
            if { $cos_angle >= $tolerance_cos_angle } {
                lappend selected_surface_ids $surface_id
            }
        }
    }
    return [objarray new intarray -values $selected_surface_ids]
}

#filter lines with tangent pointing to a direction (test_tangent unitary) with tolerance
#line_ids: list of ids of lines (e.g. [GiD_Geometry list line])
proc GidUtils::GetLinesFilterTangent { line_ids test_tangent tolerance_deviation_angle_degrees } {
    set selected_line_ids [list]
    set tolerance_cos_angle [expr cos([MathUtils::FromDegreeToRad $tolerance_deviation_angle_degrees])]
    foreach line_id $line_ids {
        set layer_name [lindex [GiD_Geometry -v2 get line $line_id] 1]
        if { ![GiD_Layers get frozen $layer_name] } {
            set line_tangent [MathUtils::VectorNormalized [GiD_Info parametric line $line_id deriv_t 0.5]]
            set cos_angle [MathUtils::VectorDotProd $line_tangent $test_tangent]
            if { $cos_angle >= $tolerance_cos_angle } {
                lappend selected_line_ids $line_id
            }
        }
    }
    return [objarray new intarray -values $selected_line_ids]
}

#element_level: line surface volume
#xyz list of x y z of the point to find the closest element (and its relative coordinates)
#element_id: the GiD element number
#return the relative coordinates and the distance2 from xyz to the projected point
#but the projection could lie outside the segment, triangle or tetrahedron
proc GidUtils::GetElementRelativeCoordinatesFromCoord { element_level xyz element_id } {
    package require math::linearalgebra
    set data [GiD_Mesh get element $element_id]
    set element_type [lindex $data 1]
    set element_nnodes [lindex $data 2]
    if { $element_type == "Linear" } {
        if { $element_level != "line" } {
            WarnWin [_ "A %s element must be clicked" $element_level]
            return
        }
        set element_connectivities [lrange $data 3 4]
        set cont 0
        foreach node_id $element_connectivities {
            set element_vertex($cont) [GiD_Mesh get node $node_id coordinates]
            incr cont
        }
        set v1 [math::linearalgebra::sub_vect $element_vertex(1) $element_vertex(0)]
        set op [math::linearalgebra::sub_vect $xyz $element_vertex(0)]

        set v11 [math::linearalgebra::dotproduct $v1 $v1]
        set v1p [math::linearalgebra::dotproduct $v1 $op]
        set u [expr {$v1p/$v11}]
        set op_projected [math::linearalgebra::scale_vect $u $v1]
        set op_normal [math::linearalgebra::sub_vect $op $op_projected]
        set d2 [math::linearalgebra::dotproduct $op_normal $op_normal]
        set res [format "%f %f" $u $d2]
    } elseif { $element_type == "Triangle" || $element_type == "Quadrilateral"} {
        if { $element_level != "surface" } {
            WarnWin [_ "A %s element must be clicked" $element_level]
            return
        }
        if { $element_type == "Triangle"} {
            set element_connectivities [lrange $data 3 5]
            set cont 0
            foreach node_id $element_connectivities {
                set element_vertex($cont) [::math::linearalgebra::mkVector 3]
                math::linearalgebra::setcol element_vertex($cont) 0 [GiD_Mesh get node $node_id coordinates]
                incr cont
            }
            set v1 [math::linearalgebra::sub_vect $element_vertex(1) $element_vertex(0)]
            set v2 [math::linearalgebra::sub_vect $element_vertex(2) $element_vertex(0)]
        } elseif { $element_type == "Quadrilateral"} {
            set element_connectivities [lrange $data 3 6]
            set cont 0
            foreach node_id $element_connectivities {
                set element_vertex($cont) [::math::linearalgebra::mkVector 3]
                math::linearalgebra::setcol element_vertex($cont) 0 [GiD_Mesh get node $node_id coordinates]
                incr cont
            }
            set v1 [math::linearalgebra::sub_vect $element_vertex(1) $element_vertex(0)]
            set v2 [math::linearalgebra::sub_vect $element_vertex(3) $element_vertex(0)]
        }
        set op [math::linearalgebra::sub_vect $xyz $element_vertex(0)]
        set v11 [math::linearalgebra::dotproduct $v1 $v1]
        set v12 [math::linearalgebra::dotproduct $v1 $v2]
        set v22 [math::linearalgebra::dotproduct $v2 $v2]
        set matrix [::math::linearalgebra::mkMatrix 2 2]
        math::linearalgebra::setrow matrix 0 [list $v11 $v12]
        math::linearalgebra::setrow matrix 1 [list $v12 $v22]
        set v1p [math::linearalgebra::dotproduct $v1 $op]
        set v2p [math::linearalgebra::dotproduct $v2 $op]
        set bvect [::math::linearalgebra::mkVector 2]
        math::linearalgebra::setcol bvect 0 [list $v1p $v2p]
        lassign [::math::linearalgebra::solveGauss $matrix $bvect] u v
        set op_projected [math::linearalgebra::scale_vect $u $v1]
        set op_projected [math::linearalgebra::add $op_projected [math::linearalgebra::scale_vect $v $v2]]
        set op_normal [math::linearalgebra::sub_vect $op $op_projected]
        set d2 [math::linearalgebra::dotproduct $op_normal $op_normal]
        set res [format "%f %f %f" $u $v $d2]
    } elseif { $element_type == "Tetrahedra" || $element_type == "Hexahedra" || $element_type == "Prism" || $element_type == "Pyramid"} {
        if { $element_level != "volume" } {
            WarnWin [_ "A %s element must be clicked" $element_level]
            return
        }
        if { $element_type == "Tetrahedra"} {
            set element_connectivities [lrange $data 3 6]
            set cont 0
            foreach node_id $element_connectivities {
                set element_vertex($cont) [::math::linearalgebra::mkVector 3]
                math::linearalgebra::setcol element_vertex($cont) 0 [GiD_Mesh get node $node_id coordinates]
                incr cont
            }
            set v1 [math::linearalgebra::sub_vect $element_vertex(1) $element_vertex(0)]
            set v2 [math::linearalgebra::sub_vect $element_vertex(2) $element_vertex(0)]
            set v3 [math::linearalgebra::sub_vect $element_vertex(3) $element_vertex(0)]
        } elseif { $element_type == "Hexahedra"} {
            set element_connectivities [lrange $data 3 10]
            set cont 0
            foreach node_id $element_connectivities {
                set element_vertex($cont) [::math::linearalgebra::mkVector 3]
                math::linearalgebra::setcol element_vertex($cont) 0 [GiD_Mesh get node $node_id coordinates]
                incr cont
            }
            set v1 [math::linearalgebra::sub_vect $element_vertex(1) $element_vertex(0)]
            set v2 [math::linearalgebra::sub_vect $element_vertex(3) $element_vertex(0)]
            set v3 [math::linearalgebra::sub_vect $element_vertex(4) $element_vertex(0)]
        } elseif { $element_type == "Prism"} {
            set element_connectivities [lrange $data 3 8]
            set cont 0
            foreach node_id $element_connectivities {
                set element_vertex($cont) [::math::linearalgebra::mkVector 3]
                math::linearalgebra::setcol element_vertex($cont) 0 [GiD_Mesh get node $node_id coordinates]
                incr cont
            }
            set v1 [math::linearalgebra::sub_vect $element_vertex(1) $element_vertex(0)]
            set v2 [math::linearalgebra::sub_vect $element_vertex(3) $element_vertex(0)]
            set v3 [math::linearalgebra::sub_vect $element_vertex(2) $element_vertex(0)]
        } elseif { $element_type == "Pyramid"} {
            set element_connectivities [lrange $data 3 7]
            set cont 0
            foreach node_id $element_connectivities {
                set element_vertex($cont) [::math::linearalgebra::mkVector 3]
                math::linearalgebra::setcol element_vertex($cont) 0 [GiD_Mesh get node $node_id coordinates]
                incr cont
            }
            set v1 [math::linearalgebra::sub_vect $element_vertex(1) $element_vertex(0)]
            set v2 [math::linearalgebra::sub_vect $element_vertex(3) $element_vertex(0)]
            set v3 [math::linearalgebra::sub_vect $element_vertex(4) $element_vertex(0)]
        }
        set op [math::linearalgebra::sub_vect $xyz $element_vertex(0)]
        set v11 [math::linearalgebra::dotproduct $v1 $v1]
        set v12 [math::linearalgebra::dotproduct $v1 $v2]
        set v13 [math::linearalgebra::dotproduct $v1 $v3]
        set v22 [math::linearalgebra::dotproduct $v2 $v2]
        set v23 [math::linearalgebra::dotproduct $v2 $v3]
        set v33 [math::linearalgebra::dotproduct $v3 $v3]
        set matrix [::math::linearalgebra::mkMatrix 3 3]
        math::linearalgebra::setrow matrix 0 [list $v11 $v12 $v13]
        math::linearalgebra::setrow matrix 1 [list $v12 $v22 $v23]
        math::linearalgebra::setrow matrix 2 [list $v13 $v23 $v33]
        set v1p [math::linearalgebra::dotproduct $v1 $op]
        set v2p [math::linearalgebra::dotproduct $v2 $op]
        set v3p [math::linearalgebra::dotproduct $v3 $op]
        set bvect [::math::linearalgebra::mkVector 3]
        math::linearalgebra::setcol bvect 0 [list $v1p $v2p $v3p]
        lassign [::math::linearalgebra::solveGauss $matrix $bvect] u v w
        set op_projected [math::linearalgebra::scale_vect $u $v1]
        set op_projected [math::linearalgebra::add $op_projected [math::linearalgebra::scale_vect $v $v2]]
        set op_projected [math::linearalgebra::add $op_projected [math::linearalgebra::scale_vect $v $v3]]
        set op_normal [math::linearalgebra::sub_vect $op $op_projected]
        set d2 [math::linearalgebra::dotproduct $op_normal $op_normal]
        set res [format "%f %f %f %f" $u $v $w $d2]
    } else {
        set res ""
    }
    return $res
}

#element_level: line surface volume
#xyz list of x y z of the point to find the closest element
#element_ids: list of element ids of candidate elements to be tried
proc GidUtils::GetClosestElement { element_level xyz element_ids } {
    set min_element_id ""
    set d2_min 1e20
    foreach element_id $element_ids {
        set res [GidUtils::GetElementRelativeCoordinatesFromCoord $element_level $xyz $element_id]
        if { $element_level == "line" } {
            lassign $res t d2
            if { $t>=0.0 && $t<=1.0 } {
                if { $d2_min > $d2 } {
                    set d2_min $d2
                    set min_element_id $element_id
                }
            }
        } elseif { $element_level == "surface" } {
            lassign $res u v d2
            if { $u>=0.0 && $u<=1.0 && $v>=0.0 && $v<=1.0  } {
                if { $d2_min > $d2 } {
                    set d2_min $d2
                    set min_element_id $element_id
                }
            }
        } elseif { $element_level == "volume" } {
            lassign $res u v w d2
            if { $u>=0.0 && $u<=1.0 && $v>=0.0 && $v<=1.0  && $w>=0.0 && $w<=1.0 } {
                if { $d2_min > $d2 } {
                    set d2_min $d2
                    set min_element_id $element_id
                }
            }
        }
    }
    return $min_element_id
}

#names of possible GiD elemements
proc GidUtils::GetElementTypes { level } {
    if { $level == "all" } {
        set element_types {Linear Triangle Quadrilateral Tetrahedra Hexahedra Prism Point Pyramid Sphere Circle}
    } elseif { $level == "point" } {
        set element_types {Point}
    } elseif { $level == "line" } {
        set element_types {Linear}
    } elseif { $level == "surface" } {
        set element_types {Triangle Quadrilateral Circle}
    } elseif { $level == "volume" } {
        set element_types {Tetrahedra Hexahedra Prism Pyramid Sphere}
    } else {
        error "unexpected level=$level"
    }
    return $element_types
}

proc GidUtils::GetElementTypesLowercase { level } {
    set element_types_lowercase [list]
    foreach element_type [GidUtils::GetElementTypes $level] {
        lappend element_types_lowercase [string tolower $element_type]
    }
    return $element_types_lowercase
}

proc GidUtils::CreateSurfaceFromBoundaryLines { line_ids } {
    set max_id_surface_before [GiD_Info geometry MaxNumSurfaces]
    GiD_Process Mescape Geometry Create NurbsSurface {*}$line_ids escape escape escape escape
    set max_id_surface_after [GiD_Info geometry MaxNumSurfaces]
    set num_created_surfaces [expr $max_id_surface_after-$max_id_surface_before]
    if { $num_created_surfaces == 1 } {
        set isurface $max_id_surface_after
    } else {
        set isurface ""
    }
    return $isurface
}

proc GidUtils::CreateVolumeFromBoundarySurfaces { surface_ids } {
    set max_id_volume_before [GiD_Info geometry MaxNumVolumes]
    GiD_Process Mescape Geometry Create Volume {*}$surface_ids escape escape escape escape
    set max_id_volume_after [GiD_Info geometry MaxNumVolumes]
    set num_created_volumes [expr $max_id_volume_after-$max_id_volume_before]
    if { $num_created_volumes == 1 } {
        set ivolume $max_id_volume_after
    } else {
        set ivolume ""
    }
    return $ivolume
}

#the main draw area could have multiple windows (each one is a single gid_togl)
proc GidUtils::GetMainDrawAreaWidget { } {
    if { [winfo exists .gid.central.wins] } {
        #frame with several gid_togls
        set w .gid.central.wins
    } else {
        #gid_togl
        set w .gid.central.s
    }
    return $w
}

#the main draw area could have multiple windows (each one is a single gid_togl)
proc GidUtils::GetMainDrawAreaSize { } {
    set widget [GidUtils::GetMainDrawAreaWidget]
    if { [winfo exists $widget] } {
        set w [winfo width $widget]
        set h [winfo height $widget]
    } else {
        set w 0
        set h 0
    }
    return [list $w $h]
}

proc GidUtils::SetMainDrawAreaSize { new_width new_height {check_values 1}} {
    set fail 0
    set widget [GidUtils::GetMainDrawAreaWidget]
    if { [winfo exists $widget] } {
        update ;#otherwise some tester case like 59AF24D35A4F7A3758ACD2A81E591883 doesn't has the required size
        set toplevel [winfo toplevel $widget]
        set offset_width [expr [winfo width $toplevel] - [winfo width $widget]]
        set offset_height [expr [winfo height $toplevel] - [winfo height $widget]]
        if { $check_values } {
            # Check values
            set max_width [expr [winfo vrootwidth .] - $offset_width]   ; # should substract the window borders too ...
            set max_height [expr [winfo vrootheight .] - $offset_height]   ; # should substract the window borders too ...
            if { $new_width > $max_width} {
                ::GidUtils::SetWarnLine \
                    [_ "Width %s greater than the maximum %s. Enter a smaller value, resize the layer and problemtype panels or close them." \
                    $new_width $max_width]
                set fail 1
            }
            if { $new_height > $max_height} {
                ::GidUtils::SetWarnLine \
                    [_ "Height %s greater than the maximum %s. Enter a smaller value, resize the layer and problemtype panels or close them." \
                    $new_height $max_height]
                set fail 1
            }
        }

        # do the resize
        if { !$fail} {
            wm geometry $toplevel [expr $offset_width + $new_width]x[expr $offset_height + $new_height]
            # move the Layer panel embedded accordingly
            if { [info exists ::GidPriv(pwCentral)] && [winfo exists $::GidPriv(pwCentral)] } {
                set wsash [lsearch [$::GidPriv(pwCentral) panes] $widget]
                if { $wsash < 0} {
                    set wsash 0
                }
                set new_sash_x [expr $new_width + 1]
                if { $wsash > 0} {
                    # there are more than one sash
                    set left_sash_coords [$::GidPriv(pwCentral) sash coord [expr $wsash - 1]]
                    set new_sash_x [expr $new_width + [lindex $left_sash_coords 0] + \
                        2 + [$::GidPriv(pwCentral) cget -sashwidth]]
                }
                update
                $::GidPriv(pwCentral) sash place $wsash $new_sash_x 0
            }
            update
        }
    } else {
        set fail 1
    }
    return $fail
}



# native types: "pure string" string int double list dict
#                cmdName parsedVarName localVarName path channel
# objarray types: chararray shortarray intarray longarray longlongarray floatarray doublearray
proc GidUtils::GetInternalRepresentation { x } {
    set type ""
    regexp {^value is a (.*?) with a refcount} [::tcl::unsupported::representation $x] -> type
    return $type
}

# [objarray type $pp] may return any of Tcl types ( string, path, etc...) or objarray_types or ""
# but this needs 'package require objarray'
# below version does not require a package require objarray
proc GidUtils::IsObjarray { x } {
    set objarray_types [ list chararray shortarray intarray longarray longlongarray floatarray doublearray ]
    set type [ GidUtils::GetInternalRepresentation $x ]
    if { [ lsearch $objarray_types $type] != -1 } {
        return 1
    } else {
        return 0
    }
}

proc GidUtils::AsObjarray { var_name obj_type } {
    upvar $var_name var_value
    set obj_array $var_value

    if { ![ GidUtils::IsObjarray $var_value]} {
	set true_type [ GidUtils::GetInternalRepresentation $var_value]
	W "Warning: '$var_name' is of type '$true_type', converting to 'objarray $obj_type'."
	if { [ llength $var_value] != 0} {
	    set obj_array [ objarray new $obj_type -values $var_value]
	} else {
	    # create objarray of size 0
	    set obj_array [ objarray new $obj_type 0]
	}
    }
    return $obj_array
}

#e.g. extension *.igs
proc GidUtils::GetFilesRecursive { dirname extension } {
    set names [glob -nocomplain -directory $dirname -types f $extension]
    foreach name [glob -nocomplain -directory $dirname -types d *] {
        lappend names {*}[GidUtils::GetFilesRecursive $name $extension]
    }
    return $names
}


proc GidUtils::ResetModel { } {
    foreach layer [GiD_Info layers] {
        GiD_Process 'Layers Delete $layer Yes escape escape
    }
    foreach group [GiD_Groups list] {
        if { [GiD_Groups exists $group] } {
            GiD_Groups delete $group
        }
    }
}

# result_group_name must not exist as group
# group_list is a list of group names that exist
# It will create a group and assign all the entities of the group_list
proc GidUtils::MergeGroups {result_group_name group_list} {
    GiD_Groups create $result_group_name
    foreach group $group_list {
        foreach entity [list points lines surfaces volumes nodes elements faces] {
            GiD_EntitiesGroups assign $result_group_name $entity [GiD_EntitiesGroups get $group $entity]
        }
    }
}

# Rhinoceros use :: as tree separator for layers (like GiD use of a//b)
# create the tree structure based on Rhino names
proc GidUtils::LayersTreeStructureFromRhinoNames { } {
    set changes 0
    foreach layername [GiD_Layers list] {
        set name_split [GidUtils::Split $layername ::]
        if { [llength $name_split] > 1 } {
            set full_parent_name [Layers::NameJoin {*}[lrange $name_split 0 end-1]]
            if { ![GiD_Layers exists $full_parent_name] } {
                GiD_Layers create $full_parent_name ;#this create parent layers if necessary
            }
            GiD_Layers edit parent $layername $full_parent_name
            set new_layername [Layers::NameJoin $full_parent_name $layername]
            GiD_Layers edit name $new_layername [lindex $name_split end]
            incr changes
        }
    }
    if { $changes } {
        GidUtils::UpdateWindow LAYER
    }
}

proc GidUtils::DeleteteLayersEmpty { } {
    foreach layer_name [GiD_Layers list] {
        if { [lindex [GiD_Info Layers -canbedeleted $layer_name] 0] == 1 } {
            GiD_Layers delete $layer_name
        }
    }
}

proc GidUtils::SeparateTopGeometryInLayers { } {
    GidUtils::DisableGraphics
    foreach entity_type {point line surface volume} {
        set prefix [string toupper [string index ${entity_type} 0]]_
        foreach entity_id [GiD_Geometry list -higherentity 0 $entity_type] {
            set layer_name ${prefix}${entity_id}
            if { ![GiD_Layers exists $layer_name] } {
                GiD_Layers create $layer_name
            }
            GiD_EntitiesLayers assign $layer_name -also_lower_entities ${entity_type}s $entity_id
        }
    }
    GidUtils::DeleteteLayersEmpty
    GidUtils::EnableGraphics
    GiD_Redraw
    after idle [list GidUtils::UpdateWindow LAYER]
}

proc GidUtils::CreateLayersFromElementMaterials { } {
    set used_materials [list]
    foreach mesh [GiD_Info mesh -pre elements any -array] {
        lappend used_materials {*}[lsort -integer -unique [lindex $mesh 3]]
    }
    set used_materials [lsort -integer -unique $used_materials]
    if { [llength $used_materials] } {
        MeshView
        GidUtils::DisableGraphics
        foreach material_id $used_materials {
            set layer_name Mat_$material_id
            if { ![GiD_Layers exists $layer_name] } {
                GiD_Layers create $layer_name
                incr changes
            }
            set element_ids [GiD_Mesh list -material $material_id element]
            GiD_EntitiesLayers assign $layer_name -also_lower_entities elements $element_ids
        }
        GidUtils::DeleteteLayersEmpty
        GidUtils::EnableGraphics
        GiD_Redraw
        after idle [list GidUtils::UpdateWindow LAYER]
    }
}

#function to standarize and facilitate the units syntax
proc GiD_Units { mode key {value ""} } {
    set result ""
    set syntax "GiD_Units edit|get magnitude_units|magnitude_from_unit|magnitudes|model_unit_length|system ?<value>?"
    if { $mode == "edit" } {
        if { [GroupsXmlDocExists] } {
            if { $key == "model_unit_length" } {
                set length_units [GiD_Units get magnitude_units L]
                set pos [lsearch $length_units $value]
                if { $pos == -1 } {
                    error "valid units are: $length_units"
                }
                gid_groups_conds::set_mesh_unit $value
            } elseif { $key == "system" } {
                set sytems [list]
                foreach item [gid_groups_conds::give_units_system_list] {
                    lappend sytems [lindex $item 0]
                }
                if { [lsearch $sytems $value] == -1 } {
                    error "valid systems are: $sytems"
                }
                gid_groups_conds::set_active_units_system $value
            } else {
                error "Expected: $syntax"
            }
        } else {
            if { $key == "model_unit_length" } {
                set length_units [GiD_Units get magnitude_units LENGTH]
                set pos [lsearch $length_units $value]
                if { $pos == -1 } {
                    error "valid units are: $length_units"
                }
                lassign [MenuUnits::GetListsUnitsDataFromMenu LENGTH] UnitsList UInfoList
                set ::ModUnitWin::LengthInfo [lindex $UInfoList $pos]
                updmdlunits
            } elseif { $key == "system" } {
                set systems [GiD_Info unitssystems prj]
                if { [lsearch $systems $value] == -1 } {
                    error "valid systems are: $sytems"
                }
                set ::ModUnitWin::CurSysName $value
                updmdlunits
            } else {
                error "Expected: $syntax"
            }
        }
    } elseif { $mode == "get" } {
        if { [GroupsXmlDocExists] } {
            if { $key == "magnitude_units" } {
                #must check that value is a valid magnitude: L ...
                set magnitudes [GiD_Units get magnitudes]
                if { [lsearch $magnitudes $value] == -1 } {
                    error "valid magnitudes are: $magnitudes"
                }
                set result [list]
                foreach priority {2 3} {
                    lappend result {*}[gid_groups_conds::_give_units_list_do [list [list $value 1]] $priority]
                }
            } elseif { $key == "magnitudes" } {
                set result [list]
                foreach item [gid_groups_conds::give_unit_all_magnitudes] {
                    lappend result [lindex $item 0]
                }
            } elseif { $key == "model_unit_length" } {
                set result [gid_groups_conds::give_mesh_unit]
            } elseif { $key == "system" } {
                set result [lindex [gid_groups_conds::give_active_units_system] 0]
            } elseif { $key == "magnitude_from_unit" } {
                set unit_magnitude ""
                if { [catch { gid_groups_conds::give_unit_magnitude_and_factor $units } ret] } {
                    set unit_magnitude ""
                } else {
                    lassign $ret unit_magnitude unit_factor addend
                }
                set result $unit_magnitude
            } else {
                error "Expected: $syntax"
            }
        } else {
            if { $key == "magnitude_units" } {
                #must check that value is a valid magnitude: LENGTH ...
                set magnitudes [GiD_Units get magnitudes]
                if { [lsearch $magnitudes $value] == -1 } {
                    error "valid magnitudes are: $magnitudes"
                }
                lassign [MenuUnits::GetListsUnitsDataFromMenu $value] UnitsList UInfoList
                set result [MenuUnits::GetListUnits $UnitsList]
            } elseif { $key == "magnitudes" } {
                set result [GiD_Info magnitudes 1] ;#1==problemtype (0==GiD initially)
            } elseif { $key == "model_unit_length" } {
                lassign [GiD_Info unitssystems modunit] magnitude_name unit_index set_index
                set result [lindex [GiD_Info magnitudes $set_index $magnitude_name $unit_index] 1]
            } elseif { $key == "system" } {
                set result [GiD_Info unitssystems prbsys]
            } elseif { $key == "magnitude_from_unit" } {
                set unit_magnitude ""
                set ::out_global_variable ""
                set found [FindUnitGidAndPrjStr $value ::out_global_variable]
                if { $found } {
                    set unit_magnitude [lindex $::out_global_variable 0]
                } else {
                    set unit_magnitude ""
                }
                set result $unit_magnitude
            } else {
                error "Expected: $syntax"
            }
        }
    } else {
        error "Expected: $syntax"
    }
    return $result
}

#bandwidth is the maximum difference of numeration of nodes or each element
#prepost could be PRE or POST, or empty to use the current one
#return 3 integers: bandwidth, amount of elements with this bandwith and the id of an arbitrary element with this bandwidth
proc GiD_Mesh_GetBandWidth { {prepost ""} } {
    if { $prepost == ""} {
        set prepost [GetCurrentPrePostMode]
    }
    set band_width_max 0
    set i_element_max -1
    foreach mesh [GiD_Info mesh -[string tolower $prepost] elements Any -array] {
        set element_ids [lindex $mesh 1]
        set connectivities [lindex $mesh 2]
        set num_elements [objarray length $element_ids]
        set num_connectivities_element [llength $connectivities]
        for {set i_connectivity 0 } {$i_connectivity<$num_connectivities_element} {incr i_connectivity} {
            set connectivities_i($i_connectivity) [lindex $connectivities $i_connectivity]
        }
        set element_nodes [objarray new intarray $num_connectivities_element]
        for {set i_element 0 } {$i_element<$num_elements} {incr i_element} {
            for {set i_connectivity 0 } {$i_connectivity<$num_connectivities_element} {incr i_connectivity} {
                objarray set $element_nodes $i_connectivity [objarray get $connectivities_i($i_connectivity) $i_element]
            }
            set min [objarray minimum $element_nodes]
            set max [objarray maximum $element_nodes]
            set local_band_width [expr $max-$min]
            if { $band_width_max <= $local_band_width } {
                incr num_elements_max($local_band_width)
                if { $band_width_max < $local_band_width } {
                    set band_width_max $local_band_width
                    set i_element_max $i_element
                }
            }
        }
    }
    set id_element_max [objarray get $element_ids $i_element_max]
    return [list $band_width_max $num_elements_max($band_width_max) $id_element_max]
}

# https://en.wikipedia.org/wiki/Genus_(mathematics)
#here the  mesh is expected to be closed and made of triangles
proc GidUtils::MeshTrianglesGetNumGenus { } {
    set num_genus ""
    set num_elements_total [GiD_Info mesh NumElements]
    set num_triangle [GiD_Info mesh NumElements triangle]
    if { $num_elements_total==$num_triangle } {
        # must check that the mesh is closed (each edge is shared by 2 triangles, all edge ehigherentities ==2)
        set num_nodes [GiD_Info mesh NumNodes]
        set num_genus [expr ($num_triangle-2*$num_nodes+4)/4]
    } else {
        # implemented only for triangles
    }
    return $num_genus
}

#e.g. to try to delete Layer0 after import if empty
proc DeleteLayerEmpty { layer } {
    if { [llength [GiD_Layers list]] >1 } {
        if { [GiD_Layers exists $layer] } {
            set num_items 0
            foreach n [concat [GiD_EntitiesLayers get $layer all_geometry -count]  [GiD_EntitiesLayers get $layer all_mesh -count]] {
                incr num_items $n
            }
            if { !$num_items } {
                GiD_Layers delete $layer
            }
        }
    }
}

proc CreateMeshFromRenderMeshOfSurfacesSelection { } {
    set surface_ids [GidUtils::PickEntities Surfaces multiple [_ "Select surfaces to create mesh from its render mesh"]]
    foreach surface_id [GidUtils::UnCompactNumberList $surface_ids] {
        set layer [lindex [GiD_Geometry -v2 get surface $surface_id] 1]
        lassign [GiD_Geometry get surface $surface_id render_mesh] elemtype elementnnodes nodes elements normals uvs
        GiD_MeshPre_Create $elemtype $elementnnodes "" $nodes "" $elements 1 $layer
    }
    GidUtils::SetWarnLine [_ "Created mesh from render mesh of %s surfaces" [llength $surface_ids]]
    MeshView
}

proc CreateMeshFromRenderMeshOfLinesSelection { } {
    set line_ids [GidUtils::PickEntities Lines multiple [_ "Select lines to create mesh from its render mesh"]]
    foreach line_id [GidUtils::UnCompactNumberList $line_ids] {
        set layer [lindex [GiD_Geometry -v2 get line $line_id] 1]
        lassign [GiD_Geometry get line $line_id render_mesh] elemtype elementnnodes nodes elements
        GiD_MeshPre_Create $elemtype $elementnnodes "" $nodes "" $elements 1 $layer
    }
    GidUtils::SetWarnLine [_ "Created mesh from render mesh of %s lines" [llength $line_ids]]
    MeshView
}

#temporary implementation until "GiD_Sets exists" subcommand is implemented
proc GiD_Sets_exists { set_id } {
    set exists 0
    if { [lsearch [GiD_Info postprocess get all_volumesets] $set_id] != -1 } {
        set exists 1
    } elseif { [lsearch [GiD_Info postprocess get all_surfacesets] $set_id] != -1 } {
        set exists 1
    } elseif { [lsearch [GiD_Info postprocess get all_cutsets] $set_id] != -1 } {
        set exists 1
    }
    return $exists
}

#temporary implementation until "GiD_Sets list" subcommand is implemented
proc GiD_Sets_list { } {
    set sets [list]
    foreach type [list Surface Volume Cut] {
        foreach set_name [GiD_Info postprocess get all_${type}sets] {
            lappend sets $set_name
        }
    }
    return $sets
}

#create in preprocess a mesh from a set of postprocess, if layer_id is "" use same layer name as the set name
proc GidUtils::CreateMeshFromSet { set_id {layer_id ""} } {
    #set t0 [clock milliseconds]
    if { ![GiD_Sets_exists $set_id] } {
        return 1
    }
    set layer_to_use_prev [GiD_Layers get to_use]
    if { $layer_id == "" } {
        set layer_id $set_id
    }
    if { ![GiD_Layers exists $layer_id] } {
        GiD_Layers create $layer_id
    }
    GiD_Layers edit to_use $layer_id
    set current_step_index [GiD_Info postprocess get cur_step_index]
    lassign [GiD_Info Mesh -post -step_index $current_step_index Nodes -array2] node_ids_total node_coordinates_total
    set node_ids_set [::GiD_EntitiesSets get $set_id nodes]
    set num_nodes_set [objarray length $node_ids_set]
    for { set i 0 } { $i< $num_nodes_set } {incr i } {
        set node_id [objarray get $node_ids_set $i]
        set index [objarray search -sorted $node_ids_total $node_id]
        #set index [expr ($node_id-1)] ;#do not require search assuming that node_ids_total is dense from 1 to n, but can have holes
        set pos [expr $index*3]
        set node_xyz [objarray range $node_coordinates_total $pos [expr $pos+2]]
        set NewNodeId($node_id) [GiD_Mesh create node append $node_xyz]
    }
    set element_ids_set [::GiD_EntitiesSets get $set_id elements]
    set num_elements_set [objarray length $element_ids_set]
    if { $num_elements_set } {
        set element_type [GiD_Sets get element_type $set_id]
        set element_nnode [GiD_Sets get element_num_nodes $set_id]
        set mest_data_several_sets [GiD_Info Mesh -post -step_index $current_step_index -set_name $set_id Elements $element_type -array2]
        if { [llength $mest_data_several_sets] == 1 } {
            set mesh_data [lindex $mest_data_several_sets 0]
            lassign $mesh_data element_type_2 element_ids_this_type element_connectivities_this_type
            set new_element_node_ids [objarray new intarray $element_nnode]
            for { set j 0 } { $j< $num_elements_set } {incr j } {
                set element_id [objarray get $element_ids_set $j]
                set index [objarray search -sorted $element_ids_this_type $element_id]
                #set index [expr ($element_id-1)] ;#do not require search assuming that element_ids_total is dense from 1 to n
                set pos [expr $index*$element_nnode]
                set element_node_ids [objarray range $element_connectivities_this_type $pos [expr $pos+$element_nnode-1]]
                for { set i_element_node 0 } { $i_element_node< $element_nnode } {incr i_element_node } {
                    set node_id [objarray get $element_node_ids $i_element_node]
                    objarray set $new_element_node_ids $i_element_node $NewNodeId($node_id)
                }
                GiD_Mesh create element append $element_type $element_nnode $new_element_node_ids
            }
        }
    }
    GiD_Layers edit to_use $layer_to_use_prev
    #set t1 [clock milliseconds]
    #W "Time GidUtils::CreateMeshFromSet [expr {($t1-$t0)/1000.0}] seconds"
    return 0
}

#create in preprocess a mesh from all sets of postprocess
proc GidUtils::CreateMeshFromPostMesh { } {
    #set t0 [clock milliseconds]
    foreach type [list Surface Volume Cut] {
        foreach set_id [GiD_Info postprocess get all_${type}sets] {
            GidUtils::CreateMeshFromSet $set_id
        }
    }
    #set t1 [clock milliseconds]
    #W "Time GidUtils::CreateMeshFromPostMesh  [expr {($t1-$t0)/1000.0}] seconds"
}

#GiD_Result get return more components that it accepts for GiD_Result create
#avoid return values of redundant components, like module of vector, or main values of matrix
#it would be better for efficiency if the C++ function was changed (at least with a flag to preserve back-compatibility)
#to return the exact values that must be used for GiD_Result create
proc GiD_ResultGetArrayOnlyStrictComponents { result_id } {
    set res_info [GiD_Result get -array $result_id]
    set type [lindex [lindex $res_info 0] 4]
    lassign [lindex $res_info 3] ids values
    set num_components [llength $values]
    if { $type == "Vector" } {
        if { $num_components != 3 } {
            lset res_info {3 1} [lrange $values 0 2]
        }
    } elseif { $type == "Matrix" } {
        if { $num_components != 6 } {
            lset res_info {3 1} [lrange $values 0 5]
        }
    } elseif { $type == "PlainDeformationMatrix" } {
        if { $num_components != 4 } {
            lset res_info {3 1} [lrange $values 0 3]
        }
    } elseif { $type == "ComplexVector" } {
        #it is returning 12, a lot of combination of modules, without practical intesest in my opinion!!
        #        X-rE         X-iE         Y-rE         Y-iE         Z-rE         Z-iE         |rE|         |iE|         mod(E)         |E|-X         |E|-Y         |E|-Z
        if { $num_components != 6 } {
            lset res_info {3 1} [lrange $values 0 5]
        }
    } elseif { $type == "ComplexMatrix" } {
        if { $num_components != 12 } {
            lset res_info {3 1} [lrange $values 0 11]
        }
    } elseif { $type == "LocalAxes" } {
        #3 euler angles
        if { $num_components != 3 } {
            lset res_info {3 1} [lrange $values 0 2]
        }
    }
    return $res_info
}

#to facilitate do this test
# which may be : all, tcltk (default), tcl, tk, import (pre and post import dll)
proc test_packages { { which_ones tcltk} { messages_in_stdout 0}} {
    source [ file join $::GIDDEFAULT scripts gid_filesystem.tcl] ; #trick because this file has procs defined in two ways depending on an if { $::gid_start_mode} expression, and the procs doesn't appear in tclIndex.tcl
    source [ file join [ gid_filesystem::get_folder_standard scripts] tests basic_test.tcl]
    do_test_packages $which_ones $messages_in_stdout
}

proc test_import {} {
    test_packages import
}

proc test_fps { { num_redraws 0} } {
    source [ file join [gid_filesystem::get_folder_standard scripts] tests basic_test.tcl]
    do_test_fps $num_redraws
}


#window dialog examples
#
#get a filename
#   set cathegory file
#to write
#   set mode write
#   set title [= "Write IGES file"]
#   set initial_filename ""
#   set file_types [list [list [= "IGES file"] ".igs .iges"] [list [= "All files"] ".*"]]
#   set default_extension .igs
#   set multiple 0
#   set extra_options [list]
#   set filename_write [MessageBoxGetFilename $cathegory $mode $title $initial_filename $file_types $default_extension $multiple extra_options]
#to read
#   set mode read
#   set title [= "Read IGES file"]
#   set filename_read [MessageBoxGetFilename $cathegory $mode $title $initial_filename $file_types $default_extension $multiple extra_options]



#user procedures that modify menu variables could be registered to be automatically called
#procname is a procedure without any argument
#GiD_RegisterPluginAddedMenuProc <procname>
#GiD_UnRegisterPluginAddedMenuProc <procname>

#user procedures to handle some file extension when dropping files on GiD
#prepost could be: PRE POST or PREPOST
#procname is a procedure with an argument, the name of the dropped file that match the extension
#GiD_RegisterExtensionProc <list of extensions> <procname> <prepost>
#GiD_UnRegisterExtensionProc <list of extensions> <prepost>
#proc MyUnvRead { file } { ... }
#GiD_RegisterExtensionProc ".unv .uff" PRE MyUnvRead


proc GidUtils::HexDump { str_in} {
    # in chunks of 16 bytes:
    set chunk 16
    set chunk_1 [expr $chunk - 1]        ;# 15
    set chunk2_1 [expr $chunk * 2 - 1]   ;# 31
    set chunk_2 [expr $chunk / 2]        ;#  8
    set chunk_2_1 [expr $chunk / 2 - 1]  ;#  7
    set ini 0
    set end $chunk
    set len_str [string length $str_in]
    set str_out ""
    set i_sep 0
    while { $ini < $len_str} {
        # does not matter if $end > $len_str
        set s [string range $str_in $ini $end]

        # Convert the data to hex and to characters.
        binary scan $s H*@0a* hex ascii

        # Replace non-printing characters in the data.
        regsub -all -- {[^[:graph:] ]} $ascii {.} ascii

        # Split the 16 bytes into two 8-byte chunks
        # set hex1   [string range $hex   0 15]
        # set hex2   [string range $hex  16 31]
        # set ascii1 [string range $ascii 0  7]
        # set ascii2 [string range $ascii 8 16]
        set hex1   [string range $hex   0 $chunk_1]
        set hex2   [string range $hex  $chunk $chunk2_1]
        set ascii1 [string range $ascii 0  $chunk_2_1]
        set ascii2 [string range $ascii $chunk_2 $chunk]

        # Convert the hex to pairs of hex digits
        regsub -all -- {..} $hex1 {& } hex1
        regsub -all -- {..} $hex2 {& } hex2

        # Put the hex and Latin-1 data to the channel
        set sep |
        incr i_sep
        if { $i_sep == 4} {
            set sep +
            set i_sep 0
        }
        append str_out [format "%08x $sep %-24s${sep} %-24s${sep} %-8s%-8s\n" $ini $hex1 $hex2 $ascii1 $ascii2]
        incr ini $chunk
        incr end $chunk
    }
    return $str_out
}

#to visually show thousands separated by a character like ,
#e.g. change 12345 by 12,345
proc GidUtils::AddThousandsSeparator { str { sep ,}} {
    # let's hide whats right at the decimal point
    while { [regsub {(.*\d\.(\d\d\d)+)(\d+.*)$} $str "\\1|!|!|!|\\3" str]} {
    }
    while { [regsub {^(.*[-+]?\d+)(\d\d\d)} $str "\\1$sep\\2" str]} {
    }
    regsub -all {\|\!\|\!\|\!\|} $str {} str
    return $str
}


#to find the elements of nodes of bad edges (neighbor elements with normal opposite, typical after edge-collapse, etc.)
proc GidUtils::GetListElementsFlipped { {normals_dotprod_threshold -0.9} } {
    # get the faces (are like edges 2-nodes) of neightbor triangles and/or quadrilaterals with normals near
    # opposite (dot_product < threshold). Normalized normals: the value could be in range [-1.0,1.0]
    lassign [GiD_Mesh list -normals_dotprod_threshold $normals_dotprod_threshold face] element_ids faces_index
    set nodes_of_faces [list]
    set n_items [objarray length $element_ids]
    for { set i 0 } { $i < $n_items } { incr i } {
        set element_id [objarray get $element_ids $i]
        set face_index [objarray get $faces_index $i]
        set face_nodes [GiD_Mesh get element $element_id face_linear $face_index]
        lappend nodes_of_faces {*}$face_nodes
    }
    set nodes_of_faces [lsort -integer -unique $nodes_of_faces]
    #get the elements of these nodes
    if { 0 } {
        set elements_of_nodes [list]
        foreach node_id $nodes_of_faces {
            # "GiD_Mesh get element from_node" is not implemented and will be expensive requiring a loop over all elements
            set elements_of_node [GiD_Mesh get element from_node $node_id]
            lappend elements_of_nodes {*}$elements_of_node
        }
        set elements_of_nodes [lsort -integer -unique $elements_of_nodes]
    } else {
        set elements_of_nodes [list]
        foreach element_id [GiD_Mesh list -element_type {Triangle Quadrilateral} element] {
            foreach node_id [GiD_Mesh get element $element_id connectivities] {
                if { [lsearch -sorted -integer $nodes_of_faces $node_id] != -1 } {
                    lappend elements_of_nodes $element_id
                    break
                }
            }
        }
    }
    return $elements_of_nodes
}

#to know is the model could be considered as 2D or 3D
#mode: mesh geometry post
proc GidUtils::GetIs2D { mode {tolerance 1e-7} } {
    set is_2d 0
    if { [lsearch [list geometry mesh post] $mode] == -1 } {
        error "valid modes: geometry mesh post"
    } else {
        lassign [GiD_Info bounding_box -pmin_pmax -$mode -layers_off_also] x_min y_min z_min x_max y_max  z_max
        if { abs($z_min)<=$tolerance && abs($z_max)<=$tolerance} {
            set is_2d 1
        }
    }
    return $is_2d
}

proc GidUtils::_setWindowAboveAnother { my_w other_w} {
    if { [winfo exists $my_w] && [winfo ismapped $my_w] } {
        if { [winfo exists $other_w] && [winfo ismapped $other_w]} {
            wm stackorder $my_w isabove $other_w
        }
    }
}

proc GidUtils::_setWindowAboveGiD { w} {
    if { [winfo exists $w] } {
        set wtop [winfo toplevel $w]
        if { [winfo ismapped $wtop] } {
            # make wtop to be above all others...
            # [winfo class .gid] == GiD and not Toplevel !!!
            GidUtils::_setWindowAboveAnother $wtop .gid
            set widgets [list]
            foreach ww [list . .gid] {
                if { [winfo exists $ww] } {
                    lappend widgets $ww
                }
            }
            foreach ww [list . .gid] {
                if { [winfo exists $ww] } {
                    lappend widgets {*}[winfo children $ww]
                }
            }
            foreach ww $widgets {
                if { [winfo class $ww] == "Toplevel"} {
                    GidUtils::_setWindowAboveAnother $wtop $ww
                }
            }
            raise $wtop
            # focus $w
        }
    }
}

# used when window can not be transient or topmost
proc GidUtils::WindowAboveGid { w } {
    # if { $::tcl_platform(platform) == "windows"} {
    #     if { [winfo exists $w]} {
    #         if { [info exists ::VISUAL_STUDIO_DEBUG] } {
    #             #to avoid enoying windows over Visual Studio editor stopped in debug
    #             after 500 [list GidUtils::_setWindowAboveGiD $w]
    #         } else {
    #             wm attributes $w -topmost 1
    #         }
    #     }
    # } else {
    #    after 500 [list GidUtils::_setWindowAboveGiD $w]
    # }
    # GidUtils::_setWindowAboveGiD $w
    if { [ winfo exists .gid ] } {
        if { [ winfo exists $w ] && ( [ winfo class $w] == "Toplevel")} {
            # for transient no need to check if .gid or $w are mapped
            # they can be declared transient when withdrawn

            set can_use_transient [gid_cross_platform::all_monitors_use_same_scaling_factor]
            # macOS and several monitors, only if mirroed, transient can be used
            #      it does not matter the scaling factor on the monitors
            if {  $::tcl_platform(os) == "Darwin"} {
                if { [gid_cross_platform::get_number_of_monitors_connected] > 1} {
                    set can_use_transient  0
                }
            }
            if { $can_use_transient} {
                wm transient $w .gid
            }

            # when withdrawn, raise does not deiconify it
            raise $w
        }
    }
}

proc GidUtils::GiDDefaultParentIfPresent {} {
    set ret_parent [list]
    if { [winfo exists .gid] && [winfo ismapped .gid]} {
        set ret_parent [list -parent .gid]
    }
    return $ret_parent
}

proc GidUtils::grab { args} {
    # grab ?-global? window
    # grab current ?window?
    # grab release window
    # grab set ?-global? window
    # grab status window

    # avoid grabbing an unmapped window
    # check if windown is mapped when doing
    #    grab ?-global? window
    #    grab set ?-global? window
    set first [lindex $args 0]
    set discard_check_list [list current release status]
    set exec_cmd 1
    if { [lsearch $discard_check_list $first] == -1} {
        # $first is a windows name or the option 'set'
        set prob_window [lindex $args end]
        if { [winfo ismapped $prob_window] == 0} {
            set exec_cmd 0
        }
    }
    set ret_value ""
    if { $exec_cmd} {
        set ret_value [::grab {*}$args]
    }
    return $ret_value
}

proc GetResultAverageOfOneSet { set_name result_name analysis_name step_value component_index} {
    lassign [GiD_Result get -sets [list $set_name] -array [list $result_name $analysis_name $step_value]] header unit components lst_idx_values

    lassign $lst_idx_values lst_indices lst_values
    set my_values [lindex $lst_values $component_index]
    set num_values [objarray length $lst_indices]
    if { $num_values != 0} {
        set average [expr [objarray sum $my_values]/$num_values]
    } else {
        set average -1.0+e38
    }
    return $average
}

proc PrintResultAverageForEachVolume { result_name analysis_name step_value component_index} {
    foreach volume_set [GiD_Info postprocess get all_volumesets] {
        set res_average [GetResultAverageOfOneSet $volume_set $result_name $analysis_name $step_value $component_index]
        W "Average result $result_name for volume '$volume_set' = $res_average"
    }
}
