# tcl_precision is higher than required to show some doubles
# but must be higth to have good precision in GiD_Info procedures
proc FormatDouble { text } {    
    if { [string is double -strict $text] && ![string is integer $text] } {    
        if { [catch { set text [format [GiD_Set RealFormat] $text] } msg] } {
            #W "FormatDouble '$text' $msg" 
            set text nan
        }    
    }
    return $text
}

set entity_types [list POINT STLINE ARCLINE SPLINELINE NURBLINE \
        POLYLINE NURBSURFACE COONSURFACE PLSURFACE CONTACTSURFACE \
        MESHSURFACE POLYSURFACE VOLUME CONTACTVOLUME NODES ELEMENTS \
        POSTNODES POSTELEMENTS]

set level_types [list points lines lines lines lines \
        lines surfaces surfaces surfaces surfaces \
        surfaces surfaces volumes volumes nodes elements \
        nodes elements]

proc GetLevelType { entity_type } {        
    set i_level [lsearch $::entity_types $entity_type]
    if { $i_level != -1 } {
        set level_type [lindex $::level_types $i_level]
    } else {
        set level_type "UNKNOWN"        
    }
    return $level_type
}

proc CreateNumericEntryFocusIn { w} {
    set wclass [ winfo class $w]
    if { $wclass == "Text"} {
        set ::GidPriv($w,old_value) [ $w get 0.0 end]
        set del_ini 0.0
    } else {
        # TEntry
        set ::GidPriv($w,old_value) [ $w get]
        set del_ini 0
    }
    set ::GidPriv($w,old_state) [ $w cget -state]
    set new_value [ regsub -all {,} $::GidPriv($w,old_value) {}]
    $w configure -state normal
    $w delete $del_ini end
    $w insert end $new_value
    $w configure -state $::GidPriv($w,old_state)
}

proc CreateNumericEntryFocusOut { w} {
    if { [ info exists ::GidPriv($w,old_value)]} {
        set wclass [ winfo class $w]
        if { $wclass == "Text"} {
            set del_ini 0.0
        } else {
            # TEntry
            set del_ini 0
        }
        $w configure -state normal
        $w delete $del_ini end
        $w insert end $::GidPriv($w,old_value)
        $w configure -state $::GidPriv($w,old_state)
    }
}

proc CreateNumericEntry { w text { l -1} } {
    ttk::entry $w
    set text [FormatDouble $text]
    $w  insert end $text
    set len [string length $text]
    incr len 4
    if { $l != -1} { set len $l}
    $w configure -width $len
    # bindings to remove ',' thousand separator when focus in to do Control-C
    bind $w <FocusIn> [ list CreateNumericEntryFocusIn $w]
    bind $w <FocusOut> [ list CreateNumericEntryFocusOut $w]
}

proc CreateEntry { w text { l -1} } {
    ttk::entry $w
    set text [FormatDouble $text]
    $w  insert end $text
    set len [string length $text]
    incr len 4
    if { $l != -1} { set len $l}
    $w configure -width $len
}

proc SelectMouselineInList { w } {
    $w.e delete 0 end
    $w.e insert 0 \
            [lindex [$w.f1.list get [lindex [$w.f1.list curselection] 0]] 0]
}

proc GoToShowEnt { w } {
    global InfoListEnt
    set iNumEnt [$w.list.e get]
    if { ![info exists InfoListEnt($iNumEnt,type)] } {
        WarnWin [_ "Before pressing 'go' enter one entity number"]
        return
    }
    ShowEntity $w.enti $iNumEnt
    destroy $w.list
}

proc OpenProjectFolder { } {    
    if { [GidUtils::ModelHasName] } {
        package require gid_cross_platform
        gid_cross_platform::open_by_extension [GiD_Info Project ModelName].gid
    }
}

proc OpenProblemtypeFolder { problemtype } {
    package require gid_cross_platform
    set full_problemtype [GidUtils::GiveProblemTypeFullname $problemtype].gid
    gid_cross_platform::open_by_extension $full_problemtype
}

proc OpenTemporaryFolder { } {    
    package require gid_cross_platform
    gid_cross_platform::open_by_extension [GiD_Info project TmpDirectory]
}

proc OpenUserSettingsFolder { } {
    package require gid_cross_platform
    set user_settings_folder [file dirname [GiD_GetUserSettingsFilename]]
    gid_cross_platform::open_by_extension $user_settings_folder
}

proc OpenProjectFolderOrPostFile { projectOrFile } {
    set folderToOpen ""
    if { [ file isdirectory $projectOrFile]} {
        set folderToOpen $projectOrFile
    } elseif { $projectOrFile != "UNNAMED"} {
        # may be it's the project's name
        set folder ${projectOrFile}.gid
        if { [ file isdirectory $folder]} {
            set folderToOpen $folder
        } else {
            # or a postprocess data file
            set folder [ file dirname $projectOrFile]
            if { [ file isdirectory $folder]} {
                set folderToOpen $folder
            }
        }
    }
    if { $folderToOpen != ""} {
        package require gid_cross_platform
        gid_cross_platform::open_by_extension $folderToOpen
    }
}

proc FillListOfEntWithEnt { w listorder} {
    global ListOfEnt InfoListEnt

    GIDReportData::Reset

    set w2 $w.list
    if { [winfo exists $w2.f2.text] } {
        catch { pack forget $w2.f2 }
        pack $w2.f1 -fill both -expand yes -side top
    }
    $w2.f1.list delete 0 end
    set num [lindex $ListOfEnt 0]
    if { ![string is integer -strict $num] } {
        return 1
    }
    if { $InfoListEnt($num,type) == "POINT" || $InfoListEnt($num,type) == "NODES" || $InfoListEnt($num,type) == "POSTNODES" } {
        foreach i $ListOfEnt {
            $w2.f1.list insert end "$i         $InfoListEnt($i,coord)"
            GIDReportData::AddData $i "$InfoListEnt($i,coord)"
        }
    } else {
        if { $InfoListEnt($num,type) == "ELEMENTS" || $InfoListEnt($num,type) == "POSTELEMENTS" } {
            foreach i $ListOfEnt {
                $w2.f1.list insert end "$i        $InfoListEnt($i,nodes)"
                GIDReportData::AddData $i "$InfoListEnt($i,nodes)"
            }
        } else {
            foreach i $ListOfEnt {
                $w2.f1.list insert end $i
                GIDReportData::AddData $i
            }
        }
    }
    set trad [_ "Entities contents"]:
    $w2.buts.text configure -text [_ "View text"]  -command \
            "FillListOfEntWithText $w [list $listorder]; $w2.buts.toReport configure -state disable; $w2.l configure -text \"$trad\""

    set InfoListEnt(ReportData) [ GIDReportData::GetDataHTML 1]
}

proc FillListOfEntWithText { w listorder} {

    set w2 $w.list
    if { ![winfo exists $w2.f2.text] } {
        PutText $w2 2 yes 15
    }
    pack forget $w2.f1
    pack $w2.f2 -fill both -expand yes -side top

    set cmd "GiD_Info list_entities -more -nice "

    foreach elem $listorder {
        append cmd " $elem"
    }
    set info [eval $cmd]
    set info [split $info \n]

    set que [ lindex $info 0]
    $w2.f2.text delete 0.0 end

    if { ( "$que" != "POSTNODES") && ( "$que" != "POSTELEMENTS") } {
        $w2.f2.text configure -font Normal
        foreach aa $info {
            $w2.f2.text insert end $aa\n
        }
    } else {
        $w2.f2.text configure -font FixedFont
        foreach aa $info {
            set idx [ string first : $aa]
            if { $idx == -1 } {
                $w2.f2.text insert end $aa\n
            } else {
                set clave [ string range $aa 0 $idx]
                switch -- $clave {

                    "Num:" {
                        $w2.f2.text insert end "\n[ lindex $aa 0] [ lindex $aa 1]\t [ lindex $aa 2]\t [ lindex $aa 3]\t [ lindex $aa 4]\n"
                    }
                
                    "Used by:" {
                        set lst_mallas [ lrange $aa 2 end]
                        $w2.f2.text insert end "Used by: "
                        if { ( [ llength $lst_mallas] == 1) && ( "$lst_mallas" == "N/A")} {
                            $w2.f2.text insert end [_ "No elements"]
                        } else {
                            set primero 1
                            foreach i_malla $lst_mallas {
                                if { !$primero} {
                                    $w2.f2.text insert end \t$i_malla\n
                                } else {
                                    $w2.f2.text insert end $i_malla\n
                                    set primero 0
                                }
                            }
                        }
                    }

                    "Current result:" {
                        set rr [ lrange $aa 2 end]
                        set nombres [ lindex $rr 0]
                        set resul_pgs [ lindex $rr 1]
                        set gausspoints ""
                        if { [ llength $nombres] == 2} {
                            set componentes [ lindex $nombres 1]
                        } else {
                            set gausspoints [ lindex $nombres 1]
                            set componentes [ lindex $nombres 2]
                        }
                        $w2.f2.text insert end "  Current result: [lindex $nombres 0]"
                        if { "$gausspoints" != ""} {
                            $w2.f2.text insert end " on '$gausspoints'"
                        }

                        if { [ llength $componentes] > 1} {
                            $w2.f2.text insert end "\n                 \t"
                            for { set i 0} { $i < [ llength $componentes]} { incr i} {
                                $w2.f2.text insert end " \t[lindex $componentes $i]"
                            }
                        }

                        for { set ipg 0} { $ipg < [ llength $resul_pgs]} { incr ipg} {
                            $w2.f2.text insert end "\n                 \t"
                            set valores [ lindex $resul_pgs $ipg]
                            if { [ llength $valores] == 1} {
                                $w2.f2.text insert end " \t[lindex $valores 0]"
                            } else {
                                set n [ llength $valores]
                                for { set i 0} { $i < $n} { incr i} {
                                    set v [ lindex $valores $i]
                                    $w2.f2.text insert end " \t$v"
                                }
                            }
                        }
                        $w2.f2.text insert end \n
                    }

                    "More results:" {
                        set todos [ lindex $aa 2 ]
                        set nt [ llength $todos]
                        for { set it 0} { $it < $nt} { incr it} {
                            set rr [ lindex $todos $it]
                            set nombre [ lindex $rr 0]
                            set gausspoints ""
                            if { [ llength $rr] == 3} {
                                set componentes [ lindex $rr 1]
                                set resul_pgs [ lindex $rr 2]
                            } else {
                                set gausspoints [ lindex $rr 1]
                                set componentes [ lindex $rr 2]
                                set resul_pgs [ lindex $rr 3]
                            }
                            if { !$it} {
                                $w2.f2.text insert end " - More results: $nombre"
                            } else {
                                $w2.f2.text insert end "                 $nombre"
                            }
                            if { "$gausspoints" != ""} {
                                $w2.f2.text insert end " on '$gausspoints'"
                            }
                            if { [ llength $componentes] > 1} {
                                $w2.f2.text insert end "\n                 \t"
                                for { set i 0} { $i < [ llength $componentes]} { incr i} {
                                    $w2.f2.text insert end " \t[lindex $componentes $i]"
                                }
                            }
                            for { set ipg 0} { $ipg < [ llength $resul_pgs]} { incr ipg} {
                                $w2.f2.text insert end "\n                 \t"
                                set valores [ lindex $resul_pgs $ipg]
                                if { [ llength $valores] == 1} {
                                    $w2.f2.text insert end " \t[lindex $valores 0]"
                                } else {
                                    set n [ llength $valores]
                                    for { set i 0} { $i < $n} { incr i} {
                                        set v [ lindex $valores $i]
                                        $w2.f2.text insert end " \t$v"
                                    }
                                }
                            }
                            $w2.f2.text insert end \n
                        }
                    }

                    "Gauss points:" {
                        set todos [ lrange $aa 2 end]
                        set nt [ llength $todos]
                        for { set it 0} { $it < $nt} { incr it} {
                            set rr [ lindex $todos $it]
                            set nombre [ lindex $rr 0]
                            set numero [ lindex $rr 1]
                            set que_coordenadas [ lindex $rr 2]
                            if { !$it} {
                                $w2.f2.text insert end " - Gauss points: '$nombre' $numero ppe with $que_coordenadas"
                            } else {
                                $w2.f2.text insert end "                 '$nombre' $numero ppe with $que_coordenadas"
                            }
                            $w2.f2.text insert end \n
                        }
                    }

                    "Analysis name:" {
                        set name [ lrange $aa 2 end]
                        $w2.f2.text insert end " - Analysis name: '$name'"
                        $w2.f2.text insert end \n
                    }

                    "Step value:" {
                        set value [ lrange $aa 2 end]
                        $w2.f2.text insert end " - Step value: '$value'"
                        $w2.f2.text insert end \n
                    }
                    default {
                        $w2.f2.text insert end $aa\n
                    }
                }
            }
        }
    }

    set trad [_ "Select entity"]:
    $w2.buts.text configure -text [_ "View list"]  -command \
            "FillListOfEntWithEnt $w [list $listorder]; $w2.buts.toReport configure -state normal; $w2.l configure -text \"$trad\""
}

proc MakeListOfEnt { w listorder } {
    global ListOfEnt InfoListEnt

    set w2 $w.list

    if { ![winfo exists $w2] } {
        InitWindow2 $w2 -title [_ "List entities"] \
            -geometryvariable PreListEntitiesWindowGeom \
            -ontop
        if { ![winfo exists $w2] } return ;# windows disabled || UseMoreWindows == 0        
        
        ttk::label $w2.l -text [_ "Select entity"]:
        pack $w2.l -side top
        ttk::entry $w2.e
        $w2.e delete 0 end
        $w2.e insert end [lindex $ListOfEnt 0]
        ttk::frame $w2.buts  -style BottomFrame.TFrame
        ttk::button $w2.buts.go -text [_ "Go"] -command "GoToShowEnt $w" -style BottomFrame.TButton
        ttk::button $w2.buts.text -text [_ "View text"] -command [list FillListOfEntWithText $w $listorder] -style BottomFrame.TButton
        ttk::button $w2.buts.close -text [_ "Close"] -command "destroy $w2" -style BottomFrame.TButton
        ttk::button $w2.buts.toReport -image [gid_themes::GetImage ToReport.png medium_icons] -style BottomFrame.TButton \
            -command {
            global InfoListEnt
            if { "$InfoListEnt(ReportData)" != ""} {
                GIDReport::AddToReport $InfoListEnt(ReportData)
            }
        }

        GidHelp $w2.buts.toReport [ GIDReport::GetHelpForImage ToReport]
        pack $w2.buts.go $w2.buts.text $w2.buts.close -side left -padx 1 -pady 5
        
        pack $w2.buts.toReport -side right
        pack $w2.buts -side bottom -fill x
        pack $w2.e -fill x -padx 5 -pady 5 -side bottom
        PutListBox $w2 1 yes 15
        
        bind $w2.f1.list <ButtonRelease-1> "SelectMouselineInList $w2"
        bind $w2.f1.list <Double-Button-1> "$w2.buts.go invoke ; break"
        bind $w2.e <Return> "$w2.buts.go invoke"
    }
    FillListOfEntWithEnt $w $listorder
}

proc ClipboardCopyText { value} {
    clipboard clear
    clipboard append $value
}

proc ShowStatus { wListEnt } {               
    wm title $wListEnt [_ "Status"]
    wm iconname $wListEnt "Status"
    $wListEnt.ltit configure -text [_ "STATUS"]
    catch { destroy  $wListEnt.enti }
    if { [winfo height $wListEnt.buts.close] > 1 } {
        pack propagate $wListEnt.buts no
    }
    pack propagate $wListEnt.d no
    $wListEnt.d configure -text {}

    pack forget $wListEnt.buts.prev $wListEnt.buts.next $wListEnt.buts.list $wListEnt.buts.toReport
    pack $wListEnt.buts.close -anchor center -padx 2 -pady 3

    bind $wListEnt <Tab> ""
    bind $wListEnt <Shift-Tab> ""

    ttk::frame $wListEnt.enti -style ridge.TFrame -borderwidth 2
    pack propagate $wListEnt.enti no
    pack $wListEnt.enti -fill both -expand yes -side top
    
    set variable_names [list project_name problem_type num_nodes num_elements temporary_folder user_settings_folder]
    set texts [list [_ "Project Name"] [_ "Problem type"] [_ "Num of nodes"] [_ "Num of elems"] [_ "Temporary folder"] [_ "User settings folder"]]

    set ilistnum 0
    set index 0
    set info [split [GiD_Info list_entities status] \n]
    set maxindex [llength $info]
    while {$index <= $maxindex} {
        set aa [lindex $info $index]
        incr index        
        switch -regexp $aa {
            "Project name:" {
                regexp {Project name: (.*)$} $aa trash project_name
            }
            "Problem type:" {
                regexp {Problem type: (.*)$} $aa trash problem_type
            }            
            "number of nodes:" {
                regexp {number of nodes: (.*)$} $aa trash num_nodes
            }
            "Total number of elements:" {
                regexp {Total number of elements: (.*)$} $aa trash num_elements
            }
            "There is no mesh" {
                # nothing
            }
            default {
                set inlist($ilistnum) $aa
                incr ilistnum
            }
        }
    }

    set temporary_folder [file join [GiD_Info project TmpDirectory]]
    set user_settings_folder  [file dirname [GiD_GetUserSettingsFilename]]
    
    if { [GetCurrentPrePostMode] == "PRE" } {
        set es_pre 1
    } else {
        set es_pre 0
    }

    set w $wListEnt.enti
    
    foreach variable_name $variable_names text $texts {
        if { ![info exists $variable_name] } {
            if { $variable_name == "num_nodes" } {
                ttk::label $w.$variable_name  -text [_ "There is no mesh"]
                pack $w.$variable_name -side top
            }
        } else {
            set value [set $variable_name]
            ttk::frame $w.$variable_name
            ttk::label $w.$variable_name.cc -text ${text}: -width 18
            # something like this
            set is_number [ string is integer $value]
            if { $is_number} {
                CreateNumericEntry $w.$variable_name.n [::GidUtils::AddThousandsSeparator $value]
            } else {
                CreateEntry $w.$variable_name.n $value
            }
            # copy to clipboard the raw value, not the formated one
            # this is wrong: [list clipboard clear ; clipboard append $value]
            # creates a list with first item, and evaluates the 2nd command
            # bind $w.$variable_name.n <Control-c> [list clipboard clear ; clipboard append $value]
            bind $w.$variable_name.n <Control-c> [ list ClipboardCopyText $value]
            if { $variable_name == "project_name" } {
                ttk::button $w.$variable_name.b_open_folder -image [gid_themes::GetImage folder.png small_icons] \
                    -command [list OpenProjectFolderOrPostFile $value]
                GidHelp $w.$variable_name.b_open_folder [_ "Open the folder in the file browser"]                
                pack $w.$variable_name.cc $w.$variable_name.b_open_folder $w.$variable_name.n -side left -anchor w
             } elseif { $variable_name == "problem_type" } {
                ttk::button $w.$variable_name.b_open_folder -image [gid_themes::GetImage folder.png small_icons] \
                    -command [list OpenProblemtypeFolder $value]
                GidHelp $w.$variable_name.b_open_folder [_ "Open the folder in the file browser"]
                pack $w.$variable_name.cc $w.$variable_name.b_open_folder $w.$variable_name.n -side left -anchor w
            } elseif { $variable_name == "temporary_folder" } {
                ttk::button $w.$variable_name.b_open_folder -image [gid_themes::GetImage folder.png small_icons] \
                    -command [list OpenTemporaryFolder]
                GidHelp $w.$variable_name.b_open_folder [_ "Open the folder in the file browser"]
                pack $w.$variable_name.cc $w.$variable_name.b_open_folder $w.$variable_name.n -side left -anchor w
            } elseif { $variable_name == "user_settings_folder" } {
                ttk::button $w.$variable_name.b_open_folder -image [gid_themes::GetImage folder.png small_icons] \
                    -command [list OpenUserSettingsFolder]
                GidHelp $w.$variable_name.b_open_folder [_ "Open the folder in the file browser"]
                pack $w.$variable_name.cc $w.$variable_name.b_open_folder $w.$variable_name.n -side left -anchor w
            } else {
                pack $w.$variable_name.cc $w.$variable_name.n -side left -anchor w
            }
            
            pack $w.$variable_name -side top -fill x -pady 2
        }
    }

    ttk::button $w.b1 -text [_ "Set Mesh Options"] -command SetMeshOptionsFromStatus
    GidHelp $w.b1 [_ "Set the current meshing parameters with the values used when the last mesh was generated"]
    pack $w.b1 -side top -anchor e -padx 2 -pady 2        
    
    if { [GiD_Info Project LastElementSize] == "NONE" } {
        $w.b1 configure -state disabled
    }

    PutListBox $w 1 yes 9
    for {set i 0} {$i < $ilistnum} {incr i} {
        if { $es_pre } {
            $w.f1.list insert end $inlist($i)
        } else {
            $w.f1.list insert end [::GidUtils::AddThousandsSeparator $inlist($i)]
        }
    }
}

proc SetMeshOptionsFromStatus { } {
    #preferences window is updated from c++    
    set prev_MesType [GiD_Set MeshType]
    GiD_SetMeshOptions
    
    if {$prev_MesType != [GiD_Set MeshType] } {
        #update menus and also preferences window
        UpdateMenus
    }

    if { [GidUtils::ExistsWindow PREFERENCES] } {
        #to immediatelly refresh the window content
        set list_var_mesh [GiD_Info variables -mesh -expand_array_names]
        foreach var $list_var_mesh {
          CreateWidgetsFromXml::ChangeVariableValue gid_preferences $var [GiD_Set $var]
        }
    }
}


proc ShowListEntities { w listorder } {
    global NumEntList ListOfEnt InfoListEnt

    set ListOfEnt ""
    set NumEntList 0
    set LastPolyNum -1 ; set LastPolyNumLine -1
    set LastPolyNumLine2 -1

    set LastPolySurfNum -1
    set LastPolySurfNumSurf -1 ; set LastPolySurfNumIntLine -1

    set info [split [GiD_Info list_entities -nice {*}$listorder] \n]
    
    set index 0
    set maxindex [llength $info]
    while {$index <= $maxindex} {
        set type [lindex $info $index]; incr index
        
        set NumToUse -1
        while 1 {
            set aa [lindex $info $index]; incr index
            if { [string match END* $aa] } {
                if { $LastPolyNum != -1 } {
                    set aa [lindex $info $index]; incr index
                    set InfoListEnt($NumToUse,orien) [lindex $aa 1]
                    set InfoListEnt($NumToUse,len) [string trimleft [lindex $aa 2] Length=]
                    set num $InfoListEnt($LastPolyNum,nlin)
                    set num2 0
                    if { $LastPolyNumLine2 != -1 } {
                        set idx "$LastPolyNum $LastPolyNumLine"
                        set num2 $InfoListEnt($idx,nlin)
                    }
                    if { ($num2>0 && $LastPolyNumLine2>=$num2) || ($num2==0 && $LastPolyNumLine >= $num) } {
                        set aa [lindex $info $index]; incr index
                        if { $LastPolyNumLine2 != -1 } {
                            set NumToUse "$LastPolyNum $LastPolyNumLine"
                            set LastPolyNumLine2 -1
                        } else {
                            set NumToUse $LastPolyNum
                            set LastPolyNum -1
                        }
                        set InfoListEnt($NumToUse,len) [string trimleft [lindex $aa 1] "Length="]
                        set InfoListEnt($NumToUse,upim) [lrange $aa 6 end]
                        continue
                    }
                } elseif { $LastPolySurfNum != -1 } {
                    if { $LastPolySurfNumSurf != -1 } {
                        set aa [lindex $info $index]; incr index
                        set InfoListEnt($NumToUse,orien) [lindex $aa 1]
                        set num $InfoListEnt($LastPolySurfNum,nsurf)
                        if { $LastPolySurfNumSurf >= $num } {
                            set NumToUse $LastPolySurfNum
                            set LastPolySurfNumSurf -1
                            continue
                        }
                    } elseif { $LastPolySurfNumIntLine != -1 } {
                        set num $InfoListEnt($LastPolySurfNum,nintln)
                        if { $LastPolySurfNumIntLine >= $num } {
                            set NumToUse $LastPolySurfNum
                            set LastPolySurfNumIntLine -1
                            set LastPolySurfNum  -1
                            continue
                        }
                    }
                }

                break
            }
            if { $index >= $maxindex } { break }
            switch -regexp -- $aa {
                Num: {
                    set NumToUse [lindex $aa 1]
                    if { $NumToUse == 0 } {
                        if { $LastPolyNum != -1 } {
                            if { $LastPolyNumLine2 != -1 } {
                                incr LastPolyNumLine2
                                set NumToUse "$LastPolyNum $LastPolyNumLine $LastPolyNumLine2"
                            } else {
                                incr LastPolyNumLine
                                set NumToUse "$LastPolyNum $LastPolyNumLine"
                            }
                        } elseif { $LastPolySurfNum != -1 } {
                            if { $LastPolySurfNumSurf != -1 } {
                                incr LastPolySurfNumSurf
                                set NumToUse "$LastPolySurfNum S$LastPolySurfNumSurf"
                            } elseif { $LastPolySurfNumIntLine != -1 } {
                                incr LastPolySurfNumIntLine
                                set NumToUse "$LastPolySurfNum L$LastPolySurfNumIntLine"
                            }
                        }

                    } else {
                        lappend ListOfEnt $NumToUse
                        incr NumEntList
                    }

                    set InfoListEnt($NumToUse,type) $type

                    if { "$type" == "POSTNODES" } {
                        set InfoListEnt($NumToUse,coord) [ lindex $aa 2]
                        set InfoListEnt($NumToUse,mat) 0
                        set InfoListEnt($NumToUse,CurrentResultName) ""
                        set InfoListEnt($NumToUse,MoreResults) ""
                        set InfoListEnt($NumToUse,AnalysisName) ""
                        set InfoListEnt($NumToUse,StepValue) ""
                        if { [ lsearch $aa GaussPoint] != -1} {
                            set InfoListEnt($NumToUse,EsPuntoGauss) 1
                        } else {
                            set InfoListEnt($NumToUse,EsPuntoGauss) 0
                        }

                    } elseif { "$type" == "POSTELEMENTS" } {

                        set InfoListEnt($NumToUse,nodes) [ lindex $aa 2]
                        set InfoListEnt($NumToUse,TipoElemento) [lindex $aa 3]
                        set InfoListEnt($NumToUse,mat) [lindex $aa 4]
                        set InfoListEnt($NumToUse,CurrentResultName) ""
                        set InfoListEnt($NumToUse,MoreResults) ""
                        set InfoListEnt($NumToUse,AnalysisName) ""
                        set InfoListEnt($NumToUse,StepValue) ""
                        set InfoListEnt($NumToUse,GaussPoints) ""
                        if { [ info exists InfoListEnt($NumToUse,SphereCircleRadius)] } {
                            unset InfoListEnt($NumToUse,SphereCircleRadius)
                        }
                        if { [ info exists InfoListEnt($NumToUse,CircleNormal)] } {
                            unset InfoListEnt($NumToUse,CircleNormal)
                        }

                    } else {
                        set InfoListEnt($NumToUse,HigherEnt) [lindex $aa 3]
                        set InfoListEnt($NumToUse,cond) [lindex $aa 5]
                        set InfoListEnt($NumToUse,mat) [lindex $aa 7]
                    }
                }
                Coord: {
                    set InfoListEnt($NumToUse,coord) [string trimleft $aa "Coord: "]
                }
                Points: {
                    set InfoListEnt($NumToUse,pts) [string trimleft $aa "Points: "]
                }
                "2D center" {
                    set InfoListEnt($NumToUse,2DCEN) [string trimleft [lindex $aa 1] "center="]
                    set InfoListEnt($NumToUse,rad) [string trimleft [lindex $aa 2] "radius="]
                    set InfoListEnt($NumToUse,iang) [string trimleft [lindex $aa 3] "initang="]
                    set InfoListEnt($NumToUse,eang) [string trimleft [lindex $aa 4] "endang="]
                }
                "Transformation Matrix:" {
                    set InfoListEnt($NumToUse,mat1) [lindex $info $index]; incr index
                    set InfoListEnt($NumToUse,mat2) [lindex $info $index]; incr index
                    set InfoListEnt($NumToUse,mat3) [lindex $info $index]; incr index
                    set InfoListEnt($NumToUse,mat4) [lindex $info $index]; incr index
                }
                "Num of Control Points" {
                    set num [string trimleft $aa "Num of Control Points="]
                    set InfoListEnt($NumToUse,NCtrl) $num
                    for {set i 1} {$i <= $num} {incr i} {
                        set aa [lindex $info $index]; incr index
                        set InfoListEnt($NumToUse,Ctrl$i) [string trimleft [lindex $aa 2] "coords"]:
                    }
                }
                "Number of Control Points" {
                    regexp {=[ ]*(.*)[ ]*Degree=[ ]*(.*)$} $aa t num degree
                    set InfoListEnt($NumToUse,NCtrl) $num
                    set InfoListEnt($NumToUse,degree) $degree
                    if { [llength $num] == 2 } {
                        set NumCtrl [expr [lindex $num 0]*[lindex $num 1]]
                    } else { set NumCtrl $num }
                    for {set i 1} {$i <= $NumCtrl} {incr i} {
                        regexp {Point (.*) coords:(.*)$} [lindex $info $index] t ord val; incr index
                        set InfoListEnt($NumToUse,Ctrl$i) [format "%-6s %s" $ord $val]
                    }
                }
                "Rational weights:" {
                    if { [llength $InfoListEnt($NumToUse,NCtrl)] == 2 } {
                        set NumCtrl [expr [lindex $InfoListEnt($NumToUse,NCtrl) 0]*[lindex $InfoListEnt($NumToUse,NCtrl) 1]]
                    } else { set NumCtrl $InfoListEnt($NumToUse,NCtrl) }
                    for {set i 1} {$i <= $NumCtrl} {incr i} {
                        set InfoListEnt($NumToUse,weights$i) [lindex $info $index]
                        incr index
                    }
                }
                "Number of knots=" {
                    set num [lindex $aa end]
                    set InfoListEnt($NumToUse,NKnots) $num
                    for {set i 1} {$i <= $num} {incr i} {
                        set aa [lindex $info $index]; incr index
                        set InfoListEnt($NumToUse,Knot$i) [string trimleft [lindex $aa 2] "value="]
                    }
                }
                "Number of knots in" {
                    regexp {in[ ]*([UVuv])=[ ]*([0-9]*)$} $aa t dir num
                    set InfoListEnt($NumToUse,NKnots$dir) $num
                    for {set i 1} {$i <= $num} {incr i} {
                        set aa [lindex $info $index]; incr index
                        set InfoListEnt($NumToUse,Knot$dir$i) [string trimleft [lindex $aa 2] "value="]
                    }
                }
                IsTrimmed: {
                    set InfoListEnt($NumToUse,IsTrimmed) [lindex $aa 1]
                }
                "TotalLength" {
                    set num [lindex $aa 1]
                    set InfoListEnt($NumToUse,TotalLength) $num
                }
                "Num of Lines" {
                    set InfoListEnt($NumToUse,nlin) [string trimleft $aa "Num of Lines="]
                    if { $LastPolyNum == -1 } {
                        set LastPolyNum $NumToUse
                        set LastPolyNumLine 0
                    } else {
                        set LastPolyNumLine2 0
                    }
                    break
                }
                NumLines: {
                    set num [lindex $aa 1]
                    set InfoListEnt($NumToUse,nlin) $num
                    for {set i 1} {$i <= $num} {incr i} {
                        set aa [lindex $info $index]; incr index
                        set InfoListEnt($NumToUse,line$i) [lindex $aa 1]
                        if { [llength $aa]== 4 } {
                            set InfoListEnt($NumToUse,orien$i) [lindex $aa 3]
                        } else {
                            set InfoListEnt($NumToUse,orien$i) [lrange $aa 2 end]
                        }
                    }
                
                }
                Center: {
                    set InfoListEnt($NumToUse,center) [lrange $aa 1 end]
                }
                Normal: {
                    set InfoListEnt($NumToUse,normal) [lrange $aa 1 end]
                }
                LAYER: {
                    set InfoListEnt($NumToUse,lay) [string trim [lrange $aa 1 end]]
                }
                "Meshing Info:" {
                    set items { Elemtype IsStructured Meshing size num_divisions weight tops SkipMesh BLMnlevels1 BLMnlevels2 BLMfirsth1 BLMfirsth2 Mesher}
                    set fields { elmtype struc meshing size num_divisions strucweight tops skipmesh nlevels1 nlevels2 firsth1 firsth2 mesher}
                    for { set i 3} { $i < [llength $aa] } { incr i} {
                        set itemandvalue [split [lindex $aa $i] =]
                        set j [lsearch $items [lindex $itemandvalue 0]]
                        if { $j == -1 } continue
                        set InfoListEnt($NumToUse,[lindex $fields $j]) [lindex $itemandvalue 1]
                        if { $j == 0 } {
                            switch -- $InfoListEnt($NumToUse,elmtype) {
                                0 { set InfoListEnt($NumToUse,elmtype) [_ "None"] }
                                1 { set InfoListEnt($NumToUse,elmtype) [_ "Linear"] }
                                2 { set InfoListEnt($NumToUse,elmtype) [_ "Triangle"] }
                                3 { set InfoListEnt($NumToUse,elmtype) [_ "Quadrilateral"] }
                                4 { set InfoListEnt($NumToUse,elmtype) [_ "Tetrahedra"] }
                                5 { set InfoListEnt($NumToUse,elmtype) [_ "Hexahedra"] }
                                6 { set InfoListEnt($NumToUse,elmtype) [_ "Prism"] }
                                7 { set InfoListEnt($NumToUse,elmtype) [_ "Only points"] }
                                8 { set InfoListEnt($NumToUse,elmtype) [_ "Pyramid"] }
                                9 { set InfoListEnt($NumToUse,elmtype) [_ "Sphere"] }
                                10 { set InfoListEnt($NumToUse,elmtype) [_ "Circle"] }
                            }
                        } elseif { $j == 2 } {
                            switch -- $InfoListEnt($NumToUse,meshing) {
                                No { set InfoListEnt($NumToUse,meshing) [_ "No"] }
                                Default { set InfoListEnt($NumToUse,meshing) [_ "Default"] }
                                Yes { set InfoListEnt($NumToUse,meshing) [_ "Yes"] }
                                No,Duplicate { set InfoListEnt($NumToUse,meshing) [_ "No"] }
                                Duplicate { set InfoListEnt($NumToUse,meshing) [_ "Duplicate"] }
                            }
                        } elseif { $j == 7 } {
                            switch -- $InfoListEnt($NumToUse,skipmesh) {
                                -1 { set InfoListEnt($NumToUse,skipmesh) [_ "No"] }
                                0 { set InfoListEnt($NumToUse,skipmesh) [_ "Automatic"] }
                                1 { set InfoListEnt($NumToUse,skipmesh) [_ "Yes"] }
                            }                        
                        } elseif { $j == 12 } {
                            switch -- $InfoListEnt($NumToUse,mesher) {                             
                                1 { set InfoListEnt($NumToUse,mesher) "RFast" }
                                2 { set InfoListEnt($NumToUse,mesher) "Rsurf" }
                                3 { set InfoListEnt($NumToUse,mesher) "DelaunaySurf3"] }
                                4 { set InfoListEnt($NumToUse,mesher) "AdvancingFront4" }
                                5 { set InfoListEnt($NumToUse,mesher) "MinElem" }
                                6 { set InfoListEnt($NumToUse,mesher) "Advancing front" }
                                7 { set InfoListEnt($NumToUse,mesher) "DelaunayVol7" }
                                8 { set InfoListEnt($NumToUse,mesher) "Isosurface" }
                                9 { set InfoListEnt($NumToUse,mesher) "Tetgen" }
                                10 { set InfoListEnt($NumToUse,mesher) "Octree" }
                                11 { set InfoListEnt($NumToUse,mesher) "PVolume11" }
                            }
                        }
                    }
                }
                "Num of Surfaces" {
                    set InfoListEnt($NumToUse,nsurf) [string trimleft $aa "Num of Surfaces="]
                    set LastPolySurfNum $NumToUse
                    set LastPolySurfNumSurf 0
                    set LastPolySurfNumIntLine -1
                    break
                }
                "Num of Interior Lines" {
                    set InfoListEnt($NumToUse,nintln) [string trimleft $aa "Num of InteriorLines="]
                    set LastPolySurfNumSurf -1
                    set LastPolySurfNumIntLine 0
                    break
                }
                NumSurfaces: {
                    set num [lindex $aa 1]
                    set InfoListEnt($NumToUse,nsur) $num
                    for {set i 1} {$i <= $num} {incr i} {
                        set aa [lindex $info $index]; incr index
                        set InfoListEnt($NumToUse,sur$i) [lindex $aa 1]
                        set InfoListEnt($NumToUse,orien$i) [lindex $aa 3]
                    }
                }
                "Segments used for special 4-sided surfaces:" {
                    set InfoListEnt($NumToUse,segments) [lindex $info $index]; incr index
                }
                "TRANSFORMATION MATRIX:" {
                    set InfoListEnt($NumToUse,matrix,row1) [lindex $info $index]; incr index
                    set InfoListEnt($NumToUse,matrix,row2) [lindex $info $index]; incr index
                    set InfoListEnt($NumToUse,matrix,row3) [lindex $info $index]; incr index
                    set InfoListEnt($NumToUse,matrix,row4) [lindex $info $index]; incr index
                }

                "Used by:" {
                    if { $InfoListEnt($NumToUse,type) == "POSTNODES" || $InfoListEnt($NumToUse,type) == "POSTELEMENTS" } {
                        set InfoListEnt($NumToUse,UsedBy) [lrange $aa 2 end]
                    }
                }

                "Current result:" {
                    if { $InfoListEnt($NumToUse,type) == "POSTNODES" } {
                        set InfoListEnt($NumToUse,CurrentResultName) [lindex $aa 2]
                        set InfoListEnt($NumToUse,CurrentResultValues) [lindex $aa 3]
                    } elseif { $InfoListEnt($NumToUse,type) == "POSTELEMENTS" } {
                        set InfoListEnt($NumToUse,CurrentResultName) [lindex $aa 2]
                        set InfoListEnt($NumToUse,CurrentResultValues) [lindex $aa 3]
                    }
                }

                "Sphere/Circle radius:" {
                    set InfoListEnt($NumToUse,SphereCircleRadius) [ lindex $aa 2]
                }

                "Circle normal:" {
                    set InfoListEnt($NumToUse,CircleNormal) [ lindex $aa 2]
                }

                "More results:" {
                    if { $InfoListEnt($NumToUse,type) == "POSTNODES" } {
                        set InfoListEnt($NumToUse,MoreResults) [ lindex $aa 2]
                    } elseif { $InfoListEnt($NumToUse,type) == "POSTELEMENTS" } {
                        set InfoListEnt($NumToUse,MoreResults) [ lindex $aa 2]
                    }
                }

                "Gauss points:" {
                    if { $InfoListEnt($NumToUse,type) == "POSTELEMENTS" } {
                        set InfoListEnt($NumToUse,GaussPoints) [ lrange $aa 2 end]
                    }
                }

                "Analysis name:" {
                    if { $InfoListEnt($NumToUse,type) == "POSTNODES" } {
                        set InfoListEnt($NumToUse,AnalysisName) [ lindex $aa 2]
                    } elseif { $InfoListEnt($NumToUse,type) == "POSTELEMENTS" } {
                        set InfoListEnt($NumToUse,AnalysisName) [ lindex $aa 2]
                    }
                }

                "Step value:" {
                    if { $InfoListEnt($NumToUse,type) == "POSTNODES" } {
                        set InfoListEnt($NumToUse,StepValue) [ lindex $aa 2]
                    } elseif { $InfoListEnt($NumToUse,type) == "POSTELEMENTS" } {
                        set InfoListEnt($NumToUse,StepValue) [ lindex $aa 2]
                    }
                }

                default {
                    if { $type == "NODES" } {
                        set InfoListEnt($NumToUse,coord) [lrange $aa 1 end]


                    }
                    if { $type == "ELEMENTS" } {
                        if { $InfoListEnt($NumToUse,mat) == 0 } {
                            set InfoListEnt($NumToUse,nodes) [lrange $aa 1 end]
                        } else {
                            set InfoListEnt($NumToUse,nodes) [lrange $aa 1 [expr [llength $aa]-2]]
                        }
                    }

                }
            }
        }
    }
}

proc PutIniEnd { w inum } {
    global InfoListEnt
    set cc [list [_ "Initial Pnt"] [_ "End Pnt"]]
    ttk::frame $w.coo
    for {set i 0} {$i < 2} {incr i} {
        ttk::label $w.coo.c$i -text [lindex $cc $i]:
        CreateEntry $w.coo.r$i [lindex $InfoListEnt($inum,pts) $i]
        pack $w.coo.c$i $w.coo.r$i -side left -anchor w
        GIDReportData::AddData [lindex $cc $i] [lindex $InfoListEnt($inum,pts) $i]
    }
    pack $w.coo -side top -fill x -pady 2

}
        

proc PutListBox { w num { havey no } { ysize 4 } } {

    ttk::frame $w.f$num
    ttk::scrollbar $w.f$num.xscroll -orient horizontal -command "$w.f$num.list xview"
    #pseudottk::
    tk::listbox $w.f$num.list -width 27 -height $ysize -xscroll "$w.f$num.xscroll set" -relief sunken -selectmode extended
    if { $havey != "no" } {
        ttk::scrollbar $w.f$num.yscroll -command "$w.f$num.list yview"
        $w.f$num.list configure -yscroll "$w.f$num.yscroll set"
        pack $w.f$num.yscroll -side right -fill y
    }
    pack $w.f$num -side top  -fill both -padx 2 -pady 2 -expand yes
    pack $w.f$num.xscroll -side bottom -fill x
    pack $w.f$num.list -fill both -side top -expand yes
    $w.f$num.list delete 0 end
}

proc PutText { w num { havey no } { ysize 4 } } {

    ttk::frame $w.f$num
    ttk::scrollbar $w.f$num.xscroll -orient horizontal -command "$w.f$num.text xview"
    tk::text $w.f$num.text -width 27 -height $ysize -xscroll "$w.f$num.xscroll set" -relief sunken
    if { $havey != "no" } {
        ttk::scrollbar $w.f$num.yscroll -command "$w.f$num.text yview"
        $w.f$num.text configure -yscroll "$w.f$num.yscroll set"
        pack $w.f$num.yscroll -side right -fill y
    }
    pack $w.f$num -side top  -fill both -padx 2 -pady 2 -expand yes
    pack $w.f$num.xscroll -side bottom -fill x
    pack $w.f$num.text -fill both -side top -expand yes
    $w.f$num.text delete 1.0 end
}

proc FillForSurface { w inum} {
    global InfoListEnt
    ttk::frame $w.coo1
    ttk::label $w.coo1.cc0 -text [_ "Num Lines"]
    CreateEntry $w.coo1.n0 $InfoListEnt($inum,nlin)
    pack $w.coo1.cc0 $w.coo1.n0 -side left -anchor w
    pack $w.coo1 -side top -fill x -pady 2
    GIDReportData::AddData [_ "Num Lines"] $InfoListEnt($inum,nlin)

    PutListBox $w 1 yes

    $w.f1.list insert end [format "%-10s%s" [_ "Line"] [_ "Orientation"]]
    GIDReportData::AddData [_ "Line"] [_ "Orientation"]
    for {set i 1} {$i <= $InfoListEnt($inum,nlin)} {incr i} {
        $w.f1.list insert end [format "%-4i        %s" $InfoListEnt($inum,line$i) $InfoListEnt($inum,orien$i)]
        GIDReportData::AddData $InfoListEnt($inum,line$i) $InfoListEnt($inum,orien$i)
    }

    set commands { center normal }
    set labels [list [_ "Label position"] [_ "Normal"]]
    foreach i $commands label $labels {
        ttk::frame $w.cou$i
        ttk::label $w.cou$i.cc -text $label:
        set coords ""
        foreach item $InfoListEnt($inum,$i) {
            lappend coords [FormatDouble $item]
        }
        CreateEntry $w.cou$i.n $coords
        pack $w.cou$i.cc -side left -anchor w
        pack $w.cou$i.n -side right -anchor e -fill x -expand yes
        pack $w.cou$i -side top -fill x -pady 2
        GIDReportData::AddData $label $coords
    }
}

proc FillForPolyline { w inum} {
    global InfoListEnt
    ttk::frame $w.coo1
    ttk::label $w.coo1.cc0 -text [_ "Num Lines"]
    CreateEntry $w.coo1.n0 $InfoListEnt($inum,nlin)
    pack $w.coo1.cc0 $w.coo1.n0 -side left -anchor w
    pack $w.coo1 -side top -fill x -pady 2
    GIDReportData::AddData [_ "Num Lines"] $InfoListEnt($inum,nlin)

    PutListBox $w 1 yes
    $w.f1.list insert end [format "%-6s%-12s%-11s%s" [_ "Line"] [_ "Orientation"] [_ "Type"] [_ "Length"]]
    GIDReportData::AddData [_ "Line"] [_ "Orientation"] [_ "Type"] [_ "Length"]

    for {set i 1} {$i <= $InfoListEnt($inum,nlin)} {incr i} {
        set length [FormatDouble $InfoListEnt($inum $i,len)]
        $w.f1.list insert end [format "%-6i%-12s%-11s%s" $i $InfoListEnt($inum $i,orien) $InfoListEnt($inum $i,type) $length]
        GIDReportData::AddData $i $InfoListEnt($inum $i,orien) $InfoListEnt($inum $i,type) $length
    }
    ttk::frame $w.buts -style raised.TFrame
    ttk::entry $w.buts.e -width 5
    $w.buts.e delete 0 end
    $w.buts.e insert end 1
    set polynum [$w.n.num2 get]
    ttk::button $w.buts.b -text [_ "See"] -command "ListPolyEntitiesEnt $w \"$polynum\" $InfoListEnt($inum,nlin) $w.buts.e"
    ttk::label $w.buts.l -text [_ "'Use Return'"]

    focus $w.buts.e
    set pp [winfo parent $w]
    bind $w.buts.e <Tab> "$pp.buts.next invoke ; break"
    bind $w.buts.e <Shift-Tab> "$pp.buts.prev invoke ; break"
    bind $w.buts.e <Return> "$w.buts.b invoke"

    bind $w.f1.list <ButtonRelease-1> "SelectMouseline $w"
    bind $w.f1.list <Double-Button-1> "$w.buts.b invoke ; break"

    pack $w.buts.e $w.buts.b $w.buts.l -side left -padx 2 -pady 2
    pack  $w.buts -side top
    if { ![info exists InfoListEnt($inum,segments)] } {
        set InfoListEnt($inum,segments) NONE
    }
    set cc [list [_ "Length"] [_ "Use Pnts in meshing"] [_ "used segments"]]
    set rr { len upim segments}
    for {set i 0} {$i < 3} {incr i} {
        ttk::frame $w.cou$i
        ttk::label $w.cou$i.cc -text [lindex $cc $i]:
        CreateEntry $w.cou$i.n $InfoListEnt($inum,[lindex $rr $i])
        pack $w.cou$i.cc  -side left -anchor w
        pack $w.cou$i.n -side left -anchor w
        pack $w.cou$i -side top -fill x -pady 2
        GIDReportData::AddData [lindex $cc $i] $InfoListEnt($inum,[lindex $rr $i])
    }
}


proc SelectMouseline { w } {
    $w.buts.e delete 0 end
    $w.buts.e insert 0 [lindex [$w.f1.list get [$w.f1.list curselection]] 0]
}

proc FrameForPolySurfLines { w iNumEnt} {
    global InfoListEnt

    ttk::frame $w.coo1
    ttk::label $w.coo1.cc0 -text [_ "Num Lines"]
    CreateEntry $w.coo1.n0 $InfoListEnt($iNumEnt,nlin)
    pack $w.coo1.cc0 $w.coo1.n0 -side left -anchor w
    pack $w.coo1 -side top -fill x -pady 2
    GIDReportData::AddData [_ "Num Lines"] $InfoListEnt($iNumEnt,nlin)

    PutListBox $w 1 yes
    $w.f1.list insert end [format "%-10s%s" [_ "Line"] [_ "Orientation"]]
    GIDReportData::AddData [_ "Line"] [_ "Orientation"]
    for {set i 1} {$i <= $InfoListEnt($iNumEnt,nlin)} {incr i} {
        $w.f1.list insert end [format "%-4i        %s" $InfoListEnt($iNumEnt,line$i) $InfoListEnt($iNumEnt,orien$i)]
        GIDReportData::AddData $InfoListEnt($iNumEnt,line$i) $InfoListEnt($iNumEnt,orien$i)
    }
    return $w
}


proc FrameForPolySurfSurfaces { w inum} {
    global InfoListEnt
    ttk::frame $w.coon1

    ttk::label $w.coon1.cc0 -text [_ "Num Surfaces"]
    CreateEntry $w.coon1.n0 $InfoListEnt($inum,nsurf)
    pack $w.coon1.cc0 $w.coon1.n0 -side left -anchor w
    pack $w.coon1 -side top -fill x -pady 2
    GIDReportData::AddData [_ "Num Surfaces"] $InfoListEnt($inum,nsurf)

    PutListBox $w 1 yes
    $w.f1.list insert end [format "%-8s%-12s%-11s" [_ "Surface"] [_ "Orientation"] [_ "Type"]]
    GIDReportData::AddData [_ "Surface"] [_ "Orientation"] [_ "Type"]
    for {set i 1} {$i <= $InfoListEnt($inum,nsurf)} {incr i} {
        $w.f1.list insert end [format "%-12i%-12s%-11s" $i $InfoListEnt($inum S$i,orien) $InfoListEnt($inum S$i,type)]
        GIDReportData::AddData $i $InfoListEnt($inum S$i,orien) $InfoListEnt($inum S$i,type)
    }
    return $w
}

proc FrameForPolySurfIntLines { w inum} {
    global InfoListEnt
    ttk::frame $w.coon2

    ttk::label $w.coon2.cc0 -text [_ "Num Int.Lines"]
    CreateEntry $w.coon2.n0 $InfoListEnt($inum,nintln)
    pack $w.coon2.cc0 $w.coon2.n0 -side left -anchor w
    pack $w.coon2 -side top -fill x -pady 2
    GIDReportData::AddData [_ "Num Int.Lines"] $InfoListEnt($inum,nintln)
    PutListBox $w 1 yes
    $w.f1.list insert end [format "%-6s%s" Line Type]
    GIDReportData::AddData Line Type
    for {set i 1} {$i <= $InfoListEnt($inum,nintln)} {incr i} {
        $w.f1.list insert end [format "%-6i%s" $i $InfoListEnt($inum L$i,type)]
        GIDReportData::AddData $i $InfoListEnt($inum L$i,type)
    }
}

proc FillForVolume { w inum } {
    global InfoListEnt
    ttk::frame $w.coo1
    ttk::label $w.coo1.cc0 -text [_ "Num Surfaces"]
    CreateEntry $w.coo1.n0 $InfoListEnt($inum,nsur)
    pack $w.coo1.cc0 $w.coo1.n0 -side left -anchor w
    pack $w.coo1 -side top -fill x -pady 2
    GIDReportData::AddData [_ "Num Surfaces"] $InfoListEnt($inum,nsur)
        
    PutListBox $w 1 yes
    $w.f1.list insert end [format "%-10s%s" [_ "Surface"] [_ "Orientation"]]
    GIDReportData::AddData [_ "Surface"] [_ "Orientation"]
    for {set i 1} {$i <= $InfoListEnt($inum,nsur)} {incr i} {
        $w.f1.list insert end [format "%-4i        %s" $InfoListEnt($inum,sur$i) $InfoListEnt($inum,orien$i)]
        GIDReportData::AddData $InfoListEnt($inum,sur$i) $InfoListEnt($inum,orien$i)
    }
    ttk::frame $w.cou
    ttk::label $w.cou.cc -text [_ "Label position"]:
    set coords ""
    foreach item $InfoListEnt($inum,center) {
        lappend coords [FormatDouble $item]
    }
    CreateEntry $w.cou.n $coords
    pack $w.cou.cc -side left -anchor w
    pack $w.cou.n -side right -anchor e -fill x -expand yes
    pack $w.cou -side top -fill x -pady 2
    GIDReportData::AddData [_ "Label position"] $coords
    bind $w.f1.list <Double-Button-1> [list SignalEntity Surfaces $w.f1.list]
}

proc FillForContactVolume { w inum} {
    global InfoListEnt

    PutListBox $w 1 no 3
    $w.f1.list insert end [format "%-10s%s" [_ "Surface"] [_ "Orientation"]]
    GIDReportData::AddData [_ "Surface"] [_ "Orientation"]
    for {set i 1} {$i <= $InfoListEnt($inum,nsur)} {incr i} {
        $w.f1.list insert end [format "%-4i        %s" $InfoListEnt($inum,sur$i) $InfoListEnt($inum,orien$i)]
        GIDReportData::AddData $InfoListEnt($inum,sur$i) $InfoListEnt($inum,orien$i)
    }
    if { [info exists InfoListEnt($inum,matrix,row1)] } {
        PutListBox $w 2 yes 5
        $w.f2.list insert end "   Transformation matrix"
        GIDReportData::AddData "Transformation matrix"
        for {set i 1} {$i <= 4} {incr i} {
            $w.f2.list insert end $InfoListEnt($inum,matrix,row$i)
            GIDReportData::AddData $InfoListEnt($inum,matrix,row$i)
        }
    }
    ttk::frame $w.cou
    ttk::label $w.cou.cc -text [_ "Label position"]:
    set coords ""
    foreach item $InfoListEnt($inum,center) {
        lappend coords [FormatDouble $item]
    }
    CreateEntry $w.cou.n $coords
    pack $w.cou.cc  -side left -anchor w
    pack $w.cou.n -side right -anchor e -fill x -expand yes
    pack $w.cou -side top -fill x -pady 2
    GIDReportData::AddData [_ "Label position"] $coords
}

proc FrameForNURBSurfLines { w iNumEnt} {
    global InfoListEnt
    ttk::frame $w.coo1
    ttk::label $w.coo1.cc0 -text [_ "Num Lines"]
    CreateEntry $w.coo1.n0 $InfoListEnt($iNumEnt,nlin)
    pack $w.coo1.cc0 $w.coo1.n0 -side left -anchor w
    pack $w.coo1 -side top -fill x -pady 2

    GIDReportData::AddData [_ "Num Lines"] $InfoListEnt($iNumEnt,nlin)

    PutListBox $w 1 yes
    $w.f1.list insert end [format "%-10s%s" [_ "Line"] [_ "Orientation"]]
    GIDReportData::AddData [_ "Line"] [_ "Orientation"]

    for {set i 1} {$i <= $InfoListEnt($iNumEnt,nlin)} {incr i} {
        $w.f1.list insert end [format "%-4i        %s" $InfoListEnt($iNumEnt,line$i) $InfoListEnt($iNumEnt,orien$i)]
        GIDReportData::AddData $InfoListEnt($iNumEnt,line$i) $InfoListEnt($iNumEnt,orien$i)
    }
    bind $w.f1.list <Double-Button-1> [list SignalEntity Lines $w.f1.list]
    return $w
}

proc SignalControlPoint { list } {
    set sel [$list curselection]
    if { [llength $sel] == 1 } {
        set coords [lindex [$list get  [$list curselection]] 1]
        if { $coords != "" } {
            GiD_Process escape escape escape escape Utilities SignalEntities Points NoJoin $coords
        }
    }
}

#type: line or surface
proc SignalKnot { type num list } {
    set sel [$list curselection]
    if { [llength $sel] == 1 } {
        set knot [lindex [$list get [$list curselection]] 1]
        if { [string is double -strict $knot] && $knot>=0.0 && $knot<=1.0 } {
            if { $type == "" } {
                set coords [GiD_Info parametric line $num coord $knot]
            } else {
                if { $type == "U" } {
                    set u $knot
                    set v 0.0
                } elseif { $type == "V" } {
                    set u 0.0
                    set v $knot
                } else {
                    return
                }
                set coords [GiD_Info parametric surface $num coord $u $v]
                
            }
            GiD_Process escape escape escape escape Utilities SignalEntities Points NoJoin $coords
        }
    }
}

proc SignalEntity { type list } {
    set sel [$list curselection]
    if { [llength $sel] == 1 } {
        set id [lindex [$list get [$list curselection]] 0]
        if { $id != "" && [string is integer $id] } {
            GiD_Process escape escape escape escape Utilities SignalEntities $type $id            
        }
    }    
}

proc FrameForNURBSurfPoints { w iNumEnt args } {
    global InfoListEnt
    ttk::frame $w.coon1
    ttk::label $w.coon1.cc0 -text [_ "Num Points"]:
    CreateEntry $w.coon1.n0 $InfoListEnt($iNumEnt,NCtrl)

    ttk::label $w.coon1.cc1 -text [_ "Degree"]:
    CreateEntry $w.coon1.n1 $InfoListEnt($iNumEnt,degree)
    GIDReportData::AddData [_ "Degree"] $InfoListEnt($iNumEnt,degree)

    pack $w.coon1.cc0 $w.coon1.n0 $w.coon1.cc1 $w.coon1.n1 -side left -anchor w
    pack $w.coon1 -side top -fill x -pady 2

    if { [info exists InfoListEnt($iNumEnt,weights1)] } {
        ttk::label $w.coon2 -text [_ "Rational"]
        GIDReportData::AddData [_ "Rational"]
    } else {
        ttk::label $w.coon2 -text [_ "Non rational"]
        GIDReportData::AddData [_ "Non rational"]
    }
    pack $w.coon2 -side top -pady 2 -anchor w

    GIDReportData::AddData [_ "Num Points"] $InfoListEnt($iNumEnt,NCtrl)

    PutListBox $w 2 yes

    bind $w.f2.list <Double-Button-1> "SignalControlPoint $w.f2.list"

    if { [llength $InfoListEnt($iNumEnt,NCtrl)] == 2 } {
        set NumCtrl [expr [lindex $InfoListEnt($iNumEnt,NCtrl) 0]*[lindex $InfoListEnt($iNumEnt,NCtrl) 1]]
    } else { set NumCtrl $InfoListEnt($iNumEnt,NCtrl) }

    if { ![info exists InfoListEnt($iNumEnt,weights1)] } {
        for {set i 1} {$i <= $NumCtrl} {incr i} {
            if { [ regexp {^([^ ]+) +([^ ]+)$} $InfoListEnt($iNumEnt,Ctrl$i) dummy id coords]} {
                set newcoords ""
                foreach item [split $coords ,] {
                    append newcoords [FormatDouble $item],
                }
                set coords [string trimright $newcoords ,]
                set text [format "%-7s" $id]$coords
                GIDReportData::AddData $id $coords
            } else {
                set text $InfoListEnt($iNumEnt,Ctrl$i)
                GIDReportData::AddData $text
            }
            $w.f2.list insert end $text
        }
    } else {
        for {set i 1} {$i <= $NumCtrl} {incr i} {
            set weight [FormatDouble $InfoListEnt($iNumEnt,weights$i)]
            if { [ regexp {^([^ ]+) +([^ ]+)$} $InfoListEnt($iNumEnt,Ctrl$i) dummy id coords]} {
                set newcoords ""
                foreach item [split $coords ,] {
                    append newcoords [FormatDouble $item],
                }
                set coords [string trimright $newcoords ,]
                set text [format "%-7s" $id]$coords
                GIDReportData::AddData $id $coords "W=$weight"
            } else {
                set text $InfoListEnt($iNumEnt,Ctrl$i)
                GIDReportData::AddData $text "W=$weight"
            }
            $w.f2.list insert end "$text W=$weight"
        }
    }
    return $w
}

proc FrameForNURBSurfKnots { w iNumEnt {type ""} args } {
    global InfoListEnt
    ttk::frame $w.coon2
    ttk::label $w.coon2.cc1 -text [_ "Num Knots"]:
    CreateEntry $w.coon2.n1 $InfoListEnt($iNumEnt,NKnots$type)
    pack $w.coon2.cc1 $w.coon2.n1 -side left -anchor w
    pack $w.coon2 -side top -fill x -pady 2
    GIDReportData::AddData [concat [_ "Num Knots" ] $type] $InfoListEnt($iNumEnt,NKnots$type)
    PutListBox $w 3 yes
    bind $w.f3.list <Double-Button-1> [list SignalKnot $type $iNumEnt $w.f3.list]
    for {set i 1} {$i<= $InfoListEnt($iNumEnt,NKnots$type)} {incr i} {
        set knot [FormatDouble $InfoListEnt($iNumEnt,Knot$type$i)]
        set text [format "%-4i" $i]$knot
        $w.f3.list insert end $text
        GIDReportData::AddData $i $knot
    }
    return $w
}

proc OnDoubleClickLinesUV { c x_screen y_screen iNumEnt } {
    set margin 15
    set width [winfo width $c]
    set height [winfo height $c]
    if { $width < $height } {
        set esc $width
    } else {
        set esc $height
    }
    set esc [expr $esc-2*$margin]
    set x [$c canvasx $x_screen]
    set y [$c canvasy $y_screen] 
    set u [expr ($x-$margin)/$esc]
    set v [expr ($height-$y-$margin)/$esc]
    if { $u>=0.0 && $u<=1.0 && $v>=0.0 && $v<=1.0 } {
        set coordinates [GiD_Info parametric surface $iNumEnt coord $u $v]
        GidUtils::DisableWarnLine
        GiD_Process Mescape Utilities SignalEntities Points FNoJoin {*}$coordinates
        GidUtils::EnableWarnLine
        set ::DrawLinesUVPriv($iNumEnt,signaled) 1
    }    
}

proc OnMotionLinesUV { c w x_screen y_screen iNumEnt } {
    set margin 15
    set width [winfo width $c]
    set height [winfo height $c]
    if { $width < $height } {
        set esc $width
    } else {
        set esc $height
    }
    set esc [expr $esc-2*$margin]
    set x [$c canvasx $x_screen]
    set y [$c canvasy $y_screen] 
    set u [expr ($x-$margin)/$esc]
    set v [expr ($height-$y-$margin)/$esc]
    if { $u>=0.0 && $u<=1.0 && $v>=0.0 && $v<=1.0 } {
        $w.f.flateral.fuv.lu configure -text u=[format "%.3f" $u]
        $w.f.flateral.fuv.lv configure -text v=[format "%.3f" $v]
        set coordinates [GiD_Info parametric surface $iNumEnt coord $u $v]
        lassign $coordinates px py pz
        $w.f.flateral.fxyz.lx configure -text x=[format "%.8g" $px]
        $w.f.flateral.fxyz.ly configure -text y=[format "%.8g" $py]
        $w.f.flateral.fxyz.lz configure -text z=[format "%.8g" $pz] 
        if { $::DrawLinesUVPriv($iNumEnt,signal) } {
            GidUtils::DisableWarnLine   
            GiD_Process Mescape Utilities SignalEntities Points FNoJoin {*}$coordinates
            GidUtils::EnableWarnLine
            set ::DrawLinesUVPriv($iNumEnt,signaled) 1 
        }
    } else {
        $w.f.flateral.fuv.lu configure -text ""
        $w.f.flateral.fuv.lv configure -text ""
        $w.f.flateral.fxyz.lx configure -text ""
        $w.f.flateral.fxyz.ly configure -text ""
        $w.f.flateral.fxyz.lz configure -text ""
    }
}

proc OnConfigureLinesUV { c iNumEnt } { 
    if { [info exists ::DrawLinesUVPriv($iNumEnt,resizing)] } {
        after cancel $::DrawLinesUVPriv($iNumEnt,resizing)
    }
    set ::DrawLinesUVPriv($iNumEnt,resizing) [after idle "DrawLinesUV $c $iNumEnt ; unset -nocomplain ::DrawLinesUVPriv($iNumEnt,resizing)"]        
}

proc OnDestroyLinesUV { iNumEnt } {
    if { [info exists ::DrawLinesUVPriv($iNumEnt,signaled)] } {
        unset ::DrawLinesUVPriv($iNumEnt,signaled)
        GiD_Process escape
    }
}

proc DrawLinesUV { c iNumEnt } {   
    if { ![winfo exists $c] } {       
        return 1
    } 
    $c configure -state disabled ;#to avoid invoke again while drawing
    set margin 15
    set width [winfo width $c]    
    set height [winfo height $c]
    if { $width < $height } {
        set esc $width
    } else {
        set esc $height
    }
    set esc [expr $esc-2*$margin]
    
    $c delete all
     
    set xy [list]
    foreach {u v} {0.0 0.0 1.0 0.0} {
        set x [expr $u*$esc+$margin]
        set y [expr $height-($v*$esc+$margin)]
        lappend xy $x $y
    }
    $c create line $xy -tag axis_u -width 1 -fill black -arrow last
    $c create text [lrange $xy end-1 end] -text u -anchor w
    set xy [list]
    foreach {u v} {0.0 0.0 0.0 1.0} {
        set x [expr $u*$esc+$margin]
        set y [expr $height-($v*$esc+$margin)]
        lappend xy $x $y
    }
    $c create line $xy -tag axis_v -width 1 -fill black -arrow last
    $c create text [lrange $xy end-1 end] -text v  -anchor s
    set xy [list]
    foreach {u v} {1.0 0.0 1.0 1.0} {
        set x [expr $u*$esc+$margin]
        set y [expr $height-($v*$esc+$margin)]
        lappend xy $x $y
    }
    $c create line $xy -width 1 -fill gray
    set xy [list]
    foreach {u v} {0.0 1.0 1.0 1.0} {
        set x [expr $u*$esc+$margin]
        set y [expr $height-($v*$esc+$margin)]
        lappend xy $x $y
    }
    $c create line $xy -width 1 -fill gray

    set num_t 51 ;#enough points to have o good approximation
    set color_lines [GiD_Set Color(Lines)]
    set surface_lines [GidUtils::GetSurfaceLines $iNumEnt]
    set num_lines [objarray length $surface_lines]
    for {set i_line 1} {$i_line <= $num_lines} { incr i_line } {
        set line_uv_id [GiD_Geometry -no_model create line append {*}[GiD_Geometry get surface $iNumEnt line_uv $i_line]]
        set xy [list]
        for {set i_t 0} {$i_t <= $num_t} {incr i_t} {
            set t [expr {double($i_t)/$num_t}]
            lassign [GiD_Info parametric -no_model line $line_uv_id coord $t] u v dummy
            set x [expr $u*$esc+$margin]
            set y [expr $height-($v*$esc+$margin)]
            lappend xy $x $y
        }
        $c create line $xy -tag $i_line -width 1 -fill $color_lines
        set line_id [objarray get $surface_lines [expr $i_line-1]]
        set t 0.5
        lassign [GiD_Info parametric -no_model line $line_uv_id coord $t] u v dummy
        set x [expr $u*$esc+$margin]
        set y [expr $height-($v*$esc+$margin)]
        $c create text $x $y -text $line_id
        GiD_Geometry -no_model delete line $line_uv_id
    }  
    $c configure -state normal    
    return 0
}

proc ShowLinesUV { iNumEnt } {
    set w .gid.linesuv-$iNumEnt
    if { [winfo exists $w] } {
        destroy $w
    }
    InitWindow2 $w -title [_ "Lines UV of surface %s" $iNumEnt] \
        -geometryvariable PreShowLinesUVWindowGeom -ontop
    ttk::frame $w.f
    canvas $w.f.c
    ttk::frame $w.f.flateral
    ttk::frame $w.f.flateral.fuv
    ttk::label $w.f.flateral.fuv.lu -width 15
    ttk::label $w.f.flateral.fuv.lv -width 15
    ttk::frame $w.f.flateral.fxyz
    ttk::label $w.f.flateral.fxyz.lx -width 15
    ttk::label $w.f.flateral.fxyz.ly -width 15
    ttk::label $w.f.flateral.fxyz.lz -width 15
    if { ![info exists ::DrawLinesUVPriv($iNumEnt,signal)] } {
        set ::DrawLinesUVPriv($iNumEnt,signal) 0
    }
    ttk::checkbutton $w.f.flateral.fxyz.signal -text [_ "Signal"] -variable ::DrawLinesUVPriv($iNumEnt,signal)
    grid $w.f.flateral.fuv.lu -sticky w
    grid $w.f.flateral.fuv.lv -sticky w
    grid $w.f.flateral.fuv -row 1 -sticky s -padx 3 -pady 3
    grid $w.f.flateral.fxyz.lx -sticky w
    grid $w.f.flateral.fxyz.ly -sticky w
    grid $w.f.flateral.fxyz.lz -sticky w
    grid $w.f.flateral.fxyz.signal -sticky w
    grid $w.f.flateral.fxyz -row 2 -sticky s -padx 3 -pady 3
    grid rowconfigure $w.f.flateral 0 -weight 1
    grid columnconfigure $w.f.flateral 0 -weight 1
    grid $w.f.c $w.f.flateral -sticky ns -padx 3 -pady 3
    grid configure $w.f.c -sticky nsew
    grid rowconfigure $w.f 0 -weight 1
    grid columnconfigure $w.f 0 -weight 1
    grid $w.f -sticky nsew
    
    ttk::frame $w.buts -style BottomFrame.TFrame
    ttk::button $w.buts.close -text [_ "Close"] -command [list destroy $w] -style BottomFrame.TButton
    grid $w.buts.close -padx 3 -pady 3
    grid $w.buts -sticky sew
    grid rowconfigure $w 0 -weight 1
    grid columnconfigure $w 0 -weight 1    
    grid anchor $w.buts center

    
    update
    DrawLinesUV $w.f.c $iNumEnt

    bind $w.f.c <Double-1> [list OnDoubleClickLinesUV $w.f.c %x %y $iNumEnt]
    bind $w.f.c <Motion> [list OnMotionLinesUV $w.f.c $w %x %y $iNumEnt]    
    bind $w.f.c <Configure> [list +OnConfigureLinesUV $w.f.c $iNumEnt]
    bind $w.f.c <Destroy> [list OnDestroyLinesUV $iNumEnt]
}

proc ShowEntity { w iNumEnt } {

    global InfoListEnt GIDDEFAULT

    GIDReportData::Reset

    catch { destroy $w }
    ttk::frame $w -style ridge.TFrame -borderwidth 2

    # para quitarlos de cuando creamos el Scrolled Canvas
    bind [winfo toplevel $w] <4> {}
    bind [winfo toplevel $w] <5> {}

    pack propagate $w no
    pack $w -fill both -expand yes -side top

    set pp [winfo parent $w]

    set type $InfoListEnt($iNumEnt,type)

    #to mark for translation
    set dummy [list [_ "POINT"] [_ "STLINE"] [_ "ARCLINE"] [_ "SPLINELINE"] [_ "NURBLINE"] \
            [_ "NURBSURFACE"] [_ "POLYLINE"] [_ "COONSURFACE"] [_ "PLSURFACE"] [_ "CONTACTSURFACE"] \
            [_ "POLYSURFACE"] [_ "VOLUME"] [_ "CONTACTVOLUME"] [_ "NODES"] [_ "ELEMENTS"] \
            [_ "POSTNODES"] [_ "POSTELEMENTS"]]
    $pp.ltit configure -text [_ $type]

    GIDReportData::AddData [_ "Entity type"] $type

    ttk::frame $w.n
    ttk::label $w.n.num1 -text [_ "Num"]
    if { "$type" != "POSTELEMENTS"} {
        CreateEntry $w.n.num2 $iNumEnt
    } else {
        set txt_ent $iNumEnt
        if { [ regexp {^([0-9]+)\(([0-9]+)\)$} $iNumEnt tnt idcorte idorig]} {
            set txt_ent "$idcorte ($idorig)"
        }
        CreateEntry $w.n.num2 $txt_ent
    }
    GIDReportData::AddData [_ "Num"] $iNumEnt

    if { $type != "POSTNODES" && $type != "POSTELEMENTS" } {
        ttk::label $w.n.lay1 -text [_ "Layer"]
        set lay "*****"
        catch { set lay $InfoListEnt($iNumEnt,lay) }
        CreateEntry $w.n.lay2 $lay
        GIDReportData::AddData [_ "Layer"] $lay

        pack $w.n.num1 $w.n.num2 $w.n.lay1 -side left -padx 5 -anchor w
        pack $w.n.lay2 -side left -padx 5 -anchor w -expand yes -fill x
        
        set HigherEnt $InfoListEnt($iNumEnt,HigherEnt)
        set cond $InfoListEnt($iNumEnt,cond)
        ttk::frame $w.n2
        ttk::label $w.n2.num1 -text [_ "HigherEnt"]:
        CreateEntry $w.n2.num2 $HigherEnt
        GIDReportData::AddData [_ "HigherEnt"] $HigherEnt

        ttk::label $w.n2.lay1 -text [_ "Cond"]:
        CreateEntry $w.n2.lay2 $cond
        GIDReportData::AddData [_ "Cond"] $cond

        if { $::GidPriv(ShowGroupsTab) } {
            set level_type [GetLevelType $InfoListEnt($iNumEnt,type)]
            if { [llength $iNumEnt] == 1 } {
                set entity_groups [GiD_EntitiesGroups entity_groups $level_type $iNumEnt]
            } else {
                #e.g a line of a polyline iNumEnt=={i_polyline i_local_subline}
                set entity_groups [list]
            }
            set num_groups [llength $entity_groups]
            ttk::label $w.n2.groups_l -text [_ "Groups"]:
            ttk::menubutton  $w.n2.groups_mb -menu $w.n2.groups_mb.menu -text $num_groups -direction right
            menu $w.n2.groups_mb.menu -tearoff 0
            foreach group $entity_groups {
                $w.n2.groups_mb.menu add command -label $group 
                #-command [list WarnWinText [GiD_EntitiesGroups get $group all_geometry]]
            }
            GIDReportData::AddData [_ "Groups"] $num_groups
            pack $w.n2.num1 $w.n2.num2 $w.n2.lay1 $w.n2.lay2 $w.n2.groups_l $w.n2.groups_mb -side left -padx 5 -anchor w           
        } else {
            pack $w.n2.num1 $w.n2.num2 $w.n2.lay1 $w.n2.lay2 -side left -padx 5 -anchor w
        }
    } else {
        pack $w.n.num1 $w.n.num2 -side left -padx 5 -anchor w
        if { $type == "POSTELEMENTS"} {
            set tipo_elemento [_ $InfoListEnt($iNumEnt,TipoElemento)]
            if { [ regexp {^[0-9]+\([0-9]+\)$} $iNumEnt]} {
                append tipo_elemento " " [_ "(Original element ID)"]
            }
            ttk::label $w.n.tel -text $tipo_elemento          
            pack $w.n.tel -side left -padx 5 -anchor w
            GIDReportData::AddData [_ "Element type"] [_ $InfoListEnt($iNumEnt,TipoElemento)]
        }

        ttk::frame $w.n2 -borderwidth 0
    }

    if { ( $type != "POSTNODES") && ( $type != "POSTELEMENTS")} {
        ttk::frame $w.m -borderwidth 0
        ttk::frame $w.m2 -borderwidth 0
        if { [catch { set elmtype $InfoListEnt($iNumEnt,elmtype) }] } {
            if { $type != "ELEMENTS" && $type != "NODES" } {
                ttk::label $w.m.l -text [_ "No meshing information"]
                pack $w.m.l -side left -ipadx 4
                GIDReportData::AddData [_ "No meshing information"]
            }
        } else {
            if { $InfoListEnt($iNumEnt,struc) == 0 || $InfoListEnt($iNumEnt,struc) == 1} {
                set lnames [list [_ "ElmType"] [_ "IsStruc"] [_ "Size"] [_ "Num\ncells"] [_ "Meshing"]]
            } elseif { $InfoListEnt($iNumEnt,struc) == 2 } {
                #2==Cartesian
                set lnames [list [_ "ElmType"] [_ "IsStruc"] [_ "Cartesian"] [_ "Meshing"]]
            } else {
                #unexpected
                set lnames [list [_ "ElmType"] [_ "IsStruc"] [_ "Size"] [_ "Num\ncells"] [_ "Meshing"]]
            }
            set inames { elmtype struc size num_divisions meshing }
            for {set i 0} {$i < 2} {incr i} {
                if { [info exists InfoListEnt($iNumEnt,[lindex $inames $i])] } {
                    if { $i == 1 &&  $InfoListEnt($iNumEnt,struc) == 2 } {
                        set text [_ "Cartesian"]
                    } else {
                        set text $InfoListEnt($iNumEnt,[lindex $inames $i])
                    }
                    ttk::label $w.m.l$i -text [lindex $lnames $i] -justify right
                    CreateEntry $w.m.i$i $text
                    GIDReportData::AddData [lindex $lnames $i] $text
                    pack $w.m.l$i $w.m.i$i -side left
                }
            }        
            for {set i 2} {$i < [llength $inames]} {incr i} {
                if { [info exists InfoListEnt($iNumEnt,[lindex $inames $i])] } {
                    if { ($i == 2 || $i==3) &&  $InfoListEnt($iNumEnt,struc) == 2 } {
                        #2==Cartesian"
                        continue
                    }
                    ttk::label $w.m2.l$i -text [lindex $lnames $i]
                    CreateEntry $w.m2.i$i $InfoListEnt($iNumEnt,[lindex $inames $i])
                    GIDReportData::AddData [lindex $lnames $i] $InfoListEnt($iNumEnt,[lindex $inames $i])
                    pack $w.m2.l$i $w.m2.i$i -side left -anchor w
                }
            }
            if { [info exists InfoListEnt($iNumEnt,skipmesh)] } {
                set i [llength $inames]
                ttk::label $w.m2.l$i -text [_ "Skip"]
                CreateEntry $w.m2.i$i $InfoListEnt($iNumEnt,skipmesh)
                GIDReportData::AddData [_ "Skip"] $InfoListEnt($iNumEnt,skipmesh)
                pack $w.m2.l$i $w.m2.i$i -side left -anchor w
                
            }
            if { [info exists InfoListEnt($iNumEnt,strucweight)] } {
                if { $type != "VOLUME" } {
                    #with weighs at start and end of line, are stored two values: (w0,w1)
                    set w0 0
                    set w1 0
                    scan $InfoListEnt($iNumEnt,strucweight) "(%f,%f)" w0 w1
                    if { $w0 != 0 || $w1 != 0 } {
                        ttk::frame $w.m25
                        ttk::label $w.m25.l -text [_ "Struct weight"]
                        CreateEntry $w.m25.i $InfoListEnt($iNumEnt,strucweight)
                        GIDReportData::AddData [_ "Struct weight"] $InfoListEnt($iNumEnt,strucweight)
                        pack $w.m25.l $w.m25.i -side left -anchor w
                    }
                }
            }
            if { [info exists InfoListEnt($iNumEnt,tops)] } {
                if { $type == "VOLUME" } {
                    #in case of volumes, this meshingdata represents semi-structured volume, and weights are top surfaces.
                    # and weights are top surfaces. if one top surface is <0 it means that that surface is the Master one.
                    ttk::frame $w.m25
                    ttk::label $w.m25.l -text [_ "Top surfaces"]
                    set a 0
                    set b 0
                    scan $InfoListEnt($iNumEnt,tops) "(%d,%d)" a b
                    set master [_ "Default"]
                    if { $a == 0 } {
                        set a [_ "Default"]
                    } elseif { $a < 0 } {
                        set a [expr abs($a)]
                        set master $a
                    }
                    if { $b == 0 } {
                        set b [_ "Default"]
                    } elseif { $b < 0 } {
                        set b [expr abs($b)]
                        set master $b
                    }
                    CreateEntry $w.m25.i "$a $b"
                    GIDReportData::AddData  [_ "Top surfaces"] "$a $b"                
                    ttk::label $w.m25.l2 -text [_ "Master surface"]:
                    CreateEntry $w.m25.i2 $master
                    GIDReportData::AddData [_ "Master surface"] $master
                    pack $w.m25.l $w.m25.i $w.m25.l2 $w.m25.i2 -side left -anchor w
                }
            }
            if { [info exists InfoListEnt($iNumEnt,mesher)] } {
                set num_mesher $InfoListEnt($iNumEnt,mesher)
                if { $num_mesher != 0 } {
                    set i 5
                    ttk::label $w.m2.l$i -text [_ "Mesher"]
                    CreateEntry $w.m2.i$i $InfoListEnt($iNumEnt,mesher)
                    GIDReportData::AddData [_ "Mesher"] $InfoListEnt($iNumEnt,mesher)
                    pack $w.m2.l$i $w.m2.i$i -side left -anchor w
                }
            }
            if { [info exists InfoListEnt($iNumEnt,nlevels1)] } {
                set firsth_1 $InfoListEnt($iNumEnt,firsth1)
                set firsth_2 $InfoListEnt($iNumEnt,firsth2)
                set nlevels_1 [expr {int($InfoListEnt($iNumEnt,nlevels1))}]
                set nlevels_2 [expr {int($InfoListEnt($iNumEnt,nlevels2))}]
                
                if { ($nlevels_1 != 0) || ($nlevels_2 != 0) } {
                    ttk::frame $w.mBLMesh
                    ttk::label $w.mBLMesh.blmlabel -text [_ "Boundary layer mesh"]:
                
                    GIDReportData::AddData  [_ "Boundary layer mesh"]:
                
                    if { ($nlevels_1 != 0) && ($nlevels_2 != 0) } {
                        ttk::label $w.mBLMesh.lab_1 -text [_ "First layer height"]:
                        CreateEntry $w.mBLMesh.fh "$firsth_1 $firsth_2"                
                        GIDReportData::AddData  [_ "First layer height"]: "$firsth_1 $firsth_2"                
                        ttk::label $w.mBLMesh.lab_2 -text [_ "NumLayers"]
                        CreateEntry $w.mBLMesh.nlev "$nlevels_1 $nlevels_2"
                        GIDReportData::AddData  [_ "NumLayers"] "$nlevels_1 $nlevels_2"
                    } else {
                        if { $nlevels_1 != 0 } {
                            set firsth $firsth_1
                            set nlevels $nlevels_1
                        } else {
                            set firsth $firsth_2
                            set nlevels $nlevels_2
                        }
                        ttk::label $w.mBLMesh.lab_1 -text [_ "First layer height"]:
                        CreateEntry $w.mBLMesh.fh "$firsth"
                        GIDReportData::AddData  [_ "First layer height"]: "$firsth"
                        ttk::label $w.mBLMesh.lab_2 -text [_ "NumLayers"]
                        CreateEntry $w.mBLMesh.nlev "$nlevels"
                        GIDReportData::AddData  [_ "NumLayers"] "$nlevels"
                    }
                    pack $w.mBLMesh.blmlabel -side top -anchor w
                    pack $w.mBLMesh.lab_1 $w.mBLMesh.fh $w.mBLMesh.lab_2 $w.mBLMesh.nlev -side left -anchor w
                }
            }
        }
        pack $w.n $w.n2 $w.m $w.m2 -side top -pady 2 -fill x
        if { [winfo exists $w.m25] } {
            pack $w.m25 -side top -pady 2 -fill x
        }
        if { [winfo exists $w.mBLMesh] } {
            pack $w.mBLMesh -side top -pady 2 -fill x
        }
    } else {
        pack $w.n $w.n2 -side top -pady 2 -fill x
    }

    if { $InfoListEnt($iNumEnt,mat) != 0 } {
        ttk::frame $w.nm
        ttk::label $w.nm.mat -text [_ "Material"]:
        if { ( "$type" != "POSTNODES") && ( "$type" != "POSTELEMENTS")} {
            CreateEntry $w.nm.mat2 $InfoListEnt($iNumEnt,mat)
            GIDReportData::AddData [_ "Material"] $InfoListEnt($iNumEnt,mat)
        } else {
            ttk::label $w.nm.mat2 -text $InfoListEnt($iNumEnt,mat)        
            GIDReportData::AddData [_ "Material"] $InfoListEnt($iNumEnt,mat)
        }
        pack $w.nm.mat $w.nm.mat2 -side left -padx 5 -anchor w
        pack $w.nm -side top -anchor w
    }

    switch -- $type {
        POINT {
            set cc { x: y: z: }
            for {set i 0} {$i < 3} {incr i} {
                ttk::frame $w.coo$i
                ttk::label $w.coo$i.cc -text [lindex $cc $i] -width 4
                CreateEntry $w.coo$i.n [lindex $InfoListEnt($iNumEnt,coord) $i]
                GIDReportData::AddData [lindex $cc $i] [lindex $InfoListEnt($iNumEnt,coord) $i]
                pack $w.coo$i.cc $w.coo$i.n -side left -anchor w
                pack $w.coo$i -side top -fill x -pady 2
            }            
        }
        STLINE {
            PutIniEnd $w $iNumEnt
        }
        ARCLINE {
            PutIniEnd $w $iNumEnt
            set cc { "2D center" Radius IniAng EndAng  }
            set rr { 2DCEN rad iang eang }
            for {set i 0} {$i < 4} {incr i} {
                ttk::frame $w.coo$i
                ttk::label $w.coo$i.cc -text [lindex $cc $i]:

                set text $InfoListEnt($iNumEnt,[lindex $rr $i])
                if { [lindex $rr $i] == "2DCEN" } {
                    set text [split $text ,]
                    set text [FormatDouble [lindex $text 0]],[FormatDouble [lindex $text 1]]
                }
                CreateEntry $w.coo$i.n $text
                GIDReportData::AddData [lindex $cc $i] $text
                pack $w.coo$i.cc  -side left -anchor w
                pack $w.coo$i.n -side left -anchor w
                pack $w.coo$i -side top -fill x -pady 2
            }
            PutListBox $w 1 yes
            for {set i 1} {$i <= 4} {incr i} {
                set text ""
                foreach item [split [string trim $InfoListEnt($iNumEnt,mat$i)]] {
                    set item [split $item =]
                    set text [concat $text [lindex $item 0]=[FormatDouble [lindex $item 1]]]
                }
                $w.f1.list insert end $text
                GIDReportData::AddData $text
            }
        }
        SPLINELINE {
            PutIniEnd  $w $iNumEnt
            set cc { "Num Points" "Num Knots" }
            set rr { NCtrl NKnots }
            ttk::frame $w.coo1
            ttk::label $w.coo1.cc0 -text [lindex $cc 0]:
            ttk::label $w.coo1.n0 -text $InfoListEnt($iNumEnt,[lindex $rr 0]) -relief sunken -width 4
            pack $w.coo1.cc0 $w.coo1.n0 -side left -anchor w
            pack $w.coo1 -side top -fill x -pady 2
        
            set antes ""
            set txt ""
            PutListBox $w 1 yes
            for {set i 1} {$i <= $InfoListEnt($iNumEnt,NCtrl)} {incr i} {
                $w.f1.list insert end [format "%-4i %s" $i $InfoListEnt($iNumEnt,Ctrl$i)]
                append txt ${antes}[format "%-4i %s" $i $InfoListEnt($iNumEnt,Ctrl$i)]
                set antes <br>
            }
            GIDReportData::AddData [lindex $cc 0] $txt

            ttk::frame $w.coo2
            ttk::label $w.coo2.cc1 -text [lindex $cc 1]:
            ttk::label $w.coo2.n1 -text $InfoListEnt($iNumEnt,[lindex $rr 1]) -relief sunken -width 4
            pack $w.coo2.cc1 $w.coo2.n1 -side left -anchor w
            pack $w.coo2 -side top -fill x -pady 2
            set antes ""
            set txt ""
            PutListBox $w 2 yes
            for {set i 1} {$i<= $InfoListEnt($iNumEnt,NKnots)} {incr i} {
                $w.f2.list insert end [format "%-4i %s" $i $InfoListEnt($iNumEnt,Knot$i)]
                append txt ${antes}[format "%-4i %s" $i $InfoListEnt($iNumEnt,Knot$i)]
                set antes <br>
            }
            GIDReportData::AddData [lindex $cc 1] $txt
        }
        NURBLINE {
            PutIniEnd  $w $iNumEnt
            ttk::frame $w.coo3
            ttk::label $w.coo3.cc0 -text [_ "Length"]:
            CreateEntry $w.coo3.n0 $InfoListEnt($iNumEnt,TotalLength)
            GIDReportData::AddData [_ "Length"] $InfoListEnt($iNumEnt,TotalLength)
            pack $w.coo3.cc0 $w.coo3.n0 -side left -anchor w
            pack $w.coo3 -side bottom -fill x -pady 2
            NoteBook $w.opt
            FrameForNURBSurfPoints [$w.opt insert end Points -text [_ "Points"]] $iNumEnt
            FrameForNURBSurfKnots [$w.opt insert end Knots -text [_ "Knots"]] $iNumEnt
            $w.opt raise Points
            pack $w.opt -fill both -expand yes -side top
        }
        NURBSURFACE {
            set commands { IsTrimmed center normal }
            set labels [list [_ "Is trimmed"] [_ "Label position"] [_ "Normal"]]
            foreach i $commands label $labels {
                ttk::frame $w.cou$i
                ttk::label $w.cou$i.cc -text $label:
                if { $i != "IsTrimmed" } {
                    set coords ""
                    foreach item $InfoListEnt($iNumEnt,$i) {
                        lappend coords [FormatDouble $item]
                    }
                } else {
                    set coords $InfoListEnt($iNumEnt,$i)
                }
                CreateEntry $w.cou$i.n $coords
                if { $i == "IsTrimmed" } {
                    ttk::button $w.cou$i.b -text [_ "Lines uv"] -command [list ShowLinesUV $iNumEnt]
                }
                GIDReportData::AddData $label $coords

                pack $w.cou$i.cc  -side left -anchor w
                if { $i == "IsTrimmed" } {
                    pack $w.cou$i.b $w.cou$i.n -side right -anchor e -fill x -expand yes
                } else {
                    pack $w.cou$i.n -side right -anchor e -fill x -expand yes
                }
                pack $w.cou$i -side bottom -fill x -pady 2
            }
            NoteBook $w.opt
            FrameForNURBSurfLines [$w.opt insert end Lines -text [_ "Lines"]] $iNumEnt
            FrameForNURBSurfPoints [$w.opt insert end Points -text [_ "Points"]] $iNumEnt
            FrameForNURBSurfKnots [$w.opt insert end KnotsU -text [_ "Knots in U"]] $iNumEnt U
            FrameForNURBSurfKnots [$w.opt insert end KnotsV -text [_ "Knots in V"]] $iNumEnt V
            $w.opt raise Lines

            pack $w.opt -fill both -expand yes -side top

        }
        POLYLINE {
            PutIniEnd $w $iNumEnt
            FillForPolyline $w $iNumEnt
        }
        COONSURFACE {
            FillForSurface $w $iNumEnt
        }
        PLSURFACE {
            FillForSurface $w $iNumEnt
        }
        CONTACTSURFACE {
            FillForSurface $w $iNumEnt
        }
        POLYSURFACE {
            set commands { center normal }
            set labels [list [_ "Label position"] [_ "Normal"]]
            foreach i $commands label $labels {
                ttk::frame $w.cou$i
                ttk::label $w.cou$i.cc -text $label:
                CreateEntry $w.cou$i.n $InfoListEnt($iNumEnt,$i)
                GIDReportData::AddData $label $InfoListEnt($iNumEnt,$i)

                pack $w.cou$i.cc  -side left -anchor w
                pack $w.cou$i.n -side right -anchor e -fill x -expand yes
                pack $w.cou$i -side bottom -fill x -pady 2
            }

            NoteBook $w.opt
            FrameForNURBSurfLines [$w.opt insert end "Lines" -text [_ "Lines"]] $iNumEnt
            FrameForPolySurfSurfaces [$w.opt insert end Surfaces -text [_ "Surfaces"]] $iNumEnt
            FrameForPolySurfIntLines [$w.opt insert end IntLines -text [_ "Int.Lines"]] $iNumEnt
            $w.opt raise Lines

            pack $w.opt -fill both -expand yes -side top
        }
        VOLUME {
            FillForVolume $w $iNumEnt
        }
        CONTACTVOLUME {
            FillForContactVolume $w $iNumEnt
        }
        NODES {
            set cc { x y z }
            for {set i 0} {$i < 3} {incr i} {
                ttk::frame $w.coo$i
                ttk::label $w.coo$i.cc -text [lindex $cc $i]: -width 5
                CreateEntry $w.coo$i.n [lindex $InfoListEnt($iNumEnt,coord) $i] 12
                GIDReportData::AddData [lindex $cc $i] [lindex $InfoListEnt($iNumEnt,coord) $i]
                pack $w.coo$i.cc $w.coo$i.n -side left -anchor w
                pack $w.coo$i -side top -fill x -pady 2
            }           
        }
        ELEMENTS {
            set info [GiD_Info list_entities -more elements $iNumEnt]
            if { ![regexp {Type=([^ ]+) Nnode=([0-9]+) Volume=([^ ]+)} $info dummy type nnode volume] } {
                set type ?
                set nnode ?
                set volume ?
            }
            ttk::frame $w.more
            ttk::label $w.more.ltype -text [_ "Type"]:
            CreateEntry $w.more.etype $type
            if { $type == "Sphere" } {
                ttk::label $w.more.lvolume -text [_ "Radius"]:
                set r [FormatDouble [expr {pow(3.0/4.0*$volume/$MathUtils::PI,1.0/3.0)}]]
                CreateEntry $w.more.evolume $r
                GIDReportData::AddData [_ "Radius"] $r
            } elseif { $type == "Circle" } {
                ttk::label $w.more.lvolume -text [_ "Radius"]:
                set r [FormatDouble [expr {sqrt($volume/$MathUtils::PI)}]]
                CreateEntry $w.more.evolume $r
                GIDReportData::AddData [_ "Radius"] $r
            } else {
                ttk::label $w.more.lvolume -text [_ "Volume"]
                CreateEntry $w.more.evolume [FormatDouble $volume]
                GIDReportData::AddData [_ "Volume"] [FormatDouble $volume]
            }
            pack $w.more.ltype $w.more.etype -side left -anchor w -ipadx 2
            pack $w.more.lvolume $w.more.evolume -side left -anchor w -ipadx 2
            pack $w.more -side top -fill x -pady 2

            ttk::frame $w.coo
            ttk::label $w.coo.cc -text [_ "Nodes"]:
            CreateEntry $w.coo.n $InfoListEnt($iNumEnt,nodes)
            GIDReportData::AddData [_ "Nodes"] $InfoListEnt($iNumEnt,nodes)
            pack $w.coo.cc $w.coo.n -side left -anchor w -ipadx 2
            pack $w.coo -side top -fill x -pady 2
        }
        POSTNODES {
            ttk::frame $w.f
            if { $InfoListEnt($iNumEnt,EsPuntoGauss)} {
                ttk::label $w.f.pgl -text [ format "+ %s:" [_ "Gauss Point"]] -font BigFont
                pack $w.f.pgl -side left -anchor w -pady 2
                GIDReportData::AddData [_ "Gauss Point"]
            }
            set cc { x y z }
            for {set i 0} {$i < 3} {incr i} {
                ttk::frame $w.f.coo$i
                ttk::label $w.f.coo$i.cc -text [lindex $cc $i]: -width 5
                CreateEntry $w.f.coo$i.n [lindex $InfoListEnt($iNumEnt,coord) $i] 12
                pack $w.f.coo$i.cc $w.f.coo$i.n -side left -anchor w
                pack $w.f.coo$i -side top -fill x -pady 2
                GIDReportData::AddData [lindex $cc $i] [lindex $InfoListEnt($iNumEnt,coord) $i]
            }            
            pack $w.f -side top -fill x -anchor w

            if { [ winfo exists $w.fc]} { destroy $w.fc}
            set c [ CreateScrolledCanvas $w.fc]
            bind [winfo toplevel $w] <4> "$c yview scroll -3 units"
            bind [winfo toplevel $w] <5> "$c yview scroll 3 units"
            pack $w.fc -fill both -expand yes

            if { [info exists InfoListEnt($iNumEnt,UsedBy)] && $InfoListEnt($iNumEnt,UsedBy) != "" } {
                ttk::frame $c.uso_sep
                ttk::label $c.uso_sep.l -text [ format "+ %s:" [_ "Used by"]] -font BigFont
                grid $c.uso_sep.l -sticky w
                GIDReportData::AddData [_ "Used by"]

                AddToScrolledCanvas $w.fc $c.uso_sep

                ttk::frame $c.conec -borderwidth 0
                set lst_mallas $InfoListEnt($iNumEnt,UsedBy)
                if { ( [ llength $lst_mallas] == 1) && ( "$lst_mallas" == "N/A")} {
                    set txt [_ "No elements"]
                    ttk::label $c.conec.l -text $txt
                    grid $c.conec.l -sticky w -padx 4
                    GIDReportData::AddData $txt
                } else {
                    set idx 0
                    foreach i_malla $lst_mallas {
                        set txt $i_malla
                        if { [ regexp {^(.*): *(([0-9]*\,? *)*)$} $i_malla dum nombre_malla lst_elems]} {
                            ttk::label $c.conec.l$idx -text $nombre_malla
                            regsub -all  {,} $lst_elems {} lst_idx
                            CreateEntry $c.conec.e$idx $lst_idx
                            grid $c.conec.l$idx $c.conec.e$idx -sticky w -padx 4
                            grid configure $c.conec.l$idx -sticky e
                        } else {
                            ttk::label $c.conec.l$idx -text $txt
                            grid $c.conec.l$idx -sticky w -padx 4
                        }
                        GIDReportData::AddData $txt
                        incr idx
                    }
                }

                AddToScrolledCanvas $w.fc $c.conec
            }

            if { [info exists InfoListEnt($iNumEnt,AnalysisName)] && $InfoListEnt($iNumEnt,AnalysisName) != "" } {
                ttk::frame $c.resAnalysisName
                set txt [_ "Analysis name: %s" $InfoListEnt($iNumEnt,AnalysisName)]
                ttk::label $c.resAnalysisName.l -text "+ $txt" -font BigFont
                pack $c.resAnalysisName.l -pady 4
                GIDReportData::AddData $txt
                AddToScrolledCanvas $w.fc $c.resAnalysisName
            }

            if { [info exists InfoListEnt($iNumEnt,StepValue)] && $InfoListEnt($iNumEnt,StepValue) != "" } {
                ttk::frame $c.resStepValue
                set txt [_ "Step value: %s" $InfoListEnt($iNumEnt,StepValue)]
                ttk::label $c.resStepValue.l -text "+ $txt" -font BigFont
                pack $c.resStepValue.l -pady 4
                GIDReportData::AddData $txt
                AddToScrolledCanvas $w.fc $c.resStepValue
            }

            if { [info exists InfoListEnt($iNumEnt,CurrentResultName)] && $InfoListEnt($iNumEnt,CurrentResultName) != "" } {
                ttk::frame $c.ressep
                ttk::label $c.ressep.l -text [ format "+ %s:" [_ "Current result"]] -font BigFont
                pack $c.ressep.l -pady 0
                GIDReportData::AddData [_ "Current result"]

                AddToScrolledCanvas $w.fc $c.ressep

                ttk::frame $c.curres -borderwidth 0
                ttk::label $c.curres.l -text [ lindex $InfoListEnt($iNumEnt,CurrentResultName) 0]
                pack $c.curres.l -side left -anchor e -pady 2 -padx 2
                
                # to create the entry:    Vector   x: 2.3  y: 3.3 z: -3.0 ||: 45.3
                set comp_desc [ lindex $InfoListEnt($iNumEnt,CurrentResultName) 1]
                set CurrentResultValues [ lindex $InfoListEnt($iNumEnt,CurrentResultValues) 0]

                set lr [ llength $CurrentResultValues]

                for { set ir 0} { $ir < $lr} { incr ir} {
                    set val [ lindex $CurrentResultValues $ir]
                    if { [ scan $val %g pp] == 1} {
                        set width_entry [ string length $val]
                    } else {
                        set width_entry [ string length $val]
                    }
                    if { $width_entry < 4} { set width_entry 4}

                    if { "[ lindex $comp_desc $ir]" == "" } {
                        CreateEntry $c.curres.e$ir $val $width_entry
                        pack $c.curres.e$ir -side left -anchor w -pady 2 -padx 2
                        if { !$ir} {
                            GIDReportData::AddData [ lindex $InfoListEnt($iNumEnt,CurrentResultName) 0] $val
                        } else {
                            GIDReportData::AddData {} $val
                        }
                    } else {
                        ttk::frame $c.curres.fe$ir -borderwidth 0
                        set desc " [ lindex $comp_desc $ir]"
                        ttk::label $c.curres.fe$ir.l -text ${desc}:
                        CreateEntry $c.curres.fe$ir.e $val $width_entry
                        pack $c.curres.fe$ir.l $c.curres.fe$ir.e -side left -anchor w
                        pack $c.curres.fe$ir -side left -anchor w -pady 2 -padx 2
                        if { !$ir} {
                            GIDReportData::AddData [ lindex $InfoListEnt($iNumEnt,CurrentResultName) 0] "$desc" $val
                        } else {
                            GIDReportData::AddData {} "$desc" $val
                        }
                    }
                }
                AddToScrolledCanvas $w.fc $c.curres
            }

            if { [info exists InfoListEnt($iNumEnt,MoreResults)] && $InfoListEnt($iNumEnt,MoreResults) != "" } {
                ttk::frame $c.ressep2
                ttk::label $c.ressep2.l -text [ format "+ %s:" [_ "All results"]] -font BigFont
                pack $c.ressep2.l -pady 0
                GIDReportData::AddData [_ "All results"]

                AddToScrolledCanvas $w.fc $c.ressep2

                set ifr 0
                ttk::frame $c.res -borderwidth 0

                foreach res $InfoListEnt($iNumEnt,MoreResults) {
                    ttk::label $c.res.l$ifr -text [ lindex $res 0]

                    set comp_desc [ lindex $res 1]
                    set resval [ lindex [ lindex $res 2] 0]
                    set lr [ llength $resval]

                    grid $c.res.l$ifr -row $ifr -column 0 -sticky e -pady 2 -padx 2
                    ttk::frame $c.res.f$ifr -border 0
                    for { set ir 0} { $ir < $lr} { incr ir} {
                        set val [ lindex $resval $ir]
                        if { [ scan $val %g pp] == 1} {
                            set width_entry [ string length $val]
                        } else {
                            set width_entry [ string length $val]
                        }
                        if { $width_entry < 4} { set width_entry 4}

                        if { "[ lindex $comp_desc $ir]" == "" } {
                            CreateEntry $c.res.f${ifr}.e$ir $val $width_entry
                            grid $c.res.f${ifr}.e$ir -row 0 -column $ir -sticky w
                            if { !$ir} {
                                GIDReportData::AddData [ lindex $res 0] $val
                            } else {
                                GIDReportData::AddData {} $val
                            }
                        } else {
                            ttk::frame $c.res.f${ifr}.fe$ir -borderwidth 0
                            set desc " [ lindex $comp_desc $ir]"

                            ttk::label $c.res.f${ifr}.fe$ir.l -text ${desc}:
                            CreateEntry $c.res.f${ifr}.fe$ir.e $val $width_entry
                            grid $c.res.f${ifr}.fe$ir.l $c.res.f${ifr}.fe$ir.e -sticky e
                            grid $c.res.f${ifr}.fe$ir -row 0 -column $ir -sticky w

                            if { !$ir} {
                                GIDReportData::AddData [ lindex $res 0] "$desc" $val
                            } else {
                                GIDReportData::AddData {} "$desc" $val
                            }
                        }
                    }
                    grid $c.res.f$ifr -row $ifr -column 1 -sticky w -pady 2 -padx 2
                    incr ifr
                }

                AddToScrolledCanvas $w.fc $c.res
            }
        }
        POSTELEMENTS {
            ttk::frame $w.coo
            ttk::label $w.coo.cc -text [_ "Nodes"]: -width 6
            CreateEntry $w.coo.n $InfoListEnt($iNumEnt,nodes)
            pack $w.coo.cc $w.coo.n -side left -anchor w -ipadx 2
            pack $w.coo -side top -fill x -pady 2
            GIDReportData::AddData [_ "Nodes"] $InfoListEnt($iNumEnt,nodes)

            if { [info exists InfoListEnt($iNumEnt,SphereCircleRadius)] || [info exists InfoListEnt($iNumEnt,CircleNormal)] } {
                ttk::frame $w.sc
                if { [ info exists InfoListEnt($iNumEnt,SphereCircleRadius)]} {
                    ttk::label $w.sc.radius -text [_ "Radius"]
                    CreateEntry $w.sc.r $InfoListEnt($iNumEnt,SphereCircleRadius)
                    pack $w.sc.radius $w.sc.r -side left -anchor w -ipadx 2
                }
                if { [ info exists InfoListEnt($iNumEnt,CircleNormal)]} {
                    ttk::label $w.sc.normal -text [_ "Normal"]
                    CreateEntry $w.sc.n $InfoListEnt($iNumEnt,CircleNormal)
                    pack $w.sc.normal $w.sc.n -side left -anchor w -ipadx 2
                }
                pack $w.sc -side top -fill x -pady 2
            }

            if { [info exists InfoListEnt($iNumEnt,GaussPoints)] && $InfoListEnt($iNumEnt,GaussPoints) != "" } {

                if { [ winfo exists $w.fc]} { destroy $w.fc}
                set c [ CreateScrolledCanvas $w.fc]

                bind [winfo toplevel $w] <4> "$c yview scroll -3 units"
                bind [winfo toplevel $w] <5> "$c yview scroll 3 units"
                pack $w.fc -fill both -expand yes
                ttk::frame $c.pgsep
                ttk::label $c.pgsep.l -text [ format "+ %s:" [_ "Gauss points"]] -font BigFont
                pack $c.pgsep.l -pady 2 -anchor w
                GIDReportData::AddData [_ "Gauss points"]

                AddToScrolledCanvas $w.fc $c.pgsep

                set ifr 0
                ttk::frame $c.pg -borderwidth 0
                foreach res $InfoListEnt($iNumEnt,GaussPoints) {
                    set gp_name "[ lindex $res 0]: "
                    ttk::label $c.pg.l$ifr -text $gp_name
                    grid $c.pg.l$ifr -row $ifr -column 0 -sticky e -pady 2
                    set desc "[ lindex $res 1] points, "
                    append desc "[ lindex $res 2]"
                    if { [ llength $res] == 4} {
                        append desc ", [ lindex $res 3]"
                    }
                    CreateEntry $c.pg.e$ifr $desc 40
                    grid $c.pg.e$ifr -row $ifr -column 1 -sticky w -pady 2

                    GIDReportData::AddData $gp_name $desc
                
                    incr ifr
                }
                AddToScrolledCanvas $w.fc $c.pg
            }

            if { [ info exists InfoListEnt($iNumEnt,AnalysisName)] && \
                    ( $InfoListEnt($iNumEnt,AnalysisName) != "") } {
                if { ![ info exists c] || ![ winfo exists $c]} {
                    if { [ winfo exists $w.fc]} { destroy $w.fc}
                    set c [ CreateScrolledCanvas $w.fc]

                    bind [winfo toplevel $w] <4> "$c yview scroll -3 units"
                    bind [winfo toplevel $w] <5> "$c yview scroll 3 units"

                    pack $w.fc -fill both -expand yes
                }
                ttk::frame $c.resAnalysisName
                set txt [_ "Analysis name: %s" $InfoListEnt($iNumEnt,AnalysisName)]
                ttk::label $c.resAnalysisName.l -text "+ $txt" -font BigFont
                pack $c.resAnalysisName.l -pady 4
                GIDReportData::AddData $txt
                AddToScrolledCanvas $w.fc $c.resAnalysisName
            }

            if { [info exists InfoListEnt($iNumEnt,StepValue)] && $InfoListEnt($iNumEnt,StepValue) != "" } {
                if { ![ info exists c] || ![ winfo exists $c]} {
                    if { [ winfo exists $w.fc]} { destroy $w.fc}
                    set c [ CreateScrolledCanvas $w.fc]

                    bind [winfo toplevel $w] <4> "$c yview scroll -3 units"
                    bind [winfo toplevel $w] <5> "$c yview scroll 3 units"

                    pack $w.fc -fill both -expand yes
                }
                ttk::frame $c.resStepValue
                set txt [_ "Step value: %s" $InfoListEnt($iNumEnt,StepValue)]
                ttk::label $c.resStepValue.l -text "+ $txt" -font BigFont
                pack $c.resStepValue.l -pady 4
                GIDReportData::AddData $txt
                AddToScrolledCanvas $w.fc $c.resStepValue
            }

            if { [info exists InfoListEnt($iNumEnt,CurrentResultName)] && $InfoListEnt($iNumEnt,CurrentResultName) != "" } {

                if { ![info exists c] || ![winfo exists $c]} {
                    if { [winfo exists $w.fc]} { destroy $w.fc}
                    set c [CreateScrolledCanvas $w.fc]

                    bind [winfo toplevel $w] <4> "$c yview scroll -3 units"
                    bind [winfo toplevel $w] <5> "$c yview scroll 3 units"

                    pack $w.fc -fill both -expand yes
                }

                ttk::frame $c.ressep
                ttk::label $c.ressep.l -text [ format "+ %s:" [_ "Current result"]] -font BigFont
                pack $c.ressep.l -pady 2
                GIDReportData::AddData [_ "Current result"]

                AddToScrolledCanvas $w.fc $c.ressep

                ttk::frame $c.curres -borderwidth 0
                ttk::frame $c.curres.fn -borderwidth 0
                ttk::label $c.curres.fn.l -text [ lindex $InfoListEnt($iNumEnt,CurrentResultName) 0]               
                ttk::label $c.curres.fn.lpg -text [_ "on '%s'" [ lindex $InfoListEnt($iNumEnt,CurrentResultName) 1]]

                set res_name [ lindex $InfoListEnt($iNumEnt,CurrentResultName) 0]
                append res_name " "
                append res_name [_ "on '%s'" [ lindex $InfoListEnt($iNumEnt,CurrentResultName) 1]]
               
                grid $c.curres.fn.l -sticky ne -pady 2
                grid $c.curres.fn.lpg -sticky ne -pady 2

                set comp_desc [ lindex $InfoListEnt($iNumEnt,CurrentResultName) 2]
                ttk::frame $c.curres.fv -borderwidth 0
                set l 0
                foreach iv $InfoListEnt($iNumEnt,CurrentResultValues) {
                    set lr [ llength $iv]                
                    # to create the entry:    Vector   x: 2.3  y: 3.3 z: -3.0 ||: 45.3
                    for { set ir 0} { $ir < $lr} { incr ir} {
                        set val [ lindex $iv $ir]
                        if { [ scan $val %g pp] == 1} {
                            set width_entry [ string length $val]
                            # 10
                        } else {
                            set width_entry [ string length $val]
                        }
                        if { $width_entry < 4} { set width_entry 4}

                        if { "[ lindex $comp_desc $ir]" == "" } {
                            CreateEntry $c.curres.fv.e${l}_$ir [ lindex $iv $ir] $width_entry
                            grid $c.curres.fv.e${l}_$ir -row $l -column $ir -sticky w
                            if { !$l && !$ir} {
                                GIDReportData::AddData $res_name [ lindex $iv $ir]
                            } else {
                                GIDReportData::AddData {} [ lindex $iv $ir]
                            }
                        } else {
                            ttk::frame $c.curres.fv.fe${l}_$ir -borderwidth 0
                            set desc " [ lindex $comp_desc $ir]"
                            ttk::label $c.curres.fv.fe${l}_$ir.l -text ${desc}:
                            CreateEntry $c.curres.fv.fe${l}_$ir.e $val $width_entry
                            grid $c.curres.fv.fe${l}_$ir.l $c.curres.fv.fe${l}_$ir.e -sticky e
                            grid $c.curres.fv.fe${l}_$ir -row $l -column $ir -sticky w
                            if { !$l && !$ir} {
                                GIDReportData::AddData $res_name "$desc" $val
                            } else {
                                GIDReportData::AddData {} "$desc" $val
                            }
                        }
                    }
                    incr l
                }

                grid $c.curres.fn $c.curres.fv -sticky ne -pady 2 -padx 2
                AddToScrolledCanvas $w.fc $c.curres

            }

            if { [info exists InfoListEnt($iNumEnt,MoreResults)] && $InfoListEnt($iNumEnt,MoreResults) != "" } {

                if { ![ info exists c] || ![ winfo exists $c]} {
                    if { [ winfo exists $w.fc]} { destroy $w.fc}
                    set c [ CreateScrolledCanvas $w.fc]

                    bind [winfo toplevel $w] <4> "$c yview scroll -3 units"
                    bind [winfo toplevel $w] <5> "$c yview scroll 3 units"

                    pack $w.fc -fill both -expand yes
                }

                ttk::frame $c.ressep2
                ttk::label $c.ressep2.l -text [ format "+ %s:" [_ "All results"]] -font BigFont
                pack $c.ressep2.l -pady 2
                GIDReportData::AddData [_ "All results"]

                AddToScrolledCanvas $w.fc $c.ressep2

                ttk::frame $c.res -borderwidth 0
                set ir 0
                foreach or $InfoListEnt($iNumEnt,MoreResults) {
                    ttk::frame $c.res.fl$ir
                    ttk::label $c.res.fl$ir.l -text [ lindex $or 0]
                    ttk::label $c.res.fl$ir.lpg -text [_ "on '%s'" [ lindex $or 1]]
                    grid $c.res.fl$ir.l -sticky ne
                    grid $c.res.fl$ir.lpg -sticky ne
                    set res_name [ lindex $or 0]
                    append res_name " "
                    append res_name [_ "on '%s'" [ lindex $or 1]]
                    set comp_desc [ lindex $or 2]
                
                    ttk::frame $c.res.f$ir -borderwidth 0
                    set ifr 0
                    foreach iv [ lindex $or 3] {
                        set lr [ llength $iv]                
                        # to create the entry:    Vector   x: 2.3  y: 3.3 z: -3.0 |v|: 45.3
                        for { set iir 0} { $iir < $lr} { incr iir} {
                            set val [ lindex $iv $iir]
                            if { [ scan $val %g pp] == 1} {
                                set width_entry [ string length $val]
                            } else {
                                set width_entry [ string length $val]
                            }
                            if { $width_entry < 4} { set width_entry 4}
                            if { "[ lindex $comp_desc $iir]" == "" } {
                                CreateEntry $c.res.f${ir}.e${ifr}_$iir $val $width_entry
                                grid $c.res.f${ir}.e${ifr}_$iir -row $ifr -column $iir -sticky w
                                if { !$ifr && !$iir} {
                                    GIDReportData::AddData $res_name $val
                                } else {
                                    GIDReportData::AddData {} $val
                                }
                            } else {
                                ttk::frame $c.res.f${ir}.fe${ifr}_$iir -borderwidth 0
                                set desc " [ lindex $comp_desc $iir]"
                                ttk::label $c.res.f${ir}.fe${ifr}_$iir.l -text ${desc}:
                                CreateEntry $c.res.f${ir}.fe${ifr}_$iir.e $val $width_entry
                                grid $c.res.f${ir}.fe${ifr}_$iir.l $c.res.f${ir}.fe${ifr}_$iir.e -sticky e
                                grid $c.res.f${ir}.fe${ifr}_$iir -row $ifr -column $iir -sticky w
                                if { !$ifr && !$iir} {
                                    GIDReportData::AddData $res_name "$desc" $val
                                } else {
                                    GIDReportData::AddData {} "$desc" $val
                                }
                            }
                        }
                        incr ifr
                    }
                    grid $c.res.fl$ir -row $ir -column 0 -sticky ne -pady 2 -padx 2
                    grid $c.res.f$ir -row $ir -column 1 -sticky nw -pady 2 -padx 2
                    incr ir
                }

                AddToScrolledCanvas $w.fc $c.res
            }

        }
    }
    set InfoListEnt(ReportData) [ GIDReportData::GetDataHTML 1]
}

proc ListPolyEntitiesEnt { w entnum maxnum entr} {
    set inum [$entr get]
    if { $inum <= 0 || $inum > $maxnum } {
        set w2 [winfo parent $w]
        mkDialogN $w2.modal [list -text [_ "Line number out of limits"] -aspect 250 -justify left] [list [_ "Ok"] OK ]
        return
    }
    ListPolyEntities $w.enti $inum $entnum $maxnum
}

proc ChangeFocusTo { w } {

    if { [winfo exists $w.enti.buts.e] } {
        focus $w.enti.buts.e
    } else { catch { focus $w.buts.next } }

}

proc ListPolyEntities { w inum entnum maxnum } {

    set inumnext [expr $inum+1]
    if { $inumnext > $maxnum } { set inumnext 1 }
    set inumprev [expr $inum-1]
    if { $inumprev <= 0 } { set inumprev $maxnum }
    if { ![winfo exists $w] } {
        InitWindow2 $w -title [_ "List Poly Entities"] \
        -geometryvariable PreListPolylinesWindowGeom -ontop
        if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0
        set x [ winfo x [winfo parent [winfo parent $w]]]
        set y [ winfo y [winfo parent [winfo parent $w]]]
        wm geometry $w 210x420+[expr $x+20]+[expr $y-100]
        wm minsize $w 210 420

        ttk::label $w.ltit -font BigFont -anchor center        
        #-style BigFont.TLabel     
        ttk::frame $w.buts -style BottomFrame.TFrame
        ttk::button $w.buts.list -text [_ "List"] -style BottomFrame.TButton
        ttk::button $w.buts.prev -text [_ "Prev"] -style BottomFrame.TButton
        ttk::button $w.buts.next -text [_ "Next"] -style BottomFrame.TButton
        bind $w <Tab> "$w.buts.next invoke ; break "
        bind $w <Shift-Tab> "$w.buts.prev invoke ; break"
        ttk::button $w.buts.close -text [_ "Close"] -command "destroy $w" -style BottomFrame.TButton
        ttk::label $w.d -text [_ "Use 'Tab' and 'Shift-Tab'"]
       
        pack $w.buts.list $w.buts.prev $w.buts.next $w.buts.close -side left

        pack $w.ltit -side top
        pack $w.buts $w.d -side bottom
    }
    $w.buts.prev configure -command "ListPolyEntities $w $inumprev \"$entnum\" $maxnum"
    $w.buts.next configure -command "ListPolyEntities $w $inumnext \"$entnum\" $maxnum"
    set entnum "$entnum $inum"
    ShowEntity $w.enti $entnum
}

proc NextEntity { iNumEntList w } {
    global ListOfEnt NumEntList
    incr iNumEntList
    if { $iNumEntList >= $NumEntList } { set iNumEntList 0 }
    ShowEntity $w.enti [lindex $ListOfEnt $iNumEntList]
    $w.buts.next configure -command "NextEntity $iNumEntList $w"
    $w.buts.prev configure -command "PrevEntity $iNumEntList $w"
}

proc PrevEntity { iNumEntList w } {
    global ListOfEnt NumEntList
    incr iNumEntList -1
    if { $iNumEntList < 0 } {
        set iNumEntList [expr $NumEntList-1]
    }
    ShowEntity $w.enti [lindex $ListOfEnt $iNumEntList]
    $w.buts.next configure -command "NextEntity $iNumEntList $w"
    $w.buts.prev configure -command "PrevEntity $iNumEntList $w"
}

proc CloseListWindow { w } {
    global NumListWindows
    incr NumListWindows -1
    if { $NumListWindows < 0 } { set NumListWindows 0 }
    if { $NumListWindows == 0 } {
        unset -nocomplain ListOfEnt
        unset -nocomplain InfoListEnt
    }
    destroy $w
}

proc ShowMassProperties { listorder } {
    WarnWinText [GiD_Info ListMassProperties {*}$listorder]
}

proc ListEntities { listorder {w .gid.listent} } {
    global wListEnt ListOfEnt NumEntList NumListWindows InfoListEnt
    
    if { ![winfo exists $w] } {
        if { ![info exists ::GidPriv(PreListEntitiesWindowGeom)] } {
            set assign_default_size 1
        } else {
            set assign_default_size 1
        }
        InitWindow2 $w -title [_ "List entities"] \
            -geometryvariable PreListEntitiesWindowGeom \
            -ontop
        if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0
        if { $assign_default_size } {
            wm geometry $w 360x480
        }
    } else {
        raise $w
    }

    if { ![info exists NumListWindows] || $NumListWindows == 0 } {
        set NumListWindows 1
    }
    if { $listorder == "" } {
        incr NumListWindows
    } else {
        for {set i 2} {$i <= $NumListWindows} {incr i} {
            CloseListWindow $w$i
        }
    }


    if { $NumListWindows != 1 } {
        if { $NumListWindows == 2 } {
            set wbefore $w
        } else { set wbefore $w[expr $NumListWindows-1] }
        catch {
        regexp {([^x]*)x([^+-]*)([+-][^+-]*)([+-].*)} [wm geometry $wbefore] trash wf hf xf yf

        set yfnew [expr $yf-20]
        if { $yfnew<0 && $yf>=0 } { set yfnew -0}
        if { $yfnew > 0 } { set yfnew +$yfnew }

        if { $yf > 0 } { set yf +$yf }
            wm geometry $w [join "$wf x $hf $xf $yfnew" ""]
        }
    }

    set wListEnt $w
    if { ![winfo exists $w.ltit] } {
        ttk::label $w.ltit -font BigFont -anchor center       
        #-style BigFont.TLabel
    }

    if { ![winfo exists $w.buts] } {
        ttk::frame $w.buts  -style BottomFrame.TFrame
        ttk::button $w.buts.list -text [_ "List"] -command "MakeListOfEnt $w [list $listorder]" -style BottomFrame.TButton
        ttk::button $w.buts.prev -text [_ "Prev"] -command "PrevEntity 0 $w" -style BottomFrame.TButton
        ttk::button $w.buts.next -text [_ "Next"] -command "NextEntity 0 $w" -style BottomFrame.TButton
        ttk::button $w.buts.close -text [_ "Close"] -command "CloseListWindow $w" -style BottomFrame.TButton        
        ttk::button $w.buts.toReport -image [gid_themes::GetImage ToReport.png medium_icons] -style BottomFrame.TButton -command {
            global InfoListEnt
            if { "$InfoListEnt(ReportData)" != ""} {
                GIDReport::AddToReport $InfoListEnt(ReportData)
            }
        }

        GidHelp $w.buts.toReport [ GIDReport::GetHelpForImage ToReport]

        ttk::label $w.d
    } else {
        $w.buts.list configure -command "MakeListOfEnt $w [list $listorder]"
    }

    $w.d configure -text [_ "Use 'Tab' and 'Shift-Tab'"]
    bind $w <Tab> "$w.buts.next invoke ; break"
    bind $w <Shift-Tab> "$w.buts.prev invoke  ; break"
    catch {
        bind $w <ISO_Left_Tab> "$w.buts.prev invoke  ; break"
    }
    pack forget $w.buts.close
    if { $listorder == "status" } {
        pack $w.buts.list $w.buts.prev $w.buts.next $w.buts.close -padx 1 -pady 3
    } else {
        pack $w.buts.list $w.buts.prev $w.buts.next $w.buts.close -side left -padx 1 -pady 5
    }
    pack $w.buts.toReport -side right

    pack $w.ltit -side top
    pack $w.buts $w.d -side bottom -fill x

    if { $listorder != "" } {
        unset -nocomplain ListOfEnt
        unset -nocomplain NumEntList
        unset -nocomplain InfoListEnt          
        if { $listorder == "status" } {   
            set NumEntList 0         
            ShowStatus $w               
        } else {
            ShowListEntities $w $listorder    
        }      
    }

    if { $NumEntList > 0 } { 
        ShowEntity $w.enti [lindex $ListOfEnt 0] 
    }


    # la ventana de "List Entitities Tot"
    set w2 $w.list
    if { [winfo exists $w2] } {
        MakeListOfEnt $w [list $listorder]
    }

    bind $w <Escape> [list CloseListWindow $w]
    if { $listorder != "status" } {
        bind $w <$::gid_right_button> [list ContextualListEntities $w %x %y %X %Y]
    }

    focus $w.buts.close
}

proc ContextualListEntities { lb x y X Y} {
    set menu $lb.menu
    destroy $menu
    menu $menu -tearoff 0

    $menu add command -label [_ "Clone"] -command {
        global NumListWindows
        ListEntities {} .gid.listent[expr $NumListWindows+1]
    }

    tk_popup $menu $X $Y
}
