
package require tooltip

set CalcPi 3.14159265358979323846

proc vect-disp_ui { base } {
    #pseudottk::
    set T [tk::listbox $base.listbox#1 \
            -xscrollcommand "$base.scrollbar#1 set" \
            -yscrollcommand "$base.scrollbar#2 set" \
            -setgrid 0 -selectmode single -exportselection 0]
    set vsb [ttk::scrollbar $base.scrollbar#2 -command "$base.listbox#1 yview" -orient v]
    set hsb [ttk::scrollbar $base.scrollbar#1 -command "$base.listbox#1 xview" -orient h]

    # Geometry management
    grid $base.listbox#1 $base.scrollbar#2 -sticky nesw
    grid configure $base.scrollbar#2 -sticky ns
    grid $base.scrollbar#1 -sticky ew
    
    grid remove $vsb $hsb
    bind $T <Configure> [list ConfigureListScrollbars $T $hsb $vsb]

    # Resize behavior management
    grid rowconfigure $base 0 -weight 1
    grid columnconfigure $base 0 -weight 1
}

proc vect-statelabel_ui { base } {
    global CalcAngleUnits
    global CalculatorPriv
    ttk::label $base.label#1 -textvariable CalcAngleUnits
    ttk::label $base.label#2 -textvariable CalculatorPriv(inverse)
    ttk::label $base.label#3 -textvariable CalculatorPriv(displayeng)
    
    ttk::button $base.button#6 -image [gid_themes::GetImage "help.png" small_icons] -width 3 -command CalcOpenHelpWindow
    # Geometry management    
    grid $base.label#1 $base.label#2 $base.label#3 -padx 2 -pady 2 -sticky e
    grid $base.button#6 -padx 2 -pady 2 -sticky w -row 0 -column 4
    grid columnconfigure $base "3" -weight 1
}

proc vect-func_ui { base } {
    ttk::button $base.button#4 -text sqrt -width 3
    ttk::button $base.button#5 -text pi -width 3
    ttk::button $base.button#13 -text <- -width 3
    ttk::button $base.button#22 -text +/- -width 3
    ttk::button $base.button#6 -text sin -width 3
    ttk::button $base.button#7 -text cos -width 3
    ttk::button $base.button#14 -text tan -width 3
    ttk::button $base.button#23 -text Exp -width 3
    ttk::button $base.button#8 -text MS -width 3
    ttk::button $base.button#9 -text MR -width 3
    ttk::button $base.button#15 -text x<>y -width 3
    ttk::button $base.button#eng -text Eng -width 3
    ttk::button $base.button#19 -text Cls -width 3
    ttk::button $base.button#20 -text Drg -width 3
    ttk::button $base.button#21 -text Inv -width 3
    ttk::button $base.button#24 -text Ent -width 3

    # Geometry management
    grid $base.button#4 $base.button#5 $base.button#13 $base.button#22
    grid $base.button#6 $base.button#7 $base.button#14 $base.button#23
    grid $base.button#8 $base.button#9 $base.button#15 $base.button#eng
    grid $base.button#19 $base.button#20 $base.button#21 $base.button#24
    #grid configure $base.button#24 -rowspan 2 -sticky nsew
}

proc vect-num_ui { base } {
    ttk::button $base.button#4 -text 7 -width 3
    ttk::button $base.button#5 -text 8 -width 3
    ttk::button $base.button#13 -text 9 -width 3
    ttk::button $base.button#22 -text / -width 3
    ttk::button $base.button#6 -text 4 -width 3
    ttk::button $base.button#7 -text 5 -width 3
    ttk::button $base.button#14 -text 6 -width 3
    ttk::button $base.button#23 -text * -width 3
    ttk::button $base.button#8 -text 1 -width 3
    ttk::button $base.button#9 -text 2 -width 3
    ttk::button $base.button#15 -text 3 -width 3
    ttk::button $base.button#24 -text - -width 3
    ttk::button $base.button#19 -text 0 -width 3
    ttk::button $base.button#20 -text . -width 3
    ttk::button $base.button#21 -text v/m -width 3
    ttk::button $base.button#25 -text + -width 3

    # Geometry management
    grid $base.button#4 $base.button#5 $base.button#13 $base.button#22
    grid $base.button#6 $base.button#7 $base.button#14 $base.button#23
    grid $base.button#8 $base.button#9 $base.button#15 $base.button#24
    grid $base.button#19 $base.button#20 $base.button#21 $base.button#25   
}

proc vect-vec_ui { base } {
    ttk::button $base.button#4 -text x^y -width 3
    ttk::button $base.button#5 -text |x| -width 3
    ttk::button $base.button#6 -text x1 -width 3
    ttk::button $base.button#7 -text x<y -width 3    
    
    ttk::menubutton $base.menubutton#2 -text [_ "More"]... -menu $base.menubutton#2.m

    menu $base.menubutton#2.m -tearoff 0
    $base.menubutton#2.m add command -label [_ "Get point"] \
            -accelerator "Control-a" \
            -command "CalcGetCalcVectorFromGid"
    $base.menubutton#2.m add command -label [_ "Get distance"] \
            -accelerator "Control-d" \
            -command "CalcGetDistanceFromGid"
    $base.menubutton#2.m add command -label [_ "Get vector"] \
            -accelerator "Control-b" \
            -command "CalcGetVectorFromGid"
    $base.menubutton#2.m add command -label [_ "Get angle"] \
            -accelerator "Control-e" \
            -command "CalcGetAngleFromGid"
    $base.menubutton#2.m add command -label [_ "Write"] -accelerator "Control-w" \
            -command CalcWriteVectorOrScalar
    
    $base.menubutton#2.m add separator
    $base.menubutton#2.m add cascade -label [_ "Local axes"] -menu $base.menubutton#2.m.la
    menu $base.menubutton#2.m.la -tearoff 0
    
    $base.menubutton#2.m.la add cascade -label [_ "trasf. local to global"] -menu \
        $base.menubutton#2.m.la.m1
    menu $base.menubutton#2.m.la.m1 -postcommand [list CalcFillLocalAxesMenu $base.menubutton#2.m.la.m1 LtoG]
    $base.menubutton#2.m.la add cascade -label [_ "trasf. global to local"] -menu \
        $base.menubutton#2.m.la.m2
    menu $base.menubutton#2.m.la.m2 -postcommand [list CalcFillLocalAxesMenu $base.menubutton#2.m.la.m2 GtoL]   

    set bt $base
    while { [winfo class $bt] != "Toplevel" } { set bt [winfo parent $bt] }

    bind $bt <${::acceleratorKey}-a> "$base.menubutton#2.m invoke 0 ; break"
    bind $bt <${::acceleratorKey}-b> "$base.menubutton#2.m invoke 2 ; break"
    bind $bt <${::acceleratorKey}-e> "$base.menubutton#2.m invoke 3 ; break"
    bind $bt <${::acceleratorKey}-d> "$base.menubutton#2.m invoke 1 ; break"
    bind $bt <${::acceleratorKey}-w> "$base.menubutton#2.m invoke 4 ; break"
    #trick the extra space is to avoid tooltip error if text start by -
    tooltip::tooltip $base.menubutton#2 " [CalcShowHelpGiveText More]"
    
    # Geometry management
    grid $base.button#4 $base.button#5
    grid $base.button#6 $base.button#7
    grid $base.menubutton#2 -columnspan 2
}

proc CalcGetCalcVectorFromGid {} {
    set xyz [::GidUtils::GetCoordinates]
    if { $xyz != "" } {
        CalcPushNum "v [lindex $xyz 0],[lindex $xyz 1],[lindex $xyz 2]"
    }
}

proc CalcGetDistanceFromGid {} {
    set dist [::GidUtils::GetDistance]
    if { $dist != "" } {
        CalcPushNum "$dist"
    }
}

proc CalcGetVectorFromGid {} {
    set p [::GidUtils::GetVector]
    if { $p != "" } {
        CalcPushNum "v [lindex $p 0],[lindex $p 1],[lindex $p 2]"
    }
}

proc CalcGetAngleFromGid {} {
    global CalcAngleUnits

    set angle [::GidUtils::GetAngle]
    if { $angle == "" } { return }

    switch $CalcAngleUnits {
        rad { set angle [expr {$angle*$MathUtils::DEGREE_TO_RAD}] }
        gra { set angle [expr {$angle*200.0/180.0}] }
    }
    CalcPushNum "$angle"
}

proc CalcWriteVectorOrScalar {} {
    global EntryVar EntryVarHilit

    CalcEndInsertion
    set num [CalcPopNum]
    CalcPushNum $num
    regexp {v (.*)} $num trash num
    set ::EntryVar $num
    set ::EntryVarHilit 1
}

proc CalcFlashlabel { num black w } {

    if { ![winfo exists $w] } { return }

    bind $w <1> "destroy $w"

    incr num
    if { $black == 0 } {
        $w configure -foreground white
        set black 1
    } else {
        $w configure -foreground black
        set black 0
    }
    if { $num < 10 } {
        after 300 CalcFlashlabel $num $black $w
    } else { destroy $w }
}

proc CalcShowError { buf } {
    global CalculatorPriv
    ttk::label $CalculatorPriv(listbox).errr -text $buf -relief ridge
    place  $CalculatorPriv(listbox).errr -x 0 -rely 1 -in \
            $CalculatorPriv(listbox) -anchor sw
    if { $::tcl_platform(os) != "Darwin"} {
        tkwait visibility $CalculatorPriv(listbox).errr
    } else {
        update
    }

    CalcFlashlabel 0 0 $CalculatorPriv(listbox).errr
}

proc CalcInsEnd { text } {
    global CalculatorPriv

    if { $CalculatorPriv(displayeng) == "" } {
        $CalculatorPriv(listbox) insert end $text
    } else {
        if { [regexp {([^0-9]*)([^,]+),([^,]+),(.*)} $text {} pref x y z] } {
            set text [format "%s%g,%g,%g" $pref $x $y $z]
        } elseif { [regexp {([^0-9]*)(.*)} $text {} pref num] } {
            set text [format "%s%g" $pref $num]
        }
        $CalculatorPriv(listbox) insert end $text
    }
}

proc CalcPushNum { num } {
    global CalculatorPriv

    set x [string range [$CalculatorPriv(listbox) get end] 3 end]
    $CalculatorPriv(listbox) delete end

    set y [string range [$CalculatorPriv(listbox) get end] 3 end]
    $CalculatorPriv(listbox) delete end

    set z [string range [$CalculatorPriv(listbox) get end] 3 end]
    $CalculatorPriv(listbox) delete end

    set t [string range [$CalculatorPriv(listbox) get end] 3 end]
    $CalculatorPriv(listbox) delete end

    CalcInsEnd $t
    CalcInsEnd "t: $z"
    CalcInsEnd "z: $y"
    CalcInsEnd "y: $x"
    CalcInsEnd "x: $num"
    $CalculatorPriv(listbox) see end
}

proc CalcPopNum {} {
    global CalculatorPriv

    set x [string range [$CalculatorPriv(listbox) get end] 3 end]
    $CalculatorPriv(listbox) delete end

    set y [string range [$CalculatorPriv(listbox) get end] 3 end]
    $CalculatorPriv(listbox) delete end

    set z [string range [$CalculatorPriv(listbox) get end] 3 end]
    $CalculatorPriv(listbox) delete end

    set t [string range [$CalculatorPriv(listbox) get end] 3 end]
    $CalculatorPriv(listbox) delete end

    set other 0.0
    if { [$CalculatorPriv(listbox) size] >0 } {
        set other [$CalculatorPriv(listbox) get end]
        $CalculatorPriv(listbox) delete end
    }

    CalcInsEnd "t: $other"
    CalcInsEnd "z: $t"
    CalcInsEnd "y: $z"
    CalcInsEnd "x: $y"
    $CalculatorPriv(listbox) see end
   
    return $x
}

proc CalcRedisplay {} {
    global CalculatorPriv

    set x [string range [$CalculatorPriv(listbox) get end] 3 end]
    $CalculatorPriv(listbox) delete end

    set y [string range [$CalculatorPriv(listbox) get end] 3 end]
    $CalculatorPriv(listbox) delete end

    set z [string range [$CalculatorPriv(listbox) get end] 3 end]
    $CalculatorPriv(listbox) delete end

    set t [string range [$CalculatorPriv(listbox) get end] 3 end]
    $CalculatorPriv(listbox) delete end

    CalcInsEnd "t: $t"
    CalcInsEnd "z: $z"
    CalcInsEnd "y: $y"
    CalcInsEnd "x: $x"
    $CalculatorPriv(listbox) see end
}

proc CalcConvertToValid { val } {
    if { ![regexp \
                {^[  ]*([+-]?)0*([0-9]*)(\.?)([0-9]*)([eE][+-]?([0-9]*))?[  ]*$}\
                $val trash sign bp p ap exp expval] } { return NaN }
    if { $bp == "" } { set bp 0}
    if { $p == "" } { set p .}
    if { [string length $ap] == 0 } { set ap 0}
    if { $exp != "" && $expval == "" } { append exp 1 }
    set num $sign$bp$p$ap$exp
    if { $num == "-0.0" } { set num 0.0 }
    return $num
}

proc CalcConvertToValidVec { val } {
    if { ![regexp {v[ ]*([^,]*),?([^,]*),?([^,]*)} \
                $val trash v(1) v(2) v(3)] } {
        return [CalcConvertToValid $val]
    }

    set v(1) [CalcConvertToValid $v(1)]
    set v(2) [CalcConvertToValid $v(2)]
    set v(3) [CalcConvertToValid $v(3)]

    return "v $v(1),$v(2),$v(3)"
}

proc CalcConvertToValidVecExtern { val } {
    if { ![regexp {[ ]*([^,]+),([^,]+),?([-+0-9.eE]*)} \
                $val trash v(1) v(2) v(3)] } {

        if { ![regexp {[ ]*([^ ]+)[ ]+([^ ]+)[ ]*([-+0-9.eE]*)} \
                    $val trash v(1) v(2) v(3)] } {
            return [CalcConvertToValid $val]
        }
    }

    set v(1) [CalcConvertToValid $v(1)]
    set v(2) [CalcConvertToValid $v(2)]
    set v(3) [CalcConvertToValid $v(3)]

    if { $v(1) == "NaN" || $v(2) == "NaN" || $v(3) == "NaN" } {
        return NaN
    }

    return "v $v(1),$v(2),$v(3)"
}

proc CalcChangeSignToNum { num { inexponent  0 } } {

    set isv [regexp {v[ ]*([^,]*),?([^,]*),?([^,]*)} \
            $num trash v(1) v(2) v(3)]

    set endi 1
    if { $isv } { set endi 3 }

    for {set i 1} {$i <= $endi} {incr i} {

        if { $isv } { set num $v($i) }

        if { !$inexponent || ![string match *E* $num] } {
            if { [string index $num 0] == "+" } {
                regsub {\+} $num {-} num
            } elseif { [string index $num 0] == "-" } {
                regsub -- {-} $num {} num
            } else { set num -$num }
        } elseif { [string match *E\+* $num] } {
            regsub {E\+} $num {E-} num
        } else { regsub {E\-} $num {E+} num }

        if { $isv } { set v($i) $num }
    }

    if { $isv } { set num "v $v(1),$v(2),$v(3)" }
    return $num
}

proc CalcInsert { what } {
    global CalculatorPriv

    if { $CalculatorPriv(insert) != ""} {
        $CalculatorPriv(listbox) delete end
    }

    set isv [regexp {v[ ]*([^,]*)(,?)([^,]*)(,?)([^,]*)} \
            $CalculatorPriv(insert) trash v(1) sp1 v(2) sp2 v(3)]

    if { !$isv } {
        set num $CalculatorPriv(insert)
        set pos 0
    } elseif { $sp2 != "" } {
        set num $v(3)
        set pos 3
    } elseif { $sp1 != "" } {
        set num $v(2)
        set pos 2
    } else {
        set num $v(1)
        set pos 1
    }

    switch -- $what {
        CHS {
            set num [CalcChangeSignToNum $num 1]
        }
        EEX {
            if { ![string match *E* $num] } {
                if { $num == "" } { set num 1.0 }
                set num ${num}E+
            } else {
                regsub {E.*} $num {} num
            }
        }
        . {
            if { ![string match *E* $num] && ![string match *.* $num] } {
                append num .
            }
        }
        v {
            if { $pos == 0 } {
                if { $num != "" } {
                    set v(1) [CalcConvertToValid $num]
                    set pos 2
                    set num ""
                } else { set pos 1 }
            } elseif { $pos < 3 } {
                set v($pos) [CalcConvertToValid $num]
                incr pos
                set num ""
            } else {
                $CalculatorPriv(listbox) insert end $CalculatorPriv(insert)
                CalcEndInsertion
                return
            }
        }
        <- {
            if { $num != "" } {
                set num [string range $num 0 [expr [string length $num]-2]]
            } elseif { $pos > 1 } {
                incr pos -1
                set num $v($pos)
            } elseif { $pos == 1 } { set pos 0 }
        }
        default {
            append num $what
        }
    }

    if { !$pos } {
        set CalculatorPriv(insert) $num
    } elseif { $pos == 3 } {
        set CalculatorPriv(insert) "v $v(1),$v(2),$num"
    } elseif { $pos == 2 } {
        set CalculatorPriv(insert) "v $v(1),$num"
    } else { set CalculatorPriv(insert) "v $num" }

    if { $CalculatorPriv(insert)!= "" } {
        $CalculatorPriv(listbox) insert end $CalculatorPriv(insert)
    }
    $CalculatorPriv(listbox) see end
}

proc CalcEndInsertion {} {
    global CalculatorPriv

    if { $CalculatorPriv(insert) != ""} {
        $CalculatorPriv(listbox) delete end
        CalcPushNum [CalcConvertToValidVec $CalculatorPriv(insert)]
        set CalculatorPriv(insert) ""
    }
}

proc CalcGeneralFunc { what } {
    global CalculatorPriv CalcAngleUnits CalcPi

    switch $what {
        CHS {
            if { $CalculatorPriv(insert) != ""} {
                CalcInsert CHS
            } else {
                set num [CalcChangeSignToNum [CalcPopNum]]
                CalcPushNum $num
            }
        }
        CLS {
            if { $CalculatorPriv(insert) != ""} {
                $CalculatorPriv(listbox) delete end
                set CalculatorPriv(insert) ""
            } else {
                CalcPopNum
            }
        }
        ENT {
            if { $CalculatorPriv(insert) != ""} {
                CalcEndInsertion
            } else {
                set num [CalcPopNum]
                CalcPushNum $num
                CalcPushNum $num
            }
        }
        inv {
            if { $CalculatorPriv(inverse) == "" } {
                set CalculatorPriv(inverse) inv
            } else {
                set CalculatorPriv(inverse) ""
            }
        }
        PI {
            if { $CalculatorPriv(insert) != ""} {
                CalcEndInsertion
            }
            CalcPushNum $CalcPi
        }
        STO {
            if { $CalculatorPriv(insert) != ""} {
                CalcEndInsertion
            }
            set CalculatorPriv(STO) [CalcPopNum]
            CalcPushNum $CalculatorPriv(STO)
        }
        RCL {
            if { $CalculatorPriv(STO) != ""} {
                if { $CalculatorPriv(insert) != ""} {
                    CalcEndInsertion
                }
                CalcPushNum $CalculatorPriv(STO)
            }
        }
        DRG {
            if { $CalculatorPriv(inverse) == "" } {
                if { $CalcAngleUnits == "rad" } {
                    set CalcAngleUnits deg
                } elseif { $CalcAngleUnits == "deg" } {
                    set CalcAngleUnits gra
                } else { set CalcAngleUnits rad }
            } else {
                if { $CalcAngleUnits == "rad" } {
                    set CalcAngleUnits gra
                } elseif { $CalcAngleUnits == "deg" } {
                    set CalcAngleUnits rad
                } else { set CalcAngleUnits deg }
                set CalculatorPriv(inverse) ""
            }
        }
        x<>y {
            CalcEndInsertion
            set num1 [CalcPopNum]
            set num2 [CalcPopNum]
            CalcPushNum $num1
            CalcPushNum $num2
        }
        ENG {
            if { $CalculatorPriv(displayeng) == "eng" } {
                set CalculatorPriv(displayeng) ""
            } else {
                set CalculatorPriv(displayeng) eng
            }
            CalcEndInsertion
            CalcRedisplay
        }
    }
}

proc CalcBinaryOp { op { printerror yes } } {
    global CalculatorPriv

    if { $CalculatorPriv(inverse) == "inv" } {
        CalcShowError "error: $op have no inverse"
        set CalculatorPriv(inverse) ""
        return
    }

    CalcEndInsertion

    set num2 [CalcPopNum]
    set num1 [CalcPopNum]

    set isv1 [regexp {v[ ]*([^,]*),?([^,]*),?([^,]*)} $num1 trash v1(1) \
            v1(2) v1(3)]
    set isv2 [regexp {v[ ]*([^,]*),?([^,]*),?([^,]*)} $num2 trash v2(1) \
            v2(2) v2(3)]

    set error ""

    if { $op == "x<y" } {
        if { !$isv1 || !$isv2 } {
            set error "operation angle only for vectors"
        } else {
            CalcPushNum $num1
            set error [CalcFuncOperation x1 no]
            if { $error != "" } {
                set error "vector y is a zero vector"
                CalcPopNum
            } else {
                CalcPushNum $num2
                set error [CalcFuncOperation x1 no]
                if { $error != "" } {
                    set error "vector x is a zero vector"
                    CalcPopNum
                } else {
                    CalcBinaryOp *
                    CalcGeneralFunc inv
                    CalcFuncOperation cos
                    set res [CalcPopNum]
                }
            }
        }
    } elseif { $op == "^" && ( !$isv1 || !$isv2 ) } {
        set error "cross product only for vectors"
    } elseif { !$isv2 && $op == "/" && $num2 == 0.0 } {
        set error "divide by 0"
    } elseif { $op == "/" && $isv2 } {
        set error "can't divide by a vector"
    } elseif { !$isv1 && !$isv2 } {
        set res [expr $num1 $op $num2]
    } elseif { $isv1 && $isv2 } {
        if { $op == "^" } {
            set resv(1) [expr $v1(2)*$v2(3)-$v2(2)*$v1(3)]
            set resv(2) [expr $v1(3)*$v2(1)-$v1(1)*$v2(3)]
            set resv(3) [expr $v1(1)*$v2(2)-$v1(2)*$v2(1)]
        } else {
            for {set i 1} {$i <= 3} {incr i} {
                set resv($i) [expr $v1($i) $op $v2($i)]
            }
        }
        if { $op == "*"} { set res [expr $resv(1)+$resv(2)+$resv(3)] }
    } elseif { $op == "+" || $op == "-" } {
        set error "can't add or substract scalar and vector"
    } else {
        for {set i 1} {$i <= 3} {incr i} {
            if { $isv1 } {
                set resv($i) [expr $v1($i) $op $num2]
            } else { set resv($i) [expr $num1 $op $v2($i)] }
        }
    }

    if { $error != "" } {
        if { $printerror == "yes" } {
            CalcShowError "error: $error"
        }
        CalcPushNum $num1
        CalcPushNum $num2
        return error
    } elseif { [info exists res] } {
        CalcPushNum $res
    } else {
        CalcPushNum "v $resv(1),$resv(2),$resv(3)"
    }
    return ""
}


proc CalcConvertAngleUnits { from to angle } {
    global CalcPi

    if { $from == "rad" && $to == "deg" } {
        return [expr $angle*180.0/$CalcPi]
    } elseif { $from == "rad" && $to == "gra" } {
        return [expr $angle*200.0/$CalcPi]
    } elseif { $from == "deg" && $to == "rad" } {
        return [expr $angle*$CalcPi/180.0]
    } elseif { $from == "deg" && $to == "gra" } {
        return [expr $angle*200.0/180.0]
    } elseif { $from == "gra" && $to == "rad" } {
        return [expr $angle*$CalcPi/200.0]
    } else {
        return [expr $angle*180.0/200.0]
    }
}

proc CalcFuncOperation { func { printerror yes } } {
    global CalculatorPriv CalcAngleUnits

    CalcEndInsertion
    set num [CalcPopNum]

    set isv [regexp {v[ ]*([^,]*),?([^,]*),?([^,]*)} $num trash v(1) \
            v(2) v(3)]

    set error ""

    if { $CalculatorPriv(inverse) == "inv" } {
        if { $func == "sin" } {
            set func asin
        } elseif { $func == "cos" } {
            set func acos
        } elseif { $func == "tan" } {
            set func atan
        } else {
            CalcShowError "error: $func have no inverse"
            CalcPushNum $num
            set CalculatorPriv(inverse) ""
            return error
        }
    }

    if { $isv && $func == "abs" } {
        CalcPushNum $num
        CalcPushNum $num
        CalcBinaryOp *
        CalcFuncOperation sqrt
    } elseif { $func == "x1" } {
        if { !$isv } {
            set error "x1 (x norma) can only  be performed on vector"
        } else {
            CalcPushNum $num
            CalcPushNum $num
            CalcFuncOperation abs
            set error [CalcBinaryOp / no]
            if { $error != "" } {
                set error "Can't normalize a zero vector"
                CalcPopNum
                CalcPopNum
            }
        }
    } elseif { $isv } {
        set error "$func can't be performed on vector"
    } elseif { $func == "sqrt" && $num < 0.0 } {
        set error "sqrt of negative number"
    } elseif { ( $func == "asin" || $func == "acos") && \
                ($num < -1.0 || $num > 1.0)} {
        set error "func $func not defined in $num"
    } else {
        if { $CalcAngleUnits != "rad" && ( $func == "sin" || $func == "cos" || \
                    $func == "tan") } {
            set num [CalcConvertAngleUnits $CalcAngleUnits rad $num]
        }
        set res [expr $func\($num)]
        if { $CalculatorPriv(inverse) == "inv" && $CalcAngleUnits != "rad" && \
                    ( $func == "asin" || $func == "acos" || \
                    $func == "atan" ) } {
            set res [CalcConvertAngleUnits rad $CalcAngleUnits $res]
        }
        CalcPushNum $res
    }
    set CalculatorPriv(inverse) ""
    if { $error != "" } {
        if { $printerror == "yes" } {
            CalcShowError "error: $error"
        }
        CalcPushNum $num
    }

}

proc CalcAddCommandToButChildren { w } {
    foreach i [winfo children $w] {
        if { [winfo class $i] == "TFrame" || [winfo class $i] == "Frame" || [winfo class $i] == "TLabelframe"} {
            CalcAddCommandToButChildren $i
        } elseif { [winfo class $i] == "TButton" || [winfo class $i] == "Button"} {
            set text [$i cget -text]
            if { $text != "" } {
                if { $text == "+/-" } {
                    set text CHS
                } elseif { $text == "Cls" } { 
                    set text CLS
                } elseif { $text == "Drg" } { 
                    set text DRG
                } elseif { $text == "Eng" } {                     
                    set text ENG
                } elseif { $text == "Ent" } {   
                    set text ENT
                } elseif { $text == "Exp" } { 
                    set text EEX
                } elseif { $text == "Inv" } { 
                    set text INV
                } elseif { $text == "MR" } { 
                    set text RCL
                } elseif { $text == "MS" } { 
                    set text STO
                } elseif { $text == "pi" } { 
                    set text PI
                } elseif { $text == "sqrt" } { 
                    set text SQRT
                }
                $i configure -command "CalcMainOp $text"
                #trick the extra space is to avoid tooltip error if text start by -
                tooltip::tooltip $i " [CalcShowHelpGiveText $text]"
            }
        }
    }
}

proc InitOperations {} {
    global CalculatorPriv tcl_platform

    set CalculatorPriv(name) ""
    set CalculatorPriv(op) ""

    set val {7 8 9 / 4 5 6 * 1 2 3 - 0 . v/m +}
    if { $tcl_platform(os) == "Darwin" } {
	# Mac OS X ...
        set acc {KeyPress-7 KeyPress-8 KeyPress-9 slash KeyPress-4 KeyPress-5 \
		     KeyPress-6 asterisk KeyPress-1 KeyPress-2 KeyPress-3 minus \
		     KeyPress-0 period {comma space} plus }
    } elseif { $::tcl_platform(platform) == "windows" } {
        set acc {KeyPress-7 KeyPress-8 KeyPress-9 slash KeyPress-4 KeyPress-5 \
                    KeyPress-6 asterisk KeyPress-1 KeyPress-2 KeyPress-3 minus \
                    KeyPress-0 period {comma space} plus }
    } else {
	# Linux ...
        set acc {{KeyPress-7 KP_Home} {KeyPress-8 KP_Up} {KeyPress-9 KP_Prior} \
                    {slash KP_Divide} {KeyPress-4 KP_Left} {KeyPress-5 KP_Begin} \
                    {KeyPress-6 KP_Right} {asterisk KP_Multiply} {KeyPress-1 KP_End} \
                    {KeyPress-2 KP_Down} {KeyPress-3 KP_Next} {minus KP_Subtract} \
                    {KeyPress-0 KP_Insert} {period KP_Delete} {comma space} \
                    {plus KP_Add}}
    }
    set func { "CalcInsert 7" "CalcInsert 8" "CalcInsert 9" "CalcBinaryOp /" \
                "CalcInsert 4" "CalcInsert 5" "CalcInsert 6" "CalcBinaryOp *" \
                "CalcInsert 1" "CalcInsert 2" "CalcInsert 3" "CalcBinaryOp -" \
                "CalcInsert 0" "CalcInsert ." "CalcInsert v" "CalcBinaryOp +"}

    append CalculatorPriv(name) " $val"
    append CalculatorPriv(acc) " $acc"
    append CalculatorPriv(op) " $func"

    set val {CHS EEX ENT}
    if { $::tcl_platform(platform) != "windows" } {
        set acc { m   e   {Return KP_Enter}}
    } else {
        set acc { m   e Return }
    }
    set func { "CalcGeneralFunc CHS" "CalcInsert EEX" "CalcGeneralFunc ENT"}

    append CalculatorPriv(name) " $val"
    append CalculatorPriv(acc) " $acc"
    append CalculatorPriv(op) " $func"

    set val {SQRT "PI" "<-" sin cos tan STO RCL x<>y CLS DRG INV ENG}
    set acc { r p "BackSpace Delete" s c t "" "" "" l d i g}
    set func { "CalcFuncOperation sqrt" "CalcGeneralFunc PI" "CalcInsert <-" \
                "CalcFuncOperation sin" \
                "CalcFuncOperation cos" "CalcFuncOperation tan" \
                "CalcGeneralFunc STO" "CalcGeneralFunc RCL" \
                "CalcGeneralFunc x<>y"  "CalcGeneralFunc CLS" \
                "CalcGeneralFunc DRG" "CalcGeneralFunc inv" "CalcGeneralFunc ENG"}

    append CalculatorPriv(name) " $val"
    append CalculatorPriv(acc) " $acc"
    append CalculatorPriv(op) " $func"

    set val [list [_ "Help"] [_ "Close"]]
    set acc { h Control-q}
    set func { CalcOpenHelpWindow "destroy $CalculatorPriv(wCalcul)" }     

    append CalculatorPriv(name) " $val"
    append CalculatorPriv(acc) " $acc"
    append CalculatorPriv(op) " $func"


    set val {x^y "|x|" "x1" "x<y" "" ""}
    set acc { "" "" "" "" "" ""}
    set func { "CalcBinaryOp ^" "CalcFuncOperation abs" "CalcFuncOperation x1" \
                "CalcBinaryOp x<y" "" "" ""}

    append CalculatorPriv(name) " $val"
    append CalculatorPriv(acc) " $acc"
    append CalculatorPriv(op) " $func"

    set inum 0
    foreach i $CalculatorPriv(op) {
        if { $i != "" } {
            set acc [lindex $CalculatorPriv(acc) $inum]
            foreach j $acc {
                if { $j != "" } {
                    bind $CalculatorPriv(wCalcul) <$j> $i
                }
            }
        }
        incr inum
    }
}

proc CalcMainOp { what } {
    global CalculatorPriv

    set pos [lsearch -exact $CalculatorPriv(name) $what]
    if { $pos != -1 } { eval [lindex $CalculatorPriv(op) $pos] }
}

proc CalculatorV { { w .gid.wcalc } } {
    global tcl_precision CalcAngleUnits CalculatorPriv GIDDEFAULT tcl_platform
   
    #set tcl_precision 17 ;#left the default value=0 from Tcl 8.5 with a special meaning

    set CalculatorPriv(displayeng) ""

    InitWindow2 $w -title [_ "Vector Calculator"] \
        -geometryvariable PreCalculatorWindowGeom \
        -initcommand CalculatorV
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0

    set CalculatorPriv(wCalcul) $w
    set CalculatorPriv(MainFrames) "$w.up $w.buttons"
   
    ttk::frame $w.buttons
    ttk::labelframe $w.buttons.vectorial -text [_ "Vector"]
    set funcs "vect-disp_ui vect-statelabel_ui vect-vec_ui vect-func_ui vect-num_ui"
    set frames "$w.disp $w.statelabel $w.buttons.vectorial.vec $w.buttons.func $w.buttons.num"
    foreach func $funcs frame $frames {
        eval $func [ttk::frame $frame]
    }
    ttk::frame $w.buts -style BottomFrame.TFrame
    ttk::button $w.buts.bclose -text [_ "Close"] -command "destroy $w" -style BottomFrame.TButton
    #
    grid $w.disp -sticky nsew -padx 2 -pady 2
    grid $w.statelabel -sticky new -padx 2 -pady 2
    grid $w.buttons.vectorial.vec -padx 2 -pady 2
    grid $w.buttons.vectorial $w.buttons.func $w.buttons.num -sticky ew -padx 2 -pady 2
    grid $w.buttons -sticky new -padx 2 -pady 2
    grid $w.buts.bclose
    grid $w.buts -sticky sew -pady {2 0}
    grid anchor $w.buts center
    
    grid columnconfigure $w.buttons {0 1 2} -weight 1
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1
    
    set CalculatorPriv(insert) ""
    set CalculatorPriv(listbox) $w.disp.listbox#1
    if { ![info exists CalcAngleUnits] || ($CalcAngleUnits != "rad" && $CalcAngleUnits != "deg" && $CalcAngleUnits != "gra") } {
        set CalcAngleUnits "rad"
    }
    set CalculatorPriv(inverse) ""

    $CalculatorPriv(listbox) insert end "t: 0.0"
    $CalculatorPriv(listbox) insert end "z: 0.0"
    $CalculatorPriv(listbox) insert end "y: 0.0"
    $CalculatorPriv(listbox) insert end "x: 0.0"

    selection handle $CalculatorPriv(listbox) CalcReturnSelectedL
    bind $CalculatorPriv(listbox) <ButtonRelease-1> \
            "selection own -command CalcLooseSelectionL $CalculatorPriv(listbox)"

    bind $CalculatorPriv(listbox) <$::gid_central_button> {
        catch {
            CalcPasteNum [string trim [selection get] " "]
        }
        break
    }
    bind $CalculatorPriv(listbox) <B2-Motion> "break"

    bind $CalculatorPriv(wCalcul) <${::acceleratorKey}-c> "CalcReturnSelectedL 0 0 ; break"
    #bind all <${::acceleratorKey}-c> "CalcReturnSelectedL 0 0 ; break"
    bind $CalculatorPriv(wCalcul) <${::acceleratorKey}-v> "CalcPasteFromClipboard ; break"
    #bind all <${::acceleratorKey}-v> "CalcPasteFromClipboard ; break"

    InitOperations
    CalcAddCommandToButChildren $w   
    wm minsize $w 350 260
}

proc CalcPasteFromClipboard {} {
    catch {
        CalcPasteNum [string trim [selection get -selection CLIPBOARD] " "]
    }
}

proc CalcReturnSelectedL { offset maxBytes } {
    global CalculatorPriv tcl_platform

    set cur [$CalculatorPriv(listbox) curselection]

    set num ""
    if { $cur != "" } {
        set num [$CalculatorPriv(listbox) get $cur]
        regexp {(x|y|z|t): (.*)} $num trash trash2 num
        regexp {v (.*)} $num trash num
    }

    if { $maxBytes == 0 } {
        clipboard clear -displayof .
        clipboard append -displayof . $num
    }
    return $num
}

proc CalcLooseSelectionL {} {
    global CalculatorPriv
    $CalculatorPriv(listbox) selection clear 0 end
}

proc CalcPasteNum { val } {
    set val [CalcConvertToValidVecExtern $val]
    if { $val == "NaN" } {
        CalcShowError "error: Not valid pasted scalar or vector"
    } else {
        CalcPushNum $val
    }
}

proc CalcOpenHelpWindow {} {
    global CalculatorPriv
    set w $CalculatorPriv(wCalcul).mhelp
    if { [winfo exists $w] } {
        destroy $w
    }

    toplevel $w
    if { $::tcl_platform(platform) == "windows" } {
        wm attributes $w -toolwindow 1
    }
    wm title $w [_ "CalculatorV Help"]
    wm iconname $w "CalcV H"
    wm geometry $w +[winfo x $CalculatorPriv(wCalcul)]+[winfo y $CalculatorPriv(wCalcul)]

    ttk::frame $w.f
    set T [text $w.f.t -wrap none -yscroll "$w.f.yscroll set" -xscroll "$w.f.xscroll set"]
    set vsb [ttk::scrollbar $w.f.yscroll -command "$w.f.t yview" -orient v]
    set hsb [ttk::scrollbar $w.f.xscroll -command "$w.f.t xview" -orient h]        
    
    ttk::frame $w.buts -style BottomFrame.TFrame
    ttk::button $w.buts.b -text [_ "Close"] -command "destroy $w" -style BottomFrame.TButton
           
    grid $T $vsb -sticky nsew
    grid configure $vsb -sticky ns
    grid $hsb -sticky ew
    grid remove $vsb $hsb
    bind $T <Configure> [list ConfigureListScrollbars $T $hsb $vsb]  
    
    grid $w.f -sticky nsew
    grid $w.buts.b
    grid $w.buts -sticky sew -pady {2 0}
    grid anchor $w.buts center
    grid columnconfigure $w.f 0 -weight 1
    grid rowconfigure $w.f 0 -weight 1

    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1
    
    $w.f.t insert end $CalculatorPriv(HelpCalcVariable)
    $w.f.t configure -state disabled
    
    bind $w <Alt-c> [list destroy $w]
    bind $w <Escape> [list destroy $w]
}

proc CalcShowHelpGiveText { what } {
    global CalculatorPriv

    if { $what == "" } { return }

    set whatsub $what
    foreach i "* + | ^ ." {
        regsub -all "\\$i" $whatsub "\\$i" whatsub
    }
    regexp -- "\n\[*\]($whatsub\[^*\]*)" \
            $CalculatorPriv(HelpCalcVariable) \
            trash text

    if { [info exists text] } {
        set text [string trimright $text \n]
    } else {
        set num [lsearch -exact $CalculatorPriv(name) $what]
        set acc [lindex $CalculatorPriv(acc) $num]
        set text "$what\nshortcuts: $acc"
    }
    return $text
}

proc  CalcFillLocalAxesMenu { menu dir } {
    $menu delete 0 end
    foreach la [GiD_Info localaxes] {
        $menu add command -label $la -command [list CalcLocalAxesTransf $la $dir]
    }
}

proc CalcLocalAxesTransf { localaxes dir } {
    
    CalcEndInsertion

    set num1 [CalcPopNum]

    set isv1 [regexp {v[ ]*([^,]*),?([^,]*),?([^,]*)} $num1 trash v1(1) \
            v1(2) v1(3)]

    set error ""
    
    if { !$isv1 } {
        set error [_ "Local axes transformation can only be applied to a vector"]
    } else {
        set mat [lindex [GiD_Info localaxes $localaxes -localaxesmat] 0]
        for { set row 0 } { $row < 3 } { incr row } {
            set rm1 [expr {$row+1}]
            set resv($rm1) 0.0
            for { set col 0 } { $col < 3 } { incr col } {
                set cm1 [expr {$col+1}]
                switch $dir {
                    LtoG { set mati [lindex $mat [expr {3*$row+$col}]] }
                    GtoL { set mati [lindex $mat [expr {3*$col+$row}]] }
                }
                set resv($rm1) [expr {$resv($rm1)+$mati*$v1($cm1)}]
            }
            set resv($rm1) [format %.5g $resv($rm1) ]
        }
    }
    if { $error != "" } {
        CalcShowError "error: $error"
        CalcPushNum $num1
        return error
    } else {
        CalcPushNum "v $resv(1),$resv(2),$resv(3)"
    }
    return ""

}

# variable for HELP
set CalculatorPriv(HelpCalcVariable)  "\
        ESCALAR AND VECTOR CALCULATOR

The operations are based on Reversed-polish
notation.

Valid format for vectors:

2.3,67,9e-5
45,-67.3
12.3 45.8 12

Pressing middle mouse button over the stack,
a scalar or a vector can be inserted in
register x:.

left mouse button over one register: selects
it to paste in another place.

right or middle mouse button over a key:
short help.


LIST OF FUNCTION KEYS

*<- backspace
Deletes last digit when entering one number
shortcuts: Backspace Delete

*CHS Change sign
Changes sign of number in register x:
If entering exponent of a number, changes
its exponent sign
shortcuts: m

*EEX Enter exponent
When entering a number,enter its exponent
if no mantisa entered, it's supposed to be 1.0
shortcuts: e

*ENT Enter
Enter number into register x or
copies register x into register y
shortcuts: Return Enter

*v/m Enter vector component
To enter in vector mode or
change to next component
shortcuts: comma space

*SQRT Square root
Argument must be scalar and > 0.0
shortcuts: r

*PI number PI
Enter number PI into register x:
shortcuts: p

*sin sinus
Argument must be scalar
Change units with DRG
if INV is set: arcsin
argument scalar -1.0<=a<=1.0
shortcuts: s

*cos cosinus
Argument must be scalar
Change units with DRG
if INV is set: arccos
argument scalar -1.0<=a<=1.0
shortcuts: c

*tan tangent
Argument must be scalar
Change units with DRG
if INV is set: arctan
argument scalar
shortcuts: t

*CLS Drop register x:
shortcuts: l

*DRG Change degree units
Pressing it goes to:
rad   radians
deg   degrees
gra   Centesimal degrees
shortcuts: d

*INV inverse of functions
Set or unset inverse. Functions that have
one inverse will use it.
example: sin converts to arcsin
see every function for details.
shortcuts: i

*STO Save register x: in memory position
Use RCL to recover.
No shortcuts

*RCL Recovers number saved in memory by
STO to register x:
No shortcuts

*x<>y Swap register x: and y:
No shortcuts

*x^y cross product of vectors
No shortcuts

*Close - Close calculator
shortcuts: Control q

*HELP opens help window
shortcuts: h

*|x| norma of vector
returns the norma of the vector
absolute value of one scalar
no shortcuts

*x1 normalize vector
Converts vector to normalized
no shortcuts

*x<y angle of vectors
Give the angle between 2 vectors
from 0 to 180 degrees in current
angle unit
no shortcuts

*ENG Change display of numbers
If it is activated, small and big
numbers are displayed with
scientific style
shortcuts: g

*x^y cross product
Cross product of 2 vectors
no shortcuts

*More contains additional functions for transfering
    data with main program and for other functions
-Get point: one point from the program
is selected and pushed in stack.
shortcuts: Control-a
-Get distance: Distance between 2 points
is selected and pushed in stack.
shortcuts: Control-d
-Get vector: a vector defined by 2 points selected
in the program is created and pushed in stack.
shortcuts: Control-b
-Get angle: an angle defined by 3 points selected
in the program is created and pushed in stack.
shortcuts: Control-e
-Write: Value in x register is transferred
to input line in main program.
shortcuts: Control-w
- Local axes: Given a vector in the stack and selecting
one of the existing local axes from the main program,
these rutines transform either the vector from local axes
into global axes or the opposite. The transformed vector
substitutes to the original vector in the stack
"
