
package require BWidget
package require msgcat
package require dialogwin

package provide texteditor 1.0

if { [info procs _] == "" } {
	proc ::_ { args } {
		if { [regexp {(.*)#C#(.*)} [lindex $args 0] {} str comm] } {
			set args [lreplace $args 0 0 $str]
		}
		return [uplevel 1 ::msgcat::mc $args]
	}
}


# lognoter, for instance, uses it's own interpreter...
if { [ tk windowingsystem] eq "aqua"} {
	set ::acceleratorKey Command
	set ::acceleratorText Command
} else {
	set ::acceleratorKey Control
	set ::acceleratorText Ctrl
}

namespace eval TextEditor {} {
	#preference to only save the parameter file, and hide New and Open buttons
	variable HIDE_NEW_AND_OPEN 0

	#
	variable version "1.0"
	variable status ""
	variable progress 0
	variable showtoolbar 1
	variable changed [_ "Unmodified"]
	variable rowandcol ""
	variable currentfile ""
	variable encoding ""
	variable cansavetovariable 0
	variable vartosave ""

	variable findparam
	set findparam(searchstring) ""
	set findparam(searchstrings) ""
	set findparam(searchpos) ""
	set findparam(searchdirection) "forward"
	set findparam(considercase) 0
	set findparam(frombeginning) 0
	set findparam(wholeword) 0

	variable img
	#widgets
	variable mainframe
	variable maintext
}


#proc encodegif { filename } {
#    set fp [open $filename r]
#    fconfigure $fp -translation binary
#    set txt " set img([ file rootname [file tail $filename]]) \[image create photo -data {\n"
#    append txt [binary encode base64 [read $fp]]
#    append txt "\n}\n\]\n"
#    close $fp
#    return $txt
#}

#proc encodegifs { dirname } {
#    set txt ""
#    foreach filename [glob -nocomplain -directory $dirname -tails *.gif] {
#        append txt "[encodegif $filename]\n"
#    }
#    return $txt
#}

proc TextEditor::CreateImages { } {
	variable img
	if { [ info exists ::GIDDEFAULT]} {

		set img(find) [ gid_themes::GetImage search.png medium_icons]
		set img(new) [ gid_themes::GetImage new.png medium_icons]
		set img(open) [ gid_themes::GetImage open.png medium_icons]
		set img(save) [ gid_themes::GetImage save.png medium_icons]
		set img(paste) [ gid_themes::GetImage paste_text.png medium_icons]
		set img(redo) [ gid_themes::GetImage redo.png medium_icons]
		set img(copy) [ gid_themes::GetImage copy_text.png medium_icons]
		set img(cut) [ gid_themes::GetImage cut_text.png medium_icons]
		set img(undo) [ gid_themes::GetImage undo.png medium_icons]
		set img(date) [ gid_themes::GetImage date.png medium_icons]

	} else {


		#size 24x24
		set img(copy) [image create photo -data {
			R0lGODlhGAAYAPcAAAAAAIAAAACAAICAAAAAgIAAgACAgP///8DcwKbK8EAg
			AGAgAIAgAKAgAMAgAOAgAABAACBAAEBAAGBAAIBAAKBAAMBAAOBAAABgACBg
			AEBgAGBgAIBgAKBgAMBgAOBgAACAACCAAECAAGCAAICAAKCAAMCAAOCAAACg
			ACCgAECgAGCgAICgAKCgAMCgAOCgAADAACDAAEDAAGDAAIDAAKDAAMDAAODA
			AADgACDgAEDgAGDgAIDgAKDgAMDgAODgAAAAQCAAQEAAQGAAQIAAQKAAQMAA
			QOAAQAAgQCAgQEAgQGAgQIAgQKAgQMAgQOAgQABAQCBAQEBAQGBAQIBAQKBA
			QMBAQOBAQABgQCBgQEBgQGBgQIBgQKBgQMBgQOBgQACAQCCAQECAQGCAQICA
			QKCAQMCAQOCAQACgQCCgQECgQGCgQICgQKCgQMCgQOCgQADAQCDAQEDAQGDA
			QIDAQKDAQMDAQODAQADgQCDgQEDgQGDgQIDgQKDgQMDgQODgQAAAgCAAgEAA
			gGAAgIAAgKAAgMAAgOAAgAAggCAggEAggGAggIAggKAggMAggOAggABAgCBA
			gEBAgGBAgIBAgKBAgMBAgOBAgABggCBggEBggGBggIBggKBggMBggOBggACA
			gCCAgECAgGCAgICAgKCAgMCAgOCAgACggCCggECggGCggICggKCggMCggOCg
			gADAgCDAgEDAgGDAgIDAgKDAgMDAgODAgADggCDggEDggGDggIDggKDggMDg
			gODggAAAwCAAwEAAwGAAwIAAwKAAwMAAwOAAwAAgwCAgwEAgwGAgwIAgwKAg
			wMAgwOAgwABAwCBAwEBAwGBAwIBAwKBAwMBAwOBAwABgwCBgwEBgwGBgwIBg
			wKBgwMBgwOBgwACAwCCAwECAwGCAwICAwKCAwMCAwOCAwACgwCCgwECgwGCg
			wICgwKCgwMCgwOCgwADAwCDAwEDAwGDAwIDAwKDAwP/78KCgpICAgP8AAAD/
			AP//AAAA//8A/wD//////yH5BAEAAAcALAAAAAAYABgAQAhhAA8IHEiwoMGD
			AP4p/JcQwMGHCBkCmNgQosWLAxMuZChR4kSMIEOKzEjx48gDGhemlHhxJceS
			FUPCLHmyps2bOC2uNCmyoU+YGH9O3BjzoVCXCh0anblRZc+mSXNKnfowIAA7
		}
		]

		set img(cut) [image create photo -data {
			R0lGODlhGAAYAPcAAAAAAIAAAACAAICAAAAAgIAAgACAgMDAwMDcwKbK8EAg
			AGAgAIAgAKAgAMAgAOAgAABAACBAAEBAAGBAAIBAAKBAAMBAAOBAAABgACBg
			AEBgAGBgAIBgAKBgAMBgAOBgAACAACCAAECAAGCAAICAAKCAAMCAAOCAAACg
			ACCgAECgAGCgAICgAKCgAMCgAOCgAADAACDAAEDAAGDAAIDAAKDAAMDAAODA
			AADgACDgAEDgAGDgAIDgAKDgAMDgAODgAAAAQCAAQEAAQGAAQIAAQKAAQMAA
			QOAAQAAgQCAgQEAgQGAgQIAgQKAgQMAgQOAgQABAQCBAQEBAQGBAQIBAQKBA
			QMBAQOBAQABgQCBgQEBgQGBgQIBgQKBgQMBgQOBgQACAQCCAQECAQGCAQICA
			QKCAQMCAQOCAQACgQCCgQECgQGCgQICgQKCgQMCgQOCgQADAQCDAQEDAQGDA
			QIDAQKDAQMDAQODAQADgQCDgQEDgQGDgQIDgQKDgQMDgQODgQAAAgCAAgEAA
			gGAAgIAAgKAAgMAAgOAAgAAggCAggEAggGAggIAggKAggMAggOAggABAgCBA
			gEBAgGBAgIBAgKBAgMBAgOBAgABggCBggEBggGBggIBggKBggMBggOBggACA
			gCCAgECAgGCAgICAgKCAgMCAgOCAgACggCCggECggGCggICggKCggMCggOCg
			gADAgCDAgEDAgGDAgIDAgKDAgMDAgODAgADggCDggEDggGDggIDggKDggMDg
			gODggAAAwCAAwEAAwGAAwIAAwKAAwMAAwOAAwAAgwCAgwEAgwGAgwIAgwKAg
			wMAgwOAgwABAwCBAwEBAwGBAwIBAwKBAwMBAwOBAwABgwCBgwEBgwGBgwIBg
			wKBgwMBgwOBgwACAwCCAwECAwGCAwICAwKCAwMCAwOCAwACgwCCgwECgwGCg
			wICgwKCgwMCgwOCgwADAwCDAwEDAwGDAwIDAwKDAwP/78KCgpICAgP8AAAD/
			AP//AAAA//8A/wD//////yH5BAEAAP8ALAAAAAAYABgAQAhVAP8JHEiwoMGD
			BQEA+KcQoUMCAiFC/DfRocWLFgkAqIixo8ePFxcKFAlyZMOSFAlIjAhSJcqX
			MGOWJEkSJE2YC2ui1FlSosuPE1dSBMoypcyjSDEGBAA7
		}
		]

		set img(find) [image create photo -data {
			R0lGODlhGAAYAPcAAAAAAIAAAACAAICAAAAAgIAAgACAgMDAwMDcwKbK8EAg
			AGAgAIAgAKAgAMAgAOAgAABAACBAAEBAAGBAAIBAAKBAAMBAAOBAAABgACBg
			AEBgAGBgAIBgAKBgAMBgAOBgAACAACCAAECAAGCAAICAAKCAAMCAAOCAAACg
			ACCgAECgAGCgAICgAKCgAMCgAOCgAADAACDAAEDAAGDAAIDAAKDAAMDAAODA
			AADgACDgAEDgAGDgAIDgAKDgAMDgAODgAAAAQCAAQEAAQGAAQIAAQKAAQMAA
			QOAAQAAgQCAgQEAgQGAgQIAgQKAgQMAgQOAgQABAQCBAQEBAQGBAQIBAQKBA
			QMBAQOBAQABgQCBgQEBgQGBgQIBgQKBgQMBgQOBgQACAQCCAQECAQGCAQICA
			QKCAQMCAQOCAQACgQCCgQECgQGCgQICgQKCgQMCgQOCgQADAQCDAQEDAQGDA
			QIDAQKDAQMDAQODAQADgQCDgQEDgQGDgQIDgQKDgQMDgQODgQAAAgCAAgEAA
			gGAAgIAAgKAAgMAAgOAAgAAggCAggEAggGAggIAggKAggMAggOAggABAgCBA
			gEBAgGBAgIBAgKBAgMBAgOBAgABggCBggEBggGBggIBggKBggMBggOBggACA
			gCCAgECAgGCAgICAgKCAgMCAgOCAgACggCCggECggGCggICggKCggMCggOCg
			gADAgCDAgEDAgGDAgIDAgKDAgMDAgODAgADggCDggEDggGDggIDggKDggMDg
			gODggAAAwCAAwEAAwGAAwIAAwKAAwMAAwOAAwAAgwCAgwEAgwGAgwIAgwKAg
			wMAgwOAgwABAwCBAwEBAwGBAwIBAwKBAwMBAwOBAwABgwCBgwEBgwGBgwIBg
			wKBgwMBgwOBgwACAwCCAwECAwGCAwICAwKCAwMCAwOCAwACgwCCgwECgwGCg
			wICgwKCgwMCgwOCgwADAwCDAwEDAwGDAwIDAwKDAwP/78KCgpICAgP8AAAD/
			AP//AAAA//8A/wD//////yH5BAEAAP8ALAAAAAAYABgAQAhgAP8JHEiwoMGD
			AgEoVPhvIUOECRs+HAhAYkWIGDMmrLhQosWJGkOKHEmxYcmLGR2qdIhxJceV
			JGPKnElTZseNKDXetCjyJUeeLX16VNnyI0OiCIUSVHoQKUWWNaNKxRgQADs=
		}
		]

		set img(new) [image create photo -data {
			R0lGODlhGAAYAPcAAAAAAIAAAACAAICAAAAAgIAAgACAgP///8DcwKbK8EAg
			AGAgAIAgAKAgAMAgAOAgAABAACBAAEBAAGBAAIBAAKBAAMBAAOBAAABgACBg
			AEBgAGBgAIBgAKBgAMBgAOBgAACAACCAAECAAGCAAICAAKCAAMCAAOCAAACg
			ACCgAECgAGCgAICgAKCgAMCgAOCgAADAACDAAEDAAGDAAIDAAKDAAMDAAODA
			AADgACDgAEDgAGDgAIDgAKDgAMDgAODgAAAAQCAAQEAAQGAAQIAAQKAAQMAA
			QOAAQAAgQCAgQEAgQGAgQIAgQKAgQMAgQOAgQABAQCBAQEBAQGBAQIBAQKBA
			QMBAQOBAQABgQCBgQEBgQGBgQIBgQKBgQMBgQOBgQACAQCCAQECAQGCAQICA
			QKCAQMCAQOCAQACgQCCgQECgQGCgQICgQKCgQMCgQOCgQADAQCDAQEDAQGDA
			QIDAQKDAQMDAQODAQADgQCDgQEDgQGDgQIDgQKDgQMDgQODgQAAAgCAAgEAA
			gGAAgIAAgKAAgMAAgOAAgAAggCAggEAggGAggIAggKAggMAggOAggABAgCBA
			gEBAgGBAgIBAgKBAgMBAgOBAgABggCBggEBggGBggIBggKBggMBggOBggACA
			gCCAgECAgGCAgICAgKCAgMCAgOCAgACggCCggECggGCggICggKCggMCggOCg
			gADAgCDAgEDAgGDAgIDAgKDAgMDAgODAgADggCDggEDggGDggIDggKDggMDg
			gODggAAAwCAAwEAAwGAAwIAAwKAAwMAAwOAAwAAgwCAgwEAgwGAgwIAgwKAg
			wMAgwOAgwABAwCBAwEBAwGBAwIBAwKBAwMBAwOBAwABgwCBgwEBgwGBgwIBg
			wKBgwMBgwOBgwACAwCCAwECAwGCAwICAwKCAwMCAwOCAwACgwCCgwECgwGCg
			wICgwKCgwMCgwOCgwADAwCDAwEDAwGDAwIDAwKDAwP/78KCgpICAgP8AAAD/
			AP//AAAA//8A/wD//////yH5BAEAAAcALAAAAAAYABgAQAhYAA8IHEiwoMGD
			ApP8W8gwicMkCAsqZEjxH8SIGDManFhx4UWNIEOKlNjRoUiOJUGirPhxpMuX
			MGOOfEizpcaVFi2GxNlQZceeN3969CnUJsKaSGUqXQoyIAA7
		}
		]

		set img(open) [image create photo -data {
			R0lGODlhGAAYAPcAAAAAAIAAAACAAICAAAAAgIAAgACAgMDAwMDcwKbK8EAg
			AGAgAIAgAKAgAMAgAOAgAABAACBAAEBAAGBAAIBAAKBAAMBAAOBAAABgACBg
			AEBgAGBgAIBgAKBgAMBgAOBgAACAACCAAECAAGCAAICAAKCAAMCAAOCAAACg
			ACCgAECgAGCgAICgAKCgAMCgAOCgAADAACDAAEDAAGDAAIDAAKDAAMDAAODA
			AADgACDgAEDgAGDgAIDgAKDgAMDgAODgAAAAQCAAQEAAQGAAQIAAQKAAQMAA
			QOAAQAAgQCAgQEAgQGAgQIAgQKAgQMAgQOAgQABAQCBAQEBAQGBAQIBAQKBA
			QMBAQOBAQABgQCBgQEBgQGBgQIBgQKBgQMBgQOBgQACAQCCAQECAQGCAQICA
			QKCAQMCAQOCAQACgQCCgQECgQGCgQICgQKCgQMCgQOCgQADAQCDAQEDAQGDA
			QIDAQKDAQMDAQODAQADgQCDgQEDgQGDgQIDgQKDgQMDgQODgQAAAgCAAgEAA
			gGAAgIAAgKAAgMAAgOAAgAAggCAggEAggGAggIAggKAggMAggOAggABAgCBA
			gEBAgGBAgIBAgKBAgMBAgOBAgABggCBggEBggGBggIBggKBggMBggOBggACA
			gCCAgECAgGCAgICAgKCAgMCAgOCAgACggCCggECggGCggICggKCggMCggOCg
			gADAgCDAgEDAgGDAgIDAgKDAgMDAgODAgADggCDggEDggGDggIDggKDggMDg
			gODggAAAwCAAwEAAwGAAwIAAwKAAwMAAwOAAwAAgwCAgwEAgwGAgwIAgwKAg
			wMAgwOAgwABAwCBAwEBAwGBAwIBAwKBAwMBAwOBAwABgwCBgwEBgwGBgwIBg
			wKBgwMBgwOBgwACAwCCAwECAwGCAwICAwKCAwMCAwOCAwACgwCCgwECgwGCg
			wICgwKCgwMCgwOCgwADAwCDAwEDAwGDAwIDAwKDAwP/78KCgpICAgP8AAAD/
			AP//AAAA//8A/wD//////yH5BAEAAP8ALAAAAAAYABgAQAhsAP8JHEiwoMGD
			/6QoRKhQCsKEEgZInCixjcOHGDMWdLjv3z4JIEE2bKixpMmTGC8mXMnQo8uO
			HVWm9BiR4kSLKHPq3MmzJ0GSPgUuZAhzJNCMUmC+9CjzYVKaNm82/fkxasWp
			G40aDcq168OAADs=
		}
		]

		set img(paste) [image create photo -data {
			R0lGODlhGAAYAPcAAAAAAIAAAACAAICAAAAAgIAAgACAgP///8DcwKbK8EAg
			AGAgAIAgAKAgAMAgAOAgAABAACBAAEBAAGBAAIBAAKBAAMBAAOBAAABgACBg
			AEBgAGBgAIBgAKBgAMBgAOBgAACAACCAAECAAGCAAICAAKCAAMCAAOCAAACg
			ACCgAECgAGCgAICgAKCgAMCgAOCgAADAACDAAEDAAGDAAIDAAKDAAMDAAODA
			AADgACDgAEDgAGDgAIDgAKDgAMDgAODgAAAAQCAAQEAAQGAAQIAAQKAAQMAA
			QOAAQAAgQCAgQEAgQGAgQIAgQKAgQMAgQOAgQABAQCBAQEBAQGBAQIBAQKBA
			QMBAQOBAQABgQCBgQEBgQGBgQIBgQKBgQMBgQOBgQACAQCCAQECAQGCAQICA
			QKCAQMCAQOCAQACgQCCgQECgQGCgQICgQKCgQMCgQOCgQADAQCDAQEDAQGDA
			QIDAQKDAQMDAQODAQADgQCDgQEDgQGDgQIDgQKDgQMDgQODgQAAAgCAAgEAA
			gGAAgIAAgKAAgMAAgOAAgAAggCAggEAggGAggIAggKAggMAggOAggABAgCBA
			gEBAgGBAgIBAgKBAgMBAgOBAgABggCBggEBggGBggIBggKBggMBggOBggACA
			gCCAgECAgGCAgICAgKCAgMCAgOCAgACggCCggECggGCggICggKCggMCggOCg
			gADAgCDAgEDAgGDAgIDAgKDAgMDAgODAgADggCDggEDggGDggIDggKDggMDg
			gODggAAAwCAAwEAAwGAAwIAAwKAAwMAAwOAAwAAgwCAgwEAgwGAgwIAgwKAg
			wMAgwOAgwABAwCBAwEBAwGBAwIBAwKBAwMBAwOBAwABgwCBgwEBgwGBgwIBg
			wKBgwMBgwOBgwACAwCCAwECAwGCAwICAwKCAwMCAwOCAwACgwCCgwECgwGCg
			wICgwKCgwMCgwOCgwADAwCDAwEDAwGDAwIDAwKDAwP/78KCgpICAgP8AAAD/
			AP//AAAA//8A/wD//////yH5BAEAAAcALAAAAAAYABgAQAiFAA8IHEiwoMGD
			AEiRAmAw4QCGBwU6JDWAFIF/ADJq/EcgosePBSdWvPivJMeOIFOqXBlRY8Z9
			+1xC9CiSos2KDz/WvLiRgE+UKX8K/cmyqNGjSF2yBIATQEynE2ciHIBRZkaF
			UhvipDjUZ9aQWweQNHkRqNabY02WNBvSZVq1bJHKnQsyIAA7
		}
		]

		set img(redo) [image create photo -data {
			R0lGODlhGAAYAPcAAAAAAIAAAACAAICAAAAAgIAAgACAgMDAwMDcwKbK8EAg
			AGAgAIAgAKAgAMAgAOAgAABAACBAAEBAAGBAAIBAAKBAAMBAAOBAAABgACBg
			AEBgAGBgAIBgAKBgAMBgAOBgAACAACCAAECAAGCAAICAAKCAAMCAAOCAAACg
			ACCgAECgAGCgAICgAKCgAMCgAOCgAADAACDAAEDAAGDAAIDAAKDAAMDAAODA
			AADgACDgAEDgAGDgAIDgAKDgAMDgAODgAAAAQCAAQEAAQGAAQIAAQKAAQMAA
			QOAAQAAgQCAgQEAgQGAgQIAgQKAgQMAgQOAgQABAQCBAQEBAQGBAQIBAQKBA
			QMBAQOBAQABgQCBgQEBgQGBgQIBgQKBgQMBgQOBgQACAQCCAQECAQGCAQICA
			QKCAQMCAQOCAQACgQCCgQECgQGCgQICgQKCgQMCgQOCgQADAQCDAQEDAQGDA
			QIDAQKDAQMDAQODAQADgQCDgQEDgQGDgQIDgQKDgQMDgQODgQAAAgCAAgEAA
			gGAAgIAAgKAAgMAAgOAAgAAggCAggEAggGAggIAggKAggMAggOAggABAgCBA
			gEBAgGBAgIBAgKBAgMBAgOBAgABggCBggEBggGBggIBggKBggMBggOBggACA
			gCCAgECAgGCAgICAgKCAgMCAgOCAgACggCCggECggGCggICggKCggMCggOCg
			gADAgCDAgEDAgGDAgIDAgKDAgMDAgODAgADggCDggEDggGDggIDggKDggMDg
			gODggAAAwCAAwEAAwGAAwIAAwKAAwMAAwOAAwAAgwCAgwEAgwGAgwIAgwKAg
			wMAgwOAgwABAwCBAwEBAwGBAwIBAwKBAwMBAwOBAwABgwCBgwEBgwGBgwIBg
			wKBgwMBgwOBgwACAwCCAwECAwGCAwICAwKCAwMCAwOCAwACgwCCgwECgwGCg
			wICgwKCgwMCgwOCgwADAwCDAwEDAwGDAwIDAwKDAwP/78KCgpICAgP8AAAD/
			AP//AAAA//8A/wD//////yH5BAEAAP8ALAAAAAAYABgAQAhDAP8JHEiwoMGD
			CBMqXMiw4UICCAlIdEixosWLBSEOnChQ48OMEjliHEmypMmTKDuGvOiRIESR
			CVtuXMlQZsqbOFMGBAA7
		}
		]

		set img(save) [image create photo -data {
			R0lGODlhGAAYAPcAAISCAI6MGtbTzhsaGgAAAP///wAAAAAAANuSRXkAABMA
			AAAAAPi/HQD0VRMAmJJFechVEwDPHv94xO4SAElNRAAAAAAArO4SAHj7EgB4
			+xIAtBtKeWgkRXn/////AQAAAL/UQQD4vx0AZgAAAMhVEwABAAAAwBx/AHTv
			EgDAHxMA+L8dAEwAAAB0EgQAAAAAEPi/HQAAAAAAAAAAAAEAAADEIBMABAEA
			AAAAAAAAAAAAwB8TAFTvEgBsAAgAIwAoAAAAAADIVRMAMNlEAFjpRwBBYnJp
			cgBKeJDvEgAkAAAAVCpHeAAAEwBIDRMAJAAAAEM6XFRlbXBcaWNvbnNcZ2lm
			cy1kaWFsb2d3aW5cMjR4MjQAc2F2ZS5naWYAZgAAAMAcfwB0EgQADQAAAP4B
			AADY7xIAAQAAAF0AEAEDAAAAAQAAAFjwEgDyh+F3hRIBbyEA8ADw7xIABAAA
			AADg/X8CAAAAAAEAAIUSAW9dAQAAAgAAAAEAAAASAAAAXgAQAQAAAAATAAAA
			XQEAAAEAAABeABABAAAAAAIAAAABAAAAEQAAAF0AEAEBAAAAAgAAAFTwEgBm
			IXJxhRIBb3zwEgAFAAAAwPESACsYdXGG8BIAkLUVAIbwEgCQtRUAZPMSAAAA
			AACFEgFvDAAAAAEAAAAAAAAAQzpcVGVtcFxpY29uc1xnaWZzLWRpYWxvZ3dp
			blwyNHgyNFxzYXZlLmdpZgAAAAAAQvISAJC1FQCFEgFv8PESAAtE9HcAAAAA
			OPISAAUAAAAI8hIAAQAAAATyEgDgdXZxAAATAAIAAAAEAAAABgAAAAEAAAAA
			8hIAJAAAAAAAAABC8hIAkLUVAIUSAW+BAgAAAQAAAAAAAADIVRMAIAEAAJAV
			fwB0EgQAdBIEAHDxEgCRgUd4QJZGeP////+A8RIAfZFFeQAAEwAIABQAGAEA
			AGYAAABmAAAAwBx/APZp4XcAAAAACAAAALBl4XcBAAAASPESAHQSBAB4+xIA
			tBtKefgjRXl0tkQAdBIEACH5BAEAAAUALAAAAAAYABgAQAh9AAsIHEiwoMGD
			BQYAICCgocOGBAAMQChQ4UICGDFClEixo0eCFjOKjMjxo8mTKAcqZPjQIcOJ
			FFe2dFkSYciRGWseFDCgp8+fPlMKHUr0I9CjQWMunAmRAEybS5lGfHpQplSd
			Bi0C2Mq1K9WsW3Fu/FpwQICLIscWXcv2ZEAAOw==
		}
		]

		set img(undo) [image create photo -data {
			R0lGODlhGAAYAPcAAAAAAIAAAACAAICAAAAAgIAAgACAgMDAwMDcwKbK8EAg
			AGAgAIAgAKAgAMAgAOAgAABAACBAAEBAAGBAAIBAAKBAAMBAAOBAAABgACBg
			AEBgAGBgAIBgAKBgAMBgAOBgAACAACCAAECAAGCAAICAAKCAAMCAAOCAAACg
			ACCgAECgAGCgAICgAKCgAMCgAOCgAADAACDAAEDAAGDAAIDAAKDAAMDAAODA
			AADgACDgAEDgAGDgAIDgAKDgAMDgAODgAAAAQCAAQEAAQGAAQIAAQKAAQMAA
			QOAAQAAgQCAgQEAgQGAgQIAgQKAgQMAgQOAgQABAQCBAQEBAQGBAQIBAQKBA
			QMBAQOBAQABgQCBgQEBgQGBgQIBgQKBgQMBgQOBgQACAQCCAQECAQGCAQICA
			QKCAQMCAQOCAQACgQCCgQECgQGCgQICgQKCgQMCgQOCgQADAQCDAQEDAQGDA
			QIDAQKDAQMDAQODAQADgQCDgQEDgQGDgQIDgQKDgQMDgQODgQAAAgCAAgEAA
			gGAAgIAAgKAAgMAAgOAAgAAggCAggEAggGAggIAggKAggMAggOAggABAgCBA
			gEBAgGBAgIBAgKBAgMBAgOBAgABggCBggEBggGBggIBggKBggMBggOBggACA
			gCCAgECAgGCAgICAgKCAgMCAgOCAgACggCCggECggGCggICggKCggMCggOCg
			gADAgCDAgEDAgGDAgIDAgKDAgMDAgODAgADggCDggEDggGDggIDggKDggMDg
			gODggAAAwCAAwEAAwGAAwIAAwKAAwMAAwOAAwAAgwCAgwEAgwGAgwIAgwKAg
			wMAgwOAgwABAwCBAwEBAwGBAwIBAwKBAwMBAwOBAwABgwCBgwEBgwGBgwIBg
			wKBgwMBgwOBgwACAwCCAwECAwGCAwICAwKCAwMCAwOCAwACgwCCgwECgwGCg
			wICgwKCgwMCgwOCgwADAwCDAwEDAwGDAwIDAwKDAwP/78KCgpICAgP8AAAD/
			AP//AAAA//8A/wD//////yH5BAEAAP8ALAAAAAAYABgAQAhDAP8JHEiwoMGD
			CAkoJICwocOHECEuZGiQYsSLGDNqPKjwn0WCHzeG3EiypMmTKFMWtNhx4EiO
			LStKnLhSpc2bOE0GBAA7
		}
		]

		set img(date) [image create photo -data {
			R0lGODlhGAAYAKIAAAAAAP///wCA/wBAgODg4P///wAAAAAAACH5BAEAAAUA
			LAAAAAAYABgAAANvWLrc/jCyQau99A3Bu/+D5mGkpWBfOp5BG2yqGhaD+5bl
			SexAbf+/mQ/wshFdgKRLaBvsCMknQLoTPgnDwFGLbDGXRuD2671OzVcrz9fa
			cpsscLc9J7+kUWgyL7wA/14agH8zDTg5EomKi4yNjg0JADs=
		}
		]
	}
}

proc TextEditor::CreateMenu { } {
	variable HIDE_NEW_AND_OPEN
	variable mainframe
	variable maintext
	variable showtoolbar

	if { $HIDE_NEW_AND_OPEN == "0" } {
		set menufile [list \
			[_ "&File"] {} {} 0 [list \
				[list command [_ "&New"] file [_ "Creates a new document"] {Ctrl n} \
				-command "TextEditor::NewFileW"] \
				[list command [_ "&Open"]... file [_ "Opens an existing document"] {Ctrl o} \
				-command "TextEditor::ReadFileW"] \
				[list command [_ "&Save"] file [_ "Saves the document"] {Ctrl s} \
				-command "TextEditor::SaveFileW save"] \
				[list command [_ "&Save as"]... file [_ "Saves the document with a new name"] {} \
				-command "TextEditor::SaveFileW saveas"] \
				{separator} \
				[list command [_ "E&xit"] {} [_ "Exit the application"] {} -command "TextEditor::Exit"] \
			] \
		]
} elseif { $HIDE_NEW_AND_OPEN == "1" } {
	set menufile [list \
		[_ "&File"] {} {} 0 [list \
			[list command [_ "&Save"] file [_ "Saves the document"] {Ctrl s} \
			-command "TextEditor::SaveFileW save"] \
			[list command [_ "&Save as"]... file [_ "Saves the document with a new name"] {} \
			-command "TextEditor::SaveFileW saveas"] \
			{separator} \
			[list command [_ "E&xit"] {} [_ "Exit the application"] {} -command "TextEditor::Exit"] \
		] \
	]
} else { #only hyde open
	set menufile [list \
		[_ "&File"] {} {} 0 [list \
			[list command [_ "&New"] file [_ "Creates a new document"] {Ctrl n} \
			-command "TextEditor::NewFileW"] \
			[list command [_ "&Save"] file [_ "Saves the document"] {Ctrl s} \
			-command "TextEditor::SaveFileW save"] \
			[list command [_ "&Save as"]... file [_ "Saves the document with a new name"] {} \
			-command "TextEditor::SaveFileW saveas"] \
			{separator} \
			[list command [_ "E&xit"] {} [_ "Exit the application"] {} -command "TextEditor::Exit"] \
		] \
	]
}

set menuedit [list \
	[_ "&Edit"] {} {} 0 [list \
		[list command [_ "&Undo"] edit [_ "Reverts the last action"] {Ctrl z} -command "TextEditor::Undo" ] \
		[list command [_ "&Redo"] edit [_ "Reverts the last undo"] {Ctrl y} -command "TextEditor::Redo"] \
		{separator} \
		[list command [_ "Cu&t"] edit [_ "Cut the selection and insert it in the clipboard"] {Ctrl x} \
		-command "tk_textCut $TextEditor::maintext" ] \
		[list command [_ "&Copy"] edit [_ "Copy the selection to the clipboard"] {Ctrl c} \
		-command "tk_textCopy $TextEditor::maintext"] \
		[list command [_ "&Paste"] edit [_ "Insert the clipboard contents"] {Ctrl v} \
		-command "tk_textPaste $TextEditor::maintext"] \
		[list command [_ "Select &all"] edit [_ "Select all the document"] {Ctrl a} \
		-command "$TextEditor::maintext tag add sel 1.0 end"] \
		{separator} \
		[list command [_ "&Find"]... edit [_ "Find the specified text"] {Ctrl f} \
		-command "TextEditor::FindW" ] \
		{separator} \
		[list command [_ "&Go to"]... edit [_ "Moves to a specifies location"] {Ctrl g} \
		-command " TextEditor::GotoLineW" ] \
		{separator} \
		[list command [_ "Insert &date"] edit [_ "Inserts the current date"] {Ctrl d} \
		-command " TextEditor::InsertCurrentDate" ] \
	] \
]

set menuoptions [list \
	[_ "&Options"] {} {} 0 [list \
		[list checkbutton [_ "Toolbar"] {} [_ "Show/hide toolbar"] {} \
		-variable TextEditor::showtoolbar \
		-command [list TextEditor::UpdateToolBar] \
	] \
] \
]

set descmenu "$menufile $menuedit $menuoptions"

MainFrame::_create_menubar $TextEditor::mainframe $descmenu
}

proc TextEditor::UpdateToolBar { } {
variable mainframe
variable showtoolbar
$mainframe showtoolbar 0 $showtoolbar
}


#filename the name of the file to open
#linenumber to see this line number
#string to find this word
#hidenewandopen to show or not the new and open buttons
#enc the set encoding to read and save files: "", utf-8
proc TextEditor::Create { { filename "" } { linenumber 0 } { string "" } { hidenewandopen 0 } { w .texteditor } {enc ""} } {
	variable HIDE_NEW_AND_OPEN
	variable mainframe
	variable maintext
	variable status
	variable percent
	variable changed
	variable rowandcol
	variable currentfile
	variable encoding
	variable cansavetovariable

	set encoding $enc
	set HIDE_NEW_AND_OPEN $hidenewandopen

	if { [winfo exists $w] && $w != "." } {
		if { $cansavetovariable } {
			SaveToVariable
		} else {
			if { [SaveFileW ask] == -1 } { return }
		}
		destroy $w
	}

	if { [winfo exists .gid.central.s] && [info procs ::InitWindow] != "" } {
		::InitWindow2 $w -title $filename \
            -geometryvariable PrePostTextEditorWindowGeom \
            -initcommand OpenNotes
		if { ![winfo exists $w] } { return close } ;# windows disabled || UseMoreWindows == 0
	} else {
		if { ![winfo exists $w] } {
			if { [winfo exists .gid.central.s] && [GiD_Project set disable_windows] } {
				return close
			}
			toplevel $w
		}
	}

	wm protocol $w WM_DELETE_WINDOW "TextEditor::Exit"

	CreateImages

	set mainframe [MainFrame $w.mainframes \
		-textvariable TextEditor::status \
		-progressvar  TextEditor::progress]

	$mainframe showstatusbar status

	set tb [$mainframe addtoolbar]
	$mainframe addindicator -textvariable TextEditor::changed -width 25
	$mainframe addindicator -textvariable TextEditor::rowandcol -width 25

	set bbox(1) [ButtonBox $tb.bbox1 -spacing 0 -padx 1 -pady 1 -homogeneous 1 -spacing 1]
	set sep1 [Separator $tb.sep1 -orient vertical]
	set bbox(2) [ButtonBox $tb.bbox2 -spacing 0 -padx 1 -pady 1 -homogeneous 1 -spacing 1]
	set sep2 [Separator $tb.sep2 -orient vertical]
	set bbox(3) [ButtonBox $tb.bbox3 -spacing 0 -padx 1 -pady 1 -homogeneous 1 -spacing 1]
	set sep3 [Separator $tb.sep3 -orient vertical]
	set bbox(4) [ButtonBox $tb.bbox4 -spacing 0 -padx 1 -pady 1 -homogeneous 1 -spacing 1]

	set frame [$mainframe getframe]

	set sw [ScrolledWindow $frame.sw]
	set maintext [text $sw.txt -wrap none -undo true]
	$sw setwidget $maintext

	set names ""
	set helps ""
	set cmds ""
	set iboxs ""
	if { $HIDE_NEW_AND_OPEN == "0" } {
		lappend names new open
		lappend helps [_ "Creates a new document"]  [_ "Opens an existing document"]
		lappend cmds  "TextEditor::NewFileW" "TextEditor::ReadFileW"
		lappend iboxs 1 1
	} elseif { $HIDE_NEW_AND_OPEN == "2" } {
		lappend names new
		lappend helps [_ "Creates a new document"]
		lappend cmds  "TextEditor::NewFileW"
		lappend iboxs 1
	}
	lappend names save find cut copy paste undo redo date
	lappend helps [_ "Saves the document"] [_ "Find the specified text"] \
		[_ "Cut the selection and insert it in the clipboard"] \
		[_ "Copy the selection to the clipboard"] [_ "Insert the clipboard contents"] \
		[_ "Reverts the last action"] [_ "Reverts the last undo"] [_ "Inserts the current date"]
	lappend cmds "TextEditor::SaveFileW save" "TextEditor::FindW" \
		"tk_textCut $maintext" "tk_textCopy $maintext" "tk_textPaste $maintext" \
		"TextEditor::Undo" "TextEditor::Redo" "TextEditor::InsertCurrentDate"
	lappend iboxs 1 2 3 3 3 3 3 4
	foreach name $names help $helps cmd $cmds ibox $iboxs {
		$bbox($ibox) add -image $TextEditor::img($name) \
			-takefocus 0 -relief link \
			-helptext $help        -command $cmd \
			-relief flat
	}
	pack $bbox(1) -side left -anchor w
	pack $sep1 -side left -fill y -padx 4 -anchor w
	pack $bbox(2) -side left -anchor w
	pack $sep2 -side left -fill y -padx 4 -anchor w
	pack $bbox(3) -side left -anchor w
	pack $sep3 -side left -fill y -padx 4 -anchor w
	pack $bbox(4) -side left -anchor w

	pack $sw -fill both -expand yes
	pack $mainframe -fill both -expand yes

	CreateMenu

	if { $filename != "" } {
		ReadFile $filename
		if { $linenumber > 0 } {
			GotoLine $linenumber $string
		}
	} else {
		set currentfile ""
	}

	bind $maintext <KeyRelease> TextEditor::ChangeRowCol
	bind $maintext <ButtonRelease> TextEditor::ChangeRowCol
	bind $maintext <<Modified>> TextEditor::Modified
	bind $maintext <${::acceleratorKey}-f> TextEditor::FindW
	bind $maintext <${::acceleratorKey}-z> "TextEditor::Undo ; break"
	bind $maintext <${::acceleratorKey}-y> "TextEditor::Redo ; break"
	bind $maintext <F3> [list TextEditor::Find $maintext]
	#to avoid past/copy twice when press this keys
	bind $maintext <${::acceleratorKey}-Key-x> "tk_textCut   $TextEditor::maintext ; break"
	bind $maintext <${::acceleratorKey}-Key-c> "tk_textCopy  $TextEditor::maintext ; break"
	bind $maintext <${::acceleratorKey}-Key-v> "tk_textPaste $TextEditor::maintext ; break"

	bind $maintext <FocusIn> [list $maintext tag remove find_tag 1.0 end]

	$maintext tag configure find_line_tag -background yellow -foreground blue
	$maintext tag configure find_tag -background black -foreground white
	$maintext tag raise sel
}

proc TextEditor::ChangeRowCol { } {
	variable maintext
	variable rowandcol
	scan [$maintext index insert] "%d.%d" row col
	incr col
	set rowandcol [_ "row %s col %s" $row $col]
}

proc TextEditor::Modified { } {
	variable maintext
	variable changed
	if { [$maintext edit modified] } {
		set changed [_ "Modified"]
	} else {
		set changed [_ "Unmodified"]
	}
}

proc TextEditor::Exit {} {
	variable mainframe
	if { [SaveFileW ask] == -1 } { return }

	if { [winfo parent $mainframe] == "." } {
		exit
	} else {
		destroy [winfo parent $mainframe]
	}
}


proc TextEditor::SaveFileW { what } {
	variable mainframe
	variable maintext

	variable currentfile
	variable cansavetovariable

	if { $what == "ask" } {
		if { ![$maintext edit modified] } { return 0 }

		if { $currentfile == "" && $cansavetovariable } {
			set ret [DialogWin::messageBox -default yes -message \
				[_ "Do you want to save the changes?"] -parent $mainframe \
				-title [_ "Save changes"] -type yesnocancel]

		} else {
			set ret [DialogWin::messageBox -default yes -message \
				[_ "Do you want to save file '%s'?" $currentfile] -parent $mainframe \
				-title [_ "Save file"] -type yesnocancel]
		}
		if { $ret == "cancel" } {
			return -1
		} elseif { $ret == "no" } {
			return 0
		} else {
			set filename $currentfile
		}
	} elseif { $what == "saveas"  } {
		set types [list \
			[list [_ "All files"] ".*"]]
		set defaultdir [file dirname $currentfile]
		set filename [tk_getSaveFile -filetypes $types -initialdir $defaultdir -parent $mainframe \
			-title [_ "Save file"]]
	} else {
		set filename $currentfile
	}

	if { $filename != "" } {
		SaveFile $filename
	} else {
		if { $cansavetovariable } {
			SaveToVariable
		} else {
			return -1
		}
	}
	return 0
}

proc TextEditor::ReadFileW { } {
	variable mainframe
	variable currentfile

	if { [SaveFileW ask] == -1 } { return }
	set types [list \
		[list [_ "All files"] ".*"]]
	set defaultdir [file dirname $currentfile]
	set filename [tk_getOpenFile -filetypes $types -initialdir $defaultdir -parent $mainframe \
		-title [_ "Open file"]]
	if { $filename == "" } { return }
	ReadFile $filename
	return 0
}

proc TextEditor::NewFileW { } {
	if { [SaveFileW ask] == -1 } { return }
	TextEditor::NewFile
}

proc TextEditor::NewFile { } {
	variable mainframe
	variable maintext
	variable currentfile

	set currentfile ""
	ResetSaveVariable

	if { [info exists maintext] && [winfo exists $maintext] } {
		$maintext delete 1.0 end
		$maintext edit reset
		$maintext edit modified false
		set w [winfo parent $mainframe]
		wm title $w $currentfile
	}
}


proc TextEditor::SaveFile { filename } {
	variable mainframe
	variable maintext
	variable encoding
	variable currentfile
	if { $filename == "" } {
		SaveFileW saveas
		return
	}
	set fp [open $filename w]
	if { $fp == "" } {
		ShowMsg [_ "Error: cannot open file '%s'" $filename] error
		return
	}
	if { $encoding != "" } {
		fconfigure $fp -encoding $encoding
	}
	puts -nonewline $fp [$maintext get 1.0 end-1c]
	close $fp

	#do not reset undo stack, the user can save by error
	#$maintext edit reset
	$maintext edit modified false
	set currentfile $filename
	ResetSaveVariable
	set w [winfo parent $mainframe]
	wm title $w $filename
	if { [file size $filename] == 0 } {
		file delete $filename
	}
}

proc TextEditor::ResetSaveVariable { } {
	variable vartosave
	set vartosave ""
}

proc TextEditor::SaveToVariable { } {
	variable maintext
	variable vartosave
	set vartosave [$maintext get 1.0 end-1c]
	$maintext edit modified false
}

proc TextEditor::ReadFromVariable { } {
	variable mainframe
	variable maintext
	variable vartosave
	variable currentfile
	set w [winfo parent $mainframe]
	$maintext delete 1.0 end
	$maintext insert end $vartosave
	set currentfile ""
	$maintext edit reset
	$maintext edit modified false
	wm title $w [_ "UNNAMED"]
}

proc TextEditor::SaveFromVariableToFile { } {
	set fail 0
	variable currentfile
	variable vartosave
	variable encoding
	if { $vartosave == "" } return

	set modelname [GiD_Info Project ModelName]
	if  { $modelname == "UNNAMED" } {
		return
	} else {
		set filename [file join $modelname.gid [file tail $modelname].txt]
	}
	set fp [open $filename w]
	if { $fp == "" } {
		ShowMsg [_ "Error: cannot open file '%s'" $filename] error
		return
	}
	if { $encoding != "" } {
		fconfigure $fp -encoding $encoding
	}
	puts -nonewline $fp $vartosave
	close $fp

	set currentfile $filename
	ResetSaveVariable
	return $fail
}

proc TextEditor::ReadFile { filename } {
	variable mainframe
	variable maintext
	variable currentfile
	variable encoding

	set w [winfo parent $mainframe]

	$maintext delete 1.0 end

	if { ![file exists $filename] } {
		set currentfile ""
		$maintext edit reset
		$maintext edit modified false
		wm title $w $filename
		ShowMsg [_ "Error: file '%s' does not exist" $filename] error
		return
	}
	set fp [open $filename r]
	if { $fp == "" } {
		ShowMsg [_ "Error: cannot open file '%s'" $filename] error
		return
	}
	if { $encoding != "" } {
		fconfigure $fp -encoding $encoding
	}
	$maintext insert end [read $fp]
	close $fp

	set currentfile $filename
	$maintext edit reset
	$maintext edit modified false
	wm title $w $filename
	ResetSaveVariable
}

proc TextEditor::FindW {} {
	variable mainframe
	variable maintext
	variable findparam

	if { [winfo exists $mainframe.__dialogwin] } {
		destroy $mainframe.__dialogwin
	}

	$maintext tag remove find_line_tag 1.0 end
	$maintext tag remove find_tag 1.0 end

	set f [DialogWin::Init $mainframe [_ "Search"] separator_nograb "" [_ "Search next"]]

	set w [winfo toplevel $f]

	label $f.l1 -text [_ "Search"]:

	set cursel [$maintext tag range sel]
	if { $cursel != "" } {
		set selection [$maintext get [lindex $cursel 0] [lindex $cursel 1]]
		if { $selection != "" && [string length $selection] < 100 } {
			set TextEditor::findparam(searchstring) $selection
		}
	}

	ComboBox $f.c1 -textvariable TextEditor::findparam(searchstring) \
		-values $TextEditor::findparam(searchstrings) -width 20 -editable 1
	set TextEditor::findparam(searchstring) [lindex $TextEditor::findparam(searchstrings) 0]

	focus $f.c1
	bind $f.c1 <Return> "DialogWin::InvokeOK"
	bind $f <Return> "DialogWin::InvokeOK"
	bind [winfo parent $f] <Return> "DialogWin::InvokeOK"
	bind [winfo parent $f] <F3> "DialogWin::InvokeOK"

	set f1 [ttk::labelframe $f.f1 -text [_ "Direction"]]
	radiobutton $f1.r1 -text [_ "Up"] -variable TextEditor::findparam(searchdirection) -value "backward"
	radiobutton $f1.r2 -text [_ "Down"] -variable TextEditor::findparam(searchdirection) -value "forward"
	grid $f1.r1 $f1.r2 -sticky w

	frame $f.f2
	checkbutton $f.f2.cb1 -text [_ "Consider case"] -variable TextEditor::findparam(considercase)

	checkbutton $f.f2.cb2 -text [_ "From beginning"] -variable TextEditor::findparam(frombeginning)

	grid $f.f2.cb1 $f.f2.cb2 -sticky w
	grid columnconf $f.f2 1 -weight 1

	checkbutton $f.cb2 -text [_ "Whole word"] -variable TextEditor::findparam(wholeword)

	grid $f.l1 $f.c1 -sticky ne
	grid $f.f1   -   -sticky nsew
	grid $f.f2   -  -sticky ew
	grid $f.cb2  -  -sticky w
	grid conf $f.c1 -sticky ew
	grid columnconf $f 1 -weight 1

	set action [DialogWin::CreateWindow]

	set lastirow -1
	set lasticol -1

	while 1 {
		if { $action == 0 } {
			DialogWin::DestroyWindow
			return
		}

		Find $maintext

		if { $TextEditor::findparam(searchpos) == ""} {
			if { $findparam(searchdirection) == "forward" } {
				set TextEditor::findparam(searchpos) "1.0"
			} else {
				set TextEditor::findparam(searchpos) end
			}
			set errstring [_ "Could not find string '%s'" $TextEditor::findparam(searchstring)]
			ShowMsg $errstring info
		}

		if { [lsearch $TextEditor::findparam(searchstrings) $TextEditor::findparam(searchstring)] == -1 } {
			set TextEditor::findparam(searchstrings) \
				[linsert $TextEditor::findparam(searchstrings) 0 $TextEditor::findparam(searchstring)]
		}
		if { [llength $TextEditor::findparam(searchstrings)] > 7 } {
			set TextEditor::findparam(searchstrings) [lrange $TextEditor::findparam(searchstrings) 0 6]
		}
		set action [DialogWin::WaitForWindow]
	}
}

proc TextEditor::Find { txtwidget } {
	variable findparam

	if { ![winfo exists $txtwidget] } return ""

	if { $findparam(searchstring) == "" } return ""
	if { $findparam(searchpos) == "" || $findparam(frombeginning) } { set findparam(searchpos) 1.0 }

	if { $findparam(searchdirection) == "forward" } {
		set limit end
	} else {
		set limit 1.0
	}

	#set cmd "$txtwidget search -$findparam(searchdirection)"
	#if { !$findparam(considercase) } { append cmd " -nocase" }
	#append cmd " [list $findparam(searchstring) $findparam(searchpos) $limit]"
	#set findparam(searchpos) [eval $cmd]

	if { $findparam(considercase) } {
		set rex (?c)
	} else {
		set rex (?i)
	}
	if { $findparam(wholeword) } {
		append rex {\m}
	}
	set ss $findparam(searchstring)
	foreach i {. \ * ? + | [ ] ( ) } {
		regsub -all \[$i\] $ss {\\&} ss
	}
	append rex $ss
	if { $findparam(wholeword) } {
		append rex {\M}
	}
	set findparam(searchpos) [$txtwidget search -$findparam(searchdirection) -regexp -- $rex $findparam(searchpos) $limit]

	if { $findparam(searchpos) != ""} {
		set len [string length $findparam(searchstring)]
		$txtwidget see $findparam(searchpos)
		$txtwidget mark set insert "$findparam(searchpos) + $len char"
		$txtwidget tag remove find_tag 1.0 end
		$txtwidget tag remove sel 1.0 end
		#$txtwidget tag add sel $findparam(searchpos) "$findparam(searchpos) + $len char"
		$txtwidget tag add find_line_tag "$findparam(searchpos) linestart" "$findparam(searchpos) lineend"
		$txtwidget tag add find_tag $findparam(searchpos) "$findparam(searchpos) + $len char"
		$txtwidget tag add sel $findparam(searchpos) "$findparam(searchpos) + $len char"
		#focus $txtwidget
		if { $findparam(searchdirection) == "forward" } {
			set findparam(searchpos) "$findparam(searchpos) + $len char"
		}
		ChangeRowCol
	}

	return $findparam(searchpos)
}

proc TextEditor::Undo { } {
	variable maintext
	catch { $maintext edit undo }
}

proc TextEditor::Redo { } {
	variable maintext
	catch { $maintext edit redo }
}

proc TextEditor::GotoLineW { } {
	variable mainframe

	set f [DialogWin::Init $mainframe [_ "Goto line"]: separator]
	set w [winfo toplevel $f]

	label $f.l -text [_ "Enter line number"]:
	entry $f.e -textvariable DialogWin::user(LineNumber)

	grid $f.l $f.e -padx 3 -pady 3
	grid $f

	focus $f.e
	bind $f.e <Return> "DialogWin::InvokeOK"
	bind $w <Return> "DialogWin::InvokeOK"

	set action [DialogWin::CreateWindow]
	while 1 {
		switch $action {
			0 {
				DialogWin::DestroyWindow
				return
			}
			1 {
				if { ![string is integer $DialogWin::user(LineNumber)] || $DialogWin::user(LineNumber) < 1 } {
					ShowMsg [_ "Bad line number"] error
				} else {
					GotoLine $DialogWin::user(LineNumber)
					DialogWin::DestroyWindow
					return
				}
			}
		}
		set action [DialogWin::WaitForWindow]
	}
}

proc TextEditor::InsertCurrentDate { } {
	variable maintext

	set cur_time [ clock seconds]
	set d [ clock format $cur_time -format %e]
	set du [ expr $d % 10]
	set lst_trad_month [ list [_ "January"] [_ "February"] [_ "March"] [_ "April"] [_ "May"] [_ "June"] \
		[_ "July"] [_ "August"] [_ "September"] [_ "October"] [_ "November"] [_ "December"]]
	set idx_month [string trimleft [clock format $cur_time -format %m] 0]
	set mes_txt [ lindex $lst_trad_month [ expr {$idx_month-1}]]
	switch [GiD_Set Language] {
		"es" {
			set fmt "%e $mes_txt %Y"
		}
		"de" {
			set fmt "%e. $mes_txt, %Y"
		}
		"en" {
			# 1st, 2nd, 3rd, 4th ..., 11th, 12th, 13th, ..., 21st, 22nd, 23rd, 24th, ...
			set ext "th"
			if { ( $du == 1) && ( $d != 11)} { set ext "st"}
			if { ( $du == 2) && ( $d != 12)} { set ext "nd"}
			if { ( $du == 3) && ( $d != 13)} { set ext "rd"}
			set fmt "%B %e$ext, %Y"
		}
		default {
			set fmt "%e $mes_txt %Y"
		}
	}
	$maintext ins end [ clock format $cur_time -format $fmt]
}

proc TextEditor::GotoLine { row {string "" } } {
	variable mainframe
	variable maintext

	$maintext see "$row.0"
	$maintext tag remove sel 1.0 end

	if { $string != "" } {
		# ramsan: added -2 lines because line number is not OK for lines
		# with continuation
		set pos [$maintext search -- $string "$row.0-2 lines" "$row.end"]
		if { $pos != "" } {
			set len [string length $string]
			$maintext mark set insert "$pos + $len char"
			$maintext tag add find_tag $pos "$pos + $len char"
			$maintext tag add sel $pos "$pos + $len char"
		} else {
			ShowMsg [_ "String '%s' does not exist at line %s" $string $row] info
			raise [winfo toplevel $maintext]
			$maintext mark set insert "$row.0"
			$maintext tag add find_tag "$row.0" "$row.end"
			$maintext tag add sel "$row.0" "$row.end"
		}
	} else {
		$maintext mark set insert "$row.0"
		$maintext tag add find_tag "$row.0" "$row.end"
		$maintext tag add sel "$row.0" "$row.end"
	}

	focus $maintext
	ChangeRowCol
}



proc TextEditor::ShowMsg { text type {title ""}} {
	variable mainframe
	if { $type == "error" } {
		if { $title == "" } { set title [_ "Error"] }
		set icon error.png
	} elseif { $type == "warning" } {
		if { $title == "" } { set title [_ "Warning"] }
		set icon warning.png
	} elseif { $type == "info" } {
		if { $title == "" } { set title [_ "Information"] }
		set icon info.png
	} elseif { $type == "question" } {
		if { $title == "" } { set title [_ "Question"] }
		set icon question.png
	}  else {
		if { $title == "" } { set title [_ "Information"] }
		set icon info.png
	}
	DialogWinTop::messageBox -title $title -icon $icon \
		-parent $mainframe -message $text -type ok
}

if { [info exists argv0] && [info script] == $argv0 } {
	#TextEditor::Create "E:/Programas/ActiveTcl8.4.1.0/lib/texteditor1.0/Operations.cc" 49 "MAXDOUBLE" 0 . utf-8
	TextEditor::Create "" "" "" 0 . utf-8
}
