# ------------------------------------------------------------------------------
# gidi18n.tcl --
#
#  This file contains routines related to internationalization = i18n
# ------------------------------------------------------------------------------

namespace eval ::GIDi18n {

}

proc GIDmcset { lan src dest } {
  ::msgcat::mcset $lan $src [subst -nocommands -novariables $dest]
}

# ------------------------------------------------------------------------------
# GIDi18n::language --
# ------------------------------------------------------------------------------

proc GIDi18n::language {{what ""}} {
  set list {
    aa Afar ab Abkhazian af Afrikaans am Amharic ar Arabic as Assamese
    ay Aymara az Azerbaijani ba Bashkir be Byelorussian bg Bulgarian
    bh Bihari bi Bislama bn Bengali bo Tibetan br Breton ca Catalan
    co Corsican cs Czech cy Welsh da Danish de German dz Bhutani
    el Greek en English eo Esperanto es Spanish et Estonian eu Basque
    fi Finnish fj Fiji fo Faroese fr French fy Frisian ga Irish
    gl Galician gn Guarani gu Gujarati ha Hausa hi Hindi hr Croatian
    hu Hungarian hy Armenian ia Interlingua ie Interlingue ik Inupiak
    in Indonesian is Icelandic it Italian iw Hebrew ja Japanese ji Yiddish
    jv Javanese ka Georgian kk Kazakh kl Greenlandic km Cambodian
    kn Kannada ko Korean ks Kashmiri ku Kurdish ky Kirghiz la Latin
    ln Lingala lo Laothian lt Lithuanian lv Latvian mg Malagasy mi Maori
    mk Macedonian ml Malayalam mn Mongolian mo Moldavian mr Marathi
    ms Malay mt Maltese my Burmese na Nauru ne Nepali nl Dutch no Norwegian
    oc Occitan om Afan(Oromo) or Oriya pa Punjabi pl Polish ps Pashto
    pt Portuguese qu Quechua rm Rhaeto-romance rn Kurundi ro Romanian
    ru Russian rw Kinyarwanda sa Sanskrit sd Sindhi sg Sangho
    sh Serbo-croatian si Singhalese sk Slovak sl Slovenian sm Samoan
    sn Shona so Somali sq Albanian sr Serbian ss Siswati st Sesotho
    su Sundanese sv Swedish sw Swahili ta Tamil te Telugu tg Tajik th Thai
    ti Tigrinya tk Turkmen tl Tagalog tn Setswana to Tonga tr Turkish
    ts Tsonga tt Tatar tw Twi uk Ukrainian ur Urdu uz Uzbek vi Vietnamese
    vo Volapuk wo Wolof xh Xhosa yo Yoruba zh Chinese zu Zulu
  }
  if {$what==""} {
    foreach {short long} $list {lappend res $short}
    return [lsort $res]
  }
  set i [lsearch -exact $list $what]
  if {$i==-1} {return "--"}
  if {$i%2} {return [lindex $list [expr $i-1]]}
  return [lindex $list [expr $i+1]]
}

# ------------------------------------------------------------------------------
# ::GIDi18n::SaveLangFile --
#
#   Create a file with the name of the languages. This file will be used by
#   RamTranslator.
# ------------------------------------------------------------------------------

proc ::GIDi18n::SaveLangFile { {fname language} } {
  set fd [open $fname "w"]
  foreach slan [language] {
    puts $fd [language $slan]
  }
  close fname
}

#converts a real Unicode string (e.g. as pasted to a widget) to \u-escaped ASCII
proc GIDi18n::u2a { s } {
    set res ""
    foreach i [split $s ""] {
        scan $i %c c
        if {$c<128} {append res $i} else {append res \\u[format %04.4X $c]}
    }
    set res
}

# ------------------------------------------------------------------------------
# namespace ::GIDi18nProblemType --
# ------------------------------------------------------------------------------

namespace eval ::GIDi18nProblemType {
}

# ------------------------------------------------------------------------------
# ::GIDi18nProblemType::load --
# ------------------------------------------------------------------------------

proc ::GIDi18nProblemType::load { dir } {
  package require tdom
  set xmlfile [file join $dir [file root [file tail $dir]]].xml
  if {[file exists $xmlfile]} {
    set xml [tDOM::xmlReadFile $xmlfile]
    if {[info exists xmldoc_msg]} {
      $xmldoc_msg delete
    }
    set xmldoc_msg [dom parse $xml]
    set root [$xmldoc_msg documentElement]
    set msgcat_node [$root select Program/MsgcatRoot]
    if {$msgcat_node ne ""} {
      set msgcat_root [$msgcat_node text]
      if {$msgcat_root eq ""} {
        WarnWinText [_ "Node MsgcatRoot is empty"]
        return
      }
      if {[file pathtype $msgcat_root] eq "absolute"} {
        set msgcat_path $msgcat_root
      } else {
        set msgcat_path [file join $dir $msgcat_root]
      }
      if {![file exists $msgcat_path]} {
        WarnWinText [_ "Requested message catalog %s doesn't exists" $msgcat_path]
        return
      }
    } else {
      set msgcat_path [file join $dir msgs]
      if {![file exists $msgcat_path]} {
        return
      }
    }
  } else {
    set msgcat_path [file join $dir msgs]
    if {![file exists $msgcat_path]} {
      return
    }
  }
  msgcat::mcload $msgcat_path
}

# ------------------------------------------------------------------------------
# ::GIDi18nProblemType::unload --
# ------------------------------------------------------------------------------

proc ::GIDi18nProblemType::unload { } {
}

# ------------------------------------------------------------------------------
# =
# ------------------------------------------------------------------------------


proc = { src args } {
    set res [namespace eval ::GIDi18nProblemType ::msgcat::mc [list $src] $args]
    regsub "#C#.*" $res {} res
    return $res
}