package provide gid_monitoring 1.24

package require gid_cross_platform
package require gid_tcl_mathfunc

namespace eval gid_monitoring {
    variable request "" ;#to only process and return the asked variables, without the overhead of compute all possible data
    set expr_procedures "" ;#to ask for defined body procedures
}

proc gid_monitoring::set_request { variable_names } {
    variable request
    set request $variable_names
}

proc gid_monitoring::set_expr_procedures { procedures } {
    variable expr_procedures
    set expr_procedures $procedures
}

proc printDebug { txt} {
    set fo [ open /tmp/gid_debug.txt a]
    puts $fo "$txt"
    close $fo
}

proc gid_monitoring::getinfo { } {
    variable request
    variable expr_procedures
    array unset MonitoringInfo

    #these process related variables are always returned, because are not expensive
    set pid [pid]
    set MonitoringInfo(workingset) [gid_cross_platform::get_process_info_memory $pid]
    set MonitoringInfo(workingsetpeak) [gid_cross_platform::get_process_info_memory_peak $pid]
    set MonitoringInfo(elapsedtime) [gid_cross_platform::get_process_info_elapsedtime $pid]
    set MonitoringInfo(gid_project_tmp_directory) [GiD_Info project TmpDirectory]

    #from here only asked variables are get and returned


    #######################
    #common information
    #######################

    set item gid_is_developer
    if { [lsearch $request $item]!=-1 } {
        set MonitoringInfo($item) [GidUtils::IsDeveloperVersion]
    }
    set item gid_version
    if { [lsearch $request $item]!=-1 } {
        set MonitoringInfo($item) [GiD_Info GiDVersion]
    }

    set item gid_opengl_is_software_renderer
    if { [ lsearch $request $item ] != -1 } {
        # ::GidUtils::OpenGLSoftwareRenderer returns
        # -1 - no OpenGL
        #  0 - hardware accelerated OpenGL
        #  1 - software OpenGL
        set MonitoringInfo($item) [ ::GidUtils::OpenGLSoftwareRenderer]
    }
    set item gid_opengl_has_vbo
    if { [ lsearch $request $item ] != -1 } {
        set MonitoringInfo($item) [ ::GidUtils::OpenGLHasVBO ]
    }

    #######################
    #preprocess information
    #######################

    ##############
    ##### Geometry
    ##############

    set item geometry_bounding_box
    if { [lsearch $request $item]!=-1 } {
        lassign [GiD_Info bounding_box -pmin_pmax -layers_off_also -geometry] x_min y_min z_min x_max y_max z_max
        set MonitoringInfo($item) [list $x_min $y_min $z_min $x_max $y_max $z_max]
    }

    set item geometry_bounding_box_sizes
    if { [lsearch $request $item]!=-1 } {
        lassign [GiD_Info bounding_box -pmin_pmax -layers_off_also -geometry] x_min y_min z_min x_max y_max z_max
        set MonitoringInfo($item) [list [expr $x_max-$x_min] [expr $y_max-$y_min] [expr $z_max-$z_min]]
    }

    set item geometry_bounding_box_diagonal
    if { [lsearch $request $item]!=-1 } {
        lassign [GiD_Info bounding_box -pmin_pmax -layers_off_also -geometry] x_min y_min z_min x_max y_max z_max
        set MonitoringInfo($item) [MathUtils::VectorDistance [list $x_min $y_min $z_min] [list $x_max $y_max $z_max]]
    }

    set item geometry_bounding_box_center_distance_to_origin
    if { [lsearch $request $item]!=-1 } {
        lassign [GiD_Info bounding_box -pmin_pmax -layers_off_also -geometry] x_min y_min z_min x_max y_max z_max
        set MonitoringInfo($item) [MathUtils::VectorModulus [MathUtils::VectorMean [list $x_min $y_min $z_min] [list $x_max $y_max $z_max]]]
    }

    set item geometry_normal_mean
    if { [lsearch $request $item]!=-1 } {
        set MonitoringInfo($item) [GidUtils::GetMeanNormal_Geometry]
    }

    foreach item {numpoints numlines numsurfaces numvolumes numdimensions} {
        if { [lsearch $request $item]!=-1 } {
            set MonitoringInfo($item) [GiD_Info Geometry $item]
        }
    }

    set ask_line_type 0
    foreach item {nurbsline stline arcline polyline} {
        if { [lsearch $request numlines_$item]!=-1 } {
            set ask_line_type 1
            break
        }
    }
    if { $ask_line_type } {
        objarray foreach num [GiD_Geometry list line] {
            incr count([lindex [GiD_Geometry -v2 get line $num] 0])
        }
        foreach item {nurbsline stline arcline polyline} {
            if { [lsearch $request numlines_$item]!=-1 } {
                if { [info exists count($item)] } {
                    set MonitoringInfo(numlines_$item) $count($item)
                } else {
                    set MonitoringInfo(numlines_$item) 0
                }
            }
        }
    }

    set ask_surface_type 0
    foreach item {nurbssurface plsurface coonsurface meshsurface contactsurface} {
        if { [lsearch $request numsurfaces_$item]!=-1 } {
            set ask_surface_type 1
            break
        }
    }
    if { $ask_surface_type } {
        objarray foreach num [GiD_Geometry list surface] {
            incr count([lindex [GiD_Geometry -v2 get surface $num] 0])
        }
        foreach item {nurbssurface plsurface coonsurface meshsurface contactsurface} {
            if { [lsearch $request numsurfaces_$item]!=-1 } {
                if { [info exists count($item)] } {
                    set MonitoringInfo(numsurfaces_$item) $count($item)
                } else {
                    set MonitoringInfo(numsurfaces_$item) 0
                }
            }
        }
    }

    set ask_volume_type 0
    foreach item {volume contactvolume} {
        if { [lsearch $request numvolumes_$item]!=-1 } {
            set ask_volume_type 1
            break
        }
    }
    if { $ask_volume_type } {
        objarray foreach num [GiD_Geometry list volume] {
            incr count([lindex [GiD_Geometry -v2 get volume $num] 0])
        }
        foreach item {volume contactvolume} {
            if { [lsearch $request numvolumes_$item]!=-1 } {
                if { [info exists count($item)] } {
                    set MonitoringInfo(numvolumes_$item) $count($item)
                } else {
                    set MonitoringInfo(numvolumes_$item) 0
                }
            }
        }
    }


    #surfaces where render meshing has failed
    set item numsurfaces_unrendered
    if { [lsearch $request $item]!=-1 } {
        set MonitoringInfo($item) [GiD_Geometry list -count -unrendered surface]
    }

    #do not force to create render mesh, only ask if exists
    foreach item {render_line_numelements render_surface_numelements} {
        set entity_type [lindex [split $item _] 1]
        if { [lsearch $request $item]!=-1 } {
            set render_num_elements 0
            objarray foreach num [GiD_Geometry list $entity_type] {
                set render_mesh [GiD_Geometry -v2 get $entity_type $num render_mesh]
                if { [llength $render_mesh] } {
                    set element_num_nodes [lindex $render_mesh 1]
                    set num_connectivities [llength [lindex $render_mesh 3]]
                    incr render_num_elements [expr $num_connectivities/$element_num_nodes]
                }
            }
            set MonitoringInfo($item) $render_num_elements
        }
    }

    set item gravity_center_volumes
    if { [lsearch $request $item]!=-1 } {
        set volume_total 0.0
        set gravity_center_volumes [list 0.0 0.0 0.0]
        objarray foreach volume_id [GiD_Geometry list volume] {
            lassign [GiD_Tools geometry mass_properties $volume_id] volume gravity_center inertias
            set volume_abs [expr abs($volume)]
            set volume_total [expr $volume_total+$volume_abs]
            set gravity_center_volumes [MathUtils::VectorSum $gravity_center_volumes [MathUtils::ScalarByVectorProd $volume_abs $gravity_center]]
        }
        if { $volume_total > 0.0 } {
            set gravity_center_volumes [MathUtils::ScalarByVectorProd [expr 1.0/$volume_total] $gravity_center_volumes]
        }
        set MonitoringInfo($item) $gravity_center_volumes
    }

    foreach item {length_lines area_surfaces volume_volumes} cathegory {line surface volume} {
        if { [lsearch $request $item]!=-1 } {
            set MonitoringInfo($item) [lindex [split [lindex [GiD_Info ListMassProperties ${cathegory}s 1:end] end] =] 1]
        }
    }

    ##############
    ##### Mesh Pre
    ##############

    foreach item {numnodes numelements maxnumnodes maxnumelements} {
        if { [lsearch $request $item]!=-1 } {
            set MonitoringInfo($item) [GiD_Info Mesh $item]
        }
    }

    set item mesh_bounding_box
    if { [lsearch $request $item]!=-1 } {
        lassign [GiD_Info bounding_box -pmin_pmax -layers_off_also -mesh] x_min y_min z_min x_max y_max z_max
        set MonitoringInfo($item) [list $x_min $y_min $z_min $x_max $y_max $z_max]
    }

    set item mesh_bounding_box_diagonal
    if { [lsearch $request $item]!=-1 } {
        lassign [GiD_Info bounding_box -pmin_pmax -layers_off_also -mesh] x_min y_min z_min x_max y_max z_max
        set MonitoringInfo($item) [MathUtils::VectorDistance [list $x_min $y_min $z_min] [list $x_max $y_max $z_max]]
    }

    set item mesh_bounding_box_center_distance_to_origin
    if { [lsearch $request $item]!=-1 } {
        lassign [GiD_Info bounding_box -pmin_pmax -layers_off_also -mesh] x_min y_min z_min x_max y_max z_max
        set MonitoringInfo($item) [MathUtils::VectorModulus [MathUtils::VectorMean [list $x_min $y_min $z_min] [list $x_max $y_max $z_max]]]
    }

    set item mesh_normal_mean
    if { [lsearch $request $item]!=-1 } {
        set MonitoringInfo($item) [GidUtils::GetMeanNormal_Mesh]
    }

    foreach item {length_elements area_elements volume_elements} cathegory {line surface volume} {
        if { [lsearch $request $item]!=-1 } {
            set ids [list]
            foreach element_type [GidUtils::GetElementTypes $cathegory] {
                lappend ids {*}[lindex [lindex [GiD_Info Mesh elements $element_type 1 end -array] 0] 1]
            }
            set MonitoringInfo($item) [lindex [split [lindex [GiD_Info ListMassProperties Elements {*}$ids] end] =] 1]
        }
    }

    set item numdistances
    if { [lsearch $request $item]!=-1 } {
        set num_distances 0
        lassign [GiD_Info Mesh EmbeddedDistances] node_ids distances
        if { $node_ids != ""} {
            set num_data_1 [objarray length $node_ids]
            set num_data_2 [objarray length $distances]
            if { $num_data_1 == $num_data_2 } {
                set num_distances $num_data_1
            }
        }
        set MonitoringInfo($item) $num_distances
    }

    foreach item [GidUtils::GetElementTypesLowercase all] {
        if { [lsearch $request numelements_$item]!=-1 } {
            set MonitoringInfo(numelements_$item) [GiD_Info Mesh numelements $item]
        }
    }

    #ElemXX
    foreach quality_criteria {Size MinEdge MaxEdge ShapeQuality MinJacobian MaxMinEdge MinAngle} {
        set quality_criteria_lowercase [string tolower $quality_criteria]
        foreach item [GidUtils::GetElementTypesLowercase all] {
            set key ${quality_criteria_lowercase}_$item
            if { [lsearch $request $key]!=-1 } {
                if { [GiD_Info Mesh numelements $item] } {
                    #ask also for its minimum criteria quality
                    if {$quality_criteria == "MaxMinEdge" } {
                        #the true quality_criteria is MinEdge, and the value is the end, not end-1
                        set MonitoringInfo($key) [lindex [GiD_Info MeshQuality ElemMinEdge $item 0 0 1] end]
                    } elseif {$quality_criteria == "MaxEdge" } {
                        set MonitoringInfo($key) [lindex [GiD_Info MeshQuality Elem${quality_criteria} $item 0 0 1] end]
                    } elseif {$quality_criteria == "MinAngle" } {
                        set MonitoringInfo($key) [lindex [GiD_Info MeshQuality $quality_criteria $item 0 0 1] end-1]
                    } else {
                        set MonitoringInfo($key) [lindex [GiD_Info MeshQuality Elem${quality_criteria} $item 0 0 1] end-1]
                    }
                } else {
                    set MonitoringInfo($key) 0.0
                }
            }
        }
    }

    set item num_reused_meshes
    if { [lsearch $request $item]!=-1 } {
        if { ![info exists ::EACH_FIGURE_WITH_ITS_MESH] } {
            #arbitrary value -1 but not 0 to pass tests in old GiD versions without this feature activated
            set MonitoringInfo($item) -1
        } else {
            set MonitoringInfo($item) [GiD_Set -meshing_parameters_model NumReusedMeshes]
        }
    }

    set item num_mesh_ogl_arrays
    if { [lsearch $request $item]!=-1 } {
        set MonitoringInfo($item) [GiD_Info Project NumMeshOglArrays]
    }

    set item fail_contact_elements_connectivities
    if { [lsearch $request $item]!=-1 } {
        lassign [GiD_Info check mesh contact_elements_connectivities] fail_contact_elements_connectivities fail_message
        set MonitoringInfo($item) $fail_contact_elements_connectivities
    }

    foreach cathegory {line surface volume boundary_layer} {
        set item num_mesh_errors_$cathegory
        if { [lsearch $request $item]!=-1 } {
            set num_errors 0
            if { [info exists ::MeshErrorsPriv($cathegory,num_rows)] } {
                set num_errors $::MeshErrorsPriv($cathegory,num_rows)
            }
            set MonitoringInfo($item) $num_errors
        }
    }

    ##############
    ##### Layers
    ##############

    set item numlayers
    if { [lsearch $request $item]!=-1 } {
        set MonitoringInfo($item) [llength [GiD_Layers list]]
    }

    foreach over {points lines surfaces volumes nodes elements faces} {
        set item entities_layers_num$over
        if { [lsearch $request $item]!=-1 } {
            set n 0
            foreach layer [GiD_Layers list] {
                incr n [GiD_EntitiesLayers get $layer $over -count]
            }
            set MonitoringInfo($item) $n
        }
    }

    set item last_layer_bounding_box_geometry
    if { [lsearch $request $item]!=-1 } {
        set layer_name [lindex [GiD_Layers list] end]
        set bbox [GiD_Layers get bounding_box $layer_name geometry]
        set MonitoringInfo($item) $bbox
    }

    ##############
    ##### Groups
    ##############

    set item numgroups
    if { [lsearch $request $item]!=-1 } {
        set MonitoringInfo($item) [llength [GiD_Groups list]]
    }

    foreach over {points lines surfaces volumes nodes elements faces} {
        set item entities_groups_num$over
        if { [lsearch $request $item]!=-1 } {
            set n 0
            foreach group [GiD_Groups list] {
                incr n [GiD_EntitiesGroups get $group $over -count]
            }
            set MonitoringInfo($item) $n
        }
    }

    foreach over {points lines surfaces nodes} {
        set item min_max_higherentities_$over
        if { [lsearch $request $item]!=-1 } {
            set MonitoringInfo($item) [GidUtils::GetMinMaxHigherEntities [string range $over 0 end-1]]
        }
    }

    ########################
    #Problemtype information
    ########################

    ################
    ##### Conditions
    ################

    foreach type {point line surface volume layer group} {
        set over over_$type
        set item numconditions_${type}s
        if { [lsearch $request $item]!=-1 } {
            set MonitoringInfo($item) [llength [GiD_Info conditions $over]]
        }
    }

    foreach type {point line surface volume layer group} {
        set over over_$type
        set item entities_conditions_num${type}s
        if { [lsearch $request $item]!=-1 } {
            set n 0
            foreach condition [GiD_Info conditions $over] {
                incr n [GiD_Info conditions $condition geometry -count]
            }
            set MonitoringInfo($item) $n
        }
    }

    foreach mesh_type {node element face} over_mesh {nodes body face} {
        set item entities_conditions_num${mesh_type}s
        if { [lsearch $request $item]!=-1 } {
            set n 0
            foreach type {point line surface volume layer group} {
                set over over_$type
                foreach condition [GiD_Info conditions $over] {
                    if { [lindex [lindex [GiD_Info conditions $condition condmeshtype] 0] 1] == $over_mesh } {
                        incr n [GiD_Info conditions $condition mesh -count]
                    }
                }
            }
            set MonitoringInfo($item) $n
        }
    }

    ################
    ##### Materials
    ################

    set item nummaterials
    if { [lsearch $request $item]!=-1 } {
        set MonitoringInfo($item) [llength [GiD_Info materials]]
    }

    foreach over {points lines surfaces volumes} {
        set item entities_materials_num$over
        if { [lsearch $request $item]!=-1 } {
            set over_singular [string range $over 0 end-1]
            set n_total [GiD_Geometry list -count $over_singular]
            set n_without_material [GiD_Geometry list -count -material 0 $over_singular]
            set n [expr $n_total-$n_without_material]
            set MonitoringInfo($item) $n
        }
    }
    foreach over {nodes elements} {
        set item entities_materials_num$over
        if { [lsearch $request $item]!=-1 } {
            set over_singular [string range $over 0 end-1]
            set n_total [GiD_Mesh list -count $over_singular]
            set n_without_material [GiD_Mesh list -count -material 0 $over_singular]
            set n [expr $n_total-$n_without_material]
            set MonitoringInfo($item) $n
        }
    }

    #######################
    #postproces information
    #######################

    ################
    ##### Mesh post
    ################

    foreach item {postnumnodes postnumelements postmaxnumnodes postmaxnumelements} {
        if { [lsearch $request $item]!=-1 } {
            set item_part [string range $item 4 end] ;#remove 'post' prefix
            set MonitoringInfo($item) [GiD_Info mesh -post $item_part]
        }
    }

    foreach item [GidUtils::GetElementTypesLowercase all] {
        if { [lsearch $request postnumelements_$item]!=-1 } {
            set MonitoringInfo(postnumelements_$item) [GiD_Info Mesh -post numelements $item]
        }
    }

    set item postmesh_bounding_box
    if { [lsearch $request $item]!=-1 } {
        lassign [GiD_Info bounding_box -pmin_pmax -layers_off_also -post] x_min y_min z_min x_max y_max z_max
        set MonitoringInfo($item) [list $x_min $y_min $z_min $x_max $y_max $z_max]
    }

    set item postmesh_bounding_box_center_distance_to_origin
    if { [lsearch $request $item]!=-1 } {
        lassign [GiD_Info bounding_box -pmin_pmax -layers_off_also -post] x_min y_min z_min x_max y_max z_max
        set MonitoringInfo($item) [MathUtils::VectorModulus [MathUtils::VectorMean [list $x_min $y_min $z_min] [list $x_max $y_max $z_max]]]
    }

    set item postmesh_bounding_box_diagonal
    if { [lsearch $request $item]!=-1 } {
        lassign [GiD_Info bounding_box -pmin_pmax -layers_off_also -post] x_min y_min z_min x_max y_max z_max
        set MonitoringInfo($item) [MathUtils::VectorDistance [list $x_min $y_min $z_min] [list $x_max $y_max $z_max]]
    }

    ################
    ##### Sets
    ################

    foreach type {VolumeSets SurfaceSets CutSets} {
        set item num[ string tolower $type]
        if { [ lsearch $request $item] != -1 } {
            set MonitoringInfo($item) [ llength [GiD_Info postprocess get all_$type]]
        }
    }

    set item last_set_bounding_box
    if { [lsearch $request $item]!=-1 } {
        set set_name [lindex [GiD_Sets_list] end]
        set bbox [GiD_Sets get bounding_box $set_name]
        set MonitoringInfo($item) $bbox
    }

    ################
    ##### Results
    ################

    foreach item {numanalysis numresultscurrentanalysis numnodalresultscurrentanalysis \
        numgaussresultscurrentanalysis numrangestables numgausspoints \
            numstepscurrentanalysis stepcurrent numgraphs numgraphsets numsets numcuts } {
            if { [lsearch $request $item]!=-1 } {
                if { $item == "numanalysis" } {
                    set MonitoringInfo($item) [llength [GiD_Info postprocess get all_analysis]]
            } elseif { $item == "numresultscurrentanalysis" } {
                set MonitoringInfo($item) [llength [GiD_Info postprocess get cur_results_list Contour_Fill]]
            } elseif { $item == "numnodalresultscurrentanalysis" } {
                set num_results_total [llength [GiD_Info postprocess get cur_results_list Contour_Fill]]
                set num_results_gauss [llength [GiD_Info postprocess get cur_results_list Smooth_Contour_Fill]]
                set MonitoringInfo($item) [expr $num_results_total-$num_results_gauss]
            } elseif { $item == "numgaussresultscurrentanalysis" } {
                set MonitoringInfo($item) [llength [GiD_Info postprocess get cur_results_list Smooth_Contour_Fill]]
            } elseif { $item == "numrangestables" } {
                set MonitoringInfo($item) [llength [GiD_Result result_ranges_table names]]
            } elseif { $item == "numgausspoints" } {
                set MonitoringInfo($item) [llength [GiD_Result gauss_point names]]
            } elseif { $item == "numstepscurrentanalysis" } {
                set MonitoringInfo($item) [llength [GiD_Info postprocess get all_steps [GiD_Info postprocess get cur_analysis]]]
            } elseif { $item == "stepcurrent" } {
                set MonitoringInfo($item) [GiD_Info postprocess get cur_step]
            } elseif { $item == "numgraphs" } {
                set MonitoringInfo($item) [llength [GiD_Graph list]]
            } elseif { $item == "numgraphsets" } {
                set MonitoringInfo($item) [llength [GiD_GraphSet list]]
            } elseif { $item == "numsets" } {
                #set MonitoringInfo($item) [llength [GiD_Sets list]]
                set MonitoringInfo($item) [expr [llength [GiD_Info postprocess get all_surfacesets]]+[llength [GiD_Info postprocess get all_volumesets]]]
            } elseif { $item == "numcuts" } {
                set MonitoringInfo($item)  [llength [GiD_Info postprocess get all_cutsets]]
            }
        }
    }

    ################
    ##### Other
    ################

    ##############
    ##### Redraw
    ##############

    set item redraw_time ;#microseconds per iteration
    if { [lsearch $request $item]!=-1 } {
        update
        set MonitoringInfo($item) [lindex [time GiD_Redraw 10] 0]
    }

    set item redraw_fps ; # frames per second
    if { [lsearch $request $item]!=-1 } {
        update
        set total_t [lindex [time GiD_Redraw 10] 0]
        set fps [ expr round( 1.0 / ( $total_t * 1e-9)) / 1000.0]
        set MonitoringInfo($item) $fps
    }

    set item redraw_fps_100 ; # frames per second
    if { [lsearch $request $item]!=-1 } {
        update
        set total_t [lindex [time GiD_Redraw 100] 0]
        set fps [ expr round( 1.0 / ( $total_t * 1e-9)) / 1000.0]
        set MonitoringInfo($item) $fps
    }

    ##############
    ##### Color
    ##############

    set item color_mean
    if { [lsearch $request $item]!=-1 } {
        set MonitoringInfo($item) [GiD_Thumbnail get_statistics_mean]
    }

    ####################
    ##### Generic checks
    ####################

    ### generic expr procedures  ###
    set cont 0
    foreach procedure $expr_procedures {
        incr cont
        if { [catch { set value [expr $procedure] } err] } {
            #set MonitoringInfo(expression-$cont) $err
            #the value is expected to be boolean, set as false to not pass the test
            set MonitoringInfo(expression-$cont) 0
        } else {
            set MonitoringInfo(expression-$cont) $value
        }
    }

    ### generic tcl global variables ###
    foreach item $request {
        if { [string range $item 0 11] == "tclvariable_" } {
            set name [string range $item 12 end]
            if { [info exists ::$name] } {
                set MonitoringInfo(tclvariable_$name) [set ::$name]
            }
        }
    }


    set res ""
    foreach item [array names MonitoringInfo] {
        lappend res $item $MonitoringInfo($item)
    }
    return $res
}

proc gid_monitoring::saveinfo { filename } {
    set fp [open $filename w]
    puts -nonewline $fp [gid_monitoring::getinfo]
    close $fp
}
