
package require Plotchart
package require no_gid

package require parse_args

package provide gid_graph 1.3

namespace eval GidGraph {

}

proc GidGraph::GetToplevelNewName {} {
    set w ""
    if { [winfo exists .gid] } {
        set base_name .gid.curveplot
    } else {
        set base_name .curveplot
    }
    set i 1
    while { 1 } {
        set w ${base_name}-$i
        if { ![winfo exists $w] } {
            break
        }
        incr i
    }
    return $w
}

proc GidGraph::CheckPoints { points type } {
    set nrows [llength $points]
    if { $nrows < 2 } {
        error [_ "Insufficient number of data"]
    }
    set ncols [llength [lindex $points 0]]
    if { $ncols < 2 } {
        error [_ "Insufficient number of data"]
    }

    #for bar first value x could be a string, not a number
    foreach point $points {
        set serie -1
        foreach value $point {
            if { $type != "bar" || $serie>=0} {
                if { ![string is double $value] } {
                    error [concat [_ "Graph can't be drawn"]. [_ "value '%s' is not a number" $value]]
                }
            }
            incr serie
        }
    }
    return 0
}

#points: list of points, each item is a point with real values like {x1 y_series_1 .. y_series_n}
#series options: see documentation of tklib plotchar package to see valid options for each type of grahp
#type "xy" "bar"

proc GidGraph::Window2 { args } {
    parse_args::parse_args $args {
        -points {}
        -title  {}
        -x_text {-default "x"}
        -y_text {-default "y"}
        -type {-default "xy"}
        -logx {-boolean}
        -logy {-boolean}
        -bar_horizontal {-boolean}
        -bar_stacked {-boolean}
        -margin {-default "50 50 50 50"}
        -background {-default white}
        -series_options {-default {}}
    }
    GidGraph::CheckPoints $points $type
    set w [GidGraph::GetToplevelNewName]
    if { [info procs InitWindow] != "" } {
        InitWindow2 $w -title $title -geometryvariable PreGraphWindowGeom
    } else {
        wm withdraw .
        toplevel $w
        wm title $w $title
    }
    set f [ttk::frame $w.f]
    grid $f -sticky nsew
    grid columnconfigure $f 0 -weight 1
    grid rowconfigure $f 0 -weight 1

    #lower buttons
    ttk::frame $w.frmButtons -style BottomFrame.TFrame
    ttk::button $w.frmButtons.btnclose -text [_ "Close"] -command [list destroy $w] -underline 0 -style BottomFrame.TButton

    grid $w.frmButtons -sticky nsew
    grid $w.frmButtons.btnclose -padx 3 -pady 3
    grid columnconfigure $w.frmButtons 0 -weight 1
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1

    wm minsize $w 300 150

    bind $w <Alt-c> [list $w.frmButtons.btnclose invoke]
    bind $w <Escape> [list $w.frmButtons.btnclose invoke]
    set xyplot [GidGraph::InFrame2 {*}$args $f]
    focus -force $w.frmButtons.btnclose
    return $xyplot
}

#to allow create the graph inside an existent frame f
proc GidGraph::InFrame2 { args } {
    parse_args::parse_args $args {
        -points {}
        -title  {}
        -x_text {-default "x"}
        -y_text {-default "y"}
        -type {-default "xy"}
        -logx {-boolean}
        -logy {-boolean}
        -bar_horizontal {-boolean}
        -bar_stacked {-boolean}
        -margin {-default "50 50 50 50"}
        -background {-default white}
        -series_options {-default {}}
        f {-required}
    }    
    
    #GidGraph::CheckPoints $points

    #::Plotchart::plotstyle load default    
    if { [winfo exists $f.c] } {
        ::Plotchart::ClearPlot $f.c
        destroy $f.c
    }
    set c [canvas $f.c]
    $c configure -background $background

    grid $c -sticky nsew
    set command_plot [list GidGraph::Plot {*}[lrange $args 0 end-1] $c]
    set xyplot [eval $command_plot]

    bind $f <Destroy> [list GidGraph::OnDestroy %W $f $c]
    bind $f <Configure> $command_plot
    #bind $f <<Copy>> [list GidGraph::ClipboardCopyImage %W]
    #$c bind data <Any-Enter> [list GidGraph::EnterPoint $c]
    #$c bind data <Any-Leave> [list GidGraph::LeavePoint $c]
    if { $::tcl_platform(os) == "Darwin" } {
        set right_button 2
    } else {
        set right_button 3
    }
    bind $c <ButtonPress-$right_button> [list +GidGraph::ShowMenu %W %x %y $points $title $x_text $y_text]

    return $xyplot
}


proc GidGraph::OnDestroy { W w c } {
    if { $W == $w } {
        ::Plotchart::ClearPlot $c
    }
}

#type "xy" "bar"
#xy_config_list: list of 2 items xconfig yconfig, each one a list of -option value of valid options (format ticklength ticklines scale),
#                e.g.  {{-format %g -ticklength 3} {}}
proc GidGraph::Plot { args } {
    parse_args::parse_args $args {
        -points {}
        -title  {}
        -x_text {-default "x"}
        -y_text {-default "y"}
        -type {-default "xy"}
        -logx {-boolean}
        -logy {-boolean}
        -bar_horizontal {-boolean}
        -bar_stacked {-boolean}
        -margin {-default {50 50 50 50}}
        -background {-default white}
        -series_options {-default {}}
        -xy_config_list {-default {}}
        c {-required}
    }

    set type_plot ""
    if { $type == "xy" } {
        set type_plot "xyplot"
        if { $logx } {
            if { $logy } {
                set type_plot "logxlogyplot"
            } else {
                set type_plot "logxyplot"
            }
        } elseif { $logy } {
            set type_plot "logy"
        }
    } elseif { $type == "bar" } {
        if { $bar_horizontal } {
            set type_plot "horizbars"
        } else {
            set type_plot "vertbars" 
        }
    } elseif { $type == "pie" } {
        set type_plot "piechart"
    } elseif { $type == "polar" } {
        set type_plot "polarplot"        
    } else {
        #default
        set type_plot "xyplot"
    }
        
    update
    if { ![winfo exists $c] } {
        return ""
    }
    if { [winfo width $c] <= 160 || [winfo height $c] <= 110 } {
        #window too small
        return ""
    }
    if { $points == "" || [llength $points]<=1 || ![winfo exists $c] } {
        return ""
    }

    ::Plotchart::ClearPlot $c

    foreach margin {top bottom left right} size $margin {
        ::Plotchart::plotconfig $type_plot margin $margin $size
    }

    set xmin 1e20
    set xmax -1e20
    set ymin 1e20
    set ymax -1e20
    foreach pt $points {
        set x [lindex $pt 0]
        if { $xmax < $x } {
            set xmax $x
        }
        if { $xmin > $x } {
            set xmin $x
        }
        if { $bar_stacked } {
            set y_sum 0.0
            foreach y [lrange $pt 1 end] {
                set y_sum [expr $y_sum+$y]
            }
            if { $ymax < $y_sum } {                    
                set ymax $y_sum
            }
            if { $ymin > $y_sum } {
                set ymin $y_sum
            }            
        } else {
            foreach y [lrange $pt 1 end] {
                if { $ymax < $y } {
                    set ymax $y
                }
                if { $ymin > $y } {
                    set ymin $y
                }
            }
        }
    }

    if { $xmax <= $xmin } {
        #e.g all points with same x, illegal but...
        set xmin [expr $xmin-1.0]
        set xmax [expr $xmax+1.0]
    }
    if { $ymax <= $ymin } {
        #e.g all points with same y
        set ymin [expr $ymin-1.0]
        set ymax [expr $ymax+1.0]
    }

    if { 0 } {
        set dy [expr {($ymax-$ymin)*0.01}]
        set ymin [expr $ymin-$dy]
        set ymax [expr $ymax+$dy] ;#else horizontal max line is not shown
    }

    if { [lsearch {xyplot logxyplot xlogyplot logxlogyplot} $type_plot] !=-1} {
        set xaxis [Plotchart::determineScale $xmin $xmax 0]
    }
    if { [lsearch {xyplot logxyplot xlogyplot logxlogyplot vertbars horizbars polarplot} $type_plot] !=-1} {
        set yaxis [Plotchart::determineScale $ymin $ymax 0]
    }

    $c delete all
    
    if { [lsearch {xyplot logxyplot xlogyplot logxlogyplot vertbars horizbars} $type_plot] !=-1} {
        ::Plotchart::plotconfig $type_plot bottomaxis format "%g"
        ::Plotchart::plotconfig $type_plot leftaxis format "%g"
    }
    

    if { $type_plot == "logxyplot" } {
        set xaxis [list $xmin $xmax 0]
        set xyplot [::Plotchart::createLogXYPlot $c $xaxis $yaxis]
    } elseif { $type_plot == "xlogyplot"} {
        #set yaxis [list $ymin $ymax 0]
        set yaxis [list $ymin $ymax 1] ;#set ystep to 1 instead 0, really is not ignored for a small bug of Plotchart package
        set xyplot [::Plotchart::createXLogYPlot $c $xaxis $yaxis]
    } elseif { $type_plot == "logxlogyplot" } {
        set xaxis [list $xmin $xmax 0]
        set yaxis [list $ymin $ymax 0]
        set xyplot [::Plotchart::createLogXLogYPlot $c $xaxis $yaxis]
    } elseif { $type_plot == "vertbars" || $type_plot == "horizbars" } {
        lset yaxis 0 0.0
        set xlabels [list]
        foreach pt $points {
            lappend xlabels [lindex $pt 0]
        }
        set nseries [expr {[llength [lindex $points 0]]-1}]
        if { $type_plot == "vertbars" } {
            if { $bar_stacked } {
                set xyplot [::Plotchart::createBarchart $c $xlabels $yaxis stacked]
            } else {
                set xyplot [::Plotchart::createBarchart $c $xlabels $yaxis $nseries]
            }
        } elseif { $type_plot == "horizbars" } {
            if { $bar_stacked } {
                set xyplot [::Plotchart::createHorizontalBarchart $c $yaxis $xlabels stacked]
            } else {
                set xyplot [::Plotchart::createHorizontalBarchart $c $yaxis $xlabels $nseries]
            }
        }
    } elseif { $type_plot == "xyplot" } {
        set xyplot [::Plotchart::createXYPlot $c $xaxis $yaxis]
    } elseif { $type_plot == "piechart" } {
        set xyplot [::Plotchart::createPiechart $c]
    } elseif { $type_plot == "polarplot" } {        
        set xyplot [::Plotchart::createPolarplot $c [lrange $yaxis 1 2]]
    }

    #set margin 40
    #set pxmin $margin
    #set pymin $margin
    #set pxmax [expr {[$c cget -width]-$margin}] ;#not change width height !!!
    #set pymax [expr {[$c cget -height]-$margin}]
    #::Plotchart::viewPort $c $pxmin $pymin $pxmax $pymax

    ::Plotchart::plotconfig $type_plot background outercolor $background
    ::Plotchart::plotconfig $type_plot background innercolor $background
    
    ::Plotchart::plotconfig $type_plot title background $background
    if { [lsearch {xyplot logxyplot xlogyplot logxlogyplot vertbars horizbars polarplot} $type_plot] !=-1} {
        $xyplot background plot gray95
        $xyplot background axes $background
    }
    
    
    $xyplot title $title center
    if { [lsearch {xyplot logxyplot xlogyplot logxlogyplot vertbars horizbars} $type_plot] !=-1} {
        $xyplot xtext $x_text
    }

    if { $xy_config_list != "" } {
        lassign $xy_config_list xconfig yconfig
        $xyplot xconfig {*}$xconfig
        $xyplot yconfig {*}$yconfig

    }
    if { [lsearch {xyplot logxyplot xlogyplot logxlogyplot horizbars} $type_plot] !=-1} {
        $xyplot xticklines ;# green
    }
    if { [lsearch {xyplot logxyplot xlogyplot logxlogyplot vertbars} $type_plot] !=-1} {
        $xyplot yticklines ;# green
    }
    
    
    #$xyplot grid {{1 2 3} {1 2 3}}  {{20 20 20} {21 21 21}}
    #type: line, symbol, or both.
    #symbol: plus, cross, circle, up, down, dot, upfilled or downfilled
    set colors {red green blue yellow magenta cyan orange gray black}
    set ncolors [llength $colors]
    set nseries [expr {[llength [lindex $points 0]]-1}]

    #$xyplot xconfig -scale $xaxis
    #$xyplot yconfig -scale $yaxis

    #add specified series options if any
    set serie 0
    foreach serie_options $series_options {
        $xyplot dataconfig $serie {*}$serie_options
        incr serie
    }

    if { [lsearch {xyplot logxyplot xlogyplot logxlogyplot polarplot} $type_plot] !=-1} {
        for {set serie 0} {$serie<$nseries} {incr serie} {
            $xyplot dataconfig $serie -color [lindex $colors [expr {$serie%$ncolors}]] -type both -symbol plus
            $xyplot legend $serie [lindex $y_text $serie]
        }
    }

    if { $type_plot == "vertbars" || $type_plot == "horizbars" } {
        for { set serie 0 } { $serie<$nseries } { incr serie } {
            set ys [list]
            foreach pt $points {
                lappend ys [lindex $pt $serie+1]
            }
            set color [lindex $colors [expr {$serie%$ncolors}]]
            $xyplot plot $serie $ys $color
        }
    } elseif { $type_plot == "piechart" } {
        if { $nseries==1 } {
            $xyplot colours {*}$colors
            for { set serie 0 } { $serie<$nseries } { incr serie } {
                set data [list]
                set ys [list]
                foreach pt $points {
                    lappend data [lindex $pt 0] [lindex $pt $serie+1]
                }
                $xyplot plot $data
            }
        }
        $xyplot explode auto
    } elseif { $type_plot == "polarplot" } {
        foreach pt $points {
            set angle_degrees [lindex $pt 0]
            set serie 0
            foreach radius [lrange $pt 1 end] {
                $xyplot plot $serie $radius $angle_degrees
                incr serie
            }
        }
        
    } else {
        foreach pt $points {
            set x [lindex $pt 0]
            set serie 0
            foreach y [lrange $pt 1 end] {
                $xyplot plot $serie $x $y
                $xyplot bindlast $serie <Enter> [list GidGraph::ShowAnnotation $xyplot %W]
                incr serie
            }
        }
    }

    if { $type_plot == "vertbars" || $type_plot == "horizbars" } {
        for {set serie 0} {$serie<$nseries} {incr serie} {
            $xyplot legend $serie [lindex $y_text $serie]
        }
    }


    return $xyplot
}

proc GidGraph::ShowAnnotation {xcoord ycoord plot w} {
    GidGraph::RemoveAnnotation $w
    $plot balloon $xcoord $ycoord [format "%g, %g" $xcoord $ycoord] north
    after 2000 [list GidGraph::RemoveAnnotation $w]
}

proc GidGraph::RemoveAnnotation {w} {
    # Use the tags to remove all annotations
    if { [winfo exists $w] } {
        $w delete BalloonText
        $w delete BalloonFrame
    }
}

proc GidGraph::ClipboardCopyImage { c } {
    package require gid_cross_platform
    set img [gid_cross_platform::capture_window $c]
    #save image to disk
    #$image write $filename -format GIF
    gid_cross_platform::image_to_clipboard $img
}

proc GidGraph:SaveImageSVG { c {filename ""} } {
    package require can2svg
    if { $filename == "" } {
        package require gid_cross_platform
        set filename [gid_cross_platform::get_unused_tmp_filename graph .svg]
    }    
    #save image to disk
    can2svg::canvas2file $c $filename
    if { 0 } {
        set img [gid_cross_platform::capture_window $w]
        #save image to disk
        #$image write $filename -format SVG
        gid_cross_platform::image_to_clipboard $img
    }
    return $filename
}

proc GidGraph::ClipboardCopyData { points title x_text y_text } {
    set data_list [list $title]
    lappend data_list [join [list $x_text {*}$y_text] \t]
    foreach pt $points {
        set x [lindex $pt 0]
        set ys [lrange $pt 1 end]
        set line_list [list $x {*}$ys]
        lappend data_list [join $line_list \t]
    }
    clipboard clear
    clipboard append [join $data_list \n]
}

proc GidGraph::ShowMenu { w x y points title x_text y_text } {
    set m .menu_contextual
    if { [winfo exists $m] } {
        destroy $m
    }
    menu $m -tearoff no
    $m add command -label [_ "Copy data"] -command [list GidGraph::ClipboardCopyData $points $title $x_text $y_text]
    $m add command -label [_ "Copy image"] -command [list after 1000 GidGraph::ClipboardCopyImage $w]
    set xx [expr [winfo rootx $w]+$x+50]
    set yy [expr [winfo rooty $w]+$y]
    tk_popup $m $xx $yy 0
}

proc GidGraph::Test { } {
    set points [list {0.0 25.0 189.0} {2.5 20 200.0} {4.3 5.0 250.0} {6.2 6.7 220.3}]    
    GidGraph::Window2 -points $points -title [_ "Test graph"] -x_text [_ "time"] -y_text [list [_ "temperature"] [_ "pressure"]] -type "xy"
}
