
package require BWidget

namespace eval GiDSelectColor {
    Widget::declare GiDSelectColor [list \
                                        [list -title    String     [_ "Select a color"] 0] \
                                        [list -parent   String     "" 0] \
                                        [list -color    TkResource "" 0 [list label -background]]]

    proc ::GiDSelectColor { path args } {
        return [GiDSelectColor::dialog $path {*}$args]
    }

    variable _baseColors {
        \#0000ff \#00ff00 \#00ffff \#ff0000 \#ff00ff \#ffff00
        \#000099 \#009900 \#009999 \#990000 \#990099 \#999900
        \#000000 \#333333 \#666666 \#999999 \#cccccc \#ffffff
    }

    variable _userColors {
        \#ffffff \#ffffff \#ffffff \#ffffff \#ffffff \#ffffff
        \#ffffff \#ffffff \#ffffff \#ffffff \#ffffff
    }

    if {![string compare $::tcl_platform(platform) "unix"]} {
        set useTkDialogue 0
    } else {
        set useTkDialogue 1
    }

    variable _selectype
    variable _selection
    variable _wcolor
    variable _image
    variable _hsv
    variable _entry_values
    variable _using_themes     -1
}


proc GiDSelectColor::menu {path placement args} {
    variable _baseColors
    variable _userColors
    variable _wcolor
    variable _selectype
    variable _selection

    Widget::init GiDSelectColor $path $args
    set top [::menu $path]
    wm withdraw  $top
    wm transient $top [winfo toplevel [winfo parent $top]]

    set frame [ttk::frame $top.frame -relief flat -borderwidth 1]

    set col    0
    set row    0
    set count  0
    set colors [concat $_baseColors $_userColors]
    foreach color $colors {
        set f [ttk::frame $frame.c$count \
                   -relief solid -borderwidth 1 \
                   -width 16 -height 16]
        bind $f <ButtonPress-1> "set GiDSelectColor::_selection $count"
        bind $f <Enter>         {focus %W}
        grid $f -column $col -row $row -padx 1 -pady 1
        bindtags $f $f
        incr count
        if {[incr col] == 6 } {
            set  col 0
            incr row
        }
    }
    set f [ttk::label $frame.c$count -image [Bitmap::get palette]]
    grid $f -column $col -row $row -padx 1 -pady 1
    bind $f <ButtonPress-1> "set GiDSelectColor::_selection $count"
    bind $f <Enter>         {focus %W}
    pack $frame

    bind $frame <ButtonPress-1> {set GiDSelectColor::_selection -1}
    bind $frame <FocusOut>      {set GiDSelectColor::_selection -2}
    eval BWidget::place $top 0 0 $placement

    wm deiconify $top
    focus -force $frame
    raise $top
    BWidget::grab set $frame

    tkwait variable GiDSelectColor::_selection
    update
    BWidget::grab release $frame
    destroy $top
    update
    Widget::destroy $top
    if {$_selection == $count} {
        return [eval dialog $path $args]
    } else {
        return [lindex $colors $_selection]
    }
}

proc GiDSelectColor::_ActualizeEntries { bg} {
    variable _widget
    variable _entry_values

    if { [ info exists _entry_values(actualizing)] && $_entry_values(actualizing)} {
        return
    }

    if { [ scan $bg #%4x%4x%4x r g b] != 3} {
        if { [ scan $bg #%2x%2x%2x r g b] != 3} {
            return
        }
    } else {
        # van de 0 a 65535
        set r [ expr int( 0.5 + $r * 255.0 / 65535.0)]
        set g [ expr int( 0.5 + $g * 255.0 / 65535.0)]
        set b [ expr int( 0.5 + $b * 255.0 / 65535.0)]
    }

    set _entry_values(actualizing) 1

    # set frame $_widget(fcolor)
    #
    #$frame.roundentries.entries.er delete 0 end
    #$frame.roundentries.entries.er insert end $r
    #$frame.roundentries.entries.eg delete 0 end
    #$frame.roundentries.entries.eg insert end $g
    #$frame.roundentries.entries.eb delete 0 end
    #$frame.roundentries.entries.eb insert end $b
    #
    #$frame.roundentries.entries.ehr delete 0 end
    #$frame.roundentries.entries.ehr insert end [ format %2x $r]
    #$frame.roundentries.entries.ehg delete 0 end
    #$frame.roundentries.entries.ehg insert end [ format %2x $g]
    #$frame.roundentries.entries.ehb delete 0 end
    #$frame.roundentries.entries.ehb insert end [ format %2x $b]

    set _entry_values(red,dec) $r
    set _entry_values(green,dec) $g
    set _entry_values(blue,dec) $b
    set _entry_values(red,hex) [ format %2x $r]
    set _entry_values(green,hex) [ format %2x $g]
    set _entry_values(blue,hex) [ format %2x $b]

    set _entry_values(actualizing) 0
}

proc GiDSelectColor::_CambioEntries { name1 name2 op} {
    variable _widget
    variable _entry_values
    upvar ${name1}($name2) entry_val

    if { "$entry_val" == ""} {
        return
    }

    set frame $_widget(fcolor)

    set r $_entry_values(red,dec)
    set g $_entry_values(green,dec)
    set b $_entry_values(blue,dec)

    if { $r < 0 || $r > 255 || $g < 0 || $g > 255 || $b < 0 || $b > 255 } {
        set rgb [ $frame.color cget -background]
        _ActualizeEntries $rgb
        bell
        return
    }

    if { [ info exists _entry_values(actualizing)] && $_entry_values(actualizing)} {
        return
    }

    set _entry_values(actualizing) 1

    set _entry_values(red,hex) [ format %2x $r]
    set _entry_values(green,hex) [ format %2x $g]
    set _entry_values(blue,hex) [ format %2x $b]

    set bg [ format #%02x%02x%02x $r $g $b]

    set _hsv [eval rgbToHsv [winfo rgb $frame $bg]]
    _set_hue_sat [lindex $_hsv 0] [lindex $_hsv 1]
    _set_value   [lindex $_hsv 2]
    $frame.color configure -background $bg

    set _entry_values(actualizing) 0
}

proc GiDSelectColor::_CambioEntriesHex { name1 name2 op} {
    variable _entry_values
    variable _widget
    upvar ${name1}($name2) entry_val

    if { "$entry_val" == ""} {
        return
    }

    if { [ info exists _entry_values(actualizing)] && $_entry_values(actualizing)} {
        return
    }

    set frame $_widget(fcolor)

    set rgb 0
    set err [ catch {
        if { [ scan $_entry_values(red,hex) %x r] == 1} { incr rgb}
        if { [ scan $_entry_values(green,hex) %x g] == 1} { incr rgb}
        if { [ scan $_entry_values(blue,hex) %x b] == 1} { incr rgb}
    }
    ]

    if { $err || $rgb != 3 || $r<0 || $r>255 || $g<0 || $g>255 || $b<0 || $b>255 } {
        set rgb [ $frame.color cget -background]
        _ActualizeEntries $rgb
        bell
        return
    }

    set _entry_values(actualizing) 1

    set _entry_values(red,dec) $r
    set _entry_values(green,dec) $g
    set _entry_values(blue,dec) $b

    set bg [ format #%02x%02x%02x $r $g $b]

    set _hsv [eval rgbToHsv [winfo rgb $frame $bg]]
    _set_hue_sat [lindex $_hsv 0] [lindex $_hsv 1]
    _set_value   [lindex $_hsv 2]
    $frame.color configure -background $bg

    set _entry_values(actualizing) 0
}

proc GiDSelectColor::GetUsingThemes { } {
    variable _using_themes

    if { $_using_themes == -1} {
        set _using_themes 0
        set check_theme ""
        catch {
            set check_theme [ gid_themes::GetCurrentTheme ]
        }
        if { $check_theme != "" } {
            set _using_themes 1
        }
    }
    return $_using_themes
}

proc GiDSelectColor::dialog {path args} {
    variable _baseColors
    variable _userColors
    variable _widget
    variable _selection
    variable _image
    variable _hsv
    variable _entry_values
    variable _transparency

    set _transparency 0

    # if we are using gid themes, enable experimental use of themes inside bwidget:
    Widget::theme [ GiDSelectColor::GetUsingThemes]

        set newargs ""
        foreach {item value} $args {
            if { $item == "-color" } {
                if { [string length $value] == 9 } {
                    #last two characters are alpha channel, separe from color
                    scan [string range $value 7 8] %2x alpha
                    set _entry_values(alpha,dec) $alpha
                    set value [string range $value 0 6]
                    set _transparency 1
                }
            }
            lappend newargs $item $value
        }
        set args $newargs

    Widget::init GiDSelectColor $path:GiDSelectColor $args
    set top   [Dialog::create $path \
                   -title  [Widget::cget $path:GiDSelectColor -title]  \
                   -parent [Widget::cget $path:GiDSelectColor -parent] \
                   -separator 0 -default 0 -cancel 1]
    wm resizable $top 0 0
    set dlgf  [$top getframe]
    set fg    [ttk::frame $dlgf.fg]
    set desc  [list \
                   base _baseColors [_ "Base colors"] \
                   user _userColors [_ "User colors"]]
    set count 0
    foreach {type varcol defTitle} $desc {
        set col   0
        set lin   0
        set title [lindex [BWidget::getname "${type}Colors"] 0]
        if {![string length $title]} {
            set title $defTitle
        }
        set titf  [ttk::labelframe $fg.$type -text $title]        
        foreach color [set $varcol] {
            set fround [ttk::frame $fg.round$count -relief sunken -borderwidth 1]
            set fcolor [tk::frame $fg.color$count -width 16 -height 12 \
                            -relief flat -borderwidth 0 \
                            -highlightthickness 0 -background $color]                            
            pack $fcolor -in $fround
            grid $fround -in $titf -row $lin -column $col -padx 1 -pady 1

            bind $fround <ButtonPress-1> "GiDSelectColor::_select_rgb $count"
            bind $fcolor <ButtonPress-1> "GiDSelectColor::_select_rgb $count"

            incr count
            if {[incr col] == 6} {
                incr lin
                set  col 0
            }
        }
        pack $titf -anchor w -pady 2
    }

    #set froundentries [ ttk::frame $fg.roundentries \
#                    -relief sunken -borderwidth 0]
 #   set fround [ttk::frame $froundentries.round \
#                    -relief sunken -borderwidth 1]
 #   set fentries [ ttk::frame $froundentries.entries \
#                    -relief sunken -borderwidth 0]
    set froundentries [ ttk::frame $fg.roundentries \
                    -relief flat -borderwidth 0]
    set fround [ttk::frame $froundentries.round \
                    -relief flat -borderwidth 1]
    set fentries [ ttk::frame $froundentries.entries \
                    -relief flat -borderwidth 0]

    ttk::label $fentries.lr -text [_ "Red"]:
    ttk::label $fentries.lg -text [_ "Green"]:
    ttk::label $fentries.lb -text [_ "Blue"]:
    ttk::entry $fentries.er -width 4 -justify right \
            -textvariable ::GiDSelectColor::_entry_values(red,dec)
    ttk::entry $fentries.eg -width 4 -justify right \
            -textvariable ::GiDSelectColor::_entry_values(green,dec)
    ttk::entry $fentries.eb -width 4 -justify right \
            -textvariable ::GiDSelectColor::_entry_values(blue,dec)
    ttk::entry $fentries.ehr -width 3 -justify right \
            -textvariable ::GiDSelectColor::_entry_values(red,hex)
    ttk::entry $fentries.ehg -width 3 -justify right \
            -textvariable ::GiDSelectColor::_entry_values(green,hex)
    ttk::entry $fentries.ehb -width 3 -justify right \
            -textvariable ::GiDSelectColor::_entry_values(blue,hex)
    grid $fentries.lr $fentries.er $fentries.ehr
    grid $fentries.lg $fentries.eg $fentries.ehg
    grid $fentries.lb $fentries.eb $fentries.ehb

    grid $fentries $fround -sticky nse
    grid configure $fentries  -sticky nsw
    grid rowconfigure $froundentries 0 -weight 1
    grid columnconfigure $froundentries 0 -weight 1
    grid columnconfigure $froundentries 1 -weight 1

    if { $_transparency } {
        set ftransparency [ttk::frame $froundentries.transparency \
                    -relief flat -borderwidth 0]
        ttk::label $ftransparency.lalpha -text [_ "Opacity:"]
        ttk::entry $ftransparency.ealpha -width 3 -textvariable ::GiDSelectColor::_entry_values(alpha,dec)
        gidscale $ftransparency.alpha -variable ::GiDSelectColor::_entry_values(alpha,dec) \
                -from 0 -to 255 -resolution 1 -orient horizontal -showvalue 0
        grid $ftransparency.lalpha $ftransparency.ealpha $ftransparency.alpha -sticky ew
        grid configure $ftransparency -columnspan 2
        grid $ftransparency -sticky ew
        grid columnconfigure $ftransparency 0 -weight 1
        grid rowconfigure $ftransparency 0 -weight 1
    }

    set fcolor [tk::frame $fg.color \
                    -width 50 \
                    -relief flat -borderwidth 0 \
                    -highlightthickness 0]
    pack $fcolor -in $fround -fill y -expand yes
    pack $froundentries -anchor e -pady 2 -fill both -expand yes

    set fd  [ttk::frame $dlgf.fd]
    #set f1  [ttk::frame $fd.f1 -style sunken.TFrame -borderwidth 1]
    #set f2  [ttk::frame $fd.f2  -style sunken.TFrame -borderwidth 1]
    set f1  [ttk::frame $fd.f1]
    set f2  [ttk::frame $fd.f2]
    set c1  [canvas $f1.c -width 200 -height 200 -borderwidth 0 -highlightthickness 0]
    set c2  [canvas $f2.c -width 15  -height 200 -borderwidth 0 -highlightthickness 0]

    for {set val 0} {$val < 40} {incr val} {
        $c2 create rectangle 0 [expr {5*$val}] 15 [expr {5*$val+5}] -tags val[expr {39-$val}]
    }
    $c2 create polygon 0 0 10 5 0 10 -fill black -outline white -tags target

    pack $c1 $c2
    pack $f1 $f2 -side left -padx 10 -anchor n

    pack $fg $fd -side left -anchor n -fill y

    bind $c1 <ButtonPress-1> "GiDSelectColor::_select_hue_sat %x %y"
    bind $c1 <B1-Motion>     "GiDSelectColor::_select_hue_sat %x %y"

    bind $c2 <ButtonPress-1> "GiDSelectColor::_select_value %x %y"
    bind $c2 <B1-Motion>     "GiDSelectColor::_select_value %x %y"

    if {![info exists _image] || [catch {image type $_image}]} {
        set _image [image create photo -width 200 -height 200]
        for {set x 0} {$x < 200} {incr x 4} {
            for {set y 0} {$y < 200} {incr y 4} {
                $_image put \
                        [eval format "\#%04x%04x%04x" \
                        [hsvToRgb [expr {$x/196.0}] [expr {(196-$y)/196.0}] 0.85]] \
                        -to $x $y [expr {$x+4}] [expr {$y+4}]
            }
        }
    }
    $c1 create image  0 0 -anchor nw -image $_image
    $c1 create bitmap 0 0 \
        -bitmap @[file join $::BWIDGET::LIBRARY "images" "target.xbm"] \
        -anchor nw -tags target

    set _selection -1
    set _widget(fcolor) $fg
    set _widget(chs)    $c1
    set _widget(cv)     $c2
    set rgb             [winfo rgb $path [Widget::cget $path:GiDSelectColor -color]]
    set _hsv            [eval rgbToHsv $rgb]
    _set_rgb     [eval format "\#%04x%04x%04x" $rgb]
    _set_hue_sat [lindex $_hsv 0] [lindex $_hsv 1]
    _set_value   [lindex $_hsv 2]

    trace variable ::GiDSelectColor::_entry_values(red,dec) w ::GiDSelectColor::_CambioEntries
    trace variable ::GiDSelectColor::_entry_values(green,dec) w ::GiDSelectColor::_CambioEntries
    trace variable ::GiDSelectColor::_entry_values(blue,dec) w ::GiDSelectColor::_CambioEntries
    trace variable ::GiDSelectColor::_entry_values(red,hex) w ::GiDSelectColor::_CambioEntriesHex
    trace variable ::GiDSelectColor::_entry_values(green,hex) w ::GiDSelectColor::_CambioEntriesHex
    trace variable ::GiDSelectColor::_entry_values(blue,hex) w ::GiDSelectColor::_CambioEntriesHex
    
    set b0 [$top add -name ok -text [_ "Ok"]]
    set b1 [$top add -name cancel -text [_ "Cancel"]]
    #FALTA -style BottomFrame.TButton y tambien del frame
    if { [ GiDSelectColor::GetUsingThemes]} {
        # buttons are almost configured, just need
        pack configure $top.bbox -fill both -expand 1
        grid configure $b0 -sticky s -pady 12 -padx 8
        grid configure $b1 -sticky s -pady 12 -padx 8
        $top.bbox configure -background [ set ::ttk::theme::[ ::gid_themes::GetTtkTheme ]::colors(-text_bg) ]
    }

    #replace use of old proc SetWidgetsWidthFromText by this block, to delete the old proc
    set len0 [string length [_ "Ok"]]
    set len1 [string length [_ "Cancel"]]
    if { $len0 > $len1 } {
        set maxlength $len0
    } else {
        set maxlength $len1
    }
    $b0 configure -width $maxlength
    $b1 configure -width $maxlength
    #end replace

    set res [$top draw]
    if {$res == 0} {
        set color [$fg.color cget -background]
        if { [ string length $color] == 13} {
            # es de la forma #rrrrggggbbbb -> cada componente 0..65535
            scan $color #%4x%4x%4x r g b
            set r [ expr int( 0.5 + $r * 255.0 / 65535.0)]
            set g [ expr int( 0.5 + $g * 255.0 / 65535.0)]
            set b [ expr int( 0.5 + $b * 255.0 / 65535.0)]
            set color [ format #%02x%02x%02x $r $g $b]
        }
    } else {
        set color ""
    }

    if { $_transparency && $color != "" } {
        append color [format "%02x" $_entry_values(alpha,dec)]
    }

    trace vdelete ::GiDSelectColor::_entry_values(red,dec) w ::GiDSelectColor::_CambioEntries
    trace vdelete ::GiDSelectColor::_entry_values(green,dec) w ::GiDSelectColor::_CambioEntries
    trace vdelete ::GiDSelectColor::_entry_values(blue,dec) w ::GiDSelectColor::_CambioEntries
    trace vdelete ::GiDSelectColor::_entry_values(red,hex) w ::GiDSelectColor::_CambioEntriesHex
    trace vdelete ::GiDSelectColor::_entry_values(green,hex) w ::GiDSelectColor::_CambioEntriesHex
    trace vdelete ::GiDSelectColor::_entry_values(blue,hex) w ::GiDSelectColor::_CambioEntriesHex

    destroy $top
    return $color
}

proc GiDSelectColor::_select_rgb {count} {
    variable _baseColors
    variable _userColors
    variable _selection
    variable _widget
    variable _hsv

    set frame $_widget(fcolor)
    if {$_selection >= 0} {
        $frame.round$_selection configure -relief sunken -borderwidth 1
    }
    $frame.round$count configure -relief flat -borderwidth 1
    focus $frame.round$count
    set _selection $count
    set bg   [$frame.color$count cget -background]
    set user [expr {$_selection-[llength $_baseColors]}]
    if {$user >= 0 &&
        ![string compare \
              [winfo rgb $frame.color$_selection $bg] \
              [winfo rgb $frame.color$_selection white]]} {
        set bg [$frame.color cget -background]
        $frame.color$_selection configure -background $bg
        set _userColors [lreplace $_userColors $user $user $bg]
    } else {
        set _hsv [eval rgbToHsv [winfo rgb $frame.color$count $bg]]
        _set_hue_sat [lindex $_hsv 0] [lindex $_hsv 1]
        _set_value   [lindex $_hsv 2]
        $frame.color configure -background $bg
    }

    _ActualizeEntries $bg
}


proc GiDSelectColor::_set_rgb {rgb} {
    variable _selection
    variable _baseColors
    variable _userColors
    variable _widget

    set frame $_widget(fcolor)
    $frame.color configure -background $rgb
    set user [expr {$_selection-[llength $_baseColors]}]
    if {$user >= 0} {
        $frame.color$_selection configure -background $rgb
        set _userColors [lreplace $_userColors $user $user $rgb]
    }

    _ActualizeEntries $rgb
}


proc GiDSelectColor::_select_hue_sat {x y} {
    variable _widget
    variable _hsv

    if {$x < 0} {
        set x 0
    } elseif {$x > 200} {
        set x 200
    }
    if {$y < 0 } {
        set y 0
    } elseif {$y > 200} {
        set y 200
    }
    set hue  [expr {$x/200.0}]
    set sat  [expr {(200-$y)/200.0}]
    set _hsv [lreplace $_hsv 0 1 $hue $sat]
    $_widget(chs) coords target [expr {$x-9}] [expr {$y-9}]
    _draw_values $hue $sat
    _set_rgb [eval format "\#%04x%04x%04x" [eval hsvToRgb $_hsv]]
}


proc GiDSelectColor::_set_hue_sat {hue sat} {
    variable _widget

    set x [expr {$hue*200-9}]
    set y [expr {(1-$sat)*200-9}]
    $_widget(chs) coords target $x $y
    _draw_values $hue $sat
}



proc GiDSelectColor::_select_value {x y} {
    variable _widget
    variable _hsv

    if {$y < 0} {
        set y 0
    } elseif {$y > 200} {
        set y 200
    }
    $_widget(cv) coords target 0 [expr {$y-5}] 10 $y 0 [expr {$y+5}]
    set _hsv [lreplace $_hsv 2 2 [expr {(200-$y)/200.0}]]
    _set_rgb [eval format "\#%04x%04x%04x" [eval hsvToRgb $_hsv]]
}


proc GiDSelectColor::_draw_values {hue sat} {
    variable _widget

    for {set val 0} {$val < 40} {incr val} {
        set l   [hsvToRgb $hue $sat [expr {$val/39.0}]]
        set col [eval format "\#%04x%04x%04x" $l]
        $_widget(cv) itemconfigure val$val -fill $col -outline $col
    }
}


proc GiDSelectColor::_set_value {value} {
    variable _widget

    set y [expr {int((1-$value)*200)}]
    $_widget(cv) coords target 0 [expr {$y-5}] 10 $y 0 [expr {$y+5}]
}


# --
#  Taken from tk8.0/demos/tcolor.tcl
# --
# The procedure below converts an HSB value to RGB.  It takes hue, saturation,
# and value components (floating-point, 0-1.0) as arguments, and returns a
# list containing RGB components (integers, 0-65535) as result.  The code
# here is a copy of the code on page 616 of "Fundamentals of Interactive
# Computer Graphics" by Foley and Van Dam.

proc GiDSelectColor::hsvToRgb {hue sat val} {
    set v [expr {round(65535.0*$val)}]
    if {$sat == 0} {
        return [list $v $v $v]
    } else {
        set hue [expr {$hue*6.0}]
        if {$hue >= 6.0} {
            set hue 0.0
        }
        set i [expr {int($hue)}]
        set f [expr {$hue-$i}]
        set p [expr {round(65535.0*$val*(1 - $sat))}]
        set q [expr {round(65535.0*$val*(1 - ($sat*$f)))}]
        set t [expr {round(65535.0*$val*(1 - ($sat*(1 - $f))))}]
        switch $i {
            0 {return [list $v $t $p]}
            1 {return [list $q $v $p]}
            2 {return [list $p $v $t]}
            3 {return [list $p $q $v]}
            4 {return [list $t $p $v]}
            5 {return [list $v $p $q]}
        }
    }
}


# --
#  Taken from tk8.0/demos/tcolor.tcl
# --
# The procedure below converts an RGB value to HSB.  It takes red, green,
# and blue components (0-65535) as arguments, and returns a list containing
# HSB components (floating-point, 0-1) as result.  The code here is a copy
# of the code on page 615 of "Fundamentals of Interactive Computer Graphics"
# by Foley and Van Dam.

proc GiDSelectColor::rgbToHsv {red green blue} {
    if {$red > $green} {
        set max $red.0
        set min $green.0
    } else {
        set max $green.0
        set min $red.0
    }
    if {$blue > $max} {
        set max $blue.0
    } else {
        if {$blue < $min} {
            set min $blue.0
        }
    }
    set range [expr {$max-$min}]
    if {$max == 0} {
        set sat 0
    } else {
        set sat [expr {($max-$min)/$max}]
    }
    if {$sat == 0} {
        set hue 0
    } else {
        set rc [expr {($max - $red)/$range}]
        set gc [expr {($max - $green)/$range}]
        set bc [expr {($max - $blue)/$range}]
        if {$red == $max} {
            set hue [expr {.166667*($bc - $gc)}]
        } else {
            if {$green == $max} {
                set hue [expr {.166667*(2 + $rc - $bc)}]
            } else {
                set hue [expr {.166667*(4 + $gc - $rc)}]
            }
        }
        if {$hue < 0.0} {
            set hue [expr {$hue + 1.0}]
        }
    }
    return [list $hue $sat [expr {$max/65535}]]
}

