
package require fulltktree

namespace eval DFRM {
    variable tables
    variable mapped { 0 }
}

proc DFRM::DeformationWin { {w .gid.deformation} } {
    variable tables
    variable currenttab
    InitWindow2 $w -title [_ "Deformation"] -geometryvariable PreDeformationWindowGeom \
        -initcommand DFRM::DeformationWin
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0

    set nb [ttk::notebook $w.nb]
    ttk::notebook::enableTraversal $nb

    set f [ttk::frame $nb.fgroups]
    $nb add $f -text [_ "Groups"] -underline 0
    set tables(group) [DeformationGroupsCreateListWin $f]

    set f [ttk::frame $nb.fparameters]
    $nb add $f -text [_ "Parameters"] -underline 0
    set tables(parameter) [DeformationParametersCreateListWin $f]

    set f [ttk::frame $nb.faxes]
    $nb add $f -text [_ "Axes"] -underline 0
    set tables(axe) [DeformationAxesCreateListWin $f]

    set f [ttk::frame $nb.fmovements]
    $nb add $f -text [_ "Movements"] -underline 0
    set tables(movement) [DeformationMovementsCreateListWin $f]

#     set f [ttk::frame $nb.fdeform]
#     $nb add $f -text [_ "Deform"] -underline 0
#     set tables(deform) [DeformationDeformCreateListWin $f]
    set tables(deform) $f

    ttk::frame $w.buts -style BottomFrame.TFrame
    ttk::button $w.buts.close -text [_ "Close"] \
      -command [list destroy $w] -style BottomFrame.TButton

    grid $w.nb -sticky nsew -pady 0

    grid $w.buts.close -sticky ew -padx 5 -pady 6
    grid $w.buts -sticky ews
    grid anchor $w.buts center
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1

    FillInfoDeformation

    if { [info exists currenttab] } {
	$nb select $currenttab
    }   

    bind $w <<NotebookTabChanged>> [list DFRM::DeformationTabChanged $nb]
}

proc DFRM::DeformationTabChanged { nb } {
    variable currenttab
    set currenttab [$nb index current]
}

proc DFRM::FillInfoDeformation { } {
    variable tables
    if { [info exists tables(group)] && [winfo exists $tables(group)] } {
	FillTableInfoDeformationGroups $tables(group)
    }
    if { [info exists tables(parameter)] && [winfo exists $tables(parameter)] } {
	FillTableInfoDeformationParameters $tables(parameter)
    }
    if { [info exists tables(axe)] && [winfo exists $tables(axe)] } {
	FillTableInfoDeformationAxes $tables(axe)
    }
    if { [info exists tables(movement)] && [winfo exists $tables(movement)] } {
	FillTableInfoDeformationMovements $tables(movement)
    }
#     if { [info exists tables(deform)] && [winfo exists $tables(deform)] } {
#         FillTableInfoDeformationDeform $tables(deform)
#    }

}

# T must be a treectrl
proc DFRM::GetNamesFromSelection { T } {
    set names ""
    foreach item [$T selection get] {
	lappend names [$T item text $item 0]
    }
    return $names
}

################################################
############## DEFORMATION GROUPS ##############
################################################

proc DFRM::DeformationGroupsCreateListWin { w } {
    set cols [list \
		  [list 15 [_ "Name"] left text 1 ] \
		  [list 15 [_ "Volume"] left text 0 ] \
		  [list 20 [_ "Points"] left text 0 ] \
		 ]
    set tbl [fulltktree $w.t -columns $cols -showlines 0]
    $tbl configure -editaccepthandler DFRM::EditHandlerDeformationGroups \
	-deletehandler DFRM::DeleteHandlerDeformationGroups \
	-contextualhandler_menu DFRM::MenuContextualDeformationGroups

    grid $tbl -sticky nsew

    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1
    return $tbl
}

proc DFRM::FillTableInfoDeformationGroups { T } {
    if { ![winfo exists $T] } { return }
    $T item delete all
    foreach name [GiD_Deformation get groups] {
	$T insert end [GiD_Deformation get group $name]
    }
}

proc DFRM::GetAutomaticGroupName { } {
    set prefix G
    set names [GiD_Deformation get groups]
    set name ""
    for {set i 0} {$i<10000} {incr i} {
	set name ${prefix}${i}
	if { [lsearch -exact $names $name] == -1 } { break }
    }
    return $name
}

proc DFRM::GroupCreate { } {
    variable tables
    GiD_Process MEscape Utilities DeformationVol CreateGroup
    FillTableInfoDeformationGroups $tables(group)
    return 0
}

proc DFRM::GroupRename { oldname newname } {
    variable tables
    GiD_Process MEscape Utilities DeformationVol RenameGroup $oldname $newname escape
    FillTableInfoDeformationGroups $tables(group)
    FillTableInfoDeformationMovements $tables(movement)
    return 0
}

proc DFRM::GroupDelete { name } {
    variable tables
    GiD_Process MEscape Utilities DeformationVol DeleteGroup $name escape
    FillTableInfoDeformationGroups $tables(group)
    FillTableInfoDeformationMovements $tables(movement)
    return 0
}

proc DFRM::GroupDeleteSelection { T } {
    variable tables
    foreach name [DFRM::GetNamesFromSelection $T] {
	GiD_Process MEscape Utilities DeformationVol DeleteGroup $name escape
    }
    FillTableInfoDeformationGroups $tables(group)
    FillTableInfoDeformationMovements $tables(movement)
}

proc DFRM::GroupDrawSelection { T } {
    foreach name [DFRM::GetNamesFromSelection $T] {
	GiD_Process MEscape Utilities DeformationVol DrawGroup $name
	break ;#now draw only the first group of the selection
    }
}

proc DFRM::EditHandlerDeformationGroups { args } {
    foreach {T item col text} $args break
    set name [$T item text $item 0]
    if { $col == 0 } {
	DFRM::GroupRename $name $text
	#$T item text $item 0 $text
    }
}

proc DFRM::DeleteHandlerDeformationGroups { args } {
    foreach {T items} $args break
    foreach item $items {
	set name [$T item text $item 0]
	DFRM::GroupDelete $name
	#$T item delete $item
    }
}

proc DFRM::MenuContextualDeformationGroups { T menu item selection } {
    $menu add command -label [_ "New group#C#deformation"] -command [list DFRM::GroupCreate]
    if { [$T selection get] == "" } {
	$menu add command -label [_ "Delete group#C#deformation"] -command [list DFRM::GroupDeleteSelection $T] -state disabled
	$menu add command -label [_ "Draw group#C#deformation"] -command [list DFRM::GroupDrawSelection $T] -state disabled
    } else {
	$menu add command -label [_ "Delete group#C#deformation"] -command [list DFRM::GroupDeleteSelection $T]
	$menu add command -label [_ "Draw group#C#deformation"] -command [list DFRM::GroupDrawSelection $T]
    }
}

################################################
############## DEFORMATION PARAMETERS ##########
################################################

proc DFRM::DeformationParametersCreateListWin { w } {
    set cols [list \
		  [list 15 [_ "Name"] left text 1 ] \
		  [list 15 [_ "Value"] left text 1 ] \
		 ]
    set tbl [fulltktree $w.t -columns $cols -showlines 0]
    $tbl configure -editaccepthandler DFRM::EditHandlerDeformationParameters \
	-deletehandler DFRM::DeleteHandlerDeformationParameters \
	-contextualhandler_menu DFRM::MenuContextualDeformationParameters
    grid $tbl -sticky nsew
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1
    return $tbl
}

proc DFRM::FillTableInfoDeformationParameters { T } {
    if { ![winfo exists $T] } { return }
    $T item delete all
    foreach name [GiD_Deformation get parameters] {
	set value [GiD_Deformation get parameter $name]
	$T insert end [list $name $value]
    }
}

proc DFRM::GetAutomaticParameterName { } {
    set prefix P
    set names [GiD_Deformation get parameters]
    set name ""
    for {set i 0} {$i<10000} {incr i} {
	set name ${prefix}${i}
	if { [lsearch -exact $names $name] == -1 } { break }
    }
    return $name
}

proc DFRM::ParameterCreate { } {
    variable tables
    set name [DFRM::GetAutomaticParameterName]
    set value 0.0
    GiD_Process MEscape Utilities DeformationVol CreateParameter $name $value escape
    FillTableInfoDeformationParameters $tables(parameter)
    return 0
}

proc DFRM::ParameterSet { name value } {
    variable tables
    variable parameterlimits
    GiD_Process MEscape Utilities DeformationVol SetParameter $name $value escape
    FillTableInfoDeformationParameters $tables(parameter)
    set parameterlimits(value,$name) $value
    return 0
}

proc DFRM::ParameterRename { oldname newname } {
    variable tables
    GiD_Process MEscape Utilities DeformationVol RenameParameter $oldname $newname escape
    FillTableInfoDeformationParameters $tables(parameter)
    FillTableInfoDeformationMovements $tables(movement)
    return 0
}

proc DFRM::ParameterDelete { name } {
    variable tables
    GiD_Process Mescape Utilities Deformation DeleteParameter $name escape
    FillTableInfoDeformationParameters $tables(parameter)
    FillTableInfoDeformationMovements $tables(movement)
    return 0
}

proc DFRM::ParameterDeleteSelection { T } {
    variable tables
    foreach name [DFRM::GetNamesFromSelection $T] {
	GiD_Process Mescape Utilities Deformation DeleteParameter $name escape
    }
    FillTableInfoDeformationParameters $tables(parameter)
    FillTableInfoDeformationMovements $tables(movement)
}

proc DFRM::MenuContextualDeformationParameters { T menu item selection } {
    $menu add command -label [_ "New parameter#C#deformation"] -command [list DFRM::ParameterCreate]
    if { [$T selection get] == "" } {
	$menu add command -label [_ "Delete parameter#C#deformation"] -command [list DFRM::ParameterDeleteSelection $T] -state disabled
    } else {
	$menu add command -label [_ "Delete parameter#C#deformation"] -command [list DFRM::ParameterDeleteSelection $T]
    }
}

proc DFRM::EditHandlerDeformationParameters { args } {
    foreach {T item col text} $args break
    set name [$T item text $item 0]
    if { $col == 0 } {
	DFRM::ParameterRename $name $text
	#$T item text $item 0 $text
    } elseif { $col == 1 } {
	DFRM::ParameterSet $name $text
	#$T item text $item 1 $text
    }
}

proc DFRM::DeleteHandlerDeformationParameters { args } {
    foreach {T items} $args break
    foreach item $items {
	set name [$T item text $item 0]
	DFRM::ParameterDelete $name
	#$T item delete $item
    }
}

################################################
############## DEFORMATION AXES ################
################################################

proc DFRM::DeformationAxesCreateListWin { w } {
    set cols [list \
	    [list 15 [_ "Name"] left text 1 ] \
	    ]
    set tbl [fulltktree $w.t -columns $cols -showlines 0]
    $tbl configure -editaccepthandler DFRM::EditHandlerDeformationAxes \
	-deletehandler DFRM::DeleteHandlerDeformationAxes \
	-contextualhandler_menu DFRM::MenuContextualDeformationAxes

    grid $tbl -sticky nsew

    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1
    return $tbl
}

proc DFRM::FillTableInfoDeformationAxes { T } {
    if { ![winfo exists $T] } { return }
    $T item delete all
    foreach name [GiD_Info localaxes] {
	$T insert end $name
    }
}

proc DFRM::GetAutomaticAxeName { } {
    set prefix A
    set names [GiD_Info localaxes]
    set name ""
    for {set i 0} {$i<10000} {incr i} {
	set name ${prefix}${i}
	if { [lsearch -exact $names $name] == -1 } { break }
    }
    return $name
}

proc DFRM::AxeCreate { } {
    variable tables
    GiD_Process MEscape Data LocalAxes DefineLocAxes [GetAutomaticAxeName]
    FillTableInfoDeformationAxes $tables(axe)
    return 0
}

proc DFRM::AxeRename { oldname newname } {
    variable tables
    GiD_Process MEscape Data LocalAxes RenameLA $oldname $newname escape
    FillTableInfoDeformationAxes $tables(axe)
    FillTableInfoDeformationMovements $tables(movement)
    return 0
}

proc DFRM::AxeDelete { name } {
    variable tables
    GiD_Process MEscape Data LocalAxes DeleteLA $name escape
    FillTableInfoDeformationAxes $tables(axe)
    FillTableInfoDeformationMovements $tables(movement)
    return 0
}

proc DFRM::AxeDeleteSelection { T } {
    variable tables
    foreach name [DFRM::GetNamesFromSelection $T] {
	GiD_Process MEscape Data LocalAxes DeleteLA $name escape
    }
    FillTableInfoDeformationAxes $tables(axe)
    FillTableInfoDeformationMovements $tables(movement)
}

proc DFRM::AxeDrawSelection { T } {
    foreach name [DFRM::GetNamesFromSelection $T] {
	GiD_Process MEscape Data LocalAxes DrawLocAxes $name
	break ;#now draw only the first axe of the selection
    }
}

proc DFRM::EditHandlerDeformationAxes { args } {
    foreach {T item col text} $args break
    set name [$T item text $item 0]
    if { $col == 0 } {
	DFRM::AxeRename $name $text
	#$T item text $item 0 $text
    }
}

proc DFRM::DeleteHandlerDeformationAxes { args } {
    foreach {T items} $args break
    foreach item $items {
	set name [$T item text $item 0]
	DFRM::AxeDelete $name
	#$T item delete $item
    }
}

proc DFRM::MenuContextualDeformationAxes { T menu item selection } {
    $menu add command -label [_ "New axis#C#deformation"] -command [list DFRM::AxeCreate]
    if { [$T selection get] == "" } {
	$menu add command -label [_ "Delete axis#C#deformation"] -command [list DFRM::AxeDeleteSelection $T] -state disabled
	$menu add command -label [_ "Draw axis#C#deformation"] -command [list DFRM::AxeDrawSelection $T] -state disabled
    } else {
	$menu add command -label [_ "Delete axis#C#deformation"] -command [list DFRM::AxeDeleteSelection $T]
	$menu add command -label [_ "Draw axis#C#deformation"] -command [list DFRM::AxeDrawSelection $T]
    }
}

################################################
############## DEFORMATION MOVEMENTS ###########
################################################

proc DFRM::DeformationMovementsCreateListWin { w } {
    set cols [list \
		  [list 15 [_ "Group"] left text 0] \
		  [list 15 [_ "Parameter"] left text 0] \
		  [list 15 [_ "Axes"] left text 0] \
		  [list 15 [_ "Component"] left text 0] \
		 ]
    set tbl [fulltktree $w.t -columns $cols -showlines 0]
    $tbl configure -deletehandler DFRM::DeleteHandlerDeformationMovements \
	-contextualhandler_menu DFRM::MenuContextualDeformationMovements

    grid $tbl -sticky nsew
    bind $tbl <Button-$::gid_right_button> "[list MenuContextualDeformationMovements %W %x %y] ; break"


    DeformationDeformCreateListWin $w

    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1

    return $tbl
}

proc DFRM::FillTableInfoDeformationMovements { T } {
    if { ![winfo exists $T] } { return }
    $T item delete all
    foreach item [GiD_Deformation get movements] {
	$T insert end $item 0
    }
    set w [winfo parent $T]
    FillTableInfoDeformationDeform $w
}

proc DFRM::MovementCreate { group parameter axe component } {
    variable tables
    GiD_Process Mescape Utilities Deformation CreateMovement $group $parameter $axe $component escape
    FillTableInfoDeformationMovements $tables(movement)
    FillTableInfoDeformationDeform $tables(deform)
    return 0
}

proc DFRM::MovementSet { name group parameter axe component } {
    variable tables
    GiD_Process Mescape Utilities Deformation DeleteMovement $name escape
    GiD_Process Mescape Utilities Deformation CreateMovement $group $parameter $axe $component escape
    FillTableInfoDeformationMovements $tables(movement)
    FillTableInfoDeformationDeform $tables(deform)
}

proc DFRM::MovementDelete { name } {
    variable tables
    GiD_Process Mescape Utilities Deformation DeleteMovement $name escape
    FillTableInfoDeformationMovements $tables(movement)
    FillTableInfoDeformationDeform $tables(deform)
    return 0
}

proc DFRM::MovementDeleteSelection { T } {
    variable tables
#     foreach name [DFRM::GetNamesFromSelection $T] {
#         GiD_Process Mescape Utilities Deformation DeleteMovement $name escape
#     }
    #warnign sort decreasing because numbers change when deleting, maybe is beter to use names
    foreach item [lsort -integer -decreasing [$T selection get]] {
	GiD_Process Mescape Utilities Deformation DeleteMovement $item escape
    }
    FillTableInfoDeformationMovements $tables(movement)
    FillTableInfoDeformationDeform $tables(deform)
}

proc DFRM::DeleteHandlerDeformationMovements { args } {
    foreach {T items} $args break
    foreach item $items {
	#set name [$T item text $item 0]
	#DFRM::MovementDelete $name
	DFRM::MovementDelete $item
	#$T item delete $item
    }
}

proc DFRM::MovementDrawGroupSelection { T } {
    foreach name [DFRM::GetNamesFromSelection $T] {
	GiD_Process MEscape Utilities DeformationVol DrawGroup $name
	break ;#now draw only the first group of the selection
    }
}

proc DFRM::MovementDrawAxeSelection { T } {
    set names ""
    foreach item [$T selection get] {
	lappend names [$T item text $item 2]
    }
    foreach name $names {
	GiD_Process MEscape Data LocalAxes DrawLocAxes $name
	break ;#now draw only the first group of the selection
    }
}

proc DFRM::MenuContextualDeformationMovements { T menu item selection } {
    $menu add command -label [_ "New movement#C#deformation"] -command [list DFRM::MovementWin]
    if { [llength [$T selection get]] == 1 } {
	$menu add command -label [_ "Edit movement#C#deformation"] -command [list DFRM::MovementWin edit [$T selection get]]
    } else {
	$menu add command -label [_ "Edit movement#C#deformation"] -command [list DFRM::MovementWin edit ""] -state disabled
    }
    if { [$T selection get] == "" } {
	$menu add command -label [_ "Delete movement#C#deformation"] -command [list DFRM::MovementDeleteSelection $T] -state disabled
    } else {
	$menu add command -label [_ "Delete movement#C#deformation"] -command [list DFRM::MovementDeleteSelection $T]
    }
    $menu add command -label [_ "Draw group#C#deformation"] -command [list DFRM::MovementDrawGroupSelection $T]
    $menu add command -label [_ "Draw axis#C#deformation"] -command [list DFRM::MovementDrawAxeSelection $T]
}

#mode: new edit
proc DFRM::MovementWin { {mode new} {name ""} {w .gid.deformation.newmovement} } {
    variable newmovement
    if { $mode == "new" } {
        InitWindow2 $w -title [_ "New Movement"] -geometryvariable PreMovementWindowGeom \
            -initcommand DFRM::MovementWin -ontop
    } else {
        #edit
        InitWindow2 $w -title [_ "Edit movement"] -geometryvariable PreMovementWindowGeom \
            -initcommand DFRM::MovementWin -ontop
        if { $name != "" } {
            set movement [lindex [GiD_Deformation get movements] [expr $name-1]]
            set newmovement(group) [lindex $movement 0]
            set newmovement(parameter) [lindex $movement 1]
            set newmovement(axe) [lindex $movement 2]
            set newmovement(component) [lindex $movement 3]
        }
    }
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0

    set f [ttk::frame $w.f]
    ttk::label $f.lgroup -text [_ "Group"]
    ttk::combobox $f.cbgroup -values [GiD_Deformation get groups] -textvariable DFRM::newmovement(group)
    ttk::label $f.lparameter -text [_ "Parameter"]
    ttk::combobox $f.cbparameter -values [GiD_Deformation get parameters] -textvariable DFRM::newmovement(parameter)
    ttk::label $f.laxe -text [_ "Axes"]
    ttk::combobox $f.cbaxe -values [GiD_Info localaxes] -textvariable DFRM::newmovement(axe)
    ttk::label $f.lcomponent -text [_ "Component"]
    ttk::combobox $f.cbcomponent -values {1 2 3} -textvariable DFRM::newmovement(component)

    ttk::frame $w.buts -style BottomFrame.TFrame
    if { $mode == "new" } {
	ttk::button $w.buts.ok -text [_ "Ok"] -command [list DFRM::NewMovementOk $name $w] \
	  -style BottomFrame.TButton
    } else {
	ttk::button $w.buts.ok -text [_ "Ok"] -command [list DFRM::EditMovementOk $name $w] \
	  -style BottomFrame.TButton
    }
    ttk::button $w.buts.close -text [_ "Cancel"] -command [list destroy $w] -style BottomFrame.TButton

    grid $f.lgroup $f.cbgroup -sticky ew
    grid $f.lparameter $f.cbparameter -sticky ew
    grid $f.laxe $f.cbaxe -sticky ew
    grid $f.lcomponent $f.cbcomponent -sticky ew
    grid $f -sticky nsew -pady 0
    grid columnconfigure $f {0 1} -weight 1

    grid $w.buts.ok $w.buts.close -sticky ew -padx 5 -pady 6
    grid $w.buts -sticky ews
    grid anchor $w.buts center
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1

    if { ![info exists newmovement(group)] } {
	set newmovement(group) [lindex [$f.cbgroup cget -values] 0]
    }
    if { ![info exists newmovement(parameter)] } {
	set newmovement(parameter) [lindex [$f.cbparameter cget -values] 0]
    }
    if { ![info exists newmovement(axe)] } {
	set newmovement(axe) [lindex [$f.cbaxe cget -values] 0]
    }
    if { ![info exists newmovement(component)] } {
	set newmovement(component) [lindex [$f.cbcomponent cget -values] 0]
    }

    bind $w <Return> "$w.buts.ok invoke"

    bind $w <Alt-c> "destroy $w"
    bind $w <Escape> "destroy $w"
}

proc DFRM::NewMovementOk { name w } {
    variable newmovement
    set group [$w.f.cbgroup get]
    set parameter [$w.f.cbparameter get]
    set axe [$w.f.cbaxe get]
    set component [$w.f.cbcomponent get]
    DFRM::MovementCreate $group $parameter $axe $component
    destroy $w
}

proc DFRM::EditMovementOk { name w } {
    variable newmovement
    set group [$w.f.cbgroup get]
    set parameter [$w.f.cbparameter get]
    set axe [$w.f.cbaxe get]
    set component [$w.f.cbcomponent get]
    MovementSet $name $group $parameter $axe $component
    destroy $w
}

################################################
############## DEFORMATION DEFORM ##############
################################################

proc DFRM::DeformationDeformCreateListWin { w } {
    variable mapped
    variable dynamicupdate
    if { $mapped } {
	set s1 normal
	set s2 disabled
    } else {
	set s1 disabled
	set s2 normal
    }
    if { ![info exists dynamicupdate] } {
	set dynamicupdate 1
    }
    set f [ttk::frame $w.f1]
    ttk::button $f.bmap -text [_ "Map"] -command DFRM::DeformationDeformMapNodes -state $s1
    ttk::button $f.bunmap -text [_ "Unmap"] -command DFRM::DeformationDeformUnmapNodes -state $s2
    ttk::button $f.bdeform -text [_ "Deform"] -command DFRM::DeformationDeformDeformNodes -state $s2
    ttk::button $f.brestorenodes -text [_ "Restore nodes"] -command DFRM::DeformationDeformRestoreNodes -state $s2
    ttk::button $f.brestorebox -text [_ "Restore box"] -command DFRM::DeformationDeformRestoreBox -state $s2
    if { ![info exists DFRM::dynamicupdate] } { set DFRM::dynamicupdate 0 }
    ttk::checkbutton $f.dynamicupdate -text [_ "Dynamic update"] -variable DFRM::dynamicupdate

    grid $f.bmap $f.bunmap $f.bdeform $f.brestorenodes $f.brestorebox -sticky ew -padx 2 -pady 2
    grid $f.dynamicupdate -sticky w -padx 2 -columnspan 5
    grid $f -sticky new
    grid columnconfigure $f {0 1 2 3} -weight 1

    FillTableInfoDeformationDeform $w

#     grid columnconfigure $w 0 -weight 1
#     grid rowconfigure $w 1 -weight 1
    return $w
}

proc DFRM::FillTableInfoDeformationDeform { w } {
    variable parameterlimits
    if { ![winfo exists $w] } { return }
    if { [winfo exists $w.f2] } {
	destroy $w.f2
    }
    set f [ttk::frame $w.f2 -style groove.TFrame]
    set usedparameters ""
    foreach item [GiD_Deformation get movements] {
	lappend usedparameters [lindex $item 1]
    }
    set usedparameters [lsort -unique $usedparameters]
    set i 1
    foreach name $usedparameters {
	set value [GiD_Deformation get parameter $name]
	
	set min [GiD_Deformation get parameter $name min]
	set max [GiD_Deformation get parameter $name max]
	if { $value>=-1.0 && $value<=1.0 } {
	    set from -1.0
	    set to 1.0
	} else {
	    set from [expr $value-1.0]
	    set to [expr $value+1.0]
	}
	if { $from < $min } {
	    set from $min
	}
	if { $to > $max } {
	    set to $max
	}
	set resolution [expr ($to-$from)*0.01]
	set bigincrement [expr ($to-$from)*0.25]
	
	set parameterlimits(value,$name) $value
	if { ![info exists parameterlimits(from,$name)] } {
	    set parameterlimits(from,$name) $from
	}
	if { ![info exists parameterlimits(to,$name)] } {
	    set parameterlimits(to,$name) $to
	}

	set sc $f.scrollparam$i
	ttk::label $f.lparam$i -text $name
	ttk::spinbox $f.spinparammin$i -width 5 -increment 1.0 -from -1000. -to 1000.0 \
	    -textvariable DFRM::parameterlimits(from,$name) \
	    -command [list DFRM::DeformationDeformChangedLimit $sc from $name]
	bind $f.spinparammin$i <Return> [list DFRM::DeformationDeformChangedLimit $sc from $name]
	gidscale $sc -from $DFRM::parameterlimits(from,$name) -to $DFRM::parameterlimits(to,$name) \
	    -resolution $resolution \
	    -orient horizontal -showvalue 1 -variable DFRM::parameterlimits(value,$name) \
	    -command [list DFRM::DeformationDeformChangedParameter $name]
	    #-bigincrement $bigincrement
	ttk::spinbox $f.spinparammax$i -width 5 -increment 1.0 -from -1000. -to 1000.0 \
	    -textvariable DFRM::parameterlimits(to,$name) \
	    -command [list DFRM::DeformationDeformChangedLimit $sc to $name]
	bind $f.spinparammax$i <Return> [list DFRM::DeformationDeformChangedLimit $sc to $name]
	
	grid $f.lparam$i $f.spinparammin$i $f.scrollparam$i $f.spinparammax$i -sticky ew -padx 2 -pady 2
	
	incr i
    }
    grid $f -sticky nsew -padx 2 -pady 2
    grid columnconfigure $f 2 -weight 1
    #grid rowconfigure $f 0 -weight 1

    SetStateDeformationDeform $w
}

proc DFRM::SetStateDeformationDeform { w } {
    variable mapped
    if { ![info exists mapped] } {
	set mapped 0
    }
    if { $mapped } {
	set s1 disabled
	set s2 normal
    } else {
	set s1 normal
	set s2 disabled
    }
    set f $w.f1
    $f.bmap configure -state $s1
    $f.bunmap configure -state $s2
    $f.bdeform configure -state $s2
    $f.brestorenodes configure -state $s2
    $f.brestorebox configure -state $s2
}

proc DFRM::DeformationDeformMapNodes { } {
    variable tables
    variable mapped
    GiD_Process Mescape Utilities Deformation MapNodes escape
    set mapped 1
    #FillTableInfoDeformationDeform $tables(deform)
    SetStateDeformationDeform $tables(deform)
}

proc DFRM::DeformationDeformUnmapNodes { } {
    variable tables
    variable mapped
    GiD_Process Mescape Utilities Deformation UnmapNodes escape
    set mapped 0
    #FillTableInfoDeformationDeform $tables(deform)
    SetStateDeformationDeform $tables(deform)
}

proc DFRM::DeformationDeformDeformNodes { } {
    GiD_Process Mescape Utilities Deformation DeformNodes escape
}

proc DFRM::DeformationDeformRestoreNodes { } {
    GiD_Process Mescape Utilities Deformation RestoreNodes escape
}

proc DFRM::DeformationDeformRestoreBox { } {
    GiD_Process Mescape Utilities Deformation RestoreBox escape
}


proc DFRM::EditHandlerDeformationDeform { args } {
    foreach {T item col text} $args break
    set name [$T item text $item 0]
    if { $col == 1 } {
	ParameterSet $name $text
	#$T item text $item 1 $text
    }
}

proc DFRM::DeformationDeformChangedLimit { w what name } {
    variable parameterlimits
    $w configure -$what $DFRM::parameterlimits($what,$name)
}

proc DFRM::DeformationDeformChangedParameter { name value } {
    variable mapped
    variable dynamicupdate
    if { $value == [GiD_Deformation get parameter $name] } {
	return
    }
    ParameterSet $name $value
    if { $mapped && $dynamicupdate } {
	DeformationDeformDeformNodes
    }
}

