#kike: tests gid_togl open gl with textures (e.g. to select in the future a region in a 3D world)
namespace eval draw_world {
    variable radius_earth 6378137.0 ;#m (equatorial radius WGS-84)
    variable DEGREE_TO_RAD 0.0174532925199
    variable show_plane 0 ;#1 to show spheric
}

proc draw_world::ll2xyz { longitude latitude {h 0.0} } {
    variable radius_earth
    variable DEGREE_TO_RAD    
    set latitude [expr {$latitude*$DEGREE_TO_RAD}]
    set longitude [expr {$longitude*$DEGREE_TO_RAD}]
    set r [expr {$radius_earth+$h}]
    set coslat [expr {cos($latitude)}]
    set x [expr {$r*cos($longitude)*$coslat}]
    set y [expr {$r*sin($longitude)*$coslat}]
    set z [expr {$r*sin($latitude)}]
    return [list $x $y $z]
}

proc draw_world::get_topography_window { } {
    set w .gid.get_topography
    InitWindow2 $w -title [_ "Get topography"] \
        -geometryvariable GiDMapGetTopographyWindowGeom \
        -initcommand draw_world::get_topography_window
    set options [list -width 256 -height 256 -rgba true -double true -depth true -ident GetTopographyWindow -privatecmap false -overlay 0]
    lappend options -createcommand draw_world::on_create -destroycommand draw_world::on_destroy
    gid_togl $w.wogl {*}$options
    bind $w.wogl <ButtonPress-1> { draw_world::rotate_set %x %y %W }
    bind $w.wogl <B1-Motion> { draw_world::rotate_drag %x %y %W }
    bind $w.wogl <ButtonPress-2> { draw_world::scale_set %x %y %W }
    bind $w.wogl <B2-Motion> { draw_world::scale_drag %x %y %W }
    #bind $w.wogl <ButtonPress-3> { draw_world::translate_set %x %y %W }
    #bind $w.wogl <B3-Motion> { draw_world::translate_drag %x %y %W }   
    bind $w <MouseWheel> { draw_world::mouse_wheel %D %W }
    grid $w.wogl -sticky nsew
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1    
    $w.wogl configure -displaycommand draw_world::on_display -reshapecommand draw_world::on_reshape
}

proc draw_world::on_reshape { togl } {
    variable radius_earth
    variable show_plane
#W "draw_world::on_reshape $togl"
    set w [winfo width $togl]
    set h [winfo height $togl]
    GiD_OpenGL draw -viewport 0 0 $w $h
    GiD_OpenGL draw -matrixmode projection
    GiD_OpenGL draw -loadidentity
    if { $show_plane } {
        lassign [list 360 180 100] sx sy sz
    } else {
        lassign [list $radius_earth $radius_earth $radius_earth] sx sy sz
    }
    #GiD_OpenGL draw -perspective [list 60.0 [expr double($w)/double($h)] 1.0 30.0]  
    GiD_OpenGL draw -ortho [list [expr -1.1*$sx] [expr 1.1*$sx] [expr -1.1*$sy] [expr 1.1*$sy] [expr -10*$sz] [expr 10*$sz]]
    GiD_OpenGL draw -matrixMode modelview
    GiD_OpenGL draw -loadidentity       
    GiD_OpenGL draw -translate [list 0.0 0.0 [expr -5*$sz]]
}

proc draw_world::on_create { togl } {
#W "draw_world::on_create $togl"
    variable angle 0.0
    variable rotx 0.0
    variable roty 0.0
    variable rotz 0.0   
    variable scale 1.0
    variable transx 0.0
    variable transy 0.0
    variable transz 0.0
    GiD_OpenGL draw -clearcolor [list 0.0 0.0 0.0 0.0]
    GiD_OpenGL draw -shademodel flat
    GiD_OpenGL draw -enable depthtest
    draw_world::texture_init
}

proc draw_world::on_destroy { togl } {
#W "draw_world::on_destroy $togl"
    draw_world::texture_end
}

proc draw_world::on_display { togl } {
#W "draw_world::on_display $togl"
    variable show_plane
    variable scale
    variable angle
    variable rotx
    variable roty
    variable rotz   
    variable transx
    variable transy
    variable transz
    set sc $scale
    $togl makecurrent
    GiD_OpenGL draw -clear colorbufferbit
    GiD_OpenGL draw -clear depthbufferbit
    GiD_OpenGL draw -pushmatrix
    #GiD_OpenGL draw -translate [list $transx $transy $transz]  
    GiD_OpenGL draw -rotate [list $rotx 1.0 0.0 0.0]
    GiD_OpenGL draw -rotate [list $roty 0.0 1.0 0.0]
    GiD_OpenGL draw -rotate [list $rotz 0.0 0.0 1.0]
    GiD_OpenGL draw -scale  [list $sc $sc $sc]
    GiD_OpenGL draw -rotate [list $angle 0.0 0.0 1.0]    
    if { $show_plane } {
        draw_world::draw_world_plane $togl
    } else {
        draw_world::draw_world_sphere $togl
    }
    GiD_OpenGL draw -popmatrix         
    $togl swapbuffers
}

proc draw_world::texture_init { } {
#W "draw_world::texture_init"
    variable texture_ids    
    set texture_ids [GiD_OpenGL draw -gentextures 1]
    set texture_id [lindex $texture_ids 0]
    GiD_OpenGL draw -bindtexture texture2d $texture_id   
    GiD_OpenGL draw -texparameter texture2d texturewraps repeat
    GiD_OpenGL draw -texparameter texture2d texturewrapt repeat
    GiD_OpenGL draw -texparameter texture2d texturemagfilter nearest
    GiD_OpenGL draw -texparameter texture2d textureminfilter nearest
    #set filename [file join [file dirname [info script]] world.png]
    set filename [file join $::GIDDEFAULT scripts tests textures world.png]    
    set img [image create photo -file $filename]
    GiD_OpenGL draw -teximage2d 0 0 $img
}

proc draw_world::texture_end { } {
#W "draw_world::texture_end"
    variable texture_ids
    GiD_OpenGL draw -deletetextures $texture_ids
    unset texture_ids
}

proc draw_world::draw_world_plane { togl } {    
#W "draw_world::draw_world_plane $togl"
    variable texture_ids
    if { [winfo exists $togl] } {        
        GiD_OpenGL draw -enable texture2d
        GiD_OpenGL draw -texenv textureenv textureenvmode decal
        set texture_id [lindex $texture_ids 0]
        GiD_OpenGL draw -bindtexture texture2d $texture_id   
        GiD_OpenGL draw -begin quads
        GiD_OpenGL draw -texcoord [list 0.0 0.0] -vertex [list -180.0 -85.05112877983284 0.0]  
        GiD_OpenGL draw -texcoord [list 0.0 1.0] -vertex [list -180.0 85.05112877983284 0.0]
        GiD_OpenGL draw -texcoord [list 1.0 1.0] -vertex [list 180.0 85.05112877983284 0.0]
        GiD_OpenGL draw -texcoord [list 1.0 0.0] -vertex [list 180.0 -85.05112877983284 0.0]           
        GiD_OpenGL draw -end
        GiD_OpenGL draw -flush
        GiD_OpenGL draw -disable texture2d
    }
}

proc draw_world::draw_world_sphere { togl } {    
#W "draw_world::draw_world_sphere $togl"
    variable texture_ids
    if { [winfo exists $togl] } {        
        GiD_OpenGL draw -enable texture2d
        GiD_OpenGL draw -texenv textureenv textureenvmode decal
        set texture_id [lindex $texture_ids 0]
        GiD_OpenGL draw -bindtexture texture2d $texture_id   
        GiD_OpenGL draw -begin quads
        lassign [list -180.0 180.0] lon0 lon1
        #lassign [list -85.05112877983284 85.0511287798328] lat0 lat1   
        lassign [list -90.0 90.0] lat0 lat1
        set dlon [expr $lon1-$lon0]
        set dlat [expr $lat1-$lat0]   
        set n_i 25
        set n_j 25
        set incre_lon [expr double($dlon)/$n_i]
        set incre_lat [expr double($dlat)/$n_j]
        for {set i 1} {$i <= $n_i} { incr i } {
            set lon [expr $i*$incre_lon+$lon0]
            set lon_prev [expr $lon-$incre_lon]
            set u [expr double($i)/$n_i]
            set u_prev [expr $u-1.0/$n_i]   
            for {set j 1} {$j <= $n_j} { incr j } {
                set lat [expr $j*$incre_lat+$lat0]                
                set lat_prev [expr $lat-$incre_lat]
                set v [expr double($j)/$n_j]
                set v_prev [expr $v-1.0/$n_j]
                GiD_OpenGL draw -texcoord [list $u_prev $v_prev] -vertex [draw_world::ll2xyz $lon_prev $lat_prev]
                GiD_OpenGL draw -texcoord [list $u $v_prev] -vertex [draw_world::ll2xyz $lon $lat_prev]
                GiD_OpenGL draw -texcoord [list $u $v] -vertex [draw_world::ll2xyz $lon $lat]
                GiD_OpenGL draw -texcoord [list $u_prev $v] -vertex [draw_world::ll2xyz $lon_prev $lat]                    
            }
        }
        GiD_OpenGL draw -end
        GiD_OpenGL draw -flush
        GiD_OpenGL draw -disable texture2d
    }
}

proc draw_world::rotate_set { x y togl } {
#W "draw_world::rotate_set $w"
    variable angle
    variable rotx
    variable roty
    variable rotz   

    variable startx 
    variable starty
    variable xangle0
    variable yangle0
    variable xangle
    variable yangle
    set startx $x
    set starty $y
    #set vPos [$togl position]
    set xangle0 $roty
    set yangle0 $rotx
}

proc draw_world::rotate_drag { x y togl } {
#W "draw_world::rotate_drag $w"
    variable angle
    variable rotx
    variable roty
    variable rotz   

    variable startx 
    variable starty
    variable xangle0
    variable yangle0
    variable xangle
    variable yangle  
    set xangle [expr $xangle0+($x-$startx)]
    set yangle [expr $yangle0+($y-$starty)]
    #$togl rotate $xangle $yangle
    set roty $xangle
    set rotx $yangle
    draw_world::on_display $togl
}

proc draw_world::scale_set { x y togl } { 
    variable scale
    variable startx 
    variable starty
    variable scale0
    set startx $x
    set starty $y
    set scale0 $scale
}

proc draw_world::scale_drag { x y togl } { 
    variable scale
    variable starty
    variable scale0
    set q [expr ($starty-$y)/400.0 ]
    set scale [expr $scale0*exp($q)]
    draw_world::on_display $togl    
}

proc draw_world::translate_set { x y togl } {     
    variable startx 
    variable starty   
    set startx $x
    set starty $y   
}

proc draw_world::translate_drag { x y togl } { 
    variable transx
    variable transy
    variable transz
    variable startx 
    variable starty
    set dx [expr ($starty-$x)/4000.0]
    set dy [expr ($starty-$y)/4000.0]              
    set transx [expr $transx+$dx]
    set transy [expr $transy+$dy]
    draw_world::on_display $togl    
}

proc draw_world::mouse_wheel { increment togl } {
    variable scale  
    if { $increment > 0 } {
        set scale [expr $scale*1.1]
    } else {
        set scale [expr $scale*0.9]
    }    
    draw_world::on_display $togl
}

#e.g. use: -np- source [file join $GIDDEFAULT scripts tests textures draw_world.tcl]
draw_world::get_topography_window 
