# -*- mode: Tcl; tcl-indent-level :  4 -*-

set ::GID_FILE_BROWSER_CLOUD_UNIT_FASTER 1 ;#1 to try to accelerate in case of cloud unit, avoiding features like read problemtype of .geo

# gidtkfbox.tcl -- dialog box to select file, directory or a GiD project
# based on tkfbox.tcl modified to allow consider .gid Project folder as a single file

#for some efficiency the window is not destroyed, but only hidden by widhdrawn, but this is problematic, 
#kike: this variable is to do tests if is better to destroy
set ::GIDTKFBOX_WITHDRAWN 1 ;# 1 to set window state to withdrawn, 0 to destroy

namespace eval ::gidtk {
    variable _busy_cursor 0
}


#######################

#return 1 if changed
proc ::gidtk::setBusyCursor {} {
    variable _busy_cursor   
    if { !$_busy_cursor } {
        set w .gid.central.s
        if { [info exists ::gidtk::Priv(dialog)] && [winfo exists $::gidtk::Priv(dialog)]} {
            set w $::gidtk::Priv(dialog)
        }
        set _busy_cursor 1
        GidUtils::WaitState $w
        return 1
    }
    return 0
}

#return 1 if changed
proc ::gidtk::clearBusyCursor {} {
    variable _busy_cursor   
    if { $_busy_cursor } {
        set w .gid.central.s
        if { [info exists ::gidtk::Priv(dialog)] && [winfo exists $::gidtk::Priv(dialog)]} {
            set w $::gidtk::Priv(dialog)
        }
        set _busy_cursor 0
        GidUtils::EndWaitState $w
        return 1
    }
    return 0
}


#check also some expected GiD model file existence inside the folder, but use the filesystem and could be slow (network)
proc ::gidtk::CouldBePrj { f ext } {
    if { ( $ext == "") || ( [ file extension $f] != $ext) || ( [ file rootname $f] == "") } {
        return 0
    } else {
        return [gid_filesystem::is_gid_project $f]
    }
}

#check only file extension
proc ::gidtk::IsProject { f ext } {
    return [ expr { \
                        ( $ext != "") && \
                        ( [file extension $f] == $ext) && \
                        ( [ file rootname $f] != "") \
                    }]
}


proc font:reqwidth {font text {w ""}} {
    for {set i 1} {[winfo exists $w.label-$i]} {incr i} {}
    set l $w.label-$i
    ttk::label $l -font $font -text $text
    set width  [winfo reqwidth  $l]
    set height [winfo reqheight $l]
    destroy $l
    return $width
}

#----------------------------------------------------------------------
#
#                       I C O N         L I S T
#
# This is a pseudo-widget that implements the icon list inside the
# ::gidtk::dialog::file:: dialog box.
#
#----------------------------------------------------------------------

# ::gidtk::IconList --
#
#         Creates an IconList widget.
#
proc ::gidtk::IconList {w args} {
    IconList_Config $w $args
    IconList_Create $w
}

proc ::gidtk::IconList_Index {w i} {
    upvar #0 ::gidtk::$w data
    upvar #0 ::gidtk::$w:itemList itemList
    if {![info exists data(list)]} {set data(list) {}}
    switch -regexp -- $i {
        "^-?[0-9]+$" {
            if { $i < 0 } {
                set i 0
            }
            if { $i >= [llength $data(list)] } {
                set i [expr {[llength $data(list)] - 1}]
            }
            return $i
        }
        "^active$" {
            return $data(index,active)
        }
        "^anchor$" {
            return $data(index,anchor)
        }
        "^end$" {
            return [llength $data(list)]
        }
        "@-?[0-9]+,-?[0-9]+" {
            lassign [scan $i "@%d,%d"] x y
            set item [$data(canvas) find closest $x $y]
            return [lindex [$data(canvas) itemcget $item -tags] 1]
        }
    }
}

proc ::gidtk::IconList_Selection {w op args} {
    upvar ::gidtk::$w data
    switch -exact -- $op {
        "anchor" {
            if { [llength $args] == 1 } {
                set data(index,anchor) [::gidtk::IconList_Index $w [lindex $args 0]]
            } else {
                return $data(index,anchor)
            }
        }
        "clear" {
            if { [llength $args] == 2 } {
                lassign $args first last
            } elseif { [llength $args] == 1 } {
                set first [set last [lindex $args 0]]
            } else {
                error "wrong # args: should be [lindex [info level 0] 0] path\
                    clear first ?last?"
            }
            set first [IconList_Index $w $first]
            set last [IconList_Index $w $last]
            if { $first > $last } {
                set tmp $first
                set first $last
                set last $tmp
            }
            set ind 0
            foreach item $data(selection) {
                if { $item >= $first } {
                    set first $ind
                    break
                }
            }
            set ind [expr {[llength $data(selection)] - 1}]
            for {} {$ind >= 0} {incr ind -1} {
                set item [lindex $data(selection) $ind]
                if { $item <= $last } {
                    set last $ind
                    break
                }
            }
            if { $first > $last } {
                return
            }
            set data(selection) [lreplace $data(selection) $first $last]
            event generate $w <<ListboxSelect>>
            IconList_DrawSelection $w
        }
        "includes" {
            set index [lsearch -exact $data(selection) [lindex $args 0]]
            return [expr {$index != -1}]
        }
        "set" {
            if { [llength $args] == 2 } {
                lassign $args first last
            } elseif { [llength $args] == 1 } {
                set last [set first [lindex $args 0]]
            } else {
                error "wrong # args: should be [lindex [info level 0] 0] path set first ?last?"
            }
            set first [IconList_Index $w $first]
            set last [IconList_Index $w $last]
            if { $first > $last } {
                set tmp $first
                set first $last
                set last $tmp
            }
            for {set i $first} {$i <= $last} {incr i} {
                lappend data(selection) $i
            }
            set data(selection) [lsort -integer -unique $data(selection)]
            event generate $w <<ListboxSelect>>
            IconList_DrawSelection $w
            IconList_OnSelectCmd $w $last
        }
    }
}

proc ::gidtk::IconList_Curselection {w} {
    upvar ::gidtk::$w data
    return $data(selection)
}

proc ::gidtk::IconList_DrawSelection {w} {
    upvar ::gidtk::$w data
    upvar ::gidtk::$w:itemList itemList
    
    if {[llength $data(selection)]} {
        $data(canvas) delete selection
        foreach item $data(selection) {
            set rTag [lindex [lindex $data(list) $item] 2]
            if { [info exists itemList($rTag)] } {
                lassign $itemList($rTag) iTag tTag text serial idata hidden
                if { $hidden && !$data(options,show_hidden) } {
                    continue
                }
                set bbox [$data(canvas) bbox $tTag]
                $data(canvas) create rect $bbox -tags selection
            }
        }
        $data(canvas) lower selection
    }
    return
}

proc ::gidtk::IconList_Length { w } {
    upvar ::gidtk::$w data
    return [llength $data(list)]
}

proc ::gidtk::IconList_Get {w item} {
    upvar ::gidtk::$w data
    upvar ::gidtk::$w:itemList itemList
    set rTag [lindex [lindex $data(list) $item] 2]
    lassign $itemList($rTag) iTag tTag text serial idata hidden
    return [list $text $idata $hidden]
}

# ::gidtk::IconList_Config --
#
#         Configure the widget variables of IconList, according to the command
#         line arguments.
#
proc ::gidtk::IconList_Config {w argList} {
    
    # 1: the configuration specs
    #
    set specs {
        {-command "" "" ""}
        {-multiple "" "" "0"}
        {-checkrenamecmd "" "" ""}
        {-menucontext "" "" ""}
        {-selectcmd "" "" ""}        
    }
    
    # 2: parse the arguments
    #
    tclParseConfigSpec ::gidtk::$w $specs "" $argList
}

proc ::gidtk::IconList_CreateMenuContextual { w file_picked } {
    upvar ::gidtk::$w data
    
    if { [info exists data(menu_context)] && [winfo exists $data(menu_context)] } {
        destroy $data(menu_context)
    }
    set data(menu_context) [menu $data(canvas).mc -tearoff 0 -borderwidth 1 -activeborder 0]
    
    set data_menu_list $data(-menucontext)
    
    if { [file extension $file_picked] == ".zip" } {
        set w_menus_cmd [winfo parent [winfo parent $w]] ;#.gid.__tk_filedialog  
        lappend data_menu_list ---
        lappend data_menu_list [list \
                -label [_ "Extract all"] \
                -cmd [list ::gidtk::dialog::file::UnzipFile $w_menus_cmd $file_picked] \
                -image [gid_themes::GetImage "zip.png" small_icons] \
                ]
    } else {
        set w_menus_cmd [winfo parent [winfo parent $w]] ;#.gid.__tk_filedialog  
        lappend data_menu_list ---
        lappend data_menu_list [list \
                -label [_ "Compress to %s" [file rootname $file_picked].zip] \
                -cmd [list ::gidtk::dialog::file::ZipFile $w_menus_cmd $file_picked] \
                -image [gid_themes::GetImage "zip.png" small_icons] \
                ]
    }
    
    set idx_entry 0
    foreach i $data_menu_list {
        if {([llength $i] == 1) && ($i == "---") } {
            $data(menu_context) add separator
            incr idx_entry
            continue
        }
        array set entry_info {
            -cmd {}
            -label {}
            -image {}
            -bitmap {}
            -bind {}
        }
        array set entry_info $i
        # accept paremeters: %M menu name, %I index of entry
        regsub -all {%M} $entry_info(-cmd) $data(canvas).mc cmd_tmp
        regsub -all {%I} $cmd_tmp $idx_entry entry_cmd
        $data(menu_context) add command -command $entry_cmd -hidemargin true
        incr idx_entry
        foreach opt {image label bitmap} {
            set has_${opt} 0
            if { $entry_info(-$opt) != "" } {
                set has_${opt} 1
                $data(menu_context) entryconf end -$opt $entry_info(-$opt)
            }
        }
        if { $has_label && ($has_image || $has_bitmap) } {
            $data(menu_context) entryconf end -compound left
        }
        if {$entry_info(-bind) != "" && $entry_info(-cmd) != ""} {
            bind $data(canvas) $entry_info(-bind) +$entry_info(-cmd)
        }
    }
    return    
}

# ::gidtk::IconList_Create --
#
#         Creates an IconList widget by assembling a canvas widget and a
#         scrollbar widget. Sets all the bindings necessary for the IconList's
#         operations.
#
proc ::gidtk::IconList_Create {w} {
    upvar ::gidtk::$w data
    
    set data(options,show_hidden) 0
    set data(dict) [dict create]
    ttk::frame $w
    set data(sbar) [ttk::scrollbar $w.sbar -orient horizontal -takefocus 0]
    set data(canvas) [canvas $w.canvas -borderwidth 1 -relief sunken -width 400 -height 120 -takefocus 1]
    
    if { [llength $data(-menucontext)] } {
        #::gidtk::IconList_CreateMenuContextual $w ""
        bind $data(canvas) <$::gid_right_button> [list ::gidtk::IconList_Btn3 $w %X %Y %x %y]
    }
    
    grid $data(canvas) -row 0 -column 0 -padx 2 -sticky "snew"
    grid $data(sbar) -row 1 -column 0 -sticky "ew"
    grid rowconfigure $w 0 -weight 1
    grid columnconfigure $w 0 -weight 1
    
    $data(sbar) configure -command [list $data(canvas) xview]
    $data(canvas) configure -xscrollcommand [list $data(sbar) set] -background  $::GidPriv(Color,BackgroundFileDialog)
    
    set data(edit) [ttk::entry $data(canvas).edit \
            -textvariable ::gidtk::${w}(edit,textvar)]
    set data(edit,undo) ""
    set data(edit,isactive) 0
    bind $data(edit) <KeyPress-Return> "[list ::gidtk::IconList_TextAccept $w] ; break"
    bind $data(edit) <KeyPress-Escape> "[list ::gidtk::IconList_TextEscape $w] ; break"
    bind $data(edit) <FocusOut>        "[list ::gidtk::IconList_TextAccept $w] ; break"
    
    # Initializes the max icon/text width and height and other variables
    #
    set data(maxIW) 1
    set data(maxIH) 1
    set data(maxTW) 1
    set data(maxTH) 1
    set data(numItems) 0
    set data(curItem)  {}
    set data(noScroll) 1
    set data(selection) {}
    set data(index,anchor) ""
    
    # Creates the event bindings.    
    bind $data(canvas) <Configure> [list ::gidtk::IconList_Arrange $w]
    bind $data(canvas) <1> [list ::gidtk::IconList_Btn1 $w %x %y]
    bind $data(canvas) <B1-Motion> [list ::gidtk::IconList_Motion1 $w %x %y]
    bind $data(canvas) <B1-Leave> [list ::gidtk::IconList_Leave1 $w %x %y]
    bind $data(canvas) <${::acceleratorKey}-1> [list ::gidtk::IconList_CtrlBtn1 $w %x %y]
    bind $data(canvas) <Shift-1> [list ::gidtk::IconList_ShiftBtn1 $w %x %y]
    bind $data(canvas) <B1-Enter> [list tk::CancelRepeat]
    bind $data(canvas) <ButtonRelease-1> [list tk::CancelRepeat]
    #bind $data(canvas) <Double-ButtonRelease-1>
    bind $data(canvas) <Double-1> [list after idle ::gidtk::IconList_Double1 $w %x %y]
    bind $data(canvas) <Up> [list ::gidtk::IconList_UpDown $w -1]
    bind $data(canvas) <Down> [list ::gidtk::IconList_UpDown $w 1]
    bind $data(canvas) <Left> [list ::gidtk::IconList_LeftRight $w -1]
    bind $data(canvas) <Right> [list ::gidtk::IconList_LeftRight $w  1]
    bind $data(canvas) <Return> [list ::gidtk::IconList_ReturnKey $w]
    bind $data(canvas) <KeyPress> [list ::gidtk::IconList_KeyPress $w %A]
    bind $data(canvas) <${::acceleratorKey}-KeyPress> ";"
    bind $data(canvas) <Alt-KeyPress> ";"
    bind $data(canvas) <FocusIn> [list ::gidtk::IconList_FocusIn $w]
    #bind $data(canvas) <FocusOut> [list ::gidtk::IconList_FocusOut $w]
    if { $data(-multiple)} {
        bind $data(canvas) <${::acceleratorKey}-a> [list ::gidtk::IconList_CtrlA $w]
        bind $data(canvas) <${::acceleratorKey}-A> [list ::gidtk::IconList_CtrlA $w]
    }
    bind $data(canvas) <F2> [list ::gidtk::dialog::file::RenameFile $::gidtk::Priv(dialog)]
    bind $data(canvas) <Delete> [list gidtk::dialog::file::RemoveFile $::gidtk::Priv(dialog)]
    return $w
}

# ::gidtk::IconList_AutoScan --
#
# This procedure is invoked when the mouse leaves an entry window
# with button 1 down.  It scrolls the window up, down, left, or
# right, depending on where the mouse left the window, and reschedules
# itself as an "after" command so that the window continues to scroll until
# the mouse moves back into the window or the mouse button is released.
#
# Arguments:
# w -                     The IconList window.
#
proc ::gidtk::IconList_AutoScan {w} {
    upvar ::gidtk::$w data
    variable ::gidtk::Priv
    
    if {![winfo exists $w]} return
    set x $Priv(x)
    set y $Priv(y)
    
    if {$data(noScroll)} {
        return
    }
    if {$x >= [winfo width $data(canvas)]} {
        $data(canvas) xview scroll 1 units
    } elseif {$x < 0} {
        $data(canvas) xview scroll -1 units
    } elseif {$y >= [winfo height $data(canvas)]} {
        # do nothing
    } elseif {$y < 0} {
        # do nothing
    } else {
        return
    }
    
    IconList_Motion1 $w $x $y
    set Priv(afterId) [after 50 [list ::gidtk::IconList_AutoScan $w]]
}

# Deletes all the items inside the canvas subwidget and reset the IconList's state.
proc ::gidtk::IconList_DeleteAll {w} {
    upvar ::gidtk::$w data
    upvar ::gidtk::$w:itemList itemList
    upvar ::gidtk::$w:textList textList
    
    $data(canvas) delete all
    catch {unset data(selected)}
    catch {unset data(rect)}
    catch {unset data(list)}
    catch {array unset itemList}
    catch {array unset textList}
    set data(maxIW) 1
    set data(maxIH) 1
    set data(maxTW) 1
    set data(maxTH) 1
    set data(numItems) 0
    set data(curItem)  {}
    set data(noScroll) 1
    set data(selection) {}
    set data(index,anchor) ""
    $data(sbar) set 0.0 1.0
    $data(canvas) xview moveto 0
    
    dict remove $data(dict) {*}[dict keys $data(dict)]
    
    # remake the edit item
    
    $data(canvas) create window -100 -100 -anchor "nw" -window $data(edit) -tags edit
    set data(edit,request) 0
}


proc ::gidtk::IconList_RenameAnyHandler {w x y} {
    upvar ::gidtk::$w data
    
    if { $x < 0 || $x > [winfo width $data(edit)] ||
        $y < 0 || $y > [winfo height $data(edit)] } {
        IconList_TextAccept $w
        #IconList_UnmapRename $w
    }
}

# Invoked when changing an item's text and one of the events occur:
# <KeyPress-Return> or <FocusOut>

proc ::gidtk::IconList_TextAccept {w} {
    upvar ::gidtk::$w data
    upvar ::gidtk::$w:itemList itemList
    upvar ::gidtk::$w:textList textList
    
    # verify if this is a valid text
    
    if { !$data(edit,isactive) } {
        return
    }
    
    set item $data(item_edited)
    if { $data(edit,textvar) != $data(edit,undo) } {
        set status_rename [eval $data(-checkrenamecmd) \
                [list $data(edit,undo)] \
                [list $data(edit,textvar)] \
                [list $data(edit,idata)]]
        
        if { ![lindex $status_rename 0] } {
            # show the message returned
            GID_tk_messageBox -type ok -parent $w -message [lindex $status_rename 1]  \
                -icon warning -title [_ "Error"]
            focus -force $data(edit)
            return
        } else {
            set item $data(item_edited)
            # change the text info
            set item_info [lindex $data(list) $item]
            if { [llength $status_rename] > 1 } {
                set iTag [lindex $item_info 0]
                $data(canvas) itemconfigure $iTag -image [lindex $status_rename 1]
                lassign [$data(canvas) bbox $iTag] x1 y1 x2 y2
                set iW [expr {$x2 - $x1}]
                set iH [expr {$y2 - $y1}]
                if {$data(maxIW) < $iW} {
                    set data(maxIW) $iW
                }
                if {$data(maxIH) < $iH} {
                    set data(maxIH) $iH
                }
            }
            set tTag [lindex $item_info 1]
            $data(canvas) itemconfigure $tTag -text $data(edit,textvar)
            lassign [$data(canvas) bbox $tTag] x1 y1 x2 y2
            set tW [expr {$x2 - $x1}]
            set tH [expr {$y2 - $y1}]
            if {$data(maxTW) < $tW} {
                set data(maxTW) $tW
            }
            if {$data(maxTH) < $tH} {
                set data(maxTH) $tH
            }
            
            set rTag [lindex $item_info 2]
            set itemList($rTag) [lreplace $itemList($rTag) 2 2 $data(edit,textvar)]
            set textList($item) [string tolower $data(edit,textvar)]
            set data(edit,undo) ""
        }
    }
    
    # hide the edit box
    IconList_UnmapRename $w
    
    ::gidtk::IconList_Arrange $w
    ::gidtk::IconList_Selection $w clear 0 end
    ::gidtk::IconList_Selection $w set $item
    ::gidtk::IconList_Selection $w anchor $item
    ::gidtk::IconList_DrawSelection $w
}

proc ::gidtk::IconList_ShowHiddenOn { w } {
    upvar ::gidtk::$w data
    
    set data(options,show_hidden) 1
    IconList_Arrange $w
}

proc ::gidtk::IconList_ShowHiddenOff { w } {
    upvar ::gidtk::$w data
    
    set data(options,show_hidden) 0
    IconList_Arrange $w
}

proc ::gidtk::IconList_UnmapRename {w} {
    upvar ::gidtk::$w data
    
    bind $data(edit) <Any> ""
    grab release $data(edit)
    
    #place forget $data(edit)
    $data(canvas) coords edit -100 -100
    
    set data(edit,isactive) 0
    focus -force $data(edit,oldfocus)
    catch {grab $data(edit,oldgrab)}
    bind $data(edit) <Any-Button> $data(edit,oldbind)
}

# Invoked when changing an item's text and <KeyPress-Escape> occur:
#

proc ::gidtk::IconList_TextEscape {w} {
    upvar ::gidtk::$w data
    
    if { !$data(edit,isactive) } {
        return
    }
    set data(edit,undo) ""
    IconList_UnmapRename $w
    ::gidtk::IconList_Arrange $w
    ::gidtk::IconList_Selection $w clear 0 end
    set item $data(item_edited)
    ::gidtk::IconList_Selection $w set $item
    ::gidtk::IconList_Selection $w anchor $item
    ::gidtk::IconList_DrawSelection $w
}

proc ::gidtk::IconList_GetText {w item} {
    upvar ::gidtk::$w data
    upvar ::gidtk::$w:itemList itemList
    upvar ::gidtk::$w:textList textList
    
    if {$item == -1} {
        if { ![info exists data(sim,id)] } {
            return [list "" ""]
        }
        set item $data(sim,id)
    }
    set text ""
    set item_data ""
    set rTag [lindex [lindex $data(list) $item] 2]
    if { [info exists itemList($rTag)] } {
        lassign $itemList($rTag) iTag tTag text serial item_data
    }
    return [list $text $item_data]
}

# Change the text of an item
#

proc ::gidtk::IconList_RenameText {w item} {
    upvar ::gidtk::$w data
    upvar ::gidtk::$w:itemList itemList
    upvar ::gidtk::$w:textList textList
    
    if {$item == -1} {
        if {[catch {set item $data(sim,id)}]} {
            return
        }
    }
    set rTag [lindex [lindex $data(list) $item] 2]
    set data(item_edited) $item
    if { ![info exists itemList($rTag)] } {
        return
    }
    if { ![info exists data(canvas)] || ![winfo exists $data(canvas)]} {
        return
    }
    
    lassign $itemList($rTag) iTag tTag text serial idata    
    lassign [$data(canvas) bbox $tTag] x1 y1 x2 y2
    set data(edit,textvar) $text
    set data(edit,idata) $idata
    set data(edit,undo) $text
    set lstr [GidUtils::GetButtonWidthFromStrings $text]
    $data(edit) configure -font $data(font) -width $lstr
    set lbbox [expr $x2-$x1]
    while { [winfo reqwidth $data(edit)] < $lbbox } {
        incr lstr
        $data(edit) configure -width $lstr
    }
    $data(edit) selection range 0 end
    
    IconList_See $w $item
    
    $data(canvas) coords edit $x1 $y1
    
    set data(edit,isactive) 1
    set data(edit,oldgrab) [grab current]
    catch {grab $data(edit)}
    set data(edit,oldfocus) [focus -force $data(edit)]
    set data(edit,oldbind) [bind $data(edit) <Any-Button>]
    bind $data(edit) <Any-Button> +[list ::gidtk::IconList_RenameAnyHandler $w %x %y]
}

# Adds an icon into the IconList with the designated image and text
#
proc ::gidtk::IconList_Add { w image items args } {
    upvar ::gidtk::$w data
    upvar ::gidtk::$w:itemList itemList
    upvar ::gidtk::$w:textList textList
    
    array set opts {
        -hidden 0
    }
    array set opts $args
    foreach item $items {
        lassign $item text item_data
        set iTag [$data(canvas) create image 0 0 -image $image -anchor nw -tags [list icon $data(numItems) item$data(numItems)]]
        set tTag [$data(canvas) create text  0 0 -text $text -anchor nw -font $data(font) \
                -tags [list text $data(numItems) item$data(numItems)]]
        set rTag [$data(canvas) create rect  0 0 0 0 -fill "" -outline "" -tags [list rect $data(numItems) item$data(numItems)]]
        
        lassign [$data(canvas) bbox $iTag] x1 y1 x2 y2
        set iW [expr {$x2 - $x1}]
        set iH [expr {$y2 - $y1}]
        if {$data(maxIW) < $iW} {
            set data(maxIW) $iW
        }
        if {$data(maxIH) < $iH} {
            set data(maxIH) $iH
        }
        
        lassign [$data(canvas) bbox $tTag] x1 y1 x2 y2
        set tW [expr {$x2 - $x1}]
        set tH [expr {$y2 - $y1}]
        if {$data(maxTW) < $tW} {
            set data(maxTW) $tW
        }
        if {$data(maxTH) < $tH} {
            set data(maxTH) $tH
        }
        lappend data(list) [list $iTag $tTag $rTag $iW $iH $tW $tH $data(numItems) $opts(-hidden)]
        set itemList($rTag) [list $iTag $tTag $text $data(numItems) $item_data $opts(-hidden)]
        #         set textList($data(numItems)) [string tolower $text]
        set textList($data(numItems)) $text
        incr data(numItems)
    }
}

# Places the icons in a column-major arrangement.
#
proc ::gidtk::IconList_Arrange {w} {
    upvar ::gidtk::$w data
    
    if {![info exists data(list)]} {
        if {[info exists data(canvas)] && [winfo exists $data(canvas)]} {
            set data(noScroll) 1
            $data(sbar) configure -command ""
        }
        return
    }
    
    set W [winfo width        $data(canvas)]
    set H [winfo height $data(canvas)]
    set pad [expr {[$data(canvas) cget -highlightthickness] + \
            [$data(canvas) cget -borderwidth]}]
    if {$pad < 2} {
        set pad 2
    }
    
    incr W -[expr {$pad*2}]
    incr H -[expr {$pad*2}]
    
    set dx [expr {$data(maxIW) + $data(maxTW) + 8}]
    if {$data(maxTH) > $data(maxIH)} {
        set dy $data(maxTH)
    } else {
        set dy $data(maxIH)
    }
    incr dy 2
    set shift [expr {$data(maxIW) + 4}]
    
    set x [expr {$pad * 2}]
    set y [expr {$pad * 1}] ; # Why * 1 ?
    set usedColumn 0
    foreach sublist $data(list) {
        set usedColumn 1
        lassign $sublist iTag tTag rTag iW iH tW tH serial hidden
        if { $data(options,show_hidden) } {
            set state "normal"
        } else {
            set state [expr { $hidden ? "hidden" : "normal" }]
        }
        foreach t [list $iTag $tTag $rTag] {
            $data(canvas) itemconfigure $t -state $state
        }
        
        if { $state == "hidden" } continue
        
        set i_dy [expr {($dy - $iH)/2}]
        set t_dy [expr {($dy - $tH)/2}]
        
        $data(canvas) coords $iTag $x                         [expr {$y + $i_dy}]
        $data(canvas) coords $tTag [expr {$x + $shift}]  [expr {$y + $t_dy}]
        $data(canvas) coords $rTag $x $y [expr {$x+$dx}] [expr {$y+$dy}]
        
        incr y $dy
        if {($y + $dy) > $H} {
            set y [expr {$pad * 1}] ; # *1 ?
            incr x $dx
            set usedColumn 0
        }
    }
    
    if {$usedColumn} {
        set sW [expr {$x + $dx}]
    } else {
        set sW $x
    }
    
    if {$sW < $W} {
        $data(canvas) configure -scrollregion [list $pad $pad $sW $H]
        $data(sbar) configure -command ""
        $data(canvas) xview moveto 0
        set data(noScroll) 1
    } else {
        $data(canvas) configure -scrollregion [list $pad $pad $sW $H]
        $data(sbar) configure -command [list $data(canvas) xview]
        set data(noScroll) 0
    }
    
    set data(itemsPerColumn) [expr {($H-$pad)/$dy}]
    if {$data(itemsPerColumn) < 1} {
        set data(itemsPerColumn) 1
    }
    
    IconList_DrawSelection $w
}

# Gets called when the user invokes the IconList (usually by double-clicking
# or pressing the Return key).
#
proc ::gidtk::IconList_Invoke {w} {
    upvar ::gidtk::$w data
    if {$data(-command) != "" && [llength $data(selection)]} {
        uplevel #0 $data(-command)
    }
}

# ::gidtk::IconList_See --
#
#         If the item is not (completely) visible, scroll the canvas so that
#         it becomes visible.
proc ::gidtk::IconList_See {w rTag} {
    upvar ::gidtk::$w data
    upvar ::gidtk::$w:itemList itemList
    
    if {$data(noScroll)} {
        return
    }
    set sRegion [$data(canvas) cget -scrollregion]
    if { $sRegion == "" } {
        return
    }
    
    if { $rTag < 0 || $rTag >= [llength $data(list)] } {
        return
    }
    
    set bbox [$data(canvas) bbox item$rTag]
    set pad [expr {[$data(canvas) cget -highlightthickness] + \
            [$data(canvas) cget -borderwidth]}]
    
    set x1 [lindex $bbox 0]
    set x2 [lindex $bbox 2]
    incr x1 -[expr {$pad * 2}]
    incr x2 -[expr {$pad * 1}] ; # *1 ?
    
    set cW [expr {[winfo width $data(canvas)] - $pad*2}]
    
    set scrollW [expr {[lindex $sRegion 2]-[lindex $sRegion 0]+1}]
    set dispX [expr {int([lindex [$data(canvas) xview] 0]*$scrollW)}]
    set oldDispX $dispX
    
    # check if out of the right edge
    #
    if {($x2 - $dispX) >= $cW} {
        set dispX [expr {$x2 - $cW}]
    }
    # check if out of the left edge
    #
    if {($x1 - $dispX) < 0} {
        set dispX $x1
    }
    
    if {$oldDispX != $dispX} {
        set fraction [expr {double($dispX)/double($scrollW)}]
        $data(canvas) xview moveto $fraction
    }
}

proc ::gidtk::IconList_Btn1 {w x y {sim 0}} {
    upvar ::gidtk::$w data
    
    focus $data(canvas)
    set x [expr {int([$data(canvas) canvasx $x])}]
    set y [expr {int([$data(canvas) canvasy $y])}]
    set i [IconList_Index $w @${x},${y}]
    if {$i==""} return
    # verify if there is only one selected
    if { !$sim && [llength $data(selection)] == 1 && [lindex $data(selection) 0] == $i } {
        # enter in rename mode
        set data(edit,request) 1
        set data(edit,afterId) [after 1000 ::gidtk::IconList_RenameText $w $i]
    } else {
        IconList_Selection $w clear 0 end
        IconList_Selection $w set $i
        IconList_Selection $w anchor $i
        # this is only needed when the WM is configure to change the
        # focus as the mouse move
        #         if {$sim} {
            set data(sim,id) $i
            #         }
        
    }
}

proc ::gidtk::IconList_OnSelectCmd { w i }  {
    upvar ::gidtk::$w data
    if {$data(-selectcmd) != ""} {
        eval $data(-selectcmd) [list [IconList_Get $w $i]]
    }
}

proc ::gidtk::IconList_CtrlBtn1 {w x y} {
    upvar ::gidtk::$w data
    
    if { $data(-multiple) } {
        focus $data(canvas)
        set x [expr {int([$data(canvas) canvasx $x])}]
        set y [expr {int([$data(canvas) canvasy $y])}]
        set i [IconList_Index $w @${x},${y}]
        if {$i==""} return
        if { [IconList_Selection $w includes $i] } {
            IconList_Selection $w clear $i
        } else {
            IconList_Selection $w set $i
            IconList_Selection $w anchor $i
        }
    }
}

proc ::gidtk::IconList_ShiftBtn1 {w x y} {
    upvar ::gidtk::$w data
    
    if { $data(-multiple) } {
        focus $data(canvas)
        set x [expr {int([$data(canvas) canvasx $x])}]
        set y [expr {int([$data(canvas) canvasy $y])}]
        set i [IconList_Index $w @${x},${y}]
        if {$i==""} return
        set a [IconList_Index $w anchor]
        if { $a == "" } {
            set a $i
        }
        IconList_Selection $w clear 0 end
        IconList_Selection $w set $a $i
    }
}

proc ::gidtk::IconList_CtrlA { w} {
    upvar ::gidtk::$w data
    upvar ::gidtk::$w:itemList itemList
    upvar ::gidtk::$w:textList textList
    
    set wt [winfo parent $w]
    set class [winfo class $wt]
    while { $class != "TkFDialog" && $class != "TkChooseDir" } {
        set wt [winfo parent $wt]
        set class [winfo class $wt]
    }
    
    set dataName [winfo name $wt]
    upvar ::gidtk::dialog::file::$dataName data2
    
    if { $data(-multiple) } {
        focus $data(canvas)
        IconList_Selection $w clear 0 end
        # buscamos el primero que no sea directorio
        set n [llength $data(list)]
        for { set idx 0} { $idx < $n} { incr idx} {
            set file [file join $data2(selectPath) $textList($idx)]
            if { ![gid_filesystem::file isdirectory $file]} {
                break
            }
        }
        if { $idx < $n} {
            IconList_Selection $w set $idx $n
        }
    }
}

# Gets called on button-1 motions
#
proc ::gidtk::IconList_Motion1 {w x y} {
    upvar ::gidtk::$w data
    variable ::gidtk::Priv
    set Priv(x) $x
    set Priv(y) $y
    set x [expr {int([$data(canvas) canvasx $x])}]
    set y [expr {int([$data(canvas) canvasy $y])}]
    set i [IconList_Index $w @${x},${y}]
    if {$i==""} return
    if { !$data(-multiple) } {
        IconList_Selection $w clear 0 end
    }
    IconList_Selection $w set $i
}

proc ::gidtk::IconList_Btn3 {w X Y x y} {
    upvar ::gidtk::$w data
    
    IconList_Btn1 $w $x $y 1
    
    set file_picked "" ;#picked item
    set x [expr {int([$data(canvas) canvasx $x])}]
    set y [expr {int([$data(canvas) canvasy $y])}]
    set i [IconList_Index $w @${x},${y}]
    if { $i != "" } {
        lassign [::gidtk::IconList_GetText $w $i] file_picked item_data
    }
    ::gidtk::IconList_CreateMenuContextual $w $file_picked
    tk_popup $data(menu_context) $X $Y
}

proc ::gidtk::IconList_Double1 {w x y} {
    upvar ::gidtk::$w data
    
    if {$data(edit,request)} {
        # avoid mapping
        set data(edit,request) 0
        after cancel $data(edit,afterId)
    }
    if {[llength $data(selection)]} {
        IconList_Invoke $w
    }
}

proc ::gidtk::IconList_ReturnKey {w} {
    IconList_Invoke $w
}

proc ::gidtk::IconList_Leave1 {w x y} {
    variable ::gidtk::Priv
    
    set Priv(x) $x
    set Priv(y) $y
    IconList_AutoScan $w
}

proc ::gidtk::IconList_FocusIn {w} {
    upvar ::gidtk::$w data
    
    if {![info exists data(list)]} {
        return
    }
    
    if {[llength $data(selection)]} {
        IconList_DrawSelection $w
    }
}

proc ::gidtk::IconList_FocusOut {w} {
    IconList_Selection $w clear 0 end
}

# ::gidtk::IconList_UpDown --
#
# Moves the active element up or down by one element
#
# Arguments:
# w -                     The IconList widget.
# amount -          +1 to move down one item, -1 to move back one item.
#
proc ::gidtk::IconList_UpDown {w amount} {
    upvar ::gidtk::$w data
    
    if {![info exists data(list)]} {
        return
    }
    
    set curr [IconList_Curselection $w]
    if { [llength $curr] == 0 } {
        set i 0
    } else {
        set i [IconList_Index $w anchor]
        if {$i==""} return
        incr i $amount
    }
    if { !$data(options,show_hidden) } {
        while { $i >= 0 && $i < [IconList_Length $w] } {
            set hidden [lindex [IconList_Get $w $i] 2]
            if {$hidden} {
                incr i $amount
            } else {
                break
            }
        }
    }
    if { $i < 0 || $i >= [IconList_Length $w] } {
        return
    }
    
    IconList_Selection $w clear 0 end
    IconList_Selection $w set $i
    IconList_Selection $w anchor $i
    IconList_See $w $i
}

# ::gidtk::IconList_LeftRight --
#
# Moves the active element left or right by one column
#
# Arguments:
# w -                     The IconList widget.
# amount -          +1 to move right one column, -1 to move left one column.
#
proc ::gidtk::IconList_LeftRight {w amount} {
    upvar ::gidtk::$w data
    
    if {![info exists data(list)]} {
        return
    }
    
    set curr [IconList_Curselection $w]
    if { [llength $curr] == 0 } {
        set i 0
    } else {
        set i [IconList_Index $w anchor]
        if {$i==""} return
        incr i [expr {$amount*$data(itemsPerColumn)}]
    }
    if { !$data(options,show_hidden) } {
        while { $i >= 0 && $i < [IconList_Length $w] } {
            set hidden [lindex [IconList_Get $w $i] 2]
            if {$hidden} {
                incr i $amount
            } else {
                break
            }
        }
    }
    if { $i < 0 || $i >= [IconList_Length $w] } {
        return
    }
    IconList_Selection $w clear 0 end
    IconList_Selection $w set $i
    IconList_Selection $w anchor $i
    IconList_See $w $i
}

#----------------------------------------------------------------------
#                 Accelerator key bindings
#----------------------------------------------------------------------

# ::gidtk::IconList_KeyPress --
#
#         Gets called when user enters an arbitrary key in the listbox.
#
proc ::gidtk::IconList_KeyPress {w key} {
    variable Priv
    
    append Priv(ILAccel,$w) $key
    IconList_Goto $w $Priv(ILAccel,$w)
    catch {
        after cancel $Priv(ILAccel,$w,afterId)
    }
    set Priv(ILAccel,$w,afterId) [after 500 [list ::gidtk::IconList_Reset $w]]
}

proc ::gidtk::IconList_Goto {w text} {
    upvar ::gidtk::$w data
    upvar ::gidtk::$w:textList textList
    
    if {![info exists data(list)]} {
        return
    }
    
    if { $text == ""} {
        return
    }
    
    if {![info exists textList(0)]} {
        return
    }
    
    if {$data(curItem) == "" || $data(curItem) == 0} {
        set start 0
    } else {
        set start $data(curItem)
    }
    
    if { $::tcl_platform(platform) == "windows" } {
        set text [string tolower $text]
    }
    set theIndex -1
    set less 0
    set len [string length $text]
    set len0 [expr {$len-1}]
    set i $start
    
    # Search forward until we find a filename whose prefix is an exact match with $text
    while {1} {
        # RAMSAN       
        if { $::tcl_platform(platform) == "windows" } {
            set sub [string tolower [string range $textList($i) 0 $len0]]
        } else {
            set sub [string range $textList($i) 0 $len0]
        }
        if { $text == $sub } {
            set theIndex $i
            break
        }
        incr i
        if {$i == $data(numItems)} {
            set i 0
        }
        if {$i == $start} {
            break
        }
    }
    
    if {$theIndex > -1} {
        IconList_Selection $w clear 0 end
        IconList_Selection $w set $theIndex
        IconList_Selection $w anchor $theIndex
        IconList_See $w $theIndex
    }
}

proc ::gidtk::IconList_Reset {w} {
    variable Priv
    
    catch {unset Priv(ILAccel,$w)}
}

#----------------------------------------------------------------------
#
#                       F I L E         D I A L O G
#
#----------------------------------------------------------------------

namespace eval ::gidtk::dialog {}
namespace eval ::gidtk::dialog::file {
    #namespace import ::gidtk::msgcat::*    
}

# ::gidtk::dialog::file:: --
#
#         Implements the TK file selection dialog. This dialog is used when
#         the tk_strictMotif flag is set to false. This procedure shouldn't
#         be called directly. Call tk_getOpenFile or tk_getSaveFile instead.
#
# Arguments:
#         type                     "open" or "save"
#         args                     Options parsed by the procedure.
#

proc ::gidtk::dialog::file:: {type args} {
    variable ::gidtk::Priv
    set dataName __tk_filedialog
    upvar ::gidtk::dialog::file::$dataName data
    
    if {[info exists data(-previewcmd)]} {
        set had_preview [expr {$data(-previewcmd) != ""}]
    } else {
        set had_preview -1
    }
    
    ::gidtk::dialog::file::Config $dataName $type $args

    # initializing file browser dialog with initial dir/file in gid cloud folder takes some time...
    set busy_cursor 0
    if { [GidDataManager::isCloudPath $data(-initialdir)] || [GidDataManager::isCloudPath $data(-initialfile)] } {
        GidUtils::SetWarnLine ([_ "Accessing cloud folder"])
        set busy_cursor 1
        GidUtils::WaitState .gid.central.s
    }

    set has_preview [expr {$data(-previewcmd) != ""}]
    set has_more    [llength $data(-moreopt)]
    
    if { $data(-parent) == "." } {
        set w .$dataName
    } else {
        set w $data(-parent).$dataName
    }
    
    # (re)create the dialog box if necessary
    if {![winfo exists $w]} {
        ::gidtk::dialog::file::Create $w TkFDialog
    } else {
        if {($had_preview != $has_preview) || [winfo class $w] != "TkFDialog" } {
            destroy $w
            ::gidtk::dialog::file::Create $w TkFDialog
        } else {
            set data(dirCombo) $w.f1.hcb
            set data(upBtn) $w.f1.up
            set data(icons) $w.ficons.icons
            set data(ent) $w.f2.ent
            set data(typeMenuLab) $w.f2.labft
            set data(typeCombo) $w.f2.tcb
            set data(okBtn) $w.f2.ok
            set data(cancelBtn) $w.f2.cancel
            ::gidtk::dialog::file::SetSelectMode $w $data(-multiple)
            set data(ficons) $w.ficons
            
            if {$has_preview} {
                set data(previewChk) $w.f1.chk_pview
                if {![winfo exists $data(previewChk)]} {
                    if { ![info exists ::gidtk::dialog::file::${dataName}(preview)] } { set ::gidtk::dialog::file::${dataName}(preview) 0 }
                    ttk::checkbutton $data(previewChk) -text [_ "Preview"] \
                        -command "::gidtk::dialog::file::OnCheckPreview $w" \
                        -variable ::gidtk::dialog::file::${dataName}(preview)
                }
                grid $data(previewChk) -row 0 -column 4
                set data(thumbnail) $data(ficons).thumb
                if {![winfo exists $data(thumbnail)]} {
                    ttk::label $data(thumbnail) -borderwidth 1 -relief sunken  -width 40 -anchor center                
                }
                $data(thumbnail) configure -text "" -image ""  -width 40 -anchor center
                if {![info exists data(preview)]} {
                    if {[info exists ::GidPriv(GiDTkFBoxPreview)]} {
                        set data(preview) $::GidPriv(GiDTkFBoxPreview)
                    } else {
                        set data(preview) 1
                    }
                }
                OnCheckPreview $w
            } else {
                array unset data previewChk
                if {[winfo exists $w.f1.chk_pview]} {
                    if {[llength [grid info $w.f1.chk_pview]]} {
                        grid remove $w.f1.chk_pview
                    }
                }
                array unset data thumbnail
                if {[winfo exists $data(ficons).thumb]} {
                    if {[lsearch [$data(ficons) panes] $data(ficons).thumb]!=-1} {
                        $data(ficons) forget $data(ficons).thumb
                    }
                    set img [$data(ficons).thumb cget -image]
                    if {$img != ""} {
                        image delete $img
                    }
                    $data(ficons).thumb configure -image {} -text {}
                }
            }
            if {$has_more} {
                grid $data(moreopt,frame)
                ClearMoreOptions $w
                FillMoreOptions $w
            } else {
                catch {
                    grid remove $data(moreopt,frame)
                }
            }
        }
    }
    
    # Dialog boxes should be transient with respect to their parent,
    # so that they will always stay on top of their parent window.  However,
    # some window managers will create the window as withdrawn if the parent
    # window is withdrawn or iconified.  Combined with the grab we put on the
    # window, this can hang the entire application.  Therefore we only make
    # the dialog transient if the parent is viewable.
    
    if {[winfo viewable [winfo toplevel $data(-parent)]] } {
        wm transient $w $data(-parent)
    }
    
    # Add traces on the selectPath variable    
    trace add variable data(selectPath) write [list ::gidtk::dialog::file::SetPath $w]
    
    # Initialize the file types menu    
    if {[llength $data(-filetypes)]} {
        set data(typeCombo,filters) {}
        set data(typeCombo,labels) {}
        foreach type $data(-filetypes) {
            lassign $type title filter
            lappend data(typeCombo,labels) $title
            lappend data(typeCombo,filters) $filter
        }
        #::gidtk::dialog::file::SetFilter $w [lindex [lindex $data(-filetypes) 0] 1] 1
        set len_labels [llength $data(typeCombo,labels)]
        TTKComboBox_configure $data(typeCombo) -state readonly -height [expr $len_labels>5?5:$len_labels] \
            -values $data(typeCombo,filters) -labels $data(typeCombo,labels)
        $data(typeMenuLab) configure -state normal
        set data(typeCombo,textvar) [lindex $data(typeCombo,filters) 0]
    } else {
        set data(filter) "*"        
        TTKComboBox_configure $data(typeCombo) -state disabled -takefocus 0
        $data(typeMenuLab) configure -state disabled
    }
    ::gidtk::dialog::file::UpdateWhenIdle $w
    
    # Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display and de-iconify it.
    
    if {$data(-previewcmd) == ""} {
        set geomname PrePostFileDialogWindowGeom
    } else {
        set geomname PrePostFileDialogPreviewWindowGeom
    }
    if {[info exists ::GidPriv($geomname)]} {
        set newgeometry [lindex $::GidPriv($geomname) 0]
        if { [IsValidWmGeometry $newgeometry] } {
            # wm geometry $w $newgeometry
            WmGidGeom $w $newgeometry
        }
    } else {
        set ::GidPriv($geomname) ""
        ::tk::PlaceWindow $w widget $data(-parent)
    }
    if { $::GIDTKFBOX_WITHDRAWN } {
        wm state $w normal
    }
    update idletasks
    wm title $w $data(-title)
    
    set is_iconic [GidUtils::IsStateIconic]
    if { $is_iconic } {
        focus $data(ent)
        #grab $w ;#if grab when iconic then is not possible to deiconify GiD in Windows !!!
    } else {
        # Set a grab and claim the focus too.
        ::tk::SetFocusGrab $w $data(ent)
    }
    $data(ent) delete 0 end
    $data(ent) insert 0 $data(selectFile)
    $data(ent) selection range 0 end
    $data(ent) icursor end
        
    # initializing file browser dialog with initial dir/file in gid cloud folder takes some time...
    if { $busy_cursor } {
        set busy_cursor 0
        GidUtils::EndWaitState .gid.central.s
    }

    # Wait for the user to respond, then restore the focus and
    # return the index of the selected button.        Restore the focus
    # before deleting the window, since otherwise the window manager
    # may take the focus away so we can't redirect it.        Finally,
    # restore any grab that was in effect.

    vwait ::gidtk::Priv(selectFilePath)
    
    if { [winfo exists $w] } {
        if { $is_iconic } {
            destroy $w
        } else {
            ::tk::RestoreFocusGrab $w $data(ent) withdraw
        }
    }
    
    # Cleanup traces on selectPath variable
    #
    
    foreach idx {selectPath selectPathCombo} {
        foreach trace [trace info variable data($idx)] {
            trace remove variable data($idx) [lindex $trace 0] [lindex $trace 1]
        }
    }
    if { [winfo exists $w] } {
        set ::GidPriv($geomname) [winfo geometry $w]
    }
    if {$data(-previewcmd) != ""} {
        set ::GidPriv(GiDTkFBoxPreview) $data(preview)
    }
    if { [winfo exists $w] } {
        destroy $w
    }
    return $Priv(selectFilePath)
}

# ::gidtk::dialog::file::Config --
#
#         Configures the TK filedialog according to the argument list
#
proc ::gidtk::dialog::file::Config {dataName type argList} {
    upvar ::gidtk::dialog::file::$dataName data
    
    set data(type) $type
    
    # 0: Delete all variable that were set on data(selectPath) the
    # last time the file dialog is used. The traces may cause troubles
    # if the dialog is now used with a different -parent option.
    
    foreach idx {selectPath selectPathCombo} {
        foreach trace [trace info variable data($idx)] {
            trace remove variable data($idx) [lindex $trace 0] [lindex $trace 1]
        }
    }
    
    # 1: the configuration specs
    # specs = {
    #    {-commandlineswitch resourceName ResourceClass defaultValue verifier}
    #    {....}
    # }
    #

    set specs {
        {-defaultextension "" "" ""}
        {-filetypes "" "" ""}
        {-initialdir "" "" ""}
        {-recentfolders "" "" ""}
        {-initialfile "" "" ""}
        {-parent "" "" "."}
        {-title {} {} {}}
        {-dirprojectext "" "" ""}
        {-previewcmd "" "" ""}
        {-moreopt "" "" ""}
        {-category "" "" "file"}
    }
    
    # The "-multiple" option is only available for the "open" file dialog.
    #
    if { $type == "open" } {
        lappend specs {-multiple "" "" "0"}
    }
    
    # 2: default values depending on the type of the dialog
    #
    if {![info exists data(selectPath)]} {
        # first time the dialog has been popped up
        set data(selectPath) [gid_filesystem::pwd]
        set data(selectFile) ""
    }
    
    # 3: parse the arguments
    #
    tclParseConfigSpec ::gidtk::dialog::file::$dataName $specs "" $argList
    
    if {$data(-title) == ""} {
        if { $type == "open" } {
            set data(-title) [_ "Open"]
        } else {
            set data(-title) [_ "Save as"]
        }
    }
    
    # 4: set the default directory and selection according to the -initial
    #         settings
    #
    if {$data(-initialdir) != ""} {
        # Ensure that initialdir is an absolute path name.
        if {[gid_filesystem::file isdirectory $data(-initialdir)]} {
            set old [gid_filesystem::pwd]
            gid_filesystem::cd $data(-initialdir)
            set data(selectPath) [gid_filesystem::pwd]
            gid_filesystem::cd $old
        } else {
            set data(selectPath) [gid_filesystem::pwd]
        }
    }
    set data(selectFile) $data(-initialfile)
    
    # 5. Parse the -filetypes option
    #
    set data(-filetypes) [::tk::FDGetFileTypes $data(-filetypes)]
    
    if {![winfo exists $data(-parent)]} {
        error "bad window path name '$data(-parent)'"
    }
    
    # Set -multiple to a one or zero value (not other boolean types
    # like "yes") so we can use it in tests more easily.
    if { $type == "save" } {
        set data(-multiple) 0
    } elseif {$data(-multiple)} {
        set data(-multiple) 1
    } else {
        set data(-multiple) 0
    }

    # category tolower
    set data(-category) [ string tolower $data(-category)]
}

#RAMSAN adding. Text is beginning of a filename. Tries to find longer

proc ::gidtk::IconList_ReturnUniqueChar {w text} {
    global tcl_platform
    upvar ::gidtk::$w data
    upvar ::gidtk::$w:textList textList
    
    if {![info exists data(list)]} {
        return ""
    }
    
    if {![info exists textList(0)]} {
        return
    }
    
    if {[string length $text] == 0} {
        return ""
    }
    
    set start  0
    
    if { $::tcl_platform(platform) == "windows" } {
        set text [string tolower $text]
    }
    set theIndex -1
    set theIndexLast -1
    set less 0
    set len [string length $text]
    set len0 [expr $len-1]
    set i $start
    
    # Search forward until we find a filename whose prefix is an exact match with $text
    while 1 {
        if { $::tcl_platform(platform) == "windows" } {
            set sub [string tolower [string range $textList($i) 0 $len0]]
        } else {
            set sub [string range $textList($i) 0 $len0]
        }
        if { $text == $sub } {
            if { $theIndex == -1 } {
                set theIndex $i
            } else {
                set theIndexLast $i
            }
        } elseif { $theIndex != -1 } { 
            break 
        }
        incr i
        if {$i == $data(numItems)} {
            set i 0
        }
        if {$i == $start} {
            break
        }
    }
    
    if { $theIndex == -1 } { return "" }
    if { $theIndexLast == -1 } {
        return [string range $textList($theIndex) $len end]
    }
    set leni [string length $textList($theIndex)]
    for { set i $len } { $i < $leni } { incr i } {
        set a [string index $textList($theIndex) $i]
        for { set j [expr $theIndex+1] } { $j <= $theIndexLast } { incr j } {
            if { $::tcl_platform(platform) == "windows" } {
                if { [string comp [string tolower $a] [string tolower [string index $textList($j) $i]]] != 0 } {
                    break
                }
            } else {
                if { [string comp $a [string index $textList($j) $i]] != 0 } {
                    break
                }
            }
        }
        if { $j <= $theIndexLast } { 
            break 
        }
    }
    return [string range $textList($theIndex) $len [expr $i -1]]
}

proc ::gidtk::dialog::file::ClearMoreOptions {w} {
    set dataName [lindex [split $w .] end]
    upvar ::gidtk::dialog::file::$dataName data
    
    foreach c [winfo children $data(moreopt,contents)] {
        destroy $c
    }
}

proc ::gidtk::dialog::file::FillMoreOptions {w} {
    set dataName [lindex [split $w .] end]
    upvar ::gidtk::dialog::file::$dataName data
    
    if { [winfo exists $data(moreopt,frame).lmore] } {
        $data(moreopt,frame).lmore configure -text [lindex $data(-moreopt) 0]
    }
    
    set dirvar [eval [lindex $data(-moreopt) 1] $data(moreopt,contents)]
    upvar \#0 $dirvar dir
    if {[catch "set dir"]} {
        set dirvar ""
    }
    set data(moreopt,dirvar) $dirvar
    if {$dirvar == ""} {
        $data(moreopt,ab) configure -dir "right"
        grid remove $data(moreopt,contents)
    } else {
        if {$dir == "open"} {
            $data(moreopt,ab) configure -dir "bottom"
            grid $data(moreopt,contents)
        } else {
            $data(moreopt,ab) configure -dir "right"
            grid remove $data(moreopt,contents)
        }
    }
}

proc ::gidtk::dialog::file::OnClickMore {w} {
    set dataName [lindex [split $w .] end]
    upvar ::gidtk::dialog::file::$dataName data
    
    upvar \#0 $data(moreopt,dirvar) dir
    
    set f3 $data(moreopt,frame)
    if {[$f3.btnmore cget -dir] == "right"} {
        grid $f3.fmore
        set dir "open"
        $f3.btnmore configure -dir "bottom"
    } else {
        grid remove $f3.fmore
        set dir "close"
        $f3.btnmore configure -dir "right"
    }
}

proc ::gidtk::dialog::file::CreateMoreOptions {w} {
    set dataName [lindex [split $w .] end]
    upvar ::gidtk::dialog::file::$dataName data
    
    set CCBG [CCColorActivo [$w cget -background]]
    # f3: the frame with extra options
    set f3 [ttk::frame $w.f3]
    
    # -background $CCBG =>  -style BottomFrame.TFrame    
    
    set data(moreopt,frame) $f3
    
    ttk::label $f3.lmore -text [lindex $data(-moreopt) 0]
    set data(moreopt,ab) [ArrowButton $f3.btnmore -type button -background $CCBG \
            -command "::gidtk::dialog::file::OnClickMore $w" \
            -borderwidth 1 -relief solid -width 20 -height 20]
    set data(moreopt,contents) [ttk::frame $f3.fmore]
    
    set ::ff3 $f3
    grid $f3.btnmore -row 0 -column 0 -sticky w
    grid $f3.lmore -row 0 -column 1 -sticky w
    grid $f3.fmore -row 1 -column 0 -sticky snew -columnspan 3
    grid columnconfigure $f3 2 -weight 1
    grid $f3 -row 2 -column 0 -sticky "ew" -padx 4 -pady 4
    if {[llength $data(-moreopt)]} {
        FillMoreOptions $w
    } else {
        grid remove $f3
    }
}

# RAMSAN adding
proc ::gidtk::dialog::file::ImproveEntry { wentry w dataName a } {
    set result -1
    upvar #0 $dataName data
    if { [regexp {^[a-zA-Z0-9_.-]+$} $a] } {
        tk::EntryInsert $wentry $a
        set text [::gidtk::IconList_ReturnUniqueChar $w [$wentry get]]
        set idx [$wentry index insert]
        if { $text != "" && [$wentry index insert] == [$wentry index end] } {
            $wentry sel range end end
            $wentry insert end $text
            $wentry sel range $idx end
        }
        set result 0
    } else { 
        set result -1
    }
    return $result
}

proc ::gidtk::dialog::file::Create {w class} {
    package require hugecombo    
    
    set dataName [lindex [split $w .] end]
    upvar ::gidtk::dialog::file::$dataName data
    variable ::gidtk::Priv
    global tk_library
    set gid_cloud_path [GidDataManager::getConnectedPath]
    toplevel $w -class $class
    if { $::GIDTKFBOX_WITHDRAWN } {
        wm state $w withdrawn
    }
    set ::gidtk::Priv(dialog) $w
    # f1: the frame with the directory option menu    
    set f1 [ttk::frame $w.f1]
    ttk::label $f1.lab -text [_ "Directory"]:
    grid $f1.lab -row 0 -column 0 -sticky "ew"
    
    set data(dirList) [list]
    set data(dirFullList) [list]
    set data(dirCombo) [hugecombo::hugecombo $f1.hcb \
            -list [format "%s(dirList)" \
                ::gidtk::dialog::file::$dataName] \
            -compund 1 -height 7 -rowheight 18 \
            -state picklist \
            -borderwidth 1]
    grid $data(dirCombo) -row 0 -column 1 -sticky "ew"
    grid columnconfigure $f1 1 -weight 1
    
    set entry_cb [$data(dirCombo) component entry]
    $entry_cb configure -borderwidth 1 -insertbackground [$entry_cb cget -background] \
        -cursor [[$data(dirCombo) component button] cget -cursor]
    bind $entry_cb <1> "$data(dirCombo) list toggle"
    
    $f1.hcb.t.f configure -borderwidth 1 -relief solid
    
    bind [$data(dirCombo) component list] <<ListboxSelect>> \
        +[namespace code "SetPathCombo $w"]
    bind [$data(dirCombo) component entry] <Return> \
        +[namespace code "SetPathCombo $w"]
    set data(upBtn) [ttk::button $f1.up]
    grid $data(upBtn) -row 0 -column 2
    #As windows, just put the same image when open or normal folder    
    # ok.png can also be in_use.png
    foreach {fimg idximg} [ list \
            folder.png folderImage \
            folder_h.png folderhImage \
            file.png fileImage \
            file_h.png filehImage \
            upfolder.png updirImage \
            newfolder.png newfolderImage \
            folder.png  openfolderImage \
            folder_h.png openfolderImage \
            delete.png trashImage \
            rename.png renImage \
            disk.png diskImage \
            remoteFolder.png remoteFolderImage \
            folder-cloud.png cloudUnitImage \
            home.png homeImage \
            ok.png checkImage \
            blank.png noCheckImage \
            camera.png galleryImage\
            ] {
        if {![info exists Priv($idximg)]} {
            set Priv($idximg) [gid_themes::GetImage $fimg small_icons]
        }
    }
    $data(upBtn) configure -image $Priv(updirImage)
    set data(newBtn) [ttk::button $f1.new -image $Priv(newfolderImage)]
    grid $data(newBtn) -row 0 -column 3
    GidHelp $data(upBtn) [_ "Go up to parent directory"]
    GidHelp $data(newBtn) [_ "Create a new directory"]
    
    set next_col 4
    if { $gid_cloud_path != ""} {
        set data(cloudBtn) [ttk::button $f1.goCloud -image $Priv(cloudUnitImage)]
        grid $data(cloudBtn) -row 0 -column $next_col
        incr next_col
        GidHelp $data(cloudBtn) [_ "Go to cloud directory"]
        $data(cloudBtn) configure -command [list ::gidtk::dialog::file::GotoCloudPathCmd $w]
    }
    if { ![info exists data(-category)] } {
        set data(-category) "file"
    }
    if { $data(-category) == "imagefile"} {
        set data(galleryBtn) [ ttk::button $f1.goGallery -image $Priv(galleryImage)]
        grid $data(galleryBtn) -row 0 -column $next_col
        incr next_col
        GidHelp $data(galleryBtn) [_ "Go to project's gallery directory"]
        $data(galleryBtn) configure -command [list ::gidtk::dialog::file::GotoGalleryPathCmd $w]
    }

    if {$data(-previewcmd) != ""} {
        if {![info exists data(preview)]} {
            if {[info exists ::GidPriv(GiDTkFBoxPreview)]} {
                set data(preview) $::GidPriv(GiDTkFBoxPreview)
            } else {
                set data(preview) 1
            }
        }
        if { ![info exists ::gidtk::dialog::file::${dataName}(preview)] } { set ::gidtk::dialog::file::${dataName}(preview) 0 }
        set data(previewChk) [ttk::checkbutton $f1.chk_pview -text [_ "Preview"] \
                -command "::gidtk::dialog::file::OnCheckPreview $w" \
                -variable ::gidtk::dialog::file::${dataName}(preview)]
        grid $data(previewChk) -row 0 -column $next_col
        incr next_col
    }
    
    set data(w,chkHidden) $w.f1.chk_hidden
    set data(chkHidden) 0
    if { ![info exists ::gidtk::dialog::file::${dataName}(chkHidden)] } { set ::gidtk::dialog::file::${dataName}(chkHidden) 0 }
    ttk::checkbutton $data(w,chkHidden) -text [_ "Show hidden"] \
        -command "::gidtk::dialog::file::OnCheckShowHidden $w" \
        -variable ::gidtk::dialog::file::${dataName}(chkHidden)
    grid $data(w,chkHidden) -row 0 -column $next_col -sticky "snew"
    incr next_col
    
    # data(icons): the IconList that list the files and directories.
    if { $class == "TkFDialog" } {
        if { $data(-multiple) } {
            set fNameCaption [_ "File names"]:
        } else {
            set fNameCaption [_ "File name"]:
        }
        set fTypeCaption [_ "Files of type"]:
        set fCaptionWidth [GidUtils::MaxStringLength $fNameCaption $fTypeCaption]
        set fCaptionWidth [expr {$fCaptionWidth<14?14:$fCaptionWidth}]
        set iconListCommand [list ::gidtk::dialog::file::OkCmd $w]
    } else {
        # $class == "TkChooseDir" ?
        set fNameCaption [_ "Selection"]:
        set fCaptionWidth [string length $fNameCaption]
        set iconListCommand [list ::gidtk::dialog::file::chooseDir::DblClick $w]
    }
    
    set data(ficons) [ttk::panedwindow $w.ficons -orient horizontal]
    grid $w.ficons -row 1 -column 0 -sticky "snew"
    grid rowconfigure $w 1 -weight 1
    grid columnconfigure $w 0 -weight 1
    set data(icons) $w.ficons.icons
    # default sort criteria = dictionary
    # values can be dictionary alphabetically date
    if { ![info exists data(sortCriteria)]} {
        set data(sortCriteria) dictionary
    }
    
    # mark with check the corresponding sort criteria ...
    foreach crit [list alphabetically date dictionary size extension] {
        if { $data(sortCriteria) == $crit} {
            set checkImage($crit) $Priv(checkImage)
        } else {
            set checkImage($crit) $Priv(noCheckImage)
        }
    }
    
    set menucontext_list [list \
            [list -label [_ "Up directory"] \
                -cmd  [list ::gidtk::dialog::file::UpDirCmd $w] \
                -image $Priv(updirImage) \
                -bind <KeyPress-BackSpace>] \
            [list -label [_ "New directory"] \
                -cmd  [list ::gidtk::dialog::file::NewDirCmd $w] \
                -image $Priv(newfolderImage)] \
            --- \
            [list -label [_ "Sort alphabetically"] \
                -cmd  [list ::gidtk::dialog::file::SortAlphabeticallyCmd $w %M %I] \
                -image $checkImage(alphabetically)] \
            [list -label [_ "Sort by date ( newer first)"] \
                -cmd  [list ::gidtk::dialog::file::SortByDateCmd $w %M %I] \
                -image $checkImage(date)] \
            [list -label [_ "Sort by dictionary"] \
                -cmd  [list ::gidtk::dialog::file::SortByDictionaryCmd $w %M %I] \
                -image $checkImage(dictionary)] \
            [list -label [_ "Sort by file size ( bigger first)"] \
                -cmd  [list ::gidtk::dialog::file::SortBySizeCmd $w %M %I] \
                -image $checkImage(size)] \
            [list -label [_ "Sort by extension"] \
                -cmd  [list ::gidtk::dialog::file::SortByExtensionCmd $w %M %I] \
                -image $checkImage(extension)] \
            --- \
            [list -label [_ "Rename"] \
                -cmd [list ::gidtk::dialog::file::RenameFile $w] \
                -image $Priv(renImage)] \
            [list -label [_ "Delete"] \
                -cmd [list ::gidtk::dialog::file::RemoveFile $w] \
                -image $Priv(trashImage) \
                -bind <KeyPress-Delete>]]

    set ::gidtk::Priv(sort,list_entries) [ list 3 4 5 6 7]
    set ::gidtk::Priv(sort,idx_alphabetically) 3
    set ::gidtk::Priv(sort,idx_date) 4
    set ::gidtk::Priv(sort,idx_dictionary) 5
    set ::gidtk::Priv(sort,idx_size) 6
    set ::gidtk::Priv(sort,idx_extension) 7

    set next_menu_entry 2
    if { $gid_cloud_path != ""} {
        set item2add [list -label [_ "Go to cloud folder"] \
                -cmd  [list ::gidtk::dialog::file::GotoCloudPathCmd $w] \
                -image $Priv(cloudUnitImage)]
        set menucontext_list [linsert $menucontext_list $next_menu_entry $item2add]
        incr next_menu_entry
        incr ::gidtk::Priv(sort,idx_alphabetically)
        incr ::gidtk::Priv(sort,idx_date)
        incr ::gidtk::Priv(sort,idx_dictionary)
        incr ::gidtk::Priv(sort,idx_size)
        incr ::gidtk::Priv(sort,idx_extension)
        set ::gidtk::Priv(sort,list_entries) [ list \
            $::gidtk::Priv(sort,idx_alphabetically) \
            $::gidtk::Priv(sort,idx_date) \
            $::gidtk::Priv(sort,idx_dictionary) \
            $::gidtk::Priv(sort,idx_size) \
            $::gidtk::Priv(sort,idx_extension) \
        ]
    }

    if { $data(-category) == "imagefile"} {
        set item2add [list -label [_ "Go to project's gallery directory"] \
                -cmd  [list ::gidtk::dialog::file::GotoGalleryPathCmd $w] \
                -image $Priv(galleryImage)]
        set menucontext_list [linsert $menucontext_list $next_menu_entry $item2add]
        incr next_menu_entry
        incr ::gidtk::Priv(sort,idx_alphabetically)
        incr ::gidtk::Priv(sort,idx_date)
        incr ::gidtk::Priv(sort,idx_dictionary)
        incr ::gidtk::Priv(sort,idx_size)
        incr ::gidtk::Priv(sort,idx_extension)
        set ::gidtk::Priv(sort,list_entries) [ list \
            $::gidtk::Priv(sort,idx_alphabetically) \
            $::gidtk::Priv(sort,idx_date) \
            $::gidtk::Priv(sort,idx_dictionary) \
            $::gidtk::Priv(sort,idx_size) \
            $::gidtk::Priv(sort,idx_extension) \
        ]
    }

    ::gidtk::IconList $data(icons) \
        -selectcmd [list ::gidtk::dialog::file::OnSelectFile $w] \
        -command        $iconListCommand \
        -multiple         $data(-multiple) \
        -checkrenamecmd [list ::gidtk::dialog::file::OnRename $w] \
        -menucontext $menucontext_list
    
    $data(ficons) add $data(icons) -weight 1
    set data(thumbnail) [ttk::label $data(ficons).thumb \
            -borderwidth 1 -relief sunken -width 40 -anchor center]
    
    if {$data(-previewcmd) != ""} {
        OnCheckPreview $w
    }
    bind $data(icons) <<ListboxSelect>> \
        [list ::gidtk::dialog::file::ListBrowse $w]
    
    CreateMoreOptions $w
    
    # f2: the frame with the OK button and the "file name" field, file type and cancel    
    set f2 [ttk::frame $w.f2 -style BottomFrame.TFrame]
    ttk::label $f2.labfn -text $fNameCaption -anchor e -style BottomFrame.TLabel
    set data(ent) [ttk::entry $f2.ent -style BottomFrame.TEntry]
    
    # The font to use for the icons. The default Canvas font on Unix is just deviant.
    set ::gidtk::$w.ficons.icons(font) [$data(ent) cget -font]
    
    # RAMSAN adding
    bind $data(ent) <KeyPress> " \
        if { \[::gidtk::dialog::file::ImproveEntry %W $data(icons) $dataName %A] ==0 } { break }"
    
    #Anna: necessary to enable copy & paste in macOS. We want to avoid to run the ImproveEntry function so we have to assign "something" to the bind. Working for Aqua
    if { $::tcl_platform(os) == "Darwin"} {
        bind $data(ent) <Mod1-KeyPress> {set macos 0}
    }
    
    
    # Make the file types bits only if this is a File Dialog
    if { $class == "TkFDialog" } {
        # The "File of types:" label needs to be grayed-out when -filetypes are not specified. The label widget does not support
        # grayed-out text on monochrome displays. Therefore, we have to use a button widget to emulate a label widget (by setting its bindtags)
        set data(typeMenuLab) [ttk::label $f2.labft -text $fTypeCaption -style BottomFrame.TLabel]
        bindtags $data(typeMenuLab) [list $data(typeMenuLab) Label [winfo toplevel $data(typeMenuLab)] all]
        set var_name ::gidtk::dialog::file::${dataName}(typeCombo,textvar)
        set $var_name ""
        set data(typeCombo) [TTKComboBox $f2.tcb -takefocus 1 -borderwidth 1 -textvariable $var_name -values "" -labels "" \
                -modifycmd [list ::gidtk::dialog::file::SetFilterCombo $w $var_name]]
        bind $data(typeMenuLab) <<AltUnderlined>> [list focus $data(typeCombo)]  
    }
    
    # the okBtn is created after the typeMenu so that the keyboard traversal is in the right order
    set width [GidUtils::GetButtonWidthFromStrings [_ "Ok"] [_ "Cancel"]]
    set data(okBtn) [ttk::button $f2.ok -text [_ "Ok"] -style BottomFrame.TButton -width $width]
    set data(cancelBtn) [ttk::button $f2.cancel -text [_ "Cancel"] -style BottomFrame.TButton -width $width]
    
    grid $f2.labfn -row 0 -column 0 -sticky "w"
    grid $f2.ent -row 0 -column 1 -sticky "ew"
    grid $data(okBtn) -row 0 -column 2 -padx 4
    if { $class == "TkFDialog" } {
        grid $data(typeMenuLab) -row 1 -column 0 -sticky "w"
        grid $data(typeCombo) -row 1 -column 1 -sticky "ew"
    }
    grid $data(cancelBtn) -row 1 -column 2 -padx 4
    
    grid $f1 -row 0 -column 0 -sticky "ew"
    grid $f2 -row 3 -column 0 -sticky "ew"
    
    grid columnconfigure $f2 1 -weight 1
    
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure         $w 1 -weight 1
    
    # Set up the event handlers that are common to Directory and File Dialogs    
    wm protocol $w WM_DELETE_WINDOW [list ::gidtk::dialog::file::CancelCmd $w]
    bind $w <Destroy> [list +::gidtk::dialog::file::OnDestroyWindow %W $w]
    $data(upBtn) configure -command [list ::gidtk::dialog::file::UpDirCmd $w]
    $data(newBtn) configure -command [list ::gidtk::dialog::file::NewDirCmd $w]
    $data(cancelBtn) configure -command [list ::gidtk::dialog::file::CancelCmd $w]
    bind $w <KeyPress-Escape> [list $data(cancelBtn) invoke]
    bind $w <Alt-Key> [list tk::AltKeyInDialog $w %A]
    
    # Set up event handlers specific to File or Directory Dialogs    
    if { $class == "TkFDialog" } {
        bind $data(ent) <Return> [list ::gidtk::dialog::file::ActivateEnt $w]
        $data(okBtn) configure -command [list ::gidtk::dialog::file::OkCmd $w]
        bind $w <Alt-t> [format {
                if { [%s cget -state] == "normal" } {
                    focus %s
                }
            } $data(typeCombo) $data(typeCombo)]        
    } else {
        bind $data(ent) <Return> [list ::gidtk::dialog::file::chooseDir::OkCmd $w]
        $data(okBtn) configure -command [list ::gidtk::dialog::file::chooseDir::OkCmd $w]
        bind $w <Alt-s> [list focus $data(ent)]
        bind $w <Alt-o> [list $data(okBtn) invoke]
    }    
    bind $data(ent) <FocusIn>  [list ::gidtk::dialog::file::EntFocusIn $w]
    bind $data(ent) <FocusOut> [list ::gidtk::dialog::file::EntFocusOut $w]
    
    return
}

# ::gidtk::dialog::file::SetSelectMode --
#
#         Set the select mode of the dialog to single select or multi-select.
#
# Arguments:
#         w                  The dialog path.
#         multi                      1 if the dialog is multi-select; 0 otherwise.
#
# Results:
#         None.

proc ::gidtk::dialog::file::SetSelectMode {w multi} {
    set dataName __tk_filedialog
    upvar ::gidtk::dialog::file::$dataName data
    if { $multi } {
        set fNameCaption [_ "File names"]:
    } else {
        set fNameCaption [_ "File name"]:
    }
    set iconListCommand [list ::gidtk::dialog::file::OkCmd $w]
    ::gidtk::IconList_Config $data(icons) \
        [list -multiple $multi -command $iconListCommand \
            -checkrenamecmd [list ::gidtk::dialog::file::OnRename $w] \
            -selectcmd [list ::gidtk::dialog::file::OnSelectFile $w]]
    return
}

# ::gidtk::dialog::file::UpdateWhenIdle --
#
#         Creates an idle event handler which updates the dialog in idle
#         time. This is important because loading the directory may take a long
#         time and we don't want to load the same directory for multiple times
#         due to multiple concurrent events.
#

proc ::gidtk::dialog::file::CatchUpdate { w } {
    set err [catch {
        ::gidtk::dialog::file::Update $w
    } err_txt]
    if { $err } {
        ::gidtk::clearBusyCursor
        
        # ErrorWin "CatchUpdate error: \n$err_txt"
        ::gidtk::dialog::file::CancelCmd $w
        # GID-1696 this is a trick to solve the problem that gid issues an tcl error when accessing
        # $HOME/Desktop Downloads Documents RemovableMedia ... on macOS Catalina
        # we return as filename [list ____TCL_ERROR____ $err_txt]
        # so that afterwards (in tclfileP.tcl proc Browser-ramR) allows us to open ( grant us access to)
        # open a standard Tk dialog into these folders. Onlly once is needed for each folder / removable/network media
        if { $::tcl_platform(os) == "Darwin"} {
            set dataName __tk_filedialog
            upvar ::gidtk::dialog::file::$dataName data
            # restricted path to go = $data(selectPath)
            set ::gidtk::Priv(selectFilePath) [list ____TCL_ERROR____ $data(selectPath)]
        }
    }
}

proc ::gidtk::dialog::file::UpdateWhenIdle {w} {    
    #after idle [list ::gidtk::dialog::file::Update $w]
    upvar ::gidtk::dialog::file::[winfo name $w] data
    if {[info exists data(updateId)]} {
        return
    } else {
        # eventually windows and linux could also use CatchUpdate but ...
        if { $::tcl_platform(os) == "Darwin"} {
            set data(updateId) [after idle [list ::gidtk::dialog::file::CatchUpdate $w]]
        } else {
            set data(updateId) [after idle [list ::gidtk::dialog::file::Update $w]]
        }
    }    
}


# ::gidtk::dialog::file::InsertNames --
#
#        Insert into the iconlist the names given in allList, if a name
#        is in the list hiddenList then it is inserted with the icon
#        iconList[1] otherwise it is inserted with the icon
#        iconList[0].
#

proc ::gidtk::dialog::file::InsertNames {w_iconlist allList hiddenList icons {project_extension ""} } {
    set tmpList [list]
    set g_ishidden -1
    foreach item $allList {
        lassign $item name info
        if { $project_extension == ""} {
            set att_hidden [expr [lsearch -exact -index 0 $hiddenList $name] != -1]
        } else {
            # if it's a gid project, $name is only the rootname of the project, 
            # but to check if it's hidden, we need to check with the full name
            set att_hidden [expr [lsearch -exact -index 0 $hiddenList ${name}${project_extension} ] != -1]
        }
        set img [lindex $icons $att_hidden]
        #if { $::tcl_platform(platform) == "windows" && $info == ".lnk" } {
            # TODO_LNK: hay que agregar la flecha de link de windows
        #}
        if { [file extension $name] == ".zip" } {
            set img [gid_themes::GetImage "zip.png" small_icons]
        }
        ::gidtk::IconList_Add $w_iconlist $img [list $item] -hidden $att_hidden
    }
}

# ::gidtk::dialog::file::Update --
#
#         Loads the files and directories into the IconList widget. Also
#         sets up the directory option menu for quick access to parent
#         directories.
#

proc ::gidtk::dialog::file::Update {w} {
    # This proc may be called within an idle handler. Make sure that the
    # window has not been destroyed before this proc is called
    if {![winfo exists $w]} {
        return
    }
    set class [winfo class $w]
    if { $class != "TkFDialog" && $class != "TkChooseDir" } {
        return
    }
    set dataName [winfo name $w]
    upvar ::gidtk::dialog::file::$dataName data
    
    variable ::gidtk::Priv
    global tk_library
    unset -nocomplain data(updateId)
    
    set folder $Priv(folderImage)
    set file   $Priv(fileImage)
    set prjdir [gid_themes::GetImage $::GidPriv(FolderGiDBitmapName) small_icons]        
    set prjdirh [GidUtils::ImageNewTransparent $prjdir]
    
    
    set current_pwd [gid_filesystem::pwd]
    set current_pwd_prev $current_pwd
    if { $current_pwd != $data(selectPath) } {
        if {[catch { gid_filesystem::cd $data(selectPath) }]} {
            # We cannot change directory to $data(selectPath). $data(selectPath)
            # should have been checked before ::gidtk::dialog::file::Update is called, so
            # we normally won't come to here. Anyways, give an error and abort action.
            GID_tk_messageBox -type ok -parent $w -message [_ "Cannot change to the directory '%s'. Permission denied." $data(selectPath)] \
                -icon warning -title [_ "Error"]
            gid_filesystem::cd $current_pwd_prev
            return
        }
        set current_pwd [gid_filesystem::pwd]
    }

    GidHelp $data(dirCombo) $data(selectPath)

    #  # Turn on the busy cursor. BUG?? We haven't disabled X events, though,
    #  # so the user may still click and cause havoc ...
    #  set entCursor [$data(ent) cget -cursor]
    #  set dlgCursor [$w cget -cursor]
    #  $data(ent) configure -cursor watch
    #  $w configure -cursor watch
    #  update
    
    set cursor_changed [gidtk::setBusyCursor]
    
    # Update the Directory: option menu
    set list ""
    set dir ""
    set index -1
    set curdir_idx -1
    set left 15
    global tcl_platform
    
    if {![info exists data(dirCombo,index)]} {
        set data(dirCombo,index) -1
        set data(dirCombo,indexbase) -1
    }
    
    set selnative [file nativename $data(selectPath)]
    set gid_cloud_path [GidDataManager::getConnectedPath]
    set is_cloud_path [GidDataManager::isCloudPath $current_pwd]
    
    set ::nocache 1 ;#disabled code because has errors? 
    if { $::nocache || ![info exists data(cache,items_conf)] || ![info exists data(cache,dirList)] ||
        ![info exists data(cache,dirFullList)]} {
        set items_conf [list]
        set data(dirList) [list]
        set data(dirFullList) [list]
        # specials folders        
        if { $::tcl_platform(platform) == "windows" } {
            set names {downloads documents desktop}
        } else {
            set names {home downloads documents desktop publicshare}
        }

        foreach name $names {
            set pfolder [gid_filesystem::get_folder_special $name]
            if { $pfolder != "" && [gid_filesystem::file isdirectory $pfolder] } {
                lappend data(dirList) [file tail $pfolder]
                lappend data(dirFullList) $pfolder
                incr index
                if { $curdir_idx == -1 && $pfolder == $selnative && $data(dirCombo,index) <= $data(dirCombo,indexbase) } {
                    set curdir_idx $index
                }
                # iterate also in images index when available, by now set folder
                lappend items_conf [list $index $left [gid_themes::GetImage $name.png small_icons]]
            }
        }
        # NextCloud shared folders, if any
        # this is not being used as [GidDataManager::get_folders_nextcloud_shared] return ""
        # ?may be it was usefull when using an exteranal synchronization program to connect to nextcloud?
        foreach pfolder [GidDataManager::get_folders_nextcloud_shared] {
            if { $pfolder != "" && [gid_filesystem::file isdirectory $pfolder] } {
                lappend data(dirList) [file tail $pfolder]
                lappend data(dirFullList) $pfolder
                incr index
                if { $curdir_idx == -1 && $pfolder == $selnative && $data(dirCombo,index) <= $data(dirCombo,indexbase) } {
                    set curdir_idx $index
                }
                lappend items_conf [list $index $left [gid_themes::GetImage folder-cloud.png small_icons]]
            }
        }

        # add fake gid-cloud folder on linux / macOS
        # on windows later on a unit with the cloud image is added
        if { $::tcl_platform(platform) != "windows" } {
            if { $gid_cloud_path != ""} {
                set destination_mount $gid_cloud_path
                set pfolder $destination_mount
                if { [gid_filesystem::file isdirectory $pfolder] } {
                    lappend data(dirList) [file tail $pfolder]
                    lappend data(dirFullList) $pfolder
                    incr index
                    if {($curdir_idx == -1) && ($pfolder == $data(selectPath)) && ($data(dirCombo,index) <= $data(dirCombo,indexbase))} {
                        set curdir_idx $index
                    }
                    lappend items_conf [list $index $left [gid_themes::GetImage folder-cloud.png small_icons]]
                }
            }
        }

        

        # GiD special folder
        set pfolder [file nativename $::GIDDEFAULT]
        lappend data(dirList) [file tail $pfolder]
        lappend data(dirFullList) $pfolder
        incr index
        if { $curdir_idx == -1 && $pfolder == $selnative && $data(dirCombo,index) <= $data(dirCombo,indexbase) } {
            set curdir_idx $index
        }
        lappend items_conf [list $index $left [gid_themes::GetImage $::GidPriv(FolderGiDBitmapName) small_icons]]
        
        # if present add also current gid working project
        if { [GidUtils::ModelHasName] } {
            set pfolder [file nativename [GiD_Info project ModelName].gid]
            lappend data(dirList) [file tail $pfolder]
            lappend data(dirFullList) $pfolder
            incr index
            if { $curdir_idx == -1 && $pfolder == $selnative && $data(dirCombo,index) <= $data(dirCombo,indexbase) } {
                set curdir_idx $index
            }
            lappend items_conf [list $index $left [gid_themes::GetImage $::GidPriv(FolderGiDBitmapName) small_icons]]
        }
        
        set data(dirCombo,indexbase) $index
        
        if { $::tcl_platform(platform) == "windows" } {
            set volumes [givewinvolumes]
        } else {
            set volumes /
        }
        
        # check if data(selectPath) is from a remote server, if so, add it to the volumes list
        if { $::tcl_platform(platform) == "windows" } {
            if { [string index $selnative 0] == "\\"} {
                # later file split is used, and as
                # file split does split //server/folder1/folder2/file into
                # //server/folder1 folder2 file
                # and not into the expected //server folder1 folder 2
                # so we'll user //server/folder1 as 'volume' instead of '//server' as it should be ...
                if { [regexp {^(\\\\[^\\]*\\[^\\]*)\\.*$} $selnative matchingString remoteVolume] == 1} {
                    lappend volumes $remoteVolume
                }
            }
        }
        set data(cache,items_conf) $items_conf
        set data(cache,dirList) $data(dirList)
        set data(cache,dirFullList) $data(dirFullList)
        set data(cache,volumes) $volumes
    } else {
        [$data(dirCombo) component list] selection clear 0 end
        set items_conf $data(cache,items_conf)
        set data(dirList) $data(cache,dirList)
        set data(dirFullList) $data(cache,dirFullList)
        set volumes $data(cache,volumes)
        foreach i $data(dirFullList) {
            incr index
            if { $curdir_idx == -1 && $i == $selnative && $data(dirCombo,index) <= $data(dirCombo,indexbase) } {
                set curdir_idx $index
                break
            }
        }
        set index $data(dirCombo,indexbase)
    }
    
    if { [llength $data(-recentfolders)] } {
        lappend data(dirList) {*}$data(-recentfolders)
        lappend data(dirFullList) {*}$data(-recentfolders)
        foreach d $data(-recentfolders) {
            incr index
            set item_img $Priv(folderImage)
            if { $gid_cloud_path != "" } {
                if { [string first $gid_cloud_path $d] != -1} {
                    set item_img $Priv(cloudUnitImage)
                }
            }
            lappend items_conf [list $index 15 $item_img]
        }
    }
    set subdirs [file split $data(selectPath)]
    set volpath [file nativename [lindex $subdirs 0]]
    foreach vol $volumes {
        lappend data(dirList) $vol
        lappend data(dirFullList) $vol
        incr index
        set volImage $Priv(diskImage)
        if { $::tcl_platform(platform) == "windows" } {
            if { [string index $vol 0] == "\\"} {
                # in Windows we may have a remote server with a remote folder also as volume
                set volImage $Priv(remoteFolderImage)
            } elseif { $gid_cloud_path != "" && $gid_cloud_path == [string range $vol 0 1] } {
                # check if volume is a gid-cloud unit on windows only
                # above a dummy entry has been added for the cloud folder for linux / macos
                set volImage $Priv(cloudUnitImage)
            }
        }
        lappend items_conf [list $index $left $volImage]
        if { $curdir_idx == -1 && $vol == $selnative } {
            set curdir_idx $index
        } elseif { $curdir_idx == -1 && $volpath == $vol } {
            # unroll the path
            set dir $vol
            foreach subdir [lrange $subdirs 1 end] {
                set dir [file nativename [file join $dir $subdir]]
                lappend data(dirList) $subdir
                lappend data(dirFullList) $dir
                incr index
                incr left 15
                lappend items_conf [list $index $left $Priv(folderImage)]
                if {($curdir_idx == -1) && ($selnative == $dir)} {
                    set curdir_idx $index
                    lappend items_conf [list $index $left $Priv(openfolderImage)]
                } else {
                    lappend items_conf [list $index $left $Priv(folderImage)]
                }
            }
            set left 15
        }
    }
    
    # now configure the elements of the hugelist
    foreach item $items_conf {
        lassign $item index left image
        $data(dirCombo) itemconfigure $index -left $left -image $image
    }
    
    $data(dirCombo) list select $curdir_idx
    $data(dirCombo) load
    set len_dirlist [llength $data(dirList)]
    $data(dirCombo) configure -height [expr $len_dirlist>7?7:$len_dirlist]
    
    ::gidtk::IconList_DeleteAll $data(icons)
    
    # Make the dir list   
    set hiddenDirs [list]
    foreach d [gid_filesystem::glob -nocomplain -type {hidden d} *] {
        lappend hiddenDirs [list $d ""]
    }
    set normalDirs [list]
    foreach d [gid_filesystem::glob -nocomplain -type d *] {
        lappend normalDirs [list $d ""]
    }

    # in linux (vfs tcl package)
    # Check if CloudFolder is hidden as it should not be
    # all webdavbv files/folders are not hidden, but they appear when hidden files/directories are listed   
    if { $gid_cloud_path != ""} {
        if { $hiddenDirs == $normalDirs} {
            # we are inside $gid_cloud_path
            set hiddenDirs {}
        } else {
            set idx_cloud [lsearch $hiddenDirs [list [file tail $gid_cloud_path] ""]]
            if { $idx_cloud != -1} {
                set hiddenDirs [lreplace  $hiddenDirs $idx_cloud $idx_cloud]
            }
        }
    }

    # Make the file list if this is a File Dialog
    set hiddenFiles [list]
    set normalFiles [list]
     if { $class == "TkFDialog" } {
        foreach f [gid_filesystem::glob -nocomplain -type {hidden f} *] {
            if { $::tcl_platform(platform) == "windows" } {
                if { [file extension $f] == ".lnk"} {
                    if { [catch {set lnk_info [gid_filesystem::read_link $f 0]} msg] } {
                        W "Error when calling gid_filesystem::read_link : $msg"
                        continue
                    } else {
                        if { [gid_filesystem::file isdirectory $lnk_info] } {
                            array set attr_info [gid_filesystem::file attributes $f]
                            set rootf [file rootname $f]
                            if { $attr_info(-hidden) } {
                                lappend hiddenDirs [list $rootf .lnk]
                            } else {
                                lappend normalDirs [list $rootf .lnk]
                            }
                            continue
                        } else {
                            set f $lnk_info
                        }
                    }
                }
            }
            lappend hiddenFiles [list $f ""]
        }
        foreach f [gid_filesystem::glob -nocomplain -type f *] {
            if { $::tcl_platform(platform) == "windows" } {
                if { [file extension $f] == ".lnk"} {
                    if { [catch {set lnk_info [gid_filesystem::read_link $f 0]} msg] } {
                        W "Error when calling gid_filesystem::read_link : $msg"
                        continue
                    } else {
                        if { [gid_filesystem::file isdirectory $lnk_info] } {
                            array set attr_info [gid_filesystem::file attributes $f]
                            set rootf [file rootname $f]
                            if { $attr_info(-hidden) } {
                                lappend hiddenDirs [list $rootf .lnk]
                            } else {
                                lappend normalDirs [list $rootf .lnk]
                            }
                            continue
                        } else {
                            set f $lnk_info
                        }
                    }
                }
            }
            lappend normalFiles [list $f ""]
        }
     }

    # values can be dictionary alphabetically date
    set completeDirList [::gidtk::dialog::file::SortFileList $w [concat $hiddenDirs $normalDirs]]
    set dirList {}
    set prjdirList {}
    set prjdirListHidden {}
    foreach f $completeDirList {
        set name [join $f ""]
        if { $name == "." || $name == ".." } {
            continue
        }
        if { [ ::gidtk::IsProject $name $data(-dirprojectext)] } {
            lappend prjdirList [list [file rootname $name] [file extension $name]]
        } else {
            lappend dirList $f
        }
    }
    
    upvar ::gidtk::$data(icons) data2 ;#to get the canvas name
    set canvas $data2(canvas)
    $canvas configure -state disabled
    InsertNames $data(icons) $dirList $hiddenDirs [list $Priv(folderImage) $Priv(folderhImage)]
    if { $class == "TkFDialog" } {
        # in linux (vfs tcl package)
        # Check if CloudFolder is hidden as it should not be
        # all webdavbv files/folders are not hidden, but they appear when hidden files/directories are listed        
        if { $gid_cloud_path != ""} {
            if { $hiddenFiles == $normalFiles} {
                # we are inside $gid_cloud_path
                set hiddenFiles {}
            }
        }
        # values can be dictionary alphabetically date
        set completeFileList [::gidtk::dialog::file::SortFileList $w [concat $hiddenFiles $normalFiles]]
        if { $data(filter) == "*" } {
            set fileList {}
            foreach item $completeFileList {
                lappend fileList $item
            }
        } else {
            set fileList {}
            foreach item $completeFileList {
                set f [lindex $item 0]
                foreach pat $data(filter) {
                    if { [string match -nocase $pat $f] } {
                        lappend fileList $item
                        break
                    }
                }
            }
        }
        if {[llength $prjdirList]} {
            set currentpt ""
            set tmplist ""
            foreach item $prjdirList {
                set pt ""
                set rootf [lindex $item 0]
                set filename_geo [file join $current_pwd ${rootf}.gid ${rootf}.geo]
                if { $::GID_FILE_BROWSER_CLOUD_UNIT_FASTER } {
                    if { !$is_cloud_path} {
                        if { [gid_filesystem::file exists $filename_geo] } {
                            set pt [file join [lindex [gid_filesystem::read_info_geo_file $filename_geo] 2]]
                        }
                    } else {
                        set pt ""
                    }
                } else {
                    if { [gid_filesystem::file exists $filename_geo] } {
                        set pt [file join [lindex [gid_filesystem::read_info_geo_file $filename_geo] 2]]
                    }
                }
                if { $tmplist == "" } {
                    lappend tmplist $item
                    set currentpt $pt
                } else {
                    if {$pt == $currentpt } {
                        lappend tmplist $item
                    } else {
                        if { [info exists ::GidPriv(problemtype,ImageFileBrowser,$currentpt)] } {
                            set img $::GidPriv(problemtype,ImageFileBrowser,$currentpt)
                            set imgh [GidUtils::ImageNewTransparent $img]
                        } else {
                            set img $prjdir
                            set imgh $prjdirh
                        }
                        InsertNames $data(icons) $tmplist $hiddenDirs [list $img $imgh] $data(-dirprojectext)
                        set tmplist [list $item]
                        set currentpt $pt
                    }
                }
            }
            if { $tmplist != "" } {
                if { [info exists ::GidPriv(problemtype,ImageFileBrowser,$currentpt)] } {
                    set img $::GidPriv(problemtype,ImageFileBrowser,$currentpt)
                    set imgh [GidUtils::ImageNewTransparent $img]
                } else {
                    set img $prjdir
                    set imgh $prjdirh
                }
                InsertNames $data(icons) $tmplist $hiddenDirs [list $img $imgh] $data(-dirprojectext)
            }
        }
        if {[llength $fileList]} {
            InsertNames $data(icons) $fileList $hiddenFiles [list $file $Priv(filehImage)]  $data(-dirprojectext)
        }
    }
    ::gidtk::IconList_Arrange $data(icons)
    $canvas configure -state normal
    if { $current_pwd_prev != $current_pwd } {
        gid_filesystem::cd $current_pwd_prev
    }
    if { $class == "TkFDialog" } {
        # Restore the Open/Save Button if this is a File Dialog
        if { $data(type) == "open" } {
            set txt [_ "Open"]
        } else {
            set txt [_ "Save"]
        }
        $data(okBtn) configure -text $txt
        set max_width [GidUtils::GetButtonWidthFromStrings $txt]
        if {$max_width>[$data(okBtn) cget -width]} {
            $data(okBtn) configure -width $max_width
            $data(cancelBtn) configure -width $max_width
        }
    }
    if { $cursor_changed} {
        gidtk::clearBusyCursor
    }
    return 0
}
 
# ::gidtk::dialog::file::SetPathSilently --
#          Sets data(selectPath) without invoking the trace procedure
proc ::gidtk::dialog::file::SetPathSilently {w path} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    
    trace remove variable data(selectPath) write [list ::gidtk::dialog::file::SetPath $w]
    set data(selectPath) $path
    trace add variable data(selectPath) write [list ::gidtk::dialog::file::SetPath $w]
}


# This proc gets called whenever data(selectPathCombo) is set
proc ::gidtk::dialog::file::SetPathCombo {w} {
    if {[winfo exists $w]} {
        upvar ::gidtk::dialog::file::[winfo name $w] data
        set data(dirCombo,index) [$data(dirCombo) list cursel]
        set data(selectPath) [lindex $data(dirFullList) $data(dirCombo,index)]
    }
}

# This proc gets called whenever data(selectPath) is set
proc ::gidtk::dialog::file::SetPath {w name1 name2 op} {
    if {[winfo exists $w]} {
        upvar ::gidtk::dialog::file::[winfo name $w] data
        ::gidtk::dialog::file::UpdateWhenIdle $w
        # On directory dialogs, we keep the entry in sync with the currentdir.
        if { [winfo class $w] == "TkChooseDir" } {
            $data(ent) delete 0 end
            $data(ent) insert end $data(selectPath)
        }
    }
}

proc ::gidtk::dialog::file::SetFilterCombo { w name args } {    
    # upvar ::gidtk::dialog::file::[winfo name $w] data
    # set idx [eval [$data(typeCombo) comp list] curselection]
    # ::gidtk::dialog::file::SetFilter $w [lindex $data(typeCombo,filters) $idx] 1
    ::gidtk::dialog::file::SetFilter $w [set $name] 1
}

# This proc gets called whenever data(filter) is set
proc ::gidtk::dialog::file::SetFilter {w filter needsupdate} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    upvar ::gidtk::$data(icons) icons
    set data(filter) $filter
    # If we aren't using a default extension, use the one supplied by the filter.
    if {![info exists data(extUsed)]} {
        if {[string length $data(-defaultextension)]} {
            set data(extUsed) 1
        } else {
            set data(extUsed) 0
        }
    }
    
    if {!$data(extUsed)} {
        # Get the first extension in the list that matches {^\*\.\w+$} and remove all * from the filter.
        set index [lsearch -regexp $data(filter) {^\*\.\w+$}]
        if {$index >= 0} {
            set data(-defaultextension) [string trimleft [lindex $data(filter) $index] "*"]
        } else {
            # Couldn't find anything!  Reset to a safe default...
            set data(-defaultextension) ""
        }
    }
    
    if { $data(type) == "save" } {
        if { $data(-defaultextension) != "" } {
            set entry_filename [$data(ent) get] 
            if { $entry_filename != "" } {
                set file_ext [file extension $entry_filename]
                if { $file_ext != $data(-defaultextension) } {                 
                    $data(ent) delete 0 end
                    $data(ent) insert 0 [file rootname $entry_filename]$data(-defaultextension)
                }
            }
        }
    }
    
    if {$needsupdate} {
        $icons(sbar) set 0.0 0.0
        ::gidtk::dialog::file::UpdateWhenIdle $w
    }
}

# tk::dialog::file::ResolveFile --
#
#         Interpret the user's text input in a file selection dialog.
#         Performs:
#
#         (1) ~ substitution
#         (2) resolve all instances of . and ..
#         (3) check for non-existent files/directories
#         (4) check for chdir permissions
#
# Arguments:
#         context:  the current directory you are in
#         text:                the text entered by the user
#         defaultext: the default extension to add to files with no extension
#
# Return vaue:
#         [list $flag $directory $file]
#
#          flag = OK           : valid input
#               = PATTERN        : valid directory/pattern
#               = PATH             : the directory does not exist
#               = FILE             : the directory exists by the file doesn't
#                           exist
#               = CHDIR              : Cannot change to the directory
#               = ERROR              : Invalid entry
#
#          directory         : valid only if flag = OK or PATTERN or FILE
#          file                 : valid only if flag = OK or PATTERN
#
#         directory may not be the same as context, because text may contain
#         a subdirectory name
#
proc ::gidtk::dialog::file::ResolveFile {w context text defaultext} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    
    set current_pwd [gid_filesystem::pwd]
    
    set path [::gidtk::dialog::file::JoinFile $context $text]
    set path [::gidtk::dialog::file::GetFullFileName0 $path]
    # If the file has no extension, append the default.  Be careful not
    # to do this for directories, otherwise typing a dirname in the box
    # will give back "dirname.extension" instead of trying to change dir.    
    if {![gid_filesystem::file isdirectory $path]} {
        set ext [string tolower [file extension $path]]
        set append_ext 1
        foreach pattern $data(filter) {
            if { [string tolower [string trimleft $pattern *]] == $ext } {
                set append_ext 0
                break
            }                     
        }
        if { $append_ext } {
            set ext_2 [string tolower [file extension [file rootname $path]]]
            if { $ext_2 != "" } {
                #path has double extension, compare it with patterns with double extension
                set double_ext ${ext_2}${ext}
                #check patterns with double extension like *.gcode.gz
                foreach pattern $data(filter) {
                    if { [file extension [file rootname $pattern]] != "" } {
                        #pattern with double extension
                        if { [string tolower [string trimleft $pattern *]] == $double_ext } {
                            set append_ext 0
                            break
                        }            
                    }            
                }
            }
        }
        
        if { $append_ext } {
            if { $data(type) == "open" } {
                set path $path$defaultext
            } elseif { $data(type) == "save" } {
                if { $defaultext != "" } {
                    if { $ext == "" } {
                        #add required extension
                        set path $path$defaultext
                    } else {
                        if { $ext == $defaultext } {
                            #do nothing, already has this extension
                        } else {
                            if { $defaultext == ".gid" } {
                                #for the special case of create a .gid project folder to not remove possible name extensions
                                set path $path$defaultext
                            } else {
                                #remove previous extension and add the required one
                                set path [file rootname $path]$defaultext
                            }
                        }                
                    }
                }
                
            } else {
                #unexpected case
            }            
        }
    }
    
    
    if {[catch {gid_filesystem::file exists $path}]} {
        # This "if" block can be safely removed if the following code
        # stop generating errors.
        #
        #         gid_filesystem::file exists ~nonsuchuser
        #
        return [list ERROR $path ""]
    }
    
    if {[gid_filesystem::file exists $path]} {
        if {$::tcl_platform(platform) == "windows"} {
            #set path exactly (upper-lowercase) as saved
            set path [gid_filesystem::file attributes $path -longname]
        }
        if {[gid_filesystem::file isdirectory $path] && ![::gidtk::IsProject $path $data(-dirprojectext)]} {
            if {[catch {gid_filesystem::cd $path}]} {
                return [list CHDIR $path ""]
            }
            set directory [gid_filesystem::pwd]
            set file ""
            set flag OK
            gid_filesystem::cd $current_pwd
        } else {
            if {[catch {gid_filesystem::cd [file dirname $path]}]} {
                return [list CHDIR [file dirname $path] ""]
            }
            set directory [gid_filesystem::pwd]
            set file [file tail $path]
            set flag OK
            gid_filesystem::cd $current_pwd
        }
    } else {
        set dirname [file dirname $path]
        if {[gid_filesystem::file exists $dirname]} {
            if {[catch {gid_filesystem::cd $dirname}]} {
                return [list CHDIR $dirname ""]
            }
            set directory [gid_filesystem::pwd]
            set file [file tail $path]
            if {[regexp {[*]|[?]} $file]} {
                set flag PATTERN
            } else {
                set flag FILE
            }
            gid_filesystem::cd $current_pwd
        } else {
            set directory $dirname
            set file [file tail $path]
            set flag PATH
        }
    }
    
    return [list $flag $directory $file]
}


# Gets called when the entry box gets keyboard focus. We clear the selection
# from the icon list . This way the user can be certain that the input in the
# entry box is the selection.
proc ::gidtk::dialog::file::EntFocusIn {w} {
    upvar ::gidtk::dialog::file::[winfo name $w] data    
    if { [$data(ent) get] != "" } {
        $data(ent) selection range 0 end
        $data(ent) icursor end
    } else {
        $data(ent) selection clear
    }    
}

proc ::gidtk::dialog::file::EntFocusOut {w} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    
    $data(ent) selection clear
}


# Gets called when user presses Return in the "File name" entry.
proc ::gidtk::dialog::file::ActivateEnt {w} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    set text [string trimright [$data(ent) get]]
    if {$data(-multiple)} {
        # For the multiple case we have to be careful to get the file
        # names as a true list, watching out for a single file with a
        # space in the name.  Thus we query the IconList directly.
        
        set data(selectFile) ""
        # TODO_LNK: _Get retorna la etiqueta mas idata que debemos asumir
        # como una extension
        foreach item [::gidtk::IconList_Curselection $data(icons)] {
            set item_info [::gidtk::IconList_Get $data(icons) $item]
            lassign $item_info text idata
            set full "${text}${idata}"
            set flag [::gidtk::dialog::file::VerifyFileName $w $full]
            
        }
        
        # miguel: if we didn't get anything, let's try the old way:
        # perhaps it's only a directory where the user wants to go into,
        # or its a filter to be applied
        if { $data(selectFile) == ""} {
            set flag [::gidtk::dialog::file::VerifyFileName $w $text]
        }
    } else {
        set flag [::gidtk::dialog::file::VerifyFileName $w $text]
    }
}

# Verification procedure
#
proc ::gidtk::dialog::file::VerifyFileName {w filename} {
    
    upvar ::gidtk::dialog::file::[winfo name $w] data
    if {[regexp {[*]|[?]} $filename]} {
        set flag PATTERN
        set path $data(selectPath)
        SetFilter $w [list $filename] 0
        set file $data(filter)
    } else {
        lassign [::gidtk::dialog::file::ResolveFile $w $data(selectPath) $filename $data(-defaultextension)] flag path file
    }
    
    switch -- $flag {
        OK {
            if { $file ==  "" } {
                # user has entered an existing (sub)directory
                set data(selectPath) $path
                $data(ent) delete 0 end
            } else {
                ::gidtk::dialog::file::SetPathSilently $w $path
                if {$data(-multiple)} {
                    lappend data(selectFile) $file
                } else {
                    set data(selectFile) $file
                }
                ::gidtk::dialog::file::Done $w
            }
        }
        PATTERN {
            set data(filter) $file
            set data(selectPath) $path
            #set data(filter) $file
        }
        FILE {
            if { $data(type) == "open" } {
                GID_tk_messageBox -icon warning -type ok -parent $w \
                    -message [_ "File '%s' does not exist." [file join $path $file]] -title [_ "Error"]
                $data(ent) selection range 0 end
                $data(ent) icursor end
            } else {
                ::gidtk::dialog::file::SetPathSilently $w $path
                if {$data(-multiple)} {
                    lappend data(selectFile) $file
                } else {
                    set data(selectFile) $file
                }
                ::gidtk::dialog::file::Done $w
            }
        }
        PATH {
            GID_tk_messageBox -icon warning -type ok -parent $w \
                -message [_ "Directory '%s' does not exist." $path] -title [_ "Error"]
            $data(ent) selection range 0 end
            $data(ent) icursor end
        }
        CHDIR {
            GID_tk_messageBox -type ok -parent $w -message \
                [_ "Cannot change to the directory '%s'. Permission denied." $path] -icon warning -title [_ "Error"]
            $data(ent) selection range 0 end
            $data(ent) icursor end
        }
        ERROR {
            GID_tk_messageBox -type ok -parent $w -message \
                [_ "Invalid file name '%s'." $path] -icon warning  -title [_ "Error"]
            $data(ent) selection range 0 end
            $data(ent) icursor end
        }
    }
    return $flag
}

# Gets called when user presses the Alt-s or Alt-o keys.
proc ::gidtk::dialog::file::InvokeBtn {w key} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    if { [$data(okBtn) cget -text] == $key } {
        ::gidtk::ButtonInvoke $data(okBtn)
    }
}

# Gets called when user presses the "parent directory" button
proc ::gidtk::dialog::file::UpDirCmd {w} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    if { $data(selectPath) != "/" } {
        set data(selectPath) [file dirname $data(selectPath)]
    }
}

proc ::gidtk::dialog::file::GotoCloudPathCmd { w} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    set gid_cloud_path [GidDataManager::getConnectedPath]
    if { $gid_cloud_path != ""} {
        if { $::tcl_platform(platform) == "windows"} {
            # so that  ::gidtk::dialog::file::Update  works correctly
            # as givewinvolumes returns volumes with '\\' : c:\\ d:\\ ...
            # [GidDataManager::getConnectedPath] returns the unit without '\\'
            append gid_cloud_path \\
        }
        set data(selectPath) $gid_cloud_path
    }
}

proc ::gidtk::dialog::file::GotoGalleryPathCmd { w} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    set create_if_not_exists 1
    set gid_gallery_path [ GidUtils::GetDirectoryModelGallery $create_if_not_exists]
    if { $gid_gallery_path != ""} {
        set data(selectPath) $gid_gallery_path
    } else {
        set text [_ "project directory does not exists or is not writable."]
        set w $data(galleryBtn)
        # set xbase [ expr 8 + [ winfo rootx $w ]]
        # set ybase [ expr 8 + [ winfo rooty $w ]]
        lassign [ winfo pointerxy $w] xbase ybase
        PopupTransparentInfo $text $xbase $ybase $data(galleryBtn) 0 5000
        # GidHelp $data(galleryBtn) [_ "Go to project's gallery directory"]
    }
}

# command used to lsort files compared by date
proc ::gidtk::dialog::file::CompareFileDates { file1 file2} {
    if { [catch { set date1 [gid_filesystem::file mtime $file1]}]} {
        set date1 0
    }
    if { [catch { set date2 [gid_filesystem::file mtime $file2]}]} {
        set date2 0
    }
    # this is for newer first:
    if { $date2 < $date1} {
        set dif -1
    } elseif { $date2 > $date1} {
        set dif 1
    } else {
        #in case of equal date sort by file name
        set dif [string compare $file2 $file1]
    }
    return $dif
}

proc ::gidtk::dialog::file::SortByChangeDataMenuContextual { w sort_criteria_index } {
    variable ::gidtk::Priv
    upvar ::gidtk::$w.ficons.icons data
    set data_menu_list $data(-menucontext)
    # uncheck other checks and check selected criteria
    foreach i $::gidtk::Priv(sort,list_entries) {
        lset data_menu_list $i 5 $Priv(noCheckImage)
    }
    lset data_menu_list $sort_criteria_index 5 $Priv(checkImage)
    set data(-menucontext) $data_menu_list
    ::gidtk::dialog::file::UpdateWhenIdle $w
}


# Gets called when user presses the "Sort alphabetically" button
proc ::gidtk::dialog::file::SortAlphabeticallyCmd {w m idx} {
    variable ::gidtk::Priv
    upvar ::gidtk::dialog::file::[winfo name $w] data
    # values can be alphabetically date dictionary size extension
    set data(sortCriteria) alphabetically
    ::gidtk::dialog::file::SortByChangeDataMenuContextual $w $::gidtk::Priv(sort,idx_alphabetically)
}

# Gets called when user presses the "Sort by date" button
proc ::gidtk::dialog::file::SortByDateCmd {w m idx} {
    variable ::gidtk::Priv
    upvar ::gidtk::dialog::file::[winfo name $w] data
    # values can be alphabetically date dictionary size extension
    set data(sortCriteria) date
    ::gidtk::dialog::file::SortByChangeDataMenuContextual $w  $::gidtk::Priv(sort,idx_date)
}

# Gets called when user presses the "Sort by dictionary" button
proc ::gidtk::dialog::file::SortByDictionaryCmd {w m idx} {
    variable ::gidtk::Priv
    upvar ::gidtk::dialog::file::[winfo name $w] data
    # values can be alphabetically date dictionary size extension
    set data(sortCriteria) dictionary
    ::gidtk::dialog::file::SortByChangeDataMenuContextual $w  $::gidtk::Priv(sort,idx_dictionary)
}

# Gets called when user presses the "Sort by size" button
proc ::gidtk::dialog::file::SortBySizeCmd {w m idx} {
    variable ::gidtk::Priv
    upvar ::gidtk::dialog::file::[winfo name $w] data
    # values can be alphabetically date dictionary size extension
    set data(sortCriteria) size
    ::gidtk::dialog::file::SortByChangeDataMenuContextual $w $::gidtk::Priv(sort,idx_size)
}

# Gets called when user presses the "Sort by extension" button
proc ::gidtk::dialog::file::SortByExtensionCmd {w m idx} {
    variable ::gidtk::Priv
    upvar ::gidtk::dialog::file::[winfo name $w] data
    # values can be alphabetically date dictionary size extension
    set data(sortCriteria) extension
    ::gidtk::dialog::file::SortByChangeDataMenuContextual $w $::gidtk::Priv(sort,idx_extension)
}

# command used to lsort files compared by date
# http://wiki.tcl.tk/13978

###############################################################################
proc DirSize_Recurs {dir {level 0}} {
    set nextLevel [expr {$level + 1}]
    catch {gid_filesystem::cd $dir} res
    if {$res != ""} {
        return 0
    }
    ;#set dirlst [gid_filesystem::glob -nocomplain *]
    set dirlst [concat [gid_filesystem::glob -nocomplain *] [gid_filesystem::glob -type hidden -nocomplain *]]

    # in linux (vfs tcl package)
    # Check if CloudFolder is hidden as it should not be
    # all webdavbv files/folders are not hidden, but they appear when hidden files/directories are listed
    # and inside CloudFolder $dirlst will all be about duplicates as same dir/files appear in hidden and not hidden lists
    set gid_cloud_path [GidDataManager::getConnectedPath]
    if { $gid_cloud_path != ""} {
        set dirlst [lsort -unique $dirlst]
    }

    set size 0
    if { $level == 0} {
    }
    foreach ndir $dirlst {
        #set ndir [file join $dir $e]
        if {![gid_filesystem::file exists $ndir]} {
            continue
        }
        if {[gid_filesystem::file isdirectory $ndir]} {
            set s [DirSize_Recurs $ndir $nextLevel]
        } else {
            set s [gid_filesystem::file size $ndir]
        }
        set size [expr {wide($size + $s)}]
    }
    gid_filesystem::cd ..
    return $size
}

proc GetSizeOfFileOrDir { name} {
    return [gid_filesystem::file size $name]    
}

proc ::gidtk::dialog::file::CompareFileSizes { file1 file2} {
    set err [catch { set size1 [GetSizeOfFileOrDir $file1]} err_txt]
    if { $err} {
        set size1 0
    }
    set err [catch { set size2 [GetSizeOfFileOrDir $file2]} err_txt]
    if { $err} {
        set size2 0
    }
    # this is for bigger first:
    if { $size2 < $size1} {
        set dif -1
    } elseif { $size2 > $size1} {
        set dif 1
    } else {
        set dif 0     
    }
    return $dif
}

proc ::gidtk::dialog::file::SortFileList { w lst_files} {
    upvar ::gidtk::dialog::file::[winfo name $w] data

    set completeList $lst_files
    # why -unique has been added to lsort? it removes file entries !!!
    switch $data(sortCriteria) {
        dictionary {
            # This is the default sort option
            set completeList [lsort -dictionary -index 0 $lst_files]
        }
        alphabetically {
            set completeList [lsort -index 0 $lst_files]
        }
        date {
            set completeList [lsort -index 0 -command ::gidtk::dialog::file::CompareFileDates $lst_files]
        }
        size {
            set completeList [lsort -index 0 -command ::gidtk::dialog::file::CompareFileSizes $lst_files]
        }
        extension {
            # trick: 
            # build tmp_list { "extension+rootname" "original item in $lst_files"}
            # sort tmp_list
            # build back lst_files list using tmp_list order
            set tmp_list [ list]
            foreach item $lst_files {
                set filename [ lindex $item 0]
                set extension [ file extension $filename]
                set rootname [ file rootname $filename]
                # add a separator between extension and rootname
                # using :::: as non-legal character for paths
                set key2sort ${extension}::::${rootname}
                lappend tmp_list [ list $key2sort $item]
            }
            set sorted_list [ lsort -dictionary -index 0 $tmp_list]
            set completeList [ list]
            foreach item $sorted_list {
                lappend completeList [ lindex $item 1]
            }
        }
    }
    return $completeList
}

# called when a rename is about to be done
# return empty if its a valid newname, and perform the renaming
# return an message error when renaming can not be done
proc ::gidtk::dialog::file::OnRename {w oldname newname idata} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    variable ::gidtk::Priv
    
    set newname $newname$idata
    set oldname $oldname$idata    
    set _newname [::gidtk::dialog::file::JoinFile $data(selectPath) "$newname"]
    set _oldname [::gidtk::dialog::file::JoinFile $data(selectPath) "$oldname"]
    if { [gid_filesystem::file exists $_newname] } {
        return [list 0 [_ "File '%s' already exists.Can not rename." $newname]]
    }
    set is_writable [gid_filesystem::file_writable $data(selectPath)]
    if { !$is_writable } {
        return [list 0 [_ "You have not write permision on the directory '%s'. Permission denied." $data(selectPath)]]
    } else {
        # catch it for possible error not considered
        if { [catch {gid_filesystem::file rename $_oldname $_newname} msgcatch] } {
            return [list 0 [_ $msgcatch]]
        }
        if { [gid_filesystem::file isdirectory $_newname] && $data(-dirprojectext) != "" } {
            if { [file extension $_newname] == $data(-dirprojectext) } {
                # change the files names inside project directory
                set prjname [file rootname $newname]
                set old_prjname [file rootname $oldname]
                foreach f [gid_filesystem::glob -nocomplain [::gidtk::dialog::file::JoinFile $_newname *]] {
                    if { $old_prjname == [file rootname [file tail $f]] } {
                        gid_filesystem::file rename $f [::gidtk::dialog::file::JoinFile $_newname ${prjname}[file extension $f]]
                    }
                }
                if { [file extension $oldname] != $data(-dirprojectext) } {
                    return [list 1 [gid_themes::GetImage $::GidPriv(FolderGiDBitmapName) small_icons]]
                }
            } else {
                if { [file extension $oldname] == $data(-dirprojectext) } {
                    return [list 1 $Priv(folderImage)]
                }
            }
        }
    }
    return [list 1]
}


proc ::gidtk::dialog::file::RemoveFile {w} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    
    # TODO_LNK:
    # aqui hay que hacer remove usando la informacion de item_data
    lassign [::gidtk::IconList_GetText $data(icons) -1] text item_data
    if { $text == "" && $item_data == "" } {
        return 1
    }
    set fname [file nativename [file join $data(selectPath) "${text}${item_data}"]]
    if { [GidDataManager::isCloudPath $fname] && [GidDataManager::isSharedReadOnly [GidDataManager::TrimLeftCloudPath $fname]] } {
        WarnWin [_ "Cannot delete '%s' because it is a shared read-only location" $fname]
        return 0
    }
    if {$::tcl_platform(platform) == "windows"} {
        if { [catch { gid_filesystem::recycle_file $fname } msg] } {
            #e.g. "$msg == "SHFileOperation failed"
            set msg [_ "File cannot be sent to recycle bin"]
            GID_tk_messageBox -type ok -parent $w -message $msg -icon warning  -title [_ "Error"]
        }
    } else {
        if { [catch {gid_filesystem::file delete -force $fname }  msg] } {
            GID_tk_messageBox -type ok -parent $w -message $msg -icon warning  -title [_ "Error"]
        } else {
            $data(ent) delete 0 end
        }
        # doit in unix
    }
    ::gidtk::dialog::file::UpdateWhenIdle $w


    return 0
}

proc ::gidtk::dialog::file::RenameFile {w} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    
    ::gidtk::IconList_RenameText $data(icons) -1
}

proc ::gidtk::dialog::file::UnzipFile {w file_picked} {
    upvar ::gidtk::dialog::file::[winfo name $w] data    
          
    # TODO_LNK:
    # aqui hay que hacer remove usando la informacion de item_data
    lassign [::gidtk::IconList_GetText $data(icons) -1] text item_data
    if { $text == "" && $item_data == "" } {
        return 1
    }
    set fname [file join $data(selectPath) "${text}${item_data}"]
    if { [catch {gid_filesystem::unzip $fname } msg] } {
        GID_tk_messageBox -type ok -parent $w -message $msg -icon warning  -title [_ "Error"]
    } else {
        #$data(ent) delete 0 end
    }    
    ::gidtk::dialog::file::UpdateWhenIdle $w
    return 0
}

proc ::gidtk::dialog::file::ZipFile {w file_picked} {
    upvar ::gidtk::dialog::file::[winfo name $w] data    
          
    # TODO_LNK:
    # aqui hay que hacer remove usando la informacion de item_data
    lassign [::gidtk::IconList_GetText $data(icons) -1] text item_data
    if { $text == "" && $item_data == "" } {
        return 1
    }
    set fname [file join $data(selectPath) "${text}${item_data}"]
    if { [catch {gid_filesystem::zip $fname } msg] } {
        GID_tk_messageBox -type ok -parent $w -message $msg -icon warning  -title [_ "Error"]
    } else {
        #$data(ent) delete 0 end
    }    
    ::gidtk::dialog::file::UpdateWhenIdle $w
    return 0
}

# Gets called when user presses the "new directory" button
proc ::gidtk::dialog::file::NewDirCmd {w} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    upvar ::gidtk::$data(icons) icons
    # verify first if we can create a new folder
    set is_writable [gid_filesystem::file_writable $data(selectPath)]    
    if { !$is_writable } {
        GID_tk_messageBox -type ok -parent $w -message \
            [_ "You have not write permision on the directory '%s'. Permission denied." $data(selectPath)] \
            -icon warning  -title [_ "Error"]
        return
    }
    # create a unique new folder name
    set mcprefix [_ "New Folder "]
    set lpath [string length $data(selectPath)]
    set nf_list [gid_filesystem::glob -nocomplain [::gidtk::dialog::file::JoinFile $data(selectPath) ${mcprefix}*]]
    set nf_id_list [list]
    foreach nf $nf_list {
        regexp "${mcprefix}(.*)" [string range $nf $lpath end] {} _id
        if { [string is integer $_id] } {
            lappend nf_id_list $_id
        }
    }
    set nf_id_list [lsort -integer $nf_id_list]
    set id 1
    foreach _id $nf_id_list {
        if { $_id <= 0 } continue
        if { $id < $_id } break
        incr id
    }
    set FolderName ${mcprefix}$id
    # create the new folder
    gid_filesystem::file mkdir [::gidtk::dialog::file::JoinFile $data(selectPath) $FolderName]
    # add to the icon list
    ::gidtk::IconList_Add $data(icons) $::gidtk::Priv(folderImage) [list [list $FolderName {}]]
    ::gidtk::IconList_Arrange $data(icons)
    
    #::gidtk::IconList_See $w.icons [expr [::gidtk::IconList_Index $w.icons end]-1]
    
    # move to the end
    $icons(canvas) xview moveto 1
    # set the selection to the end don't know why should I compute this index that way?
    set idx_end [expr [::gidtk::IconList_Index $data(icons) end]-1]
    
    ::gidtk::IconList_Selection $data(icons) clear 0 end
    #::gidtk::IconList_Selection $data(icons) set $idx_end
    #::gidtk::IconList_Selection $data(icons) anchor $idx_end
    
    # and allow changing the name
    
    ::gidtk::IconList_RenameText $data(icons) $idx_end
    
    # creata an entry over the rectangle and send all key events when the folder is created.
    
    #{::gidtk::.__tk_filedialog.icons (list)}
}

# Join a file name to a path name. The "file join" command will break if the filename begins with ~
proc ::gidtk::dialog::file::JoinFile {path file} {
    if {[string match {~*} $file] && [gid_filesystem::file exists $path/$file]} {
        return [file join $path ./$file]
    } else {
        return [file join $path $file]
    }
}

# Gets called when user presses the "OK" button
proc ::gidtk::dialog::file::OkCmd {w} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    set filenames {}
    foreach item [::gidtk::IconList_Curselection $data(icons)] {
        set item_info [::gidtk::IconList_Get $data(icons) $item]
        lassign $item_info text idata
        lappend filenames ${text}${idata}
    }
    if {([llength $filenames] && !$data(-multiple)) || ($data(-multiple) && ([llength $filenames] == 1))} {
        set filename [lindex $filenames 0]
        set file [::gidtk::dialog::file::GetFullFileName $w $filename]
        if {[gid_filesystem::file isdirectory $file] && ![::gidtk::IsProject $file $data(-dirprojectext)]} {
            ::gidtk::dialog::file::ListInvoke $w [list $file]
            return
        }
    }
    ::gidtk::dialog::file::ActivateEnt $w
}

proc ::gidtk::dialog::file::OnDestroyWindow { W w } {
    if { $W != $w } return
    #reenter multiple times, one by toplevel child, only interest w
    upvar ::gidtk::dialog::file::[winfo name $w] data    
    unset -nocomplain data(updateId)
}

# Gets called when user presses the "Cancel" button
proc ::gidtk::dialog::file::CancelCmd {w} {
    upvar ::gidtk::dialog::file::[winfo name $w] data    
    variable ::gidtk::Priv
    ::gidtk::IconList_TextEscape $data(icons)
    update
    set ::gidtk::Priv(selectFilePath) ""
    unset -nocomplain data(updateId)
}

# Gets called when user browses the IconList widget (dragging mouse, arrow keys, etc)
proc ::gidtk::dialog::file::ListBrowse {w} {
    
    upvar ::gidtk::dialog::file::[winfo name $w] data
    
    set file_list {}
    foreach item [::gidtk::IconList_Curselection $data(icons)] {
        set item_info [::gidtk::IconList_Get $data(icons) $item]
        lassign $item_info text idata
        lappend file_list "${text}${idata}"
    }
    if {[llength $file_list] == 0} {
        return
    }
    
    if { [llength $file_list] > 1 } {
        set newfile_list {}
        foreach file $file_list {
            set fullfile [::gidtk::dialog::file::GetFullFileName $w $file]
            if { ![gid_filesystem::file isdirectory $fullfile] || ($data(-dirprojectext) != "" &&
                [file extension $fullfile] == $data(-dirprojectext)) } {
                lappend newfile_list $file
            }
        }
        set file_list $newfile_list
        set isDir 0
    } else {
        set file0 [lindex $file_list 0]
        set file [::gidtk::dialog::file::GetFullFileName $w $file0]
        set isDir 0
        if { [gid_filesystem::file isdirectory $file] } {
            if { $data(-dirprojectext) != "" && [file extension $file] == $data(-dirprojectext) } {
                #do not consider as dir in this tricky special case
                set isDir 0               
            } else {
                set isDir 1
            }       
        }
        # TODO: CUIDADO CON ESTA EXPRESION Y LA SELECCION MULTIPLE Y
        # FILE0 con espacios
        set file_list $file0
    }
    if {!$isDir} {
        $data(ent) delete 0 end
        $data(ent) insert 0 $file_list
        
        if { [winfo class $w] == "TkFDialog" } {
            if { $data(type) == "open" } {
                $data(okBtn) configure -text [_ "Open"]
            } else {
                $data(okBtn) configure -text [_ "Save"]
            }
        }
    } else {
        if { [winfo class $w] == "TkFDialog" } {
            $data(okBtn) configure -text [_ "Open"]
        }
    }
}

proc ::gidtk::dialog::file::GetFullFileName0 { filename } {    
    if { [catch {set lnk_info [gid_filesystem::read_link $filename 0]} msg] } {
        set lnk_info ""
        WarnWinText $msg
    }
    if { $lnk_info != "" } {
        return $lnk_info
    } else {
        return $filename
    }
}

proc ::gidtk::dialog::file::GetFullFileName { w filename } {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    set fullname [::gidtk::dialog::file::JoinFile $data(selectPath) $filename]
    return [::gidtk::dialog::file::GetFullFileName0 $fullname]
}

# Gets called when user invokes the IconList widget (double-click, Return key, etc)
proc ::gidtk::dialog::file::ListInvoke {w filenames} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    
    if {[llength $filenames] == 0} {
        return
    }
    
    set file [::gidtk::dialog::file::GetFullFileName $w [lindex $filenames 0]]
    
    set class [winfo class $w]
    if { $class == "TkChooseDir" || ([gid_filesystem::file isdirectory $file] && ![::gidtk::IsProject $file $data(-dirprojectext)])} {
        set current_pwd [gid_filesystem::pwd]
        if {[catch {gid_filesystem::cd $file}]} {
            GID_tk_messageBox -type ok -parent $w -message \
                [_ "Cannot change to the directory '%s'.Permission denied." $file] \
                -icon warning  -title [_ "Error"]
        } else {
            gid_filesystem::cd $current_pwd
            set data(selectPath) $file
        }
    } else {
        if {$data(-multiple)} {
            # OJO: hay que resolver aqui los lnk o hacerlo en Done
            set data(selectFile) $filenames
        } else {
            set data(selectFile) $file
        }
        ::gidtk::dialog::file::Done $w
    }
}

# ::gidtk::dialog::file::Done --
#
#         Gets called when user has input a valid filename.  Pops up a
#         dialog box to confirm selection when necessary. Sets the
#         gidtk::Priv(selectFilePath) variable, which will break the "vwait"
#         loop in ::gidtk::dialog::file:: and return the selected filename to the
#         script that calls tk_getOpenFile or tk_getSaveFile
#
proc ::gidtk::dialog::file::Done {w {selectFilePath ""}} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    variable ::gidtk::Priv
    
    if { $selectFilePath == "" } {
        if {$data(-multiple)} {
            set selectFilePath {}
            foreach f $data(selectFile) {
                lappend selectFilePath [::gidtk::dialog::file::JoinFile $data(selectPath) $f]
            }
        } else {
            set selectFilePath [::gidtk::dialog::file::JoinFile $data(selectPath) $data(selectFile)]
        }
        
        set Priv(selectFile)         $data(selectFile)
        set Priv(selectPath)         $data(selectPath)
        
        if { $data(type) == "save" } {
            if {[gid_filesystem::file exists $selectFilePath]} {
                set reply [GID_tk_messageBox -icon warning -type yesno -parent $w \
                        -message [_ "File '%s' already exists. Do you want to overwrite it?" $selectFilePath] \
                        -title [_ "Warning"]]
                if { $reply == "no" } {
                    return
                }
            }
        }
    }
    set Priv(selectFilePath) $selectFilePath
}

proc ::gidtk::dialog::file::OnCheckShowHidden { w } {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    if {$data(chkHidden)} {
        ::gidtk::IconList_ShowHiddenOn $data(icons)
    } else {
        ::gidtk::IconList_ShowHiddenOff $data(icons)
    }
}

proc ::gidtk::dialog::file::OnCheckPreview { w } {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    if {$data(preview)} {
        if { [lsearch [$data(ficons) panes] $data(thumbnail)] == -1 } {
            $data(ficons) add $data(thumbnail) -weight 0
        }        
    } else {
        if { [lsearch [$data(ficons) panes] $data(thumbnail)] != -1 } {
            $data(ficons) forget $data(thumbnail)
        }
    }
}

proc ::gidtk::dialog::file::OnSelectFile { w item_info } {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    
    if {$data(-previewcmd) != ""} {
        set img [$data(thumbnail) cget -image]
        if {$img != ""} {
            catch {image delete $img}
        }
        $data(thumbnail) configure -image {} -text [_ "No Preview"]
        lassign $item_info text idata
        set name "${text}${idata}"
        set full [::gidtk::dialog::file::GetFullFileName $w $name]
        set thumb [eval $data(-previewcmd) [list $full]]
        if { [gid_filesystem::file exists $thumb] } {
            set cursor_changed [gidtk::setBusyCursor]
            set image_data [gid_filesystem::read_file $thumb "" 1]
            if { [catch { set img [image create photo -data $image_data] } error_msg] } {
                # W $error_msg
            } else {
                set should_scale 0
                if {[image width $img] > 192} {
                    set should_scale 1
                } else {
                    if {[image height $img] > 144} {
                        set should_scale 1
                    }
                }
                if {$should_scale} {
                    $data(thumbnail) configure -text [_ "Computing preview ..."]
                    update idletask
                    after idle [string map "%L $data(thumbnail) %I $img" {
                            update
                            set new_img [GidUtils::ResizeImage %I 192 144]                            
                            image delete %I
                            %L configure -image $new_img
                        }]
                } else {
                    $data(thumbnail) configure -image $img
                }  
            }
            if { $cursor_changed} {
                gidtk::clearBusyCursor
            }   
        }
    }
}
