# gidchoosedir.tcl -- dialog box to select directory
# based on choosedir.tcl modified to work together with gidtkfbox.tcl (instead tkfbox.tcl)

# Make sure the tk::dialog namespace, in which all dialogs should live, exists
namespace eval ::gidtk::dialog {}
namespace eval ::gidtk::dialog::file {}

# Make the chooseDir namespace inside the dialog namespace
namespace eval ::gidtk::dialog::file::chooseDir {
}

# ::gidtk::dialog::file::chooseDir:: --
#
#        Implements the TK directory selection dialog.
#
# Arguments:
#        args                Options parsed by the procedure.
#
proc ::gidtk::dialog::file::chooseDir:: {args} {
    variable ::gidtk::Priv
    set dataName __tk_choosedir
    upvar ::gidtk::dialog::file::$dataName data
    Config $dataName $args
    
    # initializing file browser dialog with initial dir/file in gid cloud folder takes some time...
    set busy_cursor 0
    if { [ GidDataManager::isCloudPath $data(-initialdir)] } {
        GidUtils::SetWarnLine ([_ "Accessing cloud folder"])
        set busy_cursor 1
        GidUtils::WaitState .gid.central.s
    }

    if {$data(-parent) == "."} {
        set w .$dataName
    } else {
        set w $data(-parent).$dataName
    }
    
    # (re)create the dialog box if necessary
    #
    if {![winfo exists $w]} {
        ::gidtk::dialog::file::Create $w TkChooseDir
    } elseif {[winfo class $w] != "TkChooseDir"} {
        destroy $w
        ::gidtk::dialog::file::Create $w TkChooseDir
    } else {                
        set data(dirCombo) $w.f1.hcb
        set data(upBtn) $w.f1.up
        set data(icons) $w.ficons.icons
        set data(ent) $w.f2.ent
        set data(typeMenuLab) $w.f2.labft
        set data(typeCombo) $w.f2.tcb
        set data(okBtn) $w.f2.ok
        set data(cancelBtn) $w.f2.cancel
        set data(ficons) $w.ficons
        #set data(previewChk) $w.f1.chk_pview
        
    }
    
    # When using -mustexist, manage the OK button state for validity
    $data(okBtn) configure -state normal
    if {$data(-mustexist)} {
        $data(ent) configure -validate key -validatecommand [list ::gidtk::dialog::file::chooseDir::IsOK? $w %P]
    } else {
        $data(ent) configure -validate none
    }
    
    # Dialog boxes should be transient with respect to their parent,
    # so that they will always stay on top of their parent window.  However,
    # some window managers will create the window as withdrawn if the parent
    # window is withdrawn or iconified.  Combined with the grab we put on the
    # window, this can hang the entire application.  Therefore we only make
    # the dialog transient if the parent is viewable.
    
    if {[winfo viewable [winfo toplevel $data(-parent)]] } {
        wm transient $w $data(-parent)
    }
    # wm attributes $w -topmost 1
    GidUtils::WindowAboveGid $w
    
    trace add variable data(selectPath) write [list ::gidtk::dialog::file::SetPath $w]
    
    set data(filter) "*"
    set data(previousEntryText) ""
    ::gidtk::dialog::file::UpdateWhenIdle $w
    
    # Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display (Motif style) and de-iconify it.
    
    ::tk::PlaceWindow $w widget $data(-parent)
    wm title $w $data(-title)
    
    set is_iconic [GidUtils::IsStateIconic]
    if { $is_iconic } {
        focus $data(ent)
        #grab $w ;#if grab when iconic then is not possible to deiconify GiD in Windows !!!
    } else {
        # Set a grab and claim the focus too.
        ::tk::SetFocusGrab $w $data(ent)
    }
    
    $data(ent) delete 0 end
    $data(ent) insert 0 $data(selectPath)
    $data(ent) selection range 0 end
    $data(ent) icursor end
        
    # initializing file browser dialog with initial dir/file in gid cloud folder takes some time...
    if { $busy_cursor } {
        set busy_cursor 0
        GidUtils::EndWaitState .gid.central.s
    }

    # Wait for the user to respond, then restore the focus and
    # return the index of the selected button.  Restore the focus
    # before deleting the window, since otherwise the window manager
    # may take the focus away so we can't redirect it.  Finally,
    # restore any grab that was in effect.
    
    vwait ::gidtk::Priv(selectFilePath)
    
    if { [winfo exists $w] } {
        if { $is_iconic } {
            destroy $w
        } else {
            ::tk::RestoreFocusGrab $w $data(ent) withdraw
        }
    }
    
    # Cleanup traces on selectPath variable
    #
    
    foreach trace [trace info variable data(selectPath)] {
        trace remove variable data(selectPath) [lindex $trace 0] [lindex $trace 1]
    }
    
    # Return value to user
    #
    
    return $Priv(selectFilePath)
}

# ::gidtk::dialog::file::chooseDir::Config --
#
#        Configures the Tk choosedir dialog according to the argument list
#
proc ::gidtk::dialog::file::chooseDir::Config {dataName argList} {
    upvar ::gidtk::dialog::file::$dataName data
    
    # 0: Delete all variable that were set on data(selectPath) the
    # last time the file dialog is used. The traces may cause troubles
    # if the dialog is now used with a different -parent option.
    #
    foreach trace [trace info variable data(selectPath)] {
        trace remove variable data(selectPath) [lindex $trace 0] [lindex $trace 1]
    }
    
    # 1: the configuration specs
    #
    set specs {
        {-mustexist "" "" 0}
        {-initialdir "" "" ""}
        {-recentfolders "" "" ""}
        {-parent "" "" "."}
        {-title "" "" ""}        
        {-dirprojectext "" "" ""}
        {-previewcmd "" "" ""}
        {-moreopt "" "" ""}
    }
    
    # 2: default values depending on the type of the dialog
    #
    if {![info exists data(selectPath)]} {
        # first time the dialog has been popped up
        set data(selectPath) [gid_filesystem::pwd]
    }
    
    # 3: parse the arguments
    #
    tclParseConfigSpec ::gidtk::dialog::file::$dataName $specs "" $argList
    
    if {$data(-title) == ""} {
        set data(-title) [_ "Choose Directory"]
    }
    
    # Stub out the -multiple value for the dialog; it doesn't make sense for
    # choose directory dialogs, but we have to have something there because we
    # share so much code with the file dialogs.
    set data(-multiple) 0
    
    # 4: set the default directory and selection according to the -initial
    #    settings
    #
    if {$data(-initialdir) != ""} {
        # Ensure that initialdir is an absolute path name.
        if {[gid_filesystem::file isdirectory $data(-initialdir)]} {
            set old [gid_filesystem::pwd]
            gid_filesystem::cd $data(-initialdir)
            set data(selectPath) [gid_filesystem::pwd]
            gid_filesystem::cd $old
        } else {
            set data(selectPath) [gid_filesystem::pwd]
        }
    }
    
    if {![winfo exists $data(-parent)]} {
        return -code error -errorcode [list TK LOOKUP WINDOW $data(-parent)] \
            "bad window path name \"$data(-parent)\""
    }
}

# Gets called when user presses Return in the "Selection" entry or presses OK.
#
proc ::gidtk::dialog::file::chooseDir::OkCmd {w} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    
    # This is the brains behind selecting non-existant directories.  Here's
    # the flowchart:
    # 1.  If the icon list has a selection, join it with the current dir,
    #     and return that value.
    # 1a.  If the icon list does not have a selection ...
    # 2.  If the entry is empty, do nothing.
    # 3.  If the entry contains an invalid directory, then...
    # 3a.   If the value is the same as last time through here, end dialog.
    # 3b.   If the value is different than last time, save it and return.
    # 4.  If entry contains a valid directory, then...
    # 4a.   If the value is the same as the current directory, end dialog.
    # 4b.   If the value is different from the current directory, change to
    #       that directory.
    
#     set selection [$data(icons) selection get]
#     if {[llength $selection] != 0} {
#         set iconText [$data(icons) get [lindex $selection 0]]
#         set iconText [file join $data(selectPath) $iconText]
#         Done $w $iconText
#     } else {
#         set text [$data(ent) get]
#         if {$text == ""} {
#             return
#         }
#         set text [file join {*}[file split [string trim $text]]]
#         if {![gid_filesystem::file exists $text] || ![gid_filesystem::file isdirectory $text]} {
#             # Entry contains an invalid directory.  If it's the same as the
#             # last time they came through here, reset the saved value and end
#             # the dialog.  Otherwise, save the value (so we can do this test
#             # next time).
#             if {$text == $data(previousEntryText)} {
#                 set data(previousEntryText) ""
#                 Done $w $text
#             } else {
#                 set data(previousEntryText) $text
#             }
#         } else {
#             # Entry contains a valid directory.  If it is the same as the
#             # current directory, end the dialog.  Otherwise, change to that
#             # directory.
#             if {$text == $data(selectPath)} {
#                 Done $w $text
#             } else {
#                 set data(selectPath) $text
#             }
#         }
#     }

    set text [string trim [$data(ent) get]]
    set data(selectPath) $text
    ::gidtk::dialog::file::chooseDir::Done $w
    return
}

# Change state of OK button to match -mustexist correctness of entry
#
proc ::gidtk::dialog::file::chooseDir::IsOK? {w text} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    
    set ok [gid_filesystem::file isdirectory $text]
    $data(okBtn) configure -state [expr {$ok ? "normal" : "disabled"}]
    
    # always return 1
    return 1
}

proc ::gidtk::dialog::file::chooseDir::DblClick {w} {    
    upvar ::gidtk::dialog::file::[winfo name $w] data    
    set filenames {}
    foreach item [::gidtk::IconList_Curselection $data(icons)] {
        set item_info [::gidtk::IconList_Get $data(icons) $item]
        lassign $item_info text idata
        lappend filenames ${text}${idata}
    }
    if { [llength $filenames] == 1 } {
        set filename [lindex $filenames 0]
        set file [::gidtk::dialog::file::GetFullFileName $w $filename]
        if { [gid_filesystem::file isdirectory $file] } {
            ::gidtk::dialog::file::ListInvoke $w [list $file]                        
            return
        }
    }
}

# Gets called when user browses the IconList widget (dragging mouse, arrow
# keys, etc)
#
proc ::gidtk::dialog::file::chooseDir::ListBrowse {w text} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    
    if {$text == ""} {
        return
    }
    
    set file [::gidtk::dialog::file::JoinFile $data(selectPath) $text]
    $data(ent) delete 0 end
    $data(ent) insert 0 $file
}

# ::gidtk::dialog::file::chooseDir::Done --
#
#        Gets called when user has input a valid filename.  Pops up a
#        dialog box to confirm selection when necessary. Sets the
#        Priv(selectFilePath) variable, which will break the "vwait"
#        loop in tk_chooseDirectory and return the selected filename to the
#        script that calls tk_getOpenFile or tk_getSaveFile
#
proc ::gidtk::dialog::file::chooseDir::Done {w {selectFilePath ""}} {
    upvar ::gidtk::dialog::file::[winfo name $w] data
    variable ::gidtk::Priv
    
    if {$selectFilePath == ""} {
        set selectFilePath $data(selectPath)
    }
    if {$data(-mustexist) && ![gid_filesystem::file isdirectory $selectFilePath]} {
        return
    }
    set Priv(selectFilePath) $selectFilePath    
}
