
if { $::gid_start_mode != "server" } {
    error "this file must not be sourced with ::gid_start_mode=$::gid_start_mode"
}

set ::image_screen_updating 0
set ::image_screen_lost 0

proc gid_server_debug_print { text } {        
    # W "gid_server_debug_print $text"
    #GidUtils::DebugPrint $text
}

proc gid_server_debug_open { } {
    # W gid_server_debug_open
    #GidUtils::DebugOpen
}

proc gid_server_debug_close { } {
    # W gid_server_debug_close
    #GidUtils::DebugClose
}

#trick, define emtpy outside if to be found by auto_mkindex, and source the file if invoked
#these procs must be defined by all implemented procotols
proc gid_server_connect { } { }
proc gid_server_clients_eval { script } { }
proc gid_server_clients_eval_async { script } { }
proc gid_server_get_default_port { } { }

proc gid_server_get_host { } {
    if { $::gid_start_server_ip == "" } {
        set host localhost
    } else {
        set host $::gid_start_server_ip
    }
    return $host
}

proc gid_server_get_port { } {
    if { $::gid_start_port == "" } {
        set port [gid_server_get_default_port]
    } else {
        set port $::gid_start_port
    }
    return $port
}

if { $::gid_server_client_protocol == "comm" } {
    proc _gid_server_get_client_ids { } {
        package require comm
        #first id is self, omite it
        return [lrange [::comm::comm ids] 1 end]
    }

    proc gid_server_get_default_port { } {
        return 6900 ;#comm default port
    }
  
    proc gid_server_connect { } {
        gid_server_debug_open
        package require comm
        set port [gid_server_get_port]
        ::comm::comm configure -port $port -local $::gid_start_local
    }

    #syncronous (modal)
    proc gid_server_clients_eval { script } {
        if { 0 } {
            #better do no use to avoid that server is locked waiting for a client
            return [gid_server_clients_eval_async $script]
        } else {
            set result ""
            foreach client_id [_gid_server_get_client_ids] {
                set result [::comm::comm send $client_id $script]
            }
        }
        return $result
    }

    #asyncronous (modeless)
    proc gid_server_clients_eval_async { script } {
        foreach client_id [_gid_server_get_client_ids] {
            ::comm::comm send -async $client_id $script
        }
        return ""
    }

    #asyncronous (modeless) but invoking a callback proc when processing
    #not used by now...
    proc gid_server_clients_eval_async_callback { script callback_command } {    
        foreach client_id [_gid_server_get_client_ids] {
            ::comm::comm send -command $callback_command $client_id $script
        }    
        return ""
    }

} elseif { $::gid_server_client_protocol == "rabbitmq" } {

    # to monitorize and manage rabbitmq there is a web GUI: http://localhost:15672
    # nned to enable the rabbitmq_management plug-in:
    # "C:\Program Files\RabbitMQ Server\rabbitmq_server-3.8.3\sbin\rabbitmq-plugins.bat" enable rabbitmq_management
    # user guest / password guest can enter and create users from within the web page (section Admin) with tag 'Admin'
    # Once created you need to click on the user ( in the table) and 
    # set the permission to use (cofigure/read/write) the Virtual host '/'
    # and set the topic permission to use ( read / write) the exchange AMPQ default on the virtual host '/'
    # (just select all by default and click on 'Set permission' and 'Set topic permission'
    # 
    # no need for this below, but good to know.
    # before the user must be created with the command line utility rabbitmqctl.exe
    # rabbitmqctl add_user escolano $password
    # tag the user with "administrator" for full management UI and HTTP API access
    #rabbitmqctl set_user_tags escolano administrator

    # in this first version there is a single shared queue_server_input, all client write in the same queue
    # and a queue exclusive by connected client (by now really only can work 1 client with 1 server)
    # the exchange_server_output is a fanout that write in all client queues (and each client read its queue)

    set ::rmq_use_tls 0
    set ::rmq_channel_server "" ;#internal use to be passed to procs
    if { [ info exists ::gid_start_server_sessionid] } {
        set ::rmq_sessionid $::::gid_start_server_sessionid
    } else {
        set ::rmq_sessionid 1
    }
    if { [ info exists ::gid_start_server_username] && [ info exists ::gid_start_server_password]} {
        set ::rmq_user $::gid_start_server_username
        set ::rmq_pass $::gid_start_server_password
    } else {
        # package require rmq  ; # to define ::rmq::DEFAULT_UN and ::rmq::DEFAULT_PW
        # set user $::rmq::DEFAULT_UN ;#guest
        # set pass $::rmq::DEFAULT_PW ;#guest
        set ::rmq_user guest
        set ::rmq_pass guest        
    }
    # queue_name should have a sessionID so a single user can have different sessions
    # at the moment we'll use the username so that different users can work at the same time
    # When poler-equiv is ready we'll use sessionID instead
    set ::rmq_server_input_queue_name queue_server_input-$::rmq_user-$::rmq_sessionid
    set ::rmq_exchange_server_output exchange_server_output-$::rmq_user-$::rmq_sessionid

    proc gid_server_get_default_port { } {
        package require rmq        
        if { $::rmq_use_tls } {
            # using the guide in https://www.rabbitmq.com/ssl.html
            set port 5671
        } else {
            set port $::rmq::DEFAULT_PORT ;# 5672
        }
        return $port
    }

    proc gid_server_connect { } {
        gid_server_debug_open
        package require rmq
        set port [gid_server_get_port]
        set host [gid_server_get_host]
        # set user $::rmq::DEFAULT_UN ;#guest
        # set pass $::rmq::DEFAULT_PW ;#guest
        set rqm_login [::rmq::Login new -user $::rmq_user -pass $::rmq_pass]        
        set rmq_connection [rmq::Connection new -host $host -port $port -login $rqm_login]
        if { $::rmq_use_tls } {
            $rmq_connection tlsOptions -cafile "$::env(HOME)/testca/cacert.pem" \
                 -certfile "$::env(HOME)/client/cert.pem" \
                 -keyfile "$::env(HOME)/client/key.pem"
        }
        $rmq_connection onConnected _gid_server_on_connected
        $rmq_connection onClose _gid_server_on_close
        $rmq_connection onError _gid_server_on_error
        $rmq_connection onBlocked _gid_server_on_bloqued
        $rmq_connection onFailedReconnect _gid_server_on_fail_reconnect
        
        $rmq_connection connect                      
    }

    proc gid_server_unconnect { rmq_connection } {
      $rmq_connection closeConnection
      gid_server_debug_close
    }

    #syncronous (modal)
    proc gid_server_clients_eval { script } {
        if { 1 } {
            #better do no use to avoid that server is locked waiting for a client
            return [gid_server_clients_eval_async $script]
        } else {
            set ::rmq_rpc_result ""
            set channel $::rmq_channel_server
            set data [list EVAL $script]
            set exchange_name $::rmq_exchange_server_output
            set routing_key ""
            set flags [list]
            # create a dict with the additional property needed            
            #set props [dict create delivery-mode 2] ;## to make the message persistent
            set props ""    
            $channel basicPublish $data $exchange_name $routing_key $flags $props
            vwait ::rmq_rpc_result
        }                
         
        return $::rmq_rpc_result
    }

    #asyncronous (modeless)
    proc gid_server_clients_eval_async { script } { 
        if { $::rmq_channel_server == "" } {
            #early stages
            return ""
        }
        set channel $::rmq_channel_server
        set data [list EVAL_ASYNC $script]
        set exchange_name $::rmq_exchange_server_output
        set routing_key ""
        set flags [list]
        # create a dict with the additional property needed            
        #set props [dict create delivery-mode 2] ;## to make the message persistent
        set props ""
        gid_server_debug_print "<< EVAL_ASYNC DATA $data"
        $channel basicPublish $data $exchange_name $routing_key $flags $props
        return ""
    }

    #rabbitmq particular procs
    ##########################

    proc _gid_server_on_receive { channel methodD frameD msg } {
        #GiD_Set or its eval fail and block, probably because is inside a try catch of an object and a stack of calls
        after idle [list _gid_server_on_receive_do_at_level_0 $channel $methodD $frameD $msg]
    }

    proc _gid_server_on_receive_do_at_level_0 { channel methodD frameD msg } {
        gid_server_debug_print ">> RECEIVED $msg"              
        lassign $msg action value   
        set fail 0
        set result ""
        if { $action == "EVAL" } {
            #set fail [catch [{*}$value] result]
            #gid_server_debug_print "EVAL $value"
            #gid_server_debug_print "EVAL stack trace [GidUtils::GetStackTrace]"    
            #set result [eval $value]
            set fail [catch {eval $value} result]    
            #set result [{*}$value]                       
        } elseif { $action == "EVAL_ASYNC" } {
            #gid_server_debug_print "EVAL_ASYNC $value"
            #after idle {*}$value
            after idle [list eval $value]
        } else {
            gid_server_debug_print "unexpected action $action"            
        }
        if { $action == "EVAL" } {
            #rpc. reply only to this client            
            set rpc_properties [dict get $frameD properties]  
            set data [list RESULT [list $fail $result]]
            set reply_to [dict get $rpc_properties reply-to]
            set correlation_id [dict get $rpc_properties correlation-id]
            set exchange_name ""
            set routing_key $reply_to    
            set flags [list]                           
            set props [dict create]
            dict set props correlation-id $correlation_id            
            gid_server_debug_print "<< EVAL DATA $data"
            $channel basicPublish $data $exchange_name $routing_key $flags $props
        }
        #$channel basicAck [dict get $methodD deliveryTag]
    }

    proc _gid_server_on_connected { connection } {
        #gid_server_debug_print "_gid_server_on_connected $connection"
        set channel [rmq::Channel new $connection]
        $channel on queueDeclareOk _gid_server_on_queue_declare_ok
        $channel on queueDeleteOk _gid_server_on_queue_delete_ok
        $channel on queueUnbindOk _gid_server_on_queue_unbind_ok

        #$channel basicQos 1 1;#quality of service, to not give more than one message to a consumer at a time, prefetchCount=1, globalQos=1

        # declare a fanout exchange named exchange_server_output
        #
        # warning-1, is the same RabbitMQ handle all GiD clients, a single name for exchange or queue cannot be used
        # and both GiD-server and GiD-client must know this name in two different connections+channels
        # a common unique session-id must be known by both before declare its exchange/queues
        # cannot use the username becaues then limit only run 1 GiD by username
        # there must be some kind of 'polar' application that allow a client petition (maybe using some other rabbit queue)
        # of 'new session' that start a gid.exe and assing this unique 'session-id'
        # and by security must prevent that a user try to usurpate other guessed 'session-id' (probably rabbit must not directly accesed)
        #
        #exchange types: direct (default, routing_key is the queue_name), fanout (all bound queues), headers, match, topic
        set exchange_flags [list $::rmq::EXCHANGE_AUTO_DELETE]
        # set exchange_flags [list ] ;#a comment of tclrms said that the flag EXCHANGE_AUTO_DELETE must be 0?
        $channel exchangeDeclare $::rmq_exchange_server_output "fanout" $exchange_flags
        
        # queue bit flags:
        # QUEUE_PASSIVE (queue must previously exists, used to check existence)
        # QUEUE_DURABLE (persistent re-filled restarting rabbitMQ)
        # QUEUE_EXCLUSIVE (only this channel? could consume)
        # QUEUE_AUTO_DELETE (delete the queue when last consumer unsubscribes)
        # QUEUE_DECLARE_NO_WAIT (?)
        set queue_flags [list $::rmq::QUEUE_AUTO_DELETE]
        #set queue_flags [list ]
        # Warning-1: (name cannot be a predefined constant: see warning-1 explanation in gid_server.tcl)
        $channel queueDeclare $::rmq_server_input_queue_name $queue_flags
                
        set consume_tag "" ;# empty: automatic tag will be created by the server
        # consume bit flags:
        # CONSUME_NO_LOCAL
        # CONSUME_NO_ACK
        # CONSUME_EXCLUSIVE 
        # CONSUME_NO_WAIT
        set consume_flags [list $::rmq::CONSUME_NO_ACK]
        #set consume_flags [list]
        set consume_args ""
        $channel basicConsume _gid_server_on_receive $::rmq_server_input_queue_name $consume_tag $consume_flags $consume_args      
        set ::rmq_channel_server $channel
    }

    proc _gid_server_on_queue_declare_ok { channel queue_name msg_count consumers } {   
        #gid_server_debug_print "_gid_server_on_queue_declare_ok channel=$channel queue_name=$queue_name msg_count=$msg_count consumers=$consumers"
    }

    proc _gid_server_on_queue_delete_ok { args } {
        #gid_server_debug_print "_gid_server_on_queue_delete_ok args=$args"
    }

    proc _gid_server_on_queue_unbind_ok { args } {
        #gid_server_debug_print "_gid_server_on_queue_unbind_ok args=$args"
    }

    proc _gid_server_on_close { rmq_connection closeD } {        
        #gid_server_debug_print "_gid_server_on_close rmq_connection=$rmq_connection closeD=$closeD"
        set ::rmq_channel_server ""
    }
    
    proc _gid_server_on_error { args } {
        #gid_server_debug_print "_gid_server_on_error args=$args"
    }

    proc _gid_server_on_bloqued { args } {
        #gid_server_debug_print "_gid_server_on_bloqued args=$args"
    }

    proc _gid_server_on_fail_reconnect { args } {
        #gid_server_debug_print "_gid_server_on_fail_reconnect args=$args"
    }
      
} elseif { $::gid_server_client_protocol == "websocket" } {
    #can test this websocket server with a client web browser html5+js at http://www.websocket.org/echo.html 
    #fill Location: ws://localhost:80 and press connect
    #fill Location: ws://localhost:9160 and press connect
    #fill Message: EVAL 54 "GiD_Set SurfaceMesher 1" and press send (54 is an arbitrary id of the request)
    #then the GiD surface mesher will be set to rsurf, and its code 1 will be returned
    #
    #can test it with a client tcl sourcing the code of client.tcl in a tkcon and doing
    # gid_client_connect
    # gid_server_eval {GiD_Set SurfaceMesher 1}

    set ::ws_socket_server "" ;#internal use to be passed to procs
    set ::ws_client_ids "" ;#internal use to be passed to procs

    proc _gid_server_get_client_ids { } {
        return $::ws_client_ids
    }

    proc gid_server_get_default_port { } {
        # return 80 ;#ws default port
        #return 9160 ;#ws default port
        return 443 ;#wss  default port    
    }
  
    proc _gid_server_upgrade_socket_to_websocket { channel clientaddr clientport } {  
        #W "_gid_server_upgrade_socket_to_websocket $channel $clientaddr $clientport"               
        set http_head {}
        chan configure $channel -blocking 0
        set incoming_text [chan read $channel]
        chan configure $channel -blocking 1
        if { $incoming_text != "" } {
            #W "incoming_text $incoming_text"   
            foreach line_text [split $incoming_text \n] {
                if { [regexp {[A-Z,a-z,-]*:} $line_text result] } {
                    set key [string range $result 0 end-1] 
                    set value [string trim [string range $line_text [string length $result] end]]
                    dict set http_head $key $value                     
                }	
            }
            # Must define a protocol for implementation of websocket in tcl to work            
            #dict set http_head Sec-WebSocket-protocol "gid"
            dict set http_head Sec-WebSocket-protocol ""
            #incoming text has Sec-WebSocket-Protocol with P uppercase, but add this with p lowercase 
            #to avoid error testing from http://www.websocket.org/echo.html 
            #maybe is a bug of the tcllib/websocket package v 1.4.1 at line: set protos [dict get $res protocols]; that only find in lowercase
            set is_incoming_websocket_upgrade_request [::websocket::test $::ws_socket_server $channel * $http_head]
            #W "is_incoming_websocket_upgrade_request=$is_incoming_websocket_upgrade_request"
            if { $is_incoming_websocket_upgrade_request } {
                ::websocket::upgrade $channel
                #::websocket::configure $channel -keepalive 0 ;#0 to not send ping each 30 seconds by default
            }           
        }             
    }

    proc _gid_server_on_connected { channel clientaddr clientport } {         
        fileevent $channel readable [list _gid_server_upgrade_socket_to_websocket $channel $clientaddr $clientport]
    }

    #type: request close connect disconnect binary text ping ...
    proc _gid_server_on_receive { wsock type msg } {    
        gid_server_debug_print ">> ON RECEIVE $type $msg"
        switch -nocase -- $type {
            "connect" {
                #W "Connected on $wsock"
                lappend ::ws_client_ids $wsock      
            }
            "text" {
                #W "Message $msg"
                # msg comes between quotation marks. They should be removed
                if { [string index $msg 0] == "\"" && [string index $msg end] == "\"" } {
                    set msg [string range $msg 1 end-1]
                }
                set action [lindex $msg 0]               
                set fail 0
                set result ""
                if { $action == "EVAL" } {
                    lassign $msg action correlation_id command                   
                    set fail [catch {eval $command} result]          
                    #set result [{*}$command]     
                     #rpc. reply only to this client                    
                    set data [list RESULT $correlation_id $fail $result]
                    #gid_server_debug_print "_gid_server_websocket_send $wsock text $data"                  
                    _gid_server_websocket_send $wsock text $data                  
                } elseif { $action == "EVAL_ASYNC" } {    
                    #gid_server_debug_print "EVAL_ASYNC $command"
                    lassign $msg action command                    
                    #after idle [list {*}$command]
                    after idle [list eval $command]
                } elseif { $action == "RESULT" } {
                    lassign $msg action correlation_id fail result          
                    if { [info exists ::ws_rpc_result($wsock,$correlation_id)] } {                                  
                        set ::ws_rpc_result($wsock,$correlation_id) $result   
                    } elseif { [info exists ::ws_callback($wsock,$correlation_id)] } { 
                        set callback_command ::ws_callback($wsock,$correlation_id)
                        after idle [list {*}$callback_command $fail $result]                       
                        unset ::ws_callback($wsock,$correlation_id)
                    } else {
                        error "Doesn't match correlation_id=$correlation_id fail=$fail result=$result"                            
                    }                
                } else {
                    error "unexpected action $action"        
                }               
            }
            "binary" {
                W "binary $msg"
            }
            "close" {                
                #W "Close on $wsock"                 
                set index [lsearch $::ws_client_ids $wsock]
                if { $index != -1 } {
                    set ::ws_client_ids [lreplace $::ws_client_ids $index $index] 
                } else {
                    W "$wsock not found in ::ws_client_ids list"
                }
                array unset ::ws_rpc_result $client_id,*
                array unset ::ws_callback $client_id,*
                gid_server_debug_close
            }
            "disconnect" {                
                #W "Disconnected on $wsock"                 
                #set index [lsearch $::ws_client_ids $wsock]
                #if { $index != -1 } {
                #    set ::ws_client_ids [lreplace $::ws_client_ids $index $index] 
                #} else {
                #    W "$wsock not found in ::ws_client_ids list"
                #}                    
            }
            "request" {
                #gid_server_debug_print ">> REQUEST $type $msg"
            }
            "ping" {
                #gid_server_debug_print ">> PING $type $msg"
            }
            "pong" {
                #gid_server_debug_print ">> PONG $type $msg"
            }
            default {
                error "unexpected type $type $msg"
            } 
        }
    }

    proc gid_server_connect { } {
        gid_server_debug_open
        set ::ws_client_ids [list]
        package require uuid     
        package require websocket
        ::websocket::loglevel critical ;#emergency is the lowest level
        #logger::disable emergency 
        set port [gid_server_get_port]        
        if { 1 } {
            package require tls
            # Generating SSL key is very easy, just use these two commands (e.g. in a console of git on Windows):
            #  openssl genrsa -out server-private.pem 1024
            #  openssl req -new -x509 -key server-private.pem -out server-public.pem -days 365 
            #set certfile "C:/temp/test websocket/server-public.pem"
            #set keyfile "C:/temp/test websocket/server-private.pem"
            #
            # localhost certificates are needed, follow this tutorial: https://web.dev/how-to-use-local-https/
            # mkcert -key-file localhost-key.pem -cert-file localhost.pem localhost localhost 127.0.0.1 ::1
            set certfile "C:/gid project/scripts/localhost.pem"
            set keyfile "C:/gid project/scripts/localhost-key.pem"
            ::tls::init -certfile $certfile -keyfile $keyfile -ssl2 0 -ssl3 0 -tls1 1 -require 0 -request 0;# forcibly activate support for the TLS1 protocol            
            set sock [::tls::socket -server _gid_server_on_connected $port]
        } else {
            set sock [socket -server _gid_server_on_connected $port]            
        }
        ::websocket::server $sock        
        ::websocket::live $sock * _gid_server_on_receive        
        set ::ws_socket_server $sock        
    }

    #syncronous (modal)
    #better do no use to avoid that server is locked waiting for a client
    #probably better do for all clients in parallel and wait until all finish
    #but in any case now only one client can really handle one gid
    proc gid_server_clients_eval { script } {   
        if { 0 } {
            #better do no use to avoid that server is locked waiting for a client
            return [gid_server_clients_eval_async $script]
        } else {
            set result ""
            set correlation_id [::uuid::uuid generate]
            set data [list EVAL $correlation_id $script]
            foreach client_id [_gid_server_get_client_ids] {
                set ::ws_rpc_result($client_id,$correlation_id) ""
                _gid_server_websocket_send $client_id text $data
                vwait ::ws_rpc_result($client_id,$correlation_id)        
                set result $::ws_rpc_result($client_id,$correlation_id)
                unset ::ws_rpc_result($client_id,$correlation_id)
            }
        }        
        return $result
    }

    #asyncronous (modeless)
    proc gid_server_clients_eval_async { script } {        
        set data [list EVAL_ASYNC $script]        
        foreach client_id [_gid_server_get_client_ids] {            
            _gid_server_websocket_send $client_id text $data
        }
        return ""
    }

    #asyncronous (modeless) but invoking a callback proc when processing
    #not used by now...
    proc gid_server_clients_eval_async_callback { script callback_command } {       
        set correlation_id [::uuid::uuid generate]
        set data [list EVAL_ASYNC_CALLBACK $correlation_id $script]
        foreach client_id [_gid_server_get_client_ids] {
            set ::ws_callback($client_id,$correlation_id) $callback_command
            _gid_server_websocket_send $client_id text $data            
        }
        return correlation_id
    }

    #particular procs
    ##########################
    proc _gid_server_websocket_send { wsocket type msg } {
        gid_server_debug_print "<< SEND $type $msg"
        set fail 1
        for {set i 0} { $i<20 } {incr i} {
            set num_bytes [::websocket::send $wsocket $type $msg]
            if { $num_bytes == -1 } {
                #gid_server_debug_print "_gid_server_websocket_send num_bytes=$num_bytes i=$i"
                update
            } else {
                set fail 0
                break
            }
        }
        if { $fail } {
            error "$msg"
        }
        return $fail
    }

} else {
    error  "unexpected gid_server_client_protocol=$::gid_server_client_protocol"
}

#### start alternative tests

#send syncronous and do syncronous
#pro: in some test seems perfect (it rotate perfectly fast and smooth without lost any frame), but seems that is using the net too fast and is overflowing
#contra: very often the program become locked for a while (net/queue overflow), or crash.
#LATEST GOOD NEWS: it seems that crashed because is reentering in wordcommand when calling Tcl-Events, now after fix gid.exe it is not crashing!!
proc gid_server_clients_update_image_screen_SYNCRONOUS_NOT_SUBSAMPLE { } {
    gid_server_clients_update_image_screen_do    
}

#uncompressed and already in base64, because browsers don't provida a zlib uncompress function
proc gid_server_clients_update_image_screen_do { } {    
    lassign [GiD_Thumbnail get_pixels -quality $::gid_server_client_image_quality -format $::gid_server_client_image_format] w h data
    #seems that factor 6 or 7 are the best (see C:\gid_project\scripts\gid_server_client\test_zlib_compress.xlsx)        
    #gid_server_clients_eval [list gid_client_update_image_screen $w $h [zlib compress $data 7]]
    gid_server_clients_eval [list gid_client_update_image_screen [binary encode base64 $data]]
}

#uncompressed and already in base64, because browsers don't provida a zlib uncompress function
proc gid_server_clients_update_image_screen_do_async { } {
    lassign [GiD_Thumbnail get_pixels -quality $::gid_server_client_image_quality -format $::gid_server_client_image_format] w h data
    #seems that factor 6 or 7 are the best (see C:\gid_project\scripts\gid_server_client\test_zlib_compress.xlsx)
    #gid_server_clients_eval_async [list gid_client_update_image_screen $w $h [zlib compress $data 7]]
    gid_server_clients_eval_async [list gid_client_update_image_screen [binary encode base64 $data]]
    #next line used only by gid_server_clients_update_image_screen_ASYNCRONOUS_SUBSAMPLE_AFTER_IDLE_ABORTING_PREVIOUS
    #unset -nocomplain ::after_id_gid_server_clients_update_image_screen
    update idletasks;#this somehow aid to be much faster from the client point of view!!
}


#send asyncronous
#pro: the net is not collapsed and the program is not locked
#contra: when rotating redraws are delayed and there is lot of 'inertia' 
#(the program continue doing draws several seconds after finish the rotation)
#they appear errors messages like these:
#  "Deadlock due to omp_set_lock being called on lock already owned by thread"
#  "Error (l685): Connect to remote failed: too many nested evaluations (infinite loop?)"
proc gid_server_clients_update_image_screen_ASYNCRONOUS_NOT_SUBSAMPLE { } {    
    gid_server_clients_update_image_screen_do_async    
}

#send asyncronous and do asyncrours with after idle, and subsample with after cancel to avoid capture and send previous redraw if busy
#pro: the net is not collapsed and the program is not locked
#contra: when rotating each intermediate draw is canceled, and then only a few or the last draw is effective
proc gid_server_clients_update_image_screen_ASYNCRONOUS_SUBSAMPLE_AFTER_IDLE_ABORTING_PREVIOUS { } {
    if { [info exists ::after_id_gid_server_clients_update_image_screen] } {
        after cancel $::after_id_gid_server_clients_update_image_screen
    }
    set ::after_id_gid_server_clients_update_image_screen [after idle gid_server_clients_update_image_screen_do_async]
}

#send syncronous and do syncronous trying to subsample not doing new when busy by global semaphore variable (except the last that must be always done)
#pro: the net is not collapsed and the program is not locked
#contra: really is not working because is never subsampling!!. ::image_screen_lost is always 0 because the proc is not called when processing previous!!
#NOTE: because of the problem is doing the same as *ASYNCRONOUS_NOT_SUBSAMPLE !!
proc gid_server_clients_update_image_screen_SYNCRONOUS_SUBSAMPLE_AVOIDING_NEW_WHEN_BUSY { } {
    if { $::image_screen_updating } {
        set ::image_screen_lost 1
        W "SYNCRONOUS_SUBSAMPLE_AVOIDING_NEW_WHEN_BUSY lost $::image_screen_lost" ;#the message is never showed!!
    } else {
        set :image_screen_updating 1
        gid_server_clients_update_image_screen_do
        set :image_screen_updating 0
        if { $::image_screen_lost } {
            set ::image_screen_lost 0
            gid_server_clients_update_image_screen_do
            W "SYNCRONOUS_SUBSAMPLE_AVOIDING_NEW_WHEN_BUSY doing last"
        }
    }
}

#send syncronous and do syncronous trying to subsample not doing new when time from previous draw is too small
proc gid_server_clients_update_image_screen_SYNCRONOUS_SUBSAMPLE_AVOIDING_NEW_BY_TIME { } {
    if { [info exists ::image_screen_time_previous] } {
        set time_now [clock milliseconds]
        set time_increment [expr {$time_now-$::image_screen_time_previous}]
    } else {
        set time_increment 100000 ;#big, to process it
    }
    if { $time_increment < 100 } {
        incr ::image_screen_lost
        #W "SYNCRONOUS_SUBSAMPLE_AVOIDING_NEW_BY_TIME lost by dt=$time_increment $::image_screen_lost" ;#the message is never showed!!
    } else {
        set ::image_screen_time_previous [clock milliseconds]
        gid_server_clients_update_image_screen_do
    }
}

#### end alternative tests

proc gid_server_clients_update_image_screen { } {
    #gid_server_clients_update_image_screen_SYNCRONOUS_NOT_SUBSAMPLE
    gid_server_clients_update_image_screen_ASYNCRONOUS_NOT_SUBSAMPLE
    #gid_server_clients_update_image_screen_ASYNCRONOUS_SUBSAMPLE_AFTER_IDLE_ABORTING_PREVIOUS
    #gid_server_clients_update_image_screen_SYNCRONOUS_SUBSAMPLE_AVOIDING_NEW_WHEN_BUSY
    #gid_server_clients_update_image_screen_SYNCRONOUS_SUBSAMPLE_AVOIDING_NEW_BY_TIME
}

proc gid_server_get_main_draw_area_size { } {
    if { [info commands winfo] != "" } {
        return [GidUtils::GetMainDrawAreaSize]
    } else {
        #offscreen case 
        set w [.gid.central.s cget -width]
        set h [.gid.central.s cget -height]
        return [list $w $h]
    }
}

proc gid_server_set_main_draw_area_size { width height } {
    if { [info commands winfo] != "" } {
        GidUtils::SetMainDrawAreaSize $width $height 0
    } else {
        #offscreen case
        .gid.central.s configure -width $width -height $height
        GiD_Redraw
    }
}

proc show_taskbar_menu {x y} { 
    # See http://support.microsoft.com/kb/q135788/ 
    # Without this, clicking outside the menu does not cause menu to disappear 
    # and cursor keys and ESC do not work. 
    set hwin [twapi::Twapi_GetNotificationWindow] 
    twapi::set_foreground_window $hwin 
    # Assumes you have created a menu previously and stored the command in $taskbar_menu
    package require Tk
    wm withdraw .
    set taskbar_menu .m
    if {[winfo exists $taskbar_menu] } {
        destroy $taskbar_menu
    }
    menu $taskbar_menu -tearoff 0
    $taskbar_menu add command -label [_ "Exit"] -command {GiD_Process Mescape Quit No}      
    $taskbar_menu post $x $y 
    twapi::PostMessage $hwin 0 0 0 
}

proc gid_server_sytemtray_toolbar_handle { id event loc time } {
    if { $event == "contextmenu" || $event == "select" || $event == "keyselect" } {
        lassign [twapi::get_mouse_location] x y
        show_taskbar_menu $x $y            
    }
}

proc gid_server_sytemtray_addicon { } {
    if { $::tcl_platform(platform) == "windows" } {
        package require twapi
        set ico_id [twapi::load_icon_from_file [file join [ gid_themes::GetThemesBaseFolder] gid.ico]]
        if { [catch { set tray_ico_id [twapi::systemtray addicon $ico_id gid_server_sytemtray_toolbar_handle] } msg] } {
            #e.g. avast antivirus, seems that the fist time while scanning for viruses is runnig a pseudo-copy of the exe!!
        } else {
            set port [gid_server_get_port]
            set title [_ "GiD server"]
            set tip $title
            append tip \n [_ "Port"] { } $port  
            set balloon [_ "Started on port"]
            append balloon  { } $port
            set host [gid_server_get_host]
            if { $host == "localhost" || $host == "127.0.0.1" } {
                append tip \n "localhost"
                append balloon \n "localhost"
            } else {
                #set host [lindex [::twapi::get_system_ipaddrs] 1] ;#ip of this computer
                append tip \n "IP $host"
                append balloon \n "IP $host"
            }
            append tip \n "Protocol $::gid_server_client_protocol"
            twapi::systemtray modifyicon $tray_ico_id -tip $tip
            #twapi::systemtray modifyicon $tray_ico_id -balloontitle $title -balloon $balloon
        }
    }
}

proc gid_server_init { } {
    gid_server_connect  
    gid_server_sytemtray_addicon
}

proc gid_server_before_init_gid_project { dir } {
    set problemtype_name [file rootname [GidUtils::MakeRelativePath [gid_filesystem::get_folder_standard problemtypes] $dir]]    
    
    ::GIDi18nProblemType::load $problemtype_name
    
    set filename_tcl [GidUtils::GiveFileInsideProblemType $problemtype_name .tcl]
    set filename_tbe [GidUtils::GiveFileInsideProblemType $problemtype_name .tbe]
    if { [file exists $filename_tcl] || [file exists $filename_tbe] } {
        source $filename_tcl
    }
    
    #package require gid_themes
    #gid_themes::ReadModuleThemes $dir
}
