
#########################################################################################
################################ Coordinates projections ################################
#########################################################################################

namespace eval gid_map {
    variable radius_earth 6378137.0 ;#m (equatorial radius WGS-84)
    variable DEGREE_TO_RAD 0.0174532925199
    variable RAD_TO_DEGREE 57.2957795131
    variable PI 3.1415926535897932384626433832795      
    #
    variable current_projection MERCATOR_EPSG_3857
    variable available_projections [list MERCATOR_EPSG_3857 UTM]
    #try to check if is some valid projection of mapproj package (use 0 0 as longitude and latitude of the center of the sheet )
    #
    #Allowed:
    #PlateCarree CylindricalEqualArea Mercator MillerCylindrical Sinusoidal Mollweide 
    #EckertIV EckertVI Robinson Cassini PeirceQuincuncial Orthographic Stereographic
    #Gnomonic AzimuthalEquidistant LambertAzimuthalEqualArea Hammer LambertCylindricalEqualArea 
    #Behrmann TrystanEdwards HoboDyer GallPeters Balthasart 
    #
    #Disallowed: (because require as arguments more reference latitudes)
    #ConicEquidistant AlbersEqualAreaConic LambertConformalConic 
}

proc gid_map::get_radius_earth { } {
    variable radius_earth
    return $radius_earth
}

proc gid_map::get_available_projections { } {
    variable available_projections
    return $available_projections
}

proc gid_map::set_current_projection { projection } {
    variable current_projection
    set current_projection $projection   
}

## Other datums which could be used here instead:
##
## Name                         EquatorialRadius        EccentricitySquared
## ----                         ----------------        -------------------
## Airy                         6377563                 0.00667054
## Australian National          6378160                 0.006694542
## Bessel 1841                  6377397                 0.006674372
## Bessel 1841 (Nambia)         6377484                 0.006674372
## Clarke 1866                  6378206                 0.006768658
## Clarke 1880                  6378249                 0.006803511
## Everest                      6377276                 0.006637847
## Fischer 1960 (Mercury)       6378166                 0.006693422
## Fischer 1968                 6378150                 0.006693422
## GRS 1967                     6378160                 0.006694605
## GRS 1980                     6378137                 0.00669438
## Helmert 1906                 6378200                 0.006693422
## Hough                        6378270                 0.00672267
## International                6378388                 0.00672267
## Krassovsky                   6378245                 0.006693422
## Modified Airy                6377340                 0.00667054
## Modified Everest             6377304                 0.006637847
## Modified Fischer 1960        6378155                 0.006693422
## South American 1969          6378160                 0.006694542
## WGS 60                       6378165                 0.006693422
## WGS 66                       6378145                 0.006694542
## WGS-72                       6378135                 0.006694318
## WGS-84                       6378137                 0.00669438

################################################################################################
# Universal Transverse Mercator (UTM) meters <-> geodetical (longitude and latitude, in degrees)
################################################################################################

# geodetical (longitude and latitude) -> Universal Transverse Mercator (UTM) coordinates.
proc gid_map::ll2utm { longitude latitude } {
    variable radius_earth
    variable DEGREE_TO_RAD
    #if {$longitude > 0} {set longitude [expr {-1 * $longitude}]}
    set K0 0.9996
    
    # WGS-84
    set es2 0.00669438                          ;# EccentricitySquared
    set es4 [expr {$es2*$es2}]
    set es6 [expr {$es2*$es2*$es2}]
    
    # Must be in the range -180 <= long < 180
    while {$longitude < -180.0} { 
	set longitude [expr {$longitude + 360.0}]
    }
    while {$longitude >= 180.0}  { 
	set longitude [expr {$longitude - 360.0}]
    }
    
    # Now convert
    set lat_rad [expr {$latitude*$DEGREE_TO_RAD}]
    set long_rad [expr {$longitude*$DEGREE_TO_RAD}]
    
    set zone [expr {int(($longitude+180.0)/6.0)+1.0}]
    if {$latitude >= 56.0 && $latitude < 64.0 && $longitude >= 3.0 && $longitude < 12.0} {
        set zone  32
    }
    if { $latitude >= 72.0 && $latitude < 84.0 } {
        if { $longitude >= 0.0  && $longitude <  9.0 } {set zone 31}
        if { $longitude >= 9.0  && $longitude < 21.0 } {set zone 33}
        if { $longitude >= 21.0 && $longitude < 33.0 } {set zone 35}
        if { $longitude >= 33.0 && $longitude < 42.0 } {set zone 37}
    }
    # +3 puts origin in middle of zone
    set long_origin [expr {($zone-1.0)*6.0-180.0+3.0}]
    set long_origin_rad [expr {$long_origin*$DEGREE_TO_RAD}]
    set eccPrimeSquared [expr {$es2/(1.0-$es2)}]
    set sinlat [expr sin($lat_rad)]
    set coslat [expr cos($lat_rad)]
    set tanlat [expr $sinlat/$coslat]
    set N [expr {$radius_earth/sqrt(1.0-$es2*$sinlat*$sinlat)}]
    set T [expr {$tanlat*$tanlat}]
    set C [expr {$eccPrimeSquared*$coslat*$coslat}]
    set A [expr {$coslat*($long_rad-$long_origin_rad)}]
    set M [expr {$radius_earth*((1.0-$es2/4.0-3.0*$es4/64.0-5.0*$es6/256.0)*$lat_rad-(3.0*$es2/8.0+3.0*$es4/32.0+45.0*$es6/1024.0)*sin(2.0*$lat_rad) \
				    +(15*$es4/256.0+45.0*$es6/1024.0)*sin(4.0*$lat_rad)-(35.0*$es6/3072.0)*sin(6.0*$lat_rad))}]
    set easting [expr {$K0*$N*($A+(1.0-$T+$C)*$A*$A*$A/6.0+(5.0-18.0*$T+$T*$T+72.0*$C-58.0*$eccPrimeSquared)*$A*$A*$A*$A*$A/120.0)+500000.0}]
    set northing [expr {$K0*($M+$N*tan($lat_rad )*($A*$A/2.0+(5.0-$T+9.0*$C+4.0*$C*$C)*$A*$A*$A*$A/24.0+ \
						       (61.0-58.0*$T+$T*$T+600.0*$C-330.0*$eccPrimeSquared)*$A*$A*$A*$A*$A*$A/720.0))}]   
    if {$latitude < 0} {  ;# 1e7 meter offset for southern hemisphere
        set northing [expr {$northing+10000000.0}]
    }
    
    set northing [format "%.3f" $northing]
    set easting [format "%.3f" $easting]
    if {$latitude > 84.0 || $latitude < -80.0} {
        set letter "Z"
    } else {        
        #set l [expr {int(($latitude + 80) / 8.0)}]        
        #set letter [string index "CDEFGHJKLMNPQRSTUVWXX" $l]
        if { $latitude>= 0 } {
            #return for north hemisphere first letter N (0 to 8 degrees north)
            #(to calculate lon lat is only usef if is north or south
            # be careful, N is not the first letter of North, S is also a north hemisphere letter!!
            set letter N            
        } else {
            #return for south hemisphere first letter M (0 to 8 degrees south)
            set letter M
        }
    }    
    return [list $easting $northing $zone $letter]
}

# Universal Transverse Mercator (UTM) coordinates (m) -> geodetical (longitude and latitude, in degrees)
proc gid_map::utm2ll { easting northing zone letter } {
    if { $zone == "" || $letter == "" } {
        W "gid_map::utm2ll. Must specify UTM zone and letter"
        return [list 0.0 0.0]
    }
    variable radius_earth
    variable RAD_TO_DEGREE
    set K0 0.9996    
    # WGS-84
    set es2 0.00669438 ;# EccentricitySquared
    set es2x [expr {1.0-$es2}]
    set x [expr {$easting-500000.0}]
    set northernHemisphere [expr {$letter >= "N"}]
    set y [expr {$northing-($northernHemisphere? 0.0 : 10000000.0)}]
    set long_origin [expr {($zone-1.0)*6.0-180.0+3.0}] ;# +3 puts in middle
    set ep2 [expr {$es2/$es2x}]
    set e1 [expr {(1.0-sqrt($es2x))/(1.0+sqrt($es2x))}]
    set M [expr {$y/$K0}]
    set mu [expr {$M/($radius_earth*(1.0-$es2/4.0-3.0*$es2*$es2/64.0-5.0*$es2*$es2*$es2/256.0))}]
    set phi [expr {$mu+(3.0*$e1/2.0-27.0*$e1*$e1*$e1/32.0)*sin(2.0*$mu)+(21.0*$e1*$e1/16.0-55.0*$e1*$e1*$e1*$e1/32.0)*sin(4.0*$mu)+(151.0*$e1*$e1*$e1/96.0)*sin(6.0*$mu)}]
    set sinphi [expr {sin($phi)}]
    set cosphi [expr {cos($phi)}]
    set tanphi [expr {$sinphi/$cosphi}]
    set N1 [expr {$radius_earth/sqrt(1.0-$es2*$sinphi*$sinphi)}]
    set T1 [expr {$tanphi*$tanphi}]
    set C1 [expr {$ep2*$cosphi*$cosphi}]
    set R1 [expr {$radius_earth*$es2x/pow(1.0-$es2*$sinphi*$sinphi,1.5)}]
    set D [expr {$x/($N1*$K0)}]
    set latitude [expr {$phi-($N1*$tanphi/$R1)*($D*$D/2.0-(5.0+3.0*$T1+10.0*$C1-4.0*$C1*$C1-9.0*$ep2)*$D*$D*$D*$D/24.0+
						(61.0+90.0*$T1+298.0*$C1+45.0*$T1*$T1-252.0*$ep2-3.0*$C1*$C1)*$D*$D*$D*$D*$D*$D/720.0)}]
    set latitude [expr {$latitude*$RAD_TO_DEGREE}]
    set longitude [expr {($D-(1.0+2.0*$T1+$C1)*$D*$D*$D/6.0+(5.0-2.0*$C1+28.0*$T1-3.0*$C1*$C1+8.0*$ep2+24.0*$T1*$T1)*$D*$D*$D*$D*$D/120.0)/$cosphi}]
    set longitude [expr {$long_origin + $longitude * $RAD_TO_DEGREE}]    
    return [list $longitude $latitude]
}

#convert from "degrees minutes seconds" to decimal degrees format
proc gid_map::lat2dec {value} {
    set cnt [scan "$value 0 0 0" "%g %g %g" l1 l2 l3]
    set dec [expr {abs($l1)+$l2/60.0+$l3/3600.0}]
    if {$l1 < 0} {set dec [expr {-1*$dec}]}
    return $dec
}

proc gid_map::conversion_win { {w .gid.conversion} } {    
    global C
    InitWindow2 $w -title [_ "Coordinates converter"] \
        -geometryvariable PrePostMapConversionWindowGeom \
        -initcommand gid_map::conversion_win -ontop
    ttk::label $w.lsistref -text [concat [_ "Reference system"]: ETRS89]
    ttk::labelframe $w.ll -text [_ "Lon/Lat"]
    ttk::label $w.ll.lon -text [_ "Longitude"]:
    ttk::entry $w.ll.elon -textvariable C(lon)
    ttk::label $w.ll.lat -text [_ "Latitude"]:
    ttk::entry $w.ll.elat -textvariable C(lat)
    grid $w.ll.lon $w.ll.elon -sticky w -padx 2
    grid $w.ll.lat $w.ll.elat -sticky w -padx 2
    
    ttk::labelframe $w.utm -text "UTM"
    ttk::label $w.utm.east -text [_ "Easting"]:
    ttk::entry $w.utm.eeast -textvariable C(east)
    ttk::label $w.utm.north -text [_ "Northing"]:
    ttk::entry $w.utm.enorth -textvariable C(north)
    ttk::label $w.utm.zone -text [_ "Zone"]:
    ttk::entry $w.utm.ezone -textvariable C(zone)
    ttk::label $w.utm.letter -text [_ "Letter"]:
    entry $w.utm.eletter -textvariable C(letter)
    
    grid $w.utm.east $w.utm.eeast -sticky w -padx 2   
    grid $w.utm.north $w.utm.enorth -sticky w -padx 2
    grid $w.utm.zone $w.utm.ezone -sticky w -padx 2
    grid $w.utm.letter $w.utm.eletter -sticky w -padx 2
    
    ttk::frame $w.2
    ttk::button $w.2.utm -text "==>" -command [list gid_map::doconvert 1] -width 5
    ttk::button $w.2.ll -text "<==" -command [list gid_map::doconvert 0] -width 5
    grid $w.2.utm -row 1
    grid $w.2.ll -row 2
    grid rowconfigure $w.2 "0 3" -weight 1
    
    grid $w.lsistref -sticky w -columnspan 3
    grid $w.ll $w.2 $w.utm -sticky ns -padx 2 -pady 2
    
    ttk::frame $w.frmButtons -style BottomFrame.TFrame
    ttk::button $w.frmButtons.btnclose -text [_ "Close"] \
        -command "destroy $w" -underline 0 -style BottomFrame.TButton
    grid anchor $w.frmButtons center
    
    grid $w.frmButtons.btnclose -padx 5 -pady 5
    grid $w.frmButtons -sticky ews -columnspan 7
    
    grid columnconfigure $w "0 2" -weight 1
    grid rowconfigure $w 2 -weight 1
    
    focus $w.ll.elat
}

proc gid_map::doconvert { toUTM } {
    global C
    if { $toUTM } {
        foreach var {east north zone letter} { set C($var) "" }
        foreach var {lon lat} {
            set C($var) [string trim $C($var)]
            if {$C($var) eq ""} return
            set $var [gid_map::lat2dec $C($var)]
        }
        set utm [gid_map::ll2utm $lon $lat]
        lassign $utm C(east) C(north) C(zone) C(letter)
    } else {
        foreach var {lon lat} { set C($var) "" }
        foreach var {east north zone letter} {
            set C($var) [string trim $C($var)]
            if {$C($var) eq "" && $var ne "letter"} return
        }
        set ll [gid_map::utm2ll $C(east) $C(north) $C(zone) $C(letter)]
        lassign $ll C(lon) C(lat)
    }
}
########################################################################
#spherical geodetical (longitude and latitude, in degrees) <-> cartesian
########################################################################

proc gid_map::ll2xyz { longitude latitude {h 0.0} } {
    variable radius_earth
    variable DEGREE_TO_RAD    
    set latitude [expr {$latitude*$DEGREE_TO_RAD}]
    set longitude [expr {$longitude*$DEGREE_TO_RAD}]
    set r [expr {$radius_earth+$h}]
    set coslat [expr {cos($latitude)}]
    set x [expr {$r*cos($longitude)*$coslat}]
    set y [expr {$r*sin($longitude)*$coslat}]
    set z [expr {$r*sin($latitude)}]
    return [list $x $y $z]
}

proc gid_map::xyz2ll { x y z } {
    variable radius_earth
    variable RAD_TO_DEGREE
    set r [expr {sqrt($x*$x+$y*$y+$z*$z)}]
    set h [expr ($r-$radius_earth)]
    set latitude [expr {asin($z/$r)}]
    set longitude [expr {asin($y/($r*cos($latitude)))}]
    set latitude [expr {$latitude*$RAD_TO_DEGREE}]
    set longitude [expr {$longitude*$RAD_TO_DEGREE}]
    return [list $longitude $latitude $h]
}

proc gid_map::xyhemisphere2ll { x y hemisphere } {
    variable radius_earth
    variable RAD_TO_DEGREE    
    set longitude [expr {atan2($y,$x)*$RAD_TO_DEGREE}]
    #z has two solutions, one on each hemisphere   
    if { $hemisphere == "north" } {
        set z [expr {sqrt($radius_earth*$radius_earth-($x*$x+$y*$y))}]
    } else {
        set z [expr {-sqrt($radius_earth*$radius_earth-($x*$x+$y*$y))}]
    }
    set latitude [expr {asin($z/$radius_earth)*$RAD_TO_DEGREE}]
    return [list $longitude $latitude 0.0]
}

################################################################################################
# Spherical Mercator EPSG:3857 meters <-> geodetical (longitude and latitude, in degrees)
################################################################################################

#Spherical Mercator EPSG:3857 meters -> geodetical (longitude and latitude)
# (formulae copied from gdal2tyles.py of GDAL sources)
#From MSDN documentation:
#      To simplify the calculations, we use the spherical form of projection, not
#      the ellipsoidal form. Since the projection is used only for map display,
#      and not for displaying numeric coordinates, we don't need the extra precision
#      of an ellipsoidal projection. The spherical projection causes approximately
#      0.33 percent scale distortion in the Y direction, which is not visually
#      noticeable.

proc gid_map::mercator_EPSG_3857_2ll { x y } {   
    variable radius_earth  
    variable DEGREE_TO_RAD
    variable RAD_TO_DEGREE
    variable PI  
    set longitude [expr ($x/$radius_earth)*$RAD_TO_DEGREE]
    set latitude [expr ($y/$radius_earth)*$RAD_TO_DEGREE]
    set latitude [expr $RAD_TO_DEGREE*(2.0*atan(exp($latitude*$DEGREE_TO_RAD))-$PI/2.0)]
    return [list $longitude $latitude]
}

proc gid_map::ll2mercator_EPSG_3857 { longitude latitude } { 
    variable radius_earth  
    variable DEGREE_TO_RAD
    variable RAD_TO_DEGREE
    variable PI  
    set x [expr $longitude*$DEGREE_TO_RAD*$radius_earth]
    set y [expr log(tan((90.0+$latitude)*$DEGREE_TO_RAD/2.0))*$RAD_TO_DEGREE]
    set y [expr $y*$radius_earth*$DEGREE_TO_RAD]
    return [list $x $y]
}

##################################################################################################################
#project longitude latitude to a 2D plane in meters (dimensions valid only for small regions and far of the poles)

proc gid_map::project_with_current_projection { longitude latitude } {
    variable current_projection   
    if { $current_projection == "MERCATOR_EPSG_3857" } {
        lassign [gid_map::ll2mercator_EPSG_3857 $longitude $latitude] x y
    } elseif { $current_projection == "UTM" } {
        lassign [gid_map::ll2utm $longitude $latitude] x y zone letter
        GidUtils::SetWarnLine "UTM zone=$zone letter=$letter"            
    } else {
        #try to check if is some valid projection of mapproj package. see http://wiki.tcl.tk/19833
        #Allowed:        
        #PlateCarree CylindricalEqualArea Mercator MillerCylindrical Sinusoidal Mollweide 
        #EckertIV EckertVI Robinson Cassini PeirceQuincuncial Orthographic Stereographic
        #Gnomonic AzimuthalEquidistant LambertAzimuthalEqualArea Hammer LambertCylindricalEqualArea 
        #Behrmann TrystanEdwards HoboDyer GallPeters Balthasart 
        #avoided: (require as arguments other reference latitudes) 
        #ConicEquidistant AlbersEqualAreaConic LambertConformalConic 
        package require mapproj
        set proc_name ::mapproj::to$current_projection
        if { [info procs $proc_name] != "" } {
            set num_arguments [llength [info args $proc_name]]
            if { $num_arguments == 3 } {
                #use longitude of the center of the sheet=0 degrees
                lassign [$proc_name 0 $longitude $latitude] x y
            } elseif { $num_arguments == 4 } {
                #use longitude and latitude of the center of the sheet=0 degrees
                lassign [$proc_name 0 0 $longitude $latitude] x y
            } else {
                error "unexpected current_projection=$current_projection"
            }
            set radius_earth [gid_map::get_radius_earth]
            set x [expr $x*$radius_earth]
            set y [expr $y*$radius_earth]
        } else {
            error "unexpected current_projection=$current_projection"
        }
    }
    return [list $x $y]    
}
