#this file implements GiD events to do special things
#a GiD-event is a collection of Tcl procedures with a predefined prototype invoked from GiD

set ::gid_server_client_image_format png ;#png jpeg or raw
set ::gid_server_client_image_quality 100 ;#0 to 100

if { $::gid_start_mode == "server" } {
    #implementation of events only in case server
    source [file join $::GIDDEFAULT scripts gid_implemented_events_mode_server.tcl]
} elseif { $::gid_start_mode == "normal" || $::gid_start_mode == "client" } {
    #implementation of events only in case normal or client
    source [file join $::GIDDEFAULT scripts gid_implemented_events_mode_normal_or_client.tcl]
} else {
    error "Unexpected ::gid_start_mode $::gid_start_mode"
}

#definitions common to all cases of $::gid_start_mode (normal, server or client)

proc GiD_Implement_Event_AfterNewGIDProject { } {
    GroupsAndLayersResetTreeStatus
    GidLock::ModelDeleteLockFile
    if { [GidLock::GetReadOnly] } {
        GidLock::SetReadOnly 0
        gid_client_update_title "" ""
    }
    if { !$::GidPriv(HideUndo) && ![Undo::GetSaveClearHistory] } {
        Undo::HistoryReSetLastModelRead
    }
    if { [GiD_BackgroundImageGetOnMesh] } {
        GiD_BackgroundImageSetOnMesh_Delete
    }
    if { [info procs gid_draw_opengl::draw_list_delete_all] != "" } {
        gid_draw_opengl::draw_list_delete_all
    }
}

proc GiD_Implement_Event_BeforeReadGIDProject { projectname } {
    set result ""
    set items [GidLock::ModelIsInUse $projectname]
    set is_in_use [lindex $items 0]
    set tail_projectname [file tail $projectname]
    
    if { $is_in_use  } {
        set answer ReadOnly
        lassign $items is_in_use pid username machine date
        if { [GiD_Set ShowReadOnlyWarning] } {
            set question [_ "Model '%s' is currenty in use by user '%s' on computer '%s'." $projectname $username $machine]\n
            append question [_ "Model will be opened in read-only mode."]\n
            append question [_ "Use 'Files->Save as' to save changes."]
            set answer [MessageBoxOptionsButtons [_ "Read-only file '%s'" $tail_projectname] $question [list Cancel ReadOnly] [list [_ "Don't open file"] [_ "Open read-only"]] question ""]
        } else {
            GidUtils::SetWarnLine [_ "WARNING: model opened in read-only mode because is currenty in use by user '%s' on computer '%s'." $username $machine]
        }
        if { $answer == "Cancel" } {
            set result -cancel-
        } else {
            GidLock::SetReadOnly 1
        }
    }

    # Not in use: check write permissions
    # [ gid_cross_platform::file_writable $full_projectname] checks
    # for cloud projects if ! [ GidDataManager::isSharedReadOnly $projectname]
    # for local projects if [ file writable $projectname]
    # but we want separated messages for the user.
    if { !$is_in_use} {
        set is_cloud_path [ GidDataManager::isCloudPath $projectname]
        set full_projectname $projectname
        if { [ string tolower [ file extension $full_projectname]] != ".gid" } {
            append full_projectname ".gid"
        }
        set is_writable [ gid_cross_platform::file_writable $full_projectname]

        if { $is_writable} {
            GidLock::ModelCreateLockFile $projectname
        } else {
            # it is not writable:
            # if is cloud path it is because project is SharedReadOnly
            # else ( i.e. local path) it is ue to permissions
            if { $is_cloud_path} {
                set question [_ "Model '%s' is shared with you in read-only mode by another user, or cloud folder is mounted read-only." $projectname]\n
            } else {
                set question [_ "Model '%s' is read-only or require administrator permission" $projectname]\n
            }
            set answer ReadOnly
            if { [ GiD_Set ShowReadOnlyWarning] } {
                append question [_ "Model will be opened in read-only mode."]\n
                append question [_ "Use 'Files->Save as' to save changes."]
                set answer [ MessageBoxOptionsButtons [_ "Read-only file '%s'" $tail_projectname] $question [list Cancel ReadOnly] [list [_ "Don't open file"] [_ "Open read-only"]] question ""]
            } else {
                GidUtils::SetWarnLine [_ "WARNING: opened in read-only mode because %s" $question]
            }
            if { $answer == "Cancel" } {
                set result -cancel-
            } else {
                GidLock::SetReadOnly 1
            }
        }
    }
    return $result
}

proc GiD_Implement_Event_AfterReadGIDProject { projectname } {
    if { $::CUSTOM_LIB_USE_NATIVE_GROUPS } {
        set prev_modified_flag [GiD_ModifiedFileFlag get] ;#trick to avoid modified_flag just reading a project
        if { [GroupsXmlDocExists] } {
            #this procedure only exists if package customLib has been required (e.g. loading a problemtype customLib-based)
            gid_groups_conds::syncronize_gid_groups_and_doc_groups
        }
        #to transform group conditions also if problemtype of loaded model doesn't exist or changed its version
        if { [ConditionsGroupsExists] } {
            TransformConditionGroupsAndAssignedDataToNativeGroups
        }
        GiD_ModifiedFileFlag set $prev_modified_flag
    }
    GroupsAndLayersReadTreeStatus [file join ${projectname}.gid [file tail $projectname].tree]

    if { !$::GidPriv(HideUndo) && ![Undo::GetSaveClearHistory] } {
        #if the file was read previously then must store a copy or undo won't be able to start again from the initial model because will be overwritten
        Undo::HistorySetLastModelRead $projectname
    }
}

proc GiD_Implement_Event_AfterReadGIDProjectWithError { projectname error_string } {
    # in Project.cc:6988 a messagebox has been already launched with the error string:
    #     RaiseEvent_MessageBoxModeless( projectname, error_string)
    if { $::CUSTOM_LIB_USE_NATIVE_GROUPS } {
        set prev_modified_flag [GiD_ModifiedFileFlag get] ;#trick to avoid modified_flag just reading a project
        if { [GroupsXmlDocExists] } {
            #this procedure only exists if package customLib has been required (e.g. loading a problemtype customLib-based)
            gid_groups_conds::syncronize_gid_groups_and_doc_groups
        }
        #to transform group conditions also if problemtype of loaded model doesn't exist or changed its version
        if { [ConditionsGroupsExists] } {
            TransformConditionGroupsAndAssignedDataToNativeGroups
        }
        GiD_ModifiedFileFlag set $prev_modified_flag
    }
    GroupsAndLayersReadTreeStatus [file join ${projectname}.gid [file tail $projectname].tree]

    if { !$::GidPriv(HideUndo) && ![Undo::GetSaveClearHistory] } {
        #if the file was read previously then must store a copy or undo won't be able to start again from the initial model because will be overwritten
        Undo::HistorySetLastModelRead $projectname
    }
}

#since GiD 14.1.2d inserting a model doesn't raise GiD_Event_AfterReadGIDProject but GiD_Event_AfterInsertGIDProject
proc GiD_Implement_Event_AfterInsertGIDProject { projectname } {
    if { $::CUSTOM_LIB_USE_NATIVE_GROUPS } {
        if { [GroupsXmlDocExists] } {
            #this procedure only exists if package customLib has been required (e.g. loading a problemtype customLib-based)
            gid_groups_conds::syncronize_gid_groups_and_doc_groups
        }
        #to transform group conditions also if problemtype of loaded model does't exist or changed its version
        if { [ConditionsGroupsExists] } {
            TransformConditionGroupsAndAssignedDataToNativeGroups
        }
    }
    GroupsAndLayersReadTreeStatus [file join ${projectname}.gid [file tail $projectname].tree]
}

proc GiD_Implement_Event_BeforeSaveGIDProject { modelname } {
    return [Do_BeforeSaveGIDProject $modelname]
}

proc GiD_Implement_Event_BeforeSaveAsGIDProject { old_projectname new_projectname } {
    if { $old_projectname ==  $new_projectname } {
        #save as on then same model. Close and delete lock file to allow delete the whole folder
        GidLock::ModelDeleteLockFile
    }
    #must Do_BeforeSaveGIDProject or only some parts!!!
    return [Do_BeforeSaveGIDProject $new_projectname]
}

proc GiD_Implement_Event_AfterSaveGIDProject { projectname } {
    GroupsAndLayersSaveTreeStatus [file join ${projectname}.gid [file tail $projectname].tree]
    if { [GidLock::GetLockFile] == "" } {
        GidLock::ModelCreateLockFile $projectname
    }
    GidUtils::SetFolderIconWindowsGiDProject ${projectname}.gid
}

proc GiD_Implement_Event_AfterSaveAsGIDProject { old_projectname new_projectname } {
    GroupsAndLayersSaveTreeStatus [file join ${new_projectname}.gid [file tail $new_projectname].tree]    
    GidLock::ModelCreateLockFile $new_projectname
    GidUtils::SetFolderIconWindowsGiDProject ${new_projectname}.gid
}

proc GiD_Implement_Event_AfterSavePreferences { filename } {
    if { [GiD_Set Cloud(SynchronizePreferences)] } {
        if { [GiD_Login is_logged] } {
            #do not save logged user preferences is -c (read only) alternative was set (e.g. batchs of tester)
            if { [GiD_Set SaveGidDefaults] } {
                #set file_key [[GiD_Info GiDVersion]/gid.ini]
                set file_key "gid.ini"
                if { [GetAlternativeUserSettingsFilename] != "" } {
                    #-c2 must not save as "gid.ini" but as an alternative provided name like "IBER.ini"
                    set file_key [file tail [GetAlternativeUserSettingsFilename]]
                }
                Login::SaveLoggedUserPreferences $file_key
            }
        }
    }
}

proc GiD_Implement_Event_AfterReadPreferences { filename } {
    if { [GiD_Set Cloud(SynchronizePreferences)] } {
        if { [GiD_Login is_logged] } {
            global GidPriv
            if { ![info exists ::GidPriv(user_preferences_from_cloud_were_read)] || !$::GidPriv(user_preferences_from_cloud_were_read) } {
                #do not read and apply logged user preferences is -c or -c2 alternative was set (e.g. batchs of tester)
                set file_key "gid.ini"
                if { [GetAlternativeUserSettingsFilename] != "" } {
                    #-c2 must not save as "gid.ini" but as an alternative provided name like "IBER.ini"
                    set file_key [file tail [GetAlternativeUserSettingsFilename]]
                }
                Login::ReadLoggedUserPreferences $file_key
            }
        }
    }
}

#implement GiD-Tcl events to sincronize data in both directions
proc GiD_Implement_Event_AfterChangeGroup { group property } {
    if { $::CUSTOM_LIB_USE_NATIVE_GROUPS } {
        if { [GroupsXmlDocExists] } {
            gid_groups_conds::GiD_AfterChangeGroup $group $property
        }
    }
}

proc GiD_Implement_Event_AfterChangeParentGroup { oldname newname } {
    if { $::CUSTOM_LIB_USE_NATIVE_GROUPS } {
        if { [GroupsXmlDocExists] } {
            gid_groups_conds::GiD_AfterChangeParentGroup $oldname $newname
        }
    }
}

proc GiD_Implement_Event_AfterCreateGroup { group } {
    if { $::CUSTOM_LIB_USE_NATIVE_GROUPS } {
        if { [GroupsXmlDocExists] } {
            gid_groups_conds::GiD_AfterCreateGroup $group
        }
    } else {
        DWSetCurrentGroupsToAssign
    }
}

proc GiD_Implement_Event_AfterRenameGroup { oldname newname } {
    if { $::CUSTOM_LIB_USE_NATIVE_GROUPS } {
        if { [GroupsXmlDocExists] } {
            gid_groups_conds::GiD_AfterRenameGroup $oldname $newname
        }
    } else {
        DWSetCurrentGroupsToAssign
    }
}

proc GiD_Implement_Event_AfterDeleteGroup { group } {
    if { $::CUSTOM_LIB_USE_NATIVE_GROUPS } {
        if { [GroupsXmlDocExists] } {
            gid_groups_conds::GiD_AfterDeleteGroup $group
        }
    } else {
        DWSetCurrentGroupsToAssign
    }
}


proc GiD_Implement_Event_LoadModelSPD { filename_spd } {
    if { [info exists ::problemtype_current(CustomLibAutomatic)] && $::problemtype_current(CustomLibAutomatic) } {
        set needs_transform 0
        if { [file exists $filename_spd] } {
            set versionPT [gid_groups_conds::give_data_version]
            gid_groups_conds::open_spd_file $filename_spd READING_MODEL
            set versionData [gid_groups_conds::give_data_version]
            if { [package vcompare $versionPT $versionData] != 0 } {
                #invoke a transform upgrading and also downgrading
                set needs_transform 1
            }
        }
        if { $needs_transform } {
            set new_problemtype [GidUtils::GetRelativeProblemTypePath [GiD_Info problemtypepath]]
            #after idle [list GiD_Process escape escape escape escape Data Defaults TransfProblem $new_problemtype escape]
            GiD_Project transform_problemtype $new_problemtype
        }
        # Temporary
        customlib::UpdateDocument
    }
}

proc GiD_Implement_Event_SaveModelSPD { filename_spd } {
    if { [info exists ::problemtype_current(CustomLibAutomatic)] && $::problemtype_current(CustomLibAutomatic) } {
        gid_groups_conds::save_spd_file -compress 0 -save_post 0 $filename_spd
    }
}

proc TransformMapFill { newproblemtype } {
    # try to read possible <problemtype>.transform file of newproblemtype to map an old name of condition, 
    # material of some field to the new name, to not lost them from an old model
    unset -nocomplain ::TranformMapGeneralDataQuestion
    unset -nocomplain ::TranformMapIntervalDataQuestion
    unset -nocomplain ::TranformMapCondition
    unset -nocomplain ::TranformMapConditionQuestion
    unset -nocomplain ::TranformMapMaterial
    unset -nocomplain ::TranformMapMaterialQuestion
    unset -nocomplain ::TranformMapBlockdata
    unset -nocomplain ::TranformMapBlockdataQuestion
    unset -nocomplain ::TranformMapConditionXpath
    unset -nocomplain ::TranformMapConditionCombineQuestionsInverse
    unset -nocomplain ::TranformMapConditionInverse
    unset -nocomplain ::TranformMapConditionQuestionInverse
    unset -nocomplain ::TranformMapBlockdataInverse
    unset -nocomplain ::TranformMapBlockdataQuestionInverse
    

    #must check that olproblemtype is of same type as newproblemtype, otherwise these maps doesn't have sense...
    set transform_filename [GidUtils::GiveFileInsideProblemType $newproblemtype .transform 1]
    if { $transform_filename != "" } {
        #demo syntax, probably will be xml...
        set file_content [GidUtils::ReadFile $transform_filename]
        regsub -lineanchor -all {^#[^\n]*} $file_content {} file_content
        foreach {category action data} $file_content {
            if { $category == "GENERAL_DATA" } {
                if { $action == "RENAME_QUESTION" } {
                    lassign $data old_question new_question
                    set ::TranformMapGeneralDataQuestion($old_question) $new_question
                    if { [info exists ::TranformMapGeneralDataQuestion($new_question)] && $::TranformMapGeneralDataQuestion($new_question)==$old_question } {
                        #a previous rename was reverted again to set the original name. unset it because the transform is the identity in this case
                        unset ::TranformMapGeneralDataQuestion($new_question)
                    }
                }
            } elseif { $category == "INTERVAL_DATA" } {
                if { $action == "RENAME_QUESTION" } {
                    lassign $data old_question new_question
                    set ::TranformMapIntervalDataQuestion($old_question) $new_question
                    if { [info exists ::TranformMapIntervalDataQuestion($new_question)] && $::TranformMapIntervalDataQuestion($new_question)==$old_question } {
                        #a previous rename was reverted again to set the original name. unset it because the transform is the identity in this case
                        unset ::TranformMapIntervalDataQuestion($new_question)
                    }
                }
            } elseif { $category == "CONDITION" } {
                if { $action == "RENAME" } {
                    lassign $data old_condition_name new_condition_name
                    set ::TranformMapCondition($old_condition_name) $new_condition_name
                    if { [info exists ::TranformMapCondition($new_condition_name)] && $::TranformMapCondition($new_condition_name)==$old_condition_name } {
                        #a previous rename was reverted again to set the original name. unset it because the transform is the identity in this case
                        unset ::TranformMapCondition($new_condition_name)
                    }
                } elseif { $action == "RENAME_QUESTION" } {
                    lassign $data old_condition_name old_question new_question
                    set key [list $old_condition_name $old_question]
                    set ::TranformMapConditionQuestion($key) $new_question
                    set key_new_question [list $old_condition_name $new_question]
                    if { [info exists ::TranformMapConditionQuestion($key_new_question)] && $::TranformMapConditionQuestion($key_new_question)==$old_question } {
                        #a previous rename was reverted again to set the original name. unset it because the transform is the identity in this case
                        unset ::TranformMapConditionQuestion($key_new_question)
                    }
                } elseif { $action == "XPATH" } {
                    lassign $data old_condition_name old_condition_xpath new_condition_xpath
                    lappend ::TranformMapConditionXpath($old_condition_name) [list $old_condition_xpath $new_condition_xpath]
                 } elseif { $action == "COMBINE_QUESTIONS_PROC" } {
                    lassign $data new_condition_name old_questions new_question proc_name_combine result_names
                    set key [list $new_condition_name $new_question]
                    set ::TranformMapConditionCombineQuestionsInverse($key) [list $old_questions $proc_name_combine $result_names]
                }
            } elseif { $category == "MATERIAL" } {
                if { $action == "RENAME" } {
                    lassign $data old_material_name new_material_name
                    set ::TranformMapMaterial($old_material_name) $new_material_name
                    if { [info exists ::TranformMapMaterial($new_material_name)] && $::TranformMapMaterial($new_material_name)==$old_material_name } {
                        #a previous rename was reverted again to set the original name. unset it because the transform is the identity in this case
                        unset ::TranformMapMaterial($new_material_name)
                    }
                } elseif { $action == "RENAME_QUESTION" } {
                    lassign $data old_material_name old_question new_question
                    set key [list $old_material_name $old_question]
                    set ::TranformMapMaterialQuestion($key) $new_question
                    set key_new_question [list $old_material_name $new_question]
                    if { [info exists ::TranformMapMaterialQuestion($key_new_question)] && $::TranformMapMaterialQuestion($key_new_question)==$old_question } {
                        #a previous rename was reverted again to set the original name. unset it because the transform is the identity in this case
                        unset ::TranformMapMaterialQuestion($key_new_question)
                    }
                }
            } elseif { $category == "BLOCKDATA" } {
                if { $action == "RENAME" } {
                    lassign $data old_blockdata_name new_blockdata_name
                    set ::TranformMapBlockdata($old_blockdata_name) $new_blockdata_name
                    if { [info exists ::TranformMapBlockdata($new_blockdata_name)] && $::TranformMapBlockdata($new_blockdata_name)==$old_blockdata_name } {
                        #a previous rename was reverted again to set the original name. unset it because the transform is the identity in this case
                        unset ::TranformMapBlockdata($new_blockdata_name)
                    }
                } elseif { $action == "RENAME_QUESTION" } {
                    lassign $data old_blockdata_name old_question new_question
                    set key [list $old_blockdata_name $old_question]
                    set ::TranformMapBlockdataQuestion($key) $new_question
                    set key_new_question [list $old_blockdata_name $new_question]
                    if { [info exists ::TranformMapBlockdataQuestion($key_new_question)] && $::TranformMapBlockdataQuestion($key_new_question)==$old_question } {
                        #a previous rename was reverted again to set the original name. unset it because the transform is the identity in this case
                        unset ::TranformMapBlockdataQuestion($key_new_question)
                    }
                }
            }
        }
    }
    return 0
}

proc TransformMapFillInverse { } {
    #fill inverse relations, used by customlib
    foreach old_condition_name [array names ::TranformMapCondition] {
        set new_condition_name $::TranformMapCondition($old_condition_name)
        set ::TranformMapConditionInverse($new_condition_name) $old_condition_name
    }
    foreach item [array names ::TranformMapConditionQuestion] {
        lassign $item old_condition_name old_question
        set new_condition_name $old_condition_name
        if { [info exists ::TranformMapCondition($old_condition_name)] } {
            set new_condition_name $::TranformMapCondition($old_condition_name)
        }
        set new_question $::TranformMapConditionQuestion($item)
        set key [list $new_condition_name $new_question]
        set ::TranformMapConditionQuestionInverse($key) $old_question 
    }

    foreach old_blockdata_name [array names ::TranformMapBlockdata] {
        set new_blockdata_name $::TranformMapBlockdata($old_blockdata_name)
        set ::TranformMapBlockdataInverse($new_blockdata_name) $old_blockdata_name
    }
    foreach item [array names ::TranformMapBlockdataQuestion] {
        lassign $item old_blockdata_name old_question
        set new_blockdata_name $old_blockdata_name
        if { [info exists ::TranformMapBlockdata($old_blockdata_name)] } {
            set new_blockdata_name $::TranformMapBlockdata($old_blockdata_name)
        }
        set new_question $::TranformMapBlockdataQuestion($item)
        set key [list $new_blockdata_name $new_question]
        set ::TranformMapBlockdataQuestionInverse($key) $old_question 
    }
}

proc GiD_Implement_Event_BeforeTransformProblemType { file oldproblemtype newproblemtype } {
    TransformMapFill $newproblemtype
    if { [GroupsXmlDocExists] } {
        TransformMapFillInverse
    }
    return 0
}

proc GiD_Implement_Event_AfterTransformProblemType { filename old_problemtype new_problemtype messages } {
    set ret 0
    if { [GroupsXmlDocExists] } {
        set filename_spd [file join $filename.gid [file tail $filename].spd]
        lassign [gid_groups_conds::transform_problemtype_no_messages $filename_spd] ret messages_local
        append messages $messages_local
    } elseif { [GiD_Info unitssystems] } {
        lassign [Units::FixWrongValues] ret messages_local
        append messages $messages_local
    }    
    unset -nocomplain ::TranformMapGeneralDataQuestion
    unset -nocomplain ::TranformMapIntervalDataQuestion
    unset -nocomplain ::TranformMapCondition
    unset -nocomplain ::TranformMapConditionQuestion
    unset -nocomplain ::TranformMapMaterial
    unset -nocomplain ::TranformMapMaterialQuestion
    unset -nocomplain ::TranformMapBlockdata
    unset -nocomplain ::TranformMapBlockdataQuestion
    return [list $ret $messages]
}

proc GiD_Implement_Event_AfterLoadResults { filename } {
    #delete array that cache some menus like View results->Integrate ...
    #to avoid rebuild them, but it the result change become invalid
    array unset ::ViewMenuPriv
}

proc GiD_Implement_Event_AfterCreateInterval { interval_id } {
    RefreshOnAddNewIntv
}

proc GiD_Implement_Event_AfterDeleteInterval { interval_id } {
    RefreshOnDeleteIntv $interval_id
}

proc GiD_Implement_Event_ChangeMainWindowTitle { projectName problemType} {
    ChangeWindowTitle $projectName $problemType
}

proc GiD_Implement_Event_AfterOpenFile { filename format fail } {
    if { !$fail } {
        if { $format == "IGES_FORMAT" || $format == "STEP_FORMAT" } {
            GidUtils::LayersTreeStructureFromRhinoNames
        }
    }
}

#really this and other event procs like this (operations done at server side) must not be defined for all (server, client, normal)
#but only for server, normal (not client that doesn't have the model data) !!
proc GiD_Implement_Event_AfterRepair { } {
    return [Groups::RemoveAssignedTypesDisallowedAllGroups]
}

proc GiD_Implement_Event_AfterChangeBackground { } {
    GiD_Redraw
}

proc GiD_Implement_Event_AfterDrawModel { } {
    if { [info procs gid_draw_opengl::draw_lists_draw] != "" } {
        gid_draw_opengl::draw_lists_draw
    }
}

proc GiD_Event_After_InternetRetrieve { content name version platform python_packages } {
    if { [llength $python_packages] } {
        lassign [RetrievePT::GetCurrentPlatform] currentos currentbits
        lassign $platform os bits
        if { ($os == "All" || $os == $currentos) && ($bits == "All" || $bits == $currentbits ) } {
            set parent_w [ RetrievePT::GetWindowId]
            set timeout_s 60
            GiD_Python_PipInstallMissingPackages $python_packages $timeout_s $parent_w
        }
    }
}


############## auxiliary procedures, are not events ########################
proc Do_BeforeSaveGIDProject { modelname } {
    set res ""
    if { [GidLock::GetReadOnly] } {
        if { $modelname == [GiD_Info Project ModelName] } {
            #the name is equal, then it is not 'save as' with another name but 'save' with same name
            set question [_ "Model opened in read-only mode"].\n
            append question [_ "Yo must use 'Files->Save as' to save changes"]
            MessageBoxOkModeless [_ "Model save error"] $question error
            set res -cancel-
        } else {
            #is save as with a new name
            #must really check that the new name is not also locked and must not be overwritten
            set items [GidLock::ModelIsInUse $modelname]
            set is_in_use [lindex $items 0]
            if { $is_in_use  } {
                lassign $items is_in_use pid username machine date
                set question [_ "Cannot save '%s' because model is currenty in use by user '%s' on computer '%s', save it in other place" $modelname $username $machine]
                MessageBoxOkModeless [_ "Model save error"] $question error
                set res -cancel-
            } else {
                GidLock::SetReadOnly 0
            }
        }
    } 
    if { $res != "-cancel-" } {
        if { [GidDataManager::isCloudPath $modelname] } {
            set cloud_projectname $modelname
            if { [string tolower [file extension $cloud_projectname]] != ".gid" } {
                append cloud_projectname ".gid"
            }
            set cloud_projectname [GidDataManager::TrimLeftCloudPath $cloud_projectname]
            if { [catch {
                if { [GidDataManager::isSharedReadOnly $cloud_projectname] } {
                    WarnWin [_ "Cannot save '%s' because it is a shared read-only location, save it in other place" $modelname]
                    set res -cancel-
                }
            } msg] } {
                if { $msg  == "DATAMANAGER_FILE_OR_FOLDER_NOT_FOUND" } {
                    #do not raise error if do not exists
                } else {
                    error "GidDataManager::isSharedReadOnly. $msg"
                }
            }
        }
    }
    if { $res != "-cancel-" } {
        if { !$::GidPriv(HideUndo) && ![Undo::GetSaveClearHistory] } {
            #if the file was read previously then must store a copy or undo won't be able to start again from the initial model because will be overwritten
            if { [Undo::HistoryGetLastModelRead] != "" } {
                Undo::HistorySaveModel $modelname
            }
        }
    }
    return $res
}

proc GroupsXmlDocExists { } {
    return [info exists gid_groups_conds::doc]
}

proc ConditionsGroupsExists { } {
    set conditions_groups_exists 0
    if { [lsearch [GiD_Info conditions over_point] point_groups] != -1 } {
        set conditions_groups_exists 1
    }
    return $conditions_groups_exists
}

#ask for special conditions point_groups ... volume_groups and its entities and transform it to native groups data
proc TransformConditionGroupsAndAssignedDataToNativeGroups { } {
    foreach i [list point line surface volume] {
        set over ${i}s
        foreach j [GiD_Info conditions ${i}_groups geometry] {
            lassign $j dummy num dummy group
            if { ![GiD_Groups exists $group] } {
                GiD_Groups create $group
            }
            GiD_EntitiesGroups assign $group $over $num
        }
    }
    foreach i [list point line surface volume] {
        foreach j [GiD_Info conditions ${i}_groups mesh] {
            lassign $j xx num dummy group
            if { $xx eq "E" } {
                set over element
            } elseif { $xx eq "N" } {
                set over node
            } else {
                set over face
                set num [list $num $xx]
            }
            if { ![GiD_Groups exists $group] } {
                GiD_Groups create $group
            }
            GiD_EntitiesGroups assign $group $over $num
        }
    }
    foreach i [list point line surface volume] {
        set over ${i}s
        GiD_UnAssignData condition ${i}_groups all
        GiD_CreateData delete condition ${i}_groups
    }
    return 0
}

proc SelfRegisterEvents { } {
    foreach proc_name [info procs GiD_Implement_Event_*] {
        set event_name GiD_Event_[string range $proc_name 20 end]
        GiD_RegisterEvent $event_name $proc_name GENERAL gid
    }
}

SelfRegisterEvents
