#raster is a list object with this items: num_columns num_rows xllcenter yllcenter xcellsize ycellsize nodata_value <objarray_values>
namespace eval GIS {

}

#selected_element_ids must be sorted
#cellsize: 0.0 to automatic size
#far_points_set_nodata: bool 0 or 1 (1 to set raster points further that far_points_distance as special value nodata, usually -9999 for ARC/INFO)
#far_points_distance: double, used only if far_points_set_nodata is true
proc GIS::GetRasterFromTriangles { selected_element_ids cellsize far_points_set_nodata far_points_distance } {
    lassign [GiD_Info Mesh Nodes -array] all_node_ids coordinates
    lassign $coordinates xs ys zs
    set type [objarray type $xs]
    set triangles_data [lindex [GiD_Info Mesh Elements Triangle -array] 0]
    if { [llength $triangles_data] } {
        lassign $triangles_data type all_element_ids connectivities materials
        lassign $connectivities n0s n1s n2s
        if { [objarray length $selected_element_ids] == [objarray length $all_element_ids] } {
            #all elements
            set n012s [objarray interleave [objarray concat $n0s $n1s $n2s] 3]
            set xyzs [objarray interleave [objarray concat $xs $ys $zs] 3]
            if { [objarray length $all_node_ids] != [objarray get $all_node_ids end] } {
                #must renumber, there are not holes in the node's numeration
                objarray renumber $n012s $all_node_ids [objarray new_from_to intarray 1 [objarray length $all_node_ids]]
            }
        } else {
            set num_elements [objarray length $selected_element_ids]
            set n012s [objarray new intarray [expr $num_elements*3]]
            set count -1
            for {set i_element 0} {$i_element < $num_elements} {incr i_element} {
                set element_id [objarray get $selected_element_ids $i_element]
                set pos [objarray search -sorted $all_element_ids $element_id]
                if { $pos != -1 } {
                    objarray set $n012s [incr count] [objarray get $n0s $pos]
                    objarray set $n012s [incr count] [objarray get $n1s $pos]
                    objarray set $n012s [incr count] [objarray get $n2s $pos]
                }
            }
            set selected_node_ids [objarray sort -unique [objarray clone $n012s]]
            set num_nodes [objarray length $selected_node_ids]
            objarray renumber $n012s $selected_node_ids [objarray new_from_to intarray 1 $num_nodes]
            set xyzs [objarray new $type [expr $num_nodes*3]]
            set count -1
            for {set i_node 0} {$i_node < $num_nodes} {incr i_node} {
                set node_id [objarray get $selected_node_ids $i_node]
                set pos [objarray search -sorted $all_node_ids $node_id]
                objarray set $xyzs [incr count] [objarray get $xs $pos]
                objarray set $xyzs [incr count] [objarray get $ys $pos]
                objarray set $xyzs [incr count] [objarray get $zs $pos]
            }
        }
        set options [list -far_points_set_nodata $far_points_set_nodata -far_points_distance $far_points_distance]
        if { $cellsize != 0.0 } {
            set raster [GiD_Raster create {*}$options [list triangles $xyzs $n012s] [list $cellsize]]
        } else {
            set raster [GiD_Raster create {*}$options [list triangles $xyzs $n012s]]
        }
    } else {
        set raster ""
    }
    return $raster
}

#selected_node_ids must be sorted
#cellsize: 0.0 to automatic size
#far_points_set_nodata: bool 0 or 1 (1 to set raster points further that far_points_distance as special value nodata, usually -9999 for ARC/INFO)
#far_points_distance: double, used only if far_points_set_nodata is true
proc GIS::GetRasterFromNodes { selected_node_ids cellsize far_points_set_nodata far_points_distance } {
    lassign [GiD_Info Mesh Nodes -array] all_node_ids coordinates
    lassign $coordinates xs ys zs
    if { [objarray length $selected_node_ids] == [objarray length $all_node_ids] } {
        #all nodes
        set xyzs [objarray interleave [objarray concat $xs $ys $zs] 3]
    } else {
        if { [objarray type $selected_node_ids] != "intarray" } {
            set selected_node_ids [objarray new intarray -values $selected_node_ids]
        }
        set type [objarray type $xs]
        set num_nodes [objarray length $selected_node_ids]
        set xyzs [objarray new $type [expr $num_nodes*3]]
        set count -1
        for {set i_node 0} {$i_node < $num_nodes} {incr i_node} {
            set node_id [objarray get $selected_node_ids $i_node]
            set pos [objarray search -sorted $all_node_ids $node_id]
            if { $pos != -1 } {
                objarray set $xyzs [incr count] [objarray get $xs $pos]
                objarray set $xyzs [incr count] [objarray get $ys $pos]
                objarray set $xyzs [incr count] [objarray get $zs $pos]
            }
        }
    }
    set options [list -far_points_set_nodata $far_points_set_nodata -far_points_distance $far_points_distance]
    if { $cellsize != 0.0 } {
        set raster [GiD_Raster create {*}$options [list nodes $xyzs] [list $cellsize]]
    } else {
        set raster [GiD_Raster create {*}$options [list nodes $xyzs]]
    }
    return $raster
}

proc GIS::SaveRaster_ArcInfoASCII { raster filename } {
    set fail 0
    lassign $raster num_columns num_rows xllcenter yllcenter xcellsize ycellsize nodata_value values
    if { $xcellsize != $ycellsize } {
        set fail 1
        W "GIS::SaveRaster_ArcInfoASCII: error  xcellsize=$xcellsize != ycellsize=$ycellsize. ArcInfo ASCII only support square cells"
        return $fail
    }
    set fp [open $filename w]
    if { $fp != "" } {
        puts $fp "NCOLS $num_columns"
        puts $fp "NROWS $num_rows"
        puts $fp "XLLCENTER $xllcenter"
        puts $fp "YLLCENTER $yllcenter"
        puts $fp "CELLSIZE $xcellsize"
        puts $fp "NODATA_VALUE $nodata_value"
        set count 0
        for {set i_row 0} {$i_row<$num_rows} { incr i_row} {
            set line ""
            for {set i_col 0}  {$i_col<$num_columns} { incr i_col} {
                append line " [objarray get $values $count]"
                incr count
            }
            puts $fp $line
            #debo ver si es mas eficiente range o no
            #puts $fp [objarray range $values $count [expr $count+$num_columns-1]]
            #incr count $num_columns
        }
        close $fp
    }
    return $fail
}

#return a new averaged value from elements (cells) to its nodes
proc GIS::SmoothFromCellsToNodes { num_columns num_rows values nodata_value } {
    set nnodes_x [expr $num_columns+1]
    set nnodes_y [expr $num_rows+1]
    set nnodes [expr $nnodes_x*$nnodes_y]
    # set type [objarray type $values]
    # do not preserve original type because a chararray is small to operate
    set type floatarray
    set nodal_values [objarray new $type $nnodes 0]
    for {set i_row 0} {$i_row <$num_rows} {incr i_row} {
        for {set i_col 0} {$i_col<$num_columns} {incr i_col} {
            set z [objarray get $values [expr $i_row*$num_columns+$i_col]]
            foreach i {0 1} {
                foreach j {0 1} {
                    objarray incr $nodal_values $z [expr ($i_row+$i)*$nnodes_x+($i_col+$j)]
                }
            }
        }
    }
    #multiply all items by 1/4
    set w [expr 1.0/4.0]
    objarray scale $nodal_values $w
    #multiply the borders except the corners by 2, to become (1/4)*2=1/2
    set w [expr 2.0]
    foreach i_row [list 0 [expr $nnodes_y-1]] {
        for {set i_col 1} {$i_col<$nnodes_x-1} {incr i_col} {
            objarray scale $nodal_values $w [expr $i_row*$nnodes_x+$i_col]
        }
    }
    for {set i_row 1} {$i_row <$nnodes_y-1} {incr i_row} {
        foreach i_col [list 0 [expr $nnodes_x-1]] {
            objarray scale $nodal_values $w [expr $i_row*$nnodes_x+$i_col]
        }
    }
    #multiply the corners by 4 to become (1/4)*4=1
    set w 4.0
    foreach i_row [list 0 [expr $nnodes_y-1]] {
        foreach i_col [list 0 [expr $nnodes_x-1]] {
            objarray scale $nodal_values $w [expr $i_row*$nnodes_x+$i_col]
        }
    }
    #take into account that some cells can have a special value nodata_value that must not be operated as a number
    for {set i_row 0} {$i_row <$num_rows} {incr i_row} {
        for {set i_col 0} {$i_col<$num_columns} {incr i_col} {
            set z [objarray get $values [expr $i_row*$num_columns+$i_col]]
            if { $z == $nodata_value } {
                #invalidate its neigbor nodes
                foreach i {0 1} {
                    foreach j {0 1} {
                        objarray set $nodal_values [expr ($i_row+$i)*$nnodes_x+($i_col+$j)] $nodata_value
                    }
                }
            }
        }
    }
    return [list $nnodes_x $nnodes_y $nodal_values]
}

proc GIS::CreateSurfaceParallelLines { raster {layer ""} } {
    set surface_id 0
    if { $layer == "" } {
        set layer [GiD_Info Project LayerToUse]
    }
    lassign $raster num_columns num_rows xllcenter yllcenter xcellsize ycellsize nodata_value values
    set xmin [expr {$xllcenter+0.5*$xcellsize}]
    set ymax [expr {$yllcenter+$num_rows*$ycellsize-0.5*$ycellsize}]
    set type [objarray type $values]
    set lines_near_parallel [list]
    set xs [objarray new_n_from_increment $type $num_columns $xmin $xcellsize]
    for {set i_row 0 } {$i_row <$num_rows} { incr i_row } {
        set y [expr {$ymax-$ycellsize*$i_row}]
        set ys [objarray new $type $num_columns $y]
        set zs [objarray range $values [expr $i_row*$num_columns] [expr ($i_row+1)*$num_columns-1]]
        set row_coordinates [objarray concat $xs $ys $zs]
        objarray interleave $row_coordinates 3
        set point_start [GiD_Geometry -v2 create point append $layer {*}[objarray range $row_coordinates 0 2]]
        set point_end [GiD_Geometry -v2 create point append $layer {*}[objarray range $row_coordinates end-2 end]]
        lappend lines_near_parallel [GiD_Geometry -v2 create line append nurbsline $layer $point_start $point_end [list -interpolate $row_coordinates]]
    }
    set surface_id [GiD_Geometry -v2 create surface append nurbssurface $layer -interpolate $lines_near_parallel]
    
    # if surface_id==0 fail and can be interesting to not delete the family of near-parallel lines to detect the problem
    if { $surface_id } {
        GiD_Geometry -v2 delete -also_lower_entities line $lines_near_parallel
    }
    return $surface_id
}

proc GIS::ImportRaster_Geometry { raster show_advance_bar {value_smoothed_to_nodes 1} } {
    variable percent
    variable stop
    set fail 0
    lassign $raster num_columns num_rows xllcenter yllcenter xcellsize ycellsize nodata_value values
    set nrefresh 0
    set stop 0
    if { $show_advance_bar } {
        set percent 0
        GidUtils::CreateAdvanceBar [_ "Creating geometry"] [_ "Percentage"]: ::GIS::percent 100  {set ::GIS::stop 1}
        #4 steps: points, lines |, lines _, surfaces
        set nrefresh [expr {int($num_rows/25.0)}]
        if { $nrefresh<1 } {
            set nrefresh 1
        }
    }

    set layer [GiD_Info Project LayerToUse]
    set offset_point [GiD_Info geometry MaxNumPoints]
    set offset_line [GiD_Info geometry MaxNumLines]
    set ipoint [expr {$offset_point+1}]
    array unset -nocomplain nodata
    array unset -nocomplain noline
    if { $value_smoothed_to_nodes } {
        #replace cell values by smoothed nodal values
        lassign [GIS::SmoothFromCellsToNodes $num_columns $num_rows $values $nodata_value] num_columns num_rows values
        set xmin [expr {$xllcenter}]
        set ymax [expr {$yllcenter+($num_rows-1)*$ycellsize}]
    } else {
        #xmin-xmax, ymin-ymax are the limits of the cells, but the values are on the center of cells
        #there are num_columns cells on x direction, and then num_columns-1 raster values for the cell centers,
        #and then the xmin and xmax of the raster nodes is xmin=xmin_cells+0.5*xcellsize xmax=xmax_cells-0.5*xcellsize
        set xmin [expr {$xllcenter+0.5*$xcellsize}]
        set ymax [expr {$yllcenter+$num_rows*$ycellsize-0.5*$ycellsize}]
    }
    set y $ymax
    #create points, with number by raster location
    for {set i_row 0} {$i_row <$num_rows} {incr i_row} {
        if { $show_advance_bar } {
            if { ![expr {$i_row%$nrefresh}] } {
                set percent [expr int($i_row/($num_rows-1.0)*25)]
            }
            if { $stop } {
                set fail -1
                break
            }
        }
        set x $xmin
        set zs [objarray range $values [expr $i_row*$num_columns] [expr ($i_row+1)*$num_columns-1]]
        if { ![objarray length $zs] } {
            set fail 1
            break
        }
        for {set i_col 0} {$i_col<$num_columns} {incr i_col} {
            set z [objarray get $zs $i_col]
            if { $z != $nodata_value } {
                GiD_Geometry create point $ipoint $layer $x $y $z
            } else {
                set nodata($ipoint) 1
            }
            incr ipoint
            set x [expr {$x+$xcellsize}]
        }
        set y [expr {$y-$ycellsize}]
    }
    #create all vertical lines
    set num_vertical_0 [expr {$offset_line+1}]
    set numline $num_vertical_0
    for {set i_row 0} {$i_row<$num_rows-1} {incr i_row} {
        if { $show_advance_bar } {
            if { ![expr {$i_row%$nrefresh}] } {
                set percent [expr 25+int($i_row/($num_rows-2.0)*25)]
            }
            if { $stop } break
        }
        set pos [expr {$i_row*$num_columns+$offset_point+1}]
        for {set i_col 0} {$i_col<$num_columns} {incr i_col} {
            set n1 [expr {$pos+$i_col}]
            set n2 [expr {$n1+$num_columns}]
            if { ![info exists nodata($n1)] && ![info exists nodata($n2)] } {
                GiD_Geometry create line $numline stline $layer $n1 $n2
            } else {
                set noline($numline) 1
            }
            incr numline
        }
    }
    set num_horizontal_0 $numline
    #create all horizontal lines
    for {set i_row 0} {$i_row<$num_rows} {incr i_row} {
        if { $show_advance_bar } {
            if { ![expr {$i_row%$nrefresh}] } {
                set percent [expr 50+int($i_row/($num_rows-1.0)*25)]
            }
        }
        if { $stop } break
        set pos [expr {$i_row*$num_columns+$offset_point+1}]
        for {set i_col 0} {$i_col<$num_columns-1} {incr i_col 1} {
            set n1 [expr {$pos+$i_col}]
            set n4 [expr {$n1+1}]
            if { ![info exists nodata($n1)] && ![info exists nodata($n4)] } {
                GiD_Geometry create line $numline stline $layer $n1 $n4
            } else {
                set noline($numline) 1
            }
            incr numline
        }
    }
    #create all surfaces
    for {set i_row 0} {$i_row<$num_rows-1} {incr i_row} {
        if { $show_advance_bar } {
            if { ![expr {$i_row%$nrefresh}] } {
                set percent [expr 75+int($i_row/($num_rows-2.0)*25)]
            }
        }
        if { $stop } break
        set pos_v [expr {$i_row*$num_columns+$offset_line+1}]
        set pos_h [expr {$i_row*($num_columns-1)+$num_horizontal_0}]
        for {set i_col 0} {$i_col<$num_columns-1} {incr i_col} {
            set l1 [expr {$pos_v+$i_col}]
            set l3 [expr {$l1+1}]
            set l4 [expr {$pos_h+$i_col}]
            set l2 [expr {$l4+$num_columns-1}]
            if { ![info exists noline($l1)] && ![info exists noline($l2)]  && ![info exists noline($l3)] && ![info exists noline($l4)]} {
                GiD_Geometry create surface append coonsurface $layer 4 "$l1 0" "$l2 0" "$l3 1" "$l4 1"
            }
        }
    }
    if { $show_advance_bar } {
        GidUtils::DeleteAdvanceBar
    }
    return $fail
}

#to get an unused name, based on "prefix label"
proc GIS::GetPostMeshNameAutomatic { filename label } {
    set prefix [file rootname [file tail $filename]]
    set new_name "${prefix} ${label}"
    set names [concat [GiD_Info postprocess get all_surfacesets] [GiD_Info postprocess get all_volumesets]]
    if { [lsearch $names $new_name] == -1 } {
        set name $new_name
    } else {
        set i 1
        while { 1 } {
            set name ${new_name}_$i
            if { [lsearch $names $name] == -1 } {
                break
            }
            incr i
        }
    }
    return $name
}


proc GIS::ImportRaster_Mesh { raster show_advance_bar {value_smoothed_to_nodes 1} {pre_post "pre"} } {
    set fail 0
    lassign $raster num_columns num_rows xllcenter yllcenter xcellsize ycellsize nodata_value values
    if { $value_smoothed_to_nodes } {
        #replace cell values by smoothed nodal values
        lassign [GIS::SmoothFromCellsToNodes $num_columns $num_rows $values $nodata_value] num_columns num_rows values
        set xmin [expr {$xllcenter}]
        set ymin [expr {$yllcenter}]
        # set ymax [expr {$yllcenter+($num_rows-1)*$ycellsize}]
    } else {
        #xmin-xmax, ymin-ymax are the limits of the cells, but the values are on the center of cells
        #there are num_columns cells on x direction, and then num_columns-1 raster values for the cell centers,
        #and then the xmin and xmax of the raster nodes is xmin=xmin_cells+0.5*xcellsize xmax=xmax_cells-0.5*xcellsize
        set xmin [expr {$xllcenter+0.5*$xcellsize}]
        set ymin [expr {$yllcenter+0.5*$ycellsize}]
        # set ymax [expr {$yllcenter+$num_rows*$ycellsize-0.5*$ycellsize}]
    }
    
    # create nodes and elements in a call, using advanced objarray functions (but measuring seems a time similar to one by one!!)
    set type floatarray
    set gid_element_type Quadrilateral
    set gid_element_num_nodes 4
    set max_node_id_before [GiD_Info Mesh -$pre_post MaxNumNodes]
    if { $pre_post == "pre" } {
        set gid_mesh_command GiD_MeshPre
        set meshname ""
    } else {
        set gid_mesh_command GiD_MeshPost
        set meshname [GIS::GetPostMeshNameAutomatic "raster" $gid_element_type]
    }
    set gid_node_ids {} ;#empty for implicit increasing from the last
    set xs [objarray new_n_from_increment $type $num_columns $xmin $xcellsize]
    set ys [objarray new_n_from_increment $type $num_rows $ymin $ycellsize]
    set gid_node_coordinates [objarray new_structured_coordinates [list $xs $ys]]
    # zs==0.0, set its value
    set num_nodes [objarray length $values]
    objarray deinterleave $gid_node_coordinates 3 ;# xxxxxyyyyyyzzzz
    objarray reverse $values $num_columns ;#to reverse the rows, items of each block are not reversed. values is changed !!
    objarray set $gid_node_coordinates [expr $num_nodes*2] $values
    objarray interleave $gid_node_coordinates 3 ;# xyzxyzxyz...
    set gid_element_ids {} ;#empty for implicit increasing from the last
    set gid_element_connectivities [objarray new_structured_connectivities [list $num_columns $num_rows] 0]
    for { set i_node 0 } { $i_node < $num_nodes } { incr i_node } {
        if { [objarray get $values $i_node] == $nodata_value } {
            set nodes_nodata($i_node) 1
        }
    }
    set nodes_to_delete [objarray new intarray -values [lsort -integer [array names nodes_nodata]]]
    set num_nodes_to_delete [objarray length $nodes_to_delete]
    if { $num_nodes_to_delete } {
        #remove elements from gid_element_connectivities before be created
        set num_elements [expr {($num_rows-1)*($num_columns-1)}]
        set size_elements_new [expr {$num_elements*$gid_element_num_nodes}]
        set gid_element_connectivities_new [objarray new intarray $size_elements_new]
        set pos_new 0
        for { set pos 0 } { $pos<$size_elements_new } { incr pos $gid_element_num_nodes } {
            set must_delete 0
            for { set i_element_node 0 } { $i_element_node<$gid_element_num_nodes } { incr i_element_node } {
                if { [info exists nodes_nodata([objarray get $gid_element_connectivities [expr {$pos+$i_element_node}]])] } {
                    set must_delete 1
                    break
                }
            }
            if { !$must_delete } {
                objarray set $gid_element_connectivities_new [expr $pos_new] [objarray range $gid_element_connectivities $pos [expr $pos+$gid_element_num_nodes-1]]
                incr pos_new $gid_element_num_nodes
            }
        }
        objarray resize -preserve $gid_element_connectivities_new [expr $pos_new]
        set gid_element_connectivities $gid_element_connectivities_new
    }
    $gid_mesh_command create $meshname $gid_element_type $gid_element_num_nodes -zero_based_array $gid_node_ids $gid_node_coordinates $gid_element_ids $gid_element_connectivities
    if { $num_nodes_to_delete } {
        objarray incr $nodes_to_delete [expr {$max_node_id_before+1}]
        GiD_Mesh delete node $nodes_to_delete
    }
    return $fail
}

#action can be: geometry mesh convert_file
#increment: integer >=1 (1 to use all data, >1 to subsample the array of date, jumping values)
#show_advance_bar: boolean 1 or 0 to show or not and advancing bar
#filename_out: only required by 'convert_file' case (to export raster as ArcInfo ASCII. xcellsize must be equal to ycellsize)
proc GIS::ImportRaster { raster action increment show_advance_bar {filename_out ""} {value_smoothed_to_nodes 1} {pre_post "pre"} } {
    set fail 0
    if { ![string is integer $increment] || $increment<1 } {
        set fail 1
        W "ImportRaster: bad increment=$increment"
        return $fail
    }
    GidUtils::WaitState .gid
    GidUtils::DisableGraphics
    if { $increment != 1 } {
        set raster [GiD_Raster subsample $raster $increment $increment]
    }
    if { $action == "geometry" } {
        set fail [GIS::ImportRaster_Geometry $raster $show_advance_bar $value_smoothed_to_nodes]
    } elseif { $action == "mesh" } {
        set fail [GIS::ImportRaster_Mesh $raster $show_advance_bar $value_smoothed_to_nodes $pre_post]
    } elseif { $action == "convert_file" } {
        set fail [GIS::SaveRaster_ArcInfoASCII $raster $filename_out]
    } else {
        set fail 1
        WarnWin "Import raster. Unexpected action=$action"
    }
    GidUtils::EnableGraphics
    GidUtils::EndWaitState .gid
    if { $action != "convert_file" } {
        #GiD_Redraw
        GiD_Process Mescape View Zoom Frame escape
    }
    if { $fail } {
        if { $fail == -1 } {
            GidUtils::SetWarnLine [_ "Stop at user demand"]
            WarnWin [_ "Stop at user demand"]
        } else {
            if { $action == "geometry" || $action == "mesh"} {
                GidUtils::SetWarnLine [_ "Error importing raster"]
            } elseif { $action == "convert_file" } {
                GidUtils::SetWarnLine [_ "Error exporting ArcInfo raster ASCII file '%s'" $filename_out]
            }
        }
    } else {
        if { $action == "geometry" } {
            GeometryView
            GidUtils::SetWarnLine [_ "Raster imported as geometry"]
        } elseif { $action == "mesh" } {
            MeshView 0
            GidUtils::SetWarnLine [_ "Raster imported as mesh"]
        } elseif { $action == "convert_file" } {
            GidUtils::SetWarnLine [_ "Exported as ArcInfo raster ASCII file '%s'" $filename_out]
        }

    }
    return $fail
}

#to allow Iber not lost this old feature with values 1 and 2
#preserve_z 0: set the z interpolated
#           1: set the z intepolated only if is upper than current
#           2: set the z intepolated only if is lower than current
proc GIS::MoveNodesToZRaster { raster node_ids {preserve_z 0}} {
    set fail 0
    set selected_node_ids [GidUtils::UnCompactNumberList $node_ids]
    set node_from_id [objarray get $selected_node_ids 0]
    set node_to_id [objarray get $selected_node_ids end]
    lassign [GiD_Info Mesh Nodes $node_from_id $node_to_id -array] block_node_ids coordinates
    lassign $coordinates xs ys zs_original
    set type [objarray type $xs]
    set num_nodes [objarray length $selected_node_ids]
    set xys [objarray new $type [expr $num_nodes*2]]
    set count -1
    for {set i_node 0} {$i_node < $num_nodes} {incr i_node} {
        set id [objarray get $selected_node_ids $i_node]
        set pos [objarray search -sorted $block_node_ids $id]
        if { $pos != -1 } {
            objarray set $xys [incr count] [objarray get $xs $pos]
            objarray set $xys [incr count] [objarray get $ys $pos]
        }
    }
    set nodata_value [lindex $raster 6]
    set zs_interpolated [GiD_Raster interpolate $raster [list nodes $xys]]
    for {set i_node 0} {$i_node < $num_nodes} {incr i_node} {
        set node_id [objarray get $selected_node_ids $i_node]
        set x [objarray get $xys [expr $i_node*2]]
        set y [objarray get $xys [expr $i_node*2+1]]
        set z [objarray get $zs_interpolated $i_node]
        if { $z != $nodata_value } {
            if { $preserve_z  } {
                set z_original [objarray get $zs_original $i_node]
                if { $preserve_z == 1 } {
                    if { $z<$z_original } {
                        continue
                    }
                } elseif { $preserve_z == 2 } {
                    if { $z>$z_original } {
                        continue
                    }
                }
            }
            GiD_Mesh edit node $node_id [list $x $y $z]
        }
    }
    return $fail
}

proc GIS::MoveNodesToZFilenameRaster { filename_raster node_ids {preserve_z 0}} {
    set show_advance_bar 0
    set raster [GDAL::ReadRaster $filename_raster $show_advance_bar]
    return [GIS::MoveNodesToZRaster $raster $node_ids $preserve_z]
}


#to allow Iber not lost this old feature with values 1 and 2
#preserve_z 0: set the z interpolated
#           1: set the z intepolated only if is upper than current
#           2: set the z intepolated only if is lower than current
proc GIS::MoveLinesToZRaster { raster line_ids {preserve_z 0}} {
    set fail 0
    set selected_point_ids [GidUtils::GetLinesPoints $line_ids]
    set coordinates [GidUtils::GetPointsCoordinates $selected_point_ids]
    #lassign $coordinates xs ys zs_original
    set num_points [objarray length $selected_point_ids]
    objarray deinterleave $coordinates 3
    set xs [objarray range $coordinates 0 [expr $num_points-1]]
    set ys [objarray range $coordinates $num_points [expr $num_points*2-1]]
    if { $preserve_z  } {
        set zs_original [objarray range $coordinates [expr $num_points*2] [expr $num_points*3-1]]
    }
    set type [objarray type $xs]
    set xys [objarray new $type [expr $num_points*2]]
    set count -1
    for {set i_point 0} {$i_point < $num_points} {incr i_point} {
        set id [objarray get $selected_point_ids $i_point]
        set pos [objarray search -sorted $selected_point_ids $id]
        if { $pos != -1 } {
            objarray set $xys [incr count] [objarray get $xs $pos]
            objarray set $xys [incr count] [objarray get $ys $pos]
        }
    }
    GidUtils::DisableGraphics
    set nodata_value [lindex $raster 6]
    set zs_interpolated [GiD_Raster interpolate $raster [list nodes $xys]]
    for {set i_point 0} {$i_point < $num_points} {incr i_point} {
        set point_id [objarray get $selected_point_ids $i_point]
        set x [objarray get $xys [expr $i_point*2]]
        set y [objarray get $xys [expr $i_point*2+1]]
        set z [objarray get $zs_interpolated $i_point]
        if { $z != $nodata_value } {
            if { $preserve_z  } {
                set z_original [objarray get $zs_original $i_point]
                if { $preserve_z == 1 } {
                    if { $z<$z_original } {
                        continue
                    }
                } elseif { $preserve_z == 2 } {
                    if { $z>$z_original } {
                        continue
                    }
                }
            }
            GidUtils::GiD_Geometry_Edit_Point_Coordinates $point_id [list $x $y $z]
        }
    }
    GidUtils::EnableGraphics
    return $fail
}

proc GIS::MoveLinesToZFilenameRaster { filename_raster line_ids {preserve_z 0}} {
    set show_advance_bar 0
    set raster [GDAL::ReadRaster $filename_raster $show_advance_bar]
    return [GIS::MoveLinesToZRaster $raster $line_ids $preserve_z]
}
