
proc OpenMeshCartesianData { } {
    global MeshCartesianDataPriv

    if { ![info exists MeshCartesianDataPriv(samesizes)] } {
        set MeshCartesianDataPriv(samesizes) 1
    }        
    foreach axe {x y z} {
        if { ![info exists MeshCartesianDataPriv(size,$axe)] } {
            set MeshCartesianDataPriv(size,$axe) 0.0
        }
    }


    set w .gid.cartesianmeshdata
    if { [winfo exists $w] } return

    InitWindow2 $w -title [_ "Mesh cartesian data"] \
        -geometryvariable PreMeshCartesianDataWindowGeom \
        -initcommand OpenMeshCartesianData -ontop
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0

    ttk::frame $w.f0

    if { ![info exists MeshCartesianDataPriv(samesizes)] } { set MeshCartesianDataPriv(samesizes) 0 }
    ttk::checkbutton $w.chk -text [_ "Same sizes"] -variable MeshCartesianDataPriv(samesizes)
    foreach axe {x y z} {
        ttk::label $w.f0.l$axe -text [concat [_ "Size"] $axe]
        ttk::entry $w.f0.e$axe -textvariable MeshCartesianDataPriv(size,$axe)
        GidHelp $w.f0.e$axe [concat [_ "Size that will determine the number of divisions of the grid on each direction"]. \
                                 [_ "(Set 0 to use after the general size)"]]
        bind $w.f0.e$axe <Return> "OnChangeMeshCartesianSize $axe"
        bind $w.f0.e$axe <FocusOut> "OnChangeMeshCartesianSize $axe"
    }
    ttk::label $w.f0.lcomment -text [_ "(Set 0 to use after the general size)"]
   
    ttk::frame $w.frmButtons -style BottomFrame.TFrame
    ttk::button $w.frmButtons.btnApply -text [_ "Assign"] -command [list ApplyMeshCartesianData $w] \
        -underline 0 -style BottomFrame.TButton
    ttk::button $w.frmButtons.btnclose -text [_ "Close"] -command {GiD_Process escape} \
        -underline 0 -style BottomFrame.TButton

    grid $w.chk -sticky w -columnspan 2
    foreach axe {x y z} {
        grid $w.f0.l$axe $w.f0.e$axe -sticky ew -padx 2
        grid configure $w.f0.l$axe -sticky e
    } 
    grid $w.f0.lcomment -columnspan 2
    grid $w.f0 -sticky nsew   
    grid rowconfigure $w.f0 4 -weight 1
    grid columnconfigure $w.f0 1 -weight 1   

    grid $w.frmButtons -sticky ews -columnspan 7
    grid anchor $w.frmButtons center

    grid $w.frmButtons.btnApply $w.frmButtons.btnclose -padx 5 -pady 6
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 1 -weight 1

    focus $w.frmButtons.btnApply

    bind $w <Alt-c> "$w.frmButtons.btnclose invoke"
    bind $w <Escape> "$w.frmButtons.btnclose invoke"
    bind $w <Return> "$w.frmButtons.btnApply invoke"
    #bind $w <Destroy> [list +DestroyMeshCartesianData %W $w] ;# + to add to previous script

    wm minsize $w 140 140

    if { $::tcl_platform(os) != "Darwin"} {
        #catch { tkwait visibility $w }
        update
    } else {
        update
    }
    if { [info exists $w] } {
      grab set $w
    }
}

proc OnChangeMeshCartesianSize { i } {
    global MeshCartesianDataPriv
    if { $MeshCartesianDataPriv(samesizes) } {
        foreach axe {x y z} {
            if { $axe == $i } continue
            set MeshCartesianDataPriv(size,$axe) $MeshCartesianDataPriv(size,$i)
        }
    }
}

proc ApplyMeshCartesianData { w } {
    global MeshCartesianDataPriv
    

    foreach axe {x y z} {
        if { ![info exists MeshCartesianDataPriv(size,$axe)] } {
            set MeshCartesianDataPriv(size,$axe) [string trim $MeshCartesianDataPriv(size,$axe)]
        }
    }            
    GiD_Process $MeshCartesianDataPriv(size,x) $MeshCartesianDataPriv(size,y) $MeshCartesianDataPriv(size,z)
    
    grab release $w
    #destroy $w
    FinishButton $w $w.frmButtons [_ "Press 'Finish' to end selection"] "" disableall [GiD_Set SmallWinSelecting]
}


proc CloseMeshCartesianData { } {
    set w .gid.cartesianmeshdata
    if { [winfo exists $w] } {        
        destroy $w
    }
}

namespace eval CartesianGrid {
    variable table_contents
}

#dark trick to allow set a button in preferences to open this window, really related to the model
proc xmlprograms::CartesianGridWin { args } {    
    CartesianGrid::CartesianGridWin
}

proc CartesianGrid::CartesianGridWin { } {            
    variable table_contents
    variable mesh_size
    package require Tktable
    package require autoscroll ;#tklib      
   
    set w .gid.cartesiangrid
    InitWindow2 $w -title [_ "Cartesian grid"] \
        -geometryvariable PreCartesianGridWindowGeom -ontop
   
    set ftop [ttk::frame $w.ftop]
    #set item grid_user_defined
    #variable $item
    #set $item [GiD_Set Cartesian(GridUserDefined)]
    #set f_item [ttk::frame $ftop.f$item]
    #ttk::checkbutton $f_item.chk -text [_ "Grid user defined"] -variable CartesianGrid::$item \
     -command [list CartesianGrid::OnChangeGridUserDefined $item $w]
    #grid $f_item.chk -sticky ew
    #grid $f_item -sticky ew -padx 2 -pady 2

    set item auto_calculated
    set f_item [ttk::frame $ftop.f$item]
    ttk::label $f_item.l -text [_ "Mesh size"] -width 15
    ttk::entry $f_item.e -textvariable CartesianGrid::mesh_size
    #$f_item.e insert 0 $project_mesh_size ;#unneded with -textvariable CartesianGrid::mesh_size
    if { ![info exists mesh_size] } {
        set project_mesh_size [GiD_Info project LastElementSize]
        if { $project_mesh_size == "NONE" } {
            set project_mesh_size [GiD_Info project RecommendedMeshSize]
        }            
        set mesh_size $project_mesh_size
        if { ![string is double -strict $mesh_size] || $mesh_size<=0.0 } {
            set mesh_size ""
        }
    }
    ttk::button $f_item.b -text [_ "Auto calculate"] -command [list CartesianGrid::OnAutoCalculated $item $w]    
    grid $f_item.l $f_item.e $f_item.b -sticky ew
    grid $f_item -sticky ew -padx 2 -pady 2

    foreach item {corner boxsize ngridpoints} item_text [list [_ "Corner"] [_ "Box size"] [_ "Num grid"] ] {
        #variable $item
        #lassign [GiD_Cartesian get $item] [set item](x) [set item](y) [set item](z)
        set $item [GiD_Cartesian get $item]
        set f_item [ttk::frame $ftop.f$item]
        ttk::label $f_item.l -text $item_text -width 15
        foreach dim {0 1 2} {
            ttk::entry $f_item.e$dim ;# -textvariable CartesianGrid::[set item]($dim) 
            $f_item.e$dim insert 0 [lindex [set $item] $dim]                
        }
        ttk::button $f_item.b -text [_ "Apply"] -command [list CartesianGrid::OnApply $item $w]
        grid $f_item.l $f_item.e0 $f_item.e1 $f_item.e2 $f_item.b -sticky ew
        #grid configure $f_item.l -sticky e
        grid $f_item -sticky ew -padx 2 -pady 2
    }
        
    ttk::labelframe $w.f -text [_ "Grid relative coordinates"]    
    set f [ttk::frame $w.f.f]    
    set table [table $f.table -variable CartesianGrid::table_contents -rows 1 -cols 4 \
            -multiline 0 -selectmode extended -anchor e \
            -sparsearray 0 \
            -rowstretchmode none -colstretchmode all -resizeborders col \
            -titlerows 1 -titlecols 1 -cache 0 \
            -rowseparator \n -colseparator \t \
            -bordercursor sb_h_double_arrow \
            -xscrollcommand [list $f.sh set] -yscrollcommand [list $f.sv set] \
            -drawmode fast -relief raised -borderwidth 1 -highlightthickness 0 \
            -autoclear 0]
            #-validate 1 -validatecommand { CartesianGrid::CartesianGridTableValidate %W %s %S}
            #-browsecommand { CartesianGrid::CartesianGridTableBrowse %W %s %S}
    
    $table tag configure title -relief raised -borderwidth 2 -foreground white
    $table tag configure active -background white -foreground blue

    ttk::scrollbar $f.sv -orient vertical -command [list $f.table yview]
    ttk::scrollbar $f.sh -orient horizontal -command [list $f.table xview]    
    
    autoscroll::autoscroll $f.sh
    autoscroll::autoscroll $f.sv    

    ttk::button $w.f.b -text [_ "Apply"] -command [list CartesianGrid::OnApplyChangeTableContents $table]
    
    bind $table <Return> {
        set r [%W index active row]
        set c [%W index active col]
        if {$c == 2} {
            %W activate [incr r],1
        } else {
            %W activate $r,[incr c]
        }
        %W see active
        break
    }

    grid $table $f.sv -sticky ns -padx 2 -pady 2
    grid $f.sh -sticky ew    
    grid configure $table -sticky nsew

    grid $ftop -sticky new
    grid $f -sticky nsew
    grid $w.f.b -sticky e
    grid columnconfigure $f 0 -weight 1
    grid rowconfigure $f 0 -weight 1
    grid $w.f -sticky nsew
    grid columnconfigure $w.f 0 -weight 1
    grid rowconfigure $w.f 0 -weight 1
    
    #lower buttons
    ttk::frame $w.frmButtons -style BottomFrame.TFrame    
    ttk::button $w.frmButtons.btnclose -text [_ "Close"] -command "destroy $w" -underline 0 -style BottomFrame.TButton
    
    grid $w.frmButtons -sticky ews -columnspan 7
    grid $w.frmButtons.btnclose -padx 3 -pady 3
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 1 -weight 1
    grid anchor $w.frmButtons center
    
    focus $w.frmButtons.btnclose
    bind $w <Alt-c> "$w.frmButtons.btnclose invoke"
    bind $w <Escape> "$w.frmButtons.btnclose invoke"

    CartesianGrid::CartesianGridUpdateTable $table
    #CartesianGrid::OnChangeGridUserDefined grid_user_defined $w    
}

proc CartesianGrid::GetCartesianCoordinatesRelative { data } {        
    set data_relative [list]
    foreach items $data {
        set v_0 [lindex $items 0]
        set v_end [lindex $items end]
        set v_range [expr $v_end-$v_0]
        set items_relative [list]
        foreach item $items {
            set v [expr ($item-$v_0)/$v_range]
            lappend items_relative [format %.6f $v]
        }
        lappend data_relative $items_relative
    }
    return $data_relative
}

proc CartesianGrid::GetCartesianCoordinatesAbsolute { data_relative corner boxsize } {        
    set data [list]
    foreach items $data_relative origin $corner size $boxsize {        
        set items_absolute [list]
        foreach item $items {
            set v [expr $origin+$item*$size]
            lappend items_absolute [format %.6f $v]
        }
        lappend data $items_absolute
    }
    return $data
}

proc CartesianGrid::CartesianGridUpdateTable { table } {    
    #fill table with relative locations
    set data [GiD_Cartesian get coordinates]
    set data_relative [CartesianGrid::GetCartesianCoordinatesRelative $data]    
    
    set maxrows 0
    foreach items $data_relative {
        set row [llength $items]
        if { $maxrows<$row} {
            set maxrows $row        
        }        
    }
    incr maxrows

    #$table clear all
    #$table delete
    variable table_contents
    array unset table_contents
    
    $table configure -rows $maxrows
    $table set 0,0 Num
    $table set 0,1 X
    $table set 0,2 Y
    $table set 0,3 Z
    foreach col {1 2 3} items $data_relative {
        $table set col 1,$col $items
    }
    for {set row 1} {$row <$maxrows} {incr row} {
        $table set $row,0 $row
    }    
}

proc CartesianGrid::GetDataRelativeFromTable { } {
    variable table_contents
    set data_relative [list]
    lassign [GiD_Cartesian get ngridpoints] nrows(1) nrows(2) nrows(3)
    foreach col {1 2 3} {
        set items_relative [list]
        for {set row 1} {$row<=$nrows($col)} {incr row} {            
            lappend items_relative $table_contents($row,$col)
        }
        lappend data_relative $items_relative
    }
    return $data_relative
}

proc CartesianGrid::CheckDataRelative { data_relative } {
    set wrong_pos ""
    set epsilon 1e-8
    foreach items $data_relative col {1 2 3} {
        if { [expr abs([lindex $items 0])]>$epsilon } {
            set wrong_pos 1,$col
            break
        }
        if { [expr abs([lindex $items end]-1.0)]>$epsilon } {
            set wrong_pos [llength $items],$col
            break
        }
        set row 0
        set item_prev -1.0        
        foreach item $items {
            incr row
            if { [expr $item-$item_prev]<$epsilon } {
                set wrong_pos $row,$col
                break
            }
            set item_prev $item
        }     
    }
    return $wrong_pos
}

proc CartesianGrid::OnApplyChangeTableContents { table } {                
    set data_relative [CartesianGrid::GetDataRelativeFromTable]
    if { [llength [lindex $data_relative 0]] && [llength [lindex $data_relative 1]] && [llength [lindex $data_relative 2]] } {
        set wrong_pos [CartesianGrid::CheckDataRelative $data_relative]
        if { $wrong_pos != "" } {
            W [_ "Wrong item %s: must be and increasing vector from 0.0 to 1.0" $wrong_pos]
            $table activate $wrong_pos
        } else {
            set corner [GiD_Cartesian get corner]
            set boxsize [GiD_Cartesian get boxsize]
            set data [CartesianGrid::GetCartesianCoordinatesAbsolute $data_relative $corner $boxsize]
            GiD_Cartesian set coordinates $data
        }
    }
}

proc CartesianGrid::OnApply { item w } {
    #variable $item  
    set f_item $w.ftop.f$item
    if { $item == "corner" } {
        set corner [list]
        foreach dim {0 1 2} {            
          lappend corner [$f_item.e$dim get]
        }
        set boxsize [GiD_Cartesian get boxsize]
        #recalculate coordinates, maintain ngridpoints, boxsize and coordinates relatives
        set data_initial [GiD_Cartesian get coordinates]
        if { [llength $data_initial] } {
            set data_relative [CartesianGrid::GetCartesianCoordinatesRelative $data_initial]
            set data [CartesianGrid::GetCartesianCoordinatesAbsolute $data_relative $corner $boxsize]        
            GiD_Cartesian set corner $corner
            GiD_Cartesian set coordinates $data
        }
    } elseif { $item == "boxsize" } {
        set boxsize [list]
        foreach dim {0 1 2} {
          lappend boxsize [$f_item.e$dim get]
        }
        set corner [GiD_Cartesian get corner]
        #recalculate coordinates, maintain ngridpoints, corner and coordinates relatives
        set data_initial [GiD_Cartesian get coordinates]
        if { [llength $data_initial] } {
            set data_relative [CartesianGrid::GetCartesianCoordinatesRelative $data_initial]
            set data [CartesianGrid::GetCartesianCoordinatesAbsolute $data_relative $corner $boxsize]        
            GiD_Cartesian set boxsize $boxsize
            GiD_Cartesian set coordinates $data        
        }
    } elseif { $item == "ngridpoints" } {
        set ngridpoints [list]
        foreach dim {0 1 2} {
          lappend ngridpoints [$f_item.e$dim get]
        }
        set corner [GiD_Cartesian get corner]
        set boxsize [GiD_Cartesian get boxsize]
        #recalculate coordinates, maintain corner and boxsize, do it equispaced
        #do not preserve coordinates relatives
        set data [list]
        foreach dim {0 1 2} {
            set v_0 [lindex $corner $dim]
            set v_end [expr $v_0+[lindex $boxsize $dim]]
            set n [lindex $ngridpoints $dim]
            lappend data [GidUtils::GetRange $v_0 $v_end $n]
        }
        if { [llength [lindex $data 0]] && [llength [lindex $data 1]] && [llength [lindex $data 2]] } {
            GiD_Cartesian set ngridpoints $ngridpoints
            GiD_Cartesian set coordinates $data        
            CartesianGrid::CartesianGridUpdateTable $w.f.f.table
        }
    } else {

    }
    GiD_Redraw
}

proc CartesianGrid::OnAutoCalculated { item w } {    
    set mesh_size [$w.ftop.f$item.e get]
    if { [string is double -strict $mesh_size] && $mesh_size>0.0 } {
        set mode GEOMETRY   
        GiD_Cartesian set auto_calculated $mode $mesh_size
        #to refresh the window
        destroy $w
        CartesianGrid::CartesianGridWin
        GiD_Redraw    
    }
}

proc CartesianGrid::OnChangeGridUserDefined { item w } {
    #variable $item    
    #GiD_Set Cartesian(GridUserDefined) [set $item]
    set user_defined [GiD_Set Cartesian(GridUserDefined)]
    if { $user_defined } {
        set state normal
    } else {
        set state disabled
    }    
    foreach item {corner boxsize ngridpoints auto_calculated}  {        
        RecursiveChangeState $w.ftop.f$item $state
    }
    RecursiveChangeState $w.f $state    
}
