
set GIDDEFAULTTCL [file dirname [info script]]
set GIDDEFAULT [file dirname $GIDDEFAULTTCL]
set ::env(TK_LIBRARY) [file join $GIDDEFAULTTCL tk]
lappend auto_path $GIDDEFAULTTCL $GIDDEFAULTTCL/tcl $GIDDEFAULTTCL/tk

# in linux & co. we'll use gidx, to ensure gid.exe will launch due to library dependencies
if { $::tcl_platform(platform) eq "windows" } {
    set gidexe [file join $GIDDEFAULTTCL .. gid]
} else {
    if { $::tcl_platform(os) eq "Darwin" } {
        set gidexe [file join $GIDDEFAULTTCL .. gid]
    } else {
        set gidexe [file join $GIDDEFAULTTCL .. gidx]
    }
}

# This scripte creates desktop icons in Linux
if { $::tcl_platform(os) != "Linux"} {
    exit
}

# process args:
if { [ llength $argv] != 1} {
    puts "Usage $argv0 { install | uninstall}"
    exit
}

# Configuration parameters:
set version [ exec $gidexe -version]
set bits x32
if { $::tcl_platform(pointerSize) == 8} {
    set bits x64
}

set gid_icon [ file join $GIDDEFAULT resources images scalable gid.svg]
set gid_safe_icon [ file join $GIDDEFAULT resources images scalable gid_safe_mode.svg]
set uninstall_icon [ file join $GIDDEFAULT resources images scalable uninstall.svg]
set help_icon [ file join $GIDDEFAULT resources images scalable help.svg]
set pt_help_icon [ file join $GIDDEFAULT resources images scalable problemtype_help.svg]
# End Configuration parameters

set action install
if { [ string tolower [ lindex $argv 0]] == "uninstall"} {
    set action uninstall
    puts "Uninstalling desktop icon and menu entries."
} else {
    puts "Installing desktop icon and menu entries."
}

# there are 3 types of desktop file:
# Application, Link and Directory
# Link does not appear on application menus, ... so using xdg-open instead
proc create_app_desktop_file { filename name cmd icon verbose } {
    set err [ catch {
        set fo [ open $filename w]
        puts $fo "\[Desktop Entry\]"
        puts $fo "Version=1.0"
        puts $fo "Type=Application"
        puts $fo "Name=$name"
        puts $fo "Exec=$cmd"
        puts $fo "Icon=$icon"
        puts $fo "Categories=Application;Development;Science;Graphics"
        if { $name == "GiD" } {
            # so that the icon can be pinned in App bar
            puts $fo "StartupWMClass=$name"
            # add new-window entry
            puts $fo "Actions=new-window\n"
            puts $fo "\[Desktop Action new-window\]"
            puts $fo "Name=New Window"
            puts $fo "Exec=$cmd"
        }
        # puts $fo "OnlyShowIn=Old;"
        close $fo
        file attributes $filename -permissions ugo+x
    } err_txt ]
    if { $err} {
        puts "Error creating '$filename': $err_txt"
    } else {
        if { $verbose} {
            puts "$filename created."
        }
    }
}

proc create_desktop_menu_file { filename name icon verbose} {
    set err [ catch {
        set fo [ open $filename w]
        puts $fo "\[Desktop Entry\]"
        puts $fo "Version=1.0"
        puts $fo "Type=Directory"
        puts $fo "Name=$name"
        puts $fo "Icon=$icon"
        # puts $fo "OnlyShowIn=Old;"
        close $fo
    } err_txt ]
    if { $err} {
        puts "Error creating '$filename': $err_txt"
    } else {
        if { $verbose} {
            puts "$filename created."
        }
    }
}

set ::_gid_web_mimetype "x-scheme-handler/gid"
proc create_app_url_handler_file { filename name cmd icon verbose } {
    set err [ catch {
        set fo [ open $filename w]
        puts $fo "\[Desktop Entry\]"
        puts $fo "Version=1.0"
        puts $fo "Type=Application"
        puts $fo "Name=$name"
        puts $fo "Exec=$cmd"
        puts $fo "Icon=$icon"
        puts $fo "Categories=Application;Development;Science;Graphics"

        puts $fo "NoDisplay=true"
        puts $fo "MimeType=${::_gid_web_mimetype};"
        puts $fo "StartupNotify=true"
        close $fo
        file attributes $filename -permissions ugo+x
    } err_txt ]
    if { $err} {
        puts "Error creating '$filename': $err_txt"
    } else {
        if { $verbose} {
            puts "$filename created."
        }
    }
}

proc get_program_to_open_html { } {
    if { ![ info exists ::desktop_icons_html_browser]} {
        set ::desktop_icons_html_browser ""
        foreach program [ list xdg-open konqueror firefox chrome chromium-browser] {
            set ret [ auto_execok $program]
            if { $ret != ""} {
                set ::desktop_icons_html_browser $ret
                break
            }
        }
    }
    return $::desktop_icons_html_browser
}

set basename GiD${bits}-${version}

set lst_filenames [ list \
    ${basename}-program-entry.desktop              \
    ${basename}-program-safe-entry.desktop	       \
    ${basename}-whatsnew-entry.desktop	       \
    ${basename}-tutorials-entry.desktop	       \
    ${basename}-courses-entry.desktop          \
    ${basename}-customization-entry.desktop	   \
    ${basename}-faq-entry.desktop              \
    ${basename}-manual-entry.desktop           \
]

set lst_names [ list \
    "GiD" \
    "Repair GiD" \
    "What's new in GiD ${version}" \
    "Tutorials" \
    "Courses" \
    "Customization manual" \
    "FAQs" \
    "Reference manual" \
]

set lst_cmds [ list \
    [ file join $GIDDEFAULT gid] \
    [ file join $GIDDEFAULT gid_safe_mode] \
    [ list [ get_program_to_open_html] [ file join $GIDDEFAULT info en GWN index.html]] \
    [ list [ get_program_to_open_html] [ file join $GIDDEFAULT info en GUM index.html]] \
    [ list [ get_program_to_open_html] [ file join $GIDDEFAULT info en GCS index.html]] \
    [ list [ get_program_to_open_html] [ file join $GIDDEFAULT info en GCM index.html]] \
    [ list [ get_program_to_open_html] [ file join $GIDDEFAULT info en SD index.html]] \
    [ list [ get_program_to_open_html] [ file join $GIDDEFAULT info en GRM index.html]] \
]

set lst_icons [ list \
    $gid_icon  \
    $gid_safe_icon  \
    $help_icon \
    $help_icon \
    $help_icon \
    $pt_help_icon \
    $help_icon \
    $help_icon \
]

# which entry should go in the Help sub-menu
set lst_is_help [ list 0 0 1 1 1 1 1 1]

# check if all entries are full
set num_items [ llength $lst_filenames]
set err 0
puts "Number of entries to be created = $num_items"
foreach lst [ list lst_names lst_cmds lst_icons lst_is_help ] {
    set num_items_lst [ llength [ set $lst]]
    if { $num_items_lst != $num_items} {
        puts "Review $lst number of items = $num_items_lst != $num_items"
        puts "   llength $lst = --[ llength [ set $lst]]--"
        puts "   $lst = --[ set $lst]--"
        set err 1
    }
}

if { $err} {
    puts "   llength lst_filenames = $num_items"
    puts "   lst_filenames = --$lst_filenames--"
    # set idx 0
    # puts "index = fname    name    cmd    icon"
    # foreach fname $lst_filenames name $lst_names cmd $lst_cmds icon $lst_icons {
    # 	puts "  $idx = $fname    $name    $cmd    $icon"
    # 	incr idx
    # }
    exit
}

# create temporary files to be installed in the window manager menu's
set tmpdir [ exec mktemp -d /tmp/gid-icons-XXXX]
# create menu directory files
# directory files need a vendor's prefix
set menu [ file join $tmpdir cimne-GiD.directory]
set submenu [ file join $tmpdir cimne-GiD${bits}-${version}.directory]
set helpsubmenu [ file join $tmpdir cimne-GiD${bits}-${version}-help.directory]
set verbose 0
create_desktop_menu_file $menu "GiD - Geometry and Data" $gid_icon $verbose
create_desktop_menu_file $submenu ${version} $gid_icon $verbose
# some gnomes appear entries in help-submenu also in the menu above..., so not using helpsubmenu
create_desktop_menu_file $helpsubmenu "Help" $help_icon $verbose

foreach fname $lst_filenames name $lst_names cmd $lst_cmds icon $lst_icons is_help $lst_is_help {
    set verbose 0
    set entry [ file join $tmpdir $fname]
    create_app_desktop_file $entry $name $cmd $icon $verbose
    set err [ catch {
        if { !$is_help} {
            exec xdg-desktop-menu $action $menu $submenu $entry >& /dev/null
            puts "Menu entry ${action}ed: $name"
        } else {
            exec xdg-desktop-menu $action $menu $submenu $helpsubmenu $entry >& /dev/null
            puts "Menu entry ${action}ed: Help/$name"
        }
    } err_txt]
    if { $err} {
        puts "Error ${action}ing $name : $err_txt"
        puts "   May be the package 'xdg-utils' is missing? Copying it directly to $::env(HOME)/Desktop/"
        set dst_dir [ file join $::env(HOME) Desktop]
        if { $is_help} {
            set dst_dir [ file join $dst_dir "GiD $version Help"]
            file mkdir $dst_dir
        }
        if { $action == "install"} {
            file mkdir $dst_dir
            file copy -force $entry $dst_dir
        } else {
            catch {
                file delete [ file join $dst_dir [ file tail $entry]]
                if { $is_help} {
                    # when deleting last entry, directory will be deleted too
                    file delete $dst_dir
                }
            }
        }
    }
    file delete -force $entry
}

file delete -force $helpsubmenu
file delete -force $submenu
file delete -force $menu

# Desktop icon:
# set userdesktop [ exec xdg-user-dir DESKTOP]
# set filename [ file join $userdesktop ${basename}-desktop.desktop]
set filename [ file join $tmpdir ${basename}-desktop.desktop]
set name "GiD ${bits} ${version}"
set cmd  [ file join $GIDDEFAULT gid]
set icon $gid_icon

set verbose 0
create_app_desktop_file $filename $name $cmd $icon $verbose
set err [ catch {
    exec xdg-desktop-icon $action $filename
} err_txt]
if { $err} {
    puts "Error ${action}ing $filename : $err_txt"
    puts "   May be the package 'xdg-utils' is missing? Copying it directly to $::env(HOME)/Desktop/"
    set dst_dir [ file join $::env(HOME) Desktop]
    if { $action == "install"} {
        file mkdir $dst_dir
        file copy -force $filename $dst_dir
    } else {
        catch {
            file delete [ file join $dst_dir [ file tail $filename]]
        }
    }
}
puts "Desktop icon ${action}ed: $name"
file delete -force $filename

file delete -force $tmpdir

# for firefox to open local gid with url's like    gid://open?model_id=8310   # @ apptesting.gidsimulation.com
# or when doing    xdg-open gid://open?model_id=8310
# or  gid://open?model_id=109090    # @ app.gidsimulation.com
# from https://unix.stackexchange.com/questions/497146/create-a-custom-url-protocol-handler
set dst [ file join $::env(HOME) .local share applications]
set basename gid-url-handler.desktop
set filename [ file join $dst $basename]
if { $action == "install"} {
    create_app_url_handler_file $filename \
        "GiD URL handler" \
        "[ file join $GIDDEFAULT gid] -n -t \"ExternalActions::ExternalConnect %u\"" \
        $gid_icon \
        $verbose
    # register new mime-type
    set err [ catch {
	puts "... xdg-mime default $basename $::_gid_web_mimetype"
        exec xdg-mime default $basename $::_gid_web_mimetype
	puts "... update-desktop-database $dst"
	exec update-desktop-database $dst
        puts "Mime-type 'gid' registered."
    } err_txt]
    if { $err} {
        puts "Error ${action}ing $filename : $err_txt"
        puts "   May be the package 'xdg-utils' is missing?"
        puts "   May be the package 'desktop-file-utils' is missing?"
        puts "   packages are needed to register gid web url handler."
    } else {
        puts "Please, log out and log in again to see the changes."
    }
} else {
    # uninstall
    # puts "Please, log out and log in again to see the changes."
    file delete -force $filename
    puts "Mime-type 'gid' unregistered."
}

exit
