
if { ![info exists ::GID_UNDO_USE_BACKUP] } {
    set ::GID_UNDO_USE_BACKUP 0
}

package require TclOO

oo::class create BlockCommands {
    variable _name
    variable _commands

    constructor { name } {
        set _name $name
        set _commands [list]
    }

    method SetName { name } {
        set _name $name
        return 0
    }
    method GetName { } {
        return $_name
    }
    method LappendCommand { command } {
        lappend _commands $command
        return 0
    }
    method GetCommands { } {
        return $_commands
    }
    method SetCommands { commands } {
        set _commands $commands
    }
    export SetName GetName LappendCommand GetCommands SetCommands
}


namespace eval Undo {
    variable UndoPriv
    variable UndoPrivSavedAfterBlock
    variable UndoPrivBatchCommandsRepresentBackup

    set UndoPriv(save_clear_history) 0
    #1: like always, saving clear in undo window the previous history
    #0: allow undo to a command prior to a save (but it is bad if save
    #overwriting the same model name read, the first model must be copied
    #somewhere to have a good start point)
    set UndoPriv(last_model_read) ""
}

proc Undo::ChangeState { text filename } {
    Undo::FillUndoWin $text $filename
}

proc Undo::UnsetBackupFilenameIndexBlock { type_and_filename } {
    variable UndoPrivSavedAfterBlock
    variable UndoPrivBatchCommandsRepresentBackup
    set i_block 0
    foreach item [lsort -integer [array names UndoPrivSavedAfterBlock]] {
        if { $UndoPrivSavedAfterBlock($item) == $type_and_filename } {
            set i_block $item
            break
        }
    }
    if { $i_block } {
        #if backup file is single, then second save delete a first save, must invalidate it as restore point
        array unset UndoPrivSavedAfterBlock($i_block)
    }
    #if file is saved again with the same name then will represent other commands that previous file
    lassign $type_and_filename type filename
    if { [info exists UndoPrivBatchCommandsRepresentBackup($filename)] } {
        array unset UndoPrivBatchCommandsRepresentBackup($filename)
    }
    return 0
}


proc Undo::ModelExists { modelname } {
    return [expr {[file exists $modelname] || [file exists $modelname.gid]}]
}

proc Undo::TrimLeftKeys { keys_trim command } {
    set n [llength $command]
    set i 0
    for {set i 0} {$i<$n} {incr i} {
        set word [lindex $command $i]
        if { [lsearch -nocase $keys_trim $word] == -1 } {
            if { $i } {
                set command [lrange $command $i end]
            }
            break
        }
    }
    if { $i==$n } {
        #any command different of escape or mescape
        set command [list]
    }
    return $command
}

proc Undo::TrimKeys { keys_trim command } {
    set i_start 0
    set i_end [expr [llength $command]-1]
    for {set i $i_start} {$i<=$i_end} {incr i} {
        set word [lindex $command $i]
        if { [lsearch -nocase $keys_trim $word] == -1 } {
            break
        }
        incr i_start
    }
    for {set i $i_end} {$i>=0} {incr i -1} {
        set word [lindex $command $i]
        if { [lsearch -nocase $keys_trim $word] == -1 } {
            break
        }
        incr i_end -1
    }
    set command [lrange $command $i_start $i_end]
    return $command
}

proc Undo::GetCommandsUndo { filename {also_postprocess 0} } {
    variable UndoPriv
    variable UndoPrivSavedAfterBlock
    variable UndoPrivBatchCommandsRepresentBackup

    array unset UndoPrivSavedAfterBlock
    #array unset UndoPrivBatchCommandsRepresentBackup

    #if { $filename == "" || ![file exists $filename] } return


    GiD_Project batchfile flush
    set fileid [open $filename r]
    fconfigure $fileid -encoding utf-8
    set position 0
    set UndoPriv(read_filename) ""
    set UndoPriv(read_filetype) ""

    while { ![eof $fileid] } {
        set position_before_gets [tell $fileid]
        set aa [gets $fileid]
        if { [string equal {*****MARK } [string range $aa 0 9]] } {
            set type [lindex $aa 1]
            if { $type == "START" } {
                set UndoPriv(read_filename) ""
                set UndoPriv(read_filetype) ""
                set position [tell $fileid]
            } elseif { $type == "READ" } {
                set model_name ""
                regexp {MARK READ[ ]*(.*)$} $aa trash model_name
                set filename_copied [Undo::HistoryGetCopiedModel $model_name]
                if { $filename_copied != "" } {
                    #the initial file really was modified, the initial version is in the history copy
                    set model_name $filename_copied
                }
                if { [Undo::ModelExists $model_name] } {
                    set UndoPriv(read_filename) $model_name
                    set UndoPriv(read_filetype) "MODEL"
                    set position [tell $fileid]
                }
            } elseif { $type == "BACKUP_READ" } {
                set UndoPriv(read_filename) ""
                set model_name ""
                regexp {MARK BACKUP_READ[ ]*(.*)$} $aa trash model_name
                if { [Undo::ModelExists $model_name] } {
                    set UndoPriv(read_filename) $model_name
                    set UndoPriv(read_filetype) "BACKUP"
                    set position [tell $fileid]
                }
            } elseif { $type == "SAVE" } {
                if { $UndoPriv(save_clear_history) } {
                    set model_name ""
                    regexp {MARK SAVE[ ]*(.*)$} $aa trash model_name
                    if { [Undo::ModelExists $model_name] } {
                        set UndoPriv(read_filename) $model_name
                        set UndoPriv(read_filetype) "MODEL"
                        set position [tell $fileid]
                    }
                }
            }
        }
    }


    set blocks_commands [list]
    set block_current "" ;#root, create a block by command
    set i_level_block 0 ;#to ignore nested blocks
    set lines_current_command [list]
    seek $fileid $position
    while { ![eof $fileid] } {
        gets $fileid aa
        if { [string equal {*****MARK } [string range $aa 0 9]] } {
            set type [lindex $aa 1]
            if { $type == "SAVE" } {
                set filename ""
                regexp {MARK SAVE[ ]*(.*)$} $aa trash filename
                if { [Undo::ModelExists $filename] } {
                    set type_and_filename [list MODEL $filename]
                    Undo::UnsetBackupFilenameIndexBlock $type_and_filename
                    set i_block [expr [llength $blocks_commands]-1] ;#saved just after do this command
                    set UndoPrivSavedAfterBlock($i_block) $type_and_filename
                }
                continue
            } elseif { $type == "BACKUP_SAVE" } {
                if { $::GID_UNDO_USE_BACKUP } {
                    set filename ""
                    regexp {MARK BACKUP_SAVE[ ]*(.*)$} $aa trash filename
                    if { [Undo::ModelExists $filename] } {
                        set type_and_filename [list BACKUP $filename]
                        Undo::UnsetBackupFilenameIndexBlock $type_and_filename
                        set i_block [expr [llength $blocks_commands]-1] ;#saved just after do this command
                        set UndoPrivSavedAfterBlock($i_block) $type_and_filename
                    }
                }
                continue
            } elseif { $type == "START_COMMAND" || $type == "START_COMMAND_TRANSPARENT" || $type == "START_COMMAND_TCL" } {
                #I am unable to print paired start/end, ignore all start, and use only end (implicit start after prev end)
                continue
            } elseif { $type == "END_COMMAND" || $type == "END_COMMAND_DUMMY" || $type == "END_COMMAND_TRANSPARENT" || $type == "END_COMMAND_TCL" } {
                if { [llength $lines_current_command] } {
                    set block_to_add ""
                    if { $block_current == "" } {
                        set block_to_add [BlockCommands new root]
                        lappend blocks_commands $block_to_add
                    } else {
                        set block_to_add $block_current
                    }

                    $block_to_add LappendCommand $lines_current_command
                    set lines_current_command [list]
                } else {
                    #ignore empty START_COMMAND+END_COMMAND (or maybe first unpaired END_COMMAND)
                    set to_stop 1
                }
                continue
            } elseif { $type == "START_COMMAND_VIEW" } {
                #jump the block until END_COMMAND_VIEW
                while { ![eof $fileid] } {
                    gets $fileid line_2
                    #set line_2 [string trim $line_2]]
                    if { [string equal {*****MARK END_COMMAND_VIEW} $line_2] } {
                        break
                    }
                }
                continue
            } elseif { $type == "START_BLOCK" } {
                if { $i_level_block==0 } {
                    set block_text [lrange $aa 2 end]
                    set block_current [BlockCommands new $block_text]
                    lappend blocks_commands $block_current
                }
                incr i_level_block
                continue
            } elseif { $type == "END_BLOCK" } {
                incr i_level_block -1
                if { $i_level_block==0 } {
                    #set block_text [lrange $aa 2 end]
                    set block_current ""
                }
                continue
            }
        } elseif { !$also_postprocess && [string equal -nocase "Postprocess" $aa] } {
            #jump the postprocess block
            while { ![eof $fileid] } {
                gets $fileid line_2
                #set line_2 [string trim $line_2]]
                if { [string equal -nocase "Preprocess" $line_2] } {
                    gets $fileid line_2
                    if { [string equal -nocase "Yes" $line_2] } {
                        break
                    }
                }
            }
            continue
        }
        if { $aa != "" } {
            lappend lines_current_command {*}$aa
        }
    }
    close $fileid


    if { [llength $lines_current_command] } {
        #maybe already is not finished the funcion waiting for escapes, consider the end the last line
        set block_to_add ""
        if { $block_current == "" } {
            set block_to_add [BlockCommands new root]
            lappend blocks_commands $block_to_add
        } else {
            set block_to_add $block_current
        }
        $block_to_add LappendCommand $lines_current_command
        set lines_current_command [list]
    }

    set blocks_commands_effective [list]
    set i_block_old -1
    set i_block_new -1
    set num_blocks_deleted 0
    foreach block_commands $blocks_commands {
        set block_commands_effective [list]
        set commands [$block_commands GetCommands]
        foreach command $commands {
            if { [Undo::IsEffectiveCommand $command] } {
                lappend block_commands_effective $command
            }
        }
        if { [llength $block_commands_effective] } {
            if { [llength $block_commands_effective] != [llength $commands] } {
                $block_commands SetCommands $block_commands_effective
            }
            lappend blocks_commands_effective $block_commands
            incr i_block_new
        } else {
            #must delete the whole block and renumber affected variables UndoPrivSavedAfterBlock...
            incr num_blocks_deleted
        }
        incr i_block_old
        set new_i_block($i_block_old) $i_block_new
    }

    if { $num_blocks_deleted } {
        set blocks_commands $blocks_commands_effective
    }

    if { $::GID_UNDO_USE_BACKUP } {
        if { $num_blocks_deleted } {
            #removing non effective blocks must renumber the index stored at UndoPrivSavedAfterBlock
            foreach i_block_old [lsort -integer [array names UndoPrivSavedAfterBlock]] {
                if { [info exists new_i_block($i_block_old)] } {
                    set i_block_new $new_i_block($i_block_old)
                    if { $i_block_new != $i_block_old } {
                        set UndoPrivSavedAfterBlock($i_block_new) $UndoPrivSavedAfterBlock($i_block_old)
                        unset UndoPrivSavedAfterBlock($i_block_old)
                    }
                }
            }
        }

        foreach i_block [lsort -integer [array names UndoPrivSavedAfterBlock]] {
            lassign $UndoPrivSavedAfterBlock($i_block) file_to_start_type file_to_start_name
            set UndoPrivBatchCommandsRepresentBackup($file_to_start_name) [lrange $blocks_commands 0 $i_block]
        }

        #if first command is something like
        #{-tcl- GiD_Project backup read R:/Temp/gid7/backup-1584}
        #replace it in commands by its sub-commands
        if { [llength $blocks_commands] } {
            set first_block_commands [lindex $blocks_commands 0]
            set first_commands [$first_block_commands GetCommands]
            set command [lindex $first_commands 0]
            set file_to_start_name [Undo::IsBackupReadCommand $command]
            if { $file_to_start_name != "" } {
                set file_to_start_type "BACKUP"
                if { [info exists UndoPrivBatchCommandsRepresentBackup($file_to_start_name)] } {
                    if { [llength $first_commands] > 1 } {
                        #unexpected more commands in this block of the root
                        set to_stop 1
                    }
                    set num_insert [llength $UndoPrivBatchCommandsRepresentBackup($file_to_start_name)]
                    set blocks_commands [concat $UndoPrivBatchCommandsRepresentBackup($file_to_start_name) [lrange $blocks_commands 1 end]]
                    #replace 1 block by $num_insert commands at start, UndoPrivSavedAfterBlock indices must be updated
                    foreach i_block [lsort -integer -decreasing [array names UndoPrivSavedAfterBlock]] {
                        #-decreasing here is crutial! to not overwrite data!
                        set UndoPrivSavedAfterBlock([expr $i_block+$num_insert-1]) $UndoPrivSavedAfterBlock($i_block)
                        unset UndoPrivSavedAfterBlock($i_block)
                    }
                    set UndoPrivSavedAfterBlock([expr $num_insert-1]) [list $file_to_start_type $file_to_start_name]
                }
            }
        }
        #and similar if first command is something like
        #Files Read C:/temp/test_undo_1
        #but then instead of process commands it is used UndoPriv(read_filename) and UndoPriv(read_filetype)==MODEL
        if { $UndoPriv(read_filename) != "" } {
            if { $UndoPriv(read_filetype) == "MODEL" } {
                set file_to_start_name $UndoPriv(read_filename)
                set file_to_start_type $UndoPriv(read_filetype)
                if { [info exists UndoPrivBatchCommandsRepresentBackup($file_to_start_name)] } {
                    set num_insert [llength $UndoPrivBatchCommandsRepresentBackup($file_to_start_name)]
                    set blocks_commands [concat $UndoPrivBatchCommandsRepresentBackup($file_to_start_name) $blocks_commands]
                    #insert $num_insert blocks at start, UndoPrivSavedAfterBlock indices must be updated
                    foreach i_block [lsort -integer -decreasing [array names UndoPrivSavedAfterBlock]] {
                        #-decreasing here is crutial! to not overwrite data!
                        set UndoPrivSavedAfterBlock([expr $i_block+$num_insert]) $UndoPrivSavedAfterBlock($i_block)
                        unset UndoPrivSavedAfterBlock($i_block)
                    }
                    set UndoPrivSavedAfterBlock([expr $num_insert-1]) [list $file_to_start_type $file_to_start_name]
                }

            }

        }


    }
    return $blocks_commands
}

#avoid commands that are visual only and can be avoided in batch for undo
#avoid also save commands
proc Undo::IsEffectiveCommand { command_original } {
    set is_effective 1
    set command [Undo::TrimLeftKeys [list MEscape Escape] $command_original]
    set c0 [lindex $command 0]
    if { [string index $c0 0] == "'" } {
        if { $c0 == "'Redraw" } {
            set c1 [lindex $command 1]
            if { $c1 == "" } {
                #no more commands after 'Redraw
                set is_effective 0
            }
        } elseif { $c0 == "'Groups" } {
            set c1 [lindex $command 1]
            if { $c1== "-Show_Legend" || $c1 == "Draw" } {
                set is_effective 0
            }
        } elseif { $c0 == "'HardCopy" } {
            set c1 [lindex $command 1]
            if { $c1== "CopyClipboard" } {
                set is_effective 0
            }
        }
    } elseif { $c0 == "Data" } {
        set c1 [lindex $command 1]
        if { $c1 == "Materials" } {
            set c2 [lindex $command 2]
            if { $c2 == "DrawMaterial" } {
                set is_effective 0
            }
        } elseif { $c1 == "Conditions" } {
            set c2 [lindex $command 2]
            if { $c2 == "DrawCond" } {
                set is_effective 0
            }
        }
    } elseif { $c0 == "Files" } {
        set c1 [lindex $command 1]
        if { $c1 == "SaveAs" } {
            set is_effective 0
        } elseif { $c1 == "Save" } {
            set is_effective 0
        }
    } elseif { $c0 == "Meshing" } {
        set c1 [lindex $command 1]
        if { $c1 == "DrawSizes" } {
            set is_effective 0
        }
    } elseif { $c0 == "Utilities" } {
        set c1 [lindex $command 1]
        if { $c1 == "DrawCurvature" } {
            set is_effective 0
        } elseif { $c1 == "DrawHigher" } {
            set is_effective 0
        } elseif { $c1 == "DrawNormals" } {
            set is_effective 0
        } elseif { $c1 == "Id" } {
            set is_effective 0
        } elseif { $c1 == "Dist" } {
            set is_effective 0
        }
    } elseif { ![llength $command] } {
        #check to remove command with only MEscape, like usually last line of batch
        set is_effective 0
    }
    return $is_effective
}

proc Undo::IsBackupReadCommand { command_original } {
    set backup_filename ""
    set command [lindex $command_original 0]
    set c0 [lindex $command 0]
    if { $c0 == "-tcl-" } {
        set c1 [lindex $command 1]
        if { $c1 == "GiD_Project" } {
            set c2 [lindex $command 2]
            if { $c2 == "backup" } {
                set c3 [lindex $command 3]
                if { $c3 == "read" } {
                    set backup_filename [lindex $command 4]
                }
            }
        }
    }
    return $backup_filename
}

proc Undo::FillUndoRedoWinText { blocks_commands_original text show_reversed } {
    variable UndoPriv
    if { $show_reversed} {
        set blocks_commands [lreverse $blocks_commands_original]
    } else {
        set blocks_commands $blocks_commands_original
    }
    $text configure -state normal
    $text delete 1.0 end
    set tag normal
    set i 0
    foreach block_commands $blocks_commands {
        if { [$block_commands GetName] == "root" } {
            if { $show_reversed} {
                set commands [lreverse [$block_commands GetCommands]]
            } else {
                set commands [$block_commands GetCommands]
            }

            foreach command $commands {
                incr i
                set command_decorated [Undo::TrimKeys [list MEscape Escape 'Redraw] $command]
                $text insert end "[format %4d $i]: $command_decorated\n" $tag
            }
        } else {
            incr i
            $text insert end "[format %4d $i]: block [$block_commands GetName]\n" $tag
        }
    }
    $text tag add selected 1.0 1.end
    $text tag configure selected -background "dodger blue" -foreground white
    $text yview end
    $text configure -state disabled
    $text tag configure normal -foreground black -elide 0
    #$text tag configure secondary1 -foreground blue -elide 0 # -elide 1 to hide these lines
}

proc Undo::FillUndoWin { text filename } {
    set blocks_commands [Undo::GetCommandsUndo $filename]
    Undo::FillUndoRedoWinText $blocks_commands $text 1
}

proc Undo::FillRedoWin { text } {
    variable UndoPriv
    if { [info exists UndoPriv(blocks_commands_to_redo)] && [llength $UndoPriv(blocks_commands_to_redo)] } {
        set blocks_commands $UndoPriv(blocks_commands_to_redo)
        Undo::FillUndoRedoWinText $blocks_commands $text 0
    }
}

proc Undo::ChangeSelection { text x y } {
    set y [expr $y-[winfo rooty $text]]
    regexp {([^.]*)} [$text index @$x,$y] trash line
    Undo::ChangeSelectionLine $text $line
}

proc Undo::ChangeSelectionLine { text line } {
    $text tag remove selected 1.0 end
    $text tag add selected 1.0 $line.end
}

proc Undo::tkTextAutoScanNoSel {w} {
    global tkPriv
    if {![winfo exists $w]} return
    if {$tkPriv(y) >= [winfo height $w]} {
        $w yview scroll 2 units
    } elseif {$tkPriv(y) < 0} {
        $w yview scroll -2 units
    } elseif {$tkPriv(x) >= [winfo width $w]} {
        $w xview scroll 2 units
    } elseif {$tkPriv(x) < 0} {
        $w xview scroll -2 units
    } else {
        return
    }
    set tkPriv(afterId) [after 50 Undo::tkTextAutoScanNoSel $w]
}

proc Undo::DoReadModelOrBackup { filename filetype } {
    if { [Undo::ModelExists $filename] } {
        if { $filetype == "MODEL" } {
            if { [GiD_Info Project AreChanges] } {
                GiD_Process Mescape Files Read No $filename
            } else {                
                GiD_Process Mescape Files Read $filename
            }
        } elseif { $filetype == "BACKUP" } {
            GidUtils::EvalAndEnterInBatch GiD_Project backup read $filename
        } else {
            error "Unexpected filetype $filetype"
        }
    } else {
        W "Model '$filename' not found"
    }
    return 0
}

proc Undo::DoFileNew { } {
    set old_MaintainProblemTypeInNew [GiD_Set MaintainProblemTypeInNew]
    GiD_Set MaintainProblemTypeInNew 0 ;#avoid reload problemtype to not load twice
    DoFilesNew AUTO
    GiD_Set MaintainProblemTypeInNew $old_MaintainProblemTypeInNew
}

proc Undo::DoProcessBlocksCommands { blocks_commands } {
    foreach block_command $blocks_commands {
        set block_name [$block_command GetName]
        if { $block_name != "root" } {
            GiD_EnterInBatch "*****MARK START_BLOCK $block_name\n"
        }
        foreach data [$block_command GetCommands] {
            GiD_Process {*}$data
        }
        if { $block_name != "root" } {
            GiD_EnterInBatch "*****MARK END_BLOCK $block_name\n"
        }
    }
    return 0
}

proc Undo::ExecUndo { w text } {
    set num_undo [lindex [split [$text index selected.last] .] 0]
    destroy $w
    if { $num_undo > 0 } {
        Undo::DoUndoN $num_undo
    }
    return
}

proc Undo::ExecRedo { w text } {
    set num_redo [lindex [split [$text index selected.last] .] 0]
    if { $num_redo > 0 } {
        Undo::DoRedoN $num_redo
    }
    destroy $w
}

proc Undo::DoUndoLast { } {
    return [Undo::DoUndoN 1]
}

proc Undo::DoUndoN { num_undo } {
    variable UndoPriv
    variable UndoPrivSavedAfterBlock
    variable UndoPrivBatchCommandsRepresentBackup

    set blocks_commands [Undo::GetCommandsUndo [GiD_Project batchfile get name]]

    if { [llength $blocks_commands] } {
        #set t_start [clock milliseconds]
        set UndoPriv(doing_undo) 1
        set file_to_start_name ""
        set file_to_start_type ""
        if { $UndoPriv(read_filename) != "" && $UndoPriv(read_filetype) != "BACKUP" } {
            set file_to_start_name $UndoPriv(read_filename)
            set file_to_start_type $UndoPriv(read_filetype)
        }
        #remove transparent commands starting by ' (like 'Rotate, etc)
        set i_block_to_start 0
        set i_block_to_end [expr [llength $blocks_commands]-1-$num_undo]
        if { $i_block_to_end>=$i_block_to_start } {
            #find if there are saved points between $i_line_to_start and $i_line_to_end, and use the biggest one
            foreach i_block_saved [lsort -integer -decreasing [array names UndoPrivSavedAfterBlock]] {
                if { $i_block_saved>=$i_block_to_start && $i_block_saved<=$i_block_to_end } {
                    set i_block_to_start [expr $i_block_saved+1]
                    lassign $UndoPrivSavedAfterBlock($i_block_saved) file_to_start_type file_to_start_name
                    break
                }
            }
        } else {
            if { [info exists UndoPrivBatchCommandsRepresentBackup($file_to_start_name)] } {
                #ignore the file to read
                #e.g. create line, save, create arc, undo-1, redo-1, undo-1 ,undo-1
                #the file saved represents the first block to create line, and must not be read
                #the result must be an empty model without any line
                #the file must not be read !! because this file was not explicitly the start
                #but and auxiliary read, with associated equivalent commmands (expected faster read than repeat the commands)
                set file_to_start_name ""
                set file_to_start_type ""
            }
        }
        set blocks_commands_to_process [lrange $blocks_commands $i_block_to_start $i_block_to_end]
        set blocks_commands_to_redo [lrange $blocks_commands $i_block_to_end+1 end]
        if { [info exists UndoPriv(blocks_commands_to_redo)] && [llength $UndoPriv(blocks_commands_to_redo)] } {
            set blocks_commands_to_redo [concat $blocks_commands_to_redo $UndoPriv(blocks_commands_to_redo)]
        }
        GidUtils::DisableGraphics
        GidUtils::WaitState
        set modelname_before [GiD_Info Project ModelName]
        set problemtype_before [GiD_Info Project ProblemType]
        #DWInitUserDataOptions
        if { $file_to_start_name == "" } {
            Undo::DoFileNew
        } else {
            Undo::DoReadModelOrBackup $file_to_start_name $file_to_start_type
        }
        Undo::DoProcessBlocksCommands $blocks_commands_to_process
        if { $modelname_before != "UNNAMED" } {
            GiD_SetModelName $modelname_before
        }

        GidUtils::EnableGraphics
        if { ![GroupsXmlDocExists] && [GidUtils::ExistsWindow CUSTOMLIB] } {
            #e.g. problemtype not loaded but its window not closed because disabled windows
            GidUtils::CloseWindow CUSTOMLIB
        }
        foreach window {LAYER GROUPS CUSTOMLIB} {
            GidUtils::UpdateWindow $window
        }
        GiD_Redraw
        set UndoPriv(blocks_commands_to_redo) $blocks_commands_to_redo
        if { $problemtype_before != [GiD_Info Project ProblemType] } {
            GiDMenu::UpdateMenus
        }
        GiD_Project batchfile flush
        GidUtils::EndWaitState
        set UndoPriv(doing_undo) 0
        #set t_end [clock milliseconds]
        GidUtils::SetWarnLine [_ "Done undo"]
        #GidUtils::SetWarnLine [_ "Done undo %s seconds" [expr ($t_end-$t_start)*1e-3]]
    } else {
        GidUtils::SetWarnLine [_ "Nothing to undo"]
    }
}

proc Undo::ResetRedo { } {
    variable UndoPriv
    set UndoPriv(blocks_commands_to_redo) [list]
}

proc Undo::DoRedoLast { } {
    Undo::DoRedoN 1
}

proc Undo::DoRedoN { num_redo } {
    variable UndoPriv
    if { [info exists UndoPriv(blocks_commands_to_redo)] && [llength $UndoPriv(blocks_commands_to_redo)] } {
        set UndoPriv(doing_undo) 1
        set blocks_commands_to_process [lrange $UndoPriv(blocks_commands_to_redo) 0 $num_redo-1]
        set blocks_commands_to_redo [lrange $UndoPriv(blocks_commands_to_redo) $num_redo end]
        if { [llength $blocks_commands_to_process] } {
            GidUtils::DisableGraphics
            GidUtils::WaitState
            #GiD_Process MEscape ;#to ensure finish last command
            Undo::DoProcessBlocksCommands $blocks_commands_to_process
            GidUtils::EndWaitState
            GidUtils::EnableGraphics
            foreach window {LAYER GROUPS CUSTOMLIB} {
                GidUtils::UpdateWindow $window
            }
            GiD_Redraw
        }
        set UndoPriv(blocks_commands_to_redo) $blocks_commands_to_redo
        GiD_Project batchfile flush
        set UndoPriv(doing_undo) 0
        GidUtils::SetWarnLine [_ "Done redo"]
    } else {
        GidUtils::SetWarnLine [_ "Nothing to redo"]
    }
}

proc Undo::UndoWindow { } {
    variable UndoPriv
    set w .gid.undo
    set filename [GiD_Project batchfile get name]
    if { $filename == "" } {
        WarnWin [_ "To enable Undo set on 'Write Batch file' in preferences window"]
        return
    }
    if { ![info exists UndoPriv(save_clear_history)] } {
        set UndoPriv(save_clear_history) 0
    }
    InitWindow2 $w -title [_ "Undo multiple"] \
        -geometryvariable PreUndoWindowGeom \
        -initcommand Undo::UndoWindow -ontop
    if { ![winfo exists $w] } {
        return ;# windows disabled || UseMoreWindows == 0
    }
    ttk::frame $w.fr1
    ttk::label $w.fr1.l -text [_ "Select commands to undo"]
    #ttk::checkbutton $w.fr1.save_clear_history -text [_ "Save clear history"] -variable Undo::UndoPriv(save_clear_history) -command [list Undo::ChangeState $w.t $filename]
    text $w.t -xscrollcommand [list $w.hscr set] -yscrollcommand [list $w.vscr set] -wrap none
    ttk::scrollbar $w.vscr -orient vertical -command [list $w.t yview]
    ttk::scrollbar $w.hscr -orient horizontal -command [list $w.t xview]
    ttk::frame $w.frmButtons -style BottomFrame.TFrame
    ttk::button $w.frmButtons.btnundo -text [_ "Undo"] -command [list Undo::ExecUndo $w $w.t] -underline 0 -style BottomFrame.TButton
    ttk::button $w.frmButtons.btnclose -text [_ "Close"] -command [list destroy $w] -underline 0 -style BottomFrame.TButton
    #grid $w.fr1.l $w.fr1.save_clear_history -padx 5 -pady 5
    grid $w.fr1.l -padx 5 -pady 5
    grid $w.fr1
    grid $w.t $w.vscr -sticky nsew
    grid configure $w.vscr -sticky ns -rowspan 2
    grid $w.hscr -sticky ew
    grid $w.frmButtons -sticky ews -pady {2 0} -columnspan 2
    grid $w.frmButtons.btnundo $w.frmButtons.btnclose -sticky ew -padx 5 -pady 6
    grid anchor $w.frmButtons center
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 1 -weight 1
    focus $w.frmButtons.btnclose
    bind $w <Alt-c> "$w.frmButtons.btnclose invoke"
    bind $w <Escape> "$w.frmButtons.btnclose invoke"
    Undo::FillUndoWin $w.t $filename
    $w.t yview moveto 0.0
    bind $w.t <1> "Undo::ChangeSelection $w.t %X %Y ; break"
    bind $w.t <B1-Motion> "Undo::ChangeSelection $w.t %X %Y ; break"
    bind $w.t <B1-Leave> {
        set tkPriv(x) %x
        set tkPriv(y) %y
        Undo::tkTextAutoScanNoSel %W
        break
    }
    bind $w.t <Configure> [list ConfigureListScrollbars $w.t $w.hscr $w.vscr]
    bind $w.t <$::gid_right_button> [list Undo::MenuContextualText $w.t %x %y %X %Y 1]
    grab $w
}

proc Undo::RedoWindow { } {
    variable UndoPriv
    set w .gid.redo
    if { ![info exists UndoPriv(blocks_commands_to_redo)] || ![llength $UndoPriv(blocks_commands_to_redo)] } {
        return 0
    }
    InitWindow2 $w -title [_ "Redo multiple"] \
        -geometryvariable PreRedoWindowGeom \
        -initcommand Undo::RedoWindow -ontop
    if { ![winfo exists $w] } {
        return 0;# windows disabled || UseMoreWindows == 0
    }
    ttk::frame $w.fr1
    ttk::label $w.fr1.l -text [_ "Select commands to redo"]
    text $w.t -xscrollcommand [list $w.hscr set] -yscrollcommand [list $w.vscr set] -wrap none
    ttk::scrollbar $w.vscr -orient vertical -command [list $w.t yview]
    ttk::scrollbar $w.hscr -orient horizontal -command [list $w.t xview]
    ttk::frame $w.frmButtons -style BottomFrame.TFrame
    ttk::button $w.frmButtons.btnundo -text [_ "Redo"] -command [list Undo::ExecRedo $w $w.t] -underline 0 -style BottomFrame.TButton
    ttk::button $w.frmButtons.btnclose -text [_ "Close"] -command [list destroy $w] -underline 0 -style BottomFrame.TButton
    grid $w.fr1.l -padx 5 -pady 5
    grid $w.fr1
    grid $w.t $w.vscr -sticky nsew
    grid configure $w.vscr -sticky ns -rowspan 2
    grid $w.hscr -sticky ew
    grid $w.frmButtons -sticky ews -pady {2 0} -columnspan 2
    grid $w.frmButtons.btnundo $w.frmButtons.btnclose -sticky ew -padx 5 -pady 6
    grid anchor $w.frmButtons center
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 1 -weight 1
    focus $w.frmButtons.btnclose
    bind $w <Alt-c> "$w.frmButtons.btnclose invoke"
    bind $w <Escape> "$w.frmButtons.btnclose invoke"
    Undo::FillRedoWin $w.t
    $w.t yview moveto 0.0
    bind $w.t <1> "Undo::ChangeSelection $w.t %X %Y ; break"
    bind $w.t <B1-Motion> "Undo::ChangeSelection $w.t %X %Y ; break"
    bind $w.t <B1-Leave> {
        set tkPriv(x) %x
        set tkPriv(y) %y
        Undo::tkTextAutoScanNoSel %W
        break
    }
    bind $w.t <Configure> [list ConfigureListScrollbars $w.t $w.hscr $w.vscr]
    bind $w.t <$::gid_right_button> [list Undo::MenuContextualText $w.t %x %y %X %Y 0]
    grab $w
}

proc Undo::MenuContextualText { text x y X Y reversed} {
    set menu $text.menu
    destroy $menu
    menu $menu -tearoff 0
    $menu add command -label [_ "Copy all commands"] -command [list Undo::CopyTextCommands $text $reversed]
    $menu add command -label [_ "Copy last selected"] -command [list Undo::CopyTextLastSelectedCommand $text]
    tk_popup $menu $X $Y
}

proc Undo::CopyTextCommands { text reversed } {
    set num_blocks [expr [lindex [split [$text index end] .] 0]-2] ;#I don't know why text has alwasy 2 extra empty lines!!
    set i_block_to_start 0
    set i_block_to_end [expr $num_blocks-1]
    set blocks_commands_to_process [list]
    for {set i_block $i_block_to_start} {$i_block<=$i_block_to_end} {incr i_block} {
        if { $reversed } {
            set i_text [expr $num_blocks-$i_block]
        } else {
            set i_text [expr $i_block+1]
        }
        set data [lrange [$text get -- $i_text.0 $i_text.end] 1 end]
        append blocks_commands_to_process $data\n
    }
    if {  $blocks_commands_to_process != "" } {
        clipboard clear -displayof $text
        clipboard append -displayof $text $blocks_commands_to_process
    }
}

proc Undo::CopyTextLastSelectedCommand { text } {
    set num_blocks [expr [lindex [split [$text index end] .] 0]-2] ;#I don't know why text has alwasy 2 extra empty lines!!
    set i_block [expr $num_blocks-[lindex [split [$text index selected.last] .] 0]]
    set blocks_commands_to_process ""
    set i_text [expr $num_blocks-$i_block]
    set data [lrange [$text get -- $i_text.0 $i_text.end] 1 end]
    append blocks_commands_to_process $data\n
    if {  $blocks_commands_to_process != "" } {
        clipboard clear -displayof $text
        clipboard append -displayof $text $blocks_commands_to_process
    }
}

proc Undo::GetSaveClearHistory { } {
    variable UndoPriv
    return $Undo::UndoPriv(save_clear_history)
}

proc Undo::HistorySaveModel { modelname } {
    variable UndoPriv
    set history_dir [file join [GiD_Info project TmpDirectory] history]
    if { ![file exists $history_dir] } {
        file mkdir $history_dir
    }
    set modelname_tail [file tail $modelname]
    set dest_dir [file join $history_dir $modelname_tail]
    if { [string tolower [file extension $dest_dir]] != ".gid" } {
        append dest_dir ".gid"
    }
    if { ![file exists $dest_dir] } {
        #by now do only a copy by name of the initially readed model, instead modelname_tail that can be long (Windows MAX_PATH 260 characters) can use numbers 1, 2
        set source_dir $modelname
        if { [string tolower [file extension $source_dir]] != ".gid" } {
            append source_dir ".gid"
        }
        if { [file exists $source_dir] && [file isdirectory $source_dir] } {
            file copy $source_dir $dest_dir
            set UndoPriv(history,$modelname) $dest_dir ;#to be read after instead the last saved version of the model...
        }
    } else {
        #W "Undo::HistorySaveModel $modelname, don't do another copy"
    }
    return 0
}


proc Undo::HistorySetLastModelRead { modelname } {
    variable UndoPriv
    set UndoPriv(last_model_read) $modelname
}

proc Undo::HistoryGetLastModelRead { } {
    variable UndoPriv
    return $UndoPriv(last_model_read)
}

proc Undo::HistoryReSetLastModelRead { } {
    variable UndoPriv
    set UndoPriv(last_model_read) ""
    array unset UndoPriv history,*
    set history_dir [file join [GiD_Info project TmpDirectory] history]
    if { [file exists $history_dir] && [file isdirectory $history_dir] } {
        file delete -force $history_dir
    }
}

proc Undo::HistoryGetCopiedModel { modelname } {
    variable UndoPriv
    set copied_modelname ""
    if { [info exists UndoPriv(history,$modelname)] } {
        set copied_modelname $UndoPriv(history,$modelname)
    }
    return $copied_modelname
}
