
# -----------------------------------------------------------------------------
# GiDHelpWindow
#
# Html based help system. Show the help window.
#
# Arguments -- pairs of <option value>, where option could be:
#
#   -title : window title, defaults to "Help on <ProblemType name>"
#
#   -dir : path to help content. If -dir is missing it defaults to
#          "<ProblemType dir>/html". Multilingual content could be present, in
#          such case it is assumed that there is a directory for each language
#          provided. If the currente language is not found, language 'en'
#          (for english) is tried. Finally if 'en' is not found the value
#          provided to the option -dir is assumed as the base directory for the
#          help content.
#
#   -start : is a path relative to the value of -dir to an html link. For
#            instance:
#             -start html-version
#             -start html-tutorials/tutorial_1
#
#   -report : boolean value indicating if the the window format is report. If
#             report is requested no tree is showed, only the content pane is
#             showed.
#
# Structure of the help contents--
#

#   Assuming we have chosen html as the base directory for the multilingual
#   help contents we can have the following structure:
#
#      html
#       \__ en - English content
#
#       \__ es - Spanish content
#
#   Each content will provably have a directory structure to organize the
#   information. By default the help system build a tree resembling the
#   directory structre of the help content. In this sense there will be an
#   internal node for each subdirectory, finally the html documents will be
#   the terminal nodes of the tree.
#
#   We can also provide a help.conf configuration file in order to provide
#   more information about the estructure of the help. In a help file we can
#   specify a table of contents (TocPage), help subdirectories (HelpDirs)
#   and topic index (IndexPage).
#
# HelpDirs --
#
#  With HelpDirs we can specify wich of the subdirectories will be internal
#  nodes of the help tree. Also we can specify label for the node and a link
#  to load when the noded is clicked. The link is relative the node. For
#  instance:
#
#      HelpDirs {html-version "GiD Help" "intro/intro.html"} \
#               {html-customization "GiD Customization"} \
#               {html-faq "Frequently Asked Questions"} \
#               {html-tutorials "GiD Tutorials" "tutorials_toc.html"} \
#               {html_whatsnew "Whats New"}
#
# TocPage --
#
#  TocPage define an html page as a page describing a table of contents of
#  the current node (current directory). We have considered two ways for
#  specifying a table of content:
#
#     1- <UL> <LI> ... </UL> (default)
#     2- <DT> <DL> ... </DT>
#
#  The first is the one generated by texinfo.
#
#  For instance:
#
#    TocPage gid_toc.html
#
#    TolcPage contents.ht DT
#
# IndexPage --
#
#  If we specify by IndexPage a topic index we can take advantage of the
#  search index. In indexPage we can provide a set of html index pages along
#  with the structure type of the index. The type of the index could be:
#
#    1- <DIR> <LI> ... </DIR> (default)
#    2- <UL> <LI> ... </UL> (only one level of <UL>)
#
#  The first is the one generated by texinfo.
#
#  For instance:
#
#    IndexPage html-version/gid_18.html html-faq/faq_11.html
#
#
# help.conf example, taken from info dir in gid instalation:
#
#    HelpDirs \
#      {html-version "GiD Help" "intro/intro.html"} \
#      {html-customization "GiD Customization"} \
#      {html-faq "Frequently Asked Questions"} \
#      {html-tutorials "GiD Tutorials" "tutorials_toc.html"} \
#      {html_whatsnew "Whats New"}
#
#      IndexPage html-version/gid_18.html html-faq/faq_11.html
# -----------------------------------------------------------------------------

proc GiDGetHelpLangDir { base_dir } {
    # string trim to work also if language ies "en " by some old wrong gid.ini
    set lan [string trim [GiD_Set Language]]
    set found 0
    foreach dl [list $lan en] {
        set lan_dir [file join $base_dir $dl]
        if {[file exists $lan_dir]} {
            set found 1
            break
        }
    }
    if {$found} {
        return $lan_dir
    }
    return $base_dir
}

proc GiDGetInfoDir { } {
    GiDGetHelpLangDir [file join $::GIDDEFAULT info]
}


proc GiDOpenLognoter { dbfile {open_page ""} {select_in_tree ""} {key ""} } {
    set execute_commands ""
    if { $open_page ne "" } {
        append execute_commands ";[list showpage $open_page]"
    }
    if { $select_in_tree ne "" } {
        append execute_commands ";[list toctree_process expand_page $select_in_tree]"
        append execute_commands ";[list toctree_process select_in_tree [list $select_in_tree]]"
    }

    # string trim to work also if language ies "en " by some old wrong gid.ini
    set lan [string trim [GiD_Set Language]]
    set argv [list -readprefs 0 -readonly 1 -language $lan \
        -manageicontray 0 -prefs_data \
        [list view_languages_toolbar 0 add_more_pages_to_tree 0 \
        view_main_menu 0 geometry_default {{} 940x700 {tree 200}}] \
        -hide_on_exit 0 -execute_commands $execute_commands \
        -title [_ "Help viewer"] -saveprefs 0 \
        -dbtype sqlite -notebookname $dbfile -key $key]

    if { ![interp exists lognoter_intp] } {
        interp create lognoter_intp
        lognoter_intp eval lappend auto_path $::auto_path
        #lognoter_intp eval [list set auto_path $::auto_path]
        lognoter_intp eval [list set argv [list -language $lan]]
        lognoter_intp eval [list set ::GIDDEFAULT $::GIDDEFAULT]
    }

    package require Lognoter

    #interp alias lognoter_intp exit "" return
    #lappend argv -external_master_callback eval_master

    # necessary to avoid loading different versions in main interp and slave interp
    #lognoter_intp eval [list package require -exact treectrl [package require treectrl]]

    set r [lognoter_intp eval lognoter .%AUTO% $argv]
    #interp alias lognoter_intp eval_master "" gid_groups_conds::_report_data_from_lognoter_cmd
    if { ![winfo exists .gid]  } {
        #not exit if opened from GiD
        interp alias lognoter_intp exit {} exit ;#else not exit or exit raise an error when closing lognoter
    } else {
        if { $::tcl_platform(os) == "Darwin"} {
            # if GiD exists, if lognoter exists, do not exit from GiD
            proc ::DoNothing  {} {}
            interp alias lognoter_intp exit {} ::DoNothing
        }
    }

    # we can not make a source of the file containing InitializeGiDThemes, it does several other things we don't like
    lognoter_intp eval [ list set ::GidPriv(ThemeSelection) $::GidPriv(ThemeSelection)]
    lognoter_intp eval [ info body InitializeGiDThemes]
}

proc GiDCloseLognoter {} {
    if { [interp exists lognoter_intp] } {
        interp delete lognoter_intp
        # interp eval lognoter_intp [ list destroy .lognoter3]
        # interp eval lognoter_intp [ list destroy .]
        # interp eval lognoter_intp exit
        # must forget in order to be able to open again lognoter with GiDOpenLognoter
        package forget Lognoter
    }
}

proc GiDHelpWindowLognoter { {filename gid_reference_manual.wnl} } {
    if { [file pathtype $filename] == "relative" } {
        set full_filename [file join [GiDGetInfoDir] $filename]
    } else {
        set full_filename $filename
    }
    if { [file exists $full_filename] } {
        set open_page ""
        set select_in_tree ""
        set key ""
        GiDOpenLognoter $full_filename $open_page $select_in_tree $key
    }
}

proc GiDHelpWindow { args } {
    package require gid_helpviewer
    array set options {
        -type "REFERENCE"
        -dir ""
        -start ""
        -report 0
        -toplevel ""
        -tab tree
    }
    set options(-title) [_ "Help"]
    array set options $args

    switch [string toupper $options(-type)] {
        REFERENCE {
            set options(-dir) [GiDGetInfoDir]
            set options(-title) [_ "%s Reference" $::GidPriv(ProgName)]
        }
        REPORT {
            set options(-title) [_ "Report"]
            set options(-report) 1
        }
        CUSTOM {
            set options(-toplevel) .ctmhelp
        }
    }

    if {$options(-dir) eq ""} {
        error [_ "Calling GiDHelpWindow: should provide a -dir option"]
    }

    # on fedora 30 tkhtml crashes
    set is_fedora_30 0
    set err_txt ""
    catch {
        set fi [ open /etc/fedora-release]
        set contents [ read $fi]
        close $fi
        scan $contents "Fedora release %d" version
        if { $version >= 30} {
            set is_fedora_30 1
        }
    }
    # if there is an error (catched) then it is not fedora >= 30
    if { $is_fedora_30} {
        set try_toc [ file join $options(-dir) $options(-start) *toc.html]
        set toc [ glob -nocomplain $try_toc]
        if { $toc == ""} {
            # try index file
            set try_toc [ file join $options(-dir) $options(-start) *index.html]
            set toc [ glob -nocomplain $try_toc]
            if { $toc == ""} {
                # get the first .html file
                set try_toc [ file join $options(-dir) $options(-start) *.html]
                set toc [ glob -nocomplain $try_toc]
            }
        }
        if { $toc != ""} {
            package require gid_cross_platform
            gid_cross_platform::open_by_extension $toc
        } else {
            ErrorWin [_ "Could not open *toc.html|*index.html|*.html at '%s'" \
                [ file join $options(-dir) $options(-start)]]
        }
        return
    }

    if {$options(-toplevel) eq ""} {
        GiDHelpViewer::Show $options(-dir) -title $options(-title) -try $options(-start) -report $options(-report) -tab $options(-tab)
    } else {
        GiDHelpViewer::Show $options(-dir) -title $options(-title) -try $options(-start) -report $options(-report) -tab $options(-tab) -base $options(-toplevel)
    }
}

proc HelpManualsAddCommands { m } {
    $m add command -label [_ "Help"]... -command {GiDHelpWindow -type REFERENCE -start GRM}
    $m add command -label [_ "Customization help"]... -command {GiDHelpWindow -type REFERENCE -start  GCM}
    $m add command -label [_ "Tutorials"]... -command {GiDHelpWindow -type REFERENCE -start GUM}
    $m add command -label [_ "What's new"]...  -command {GiDHelpWindow -type REFERENCE -start GWN}
    $m add command -label [_ "FAQ"]... -command {GiDHelpWindow -type REFERENCE -start  SD}
    $m add command -label [_ "Courses"]... -command {GiDHelpWindow -type REFERENCE -start  GCS}
}

proc HelpManualsMenu { w } {
    set m $w.m
    if { ![winfo exists $m] } {
        menu $m -tearoff no -borderwidth 1
        HelpManualsAddCommands $m
    }
    set x [expr [winfo rootx $w]]
    set y [expr [winfo rooty $w]+[winfo height $w]]
    GiD_PopupMenu $m $x $y
}

proc CreateGiDInfo_win {} {
    global GIDDEFAULT GidPriv tcl_platform

    set w .gid.info_gid
    InitWindow2 $w -title [_ "%s Info" $::GidPriv(ProgName)] \
        -geometryvariable PreGiDInfoWindowGeom \
        -initcommand CreateGiDInfo_win -onlygeometry
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0

    set colorbackground #ced1d6
    $w configure -background $colorbackground

    set image_top [gid_themes::GetImage gidinfo.png] ;#now not include any version logo
    set filename gid.svg
    set src [gid_themes::GetImageCommon $filename 2.66667]
    set bd 10
    set x [expr [image width $image_top]-[image width $src]-$bd]
    $image_top copy $src -to $x $bd

    set image_but1 [gid_themes::GetImage buton_about.png]
    set image_but2 [gid_themes::GetImage buton_help.png]
    set image_but3 [gid_themes::GetImage buton_tutorials.png]
    set image_but4 [gid_themes::GetImage buton_goto.png]
    set image_but5 [gid_themes::GetImage buton_register.png]
    set image_but6 [gid_themes::GetImage buton_atsign.png]

    set image_but1_over [gid_themes::GetImage buton_about_o.png]
    set image_but2_over [gid_themes::GetImage buton_help_o.png]
    set image_but3_over [gid_themes::GetImage buton_tutorials_o.png]
    set image_but4_over [gid_themes::GetImage buton_goto_o.png]
    set image_but5_over [gid_themes::GetImage buton_register_o.png]
    set image_but6_over [gid_themes::GetImage buton_atsign_o.png]

    tk::label $w.label_imgtop -image $image_top -borderwidth 0
    set f [frame $w.f -background $colorbackground]
    set text_color #336699
    set text_font BoldFont
    tk::label $f.label_about -text [_ "About %s" $::GidPriv(ProgName)] -font $text_font -foreground $text_color -background $colorbackground
    tk::label $f.label_help -text [_ "%s help" $::GidPriv(ProgName)] -font $text_font -foreground $text_color -background $colorbackground
    tk::label $f.label_tutorials -text [_ "%s tutorials" $::GidPriv(ProgName)] -font $text_font -foreground $text_color -background $colorbackground
    tk::label $f.label_web -text [_ "Visit %s web" $::GidPriv(ProgName)] -font $text_font -foreground $text_color -background $colorbackground
    tk::label $f.label_register -text [_ "Register %s" $::GidPriv(ProgName)] -font $text_font -foreground $text_color -background $colorbackground
    tk::label $f.label_contact  -text [_ "Contact"] -font $text_font -foreground $text_color -background $colorbackground

    tk::button $f.button#1 -image $image_but1 -command CreateWelcomeWin -borderwidth 0

    proc PostMenuButton { but menu } {
        $menu post {*}[ winfo pointerxy $but]
    }
    tk::button $f.button#2 -image $image_but2 -borderwidth 0 -command [list PostMenuButton $f.button#2 $f.button#2.m]
    menu $f.button#2.m -tearoff no -borderwidth 1
    HelpManualsAddCommands $f.button#2.m
    tk::button $f.button#3 -image $image_but3 -borderwidth 0 -command {GiDHelpWindow -type REFERENCE -start GUM ; destroy .gid.info_gid}
    tk::button $f.button#4 -image $image_but4 -command [list VisitWeb $GidPriv(ProgWeb)] -borderwidth 0
    tk::button $f.button#5 -image $image_but5 -borderwidth 0 -command [list PostMenuButton $f.button#5 $f.button#5.m]

    menu $f.button#5.m -tearoff no -borderwidth 1
    menu $f.button#5.m.m1 -tearoff no -borderwidth 1
    $f.button#5.m add cascade -label [_ "Register %s" $::GidPriv(ProgName)] -menu $f.button#5.m.m1
    $f.button#5.m.m1 add command -label [_ "Local"]... -command {Register::GiD_Local 0}
    $f.button#5.m.m1 add command -label [_ "Usb"]... -command {Register::GiD_Usb 0}
    $f.button#5.m.m1 add command -label [_ "Floating"]... -command {Register::GiD_Floating 0}
    $f.button#5.m.m1 add command -label [_ "Named user"]... -command {Login::Window}

    menu $f.button#5.m.m2 -tearoff no -borderwidth 1
    $f.button#5.m add cascade -label [_ "Register problem type"] -menu $f.button#5.m.m2
    $f.button#5.m.m2 add command -label [_ "Local"]... -command {Register::GiD_Local 1}
    $f.button#5.m.m2 add command -label [_ "Usb"]... -command {Register::GiD_Usb 1}
    $f.button#5.m.m2 add command -label [_ "Floating"]... -command {Register::GiD_Floating 1}

    tk::button $f.button#6 -image $image_but6 -command [list VisitWeb mailto:$GidPriv(ContactMail)] -borderwidth 0

    #binds - on mouse over
    bind $f.button#1 <Enter> [list %W configure -image $image_but1_over]
    bind $f.button#1 <Leave> [list %W configure -image $image_but1]
    bind $f.button#2 <Enter> [list %W configure -image $image_but2_over]
    bind $f.button#2 <Leave> [list %W configure -image $image_but2]
    bind $f.button#3 <Enter> [list %W configure -image $image_but3_over]
    bind $f.button#3 <Leave> [list %W configure -image $image_but3 ]
    bind $f.button#4 <Enter> [list %W configure -image $image_but4_over]
    bind $f.button#4 <Leave> [list %W configure -image $image_but4 ]
    bind $f.button#5 <Enter> [list %W configure -image $image_but5_over]
    bind $f.button#5 <Leave> [list %W configure -image $image_but5]
    bind $f.button#6 <Enter> [list %W configure -image $image_but6_over]
    bind $f.button#6 <Leave> [list %W configure -image $image_but6]

    # Geometry management
    grid $w.label_imgtop -columnspan 6 -sticky ne
    grid $f -sticky ew -padx 20
    grid $f.button#1 $f.label_about $f.button#3 $f.label_tutorials -pady 5
    grid configure $f.label_about $f.label_tutorials -sticky nw
    grid $f.button#2 $f.label_help $f.button#5 $f.label_register -pady 5
    grid configure $f.label_help $f.label_register -sticky nw
    grid $f.button#4 $f.label_web $f.button#6 $f.label_contact -pady 5
    grid configure $f.label_web $f.label_contact -sticky nw

    # Resize behavior management
    grid rowconfigure $w 0 -weight 1
    grid rowconfigure $w 2 -weight 0 -minsize 10

    grid columnconfigure $f 0 -weight 0 -minsize 15
    grid columnconfigure $f {1 2 3} -weight 0 -minsize 30
    grid columnconfigure $f 4 -weight 0 -minsize 40
    grid columnconfigure $f 5 -weight 0 -minsize 10
}


proc VisitWeb { url } {
    package require gid_cross_platform
    gid_cross_platform::open_by_extension $url
}

