
namespace eval ::gidttk {
    variable options
    variable conf.actions ""
    
    proc on.theme { } {
        variable options
        
        array set options [ttk::style configure .]
    }
    
    proc cget { option } {
        variable options
        
        if {![info exists options]} {
            on.theme
        }
        
        set options($option)
    }
}

#--------------------------------------General. For all of them---------

catch {
image create bitmap DownImage -data {
    #define downbut_width 14
    #define downbut_height 14
    static char downbut_bits[] = {
        0x00, 0x00, 0xe0, 0x01, 0xe0, 0x01, 0xe0, 0x01, 0xe0, 0x01, 0xfc, 0x0f,
        0xf8, 0x07, 0xf0, 0x03, 0xe0, 0x01, 0xc0, 0x00, 0x00, 0x00, 0xfe, 0x1f,
        0xfe, 0x1f, 0x00, 0x00};
}

image create bitmap UpImage -data {
    #define upbut_bits_width 14
    #define upbut_bits_height 14
    static unsigned char upbut_bits_bits[] = {
        0x00, 0x00, 0xfe, 0x1f, 0xfe, 0x1f, 0x00, 0x00, 0xc0, 0x00, 0xe0, 0x01,
        0xf0, 0x03, 0xf8, 0x07, 0xfc, 0x0f, 0xe0, 0x01, 0xe0, 0x01, 0xe0, 0x01,
        0xe0, 0x01, 0x00, 0x00};
}

image create bitmap LeftImage -data {
    #define leftbut_bits_width 14
    #define leftbut_bits_height 14
    static unsigned char leftbut_bits_bits[] = {
        0x00, 0x00, 0x06, 0x00, 0x06, 0x01, 0x86, 0x01, 0xc6, 0x01, 0xe6, 0x1f,
        0xf6, 0x1f, 0xf6, 0x1f, 0xe6, 0x1f, 0xc6, 0x01, 0x86, 0x01, 0x06, 0x01,
        0x06, 0x00, 0x00, 0x00};
}

image create bitmap RightImage -data {
    #define rightbut_bits_width 14
    #define rightbut_bits_height 14
    static unsigned char rightbut_bits_bits[] = {
        0x00, 0x00, 0x00, 0x18, 0x20, 0x18, 0x60, 0x18, 0xe0, 0x18, 0xfe, 0x19,
        0xfe, 0x1b, 0xfe, 0x1b, 0xfe, 0x19, 0xe0, 0x18, 0x60, 0x18, 0x20, 0x18,
        0x00, 0x18, 0x00, 0x00};
}

image create bitmap CloseImage -data {
    #define closebut_bits_width 14
    #define closebut_bits_height 14
    static unsigned char closebut_bits_bits[] = {
        0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x18, 0x0c, 0x30, 0x06, 0x60, 0x03,
        0xc0, 0x01, 0xc0, 0x01, 0x60, 0x03, 0x30, 0x06, 0x18, 0x0c, 0x0c, 0x18,
        0x00, 0x00, 0x00, 0x00};
}
}

proc CheckCnd { } {
    foreach type {point line surface volume layer group} {
        set over over_$type
        set CndNameList [GiD_Info conditions $over]
        if { [llength $CndNameList] } {
            return $type
        }
    }
    return ""
}

proc CheckMat { } {
    set MatList [GiD_Info materials]
    
    if { [llength $MatList] } {
        return 1
    } else {
        return ""
    }
}

proc CheckIntv { } {
    set PropList [GiD_Info intvdata]
    if { [llength $PropList] > 1 } {
        return 1
    } else {
        return ""
    }
}

proc CheckPrbD { } {
    set PropList [GiD_Info gendata]
    if { [llength $PropList] > 1 } {
        return 1
    } else {
        return ""
    }
}

proc CheckDataTree { } {
    if { [GroupsXmlDocExists] } {
        return 1
    } else {
        return ""
    }
}

proc FillMenuData { w } {
    global MenuDataEntries MenuDataCommands MenuDataAcc GidPriv
    global gidDataLabels gidDataCommands
    if { $::GidPriv(POSTDATA) == 0 } {
        return
    }
    # WarnWinText "FillMenuData $w"
    
    unset -nocomplain MenuDataEntries MenuDataCommands MenuDataAcc
      
    if { ![winfo exists $w] } {
        return 
    }
    ClearMenu $w
    
    set UnderChars {}
    set IdxForData 0
    set num_opt_added 0
    set entries [list [_ "Problem type#C#menu"] [_ "Data units#C#menu"] [_ "Interval#C#menu"] [_ "Local axes#C#menu"]]
    set commands {{Problem type} {Data units} {Interval} {Local axes}}
    foreach opt $commands optlabel $entries {
        if { ![info exists gidDataLabels($opt)]} {
            continue
        }
        if { $opt == "Data units" && ([GiD_Info Project ProblemType] != "UNKNOWN" && ![GiD_Info unitssystems]) } {
            continue
        }

        if { $opt == "Interval" && ([GiD_Info Project ProblemType] == "UNKNOWN" || ![lindex [GiD_Info intvdata]  0] )} {
            #not if there are not in .prb any interval data the menu Interval is not added 
            #if conditions require intervals must define the interval data block with some field, this field could be hidded
            continue
        }
        set pos_opt($opt) $num_opt_added
        lappend MenuDataEntries(0) $optlabel
        lappend MenuDataCommands(0) $gidDataCommands($opt)
        lappend MenuDataAcc(0) ""
        lappend MenuDataEntries(0) "---"
        lappend MenuDataCommands(0) ""
        lappend MenuDataAcc(0) ""
        lappend UnderChars [string tolower [string index $opt 0]]
        incr num_opt_added
    }
    # los nombres de los widgets para poderlos cambiar mas tarde
    set ::gidDataMenu(wid) $w
    set ::gidDataMenu(PT,wid) $w.0
    
    if { $num_opt_added } {
        set MenuDataEntries(0) [lrange $MenuDataEntries(0) 0 [expr [llength $MenuDataEntries(0)]-2]]
        set MenuDataCommands(0) [lrange $MenuDataCommands(0) 0 [expr [llength $MenuDataCommands(0)]-2]]
        set MenuDataAcc(0) [lrange $MenuDataAcc(0) 0 [expr [llength $MenuDataAcc(0)]-2]]
    }
    
    if { [info exists "gidDataLabels(Problem type)"] } {
        set MenuDataEntries(0,0) ""
        set IdxForData 2
    }
    
    if { [info exists gidDataLabels(Interval)] && [info exists pos_opt(Interval)] } {
        set idx_opt [expr {$pos_opt(Interval)*2}]
        set MenuDataEntries(0,$idx_opt) [list [_ "New#C#menu"] [_ "Current#C#menu"] [_ "Delete#C#menu"]]
        set MenuDataCommands(0,$idx_opt) {{Data Intervals NewInterval} {Data Intervals ChangeInterval} {Data Intervals DeleteInterval}}
    }
    
    if { [info exists "gidDataLabels(Local axes)"] } {
        set idx_opt [expr {$pos_opt(Local axes)*2}]
        set MenuDataEntries(0,$idx_opt) [list [_ "Define#C#menu"] [_ "Draw#C#menu"] [_ "Draw all#C#menu"] [_ "Delete#C#menu"] [_ "Delete all#C#menu"]]
        set MenuDataCommands(0,$idx_opt) {{Data LocalAxes DefineLocAxes} {Data LocalAxes DrawLocAxes} {Data LocalAxes DrawLocAxes -All-} \
            {Data LocalAxes DeleteLA} {Data LocalAxes DeleteAllLA}}
    }
    if { $num_opt_added } {
        CreateCascadedMenu $w MenuDataEntries MenuDataCommands "" 0 ""
    }
    
    UpdateDataOptions $w $IdxForData UnderChars
    
    set ::GidPriv(POSTDATA) 0
    
}

#can change LabelName and ListName contents
proc FindUnderChar { LabelName ListName } {
    upvar $LabelName text
    upvar $ListName UnderLineList
    
    set len [string length $text]
    set under [string first & $text]
    if { $under != -1 && $under < [expr {$len-1}] } {
        set nextchar [expr {$under+1}]
        set letter [string tolower [string index $text $nextchar]]
        if { [regexp {[a-zA-Z]} $letter] && [lsearch $UnderLineList $letter] == -1 } {
            #change text removing &
            lappend UnderLineList $letter
            set text [string replace $text $under $nextchar [string index $text $nextchar]]
        } else {
            set under -1
        }
    }
    if { $under == -1 } {
        for {set i 0 } { $i < $len } { incr i } {
            set letter [string tolower [string index $text $i]]
            if { ![regexp {[a-zA-Z]} $letter] } { continue }
            if { [lsearch $UnderLineList $letter] == -1 } {
                lappend UnderLineList $letter
                set under $i
                break
            }
        }
    }
    
    #this is an error of tkWinMenu.c at least until tcl 8.4.13
    #TkWinHandleMenuEvent(phwnd, pMessage, pwParam, plParam, plResult)
    #instead label[underline], Tcl_UtfAtIndex(label, underline) must be used for multibyte characters
    #and can't be corrected at tcl level, because
    #if use the character index, the submenu underline the right letter, but doesn't work
    #and if use the byte intex, it works (using <alt>-character), but the underline is showed badly
    
    #if { $under != -1 } {
        ##for multiple bytes characters, the -underline menu option require the byte number,
        ##not the character number (in japanese fail Alt-letter to invoke the menu)
        #set under [string bytelength [string range $text 0 [expr $under-1]]]
        #}
    return $under
}

proc DWInitUserDataOptions {} {
    
    global gidUserDataOptions
    global gidDataLabels
    global gidDataCommands
    global gidDataBooksHidden
    
    set gidUserDataOptions {}
    foreach dl {{Problem type} {Data units} Interval {Local axes} Conditions Materials {Interval data} {Problem data} DataTree} {
        set gidDataLabels($dl) ""
        set gidDataLabels($dl,singular) ""
    }
    array set gidDataCommands {
        {Problem type} {-np- FillMenuProblemtype %W}
        {Data units} {-np- DWDataUnits}
        Interval ""
        {Local axes} ""
    }
    
    array set gidDataBooksHidden {
        conditions ""
        materials ""
        intvdata ""
        gendata ""
    }
}

proc ShowDefaultDataInfo { } {
    WarnWin [_ "Conditions, Materials, Interval data and Problem data are option available only when defined inside a problem type"]
}

proc SetDefaultDataMenu { DataMenu IndexEntry ListName } {
    upvar $ListName UnderChars
    
    if { ![info exists UnderChars] } {
        set UnderChars "" ;#it is a list of used chars shared between procs
    }
    set DataEntry [list [list [_ "Conditions#C#menu"] condition.png] \
            [list [_ "Materials#C#menu"] material.png] \
            [list [_ "Interval data#C#menu"] ""] \
            [list [_ "Problem data#C#menu"] prdata.png]]
    
    foreach me $DataEntry {
        set text [lindex $me 0]
        set littleicon [lindex $me 1]
        set UnderIdx [FindUnderChar text UnderChars]
        $DataMenu insert $IndexEntry command -label $text -command ShowDefaultDataInfo \
            -underline $UnderIdx
        AddLittleIcon $littleicon $DataMenu $IndexEntry
        incr IndexEntry
    }
    
    
    if { $::tcl_platform(platform) != "windows" } {
        SetDataMenuState
    }
}

proc ExistsConditions { } {
    set exists 0
    foreach type {point line surface volume layer group} {
        set over over_$type
        if { [llength [GiD_Info conditions $over]] } {
            set exists 1
            break
        }
    }
    return $exists
}

proc ExistsMaterials { } {
    set exists 0
    if { [llength [GiD_Info materials]] } {
        set exists 1
    }
    return $exists
}

proc UpdateDataOptions { DataMenu IndexEntry ListName } {
    global DataMenuInfo gidDataLabels GidPriv
    global gidDataBooksHidden
    upvar $ListName UnderChars
    
    if { [GiD_Info Project ProblemType] == "UNKNOWN" && ![ExistsConditions] && ![ExistsMaterials]} {
        SetDefaultDataMenu $DataMenu $IndexEntry UnderChars
        return
    }
    
    set DataEntry [list [list [_ "Conditions#C#menu"] Conditions DWCondWindow conditions CheckCnd condition.png] \
            [list [_ "Materials#C#menu"] Materials DWMatWindow materials CheckMat ""] \
            [list [_ "Interval data#C#menu"] {Interval data} DWIntervalDataWindow intvdata CheckIntv ""] \
            [list [_ "Problem data#C#menu"] {Problem data} DWProblemDataWindow gendata CheckPrbD prdata.png]]
    if { [GroupsXmlDocExists] } {
        if { [info exists ::problemtype_current(CustomLibAutomatic)] && $::problemtype_current(CustomLibAutomatic) } { 
          #do it only if is allowed in the xml declaration by CustomLibAutomatic, to not change the menu for other problemtypes like CompassFEM
          lappend DataEntry [list [_ "Data tree#C#menu"] {DataTree} [list GidUtils::ToggleWindow CUSTOMLIB] "" CheckDataTree ""]
        }
    }
    
    #set IndexEntry 4
    
    array unset DataMenuInfo
    
    
    set DataMenuInfo(menu) $DataMenu
    
    # Precondicion: END
    
    foreach m $DataEntry {
        set Class [lindex $m 1]
        
        if ![info exists gidDataLabels($Class)] continue
        
        if { $gidDataLabels($Class) == "" } {
            set ClassLabel  [lindex $m 0]
        } else {
            set ClassLabel $gidDataLabels($Class)
        }
        set Cmd [lindex $m 2]
        set AskBook [lindex $m 3]
        set CheckFunc [lindex $m 4]
        set littleicon [lindex $m 5]
        set UnderIdx [FindUnderChar ClassLabel UnderChars]
        
        if { [$CheckFunc] != "" } {
            if { $AskBook == "" } {
                set Books [list]
            } else {
                set Books [GiD_Info $AskBook BOOKS]
            }            
            set nBooks [llength $Books]
            set DataMenuInfo($Class,COUNTER) 0
            set DataMenuInfo($Class,entry) $IndexEntry
            if { $nBooks == 0 } {
                set DataMenuInfo($Class) {Default}
                $DataMenu insert $IndexEntry command -label $ClassLabel -command $Cmd -underline $UnderIdx
                AddLittleIcon $littleicon $DataMenu $IndexEntry
            } else {
                set SubMenu $DataMenu.$AskBook
                if { [winfo exists $SubMenu] } {
                    destroy $SubMenu
                }
                $DataMenu insert $IndexEntry cascade -label $ClassLabel -menu $SubMenu -underline $UnderIdx
                AddLittleIcon $littleicon $DataMenu $IndexEntry
                menu $SubMenu
                set DataMenuInfo($Class,submenu) $SubMenu
                set i 0
                set BookOrd 1
                set UnderCharsLocal {}
                set hidden 0
                foreach e $Books {
                    if { $i == 0 } {
                        if { [lsearch $gidDataBooksHidden($AskBook) $e] == -1 } {
                            lappend DataMenuInfo($Class) $e
                            set book $e
                        } else {
                            set hidden 1
                        }
                        set i 1
                    } else {
                        set i 0
                        if !$hidden {
                            set Label [= [DWUnder2Space $book]]
                            set UnderIdx [FindUnderChar Label UnderCharsLocal]
                            $SubMenu add command -label $Label -command [list $Cmd $book $BookOrd] -underline $UnderIdx
                            #AddLittleIcon $littleicon $DataMenu $IndexEntry
                        } else {
                            set hidden 0
                        }
                        incr BookOrd
                    }
                }
            }
            incr IndexEntry
        }
    }
    
    GidUpdateUserDataOptions $UnderChars
    
    if { $::tcl_platform(platform) != "windows" } {
        SetDataMenuState
    }
}

proc GetDataBooks { name } {
    
    upvar $name rname
    
    set DataEntry [list [list [_ "Conditions#C#menu"] Conditions DWCondWindow conditions CheckCnd] \
            [list [_ "Materials#C#menu"] Materials DWMatWindow materials CheckMat] \
            [list [_ "Interval data#C#menu"] {Interval data} DWIntervalDataWindow intvdata CheckIntv] \
            [list [_ "Problem data#C#menu"] {Problem data} DWProblemDataWindow gendata CheckPrbD]]
    
    
    foreach m $DataEntry {
        set ClassLabel [lindex $m 0]
        set Class [lindex $m 1]
        set Cmd [lindex $m 2]
        set AskBook [lindex $m 3]
        set CheckFunc [lindex $m 4]
        
        if { [$CheckFunc] != "" } {
            set Books [GiD_Info $AskBook BOOKS]
            
            set nBooks [llength $Books]
            set rname($Class,COUNTER) 0
            if { $nBooks == 0 } {
                set rname($Class) {Default}
            } else {
                set i 0
                set BookOrd 1
                foreach e $Books {
                    if { $i == 0 } {
                        lappend rname($Class) $e
                        set i 1
                    } else {
                        set i 0
                    }
                }
            }
        }
    }
}

# revisa por que tienes que hacer esto
set ::gidSubstDataBooks 0

proc SubstDataBooks { } {
    global gidSubstDataBooks
    
    set gidSubstDataBooks 1
    
    if { [GidUtils::IsTkDisabled] } {
        return
    }    
    
    if { [info exists ::ModUnitWin::win] && [winfo exists $::ModUnitWin::win] } {
        destroy $::ModUnitWin::win
    }
    
    GetDataBooks rBooks
    
    set ListGDN [info globals GD_*]
    
    foreach GDN $ListGDN {
        upvar #0 $GDN GidData
        
        set wname $GidData(WINDOW)
        
        if { [winfo exists $wname] } {
            set class $GidData(CLASS)
            set CallWin $GidData(CALLWIN)
            
            if { $class == "Conditions" && [winfo exists $wname.cndwnd] } {
                set HasCEW 1
                destroy $wname.cndwind
            } else {
                set HasCEW 0
            }
            #destroy $wname
            if { [info exists rBooks($class,COUNTER)] } {
                set idx $rBooks($class,COUNTER)
                set book [lindex $rBooks($class) $idx]
                if { $book != "" } {
                    incr idx
                    set GDN [ReplaceDW $GDN $CallWin $book $idx]
                    if { $HasCEW } {
                        DWShowCondEntitiesWin $GDN thiscond
                    }
                    incr rBooks($class,COUNTER)
                } else {
                    destroy $wname
                }
            } else {
                destroy $wname
            }
        }
    }
    set gidSubstDataBooks 0
}

proc RefreshDataWindows { } {
    set ListGDN [info globals GD_*]
    foreach GDN $ListGDN {
        upvar #0 $GDN GidData        
        if { [winfo exists $GidData(WINDOW)] } {
            destroy $GidData(WINDOW)
            $GidData(CALLWIN) $GidData(BOOK) $GidData(BOOKORD)
        }
    }
}

# ------------------------------------------------------------------------------
# RefreshOnAddNewIntv --
#
#   Called from C when a new interval is added.
#
# ------------------------------------------------------------------------------

proc RefreshOnAddNewIntv {} {
    set ListGDN [info globals GD_INT*]
    foreach GDN $ListGDN {
        upvar #0 $GDN GidData
        
        if { [winfo exists $GidData(WINDOW)] } {
            DWOnAddNewInterval $GDN
        }
    }
}

# ------------------------------------------------------------------------------
# RefreshOnDeleteIntv --
#
#   Called from C when the current interval is deleted.
#
# ------------------------------------------------------------------------------

proc RefreshOnDeleteIntv { I } {
    set ListGDN [info globals GD_INT*]
    foreach GDN $ListGDN {
        upvar #0 $GDN GidData
        if { [winfo exists $GidData(WINDOW)] } {
            DWOnDeleteInterval $GDN $I
        }
    }
}

proc TriggerDependencies { GDN } {
    upvar #0 $GDN GidData
    set TYPE [$GidData(TYPEFUNC) $GDN]
    #array names return items in random order, 
    # and want to raise dependencies ordered by increasing question index, sort them as integer
    set prefix $TYPE,DEPENDENCIES,
    set len_prefix [string length $prefix]
    set IdxFields [list]
    foreach item [array names GidData $prefix*] {
        lappend IdxFields [string range $item $len_prefix end]
    }
    set IdxFields [lsort -integer $IdxFields]
    foreach IdxField $IdxFields {
        if { [info exists GidData($TYPE,STATE,$IdxField)] && ($GidData($TYPE,STATE,$IdxField) == "hidden" || $GidData($TYPE,STATE,$IdxField) == "disabled") } {
            #avoid to raise dependencies of hidden or disabled fields
        } else {
            set GidData($TYPE,VALUE,$IdxField) $GidData($TYPE,VALUE,$IdxField)
        }
    }
    set GidData($TYPE,SYNC,TKWIDGET) 0
    if { [string equal $GidData(CLASS) "Conditions"] } {
        set GidData(FIRSTSTATE) ""
    } else {
        set GidData(FIRSTSTATE) [BuildStringValues $GDN]
    }
    set GidData($TYPE,SYNC,TKWIDGET) 1
}

proc GiveGlobalXY { frame } {
    
    set x 0 ; set y 0
    set wtmp $frame
    while 1 {
        incr x [winfo x $wtmp]
        incr y [winfo y $wtmp]
        if { [winfo class $wtmp] == "Toplevel" } { break }
        set wtmp [winfo parent $wtmp]
    }
    return [list $x $y]
}

proc DWVarValueWinChange { GDN w N i what numwidth mxlenvalue } {
    global $GDN
    upvar #0 $GDN GidData
    
    set totalold $GidData($N,VALUE,$i,0)
    
    set fvw [$w.sw.sf getframe]
    
    #     grid rowconf $fvw [expr $totalold/$numwidth+2] -weight 0
    
    switch $what {
        down { incr GidData($N,VALUE,$i,0) $numwidth }
        up {
            if { $GidData($N,VALUE,$i,0) <= $numwidth } {
                if {[wm overrideredirect $w]} {
                    # lower $w
                }
                WarnWin [_ "There must be at least one value for field"]
                if {[wm overrideredirect $w]} {
                    raise $w
                }
                return
            }
            incr GidData($N,VALUE,$i,0) [expr $numwidth*-1]
        }
    }
    set total $GidData($N,VALUE,$i,0)
    
    for { set j [expr $total+1] } { $j <= $totalold } { incr j } {
        destroy $fvw.l$j
    }
    
    if { [wm overrideredirect $w] } {
        set v 0
    } else {
        set v 1
    }
    incr v [expr $total/$numwidth]
    set u 0
    for { set j [expr $totalold+1] } { $j <= $total } { incr j } {
        ttk::entry $fvw.l$j -textvar $GDN\($N,VALUE,$i,$j\) -width 3
        set GidData($N,VALUE,$i,$j) $GidData($N,VALUE,$i,[expr $j-$numwidth])
        grid $fvw.l$j  -row $v -column $u -sticky ew
        incr u
        if { $u == [expr $numwidth+1] && $j < $total} {
            set u 0; incr v
        }
    }
    #     grid rowconf $fvw $v -weight 1
    
    regexp {^([0-9]*)x([0-9]*)([-+]*[0-9]*)([-+]*[0-9]*)$} \
        [winfo geom $w] {} width height x y
    
    set vmax $v
    if { $vmax>12 } { set vmax 12 }
    set height [expr [winfo reqheight $fvw.l1]*$vmax+\
            [winfo reqheight $w.f.up]+10]
    if { [winfo exists $fvw.title1] } {
        incr height [winfo reqheight $fvw.title1]
    }
    wm geometry $w ${width}x${height}${x}${y}
    set heightint [expr [winfo reqheight $fvw.l1]*$v]
    #$fvw configure -width [expr $width-15] -height $heightint
    #$w.canvas configure -scrollregion "0 0 [expr $width-15] $heightint"
    
    #DWReconfigureTextGrid $w.canvas $w.xscroll $w.yscroll
    #update
}


set GRID_DATA_USE_TREECTRL 1
proc DWVarValueWin { GDN title w N i fieldnames frame mxlenvalue { transient yes } } {
    if { $::GRID_DATA_USE_TREECTRL == 0 } {    
        DWVarValueWinOld $GDN $title $w $N $i $fieldnames $frame $mxlenvalue $transient
    } else {
        GridData::DWVarValueWin $GDN $title $w $N $i $fieldnames $frame $mxlenvalue $transient
    }
}

#*****START OLD GRID DATA USING TREECTRL
proc DWVarValueWinClick { W x y ofocus } {
    set w [winfo width $W]
    set h [winfo height $W]
    if { $x < 0 || $x > $w || $y < 0 || $y > $h } {
        grab release $W
        focus -force $ofocus
        destroy $W
    }
}

proc DWVarValueWinOld { GDN title w N i fieldnames frame mxlenvalue { transient yes } } {
    global ExtendInfo
    global $GDN
    upvar #0 $GDN GidData
    
    set ofocus [focus]
    
    set parent $w
    set w $frame.vw
    
    if {[winfo exists $frame.l0]} {
        set minwidth [winfo width $frame.l0]
    } else {
        set minwidth 0
    }
    if { [winfo exists $w] } {
        destroy $w
    }
    toplevel $w -borderwidth 2 -relief ridge
    if { $::tcl_platform(platform) == "windows" } {
        wm attributes $w -toolwindow 1
    }
    wm state $w withdrawn
    wm attributes $w -topmost 1
    ScrolledWindow $w.sw -auto both
    ScrollableFrame $w.sw.sf -constrainedheight 0 -constrainedwidth 0
    $w.sw setwidget $w.sw.sf
    set fvw [$w.sw.sf getframe]
    grid $w.sw -row 0 -column 0 -sticky "snew"
    grid rowconf    $w 0 -weight 1
    grid columnconf $w 0 -weight 1
    
    set v 0
    if { $transient == "yes" } {
        wm transient $w $parent
        wm overrideredirect $w 1
        grab $w
        $w configure -cursor top_left_arrow
        bind $w <1> [list DWVarValueWinClick %W %x %y $ofocus]
        #         bind $w.f.f$i.vw <1> {
            #             set w [winfo width %W]
            #             set h [winfo height %W]
            #             if { %x < 0 || %x > $w || %y < 0 || %y > $h } {
                #                 grab release %W
                #                 destroy %W
                #             }
            #         }
        bind $w <Escape> "grab release $w; focus -force $ofocus; destroy  $w; break"
    } else {
        wm title $w $title
        set u 0
        foreach j $fieldnames {
            ttk::label $fvw.title$u -text $j -relief ridge -borderwidth 1 -width 10
            grid $fvw.title$u -row $v -column $u -sticky ew
            grid columnconf $fvw $u -weight 1 -minsize $minwidth
            incr u
        }
        incr v
    }
    
    set numwidth [llength $fieldnames]
    # foreach "x y" [GiveGlobalXY $frame] {}
    set x [winfo rootx $frame]
    set y [expr {[winfo rooty $frame]+[winfo height $frame]}]
    # incr y [winfo height $frame]
    set width [winfo width $frame]
    set total $GidData($N,VALUE,$i,0)
    if { $total < $numwidth } {
        set GidData($N,VALUE,$i,0) $numwidth
        set total $numwidth
    }
    set u 0
    for { set j 1 } { $j <= $total } { incr j } {
        if { $v == 0 } { grid columnconf $fvw $u -weight 1 }
        ttk::entry $fvw.l$j -textvar $GDN\($N,VALUE,$i,$j\) -width 10
        grid $fvw.l$j  -row $v -column $u -sticky ew
        grid columnconfigure $fvw $u -minsize $minwidth
        incr u
        if { $u == $numwidth } {
            set u 0 ; incr v
        }
    }
    
    grid rowconf $fvw $v -weight 1
    
    ttk::frame $w.f
    ttk::button $w.f.down -image DownImage -command \
        [list DWVarValueWinChange $GDN $w $N $i down $numwidth $mxlenvalue]
    GidHelp $w.f.down [_ "Pressing this button, one new line will be added to this window"]
    focus $w.f.down
    ttk::button $w.f.up -image UpImage -command \
        [list DWVarValueWinChange $GDN $w $N $i up $numwidth $mxlenvalue]
    GidHelp $w.f.up [_ "Pressing this button, one line is deleted from this window"]
    ttk::button $w.f.right -image RightImage -command \
        [list DWVarValueWin $GDN $title $w $N $i $fieldnames $frame $mxlenvalue no]
    GidHelp $w.f.right [_ "Converts this menu to a normal window"]
    ttk::button $w.f.close -image CloseImage -command \
        [list destroy $w]
    GidHelp $w.f.close [_ "Closes this menu or window"]
    
    pack $w.f.down $w.f.up $w.f.right \
        $w.f.close -side left
    
    grid $w.f -column 0 -row 2 -pady 2 -columnspan 2
    update
    set vmax $v
    if { $vmax>11 } { set vmax 11 }
    set height [expr ([winfo reqheight $fvw.l1]+1)*$vmax+\
            [winfo reqheight $w.f.up]+20]
    if { [winfo exists $fvw.title1] } {
        incr height [winfo reqheight $fvw.title1]
    }
    wm geometry $w ${width}x${height}+${x}+${y}
    
    set heightint [expr [winfo reqheight $fvw.l1]*[expr $v-1]]
    #$fvw configure -width [expr $width-15] -height $heightint
    update idletasks
    wm state $w normal
    raise $w
    update
}
#*****END OLD GRID DATA USING TREECTRL  

#*****START NEW GRID DATA USING TREECTRL   
namespace eval GridData {
    
}

proc GridData::DestroyWindow { W w } {
    if { $W == $w } {
        #unset -nocomplain ::GidPriv(grid,treectrl)
    }
}

proc GridData::_destroy_if_outside { GDN N i T w x y } {
    if { ![winfo exists $w] } { return }
    if { $x < [winfo rootx $w] || $x > [winfo rootx $w]+[winfo width $w] || \
        $y < [winfo rooty $w] || $y > [winfo rooty $w]+[winfo height $w] } {
        #DWVarValueWinNew [winfo parent $w] [winfo parent $w]
        GridData::_destroy $GDN $N $i $T $w
    }
}

proc GridData::_destroy { GDN N i T w } {    
    GridData::FillGiDFromTree $GDN $N $i $T
    destroy $w    
}

proc GridData::CreateTreectrl { GDN N i fieldnames w } {
    package require fulltktree
    
    # width name justify type is_editable
    set columns ""
    foreach name $fieldnames {
        lappend columns [list 9 $name right text 1 1]
    }
    
    set T [fulltktree $w.t -columns $columns -showlines 0 -showbuttons 0 -showheader 1 -expand 1]
    $T configure -editbeginhandler GridData::EditBegin
    #$T configure -editaccepthandler [list GridData::EditAccept]
    $T configure -editaccepthandler [list GridData::EditAcceptUpdateGDN $GDN $N $i]
    $T configure -deletehandler GridData::RemoveRows    
    $T configure -contextualhandler_menu GridData::ContextualMenu
    $T column configure all -button 0    
    $T column configure all -minwidth {} -squeeze 0 -weight 1 ;#trick to modify fulltktree settings to have more equispaced columns
    
    #$T style layout window e_window -sticky w -iexpand "" -expand e -detach 1    
    
    grid $T -sticky nsew

    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1
    
    bind $T <<Paste>> [list ::GridData::Paste $T CLIPBOARD]
    bind $T <<PasteSelection>> [list ::GridData::Paste $T PRIMARY]
    bind $T <<Cut>> [list GridData::Cut $T]
    bind $T <<Copy>> [list GridData::Copy $T]
    bind [$T givetreectrl] <${::acceleratorKey}-c> "" ;#to remove the default binding of fulltktree that capture also header
    #bind $T <Delete> [list GridData::RemoveSelectedRows $T]    
    $T notify bind $T <ItemVisibility> [list GridData::ResizeHight $T]    
    return $T
}
    
proc GridData::EditBegin { w item col } {
    return entry
}

proc GridData::EditAcceptUpdateGDN { GDN N i w item col text } {    
    set T [winfo parent $w]
    if { [$T item text $item $col] eq $text } { 
        return
    }
    global $GDN
    upvar #0 $GDN GidData
    set ncols [$T column count]
    set row $item
    set GidData($N,VALUE,$i,[expr {($row-1)*$ncols+($col+1)}]) $text
    #$T item element configure $item $col e_text_sel -text $text
    $T item text $item $col $text
}

proc GridData::EditAccept { w item col text } {    
    set T [winfo parent $w]
    if { [$T item text $item $col] eq $text } { 
        return
    }
    $T item text $item $col $text
}

proc GridData::Copy { T } {   
    set dataList ""
    if { 0 } {
        #append also column header
        set cols [$T cget -columns]
        set lineList ""
        foreach i $cols {
            lappend lineList [lindex $i 1]
        }
        lappend dataList [join $lineList \t]
    }
    set ncols [llength [$T cget -columns]]
    set items [$T selection get]
    if { ![llength $items] } {
        set items [$T item children 0]
    }
    foreach item [lsort -integer $items] {
        set lineList ""
        for { set col 0 } { $col < $ncols } { incr col } {
            lappend lineList [$T item text $item $col]
        }
        lappend dataList [join $lineList \t]
    }
    clipboard clear
    clipboard append [join $dataList \n]
}

proc GridData::Cut { T } {
    GridData::RemoveSelectedRows $T
}

proc GridData::Paste { T sel } {
    #selection get -selection CLIPBOARD -displayof $T
    #is equivalent to
    #clipboard get -displayof $T
    set data [selection get -selection $sel -displayof $T]
    if { $data != "" } {
        GridData::SetData $T $data
    }
}

proc GridData::SetData { T data } {
    if { ![winfo exists $T]} {
        return
    }
    if { ![string is list $data] } {
        return
    }
    set ntotal [llength $data]
    set ncols [$T column count]
    if {  [expr {$ntotal%$ncols}] != 0 } {
        #is not a multiple of ncols
        return
    }
    $T item delete 0 end
    set nrows [expr $ntotal/$ncols]
    set start 0
    for { set row 1 } { $row <= $nrows } { incr row } {        
        set end [expr {$start+$ncols-1}]
        set rowdata [lrange $data $start $end]
        set item [$T insert end $rowdata]
        #$T item style set $item 0 window
        #$T item element configure $item 0 e_window -window .t.m$idx
        incr start $ncols
    }
    
    #GridData::ResizeHight $T
}

proc GridData::GetDataAllItems { T } {
    if { ![winfo exists $T] } {
        return ""
    }
    set data ""
    foreach item [$T item range first end] {
        if { $item != 0 } {
            set datarow [$T item text $item]
            if { $datarow != "" } {
                lappend data {*}$datarow
            }
        }
    }  
    return $data
}

proc GridData::GetDataOfItems { T items } {
    if { ![winfo exists $T]} {
        return ""
    }
    set data ""
    foreach item $items {
        set datarow [$T item text $item]
        if { $datarow != "" } {
            lappend data {*}$datarow       
        }
    }  
    return $data    
}

proc GridData::FillGiDFromTree { GDN N i T } {     
    set data [GridData::GetDataAllItems $T]
    global $GDN
    upvar #0 $GDN GidData
    set n [llength $data]
    set j 1
    foreach v $data {
        set GidData($N,VALUE,$i,$j) $v
        incr j
    }
    set GidData($N,VALUE,$i,0) $n
    #set also VALUE, to be ready to be asked by DWLocalGetValue
    set GidData($N,VALUE,$i) "#N# $n $data"
}

proc GridData::FillTreeFromGiD { GDN N i T } {
    global $GDN
    upvar #0 $GDN GidData
    set ntotal $GidData($N,VALUE,$i,0)
    set data ""
    for { set j 1 } { $j <= $ntotal } { incr j } {
        lappend data $GidData($N,VALUE,$i,$j)        
    }    
    GridData::SetData $T $data    
}

proc GridData::AppendRows { T nrowsmore } {
    set data [GridData::GetDataAllItems $T]
    if { [$T item count]>1 } {
        set lastrowdata [$T item text last]
    } else {
        set ncols [$T column count]
        set lastrowdata [lrepeat $ncols 0.0]
    }
    for {set row 0} {$row < $nrowsmore} {incr row} {
        lappend data {*}$lastrowdata
    }
    GridData::SetData $T $data
}

proc GridData::RemoveLastRows { T nrowsless } {
    set data [GridData::GetDataAllItems $T]
    set ncols [$T column count]
    set last [expr {[llength $data]-$nrowsless*$ncols-1}]
    if { $last<0 } {
        set data ""
    } else {
        set data [lrange $data 0 $last]
    }
    GridData::SetData $T $data    
}

proc GridData::RemoveRows { T items } {
    if { ![winfo exists $T] } {
        return
    }
    set complementaryitems ""
    foreach item [$T item range first end] {
        if { $item != 0 } {
            if { [lsearch $items $item] == -1 } {
                lappend complementaryitems $item
            }
        }
    }    
    set data [GetDataOfItems $T $complementaryitems]
    GridData::SetData $T $data    
}

proc GridData::RemoveSelectedRows { T } {
    GridData::RemoveRows $T [$T selection get]    
}

proc GridData::DWVarValueWin { GDN title w N i fieldnames frame mxlenvalue { transient yes } } {
    #     global GidPriv
    set open_over $frame
    set parent $frame ;#or .gid?
    set w $parent.m
    if { [winfo exists $w] } {
        destroy $w
        #return
    }
    toplevel $w -borderwidth 1 -relief raised
    if { $::tcl_platform(platform) == "windows" } {
        wm attributes $w -toolwindow 1
    }
    wm withdraw $w
    wm attributes $w -topmost 1
    wm overrideredirect $w 1
    if { $::tcl_platform(platform) != "windows" } {
        update
    }

    set orient "horizontal"
    if { $orient == "vertical" } {
        set x [expr {[winfo rootx $open_over]+[winfo width $open_over]}]
        set y [expr {[winfo rooty $open_over]}]        
    } else {
        set x [expr {[winfo rootx $open_over]}]
        #set y [expr {[winfo rooty $open_over]+[winfo height $open_over]}] ;#in case of hide the row of titles of the table
        set y [expr {[winfo rooty $open_over]}]
    }
    
    set width [winfo width $open_over]
    set height 100
    wm geometry $w ${width}x${height}+$x+$y
    
    ttk::frame $w.f1
    
    set T [GridData::CreateTreectrl $GDN $N $i $fieldnames $w.f1]        
    
    #focus $tbl
    #bind $tbl <Double-Button-1> "+after idle destroy $w"
            
    ttk::frame $w.f2
    set nrowsmore 1
    set nrowsless 1
    ttk::button $w.f2.b1 -image DownImage -text [_ "Add row"] \
        -command [list GridData::AppendRows $T $nrowsmore]
    GidHelp $w.f2.b1 [_ "Pressing this button, one new line will be added to this window"]    
    ttk::button $w.f2.b2 -image UpImage -text [_ "Remove row"] \
        -command [list GridData::RemoveLastRows $T $nrowsless]
    GidHelp $w.f2.b2 [_ "Pressing this button, one line is deleted from this window"]    
    ttk::button $w.f2.b3 -image [gid_themes::GetImage graphs.png small_icons] -text [_ "Graph"] \
        -command [list GridData::PressDrawGraph $GDN $N $i $w $T $title]
    GidHelp $w.f2.b3 [_ "Pressing this button, a XY graph will be drawn"]
    set ncols [$T column count]
    if { $ncols >= 2 } {
        grid $w.f2.b1 $w.f2.b2 $w.f2.b3 ;#-sticky ew
    } else {
        grid $w.f2.b1 $w.f2.b2 ;#-sticky ew
    }
    
    grid $w.f1 - -sticky nsew -pady 0
    grid $w.f2 - -sticky nsew -pady 0
    grid columnconfigure $w.f2 "0 1 2" -weight 1
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1                 
    
    GridData::FillTreeFromGiD $GDN $N $i $T
    

    bind $w <Destroy> [list +GridData::DestroyWindow %W $w]    
    bind $w <Escape> [list GridData::_destroy $GDN $N $i $T $w]
    bind $w <1> [list GridData::_destroy_if_outside $GDN $N $i $T $w %X %Y]    
    
    update idletasks
    wm deiconify $w
    update
    if { ![winfo exists $w] } { 
        return }
    
    if { $::tcl_platform(platform) == "windows" } {  
        grab $w
    } else {
        grab -global $w
    }
}

proc GridData::PressDrawGraph { GDN N i w T title } {
    GridData::FillGiDFromTree $GDN $N $i $T
    grab release $w
    wm withdraw $w
    GridData::PlotCurveWin $T $title
}

proc GridData::ContextualMenu { w menu active sel  } {
#     $menu add command -label [_ "Copy"] -command [list GridData::Copy $w]
    $menu add command -label [_ "Copy"] -command [list event generate $w <<Copy>>]
    $menu add command -label [_ "Paste"] -command [list event generate $w <<Paste>>]
    #tk_popup $menu $X $Y
}

proc GridData::ResizeHight { T } {
    set w [winfo toplevel $T]
    set nrows [expr {[$T item count]}]
    if { $nrows > 25 } {
        set nrows 25
    } elseif { $nrows < 3 } {
        set nrows 3
    }
    #set itemheight 18    
    set itemheight [$T cget -minitemheight]
    if { [$T cget -itemheight] != 0 } {
        set itemheight [$T cget -itemheight]
    }
    update idletasks ;#else extraheight will be 1

    if { [winfo exists $w.f2] } {
        set extraheight [winfo reqheight $w.f2]
    } else {
        set extraheight 0
    }
    set height [expr {($nrows)*($itemheight+1)+$extraheight+2}]
    set width [winfo width $w]
    wm geometry $w ${width}x${height}
}

proc GridData::PlotCurveWin { T title {series_options ""} } {
    package require gid_graph
    
    if { ![winfo exists $T] } {
        return
    }
    set ncols [$T column count]
    if { $ncols < 2 } {
        WarnWin [_ "Insufficient number of data"]
        return
    }
    set data [GridData::GetDataAllItems $T]
    set ntotal [llength $data]
    set nrows [expr $ntotal/$ncols]
    if { $nrows < 2 } {
        WarnWin [_ "Insufficient number of data"]
        return
    }
    
    foreach coord $data {
        if { ![string is double $coord] } {
            WarnWin [concat [_ "Graph can't be drawn"]. [_ "value '%s' is not a number" $coord]]
            return
        }
    }
    
    set x_text [$T column cget 0 -text]
    set y_text ""
    for {set col 1} {$col<$ncols} {incr col} {
        lappend y_text [$T column cget $col -text]
    }
    set points ""
    set ndata [llength $data]
    set pos 0
    while { $pos<$ndata } {
        set pt [lrange $data $pos [expr {$pos+$ncols-1}]]
        lappend points $pt
        incr pos $ncols
    }
    
    GidGraph::Window2 -points $points -title $title -x_text $x_text -y_text $y_text -type "xy" -series_options $series_options
}

#*****END NEW GRID DATA USING TREECTRL


proc is_true_word { text } {
    set opt [string tolower [string trim $text]]
    if { $opt == "true" || $opt == "yes" || $opt == "si" || $opt == "1" } {
        return 1
    } else { 
        return 0 
    }
}

proc is_false_word { text } {
    set opt [string tolower [string trim $text]]
    if { $opt == "false" || $opt == "no" || $opt == "0" } {
        return 1
    } else { 
        return 0 
    }
}

proc are_true_false_words { a b } {
    if { [is_true_word $a] && [is_false_word $b] } {
        return 1
    } elseif { [is_true_word $b] && [is_false_word $a] } {
        return -1
    } else {
        return 0
    }
}

proc EndDWDefineLA { GDN TYPE Idx localaxesold } {
    upvar #0 $GDN GidData
    
    set localaxesnew [GiD_Info localaxes]
    
    if { [llength $localaxesnew] != [llength $localaxesold]+1 } {
        return
    }
    if { [lrange $localaxesnew 0 end-1] != $localaxesold } {
        return
    }
    set GidData($TYPE,OPTIONS1,$Idx) $localaxesnew
    set GidData($TYPE,VALUE,$Idx) [lindex $localaxesnew end]
    set GidData($TYPE,CBTEXT,$Idx) [= [DWUnder2Space $GidData($TYPE,VALUE,$Idx)]]
}

proc DWDefineLA { GDN TYPE Idx } {
    
    GiD_Process Mescape Data LocalAxes DefineLocAxes
    delayedop changefunc [list EndDWDefineLA $GDN $TYPE $Idx [GiD_Info localaxes]]
}

proc DWActivateCBmenu { GDN TYPE Idx menu } {
    upvar #0 $GDN GidData
    
    $menu delete 0 end
    
    set cv $GidData($TYPE,VALUE,$Idx)
    set io 0
    if {$GidData($TYPE,TYPE,$Idx) eq "MAT"} {
        set GidData($TYPE,OPTIONS,$Idx) [lindex [DWGetMaterials \
                    $GidData($TYPE,MATFIELD,BOOKS,$Idx)] 1]
    }
    foreach l $GidData($TYPE,OPTIONS,$Idx) {
        
        if ![string compare $l $cv] {
            set GidData($TYPE,POSTING,$Idx) 1
            set GidData($TYPE,CURSEL,$Idx) $io
            set GidData($TYPE,POSTING,$Idx) 0
        }
        set label [= [DWUnder2Space $l]]
        $menu add radiobutton -value $io -label $label -variable $GDN\($TYPE,CURSEL,$Idx\)
        incr io
    }
}

proc DWActivateLAmenu { GDN TYPE Idx menu } {
    upvar #0 $GDN GidData
    
    set GidData($TYPE,POSTING,$Idx) 1
    $menu delete 0 end
    
    set cv $GidData($TYPE,VALUE,$Idx)
    set io 0
    foreach l $GidData($TYPE,OPTIONS,$Idx) {
        if ![string compare $l $cv] {
            #set GidData($TYPE,POSTING,$Idx) 1
            set GidData($TYPE,CURSEL,$Idx) $io
            #set GidData($TYPE,POSTING,$Idx) 0
        }
        set label [= [DWUnder2Space $l]]
        $menu add radiobutton -value $io -label $label -variable $GDN\($TYPE,CURSEL,$Idx\)
        incr io
    }
    set widget [lindex $GidData($TYPE,WIDGET,$Idx) 0]
    set curlen [$widget cget -width]
    set mxlen $curlen
    set GidData($TYPE,OPTIONS1,$Idx) [GiD_Info localaxes]
    foreach l $GidData($TYPE,OPTIONS1,$Idx) {
        set lenopt [string length $l]
        if { $lenopt > $mxlen } {
            set mxlen $lenopt
        }
        set label [= [DWUnder2Space $l]]
        $menu add radiobutton -value $io -label $label -variable $GDN\($TYPE,CURSEL,$Idx\)
        incr io
    }
    $menu add separator
    $menu add command -label [_ "Define"]... -command [list DWDefineLA $GDN $TYPE $Idx]
    if { $mxlen > $curlen } {
        $widget configure -width $mxlen
    }
    set GidData($TYPE,POSTING,$Idx) 0
}

proc ParserNumberUnit { Value Number Unit } {
    upvar $Number n
    upvar $Unit u
    
    set numreg {^[  ]*([+-]?[0-9]*\.?[0-9]*([eE][+-]?[0-9]+)?)[  ]*}
    set n "" ; set u ""
    regexp "${numreg}(.*)" $Value {} n {} u
    set u [string trim $u]
    return 1
}

proc DWFillText { GDN f N from to } {
    global $GDN
    global ExtendInfo
    
    upvar #0 $GDN GidData
    
    set GidData($N,CONTAINER) $f
    
    set currentrow 0
    set comments [array names GidData $N,COMMENT,*]
    if { $comments != "" } {
        ttk::label $f.comment -justify left
        grid $f.comment -row $currentrow -column 0 -columnspan 2 -sticky wn
        incr currentrow
        set labelcanvas ""
        foreach i $comments {
            if { $labelcanvas != "" } { append labelcanvas \n }
            append labelcanvas [subst -nocommands -novariables $GidData($i)]
        }
        $f.comment configure -text $labelcanvas
    }
    
    set row_weight_requested    0
    set column_weight_requested 0
    
    set thereishelp 0
    set result ""
    if { $from == 1 && [info exists GidData($N,TKWIDGET,0)] } {
        set save_row $currentrow
        if { [catch "$GidData($N,TKWIDGET,0) INIT $f currentrow $GDN [list $N] 0" result] } {
            W [concat [_ "Error in tcl script sending event INIT to"] "$GidData($N,TKWIDGET,0): $result"]
            set currentrow $save_row
        } else {
            set len_res [llength $result]
            if { $len_res > 1 } {
                if { $len_res != 2 } {
                    W [_ "Error '%s' in result from INIT event: must be a 2-element list" $result]
                    set currentrow $save_row
                } else {
                    set status [string toupper [lindex $result 0]]
                    if { [string equal $status ERROR] || [string equal $status WARNING] } {
                        set msg [lindex $result 1]
                        if { ![string length $msg] } {
                            W $msg
                        }
                        # ignoro el 3 elemento correspondiente a la accion
                    }
                }
            } else {
                if { $len_res } {
                    grid $result -row $currentrow -column 0 -columnspan 2 -sticky w
                } else {
                }
                for { set i $save_row } { $i <= $currentrow } { incr i } {
                    array set rowinfo [grid rowconfigure $f $i]
                    if { $rowinfo(-weight) } {
                        set row_weight_requested 1
                        break
                    }
                }
                incr currentrow
            }
        }
    }
    
    set colwidth 0
    
    for { set i $from } { $i <= $to } { incr i } {
        set field $GidData($N,LABELQ,$i)
        set value $GidData($N,VALUE,$i)
        set type  $GidData($N,TYPE,$i)
        set widget_options $GidData($N,WIDGET_OPTIONS,$i)

        set widget_width 25

        # verfico por argumentos
        
        set haveARGS 1
        if  { [info exists GidData($N,HEADINGS,$i)] } {
            upvar #0 $GDN\($N,HEADINGS,$i\) argRef
        } elseif { [info exists GidData($N,OPTIONS,$i)] } {
            upvar #0 $GDN\($N,OPTIONS,$i\) argRef
        } else {
            set haveARGS 0
        }
        if { $haveARGS } {
            set mxlenvalue 0
            set args {}
            foreach a $argRef {
                set a [DWUnder2Space $a]
                set l [string length $a]
                if { $l > $mxlenvalue } {
                    set mxlenvalue $l
                }
                lappend args $a
            }
            incr mxlenvalue
        }
        
        set dolabel 1
        
        switch $type {
            "ENTRY" {
                set widget [ttk::entry $f.e$i -width $widget_width -textvar $GDN\($N,VALUE,$i\)]               
                if { $widget_options != ""} {
                    $widget configure {*}$widget_options
                }
            }
            "FUNC" {
                #WARNING func
                set cond_name_splitted [split $N ,]
                if { [lindex $cond_name_splitted 0] == "CND" } {
                    set cond_name [lindex $cond_name_splitted 1]
                    set cond_info [lrange [GiD_Info Conditions $cond_name] 2 end]
                    set full_question [lindex $cond_info [expr ($i-1)*2]]
                    regexp -nocase {([^#]+)(?:#(FUNC)#(.*))?$} $full_question {} field type args
                    set field [= [DWUnder2Space $field]]
                } else {
                    set args ""
                }
                if { $args == "(NumPointToNumNode)" } {
                    #create an entry to allow the user to set manually the point/node number
                    set widget [ttk::entry $f.e$i -width $widget_width -textvar $GDN\($N,VALUE,$i\)]
                    if { $widget_options != ""} {
                        $widget configure {*}$widget_options
                    }                    
                } else {                   
                    set widget [ttk::label $f.e$i -width $widget_width -textvar $GDN\($N,VALUE,$i\)]
                    if { $widget_options != ""} {
                        $widget configure {*}$widget_options
                    }
                }
            }
            "UNITS" {
                if { !$GidData($N,UNITVALID,$i) } {
                    set widget [ttk::entry $f.e$i -width $widget_width -textvar $GDN\($N,VALUENUM,$i\)]
                    if { $widget_options != ""} {
                        $widget configure {*}$widget_options
                    }
                } else {
                    if { $GidData($N,UNITVALID,$i) == 1 } {
                        set widget [ttk::frame $f.f$i]                        
                        ttk::entry $f.f$i.e -width $widget_width -textvar $GDN\($N,VALUENUM,$i\)
                        if { $widget_options != ""} {
                            $f.f$i.e configure {*}$widget_options
                        }
                        canvas $f.f$i.c
                        set cgeo [GidUnit::DrawUnit $GidData($N,NUMERATOR,$i) $GidData($N,DENOMINATOR,$i) black $f.f$i.c]
                        $f.f$i.c configure \
                            -width [lindex $cgeo 0] -height [lindex $cgeo 1]
                        pack $f.f$i.c -side right
                        pack $f.f$i.e -side right -fill x -expand 1
                    } else {
                        set gu [GidUnit::CreateInstance]
                        set widget $f.f$i
                        $gu\::Create $widget $GidData($N,MAGNITUD,$i) $GDN\(MAG,$GidData($N,MAGNITUD,$i)\) $GDN\($N,IDXUNIT,$i\) $GDN\($N,VALUENUM,$i\)
                        set GidData($N,INSTANCE,$i) $gu
                    }
                }
            }
            "MAT" -
            "CB" {
                switch [lindex $GidData($N,CBTYPE,$i) 0] {
                    "LABEL" {
                        set widget [ttk::label $f.l$i -textvar $GDN\($N,VALUE,$i\)]
                    }
                    "CHECKBOX" {
                        set dolabel 0
                        if { ![info exists ${GDN}($N,VALUE,$i)] } { 
                            set ${GDN}($N,VALUE,$i) [lindex $args [lindex $GidData($N,CBTYPE,$i) 2]] 
                        } 
                        set widget [ttk::checkbutton $f.cb$i \
                                -text [subst -nocommands -novariables $field] \
                                -onvalue [lindex $args [lindex $GidData($N,CBTYPE,$i) 1]] \
                                -offvalue [lindex $args [lindex $GidData($N,CBTYPE,$i) 2]] \
                                -variable $GDN\($N,VALUE,$i\)]
                    }
                    "COMBO" {
                        # OJO: mxlenvalue depende de la traduccion y no se esta haciendo!!!!
                        ttk::menubutton $f.m$i -width $mxlenvalue \
                            -textvariable $GDN\($N,CBTEXT,$i\) -menu $f.m$i.m \
                            -direction flush
                        menu $f.m$i.m -tearoff 0 -borderwidth 1 -activeborderwidth 1 \
                            -postcommand [list DWActivateCBmenu $GDN $N $i $f.m$i.m]
                        set widget $f.m$i
                    }
                }
            }
            "LA" {
                ttk::menubutton $f.m$i -width $mxlenvalue \
                    -textvariable $GDN\($N,CBTEXT,$i\) -menu $f.m$i.m \
                    -direction flush
                menu $f.m$i.m  -tearoff 0 -postcommand \
                    [list DWActivateLAmenu $GDN $N $i $f.m$i.m] -borderwidth 1 \
                    -activeborderwidth 1
                set widget $f.m$i
            }
            "GRID" {
                if { $mxlenvalue < 8 } { set mxlenvalue 8 }
                set widget [ttk::frame $f.f$i]
                set fieldnames_translated [list]
                foreach fieldname $args {
                    lappend fieldnames_translated [= $fieldname]
                }
                set len [llength $args]
                for { set j 0 } { $j < $len } { incr j } {
                    ttk::button $f.f$i.l$j -text [= [lindex $args $j]] -command \
                        [list DWVarValueWin $GDN $field $widget $N $i $fieldnames_translated \
                            $f.f$i $mxlenvalue] -width $mxlenvalue \
                        -takefocus 0
                    pack $f.f$i.l$j -side left
                }
                ttk::button $f.f$i.b -image DownImage -command \
                    [list DWVarValueWin $GDN $field $f $N $i $fieldnames_translated \
                        $f.f$i $mxlenvalue]
                pack $f.f$i.b -side left
            }
        }
        
        set WidgetLabel {}
        if { $dolabel } {
            set WidgetLabel $f.question$i
            
            ttk::label $WidgetLabel -text [subst -nocommands -novariables $field] \
                -wraplength 50m -justify left
            
            set reqwidth [winfo reqwidth $WidgetLabel]
            if { $reqwidth > $colwidth } {
                set colwidth $reqwidth
            }
            
            grid $WidgetLabel -row $currentrow -column 0 -sticky e
            grid $widget -row $currentrow -column 1 -sticky w
        } else {
            grid $widget -row $currentrow -column 0 -columnspan 2 \
                -sticky w -padx 4
        }
        
        if { [info exists GidData($N,STATE,$i)] } {
            set wst $GidData($N,STATE,$i)
            if { $wst == "hidden" } {
                grid remove $widget
                if { $dolabel } {
                    grid remove $WidgetLabel
                }
            } else {
                if { $type == "UNITS" && $GidData($N,UNITVALID,$i) } {
                    if { $GidData($N,UNITVALID,$i) == 2 } {
                        if { $wst == "normal" } {
                            $GidData($N,INSTANCE,$i)::\Enabled
                        } else {
                            $GidData($N,INSTANCE,$i)::\Disabled
                        }
                    } else  {
                        # Puede ser ser solo un edit.
                        
                        catch {
                            $widget configure -state $wst
                        }
                        
                        # y esto en caso de frame
                        
                        foreach child [winfo children $widget] {
                            catch {
                                $child configure -state $wst
                            }
                        }
                    }
                } else {
                    catch {
                        $widget configure -state $wst
                    }
                }
            }
        }

        if { $dolabel } {
            set widgets [list $f.question$i $widget]
        } else {
            set widgets [list $widget]
        }        
        if { [info exists $GDN\($N,HELP,$i\)] } {
            setTooltip $widgets [= [DWUnder2Space [subst -nocommands -novariables $GidData($N,HELP,$i)]]]
            set thereishelp 1
        } elseif { [info exists $GDN\($N,HELP,0\)] } {
            setTooltip $widgets [= [DWUnder2Space [subst -nocommands -novariables $GidData($N,HELP,0)]]]
            set thereishelp 1
        }
        
        if { [info exists $GDN\($N,TKWIDGET,$i\)] } {
            set save_row $currentrow
            incr currentrow
            if { [catch "$GidData($N,TKWIDGET,$i) INIT $f currentrow $GDN [list $N] [list $GidData($N,QUESTION,$i)]" result] } {
                W [concat [_ "Error in tcl script sending event INIT to"] "$GidData($N,TKWIDGET,$i): $result"]
                set currentrow $save_row
            } else {
                set len_res [llength $result]
                if { $len_res > 1 } {
                    if { $len_res != 2 } {
                        W [_ "Error '%s' in result from INIT event: must be a 2-element list" $result]
                        set currentrow $save_row
                    } else {
                        set status [lindex $result 0]
                        if { [string equal $status ERROR] || [string equal $status WARNING] } {
                            set msg [lindex $result 1]
                            if { ![string length $msg] } {
                                W $msg
                            }
                            # ignoro el 3 elemento correspondiente a la accion
                        }
                    }
                } else {
                    if { $len_res } {
                        grid $result -row $currentrow -column 0 -columnspan 2 -sticky w
                    } else {
                    }
                    for { set row_used $save_row } { $row_used <= $currentrow } { incr row_used } {
                        array set rowinfo [grid rowconfigure $f $row_used]
                        if { $rowinfo(-weight) } {
                            set row_weight_requested 1
                            break
                        }
                    }
                }
            }
        }
        set GidData($N,WIDGET,$i) [list $widget $WidgetLabel]
        incr currentrow
    }
    if { $thereishelp } {
        grid $GidData(HELPBUTTON)
    } else {
        grid remove $GidData(HELPBUTTON)
    }
    
    #grid $f -row 0 -column 0 -sticky snew
    
    for { set col_used 0 } { $col_used < 10 } { incr col_used } {
        array set colinfo [grid columnconfigure $f $col_used]
        if { $colinfo(-weight) } {
            set column_weight_requested 1
            break
        }
    }
    
    if { !$row_weight_requested } {
        grid rowconf $f $currentrow -weight 1
    }
    if { !$column_weight_requested } {
        grid columnconf $f 2 -weight 1
    }
    grid columnconf $f 0 -minsize [expr $colwidth + 4] -pad 4
    list $row_weight_requested $column_weight_requested
}



# CreateTraceProc -
#
# Arguments:
#
# Results:
#

proc CreateTraceProc { GDN TYPE Idx } {
    upvar #0 $GDN GidData
    
    
    set ProcName [DWGetDepProcName $GDN $TYPE $Idx]
    if { [namespace which $ProcName] != "" } {
        rename $ProcName ""
    }
    proc $ProcName { } $GidData($TYPE,DEPENDENCIES,$Idx)
    #eval "proc $ProcName \{ \} $GidData($TYPE,DEPENDENCIES,$Idx)"
    return $ProcName
}

# RemoveTraceProcs -
#
# Arguments:
#
# Results:
#

proc RemoveTraceProcs { GDN } {
    foreach P [info commands $GDN*] {
        rename $P ""
    }
}

proc DWReconfigureTextGrid { text xscroll yscroll } {
    set b [bind $text <Configure>]
    bind $text <Configure> ""
    if { [winfo exists $xscroll] } { grid forget $xscroll }
    if { [winfo exists $yscroll] } { grid forget $yscroll }
    update idletasks
    if { [winfo exists $xscroll] } {
        if { [lindex [$text xview] 0] == 0 &&  [lindex [$text xview] 1] == 1 } {
            grid forget $xscroll
        } else { grid $xscroll -row 2 -column 1 -sticky ew}
    }
    
    if { [winfo exists $yscroll] } {
        
        if { [lindex [$text yview] 0] == 0 && [lindex [$text yview] 1] == 1 } {
            grid forget $yscroll
        } else { grid $yscroll -row 1 -column 2 -sticky ns }
    }
    after 100 [list bind $text <Configure> $b]
}

proc DWReconfigureText { text xscroll yscroll } {
    upvar ::ExtendInfo($text) geometry
    
    set b [bind $text <Configure>]
    bind $text <Configure> ""
    grid remove $xscroll
    grid remove $yscroll
    
    if { ![llength $geometry] } {
        foreach {x1 y1 x2 y2} [$text bbox frame] {}
        set geometry [list [expr $x2-$x1] [expr $y2-$y1]]
        #set geometry [list [winfo reqwidth $text.f] [winfo reqheight $text.f]]
        #$text create line $x2 $y1 $x2 $y2 -fill red
    }
    set x1_i 0 ; set y1_i 0 ; set x2_i 0 ; set y2_i 0
    foreach "x1_i y1_i x2_i y2_i" [$text bbox images] break
    set width_i [expr $x2_i-$x1_i]
    
    
    set width_w  [lindex $geometry 0]
    set height_w [lindex $geometry 1]
    set width_c  [winfo width $text]
    set height_c [winfo height $text]
    
    set change_geometry 0
    if { $width_w <= $width_c && $width_i == 0 } {
        set new_width $width_c
        set change_geometry 1
    } else {
        set new_width $width_w
    }
    if { $height_w <= $height_c && $width_i == 0} {
        set new_height $height_c
        set change_geometry 1
    } else {
        set new_height $height_w
    }
    
    if { $change_geometry } {
        $text itemconfigure frame -width $new_width -height $new_height
    }
    if { [lindex [$text xview] 0] == 0 &&  [lindex [$text xview] 1] == 1 } {
        grid remove $xscroll
    } else {
        grid $xscroll
    }
    
    if { [lindex [$text yview] 0] == 0 && [lindex [$text yview] 1] == 1 } {
        grid remove $yscroll
    } else {
        grid $yscroll
    }
    after 100 [list bind $text <Configure> $b]
}

proc DWDeleteAssociatedImage { GDN name1 name2 op } {
    image delete $GDN
}

proc DWCreateText { GDN w name from to } {
    global $GDN
    upvar #0 $GDN GidData
    set dir $GidData(DirPT)
    set sw $w.sw
    set sf $sw.sf
    
    catch {
        destroy $sw;
    }
    #WarnWin "DWCreateText: ... ---$w---$name---$from---$to-"
    set ::gsw $sw
    set ::gsf $sf
    
    ScrolledWindow $sw
    ScrollableFrame $sf
    
    set f [$sf getframe]
    set ::GidData([winfo toplevel $w],SF) $f
    
    set fdata [ttk::frame $f.fdata]
    set fimg [ttk::frame $f.fimg]   
    
    foreach opt {-constrainedheight -constrainedwidth} val [DWFillText $GDN $fdata $name $from $to] {
        $sf configure $opt $val
    }
    set img 0
    foreach i [array names GidData $name,IMGOBJ,*] {
        if {[regexp {.*,([0-9]*)} $i {} num] &&
            (($num >= $from && $num <= $to) ||
            ($num == 0 && $from == 1))} {
            lappend GidData($name,WIDGET,$num) $fimg.l$img
            grid [label $fimg.l$img -image $GidData($i)] \
                -row $img -column 0 -sticky "nw"
            if {[info exists GidData($name,STATE,$num)] &&
                $GidData($name,STATE,$num) == "hidden"} {
                grid remove $fimg.l$img
            }
            incr img
        }
    }
    grid $fdata -row 0 -column 0 -sticky "snew"
    if {$img} {
        grid $fimg -row 0 -column 1 -sticky "snew"
    } else {
        destroy $fimg
    }
    grid columnconfigure $f 0 -weight 1
    grid rowconfigure $f 0 -weight 1
    $sw setwidget $sf
    grid $sw -row 0 -column 0 -sticky "snew"
    grid rowconfigure $w 0 -weight 1
    grid columnconfigure $w 0 -weight 1
}

proc DWCreateTabOfTexts { GDN w P basename} {
    # kike comment again: global $GDN
    # it is necessary declare global when GidData alias is not valid, like when
    #    * is used as -variable of some widget  
    #    * is passed to other proc that do another upvar #0 like $gu\::Create  ... upvar \#0 $UnitsListName UnitsList
    upvar #0 $GDN GidData
    
    #destruyo la informacion de los widgets
    foreach i [array names GidData *,WIDGET,*] {
        set GidData($i) {}
    }
    
    foreach i [array names GidData *,INSTANCE,*] {
        unset GidData($i)
    }
    
    #lanzo las dependencias iniciales
    set GidData($P,VISIBLE) 0
    TriggerDependencies $GDN
    
    set fieldsnum [llength [array names GidData $P,QUESTION,*]]
    if { ![info exists GidData($P,TITLE,0)] } {
        set GidData($P,TITLE,0) $basename
    }
    
    set titlesnums [list]
    set prefix $P,TITLE,
    set prefix_length [string length $prefix]
    foreach i [array names GidData ${prefix}*] {
        lappend titlesnums [string range $i $prefix_length end]        
    }    
    set titlesnums [lsort -integer $titlesnums]
    if { [llength $titlesnums] == 1 } {
        catch { destroy $w.c }
        ttk::frame $w.c
        DWCreateText $GDN $w.c $P 1 $fieldsnum
        pack $w.c -fill both -expand yes
        set ::GidData([winfo toplevel $w],has,notebook) 0
    } else {
        catch {
            destroy $w.c
        }
        set ::GidData([winfo toplevel $w],has,notebook) 1
        set GidData($P,notebook) [NoteBook $w.c]
        set GidData($P,notebook,all_pages) [list]
        set first_vis 0
        for { set i 0 } { $i < [llength $titlesnums] } { incr i } {
            set inipos [expr [lindex $titlesnums $i]+1]
            if { $i < [expr [llength $titlesnums]-1] } {
                set endpos [lindex $titlesnums [expr $i + 1]]
            } else {
                set endpos $fieldsnum
            }
            set page_id $GidData($P,TITLE,[lindex $titlesnums $i])
            set page_path $GidData($P,TITLEINFO,$page_id,path)
            set page_txt $GidData($P,TITLEINFO,$page_id,text)
            lappend GidData($P,notebook,all_pages) $page_id
            set fpage [$w.c insert end $page_path -text $page_txt]
            set GidData($P,TITLEINFO,$page_id,frame) $fpage
            if {$GidData($P,TITLEINFO,$page_id,state) eq "hidden"} {
                # create the frame contents in order to be reused
                DWCreateText $GDN $fpage $P $inipos $endpos
                # remove it without destroying the frame contents
                $w.c delete $page_path 0
            } else {
                set GidData($P,TITLEINFO,$page_id,createcmd) \
                    [list DWCreateText $GDN $fpage $P $inipos $endpos]
                if {!$first_vis} {
                    eval $GidData($P,TITLEINFO,$page_id,createcmd)
                    set first_vis 1
                } else {
                    $w.c itemconfigure $page_path -createcmd \
                        $GidData($P,TITLEINFO,$page_id,createcmd)
                }
                $w.c itemconfigure $page_path \
                    -state $GidData($P,TITLEINFO,$page_id,state)
            }
        }
        $w.c raise [lindex [$w.c pages] 0]
        pack $w.c -fill both -expand yes
        
        # now it's visible
        set GidData($P,VISIBLE) 1
    }
}

#----------------------------------------PROBLEM DATA--------------

proc DWReadPRBfile { GDN filename } {
    global $GDN
    upvar #0 $GDN GidData
    
    foreach i [array names GidData PD*] { unset GidData($i) }
    foreach i [array names GidData ID*] { unset GidData($i) }
    
    set f [open $filename r]
    set inum 0
    gets $f aa
    while { ![eof $f] } {
        gets $f aa
        if { $aa == "" ||  [string index $aa 0] == "#" } { continue }
        regexp {^([^ ]*) (.*)$} $aa {} field value
        if { $field == "END" } { break }
        if { $field == "QUESTION:" } { incr inum }
        set GidData(PD,$field,$inum) $value
    }
    set inum 0
    gets $f aa
    while { ![eof $f] } {
        gets $f aa
        if { [string index $aa 0] == "#" } { continue }
        regexp {^([^ ]*) (.*)$} $aa {} field value
        if { $field == "END" } { break }
        if { $field == "QUESTION:" } { incr inum }
        set GidData(ID,$field,$inum) $value
    }
    close $f
}


# ------------------------------------------------------------------------------
# DWOnSelectInterval --
#
#  Invoked under -modifycmd of intervals ComboBox. It should change the current
#  interval showed in the data window.
# ------------------------------------------------------------------------------

proc DWOnSelectInterval { GDN {current -1}} {
    upvar #0 $GDN GidData
    
    if { ![ info exists GidData(ID,current)]} {
        return
    }
    
    if {$current==-1} {
        set current [$GidData(ID,combo) get]
        if {$GidData(ID,current) == $current} {
            return
        }
        if {[catch "CheckDirtyState $GDN" result]} {
            if [string equal $result CANCEL] {
                return
            } else {
                error $result
            }
        }
    }
    set GidData(ID,current) $current
    
    # Hay que pedirle a Gid que cambie de intervalo activo.
    DWSetActiveInterval $GidData(ID,current)
    # Hay que limpiar GidData para
    # luego pedirle los datos a Gid de este intervalo y ponerlos en GidData.
    # disable dependencies to refill data
    set GidData(DEPDISABLED) 1
    DWRefillIntervalData $GDN
    # now enable again dependencies
    set GidData(DEPDISABLED) 0
    TriggerDependencies $GDN
}

proc DWInitComboInterval { GDN } {
    upvar #0 $GDN GidData
    lassign [GiD_Info intvdata num] GidData(ID,current) max
    set values [list]
    for {set i 1} {$i <= $max} {incr i} {
        lappend values $i
    }
    $GidData(ID,combo) configure -values $values
    # $GidData(ID,combo) set @[expr $GidData(ID,current)-1]
    #$GidData(ID,combo) set [ lindex $values 0]
    $GidData(ID,combo) set $GidData(ID,current)
}

proc DWIntervalDataWindow { {Book "Default"} {BookOrd 1} {UseWindow ""}} {
    
    if { $Book == "Default" } {
        set books_and_indices [GiD_Info intvdata books]
        if { [llength $books_and_indices] } {
            set Book [lindex $books_and_indices 0]
        }
    }
    if { [DWInitWindow INT $Book $BookOrd $UseWindow w] != "created" } {
        return
    }
    
    set GDN GD_INT_$BookOrd
    global $GDN
    globalarray $GDN
    upvar #0 $GDN GidData
    
    set GidData(CALLWIN) DWIntervalDataWindow
    set GidData(BOOKORD) $BookOrd
    set GidData(WINDOW) $w
    
    # manipulador que atendera la validacion de los datos
    wm protocol $w WM_DELETE_WINDOW [list DWCloseWindow $GDN]
    
    # Activa el el primer intervalo pues es que define los books
    
    if { ![GenericFillGidData $GDN ID $Book] } {
        destroy $w
        return
    }
    
    #set GidData(ID,maxinterval) 0
    
    ttk::frame $w.level
    
    # A partir de aqui: controles de la interface.    
    TTKComboBox $w.level.combo -state readonly -modifycmd [list DWOnSelectInterval $GDN]
    set GidData(ID,combo) $w.level.combo
    
    ttk::button $w.level.new -image [gid_themes::GetImage new.png medium_icons] -command [list DWNewInterval $GDN 0]
    ttk::button $w.level.insert_new -image [gid_themes::GetImage insert_new.png medium_icons] -command [list DWNewInterval $GDN 1]
    ttk::button $w.level.delete -image [gid_themes::GetImage delete.png medium_icons] -command [list DWDeleteActiveInterval]
   
    GidHelp $w.level.new [_ "New interval"]
    GidHelp $w.level.insert_new [_ "Insert a new interval before the current"]
    GidHelp $w.level.delete [_ "Delete the current interval"]
    
    grid $w.level.combo $w.level.new $w.level.insert_new $w.level.delete -sticky "ew"
    CreateJumpButton $GDN $w.level 4
    grid columnconfigure $w.level {0 4} -weight 1   
    
    ttk::frame $w.f -style groove.TFrame
    
    set GidData(ID,mainframe) $w.f
    
    ttk::frame $w.but -style BottomFrame.TFrame
    ttk::button $w.but.accept -text [_ "Accept"] \
        -command [list DWIntervalAccept $GDN] -underline 0 -style BottomFrame.TButton
    ttk::button $w.but.close -text [_ "Close"] -underline 0 \
        -command [list DWCloseWindow $GDN] -style BottomFrame.TButton
       
    grid $w.but.accept -row 1 -column 1 -sticky e -padx 5 -pady 5
    grid $w.but.close -row 1 -column 2 -sticky w -padx 5 -pady 5
    
    
    grid $w.level -stick ew -padx 2 -pady 2
    grid $w.f -stick nsew
    grid $w.but -stick ew -padx 2 -pady 2
    
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 1 -weight 1
    
    grid anchor $w.but center
    
    
    foreach i "$w.but.accept $w.but.close" {
        set l [string tolower [string index [$i cget -text] [$i cget -underline]]]
        # this may fail when $l is not a valid keysym
        catch {
            bind $w <Alt-$l> "$i invoke;break"
        }
    }
    bind $w <Escape> [list $w.but.close invoke]
    
    DWInitComboInterval $GDN
    DWCreateTabOfTexts $GDN $GidData(ID,mainframe) ID "Base data"   
        
    DWMapWindow $w
    focus $w.but.accept
    
    return $GDN
}

proc UnsetGDN { GDN } {
    global $GDN
    upvar #0 $GDN GidData
    unset GidData
}

#  GetDirectoryProblemType --
#    Returns the path where the problemtype is

proc GetDirectoryProblemType { } {
    set r ""
    catch {
        set r [GiD_Info problemtypepath]
    }
    return $r
}

proc DWProblemDataWindow { {Book "Default"} {BookOrd 1} {UseWindow ""} } {
    
    if { $Book == "Default" } {
        set books_and_indices [GiD_Info gendata books]
        if { [llength $books_and_indices] } {
            set Book [lindex $books_and_indices 0]
        }
    }
    if { [DWInitWindow PD $Book $BookOrd $UseWindow w] != "created" } {
        return
    }
    
    set GDN GD_PD_$BookOrd
    
    globalarray $GDN
    global $GDN
    upvar #0 $GDN GidData
    
    set GidData(CALLWIN) DWProblemDataWindow
    set GidData(BOOKORD) $BookOrd
    set GidData(WINDOW) $w
    
    # manipulador que atendera la validacion de los datos
    wm protocol $w WM_DELETE_WINDOW "DWCloseWindow $GDN"
    
    if {![GenericFillGidData $GDN PD $Book] } {
        destroy $w
        return
    }
    
    ttk::frame $w.ftop
    CreateJumpButton $GDN $w.ftop 1
    grid $w.ftop -row 0 -column 0 -sticky "ew"
    grid columnconfigure $w.ftop 0 -weight 1
    ttk::frame $w.f -style groove.TFrame
    set GidData(PD,mainframe) $w.f
    
    ttk::frame $w.but -style BottomFrame.TFrame
    ttk::button $w.but.accept -text [_ "Accept"] \
        -command [list DWProblemAccept $GDN] -underline 0 \
        -style BottomFrame.TButton
    ttk::button $w.but.close -text [_ "Close"] \
        -command [list DWCloseWindow $GDN] -underline 0 \
        -style BottomFrame.TButton  
    
    grid $w.but.accept $w.but.close -padx 5 -pady 5
    grid $w.but -row 2 -sticky snew
    grid anchor $w.but center
    
    DWCreateTabOfTexts $GDN $GidData(PD,mainframe) PD "Base data"
    grid $GidData(PD,mainframe) -row 1 -column 0 -sticky "snew"
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 1 -weight 1    
    DWMapWindow $w
    focus $w.but.accept
    
    foreach i "$w.but.accept $w.but.close" {
        set l [string tolower [string index [$i cget -text] [$i cget -underline]]]
        # this may fail when $l is not a valid keysym
        catch {
            bind $w <Alt-$l> "$i invoke;break"
        }
    }
    bind $w <Escape> [list $w.but.close invoke]
    
    return $GDN
}

# ------------------------------------------------------------------------------
# DWOnSelectMaterial --
#
#  Invoked under -modifycmd of materials ComboBox. It should change the material
#  info showed in the data window.
# ------------------------------------------------------------------------------

proc DWOnSelectMaterial {GDN} {
    upvar #0 $GDN GidData
    
    if { ![ info exists GidData(MAT,cursel)]} {
        return
    }
    
    if {$GidData(MAT,cursel) == [set sel [$GidData(MAT,combo) get]]} {
        return
    }
    # check if material needs to be loaded
    set idx [ lsearch [ $GidData(MAT,combo) cget -values] $sel]
    set mat [DWGetGIDMaterial $GDN $idx]
    if { !$GidData(MAT,$mat,isloaded)} {
        DW_LoadMaterial $GDN $mat
    }
    if {$GidData(MAT,cursel)!=-1} {
        CheckDirtyState $GDN
    }
    set GidData(MAT,cursel) $sel
    
    # BACKWARD COMPATIBILITY
    set GidData(MAT,matname) [DWGetGUIMaterial $GDN]
    # REMOVE AFTER VERSION 8.0
    DWMaterialUpdateProperties $GDN
    DWCreateMaterialFrame $GDN
}

proc DWMatChangeCurrent { GDN matName } {
  upvar #0 $GDN GidData
  
  set idx [ lsearch $GidData(MAT,materials) $matName ]
  if { $idx != -1 } {
    $GidData(MAT,combo) current $idx
  }
}

# ------------------------------------------------------------------------------
# DWCreateMaterialFrame --
#
#  Fill in frame of GDN the info corresponding to current material.
# ------------------------------------------------------------------------------

proc DWCreateMaterialFrame { GDN } {
    upvar #0 $GDN GidData    
    set typemat [DWGetTypeMAT $GDN]    
    DWCreateTabOfTexts $GDN $GidData(MAT,mainframe) $typemat {Base properties}
}

#BACKWARD COMPATIBILITY
proc DWGetCurrentMaterial { GDN } {
    DWGetGIDMaterial $GDN
}

# ------------------------------------------------------------------------------
# DWGetGIDMaterial --
#
#  Return the current original (untranslated) material name
# ------------------------------------------------------------------------------

proc DWGetGIDMaterial { GDN {idx -1}} {
    upvar \#0 $GDN GidData
    
    # lindex $GidData(MAT,materials) [expr ($idx==-1)?$GidData(MAT,cursel):$idx]
    if { $idx == -1} {
        # return $GidData(MAT,cursel)
        set idx [ lsearch [ $GidData(MAT,combo) cget -values] $GidData(MAT,cursel)]
        return [ lindex $GidData(MAT,materials) $idx]
    } else {
        return [ lindex $GidData(MAT,materials) $idx]
    }
}

proc DWGetGUIMaterial { GDN } {
    upvar \#0 $GDN GidData
    
    # lindex [$GidData(MAT,combo) cget -values] $GidData(MAT,cursel)
    return $GidData(MAT,cursel)
}

#----------------------------------------MATERIALS--------------------
# KILL THIS PROC
proc DWChangeMatName { GDN } {
    global $GDN
    upvar \#0 $GDN GidData
    
    if { $GidData(MAT,matname) == "" } { return }
    
    if { $GidData(MAT,prevmatname) == $GidData(MAT,matname) } {
        return
    }
    
    if { $GidData(MAT,prevmatname) != "" && $GidData(MAT,prevmatname) != -1} {
        
        CheckDirtyState $GDN
        
        #        if [catch "CheckDirtyState $GDN" result] {
            #            if [string equal $result CANCEL] {
                #                return
                #            } else {
                #                error $result
                #            }
            #        }
    }
    
    set GidData(MAT,prevmatname) $GidData(MAT,matname)
    
    #DWCreateTabOfTexts $GDN $GidData(MAT,mainframe) MAT,[DWSpace2Under $GidData(MAT,matname)] {Base properties}
    DWCreateTabOfTexts $GDN $GidData(MAT,mainframe) MAT,$GidData(MAT,matname) {Base properties}
}

proc DWUpFirstLetter { str } {
    set expression {^[0-9A-Z_](.*)}
    if { [ regexp $expression $str ] } {
        return $str
    }
    append uf [string toupper [string index $str 0]] [string range $str 1 end]
    return $uf
}

proc DWDownFirstLetter { str } {
    if [regexp {^[0-9A-Z_]+$} $str] {
        return $str
    }
    append df [string tolower [string index $str 0]] [string range $str 1 end]
    return $df
}

proc DWMatWindow { {Book "Default"} {BookOrd 1} {UseWindow ""} } {
    global GidPriv
    global gidDataLabels

    if { $Book == "Default" } {
        set books_and_indices [GiD_Info materials books]
        if { [llength $books_and_indices] } {
            set Book [lindex $books_and_indices 0]
        }
    }
    
    if { [DWInitWindow MAT $Book $BookOrd $UseWindow w] != "created" } {
        return
    }
    
    set GDN GD_MAT_$BookOrd
    
    global $GDN
    globalarray $GDN
    upvar \#0 $GDN GidData
    
    set GidData(CALLWIN) DWMatWindow
    set GidData(BOOKORD) $BookOrd
    set GidData(WINDOW) $w
    
    # manipulador que atendera la validacion de los datos
    wm protocol $w WM_DELETE_WINDOW "DWCloseWindow $GDN"
    
    if { ![MATFillGidData $GDN $Book] } {
        destroy $w
        return
    }
    
    ttk::frame $w.level
    
    TTKComboBox $w.level.combo -state readonly -modifycmd [list DWOnSelectMaterial $GDN]
    set GidData(MAT,combo) $w.level.combo
    
    ttk::button $w.level.mod -image [gid_themes::GetImage prefs.png medium_icons] -command [list DWModifyMat $GDN] -underline 0
    GidHelp $w.level.mod [_ "Update changes"]
    
    ttk::button $w.level.new -image [gid_themes::GetImage new.png medium_icons] -command [list DWNewMat $GDN $w] -underline 0
    GidHelp $w.level.new [_ "New %s" $GidData(MAT,singular)]
    
    ttk::button $w.level.del -image [gid_themes::GetImage delete.png medium_icons] -command [list DWDeleteMat $GDN] -underline 0
    GidHelp $w.level.del [_ "Delete %s" $GidData(MAT,singular)]

    ttk::button $w.level.rename -image [gid_themes::GetImage rename.png medium_icons] -command [list DWRenameMat $GDN] -underline 0
    GidHelp $w.level.rename [_ "Rename %s" $GidData(MAT,singular)]

    set GidData(widgets,modify)  $w.level.mod
    set GidData(widgets,new) $w.level.new
    set GidData(widgets,delete) $w.level.del
    set GidData(widgets,rename) $w.level.rename
    
    grid $w.level.combo $w.level.mod $w.level.new $w.level.del $w.level.rename -sticky nsew
    
    set jumpbutton [CreateJumpButton $GDN $w.level 5]
    
    #grid $jumpbutton    -row 0 -column 5 -sticky "ew"
    grid columnconfigure $w.level 0 -weight 3
    grid columnconfigure $w.level 5 -weight 1
    
    ttk::frame $w.f -style groove.TFrame
    set GidData(MAT,mainframe) $w.f
    
    ttk::frame $w.but -style ridge.TFrame -borderwidth 2
    # maybe not -style BottomFrame.TFrame
    
    # ----- Opcion Assign -----
    
    set MenuAssign $w.but.assign.menu
    ttk::menubutton $w.but.assign -text [_ "Assign"] -underline 0 \
        -menu $MenuAssign 
        #-style Horizontal.IconButtonWithMenu
    set GidData(widgets,assign) $w.but.assign
    
    menu $MenuAssign -borderwidth 1 -activeborderwidth 1 \
        -postcommand [list PostCmdAssignMat $GDN $w $MenuAssign]
    
    # ----- Opcion Draw -----
    
    set MenuDraw $w.but.draw.menu
    ttk::menubutton $w.but.draw -text [_ "Draw"] -underline 0 \
        -menu $MenuDraw 
        #-style Horizontal.IconButtonWithMenu
    set GidData(widgets,draw) $w.but.draw
    
    menu $MenuDraw -borderwidth 1 -activeborderwidth 1 \
        -postcommand [list DWPostDrawMaterial $GDN $MenuDraw]
    
    $MenuDraw  add command -label XXXX \
        -command [list DWDrawMaterials $GDN $w onemat]
    
    set labelmat ""
    if { [info exists gidDataLabels(Materials)] } {
        set labelmat $gidDataLabels(Materials)
    }
    if { $labelmat == "" } {
        set labelmat [_ "Materials"]
    }
    
    $MenuDraw add command -label [_ "All %s" [DWUnder2Space $labelmat]] -command \
        [list DWDrawMaterials $GDN $w allmats]
    
    # ----- Opcion Unassign -----
    set MenuUnassign $w.but.unassign.menu
    ttk::menubutton $w.but.unassign -text [_ "Unassign"] -underline 0 \
        -menu $MenuUnassign 
        #-style Horizontal.IconButtonWithMenu
    set GidData(widgets,unassign) $w.but.unassign
    menu $MenuUnassign -borderwidth 1 -activeborderwidth 1 \
        -postcommand [list DWUnassignMat $w $GDN]
    
    # ----- Opcion Import/Export -----
    ttk::button $w.but.impexp -text [_ "Exchange"] \
        -command [list DWMatImportExport $w $Book] \
        -style Window.TButton
    set GidData(widgets,impexp) $w.but.impexp
    
    grid $w.but.assign $w.but.draw $w.but.unassign $w.but.impexp -padx 2 -sticky ew
    #grid configure $w.but.impexp -sticky "ns" ;#because now the ttk::menubutton heigh is bigger that button height
    #grid columnconfigure $w.but 4 -weight 1
    grid rowconfigure $w.but 0 -weight 1
    grid columnconfigure $w.but "0 1 2 3" -weight 1    
    
    ttk::frame $w.but2 -style BottomFrame.TFrame
    ttk::button $w.but2.close -text [_ "Close"] \
        -command [list DWCloseWindow $GDN] -underline 0 -style BottomFrame.TButton   
    pack $w.but2.close -pady 5
    
    pack $w.level -fill x  -side top -padx 2 -pady 2
    pack $w.but2 -side bottom -fill x
    pack $w.but -ipady 3 -fill x -side bottom
    pack $w.f -fill both -expand yes
    

    set combo_info [list]
    foreach mat $GidData(MAT,materials) {
        lappend combo_info [= [DWUnder2Space $mat]]
    }
    $GidData(MAT,combo) configure -values $combo_info
    #$GidData(MAT,combo) set first
    $GidData(MAT,combo) set [ lindex $combo_info 0]
    #  set GidData(MAT,prevmatname) ""
    
    #set GidData(MAT,matname) [lindex $GidData(MAT,nameslist) 0]
    # set GidData(MAT,cursel) 0
    set GidData(MAT,cursel) [ lindex $combo_info 0]
    # Load first material
    set material [DWGetGIDMaterial $GDN 0]
    if { $material != "" } {
        DW_LoadMaterial $GDN $material
        DWCreateMaterialFrame $GDN
        # BACKWARD COMPATIBILITY
        set material_translated [DWGetGUIMaterial $GDN]
        set GidData(MAT,matname) $material_translated
        # REMOVE AFTER VERSION 8.0
    }
    
    DWMaterialUpdateProperties $GDN
        
    DWMapWindow $w
    focus $w.but.assign
    
    foreach i [list $w.but.assign $w.but.draw $w.but.unassign $w.but2.close] {
        set l [string tolower [string index [$i cget -text] [$i cget -underline]]]
        # this may fail when $l is not a valid keysym
        catch {
            bind $w <Alt-$l> "$i invoke;break"
        }
    }
    
    bind $w <Escape> [list $w.but2.close invoke]
    
    return $GDN
}

proc DWIsGeomtry { } {
    if { [GiD_Info Project ViewMode] eq "GEOMETRYUSE" } {
        return 1
    }
    return 0
}

proc DWMaterialUpdateProperties { GDN } {
    upvar \#0 $GDN GidData
    
    set book $GidData(BOOK)
    set mat [DWGetGIDMaterial $GDN]
    set opt_list [DataBehaviour::GetMaterialsOpts]
    array set book_values [DataBehaviour::Materials 1 $book $opt_list]
    array set mat_values [DataBehaviour::Materials 0 $mat $opt_list]
    foreach opt $opt_list {
        set b $book_values($opt)
        set m $mat_values($opt)
        if {$m eq "hide" || ($m eq "default" && $b eq "hide")} {
            # we must hide the option
            if {[llength [grid info $GidData(widgets,$opt)]]} {
                grid remove $GidData(widgets,$opt)
                # better disabled than removed
                # $GidData(widgets,$opt) configure -state disable
            }
        } else {
            # we must ensure the the option is visible
            if {![llength [grid info $GidData(widgets,$opt)]]} {
                grid $GidData(widgets,$opt)
            }
            if {$m eq "disable" || ($m eq "default" && $b eq "disable")} {
                # visible but disabled
                $GidData(widgets,$opt) configure -state disabled
            } else {
                # visible and enabled
                $GidData(widgets,$opt) configure -state normal
            }
        }
    }
}

#----------------------------------------CONDITIONS------------------

proc DWReadCondfile { filename } {
    global GidData
    foreach i [array names GidData CND,*] { unset GidData($i) }
    set GidData(CND,nameslist) ""
    
    set f [open $filename r]
    while { ![eof $f] } {
        gets $f aa
        if { $aa == "" || [string index $aa 0] == "#" } { continue }
        if { ![regexp {NUMBER:[ ]*[0-9]*[ ]*CONDITION:?[ ]*([^ ]*)} \
            $aa {} cndname] } { break }
    lappend GidData(CND,nameslist) $cndname
    set inum 0
    while { ![eof $f] } {
        gets $f aa
        if { [string index $aa 0] == "#" } { continue }
        regexp {^([^ ]*) (.*)$} $aa {} field value
        if { $field == "END" } { break }
        if { $field == "QUESTION:" } { incr inum }
        set GidData(CND,$cndname,$field,$inum) $value
    }
}
close $f
}

# try to preserve internally true names, only decorate for labels
#proc DWSpace2Under { str } {
#    regsub -all { } $str {_} str
#    return $str
#}

proc DWUnder2Space { str } {
    regsub -all {_} $str { } str
    return [string trim $str]
}

# ------------------------------------------------------------------------------
# DWOnSelectCNDLevel --
#
#  Invoked when a level button is pressed.
# ------------------------------------------------------------------------------

proc DWOnSelectCNDLevel { GDN {cnd_idx 0}} {
    upvar \#0 $GDN GidData
    
    if {[set ov0 $GidData(CND,level)] == [set ov1 $GidData(CND,varlevel)] } {
        # ramsan adding
        # $GidData(CND,combo) set $cnd_idx
        $GidData(CND,combo) set [ lindex [ $GidData(CND,combo) cget -values] $cnd_idx]
        DWOnSelectCondition $GDN $cnd_idx
        return
    }
    
    if {$ov0 ne ""} {
        # set cnd_sel [lindex $GidData(CND,$ov0,GUINAMES) \
            #                  [$GidData(CND,combo) get]]
        set cnd_sel [$GidData(CND,combo) get]
        set match_idx [lsearch $GidData(CND,$ov1,GUINAMES) $cnd_sel]
        if {$match_idx != -1} {
            # set cnd_idx @$match_idx
            set cnd_idx $match_idx
        }
    }
    set GidData(CND,level) $ov1
    
    $GidData(CND,combo) configure -values $GidData(CND,$ov1,GUINAMES)
    if { [string equal $ov1 "over_layer"] } {
        grid remove $GidData(CND,ASSIGN-ENTITY)
        grid remove $GidData(CND,ASSIGN-GROUP)
        grid $GidData(CND,ASSIGN-LAYER) -row 0 -column 0 -sticky "ew" -padx 2
    } elseif { [string equal $ov1 "over_group"] } {
        grid remove $GidData(CND,ASSIGN-ENTITY)
        grid remove $GidData(CND,ASSIGN-LAYER)
        grid $GidData(CND,ASSIGN-GROUP) -row 0 -column 0 -sticky "ew" -padx 2
    } else {
        grid remove $GidData(CND,ASSIGN-LAYER)
        grid remove $GidData(CND,ASSIGN-GROUP)
        grid $GidData(CND,ASSIGN-ENTITY) -row 0 -column 0 -sticky "ew" -padx 2
    }
    # show the condition requested, normally the first one
    # $GidData(CND,combo) set $cnd_idx
    $GidData(CND,combo) set [ lindex [ $GidData(CND,combo) cget -values] $cnd_idx]
    if { ![ info exists GidData(CND,cursel)]} {
        set GidData(CND,cursel) [ lindex $GidData(CND,$ov1,GUINAMES) $cnd_idx]
    }
    DWOnSelectCondition $GDN $cnd_idx
}

proc DWSetCNDLevel { GDN level {cnd_idx 0}} {
    upvar \#0 $GDN GidData
    
    set GidData(CND,varlevel) $level
    DWOnSelectCNDLevel $GDN $cnd_idx
}

# ------------------------------------------------------------------------------
# DWAfterFinishButton --
#
# ------------------------------------------------------------------------------

proc DWAfterFinishButton { GDN } {
    GiD_Process escape escape escape escape
    # restore the state of the widgets.
    #TriggerDependencies $GDN

    global $GDN
    upvar #0 $GDN GidData
    set w $GidData(WINDOW)
    if { [ winfo exist .gid] && [ winfo exists $w] && [ winfo ismapped .gid] && [ winfo ismapped $w]} {
        raise $w .gid
    }
}

# DWDrawConds --
#
#       Muestra graficamente las condiciones asignadas.
#
# Argumentos:
#
#      wCond --> ventana de condiciones desde donde fue invocado.
#      what --> subcomando de draw. Puede tomar los valores:
#
#               - thiscond
#                  se refiere a la condicion actual $GidData(CND,cndname)
#                  si field es "" dibuja todas las condiones, en caso contrario dibuja
#                  dibuja las asociadas al campo $field. Las condiciones se dibujan
#                  con simbolos.
#
#               - thisbook
#
#               - colors
#
#               - allconds
#
#               - allcondsLA
#
#               - allcondsLAC
#      field --> campo asociado se interpreta en funcion del valor del argumento what
#
# Resultado:
#
#      Las condiciones dibujadas hasta que el usuario presione escape o el boton finish.

proc DWDrawConds { GDN what {field ""}} {
    global $GDN
    upvar \#0 $GDN GidData
    
    set _CndName [DWGetGIDCondName $GDN]
    set wCond $GidData(WINDOW)
    
    set singular $GidData(CND,singular)
    
    switch $what {
        thiscond {
            if { $_CndName == "" } {
                WarnWin [_ "Sorry, but first a %s must be selected" $singular] $wCond
                return
            }
            if { $field == "" } {
                GiD_Process Mescape Data Conditions DrawCond $_CndName -draw-
            } else {
                GiD_Process Mescape Data Conditions DrawCond $_CndName $field
            }
        }
        thisbook {
            GiD_Process Mescape Data Conditions DrawCond -BOOKS- $GidData(BOOK) -draw-
        }
        thisbookLA {
            GiD_Process Mescape Data Conditions DrawCond -LocalAxes- -BOOKS- $GidData(BOOK) -draw-
        }
        thisbookLAC {
            GiD_Process Mescape Data Conditions DrawCond -LocalAxes- -LocalAxes- -BOOKS- $GidData(BOOK) -draw-
        }
        colors {
            if { $_CndName == ""} {
                WarnWin [_ "Sorry, but first a %s must be selected" $singular] $wCond
                return
            }
            if { $field == "" } {
                GiD_Process Mescape Data Conditions DrawCond -ByColor- $_CndName -draw-
            } else {
                GiD_Process Mescape Data Conditions DrawCond -ByColor- $_CndName $field
            }
        }
        allconds {
            GiD_Process Mescape Data Conditions DrawCond -drawall-
        }
        thiscondLA {
            if { $_CndName == ""} {
                WarnWin [_ "Sorry, but first a %s must be selected" $singular] $wCond
                return
            }
            GiD_Process Mescape Data Conditions DrawCond -LocalAxes- $_CndName -draw-
        }
        thiscondLAC {
            if { $_CndName == ""} {
                WarnWin [_ "Sorry, but first a %s must be selected" $singular] $wCond
                return
            }
            GiD_Process Mescape Data Conditions DrawCond -LocalAxes- -LocalAxes- $_CndName -draw-
        }
        allcondsLA {
            GiD_Process Mescape Data Conditions DrawCond -LocalAxes- -drawall-
        }
        allcondsLAC {
            GiD_Process Mescape Data Conditions DrawCond -LocalAxes- -LocalAxes- -drawall-
        }
    }
    
    set SmallWinSelecting [GiD_Set SmallWinSelecting]
    FinishButton $wCond $wCond.but [_ "Press 'Finish' to end selection"] \
        "DWAfterFinishButton $GDN" \
        disableall $SmallWinSelecting
}

proc DWUpdateCondsFromListbox { GDN listbox level cond } {
    if { ![winfo exists $listbox] } {
        return
    }
    set line [DWGetSelection $listbox]
    DWUpdateConds $GDN $line $level $cond
}

proc DWEditCondFromListbox { GDN listbox level cond } {
    if { ![winfo exists $listbox] } {
        return
    }
    set line [DWGetSelection $listbox]
    DWEditCond $GDN $line $level $cond
}


proc DWEditCond { GDN line level cond } {
    upvar \#0 $GDN GidData
    set w .gid.edit_condition
    set figure_id [lindex $line 1]

    set condition_name [DWGetGIDCondName $GDN]
    
    #set over [lindex [GiD_Info conditions $condition_name] 0]
    set level $GidData(CND,level)
    set viewmode [GiD_Info Project ViewMode]
    if { $viewmode == "GEOMETRYUSE" } {
        if { $level == "over_point" } {
            set over points
        } elseif { $level == "over_line" } {
            set over lines
        } elseif { $level == "over_surface" } {
            set over surfaces
        } elseif { $level == "over_volume" } {
            set over volumes
        } elseif { $level == "over_layer" } {
            set over layers
        } elseif { $level == "over_group" } {
            set over groups
        } else {
            return
        }
        set id [lindex $line 1]
    } else {
        if { [lindex $line 0] == "N" } {
            set over nodes
            set id [lindex $line 1]
        } elseif { [lindex $line 0] == "E" } {
            #body elements
            set over elements
            set id [lindex $line 1]
        } elseif { [lindex $line 0] == "L" } {
            set over layers
            set id [lindex $line 1]
        } elseif { [lindex $line 0] == "G" } {
            set over groups
            set id [lindex $line 1]
        } else {
            #face elements
            set over face_elements
            set id [lrange $line 0 1]
        }
    }
    
    
    InitWindow2 $w -title [_ "Edit condition %s over %s %s" $condition_name $over $id] \
        -geometryvariable PreEditConditionWindowGeom \
        -initcommand EditCondition \
        -ontop
    ttk::frame $w.f
    grid $w.f -sticky nsew
    grid rowconfigure $w "0" -weight 1
    grid columnconfigure $w "0" -weight 1
    
    DWCreateTabOfTexts $GDN $w.f [DWGetTypeCND $GDN] "Base properties"
    DWUpdateConds $GDN $line $level $cond

    #add lower buttons
    ttk::frame $w.buts -style BottomFrame.TFrame
    ttk::button $w.buts.apply -text [_ "Accept"] -command [list DWEditCondApply $GDN $condition_name $over $id] -style BottomFrame.TButton
    ttk::button $w.buts.close -text [_ "Close"] -command [list destroy $w] -style BottomFrame.TButton
    grid $w.buts.apply $w.buts.close -sticky ew -padx 5 -pady 6
    grid $w.buts -sticky ews -pady {2 0}
    grid anchor $w.buts center

}

proc DWEditCondApply { GDN condition_name over id } {
    if [catch "BuildListValues $GDN" values] {
        if [string equal $values CANCEL] {
            return
        } else {
            error $values
        }
    }    
    set selection [list {*}$id]
    set index_and_values [list]
    set i 0
    foreach value $values {
        lappend index_and_values $i $value
        incr i
    }
    if { [llength $index_and_values] && [llength $selection] } {
        GiD_AccessValueAssignedCondition -field_index set $condition_name $over $index_and_values $selection   
        GidUtils::SetWarnLine [_ "Condition %s changed for %s %s " $condition_name $over $id]
    }
}

proc SignalEntityFromCondListbox { listbox level viewmode } {
    if { ![winfo exists $listbox] } {
        return
    }
    set line [DWGetSelection $listbox]    
    SetUse $viewmode
    if { $viewmode == "GEOMETRYUSE" } {
        if { $level == "over_point" } {
            set over Points
        } elseif { $level == "over_line" } {
            set over Lines
        } elseif { $level == "over_surface" } {
            set over Surfaces
        } elseif { $level == "over_volume" } {
            set over Volumes
        } elseif { $level == "over_layer" } {
            return
        } elseif { $level == "over_group" } {
            return
        } else {            
            return
        }
        set id [lindex $line 1]
    } else {      
        if { [lindex $line 0] == "N" } {
            set over Nodes
            set id [lindex $line 1]
        } elseif { [lindex $line 0] == "E" } {
            #body elements
            set over Elements
            set id [lindex $line 1]
        } elseif { [lindex $line 0] == "L" } {
            #ovlayer
            return
        } elseif { [lindex $line 0] == "G" } {
            #ovgroup
            return
        } else {
            #face elements
            set over Elements
            set id [lindex $line 0]
        }
    }
    if { ![string is integer $id] } {
        return
    }
    GiD_Process MEscape Utilities SignalEntities $over $id
}

proc DWUpdateConds { GDN line level cond } {
    global $GDN
    upvar \#0 $GDN GidData
    
    if { $line == "" } {
        return
    }
    
    # Cambio de level       
    DWSetCNDLevel $GDN $level [lsearch -exact $GidData(CND,$level,GIDNAMES) $cond]
        
    if { [llength [lrange $line 3 end]] == $GidData(CND,$cond,numprop) } {
        set i 1
        foreach item [lrange $line 3 end] {
            if { [info exists GidData(CND,$cond,STATE,$i)] && [info exists GidData(CND,$cond,DEPENDENCIES,$i)] && \
                $GidData(CND,$cond,DEPENDENCIES,$i)!= "" && \
                    ($GidData(CND,$cond,STATE,$i) == "hidden" || $GidData(CND,$cond,STATE,$i) == "disabled" )} {
                #avoid to raise dependencies of hidden or disabled fields
                set infotrace [trace info variable GidData(CND,$cond,VALUE,$i)]
                if { $infotrace != "" } {
                    foreach trace $infotrace {
                        trace remove variable GidData(CND,$cond,VALUE,$i) {*}$trace
                    }
                }
                set GidData(CND,$cond,VALUE,$i) $item
                if { $infotrace != "" } {
                    foreach trace $infotrace {
                        trace add variable GidData(CND,$cond,VALUE,$i) {*}$trace
                    }
                }                
            } else {
                set GidData(CND,$cond,VALUE,$i) $item
            }
            incr i
        }
    }

}

# DWGetSelection --
#
# Arguments:
#
# Results:
#

proc DWGetSelection { lb } {
    set sel [$lb curselection]
    if { [llength $sel] == 0 } {
        if { [$lb size] == 0 } { return "" }
        $lb sel set 0
        set sel [$lb curselection]
    }
    return [$lb get [lindex $sel 0]]
}

proc DWGetBSelection { lb } {
    set sel [$lb curselection]
    if { [llength $sel] == 0 } {
        return ""
    }
    set idx [lindex $sel 0]
    return [list $idx [$lb get $idx]]
}


proc MenuContextualCondEntities { GDN T level cond viewmode x y } {
    set w $T.menucontextualcondentities
    if { [winfo exists $w] } {
        destroy $w
    }
    menu $w

    $w add command -label [_ "Edit"] -command [list DWEditCondFromListbox $GDN $T $level $cond]    
    $w add command -label [_ "Transfer"] -command [list DWUpdateCondsFromListbox $GDN $T $level $cond]
    $w add command -label [_ "Signal"] -command [list SignalEntityFromCondListbox $T $level $viewmode]

    
    set x [expr [winfo rootx $T]+$x+2]
    set y [expr [winfo rooty $T]+$y]
    GiD_PopupMenu $w $x $y
}


# DWShowCondEntitiesWin --
#
# Arguments:
#
# Results:
#
proc DWShowCondEntitiesWin { GDN ThisOrAll } {
    global $GDN
    upvar \#0 $GDN GidData
    
    set wCond $GidData(WINDOW)
    set CndName [DWGetGUICondName $GDN]
    set _CndName [DWGetGIDCondName $GDN]
    set singular $GidData(CND,singular)
    if { $_CndName == ""} {
        WarnWin [_ "Sorry, but first a %s must be selected" $singular] $wCond
        return
    }
    
    set w $wCond.cndwind
    set GeoName PreConditionListEntities$GidData(BOOKORD)\WindowGeom
    
    if { $ThisOrAll != "allcond" } {
        if { $GidData(BOOK) == "Default" } {
            set Title $CndName
        } else {
            set Title "[= [DWUnder2Space $GidData(BOOK)]] : $CndName"
        }
    } else {
        set Title [_ "All %s" $GidData(CND,plural)]
    }
    
    InitWindow2 $w -title $Title -geometryvariable $GeoName \
        -ontop
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0
    
    # MakeAlwaysOnTop $w
    
    wm minsize  $w 1 1
    
    ttk::frame $w.lb   
    ttk::scrollbar $w.lb.yscroll -command [list $w.lb.list yview]        
    ttk::scrollbar $w.lb.xscroll -orient horizontal -command [list $w.lb.list xview]   
    #pseudottk::
    tk::listbox $w.lb.list -yscroll [list $w.lb.yscroll set] \
        -xscroll [list $w.lb.xscroll set] \
        -selectmode single -width 30
    
    pack $w.lb.yscroll -side right -fill y
    pack $w.lb.xscroll -side bottom -fill x
    pack $w.lb.list -expand yes -fill both
    
    ttk::frame $w.but -style BottomFrame.TFrame

    ttk::button $w.but.edit -text [_ "Edit"] -command [list DWEditCondFromListbox $GDN $w.lb.list $GidData(CND,level) $_CndName] -style BottomFrame.TButton
    ttk::button $w.but.transfer -text [_ "Transfer"] -command [list DWUpdateCondsFromListbox $GDN $w.lb.list $GidData(CND,level) $_CndName] -style BottomFrame.TButton
    ttk::menubutton $w.but.back -text "[_ "Send"]..." -menu $w.but.back.m -style BottomFrame.TButton    
    ttk::button $w.but.close -text [_ "Close"] -command [list destroy $w] -style BottomFrame.TButton
    
    bind $w.but.close <Return> [list $w.but.close invoke]
    bind $w <Escape> [list $w.but.close invoke]
    
    set viewmode [GiD_Info Project ViewMode]
    if { $viewmode == "GEOMETRYUSE" } {
        set mode geometry
    } else {
        set mode mesh
    }
    
    set ::GidPriv(LayersAlsoLower) [GiD_Set LayersAlsoLower]
    set ::GidPriv(LayersAlsoHigher) [GiD_Set LayersAlsoHigher]
    
    set current_intv [lindex [GiD_Info intvdata num] 0]
    
    menu $w.but.back.m
    $w.but.back.m add checkbutton -label [_ "Also lower entities"] -variable ::GidPriv(LayersAlsoLower) -command {GiD_Set LayersAlsoLower $::GidPriv(LayersAlsoLower)}
    $w.but.back.m add checkbutton -label [_ "Also higher entities"] -variable ::GidPriv(LayersAlsoHigher) -command {GiD_Set LayersAlsoHigher $::GidPriv(LayersAlsoHigher)}
    $w.but.back.m add command -label [_ "Send others to layer back"] -command [list DWSendEntitiesToBack $w $_CndName $mode $w.lb.list $current_intv tobackopposite]
    $w.but.back.m add command -label [_ "Send to layer back"] -command [list DWSendEntitiesToBack $w $_CndName $mode $w.lb.list $current_intv toback]
    $w.but.back.m add command -label [_ "Bring all to front"] -command [list DWSendEntitiesToBack $w $_CndName $mode $w.lb.list $current_intv alltofront]
    $w.but.back.m add command -label [_ "Help"] -command [list DWSendEntitiesToBack $w $_CndName $mode $w.lb.list $current_intv help]
    
    if { $ThisOrAll != "allcond" } {
        grid $w.but.edit $w.but.transfer $w.but.back  $w.but.close -padx 5 -pady 5
    } else {
        grid $w.but.close -padx 5 -pady 5
    }
    
    grid $w.lb -sticky nsew
    grid $w.but -sticky sew
    grid rowconfigure $w 0 -weight 1
    grid columnconfigure $w 0 -weight 1
    
    grid anchor $w.but center
    
    if { $ThisOrAll != "allcond" } {
        set aa [GiD_Info conditions $_CndName $mode]
        if { [lsearch [GiD_Info conditions over_layer] $_CndName] != -1 } {
            set aa [lsort -dictionary -index 1 $aa]
        } elseif { [lsearch [GiD_Info conditions over_group] $_CndName] != -1 } {
            set aa [lsort -dictionary -index 1 $aa]
        } else {
            #some unusual case can have mixed elements and face elements
            set a0 ""
            set a1 ""
            foreach i $aa {
                if { [string is integer [lindex $i 0]] } {
                    lappend a0 $i
                } else {
                    lappend a1 $i
                }
            }
            set aa [concat [lsort -integer -index 0 $a0] [lsort -integer -index 1 $a1]]
        }
        foreach i $aa { $w.lb.list insert end $i }
        #bind $w.lb.list <Double-1> [list DWUpdateCondsFromListbox $GDN %W $GidData(CND,level) $_CndName]
        bind $w.lb.list <Double-1> [list SignalEntityFromCondListbox %W $GidData(CND,level) $viewmode]
        bind $w.lb.list <Button-$::gid_right_button> "[list MenuContextualCondEntities $GDN %W $GidData(CND,level) $_CndName $viewmode %x %y]; break"
    } else {
        foreach type {point line surface volume layer group} {
            set over over_$type
            set name [GiD_Info conditions $over]
            if { $name != "" } {
                $w.lb.list insert end [concat [_ "Applied"]: $over]
                foreach iname $name {
                    set aa [GiD_Info conditions $iname $mode]
                    if { $over == "over_layer" } {
                        set aa [lsort -dictionary -index 1 $aa]
                    } elseif { $over == "over_group" } {
                        set aa [lsort -dictionary -index 1 $aa]
                    } else {
                        #some unusual case can have mixed elements and face elements
                        set a0 ""
                        set a1 ""
                        foreach i $aa {
                            if { [string is integer [lindex $i 0]] } {
                                lappend a0 $i
                            } else {
                                lappend a1 $i
                            }
                        }
                        set aa [concat [lsort -integer -index 0 $a0] [lsort -integer -index 1 $a1]]
                    }
                    if { $aa != "" } {
                        $w.lb.list insert end [concat [_ "Name"]: $iname]
                        foreach i $aa { $w.lb.list insert end $i }
                    }
                }
            }
        }
    }
    focus -force $w.lb.list
}

proc DWSendEntitiesToBack { w CndName mode list current_intv what } {
    
    if { $what == "help" } {
        set text [_ "These commands send all entities that have the same condition with the\
                same data than the selected entities to layers back or the opposite.\n\
                When an entity is in the layer back, it cannot be viewed neither selected.\n\
                To send entities again to front, use: Bring All to front"]
        WarnWin $text
        return
    }
    if { $what == "alltofront" } {
        GidUtils::WaitState        
        GiD_Process 'Layers BringToFrontAll escape        
        GidUtils::EndWaitState
        return
    }
    set selvalues ""
    foreach i [$list curselection] {
        lappend selvalues [lrange [$list get $i] 3 end]
    }
    if { $selvalues == "" } {
        WarnWin [_ "Error: it is necessary to select one or more entities"]
        return
    }
    
    GidUtils::DisableGraphics
    GidUtils::WaitState $w
    
    foreach "current_intv_new -" [GiD_Info intvdata num] break
    if { $current_intv != $current_intv_new } {
        GiD_Process Mescape Data Intervals ChangeInterval $current_intv escape
    }
    
    set isnode 0
    set entities ""
    foreach i [GiD_Info conditions $CndName $mode] {
        set val [lrange $i 3 end]
        #use exact because if val has some [] lsearch not found it !!
        if { [lsearch -exact $selvalues $val] != -1 } {
            if { [lindex $i 0] == "N" } {
                set isnode 1
                lappend entities [lindex $i 1]
            } elseif { [lindex $i 0] == "E" } {
                set isnode 0
                lappend entities [lindex $i 1]
            } elseif { [lindex $i 0] == "L" } {
                lappend entities [lindex $i 1]
            } else {
                set isnode 0
                lappend entities [lindex $i 0]
            }
        }
    }
    
    if { $current_intv != $current_intv_new } {
        GiD_Process Mescape Data Intervals ChangeInterval $current_intv_new escape
    }
    
    if { $entities == "" } {
        GidUtils::EnableGraphics
        GidUtils::EndWaitState $w
        WarnWin [_ "No entity sent"]
        return
    }
    
    set comm ""
    switch $what {
        toback {
            lappend comm Layers SendToBack
        }
        tobackopposite {
            lappend comm Layers SendToBackOpposite
        }
    }
    if { [GiD_Set LayersAlsoLower] } {
        lappend comm LowerEntities
    }
    if { [GiD_Set LayersAlsoHigher] } {
        lappend comm HigherEntities
    }
    set ovwhat [lindex [GiD_Info conditions $CndName] 0]
    switch $mode {
        geometry {
            switch $ovwhat {
                ovpnt {
                    lappend comm Points
                }
                ovline {
                    lappend comm Lines
                }
                ovsurf {
                    lappend comm Surfaces
                }
                ovvol {
                    lappend comm Volumes
                }
                ovlayer {
                    lappend comm All
                }
                ovgroup {
                    lappend comm All
                }
            }
        }
        mesh {
            if { $ovwhat == "ovlayer" } {
                lappend comm All
            } elseif { $ovwhat == "ovgroup" } {
                lappend comm All
            } elseif { $isnode } {
                lappend comm Nodes
            } else {
                lappend comm Elements
            }
        }
    }
    GiD_Process {*}$comm
    if { $ovwhat == "ovlayer" } {
        foreach i $entities {
            GiD_Process Layer:$i
        }
    } elseif { $ovwhat == "ovgroup" } {
        foreach i $entities {
            GiD_Process Group:$i
        }
    } else {
        while 1 {
            GiD_Process {*}[lrange $entities 0 100]
            set entities [lrange $entities 101 end]
            if { $entities == "" } { break }
        }       
    }
    GiD_Process escape
    GidUtils::EnableGraphics
    GidUtils::EndWaitState $w
    GiD_Redraw
}

# ------------------------------------------------------------------------------
# DWCreateConditionFrame --
#
#  Fill in frame of GDN the info corresponding to current condition.
# ------------------------------------------------------------------------------

proc DWCreateConditionFrame { GDN } {
    upvar \#0 $GDN GidData
    
    DWCreateTabOfTexts $GDN $GidData(CND,mainframe) [DWGetTypeCND $GDN] "Base properties"
}

# ------------------------------------------------------------------------------
# DWOnSelectCondition --
#
#  Invoked under -modifycmd of conditions ComboBox. It should change the condition
#  info showed in the data window.
# ------------------------------------------------------------------------------

proc DWOnSelectCondition { GDN {sel -1}} {
    upvar \#0 $GDN GidData
    
    if { ![ info exists GidData(CND,cursel)]} {
        return
    }
    # WarnWin "$sel -->  $GidData(CND,cursel)==[$GidData(CND,combo) get] "
    if {$sel==-1} {
        if {$GidData(CND,cursel)==[set sel [$GidData(CND,combo) get]]} {
            return
        }
    } else {
        set sel [$GidData(CND,combo) get]
    }
    set GidData(menudraw,needpost) 1
    set GidData(CND,cursel) $sel
    
    # BACKWARD COMPATIBILITY
    set GidData(CND,condname) [DWGetGUICondName $GDN]
    # REMOVE AFTER VERSION 8.0
    # WarnWin "CreatingConditionFrame - $GidData(CND,condname) -"
    DWCreateConditionFrame $GDN
    focus $GidData(CND,mainframe)

    SetStateConditionGroupsAllowedTypes $GDN
}

proc DWInitWindow { GDclass Book BookOrd UseWindow wvar } {
    global gidDataLabels
    
    upvar $wvar w
    
    array set DataList {
        CND {"Conditions"    .gid.conditions PreConditions  conditions}
        MAT {"Materials"     .gid.materials  PreMaterials   materials}
        INT {"Interval data" .gid.intvdata   PreIntervals   intvdata}
        PD  {"Problem data"  .gid.prbdata    PreProblemData gendata}
    }
    # trick to found this strings from ramtranslator, don't delete next commentary
    # [_ "Conditions"] [_ "Materials"] [_ "Interval data"] [_ "Problem data"]

    if {![info exists ::GidData(open,$GDclass,book,$Book)] &&
        [lsearch $::gidDataBooksHidden([lindex $DataList($GDclass) 3]) $Book] != -1} {
        return "hidden"
    }
    
    set w [ExistsDataWindow $GDclass $BookOrd]
    if { $w != "" } {
        if { [wm state $w] != "normal" } {
            wm deiconify $w
        }
        raise $w
        focus $w
        return raised
    }
    set wdata $DataList($GDclass)
    
    if { $Book == "Default" } {
        set Title [lindex $wdata 0]
        if { [info exists gidDataLabels($Title)] && $gidDataLabels($Title) != "" } {
            set Title $gidDataLabels($Title)
        }
        set Title [_ $Title]
    } else {
        set Title [= [DWUnder2Space $Book]]
    }
    if { $UseWindow == "" } {
        set GeoName "[lindex $wdata 2]${BookOrd}WindowGeom"
        set w [DWGetWindowName [lindex $wdata 1] $BookOrd]
               
        set ::do_not_geo 0
        if {[info exists ::do_not_geo] && $::do_not_geo} {
            array unset ::GidPriv $GeoName
        }
        InitWindow2 $w -title $Title -geometryvariable $GeoName -ontop
        if { ![winfo exists $w] } {
            return close ;# windows disabled || UseMoreWindows == 0
        }       
        wm state $w withdrawn
        set ::geo [list [wm geometry $w] [winfo reqheight $w] [winfo reqheight $w]]
        wm deiconify $w
        # MakeAlwaysOnTop $w
        wm deiconify $w
    } else {
        set w $UseWindow
        wm title $w $Title
    }
    return created
}

proc DWCreateGroupBySelection { GDN cb } {
    upvar #0 $GDN GidData
    set _CndName [DWGetGIDCondName $GDN]

    if [catch "BuildStringValues $GDN" Values] {
        if [string equal $Values CANCEL] {
            return
        } else {
            error $Values
        }
    }

    set allowed_types [GiD_Info conditions $_CndName GroupAllow]
       
    #set over lines ;#esto debe ser alguno seleccionado de entre allowed_types para el modo actual geometria o malla, o AllTypes

    set over $::SelectedConditionGroupEntityType
    if { $over == "" } {
        return
    }
   
    set group_name [string trim [$cb get]]
    if { $group_name != "" && [GiD_Groups exists $group_name] } {
        ::GidUtils::SetWarnLine [_ "Group '%s' already exists, use an automatic name" $group_name]
        set group_name ""
    }
    if { $group_name == "" } {
        set num 1        
        set group_name $_CndName-auto-$num
        while { [GiD_Groups exists $group_name] } {
            incr num
            set group_name $_CndName-auto-$num
        }        
    }
     

    if { 0 } {
        if { $over == "alltypes" } {
            set type AllTypes
        } else {            
            #for GidUtils::PickEntities can be "Points" "Lines" "Surfaces" "Volumes" "Dimensions" "AllTypes" "Nodes" or "Elements"    
            set type [string totitle $over] ;#first letter uppercase
        }
        set selection [GidUtils::PickEntities $type multiple]
        if { $selection != "" } {        
            GiD_Groups create $group_name
            GiD_Groups edit allowed_types $group_name $allowed_types        
            GiD_EntitiesGroups assign $group_name $over [GidUtils::UnCompactNumberList $selection]        
            GiD_AssignData condition $_CndName groups $Values $group_name            
            Groups::FillInfo
        }
    } else {
        if { $over == "alltypes" } {
            set viewmode [GiD_Info Project ViewMode]
            if { $viewmode == "GEOMETRYUSE" } {        
                set over all_geom_types
            } else {
                set over all_mesh_types
            }
        }
        GiD_Process 'Groups Create $group_name escape
        GiD_Process 'Groups Edit Allowed_types $group_name {*}$allowed_types escape
        #GiD_Process MEscape Data Condition AssignCond $_CndName Change {*}$Values $group_name escape
        GiD_Process MEscape Utilities EntitiesGroups Assign $group_name $over        
        delayedop changefunc [list Callback_EndEntitiesGroups $group_name $_CndName $Values]
    }    
}

proc Callback_EndEntitiesGroups { group_name cond_name cond_values } {       
    if { [GiD_Groups get num_entities $group_name ]} {
        #WarnWinText "must assign the condition $cond_name"
        GiD_Process MEscape Data Condition AssignCond $cond_name Change {*}$cond_values $group_name escape
    } else {
        #WarnWinText "must delete empty group"
        GiD_Process 'Groups Delete $group_name escape
    }    
    Groups::FillInfo
}

proc PTClassic_GetConditionDisallowedTypesGeometry { condition_name } {
    set allowed_types [GiD_Info conditions $condition_name GroupAllow]   
    set forbidden_types [list]
    foreach i {points lines surfaces volumes} {
        if { [lsearch $allowed_types $i] == -1 } {
            lappend forbidden_types $i
        }
    }
    return $forbidden_types   
}

proc PTClassic_GetConditionDisallowedTypesMesh { condition_name } {
    set allowed_types [GiD_Info conditions $condition_name GroupAllow]
    set forbidden_types [list]
    #when meshing always the nodes are added to the parent group                
    #then not reject if nodes is not on the list of allowed_types of the condition
    foreach i {elements faces} {
        if { [lsearch $allowed_types $i] == -1 } {
            lappend forbidden_types $i
        }
    }
    return $forbidden_types    
}

proc PTClassic_GetConditionDisallowedTypes { condition_name } {
    set condition_disallowed_types_geometry [PTClassic_GetConditionDisallowedTypesGeometry $condition_name]
    set condition_disallowed_types_mesh [PTClassic_GetConditionDisallowedTypesMesh $condition_name]
    return [concat $condition_disallowed_types_geometry $condition_disallowed_types_mesh]
}

#invoked from other procs like Groups::FillInfo or GiD_AfterCreateGroup it is not possible to provide GDN !!
proc DWSetCurrentGroupsToAssign { {GDN ""} } {
    set condition_disallowed_element_types [list] ;#classical conditions dones't forbid element types by now...
    if { $GDN != "" } {
        upvar #0 $GDN GidData
        set AssignGroup $GidData(CND,ASSIGN-GROUP)
        set condition_name [DWGetGIDCondName $GDN]                
        set condition_disallowed_types [PTClassic_GetConditionDisallowedTypes $condition_name]        
    } else {
        set AssignGroup .gid.conditions1.but.assign_group ;#this object name is the ttk:combobox of DWCreateGroupBySelection   
        set condition_disallowed_types [list]
    }
    if { ![GidUtils::IsTkDisabled] && [winfo exists $AssignGroup.cb0] } {
        set groups [Groups::ListGroups normal $condition_disallowed_types $condition_disallowed_element_types] ;#hidden groups are not listed        
        $AssignGroup.cb0 configure -values [concat $groups {{}}]
        $AssignGroup.cb0 set ""     
        $AssignGroup.b0 configure -state disabled
    }
}

proc DWAssignCondOverGroup { GDN cb } {
    upvar #0 $GDN GidData
    set _CndName [DWGetGIDCondName $GDN]
    set group [$cb get]
    if { $group == "" } return
    if { [GiD_Groups exists $group] } {
        DWAssignCond $GDN $group        
    } else {
        WarnWinText [_ "Group %s not exists" $group]
    }
}

proc OnConditionGroupSelected { cb b } {
    set group [$cb get]
    if { $group == "" || ![GiD_Groups exists $group] } {
        set state disabled
    } else {
        set state normal
    }
    $b configure -state $state   
}

#to update the list, in case of groups are created or deleted for example in the groups window
proc OnConditionGroupPost { GDN } {
    DWSetCurrentGroupsToAssign $GDN
}

#invoked automatically from DWOnSelectCondition
proc SetStateConditionGroupsAllowedTypes { GDN } {
    upvar #0 $GDN GidData

    if { ![info exists GidData(CND,level)] } {
        return
    }
    if { [string equal $GidData(CND,level) "over_group"] } {

        set AssignGroup $GidData(CND,ASSIGN-GROUP)
        if { ![winfo exists $AssignGroup] } {
            return
        }

        if { ![winfo exists $AssignGroup.cb0] } {
            ttk::combobox $AssignGroup.cb0 -width 12 -postcommand [list OnConditionGroupPost $GDN]
            bind $AssignGroup.cb0 <<ComboboxSelected>> [list OnConditionGroupSelected %W $AssignGroup.b0]

            ttk::button $AssignGroup.b0 -image [gid_themes::GetImage ok.png small_icons] \
                -command [list DWAssignCondOverGroup $GDN $AssignGroup.cb0] -underline 0
            #-text [_ "Assign"] -compound left 
            
            ttk::frame $AssignGroup.f0
            foreach img {point line surface volume alltypes} value {points lines surfaces volumes alltypes} {
                ttk::radiobutton $AssignGroup.f0.rb$value -image [gid_themes::GetImage $img.png small_icons] \
                    -variable ::SelectedConditionGroupEntityType -value $value -style Toolbutton
            }            
            foreach img {node element face} value {nodes elements faces} {
                ttk::radiobutton $AssignGroup.f0.rb$value -image [gid_themes::GetImage $img.png small_icons] \
                    -variable ::SelectedConditionGroupEntityType -value $value -style Toolbutton
            }
            
            ttk::button $AssignGroup.b1 -image [gid_themes::GetImage group_assign.png small_icons] \
                -command [list DWCreateGroupBySelection $GDN $AssignGroup.cb0] -underline 0
            #-text [_ "Select"] -compound left
            GidHelp $AssignGroup.b0 [_ "Assign the condition to the group"]
            GidHelp $AssignGroup.b1 [_ "Create automatically a new group with the selected entities and assign to it the condition"]
            grid $AssignGroup.cb0 $AssignGroup.b0 -sticky w
            grid configure $AssignGroup.cb0 -sticky ew
            grid columnconfigure $AssignGroup 0 -weight 1
            grid $AssignGroup.f0 $AssignGroup.b1 -sticky w
        }
        
        set condition_name [DWGetGIDCondName $GDN]
        set allowed_types [GiD_Info conditions $condition_name GroupAllow]        
                
        set allowed_types_current_viewmode [list]
        set viewmode [GiD_Info Project ViewMode]
        set all_geometry_allowed 1
        if { $viewmode == "GEOMETRYUSE" } {            
            foreach i {points lines surfaces volumes} {
                if { [lsearch $allowed_types $i] != -1 } {
                    lappend allowed_types_current_viewmode $i           
                } else {
                    set all_geometry_allowed 0
                }             
            }            
        } else {                       
            foreach i {nodes elements } {
                if { [lsearch $allowed_types $i] != -1 } {
                    lappend allowed_types_current_viewmode $i
                } else {
                    set all_geometry_allowed 0
                }        
            }
            foreach i { faces } {
                if { [lsearch $allowed_types $i] != -1 } {
                    lappend allowed_types_current_viewmode $i
                }
            }
        }
        set i alltypes
        if { $all_geometry_allowed  } {
            lappend allowed_types_current_viewmode $i
        }     
        #ungrid all previous buttons if any
        foreach i  [concat {points lines surfaces volumes alltypes} {nodes elements faces}] {
            grid forget $AssignGroup.f0.rb$i           
        }    
        set column 0
        foreach i $allowed_types_current_viewmode {
            grid $AssignGroup.f0.rb$i -row 0 -column $column -sticky w
            incr column
        }        

        if { ![info exists ::SelectedConditionGroupEntityType] || [lsearch $allowed_types_current_viewmode $::SelectedConditionGroupEntityType] == -1 } {
            set ::SelectedConditionGroupEntityType [lindex $allowed_types_current_viewmode 0]
        }                    
        DWSetCurrentGroupsToAssign $GDN
    }    
}

proc DWCondWindow { {Book "Default"} {BookOrd 1} {UseWindow ""} } {
    global GidPriv
    global gidDataLabels
    if { $Book == "Default" } {
        set books_and_indices [GiD_Info conditions books]
        if { [llength $books_and_indices] } {
            set Book [lindex $books_and_indices 0]
        }
    }
    
    set GDN GD_CND_$BookOrd
    if { [DWInitWindow CND $Book $BookOrd $UseWindow w] != "created" } {
        return $GDN
    }
    global $GDN
    globalarray $GDN
    upvar #0 $GDN GidData
    
    set GidData(CALLWIN) DWCondWindow
    set GidData(BOOKORD) $BookOrd
    set GidData(WINDOW) $w
    
    # manipulador que atendera la validacion de los datos
    wm protocol $w WM_DELETE_WINDOW "DWCloseWindow $GDN"
    
    if { ![CNDFillGidData $GDN $Book] } {
        destroy $w
        return
    }
    
    set singular $GidData(CND,singular)
    set Singular $GidData(CND,singular)
    set plural   $GidData(CND,plural)
    set Plural   $GidData(CND,plural)
    
    ttk::frame $w.level
    set FirstLevel ""
    foreach type {point line surface volume layer group} image {point line surface volume layers groups} {
        set over over_$type
        if { [lsearch $GidData(levels) $over]!=-1 } {
            if { $FirstLevel == "" } {
                set FirstLevel $over
            }
            ttk::radiobutton $w.level.$type -image [gid_themes::GetImage ${image}.png small_icons] \
                -variable $GDN\(CND,varlevel\) -value $over -style Toolbutton \
                -command [list DWOnSelectCNDLevel $GDN]
            pack $w.level.$type -side left
            GidHelp $w.level.$type [_ "Selecting points, lines, surfaces, volumes, layers or groups only %s belonging to that level are displayed" $plural]
        }
    }
    
    set GidPriv(ShowIntervalInCond) 0
    #kike: para mostrar el intervalo actual en la ventana de condiciones
    #problema, solo se refresca el valor al abrir la ventana.
    #No se actualiza la etiqueta al cambiar en C el valor de project->IntvToUse
    #LinkIntVariableTCL("GidPriv(CurrentInterval)",(char*)&project->IntvToUse);
    #a pesar de que si cambia GidPriv(CurrentInterval)
    #para eso puedo usar set GidPriv(CurrentInterval) [DWGetActiveInterval]
    if { $GidPriv(ShowIntervalInCond)==1 } {
        #if { ![info exists GidPriv(CurrentInterval)] } {
            #  set GidPriv(CurrentInterval) [DWGetActiveInterval]
            #}
        set GidPriv(CurrentInterval) [DWGetActiveInterval]
        label $w.level.l1 -text [_ "Current Interval"]:
        label $w.level.l2 -textvariable GidPriv(CurrentInterval)
        pack  $w.level.l2 $w.level.l1 -side right
    }
    
    ttk::frame $w.select
    
    TTKComboBox $w.select.combo -state readonly -modifycmd "DWOnSelectCondition $GDN"
    set GidData(CND,combo) $w.select.combo
    
    CreateJumpButton $GDN $w.select 2
    
    grid $w.select.combo -row 0 -column 0 -sticky "snew"
    #grid $jumpbutton -row 0 -column 2 -sticky "snew"
    grid columnconfigure $w.select 0 -weight 3
    grid columnconfigure  $w.select 1 -weight 1
    
    ttk::frame $w.f -style groove.TFrame
    set GidData(CND,mainframe) $w.f
    
    ttk::frame $w.but -style ridge.TFrame -borderwidth 2 
    #maybe not -style BottomFrame.TFrame
    set AssignEntity $w.but.assign
    ttk::button $AssignEntity -text [_ "Assign"] \
        -command [list DWAssignCond $GDN] -underline 0
        #-style Window.TButton
    set AssignLayer $w.but.assign_layer
    set MenuLayer $AssignLayer.menu
    ttk::menubutton $AssignLayer -text [_ "Assign"] -underline 0 \
        -menu $MenuLayer 
        #-style Horizontal.IconButtonWithMenu
    menu $MenuLayer -borderwidth 1 -activeborderwidth 1 \
        -postcommand [list DWPostLayers $GDN $MenuLayer DWAssignCond 0]
    
    set AssignGroup $w.but.assign_group

    if { 0 } {
        set MenuGroup $AssignGroup.menu
        ttk::menubutton $AssignGroup -text [_ "Assign"] -underline 0 \
            -menu $MenuGroup 
            #-style Horizontal.IconButtonWithMenu
        menu $MenuGroup -borderwidth 1 -activeborderwidth 1 \
            -postcommand [list DWPostGroups $GDN $MenuGroup DWAssignCond 0]
    } else {
        ttk::frame $AssignGroup                
    }
    set GidData(CND,ASSIGN-ENTITY) $AssignEntity
    set GidData(CND,ASSIGN-LAYER) $AssignLayer
    set GidData(CND,ASSIGN-GROUP) $AssignGroup
    
    # ----- Opcion ENTITIES -----
    set MenuEntities $w.but.ent.menu
    ttk::menubutton $w.but.ent -text [_ "Entities"] -underline 0 \
        -menu $MenuEntities 
        #-style Horizontal.IconButtonWithMenu
    menu $MenuEntities -borderwidth 1 -activeborderwidth 1 \
        -postcommand [list DWPostEntitiesCnd $GDN $MenuEntities]
    $MenuEntities add command -label  "XXXX" \
        -command [list DWEntitiesCond $GDN thiscond]
    $MenuEntities add command -label [_ "All conditions"] \
        -command [list DWEntitiesCond $GDN allcond]
    set num_intervals [lindex [GiD_Info intvdata num] 1]
    if { $num_intervals > 1 } {
        $MenuEntities add command -label [concat "XXXX" [_ "all intervals"]] \
            -command [list DWEntityCondAlongIntervals $GDN thiscond]
        $MenuEntities add command -label [_ "All conditions all intervals"] \
            -command [list DWEntityCondAlongIntervals $GDN allcond]
    }
    # ----- Opcion DRAW -----
    
    set MenuDraw $w.but.draw.menu
    ttk::menubutton $w.but.draw -text [_ "Draw"] -underline 0 \
        -menu $MenuDraw 
        #-style Horizontal.IconButtonWithMenu
    menu $MenuDraw -borderwidth 1 -activeborderwidth 1 \
        -postcommand [list DWPostDrawCond $GDN $MenuDraw]
    
    # ----- Opcion Unassign -----
    
    set MenuUnassign $w.but.unassign.menu
    ttk::menubutton $w.but.unassign  -text [_ "Unassign"] -underline 0 \
        -menu $MenuUnassign 
        #-style Horizontal.IconButtonWithMenu
    
    menu $MenuUnassign -borderwidth 1 -activeborderwidth 1 \
        -postcommand [list DWPostUnassign $GDN $MenuUnassign]
    
    grid $w.but.assign $w.but.ent $w.but.draw $w.but.unassign -sticky "ew" -padx 2
    grid configure $w.but.assign -sticky "nsew";#because now the ttk::menubutton heigh is bigger that button height
    #grid columnconfigure $w.but 4 -weight 1
    grid columnconfigure $w.but "0 1 2 3" -weight 1    
    
    ttk::frame $w.but1 -style BottomFrame.TFrame
    ttk::button $w.but1.close -text [_ "Close"] \
        -command [list DWCloseWindow $GDN] -underline 0 \
        -style BottomFrame.TButton    

    pack $w.but1.close -pady 5
    
    grid $w.level -row 0 -column 0 -sticky snew
    grid $w.f -row 2 -column 0 -sticky snew
    grid $w.select -row 1 -column 0 -sticky snew
    grid $w.but -row 3 -column 0 -sticky we
    
    grid $w.but1 -row 4 -column 0 -sticky snew
    grid anchor $w.but center
    
    grid rowconfigure $w 2 -weight 1
    
    grid columnconfigure $w 0 -weight 1
    
    set GidData(CND,level) ""
    DWSetCNDLevel $GDN $FirstLevel
    
    foreach i "$w.but.assign $w.but.ent $w.but.draw $w.but.unassign $w.but1.close" {
        set l [string tolower [string index [$i cget -text] [$i cget -underline]]]
        # this may fail when $l is not a valid keysym
        catch {
            bind $w <Alt-$l> "$i invoke;break"
        }
    }
    bind $w <Escape> [list $w.but1.close invoke]
        
    DWMapWindow $w
    focus $w.but.assign
    
    return $GDN
}

proc DWMapWindow { w } {
    if {[wm state $w] eq "normal"} {
        return
    }
    update
    set ::hasg $::GidPriv($w,has,geom)
    if {!$::GidPriv($w,has,geom)} {
        set rh  [winfo reqheight $w]
        set rw  [winfo reqheight $w]
        set geo [wm geometry $w]
        set sw  [winfo screenwidth $w]
        set sh  [winfo screenheight $w]
        
        #     set nw [expr {round($rw*1.1)}]
        #     if {$nw < $sw} {
            #       set rw $nw
            #     }
        #     set nh [expr {round($rh*1.1)}]
        #     if {$nh < $sh} {
            #       set rh $nh
            #     }
        if {$::GidData($w,has,notebook)} {
            incr rw [expr {round([winfo reqwidth  $::GidData($w,SF)]*1.1)}]
            incr rh [expr {round([winfo reqheight $::GidData($w,SF)]*1.1)}]
        }
        
        if {$rw < 200} {
            set rw 200
        }
        if {$rh < 200} {
            set rh 200
        }
        set x [expr {($sw-$rw)/2}]
        set y [expr {($sh-$rh)/2}]
        set ::ngeo "${rw}x${rh}+${x}+${y}"
        wm geometry $w "${rw}x${rh}+${x}+${y}"
    }
    update idletask
    wm state $w normal
}

proc DWPostUnassign { GDN m } {
    upvar #0 $GDN GidData
    
    set cnd [DWGetGIDCondName $GDN]
    set Singular $GidData(CND,singular)
    set Plural   $GidData(CND,plural)
    set Book     $GidData(BOOK)
    
    $m delete 0 end
    set label [= [DWUnder2Space $cnd]]
    if { [string equal $GidData(CND,level) "ovlayer"] } {
        $m add cascade -label [_ "%s Layer" $cnd ] -menu $m.mlayer
        if ![winfo exists $m.mlayer] {
            menu $m.mlayer -borderwidth 1 -activeborderwidth 1
        }
        DWPostLayers $GDN $m.mlayer DWUnassignSome 1
        if ![llength [GiD_Layers list]] {
            $m.mlayer entryconfigure end -state disabled
        }
    } elseif { [string equal $GidData(CND,level) "ovgroup"] } {
        $m add cascade -label [_ "%s Group" $cnd ] -menu $m.mgroup
        if ![winfo exists $m.mgroup] {
            menu $m.mgroup -borderwidth 1 -activeborderwidth 1
        }
        DWPostGroups $GDN $m.mgroup DWUnassignSome 1
        if { ![llength [ normal "" ""]] } {
            $m.mgroup entryconfigure end -state disabled
        }
    } else {
        $m add command -label [_ "%s Entities" $label] \
            -command [list DWUnassignSome $GDN]
        
        if { 0 } {
            $m add cascade -label [_ "%s Entities where" $label] -menu $m.mfield
            if ![winfo exists $m.mfield] {
                menu $m.mfield -borderwidth 1 -activeborderwidth 1 \
                    -postcommand [list DWPostUnassignSomeWhere $GDN $m.mfield]
            }
        }
    }
    $m add command -label [_ "All %s" $label] -command [list DWUnassignAllThis $GDN]
    if { $Book != "Default" } {
        set label [= [DWUnder2Space $Book]]
        $m add command -label [_ "All %s" $label] -command [list DWUnassignAllBookCND $GDN]
    }
    $m add command -label [_ "All conditions"] -command [list DWUnassignAllCND $GDN]
}

proc DWPostUnassignSomeWhere { GDN m } {
    upvar #0 $GDN GidData
    set cnd [DWGetGIDCondName $GDN]
    $m delete 0 end
    set otherfields [lrange [GiD_Info conditions $cnd otherfields] 1 end]
    set pos 0
    foreach {Question DefValue} [lrange [GiD_Info conditions $cnd] 2 end] {
        incr pos
        set hiddenfield 0
        foreach {ftype fval fpos} $otherfields {
            if { $fpos == $pos && $ftype == "STATE" && $fval == "HIDDEN" } {
                #don't show hidden fields
                set hiddenfield 1
                break
            }
        }
        if { $hiddenfield } {
            continue
        }
        
        if { [regexp -nocase {([^#]+)(?:#(MAT|CB|LA|FUNC|UNITS)#(.*))?$} $Question {} field type args] } {
            if { $type == {CB} || $type == {LA} || $type == {MAT} } {
                set values [split [string range $args 1 end-1] ,]
                if { $type == {MAT} } {
                    set values [GiD_Info Materials]
                    set book ""
                    if { $args != "" } {
                        set bookname [string range $args 1 end-1]
                        foreach {ibook i} [GiD_Info materials books] {
                            if { $bookname == $ibook } {
                                set book $bookname
                                break
                            }
                        }
                    }
                    if { $book != "" } {
                        #add only materials of this book
                        set newvalues ""
                        foreach mati $values {
                            if { $book == [GiD_Info materials $mati book] } {
                                lappend newvalues $mati
                            }
                        }
                        set values $newvalues
                    }
                } elseif { $type == {LA} } {
                    set values [concat $values [GiD_Info localaxes]]
                }
                set label [= [DWUnder2Space $field]]
                $m add cascade -label $label -menu $m.mvalues$pos
                if { ![winfo exists $m.mvalues$pos] } {
                    menu $m.mvalues$pos -borderwidth 1 -activeborderwidth 1
                }
                $m.mvalues$pos delete 0 end
                foreach i $values {
                    set label [= [DWUnder2Space $i]]
                    $m.mvalues$pos add command -label $label \
                        -command [list DWUnassignSomeWhere $GDN $cnd $field $i]
                }
            } else {
                set label [= [DWUnder2Space $field]]
                $m add command -label $label -command [list DWUnassignSomeWhere $GDN $cnd $field]
            }
        } else {
            set field $Question
            set label [= [DWUnder2Space $field]]
            $m add command -label $label -command [list DWUnassignSomeWhere $GDN $cnd $field]
        }
    }
}

proc DWUnassignSomeWhere { GDN cnd field {i ""}} {
    global $GDN
    upvar #0 $GDN GidData
    set w $GidData(WINDOW)
    if { $i != "" } {
        GiD_Process MEscape Data Conditions AssignCond $cnd UnAssign Field $field $i
    } else {
        set retval [MessageBoxEntry [_ "Enter value window"] [_ "Enter field %s value" $field] any "" 0 question.png]
        if { $retval != "" } {
            GiD_Process MEscape Data Conditions AssignCond $cnd UnAssign Field $field $retval
            set SmallWinSelecting [GiD_Set SmallWinSelecting]
            FinishButton $w $w.but.assign [_ "Press 'Finish' to end selection"] [list DWAfterFinishButton $GDN] disableall $SmallWinSelecting
        }
    }
}

proc DWPostLayers { GDN m cmd {chk 0} } {
    $m delete 0 end
    if { $chk } {
        #for unassign: to show only layers with conditions
        upvar \#0 $GDN GidData
        set cnd [DWGetGIDCondName $GDN]
        set aa {}
        foreach e [GiD_Info conditions $cnd geometry] {
            lappend aa [lindex $e 1]
        }
        foreach l [lsort -unique $aa] {
            $m add command -label $l -command [list $cmd $GDN $l]
        }
    } else {
        foreach l [lsort -unique -dictionary [GiD_Layers list]] {
            $m add command -label $l -command [list $cmd $GDN $l]
        }
    }
}

proc DWPostGroups { GDN m cmd {chk 0} } {
    $m delete 0 end
    if { $chk } {
        #for unassign: to show only groups with conditions
        upvar #0 $GDN GidData
        set cnd [DWGetGIDCondName $GDN]
        set aa {}
        foreach e [GiD_Info conditions $cnd geometry] {
            lappend aa [lindex $e 1]
        }
        foreach l [lsort -unique $aa] {
            $m add command -label $l -command [list $cmd $GDN $l]
        }
    } else {
        foreach l [Groups::ListGroups normal "" ""] {
            $m add command -label $l -command [list $cmd $GDN $l]
        }
    }
}


proc DWSendMessage { GDN TYPE itk } {
}

proc DWCheckClose { GDN } {
    upvar #0 $GDN GidData
    
    set result ""
    if [catch "CheckDirtyState $GDN" result] {
        if [string equal $result CANCEL] {
            error CANCEL
        }
        # error in tcl script, show it & continue closing
        W $result
    }
    
    set TYPE [$GidData(TYPEFUNC) $GDN]
    if { $GidData($TYPE,SYNC,TKWIDGET) } {
        foreach iproc [array names GidData $TYPE,TKWIDGET*] {
            set itk 0
            regexp (?:.)TKWIDGET,(.+) $iproc {} itk
            if { $itk } {
                set QUESTION $GidData($TYPE,QUESTION,$itk)
            } else {
                set QUESTION 0
            }
            if [catch [list {*}$GidData($iproc) CLOSE $GDN $TYPE $QUESTION] result] {
                # follow up the error
                error [_ "Error in tcl script sending event CLOSE to %s: %s" $GidData($iproc) $result]
            } else {
                set len_res [llength $result]
                if { $len_res > 1 } {
                    if { $len_res != 2 } {
                        W [_ "Error '%s' in result from CLOSE event: must be a 2-element list" $result]
                    } else {
                        set status [string toupper [lindex $result 0]]
                        set isERROR [string equal $status ERROR]
                        if { $isERROR || [string equal $status WARNING] } {
                            set msg [lindex $result 1]
                            if [string length $msg] {
                                W $msg
                            }
                            if $isERROR {
                                error CANCEL
                            }
                        } else {
                            W [concat [_ "Invalid status '%s' in CLOSE event: must be" $status] \
                                    "ERROR or WARNING"]
                        }
                    }
                }
            }
        }
    }
}

proc DWCloseWindow { GDN {after_close ""} } {
    upvar #0 $GDN GidData
    
    if [catch "DWCheckClose $GDN" result] {
        if [string equal $result CANCEL] {
            return
        }
        if {$after_close ne ""} {
            eval $after_close
        }
        destroy $GidData(WINDOW)
        unset GidData
        error $result
    }
    if {$after_close ne ""} {
        eval $after_close
    }
    destroy $GidData(WINDOW)
    unset GidData
}

# DWCreateGrid --
#
#        From the string defining a grid #N# N g1 g2 ... gN,
#        Creates the structure with pattern GidData($HEAD,VALUE,i,j) that get
#        the values in gk.
# Arguments:
#
#        HEAD --> header of the structure.
#                 e.g. - CND,Point-Constraint
#                      - MAT,Steel
#       iField --> index of the associated field
#       StrValue --> string of values
#
# Results:
#

proc DWCreateGrid { GDN HEAD iField StrValues } {
    global $GDN
    upvar #0 $GDN GidData
    
    set j 0
    
    catch {
        foreach i [array names GidData($HEAD,VALUE,$iField,*)] {
            unset GidData($i)
        }
    }
    foreach GridValue $StrValues {
        set GidData($HEAD,VALUE,$iField,$j) $GridValue
        incr j 1
    }
    incr j -1
    if { $GidData($HEAD,VALUE,$iField,0) != $j } {
        W [concat [_ "Incorrect number of values for GRID field"] $GidData($HEAD,QUESTION,$iField)]
        return 0
    }
    return 1
}

proc VerifyProblemTypeDir { GDN } {
    upvar #0 $GDN GidData
    
    if { [set GidData(DirPT) [GetDirectoryProblemType]] == "" } {
        return 0
    }
    
    return 1
}

proc DWMakeSingular { str } {
    set str [string trimright $str]
    set l [string length $str]
    if { $l > 1 } {
        set last [string index $str [expr $l-1]]
        if { $last == "s" || $last == "S" } {
            if ![regexp -nocase ".*CLASS$" $str] {
                set str [string range $str 0 [expr $l-2]]
            }
        }
    }
    
    return $str
}

#  CNDFillGidGata --
#     Construye dentro de CNDDataName la estructura con el patron
#     CNDData(CND,*).
#  Argumentos:
#     CNDDataName --> variable name where the struct will be created
#  Resultados:
#     Las subestructuras que se generan en CNDDataName son las siguientes:
#       CNDDataName(CND,nameslist) --> Lista de los nombres de las condiciones
#       definidas. Ej.: Point-Constraints Line-Constraints ...
#
#       CNDDataName(CND,<name>,CONDTYPE,0) --> descripcion del tipo de condicion
#            para la condicion <name>. Puede tomar uno de los valores:
#            over points, over lines, over surfaces, over volumes.
#
#       CNDDataName(CND,<name>,DIRTY) --> indica si los VALUES han cambiado.
#          Inicialmente no han cnmbiado.
#          0 --> no han cambiado.
#          1 --> han cambiado
#
#       CNDDataName(CND,<name>,numprop) --> cantidad de propiedades.
#
#       CNDDataName(CND,<name>,QUESTION,i) --> valor de la i-esima QUESTION
#            para la condicion <name>. i >= 1.
#
#       CNDDataName(CND,<name>,VALUE,i) --> valor del i-esima VALUE
#            para la condicion <name>. i >= 1.
#
#       CNDDataName(CND,<name>,HELP,i) --> valor de la i-esima HELP asociada a
#            la i-esima QUESTION para la condicion <name>. i >= 1.
#

proc CNDFillGidData { GDN {BOOK Default} } {
    global $GDN
    global gidDataLabels
    upvar #0 $GDN GidData
    
    
    VerifyProblemTypeDir $GDN
    
    set GidData(BOOK) $BOOK
    #    set GidData(TYPEFUNC) CNDGetType
    set GidData(TYPEFUNC) DWGetTypeCND
    set GidData(CLASS) "Conditions"
    set GidData(top) "CND"
    set GidData(DEPDISABLED) 0
    set GidData(CHANGECND) 1
    
    if { $BOOK == "Default" } {
        if { [info exists gidDataLabels(Conditions)] && $gidDataLabels(Conditions) != "" } {
            set GidData(CND,plural) $gidDataLabels(Conditions)
        } else {
            set GidData(CND,plural) [_ "Conditions"]
        }
        
        if { [info exists gidDataLabels(Conditions,singular)] && $gidDataLabels(Conditions,singular) != "" } {
            set GidData(CND,singular) $gidDataLabels(Conditions,singular)
        } else {
            set GidData(CND,singular) [_ "Condition"]
        }
    } else {
        set GidData(CND,plural) [= [DWUnder2Space $BOOK]]
        set GidData(CND,singular) [DWMakeSingular  $GidData(CND,plural)]
    }
    
    
    array set PatternNames {
        over_point  POINT
        over_line LINE
        over_surface SURFACE
        over_volume  VOLUME
        over_layer LAYER
        over_group GROUP
    }
    set CndTypeList { over_point over_line over_surface over_volume over_layer over_group }
    set CndTypeNameList { "over points" "over lines" "over surfaces" "over volumes" "over layers" "over groups"}
    
    foreach CndType $CndTypeList CndTypeName $CndTypeNameList {
        set CurCndNameList [GiD_Info conditions $CndType]
        #lappend GidData(CND,nameslist) $CurCndNameList
        
        foreach CndName $CurCndNameList {
            if {[lindex [::GiD_ConditionBehaviour $CndName visible] 1] eq "disable"} {
                # this condition is not visible
                continue
            }
            # Verifico el book.
            
            set CndBook [GiD_Info conditions $CndName BOOK]
            
            if { $CndBook == $BOOK } {
                lappend GidData(levels) $CndType
                set _CndName $CndName
                
                # Convierto _ a espacio.
                
                regexp -nocase ^$PatternNames($CndType)_(.+) $CndName {} CndName
                set CndName [= [DWUnder2Space $CndName]]
                lappend GidData(CND,$CndType,GUINAMES) $CndName
                lappend GidData(CND,$CndType,GIDNAMES) $_CndName
                
                set OtherFieldsList [GiD_Info conditions $_CndName OTHERFIELDS]
                set OtherFieldsNum  [lindex $OtherFieldsList 0]
                
                set GidData(CND,$_CndName,VISIBLE) 0
                
                set GidData(CND,$_CndName,STATE) 1
                set GidData(CND,$_CndName,HASLA) 0
                set GidData(CND,$_CndName,CONDTYPE,0) $CndTypeName
                set PropList [GiD_Info conditions $_CndName]
                set PropList [lrange $PropList 1 end]
                set NumProp  [lindex $PropList 0]
                set GidData(CND,$_CndName,numprop) $NumProp
                for { set i 1 } { $i <= $NumProp } { incr i 1 } {
                    set qValue [lindex $PropList [expr $i*2-1]]
                    set StringValues [lindex $PropList [expr $i*2]]
                    ProcessField $GDN CND,$_CndName $i $qValue $StringValues
                }
                set GidData(BOOK,HASLA) $GidData(CND,$_CndName,HASLA)
                
                for { set i 1 } { $i <= $OtherFieldsNum } { incr i 1 } {
                    set OFType [lindex $OtherFieldsList [expr $i*3-2]]
                    set OFPos [lindex $OtherFieldsList [expr $i*3]]
                    set OFValue [lindex $OtherFieldsList [expr $i*3-1]]
                    ProcessOtherField $GDN CND,$_CndName $OFType $OFPos $OFValue
                }
                #GenScriptDependencies $GDN "CND,$_CndName"
                DWBindTrace $GDN CND,$_CndName
            }
        }
    }
    return 1
}

#  MATFillGidGata --
#     Construye dentro de CNDDataName la estructura con el patron
#     MATDataName(MAT,*).
#  Argumentos:
#     MATDataName --> variable name where the struct will be created
#  Resultados:
#     Las subestructuras que se generan en MATDataName son las siguientes:
#       MATDataName(MAT,nameslist) --> Lista de los nombres de los materiales
#       definidos. Ej.: Steel Concrete ...
#
#       MATDataName(MAT,<name>,DIRTY) --> indica si los VALUES han cambiado.
#          Inicialmente no han cnmbiado.
#          0 --> no han cambiado.
#          1 --> han cambiado
#
#       MATDataName(MAT,<name>,numprop) --> cantidad de propiedades.
#
#       MATDataName(MAT,<name>,QUESTION,i) --> valor de la i-esima QUESTION
#            para el material <name>. i >= 1.
#
#       MATDataName(MAT,<name>,VALUE,i) --> valor del i-esima VALUE
#            para el material <name>. i >= 1.
#
#       MATDataName(MAT,<name>,<FIELD>,i) --> valor del campo <FIELD>
#            asociado a la i-esima QUESTION para el material <name>. i >= 1.

proc MATFillGidData { GDN { BOOK Default } } {
    global $GDN
    global gidDataLabels
    upvar #0 $GDN GidData
    
    VerifyProblemTypeDir $GDN
    
    #set GidData(MAT,nameslist) ""
    set GidData(MAT,materials) [list]
    set GidData(BOOK) $BOOK
    set GidData(CLASS) "Materials"
    set GidData(top) "MAT"
    #set GidData(TYPEFUNC) MATGetType
    set GidData(TYPEFUNC) DWGetTypeMAT
    set GidData(FILLFUNC) MATFillGDOnlyValues
    set GidData(SAVEFUNC) DWModifyMat
    set GidData(DEPDISABLED) 0
    
    if { $BOOK == "Default" } {
        if { [info exists gidDataLabels(Materials)] && $gidDataLabels(Materials) != "" } {
            set GidData(MAT,plural) $gidDataLabels(Materials)
        } else {
            set GidData(MAT,plural) [_ "Materials"]
        }
        
        if { [info exists gidDataLabels(Materials,singular)] && $gidDataLabels(Materials,singular) != "" } {
            set GidData(MAT,singular) $gidDataLabels(Materials,singular)
        } else {
            set GidData(MAT,singular) [_ "Material"]
        }
    } else {
        set GidData(MAT,plural) [= [DWUnder2Space $BOOK]]
        set GidData(MAT,singular) [DWMakeSingular $GidData(MAT,plural)]
    }
    
    
    set MatNameList [GiD_Info materials]
    
    foreach _MatName $MatNameList {
        
        set MatBook [GiD_Info materials $_MatName BOOK]
        
        if { $MatBook == $BOOK } {
            set GidData(MAT,$_MatName,isloaded) 0
            #       set OtherFieldsList [GiD_Info materials $_MatName OTHERFIELDS]
            #       set OtherFieldsNum [lindex $OtherFieldsList 0]
            lappend GidData(MAT,materials) $_MatName
            #      set PropList [GiD_Info materials $_MatName]
            
            #      set GidData(MAT,$_MatName,VISIBLE) 0
            
            #      set NumProp [lindex $PropList 0]
            #      set GidData(MAT,$_MatName,numprop) $NumProp
            
            #       for { set i 1 } { $i <= $OtherFieldsNum } { incr i 1 } {
                #         set OFType [lindex $OtherFieldsList [expr $i*3-2]]
                #         set OFPos [lindex $OtherFieldsList [expr $i*3]]
                #         set OFValue [lindex $OtherFieldsList [expr $i*3-1]]
                #         ProcessOtherField $GDN "MAT,$_MatName" $OFType $OFPos $OFValue
                #       }
            #       for { set i 1 } { $i <= $NumProp } { incr i 1 } {
                #         set qValue [lindex $PropList [expr $i*2-1]]
                #         set StringValues [lindex $PropList [expr $i*2]]
                #         ProcessField $GDN MAT,$_MatName $i $qValue $StringValues
                #       }
            #       #GenScriptDependencies $GDN MAT,$_MatName
            #       DWBindTrace $GDN MAT,$_MatName
        }
    }
    return 1
}

# ----------------------------------------------------------------------------
# DW_LoadMaterial --
#
# Fill in GidData the material properties of material mat.
#
# ----------------------------------------------------------------------------

proc DW_LoadMaterial {GDN mat} {
    if { $mat == "" } {
        return
    }
    upvar #0 $GDN GidData
    
    set GidData(MAT,$mat,VISIBLE) 0
    set OtherFieldsList [GiD_Info materials $mat OTHERFIELDS]
    set OtherFieldsNum [lindex $OtherFieldsList 0]
    set PropList [GiD_Info materials $mat]
    set NumProp [lindex $PropList 0]
    set GidData(MAT,$mat,numprop) $NumProp
    for { set i 1 } { $i <= $OtherFieldsNum } { incr i 1 } {
        set OFType [lindex $OtherFieldsList [expr $i*3-2]]
        set OFPos [lindex $OtherFieldsList [expr $i*3]]
        set OFValue [lindex $OtherFieldsList [expr $i*3-1]]
        ProcessOtherField $GDN MAT,$mat $OFType $OFPos $OFValue
    }
    for { set i 1 } { $i <= $NumProp } { incr i 1 } {
        set qValue [lindex $PropList [expr $i*2-1]]
        set StringValues [lindex $PropList [expr $i*2]]
        ProcessField $GDN MAT,$mat $i $qValue $StringValues
    }
    DWBindTrace $GDN MAT,$mat
    set GidData(MAT,$mat,isloaded) 1
}

namespace eval GID_Transform {
    # returns a list with the {type, label, arguments}
    #
    proc parse_question {Q} {
        
    }
    
    # try to convert the pair <Q1,V1> to <Q2,V2>
    # return the new value in case of matching or empty in other case
    #
    proc match_question {Q1 V1 Q2 V2} {
        set Q1_info [parse_question $Q1]
        set Q2_info [parse_question $Q2]
        if {$Q1_info eq "" || $Q2_info eq ""} {
            return ""
        }
        set type1 [lindex $Q1_info 0]
        set type2 [lindex $Q2_info 0]
        if {$type1 ne $type2} {
            return ""
        }
        set V_new [list]
        if {[lindex $Q1_info 1] eq [lindex $Q2_info 1]} {
            # label & type matches
            if {$type1 eq "GRID"} {
                set len1 [llength [lindex $Q2_info 2]]
                set len2 [llength [lindex $Q2_info 2]]
                if {$len1 == $len2} {
                    return $V1
                }
                set count1 [lindex $V1 1]
                set rows1 [expr {$len1/$count1}]
                set count2 [lindex $V2 1]
                set rows2 [expr {$len2/$count2}]
                set V_new [list "#N#" [expr {$count1+($len2-$len1)/$rows1}]]
                set start1 2
                set end1 [expr {$len1 + 1}]
                set start2 [expr {$end1+1}]
                set end2 [expr {$len2 + 1}]
                if {$len1 < $len2} {
                    for {set r 0} {$r < $rows1} {incr r} {
                        lappend V_new {*}[lrange $V1 $start1 $end1]
                        set complete [lrange $V2 $start2 $end2]
                        if {$complete eq ""} {
                            set complete $default_complete
                        } else {
                            set default_complete $complete
                        }
                        lappend V_new {*}$complete
                        incr start1 $len1
                        incr end1 $len1
                        incr start2 $len2
                        incr end2 $len2
                    }
                } else {
                }
            }
        }
        set $V_new
    }
}

array set TypeInfo {
    MAT,ARGS      -1
    CB,ARGS       2
    FUNC,ARGS     2
    LA,ARGS       2
    UNITS,ARGS    0
}

proc ParserType { Question } {
    global TypeInfo
    
    # reconce una cadena seguida de argmentos entre ()
    #     set re "(.+)\#(MAT|CB|LA|FUNC|UNITS|TKWIDGET)\#(.*)\$"
    set re "(.+)\#(MAT|CB|LA|FUNC|UNITS)\#(.*)\$"
    
    if [regexp -nocase $re $Question {} id type args] {
        set id   [string trim $id]
        set type [string toupper $type]
        set args [string trim $args]
        if { $args == "" } {
            if { $TypeInfo($type,ARGS) > 0 } {
                W [_ "In %s: field type %s does needs arguments. Assuming ENTRY" $Question $type]
                set type "ENTRY"
            }
            return [list $type $id]
        }
        if ![regexp \\((.+)\\)\$ $args {} args] {
            W [_ "In %s : invalid arguments in %s field. Assuming ENTRY" $Question $type]
            return [list ENTRY $id]
        }
        
        # Proceso los argumentos
        
        set _args [split $args ","]
        if { [llength $_args] == 1 } {
            set _args [split $args " "]
        }
        set args {}
        foreach a $_args {
            set a [string trim $a]
            if [string length $a] {
                lappend args [string trim $a]
            }
        }
        
        if { $TypeInfo($type,ARGS) == 1 } {
            if { [llength $args] > 1 } {
                set args [lindex $args 0]
                W [_ "In %s: only one argument allowed in %s. Using the first '%s'" $Question $type $args]
            }
        }
        return [list $type $id $args]
    }
    
    set re {([^\(]+)\((.+)\)$}
    if [regexp $re $Question {} id args] {
        # es una grid
        # Proceso los argumentos
        
        set _args [split $args ","]
        if { [llength $_args] == 1 } {
            set _args [split $args " "]
        }
        set args {}
        foreach a $_args {
            lappend args [string trim $a]
        }
        return [list "GRID" $id $args]
    }
    return [list "ENTRY" $Question]
}

proc ProcessUnitValue { GDN HEAD Idx ValueName } {
    global $GDN
    upvar #0 $GDN GidData
    upvar $ValueName Value
    
    set GidData($HEAD,NUMERATOR,$Idx) ""
    set GidData($HEAD,DENOMINATOR,$Idx) ""
    set GidData($HEAD,MAGNITUD,$Idx) ""
    set GidData($HEAD,IDXUNIT,$Idx)  -1
    
    ParserNumberUnit $Value n u
        
    if { $n == "" || $u == ""} {
        return "There must be one number and one unit in expresion $Value"
    } else {
        #trick to see if current unit is not allowed by current unit system
        ParserUnit::Init $u
        ParserUnit::Start
        if { [lindex $ParserUnit::Status 0] != "ERROR" } {
            CreateExpTextList $ParserUnit::Numerator NumeratorTextList
            CreateExpTextList $ParserUnit::Denominator DenominatorTextList
            set unitinfo_tmp [list {} {} {} $NumeratorTextList $DenominatorTextList]
            if { [Units::IsUnitDisallowed $unitinfo_tmp] } {
                set Value [GidConvertValueUnit $Value]
                ParserNumberUnit $Value n u
            }                      
        }
        #end trick
        set GidData($HEAD,VALUENUM,$Idx) $n
        regsub -all { } $u {} u
        set GidData($HEAD,VALUEUNIT,$Idx) $u
        ParserUnit::Init $u
        ParserUnit::Start
        if { [lindex $ParserUnit::Status 0] != "ERROR" } {
            CreateExpTextList $ParserUnit::Numerator NumeratorTextList
            CreateExpTextList $ParserUnit::Denominator DenominatorTextList
            set GidData($HEAD,NUMERATOR,$Idx) $NumeratorTextList
            set GidData($HEAD,DENOMINATOR,$Idx) $DenominatorTextList
            set UInfo [FindUnitGidAndPrj $ParserUnit::Numerator $ParserUnit::Denominator]            
            if { [llength $UInfo] } {
                set Value "$GidData($HEAD,VALUENUM,$Idx)$GidData($HEAD,VALUEUNIT,$Idx)"
                set GidData($HEAD,UNITVALID,$Idx) 2
                lassign $UInfo umag uidx utable
                if ![info exists GidData(MAG,$umag)] {
                    #1==problemtype defined magnitudes
                    set GidData(MAG,$umag) [GiD_Info magnitudes 1 $umag units]
                    if { $utable == 1 } {
                        #units defined in the problemtype
                        set IdxNeedUpdate 0
                        set gidx 0
                        set GidData($HEAD,IDXUNIT,$Idx) $uidx
                    } else {
                        #units defined in GiD
                        set IdxNeedUpdate 1
                        set gidx [llength $GidData(MAG,$umag)]
                        incr gidx -1
                    }
                    set lidx 0
                    #0==GiD defined magnitudes
                    foreach u [GiD_Info magnitudes 0 $umag units] {
                        set N [lindex $u 1]
                        set D [lindex $u 2]
                        if { [FindUnitInList $N $D $GidData(MAG,$umag)] == {} } {
                            lappend GidData(MAG,$umag) $u
                            incr gidx
                        }
                        if { $IdxNeedUpdate && $lidx == $uidx } {
                            set GidData($HEAD,IDXUNIT,$Idx) $gidx
                        }
                        incr lidx
                    }
                } else {
                    set GidData($HEAD,IDXUNIT,$Idx) [FindUnitInList $ParserUnit::Numerator $ParserUnit::Denominator $GidData(MAG,$umag)]
                }
                set GidData($HEAD,MAGNITUD,$Idx) $umag
            } else {
                set GidData($HEAD,UNITVALID,$Idx) 1
            }
        } else {
            set GidData($HEAD,UNITVALID,$Idx) 0
        }
    }
    return ""
}

proc DWGetMaterials { books } {
    set strError {}
    if { [llength $books] } {
        set loptions {}
        set visited {}
        set alreadyVisited {}
        set notVisited {}
        set materials [GiD_Info materials]
        foreach b $books {
            if { [lsearch -exact $visited $b] == -1 } {
                lappend visited $b
                set found 0
                foreach m $materials {
                    if ![string compare [GiD_Info materials $m BOOK] $b] {
                        lappend loptions $m
                        set found 1
                    }
                }
                if !$found {
                    lappend notVisited $b
                }
            } else {
                lappend alreadyVisited $b
            }
        }
        set strError ""
        if [llength $notVisited] {
            set strError [concat [_ "The following books are not defined"]: $notVisited]
        }
        if [llength $alreadyVisited] {
            if [string length $strError] {
                append strError "\n"
            }
            append strError [concat [_ "The following books are referenced more than once"]: $alreadyVisited]
        }
    } else {
        set loptions [GiD_Info materials]
        
    }
    if ![llength $loptions] {
        append strError [_ "Requested materials not found"]
    }
    return [list $strError $loptions]
}

proc ProcessField { GDN HEAD Idx Question Value } {
    global $GDN
    upvar #0 $GDN GidData
    
    set Question [string trim $Question]
    set Value [string trim $Value]
    set typeInfo [ParserType $Question]
    
    set FType [lindex $typeInfo 0]
    set fieldName [lindex $typeInfo 1]
    #if { [llength $fieldName] != 1 } {
        #  # Por que no puede contener espacios?????
        #  # investigalo.
        #  error "Invalid field name '$fieldName' in question '$Question' at $GidData(CLASS)"
        #}
    # WarnWinText "ProcessField---$Question---$Value---$typeInfo"
    set widget_options ""
    switch $FType {
        "GRID" {
            set GidData($HEAD,HEADINGS,$Idx) [lindex $typeInfo end]
            if { [llength $Value] >= 2 && [lindex $Value 0] == "#N#" } {
                if { [lindex $Value 1] != [expr [llength $Value]-2] } {
                    error [_ "Error in %s: #N# GRID field %s, declared %s values but exists %s" $HEAD \
                            $fieldName [lindex $Value 1] [expr [llength $Value]-2]]
                }
                DWCreateGrid $GDN $HEAD $Idx [lrange $Value 1 end]
            } else {
                set FType "ENTRY"
                set fieldName $Question
            }
        }
        "UNITS" {
            set err_str [ProcessUnitValue $GDN $HEAD $Idx Value]
            if { $err_str != "" } {
                error [_ "%s in field: %s" $err_str $Question]
            }
            
        }
        "MAT" -
        "LA" -
        "CB" {
            if { $FType == "LA" } {
                set GidData($HEAD,CBTYPE,$Idx) "COMBO"
                set GidData($HEAD,OPTIONS1,$Idx) [GiD_Info localaxes]
                set GidData($HEAD,OPTIONS,$Idx) {}
                foreach i [lindex $typeInfo end] {
                    regsub -nocase (\#A\#|\#G\#)$ $i {} text
                    lappend GidData($HEAD,OPTIONS,$Idx) $text
                }
                set GidData($HEAD,CBTEXT,$Idx) [= [DWUnder2Space $Value]]
                set GidData($HEAD,POSTING,$Idx) 0
                set GidData($HEAD,HASLA) 1
            } else {
                if { $FType == "MAT" } {
                    set GidData($HEAD,MATFIELD,BOOKS,$Idx) [lindex $typeInfo 2]
                    set infomat [DWGetMaterials [lindex $typeInfo 2]]
                    set strError [lindex $infomat 0]
                    if [string length $strError] {
                        W $strError
                    }
                    if [llength [set loptions [lindex $infomat 1]]] {
                        set GidData($HEAD,OPTIONS,$Idx) $loptions
                        set isCombo 1
                    } else {
                        set FType "ENTRY"
                        set isCombo 0
                        W [_ "In %s: requested materials not found. Assuming ENTRY field" $Question]
                    }
                } else {
                    set GidData($HEAD,OPTIONS,$Idx) [lindex $typeInfo end]
                    set len [llength $GidData($HEAD,OPTIONS,$Idx)]
                    set isCombo 0
                    if { $len == 1 } {
                        set GidData($HEAD,CBTYPE,$Idx) "LABEL"
                    } elseif { $len == 2 } {
                        set true_false [are_true_false_words [lindex $GidData($HEAD,OPTIONS,$Idx) 0] [lindex $GidData($HEAD,OPTIONS,$Idx) 1]]                     
                        if { $true_false == 1 } {
                            set GidData($HEAD,CBTYPE,$Idx) [list "CHECKBOX" 0 1]
                        } elseif { $true_false == -1 } {
                            set GidData($HEAD,CBTYPE,$Idx) [list "CHECKBOX" 1 0]
                        } else {
                            set isCombo 1
                        }
                    } else {
                        set isCombo 1
                    }
                }
                if $isCombo {
                    set GidData($HEAD,CBTYPE,$Idx) "COMBO"
                    if { [set GidData($HEAD,CURSEL,$Idx) [lsearch -exact $GidData($HEAD,OPTIONS,$Idx) $Value]] == -1 } {
                        set GidData($HEAD,CURSEL,$Idx) 0
                        set Value [lindex $GidData($HEAD,OPTIONS,$Idx) 0]
                    }
                    set GidData($HEAD,CBTEXT,$Idx) [= [DWUnder2Space $Value]]
                    set GidData($HEAD,POSTING,$Idx) 0
                }
            }
        }
        default {
            set Value [string trim $Value]

            if { [ regexp {(.*)#WIDTH#\(([0-9]+)\)} $Value dum true_value custom_width] } {
                set Value $true_value
                if { $custom_width >= 1} {
                    append widget_options " -width $custom_width"
                }
            }
        }
    }
    set GidData($HEAD,QUESTION,$Idx) $fieldName
    set GidData($HEAD,LABELQ,$Idx) [= [DWUnder2Space $fieldName]]
    set GidData($HEAD,VALUE,$Idx) $Value
    set GidData($HEAD,TYPE,$Idx) $FType

    set GidData($HEAD,WIDGET_OPTIONS,$Idx) $widget_options
}


# Este proc es utilizado for GenericFillGDFull en los eventos accept
# de los datos parciales intvdata y gendata

proc ProcessFieldOnlyValue { GDN HEAD Idx Question Value } {
    global $GDN
    upvar #0 $GDN GidData
    
    set Question [string trim $Question]
    set Value [string trim $Value]
    set typeInfo [ParserType $Question]
    set FType [lindex $typeInfo 0]
    set fieldName [lindex $typeInfo 1]
    
    # QUE impide que un campo tenga espacios?
    
    if { [llength $fieldName] != 1 } {
        error [_ "Invalid field name '%s' in question '%s' at %s" $fieldName $Question $GidData(CLASS)]
    }
    switch $FType {
        "GRID" {
            set GidData($HEAD,HEADINGS,$Idx) [lindex $typeInfo end]
            if { [llength $Value] >= 2 && [lindex $Value 0] == "#N#" } {
                DWCreateGrid $GDN $HEAD $Idx [lrange $Value 1 end]
            } else {
                set FType "ENTRY"
                set fieldName $Question
            }
        }
        UNITS {
            if { [set err_str [ProcessUnitValue $GDN $HEAD $Idx Value]] != "" } {
                WarnWin [join [list $err_str " [_ "in field"]: $Question"] ""]
            }
        }
    }
    set GidData($HEAD,QUESTION,$Idx) $fieldName
    set GidData($HEAD,VALUE,$Idx) $Value
    set GidData($HEAD,TYPE,$Idx) $FType
}


proc MATFillGDOnlyValues { GDNOV GDN } {
    global $GDN $GDNOV
    upvar #0 $GDN GidData
    upvar #0 $GDNOV GDOnlyValue
    
    # Nombre del material actual.    
    set _MatName [DWGetGIDMaterial $GDN]
    
    set PropList [GiD_Info materials $_MatName]
    set NumProp [lindex $PropList 0]
    set GDOnlyValue(MAT,$_MatName,numprop) $NumProp
    
    for { set i 1 } { $i <= $NumProp } { incr i 1 } {
        set qValue [lindex $PropList [expr $i*2-1]]
        set StringValues [lindex $PropList [expr $i*2]]
        ProcessField $GDNOV "MAT,$_MatName" $i $qValue $StringValues
    }
}

proc CheckDirtyState { GDN } {
    global $GDN gidSubstDataBooks
    global gidSubstDataBooks
    
    upvar #0 $GDN GidData
    
    if { $gidSubstDataBooks || ![info exists GidData(SAVEFUNC)] } {
        return
    }
    
    # OJO el type ya va en GidData, revisar BuildStringValues
    set CurValues  [BuildStringValues $GDN]
    
    set TYPE [$GidData(TYPEFUNC) $GDN]
    
    if { [string compare $CurValues $GidData(FIRSTSTATE)] } {
        set strMes [_ "Data has changed. Do you want to save it?"]
        set Title [_ "Warning"]:[_ $GidData(CLASS)]-[= [DWUnder2Space $GidData(BOOK)]]
        set answer [MessageBoxOptionsButtons $Title $strMes \
            {0 1} [list [_ "Yes"] [_ "No#C#I don't accept"]] question ""]
        switch $answer {
            0 {
                set GidData($TYPE,SYNC,TKWIDGET) 0
                $GidData(SAVEFUNC) $GDN
                set GidData($TYPE,SYNC,TKWIDGET) 1
            }
            1 {
                # do not save the data & continue
            }
            2 {
                # abort the operation
                error CANCEL
            }
        }
    }
}

proc DWGetGIDCondName { GDN } {
    upvar #0 $GDN GidData
    
    # lindex $GidData(CND,$GidData(CND,level),GIDNAMES) $GidData(CND,cursel)
    #return $GidData(CND,cursel)
    set idx [ lsearch [ $GidData(CND,combo) cget -values] $GidData(CND,cursel)]
    lindex $GidData(CND,$GidData(CND,level),GIDNAMES) $idx
}

proc DWGetGUICondName { GDN } {
    upvar #0 $GDN GidData
    
    # lindex $GidData(CND,$GidData(CND,level),GUINAMES) $GidData(CND,cursel)
    return $GidData(CND,cursel)
}

proc DWGetTypeCND { GDN } {
    upvar #0 $GDN GidData
    
    return "CND,[DWGetGIDCondName $GDN]"
}

proc DWGetTypeID { GDN } {
    return ID
}

proc DWGetTypePD { GDN } {
    return PD
}

# ------------------------------------------------------------------------------
# DWGetTypeMAT --
#
#  return the prefix of the element's index in GDN for the current material
# ------------------------------------------------------------------------------

proc DWGetTypeMAT { GDN } {
    upvar #0 $GDN GidData    
    return MAT,[DWGetGIDMaterial $GDN]
}

#  GenericFillGidGata --
#     Construye dentro de GidDataName la estructura con el patron
#     GidDataName($TYPE,*).
#  Argumentos:
#     TYPE --> Es el nombre del tipo de estructura. Puede tomar los valores:
#       ID : interval data.
#       PD : problem data.
#     GidDataName --> variable name where the struct will be created
#  Resultados:
#     Las subestructuras que se generan en GidDataName son las siguientes:
#
#       GidDataName($TYPE,DIRTY) --> indica si los VALUES han cambiado.
#          Inicialmente no han cnmbiado.
#          0 --> no han cambiado.
#          1 --> han cambiado
#
#       GidDataName($TYPE,numprop) --> numero de propiedades.
#
#       GidDataName($TYPE,QUESTION,i) --> valor de la i-esima QUESTION
#
#       GidDataName($TYPE,VALUE,i) --> valor del i-esima VALUE
#
#       GidDataName($TYPE,<FIELD>,k) --> valor del campo <FIELD>
#            asociado a la k-esima QUESTION. k >= 1.

proc GenericFillGidData { GDN TYPE {BOOK Default} } {
    global $GDN
    global gidDataLabels
    upvar #0 $GDN GidData
    
    VerifyProblemTypeDir $GDN
    
    switch $TYPE {
        "ID" {
            set DataArg intvdata
            set OFArg {intvdata -interval 1}
            set GidData(TYPEFUNC) DWGetTypeID
            set GidData(SAVEFUNC) DWIntervalAccept
            set GidData(CLASS) "Interval data"
            
            # Activo el primer intervalo, es donde se definen los BOOKS.
            
            #DWSetActiveInterval 1
            set GidData(ID,restore-active) [DWGetActiveInterval]
        }
        "PD" {
            set DataArg gendata
            set OFArg gendata
            set GidData(TYPEFUNC) DWGetTypePD
            set GidData(SAVEFUNC) DWProblemAccept
            set GidData(CLASS) "Problem Data"
        }
        default {
            error [_ "Invalid data type %s" $TYPE]
        }
    }
    set GidData(top) $TYPE
    set GidData(FILLFUNC) GenericFillGDOnlyValues
    set GidData(BOOK) $BOOK
    set GidData(DEPDISABLED)   0
    set GidData($TYPE,VISIBLE) 0
    
    set name $GidData(CLASS)
    if { $BOOK == "Default" } {
        if { [info exists gidDataLabels($name)] && $gidDataLabels($name) != "" } {
            set GidData($TYPE,plural) $gidDataLabels($name)
        } else {
            set GidData($TYPE,plural) [_ $name]
        }
        
        if { [info exists gidDataLabels($name,singular)] && $gidDataLabels($name,singular) != "" } {
            set GidData($TYPE,singular) $gidDataLabels($name,singular)
        } else {
            set GidData($TYPE,singular) [_ $name]
        }
    } else {
        set GidData($TYPE,plural) [= [DWUnder2Space $BOOK]]
        set GidData($TYPE,singular) [DWMakeSingular $GidData($TYPE,plural)]
    }
    
    set PropList [GiD_Info {*}$DataArg]
    
    set NumProp [lindex $PropList 0]
    set NumPropReal 0
    
    set OtherFieldsList [GiD_Info {*}$OFArg OTHERFIELDS]
    
    set OtherFieldsNum  [lindex $OtherFieldsList 0]
    
    set BookStart [lsearch -exact $OtherFieldsList $BOOK]
    
    if { $BookStart != -1 } {
        set RestList [lrange $OtherFieldsList $BookStart end]
        set BookStart [expr [lindex $RestList 1]+ 1]
        set BookEnd [lsearch -exact $RestList BOOK]
        if { $BookEnd == -1 } {
            set BookEnd $NumProp
        } else {
            set BookEnd [lindex $RestList [expr $BookEnd + 2]]
        }
    } else {
        set BookStart 1
        set BookEnd $NumProp
    }
    
    set GidData($TYPE,numprop) [expr $BookEnd - $BookStart + 1]
    set GidData($TYPE,BSTART) $BookStart
    set GidData($TYPE,BEND) $BookEnd
    
    for { set i $BookStart; set j 1 } { $i <= $BookEnd } { incr i; incr j} {
        set qValue [lindex $PropList [expr $i*2-1]]
        set StringValues [lindex $PropList [expr $i*2]]
        
        ProcessField $GDN "$TYPE" $j $qValue $StringValues
    }
    
    incr BookStart -1
    
    for { set i 1 } { $i <= $OtherFieldsNum } { incr i 1 } {
        set OFPos [lindex $OtherFieldsList [expr $i*3]]
        if { ($OFPos >= $BookStart) && ($OFPos <= $BookEnd) } {
            set OFType [string toupper [lindex $OtherFieldsList [expr $i*3-2]]]
            if { $OFPos == $BookStart && $OFType != "TITLE" && $OFType != "TKWIDGET" } {
                continue
            }
            if { $OFPos == $BookEnd && $OFType == "TITLE" } {
                continue
            }
            set OFValue [lindex $OtherFieldsList [expr $i*3-1]]
            set OFIdx [expr $OFPos - $BookStart]
            ProcessOtherField $GDN $TYPE $OFType $OFIdx $OFValue
        }
    }
    
    DWBindTrace $GDN $TYPE
    return 1
}

proc DWRefillIntervalData { GDN } {
    global $GDN
    upvar #0 $GDN GidData
    
    set BOOK $GidData(BOOK)
    
    set PropList [GiD_Info intvdata]
    set NumProp [lindex $PropList 0]
    
    set BookStart $GidData(ID,BSTART)
    set BookEnd   $GidData(ID,BEND)
    
    for { set i $BookStart; set j 1 } { $i <= $BookEnd } { incr i; incr j} {
        set StringValues [lindex $PropList [expr $i*2]]
        set GidData(ID,VALUE,$j) $StringValues
        if { [lindex $StringValues 0] == "#N#" } {
            DWCreateGrid $GDN ID $j [lrange $StringValues 1 end]
        }
    }
}



proc GenericFillGDFull { GDN TYPE FGDN } {
    global $GDN $FGDN
    upvar #0 $GDN GidData
    upvar #0 $FGDN FGidData
    
    switch $TYPE {
        "ID" {
            set DataArg intvdata
        }
        "PD" {
            set DataArg gendata
        }
        default {
            error [_ "Invalid data type %s" $TYPE]
        }
    }
    
    set FGidData($TYPE,SYNC,TKWIDGET) 0
    set FGidData(TYPEFUNC) $GidData(TYPEFUNC)
    
    set PropList [GiD_Info $DataArg]
    set NumProp [lindex $PropList 0]
    
    set FGidData($TYPE,numprop) $NumProp
    
    for { set i 1 } { $i <= $NumProp } { incr i } {
        set qValue [lindex $PropList [expr $i*2-1]]
        set StringValues [lindex $PropList [expr $i*2]]        
        ProcessFieldOnlyValue $FGDN $TYPE $i $qValue $StringValues
    }
    
    # No necesito los OTHERFIELDS.
    
    # Ahora sustituyo los valores de FGidData por los de GidData
    # Hay que tener cuidado con las Grid y las unidades
    
    set NumProp $GidData($TYPE,numprop)
    set Start $GidData($TYPE,BSTART)
    set End   $GidData($TYPE,BEND)
    
    for { set i 1; set j $Start } { $i <= $NumProp } { incr i; incr j } {
        set Value $GidData($TYPE,VALUE,$i)
        switch $GidData($TYPE,TYPE,$i) {
            GRID {
                set NumCells $GidData($TYPE,VALUE,$i,0)
                set FNumCells $FGidData($TYPE,VALUE,$j,0)
                set FGidData($TYPE,VALUE,$j,0) $NumCells
                for {set k 1} {$k <= $NumCells} {incr k} {
                    set FGidData($TYPE,VALUE,$j,$k) $GidData($TYPE,VALUE,$i,$k)
                }
                for {} { $k <= $FNumCells } { incr k } {
                    unset FGidData($TYPE,VALUE,$j,$k)
                }
            }
            UNITS {
                set FGidData($TYPE,VALUE,$j) $Value
            }
            default {
                set FGidData($TYPE,VALUE,$j) $Value
            }
        }
    }
}


proc DWBindTrace { GDN TYPE } {
    upvar #0 $GDN GidData
    
    set until $GidData($TYPE,numprop)
    for { set i 1 } { $i <= $until } { incr i } {
        trace remove variable GidData($TYPE,STATE,$i) write "[list DWOnChangeSTATE $GDN $TYPE $i];\#"
        trace add variable GidData($TYPE,STATE,$i) write "[list DWOnChangeSTATE $GDN $TYPE $i];\#"
        # enlazo las dependencias primero
        if [info exists GidData($TYPE,DEPSOURCE,$i)] {
            if { [ScriptGen $GDN $TYPE $i $GidData($TYPE,DEPSOURCE,$i) GidData($TYPE,DEPENDENCIES,$i) WarnStr] } {
                CreateTraceProc $GDN $TYPE $i
                trace remove variable GidData($TYPE,VALUE,$i) write "[list [DWGetDepProcName $GDN $TYPE $i]];\#"
                trace add variable GidData($TYPE,VALUE,$i) write "[list [DWGetDepProcName $GDN $TYPE $i]];\#"
            } else {
                unset GidData($TYPE,DEPENDENCIES,$i)
                W [concat [_ "Error in dependencies : %s\n Assuming no dependencies." $WarnStr] \
                        ($GidData($TYPE,DEPSOURCE,$i))]
            }
        }
        set ftype $GidData($TYPE,TYPE,$i)
        switch $ftype {
            "MAT" -
            "LA" -
            "CB" {
                if { $GidData($TYPE,CBTYPE,$i) == "COMBO" } {
                    # limpio el trace de CURSEL                    
                    trace remove variable GidData($TYPE,CURSEL,$i) write "[list DWOnChangeCURSEL $GDN $TYPE $i];\#"
                    trace add variable GidData($TYPE,CURSEL,$i) write "[list DWOnChangeCURSEL $GDN $TYPE $i];\#"
                    # limpio el trace de VALUE
                    trace remove variable GidData($TYPE,VALUE,$i) write "[list DWOnChangeCBVALUE $GDN $TYPE $i];\#"
                    trace add variable GidData($TYPE,VALUE,$i) write "[list DWOnChangeCBVALUE $GDN $TYPE $i];\#"
                    set GidData($TYPE,SYNC,$i) 1
                }
            }
            "GRID" {
                set GidData($TYPE,OLDVALUE,$i) $GidData($TYPE,VALUE,$i)
                trace remove variable GidData($TYPE,VALUE,$i) write "[list DWOnChangeGRIDVALUE $GDN $TYPE $i];\#"
                trace add variable GidData($TYPE,VALUE,$i) write "[list DWOnChangeGRIDVALUE $GDN $TYPE $i];\#"
            }
            "UNITS" {
                set GidData($TYPE,OLDVALUENUM,$i) $GidData($TYPE,VALUENUM,$i)
                trace remove variable GidData($TYPE,VALUENUM,$i) write "[list DWOnChangeVALUENUM $GDN $TYPE $i];\#"
                trace add variable GidData($TYPE,VALUENUM,$i) write "[list DWOnChangeVALUENUM $GDN $TYPE $i];\#"
                trace remove variable GidData($TYPE,VALUE,$i) write "[list DWOnChangeUNITFIELD $GDN $TYPE $i];\#"
                trace add variable GidData($TYPE,VALUE,$i) write "[list DWOnChangeUNITFIELD $GDN $TYPE $i];\#"
                trace remove variable GidData($TYPE,IDXUNIT,$i) write "[list DWUpdateStrUnit $GDN $TYPE $i];\#"
                if { $GidData($TYPE,UNITVALID,$i) == 2 } {
                    trace add variable GidData($TYPE,IDXUNIT,$i) write "[list DWUpdateStrUnit $GDN $TYPE $i];\#"
                }
                set GidData($TYPE,SYNC,$i) 1
            }
        }
    }
}

proc DWOnChangeGRIDVALUE { GDN TYPE Idx } {
    upvar #0 $GDN GidData
    
    set NewValue $GidData($TYPE,VALUE,$Idx)
    
    if { ([lindex $NewValue 0] != "\#N\#") || ![string is integer -strict [lindex $NewValue 1]] } {
        W [_ "Invalid value %s for GRID field %s" $NewValue $GidData($TYPE,QUESTION,$Idx)]
    } else {
        if [DWCreateGrid $GDN $TYPE $Idx [lrange $NewValue 1 end]] {
            set GidData($TYPE,OLDVALUE,$Idx) $NewValue
        } else {
            DWCreateGrid $GDN $TYPE $Idx [lrange $GidData($TYPE,OLDVALUE,$Idx) 1 end]
            set GidData($TYPE,VALUE,$Idx) $GidData($TYPE,OLDVALUE,$Idx)
        }
    }
}

proc DWOnChangeSTATE { GDN TYPE Idx } {
    upvar #0 $GDN GidData
    
    set cm [split $TYPE ","]
    set top [lindex $cm 0]
    set member [lindex $cm 1]
    set class [ClassInfo::GetClass $top]
    set question $GidData($TYPE,QUESTION,$Idx)
    if { [catch {GiD_AccessValue set $class $member $question STATE $GidData($TYPE,STATE,$Idx)}] } {
        
    }
}

proc DWUpdateStrUnit { GDN TYPE Idx } {
    upvar #0 $GDN GidData
    
    set UInfo [lindex $GidData(MAG,$GidData($TYPE,MAGNITUD,$Idx)) $GidData($TYPE,IDXUNIT,$Idx)]
    set GidData($TYPE,VALUEUNIT,$Idx) [lindex $UInfo end]
}

proc DWOnChangeVALUENUM { GDN TYPE Idx } {
    upvar #0 $GDN GidData
    
    if $GidData($TYPE,SYNC,$Idx) {
        if { $GidData($TYPE,VALUENUM,$Idx) == "" || $GidData($TYPE,VALUENUM,$Idx) == "-" } return
        set GidData($TYPE,SYNC,$Idx) 0
        set GidData($TYPE,VALUE,$Idx) $GidData($TYPE,VALUENUM,$Idx)$GidData($TYPE,VALUEUNIT,$Idx)
        
        set GidData($TYPE,SYNC,$Idx) 1
    }
}

proc DWOnChangeUNITFIELD { GDN TYPE Idx } {
    upvar #0 $GDN GidData
    
    if $GidData($TYPE,SYNC,$Idx) {
        
        ParserNumberUnit $GidData($TYPE,VALUE,$Idx) n u
        
        if { $n == "" || $u == "" } {
            # Muesto el error
            W [_ "Invalid unit expresion %s in question %s. There must be one number and one unit" \
                    $GidData($TYPE,VALUE,$Idx) $GidData($TYPE,QUESTION,$Idx)]
            # Restauro el valor anterior valido
            set GidData($TYPE,VALUE,$Idx) $GidData($TYPE,VALUENUM,$Idx)$GidData($TYPE,VALUEUNIT,$Idx)
        } else {
            regsub -all { } $u {} u
            ParserUnit::Init $u
            ParserUnit::Start
            if { [lindex $ParserUnit::Status 0] == "ERROR" } {
                # Muesto el error
                W [_ "Invalid unit expresion %s in question %s" \
                        $GidData($TYPE,VALUE,$Idx) $GidData($TYPE,QUESTION,$Idx)]
                # Restauro el valor anterior valido
                set GidData($TYPE,VALUE,$Idx) $GidData($TYPE,VALUENUM,$Idx)$GidData($TYPE,VALUEUNIT,$Idx)
            } else {
                set GidData($TYPE,SYNC,$Idx) 0
                if { $GidData($TYPE,UNITVALID,$Idx) == 2 } {
                    set UInfo [FindUnitGidAndPrj $ParserUnit::Numerator $ParserUnit::Denominator]
                    if [llength $UInfo] {
                        set umag [lindex $UInfo 0]
                        if { $umag == $GidData($TYPE,MAGNITUD,$Idx) } {
                            set GidData($TYPE,VALUENUM,$Idx) $n
                            set GidData($TYPE,IDXUNIT,$Idx) [FindUnitInList $ParserUnit::Numerator $ParserUnit::Denominator $GidData(MAG,$umag)]
                        } else {
                            W [_ "Incompatible unit %s for question %s" $u $GidData($TYPE,QUESTION,$Idx)]
                            # Restauro el valor anterior valido
                            set GidData($TYPE,VALUE,$Idx) $GidData($TYPE,VALUENUM,$Idx)$GidData($TYPE,VALUEUNIT,$Idx)
                        }
                    } else {
                        # Muesto el error
                        W [_ "Unit %s not defined in question %s" $u $GidData($TYPE,QUESTION,$Idx)]
                        # Restauro el valor anterior valido
                        set GidData($TYPE,VALUE,$Idx) $GidData($TYPE,VALUENUM,$Idx)$GidData($TYPE,VALUEUNIT,$Idx)
                    }
                } else {
                    set GidData($TYPE,VALUENUM,$Idx) $n
                }
                set GidData($TYPE,SYNC,$Idx) 1
            }
        }
    }
}



# Este es el trace lanzado por CURSEL, cuando se escoge una opcion en el CB

proc DWOnChangeCURSEL { GDN TYPE Idx } {
    upvar #0 $GDN GidData
    
    if !$GidData($TYPE,POSTING,$Idx) {
        if $GidData($TYPE,SYNC,$Idx) {
            
            set GidData($TYPE,SYNC,$Idx) 0
            set m [lindex $GidData($TYPE,WIDGET,$Idx) 0].m
            set i $GidData($TYPE,CURSEL,$Idx)
            set l [llength $GidData($TYPE,OPTIONS,$Idx)]
            if { $i < $l } {
                set GidData($TYPE,VALUE,$Idx) [lindex $GidData($TYPE,OPTIONS,$Idx) $i]
            } else {
                set GidData($TYPE,VALUE,$Idx) [lindex $GidData($TYPE,OPTIONS1,$Idx) [expr $i-$l]]
            }
            set GidData($TYPE,SYNC,$Idx) 1
        }
        # Actualizo el texto del boton
        set GidData($TYPE,CBTEXT,$Idx) [= [DWUnder2Space $GidData($TYPE,VALUE,$Idx)]]
    }
}

proc DWParseCBValue { value chkType } {
    
    set value [string trim $value]
    set re "^(.*)\#(MAT|CB)\#(.*)\$"
    
    set strError ""
    
    if [regexp -nocase $re $value {} value type args] {
        set value [string trim $value]
        set type [string toupper $type]
        set args [string trim $args]
        
        if [string compare $chkType $type] {
            return [list "type mismatch ($type != $chkType): must be $chkType" {} {}]
        }
        
        if [string length $args] {
            if ![regexp \\((.+)\\)\$ $args {} args] {
                return [list "invalid arguments in $type field." {} {}]
            }
            set _args [split $args ","]
            if { [llength $_args] == 1 } {
                set _args [split $args " "]
            }
            set args {}
            foreach a $_args {
                set a [string trim $a]
                if [string length $a] {
                    lappend args [string trim $a]
                }
            }
        }
        
        if ![string compare $type "MAT"] {
            set infomat [DWGetMaterials $args]
            set strError [lindex $infomat 0]
            set args [lindex $infomat 1]
            
        } elseif ![llength $args] {
            set strError [_ "CB field needs arguments"]
            set value {}
        }
        
        return [list $strError $value $args]
    }
    
    return [list "" $value]
}

proc DWRestoreCBValue { GDN TYPE Idx } {
    upvar #0 $GDN GidData
    
    set loptions $GidData($TYPE,OPTIONS,$Idx)
    if ![string compare $GidData($TYPE,TYPE,$Idx) "LA"] {
        lappend loptions $GidData($TYPE,OPTIONS1,$Idx)
    }
    set GidData($TYPE,VALUE,$Idx) [lindex $loptions $GidData($TYPE,CURSEL,$Idx)]
}

# Este es el trace lanzado por VALUE,
# por ejemplo desde una dependencia.

proc DWOnChangeCBVALUE { GDN TYPE Idx } {
    upvar #0 $GDN GidData
    
    if $GidData($TYPE,SYNC,$Idx) {
        set GidData($TYPE,SYNC,$Idx) 0
        set changeOptions 0
        set co -1
        
        if [string compare $GidData($TYPE,TYPE,$Idx) "LA"] {
            # Es #MAT# o #CB#
            
            # No usar ParseType, permitir valor vacio ==> mantener el valor actual
            
            set cbInfo [DWParseCBValue $GidData($TYPE,VALUE,$Idx) $GidData($TYPE,TYPE,$Idx)]
            
            if ![string length [set strError [lindex $cbInfo 0]]] {
                
                set cv [lindex $cbInfo 1]
                if { [llength $cbInfo] > 2 } {
                    # Se modifica el contenido del CB
                    set changeOptions 1
                    set loptions [lindex $cbInfo 2]
                    if [string length $cv] {
                        if { [set co [lsearch -exact $loptions $cv ]] == -1 } {
                            W [concat "GidData($TYPE,QUESTION,$Idx):" \
                                    [_ "option %s not found within set (%s). Using the first one" $cv $loptions]]
                            set co 0
                            set GidData($TYPE,VALUE,$Idx) [lindex $loptions 0]
                        } else {
                            # Value valido
                            set GidData($TYPE,VALUE,$Idx) $cv
                        }
                    } else {
                        # Valor por omision de la lista
                        set cv [lindex $GidData($TYPE,OPTIONS,$Idx) $GidData($TYPE,CURSEL,$Idx)]
                        if { [set co [lsearch -exact $loptions $cv]] == -1 } {
                            # Asumo la primer option
                            set co 0
                            set GidData($TYPE,VALUE,$Idx) [lindex $loptions 0]
                        } else {
                            set GidData($TYPE,VALUE,$Idx) $cv
                        }
                    }
                } else {
                    # No se modifica el contenido de la lista
                    set loptions $GidData($TYPE,OPTIONS,$Idx)
                }
            }
        } else {
            set loptions [concat $GidData($TYPE,OPTIONS,$Idx) $GidData($TYPE,OPTIONS1,$Idx)]
            set cv $GidData($TYPE,VALUE,$Idx)
        }
        
        if $changeOptions {
            set GidData($TYPE,OPTIONS,$Idx) $loptions
        } elseif { [set co [lsearch -exact $loptions $GidData($TYPE,VALUE,$Idx)]] == -1 } {
            set strError [concat "GidData($TYPE,QUESTION,$Idx):" \
                    [_ "option %s not found within set (%s). Using previous value" $cv $loptions]]
        }
        
        if { $co == -1 } {
            # No se pudo encontrar una option valida
            W $strError
            DWRestoreCBValue $GDN $TYPE $Idx
        } else {
            set GidData($TYPE,CURSEL,$Idx) $co
        }
        set GidData($TYPE,SYNC,$Idx) 1
    }
}

proc BuildStringValues { GDN } {
    global $GDN
    upvar #0 $GDN GidData
    # Sincronizar
    set StructName [$GidData(TYPEFUNC) $GDN]
    DWInvokeTkWidget $GDN
    set StringValues ""
    set NumProp $GidData($StructName,numprop)
    for {set i 1} {$i <= $NumProp} {incr i 1} {
        set field_value [Field2String $GDN $StructName $i]
        if {$GidData($StructName,TYPE,$i) eq "GRID"} {
            foreach it $field_value {
                lappend StringValues $it
            }
        } else {
            lappend StringValues $field_value
        }
    }    
    return $StringValues
}

#like BuildStringValues but not flat "GRID" fields, to maintain the amount of items
proc BuildListValues { GDN } {
    global $GDN
    upvar #0 $GDN GidData    
    # Sincronizar    
    set StructName [$GidData(TYPEFUNC) $GDN]    
    DWInvokeTkWidget $GDN    
    set list_values ""        
    set NumProp $GidData($StructName,numprop)    
    for {set i 1} {$i <= $NumProp} {incr i 1} {
        set field_value [Field2String $GDN $StructName $i]
        lappend list_values $field_value
    }    
    return $list_values
}


proc DWSendTKWIDGETEvent {GDN TYPE idx Ev args} {
    upvar #0 $GDN GidData
    
    if {[catch {set result [{*}$GidData($TYPE,TKWIDGET,$idx) $Ev $GDN $TYPE $GidData($TYPE,QUESTION,$idx) {*}$args]} msg]} {
        # follow up the error
        error [_ "Error in tcl script sending event %s to %s: %s" $Ev $GidData($TYPE,TKWIDGET,$idx) $::errorInfo]
    } else {
        set len_res [llength $result]
        if { $len_res > 1 } {
            if { $len_res != 2 } {
                W [_ "Error '%s' in result from %s event: must be a 2-element list" $result $Ev]
            } else {
                set status [string toupper [lindex $result 0]]
                set isERROR [string equal $status "ERROR"]
                if {$isERROR || ($status eq "WARNING") } {
                    set msg [lindex $result 1]
                    if [string length $msg] {
                        W $msg
                    }
                    if $isERROR {
                        return CANCEL
                    }
                } else {
                    W [_ "Invalid status '%s' in %s event: must be ERROR or WARNING" $status $Ev]
                }
            }
        }
    }
    return OK
}

proc DWInvokeTkWidget { GDN } {
    upvar #0 $GDN GidData    
    set StructName [$GidData(TYPEFUNC) $GDN]        
    if { $GidData($StructName,SYNC,TKWIDGET) } {
        foreach iproc [array names GidData $StructName,TKWIDGET*] {
            set itk 0
            regexp (?:.)TKWIDGET,(.+) $iproc {} itk
            if { $itk } {
                set QUESTION $GidData($StructName,QUESTION,$itk)
            } else {
                set QUESTION 0
            }
            set result ""
            if [catch [list {*}$GidData($iproc) SYNC $GDN $StructName $QUESTION] result] {
                # follow up the error
                error [_ "Error in tcl script sending event SYNC to %s: %s" $GidData($iproc) $::errorInfo]
            } else {
                set len_res [llength $result]
                if { $len_res > 1 } {
                    if { $len_res != 2 } {
                        W [_ "Error '%s' in result from SYNC event: must be a 2-element list" $result]
                    } else {
                        set status [string toupper [lindex $result 0]]
                        set isERROR [string equal $status ERROR]
                        if { $isERROR || [string equal $status WARNING] } {
                            set msg [lindex $result 1]
                            if [string length $msg] {
                                W $msg
                            }
                            if $isERROR {
                                error CANCEL
                            }
                        } else {
                            W [_ "Invalid status '%s' in SYNC event: must be ERROR or WARNING" $status]
                        }
                    }
                }
            }
        }
    }
}

proc DW_MakeValue { str } {
    set value [string trim $str]
    set err_cond [catch {
            if {[llength $value] != 1} {
                set value [list $value]
            }
        }]
    if {$err_cond} {
        set value [list $value]
    }
    set value
}

proc Field2String { GDN StructName i } {
    upvar #0 $GDN GidData
    
    set FieldType $GidData($StructName,TYPE,$i)
    
    switch $FieldType {
        GRID {
            set NumCells $GidData($StructName,VALUE,$i,0)
            set value [list "#N#" $NumCells]
            for {set j 1} {$j <= $NumCells} {incr j 1} {
                lappend value [DW_MakeValue $GidData($StructName,VALUE,$i,$j)]
            }
        }
        UNITS {
            if { [info exists GidData($StructName,INSTANCE,$i)] &&
                $GidData($StructName,INSTANCE,$i) != ""} {
                if {[$GidData($StructName,INSTANCE,$i)\::ValidateReal]==-1} {
                    error CANCEL
                }
            }
            set value [string trim $GidData($StructName,VALUE,$i)]
        }
        ENTRY {
            set value [string trim $GidData($StructName,VALUE,$i)]
        }
        default {
            set value [string trim $GidData($StructName,VALUE,$i)]
        }
    }
    set value
}


# Eventos disparados en la ventana de condiciones.

# ------------------------------------------------------------------------------
# DWPostEntitiesCnd --
# ------------------------------------------------------------------------------

proc DWPostEntitiesCnd { GDN m } {
    $m entryconfigure 0 -label [DWGetGUICondName $GDN]    
    if { [$m index end] >= 2 } {
        $m entryconfigure 2 -label [concat [DWGetGUICondName $GDN] [_ "all intervals"]]
    }
}

# --------------------------------------------------------------------------
# DWPostDrawCond --
# --------------------------------------------------------------------------

proc DWPostDrawCond { GDN m } {
    upvar #0 $GDN GidData
    
    if { !$GidData(menudraw,needpost) ||
        [$GidData(CND,combo) get] == -1 } {
        return
    }
    set GidData(menudraw,needpost) 0
    set _CndName [DWGetGIDCondName $GDN]
    
    $m delete 0 end
    if {$GidData(BOOK) ne "Default"} {
        # draw book options
        if { $GidData(BOOK,HASLA) } {
            $m add cascade -label [_ "All %s" $GidData(CND,plural)] -menu $m.book
            if { ![winfo exists $m.book] } {
                menu $m.book -borderwidth 1 -activeborderwidth 1
            } else {
                $m.book delete 0 end
            }
            $m.book add command -label [_ "Exclude local axes"] -command [list DWDrawConds $GDN thisbook]
            $m.book add command -label [_ "Only local axes"] -command [list DWDrawConds $GDN thisbookLA]
            $m.book add command -label [_ "Include local axes"] -command [list DWDrawConds $GDN thisbookLAC]
        } else {
            $m add command -label [_ "All %s" $GidData(CND,plural)] \
                -command [list DWDrawConds $GDN thisbook]
        }
    }
    # thiscond options
    if $GidData(CND,$_CndName,HASLA) {
        $m add cascade -label [DWGetGUICondName $GDN] -menu $m.cond
        if { ![winfo exists $m.cond] } {
            menu $m.cond -borderwidth 1 -activeborderwidth 1
        } else {
            $m.cond delete 0 end
        }
        $m.cond add command -label [_ "Exclude local axes"] -command [list DWDrawConds $GDN thiscond]
        $m.cond add command -label [_ "Only local axes"] -command [list DWDrawConds $GDN thiscondLA]
        $m.cond add command -label [_ "Include local axes"] -command [list DWDrawConds $GDN thiscondLAC]
    } else {
        $m add command -label [DWGetGUICondName $GDN] \
            -command [list DWDrawConds $GDN thiscond]
    }
    $m add command -label [_ "Colors"] -command [list DWDrawConds $GDN colors]
    $m add cascade -label [_ "All conditions"] -menu $m.all
    if { ![winfo exists $m.all] } {
        menu $m.all -borderwidth 1 -activeborderwidth 1
    } else {
        $m.all delete 0 end
    }
    $m.all add command -label [_ "Exclude local axes"] \
        -command [list DWDrawConds $GDN allconds]
    $m.all add command -label [_ "Only local axes"] \
        -command [list DWDrawConds $GDN allcondsLA]
    $m.all add command -label [_ "Include local axes"] \
        -command [list DWDrawConds $GDN allcondsLAC]
    
    $m add cascade -label [_ "Field's value"] -menu $m.mfield
    $m add cascade -label [_ "Field's color"] -menu $m.mcolor
    if { ![winfo exists $m.mfield] } {
        menu $m.mfield -borderwidth 1 -activeborderwidth 1 -postcommand [list DWPostDrawField $GDN $m.mfield thiscond]
    }    
    if { ![winfo exists $m.mcolor] } {
        menu $m.mcolor -borderwidth 1 -activeborderwidth 1 -postcommand [list DWPostDrawField $GDN $m.mcolor colors]
    }
}

#mode must be colors or thiscond
proc DWPostDrawField { GDN m mode } {
    upvar #0 $GDN GidData
    set cnd [DWGetGIDCondName $GDN]
    $m delete 0 end    
    foreach i [array names GidData CND,$cnd,QUESTION,*] {
        regexp {([^\#]+)(.*)} $GidData($i) {} _field
        set num [lindex [split $i ,] end]
        set name CND,$cnd
        if { [info exists GidData($name,STATE,$num)] && $GidData($name,STATE,$num) == "hidden"} {
            #not to show
        } else {
            set label [= [DWUnder2Space $_field]]
            $m add command -label $label -command [list DWDrawConds $GDN $mode $_field]
        }
    }    
}

# Metodo (evento) DWAssingCond
#        Asigna las condiciones a ?. Es invocado por el boton
#        Asign de la ventana de condiciones que se crea en
#        DWCondWindow.
# Argumentos:
#        w  --> ventana desde donde fue invocado.
# Resultado:
#

proc DWAssignCond { GDN {l ""} } {
    global $GDN
    upvar #0 $GDN GidData
    
    set w $GidData(WINDOW)
    
    set _CndName [DWGetGIDCondName $GDN]
    
    if [catch "BuildStringValues $GDN" Values] {
        if [string equal $Values CANCEL] {
            return
        } else {
            error $Values
        }
    }
    
    if { $l != "" } {
        GiD_Process Mescape Data Conditions AssignCond $_CndName Change {*}$Values $l escape
    } else {
        GiD_Process Mescape Data Conditions AssignCond $_CndName Change {*}$Values
        set SmallWinSelecting [GiD_Set SmallWinSelecting]
        FinishButton $w $w.but.assign [_ "Press 'Finish' to end selection"] \
            [list DWAfterFinishButton $GDN] disableall $SmallWinSelecting
    }
    
    # lower $w .gid
}

# not remove this method, because is used by some problemtypes, like ramseries 5.9.5
#
# Metodo (evento)DWEntitiesCond
#        ?. Es invocado por el boton
#        Entities de la ventana de condiciones que se crea en DWCondWindow.

proc DWEntitiesCond { GDN ThisOrAll } {
    DWShowCondEntitiesWin $GDN $ThisOrAll
}

proc DWEntityCondAlongIntervals { GDN what } {
    global $GDN
    upvar #0 $GDN GidData
    set viewmode [GiD_Info Project ViewMode]
    if { $viewmode == "GEOMETRYUSE" } {
        set type geometry
    } else {
        set type mesh
    }
    set num_intervals [lindex [GiD_Info intvdata num] 1]
    if { $what == "allcond" } {
        set condition_names {}
        foreach over {over_point over_line over_surface over_volume over_layer over_group} {
            lappend condition_names {*}[GiD_Info conditions $over]
        }
        set title [_ "All conditions all intervals"]
    } else {
        set condition_name [DWGetGIDCondName $GDN]
        lappend condition_names $condition_name
        set title [concat [_ $condition_name] [_ "all intervals"]]
    }   
    WarnWinText [list [_ "Entity"] [_ "Interval"] [_ "Values"]] $title
    foreach condition_name $condition_names {
        set id_type "integer"
        for {set interval 1} {$interval <= $num_intervals} {incr interval} {
            foreach item [GiD_Info conditions -interval $interval $condition_name $type] {
                set on [lindex $item 0]
                if { $on == "N" || $on == "E" } {
                    set id [lindex $item 1]
                } elseif { $on == "L" } {
                    set id [lindex $item 1]
                    set id_type "string"
                } else {
                    set id [lrange $item 0 1]
                    set id_type "two_integers"
                }
                set values [lrange $item 3 end]
                lappend entity_values($id) [list $interval $values]
            }
        }
        if { [array exists entity_values] } {
            WarnWinText $condition_name
            set ids [array names entity_values] 
            if { $id_type =="integer" } {
                set ids [lsort -integer $ids]
            } elseif { $id_type =="string" } {
                set ids [lsort -dictionary $ids]
            } elseif { $id_type =="two_integers" } {
                set ids [lsort -index 0 -integer $ids]
            }
            foreach id $ids {
                foreach item $entity_values($id) {
                    lassign $item interval values
                    WarnWinText [list $id $interval {*}$values]
                }
            }
            array unset entity_values
        }
    }
}


# DWUnassignSome --
#
# Arguments:
#
# Results:

proc DWUnassignSome { GDN {l ""}} {
    global $GDN
    upvar #0 $GDN GidData
    
    set w $GidData(WINDOW)
    
    if [string length $l] {
        GiD_Process Mescape Data Conditions AssignCond [DWGetGIDCondName $GDN] UnAssign $l escape
    } else {
        GiD_Process Mescape Data Conditions AssignCond [DWGetGIDCondName $GDN] UnAssign
        
        set SmallWinSelecting [GiD_Set SmallWinSelecting]
        FinishButton $w $w.but.assign [_ "Press 'Finish' to end selection"] \
            [list DWAfterFinishButton $GDN] disableall $SmallWinSelecting
    }
}


# DWUnassignAllThis --
#
# Arguments:
#
# Results:

proc DWUnassignAllThis { GDN } {
    global $GDN
    upvar #0 $GDN GidData
    
    set w $GidData(WINDOW)
    set singular $GidData(CND,singular)
    set aa [MessageBoxOptionsButtons [_ "Unassign conditions"] \
            [_ "Sure, unassign all current %s's entities for this interval?" $singular]\
            {0 1} [list [_ "OK"] [_ "Cancel"]] question ""]
    
    if { $aa == 1 } { return }
    
    GiD_Process Mescape Data Conditions AssignCond [DWGetGIDCondName $GDN] DeleteAll Yes escape
}

# DWUnassignAllCND --
#
# Arguments:
#
# Results:

proc DWUnassignAllCND { GDN } {
    upvar #0 $GDN GidData    
    set w $GidData(WINDOW)    
    set aa [MessageBoxOptionsButtons [_ "Unassign conditions"] \
            [_ "Sure, unassign all conditions' entities?"] \
            {0 1} [list [_ "OK"] [_ "Cancel"]] question ""]
    
    if { $aa == 1 } { return }
    
    GiD_Process Mescape Data Conditions UnAssign Yes escape
}

# DWUnassignAllBookCND --
#
# Arguments:
#
# Results:

proc DWUnassignAllBookCND { GDN } {
    upvar #0 $GDN GidData
    
    set w $GidData(WINDOW)
    set plural $GidData(CND,plural)
    set Plural $GidData(CND,plural)
    set aa [MessageBoxOptionsButtons [_ "Unassign %s" $Plural] \
            [_ "Sure, unassign all %s' entities?" $plural] \
            {0 1} [list [_ "OK"] [_ "Cancel"]] question ""]
    
    if { $aa == 1 } { return }
    
    
    foreach idx [array names GidData CND,*,GIDNAMES] {
        foreach cnd $GidData($idx) {
            GiD_Process Mescape Data Conditions AssignCond $cnd DeleteAll Yes escape
        }
    }
}


# Eventos disparados en la ventana de intervalos.

# Evento DWNewInterval --
#    Es invocado por el boton NEW del dialogo de los intervalos. Agrega
#    un nuevo intervalo. Refresca el combo de los intervalos.
#
# Argumentos:
#
# Resultados:
#    Adicioan un nuevo intervalo y actualiza el combo de los intervalos.
#
proc DWNewInterval { GDN insert } {
    global $GDN
    upvar #0 $GDN GidData
    
    if { $insert } {
        set i_insertion $GidData(ID,current)        
        set title [_ "Insert new interval in position %s" $i_insertion]        
    } else {
        set title [_ "Append new interval"]        
    }    
    set answer [MessageBoxOptionsButtons $title \
            [_ "Copy conditions' entities from interval %s?" $GidData(ID,current)] \
            {0 1 2} [list [_ "Yes"] [_ "No#C#I don't want to do that"] [_ "Cancel"]] question ""]
    switch $answer {
        0 {
            set copy_conditions Yes
        }
        1 {
            set copy_conditions No
        }
        default {
            return
        }
    }       
    if { $insert } {
        GiD_Process Mescape Data Intervals NewInterval $i_insertion Yes $copy_conditions
    } else {
        GiD_Process Mescape Data Intervals NewInterval Yes $copy_conditions
    }
}

proc DWOnAddNewInterval { GDN } {
    upvar #0 $GDN GidData
    
    set cb $GidData(ID,combo)
    set values [$cb cget -values]
    set new_max [expr [lindex $values end]+1]
    lappend values $new_max
    $cb configure -values $values
    
    # cambiar el contenido de la ventana al nuevo intervalo
    
    $cb set $new_max
    # quizas un after idle para cambiar el copntenido al intervalo activo!
}

proc DWOnDeleteInterval { GDN I } {
    upvar #0 $GDN GidData
    
    set cb $GidData(ID,combo)
    $cb configure -values [lrange [$cb cget -values] 0 end-1]
    set sel [expr $I-1]
    if { $sel < 1 } {
        #now allow delete first interval
        set sel 1
    }
    $cb set $sel
    if { $GidData(ID,current) == $I } {
        # DW contents should change
        DWOnSelectInterval $GDN $sel
    }
}

# Evento DWIntervalAccept --
#    Es invocado por el boton Accept del dialogo de los intervalos. Acepta
#    los cambios realizado al intervalo actual.
#
# Argumentos:
#    w --> ventana desde donde se invoca.
#
# Resultados:
#    Le informa GID de los cambios en los values del intervalo actual.

proc DWIntervalAccept { GDN } {
    global $GDN 
    global TmpGidData
    upvar #0 $GDN GidData
    
    # Hay que guardar el intervalo activo.
    
    set ISaved [DWGetActiveInterval]
    
    # Activar el GidData(ID,currentinterval)
    # ya que GenericFillGD trabaja con el intervalo activo
    
    DWSetActiveInterval $GidData(ID,current)
    
    set result ""
    if { [catch "DWInvokeTkWidget $GDN" result] } {
        DWSetActiveInterval $ISaved
        if [string equal $result CANCEL] {
            return
        } else {
            error $result
        }
    }
    
    # validate the units values
    
    if { [catch "DWValidateUnitFields $GDN" result] } {
        DWSetActiveInterval $ISaved
        if [string equal $result CANCEL] {
            return
        } else {
            error $result
        }
    }
    
    GenericFillGDFull $GDN ID TmpGidData
    
    # este no lanza InvokeTkWidget, garantizado por GenericFillGDFull
    set Values [BuildStringValues TmpGidData]
    
    GiD_Process Mescape Data IntervalData {*}$Values
    
    set GidData(ID,SYNC,TKWIDGET) 0
    set GidData(FIRSTSTATE) [BuildStringValues $GDN]
    set GidData(ID,SYNC,TKWIDGET) 1
    
    FloatMessage $GidData(ID,mainframe).mess [_ "%s modified" $GidData(ID,plural)]
    
    DWSetActiveInterval $ISaved
    
    unset TmpGidData
}

proc DWGetActiveInterval { } {
    lindex [GiD_Info intvdata num] 0
}

proc DWSetActiveInterval { I } {
    GiD_Process Mescape Data Intervals ChangeInterval $I
}

proc DWDeleteActiveInterval { } {
    GiD_Process Mescape Data Intervals DeleteInterval
}

# Eventos disparados en la ventana de materiales

# ------------------------------------------------------------------------------
# DWPostDrawMaterial --
# ------------------------------------------------------------------------------

proc DWPostDrawMaterial {GDN m} {
    $m entryconfigure 0 -label [DWGetGUIMaterial $GDN]
}

# PostCmdAssignMat --
#
# Arguments:
#
# Results:

proc PostCmdAssignMat { GDN w Menu } {
    upvar #0 $GDN GidData
    
    set geom_labels [GetGeometryLabels]
    set geom_commands [GetGeometryCommands]
    set mesh_labels [list [_ "Elements"]]
    set mesh_commands {Elements}
    
    $Menu delete 0 end
    if { [catch { set whatuse [GiD_Info Project ViewMode] }] } {
        set whatuse "GEOMETRYUSE"
    }
    if {$whatuse eq "GEOMETRYUSE"} {
        set etype "geomlist"
        set elist [GetGeometryCommands]
        set elist_names [GetGeometryLabels]
    } else {
        set etype "meshlist"
        set elist [list Elements]
        set elist_names [list [_ "Elements"]]
    }
    set b_ent [lindex [::DataBehaviour::Materials 1 $GidData(BOOK) $etype] 1]
    set m_ent [lindex [::DataBehaviour::Materials 0 [DWGetGIDMaterial $GDN] $etype] 1]
    foreach i $elist j $elist_names {
        set e_t [string tolower $i]
        if {($m_ent eq "default" && ($b_ent eq "default" ||
            [lsearch $b_ent $e_t]!=-1)) ||
            ([lsearch $m_ent $e_t] != -1)} {
            $Menu add command -label $j -command [list DWAssignMaterials $GDN $w $i]
        }
    }
}

# DWAssignMat --
#
# Arguments:
#
# Results:

proc DWAssignMaterials  { GDN w type } {
    global $GDN
    upvar #0 $GDN GidData
    set _MatName [DWGetGIDMaterial $GDN]
    
    if [catch "CheckDirtyState $GDN" result] {
        if [string equal $result CANCEL] {
            return
        } else {
            error $result
        }
    }
    
    if { $_MatName == ""} {
        WarnWin [_ "Sorry, but first a %s must be selected" $GidData(MAT,singular)] $w
        return
    }
    
    if { $type != "Elements" } {
        GiD_Process Mescape Data Materials AssignMaterial $_MatName $type
    } else {
        GiD_Process Mescape Data Materials AssignMaterial $_MatName
    }
    set SmallWinSelecting [GiD_Set SmallWinSelecting]
    FinishButton $w $w.but.assign [_ "Press 'Finish' to end selection"] \
        "DWAfterFinishButton $GDN" disableall $SmallWinSelecting
    
    # lower $w .gid
}

# DWDrawMaterials --
#
# Arguments:
#
# Results

proc DWDrawMaterials { GDN w what } {
    global $GDN
    upvar #0 $GDN GidData
    
    switch $what {
        onemat {
            GiD_Process Mescape Data Materials DrawMaterial [DWGetGIDMaterial $GDN]
        }
        allmats {
            GiD_Process Mescape Data Materials DrawMaterial -drawall-
        }
    }
    set SmallWinSelecting [GiD_Set SmallWinSelecting]
    FinishButton $w $w.but.assign [_ "Press 'Finish' to end selection"] \
        "DWAfterFinishButton $GDN" disableall $SmallWinSelecting
}

# Evento DWUnassignMat --
#    Es invocado por el boton Unassign del dialogo de los materiales. Hace ?
#
# Argumentos:
#    w --> ventana desde donde se invoca.
#
# Resultados:
#    ?

proc DWUnassignMat {w GDN} {
    upvar #0 $GDN GidData        
    set MatName [DWGetGUIMaterial $GDN]
    set _MatName [DWGetGIDMaterial $GDN]
    set Book $GidData(BOOK)    
    
    set m $w.but.unassign.menu
    $m delete 0 end
    set sm $m.ent
    $m add cascade -label [_ "Entities"] -menu $sm
    if {![winfo exists $sm]} {
        menu $sm -borderwidth 1 -activeborderwidth 1 -postcommand [list PostCmdUnassignSomeMat $w $GDN $sm]
    }
    set singular $GidData(MAT,singular)
    $m add command -label $MatName -command [list DWUnassignAllThisMat $w $GDN]
    if { $Book != "Default" } {
        $m add command -label [_ "All book %s" [= [DWUnder2Space $Book]]] -command [list DWUnassignAllBookMat $w $GDN]
    }
    $m add command -label [_ "All materials"] -command [list DWUnassignAllMat $w $GDN]
}

proc PostCmdUnassignSomeMat { w GDN Menu } {
    set geom_labels [GetGeometryLabels]
    set geom_commands [GetGeometryCommands]
    set mesh_labels [list [_ "Elements"]]
    set mesh_commands {}
    
    $Menu delete 0 end
    switch [GiD_Info Project ViewMode] {
        GEOMETRYUSE {
            foreach i $geom_commands j $geom_labels {
                $Menu add command -label $j -command [list DWUnassignSomeMat $w $GDN $i]
            }
        }
        MESHUSE {
            foreach i $mesh_commands j $mesh_labels {
                $Menu add command -label $j -command [list DWUnassignSomeMat $w $GDN $i]
            }
        }
    }
}

proc DWUnassignSomeMat { w GDN type} {
    GiD_Process Mescape Data Materials UnAssign Entities $type
    set SmallWinSelecting [GiD_Set SmallWinSelecting]
    FinishButton $w $w.but.assign [_ "Press 'Finish' to end selection"] \
        "DWAfterFinishButton $GDN" disableall $SmallWinSelecting
}

proc DWUnassignAllThisMat { w GDN } {
    upvar #0 $GDN GidData    
    set _MatName [DWGetGIDMaterial $GDN]
    set MatName [DWGetGUIMaterial $GDN]
    if { $MatName == ""} {
        WarnWin [_ "Sorry, but first a material must be selected"] $w
        return
    }
    
    set aa [MessageBoxOptionsButtons [_ "Unassign materials"] \
            [_ "Sure, you want to unassign material '%s' of all entities?" $MatName] \
            {0 1} [list [_ "OK"] [_ "Cancel"]] question ""]
    if { $aa == "0" } {
        GiD_Process Mescape Data Materials UnAssign SomeMat $_MatName
    }
}

proc DWUnassignAllBookMat { w GDN } {
    upvar #0 $GDN GidData    
    set MatName [DWGetGUIMaterial $GDN]
    set Book $GidData(BOOK)
    
    if { $Book == "Default"} {
        WarnWin [_ "Sorry, but first a material book must be selected"] $w
        return
    }
    
    set aa [MessageBoxOptionsButtons [_ "Unassign %s" $GidData(MAT,singular)] \
            [_ "Sure, you want to unassign all book %s?" [= $Book]] \
            {0 1} [list [_ "OK"] [_ "Cancel"]] question ""]
    if { $aa == "0" } {
        GiD_Process Mescape Data Materials UnAssign SomeBook $Book
    }
}

proc DWUnassignAllMat { w GDN } {
    GiD_Process Mescape Data Materials UnAssign AllMat
}



# Evento DWImportExport --
#
# Arguments:
#
# Results:
#

proc DWMatImportExport { w Book } {
    set filename [MessageBoxGetFilename file read [_ "Load Database"] \
                      "" [list [list [_ "Materials"] ".mat"] [list [_ "All files"] ".*"]] {} 0]
    if { [string compare $filename ""] != 0 } {
        saveMaterialsWindow $filename $Book
    }
}

# Evento DWModifyMat --
#    Es invocado por el boton Modify del dialogo de los materiales. Debe
#    pasar los valores editados al GID.
#
# Argumentos:
#    w --> ventana desde donde se invoca.
#
# Resultados:
#    Envia los valores de los campos a GID utilizando el pipeline de process.
#    Los valores de los campos se obtienen de BuildStringValues

proc DWModifyMat { GDN } {
    global $GDN
    upvar #0 $GDN GidData
    set _MatName [DWGetGIDMaterial $GDN]
    if { $_MatName == "" } {
        return
    }
    if { [catch {set Values [BuildStringValues $GDN]} msg] } {
        if [string equal $msg CANCEL] {
            return
        } else {
            error $msg
        }
    }    
    GiD_Process Mescape Data Materials NewMaterial $_MatName $_MatName Yes {*}$Values escape
    FloatMessage $GidData(WINDOW)\.level.mess [_ "%s %s modified" $GidData(MAT,singular) [DWGetGUIMaterial $GDN]]
    set GidData(FIRSTSTATE) $Values    
}


proc DWGetAutomaticMaterialName { {prefix ""} } {
    set name ""
    set i 2
    if { $prefix == "" } {
        set prefix material
    } else {
        set pos [string last - $prefix]
        if { $pos != -1 } {
            set last_part [string range $prefix [expr $pos+1] end]
            if { [string is integer -strict $last_part] } {
                set i [expr $last_part+1]
                set prefix [string range $prefix 0 [expr $pos-1]]
            }
        }
    }
    set material_names [GiD_Info materials]    
    for {} {$i<10000} {incr i} {
        set fullname ${prefix}-${i}   
        if { [lsearch -exact $material_names $fullname] == -1 } { 
            set name $fullname
            break 
        }
    }
    return $name
}

# Evento DWNewMat --
#    Es invocado por el boton New del dialogo de los materiales. Debe
#    create a new material from another with its same fields.
#
# Argumentos:
#    w --> ventana desde donde se invoca.
#
# Resultados:
#    ?


# Revisa el codigo este

proc DWNewMat { GDN w } {
    global $GDN
    upvar #0 $GDN GidData
    
    set material_base_name [DWGetGIDMaterial $GDN]
    if { $material_base_name == "" } {
        return
    }
    set materialstr $GidData(MAT,singular)

    set title [_ "New %s" $materialstr]
    set question [_ "Enter new %s name" $materialstr]
    set new_name [DWGetAutomaticMaterialName $material_base_name]
    set new_name [MessageBoxEntry $title $question any $new_name 0 question.png]
    if { $new_name != "--CANCEL--" } {
        set NewMatName_translated [= $new_name]        
        # Verificar que el material no exista        
        if { ![catch [list GiD_Info materials $new_name]] } {
            WarnWin [_ "Invalid new %1\$s name: %2\$s already exists in data base.\nTo override it use Modify" \
                    $materialstr $NewMatName_translated]
        } else {            
            if { [AddNewMaterial $GDN $new_name $material_base_name] } {
                # inherit the data behaviour
                set base_props [DataBehaviour::GetMaterialsOpts]
                lappend base_props geomlist meshlist
                set behaviour {}
                foreach {p v} [GiD_DataBehaviour material $material_base_name $base_props] {
                    if {$v ne "default"} {
                        lappend behaviour $p $v
                    }
                }
                foreach {p v} $behaviour {
                    if {$p eq "meshlist" || $p eq "geomlist"} {
                        GiD_DataBehaviour material $new_name $p $v
                    } else {
                        GiD_DataBehaviour material $new_name $v $p
                    }
                }
                CopyMatGidData $GDN $new_name $material_base_name
                set newLstMat [list {*}[$GidData(MAT,combo) cget -values] $NewMatName_translated]
                $GidData(MAT,combo) configure -values $newLstMat
                $GidData(MAT,combo) set [lindex $newLstMat end]
                DWOnSelectMaterial $GDN
                FloatMessage $w.level.mess [_ "%s %s added" $materialstr $NewMatName_translated]
            }
        }
    }
}

proc DWDeleteMat { GDN } {
    global DataMenuInfo
    upvar #0 $GDN GidData    
    #   set matname $GidData(MAT,matname)
    set matname [DWGetGUIMaterial $GDN]
    if { $matname == "" } {
        return
    }
    set strmes [_ "Are you sure to delete %1\$s %2\$s?" $GidData(MAT,singular) $matname]
    set ret [MessageBoxOptionsButtons [_ "Confirm: %s" $GidData(MAT,plural)] $strmes \
        {0 1} [list [_ "Yes"] [_ "No#C#I don't want to do that"]] question ""]
    if { $ret == 0 } {
        set _matname [DWGetGIDMaterial $GDN]
        GiD_Process Mescape Data Materials DeleteMaterial $_matname
 
        if { [lsearch [GiD_Info materials] $_matname] == -1 } {
            #material was really deleted, remove it from window
            array unset GidData MAT,$_matname,*
            DataBehaviour::DeleteMaterial $_matname
            set combo $GidData(MAT,combo)
            set idx [$combo current]
            $combo configure -values [lreplace [$combo cget -values] $idx $idx]
            set GidData(MAT,materials) [lreplace $GidData(MAT,materials) $idx $idx]
            set n [llength [$combo cget -values]]
            if { $n>0 } {
                if { $idx<$n } {
                    set current $idx
                } else {
                    set current [expr $n-1]
                }        
                #$combo current $current ;#using 'current' raise errors with TTKComboBox
                set currentmatname [lindex [$combo cget -values] $current] 
                $combo set $currentmatname ;#must use the 'visible decorated' name
                set GidData(MAT,cursel) -1
                DWOnSelectMaterial $GDN
            } else {
                destroy $GidData(WINDOW)
                if { $GidData(BOOK) == "Default" } {
                    set i $DataMenuInfo(Materials,entry)
                    $DataMenuInfo(menu) entryconfigure $i -state disabled
                } else {
                    set i [lsearch $DataMenuInfo(Materials) $GidData(BOOK)]
                    $DataMenuInfo(Materials,submenu) entryconfigure $i -state disabled
                }
                unset GidData
            }
        }
    }
}

proc DWRenameMat { GDN } {   
    upvar #0 $GDN GidData
    set old_name [DWGetGIDMaterial $GDN]
    if { $old_name == "" } {
        return
    }
    set old_material_label [DWGetGUIMaterial $GDN]
    set materialstr $GidData(MAT,singular)
    set title [_ "Rename %s" $materialstr]
    set question [_ "Enter new name of %s '%s'" $materialstr $old_material_label]
    set new_name [MessageBoxEntry $title $question any $old_material_label 0 question.png]
    if { $new_name == "--CANCEL--" } {
        return
    }
               
    #check if material exists
    if { [lsearch [GiD_Info materials] $new_name] != -1 } {   
        WarnWin [_ "Invalid new %1\$s name: %2\$s already exists in data base.\nTo override it use Modify" \
                     $materialstr $new_name]
    } else {        
        set book [GiD_Info materials $old_name book]
        GiD_Process Mescape Data Materials RenameMaterial $old_name $new_name escape        
        
        if { [lsearch [GiD_Info materials] $new_name] != -1 } {
            #update window information
            set len [string length MAT,$old_name]
            foreach i [array names GidData MAT,$old_name,*] {
                set suffix [string range $i $len end]
                set j MAT,${new_name}${suffix}
                set GidData($j) $GidData($i)
                if { [string range $suffix 1 6] == "IMGOBJ" } {
                    #transfer the trace that delete the image when unsetting the variable
                    trace remove variable GidData($i) unset [list DWDeleteAssociatedImage $GidData($i)]
                    trace add variable GidData($j) unset [list DWDeleteAssociatedImage $GidData($j)]
                }
                unset GidData($i)          
            }
            DataBehaviour::RenameMaterial $old_name $new_name
            set combo $GidData(MAT,combo)
            set idx [$combo current]
            $combo configure -values [lreplace [$combo cget -values] $idx $idx $new_name]
            set GidData(MAT,materials) [lreplace $GidData(MAT,materials) $idx $idx $new_name]
            set currentmatname [lindex [$combo cget -values] $idx] 
            $combo set $currentmatname ;#must use the 'visible decorated' name
            set GidData(MAT,cursel) -1
            DWOnSelectMaterial $GDN     
            DWBindTrace $GDN MAT,$new_name
        }
    }    
}

# Eventos disparados en la ventana de problemas

# Evento DWProblemAccept --
#    Es invocado por el boton Accept del dialogo de los datos del problema. Hace ?
#
# Argumentos:
#    w --> ventana desde donde se invoca.
#
# Resultados:
#    ?

proc DWProblemAccept { GDN } {
    global TmpGidData
    upvar #0 $GDN GidData
    
    set result ""
    if [catch "DWInvokeTkWidget $GDN" result] {
        if [string equal $result CANCEL] {
            return
        } else {
            error $result
        }
    }
    
    # validate units
    
    if [catch "DWValidateUnitFields $GDN" result] {
        if [string equal $result CANCEL] {
            return
        } else {
            error $result
        }
    }
    
    GenericFillGDFull $GDN PD TmpGidData
    
    # no lanza InvokeTkWidget, garantizado por GenericFill...
    set Values [BuildStringValues TmpGidData]
    
    GiD_Process Mescape Data ProblemData {*}$Values
    
    FloatMessage $GidData(PD,mainframe).mess [_ "%s modified" $GidData(PD,plural)]
    set GidData(PD,SYNC,TKWIDGET) 0
    set GidData(FIRSTSTATE) [BuildStringValues $GDN]
    set GidData(PD,SYNC,TKWIDGET) 1
    
    unset TmpGidData
}

proc DWValidateUnitFields { GDN } {
    upvar #0 $GDN GidData
    
    set TYPE [$GidData(TYPEFUNC) $GDN]
    foreach idx [array names GidData $TYPE,INSTANCE,*] {
        # ValidateReal modifies VALUENUM and this the VALUE
        if {[$GidData($idx)\::ValidateReal]==-1} {
            error CANCEL
        }
    }
}

# -- Procedimientos auxiliares --

# AddNewMaterial --
#
#        Adiciona un nuevo material basado en uno existente.
#
# Argumentos:
#
#        NewMatName --> nombre del nuevo material.
#        BaseMatName --> nombre del material base.
#
# Resultados:
#
#        Agrega dentro de GID un nuevo material. El nombre del
#        nuevo material debe ser distinto del material base. Si
#        la operacion fue exitosa retorna 1, en caso contrario
#        retorna 0.

proc AddNewMaterial { GDN NewMatName BaseMatName } {
    if { $NewMatName != "" } {
        if [catch "BuildStringValues $GDN" Values] {
            if [string equal $Values CANCEL] {
                return 0
            } else {
                error $Values
            }
        }
        GiD_Process Mescape Data Materials NewMaterial $BaseMatName $NewMatName {*}$Values
        return 1
    }
    return 0
}

# CopyMatGidData --
#
#        Crea la estrucra GidData para un nuevo material y copia los
#        valores a partir de un material existente.
#
# Argumentos:
#
#        NewMatName --> nombre del nuevo material.
#        BaseMatName --> nombre del material base.
#
# Resultados:
#
#        GidData contiene la informacion referente al nuevo material.
#        GidData(MAT,$NewMatName,*)
#        GidData(MAT,nameslist) += $NewMatName

proc CopyMatGidData { GDN NewMatName BaseMatName } {
    global $GDN
    upvar #0 $GDN GidData
    lappend GidData(MAT,materials) $NewMatName
    set prefix MAT,$BaseMatName,
    set prefix_length [string length $prefix]
    set prefix_new MAT,$NewMatName,
    set lidx [array names GidData ${prefix}*]
    foreach i $lidx {
        set j ${prefix_new}[string range $i $prefix_length end]
        set GidData($j) $GidData($i)
    }
    #GenScriptDependencies $GDN MAT,$NewMatName
    DWBindTrace $GDN MAT,$NewMatName
}

# globalarray --
#
# Arguments:
#
# Results:
#

proc globalarray { name } {
    global $name
    upvar #0 $name GD
    unset -nocomplain GD
    set GD(0) 1
    unset GD(0)
    return $name
}

proc DWGetWindowName { head bord } {
    set w $head$bord
    while { [winfo exists $w] } {
        set w $w$bord
    }
    return $w
}

proc ExistsDataWindow { GDclass BookOrd } {
    set GDN GD_$GDclass\_$BookOrd
    
    if { [info globals $GDN] != {} } {
        upvar #0 $GDN GidData
        
        if { [info exists GidData(WINDOW)] } {
            set w $GidData(WINDOW)
            if { [winfo exists $w] } {
                return $w
            }
        }
    }
    return ""
}

proc DWDestroyChildren { w } {
    foreach i [winfo children $w] {
    }
}

proc ReplaceDW { GDN NewDWCmd {Book Default} {BookOrd 1} } {
    upvar #0 $GDN GidData
    
    if [catch "DWCheckClose $GDN" result] {
        if [string equal $result CANCEL] {
            return
        }
        W [concat "ReplaceDW:" $result]
        set w $GidData(WINDOW)
        
        set geom [winfo geom $w]
        foreach i [winfo children $w] {
            destroy $i
        }
        
        unset GidData
        set retval [$NewDWCmd [list $Book] $BookOrd $w]
        wm geometry $w $geom
        return $retval
    }
    
    set w $GidData(WINDOW)
    
    set geom [winfo geom $w]
    foreach i [winfo children $w] {
        if { ![winfo exists $i] } {
            continue
        }
        array set gridinfo [grid info $i]
        if {[llength [array names gridinfo]]} {
            set row($gridinfo(-row)) 1
            set column($gridinfo(-column)) 1
        }
        array unset gridinfo
        destroy $i
    }
    foreach i {row column} {
        foreach v [array names $i] {
            grid ${i}configure $w $v -minsize 0 -pad 0 -uniform {} -weight 0
        }
    }
    unset GidData
    set retval [$NewDWCmd $Book $BookOrd $w]
    wm geometry $w $geom
    return $retval
}


proc FillJumpMenu { GDN } {
    global gidDataLabels
    global gidDataBooksHidden
    
    upvar #0 $GDN GidData
    
    
    set DataEntry [list [list [_ "Conditions#C#menu"] "Conditions" \
                conditions CheckCnd DWCondWindow CND] \
            [list [_ "Materials#C#menu"] "Materials"  \
                materials CheckMat DWMatWindow MAT] \
            [list [_ "Interval data#C#menu"] "Interval data" \
                intvdata CheckIntv DWIntervalDataWindow INT] \
            [list [_ "Problem data#C#menu"] "Problem data" \
                gendata CheckPrbD DWProblemDataWindow PD]]
    
    set jb      $GidData(JUMPBUTTON)
    set Class   $GidData(CLASS)
    set Book    $GidData(BOOK)
    set BookOrd $GidData(BOOKORD)
    
    set m [$jb cget -menu]
    
    $m delete 0 last
    
    set UnderChars {}
    
    foreach i $DataEntry {
        set theClass  [lindex $i 1]
        
        # if ![info exists gidDataLabels($theClass)] continue
        
        set isDeleted [expr ![info exists gidDataLabels($theClass)]]
        
        if { $isDeleted || $gidDataLabels($theClass) == "" } {
            set ClassLabel [lindex $i 0]
        } else {
            set ClassLabel $gidDataLabels($theClass)
        }
        set AskBook   [lindex $i 2]
        set CheckFunc [lindex $i 3]
        set DWCmd     [lindex $i 4]
        set GD        [lindex $i 5]
        
        if { [$CheckFunc] != "" } {
            set BookList [GiD_Info $AskBook BOOKS]
            
            if { [llength $BookList] == 0 } {
                if { $isDeleted } continue
                if { $Class == $theClass || [ExistsDataWindow $GD 1] != "" } {
                    set bAddSeparator 0
                } else {
                    set Label [= [DWUnder2Space $ClassLabel]]
                    set UnderIdx [FindUnderChar Label UnderChars]
                    $m add command -label $Label -command [list ReplaceDW $GDN $DWCmd] \
                        -underline $UnderIdx
                    set bAddSeparator 1
                }
            } else {
                set bAddSeparator 0
                set isbook 1
                set bord   1
                
                foreach b $BookList {
                    if { $isbook } {
                        if { [lsearch $gidDataBooksHidden($AskBook) $b] == -1 } {
                            if { $b != $Book && [ExistsDataWindow $GD $bord] == ""} {
                                set Label [= [DWUnder2Space $b]]
                                set UnderIdx [FindUnderChar Label UnderChars]
                                $m add command -label $Label -command [list ReplaceDW $GDN $DWCmd $b $bord] \
                                    -underline $UnderIdx
                                set bAddSeparator 1
                            }
                        }
                        set isbook 0
                        incr bord
                    } else {
                        set isbook 1
                    }
                }
            }
            if { $bAddSeparator } {
                $m add separator
            }
        }
    }
    $m delete last
}


proc AddHelpTagFirst { w } {
    set cur ""
    catch { set cur [$w cget -cursor] }
    if { $cur == "question_arrow" } { set cur "" }
    set tags [list Help $cur]
    bindtags $w [concat $tags [bindtags $w]]
    catch { $w configure -cursor question_arrow }
    foreach i [winfo children $w] {
        AddHelpTagFirst $i
    }
}

proc RemoveHelpTag { w } {
    if { [lindex [bindtags $w] 0] == "Help" } {
        catch { $w configure -cursor [lindex [bindtags $w] 1] }
        bindtags $w [lrange [bindtags $w] 2 end]
    }
    foreach i [winfo children $w] {
        RemoveHelpTag $i
    }
}

proc EndPickHelp { w wh x y } {
    RemoveHelpTag $w
#     set pos [lsearch [bind $wh <ButtonPress-$::gid_right_button>] DisplayGidHelpMenu]
#     if { $pos != -1 } {
#         incr pos 2
#         set text [lindex [bind $wh <ButtonPress-$::gid_right_button>] $pos]
#     } else {
#         set text [_ "There is no help for this item"]
#     }
    set enter_binding [bind $wh <Enter>]
    if { [lindex [lindex $enter_binding 2] 0] == "showTooltip" } {
        set text [lindex [lindex [lindex $enter_binding 2] 2] 0]
    } else {
        set text [_ "There is no help for this item"]
    }
    DisplayGidHelp $wh $text menu $x $y
}

proc PickHelp { w } {
    
    set w [winfo toplevel $w]
    AddHelpTagFirst $w
    bind Help <ButtonPress-1> "EndPickHelp $w %W %X %Y; break"
    bind Help <ButtonRelease-1> "break"
}

proc CreateJumpButton { GDN w col {dir right}} {
    upvar #0 $GDN GidData
    
    set hp $w.hp
    set jb $w.jb
    
    set m  $jb.m

    ttk::button $hp -image [gid_themes::GetImage questionarrow.png small_icons] \
        -takefocus 0 -command [list PickHelp $hp]
    ttk::menubutton $jb -image [gid_themes::GetImage note.png small_icons] \
        -menu $m -underline 0 -direction $dir 
        #-style Horizontal.IconButtonWithMenu
    menu $m -postcommand [list FillJumpMenu $GDN] -borderwidth 1 -activeborderwidth 1
    grid $hp -row 0 -column $col -sticky "snew"
    incr col
    grid $jb -row 0 -column $col -sticky "snew"
    
    set GidData(JUMPBUTTON) $jb
    set GidData(HELPBUTTON) $hp
    
    GidHelp $jb [_ "Open other data option using this window"]
    
    list $hp $jb
}

proc DWDataUnits { } {
    if { [GiD_Info unitssystems] && [ModUnitWin::Init .gid.dataunits] } {
        ModUnitWin::Show
    } else {
        WarnWin [_ "There are not units defined"]
    }
}

# ------------- UTILES ------------

proc FloatMessage { w mes } {
    if [winfo exists $w] {
        return
    }
    ttk::label $w -text $mes -relief ridge
    place $w -x 0 -rely 1 -in [winfo parent $w] -anchor sw
    tkwait visibility $w
    FlashLabel 0 0 $w
}

proc FlashLabel { num black w } {
    if ![winfo exists $w] {
        return
    }
    incr num
    if $black {
        $w configure -foreground black
        set black 0
    } else {
        $w configure -foreground white
        set black 1
    }
    if { $num < 10 } {
        after 200 FlashLabel $num $black $w
    } else {
        destroy $w
    }
}


# ----------------- User functions -----------------

# Procedure GidAddUserDataOptions --
#
#    Add a new entry to the data menu of GiD.
#
# Arguments:
#    L   --> Label of the option. In order to add a separator entry
#            it must be passed the value "---"
#    A   --> script to execute when the option is selected
#    W   --> Position of the option:
#                an integer or end to indicate the last position in the menu.
#                end is the default value if you omite this argument.
#    Acc --> Acceleration key combination.
#
#
# Results:
#

proc GidAddUserDataOptions { L {A ""} {W end} {Acc ""} } {
    global gidUserDataOptions
    
    if { $L == "---" } {
        lappend gidUserDataOptions $W
    } else {
        if ![regexp {^(end|[0-9]+)$} $W] {
            W [_ "Error: Argument position in GidAddUserDataOptions should be an integer or end."]
            return
        }
        lappend gidUserDataOptions [list $L $A $W $Acc]
    }
    
    # fuerzo redibujar el menu de datos.
    
    CacheReset menu_data
}

# Creates a submenu
# action will be converted into a postcommand, substituting %W for the menu name

proc GidAddUserDataOptionsMenu { L {A ""} {W end} } {
    global gidUserDataOptions
    
    if ![regexp {^(end|[0-9]+)$} $W] {
        W [_ "Error: Argument position in GidAddUserDataOptionsMenu should be an integer or end."]
        return
    }
    lappend gidUserDataOptions [list $L $A $W "" menu]
    
    # fuerzo redibujar el menu de datos.
    
    CacheReset menu_data
}

#to remove all GidAddUserDataOptions added items
proc GidClearUserDataOptions { } {
    global gidUserDataOptions
    set gidUserDataOptions {}    
    CacheReset menu_data
}

#to remove an GidAddUserDataOptions added item
#to remove a separator must specify "---$i" where $i is is the index of separator, 
#starting by 0 but taken into account only the ones added by GidAddUserDataOptions
#
#NOTE not all items could be removed, only the ones added with GidAddUserDataOptions 
#(e.g. "Problem type" and first separator can't be removed with this procedure)
proc GidRemoveUserDataOptions { L } {
    global gidUserDataOptions
    if { [string range $L 0 2] == "---" } {
        set nseparator [string range $L 3 end]
        if { $nseparator == "" } {
            set nseparator 0
        }
        set L "---"
        set iseparator 0
    }   
    set index -1
    set count 0
    foreach item $gidUserDataOptions {
        if { [llength $item] == 1 } {
            #it is a separator ---
            if { $iseparator == $nseparator } {
                set index $count                   
                break
            }
            incr iseparator              
        } else {
            if { [lindex $item 0] == $L } {
                set index $count
                break
            }
        }
        incr count
    }    
    if { $index != -1 } {
        #remove item index of the list and decrease index values stored in next items
        set newgidUserDataOptions [list]
        set count 0
        foreach item $gidUserDataOptions {
            if {$count < $index } {
                lappend newgidUserDataOptions $item
            } elseif {$count == $index } {
                #item to be removed
            } else {
                if { [llength $item] == 1 } {
                    #it is the separator index, decrease it
                    incr item -1
                } else {
                    lset item 2 [expr {[lindex $item 2]-1}]
                }
                lappend newgidUserDataOptions $item
            }
            incr count
        }
        set gidUserDataOptions $newgidUserDataOptions        
        CacheReset menu_data
    }   
}


# Procedure GidChangeDataLabel
#    Modify the text of GiD data menus:
#        - Conditions, Materials, Problem data, Interval data,...
# Arguments:
#    GidLabel --> Text key (in english, not translate it!!), could be some of:
#        "Problem type" "Data units" "Interval" "Local axes" "Conditions" "Materials"
#         "Interval data" "Problem data"
#    NewLabel --> New text (translated)
# Results:

proc GidChangeDataLabel { GidLabel NewLabel {Singular ""} } {
    global gidDataLabels
    
    if { $GidLabel == "Problem Data" && $NewLabel == "" } {
        #for back compatibility: old name "Problem Data" is now "Problem data"
        set GidLabel "Problem data"
    } elseif { $GidLabel == "Interval Data" && $NewLabel == "" } {
        #for back compatibility: old name "Interval Data" is now "Interval data"
        set GidLabel "Interval data"
    }

    if { $NewLabel == "" } {
        unset -nocomplain gidDataLabels($GidLabel)
    } else {
        set gidDataLabels($GidLabel) $NewLabel
        if { $Singular == "" } {
            set gidDataLabels($GidLabel,singular) [DWMakeSingular $NewLabel]
        } else {
            set gidDataLabels($GidLabel,singular) $Singular
        }
    }
    CacheReset menu_data
}

proc GiD_ShowBook { class {book ""} {show 1} } {
    global gidDataBooksHidden
    
    set lclass [array names gidDataBooksHidden]
    if { [set idx [lsearch $lclass $class*]] == -1 } {
        W [concat [_ "Data class '%s' not found in" $class] "GiD_ShowBook"]
        return
    }
    set class [lindex $lclass $idx]
    set lbook [GiD_Info $class BOOKS]
    if { $book == "" } {
        if { !$show } {
            set gidDataBooksHidden($class) {}
            foreach {b p} $lbook {
                lappend gidDataBooksHidden($class) $b
            }
        } else {
            set gidDataBooksHidden($class) {}
        }
    } else {
        if { [set idx [lsearch -exact $lbook $book]] == -1 } {
            W [concat [_ "Book '%s' not found in" $book] "GiD_ShowBook"]
            return
        }
        
        set rbook [lindex $lbook $idx]
        
        set idx [lsearch -exact $gidDataBooksHidden($class) $rbook]
        if { !$show } {
            if { $idx == -1 } {
                lappend gidDataBooksHidden($class) $rbook
            }
        } elseif { $idx != -1 } {
            set gidDataBooksHidden($class) [lreplace $gidDataBooksHidden($class) $idx $idx]
        }
    }
}


proc DWFindBookOrdinal { Class Book } {
    
    if {$Class eq "intvdata"} {
        set BooksList [GiD_Info "intvdata" -interval 1 BOOKS]
    } else {
        set BooksList [GiD_Info $Class BOOKS]
    }
    
    if { ![llength $BooksList] } {
        if { $Book == "Default" } {
            return 1
        }
        return 0
    } else {
        set idx [lsearch $BooksList $Book]
        if { $idx == -1 } {
            return 0
        }
        return [expr $idx/2+1]
    }
}

# Procedure GidOpenConditions --
#
#    Open the conditions window for a specific book.
#
# Arguments:
#    Book --> Book's name. If you omit this argument the default will
#        be used
#
# Results:
#

proc GidOpenConditions { {Book "Default"} } {
    set bord [DWFindBookOrdinal conditions $Book]
    if $bord {
        set ::GidData(open,CND,book,$Book) 1
        set GDN [DWCondWindow $Book $bord]
        unset ::GidData(open,CND,book,$Book)
    } else {
        W [concat "GidOpenConditions: Book" $Book [_ "not found in Conditions"]]
        set GDN ""
    }
    return $GDN
}

# Procedure GidOpenMaterials --
#
#    Open de materials window for a specific book.
#
# Arguments:
#    Book --> Book's name. If you omit this argument the default will
#        be used
#
# Results:
#

proc GidOpenMaterials { {Book "Default"} {Material ""} } {
    set bord [DWFindBookOrdinal materials $Book]
    if $bord {
        set ::GidData(open,MAT,book,$Book) 1
        DWMatWindow $Book $bord
        unset ::GidData(open,MAT,book,$Book)
        if { $Material != "" } {
            DWMatChangeCurrent GD_MAT_$bord $Material
        }
    } else {
        W [_ "Book %s not found in Materials" $Book]
    }
}

# Procedure GidOpenIntervals --
#
#    Open de intervals window for a specific book.
#
# Arguments:
#    Book --> Book's name. If you omit this argument the default will
#        be used
#
# Results:
#

proc GidOpenIntervals { {Book "Default"} } {
    set bord [DWFindBookOrdinal intvdata $Book]
    if $bord {
        set ::GidData(open,INT,book,$Book) 1
        DWIntervalDataWindow $Book $bord
        unset ::GidData(open,INT,book,$Book)
    } else {
        W [_ "Book %s not found in Interval data" $Book]
    }
}

# Procedure GidOpenProblemData --
#
#    Open the problem data window for a specific book.
#
# Arguments:
#    Book --> Book's name. If you omit this argument the default will
#        be used
#
# Results:
#

proc GidOpenProblemData { {Book "Default"} } {
    set bord [DWFindBookOrdinal gendata $Book]
    
    if $bord {
        set ::GidData(open,PD,book,$Book) 1
        DWProblemDataWindow $Book $bord
        unset ::GidData(open,PD,book,$Book)
    } else {
        W [_ "Book %s not found in Problem Data" $Book]
    }
}


#like LabelField but with a better name and using the GidData($TYPE,LABEL,$FIELD) cache value if exists
proc DWGetQuestionIndex { GDN STRUCT QUESTION } {
    upvar #0 $GDN GidData
    set question_index -1
    if { [info exists GidData($STRUCT,LABEL,$QUESTION)] } {
        set question_index $GidData($STRUCT,LABEL,$QUESTION)
    } else {
        set question_index [LabelField $GDN $STRUCT $QUESTION]
    }
    return $question_index
}

# -------------------------------------------------------------------
#  Procedure DWLocalSetValue:
#
#      Modifca una QUESTION perteneciente a un "conjunto de datos"
#      abiertos identificados por {GDN STRUCT}. Le asigna el nuevo
#      valor VALUE y cambia el estado si este se especifica.
#      STATE: normal, hidden, disabled
#
# -------------------------------------------------------------------

proc DWLocalSetValue { GDN STRUCT QUESTION VALUE {STATE ""}} {
    upvar #0 $GDN GidData
    set index_question [DWGetQuestionIndex $GDN $STRUCT $QUESTION]
    if { $index_question == -1 } {
        error [_ "Invalid question access in DWSetValue for : %s" $QUESTION]
    } else {
        if {$STATE eq ""} {
            set GidData($STRUCT,VALUE,$index_question) $VALUE
        } else {
            array set action {
                normal   DepActionRESTORE
                hidden   DepActionHIDE
                disabled DepActionDISABLE
                set DepActionSET
            }
            # no state checking!!!
            $action($STATE) $GDN $STRUCT $index_question $VALUE
        }
    }
}

# -------------------------------------------------------------------
#  Procedure DWLocalGetValue:
#
#      Obtiene el valor asociado a una QUESTION perteneciente a un
#     "conjunto de datos" abiertos identificados por {GDN STRUCT}.
#
# -------------------------------------------------------------------

proc DWLocalGetValue { GDN STRUCT QUESTION } {
    upvar #0 $GDN GidData
    set index_question [DWGetQuestionIndex $GDN $STRUCT $QUESTION]    
    if { $index_question == -1 } {
        error [_ "Invalid question access in DWGetValue for : %s" $QUESTION]
    } else {
        return $GidData($STRUCT,VALUE,$index_question)
    }
}

# ------------------------------------------------------------------------------
# DWGetBook --
#
#  Returns the book name associated to this GDN.
# ------------------------------------------------------------------------------

proc DWGetBook { GDN } {
    upvar #0 $GDN GidData
    
    set GidData(BOOK)
}

# ------------------------------------------------------------------------------
# DWGetTopLevel --
#
#  Return the widget path of the toplevel window associated to this GDN.
# ------------------------------------------------------------------------------

proc DWGetTopLevel { GDN } {
    upvar #0 $GDN GidData
    
    set GidData(WINDOW)
}

# ------------------------------------------------------------------------------
# DWGetStruct --
#
#   Used to retrieve the STRUCT argument needed in functions like DWLocalGetValue
#   As you may think this is not necesary if we modify DWLocalGetValue but as this
#   proc is called in olders prbtypes we should keep it for backward compatibility.
# ------------------------------------------------------------------------------
proc DWGetStruct { GDN } {
    upvar #0 $GDN GidData
    
    $GidData(TYPEFUNC) $GDN
}

namespace eval DataBehaviour {
    variable data
    variable properties
    
    array set properties {
        materials {assign draw unassign impexp new modify delete rename}
        geomlist {points lines surfaces volumes}
        meshlist {elements}
        materials,cmds {show disable hide geomlist meshlist}
    }
    #   array set imply {
        #     assign,disable,1
        #  }
}

proc DataBehaviour::GetMaterialsOpts { } {
    variable properties
    
    set properties(materials)
}

proc DataBehaviour::GetEntityList { what } {
    variable properties
    
    set properties($what)
}

proc DataBehaviour::IsValidProperty {data_t prop} {
    variable properties
    
    if {$data_t ne "materials"} {
        error [_ "DataBehaviour only implemented on materials and %s was requested" $data_t]
    }
    expr [lsearch $properties($data_t) $prop] != -1
}

proc DataBehaviour::HumanizeList { items } {
    if {[set ll [llength $items]]<=1} {
        return [lindex $items 0]
    }
    set result {}
    foreach i $items {
        incr ll -1
        if {$ll > 1} {
            append result "${i},"
        } elseif {$ll == 1} {
            append result "${i} or "
        }
    }
    append result [lindex $items end]
}

proc DataBehaviour::ProcessEntityList { key entities } {
    variable properties
    
    set idx 0
    set elist $properties($key)
    foreach e  $elist {
        set list_map($e) 0
        set list_index($idx) $e
        incr idx
    }
    foreach e_ $entities {
        set e [string tolower $e_]
        if {[lsearch $elist $e]==-1} {
            error [_ "invalid entity type '%s' in %s: sould be '%s'" $e_ $key [HumanizeList $elist]]
        }
        set list_map($e) 1
    }
    set result [list]
    foreach i [array names list_index] {
        set e $list_index($i)
        if {$list_map($e)} {
            lappend result $e
        }
    }
    set result
}

proc DataBehaviour::Conditions {name args} {
    error [_ "DataBehaviour not implemented on Conditions"]
}

proc DataBehaviour::Materials {isbook name args} {
    variable data
    variable properties
    
    set key [expr {$isbook?"bookmat":"material"}]
    if {[llength $args]==1} {
        set tprop [lindex $args 0]
        set cmd query
    } else {
        lassign $args cmd_ tprop
        set cmd [string tolower $cmd_]
        if {[lsearch $properties(materials,cmds) $cmd]==-1} {
            error [_ "invalid cmd '%s' in DataBehaviour::Materials: should be '%s'" $cmd [HumanizeList $properties(materials,cmds)]]
        }
        set assign [expr {$cmd eq "meshlist" || $cmd eq "geomlist"}]
    }
    set result [list]
    if {$cmd eq "query"} {
        foreach p_ $tprop {
            set p [string tolower $p_]
            set assign [expr {$p eq "meshlist" || $p eq "geomlist"}]
            if {$assign || [IsValidProperty materials $p]} {
                lappend result $p [expr {[info exists data($key,$name,$p)]?\
                        $data($key,$name,$p):\
                            "default"}]
            } else {
                set tmplist $properties(materials)
                lappend tmplist "geomlist" "meshlist"
                error [_ "invalid property '%s' in DataBehaviour::Materials: should be '%s'" $p_ [HumanizeList $tmplist]]
            }
        }
    } elseif {$assign} {
        set data($key,$name,$cmd) [ProcessEntityList $cmd $tprop]
    } else {
        # enable or disable comands
        foreach p_ $tprop {
            set p [string tolower $p_]
            if {[IsValidProperty materials $p]} {
                set data($key,$name,$p) $cmd
            } else {
                error [_ "invalid property '%s' in DataBehaviour::Materials: should be '%s'" $p_ [HumanizeList $properties(materials)]]
            }
        }
    }
    return $result
}

proc DataBehaviour::RenameMaterial { old_name new_name } {
    variable data
    set len [string length material,$old_name]
    foreach i [array names data material,$old_name,*] {
        set suffix [string range $i $len end]
        set j material,${new_name}${suffix}
        set data($j) $data($i)
        unset data($i)
    }
}

proc DataBehaviour::DeleteMaterial { name } {
    variable data
    array unset data material,$name,*
}

# -----------------------------------------------------------------------------
# DataBehaviour::Condition --
#
# -----------------------------------------------------------------------------

proc DataBehaviour::Condition { name args} {
    variable data
    
    set ll [llength $args]
    if {$ll == 1} {
        set cmd query
        set tprop [lindex $args 0]
    } else {
        foreach {cmd_ tprop} $args break
        set cmd [string tolower $cmd_]
        if {$cmd ne "enable" && $cmd ne "disable"} {
            error [_ "invalid cmd '%s' in DataBehaviour::Conditions: should be 'enable or disable'" $cmd_]
        }
    }
    set result [list]
    set lprop {visible draw unassign}
    foreach p_ $tprop {
        set p [string tolower $p_]
        if {[lsearch $lprop $p]==-1} {
            error [_ "invalid property '%s' in DataBehaviour::Conditions: should be '%s'" $p_ [HumanizeList $lprop]]
        }
        if {$cmd eq "query"} {
            lappend result $p [expr {[info exists data(condition,$name,$p)]?\
                    $data(condition,$name,$p):"default"}]
        } else {
            set data(condition,$name,$p) $cmd
        }
    }
    set result
}

# -----------------------------------------------------------------------------
# GiD_DataBehaviour --
#
#   This command controls properties of data windows for materials and
#   conditions (not currently implemented). For materials we can modify
#   the behaviour of assign, draw, unassign, impexp, new, modify, delete and rename.
#   We can also specify the entity type list in assign option throught the
#   cmds geomlist and meshlist.
#
#   Syntax:
#
#     GiD_DataBehaviour data_class name ?cmd? proplist
#
#     data_class could be: "material" if we want to modify the behaviour of a
#     particular material if whole book must be modified the data_class should
#     take the value "materials". Depending on the value of data_class name
#     takes the value of a material's name or a book's name. Argument cmd can
#     take one of the values: show, hide, disable, geomlist and meshlist.
#     Argument proplist is a list of options or entity type. When cmd is show,
#     hide or disable then proplist could a subset of {assign draw unassign
    #     impexp new modify delete rename}. The command show make the option visible,
#     if the value is hide then the option is not visible and when the value is
#     disable then the option is visible but unavailable. When cmd if geomlist
#     then then property list can take a subset of {points lines surfaces volumes}
#     and define the entities that can get the material assigned when in geometry
#     mode, if the value of cmd is meshlist the proplist can take the value
#     elements (only elements can gets a material assigned in mesh mode).
#
#   Examples:
#
#     GiD_DataBehaviour materials Table geomlist {surfaces volumes}
#     GiD_DataBehaviour materials Solid hide {delete impexp}
#
#  Note: GiD_DataBehaviour is only implemented for material and books of materials.
# -----------------------------------------------------------------------------

proc GiD_DataBehaviour {data_t name args} {
    if {[set ll [llength $args]]>2 || $ll==0} {
        error [_ "wrong # of arguments: should be 'GiD_DataBehaviour data_class name ?cmd? proplist'"]
    }
    if {$name eq ""} {
        error [_ "invalid book name '%s': should be not empty" $name]
    }
    set data_t_ [string toupper $data_t]
    if {$data_t_ eq "CONDITIONS"} {
        set result [DataBehaviour::Conditions $name {*}$args]
    } elseif {$data_t_ eq "MATERIALS"} {
        set result [DataBehaviour::Materials 1 $name {*}$args]
    } elseif {$data_t_ eq "MATERIAL"} {
        set result [DataBehaviour::Materials 0 $name {*}$args]
    } else {
        error [_ "invalid DataBehaviour request on unknown data type '%s': should be 'condition, conditions, material or materials'" $data_t]
    }
    return $result
}

# -----------------------------------------------------------------------------
# GiD_ConditionBehaviour --
#
#   This command controls properties of conditions not defined in .cnd file.
#   The properties are : visible, draw and unassign.
#
#   GiD_ConditionBehaviour condname [property | cmd property]
#
#   cmd could be enable or disable.
#
#   examples
#
#     GiD_ConditionBehaviour Line-Constraints disable visible
#
#   when querying the properties you can provide a list of properties and
#   the result is the state for each property. The return value is a list of pairs
#   name value, like this:
#
#              {visible disable draw default unassign default}
#
#   the value "default" means that nothing has being set for this property.
#
#  Note: draw and unassign can be set or query but
#
# -----------------------------------------------------------------------------

proc GiD_ConditionBehaviour {name args} {
    set ll [llength $args]
    if {$ll < 1 || $ll > 2} {
        error [_ "wrong # of arguments: should be 'GiD_ConditionBehaviour name \[property | cmd property\]'"]
    }
    DataBehaviour::Condition $name {*}$args
}

proc GidUpdateUserDataOptions { UnderChars } {
    global gidUserDataOptions
    global DataMenuInfo
    
    set wmenu_data $DataMenuInfo(menu)
    
    if ![info exists gidUserDataOptions] {
        set gidUserDataOptions {}
    }
    set littleicon ""
    foreach opt $gidUserDataOptions {
        if { [llength $opt] == 1 } {
            $wmenu_data insert $opt separator
        } elseif { [llength $opt] == 5 } {
            set Label [lindex $opt 0]
            set UnderIdx [FindUnderChar Label UnderChars]
            set i 1
            while { [winfo exists $wmenu_data.m$i] } { incr i }
            set menu [menu $wmenu_data.m$i]
            regsub -all {([^%]|^)%W} [lindex $opt 1] "\\1$menu" Action
            $menu configure -postcommand $Action
            set Where [lindex $opt 2]
            if { $Where == "end" } {
                $wmenu_data add cascade -label $Label -underline $UnderIdx -menu $menu
                set Where [$wmenu_data index end]
            } else {
                $wmenu_data insert $Where cascade -label $Label -underline $UnderIdx -menu $menu
            }
            AddLittleIcon $littleicon $wmenu_data $Where
        } else {
            set Label [lindex $opt 0]
            set UnderIdx [FindUnderChar Label UnderChars]
            set Action [lindex $opt 1]
            set Where [lindex $opt 2]
            set Accel [lindex $opt 3]
            if { $Where == "end" } {
                $wmenu_data add command -label $Label -command $Action -underline $UnderIdx
                set Where [$wmenu_data index end]
            } else {
                $wmenu_data insert $Where command -label $Label -command $Action -underline $UnderIdx
            }
            AddAccelerator $Accel $wmenu_data $Where
            AddLittleIcon $littleicon $wmenu_data $Where
        }
    }
}

proc DWGetDepProcName { GDN TYPE Idx } {
    regsub -all , $TYPE "" TYPE
    return $GDN$TYPE\_$Idx
}

proc ProcessOtherField { GDN TYPE OFT OFI OFV } {
    upvar #0 $GDN GidData
    
    set OFT [string toupper $OFT]
    
    switch $OFT {
        TITLE {            
            set GidData($TYPE,TITLEINFO,$OFV,state) normal
            set GidData($TYPE,TITLEINFO,$OFV,path) \
                [regsub -all {[\.\s]} $OFV "_"]
            # it is better to index by name rather than by position
            set GidData($TYPE,TITLEINFO,$OFV,text) [= [DWUnder2Space $OFV]]
        }
        IMAGE {
            if { ![file exist [set f [file join $GidData(DirPT) $OFV]]] } {
                W [_ "File '%s' does not exist" $f]
                return
            }
            if [catch {set GidData($TYPE,IMGOBJ,$OFI) [image create bitmap -file $f]}] {
                if [catch {set GidData($TYPE,IMGOBJ,$OFI) [image create photo -file $f]}] {            
                    W [_ "Image '%s' is not correct" $f]
                    return
                }
            }
            trace add variable GidData($TYPE,IMGOBJ,$OFI) unset [list DWDeleteAssociatedImage $GidData($TYPE,IMGOBJ,$OFI)]
        }
        LABEL {
            set GidData($TYPE,LABEL,$OFV) $OFI
            return
        }
        STATE {
            switch [string toupper $OFV] {
                DISABLED {
                    set OFV disabled
                }
                HIDDEN {
                    set OFV hidden
                }
                NORMAL -
                default {
                    # asumo estado normal
                    set OFV normal
                }
                
            }
        }
        DEPENDENCIES {
            ParserDep::Init $OFV
            if { ![ParserDep::Start] } {
                set WarnStr "Syntax error in dependencies at field $OFI. "
                append WarnStr "Assuming no dependencies\n"
                append WarnStr $ParserDep::ErrFound
                W $WarnStr
                return
            }
            # Si ya existen dependencias asociadas a este campo tenemos que
            # anexarlas.
            
            append GidData($TYPE,DEPSOURCE,$OFI) $ParserDep::DepList " "
            #set GidData($TYPE,DEPENDENCIES,$OFI) $ParserDep::DepList
            return
        }
    }
    set GidData($TYPE,$OFT,$OFI) $OFV
    if {!$OFI} {
        set GidData($TYPE,STATE,$OFI) "normal"
    }
}
