
proc RedrawArc { canvas } {
    global CEPriv
    if { ![winfo exists $canvas] } {
        return
    }    
    foreach value [list $CEPriv(radius) $CEPriv(length) $CEPriv(chordalerror)] {
        if { ![string is double -strict $value] || $value <=0 } {
            return
        }        
    }    

    set width [winfo width $canvas]
    if { $width < 10 } {
        set width [winfo reqwidth $canvas]
    }
    set height [winfo height $canvas]
    if { $height < 10 } {
        set height [winfo reqheight $canvas]
    }

    set r $CEPriv(radius)

    if { $CEPriv(length) < 0 || $CEPriv(length) > 1e5 } {
        set CEPriv(length) 1
    }
    set l $CEPriv(length)
    if { $CEPriv(chordalerror) < 0 || $CEPriv(chordalerror) > 1e5 } {
        set CEPriv(chordalerror) .1
    }
    set CE $CEPriv(chordalerror)

    if { $r == "Inf." } {
        set NE 1
    } else {
        set discr [expr {2.0*$r*$CE-$CE*$CE} ]
        if { $discr > 0 } {
            set size [expr 2.0*sqrt(2.0*$r*$CE-$CE*$CE)]
        } else {
            set size 0
        }
        if { $size == 0 } {
            set NE 1
        } else {
            set NE [expr int(ceil($l/$size))]
        }
        
        if { $NE < 1 } { set NE 1 }
    }
    set CEPriv(numelems) [concat [_ "Mesh elements"]: $NE]

    if { $r != "Inf." } {
        set f [expr double($width)/$l]
        
        set x1 [expr $width/2-$f*$r]
        set x2 [expr $width/2+$f*$r]
        
        set y1 40
        set y2 [expr 40+$f*(2*$r)]
        
        set angle [expr 0.9*180.0/3.1416*$l/$r]
        if { $angle > 360 } { set angle 359 }
        
        $canvas delete temp
        set start [expr 90-$angle/2]
        $canvas create arc $x1 $y1 $x2 $y2 -outline red -start $start \
            -extent $angle -width 2 -style arc -tags temp
        $canvas create line [expr $width/2] 150 [expr $width/2] 42 -arrow last \
            -fill red -tags temp
        $canvas create line [expr $width/2-10] 150 [expr $width/2+10] 150 \
            -fill red -tags temp
        
        set cx [expr ($x1+$x2)/2.0]
        set cy [expr ($y1+$y2)/2.0]
        set rd [expr $f*$r]
        
        if { $NE <= 50 } {
            for { set i 0 } { $i <= $NE } { incr i } {
                set a [expr $i*$angle/$NE]
                set x [expr $cx+$rd*cos(($a+$start)*3.1416/180.0)]
                set y [expr $cy-$rd*sin(($a+$start)*3.1416/180.0)]
                if { $a > 0 } {
                    $canvas create line $xold $yold $x $y -fill black \
                        -tags temp
                
                }
                $canvas create oval [expr $x-2] [expr $y-2]  [expr $x+2] \
                    [expr $y+2] \
                    -fill black -tags temp
                set xold $x
                set yold $y
            }
        }
    } else {
        $canvas delete temp
        $canvas create line 10 42 [expr $width-20] 42 -fill black \
            -tags temp
        $canvas create oval 8 40 12 44 -fill black -tags temp
        $canvas create oval [expr $width-20-2] 40  [expr $width-20+2] \
            44 -fill black -tags temp
        $canvas create line [expr $width/2] 150 [expr $width/2] 42 -arrow last \
            -fill red -tags temp
        $canvas create line [expr $width/2-10] 150 [expr $width/2+10] 150 \
            -fill red -tags temp
    }

    #$canvas coords $CEPriv(radiuspos) [expr $width/2+10] 95
    $canvas coords radpos [expr $width/2+10] 95
}


proc CECLoseTest { w but } {
    global CEPriv GIDDEFAULT

    #grid rowconf $w 4 -weight 0
    $but configure -image [gid_themes::GetImage ArrowDown.png small_icons] \
        -command [list CEOpenTest $w $but]
    grid remove $w.ftest
    #destroy $w.ftest
}

proc CESelLine { w } {
    global CEPriv

    set ne $CEPriv(numelems)
    set CEPriv(numelems) [_ "Select one line of the model"]
    $w.ftest.line configure -command {GiD_Process Mescape} -state disable

    set nnum [GidUtils::PickEntities Lines single]
    $w.ftest.line configure -command  [list CESelLine $w] -state normal
    set CEPriv(numelems) $ne

    if { $nnum != "" } {
        set retval [GiD_Info list_entities -more Lines $nnum]
        regexp {Length=[ ]*([0-9.]*)[ ]*Radius=([0-9.]*)} $retval {} l r
        set CEPriv(length) $l
        if { $r < 0 || $r >= 1e5  } {
            set CEPriv(radius) Inf.
        } else {
            set CEPriv(radius) $r
        }
        raise $w
    }

}

proc CEOpenTest { w but} {
    global CEPriv GIDDEFAULT GidPriv

    wm geometry $w ""
    grid rowconf $w 4 -weight 1
    $but configure -image [gid_themes::GetImage ArrowUp.png small_icons] \
        -command "CECLoseTest $w $but"

    grid $w.ftest

    if { ![winfo exists $w.ftest.c] } {
        
        canvas $w.ftest.c -borderwidth 2 -relief groove -height 200 -takefocus 0
        
        grid $w.ftest.c -sticky ewns -row 1 -column 1
        grid columnconfigure $w.ftest 1 -weight 1
        grid rowconfigure $w.ftest 1 -weight 1
        ttk::button $w.ftest.line -image [gid_themes::GetImage "line.png" small_icons]  \
            -command [list CESelLine $w]
        place $w.ftest.line -in $w.ftest.c -anchor se -relx 1 -rely 1 -x -4 -y -4
        GidHelp $w.ftest.line [_ "A line can be selected to obtain its length and curvature radius"]
        
        ttk::frame $w.ftest.c.length
        label $w.ftest.c.length.l -text [_ "Length"]:
        ttk::entry $w.ftest.c.length.e -textvar CEPriv(length) -width 9
        pack $w.ftest.c.length.l $w.ftest.c.length.e -side left
        
        set CEPriv(length) 1
        
        ttk::frame $w.ftest.c.radius
        ttk::label $w.ftest.c.radius.l -text [_ "Radius"]:
        ttk::entry $w.ftest.c.radius.e -textvar CEPriv(radius) -width 9
        pack $w.ftest.c.radius.l $w.ftest.c.radius.e -side left
        set CEPriv(radius) 1
        
        ttk::label $w.ftest.c.numelems -textvar CEPriv(numelems) -relief ridge -borderwidth 2 \
            -width 30 -anchor w
        place $w.ftest.c.numelems -in $w.ftest.c -anchor sw -relx 0 -rely 1 \
            -x +4 -y -5
        set CEPriv(numelems) [_ "Mesh elements"]:
        
        $w.ftest.c create window 20 10 -window $w.ftest.c.length -anchor nw
        set CEPriv(radiuspos) [$w.ftest.c create window 30 10 \
                -window $w.ftest.c.radius -anchor w -tags radpos]
        
        GidHelp $w.ftest.c [_ "This graphical window is used to test the\
                chordal error.\nGiven a length of the line, its radius and\
                the chordal error, it calculates approximately the number of\
                elements that will be meshed over it.\n\
                The length and radius can be selected from a line in the model."]
        
        
        trace add variable  CEPriv(length) write "RedrawArc $w.ftest.c;#"
        trace add variable  CEPriv(radius) write "RedrawArc $w.ftest.c;#"
        trace add variable  CEPriv(chordalerror) write "RedrawArc $w.ftest.c;#"
    }

    RedrawArc $w.ftest.c

}

proc DestroyChordalErrorWindow { W w } {
    if { $W != $w } return
    global CEPriv
    #avoid reenter multiple times, one by child
    trace remove variable  CEPriv(length) write "RedrawArc $w.ftest.c;#"
    trace remove variable  CEPriv(radius) write "RedrawArc $w.ftest.c;#"
    trace remove variable  CEPriv(chordalerror) write "RedrawArc $w.ftest.c;#"
}

proc CEprocessinfo { w } {
    global CEPriv
    foreach value [list $CEPriv(maxmeshsize) $CEPriv(chordalerror)] {
        if { ![string is double -strict $value] || $value <=0 } {
            WarnWin [_ "Invalid value: %s, must be a real number" $value]
            return
        }        
    }            
    set minsize $CEPriv(minmeshsize)
    if { $minsize == "Automatic" } {
        set minsize [expr $CEPriv(maxmeshsize)/500.0]
    }
    foreach value [list $minsize] {
        if { ![string is double -strict $value] || $value <=0 } {
            WarnWin [_ "Invalid value: %s, must be a real number" $value]
            return
        }        
    } 
    GiD_Process Mescape Meshing AssignSizes ByChordalError $minsize $CEPriv(maxmeshsize) $CEPriv(chordalerror) escape
    destroy $w
}

proc CEprocessinfoCS { w } {
    global CEPriv    
    foreach value [list $CEPriv(maxmeshsize)] {
        if { ![string is double -strict $value] || $value <=0 } {
            WarnWin [_ "Invalid value: %s, must be a real number" $value]
            return
        }        
    }  
    if { !$CEPriv(bygeometry) && !$CEPriv(correctsizes) } {
        WarnWin [_ "No action has been performed. Check one or more options to proceed"]
        return
    }
    
    set minsize $CEPriv(minmeshsize)
    if { $minsize == "Automatic" } {
        set minsize [expr $CEPriv(maxmeshsize)/500.0]
    }    
    if { $CEPriv(bygeometry) } {
        GiD_Process Mescape Meshing AssignSizes ByGeometry $minsize $CEPriv(maxmeshsize) escape
    }
    if { $CEPriv(correctsizes) } {
        GiD_Process Mescape Meshing AssignSizes CorrectSizes $CEPriv(maxmeshsize) escape
    }
    destroy $w
}

proc CEGiveSizeFromBBox { bbox } {
    set err [catch {expr {abs([lindex $bbox 0]-[lindex $bbox 1])}} l1]
    if { $err } { 
        set l1 1
    }
    set err [catch {expr {abs([lindex $bbox 2]-[lindex $bbox 3])}} l2]
    if { $err } {
        set l2 1
    }
    set err [catch {expr {abs([lindex $bbox 4]-[lindex $bbox 5])}} l3]
    if { $err } {
        set l3 1
    }
    set retval [expr ($l1+$l2+$l3)/30.0]
    if { $retval == 0 } {
        return 10
    } else {
        return [format "%g" $retval] 
    }
}

proc CEBeginCorrectSizes { { w .gid.correctsizes } } {
    global CEPriv GIDDEFAULT

    if { ![winfo exists $w] } {
        InitWindow2 $w -title [_ "Correct meshing sizes"] \
            -geometryvariable PreCorrectMeshSizesWindowGeom \
            -initcommand CEBeginCorrectSizes \
            -onlygeometry -ontop
        if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0
        
        set retval [GiD_Info Project LastElementSize]
        set bbox [GiD_Info layers -bounding_box]
        set calcsize [CEGiveSizeFromBBox $bbox]
        if { ![info exists CEPriv(maxmeshsize)] || $CEPriv(maxmeshsize) == "" } {
            if { $retval != "NONE" } {
                set CEPriv(maxmeshsize) $retval
            } else {
                set CEPriv(maxmeshsize) $calcsize
            }
        }
        
        label $w.l2 -text [_ "Maximum meshing size"]:
        ttk::entry $w.c2 -textvariable CEPriv(maxmeshsize)
        
        GidHelp "$w.l2 $w.c2" [_ "Typically, it is the meshing size that will be used\
                later as default for meshing."]
        
        label $w.l3 -text [_ "Minimum meshing size"]:
        combobox $w.c3 -editable 1 -textvariable CEPriv(minmeshsize)
        
        if { ![info exists CEPriv(minmeshsize)] || $CEPriv(minmeshsize) == "" } {
            set CEPriv(minmeshsize) Automatic
            $w.c3 add Automatic
        } else {
            $w.c3 add $CEPriv(minmeshsize)
            if { [string compare $CEPriv(minmeshsize) "Automatic"] } {
                $w.c3 add Automatic
            }
        }
        
        GidHelp "$w.l3 $w.c3" [_ "The algorithm will not assign any size smaller\
                than this to any geometrical entity."]
        
        if { ![info exists CEPriv(bygeometry)] } {
            set CEPriv(bygeometry) 1
        }
        if { ![info exists CEPriv(bygeometry)] } { set CEPriv(bygeometry) 0 }
        ttk::checkbutton $w.ce1 -text [_ "By geometry"] -variable CEPriv(bygeometry)

        if { ![info exists CEPriv(correctsizes)] } {
            set CEPriv(correctsizes) 1
        }
        if { ![info exists CEPriv(correctsizes)] } { set CEPriv(correctsizes) 0 }
        ttk::checkbutton $w.ce2 -text [_ "Correct incompatible sizes"] \
                -variable CEPriv(correctsizes)
        
        GidHelp "$w.ce1" [_ "Option 'By geometry' assign smaller meshing sizes\
                to details in geometry.\nNote: Option 'Correct meshing sizes'\
                in Preferences perform both actions automatically."]

        GidHelp "$w.ce2" [_ "Option 'Correct incompatible sizes' compares close\
                geometrical entities and change its sizes to make them\
                compatible.\nNote: Option 'Correct meshing sizes'\
                in Preferences perform both actions automatically."]
        
        ttk::frame $w.buts -style BottomFrame.TFrame
        ttk::button $w.buts.ok -text [_ "OK"] -command [list CEprocessinfoCS $w] -underline 0 -style BottomFrame.TButton
        ttk::button $w.buts.cancel -text [_ "Cancel"] -command "destroy $w" -underline 0 -style BottomFrame.TButton
        
        bind $w <Alt-o> "$w.buts.ok invoke"
        bind $w.c2 <Return> "$w.buts.ok invoke"
        
        bind $w.c2 <FocusIn> {
            %W sel range 0 end
            %W icursor end
        }
        bind $w.c2 <FocusOut> { %W sel clear }
        
        bind $w.c3 <FocusIn> {
            %W sel range 0 end
            %W icursor end
        }
        bind $w.c3 <FocusOut> { %W sel clear }
        
        grid $w.buts.ok $w.buts.cancel -padx 2 -pady 2
        
        grid $w.l2 -row 1 -column 1  -sticky e
        grid $w.c2 -row 1 -column 2 -sticky ew
        
        grid $w.l3 -row 2 -column 1  -sticky e
        grid $w.c3 -row 2 -column 2 -sticky ew
        
        grid $w.ce1 -row 3 -column 1  -sticky w -columnspan 2
        grid $w.ce2 -row 4 -column 1 -sticky  w -columnspan 2
        
        grid $w.buts -row 6 -column 1 -columnspan 2 -sticky sew
        grid anchor $w.buts center
        grid rowconfigure $w "5" -weight 1
        grid columnconfigure $w "2" -weight 1
    }

    focus $w.c2
    $w.c2 sel range 0 end
    $w.c2 icursor end

}

proc CEBegin { { w .gid.chordalerror } } {
    global CEPriv GIDDEFAULT

    if { ![winfo exists $w] } {
        InitWindow2 $w -title [_ "Assign sizes by chordal error"] \
            -geometryvariable PreMeshSizesChordalErrorWindowGeom \
            -initcommand CEBegin \
            -onlygeometry -ontop
        if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0
        
        if { ![info exists CEPriv(chordalerror)] || $CEPriv(chordalerror) == "" } {
            set CEPriv(chordalerror) 0.1
        }
        
        set retval [GiD_Info Project LastElementSize]
        set bbox [GiD_Info layers -bounding_box]
        set calcsize [CEGiveSizeFromBBox $bbox]
        if { ![info exists CEPriv(maxmeshsize)] || $CEPriv(maxmeshsize) == "" } {
            if { $retval != "NONE" } {
                set CEPriv(maxmeshsize) $retval
            } else {
                set CEPriv(maxmeshsize) $calcsize
            }
        }
        
        label $w.l1 -text [_ "Chordal Error"]:
        ttk::entry $w.c1 -textvariable CEPriv(chordalerror)
        
        GidHelp "$w.l1 $w.c1" [_ "Smaller chordal errors will asign smaller sizes to\
                entities with high curvature"]
        
        label $w.l2 -text [_ "Maximum meshing size"]:
        ttk::entry $w.c2 -textvariable CEPriv(maxmeshsize)
        
        GidHelp "$w.l2 $w.c2" [_ "Typically, it is the meshing size that will be used\
                later as default for meshing."]
        
        label $w.l3 -text [_ "Minimum meshing size"]:
        combobox $w.c3 -editable 1 -textvariable CEPriv(minmeshsize)
        
        if { ![info exists CEPriv(minmeshsize)] || $CEPriv(minmeshsize) == "" } {
            set CEPriv(minmeshsize) Automatic
            $w.c3 add Automatic
        } else {
            $w.c3 add $CEPriv(minmeshsize)
            if { [string compare $CEPriv(minmeshsize) "Automatic"] } {
                $w.c3 add Automatic
            }
        }
        GidHelp "$w.l3 $w.c3" [_ "The algorithm will not assign any size smaller\
                than this to any geometrical entity."]
        
        ttk::frame $w.ftest
        
        ttk::frame $w.buts -style BottomFrame.TFrame
        ttk::button $w.buts.ok -text [_ "OK"] -command [list CEprocessinfo $w] -underline 0 -style BottomFrame.TButton
        ttk::button $w.buts.cancel -text [_ "Cancel"] -command [list destroy $w] -underline 0 -style BottomFrame.TButton
        
        ttk::button $w.buts.test -image [gid_themes::GetImage ArrowDown.png small_icons] \
            -command "CEOpenTest $w $w.buts.test" -style BottomFrame.TButton
        
        bind $w <Alt-o> "$w.buts.ok invoke"
        bind $w <Alt-c> "$w.buts.cancel invoke"
        bind $w.c1 <Return> "$w.buts.ok invoke"
        bind $w.c2 <Return> "$w.buts.ok invoke"
        #bind $w.c3 <Return> "$w.buts.ok invoke"
        bind $w <Escape> "$w.buts.cancel invoke"
        
        bind $w.c1 <FocusIn> {
            %W sel range 0 end
            %W icursor end
        }
        bind $w.c1 <FocusOut> { %W sel clear }
        bind $w.c2 <FocusIn> {
            %W sel range 0 end
            %W icursor end
        }
        bind $w.c2 <FocusOut> { %W sel clear }
        bind $w.c3 <FocusIn> {
            %W sel range 0 end
            %W icursor end
        }
        bind $w.c3 <FocusOut> { %W sel clear }
        
        GidHelp $w.buts.test [_ "A graphical window to test the chordal error is given"]
        
        grid $w.buts.ok -row 1 -column 1 -padx 2 -pady 2
        grid $w.buts.cancel -row 1 -column 2 -padx 2 -pady 2
        place $w.buts.test -in $w.buts -relx 1 -rely 1 -anchor se
        
        grid $w.l1 -row 1 -column 1 -sticky ne -pady 2
        grid $w.c1 -row 1 -column 2 -sticky new
        
        grid $w.l2 -row 2 -column 1  -sticky e -pady 2
        grid $w.c2 -row 2 -column 2 -sticky ew
        
        grid $w.l3 -row 3 -column 1  -sticky e -pady 2
        grid $w.c3 -row 3 -column 2 -sticky ew
        
        grid $w.ftest -row 4 -column 1 -columnspan 2 -sticky nsew
        grid $w.buts -row 5 -column 1 -columnspan 2 -sticky sew
        grid anchor $w.buts center
        grid rowconfigure $w 4 -weight 1
        grid columnconfigure $w 2 -weight 1
    }

    focus $w.c1
    $w.c1 sel range 0 end
    $w.c1 icursor end

    bind $w <Destroy> [list +DestroyChordalErrorWindow %W $w] ;# + to add to previous script
}

