namespace eval Shapefile {
    variable shape_types
    variable gid_entity_types

    array set shape_types {
        0 Null
        1 Point
        3 PolyLine
        5 Polygon
        8 MultiPoint
        11 PointZ
        13 PolyLineZ
        15 PolygonZ
        18 MultiPointZ
        21 PointM
        23 PolyLineM
        25 PolygonM
        28 MultiPointM
        31 MultiPatch
    }

    array set gid_entity_types {
        Null ""
        Point point
        PolyLine line
        Polygon surface
        MultiPoint point
        PointZ point
        PolyLineZ line
        PolygonZ surface
        MultiPointZ point
        PointM point
        PolyLineM line
        PolygonM surface
        MultiPointM point
        MultiPatch surface
    }
}

proc Shapefile::GetHeader { filename } {
    set values [list]
    if { [file exists $filename] } {
        set fp [open $filename rb]
        fconfigure $fp -encoding binary -translation binary -eofchar {}
        if { $fp != "" } {
            set header [read $fp 100]
            close $fp
            binary scan $header IIIIIIIiidddddddd file_code d0 d1 d2 d3 d4 file_length version i_shape_type x0 y0 x1 y1 z0 z1 m0 m1
            if { $file_code == 9994 } {
                lappend values $file_code $d0 $d1 $d2 $d3 $d4 $file_length $version $i_shape_type $x0 $y0 $x1 $y1 $z0 $z1 $m0 $m1
            }
        }
    }
    return $values
}

proc Shapefile::GetShapeType { filename } {
    variable shape_types
    set shape_type ""
    set header_values [Shapefile::GetHeader $filename]
    set i_shape_type [lindex $header_values 8]
    if { [info exists shape_types($i_shape_type)] } {
        set shape_type $shape_types($i_shape_type)
    }
    return $shape_type
}

proc Shapefile::GetGidEntityType { shape_type } {
    variable gid_entity_types
    set gid_entity_type ""
    if { [info exists gid_entity_types($shape_type)] } {
        set gid_entity_type $gid_entity_types($shape_type)
    }
    return $gid_entity_type
}

proc Shapefile::GetGidConditionOver { gid_entity_type } {
    set over [list]
    if { $gid_entity_type == "point" } {
        set over [list over_point over_node]
    } elseif { $gid_entity_type == "line" } {
        set over [list over_line over_element]
    } elseif { $gid_entity_type == "surface" } {
        set over [list over_surface over_element]
    } else {
        error "Shapefile::GetGidOver: unexpected gid_entity_type=$gid_entity_type"
    }
    return $over
}

proc Shapefile::GetAutomaticName { gid_entity_type } {
    set prefix ${gid_entity_type}_
    set over_geometry [lindex [Shapefile::GetGidConditionOver $gid_entity_type] 0]
    set names [GiD_Info conditions $over_geometry]
    set fullname ""
    for {set i 1} {$i<10000} {incr i} {
        set fullname ${prefix}imported-${i}
        if { [lsearch -exact $names $fullname] == -1 } {
            break
        }
    }
    return $fullname
}

proc Shapefile::MoveEntitiesZAbsolute { gid_entity_type entities z } {
    #assume all entities in zame z, use center of first
    set first_entity [lindex $entities 0]
    if { $first_entity != "" } {
        set z_current [lindex [GidUtils::GetEntityCenter $gid_entity_type $first_entity] 2]
        set z_increment [expr $z-$z_current]
        if { $z_increment != 0 } {
            GiD_Process Mescape Utilities Move ${gid_entity_type}s Duplicate MaintainLayers Translation FNoJoin 0,0,0 FNoJoin 0,0,$z_increment {*}$entities escape escape
        }
    }
}

#this key "ZValue" mean a building height, at seast in some of tested examples
proc Shapefile::IsHeighKey { key } {
    if { [lsearch {ZValue} $key] != -1 } {
        set result 1
    } else {
        set result 0
    }
    return $result
}

proc Shapefile::ReadDBF { filename create_conditions move_z_to_height } {
    if { [file exists $filename] } {
        set shape_type [Shapefile::GetShapeType [file rootname $filename].shp]
        set gid_entity_type [Shapefile::GetGidEntityType $shape_type]
        if { $gid_entity_type != "" } {
            package require dbf
            set ok [dbf db_handler -open $filename -readonly]
            if { $ok } {
                set created_conditions [list]
                set pos_height -1
                if { $create_conditions } {
                    lassign [Shapefile::GetGidConditionOver $gid_entity_type] over_geometry over_mesh
                    lassign [$db_handler info] num_records num_fields
                    set fields [$db_handler fields]
                    set questions [list]
                    foreach field $fields {
                        lappend questions [lindex $field 0]
                    }
                    set condition_name [Shapefile::GetAutomaticName $gid_entity_type]
                    GiD_CreateData create condition $condition_name $over_geometry $over_mesh [list $questions]
                    foreach entity_id [GiD_Geometry list $gid_entity_type] {
                        set material_id [GiD_Geometry get $gid_entity_type $entity_id material]
                        lappend entities_by_material($material_id) $entity_id
                    }
                    for { set i_record 0 } { $i_record < $num_records } { incr i_record } {
                        set values [$db_handler record $i_record]
                        set material_id [expr {$i_record+1}]
                        #set entities [GiD_Geometry list -material $material_id $gid_entity_type] ;#slow command to be used inside the loop
                        if { [info exists entities_by_material($material_id)] } {
                            set entities $entities_by_material($material_id)
                            GiD_AssignData condition $condition_name ${gid_entity_type}s $values $entities
                        } else {
                            W "Shapefile::ReadDBF record $i_record material_id $material_id values '$values' not assigned to entities"
                        }
                    }
                    array unset entities_by_material
                    lappend created_conditions $condition_name
                }
                if { $move_z_to_height } {
                    lassign [$db_handler info] num_records num_fields
                    set fields [$db_handler fields]
                    set i_field 0
                    foreach field $fields {
                        if { [Shapefile::IsHeighKey [lindex $field 0]] } {
                            set pos_height $i_field
                            break
                        }
                        incr i_field
                    }
                    if { $pos_height != -1 } {
                        set ::parar_a_ver 0
                        GidUtils::DisableGraphics
                        for { set i_record 0 } { $i_record < $num_records } { incr i_record } {
                            set values [$db_handler record $i_record]
                            set entities [GiD_Geometry list -material [expr {$i_record+1}] $gid_entity_type]
                            set height_relative_to_ground [lindex $values $pos_height]
                            if { $::parar_a_ver } {
                                break
                            }
                            Shapefile::MoveEntitiesZAbsolute $gid_entity_type $entities $height_relative_to_ground
                        }
                        GidUtils::EnableGraphics
                    }
                }
                $db_handler close
                if { [llength $created_conditions] } {
                    CacheReset menu_data
                    GidUtils::SetWarnLine [_ "Created %s new conditions representing shapefile attached data" [llength $created_conditions]]
                }
                if { $pos_height != -1} {
                    GidUtils::SetWarnLine [_ "Surfaces moved to z representing building height"]
                }
            } else {
                error "Shapefile::ReadDBF: dbf file $filename, wrong format"
            }
        }
    } else {
        error "Shapefile::ReadDBF: shp file $filename, not found"
    }
}

#invoked from C++
proc Shapefile::AfterOpenFile { filename create_conditions move_z_to_height } {
    set filename_dbase [file rootname $filename].dbf
    if { ![file exists $filename_dbase] } {
        # try uppercase extension !!!
        set filename_dbase [file rootname $filename].DBF
    }
    if { [file exists $filename_dbase] } {
        Shapefile::ReadDBF $filename_dbase $create_conditions $move_z_to_height
    }
    set filename_project [file rootname $filename].prj
    if { ![file exists $filename_project] } {
        # try uppercase extension !!!
        set filename_project [file rootname $filename].PRJ
    }
    if { [file exists $filename_project] } {
        set content [GidUtils::ReadFile $filename_project "" 0]
        if { $content != "" } {
            SetNotes $content
        }
    }
    #the original surface id is stored in the material, unassign materials
    #(e.g. to not prevent join planar surfaces because different material)
    #NOTE: this could be bad, unassigning all prevoius materials of all surfaces and lines!!
    #really must restore the previous material id of possible old entities and set to 0 for new entities
    GiD_UnAssignData material * surfaces 1:end
    GiD_UnAssignData material * lines 1:end
    return 0
}
