
#############################################
##########   useful procedures    ###########
#############################################

# to get the full path to the exe that starts python
proc GiD_Python_GetPythonExe { } {
    if { $::tcl_platform(platform) == "windows" } {
        if { $::tcl_platform(pointerSize) == 8 } {
            set bits 64
        } else {
            set bits 32
        }
        set python_exe_path [file join [gid_filesystem::get_folder_standard scripts] tohil/python/bin/x${bits}/python.exe]
    } else {
        #must be changed by the real location in case of linux!!!
        set python_exe_path [file join [gid_filesystem::get_folder_standard scripts] tohil/python/python]
    }
    if { ![file exists $python_exe_path] } {
        set python_exe_path ""
    }
    return $python_exe_path
}

# to get the full path to place where third part packages are stored
proc GiD_Python_GetSitePackagesPath { {pure_scripts 0}} {
    if { $::tcl_platform(platform) == "windows" } {
        set site_packages_path [file join [gid_filesystem::get_folder_standard scripts] tohil/python/lib/site-packages]
    } else {
        # if { $pure_scripts } {
        #     #for pure script modules
        #     set site_packages_path [file join [gid_filesystem::get_folder_standard scripts] tohil/python/lib/python3.10/site-packages]
        # } else {
        #     #for compiled modules
        #     set site_packages_path [file join [gid_filesystem::get_folder_standard scripts] tohil/python/x64/lib/python3.10/site-packages]
        # }
        package require tohil
        set err [ catch {
        set python_version [ tohil::run {import sys;print ( str( sys.version_info[ 0]) + "." + str( sys.version_info[ 1]));}]
        set site_packages_path [ file join [ gid_filesystem::get_folder_standard scripts] tohil/python/lib/python${python_version}/site-packages]
        } err_txt]
    }
    return $site_packages_path
}

proc GiD_Python_PythonArrayToTclList { python_array } {
    return [string map {[ \{ ] \}} $python_array]
}

proc GiD_Python_TclListToPythonArray { tcl_list } {
    return [string map {\{ [ \} ]} $tcl_list]
}

#############################################
########## pip related procedures ###########
#############################################

#packages is a list, and the package can specify a version with syntax $name==$version (e.g. numpy==1.24.2)
#return a list with the packages that were not intalled
proc GiD_Python_PipInstall { packages {upgrade 0} {timeout_seconds 60} } {
    set packages_fail [list]
    if { [llength $packages] } {
        set action [list install]
        if { $upgrade } {
            lappend action --upgrade
        }
        lappend action --no-cache-dir --disable-pip-version-check --no-warn-script-location
        set python_path [GiD_Python_GetPythonExe]
        #-E python flag is to say python.exe to ignore PYTHON* environment variables like PYTHONPATH that become a problem pointing to other versions...
        # W "is writable [GiD_Python_GetSitePackagesPath] : [file writable [GiD_Python_GetSitePackagesPath]]"
        if { [file writable [GiD_Python_GetSitePackagesPath]] } {
            set err [ catch {exec $python_path -E -m pip {*}$action {*}$packages} msg]
            if { $err} {
                W "error instaling python packages with:"
                W "    $python_path -E -m pip {*}$action {*}$packages"
                W "--> $msg"
            } else {
                # W "OK instaling python packages with:"
                # W "    $python_path -E -m pip {*}$action {*}$packages"
                # W "--> $msg"
            }
            set packages_fail [GiD_Python_PipGetMissingPackages $packages]
            # WV packages_fail
        } else {
            # W "error needs root permision!!"
            # return 1
            package require gid_cross_platform
            set t0 [clock seconds]
            catch { gid_cross_platform::run_as_administrator -show hide $python_path -E -m pip {*}$action {*}$packages } msg
            #want to wait until packages are downloaded before continue, try until N times
            while { ( !$timeout_seconds || [expr [clock seconds]-$t0]<$timeout_seconds) && [llength [GiD_Python_PipGetMissingPackages $packages]] } {
                after 100
            }
            set packages_fail [GiD_Python_PipGetMissingPackages $packages]
        }
    }
    return $packages_fail
}

proc GiD_Python_PipUnInstall { packages {timeout_seconds 60} } {
    set packages_fail [list]
    set packages [GiD_Python_PipGetExistingPackages $packages]
    if { [llength $packages] } {
        set python_path [GiD_Python_GetPythonExe]
        #-E python flag is to say python.exe to ignore PYTHON* environment variables like PYTHONPATH that become a problem pointing to other versions...
        if { [file writable [GiD_Python_GetSitePackagesPath]] } {
            catch {exec $python_path -E -m pip uninstall -y {*}$packages} msg
            set packages_fail [GiD_Python_PipGetExistingPackages $packages]
        } else {
            package require gid_cross_platform
            set t0 [clock seconds]
            catch { gid_cross_platform::run_as_administrator -show hide $python_path -E -m pip uninstall -y {*}$packages }
            #want to wait until packages are downloaded before continue, try until N times
            while { ( !$timeout_seconds || [expr [clock seconds]-$t0]<$timeout_seconds) && [llength [GiD_Python_PipGetExistingPackages $packages]] } {
                after 100
            }
            set packages_fail [GiD_Python_PipGetExistingPackages $packages]
        }
    }
    return $packages_fail
}

proc GiD_Python_VersionSatisfy { present comparation required } {
    set satisfy 0
    if { $required == "" } {
        set satisfy 1
    } else {
        if { $present == "" } {
            set satisfy 0
        } else {
            if { $comparation == "==" } {
                set cmp [GidUtils::TwoVersionsCmp $present $required] 
                set satisfy [expr {$cmp == 0}]
            } elseif { $comparation == ">=" } {
                set cmp [GidUtils::TwoVersionsCmp $present $required] 
                set satisfy [expr {$cmp >= 0}]
            } else {
                set satisfy 1
            }
        }
    }
    return $satisfy
}

proc GiD_Python_SplitPackageComparationVersion { item } {
    set package ""
    set comparation ""
    set version ""
    if { [string first == $item] !=-1 } {
        #specific_version
        set comparation ==
        lassign [GidUtils::Split $item ==] package version
    } elseif { [string first >= $item] !=-1 } {
        #minimum_version
        set comparation >=
        lassign [GidUtils::Split $item >=] package version
    } else {
        #last_version
        set comparation ""
        set package $item
        set version ""
    }
    return [list $package $comparation $version]
}

proc GiD_Python_PipGetMissingPackages { required_packages } {
    set packages_and_versions [GiD_Python_PipListPackages]
    set missing_packages [list]
    foreach item $required_packages {
        lassign [GiD_Python_SplitPackageComparationVersion $item] package comparation version
        set pos [lsearch -index 0 $packages_and_versions $package]
        if { $pos == -1 } {
            lappend missing_packages $item
        } else {
            #if requiring a specific version must check if match the current
            if { ![GiD_Python_VersionSatisfy [lindex $packages_and_versions $pos 1] $comparation $version] } {
                lappend missing_packages $item
            }
        }
    }
    return $missing_packages
}

proc GiD_Python_PipGetExistingPackages { required_packages } {
    set packages_and_versions [GiD_Python_PipListPackages]
    set existing_packages [list]
    foreach item $required_packages {
        lassign [GiD_Python_SplitPackageComparationVersion $item] package comparation version
        set pos [lsearch -index 0 $packages_and_versions $package]
        if { $pos != -1 } {
            #if requiring a specific version must check if match the current
            if { [GiD_Python_VersionSatisfy [lindex $packages_and_versions $pos 1] $comparation $version] } {
                lappend existing_packages $package
            }
        }
    }
    return $existing_packages
}

proc GiD_Python_PipInstallMissingPackages { required_packages {timeout_seconds 60} {parent ""}} {
    set packages_to_install [GiD_Python_PipGetMissingPackages $required_packages]
    set packages_fail [list]
    if { [llength $packages_to_install] } {
        set answer 1
        if { [GiD_Set AutomaticAnswers] } {
            set answer 1
        } else {
            set answer [ MessageBoxOptionsButtons [_ "Install python packages"] \
                [_ "Some Python packages require be installed. Continue?"] {1 0} [list [_ "Ok"] [_ "Cancel"]] \
                             question "" 0 $parent]
            if { $answer == ""} {
                # may be MessageBox could not be openned, asuming '1' as (AutomaticAnswer) above
                set answer 1
            }
        }
        if { $answer == 1 } {
            GidUtils::SetWarnLine [_ "Installing %s Python packages: %s" [llength $packages_to_install] $packages_to_install]
            GidUtils::WaitState
            set upgrade 0
            set packages_fail [GiD_Python_PipInstall $packages_to_install $upgrade $timeout_seconds]
            GidUtils::EndWaitState
             if { [llength $packages_fail] } {
                GidUtils::SetWarnLine [_ "%s missing Python packages cannot be installed: %s" [llength $packages_fail] $packages_fail]
            } else {
                GidUtils::SetWarnLine [_ "Installed %s packages" [llength $packages_to_install]]
            }
        } else {
            set packages_fail $packages_to_install
        }
    }
    return $packages_fail
}

proc GiD_Python_PipListPackages { } {
    set packages_and_versions [list]
    set python_path [GiD_Python_GetPythonExe]
    #-E python flag is to say python.exe to ignore PYTHON* environment variables like PYTHONPATH that become a problem pointing to other versions...
    set pip_packages_installed_raw [exec $python_path -E -m pip list --format=freeze --disable-pip-version-check]
    foreach item $pip_packages_installed_raw {
        lassign [GiD_Python_SplitPackageComparationVersion $item] package comparation version
        lappend packages_and_versions [list $package $version]
    }
    return $packages_and_versions
}

proc GiD_Python_PipPackageVersion { package } {
    set version ""
    set packages_and_versions [GiD_Python_PipListPackages]
    set pos [lsearch -index 0 $packages_and_versions $package]
    if { $pos != -1 } {
        set version [lindex $packages_and_versions $pos 1]
    }
    return $version
}

#############################################
############## tohil  related ###############
#############################################

# decoration procs that simply wrap or alias the use of tohil with GiD_Python_XX proc names
# in fact probably it is more effient use Tcl interp alias command that define an extra proc


#to allow a remote debug from VSCode, at localhost, port 5678 (can attach to a process also, that use the same values)
#warning: do not define it in a namespace of will fail

#it is necessary to ad inside the folder .vscode a launch.json file like this:
#
#{
#    // Use IntelliSense to learn about possible attributes.
#    // Hover to view descriptions of existing attributes.
#    // For more information, visit: https://go.microsoft.com/fwlink/?linkid=830387
#    "version": "0.2.0",
#    "configurations": [
#        {
#            "name": "Python: Attach localhostt:5678",
#            "type": "python",
#            "request": "attach",
#            "connect": {
#                "host": "localhost",
#                "port": 5678
#              },
#            //"processId": "${command:pickProcess}",
#            //"justMyCode": true
#        }
#    ]
#}

proc GiD_Python_StartDebuggerServer { } {
    package require tohil
    tohil::import debugpy
    #tohil::exec "debugpy.log_to('[gid_cross_platform::get_tmp]')"
    set my_python [GiD_Python_GetPythonExe]
    #to avoid a bug of debugpy that use sys.executable and try to run another gid.exe!!. see https://github.com/microsoft/debugpy/issues/262
    tohil::exec "debugpy.configure(python='$my_python')"
    tohil::exec "debugpy.listen(5678)"
    #tohil::exec "debugpy.wait_for_client()"
    return 0
}


#evaluate a block of code without any return
proc GiD_Python_Exec { python_block_code } {
    package require tohil
    tohil::exec $python_block_code
    return 0
}

#evaluate a single python command and return its result
proc GiD_Python_Eval { python_single_code } {
    package require tohil
    return [tohil::eval $python_single_code]
}

#call an existing python function
proc GiD_Python_Call { function_name args } {
    package require tohil
    return [tohil::call $function_name {*}$args]
}

#import a python module
proc GiD_Python_Import { module_name } {
    package require tohil
    return [tohil::import $module_name]
}

proc GiD_Python_GetPath { } {
    package require tohil
    tohil::import sys
    return [tohil::eval sys.path]
}

proc GiD_Python_PathAppend { dir_name } {
    if { [lsearch [GiD_Python_GetPath] $dir_name] == -1} {
        tohil::call sys.path.append $dir_name
    }
}

proc GiD_Python_PathInsert { index dir_name } {
    if { [lindex [GiD_Python_GetPath] $index] !=$dir_name } {
        tohil::exec "sys.path.insert($index,'$dir_name')"
    }
}

proc GiD_Python_PathRemove { dir_name } {
    if { [lsearch [GiD_Python_GetPath] $dir_name] != -1} {
        tohil::exec "sys.path.remove('$dir_name')"
    }
}


#interesting to reload a file that is changed in develop, because import don't reload it
#e.g. GiD_Python_Source "C:/gid project/plugins/Import/meshio/gid_meshio.py"
proc GiD_Python_Source { python_filename } {
    package require tohil
    set module_name [file rootname [file tail $python_filename]]
    tohil::exec "from pydoc import importfile"
    tohil::exec "$module_name=importfile('$python_filename')"
}

#e.g. GiD_Python_Import_File "C:/gid project/plugins/Import/meshio/gid_meshio.py"
proc GiD_Python_Import_File { python_filename } {
    package require tohil
    set module_name [file rootname [file tail $python_filename]]
    set dir_name [file dirname $python_filename]
    tohil::import sys
    tohil::call sys.path.append $dir_name
    tohil::import $module_name
}

proc GiD_Python_Open_IDLE_Shell { } {
    package require tohil
    tohil::exec {
        #load tcl functions only to facilitate GiD calls
        tcl=tohil.import_tcl()
        import sys
        sys.argv=['','-n','-t','GiD python IDLE shell']
        import idlelib.pyshell
        idlelib.pyshell.main()
    }
}


