package require TclOO


if { [info commands GiDMeshVariables] != "" } {
    #to protect of multiple source
    GiDMeshVariables destroy
}

#owner PREFERENCES,MODEL,MESHING
oo::class create GiDMeshVariables {
    variable Owner
    constructor { v } {
        my Reset
        set [my varname Owner] $v
        set Owner $v
        if { $v != "PREFERENCES" && $v != "MODEL" && $v !="MESHING" } {
            WarnWinText "GiDMeshVariables: wrong owner, must be PREFERENCES MODEL MESHING"
        }
    }

    #get the variables of the top class of the object
    method GetVariables { } {
        return [info class variables [info object class [self object]]]
    }

    method Serialize { } {
        set data [list]
        foreach item [my GetVariables] {
            lappend data $item [my Get $item]
        }
        return $data
    }

    method DeSerialize { data } {
        foreach {item value} $data {
            my Set $item $value
        }
        return 0
    }

    method Set { key value } {
        set [my varname $key] $value
    }

    method Get { key } {
        set [my varname $key]
    }

    method GetDefault { key } {
        if { $key == "Owner" } {
            set value ""
        } else {
            set value 0
            WarnWinText "GiDMeshVariables::GetDefault. Unexpected key $key"
        }
        return $value
    }

    method Reset {} {
        foreach item [my GetVariables] {
            my Set $item [my GetDefault $item]
        }
        return 0
    }

    method Clear {} {
        foreach item [my GetVariables] {
            my Set $item -1
        }
        return 0
    }

    method SetParametersFromOther { obj } {
        foreach item [my GetVariables] {
            my Set $item [$obj Get $item]
        }
        return 0
    }

    method IsValid {} {
        return 1
    }

    method SetAsDefaultTheNonValids {} {
        set res 0
        if { ![my IsValid] } {
            my Reset
            set res 1
        }
        return $res
    }

    method Print { } {
        set res [list]
        foreach item [my GetVariables] {
            lappend res [list $item [my Get $item] [my GetDefault $item]]
        }
        return $res
    }

    export Clear DeSerialize Get GetDefault GetVariables IsValid Print Reset Serialize Set SetAsDefaultTheNonValids SetParametersFromOther
}


#to register this class as a inner mesher and handle its variables (used copy in the model, in preferences and the ones to use meshing).
#class_name must be inherit of GiDMeshVariables class to implement its expected methods
proc GiD_RegisterPluginMeshVariablesClass { class_name } {
    if { ![info object isa object $class_name] || ![info object isa class $class_name] } {
        return -code error [_ "class %s not found" $class_name]
    }
    if { ![info exists ::GidPriv(RegisteredPluginMeshVariablesClass)] } {
        set pos -1
    } else {
        set pos [lsearch $::GidPriv(RegisteredPluginMeshVariablesClass) $class_name]
    }
    if { $pos == -1 } {
        lappend ::GidPriv(RegisteredPluginMeshVariablesClass) $class_name
        PluginMeshVariablesClass_CreateObjects PREFERENCES
        PluginMeshVariablesClass_CreateObjects MODEL
        PluginMeshVariablesClass_CreateObjects MESHING
    }
}

proc GiD_UnRegisterPluginMeshVariablesClass { class_name } {
    if { ![info exists ::GidPriv(RegisteredPluginMeshVariablesClass)] } {
        return 1
    }
    set pos [lsearch $::GidPriv(RegisteredPluginMeshVariablesClass) $class_name]
    if { $pos == -1 } {
        set ::GidPriv(RegisteredPluginMeshVariablesClass) [lreplace $::GidPriv(RegisteredPluginMeshVariablesClass) $pos $pos]
    }
}

#owner PREFERENCES,MODEL,MESHING
proc PluginMeshVariablesClass_CreateObjects { owner } {
    if { ![info exists ::GidPriv(RegisteredPluginMeshVariablesClass)] } {
        return 1
    }
    foreach class_name $::GidPriv(RegisteredPluginMeshVariablesClass) {
        set obj ""
        foreach item [info class instances $class_name] {
            if { [$item Get Owner] == $owner } {
                set obj $item
                $obj Reset
                break
            }
        }
        if { $obj == "" } {
            set obj [$class_name new $owner]
        }
    }
    return 0
}

proc PluginMeshVariablesClass_SetParametersFromOther { owner_other owner_dst } {
    #WarnWinText "PluginMeshVariablesClass_SetParametersFromOther $owner_other $owner_dst"
    if { [info exists ::GidPriv(RegisteredPluginMeshVariablesClass)] } {
        foreach class_name $::GidPriv(RegisteredPluginMeshVariablesClass) {
            set obj_other [PluginMeshVariablesClass_GetClassObject $class_name $owner_other]
            set obj_dst [PluginMeshVariablesClass_GetClassObject $class_name $owner_dst]
            $obj_dst SetParametersFromOther $obj_other
        }
    }
}

#set un-initialized invalid values
proc PluginMeshVariablesClass_Clear { owner } {
    #WarnWinText "PluginMeshVariablesClass_Clear $owner"
    if { [info exists ::GidPriv(RegisteredPluginMeshVariablesClass)] } {
        foreach class_name $::GidPriv(RegisteredPluginMeshVariablesClass) {
            #WarnWinText "  Clear class=$class_name"
            set obj [PluginMeshVariablesClass_GetClassObject $class_name $owner]
            $obj Clear
        }
    }
}

proc PluginMeshVariablesClass_Reset { owner } {
    #WarnWinText "PluginMeshVariablesClass_Reset $owner"
    if { [info exists ::GidPriv(RegisteredPluginMeshVariablesClass)] } {
        foreach class_name $::GidPriv(RegisteredPluginMeshVariablesClass) {
            #WarnWinText "  Reset class=$class_name"
            set obj [PluginMeshVariablesClass_GetClassObject $class_name $owner]
            $obj Reset
        }
    }
}

#get only the non-default values
proc PluginMeshVariablesClass_GetModelParameters { } {
    set res ""
    if { [info exists ::GidPriv(RegisteredPluginMeshVariablesClass)] } {
        foreach class_name $::GidPriv(RegisteredPluginMeshVariablesClass) {
            set obj [PluginMeshVariablesClass_GetClassObject $class_name MODEL]
            foreach item [lsort -dictionary [$obj GetVariables]] {
                set value [$obj Get $item]
                if { [$obj GetDefault $item] != $value } {
                    append res "    $class_name $item: $value\n"
                }
            }
        }
    }
    return $res
}

proc PluginMeshVariablesClass_SaveObjects { filename owner } {
    if { $owner == "MODEL" } {
        set must_save 0
        set data_status [GiD_Info list_entities Status]
        set sphere_mesher_used -1
        regexp  {Sphere mesher: ([0-9]*)} $data_status dummy sphere_mesher_used
        set circle_mesher_used -1
        regexp  {Circle mesher: ([0-9]*)} $data_status dummy circle_mesher_used
        if { ($sphere_mesher_used >= 1 && [GiD_Info Mesh NumElements Sphere] ) || $circle_mesher_used >= 1 && [GiD_Info Mesh NumElements Circle] } {
            #save only the .model file if has been used for a custom mesher of spheres or circles
            set must_save 1
        }
        if { !$must_save } {
            #if saving without use any custom sphere or circle mesher must delete old possible .model file
            if { [file exists $filename] } {
                file delete $filename
            }
        }
    } elseif { $owner == "PREFERENCES" } {
        set must_save 1
        #.preferences file
    } else {
        WarnWinText "PluginMeshVariablesClass_SaveObjects: Unexpected owner $owner"
        return 1
    }
    if { $must_save && [info exists ::GidPriv(RegisteredPluginMeshVariablesClass)] && [llength $::GidPriv(RegisteredPluginMeshVariablesClass)] } {
        set fp [open $filename w]
        if { $fp != "" } {
            foreach class_name $::GidPriv(RegisteredPluginMeshVariablesClass) {
                puts $fp "<class name='${class_name}'>"
                foreach obj [info class instances $class_name] {
                    set obj_owner [$obj Get Owner]
                    if { $obj_owner == $owner } {
                        #only save to disk the model set of meshing variables
                        puts $fp "<variables owner='[$obj Get Owner]'>"
                        foreach var [$obj GetVariables] {
                            puts $fp [list $var [$obj Get $var]]
                        }
                        puts $fp "</variables>"
                    } else {
                        #ignore other owner's objects
                    }
                }
                puts $fp "</class>"
            }
            close $fp
        }
    }
    return 0
}

proc PluginMeshVariablesClass_ReadObjects { filename owner } {
    if { ![file exists $filename] } {
        if { $owner == "PREFERENCES" } {
            #do not do this if $owner == "MODEL"
            set similar_gid_defaults [GetSimilarDefaultsFile [file tail $filename]]
            if { [file exists $similar_gid_defaults] } {
                set filename $similar_gid_defaults
            }
        }
    }
    if { [file exists $filename] } {
        set fp [open $filename r]
        while { ![eof $fp] } {
            gets $fp line ;#<class name='$xxx'>
            if { $line == "" } {
                continue
            }
            set class_name [string range $line 13 end-2]
            if { [info object isa object $class_name] && [info object isa class $class_name] } {
                gets $fp line ;#<variables owner='yyy'>
                set variable_owner [string range $line 18 end-2]
                if { $variable_owner == $owner } {
                    set obj ""
                    foreach item [info class instances $class_name] {
                        if { [$item Get Owner] == $variable_owner } {
                            set obj $item
                            $obj Reset
                            break
                        }
                    }
                    if { $obj == "" } {
                        set obj [$class_name new $owner]
                    }
                    while { ![eof $fp] } {
                        gets $fp line
                        if { $line == "</variables>"} {
                            gets $fp line ;#</class>
                            break
                        } else {
                            lassign $line key value
                            $obj Set $key $value
                        }
                    }
                } else {
                    #ignore other owner's objects
                }
            } else {
                #ignore this class that is not defined (e.g. the plugin has been removed)
                while { ![eof $fp] } {
                    gets $fp line
                    if { $line == "</class>"} {
                        break
                    }
                }
            }
        }
        close $fp
    }
    return 0
}

proc PluginMeshVariablesClass_GetClassObject { class_name owner } {
    #find the only obj of this class for this owner
    set the_obj ""
    foreach obj [info class instances $class_name] {
        set obj_owner [$obj Get Owner]
        if { $obj_owner == $owner } {
            set the_obj $obj
            break
        }
    }
    return $the_obj
}

proc PluginMeshVariablesClass_VariableManager { class_name operation var {value ""} } {
    set obj [PluginMeshVariablesClass_GetClassObject $class_name PREFERENCES]
    #obj is the obj of class class_name (inherit of GiDMeshVariables) that store PREFERENCES
    if { $obj != "" } {
        if { [lsearch [$obj GetVariables] $var] != -1 } {
            if { $operation == "GetValue" } {
                set res [$obj Get $var]
            } elseif { $operation == "SetValue" } {
                set res [$obj Set $var $value]
            } elseif { $operation == "GetDefaultValue" } {
                set res [$obj GetDefault $var]
            } else {
                WarnWinText "invalid operation $operation"
                set res 0
            }
        } else {
            WarnWinText "variable $var of class $class_name not found"
            set res 0
        }
    } else {
        WarnWinText "Object of class $class_name not found"
        set res 0
    }
    return $res
}

#to register this proc to modify the preferences window to add or remove thinks
proc GiD_RegisterPluginPreferencesProc { proc_name } {
    if { [info procs $proc_name] != "" && [llength [info args $proc_name]] != 1 } {
        return -code error [_ "Wrong procedure prototype, must be '%s'." "proc procname { root } body"]
    }
    if { ![info exists ::GidPriv(RegisteredPluginPreferencesProc)] } {
        set pos -1
    } else {
        set pos [lsearch $::GidPriv(RegisteredPluginPreferencesProc) $proc_name]
    }
    if { $pos == -1 } {
        lappend ::GidPriv(RegisteredPluginPreferencesProc) $proc_name
        set xmlfile Preferences.xml
        if { [info exists CreateWidgetsFromXml::fileread($xmlfile)] } {
            #the file has already been read when registering this proc, invoke it now because won't be invoked by CreateWidgetsFromXml::ReadXmlFile
            set root [CreateWidgetsFromXml::GetPreferencesXml]
            $proc_name $root
        }
    }
}

proc GiD_UnRegisterPluginPreferencesProc { proc_name } {
    if { ![info exists ::GidPriv(RegisteredPluginPreferencesProc)] } {
        return 1
    }
    set pos [lsearch $::GidPriv(RegisteredPluginPreferencesProc) $proc_name]
    if { $pos == -1 } {
        set ::GidPriv(RegisteredPluginPreferencesProc) [lreplace $::GidPriv(RegisteredPluginPreferencesProc) $pos $pos]
    }
}

#to be invoked when creating the data information of the preferences window
proc PluginPreferencesProcs_Eval { root } {
    PluginGenerateCircleMesh_UpdatePreferencesWindow
    PluginGenerateSphereMesh_UpdatePreferencesWindow
    if { [info exists ::GidPriv(RegisteredPluginPreferencesProc)] && [llength $::GidPriv(RegisteredPluginPreferencesProc)] } {
        foreach proc_name $::GidPriv(RegisteredPluginPreferencesProc) {
            Eval_GenericProc $proc_name $root
        }
    }
    return 0
}


proc GiD_RegisterPluginGenerateSphereMesh { value procedure label require_tetrahedra file_to_source } {
    set fail 0
    if { ![string is integer $value] } {
        set fail 1
        WarnWinText "GiD_RegisterPluginGenerateSphereMesh, invalid value=$value for SphereMesher"
    } elseif { $value == 0 } {
        set fail 1
        WarnWinText "GiD_RegisterPluginGenerateSphereMesh, invalid value=$value for SphereMesher (reserved for inner mesher)"
    } elseif { [info exists ::GidPriv(GenerateSphereMesh,$value,MeshProcedure)] } {
        set fail 1
        WarnWinText "GiD_RegisterPluginGenerateSphereMesh, invalid value=$value for SphereMesher (used for other plugin mesher)"
    } else {
        if { [llength $procedure] != 1 } {
            set fail 1
        } elseif { [info procs $procedure] != "" && [llength [info args $procedure]] != 4 } {
            set fail 1
        }
        if { $fail } {
            WarnWinText "GiD_RegisterPluginGenerateSphereMesh, invalid GenerateSphereMesh procedure, its prototype must be:"
            WarnWinText "proc procname { id_entity input_mesh input_boundary_parts size } {\n... \nreturn \[list \$nodes_coordinates \$element_radius \$output_boundary_parts]\n}"
        }
    }
    if { !$fail } {
        set ::GidPriv(GenerateSphereMesh,$value,MeshProcedure) $procedure
        set ::GidPriv(GenerateSphereMesh,$value,Labels) $label
        set ::GidPriv(GenerateSphereMesh,$value,RequireTetrahedra) $require_tetrahedra
        set ::GidPriv(GenerateSphereMesh,$value,FileToSource) $file_to_source
        #PluginGenerateSphereMesh_UpdatePreferencesWindow
        CreateWidgetsFromXml::ClearCachePreferences
    }
    return $fail
}

proc GiD_UnRegisterPluginGenerateSphereMesh { value procedure label } {
    set fail 0
    if { ![string is integer $value] } {
        set fail 1
        WarnWinText "GiD_UnRegisterPluginGenerateSphereMesh, invalid value=$value for SphereMesher"
    } elseif { $value == 0 } {
        set fail 1
        WarnWinText "GiD_UnRegisterPluginGenerateSphereMesh, invalid value=$value for SphereMesher (reserved for inner mesher)"
    } elseif { ![info exists ::GidPriv(GenerateSphereMesh,$value,MeshProcedure)] } {
        set fail 1
        WarnWinText "GiD_UnRegisterPluginGenerateSphereMesh, invalid value=$value for SphereMesher (was not registered)"
    } else {
        if { $procedure != $::GidPriv(GenerateSphereMesh,$value,MeshProcedure) } {
            set fail 1
            WarnWinText "GiD_UnRegisterPluginGenerateSphereMesh, procedure=$procedure not match $::GidPriv(GenerateSphereMesh,$value)"
        }
    }
    if { !$fail} {
        unset ::GidPriv(GenerateSphereMesh,$value,MeshProcedure)
        unset ::GidPriv(GenerateSphereMesh,$value,Labels)
        unset ::GidPriv(GenerateSphereMesh,$value,RequireTetrahedra)
        unset ::GidPriv(GenerateSphereMesh,$value,FileToSource)
        #PluginGenerateSphereMesh_UpdatePreferencesWindow
        CreateWidgetsFromXml::ClearCachePreferences
    }
    return $fail
}

proc PluginGenerateSphereMesh_GetIdsRegistered { } {
    set ids [list]
    foreach item [array names ::GidPriv GenerateSphereMesh,*,MeshProcedure] {
        lappend ids [lindex [split $item ,] 1]
    }
    return $ids
}

proc PluginGenerateSphereMesh_UpdatePreferencesWindow { } {
    #modify the window
    set ids [PluginGenerateSphereMesh_GetIdsRegistered]
    if { [llength $ids] } {
        set root [CreateWidgetsFromXml::GetPreferencesXml]
        set name sphere_mesher_rball
        set findnode [$root find "name" $name]
        if { $findnode != "" } {
            set xml_data {}
            foreach i $ids {
                set label [_ $::GidPriv(GenerateSphereMesh,$i,Labels)]
                append xml_data "<option value='$i' label='$label'/>"
            }
            CreateWidgetsFromXml::AddAfterName $root $name $xml_data
        }
    }
    return 0
}

proc GiD_GetRegisteredProcedureGenerateSphereMesh { value } {
    if { [info exists $::GidPriv(GenerateSphereMesh,$value,MeshProcedure)] } {
        set procedure $::GidPriv(GenerateSphereMesh,$value,MeshProcedure)
    } else {
        set procedure ""
    }
    return $procedure
}

####### similar for Circle mesher

proc GiD_RegisterPluginGenerateCircleMesh { value procedure label require_triangles file_to_source } {
    set fail 0
    if { ![string is integer $value] } {
        set fail 1
        WarnWinText "GiD_RegisterPluginGenerateCircleMesh, invalid value=$value for CircleMesher"
    } elseif { $value == 0 } {
        set fail 1
        WarnWinText "GiD_RegisterPluginGenerateCircleMesh, invalid value=$value for CircleMesher (reserved for inner mesher)"
    } elseif { [info exists ::GidPriv(GenerateCircleMesh,$value,MeshProcedure)] } {
        set fail 1
        WarnWinText "GiD_RegisterPluginGenerateCircleMesh, invalid value=$value for CircleMesher (used for other plugin mesher)"
    } else {
        if { [llength $procedure] != 1 } {
            set fail 1
        } elseif { [info procs $procedure] != "" && [llength [info args $procedure]] != 4 } {
            set fail 1
        }
        if { $fail } {
            WarnWinText "GiD_RegisterPluginGenerateCircleMesh, invalid GenerateCircleMesh procedure, its prototype must be:"
            WarnWinText "proc procname { id_entity input_mesh input_boundary_parts size } {\n... \nreturn \[list \$nodes_coordinates \$element_radius \$output_boundary_parts]\n}"
        }
    }
    if { !$fail } {
        set ::GidPriv(GenerateCircleMesh,$value,MeshProcedure) $procedure
        set ::GidPriv(GenerateCircleMesh,$value,Labels) $label
        set ::GidPriv(GenerateCircleMesh,$value,RequireTriangles) $require_triangles
        set ::GidPriv(GenerateCircleMesh,$value,FileToSource) $file_to_source
        #PluginGenerateCircleMesh_UpdatePreferencesWindow
        CreateWidgetsFromXml::ClearCachePreferences
    }
    return $fail
}

proc GiD_UnRegisterPluginGenerateCircleMesh { value procedure label } {
    set fail 0
    if { ![string is integer $value] } {
        set fail 1
        WarnWinText "GiD_UnRegisterPluginGenerateCircleMesh, invalid value=$value for CircleMesher"
    } elseif { $value == 0 } {
        set fail 1
        WarnWinText "GiD_UnRegisterPluginGenerateCircleMesh, invalid value=$value for CircleMesher (reserved for inner mesher)"
    } elseif { ![info exists ::GidPriv(GenerateCircleMesh,$value,MeshProcedure)] } {
        set fail 1
        WarnWinText "GiD_UnRegisterPluginGenerateCircleMesh, invalid value=$value for CircleMesher (was not registered)"
    } else {
        if { $procedure != $::GidPriv(GenerateCircleMesh,$value,MeshProcedure) } {
            set fail 1
            WarnWinText "GiD_UnRegisterPluginGenerateCircleMesh, procedure=$procedure not match $::GidPriv(GenerateCircleMesh,$value)"
        }
    }
    if { !$fail} {
        unset ::GidPriv(GenerateCircleMesh,$value,MeshProcedure)
        unset ::GidPriv(GenerateCircleMesh,$value,Labels)
        unset ::GidPriv(GenerateCircleMesh,$value,RequireTriangles)
        #PluginGenerateCircleMesh_UpdatePreferencesWindow
        CreateWidgetsFromXml::ClearCachePreferences
    }
    return $fail
}

proc PluginGenerateCircleMesh_GetIdsRegistered { } {
    set ids [list]
    foreach item [array names ::GidPriv GenerateCircleMesh,*,MeshProcedure] {
        lappend ids [lindex [split $item ,] 1]
    }
    return $ids
}

proc PluginGenerateCircleMesh_UpdatePreferencesWindow { } {
    #modify the window
    set ids [PluginGenerateCircleMesh_GetIdsRegistered]
    if { [llength $ids] } {
        set root [CreateWidgetsFromXml::GetPreferencesXml]
        set name circle_mesher_rball
        set findnode [$root find "name" $name]
        if { $findnode != "" } {
            set xml_data {}
            foreach i $ids {
                set label [_ $::GidPriv(GenerateCircleMesh,$i,Labels)]
                append xml_data "<option value='$i' label='$label'/>"
            }
            CreateWidgetsFromXml::AddAfterName $root $name $xml_data
        }
    }
    return 0
}

proc GiD_GetRegisteredProcedureGenerateCircleMesh { value } {
    if { [info exists $::GidPriv(GenerateCircleMesh,$value,MeshProcedure)] } {
        set procedure $::GidPriv(GenerateCircleMesh,$value,MeshProcedure)
    } else {
        set procedure ""
    }
    return $procedure
}
