
namespace eval GidDataManager {
    # set _host b2drop.bsc.es

    # production server:
    set _service      https
    # _host_proc is just to check against which data managet (_host) we are working.
    set _host_prod         neco.gidsimulation.com
    set _host         neco.gidsimulation.com

    # check env
    set _host_env ""
    if { [ info exists ::env(GID_NECO_HOSTNAME)]} {
        set _host_env $::env(GID_NECO_HOSTNAME)
    } 
    if { $_host_env != ""} {
        set _host $_host_env
    }

    # test server:
    # set _service      https
    # set _host         test-data.gidsimulation.com

    # front end url:
    set _front_end_url  https://app.gidsimulation.com
    # check env
    set _front_end_url_env ""
    if { [ info exists ::env(GID_PINGU_HOSTNAME)]} {
        set _front_end_url_env $::env(GID_PINGU_HOSTNAME)
    } 
    if { $_front_end_url_env != ""} {
        set _front_end_url $_front_end_url_env
    }
    
    # back end url:
    set _back_end_url  https://backend-app.gidsimulation.com
    # check env
    set _back_end_url_env ""
    if { [ info exists ::env(GID_BACKEND_HOSTNAME)]} {
        set _back_end_url_env $::env(GID_BACKEND_HOSTNAME)
    } 
    if { $_back_end_url_env != ""} {
        set _back_end_url $_back_end_url_env
    }

    set _webdav_path  /remote.php/dav/files/
    set _username     ""
    set _dama_token_pass     ""
    set _connected_path   ""
    set _delayed_connect  ""
    set _frame_in_window   ""
    set _show_datamanager_page_button    1
    set _busy_login    0
    set _lock_file     ""

    set _linux_mount_type    "davfs2"    ;# one of "wdfs" (user mount) or "davfs2" (requires root privileges)
    # with "davfs2" when doing ::connect {} will check if this is possible, if not will fall back to 'wdfs'

    set _debug 0

    array set _http_codes_msg {
        200 "DATAMANAGER_OK"
        201 "DATAMANAGER_FILE_OR_FOLDER_CREATED"
        204 "DATAMANAGER_FILE_OR_FOLDER_OVERWRITTEN_OR_DELETED"
        207 "DATAMANAGER_MULTISTATUS"
        207 "DATAMANAGER_ALREADY_REPORTED"
        401 "DATAMANAGER_NOT_AUTHENTICATED"
        404 "DATAMANAGER_FILE_OR_FOLDER_NOT_FOUND"
        405 "DATAMANAGER_FOLDER_ALREADY_EXISTS"
        409 "DATAMANAGER_PARENT_NODE_DOES_NOT_EXIST"
        507 "DATAMANAGER_INSUFFICIENT_STORAGE"
    }
    
    # extra errors, ?may be they are used?
    # 202 "DATAMANAGER_ACCEPTED"
    # 302 "Found"
    # 304 "Not Modified"
    # 400 "Bad Request"
    # 403 "Permission denied"
    # 408 "Request Timeout"
    # 411 "Length Required"
    # 419 "Expectation Failed"
    # 500 "Server Internal Error"
    # 501 "Server Busy"
    # 502 "Service Unavailable (Bad Gateway)"
    # 503 "Service Unavailable"
    # 504 "Service Temporarily Unavailable"
    # 505 "Internal Server Error"

    proc getHttpCodeMessage { code} {
        variable _http_codes_msg
        set msg "DATAMANAGER_UNKNOWN_CODE_$code"
        if { [ info exist _http_codes_msg($code)]} {
            set msg $_http_codes_msg($code)
        }
        return $msg
    }
}

if { $::tcl_platform(platform) == "windows" } {
    package require twapi
    package require twapi_share
}
package require percentencoding
package require base64

# required for GidDataManager::direct* and GidDataManager::http* commands
package require http
package require tls
if { [info exists ::env(http_proxy)] || [info exists ::env(https_proxy)] } {
    package require autoproxy
    autoproxy::init
    # it seems that here -autoservername 1 option is not valid
    http::register https 443 [list autoproxy::tls_socket]
} else {
    http::register https 443 [list ::tls::socket -autoservername 1]
}
package require tdom


proc GidDataManager::checkExecutionEnvirontment { } {
    variable _host_prod
    variable _host

    if { $_host != $_host_prod} {
        set msg "DATA MANAGER using DEVELOPMENT SERVER : $_host    Production server is : $_host_prod"
        GidDataManager::showMessage $msg
        W $msg
    }
}

proc GidDataManager::getHostUrl { } {
    variable _host
    variable _service
    return "${_service}://${_host}"
}

proc GidDataManager::getUsername {} {
    variable _username
    return $_username
}

proc GidDataManager::getUserpass { } {
    variable _dama_token_pass
    return $_dama_token_pass
}

proc GidDataManager::getUserPassHostUrl { } {
    variable _host
    variable _username
    variable _dama_token_pass
    variable _service

    return "${_service}://${_username}:${_dama_token_pass}@${_host}"
}

proc GidDataManager::getUserFoldername { } {
    variable _username
    # if { $::tcl_platform(os) == "Darwin"} {
    #     # on macOS it translates automatically username@ to username%40
    #     set user_folder ${_username}
    # } else {
    #     set user_folder [ percentencoding::encode ${_username}]
    # }
    if { $::tcl_platform(platform) == "windows" } {
        set user_folder [ percentencoding::encode ${_username}]
    } else {
        set user_folder ${_username}
    }
    return $user_folder
}

proc GidDataManager::getHostWebDavUrl { } {
    variable _host
    variable _webdav_path
    return "[ getHostUrl]${_webdav_path}"
}

proc GidDataManager::getWebDavPath { } {
    variable _host
    variable _webdav_path
    set user_folder [ GidDataManager::getUserFoldername]
    return "[ GidDataManager::getHostWebDavUrl]${user_folder}/"
}

proc GidDataManager::getUserWebDavPath { } {
    variable _webdav_path
    
    set user_folder [ GidDataManager::getUserFoldername]
    # net use x: needs /remote.php/dav/... and not /remote.php/webdav/....
    set ret_path "[ getUserPassHostUrl]${_webdav_path}${user_folder}/"
}

proc GidDataManager::encodeUrlPath { remote_path} {
    set lst_parts [ split $remote_path /]
    set lst_parts_url [ list]
    foreach part $lst_parts {
        set part_url [ percentencoding::encode [ encoding convertto utf-8 $part]]
        lappend lst_parts_url $part_url
    }
    return [ join $lst_parts_url /]
}

proc GidDataManager::getTwapiWnetSharePath { } {
    # set lst_shares [ twapi::wnet_connected_resources -type file]
    # if there is a share is like these:
    # {Y: {\\147.83.143.171\remote.php\dav\files\gid-admin}}
    # {Y: {\\147.83.143.171\remote.php\dav\files\miguel%40cimne.upc.edu}}
    # file nativename removes last '/'
    # if using https, the list is like this:
    # {Z: {\\neco.gidsimulation.com@SSL\remote.php\dav\files\miguel%40cimne.upc.edu}}
    # {Y: {\\neco.gidsimulation.com@SSL\remote.php\dav\files\miguel%40cimne.upc.edu}}
    variable _service
    variable _host
    variable _webdav_path

    if { [ string tolower $_service] == "https"} {
        set datamanager_share_path [ file nativename //${_host}@SSL${_webdav_path}]\\
    } else {
        # it's http
        set datamanager_share_path [ file nativename //${_host}${_webdav_path}]\\
    }
    return $datamanager_share_path
}

proc GidDataManager::showMessage { txt { parent_frame ""}} {
    GidUtils::SetWarnLine "GidDataManager: $txt"
    if { ( $parent_frame != "") && [ winfo exists $parent_frame] && [ winfo exists $parent_frame.ldata_manager]} {
        # eventually configure them to disconnect by the user ?
        $parent_frame.ldata_manager configure -text $txt
        # $parent_frame.data_manager configure -state disabled
    }
}

proc GidDataManager::_set_linux_mount_type { type } {
    variable _linux_mount_type
    if { $type == "wdfs"} {
        set _linux_mount_type $type
    } elseif { $type == "davfs2"} {
        # check if it is possible
        set mount_exists ""
        catch {
            set mount_exists [ exec which mount.davfs]
        }
        if { $mount_exists != ""} {
            set _linux_mount_type $type
        } else {
            set msg "mount.davfs not found in system. Please install davfs2 first, for instance 'sudo apt-get install davfs2' for better performance."
            set _linux_mount_type "wdfs"
            return -code error $msg
        }
    } else {
        set _linux_mount_type "wdfs"
        return -code error "Invalid linux mount type '$type'. Should be one of: wdfs davfs2."
    }
}

proc GidDataManager::getMountType { } {
    variable _linux_mount_type

    set mount_type $_linux_mount_type
    if { $::tcl_platform(platform) == "windows" } {
        set mount_type "twapi/netbios"
    } elseif { $::tcl_platform(os) == "Darwin"} {
        set mount_type "AppleScript"
    } else {
        # linux, 
        set mount_type $_linux_mount_type
    }
    return $mount_type
}

proc GidDataManager::_window_bottom_buttons { w_parent args } {
    #  usage: w_parent_window list_of_triplets{w_button_name txt cmd}
    # GidDataManager::_window_bottom_buttons w_name_ok ok ok_cmd w_name_cancel cancel cancel_cmd ...
    ttk::frame $w_parent.fButtons -takefocus 0
    set b_idx 0
    foreach [ list wb txt cmd] $args {
        ttk::button $w_parent.fButtons.${wb} -takefocus 1 \
            -text $txt -command $cmd
        grid $w_parent.fButtons.${wb} -padx 2 -pady 4 \
            -row 0 -column $b_idx
        grid columnconfigure $w_parent.fButtons $b_idx -weight 1
        incr b_idx
    }
    grid rowconfigure $w_parent.fButtons 0 -weight 1
    return $w_parent.fButtons
}

proc GidDataManager::_getLinuxWDFSexename {} {
    set full_exe [ file join $::GIDDEFAULT lib wdfs]
    if { ![ file exists $full_exe] || ![ file executable $full_exe]} {
        # try development location
        set full_exe [ file join $::GIDDEFAULT .. lib wdfs]
    }
    return $full_exe
}

# txt = prefix message when error
# args = cmd arg1 arg2 arg 3 ...
proc GidDataManager::_execute { txt args } {
    set err_txt --
    set err [
             catch {
                 exec {*}$args
             } err_txt
            ]
    if { $err} {
        W "$txt ERROR executing $args"
        W "    err_txt = $err_txt"
    }
    return $err
}

# same but dont display any error message
proc GidDataManager::_execute_no_error { txt args } {
    variable _debug
    
    set err_txt --
    set err [
             catch {
                 exec {*}$args
             } err_txt
            ]
    if { $err} {
        if { $_debug} {
            W "$txt EXECUTE_NO_ERROR executing $args"
            W "    err_txt = $err_txt"
        }
    }
    return $err
}

proc GidDataManager::_getDamaTokenPassFromLoginToken { username login_token} {
    # get payload of login_token
    if { $login_token == ""} {
        return ""
    }
    lassign [ split $login_token .] header payload signature
    set header [ binary decode base64 $header]
    set payload [ binary decode base64 $payload]

    # W "GidDataManager::_getDamaTokenPassFromLoginToken"
    # WV login_token "    "
    # WV "header payload" "    "

    # payload is a json object
    package require json

    set login_dict [ ::json::json2dict $payload]
    set dama_jwt ''
    if { [ dict exists $login_dict dama_jwt]} {
        set dama_jwt [ dict get $login_dict dama_jwt]
    }

    # check id dama_jwt is correct (2022-04-28 Licence manager generates an incorrect dama_jwt.uid  )
    # get dama_jwt payload
    lassign [ split $dama_jwt .] header payload signature
    set header [ binary decode base64 $header]
    set payload [ binary decode base64 $payload]
    set dama_dict [ ::json::json2dict $payload]

    set dama_uid ''
    set is_valid 0
    if { [ dict exists $dama_dict uid]} {
        set dama_uid [ dict get $dama_dict uid]
        lassign [ split $dama_uid :] lm_uid lm_email
        # WV "lm_uid lm_email"
        if { $lm_email == $username} {
            set is_valid 1
        }
    }
    set ret_dama_token_pass ''
    # WV is_valid
    if { $is_valid} {
        lassign [ split $dama_jwt .] header payload signature
        set ret_dama_token_pass ${payload}.${signature}
    }
    return $ret_dama_token_pass
}

proc GidDataManager::_getUsernameAndDamaTokenFromLogin { } {
    variable _username
    variable _dama_token_pass

    set _username [ GiD_Login logged_username ]
    set login_token [ GiD_Login logged_user_token ]
    # W "GidDataManager::_getDamaTokenPassFromLoginToken [ GiD_Login logged_username ] [ GiD_Login logged_user_token ]"
    set _dama_token_pass [ GidDataManager::_getDamaTokenPassFromLoginToken $_username $login_token]
    if { $_dama_token_pass == ""} {
        GidDataManager::showMessage "Invalid credentials to access data manager, please try signing out and in again."
    }
}

proc GidDataManager::_linux_check_mount_entry { mount_entry} {
    variable _linux_mount_type
    variable _debug

    set ret_unit_user [ list]

    # should be a single line
    # should be of the form:
    # wdfs (http://147.83.143.171/remote.php/dav/files/miguel%40cimne.upc.edu/) on /home/miguel/GiD-Cloud type fuse (rw,nosuid,nodev,relatime,user_id=1000,group_id=1000)
    # type (server) on mount_point
    #
    # macOs is very similar to Linux
    # but mount output is of the form:
    # http://147.83.143.171/remote.php/dav/files/miguel%2540cimne.upc.edu/ on /Volumes/miguel@cimne.upc.edu (webdav, nodev, noexec, nosuid, mounted by macuser)
    # server on mount_point

    # on Darwin last char is new line
    set last_char [ string index $mount_entry end]
    if { ( $last_char == "\n") || ( $last_char == "\r")} {
        set mount_entry [ string range $mount_entry 0 end-1]
    }

    set regexp_ok 0
    if { $::tcl_platform(os) == "Darwin"} {
        set regexp_mount_point {^([^\)]+) on (.*) \(.+\)$}
        set regexp_ok [ regexp $regexp_mount_point $mount_entry dummy server_path mount_path]
    } else {
        # linux: first try with wdfs mount type, then davfs2
        set regexp_mount_point {^wdfs \(([^\)]+)\) on (.*) type fuse \(.+$}
        set regexp_ok [ regexp $regexp_mount_point $mount_entry dummy server_path mount_path]
        if { $regexp_ok == 1} {
            set _linux_mount_type "wdfs"
        } else {
            set regexp_mount_point {^([^\)]+) on (.*) type fuse \(.+\)$}
            set regexp_ok [ regexp $regexp_mount_point $mount_entry dummy server_path mount_path]
            if { $regexp_ok == 1} {
                set _linux_mount_type "davfs2"
            } else {
                # set _linux_mount_type ""
            }
        }
    }
    if { $regexp_ok == 1} {
        # now get the info
        set webdav_url [ GidDataManager::getHostWebDavUrl]
        if { [ regexp "^${webdav_url}(.*)/$$" $server_path dummy user_folder_name] == 1} {
            # $user_folder_name is using percent encoding
            set ret_unit_user [ list $mount_path $user_folder_name]
        } else {
            # W "could not parse $server_path != $webdav_url"
            # is not out datamanager_server url ($datamanager_share_path)
        }
    }
    if { $_debug} {
        if { $regexp_ok == 1} {
            W "Mount point found $ret_unit_user. Type = $_linux_mount_type"
        } else {
            W "Mount point not found."
        }
    }
    return $ret_unit_user
}

# returns {{local_path_to_share remote_user_folder_name} {.. ..}}
# for instance in windows : { {y: miguel%40cimne.upc.edu} {z: gid-admin}}
proc GidDataManager::_getListExistingConnections { } {
    variable _host
    variable _webdav_path

    set ret_unit_user [ list]
    if { $::tcl_platform(platform) == "windows" } {
        set lst_shares [ twapi::wnet_connected_resources -type file]
        # if there is a share is like these:
        # {Y: {\\147.83.143.171\remote.php\dav\files\gid-admin}}
        # {Y: {\\147.83.143.171\remote.php\dav\files\miguel%40cimne.upc.edu}}
        # file nativename removes last '/'
        # if using https, the list is like this:
        # {Z: {\\neco.gidsimulation.com@SSL\remote.php\dav\files\miguel%40cimne.upc.edu}} 
        # {Y: {\\neco.gidsimulation.com@SSL\remote.php\dav\files\miguel%40cimne.upc.edu}}
        set datamanager_share_path [ GidDataManager::getTwapiWnetSharePath ]
        foreach share_entry $lst_shares {
            lassign $share_entry share_unit server_path
            # sometimes twapi::wnet_connected_resources returns shares with no unit connection...
            if { $share_unit != ""} {
                set regexp_path [ regsub -all {\\} $datamanager_share_path {\\\\}]
                if { [ regexp "^${regexp_path}(.*)$$" $server_path dummy user_folder_name] == 1} {
                    # $user_folder_name is using percent encoding
                    lappend ret_unit_user [ list $share_unit $user_folder_name]
                } else {
                    # W "could not parse $server_path != $datamanager_share_path"
                    # is not out datamanager_server url ($datamanager_share_path)
                }
            }
        }
    } elseif { ( $::tcl_platform(os) == "Linux") || ( $::tcl_platform(os) == "Darwin") } {
        # WarnWinText "mount |grep 147"
        # or  WarnWinText "mount -l |grep 147"
        set fi [ file tempfile tmp_filename]
        set err [ GidDataManager::_execute_no_error "GidDataManager::_getListExistingConnections" \
                      mount | grep ${_host} > $tmp_filename \
                     ]
        if { !$err} {
            set mount_entry [ read $fi]

            set unit_user [ GidDataManager::_linux_check_mount_entry $mount_entry]

            if { [ llength $unit_user] != 0} {
                lappend ret_unit_user $unit_user
            }
        }
        close $fi
        file delete -force $tmp_filename
    } else {
        WarnWinText "Unknown platform $::tcl_platform(platform) / $::tcl_platform(os)"
    }
    return $ret_unit_user
}

proc GidDataManager::doWindowsTrickToConnect { } {
    variable _host
    variable _username
    variable _dama_token_pass
    variable _debug

    if { $_debug} {
        GidDataManager::showMessage "Problems found with DNS+Netbios: trying something ..."
    }
    GidDataManager::showMessage "... trying again 1 ..."

    # # check again status for the case:
    # set full_status [ GidDataManager::getStatus]
    # lassign $full_status connection_status connected_path connected_user_folder
    # if { $connection_status != "NOT CONNECTED"} {
    #     if { "$connection_status" == "CONNECTED"} {
    #         if { $connected_path == "" } {
    #             #it seems that is connected but not mapped to a unit
    #             GidDataManager::_doConnect
    #         }
    #         GidDataManager::setConnectedPath $connected_path
    #         GidDataManager::showMessage "Already connected ( [ GidDataManager::getConnectedPath] ) (2)"
    #         GidDataManager::_doManageConnectEvents
    #         return
    #     }
    # }

    set err 0
    set err_txt ""
    # https://www.tcl.tk/community/tcl2004/Tcl2003papers/kupries-doctools/tcllib.doc/dns/tcllib_dns.html
    # 1st get data manager's ip
    if { $_debug} {
        GidDataManager::showMessage "...getting $_host ip"
    }
    package require dns
    # default timeout is 30000ms
    ::dns::configure -timeout 10000
    # CIMNE's dns is broken (147.83.143.133)
    set dns_token [ ::dns::resolve $_host -server 8.8.8.8]
    set status [ ::dns::wait $dns_token]
    if { $_debug} {
        W "dns status '$status' : [ ::dns::error $dns_token]"
    }
    if { $status != "ok"} {
        # may be error, timeout or eof
        set err 1
        set err_txt "dns error '$status' : [ ::dns::error $dns_token]"
        # GidDataManager::showMessage $err_txts
        ::dns::cleanup $dns_token
        return [ list $err $err_txt ]
    }

    # 2nd try to connect using the IP directly --> should fail
    # neco.gidsimulation.com uses CloudFront thus
    # ::dns::address $dns_token returns a list of addresses from CloudFront
    # use 1st one
    set host_ip [ lindex [ ::dns::address $dns_token] 0]
    if { $_debug} {
        GidDataManager::showMessage "...connecting to $host_ip --> should fail"
        # should fail because CloudFront only forwards petitions using the URL
        # i.e. only using neco.gidsimulation.com will forward the calls to the correct server
        # i.e. petitions that use IP @ will not work
    }
    # try again but with the ip, will issue an error, but then we can use it as always
    set old_host $_host
    set _host $host_ip
    if { $_debug} {
        W "trying twapi::connect_share [ GidDataManager::getWebDavPath] -user ${_username} -password ${_dama_token_pass} -localdevice *"
    }
    set err [
        catch {
            set unit [ twapi::connect_share [ GidDataManager::getWebDavPath] -user ${_username} -password ${_dama_token_pass} -localdevice * ]
        } err_txt
    ]
    if { $_debug} {
        WV "err err_txt"
    }
    if { !$err} {
        # disconnect
        set err [ catch {
            twapi::disconnect_share $unit
        } err_txt]
        if { $_debug} {
            GidDataManager::showMessage "disconnecting $err : $err_txt"
        }
    } else {
        if { $_debug} {
            GidDataManager::showMessage "got expected error: $err_txt"
        }
    }

    # 3rd try again
    set _host $old_host
    if { $_debug} {
        GidDataManager::showMessage "...now trying again with $_host"
    }
    GidDataManager::showMessage "... trying again 2 ..."
    if { $_debug} {
        W "trying twapi::connect_share [ GidDataManager::getWebDavPath] -user ${_username} -password ${_dama_token_pass} -localdevice *"
    }
    set err [
        catch {
            set unit [ twapi::connect_share [ GidDataManager::getWebDavPath] -user ${_username} -password ${_dama_token_pass} -localdevice * ]
            set err_txt $unit
        } err_txt
    ]
    if { $_debug} {
        WV "err err_txt"
    }
    ::dns::cleanup $dns_token

    # 4th try
    if { $err} {
        if { $_debug} {
            GidDataManager::showMessage "...now trying again with net use"
        }
        GidDataManager::showMessage "... trying again 3 ..."
        set ok ""
        set err [
            catch {
                set ok [ exec net use [ GidDataManager::getWebDavPath] ${_dama_token_pass} /USER:${_username}]
            } err_txt
        ]
        if { !$err} {
            # it was succesfull, but don't know which unit
            set full_status [ GidDataManager::getStatus]
            lassign $full_status connection_status connected_path connected_user_folder
            if { $connection_status == "CONNECTED"} {
                set unit $connected_path
                set err_txt $unit
            }
        }
        if { $_debug} {
            WV "ok err err_txt"
        }
    }
    return [ list $err $err_txt]
}

proc GidDataManager::_linux_wdfs_mount_user_folder { destination_mount} {
    variable _username
    variable _dama_token_pass
    variable _debug

    # set wdfs_options "-o auto_cache -o accept_sslcert"
    # set wdfs_options "-o auto_cache -o accept_sslcert -o big_writes -o splice_write"
    # set wdfs_options "-o accept_sslcert -o kernel_cache"
    # with big_writes, defaults to 128K ( 0x00020000 )bytes in a single write
    # which is the maximum hard-coded in fuse, despite -o max_write=4194304
    set wdfs_options "-o accept_sslcert -o big_writes -o splice_write -o splice_read -o splice_move"
    append wdfs_options " -o atomic_o_trunc -o kernel_cache"
    # append wdfs_options " -o locking"  ;# this option causes some troubles with editors like vi or emacs
    set err [ GidDataManager::_execute_no_error "GidDataManager::_linux_wdfs_mount..." \
                  [ GidDataManager::_getLinuxWDFSexename] [ GidDataManager::getWebDavPath] $destination_mount \
                  -o username=${_username} -o password=${_dama_token_pass} \
                  {*}$wdfs_options \
                 ]
    if { $_debug} {
        W "[ GidDataManager::_getLinuxWDFSexename] [ GidDataManager::getWebDavPath] $destination_mount \
            -o username=${_username} -o password=${_dama_token_pass} \
             $wdfs_options "
        W "err = $err"
    }
}

proc GidDataManager::_linux_davfs2_mount_user_folder { destination_mount} {
    variable _username
    variable _dama_token_pass
    variable _debug

    set whoami [ gid_cross_platform::getUsername]
    set mount_options "-o _netdev,noauto,user,uid=${whoami},gid=${whoami}"
    set fo [ file tempfile tmp_filename]
    puts $fo ${_username}
    puts $fo ${_dama_token_pass}
    close $fo
    if { $whoami == "root"} {
        set cmdline2exec [ list \
            mount -t davfs {*}$mount_options \
                [ GidDataManager::getWebDavPath] $destination_mount \
                < $tmp_filename]
    } else {
        # needs sudo like gui program to elevate privileges
        set cmdline2exec [ gid_cross_platform::get_run_as_administrator_cmdline \
            mount -t davfs {*}$mount_options \
                [ GidDataManager::getWebDavPath] $destination_mount \
                < $tmp_filename]
    }
    set err 0
    set err [ GidDataManager::_execute_no_error "GidDataManager::_linux_davfs2_mount..." \
        {*}$cmdline2exec]
    file delete -force $tmp_filename
    if { $_debug} {
        W "$cmdline2exec"
        W "err = $err"
    }
    return $err
}

proc GidDataManager::_doConnect {} {
    variable _host
    variable _username
    variable _dama_token_pass
    variable _linux_mount_type
    variable _debug

    # W GidDataManager::_doConnect
    # W "    _username = $_username"
    # W "    _dama_token_pass = $_dama_token_pass"

    if { ( $_username == "") || ( $_dama_token_pass == "")} {
        GidDataManager::showMessage "Invalid or null credentials to access data manager."
        set err 1
        return $err
    }
    
    set err 0

    if { $::tcl_platform(platform) == "windows" } {        
        set user_folder [ GidDataManager::getUserFoldername]
        # net use * [ GidDataManager::getWebDavPath]  XXpasswordXX /USER:${_username}
        # net use x: needs /remote.php/dav/... and not /remote.php/webdav/....
        # W "doing twapi::connect_share [ GidDataManager::getWebDavPath] -user ${_username} -password XXXXXXXXXX -localdevice *"
        # W "token = ${_dama_token_pass}"
        if { $_debug} {
            W "twapi::connect_share [ getWebDavPath] -user ${_username} -password ${_dama_token_pass} -localdevice *"
        }
        set err [
            catch {
                set unit [ twapi::connect_share [ GidDataManager::getWebDavPath] -user ${_username} -password ${_dama_token_pass} -localdevice * ]
                # set unit [ twapi::connect_share [ GidDataManager::getWebDavPath] -user ${_username} -password [ string range [ GiD_Login logged_user_token ] 0 254] -localdevice * ]
            } err_txt
        ]
        if { $_debug} {
            WV "err err_txt"
        }

        if { $err} {
            lassign [ GidDataManager::doWindowsTrickToConnect] err err_txt
            if { !$err} {
                # if succeded, err_txt contains the unit
                set unit $err_txt
            }
        }

        if { !$err} {
            GidDataManager::setConnectedPath $unit
        } else {
            # like linux and macos using ::_execute ....
            set txt "twapi::connect_share"
            set argss "[ GidDataManager::getWebDavPath] -user ${_username}"            
            if { [GiD_Set Cloud(AutomaticConnect)] } {
                set msg [_ "%s connect error" $txt]
                append msg ": $err_txt"
                GidDataManager::showMessage $msg
            } else {
                set msg "$txt ERROR mounting"
                append msh \n $argss
                append msg \n "    err_txt = $err_txt"
                GidDataManager::showMessage $msg
            }
        }

        # test
        # set result [glob -directory $unit *]
        # puts $result
        # return $result
    } elseif { $::tcl_platform(os) == "Darwin"} {
        # W "mount volume \"[ GidDataManager::getWebDavPath]\" as user name \"${_username}\" with password \"${_dama_token_pass}\"\n"
        set osa_script "tell application \"Finder\"\n"
        append osa_script "try\n"
        append osa_script "mount volume \"[ GidDataManager::getWebDavPath]\" as user name \"${_username}\" with password \"${_dama_token_pass}\"\n"
        append osa_script "end try\n"
        append osa_script "end tell\n"
        if { $_debug} {
            W "osascript -e $osa_script"
        }
        set err [ GidDataManager::_execute "GidDataManager::_doConnect" \
                      osascript -e $osa_script \
                     ]
        if { !$err} {
            GidDataManager::setConnectedPath [ file join /Volumes ${_username}]
        } else {
            if { $_debug} {
                W "osascript mount error = $err"
            }
        }
    } elseif { $::tcl_platform(os) == "Linux" } {
        # WarnWinText "wdfs http://147.83.143.171/remote.php/dav/files/miguel%40cimne.upc.edu/ $HOME/pepe \
        #            -o username=<user-name> -o password=<password> -o accept_sslcert"
        # set destination_mount [ file join $::env(HOME) "GiD Cloud"]
        # using 'GiD-Clound' instead of 'GiD Cloud' so that mount.davfs can use it
        set destination_mount [ file join $::env(HOME) "GiD-Cloud"]
        # first check if it exists
        if { ![ file isdirectory $destination_mount]} {
            set err [
                     catch {
                         file mkdir $destination_mount
                     } err_txt
                    ]
            if { $err} {                
                if { [GiD_Set Cloud(AutomaticConnect)] } {
                    GidDataManager::showMessage [_ "Could not create mount directory %s" $destination_mount]\n[_ "Please, check permissions."]
                } else {
                    WarnWinText [_ "Could not create mount directory %s" $destination_mount]\n[_ "Please, check permissions."]
                }
                return $err
            }
        }

        set mnt_lst_prev [ GidDataManager::_getListExistingConnections]

        if { $_linux_mount_type == "davfs2"} {
            # davfs2 type
            # run_as_administrator (requires graphics):
            set whoami [ gid_cross_platform::getUsername]
            # mount davfs only if ( we have graphics) or ( we are root)
            if { ( $whoami == "root") || ![ GidUtils::IsTkDisabled] } {
                GidDataManager::_linux_davfs2_mount_user_folder $destination_mount
            } else {
                # we are not root and without graphics, mouont as wdfs
                GidDataManager::_set_linux_mount_type "wdfs"
                # we will enter in the next if {} and mount folder
            }
        }
                
        if { $_linux_mount_type == "wdfs"} {
            GidDataManager::_linux_wdfs_mount_user_folder $destination_mount
        }

        # using ::_execute_no_error to avoid displaying error when execution was ok (i.e. mounted)
        # wdfs displays WARNING: untrusted server certificate for '*.neco.gidsimulation.com' for Let's encrypt
        # also returns an error but the share is mounted, so let's check if it's been mounted
        set mnt_lst [ GidDataManager::_getListExistingConnections]
        # W "mount list before = $mnt_lst_prev"
        # W "mount list after = $mnt_lst"
        lassign [ lindex $mnt_lst 0] mnt_pnt mnt_user
        # WV "mnt_pnt mnt_user"
        if { ( $mnt_pnt == $destination_mount) && ( $mnt_user == [ GidDataManager::getUserFoldername])} {
            # it was mounted correctly...
            set err 0
        }
        if { !$err} {
            GidDataManager::setConnectedPath $destination_mount
        }
    } else {
        WarnWinText "Unknown platform $::tcl_platform(platform) / $::tcl_platform(os)"
    }
    if { $err} {
        GidDataManager::showMessage "Invalid credentials to access data manager, please try signing out and in again."
        set _username ""
        set _dama_token_pass ""
    } else {
        GidDataManager::showMessage "Connected to [ GidDataManager::getConnectedPath]"
    }
    return $err
}

proc GidDataManager::_doDisconnect {} {
    variable _host
    variable _username
    variable _dama_token_pass

    # now gid creates and maintains open the file modelname.gid/modelname.lck
    set projectname [ GiD_Info project modelname]
    if { [GidDataManager::isCloudPath $projectname] } {
        # There is no 'close project' in GiD
        # But need to start a new project so that it releases del modelname.gid/modelname.lck
        # and the cloud folder can be unmounted
        
        # without parameter to ask the user to save changes if present:
        DoFilesNew ASK
    }
    
    # when disconnecting close and delete the file created and left open when ::connect 
    # (which was done to avoid other gid's to disconnect the mounted shared network path)

    set err_txt ""
    set cloud_path [ GidDataManager::getConnectedPath]
    if { $::tcl_platform(platform) == "windows" } {
        # in windows cloud_path is a unit
        if { $cloud_path != "" } {
            set err [ catch {
                twapi::disconnect_share $cloud_path
            } err_txt]
            if { !$err} {
                set err_txt ""
                GidDataManager::unsetConnectedPath
            }
        }
    } elseif { ( $::tcl_platform(os) == "Linux") || ( $::tcl_platform(os) == "Darwin") } {
        # very similar Linux and macOS
        # WarnWinText "fusermount -u $HOME/pepe/"

        # TODO  check this:
        # if mounted with "sudo mount /home/gid/GiD-Cloud"
        # the this here triggers an error
        # but command line "fusermount -u /home/gid/GiD-Cloud" does work.

        set cmd "fusermount -u -z"
        # -z flag = lazy unmount (works even if resource is still busy)
        if {  $::tcl_platform(os) == "Darwin"} {
            set cmd umount
        }
        # in linux or macos cloud_path is a path ( from $HOME)
        if { $cloud_path != "" } {
            # give it some retries
            set num_retries 3
            set print_cmd W
            if { [GiD_Set Cloud(AutomaticDisconnect)]} {
                set print_cmd GidDataManager::showMessage
            }
            for { set idx 0} { $idx < $num_retries} { incr idx} {
                set err [
                    catch {
                        exec {*}$cmd $cloud_path 
                    } err_txt
                ]
                if { !$err} {
                    GidDataManager::unsetConnectedPath
                    break
                }
                $print_cmd "try # [ expr $idx + 1] of $num_retries : $err_txt"
                if { [ string first busy $err_txt] == -1} {
                    #there was an error
                    $print_cmd "GidDataManager::_doDisconnect ERROR executing $cmd $cloud_path"
                    $print_cmd "    $err_txt"
                    break
                }
                after 1000
            }
        }
    } else {
        WarnWinText "Unknown platform $::tcl_platform(platform) / $::tcl_platform(os)"
    }

    # may be the path was already unmounted by another GiD:
    set full_status [ GidDataManager::getStatus]
    lassign $full_status connection_status connected_path connected_user_folder
    if { $connection_status != "CONNECTED"} {
        GidDataManager::unsetConnectedPath
        if { $err_txt != ""} {
            GidDataManager::showMessage $err_txt
        }
        set err_txt ""
    }
    return $err_txt
}

# returns 1 if disconnected
proc GidDataManager::askAndDisconnect { { f_frame "" }} {    
    set disconnected 0
    set local_cloud_path [ GidDataManager::getConnectedPath]
    if { $local_cloud_path != ""} {
        # should also check if a calculation is running ?ussing this share path?
        lassign [ AreProcessRunning ] runninglocal runningremote comunicatingremote
        set in_use_by_calculation 0
        if { $runninglocal} {
            # check if output file or project is in connected share
            set output_filenames [ PWFIndOutputFilenameCurrent]
            foreach outfile $output_filenames {
                if { [ string first $local_cloud_path $outfile] == 0} {
                    set in_use_by_calculation 1
                    break
                }
            }
            if { !$in_use_by_calculation} {
                set current_project_name [ GiD_Info Project ModelName]
                if { [ string first $local_cloud_path $outfile] == 0} {
                    set in_use_by_calculation 1
                }
            }
        }
        set question_msg [_ "Do you want to disconnect from data manager?"]
        set text ""
        if { $::tcl_platform(platform) == "windows" } {
            append text [_ "Unit %s" $local_cloud_path]
        } else {
            append text [_ "Path %s" $local_cloud_path]
        }
        if { $in_use_by_calculation} {
            append text "\n"
            append text [_ "(Used by one process running.)"]
        }
        
        # check if other gid's use it
        set number_of_gids_left 0;#[ GidDataManager::decrNumberOfGiDs]
        if { $number_of_gids_left != 0} {
            append text "\n"
            append text [_ "(Used by %d other %s's)" $number_of_gids_left $::GidPriv(ProgName,Default)]
        }
        
        set parent [ winfo toplevel $f_frame]
        if { ![ winfo exist $parent]} {
            if { [ winfo exists .gid]} {
                set parent .gid
            } else {
                set parent .
            }
        }
        set retval [MessageBoxOptionsButtons [_ "Data manager"] "$question_msg\n\n$text" {yes no} [list [_ "Yes"] [_ "No"]] question "" 0 $parent]
        if { $retval == "yes"} {
            GidDataManager::showMessage "Disconnecting  ( [ GidDataManager::getConnectedPath] ) ..." $f_frame
            set err_txt [ GidDataManager::_doDisconnect]
            if { $err_txt != ""} {
                if { [GiD_Set Cloud(AutomaticDisconnect)] } {
                    set msg [_ "Problems disconnecting '%s'" $local_cloud_path]
                    append msg ": $err_txt"
                    GidDataManager::showMessage $msg
                } else {
                    WarnWin [_ "Problems disconnecting '%s':" $local_cloud_path]\n$err_txt\n[_ "Please resolve issue and try again."]
                }
            } else {
                set disconnected 1
                GidDataManager::showMessage "Disconnected" $f_frame
            }
        }
    }
    
    return $disconnected
}

# returns 1 if disconnected
proc GidDataManager::HandleDisconnect { force_disconnect {f_frame ""}} {    
    #kike: changed behaviour: always must be automatic, never with user intervention (at least in start/end, it can ask is an user click some button)
    #  if is set the checkbox Cloud(AutomaticDisconnect)==1 then automatically try to disconnect
    #  but otherwise do not ask the user every time to disconnect or not, simply do not try to disconnect if was connected
    set disconnected 0
    set cloud_automatic_disconnect [GiD_Set Cloud(AutomaticDisconnect)]
    if { $force_disconnect || $cloud_automatic_disconnect } {
        set local_cloud_path [ GidDataManager::getConnectedPath]
        if { $local_cloud_path != ""} {            
            GidDataManager::showMessage "Disconnecting  ( [ GidDataManager::getConnectedPath] ) ..." $f_frame
            set err_txt [ GidDataManager::_doDisconnect]
            if { $err_txt != ""} {
                if { $cloud_automatic_disconnect } {
                    set msg [_ "Problems disconnecting '%s'" $local_cloud_path]
                    append msg ": $err_txt"
                    GidDataManager::showMessage $msg
                } else {
                    WarnWinText [_ "Problems disconnecting '%s':" $local_cloud_path]\n$err_txt\n[_ "Please resolve issue and try again."]
                }
            } else {
                set disconnected 1
                GidDataManager::showMessage "Disconnected." $f_frame
            }     
        } else {
            set disconnected 1
            GidDataManager::showMessage "already disconnected." $f_frame
        }
    }
    return $disconnected
}

proc GidDataManager::HandleDisconnectBeforeExit { } {   
    # to GidDataManager::_doManageDisconnectEvents and
    # launch GiD_Event_AfterDataManagerDisconnect
    # W "GidDataManager::HandleDisconnectBeforeExit"
    set force_disconnect false
    set disconnected [ GidDataManager::disconnect $force_disconnect]
}

proc GidDataManager::doDelayedConnect { } {
    variable _delayed_connect
    GidDataManager::connect
    set _delayed_connect ""
}

proc GidDataManager::checkStatusAndAutoConnect { } {
    variable _username
    variable _dama_token_pass
    variable _delayed_connect    

    # W "GidDataManager::checkStatusAndAutoConnect"
    # GidDataManager::showMessage  "GidDataManager::checkStatusAndAutoConnect"

    # eventually it has not been read
    # lassign [ GidDataManager::ReadTokenFromFile] token_user token_pass
    # if { ( $token_user != "") && ( $token_user == $_username )} {
    #     set _dama_token_pass $token_pass
    # }

    if { $::GID_ENABLE_DATAMANAGER == 2} {
        # only direct/http mode allowed
        return
    }

    # some way to get password...
    # check status
    # first check if there is already connection to datamanger server
    set full_status [ GidDataManager::getStatus]
    lassign $full_status connection_status connected_path connected_user_folder
    # set connected_user [ percentencoding::decode $connected_user_folder]
    # W "    $connection_status"
    set cloud_automatic_connect [GiD_Set Cloud(AutomaticConnect)]

    if { "$connection_status" == "CONNECTED"} {
        if { $connected_path == "" } {
            #it seems that is connected but not mapped to a unit
            GidDataManager::_doConnect
        }
        GidDataManager::setConnectedPath $connected_path
        GidDataManager::showMessage "Already connected ( [ GidDataManager::getConnectedPath] , [ GidDataManager::getMountType] )"
        GidDataManager::_doManageConnectEvents
    } elseif { "$connection_status" == "OTHER USER CONNECTED"} {
        # ?ask to disconnect and reconnect?
        # give GiD some time to load (gid_theme on Darwin)
        if { $cloud_automatic_connect} {
            if { $_delayed_connect == ""} {
                set _delayed_connect [ after 500 GidDataManager::doDelayedConnect]
            }
        }
    } else {
        # "NOT CONNECTED"
        # give GiD some time to load (gid_theme on Darwin)
        if { $cloud_automatic_connect} {
            if { $_delayed_connect == ""} {
                set _delayed_connect [ after 500 GidDataManager::doDelayedConnect]
            }
        }
    }

    if { $::GID_ENABLE_DATAMANAGER == 1 } {
        #it could be a problem if cannot set it (e.g. because doesn' has privileges) asking again and again
        #must do something (like save in gid.ini to not ask more?)
        set action 0 ;#0 do nothing, 1 ask to increase now, 2 show message that explain how to do it
        if { $action!=0 && $::tcl_platform(platform) == "windows" } {
            #increase Windows default filesizelimit of about 50Mb to 4GB
            set filesizelimit [webclient_get_filesizelimit]
            if { $filesizelimit != -1 && $filesizelimit != 4294967295} {
                #-1==4294967295 (4GB is returning -1 because is considered signed int)
                if { $action == 1 } {
                    set message [_ "Current file size limit %s MB too small, Try to increase it to 4GB?" [format "%.1f" [expr double($filesizelimit)/(1024*1024)]]]
                    set answer [MessageBoxOptionsButtons [_ "Data manager"] $message {0 1} [list [_ "Yes"] [_ "No"]] question ""]
                    if { $answer == 0 } {
                        package require gid_cross_platform
                        if { [lindex [file split [gid_filesystem::get_folder_gid]] 1] == "gid_project" } {
                            #developement environment
                            #in developer environment there is a problem, gid_64.exe cannot be directly run 
                            #without set previously add to environment variable PATH pointing to lib\x64 (or x32 respectively)
                            set exe [file join [gid_filesystem::get_folder_gid] gid_32.exe]
                        } else {
                            set exe [GidUtils::GetFullPathGiDExe]
                        }
                        #gid_cross_platform::run_as_administrator $exe -tclsh [file join [gid_filesystem::get_folder_standard scripts] run_as_administrator_procs.tcl] webclient_set_filesizelimit 4294967295
                        gid_cross_platform::run_as_administrator $exe -tclsh [file join [gid_filesystem::get_folder_standard scripts] run_as_administrator_procs.tcl] webclient_set_filesizelimit_and_restart_webclient 4294967295
                    }
                } elseif { $action == 2 } {
                    set message [_ "Current file size limit %s MB too small. To to increase it start GiD once as administrator and write in the command line" [format "%.1f" [expr double($filesizelimit)/(1024*1024)]]]
                    GidUtils::SetWarnLine "$message: -np- webclient_set_filesizelimit_and_restart_webclient"
                }
            }
        }
    }
}


proc GidDataManager::disconnect { force_disconnect { f ""}} {
    set disconnected [ GidDataManager::HandleDisconnect $force_disconnect $f]
    if { $disconnected} {
        GidDataManager::_doManageDisconnectEvents
    }
    return $disconnected
}

proc GidDataManager::_doManageConnectEvents {} {
    # Avoid double registration to avoid being called twice consecutively
    # if DataManager has not registered to BeforeExit
    # then it's first time it registers --> register and call user AfterDataManagerConnect event
    if { ! [ GiD_GetIsRegisteredEventProc GiD_Event_BeforeExit GidDataManager::HandleDisconnectBeforeExit]} {
        GiD_RegisterEvent GiD_Event_BeforeExit GidDataManager::HandleDisconnectBeforeExit
        after idle [ list GiD_RaiseEvent GiD_Event_AfterDataManagerConnect [ GidDataManager::getConnectedPath ]]
    }
}

proc GidDataManager::_doManageDisconnectEvents { } {
    # may be we cloud unit has already been disconnected, and event unregistered...
    # need to check
    set lst [ GiD_GetRegisteredEventProcs GiD_Event_BeforeExit]
    if { [ lsearch $lst GidDataManager::HandleDisconnectBeforeExit] != -1} {
        # unregister BeforeExit and call user AfterDataManagerDisconnect
        GiD_UnRegisterEvent GiD_Event_BeforeExit GidDataManager::HandleDisconnectBeforeExit
        # when doing GidDataManager::HandleDisconnectBeforeExit
        # this after idle will never be called as the Tcl interpreter's will be destroyed.
        after idle [ list GiD_RaiseEvent GiD_Event_AfterDataManagerDisconnect ]
    }
}

proc GidDataManager::isConnected {} {
    set is_doConnected 0
    if { [ GidDataManager::getConnectedPath] != ""} {
        set is_doConnected 1
    }
    return $is_doConnected
}

proc GidDataManager::getStatus { } {
    # returns {status path_if_connected user_if_connected}
    # status = "CONNECTED" "NOT CONNECTED" "OTHER USER CONNECTED"
    # path_if_connected = "" "y:" or "/a/b/c"
    # user_if_connected = "" "username@cimne.upc.edu" "local_user"

    set lst_unit_userfolder [ GidDataManager::_getListExistingConnections]
    set ret_status "NOT CONNECTED"
    set ret_path ""
    set ret_user ""
    if { [ llength $lst_unit_userfolder] != 0} {
        # go through all connections and check if our user is connected
        set my_user_connected 0
        set other_user_connected 0
        foreach connected $lst_unit_userfolder {
            lassign $connected connected_path connected_user_folder
            set my_folder_name [ GidDataManager::getUserFoldername]
            #kike: GidDataManager::_getListExistingConnections using twapi return escolano@cimne.upc.edu, 
            #      but GidDataManager::getUserFoldername replace @ by %40 and then was not matching,
            #      what convenio must be used of both?? by now compare and accept both
            if { $my_folder_name == $connected_user_folder || $my_folder_name == [percentencoding::encode $connected_user_folder]} {
                set my_user_connected 1
                set my_user_path $connected_path
                set my_username $connected_user_folder
            } else {
                set other_user_connected 1
                set other_user_path $connected_path
                set other_username $connected_user_folder
            }
        }
        if { $other_user_connected} {
            set ret_status "OTHER USER CONNECTED"
            set ret_path $other_user_path
            set ret_user $other_username
        }
        if { $my_user_connected} {
            set ret_status "CONNECTED"
            set ret_path $my_user_path
            set ret_user $my_username
        }
        # in windows, in theory a local user can not be connected to same servername with two different credentials
    }
    return [ list $ret_status $ret_path $ret_user]
}

proc GidDataManager::setConnectedPath { connected_path} {    
    variable _connected_path
    # after 1000 [ list W "setConnectedPath\n[GidUtils::GetStackTrace]"]    
    set _connected_path $connected_path
    GiD_RaiseEvent GiD_Event_AfterDataManagerSetCloudFolder $connected_path
}

proc GidDataManager::unsetConnectedPath { } { 
    variable _connected_path
    # after 1000 [ list W "unsetConnectedPath\n[GidUtils::GetStackTrace]"]
    set _connected_path ""
    GiD_RaiseEvent GiD_Event_AfterDataManagerSetCloudFolder ""
}

# in windows it's a unit like Z:
# in linux and macos its's a path (from $HOME)
proc GidDataManager::getConnectedPath { } { 
    variable _connected_path
    set ret_path ""
    if { $::GID_ENABLE_DATAMANAGER } {
        set ret_path $_connected_path
    }
    return $ret_path
}

proc GidDataManager::getCloudPath { } { 
    return [ GidDataManager::getConnectedPath]
}

#return 1 if path match the left part of the data manager connected path
proc GidDataManager::isCloudPath { path } {
    set path_is_cloud 0
    if { $path != "" } {
        set gid_cloud_path [ GidDataManager::getConnectedPath]
        if { $gid_cloud_path != "" } {
            if { $::tcl_platform(platform) == "windows" } {
                if { [ string compare -nocase -length [ string length $gid_cloud_path] $gid_cloud_path $path] == 0 } {
                    set path_is_cloud 1
                }
            } else {
                if { [ string compare -length [ string length $gid_cloud_path] $gid_cloud_path $path] == 0 } {
                    set path_is_cloud 1
                }
            }
        }
    }
    return $path_is_cloud
}

#trim left part of path e.g. Windows Z:/hello.gid/hello.geo -> hello.gid/hello.geo
proc GidDataManager::TrimLeftCloudPath { path } {
    set path_nextcloud $path
    if { $::tcl_platform(platform) == "windows" } {
        #need the full path without Z: for GidDataManager::httpDownloadDataRange
        set parts [lrange [file split $path] 1 end]
        if { [llength $parts] } {
            set path_nextcloud [file join {*}$parts]
        } else {
            set path_nextcloud ""
        }
    } else {
        #how is current_pwd in linux in case of cloud?
        set gid_cloud_path [GidDataManager::getConnectedPath]
        #it it is not mounted can wait a little until it is ready...
        set start_idx [expr [string length $gid_cloud_path]+1]
        set path_nextcloud [string range $path $start_idx end]
    }
    return $path_nextcloud
}

proc GidDataManager::isSharedReadOnly { path } {
    set is_shared_read_only 0
    set dict_attributes [GidDataManager::directGetAttributes $path 1]
    set permissions [dict get $dict_attributes permissions]
    # see meaning of permission letters at https://bitbucket.org/gidteam/datamanager-nc/src/main/API-nextcloud.md
    if { [string first S $permissions] != -1 } {
        # S == shared
        set type [dict get $dict_attributes type]
        if { $type == "file" } {
            if { [string first W $permissions] != -1 } {
                # W can write file
            } else {
                set is_shared_read_only 1
            }
        } elseif { $type == "directory" } {
            if { [string first C $permissions] != -1 && [string first K $permissions] != -1 } {
                # C can create file in this folder
                # K can create folder in this folder
            } else {
                set is_shared_read_only 1
            }
        } else {
            #unexpected
            error "GidDataManager::isSharedReadOnly. Unexpected type $type"
        }
    }
    return $is_shared_read_only
}

proc GidDataManager::SetUsernameAndDamaTokenFromLoginToken { username login_token} {
    variable _username
    variable _dama_token_pass

    set _username ""
    set _dama_token_pass ""
    set dama_token_pass [ GidDataManager::_getDamaTokenPassFromLoginToken $username $login_token]
    # W "SetUsernameAndDamaTokenFromLoginToken result = -$dama_token_pass-"
    if { $dama_token_pass != ""} {
        set is_ok 1
        set _username $username
        set _dama_token_pass $dama_token_pass
        # W "user info set"
    }
    return $_dama_token_pass
}

proc GidDataManager::connect { { parent_frame ""}} {
    variable _username
    variable _dama_token_pass
    variable _linux_mount_type
    
    # when connecting create a file an maintain it open to avoid other gid's to disconnect the mounted shared network path

    # W GidDataManager::connect
    # WV "_username _dama_token_pass" "    "
    # if { $_username == ""} {
    #     ErrorWin "username is empty"
    # }
    # if { $_dama_token_pass == ""} {
    #     ErrorWin "dama_token is empty"
    # }

    GidDataManager::checkExecutionEnvirontment

    if { ( $_username == "") || ( $_dama_token_pass == "")} {
        GidDataManager::_getUsernameAndDamaTokenFromLogin
    }

    if { $::GID_ENABLE_DATAMANAGER == 2} {
        # only direct/http mode allowed
        # store username and password only
        return
    }

    set parent .gid
    if { ( $parent_frame != "") && [ winfo exists $parent_frame]} {
        set parent [ winfo toplevel $parent_frame]
    }

    # first check if there is already connection to datamanger server
    set full_status [ GidDataManager::getStatus]
    lassign $full_status connection_status connected_path connected_user_folder

    set connected_user [ percentencoding::decode $connected_user_folder]
    if { "$connection_status" == "CONNECTED"} {
        GidDataManager::setConnectedPath $connected_path
        GidDataManager::_doManageConnectEvents
        GidDataManager::showMessage "Already connected ( [ GidDataManager::getConnectedPath] , [ GidDataManager::getMountType] )" $parent_frame
        return
    } elseif { "$connection_status" == "OTHER USER CONNECTED"} {    
        set cloud_automatic_disconnect [GiD_Set Cloud(AutomaticDisconnect)]
        if { $cloud_automatic_disconnect } {
            set retval yes
        } else {
            #set text [_ "Unit '%s' already connected for user '%s'. \nTo connect new share, previous one needs to be disconnected. \nProceed to disconnect unit '%s'?" $connected_path $connected_user $connected_path]
            #set retval [MessageBoxOptionsButtons [_ "Data manager"] $text {yes no} [list [_ "Yes"] [_ "No"]] question ""]
            #kike: never ask the user, this prevent automation of batch and test, and is ennoying for a real user ask every time!!!
            GidDataManager::showMessage [_ "Unit '%s' already connected for user '%s'. disconnecting to connect a new share" $connected_path $connected_user]
            set retval yes
        }
        if { $retval == yes} {
            GidDataManager::setConnectedPath $connected_path
            set err_txt [ GidDataManager::_doDisconnect]
            if { $err_txt != ""} {
                if { $cloud_automatic_disconnect } {
                    set msg [_ "Problems disconnecting '%s'" [ GidDataManager::getConnectedPath]]
                    append msg ": $err_txt"
                    GidDataManager::showMessage $msg
                } else {
                    WarnWinText [_ "Problems disconnecting '%s':" [ GidDataManager::getConnectedPath]]\n$err_txt\n[_ "Please resolve the issue and try again."]
                }
                return
            }
        } else {
            return
        }
    }

    # try to use davfs2 if present
    if { $::tcl_platform(os) == "Linux"} {
        # if preference is set to davfs2 check if we can
        if { $_linux_mount_type == "davfs2"} {
            set err [ catch {
                GidDataManager::_set_linux_mount_type davfs2
            } err_txt]
            if { $err} {
                W "$err_txt"
                W "using 'wdfs' to mount gid cloud unit"
                GidDataManager::_set_linux_mount_type wdfs
            }
        }
    }

    GidDataManager::showMessage "Connecting ..." $parent_frame
    update
    set err [ GidDataManager::_doConnect]
    if { !$err} {
        GidDataManager::_doManageConnectEvents
        GidDataManager::showMessage "Connected ( [ GidDataManager::getConnectedPath] , [ GidDataManager::getMountType] )" $parent_frame
    } else {
        if { $::tcl_platform(platform) == "windows" } {
            GidDataManager::showMessage "Error when connecting to server ( check if SMB CIFS File Sharing Support is enabled in Control Panel > Turn Windows features on or off )"
        } else {
            GidDataManager::showMessage "Error when connecting to server"
        }
    }
}

proc GidDataManager::GetLocalPathFromModelId {model_id} {
    # call to backend

    variable _back_end_url
    set token [GiD_Login logged_user_token]
    # set headers Authorization Bearer $token
    set extraHeadersList [ list Authorization \
                            [ list Bearer $token]]

    set url ${_back_end_url}/v1/models/${model_id}/path
    
    set token [ http::geturl $url -headers $extraHeadersList -method GET -binary 0]
    http::wait $token
    upvar #0 $token state
    set httpcode [lindex $state(http) 1]
    
    if { $httpcode == 200} {
        set err 0
        set err_txt ""
        set ret $state(body)
        # set ok 1
    } else {
        set err 1
        set err_txt [ GidDataManager::getHttpCodeMessage $httpcode]
        # set ok 0
        set ret -1
    }

    # remove starting and ending ", if exist
    set ret [ string map {\" ""} $ret]

    # remove starting and ending /, if exist
    if { [ string index $ret 0] == "/"} {
        set ret [ string range $ret 1 end]
    } 
    if { [ string index $ret end] == "/"} {
        set ret [ string range $ret 0 end-1]
    }

    # ret can be url encoded, decode it
    set ret [ percentencoding::decode $ret]

    return $ret

}

# frame in login window:
proc GidDataManager::frameInWindowUpdateState { } {
    variable _frame_in_window
    if { $::GID_ENABLE_DATAMANAGER == 2} {
        # only direct/http mode allowed
        return
    }
    set f $_frame_in_window
    if { [ winfo exists $f]} {
        if { [ GidDataManager::isConnected] } {
            # button icon represents status:
            # $f.ldata_manager configure -text [_ "Disconnect"]
            $f.ldata_manager configure -text [_ "Cloud disk status: connected"]
            $f.data_manager configure -image [gid_themes::GetImage cloud_on.png toolbar] \
                -command "GidDataManager::askAndDisconnect $f ; GidDataManager::frameInWindowUpdateState"
        } else {
            # button icon represents status:
            # $f.ldata_manager configure -text [_ "Connect"]...
            $f.ldata_manager configure -text [_ "Cloud disk status: disconnected"]
            $f.data_manager configure -image [ gid_themes::GetImage cloud_off.png toolbar] \
                -command "GidDataManager::connect $f ; GidDataManager::frameInWindowUpdateState"

        }
    }
}

proc GidDataManager::gotoDataManager {} {
    ### variable _username
    ### variable _dama_token_pass
    set server_url [ getHostUrl]/
    ### if { [ GidDataManager::isConnected] } {
    ###     package require http
    ###     set query [eval http::formatQuery [list username $_username password $_dama_token_pass]] 
    ###     VisitWeb "$server_url/apps/files"
    ### } else {  
    VisitWeb "$server_url/login"
    ### }
}

proc GidDataManager::gotoFrontEnd {} {
    variable _front_end_url
    VisitWeb "$_front_end_url"
}

proc GidDataManager::frameInWindow { f} {        
    variable _frame_in_window
    variable _show_datamanager_page_button

    set _frame_in_window $f

    if { $::GID_ENABLE_DATAMANAGER == 2} {
        # only direct/http mode allowed
        return
    }

    if { $_show_datamanager_page_button} {
        ttk::label $f.lweb_page -text [_ "Model manager page"]...
        ttk::button $f.bweb_page -image [gid_themes::GetImage folder-cloud.png toolbar] \
            -command GidDataManager::gotoFrontEnd
            # -command GidDataManager::gotoDataManager
    }

    # ttk::label $f.ldata_manager -text [_ "Connect"]...
    ttk::label $f.ldata_manager -text ""
    ttk::button $f.data_manager -image [gid_themes::GetImage cloud_list_files.png toolbar] \
        -command [ list GidDataManager::connect $f]
    GidHelp $f.data_manager [_ "Data manager"]

    frameInWindowUpdateState        

    if { $_show_datamanager_page_button} {
        grid $f.lweb_page $f.bweb_page -sticky e -padx 4
    }
    grid $f.ldata_manager $f.data_manager -sticky e -padx 4        
}


### # test
### proc processAfterDataManagerConnect { path} {
###     InfoWin "CONNECTED to DataManager using:\npath = $path\n( This will be the last message )"
###     GiD_UnRegisterEvent GiD_Event_AfterDataManagerConnect processAfterDataManagerConnect
### }
### GiD_RegisterEvent GiD_Event_AfterDataManagerConnect processAfterDataManagerConnect
### 
### proc processAfterDataManagerDisconnect { } {
###     InfoWin "DISCONNECTED from DataManager.\n( This will be the last message )"
###     GiD_UnRegisterEvent GiD_Event_AfterDataManagerDisconnect processAfterDataManagerDisconnect
### }
### GiD_RegisterEvent GiD_Event_AfterDataManagerDisconnect processAfterDataManagerDisconnect

proc GidDataManager::httpGetHeaderList { } {
    variable _username
    variable _dama_token_pass

    if { $_username == ""} {
        if { [GiD_Login is_logged] } {
            # set _username [GiD_Login logged_username]
            # lassign [ GidDataManager::ReadTokenFromFile] token_user token_pass
            # if { ( $token_user != "") && ( $token_user == $_username )} {
            #     set _dama_token_pass $token_pass
            # }
        }
        WarnWin "GidDataManager::httpGetHeaderList empty username and password"
    }

    set extraHeadersList [ list Authorization \
                            [ list Basic \
                                [ binary encode base64 ${_username}:${_dama_token_pass}]]]
    return $extraHeadersList
}

## http webdav functions management
proc GidDataManager::httpCheckAndCreateFolderPath { path} {
    set extraHeadersList [ GidDataManager::httpGetHeaderList]
    set url [ GidDataManager::getWebDavPath]/$path
    set token [ http::geturl $url -headers $extraHeadersList -method MKCOL]
    http::wait $token
    upvar #0 $token state
    set httpcode [lindex [split $state(http) " "] 1]
    if { ( $httpcode == 201) || ( $httpcode == 405)} {
        # path is new (201) or already exists (405)
        set err 0
        set err_txt ""
        set ok 1
    } else {
        set err 1
        set err_txt [ GidDataManager::getHttpCodeMessage $httpcode]
        set ok $err_txt
    }
    return -code $err -errorinfo $err_txt $ok
}

proc GidDataManager::httpUploadData { remote_path data} {
    set extraHeadersList [ GidDataManager::httpGetHeaderList]

    set remote_path_url [ GidDataManager::encodeUrlPath $remote_path]
    set url [ GidDataManager::getWebDavPath]/$remote_path_url
    set token [ http::geturl $url -headers $extraHeadersList -method PUT -binary 1 -query $data]
    http::wait $token
    upvar #0 $token state
    set httpcode [lindex [split $state(http) " "] 1]
    if { ( $httpcode == 201) || ( $httpcode == 204)} {
        # file is new (201) or already exists (204)
        set err 0
        set err_txt ""
        set ok 1
    } else {
        set err 1
        set err_txt [ GidDataManager::getHttpCodeMessage $httpcode]
        set ok $err_txt
    }
    return -code $err -errorinfo $err_txt $ok
}

proc GidDataManager::httpDownloadData { remote_path} {
    set extraHeadersList [ GidDataManager::httpGetHeaderList]

    set remote_path_url [ GidDataManager::encodeUrlPath $remote_path]
    set url [ GidDataManager::getWebDavPath]/$remote_path_url
    set token [ http::geturl $url -headers $extraHeadersList -method GET -binary 1]
    http::wait $token
    upvar #0 $token state
    set httpcode [lindex $state(http) 1]
    if { $httpcode == 200} {
        set err 0
        set err_txt ""
        set ret $state(body)
        # set ok 1
    } else {
        set err 1
        set err_txt [ GidDataManager::getHttpCodeMessage $httpcode]
        # set ok 0
        set ret $err_txt
    }
    return -code $err -errorinfo $err_txt $ret
}


#to download only part of a big file
#e.g. GidDataManager::httpDownloadDataRange kk.gid/kk.geo bytes=0-500
proc GidDataManager::httpDownloadDataRange { remote_path range } {
    set extraHeadersList [ GidDataManager::httpGetHeaderList]
    lappend extraHeadersList Range $range
    set remote_path_url [ GidDataManager::encodeUrlPath $remote_path]
    set url [ GidDataManager::getWebDavPath]/$remote_path_url
    set token [ http::geturl $url -headers $extraHeadersList -method GET -binary 1]
    http::wait $token
    upvar #0 $token state
    set httpcode [lindex $state(http) 1]
    if { $httpcode == 200} {
        set err 0
        set err_txt ""
        set ret $state(body)
        # set ok 1
    } elseif { $httpcode == 206} {
        #206==Partial Content
        set err 0
        set err_txt ""
        set ret $state(body)
        # set ok 1
    } else {
        set err 1
        set err_txt [ GidDataManager::getHttpCodeMessage $httpcode]
        # set ok 0
        set ret $err_txt
    }
    return -code $err -errorinfo $err_txt $ret
}

package require promise

#common proc, to avoid repeat its code
proc GidDataManager::GetUrlAndHeaders { remote_path } {
    set extraHeadersList [GidDataManager::httpGetHeaderList]
    set remote_path_url [ GidDataManager::encodeUrlPath $remote_path]
    set url [GidDataManager::getWebDavPath]/$remote_path_url
    return [list $url $extraHeadersList]
}

#convenience proc to be used as error callback for pgeturl promises
proc GidDataManager::ShowErrorPGetUrl { message edict } {
    set http_state [dict get $edict http_state]
    set httpcode [lindex [dict get $http_state http] 1]    
    W "Preferences from cloud: error $message. httpcode $httpcode"
}

#return a promise
proc GidDataManager::pDownloadData { remote_path } {
    lassign [GidDataManager::GetUrlAndHeaders $remote_path] url extraHeadersList
    return [promise::pgeturl $url -headers $extraHeadersList -method GET -binary 1]
}

#return a promise
proc GidDataManager::pUploadData { remote_path data } {
    lassign [GidDataManager::GetUrlAndHeaders $remote_path] url extraHeadersList
    return [promise::pgeturl $url -headers $extraHeadersList -method PUT -binary 1 -query $data]
}

#return a promise
proc GidDataManager::pCheckAndCreateFolderPath { remote_path } {
    lassign [GidDataManager::GetUrlAndHeaders $remote_path] url extraHeadersList
    return [promise::pgeturl $url -headers $extraHeadersList -method MKCOL]
}

#return a promise
proc GidDataManager::pDeletePath { remote_path } {
    lassign [GidDataManager::GetUrlAndHeaders $remote_path] url extraHeadersList
    return [promise::pgeturl $url -headers $extraHeadersList -method DELETE]
}

#return a promise
proc GidDataManager::pGetAttributes { remote_path extra_attributes } {
    lassign [GidDataManager::GetUrlAndHeaders $remote_path] url extraHeadersList
     if { $extra_attributes } {
        lappend extraHeadersList Depth 0
        set query {<?xml version='1.0' encoding='UTF-8'?>
        <d:propfind xmlns:d='DAV:'>
            <d:prop xmlns:oc='http://owncloud.org/ns'>
                <d:getlastmodified/>
                <d:getcontentlength/>
                <d:getcontenttype/>
                <oc:permissions/>
                <oc:owner-display-name />
                <oc:share-types />
                <d:resourcetype/>
                <d:getetag/>
            </d:prop>
        </d:propfind>}
        set result [promise::pgeturl $url -headers $extraHeadersList -method PROPFIND -query $query]
    } else {
        set result [promise::pgeturl $url -headers $extraHeadersList -method PROPFIND]
    }
    return $result
}

#return a promise, to create all intermediate paths if required
proc GidDataManager::pCheckAndCreateFolderPathRecursive { path } {
    set p_last ""
    set path_splitted [file split $path]
    set n [llength $path_splitted]
    if { $n } {        
        for {set i 0} {$i<$n } {incr i} {
            set path_i [file join {*}[lrange $path_splitted 0 $i]]            
            if { $p_last == ""} {                
                set p_last [GidDataManager::pCheckAndCreateFolderPath $path_i]
            } else {
                set p_last [$p_last then [promise::lambda { path_i http_state } {
                    set httpcode [lindex [dict get $http_state http] 1]
                    if { ( $httpcode == 201) || ( $httpcode == 405)} {
                        #201 {DATAMANAGER_FILE_OR_FOLDER_CREATED}
                        #405 {DATAMANAGER_FOLDER_ALREADY_EXISTS}
                        promise::then_chain [GidDataManager::pCheckAndCreateFolderPath $path_i]
                    } else  {
                        #must create some kind of dummy promise and set as reject to return a promise always and with a http_state dict?
                        #W "pCheckAndCreateFolderPathRecursive httpcode $httpcode"
                        #dict set edict http_state [array get http_state]
                        #promise::then_reject $httpcode $edict
                        #error $httpcode
                        throw $httpcode [GidDataManager::getHttpCodeMessage $httpcode]
                    }
                } $path_i]]
            }
        }
    } else {
        set p_last [promise::pfulfilled [list http 200]]
    }
    return $p_last
}

#return a promise, to upload the data to the file, creating before paths if required
proc GidDataManager::pCreateFoldersAndUploadData { remote_path data } {        
    if { [string index $remote_path 0] == "/"} {
        set remote_path [string range $remote_path 1 end]
    }
    set p1 ""
    set path_tree [file dirname $remote_path]
    if { $path_tree != "."} {              
        set p1 [[GidDataManager::pGetAttributes $path_tree 0] then [promise::lambda { path_tree http_state } {
            set httpcode [lindex [dict get $http_state http] 1] 
            if { $httpcode == 200 || $httpcode == 207 } {
                #200 {DATAMANAGER_OK}
                #207 {DATAMANAGER_MULTISTATUS}        
                set attributes [GidDataManager::processXMLattributes [dict get $http_state body]]
                #attributes has something like: type directory size 6114 mtime 1640619663 numberofentries 1
                if { [dict get $attributes type] == "directory" } {
                    #the directory already exists, must not try to create again
                } else {
                    #the file exists, but is is a file, not a directory
                    W "GidDataManager::pCreateFoldersAndUploadData error: $path_tree exists but it is not a directory. attributes $attributes"
                }                
            } elseif { $httpcode == 404 } {
                #404 {DATAMANAGER_FILE_OR_FOLDER_NOT_FOUND}                
                # parent folder doesn't exists, create it
                promise::then_chain [GidDataManager::pCheckAndCreateFolderPathRecursive $path_tree]
            } else {                
                throw $httpcode [GidDataManager::getHttpCodeMessage $httpcode]
            }
        } $path_tree]]
    }
    if { $p1== "" } {        
        set p [GidDataManager::pUploadData $remote_path $data]
    } else {
        set p [$p1 then [promise::lambda { remote_path data http_state } {            
            promise::then_chain [GidDataManager::pUploadData $remote_path $data]
        } $remote_path $data]]
    }
    return $p
}

#test using async/await
promise::async GidDataManager::AsyncCreateFoldersAndUploadData { remote_path data } {
    set fail 0
    if { [string index $remote_path 0] == "/"} {
        set remote_path [string range $remote_path 1 end]
    }
    set path_tree [file dirname $remote_path]
    if { $path_tree != "."} {       
        set p0 [GidDataManager::pGetAttributes $path_tree 0]
        set http_state_attributes [promise::await $p0]
        set httpcode_attributes [lindex [dict get $http_state_attributes http] 1]
        if { $httpcode_attributes == 200 || $httpcode_attributes == 207 } {
            #200 {DATAMANAGER_OK}
            #207 {DATAMANAGER_MULTISTATUS}
            set attributes [GidDataManager::processXMLattributes [dict get $http_state_attributes body]]
            #attributes has something like: type directory size 6114 mtime 1640619663 numberofentries 1
            if { [dict get $attributes type] == "directory" } {
                #the directory already exists, must not try to create again
            } else {
                #the file exists, but is is a file, not a directory
                set fail 1
                W "GidDataManager::AsyncCreateFoldersAndUploadData error: $path_tree exists but it is not a directory. attributes $attributes"
            }
        } elseif { $httpcode == 404 } {
            #404 {DATAMANAGER_FILE_OR_FOLDER_NOT_FOUND}
            # parent folder doesn't exists, create it
            set http_state_create_folders [promise::await [GidDataManager::pCheckAndCreateFolderPathRecursive $path_tree]]
            set httpcode_create_folders [lindex [dict get $http_state_create_folders http] 1]
            if { $httpcode_create_folders == 201 || $httpcode_create_folders == 405 } {
                #201 {DATAMANAGER_FILE_OR_FOLDER_CREATED}
                #405 {DATAMANAGER_FOLDER_ALREADY_EXISTS}            
            } else {
                set fail 1
            }
        } else {
            set fail 1
            throw $httpcode [GidDataManager::getHttpCodeMessage $httpcode]
        }
    }
    set http_state_upload [promise::await [GidDataManager::pUploadData $remote_path $data]]
    set httpcode_upload  [lindex [dict get $http_state_upload http] 1]
    if { $httpcode_upload == 201 || $httpcode_upload == 204 } {
        #201 {DATAMANAGER_FILE_OR_FOLDER_CREATED}
        #204 {DATAMANAGER_FILE_OR_FOLDER_OVERWRITTEN_OR_DELETED}        
    } else {
        set fail 1
    }
    if { $fail } {
        GidUtils::SetWarnLine [_ "Preferences problems saving to cloud"]
    } else {
        GidUtils::SetWarnLine [_ "Preferences saved to cloud"]
    }
    return $fail ;#Note: async always return a promise althougth an explicit return, the return value is used to fill the done ok value
}

proc GidDataManager::httpDeletePath { remote_path} {
    set extraHeadersList [ GidDataManager::httpGetHeaderList]

    set remote_path_url [ GidDataManager::encodeUrlPath $remote_path]
    set url [ GidDataManager::getWebDavPath]/$remote_path_url
    set token [ http::geturl $url -headers $extraHeadersList -method DELETE]
    http::wait $token
    upvar #0 $token state
    set httpcode [lindex [split $state(http) " "] 1]
    if { $httpcode == 204} {
        set err 0
        set err_txt ""
        # set ret $state(body)
        set ok 1
    } else {
        set err 1
        set err_txt [ GidDataManager::getHttpCodeMessage $httpcode]
        set ok 0
        # set ret $err_txt
    }
    return -code $err -errorinfo $err_txt $ok
}

#extra_attributes 0 or 1 (1 to ask for extra nextcloud non-webdav default attributes like permission of shared-user-name)
proc GidDataManager::httpGetAttributes { remote_path extra_attributes } {
    set remote_path_url [ GidDataManager::encodeUrlPath $remote_path]
    set url [ GidDataManager::getWebDavPath]/$remote_path_url
    set extraHeadersList [ GidDataManager::httpGetHeaderList]
    if { $extra_attributes } {
        lappend extraHeadersList Depth 0
        set query {<?xml version='1.0' encoding='UTF-8'?>
        <d:propfind xmlns:d='DAV:'>
            <d:prop xmlns:oc='http://owncloud.org/ns'>
                <d:getlastmodified/>
                <d:getcontentlength/>
                <d:getcontenttype/>
                <oc:permissions/>
                <oc:owner-display-name />
                <oc:share-types />
                <d:resourcetype/>
                <d:getetag/>
            </d:prop>
        </d:propfind>}
        set token [http::geturl $url -headers $extraHeadersList -method PROPFIND -query $query]
    } else {
        set token [http::geturl $url -headers $extraHeadersList -method PROPFIND]
    }
    http::wait $token
    upvar #0 $token state
    set httpcode [lindex [split $state(http) " "] 1]
    if { ( $httpcode == 200) || ( $httpcode == 207) } {
        set err 0
        set err_txt ""
        set ret $state(body)
    } else {
        set err 1
        set err_txt [ GidDataManager::getHttpCodeMessage $httpcode]
        set ret $err_txt
    }
    return -code $err -errorinfo $err_txt $ret
}

proc GidDataManager::processXMLattributes { xml_data} {
    set attributes [ dict create]
    set xmldoc [ dom parse $xml_data]

    # attribute: type
    set resourcetype [ $xmldoc selectNodes -namespaces {d DAV:} {/d:multistatus/d:response/d:propstat/d:prop/d:resourcetype/d:collection}]
    if { $resourcetype != "" } {
        set type "directory"
    } else {
        set type "file"
    }
    dict set attributes type $type

    # attribute: size
    if { $type == "file"} {
        set contentlength [ $xmldoc selectNodes -namespaces {d DAV:} {/d:multistatus/d:response/d:propstat/d:prop/d:getcontentlength/text()}]
    } else {
        # type == "directory"
        set contentlength [ $xmldoc selectNodes -namespaces {d DAV:} {/d:multistatus/d:response/d:propstat/d:prop/d:quota-used-bytes/text()}]
        # as it's a directory, the xml contains information of all files and folders inside provided_path
        # i.e. the above query may be a list of matches, one for the provided_path and one for each sub-folders
        # so let get the first one
        set contentlength [ lindex $contentlength 0]
    }
    if { $contentlength != "" } {
        # set contentlength [$contentlength stringValue]
        set contentlength [ $contentlength nodeValue ]
    } else {
        set contentlength 0
    }
    dict set attributes size $contentlength

    # attribute: mtime
    set lastmodified [ $xmldoc selectNodes -namespaces {d DAV:} {/d:multistatus/d:response/d:propstat/d:prop/d:getlastmodified/text()}]
    if { $type == "directory"} {
        # as it's a directory, the xml contains information of all files and folders inside provided_path
        # i.e. the above query may be a list of matches, one for the provided_path and one for each sub-folders
        # so let get the first one
        set lastmodified [ lindex $lastmodified 0]
    }
    if { $lastmodified != "" } {
        set lastmodified [ $lastmodified nodeValue ]
        # convert 'Tue, 28 Sep 2021 07:52:59 GMT' to clock seconds
        set lastmodified [ clock scan $lastmodified]
    } else {
        set lastmodified 0
    }
    dict set attributes mtime $lastmodified

    # attribute: contenttype
    if { $type == "file"} {
        set contenttype [ $xmldoc selectNodes -namespaces {d DAV:} {/d:multistatus/d:response/d:propstat/d:prop/d:getcontenttype/text()}]
        if { $contenttype != "" } {
            set contenttype [ $contenttype nodeValue ]
        } else {
            set contenttype 0
        }
        dict set attributes contenttype $contenttype
    } else {
        # type == "directory"
        # here we could check if inside there is a .geo, .msh, ... and
        # set contenttype "gid-project"    ;# or "application/gid-project"
        # set contenttype "problem-type"    ;# or "application/problem-type"
    }

    # attribute: numberofentries
    if { $type == "directory"} {
        set list_entries [ $xmldoc selectNodes -namespaces {d DAV:} {/d:multistatus/d:response/d:href/text()}]
        # list_entries also includes the querie remote_path itself
        set numberofentries [ expr [ llength $list_entries] - 1]
        dict set attributes numberofentries $numberofentries
    } else {
        # type == "file"
    }
    
    
    set permissions [ $xmldoc selectNodes -namespaces {d DAV: oc http://owncloud.org/ns} {/d:multistatus/d:response/d:propstat/d:prop/oc:permissions/text()}]
    if { $permissions !="" } {
        dict set attributes permissions [$permissions nodeValue]
    }
    
    set owner_display_name [ $xmldoc selectNodes -namespaces {d DAV: oc http://owncloud.org/ns} {/d:multistatus/d:response/d:propstat/d:prop/oc:owner-display-name/text()}]
    if { $owner_display_name !="" } {
        dict set attributes owner-display-name [$owner_display_name nodeValue]
    }
    

    return $attributes
}

#####
# exposed functions
#####

# returns 1 if folder exists or created
# returns 0 on error
proc GidDataManager::directCheckAndCreateFolderPath { path} {
    set lst_parents [ file split $path]
    set full_path ""
    foreach subfolder $lst_parents {
        set full_path [ file join $full_path $subfolder]
        set ok [ GidDataManager::httpCheckAndCreateFolderPath $full_path]
        # W "creating path $full_path returned $ret_code"
    }
    return $ok
}

# returns 1 if uploaded
# returns 0 on error
proc GidDataManager::directUploadData { remote_path data} {
    # interesting link:
    # https://wiki.tcl-lang.org/page/File+Upload+with+tcl%27s+http

    # parse remote_path
    # split it to check if parent folders exists
    if { [ string index $remote_path 0] == "/"} {
        set remote_path [ string range $remote_path 1 end]
    }

    set path_tree [ file dirname $remote_path]
    set remote_filename [ file tail $remote_path]
    if { $path_tree != "."} {
        GidDataManager::directCheckAndCreateFolderPath $path_tree
    }

    set ok 0
    set err [ catch {
        set ok [ GidDataManager::httpUploadData $remote_path $data]
    } err_txt]
    if { $err} {
        # return error code message
        set ok $err_txt
    }
    # W "uploading $remote_path returned $ret_code"
    return -code $err -errorinfo $err_txt $ok
}

proc GidDataManager::directDownloadData { remote_path} {
    set data ""
    # may be some path checking instead of catching an error?
    set err [ catch {
        set data [ GidDataManager::httpDownloadData $remote_path]
    } err_txt]
    if { $err} {
        # return error code message
        set data $err_txt
    }
    return -code $err -errorinfo $err_txt $data
}

#download and save as file
proc GidDataManager::directDownloadFile { remote_path local_filename } {
    set fail 1
    set err [ catch {
        set data [GidDataManager::directDownloadData $remote_path]
    } err_txt]
    if { $err} {
        # return error code message
        set data $err_txt
    } else {
        set err [ catch {
            set fp [open $local_filename "wb"]
        } err_txt]
        if { $err} {
          # return error code message
        } else {
            puts -nonewline $fp $data
            close $fp
            set fail 0
        }
    }
    return -code $err -errorinfo $err_txt $fail
}

proc GidDataManager::directDownloadDataRange { remote_path range } {
    set data ""
    # may be some path checking instead of catching an error?
    set err [ catch {
        set data [ GidDataManager::httpDownloadDataRange $remote_path $range]
    } err_txt]
    if { $err} {
        # return error code message
        set data $err_txt
    }
    return -code $err -errorinfo $err_txt $data
}

#extra_attributes 0 or 1 (1 to ask for extra nextcloud non-webdav default attributes like permission of shared-user-name)
proc GidDataManager::directGetAttributes { remote_path extra_attributes } {
    # TODO
    # as in vfs/webdav
    set data ""
    # may be some path checking instead of catching an error?
    set err [ catch {
        set data [ GidDataManager::httpGetAttributes $remote_path $extra_attributes]
    } err_txt]
    if { $err} {
        # return error code message
        set data $err_txt
    } else {
        # process xml info from $data
        set data [ GidDataManager::processXMLattributes $data]
    }
    return -code $err -errorinfo $err_txt $data
}

#type: directory file or "" (if empty then return true is exists as file or directory or other type)
proc GidDataManager::directExists { remote_path type } {
    set exists 0
    if { [catch { set attributes [GidDataManager::directGetAttributes $remote_path 0] } msg] } {
        if { $msg == "DATAMANAGER_FILE_OR_FOLDER_NOT_FOUND" }  {
            set exists 0
        } else {
            #other kind of unexpected bug, I am not sure that do not exists
            set exists -1
        }

    } else {
        if { $type== "" || [dict get $attributes type] == $type } {
            set exists 1
        } else {
            set exists 0
        }
    }
    return $exists
}

proc GidDataManager::directFileExists { remote_path } {
    return [GidDataManager::directExists $remote_path file]
}

proc GidDataManager::directDirectoryExists { remote_path } {
    return [GidDataManager::directExists $remote_path directory]
}
    
# it deletes remote_path
# if it's a folder, it deletes everything inside it, including all sub-folders !!!
# TODO 
#   * detect if there is something inside $remotepath
#   * if so, raise error
#   * if -force is provided , delete  it
# directDeletePath ?-force? path_to_delete
proc GidDataManager::directDeletePath { args} {
    set force 0
    if { [ llength $args] == 1} {
        set remote_path $args
    } else {
        set option [ lindex $args 0]
        if { $option == "-force"} {
            set force 1
        } else {
            return -code 1 -errorinfo "DATAMANAGER_UNKNOWN_OPTION" "DATAMANAGER_UNKNOWN_OPTION $option"
        }
        set remote_path [ lindex $args 1]
    }

    # if $force no need to get attributes and check anything
    if { !$force} {
        # first check if path is file or folder:
        set dict_attr [ GidDataManager::directGetAttributes $remote_path 0]
        if { ![ dict exists $dict_attr type] || ![ dict exists $dict_attr numberofentries]} {
            return -code 1 -errorinfo "DATAMANAGER_ATTRIBUTES_NOT_FOUND" "DATAMANAGER_ATTRIBUTES_NOT_FOUND"
        }
        if { ( [ dict get $dict_attr type] == "directory") && ( [ dict get $dict_attr numberofentries] != 0)} {
            return -code 1 -errorinfo "DATAMANAGER_FOLDER_NOT_EMPTY" "DATAMANAGER_FOLDER_NOT_EMPTY"
        }
    }

    # may be some path checking instead of catching an error?
    set err [ catch {
        set ok [ GidDataManager::httpDeletePath $remote_path]
    } err_txt]
    if { $err} {
        # return error code message
        set ok $err_txt
    }
    return -code $err -errorinfo $err_txt $ok
}

proc GidDataManager::_doProcessLogin {} {
    variable _username
    variable _dama_token_pass
    variable _frame_in_window

    # W "GidDataManager::_doProcessLogin "    
    GidDataManager::_getUsernameAndDamaTokenFromLogin
    # WV "_username _dama_token_pass" "    "

    if { ( $_username != "") && ( $_dama_token_pass != "")} {
        GidDataManager::checkStatusAndAutoConnect
    } else {
        set _username ""
        set _dama_token_pass ""
        GidDataManager::showMessage "Invalid credentials to access data manager" $_frame_in_window
    }
    # W "GidDataManager::_doProcessLogin DONE"
}

# events BeforeLogin and AfterLogin may be raised together, because inside gid_login_inner
# there are after idle [ GiD_Event raise ....]
proc GidDataManager::_processBeforeLogin { username user_token} {
    variable _busy_login
    if { $_busy_login == 0} {
        set _busy_login 1
        # W "_processBeforeLogin"
        GidDataManager::_doProcessLogin
        # W "_processBeforeLogin DONE"
        set _busy_login 0
    }
}
proc GidDataManager::_processAfterLogin { username } {
    variable _busy_login
    if { $_busy_login == 0} {
        set _busy_login 1
        # W "_processAfterLogin"
        GidDataManager::_doProcessLogin
        # W "_processAfterLogin DONE"
        set _busy_login 0
    }
}

proc GidDataManager::_processAfterLogout { } {
    variable _username
    variable _dama_token_pass

    # W "GidDataManager::_processAfterLogout"
    if { $::GID_ENABLE_DATAMANAGER == 2} {
        set _username ""
        set _dama_token_pass ""
        return
    }
    
    set local_cloud_path [ GidDataManager::getConnectedPath]
    # WV local_cloud_path "    "
    if { $local_cloud_path != ""} {
        set force_disconnect true
        set disconnected [ GidDataManager::disconnect $force_disconnect]
        # W "    disconnected = $disconnected"
    }
    set _username ""
    set _dama_token_pass ""
}

proc GidDataManager::registerDataManagerEvents { } {
    GiD_RegisterEvent GiD_Event_BeforeLogin GidDataManager::_processBeforeLogin
    GiD_RegisterEvent GiD_Event_AfterLogin GidDataManager::_processAfterLogin
    GiD_RegisterEvent GiD_Event_AfterLogout GidDataManager::_processAfterLogout
    # GiD_RegisterEvent GiD_Event_BeforeExit
}

proc GidDataManager::unregisterDataManagerEvents { } {
    GiD_UnRegisterEvent GiD_Event_BeforeLogin GidDataManager::_processBeforeLogin
    GiD_UnRegisterEvent GiD_Event_AfterLogin GidDataManager::_processAfterLogin
    GiD_UnRegisterEvent GiD_Event_AfterLogout GidDataManager::_processAfterLogout
    # GiD_RegisterEvent GiD_Event_BeforeExit
}

# -np- set data [ binary decode base64 [ [ gid_themes::GetImage about.png] data -format png]]
# -np- W [ GidDataManager::directUploadData .pepito.png [ binary decode base64 [ [ gid_themes::GetImage about.png] data -format png]]]
# -np- W [ GidDataManager::directUploadData .a/b/.pepito.png [ binary decode base64 [ [ gid_themes::GetImage about.png] data -format png]]]
# -np- W [ binary encode base64 [ GidDataManager::directDownloadData .pepito.png]]
# -np- catch { GidDataManager::directDownloadData .pepito2.png} err_txt ; W $err_txt
# -np- W [ GidataManager::directDeletePath .pepito.png ]

# -np- catch { GidDataManager::directDeletePath .pepito2.png} err_txt ; W $err_txt
# -np- catch { GidDataManager::directCheckAndCreateFolderPath 2delete} err_txt; W $err_txt
# -np- catch { GidDataManager::directDeletePath 2delete} err_txt ; W $err_txt
# -np- catch { GidDataManager::directCheckAndCreateFolderPath 2delete/more2delete} err_txt; W $err_txt
# 2delete/more2delete can be deleted as both folders are empty ( size == 0)
# -np- catch { GidDataManager::directDeletePath 2delete} err_txt ; W $err_txt
# -np- W [ GidDataManager::directUploadData 2delete/more2delete/.pepito.png [ binary decode base64 [ [ gid_themes::GetImage about.png] data -format png]]]
# this causes error as it's not empty
# -np- catch { GidDataManager::directDeletePath 2delete} err_txt ; W $err_txt
# delete anyway:
# -np- catch { GidDataManager::directDeletePath -force 2delete} err_txt ; W $err_txt

# -np- catch { GidDataManager::directGetAttributes .pepito.png 0} err_txt ; W $err_txt
# empty folder:
# -np- catch { GidDataManager::directGetAttributes win 0} err_txt ; W $err_txt
# folder with contents
# -np- catch { GidDataManager::directGetAttributes testKratosFluid2D.gid 0} err_txt ; W $err_txt
# user's folder
# -np- catch { GidDataManager::directGetAttributes . 0} err_txt ; W $err_txt

proc GidDataManager::doTimmings { args} {
    variable _linux_mount_type

    set cloud_path [ GidDataManager::getCloudPath]
    if { $cloud_path == ""} {
        return -code error "GiD Cloud unit not connected"
    }

    # parsing arguments:
    # defaults
    set opt(-mount_type) ""    ;# what is now using gid_data_manager
    set opt(-src_path) ""      ;# what is now using gid_data_manager
    set opt(-verbose) true
    set lst_opt [ list -mount_type -src_path -verbose]
    if { [ expr [ llength $args] % 2] != 0} {
        set lst_arguments ""
        foreach k $lst_opt {
            append lst_arguments "?$k value? "
        }
        return -code error "Invalid number of arguments, should be $lst_arguments"
    }
    foreach [ list key value] $args {
        if { [ lsearch $lst_opt $key] != -1} {
            set opt($key) $value
        } else {
            return -code error "Unknown option $key, should be one of $lst_opt"
        }
    }
    # end parsing arguments

    set src_path $opt(-src_path)
    set type $opt(-mount_type)
    set verbose $opt(-verbose)

    if { $src_path == ""} {
        set src_path $cloud_path
        if { $::tcl_platform(platform) == "windows" } {
            set src_path "$cloud_path/"
        }
    }
    if { $type == ""} {
        set type [ GidDataManager::getMountType]
    }
    set dest_tmp_path "tmp"

    # discading Kratos project as there are blocking Tcl-errors in the problem-type
    # testKratosFluid2D.gid

    set lst_projects [ list GID-2631-big-geo.gid new_project.gid testCmas2D.gid testExKratosFluid2D.gid \
        Modelos_Abel/casco_barco_iges_abel.gid Modelos_Abel/f1_model.gid Modelos_Abel/test_aircraft_signal_coordinate.gid]

    set total_time_read_us 0
    set total_time_save_us 0
    # for projects with Kratos problem-types a modal blocking window appears to ask for user intervention
    # we don't want this.
    set ::Kratos_AskToTransform 0

    # array to save timing data
    set gdm_timmings(projects) $lst_projects
    foreach project_path $lst_projects {
        DoFilesNew ASK
        if { $verbose} {
            W "$project_path    READ    SAVE"
        }

        # read test
        set src_project [ file join $src_path $project_path]
        set time_read_us [ lindex [ time [ list GiD_Process Mescape Files Read $src_project]] 0]

        # save/write test
        set dst_project [ file join $src_path $dest_tmp_path $project_path]

        # create render mesh to save
        GiD_Process 'Render Flat
        GiD_Process 'Render Smooth
        GiD_Process 'Render Normal

        # remove destination if it exists
        file delete -force $dst_project
        set time_save_us [ lindex [ time [ list GiD_Process Mescape Files SaveAs $dst_project]] 0]

        # print test
        set read_s [ format "%.3f" [ expr $time_read_us / 1000000.0]]
        set save_s [ format "%.3f" [ expr $time_save_us / 1000000.0]]
        if { $verbose} {
            W "$type    $read_s    $save_s"
        }

        set total_time_read_us [ expr $total_time_read_us + $time_read_us]
        set total_time_save_us [ expr $total_time_save_us + $time_save_us]

        if { $::tcl_platform(platform) == "windows" } {
            set git_scm_unix_tools_location "C:/Program Files/Git/usr/bin"
            set du [ file join $git_scm_unix_tools_location du]
            set gdm_timmings($project_path,read_size) [ lindex [ exec $du -h $src_project] 0]
            set gdm_timmings($project_path,save_size) [ lindex [ exec $du -h $dst_project] 0]
        } else {
            # linux, macos
            set gdm_timmings($project_path,read_size) [ lindex [ exec du -h $src_project] 0]
            set gdm_timmings($project_path,save_size) [ lindex [ exec du -h $dst_project] 0]
        }
        set gdm_timmings($project_path,read_time_s) $read_s
        set gdm_timmings($project_path,save_time_s) $save_s
        set gdm_timmings($project_path,mount_type) $type
    }
    
    DoFilesNew ASK
    set total_time_read_s [ format "%.3f" [ expr $total_time_read_us / 1000000.0]]
    set total_time_save_s [ format "%.3f" [ expr $total_time_save_us / 1000000.0]]
    if { $verbose} {
        W "Total Time    READ    SAVE"
        W "$type    $total_time_read_s    $total_time_save_s"
    }
    return [ array get gdm_timmings]
}

proc GidDataManager::viewTimmings {} {
    array set data [GidDataManager::doTimmings]
    W "| project name | type | read time | save time | read size | save size |"
    W "|--------------|:----:|----------:|----------:|----------:|----------:|"
    set total_read_s 0
    set total_save_s 0
    foreach project $data(projects) {
        set read_time $data($project,read_time_s)
        set save_time $data($project,save_time_s)
        set read_size $data($project,read_size)
        set save_size $data($project,save_size)
        set type $data($project,mount_type)
        set total_read_s [ expr $total_read_s + $read_time]
        set total_save_s [ expr $total_save_s + $save_time]
        W "| $project | $type | $read_time s | $save_time s | $read_size | $save_size |"
    }
    W "| Total | $type | $total_read_s | $total_save_s |"
}

#get a list of local folders set to be synchronized with nextcloud remote storage
proc GidDataManager::get_folders_nextcloud_shared {} {
    set folders [list]
    if { $::tcl_platform(platform) == "windows" } {
        #Windows: %APPDATA%\Nextcloud\nextcloud.cfg
        set filename_cft [file join [gid_cross_platform::get_folder_shell appdata] Nextcloud/nextcloud.cfg]
    } elseif { $::tcl_platform(os) == "Darwin"}  {
        #macOs: $HOME/Library/Preferences/Nextcloud/nextcloud.cfg
        set filename_cft [file join [gid_cross_platform::get_folder_special home] Library/Preferences/Nextcloud/nextcloud.cfg]
    } else {
        #Linux: $HOME/.config/Nextcloud/nextcloud.cfg
        set filename_cft [file join [gid_cross_platform::get_folder_special home] .config/Nextcloud/nextcloud.cfg]
    }
    if { [file exists $filename_cft] } {
        set data [GidUtils::ReadFile $filename_cft utf-8 0]
        set section_accounts 0
        foreach line [split $data \n] {
            if { $section_accounts } {
                #find something like #0\FoldersWithPlaceholders\1\localPath=C:/Users/escolano/Nextcloud/
                if { [string index $line 0] == {[} } {
                    set sections_account 0
                } else {
                    #0\FoldersWithPlaceholders\1\localPath=C:/Users/escolano/Nextcloud/
                    set line_splitted [split $line \\]
                    if { [lindex $line_splitted 1] == "FoldersWithPlaceholders" &&  [string range [lindex $line_splitted 3] 0 9] == "localPath=" } {
                        set local_path [file join [lindex [split [lindex $line_splitted 3] =] 1]]
                        lappend folders $local_path
                    }
                }
            } elseif {$line == {[Accounts]} } {
                set section_accounts 1
            }
        }
    }
    return $folders
}

proc GidDataManager::NavigateToGiDModelManager { } {
    
    variable _front_end_url

    set user_token [GiD_Login logged_user_token]
    set model_path [GidUtils::GetDirectoryModel]
    set is_cloud_path [GidDataManager::isCloudPath $model_path]
    if {$is_cloud_path} {
        set model_path [GidDataManager::TrimLeftCloudPath $model_path]
    } else {
        set model_path ""
    }
    VisitWeb "${_front_end_url}/landgid?user_token=${user_token}&model_path=${model_path}"
}
