
namespace eval RetrievePT {
    variable w
    variable progressbar
    variable progresslabel
    variable FileSize
    variable LastTimeMiliSec 0
    variable num_modules
    variable modulesarray
    variable lists
    variable currenttab
    if { 0 } {
        #old server
        variable server www.gidhome.com
        variable relativedir [file join ftp pub gidmodules]
    } else {
        #Amazon S3
        variable server downloads.gidsimulation.com
        variable relativedir gidmodules
    }
    variable blocksize 4096 ;#default value 4096
    variable contcallbacks 0
    variable show_content
    array set show_content {Program 1 Interface 1 Plugin 1 Example 1 Document 1 Theme 1 Fonts 1 Resource 1 Package 1}
    variable show_filter
    array set show_filter {NotInstalled 1 CompatibleVersion 1}
}

proc RetrievePT::Close {} {
    variable w
    variable progresslabel
    destroy $w
}

proc RetrievePT::HttpProgress { token total bytes } {
    # upvar #0 $token state
    variable FileSize
    if { $total != 0} {
        set FileSize $total
    }
    variable progressbar
    variable progresslabel
    variable LastTimeMiliSec
    variable blocksize
    variable contcallbacks
    if { $contcallbacks == [expr int($FileSize/($blocksize*10))] } {
        set contcallbacks 0
        set percent [expr 10+90*$bytes/double($FileSize)]
        set progressbar $percent
        set MBytes_total [expr {double($FileSize)/(1024*1024)}]
        set MBytes [expr {double($bytes)/(1024*1024)}]
        set TimeMiliSec [clock clicks -milliseconds]
        set speed [expr $MBytes/(($TimeMiliSec-$LastTimeMiliSec)*1e-3)]
        set progresslabel [format "%.1f/%.1f MB (%.1f MB/s)" $MBytes $MBytes_total $speed]
        update idletasks
    } else {
        incr contcallbacks
    }
}

## example from: https://www.tcl.tk/man/tcl8.4/TclCmd/http.htm#M20
## # Copy a URL to a file and print meta-data
## proc httpcopy { url file {chunk 4096} } {
##    set out [open $file w]
##    set token [::http::geturl $url -channel $out \
##           -progress httpCopyProgress -blocksize $chunk]
##    close $out
##
##    # This ends the line started by httpCopyProgress
##    W "ERROR: "
##
##    upvar #0 $token state
##    set max 0
##    foreach {name value} $state(meta) {
##       if {[string length $name] > $max} {
##          set max [string length $name]
##       }
##       if {[regexp -nocase ^location$ $name]} {
##          # Handle URL redirects
##          W "ERROR Location:$value"
##          return [httpcopy [string trim $value] $file $chunk]
##       }
##    }
##    incr max
##    foreach {name value} $state(meta) {
##       W [format "%-*s %s" $max $name: $value]
##    }
##
##    return $token
## }
## proc httpCopyProgress { token total current } {
##     WV [list token total current]
## }

proc RetrievePT::HttpGetFileUrlName { remote_filename} {
    variable relativedir
    variable server
    set file_url https://${server}/${relativedir}/$remote_filename
    return $file_url
}

proc RetrievePT::HttpGetFile { file_url} {
    variable w

    if { [info exists w] && [winfo exists $w] } {
        set with_window 1
    } else {
        set with_window 0
    }

    # using https
    package require http
    package require tls
    if { [info exists ::env(http_proxy)] || [info exists ::env(https_proxy)] } {
        package require autoproxy
        autoproxy::init
        # it seems that here -autoservername 1 option is not valid
        http::register https 443 [list autoproxy::tls_socket]
    } else {
        http::register https 443 [list ::tls::socket -autoservername 1]
    }

    set err [catch {
        if { $with_window} {
            set ::RetrievePT::progresslabel [format "%s '%s' ..." [_ "Downloading"] [file tail $file_url]]
            update
            set contents [::http::geturl $file_url -progress RetrievePT::HttpProgress]
        } else {
            set contents [::http::geturl $file_url]
        }
    } err_txt]
    if { $err} {
        return -1
    }
    set exit_code [set ${contents}(http)]
    # WV exit_code

    # may be it's "HTTP/1.1 404 Not Found"
    if { $exit_code == "HTTP/1.1 200 OK"} {
        set ret_txt [set ${contents}(body)]
    } else {
        set ret_txt -1
    }
    return $ret_txt
}

proc RetrievePT::Retrievelist {} {
    variable w
    variable progressbar
    variable progresslabel
    variable FileSize
    variable modulesarray
    variable num_modules

    set num_modules 0
    if { [info exists w] && [winfo exists $w] } {
        set with_window 1
    } else {
        set with_window 0
    }

    set filename_info modulesinfo

    # using http
    set modulesinfo [RetrievePT::HttpGetFile [RetrievePT::HttpGetFileUrlName $filename_info]]
    if { $modulesinfo == -1} {
        if { $with_window } {
            set progresslabel [_ "Error retrieving list"]
            GidUtils::EndWaitState $w
            update
            after 1000 {set ::RetrievePT::progresslabel ""}
        }
        return
    }

    regsub -lineanchor -all {^#[^\n]*} $modulesinfo {} modulesinfo
    unset -nocomplain modulesarray

    catch {
        foreach {name value} $modulesinfo {
            if { $name == "Name" } {
                #new style, ignore Name and use instead ProgramName,
                #but not replace Name for back compatibility with previous GiD's
                continue
            } elseif { $name == "ProgramName" } {
                set name Name
                incr num_modules
            }
            set modulesarray($num_modules,$name) $value
        }
    }
    if { $with_window } {
        CheckModulesInstalled

        set progressbar 100
        set progresslabel ""
        GidUtils::EndWaitState $w
        update
        after 1000 set RetrievePT::progressbar 0
    }
}

proc RetrievePT::MustShowInTable { i_module } {
    variable modulesarray
    variable show_content
    variable show_filter
    set content $modulesarray($i_module,Content)
    if { [info exists show_content($content)] && $show_content($content) } {
        if { ![info exists modulesarray($i_module,IsInstalled)] } {
            RetrievePT::CheckModuleInstalled $i_module
        }
        if { $show_filter(CompatibleVersion) && $modulesarray($i_module,IsInstalled) == -1 } {
            #could also use ![IsVersionOKForCurrentGiD $i_module]
            set must_show 0
        } else {
            if { $show_filter(NotInstalled) && $modulesarray($i_module,IsInstalled) == 1 } {
                set must_show 0
            } else {
                set must_show 1
            }
        }
    } else {
        set must_show 0
    }
    return $must_show
}

proc RetrievePT::FillTable { list i } {
    variable num_modules
    variable modulesarray

    if { ![winfo exists $list] } { return }

    lassign [RetrievePT::GetCurrentPlatform] currentos currentbits

    $list delete 0 end
    for {set i_module 1} {$i_module<=$num_modules} {incr i_module} {
        if { [RetrievePT::MustShowInTable $i_module] } {
            set name $modulesarray($i_module,Name)
            set platform $modulesarray($i_module,Platform)
            lassign $platform os bits
            if { ($os == "All" || $os == $currentos) && ($bits == "All" || $bits == $currentbits ) } {
                if { $i == 1 } continue
            } else {
                if { $i == 0 } continue
            }
            if { [info exists modulesarray($i_module,Version)] } {
                set version $modulesarray($i_module,Version)
            } else {
                set version ""
            }
            if { [info exists modulesarray($i_module,Date)] } {
                set date [string trim $modulesarray($i_module,Date)]
            } else {
                set date ""
            }
            if { [info exists modulesarray($i_module,ShortDescription)] } {
                set shortdesc [string trim $modulesarray($i_module,ShortDescription)]
            } else {
                set shortdesc ""
            }
            $list insert end [list $name $version $platform "" $date $shortdesc]
        }
    }

    set row 0
    for {set i_module 1} {$i_module<=$num_modules} {incr i_module} {
        if { [RetrievePT::MustShowInTable $i_module] } {
            lassign $modulesarray($i_module,Platform) os bits
            if { ($os == "All" || $os == $currentos) && ($bits == "All" || $bits == $currentbits ) } {
                if { $i == 1 } continue
            } else {
                if { $i == 0 } continue
            }
            if { ![info exists modulesarray($i_module,IsInstalled)] } {
                RetrievePT::CheckModuleInstalled $i_module
            }
            if { $modulesarray($i_module,IsInstalled) == 1 } {
                $list cellconfigure $row,3 -image [gid_themes::GetImage in_use.png small_icons]
            } elseif { $modulesarray($i_module,IsInstalled) == -1 } {
                #GiD and the module are not compatible
                $list cellconfigure $row,3 -image [gid_themes::GetImage delete.png small_icons]
            } else {
                $list cellconfigure $row,3 -image [gid_themes::GetImage blank.png small_icons]
            }
            incr row
        }
    }
}

proc RetrievePT::GetModuleIndex { name version platform } {
    variable num_modules
    variable modulesarray
    lassign $platform os bits
    set same_platform_other_versions {}
    set other_platforms {}
    for {set i_module 1} {$i_module<=$num_modules} {incr i_module} {
        if { ![string compare -nocase $modulesarray($i_module,Name) $name] } {
            set repository_module_version $modulesarray($i_module,Version)
            set repository_module_platform $modulesarray($i_module,Platform)
            lassign $repository_module_platform repository_module_os repository_module_bits
            if { ($repository_module_os == $os || $repository_module_os == "All") &&
                ($repository_module_bits == $bits || $repository_module_bits == "All") } {
                if { $repository_module_version == $version || $version == "" } {
                    #exact version found
                    return [list $i_module 0]
                } else {
                    lappend same_platform_other_versions $i_module
                }
            } else {
                lappend other_platforms $i_module
            }
        }
    }
    set i_best 0
    if { [llength $same_platform_other_versions] } {
        set i_best [lindex $same_platform_other_versions 0]
        foreach i_module [lrange $same_platform_other_versions 1 end] {
            if { [GidUtils::TwoVersionsCmp $modulesarray($i_best,Version) $modulesarray($i_module,Version)] == -1 } {
                #second version is greater
                set i_best $i_module
            }
        }
    } else {
        if { [llength $other_platforms] } {
            set i_best [lindex $other_platforms 0]
            foreach i_module [lrange $other_platforms 1 end] {
                if { [GidUtils::TwoVersionsCmp $modulesarray($i_best,Version) $modulesarray($i_module,Version)] == -1 } {
                    #second version is greater
                    set i_best $i_module
                }
            }
        }
    }
    return [list 0 $i_best]
}

proc RetrievePT::IsVersionOKForCurrentGiD { i_module } {
    variable modulesarray
    if { [info exists modulesarray($i_module,MinimumGiDVersion)] } {
        set module_min_required_gid_version $modulesarray($i_module,MinimumGiDVersion)
    } else {
        set module_min_required_gid_version [GiD_Info GiDVersion]
    }
    if { [info exists modulesarray($i_module,MaximumGiDVersion)] } {
        set module_max_required_gid_version $modulesarray($i_module,MaximumGiDVersion)
    } else {
        set module_max_required_gid_version [GiD_Info GiDVersion]
    }
    set comp [GidUtils::VersionCmp $module_min_required_gid_version]
    if { $comp < 0 } {
        #GiD is Old
        return 0
    }
    set comp [GidUtils::VersionCmp $module_max_required_gid_version]
    if { $comp > 0 } {
        #GiD is too new
        return 0
    }
    #GiD is OK
    return 1
}

proc RetrievePT::GetWarningMessageAboutGiDVersion { i_module } {
    variable modulesarray

    set GidVersion [GiD_Info GiDVersion]
    if { [info exists modulesarray($i_module,MinimumGiDVersion)] } {
        set module_min_required_gid_version $modulesarray($i_module,MinimumGiDVersion)
    } else {
        set module_min_required_gid_version $GidVersion
    }
    if { [info exists modulesarray($i_module,MaximumGiDVersion)] } {
        set module_max_required_gid_version $modulesarray($i_module,MaximumGiDVersion)
    } else {
        set module_max_required_gid_version $GidVersion
    }

    set comp [GidUtils::VersionCmp $module_min_required_gid_version]
    if { $comp < 0 } {
        #GiD is Old
        set desc [_ "Warning: This module requires %s version %s and current version is %s. There can be incompatibilities." \
            $::GidPriv(ProgName) $module_min_required_gid_version $GidVersion]
    } else {
        set comp [GidUtils::VersionCmp $module_max_required_gid_version]
        if { $comp > 0 } {
            #GiD is too new
            set desc [_ "Warning: This module requires %s version %s or less, and current version is %s. There can be incompatibilities." \
                $::GidPriv(ProgName) $module_min_required_gid_version $GidVersion]
        } else {
            set desc [_ "%s version %s is supported but this module." $::GidPriv(ProgName) $GidVersion]
        }
    }
    return $desc
}

proc RetrievePT::GetLocalFolder { content } {
    if { $content == "Program" || $content == "Interface" } {
        set folder [gid_filesystem::get_folder_standard problemtypes]
    } elseif { $content == "Plugin" } {
        set folder [gid_filesystem::get_folder_standard plugins]
    } elseif { $content == "Example" } {
        set folder [gid_filesystem::get_folder_standard Examples]
    } elseif { $content == "Document" } {
        set folder [gid_filesystem::get_folder_standard info]
    } elseif { $content == "Theme" } {
        # set folder [gid_filesystem::get_folder_standard themes]
        set folder [gid_themes::GetThemesBaseFolder]
    } elseif { $content == "Fonts" } {
        set folder [gid_filesystem::get_folder_standard Fonts]
    } elseif { $content == "Resource" } {
        set folder [gid_filesystem::get_folder_standard resources]
    } elseif { $content == "Package" } {
        set folder [gid_filesystem::get_folder_standard scripts]
    } else {
        #unexpected
        set folder ""
    }
    return $folder
}

#in case that I want to check the root name of the xml file
proc RetrievePT::GetXmlRootName { content } {
    if { $content == "Program" || $content == "Interface" } {
        set xml_root_name Infoproblemtype
    } elseif { $content == "Plugin" } {
        set xml_root_name InfoPlugin
    } elseif { $content == "Example" } {
        set xml_root_name InfoExample
    } elseif { $content == "Document" } {
        set xml_root_name InfoDocument
    } elseif { $content == "Theme" } {
        set xml_root_name InfoTheme
    } elseif { $content == "Fonts" } {
        set xml_root_name InfoFont
    } elseif { $content == "Resource" } {
        set xml_root_name InfoResource
    } elseif { $content == "Package" } {
        set xml_root_name InfoPackage
    } else {
        #unexpected
        set xml_root_name ""
    }
    return $xml_root_name
}

proc RetrievePT::GetCallbackAfterUnzip { content module_xml_file shortdesc} {
    set desc [regsub -all \n $shortdesc {}]
    if { $content == "Program" || $content == "Interface" } {
        set callback_after_unzip [list RetrievePT::UpdateMenuProblemtypes $desc]
    } elseif { $content == "Plugin" } {
        set callback_after_unzip [list RetrievePT::LoadPlugin [file dirname $module_xml_file]]
    } elseif { $content == "Example" } {
        set callback_after_unzip [list GidUtils::SetWarnLine [format "%s '%s'" [_ "Module downloaded"] $desc]]
    } elseif { $content == "Document" } {
        set callback_after_unzip [list GidUtils::SetWarnLine [format "%s '%s'" [_ "Module downloaded"] $desc]]
    } elseif { $content == "Theme" } {
        set callback_after_unzip [list GidUtils::SetWarnLine [_ "Theme downloaded '%s'. Please, check Preferences -> Graphical -> Appearance" $desc]]
    } elseif { $content == "Fonts" } {
        # Rebuild font list already exists as messahe in preferences file
        set callback_after_unzip [list GidUtils::SetWarnLine [_ "Font(s) downloaded '%s'. Please, check Preferences -> Fonts -> %s" $desc [_ "Rebuild font list"]]]
    } elseif { $content == "Resource" } {
        set callback_after_unzip [list GidUtils::SetWarnLine [_ "Resources downloaded '%s'. Please, check Preferences." $desc]]
    } elseif { $content == "Package" } {
        set callback_after_unzip [list GidUtils::SetWarnLine [format "%s '%s'" [_ "Package downloaded"] $desc]]
    } else {
        # unexpected
        set callback_after_unzip [list GidUtils::SetWarnLine [_ "%s downloaded '%s'" $content $desc]]
    }
    return $callback_after_unzip
}


proc RetrievePT::CheckModuleInstalled { i_module } {
    variable modulesarray
    set name $modulesarray($i_module,Name)
    set version $modulesarray($i_module,Version)
    set platform $modulesarray($i_module,Platform)
    set content $modulesarray($i_module,Content)
    set localdir [RetrievePT::GetLocalFolder $content]
    set problemtype_exists 0
    set module_xml_file [file join $localdir $modulesarray($i_module,XmlFile)]
    set problemtype_exists [RetrievePT::GetModuleExists $module_xml_file $name $version $platform]
    if { $problemtype_exists } {
        set modulesarray($i_module,IsInstalled) 1
    } elseif { ![IsVersionOKForCurrentGiD $i_module] } {
        set modulesarray($i_module,IsInstalled) -1
    } else {
        set modulesarray($i_module,IsInstalled) 0
    }
}

proc RetrievePT::CheckModulesInstalled {} {
    variable num_modules
    for {set i_module 1} {$i_module<=$num_modules} {incr i_module} {
        if { [RetrievePT::MustShowInTable $i_module] } {
            RetrievePT::CheckModuleInstalled $i_module
        }
    }
}

proc RetrievePT::RetrieveSelectedModules { } {
    variable lists
    variable currenttab

    set widget $lists($currenttab)
    set selection [$widget curselection]
    if { [llength $selection] == 0 } {
        WarnWin [_ "Select first one module"] $widget
        return
    }
    set selected_modules {}
    foreach item $selection {
        set values [$widget get $item]
        set name [lindex $values 0]
        set version [lindex $values 1]
        set platform [lindex $values 2]
        lappend selected_modules [list $name $version $platform]
    }

    RetrievePT::RetrieveModules $selected_modules "" ""
}

proc RetrievePT::GetModuleExists { module_xml_file name version platform} {
    set problemtype_exists 0
    if { ![file exists $module_xml_file] } {
        #some part of the path could be a link
        set real_name [GidUtils::GiveRealFilenameDereferencingLinks $module_xml_file 1]
        if { $real_name != "" } {
            set module_xml_file $real_name
        }
    }
    set xml_root_name "" ;#empry to not try to check if "Infoproblemtype" or "InfoPlugin", etc.
    set data [ReadProblemtypeXml $module_xml_file $xml_root_name {Name Version Platform}]
    if { $data != "" } {
        array set problemtype_local $data
        if { ![info exists problemtype_local(Name)] } {
            W "Error in file $module_xml_file, expected a node 'Name'"
            continue
        }
        if { ![string compare -nocase $name $problemtype_local(Name)] } {
            # $version == $problemtype_local(Version)
            # if installed version is equal or greather than version in list/download.xml
            if { ![info exists problemtype_local(Version)] } {
                W "Error in file $module_xml_file, expected a node 'Version'"
                continue
            }
            if { $version == "" || [::GidUtils::TwoVersionsCmp $problemtype_local(Version) $version] >= 0 } {
                if { ![info exists problemtype_local(Platform)] || $problemtype_local(Platform) == "" } {
                    # not defined in XML
                    set problemtype_local(Platform) [RetrievePT::GetCurrentPlatform]
                }
                lassign $problemtype_local(Platform) local_os local_bits
                lassign $platform listed_os listed_bits
                # considering as installed the listed_32 bits version on a 64bits OS will also give
                # a false positive when the installed version is the 64bits one !
                # i.e. PT-OS-64 is installed, then both listed PT will be checked: PT-OS-64 AND PT-OS-32 !!!!
                if { $local_os == $listed_os || $local_os == "All" || $listed_os == "All"} {
                    if { $local_bits == $listed_bits ||
                        $local_bits == "All" || $listed_bits == "All" ||
                        (($local_bits == 64) && ($listed_bits == 32))} {
                        #current problemtype theorically match all requirements
                        # or the 32 bits version is installed in a 64 bits OS
                        return 1
                    }
                }
            }
        }
    }
    return 0
}

#this procedure try to create localdir or an auxiliary file
proc RetrievePT::CanWrite { localdir } {
    if { ![file exists $localdir] } {
        file mkdir $localdir
        if { [file exists $localdir] && [file isdirectory $localdir] } {
            file delete $localdir
            set can_write 1
        } else {
            set can_write 0
        }
    } else {
        set tmpname ""
        for { set i 0 } { $i < 1000 } {incr i} {
            set tmpname [file join $localdir gid$i].txt
            if { ![file exists $tmpname] } break
        }
        set fp ""
        catch {set fp [open $tmpname w]} msg
        if { $fp != "" } {
            close $fp
            file delete $tmpname
            set can_write 1
        } else {
            set can_write 0
        }
    }
    return $can_write
}

#modules is a list of modules to get, where each item specify {name version platform}
#callback could be set to invoke our own procedure after finish uncompression
#the callback could have arguments like %f that will be replaced by the temporary filename (to allow delete this file)
proc RetrievePT::RetrieveModules { modules {callback_unzipping ""} {callback_after_unzip ""}} {
    variable w
    variable modulesarray
    variable progressbar
    variable progresslabel
    variable FileSize
    variable LastTimeMiliSec

    package require gid_cross_platform

    if { [info exists w] && [winfo exists $w] } {
        set with_window 1
    } else {
        set with_window 0
    }

    if { ![llength [array names RetrievePT::modulesarray]] } {
        #if list was not retrieved try to get it now
        RetrievePT::Retrievelist
    }

    set GidVersion [GiD_Info GiDVersion]

    foreach item $modules {
        set name [lindex $item 0]
        set version [lindex $item 1]
        set platform [lindex $item 2] ;#a list with two words like "Windows 32"
        lassign $platform os bits

        lassign [RetrievePT::GetModuleIndex $name $version $platform] i_module i_best
        if { $i_module == 0 && $i_best != 0 } {
            # if 64 bits not found, try 32 bits...
            if { $bits == 64} {
                set platform32 [list [lindex $platform 0] 32]
                lassign [RetrievePT::GetModuleIndex $name $version $platform32] i_module32 i_best32
                if { $i_module32 != 0} {
                    set i_best $i_module32
                } elseif { $i_best32 != 0} {
                    # only set new best if previous best platform is different than demanded
                    if { ( $i_best == 0) || ( [lindex $modulesarray($i_best,Platform)] != [lindex $platform]) } {
                        set i_best $i_best32
                    }
                }
            }
            set text [_ "Warning: Required module '%s' version '%s' for '%s' and found version '%s' for '%s'. Continue?" \
                $name $version $platform $modulesarray($i_best,Version) $modulesarray($i_best,Platform)]

            set retval [GID_tk_messageBox -default ok -icon question -message $text -type okcancel -parent $w]
            if { $retval == "cancel" } {
                set progresslabel ""
                if { $with_window } {
                    set progressbar 0
                    GidUtils::EndWaitState $w
                }
                return 1
            } else {
                set i_module $i_best
            }
        }
        if { $i_module == 0 } {
            WarnWinText [_ "Warning: Required module '%s' not found" $name]
            return 1
        }
        if { ![IsVersionOKForCurrentGiD $i_module] } {
            set text [RetrievePT::GetWarningMessageAboutGiDVersion $i_module]
            append text " \n"
            append text [_ "Proceed?"]

            set retval [GID_tk_messageBox -default ok -icon question -message $text -type okcancel  -parent $w]
            if { $retval == "cancel" } {
                set progresslabel ""
                if { $with_window } {
                    set progressbar 0
                    GidUtils::EndWaitState $w
                }
                return 1
            }
        }

        set content $modulesarray($i_module,Content)
        set localdir [RetrievePT::GetLocalFolder $content]
        set problemtype_exists 0
        #find inside the xml file if match all requirements
        set module_xml_file [file join $localdir $modulesarray($i_module,XmlFile)]
        set problemtype_exists [GetModuleExists $module_xml_file $name $version $platform]
        if { $problemtype_exists  } {
            #already downloaded
            set text [_ "Warning: module '%s' is already installed in the local system. Are you sure to proceed?" $name]
            set retval [GID_tk_messageBox -default ok -icon question -message $text -type okcancel -parent $w]
            if { $retval == "cancel" } {
                set progresslabel ""
                if { $with_window } {
                    set progressbar 0
                    GidUtils::EndWaitState $w
                }
                return 1
            }
        }

        if { 0 } {
            #without administrative rights the folder maybe cannot be created
            if { ![file exists $localdir] } {
                file mkdir $localdir
            }

            if { ![file exists $localdir] || ![file isdirectory $localdir] } {
                WarnWin [_ "Directory %s does not exist" $localdir]
                set progresslabel ""
                if { $with_window } {
                    set progressbar 0
                    catch { GidUtils::EndWaitState $w }
                }
                return 1
            }
        }

        unset -nocomplain modulesarray($i_module,IsInstalled) ;#to refresh it
        set DownloadUrl ""
        if { [info exists modulesarray($i_module,DownloadUrl)]} {
            set DownloadUrl $modulesarray($i_module,DownloadUrl)
        }
        set DownloadFile $modulesarray($i_module,DownloadFile)
        # FileSize will be filled when the download starts

        set LastTimeMiliSec [clock clicks -milliseconds]
        set tmpfilename [gid_cross_platform::get_unused_tmp_filename download [file extension $DownloadFile]]

        set fo [open $tmpfilename w]
        fconfigure $fo -translation binary
        set err 0
        set file_data -1
        if { $DownloadUrl != "" } {
            set file_data [RetrievePT::HttpGetFile $DownloadUrl]
            # GidUtils::SetWarnLine "Retrieved from $DownloadUrl"
        }
        if { $file_data == -1} {
            # try DownloadFile
            set file_data [RetrievePT::HttpGetFile [RetrievePT::HttpGetFileUrlName $DownloadFile]]
            # GidUtils::SetWarnLine "Retrieved from [RetrievePT::HttpGetFileUrlName $DownloadFile]"
        }
        puts -nonewline $fo $file_data
        close $fo
        if { $file_data == -1} {
            # if error retireving file, just delete the temporary file
            file delete -force $tmpfilename
        }
        if { ![file exists $tmpfilename] } {
            set progresslabel ""
            if { $with_window } {
                set progressbar 0
                catch { GidUtils::EndWaitState $w }
            }
            WarnWin [_ "Problems retrieving the module"]
            return 1
        }
        set progresslabel "$FileSize/$FileSize"
        if { $with_window } {
            GidUtils::WaitState $w
            update
        }

        if { [file size $tmpfilename] == $FileSize } {
            set pwd [pwd]
            if { ![file exists $localdir] || ![file isdirectory $localdir] } {
                if { [catch { file mkdir $localdir } msg] } {
                    gid_cross_platform::run_as_administrator [GidUtils::GetFullPathGiDExe] -tclsh [file join [gid_filesystem::get_folder_standard scripts] run_as_administrator_procs.tcl] file mkdir $localdir
                    if { ![file exists $localdir] || ![file isdirectory $localdir] } {
                        set progresslabel ""
                        if { $with_window } {
                            set progressbar 0
                            catch { GidUtils::EndWaitState $w }
                        }
                        WarnWin $msg
                        return 1
                    }
                }
            }
            cd $localdir
            #set can_write [GiD_IsRunAsAdministrator]
            set can_write [RetrievePT::CanWrite $localdir]
            #arranco un proceso sin bloqueo con callback
            if { $callback_unzipping != "" } {
                regsub -all -- {%f} $callback_unzipping $tmpfilename callback_unzipping
            }
            if { [info exists modulesarray($i_module,ShortDescription)] } {
                set shortdesc [string trim $modulesarray($i_module,ShortDescription)]
            } else {
                set shortdesc ""
            }
            if { ![llength $callback_after_unzip] } {
                lappend callback_after_unzip [RetrievePT::GetCallbackAfterUnzip $content $module_xml_file $shortdesc]
            }
            set python_packages [list]
            if { [info exists modulesarray($i_module,PythonPackages)] } {
                set python_packages $modulesarray($i_module,PythonPackages)
            }
            lappend callback_after_unzip [list GiD_RaiseEvent GiD_Event_After_InternetRetrieve $content $name $version $platform $python_packages]

            if { $::tcl_platform(platform) == "windows"} {
                if { $callback_unzipping == "" } {
                    set callback_unzipping [list RetrievePT::UnzipProgress finished $tmpfilename $callback_after_unzip $shortdesc]
                }
                if { $can_write } {
                    set run_as_administrator 0
                    gid_cross_platform::gid_unzip $tmpfilename $localdir $run_as_administrator
                } else {
                    set run_as_administrator 1
                    set exe_name_process_to_check [file tail [GidUtils::GetFullPathGiDExe]] ;#e.g. gid.exe
                    set num_instancies_before [llength [gid_cross_platform::get_process_ids_from_name $exe_name_process_to_check]]
                    gid_cross_platform::gid_unzip $tmpfilename $localdir $run_as_administrator
                    # retrieving is asynchronous, so we'll wait until the new process with this name has finished
                    # maximum of 300 seconds (e.g. kratos expend a lot of time, 60 seconds are not enought)
                    set timeout 300
                    for { set i 0 } { $i < $timeout } { incr i } {
                        if { [llength [gid_cross_platform::get_process_ids_from_name $exe_name_process_to_check]] == $num_instancies_before} {
                            break
                        }
                        after 1000
                    }
                }
                #raise the callback
                uplevel \#0 $callback_unzipping
            } else {
                if { $callback_unzipping == "" } {
                    set callback_unzipping [list RetrievePT::UnzipProgress %m $tmpfilename $callback_after_unzip $shortdesc]
                }
                if { [string tolower [file extension $DownloadFile]] == ".tgz" } {
                    # set exe "gunzip -c $tmpfilename | tar -xf -"
                    set exe "tar -C \"$localdir\" -zxf $tmpfilename"
                } else {
                    # flags para ficheros ASCII (CR LF => LF)
                    # may be the selection of the uncompress program should be done in gid_cross_platform package
                    # but this is the only location using unzip
                    set unzip_exe [gid_cross_platform::get_unzip_exe]
                    if { [auto_execok $unzip_exe] != ""} {
                        set exe "$unzip_exe -a -o $tmpfilename -d \"$localdir\""
                    } elseif { [auto_execok 7z] != ""} {
                        # trying 7z ...
                        set exe "7z x -o\"$localdir\" -aoa -- $tmpfilename"
                        GidUtils::SetWarnLine [_ "'unzip' not found, using '7z' instead. Be careful with the line endings in text files."]
                    } else {
                        WarnWin [_ "Neither 'unzip' nor '7z' could be found to uncompress the retrieved package. Please install one of them."] $w
                        cd $pwd
                        return 0
                    }
                }
                if { !$can_write} {
                    # on macOs it is not a single command it is something like
                    # set exe [concat [gid_cross_platform::get_run_as_administrator_cmd] $exe]
                    set exe [gid_cross_platform::get_run_as_administrator_cmdline {*}$exe]
                }
                runExe run $exe -blocking false -callback $callback_unzipping -timeout 3600
            }
            cd $pwd
        } elseif { [winfo exists $w] } {
            WarnWin [_ "Error retrieving the module"] $w
        }
    }
    return 0
}

proc RetrievePT::DeleteFile { filename } {
    if { [catch { file delete $filename  } err_txt] } {
        gid_cross_platform::run_as_administrator [GidUtils::GetFullPathGiDExe] -tclsh [file join [gid_filesystem::get_folder_standard scripts] run_as_administrator_procs.tcl] file delete $filename
    }
}

proc RetrievePT::FormatDescriptionForProgressLabel { txt} {
    set desc [regsub -all \n $txt {} ]
    if { [string length $desc] >= 75} {
        # the '...' will be added in the calling function
        set desc [string range $desc 0 74]
    }
    return $desc
}

proc RetrievePT::UnzipProgress { mode args } {
    switch -exact -- $mode {
        prepare  {
            set desc [RetrievePT::FormatDescriptionForProgressLabel [lindex $args 2]]
            set ::RetrievePT::progresslabel [format "%s %s..." [_ "Unzipping"] $desc]
            update
        }
        output {
            #set ::RetrievePT::progresslabel [lindex $args 0]
        }
        finished {
            lassign $args download_file callback_after_unzip desc
            RetrievePT::DeleteFile $download_file
            RetrievePT::UpdateDownloadModuleWindow $desc
            foreach command $callback_after_unzip {
                uplevel \#0 $command
            }
        }
    }
}

proc RetrievePT::UpdateDownloadModuleWindow { desc } {
    variable w
    variable lists
    variable currenttab

    set desc [RetrievePT::FormatDescriptionForProgressLabel $desc]
    set ::RetrievePT::progresslabel [format "%s %s..." [_ "Module downloaded"] $desc]
    if { [info exists w] && [winfo exists $w] } {
        set with_window 1
    } else {
        set with_window 0
    }
    if { $with_window } {
        #::RetrievePT::CheckModulesInstalled
        GidUtils::EndWaitState $w
        set ::RetrievePT::progressbar 0
        set list $lists($currenttab)
        RetrievePT::FillTable $list $currenttab
        update
        after 3000 { set ::RetrievePT::progresslabel "" }
    }
}

proc RetrievePT::UpdateMenuProblemtypes { desc } {
    variable w
    variable lists
    variable currenttab

    GidUtils::SetWarnLine [format "%s '%s'" [_ "Module downloaded"] $desc]
    ClearCacheFormProblemtypeMenuDir
    #if the problemtypes menu must be deleted to use FastCacheFromProblemtypeMenu
    #FastCacheFromProblemtypeMenu

}

proc RetrievePT::LoadProblemtype { projecttype } {
    #already downloaded and unzipped (or timeout exceeded)
    RetrievePT::Close ;#close because have used RetrievePT::OpenGUI to show progress downloading (but not unzipping)
    GidUtils::SetWarnLine [_ "Problemtype downloaded"]
    #try to load now this downloaded problemtype, when uncompression is finished
    SetProblemTypeChoose $projecttype 0
}

proc RetrievePT::LoadPlugin { plugin_folder } {
    #already downloaded and unzipped (or timeout exceeded)
    GidUtils::SetWarnLine [_ "Plugin downloaded"]
    #try to load now this downloaded plugin, when uncompression is finished
    SourcePlugins $plugin_folder
}

proc RetrievePT::GetWindowId {} {
    variable w
    return $w
}

proc RetrievePT::SetProblemTypeIfNotExitsTryDownload { projecttype } {
    variable w
    set fail 0
    set download_filename [file join [file dirname [GidUtils::GiveFileInsideProblemType $projecttype "" 0]] download.xml]
    set data [ReadProblemtypeXml $download_filename DownloadProblemtype {Name Version ShortDescription}]
    if { $data != "" } {
        array set module_download $data
        set module_download(Platform) [RetrievePT::GetCurrentPlatform]
        set module_xml_file [GidUtils::GiveFileInsideProblemType $projecttype .xml 1]
        set problemtype_exists [RetrievePT::GetModuleExists $module_xml_file $module_download(Name) $module_download(Version) $module_download(Platform)]
        if { $problemtype_exists } {
            #ok, already downloaded
            SetProblemTypeChoose $projecttype 0 ;#invoke it again to continue loading
        } else {
            set text [_ "Module '%s' version '%s' will be download from Internet. Continue?" $module_download(Name) $module_download(Version)]
            set text [concat $text \n\n\n [_ "Description"]: $module_download(ShortDescription)]
            set retval [GID_tk_messageBox -default ok -icon question -message $text -type okcancel \
                            {*}[GidUtils::GiDDefaultParentIfPresent]]
            if { $retval == "cancel" } {
                set fail 1
            } else {
                set modules [list [list $module_download(Name) $module_download(Version) $module_download(Platform)]]
                RetrievePT::OpenGUI
                #the callback could have arguments like %f that will be replaced by the temporary filename
                set callback_unzipping "" ;#if replace it must raise in the procedure the callback_after_unzip when finised!!!
                lappend callback_after_unzip [list RetrievePT::LoadProblemtype $projecttype]
                set fail [RetrievePT::RetrieveModules $modules $callback_unzipping $callback_after_unzip]
                if { $fail } {
                    GidUtils::SetWarnLine [_ "Error downloading module '%s %s'" $module_download(Name) $module_download(Version)]
                }
            }
        }
    } else {
        SetProblemTypeChoose $projecttype 0 ;#invoke it again to continue loading
    }
    return $fail
}


proc RetrievePT::DisplayText { parent textinfo } {

    set w $parent.retrievepttext

    InitWindow2 $w -title [_ "Display info"] \
        -geometryvariable PreRetrievePTDisplayHelpWindowGeom \
        -ontop
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0

    set sw [ScrolledWindow $w.lf \
        -auto both -relief sunken -borderwidth 2]
    set text [text $sw.text -width 60 -height 20]
    bind $text <1> "focus $text"
    $sw setwidget $text

    ttk::button $w.close -text [_ "Close"] -width 10 -command "destroy $w" -style BottomFrame.TButton
    grid $sw -sticky nsew
    grid $w.close

    grid columnconf $w 0 -weight 1
    grid rowconf $w 0 -weight 1

    $text insert end $textinfo
    $text configure -state disabled
}

proc RetrievePT::ModuleInfo {} {
    variable modulesarray
    variable lists
    variable currenttab

    set widget $lists($currenttab)
    set item [$widget curselection]

    if { [llength $item] == 0 } {
        WarnWin [_ "Select first one module"] $widget
        return
    }
    if { [llength $item] > 1 } {
        WarnWin [_ "Select only one module"] $widget
        return
    }

    set values [$widget get $item]
    set name [lindex $values 0]
    set version [lindex $values 1]
    set platform [lindex $values 2]
    lassign [RetrievePT::GetModuleIndex $name $version $platform] i_module i_best
    if { $i_module == 0 } {
        WarnWinText [_ "Warning: Required module '%s' not found" $name]
        return 1
    }
    set desc ""
    if { ![IsVersionOKForCurrentGiD $i_module] } {
        set desc [RetrievePT::GetWarningMessageAboutGiDVersion $i_module]
    }
    if { [info exists modulesarray($i_module,Description)] } {
        append desc $modulesarray($i_module,Description)
    } elseif { [info exists modulesarray($i_module,ShortDescription)] } {
        append desc $modulesarray($i_module,ShortDescription)
    } else {
        WarnWin [_ "This module has no description"] $widget
        return
    }
    RetrievePT::DisplayText $widget $desc
}

proc RetrievePT::ModuleNews {} {
    variable modulesarray
    variable lists
    variable currenttab

    set widget $lists($currenttab)
    set item [$widget curselection]

    if { [llength $item] == 0 } {
        WarnWin [_ "Select first one module"] $widget
        return
    }
    if { [llength $item] > 1 } {
        WarnWin [_ "Select only one module"] $widget
        return
    }

    set values [$widget get $item]
    set name [lindex $values 0]
    set version [lindex $values 1]
    set platform [lindex $values 2]
    lassign [RetrievePT::GetModuleIndex $name $version $platform] i_module i_best
    if { $i_module == 0 } {
        WarnWinText [_ "Warning: Required module '%s' not found" $name]
        return 1
    }

    set desc ""
    if { ![IsVersionOKForCurrentGiD $i_module] } {
        set desc [RetrievePT::GetWarningMessageAboutGiDVersion $i_module]
    }
    if { [info exists modulesarray($i_module,NewsInVersion)] } {
        append desc $modulesarray($i_module,NewsInVersion)
    } else {
        WarnWin [_ "This module has no news information"] $widget
        return
    }
    RetrievePT::DisplayText $widget $desc
}

proc RetrievePT::GetCurrentPlatform {} {
    set is64bits 0
    if { $::tcl_platform(pointerSize) == 8 } {
        set is64bits 1
    }
    if { $::tcl_platform(platform) == "windows" } {
        if { $is64bits } {
            set myplatform "Windows 64"
        } else {
            set myplatform "Windows 32"
        }
    } elseif { $::tcl_platform(os) == "Linux" } {
        if { $is64bits } {
            set myplatform "Linux 64"
        } else {
            set myplatform "Linux 32"
        }
    } elseif { $::tcl_platform(os) == "Darwin" } {
        if { $is64bits } {
            set myplatform "MacOSX 64"
        } else {
            set myplatform "MacOSX 32"
        }
    } else {
        #AIX , HP-UX, IRIX, SunOS, DragonFly, FreeBSD, NetBSD, OpenBSD, OSF1
        set myplatform "$::tcl_platform(os) ?"
    }
    return $myplatform
}

proc RetrievePT::OpenGUI { { parent .gid } } {
    variable w
    variable progressbar
    variable progresslabel
    variable lists
    variable currenttab
    variable show_content
    variable show_filter

    package require tablelist_tile

    set w $parent.retrievept

    catch { destroy $parent.retrievept }

    InitWindow2 $w -title [_ "Retrieve modules"] \
        -geometryvariable PreRetrievePTWindowGeom \
        -initcommand RetrievePT::OpenGUI
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0

    wm protocol $w WM_DELETE_WINDOW RetrievePT::Close


    ttk::label $w.l1 -text \
        [_ "Note: In order to use this function, it is necessary to be connected to internet"] \
        -font BoldFont
    ttk::frame $w.fshow
    ttk::labelframe $w.fshow.category -text [_ "Category"]
    set contents {Program Interface Plugin Example Document Theme Fonts Resource Package}
    set content_labels [list [_ "Simulation programs"] [_ "Interfaces"] [_ "Plugins"] [_ "Examples"] [_ "Documents"] [_ "Themes"] [_ "Fonts"] [_ "Resources"] [_ "Packages"]]
    set content_helps  [list [_ "A problemtype with the interface and the calculation solver"] \
        [_ "a problemtype that doesn't include the third part calculation solver"] \
        [_ "A plugin is a loadable code, located in the 'plugins' folder, that implements extra features"] \
        [_ "Examples of GiD or other modules"] \
        [_ "Documents of GiD or other modules"] \
        [_ "Themes and skins with new icons and colours combinations"] \
        [_ "Extra Fonts files, for instance for non-latin languages"] \
        [_ "Resources, such as message catalogs for other languages"] \
        [_ "Tcl packages"]]
    set content_columns {0 1 2 3 4 5 6 7 8}
    foreach content $contents text $content_labels help $content_helps column $content_columns {
        ttk::checkbutton $w.fshow.category.cb$column -variable RetrievePT::show_content($content) -text $text
        GidHelp $w.fshow.category.cb$column $help
    }
    ttk::labelframe $w.fshow.filter -text [_ "Filter"]
    set filters {NotInstalled CompatibleVersion}
    set filter_labels [list [_ "Not installed"] [_ "Compatible"]]
    set filter_helps [list [_ "To hide modules already installed"] [_ "To hide modules that require a more modern GiD version"]]
    set filter_columns {0 1}
    foreach filter $filters text $filter_labels help $filter_helps column $filter_columns {
        ttk::checkbutton $w.fshow.filter.cb$column -variable RetrievePT::show_filter($filter) -text $text
        GidHelp $w.fshow.filter.cb$column $help
    }

    set nb [ttk::notebook $w.nb]
    ttk::notebook::enableTraversal $nb

    set f1 [ttk::frame $nb.f1]
    $nb add $f1 -text [RetrievePT::GetCurrentPlatform] -underline 0
    set lists(0) [RetrievePT::CreateTable $f1]

    set f2 [ttk::frame $nb.f2]
    $nb add $f2 -text [_ "Other OS"] -underline 0
    set lists(1) [RetrievePT::CreateTable $f2]

    set wp [ttk::progressbar  $w.f -variable ::RetrievePT::progressbar -mode determinate -maximum 100.0]
    ttk::label $w.l2 -textvariable RetrievePT::progresslabel
    set progressbar 0
    set progresslabel ""

    grid $w.l1 -sticky ew -padx 2

    grid $w.fshow.category.cb0 $w.fshow.category.cb1 $w.fshow.category.cb2 $w.fshow.category.cb3 $w.fshow.category.cb4 -sticky w
    grid $w.fshow.category.cb5 $w.fshow.category.cb6 $w.fshow.category.cb7 $w.fshow.category.cb8 -sticky w
    grid $w.fshow.filter.cb0 $w.fshow.filter.cb1 -sticky w
    grid $w.fshow.category $w.fshow.filter -sticky new
    grid $w.fshow -sticky new
    grid columnconfigure $w.fshow {0 1} -weight 1

    grid $nb -sticky nsew
    grid $w.l2 -sticky ew -padx 2
    grid $wp -sticky ew -padx 2
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 2 -weight 1

    ttk::frame $w.buts -style BottomFrame.TFrame

    ttk::button $w.buts.b2 -text [_ "Retrieve module"] -command RetrievePT::RetrieveSelectedModules \
        -style BottomFrame.TButton
    ttk::button $w.buts.b3 -text [_ "Module information"] -command RetrievePT::ModuleInfo \
        -style BottomFrame.TButton
    ttk::button $w.buts.b4 -text [_ "Module news"] -command RetrievePT::ModuleNews \
        -style BottomFrame.TButton
    ttk::button $w.buts.b5 -text [_ "Close"] -command RetrievePT::Close \
        -style BottomFrame.TButton

    grid $w.buts.b2 $w.buts.b3 $w.buts.b4 $w.buts.b5 -padx 2 -pady 3

    grid $w.buts -sticky ew
    grid columnconf $w.buts {0 1 2 3} -weight 1
    grid anchor $w.buts center

    update idletasks

    set progresslabel [_ "Retrieving module list"]...
    update
    RetrievePT::Retrievelist
    foreach tab {0 1} {
        RetrievePT::FillTable $lists($tab) $tab
        $lists($tab) sortbycolumn 4 -decreasing
    }
    if { ![info exists currenttab] } {
        set currenttab 0
    }
    $nb select $currenttab

    set progresslabel ""
    update
    bind $w <<NotebookTabChanged>> [list RetrievePT::TabChanged $nb]
    bind $w <Destroy> [list +RetrievePT::OnDestroyWindow %W $w] ;# + to add to previous script
    foreach content $contents {
        trace add variable RetrievePT::show_content($content) write [list RetrievePT::OnChangeShowContent]
    }
    foreach filter $filters {
        trace add variable RetrievePT::show_filter($filter) write [list RetrievePT::OnChangeShowContent]
    }
}

proc RetrievePT::OnDestroyWindow { W w } {
    variable show_content
    variable show_filter
    if { $W != $w } return
    set contents {Program Interface Plugin Example Document Theme Fonts Resource Package}
    foreach content $contents {
        trace remove variable RetrievePT::show_content($content) write [list RetrievePT::OnChangeShowContent]
    }
    set filters {NotInstalled CompatibleVersion}
    foreach filter $filters {
        trace remove variable RetrievePT::show_filter($filter) write [list RetrievePT::OnChangeShowContent]
    }
}

proc RetrievePT::OnChangeShowContent { args } {
    variable lists
    RetrievePT::FillTable $lists(0) 0
    RetrievePT::FillTable $lists(1) 1
}

proc RetrievePT::TabChanged { nb } {
    variable currenttab
    set currenttab [$nb index current]
}

proc RetrievePT::CompareData { item1 item2 } {
    lassign [split $item1 .] d1 m1 y1
    lassign [split $item2 .] d2 m2 y2
    set n1 [format %04s%02s%02s $y1 $m1 $d1]
    set n2 [format %04s%02s%02s $y2 $m2 $d2]
    if { $n1 < $n2 } {
        return -1
    } elseif { $n1 > $n2 } {
        return 1
    } else {
        return 0
    }
}

proc RetrievePT::CreateTable { w } {
    set sw [ScrolledWindow $w.lf \
        -auto both -relief sunken -borderwidth 2]

    package require tablelist_tile
    set list [tablelist::tablelist $sw.lb \
        -exportselection 0 \
        -columns [list \
            18 [_ "Module"] left \
            8 [_ "Version"] left \
            12 [_ "Platform"] left \
            8 [_ "Installed"] center \
            10 [_ "Publish date"] center \
            20 [_ "Characteristics"] left \
        ] \
        -labelcommand tablelist::sortByColumn \
        -stretch end -selectmode extended]

    $list columnconfigure 4 -sortmode command -sortcommand RetrievePT::CompareData
    $sw setwidget $list

    bind [$list bodytag] <Button-$::gid_right_button> "[list RetrievePT::MenuContextualRetrievePT %W %x %y] ; break"
    grid $sw -sticky nsew
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1
    return $list
}

proc RetrievePT::MenuContextualRetrievePT { T x y } {
    if { [winfo class [winfo parent $T]] != "Tablelist" } {
        return
    }
    set txt [[winfo parent $T] cellcget active -text]
    if { $txt == "" } return
    set w $T.menucontextualretrievept
    if { [winfo exists $w] } {
        destroy $w
    }
    menu $w
    $w add command -label [_ "Retrieve %s" $txt] -command RetrievePT::RetrieveSelectedModules
    $w add command -label [_ "%s info" $txt] -command RetrievePT::ModuleInfo
    $w add command -label [_ "%s news" $txt] -command RetrievePT::ModuleNews
    set x [expr [winfo rootx $T]+$x+2]
    set y [expr [winfo rooty $T]+$y]
    GiD_PopupMenu $w $x $y
}

