
global MeshErrorsPriv 
#don't change global MeshErrorsPriv to a MeshErrors namespace variable, 
#there are problems to set it with Tcl_SetVar2 in this case, probably because this tcl
#code was not sourced and the namespace doesn't exist until its procedures are called trough tclIndex

namespace eval MeshErrors {    
}

proc MeshErrors::DestroyMeshErrors { } {
    set w .gid.mesherrors
    if { [winfo exists $w] } {
        destroy $w
    }
}

proc MeshErrors::CreateWindow { } {        
    if { ![info exists ::MeshErrorsPriv] } {
        WarnWin [_ "Meshing error information not available"]
        return 1
    }
    set w .gid.mesherrors
    InitWindow2 $w -title [_ "Mesh errors"] \
        -geometryvariable PreMeshErrorsWindowGeom -ontop
    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0

    ttk::frame $w.fup
    
    set def_back [$w cget -background]    
    set nb [NoteBook $w.fup.opt -internalborderwidth 1 -activebackground [CCColorActivo $def_back]]

    foreach entity_type {line surface volume boundary_layer} label [list [_ "Lines"] [_ "Surfaces"] [_ "Volumes"] [_ "Boundary Layer"]] {
        if { [info exists ::MeshErrorsPriv($entity_type,num_rows)] } {
            set f [$nb insert end $entity_type -text $label -raisecmd [list set ::MeshErrorsPriv(current_type) $entity_type]]
            MeshErrors::CreateWidgetsFrame $f
            $nb raise $entity_type
        }
    }
    
    grid $w.fup -sticky nsew
    grid $nb -sticky nsew
    grid rowconfigure $w.fup 0 -weight 1
    grid columnconfigure $w.fup 0 -weight 1
    update idletasks

    #lower buttons
    ttk::frame $w.frmButtons  -style BottomFrame.TFrame

    ttk::button $w.frmButtons.btnsignal -text [_ "Signal"] \
            -command [list MeshErrors::SignalSelected2 $nb] -underline 0 -style BottomFrame.TButton
    ttk::button $w.frmButtons.btnclose -text [_ "Close"] \
            -command [list destroy $w] -underline 0 -style BottomFrame.TButton

    grid $w.frmButtons -sticky ews -columnspan 7
    grid anchor $w.frmButtons center
    grid $w.frmButtons.btnsignal $w.frmButtons.btnclose -padx 3 -pady 3
    grid columnconf $w 0 -weight 1
    grid rowconfigure $w 0 -weight 1

    focus $w.frmButtons.btnclose   

    wm minsize $w 160 200

    update idletasks

    #fill values
    foreach entity_type {line surface volume boundary_layer} label [list [_ "Lines"] [_ "Surfaces"] [_ "Volumes"] [_ "Boundary Layer"]] {
        if { [info exists ::MeshErrorsPriv($entity_type,num_rows)] } {
            set f $nb.f$entity_type
            set num_rows $::MeshErrorsPriv($entity_type,num_rows)
            $f.lbl configure -text [_ "%s wrong %s" $num_rows $label]
            $f.frtxt.t delete 0 end
            $f.frtxt.t insert end [_ "Num"][string repeat " " 7][_ "Description"]
            
            for {set row 1} {$row <= $num_rows } {incr row} {
                set text  ""
                foreach column_type {entity_id description} {
                    if { [info exists ::MeshErrorsPriv($entity_type,$row,$column_type)] } {
                        append text $::MeshErrorsPriv($entity_type,$row,$column_type)
                    } else {
                        append text ?
                    }
                    append text "   "
                }
                $f.frtxt.t insert end [string trim $text]
            }
            bind $f.frtxt.t <Double-Button-1> [list MeshErrors::SignalSelected %W]
        }
    }    
    return 0
}

proc MeshErrors::SignalSelected2 { nb } {
    global MeshErrorsPriv
    if { ![info exists ::MeshErrorsPriv(current_type)] } {
        WarnWin [_ "Meshing error information not available"]     
    } else {
        set entity_type $::MeshErrorsPriv(current_type)
        set f $nb.f$entity_type
        set list $f.frtxt.t
        if { [$list curselection] == "" } {
            WarnWin [_ "Must select the error line to point near the problematic region"]
        } else {
            MeshErrors::SignalSelected $list
        }
    }
}

proc MeshErrors::SignalSelected { list } {
    global MeshErrorsPriv
    set row [$list curselection]
    set entity_type $::MeshErrorsPriv(current_type)
    set num_rows $::MeshErrorsPriv($entity_type,num_rows)
    if { $row < 1  || $row > $num_rows } return
    set view_mode [GiD_Info Project ViewMode]
    if { [info exists ::MeshErrorsPriv($entity_type,$row,coordinate)] } {        
        if { $view_mode == "GEOMETRYUSE" } {
            GiD_Process Mescape Utilities SignalEntities Points FNoJoin {*}$::MeshErrorsPriv($entity_type,$row,coordinate)
        } elseif { $view_mode == "MESHUSE" }  {
            GiD_Process Mescape Utilities SignalEntities Nodes FNoJoin {*}$::MeshErrorsPriv($entity_type,$row,coordinate)
        } else {
            
        }
    } elseif { [info exists ::MeshErrorsPriv($entity_type,$row,entity_id)] } {
        if  { $entity_type == "line" } {
            GiD_Process Mescape Utilities SignalEntities Lines $::MeshErrorsPriv($entity_type,$row,entity_id)
        } elseif  { $entity_type == "surface" } {
            GiD_Process Mescape Utilities SignalEntities Surfaces $::MeshErrorsPriv($entity_type,$row,entity_id)
        } elseif  { $entity_type == "volume" } {
            GiD_Process Mescape Utilities SignalEntities Volumes $::MeshErrorsPriv($entity_type,$row,entity_id)
        } else {
            #ignore it
        }
    } else {
        #ignore it
    }   
}

proc MeshErrors::CreateWidgetsFrame { fr } {
    ttk::label $fr.lbl -text [_ "Number of wrong entities"]:
    ttk::frame $fr.frtxt -borderwidth 0

    grid $fr.lbl -sticky new
    grid $fr.frtxt -sticky nsew -columnspan 7

    grid rowconf $fr 1 -weight 1
    grid columnconf $fr 0 -weight 1

    ttk::scrollbar $fr.frtxt.scrolly -command [list $fr.frtxt.t yview] -orient vertical
    ttk::scrollbar $fr.frtxt.scrollx -command [list $fr.frtxt.t xview] -orient horizontal

    tk::listbox $fr.frtxt.t -font SmallFont -yscrollcommand "$fr.frtxt.scrolly set" \
            -xscrollcommand "$fr.frtxt.scrollx set" -width 40 -height 5
    grid $fr.frtxt.t -row 0 -column 0 -sticky nsew
    grid $fr.frtxt.scrolly -row 0 -column 1 -sticky ns
    grid $fr.frtxt.scrollx -row 1 -column 0 -sticky ew
    grid rowconf $fr.frtxt 0 -weight 1
    grid columnconf $fr.frtxt 0 -weight 1

    grid remove $fr.frtxt.scrolly
    grid remove $fr.frtxt.scrollx
    bind $fr.frtxt.t <Configure> [list MeshErrors::ShowOrHideScroll $fr.frtxt.t $fr.frtxt.scrollx $fr.frtxt.scrolly]
    bind $fr.frtxt.t <$::gid_right_button> [list MeshErrors::ContextualMeshErrors $fr.frtxt.t %x %y %X %Y]
}

proc MeshErrors::ContextualMeshErrors { lb x y X Y} {
    if { [$lb curselection] != [$lb index @$x,$y] } {
        $lb selection clear 0 end
        $lb selection set "@$x,$y"
    }
    set menu $lb.menu
    destroy $menu
    menu $menu -tearoff 0
    $menu add command -label [_ "Signal problematic point"] -command [list MeshErrors::SignalSelected $lb]
    $menu add command -label [_ "More help"]... -command [list MeshErrors::MeshErrorsHelp $lb]
    $menu add separator
    $menu add command -label [_ "List"]... -command [list MeshErrors::Details]
    tk_popup $menu $X $Y
}

#text must be a listbox, text or canvas widget and sx,sy scrollbar widgets
proc MeshErrors::ShowOrHideScroll { text sx sy } {
    foreach i "x y" {
        if { ![info exists s${i}] || ![winfo exists [set s${i}]] } { continue }
        foreach "${i}1 ${i}2" [$text ${i}view] break
        
        if { [set ${i}1] == 0 && [set ${i}2] == 1 } {
            grid remove [set s${i}]
        } else {
            grid [set s${i}]
        }
    }
}

proc MeshErrors::Details { } {
    global MeshErrorsPriv
    set text ""
    foreach entity_type {line surface volume boundary_layer} label [list [_ "Lines"] [_ "Surfaces"] [_ "Volumes"] [_ "Boundary Layer"]] {
        if { [info exists ::MeshErrorsPriv($entity_type,num_rows)] } {     
            set num_rows $::MeshErrorsPriv($entity_type,num_rows)
            append text [_ "%s problematic %s" $num_rows $label]:
            for {set row 1} {$row <= $num_rows } {incr row} {
                append text " "
                append text $::MeshErrorsPriv($entity_type,$row,entity_id)
            }
            append text "\n"
        }
    }    
    WarnWinText $text [_ "Problems meshing"]
}

proc MeshErrors::ExistsMessages {} {
  return [info exists ::MeshErrorsPriv]
}

# return a list with errors by cathegory (line surface volume boundary_layer) 
# with a sublist of several messages, each one with the entity_id error_description and coordinates ("" if not available)
proc MeshErrors::GetMessages {} {
    global MeshErrorsPriv
    set data [list]
    foreach entity_type {line surface volume boundary_layer} {
        if { [info exists ::MeshErrorsPriv($entity_type,num_rows)] } {
            set item $entity_type
            set num_rows $::MeshErrorsPriv($entity_type,num_rows)
            set values [list]
            for {set row 1} {$row <= $num_rows } {incr row} {
                set values_i [list]
                foreach question {entity_id description coordinate} {
                    set value ""
                    if { [info exists ::MeshErrorsPriv($entity_type,$row,$question)] } {
                        set value $::MeshErrorsPriv($entity_type,$row,$question)
                    }
                    lappend values_i $value
                }
                lappend values $values_i
            }
            lappend data [list $item $values]
        }
    }
    return $data
}

#data must have the same list information that MeshErrors::GetMessages returns
proc MeshErrors::SetMessages { data } {
    array unset ::MeshErrorsPriv
    foreach item $data {
        lassign $item entity_type values
        foreach values_i $values {
            lassign $values_i entity_id description coordinate
            incr num_rows($entity_type)
            foreach question {entity_id description coordinate} {
                set row $num_rows($entity_type)
                set ::MeshErrorsPriv($entity_type,$row,$question) [set $question]
            }
        }
    }
    foreach entity_type {line surface volume boundary_layer} {
        if { [info exists num_rows($entity_type)] } {
            set ::MeshErrorsPriv($entity_type,num_rows) $num_rows($entity_type)
        }
    }
    return 0
}

proc MeshErrors::MeshErrorsHelp { list } {
    global MeshErrorsPriv
    if { [$list curselection] == "" } return
    #WarnWinText "show more inforation and advices about the error  [$lb get [$lb curselection]]"
    #set info  [$lb get [$lb curselection]]
    #set i [lindex $info 0]
    #set description [lrange $info 1 end]

    set row [$list curselection]
    set entity_type $::MeshErrorsPriv(current_type)
    set num_rows $::MeshErrorsPriv($entity_type,num_rows)
    if { $row < 1  || $row > $num_rows } return

    set entity_id $::MeshErrorsPriv($entity_type,$row,entity_id)
    if { $entity_type == "line" } {
        set text [_ "Not available more information about the error '%s'" $::MeshErrorsPriv($entity_type,$row,description)]
    } elseif { $entity_type == "surface" } {
        set text [_ "Surface number %d" $entity_id]
        switch -regexp $::MeshErrorsPriv($entity_type,$row,description) {
            "Couldn't map this point" {
                #"Check self intersections
                append text \n[_ "Check if the boundaries have self intersections."]
                if { $entity_id > 0 } {
                    set self_intersected [GiD_Info check $entity_type $entity_id selfintersection]
                    if { $self_intersected != "" } {
                        append text [_ "Detected %s self intersections." [llength $self_intersected]]
                    }
                }
            }
            "Failed entering octtree" {
                #"Check self intersections, or bad oriented boundary"
                append text \n[_ "Check if the boundaries have self intersections."]
                if { $entity_id > 0 } {
                    set self_intersected [GiD_Info check $entity_type $entity_id selfintersection]
                    if { $self_intersected != "" } {
                        append text [_ "Detected %s self intersections." [llength $self_intersected]]
                    }
                }
                append text \n[_ "Also verify if the surface normal is agree with its boundary lines sense (right hand rule)"]
            }
            "Failed find close faces to new node" {
                #"Check self intersections, or bad oriented boundary"
                append text \n[_ "Check if boundaries have self intersections or some very small corner angle."]
                if { $entity_id > 0 } {
                    set self_intersected [GiD_Info check $entity_type $entity_id selfintersection]
                    if { $self_intersected != "" } {
                        append text [_ "Detected %s self intersections." [llength $self_intersected]]
                    }
                    #set is_degenerated [GiD_Info check $entity_type $entity_id isdegeneratedboundary]
                    set is_closed [GiD_Info check $entity_type $entity_id isclosed]
                    if { $is_closed > 0 } {
                        if { $isclosed == 1 } {
                            set sense "u"
                        } elseif { $isclosed == 2 } {
                            set sense "v"
                        } else {
                            set sense "u,v"
                        }
                        append text [_ "closed surface: Try to divide in %s" $sense]
                    }
                }
            }
            "Failed FillInitialFaces" {
                set area 1
                regexp {Area=([^\n]+)\n} [GiD_Info list_entities -more surfaces $entity_id] {} area
                if { $area <=  1e-25 } {
                    append text  \n[_ "The surface has zero area, probably must be deleted. Can try to collapse the model with a small tolerence"]
                } else {
                    if { [GiD_Set AutomaticCorrectSizes] == 0 } {
                        append text  \n[_ "The preference 'Automatic correct sizes' is unselected: must accurately assign mesh sizes"]
                    } else {
                        append text  \n[_ "Check the assignment of mesh sizes"]
                    }
                }
            }
            "Failed calc position of new node" {
                if { [GiD_Set AutomaticCorrectSizes] == 0 } {
                    append text  \n[_ "The preference 'Automatic correct sizes' is unselected: must accurately assign mesh sizes"]
                } else {
                    append text  \n[_ "Check the assignment of mesh sizes"]
                }
            }
            default {
                append text  \n[_ "Not available more information about the error '%s'" $::MeshErrorsPriv($entity_type,$row,description)]
            }
        }
        set gentext [_ "General advices to generate the surface mesh"]:
        append gentext \n-[_ "Avoid corners with very small angles"]
        append gentext \n-[_ "Divide in two parts closed surfaces: e.g. instead a cylinder make two semi-cylinders"]
        append gentext \n-[_ "Try to delete and create again from his boundaries the problematic surface"]
        if { [GiD_Set AutomaticCorrectSizes] == 0 } {
            append gentext \n-[_ "When the preference 'Automatic correct sizes' is unselected, must manually assign appropiated mesh sizes to geometric entities"]
        }
        append text "\n\n$gentext"
    } elseif { $entity_type == "volume" } {
        set text [_ "Volume number %d" $entity_id]
        switch -regexp $::MeshErrorsPriv($entity_type,$row,description) {
            "Contour element ([0-9])+ is not correct" {
                #"Contour element 126 is not correct"
                #ejemplo: caso de un cono con una sola superficie de revolucion de 360 grados
                append text \n[_ "If the boundary surface is closed, try to divide it in two parts"]
            }
            "Couldn't mesh: not enough space in octtree -2-" {
                append text \n[_ "If the boundary surface is closed try to divide it in two parts: e.g. instead a cylinder make two semi-cylinders"]
                if { $entity_id > 0 } {
                    set bad_oriented [GiD_Info check $entity_type $entity_id orientation]
                    if { [lindex $bad_oriented 0] > 0 } {
                        append text [_ "%s bad oriented surfaces: %s. Use repair to correct it" [lindex $bad_oriented 0] [lindex $bad_oriented 1]]                    
                    }
                }
            }
            "Delaunay mesher fail. Try advancing front mesher" {
                append text \n[_ "Probably the volume Delaunay mesher is unable to create a mesh constrained to the boundary. Try other sizes or change mesher"]
            }
            default {
                append text \n[_ "Not available more information about the error '%s'" $::MeshErrorsPriv($entity_type,$row,description)]
                
            }
        }
        set gentext [_ "General advices to generate the volume mesh"]:
        append gentext \n[_ "1- Force the mesh generation for the volume surfaces (Mesh criteria->Mesh->Surfaces)"]
        append gentext \n[_ "2- See this surface mesh near the problematic point."]
        append gentext \n[_ "Typical problems are element intersections, small angles, big elements near small ones, etc."]
        append gentext \n[_ "3- Modify the geometry, or the assignment of mesh sizes to solve the problem."]
        if { [GiD_Set AutomaticCorrectSizes] == 0 } {
            append gentext \n-[_ "When the preference 'Automatic correct sizes' is unselected, must manually assign appropiated mesh sizes to geometric entities"]
        }
        append text "\n\n$gentext"
    } elseif { $entity_type == "boundary_layer" } {
        set text [_ "Not available more information about the error '%s'" $::MeshErrorsPriv($entity_type,$row,description)]
    } else {
        set text [_ "Unexpected entity type %s" $entity_type]
    }
    WarnWinText $text [_ "Problems meshing"]
}
