
namespace eval BoundaryLayer {
    variable NumberOfLayers 1
    variable DistanceFirst 0.1
}

proc BoundaryLayer::ActualizeDraw { c args } {
    variable NumberOfLayers
    variable DistanceFirst
    
    set w .gid.wBoundaryLayer


    if { ![string is integer -strict $NumberOfLayers] || $NumberOfLayers <= 0} {
        return
    }
    if { ![string is double -strict $DistanceFirst] || $DistanceFirst <= 0} {
        return
    }        
    
    set totaldistance 0
    set grow_factor [GiD_Set BoundaryLayer(GrowFactor)]
    switch [GiD_Set BoundaryLayer(GrowLaw)] {
        0 {
            #geometric
            set totaldistance 0
            set ylines ""
            if { $grow_factor == 1 } {
                for { set i 0 } { $i < $NumberOfLayers } { incr i } {
                    set totaldistance [expr $DistanceFirst*($i+1)*$grow_factor]
                    lappend ylines $totaldistance
                }
            } else {
                for { set i 0 } { $i < $NumberOfLayers } { incr i } {
                    set totaldistance [expr $DistanceFirst*(1-pow($grow_factor,$i+1))/(1-$grow_factor)]
                    lappend ylines $totaldistance
                }
            }
        }
        1 {
            #exponential
            set totaldistance 0
            set ylines ""
            for { set i 0 } { $i < $NumberOfLayers } { incr i } {
                set totaldistance [expr exp(log($DistanceFirst)+($grow_factor*$i))]
                lappend ylines $totaldistance
            }
        }
        2 {
            #geometric_mod
            set totaldistance 0
            set ylines ""
            for { set i 0 } { $i < $NumberOfLayers } { incr i } {
                set totaldistance [expr $DistanceFirst*(1+$i*(1+$grow_factor*(1+$grow_factor)*$i))]
                lappend ylines $totaldistance
            }
        }
        default {
            error "unexpected boundary layer grow law"            
        }
    }
    
    set c_width [$c cget -width]
    set c_height [$c cget -height]
    set border 30

    set x_min $border
    set x_max [expr $c_width-$border]
    set y_min $border
    set y_max [expr $c_height-$border]
    set y_size [expr $y_max-$y_min]

    $c delete all
    
    set d 25
    set x_min_block [expr $x_min+$d]
    set x_max_block [expr $x_max-$d]
    
    set color #90ffff
    set value_prev $y_max
    foreach value $ylines {
        set value [expr round($y_max-$value*($y_size/$totaldistance))]
        if { $value>=$y_max } {
            set value [expr $y_max-1]
        }
        $c create rectangle $x_min_block $value_prev $x_max_block $value -fill $color -outline #000000
        set value_prev $value
        if { $color == "#90ffff" } {
            set color #90c8c8
        } else {
            set color #90ffff
        }
    }

    
    $c create line $x_min $y_max $x_min $y_min -arrow both -tags totaldist    
    $c create text [expr $x_min-10] [expr ($y_min+$y_max)/2.0] -text [format "%.3g" $totaldistance] -tags totaldist -angle 90
    set value [expr round($y_max-$DistanceFirst*($y_size/$totaldistance))]
    if { $value>=$y_max } { 
        set value [expr $y_max-1]
    }
    $c create line $x_max $y_max $x_max $value -arrow both    
    $c create text [expr $x_max-10] [expr ($y_max+$value)/2] -text [format "%.3g" $BoundaryLayer::DistanceFirst] -angle 90
    
}

proc BoundaryLayer::OnResize { c } {
    set c_parent [winfo parent $c]
    set winfo_width [winfo width $c_parent]
    set winfo_height [winfo height $c_parent]
    $c configure -width $winfo_width -height $winfo_height        
    BoundaryLayer::ActualizeDraw $c    
}


proc BoundaryLayer::OpenWindow {  } {       
    set w .gid.wBoundaryLayer
    InitWindow2 $w -title [_ "Boundary Layer Mesh"] \
        -geometryvariable PreBoundaryLayerWindowGeom \
        -initcommand BoundaryLayerMesh

    if { ![winfo exists $w] } return ;# windows disabled || UseMoreWindows == 0
    wm minsize $w 180 200

    ttk::frame $w.f0
    
    ttk::frame $w.f0.fc
    set c [canvas $w.f0.fc.dib -width 300 -height 400]

    ttk::label $w.f0.la1 -text [_ "Number of layers"]:
    UpDownEntry $w.f0.b1 -textvariable ::BoundaryLayer::NumberOfLayers \
        -entry-options "-width 3" \
        -up-command "incr ::BoundaryLayer::NumberOfLayers 1"\
        -down-command {
        if { $::BoundaryLayer::NumberOfLayers != 1 } {
            incr ::BoundaryLayer::NumberOfLayers -1
        
        }
    }
    
    ttk::label $w.f0.la2 -text [_ "First layer height"]:
    ttk::entry $w.f0.e -textvariable ::BoundaryLayer::DistanceFirst -width 18

    ttk::label $w.f0.law -text [_ "Grow Law"] -font BigFont
    switch [GiD_Set BoundaryLayer(GrowLaw)] {
        0 { 
            set TextToDispay [_ "Geometric"]
        }
        1 { 
            set TextToDispay [_ "Exponential"]
        }
        2 { 
            set TextToDispay [_ "Geometric Mod"]
        }
        default { 
            set TextToDispay [_ "Not Known" ]
        }
    }
    ttk::label $w.f0.lawsel -text $TextToDispay -font BigFont
    ttk::label $w.f0.fac -text [_ "Grow Factor"]

    ttk::label $w.f0.facsel -text [format "%.2f" [GiD_Set BoundaryLayer(GrowFactor)]]
    GidHelp "$w.f0.law $w.f0.lawsel $w.f0.fac $w.f0.facsel" \
        [_ "Stretching function and grow factor used.\n To change this values go to Utilities->Preferences->Meshing"]

    ttk::frame $w.frmButtons -style BottomFrame.TFrame
    ttk::button $w.frmButtons.btnApply -text [_ "Assign"] -command [list BoundaryLayer::AssignBoundaryLayerCmd $w ] -underline 0 -style BottomFrame.TButton
    ttk::button $w.frmButtons.btnclose -text [_ "Close"] -command [list destroy $w] -underline 0 -style BottomFrame.TButton

    grid $w.f0.law $w.f0.lawsel -sticky ew
    grid $w.f0.fac $w.f0.facsel -sticky ew   
    grid $w.f0.fc -columnspan 2 -sticky nsew
    grid $w.f0.la2 $w.f0.e -sticky ew   
    grid $w.f0.la1 $w.f0.b1 -sticky ew
        
    grid rowconfigure $w.f0.fc 0 -weight 1
    grid columnconfigure $w.f0.fc 0 -weight 1
    grid $c ;#-sticky nsew

    grid rowconfigure $w.f0 2 -weight 1
    grid columnconfigure $w.f0 1 -weight 1

    grid $w.f0 -sticky news -padx 3 -pady 3
    grid rowconfigure $w 0 -weight 1
    grid columnconfigure $w 0 -weight 1

    grid $w.frmButtons -sticky ews -columnspan 7
    grid anchor $w.frmButtons center
    grid $w.frmButtons.btnApply $w.frmButtons.btnclose -padx 5 -pady 6

    bind $w <Alt-c> [list $w.frmButtons.btnclose invoke]
    bind $w <Escape> [list $w.frmButtons.btnclose invoke]
    bind $w <Return> [list $w.frmButtons.btnApply invoke]

    bind $w <Destroy> [list +BoundaryLayer::DestroyBoundLayer %W $w]
    bind $w.f0.fc <Configure> [list BoundaryLayer::OnResize $c]
    
    trace add variable ::BoundaryLayer::DistanceFirst write [list BoundaryLayer::OnChangeParameters]
    trace add variable ::BoundaryLayer::NumberOfLayers write [list BoundaryLayer::OnChangeParameters]
    after idle [list BoundaryLayer::ActualizeDraw $c]
}

proc BoundaryLayer::OnChangeParameters { name1 name2 op } {
    set w .gid.wBoundaryLayer
    set c $w.f0.fc.dib
    BoundaryLayer::ActualizeDraw $c
}

proc BoundaryLayer::AssignBoundaryLayerCmd { w } {
    variable NumberOfLayers
    variable DistanceFirst
    GiD_Process Assign $::BoundaryLayer::NumberOfLayers $::BoundaryLayer::DistanceFirst 
    FinishButton $w $w.frmButtons [_ "Press 'Finish' to end selection"] "" disableall [GiD_Set SmallWinSelecting]
}

proc BoundaryLayer::DestroyBoundLayer { W w } {
    if { $W != $w } return
    set c $w.f0.fc.dib
    trace remove variable ::BoundaryLayer::DistanceFirst write [list BoundaryLayer::ActualizeDraw $c]
    trace remove variable ::BoundaryLayer::NumberOfLayers write [list BoundaryLayer::ActualizeDraw $c]
    after idle [list GiD_Process Mescape]
}

proc BoundaryLayer::CloseWindow {  } {
    set w .gid.wBoundaryLayer
    if { ![winfo exists $w] } {
        return
    }       
    destroy $w
}



