#progress bar to show render progress, invoked from events: GiD_Event_BeforeRenderProgress GiD_Event_RenderProgress GiD_Event_AfterRenderProgress
#window to show meshing progress, invoked from GiD-events: GiD_Event_BeforeMeshProgress GiD_Event_MeshProgress GiD_Event_AfterMeshProgress GiD_Event_AfterMeshGeneration

namespace eval ProgressInRender {
}

proc ProgressInRender::Init { num_surfaces } {
    if { [GidUtils::AreWindowsDisabled] } {
        return 1
    }
    AdvanceBar 0 100 0 [_ "Rendering %d surfaces" $num_surfaces] [_ "Percentage"]
}

#return -cancel- ;#to set user stop
proc ProgressInRender::RenderProgress { total_percent n_nodes n_elems } {
    set res 0
    AdvanceBar $total_percent 100 $total_percent
    if { [info exists ::GidPriv(StopLoading)] && $::GidPriv(StopLoading) } {
        set res -cancel-
    }
    return $res
}

proc ProgressInRender::End { } {
    if { [GidUtils::AreWindowsDisabled] } {
        return 1
    }
    AdvanceBar 100 100 100
}


namespace eval ProgressInMeshing {
    variable user_stop
    variable progress_var
    variable label
    variable busy
    variable graph
    variable window_topmost
}

proc ProgressInMeshing::GetMeshTxt { } {
    set msg ""
    foreach element_type [GidUtils::GetElementTypes all] {
        set ne [GiD_Info Mesh NumElements $element_type]
        if { $ne } {
            append msg [_ "Num. of %s elements=%s" $element_type [ProgressInMeshing::AddThousandSeparator $ne]]\n
        }
    }
    set nn [GiD_Info Mesh NumNodes]
    append msg [_ "Num. of nodes=%s" [ProgressInMeshing::AddThousandSeparator $nn]]\n
    if { [GiD_Info Project Quadratic]>0 && ![GiD_Cartesian get iscartesian] } {
        append msg [_ "Quadratic mesh"]
    }
    return $msg
}

proc ProgressInMeshing::AfterMeshGeneration { fail } {
    if { [GidUtils::AreWindowsDisabled] } {
        return
    }
    set w .gid.wprogress
    if { [winfo exists $w] } {
        if { $fail } {
            ProgressInMeshing::End
        } else {
            if { [winfo exists $w.fnum.ltitle] } {
                #rare case: in batch meshing twice the first window could exists
                destroy $w.fnum.ltitle
                destroy $w.fnum.lmsg
                destroy $w.fnum.msg
            }
            set title [_ "Mesh generated."]
            grid forget $w.fbars
            grid forget $w.fnum.lnnodes $w.fnum.nnodes $w.fnum.lnelems $w.fnum.nelems $w.fnum.ltime $w.fnum.time
            ttk::label $w.fnum.ltitle -text $title -font BigFont
            set msg [ProgressInMeshing::GetMeshTxt]
            ttk::label $w.fnum.lmsg -image [gid_themes::GetImage info.png medium_icons]
            text $w.fnum.msg -relief groove -height [llength [split $msg \n]]

            set wt $w.fnum.msg
            bind $wt <FocusIn> [ list ::CreateNumericEntryFocusIn $wt]
            bind $wt <FocusOut> [ list ::CreateNumericEntryFocusOut $wt]

            $w.fnum.msg insert end $msg
            $w.fnum.msg configure -state disabled
            grid $w.fnum.ltitle -sticky ew -pady 3 -padx 4 -columnspan 2
            grid $w.fnum.lmsg $w.fnum.msg -sticky nsew -pady 3 -padx 4
            grid $w.frmButtons.btnmeshview
            $w.frmButtons.btnclose configure -text [_ "Close"] -command {ProgressInMeshing::End}
            # focus on view mesh button, as enter invokes this button
            focus $w.frmButtons.btnmeshview
            bind $w <Return> [list $w.frmButtons.btnmeshview invoke]
        }
    }
    return ok
    # do not return -cancel- avoid printing C messages like: reached limit of 1010 nodes
    # return -cancel- ;#to avoid that gid show its internal message
}

proc ProgressInMeshing::Init  { n0 n1 n2 n3 } {
    variable user_stop
    variable graph
    variable busy
    variable window_topmost
    if { [GidUtils::AreWindowsDisabled] } {
        unset -nocomplain busy
        return 1
    }

    # if { ( $::tcl_platform(platform) != "windows" ) || [ info exists ::env(VisualStudioVersion)] } {
    #     #the meshing window on top is annoying when debugging with Visual Studio
    #     # and on linux and macOS
    #     set window_topmost 0
    # } else {
    #     set window_topmost 1
    # }
    # do not make it topmost but transient
    set window_topmost 0
    package require gid_cross_platform
    package require Plotchart

    set user_stop 0
    set busy 0
    set graph(xs) [list 0]
    set graph(ys_nodes) [list 0]
    set graph(ys_memory) [list 0]
    set graph(t0) [clock seconds]
    set graph(memory0) [ProgressInMeshing::GetMemory]
    set graph(t_prev) $graph(t0)
    set graph(t_refresh) 1 ;#is updated dinamically
    unset -nocomplain graph(plot)

    if { ![info exists ::GidPriv(ProgressInMeshing,More)] } {
        set ::GidPriv(ProgressInMeshing,More) 1
    }
    ProgressInMeshing::CreateWindow $n0 $n1 $n2 $n3
    return 0
}

proc ProgressInMeshing::End { } {
    if { [GidUtils::IsTkDisabled] } {
        return 1
    }
    set w .gid.wprogress
    if { [winfo exists $w] } {
        destroy $w
    }
    return 0
}

proc ProgressInMeshing::CreateWindow  { n0 n1 n2 n3 } {
    variable user_stop
    variable progress_var
    variable label
    variable graph
    variable busy
    variable window_topmost

    set busy 1
    set w .gid.wprogress

    set avoid_InitWindow 1
    if { $avoid_InitWindow } {
        if { [winfo exists $w] } {
            destroy $w
        }
        toplevel $w
        wm withdraw $w
        if { $::tcl_platform(platform) == "windows" } {
            wm attributes $w -toolwindow 1
        }
        wm title $w [_ "Progress in meshing"]
        # later it will be made transient
    } else {
        # now has some problem with canvas resize
        InitWindow2 $w -title [_ "Progress in meshing"] \
            -geometryvariable PreProgressInMeshingWindowGeom -ontop
        # wm attributes $w -topmost 0
        # set parent [ winfo parent $w]
        # if { $parent == ""} {
        #     set parent .gid
        # }
        # wm transient $w $parent
    }

    ttk::frame $w.fbars
    label $w.fbars.l1 -text [_ "Percentage done"]:
    grid $w.fbars.l1 -sticky w -columnspan 2
    set n_sum [expr {$n0+$n1+$n2+$n3}]

    set test 1
    if { $test == 0 } {
        set ns [list $n0 $n1 $n2 $n3 $n_sum]
        set types [list other line surface volume total]
        set texts [list [_ "Other"] [_ "Lines"] [_ "Surfaces"] [_ "Volumes"] [_ "Total"]]
    } elseif { $test == 1 } {
        set ns [list $n0 $n2 $n3]
        set types [list other surface volume]
        set texts [list [_ "Other"] [_ "Surfaces"] [_ "Volumes"]]
    } else {
        set ns [list $n_sum]
        set types [list total]
        set texts [list [_ "Total"]]
    }
    foreach n $ns type $types text $texts {
        if { $n } {
            label $w.fbars.l$type -text ${text}:
            set progress_var($type) $w.fbars.p$type
            ttk::progressbar $progress_var($type)
            grid $w.fbars.l$type $w.fbars.p$type -sticky w -padx 4
            grid configure $w.fbars.p$type -sticky ew
        } else {
            unset -nocomplain progress_var($type)
        }
    }

    ttk::frame $w.fnum
    ttk::label $w.fnum.lnnodes -text [_ "Number of nodes"]:
    set label(n_nodes) [ttk::label $w.fnum.nnodes -text 0 -relief groove -anchor center -width 45] ;#set width 45 to force window size
    ttk::label $w.fnum.lnelems -text [_ "Number of elements"]:
    set label(n_elems) [ttk::label $w.fnum.nelems -text 0 -relief groove -anchor center]
    ttk::label $w.fnum.ltime -text [_ "Elapsed seconds"]:
    set label(time) [ttk::label $w.fnum.time -text 0 -relief groove -anchor center]
    grid $w.fnum.lnnodes $w.fnum.nnodes -sticky ew -pady 3 -padx 4
    grid $w.fnum.lnelems $w.fnum.nelems -sticky ew -pady 3 -padx 4
    grid $w.fnum.ltime $w.fnum.time -sticky ew -pady 3 -padx 4


    ttk::frame $w.fplot

    ttk::button $w.bmore -command [list ProgressInMeshing::SwapDetail $w]
    GidHelp $w.bmore [_ "Show more or less detail of the mesh generation process"]

    grid $w.fbars -sticky new
    grid columnconfigure $w.fbars 1 -weight 1
    grid $w.fnum -sticky new
    grid columnconfigure $w.fnum 1 -weight 1
    grid $w.fplot -sticky nsew
    grid rowconfigure $w.fplot 0 -weight 1
    grid columnconfigure $w.fplot 0 -weight 1
    grid $w.bmore -sticky w

    ttk::frame $w.frmButtons -style BottomFrame.TFrame
    ttk::button $w.frmButtons.btnmeshview -text [_ "View mesh"] -command {MeshView 1 ; ProgressInMeshing::End} -style BottomFrame.TButton
    ttk::button $w.frmButtons.btnclose -text [_ "Stop"] -command {set ::ProgressInMeshing::user_stop -cancel-} \
        -style BottomFrame.TButton

    grid $w.frmButtons.btnmeshview $w.frmButtons.btnclose -padx 5 -pady 6
    grid remove $w.frmButtons.btnmeshview

    grid $w.frmButtons -sticky ews
    grid anchor $w.frmButtons center

    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 2 -weight 1

    # focus on btnclose button, as enter invokes this button to stop while meshing
    focus $w.frmButtons.btnclose
    if { $avoid_InitWindow } {
        bind $w <Alt-c> [list $w.frmButtons.btnclose invoke]
        bind $w <Escape> [list $w.frmButtons.btnclose invoke]
    }
    bind $w <Return> [list $w.frmButtons.btnclose invoke]
    bind $w <Up> [list ProgressInMeshing::HideDetail $w]
    bind $w <Down> [list ProgressInMeshing::ShowDetail $w]

    bind $w <Destroy> [list +ProgressInMeshing::DestroyWindow %W $w] ;# + to add to previous script
    bind $w <Configure> [list +ProgressInMeshing::Resize $w]

    # Make it topmost to avoid putting the window behind gid's.
    if { $window_topmost } {
        # the meshing window on top is annoying always because is above all windows, not only gid's
        # wm attributes $w -topmost 1
        # meshing window isn't mapped yet, change stackorder after wm deiconify
        # wm stackorder $w isabove .gid
        GidUtils::WindowAboveGid $w
        # raise $w
        # to not grab the focus, you don't want this, otherwise
        # while the user is writting or press enter on another window
        # it may press enter in this window and will cancel the mesh process....
        # grab $w
    }

    ProgressInMeshing::UpdateDetail $w

    if { $avoid_InitWindow } {
        update
        if { [winfo exists $w] } {
            set parent [winfo parent $w]
            set reqwidth [winfo reqwidth $w]
            set reqheight [winfo reqheight $w]
            set x [expr [winfo x $parent]+[winfo width $parent]/2-$reqwidth/2]
            set y [expr [winfo y $parent]+[winfo height $parent]/2-$reqheight/2]
            WmGidGeom $w +$x+$y
            wm deiconify $w
        }
    } else {
        #update
        #wm deiconify $w
        #after idle [list ProgressInMeshing::DoResize $w]
    }
    if { $window_topmost } {
        # the meshing window on top is annoying always because is above all windows, not only gid's
        # wm attributes $w -topmost 1
        # meshing window isn't mapped yet, change stackorder after wm deiconify
        GidUtils::WindowAboveGid $w
        # if { [winfo ismapped .gid] } {
        #     wm stackorder $w isabove .gid
        # }
    } elseif { $avoid_InitWindow} {
        if { [winfo exists $w] } {
             # make it transient
            set parent [winfo parent $w]
            if { $parent == "" } {
                set parent .gid
            }
            wm transient $w $parent
        }
    }

    set busy 0
}

proc ProgressInMeshing::CreateGraph { w } {
    variable graph

    #::Plotchart::plotstyle load default
    ::Plotchart::plotconfig xyplot margin bottom 40
    ::Plotchart::plotconfig xyplot margin top 40

    canvas $w.fplot.c
    $w.fplot.c configure -background [Plotchart::plotconfig xyplot background innercolor]
    set text_colour black
    if { [gid_themes::GetCurrentTheme] == "GiD_black" } {
        set text_colour white
    }
    #set graph(xscale) [list 0 60 10]
    set graph(xscale) [list 0 15 5]
    set graph(yscale) [list 0 10 2]
    set graph(plot) [::Plotchart::createXYPlot $w.fplot.c $graph(xscale) $graph(yscale)]
    ProgressInMeshing::ConfigureGraph $graph(plot) $w.fplot.c
    grid $w.fplot.c -sticky nsew
}

proc ProgressInMeshing::ConfigureGraph { g c } {
    variable graph
    if { ![ winfo exists $c]} {
        return
    }
    set text_colour black
    #set mem_colour #007f7f
    set mem_colour green
    set legend_bg [ $c cget -background]
    if { [gid_themes::GetCurrentTheme] == "GiD_black" } {
        set text_colour white
    }
    $g xtext [_ "Time (seconds)"]
    # $g ytext [concat [_ "Num nodes"] & [concat [_ "Memory"] " "  KB]]
    # $g dataconfig nodes -color red
    # $g dataconfig memory -color cyan
    # $g xticklines green
    # $g yticklines green
    $g dataconfig nodes -color red
    $g dataconfig memory -color $mem_colour
    $g xticklines grey70
    $g yticklines grey70
    #$g legendconfig -position bottom-right
    #$g legendconfig -background $legend_bg
    #$g legendconfig -border $text_colour
    #$g legend nodes [_ "Num nodes"]
    #$g legend memory [concat [_ "Memory"] " "  KB]

    # a little bit nasty to access internal variables of Plotchart config,
    # but there is no other way until we get Plotchart 2.2.0 where we can do something like this:
    # Plotchart::plotconfig xyplot textcolor $text_color
    # Plotchart adds a prefix in its objects with the type of graph, underscore, a number and the canvas name
    # following the ::Plotchart::GetCanvas which does
    # regsub {^[^_]+_%} $cmd "" w
    # return $w ( w = canvas name)
    set w_plotchart --
    if { [ regexp {^[^_]+_(\d+.*)$} $g dum w_plotchart] == 1} {
        set ::Plotchart::config($w_plotchart,title,textcolor) $text_colour
    }
    $g title [_ "Meshing evolution"] center
    # configure color of text and add vertical axis info:
    $c create text 100 10 -text [_ "Knodes"] -anchor e -justify right -fill red
    $c create text 100 24 -text [ concat [_ "Memory"] " " MB] -anchor e -justify right -fill $mem_colour
    $c itemconfigure legend_nodes -fill $text_colour
    $c itemconfigure legend_memory -fill $text_colour
}

proc ProgressInMeshing::ShowDetail { w } {
    set ::GidPriv(ProgressInMeshing,More) 1
    ProgressInMeshing::UpdateDetail $w
}

proc ProgressInMeshing::HideDetail { w } {
    set ::GidPriv(ProgressInMeshing,More) 0
    ProgressInMeshing::UpdateDetail $w
}

proc ProgressInMeshing::SwapDetail { w } {
    set ::GidPriv(ProgressInMeshing,More) [expr !$::GidPriv(ProgressInMeshing,More)]
    ProgressInMeshing::UpdateDetail $w
}

proc ProgressInMeshing::UpdateDetail { w } {
    if { $::GidPriv(ProgressInMeshing,More) } {
        if { ![winfo exists $w.fplot.c] } {
            ProgressInMeshing::CreateGraph $w
        }
        grid $w.fplot
        $w.bmore configure -image [gid_themes::GetImage ArrowUp.png small_icons]
        GidHelp $w.bmore [_ "Less detail"]
    } else {
        grid remove $w.fplot
        $w.bmore configure -image [gid_themes::GetImage ArrowDown.png small_icons]
        GidHelp $w.bmore [_ "More detail"]
    }
}

proc ProgressInMeshing::SetProgressBarValue { pb value } {
    $pb configure -value $value
}

proc ProgressInMeshing::AddGraphPoint { t_relative n_nodes memory } {
    variable graph
    lappend graph(xs) $t_relative
    lappend graph(ys_nodes) $n_nodes
    lappend graph(ys_memory) $memory
    if { $::GidPriv(ProgressInMeshing,More) && [llength $graph(xs)] >= 2 } {
        $graph(plot) deletedata
        lassign $graph(xscale) min max increment
        if { [lindex $graph(xs) end] > $max } {
            #set graph(t_refresh) [expr {$graph(t_refresh)*2}]
            set dynamic_t_refresh [expr {$graph(t_refresh)*1.5}]
            if {$dynamic_t_refresh>5} {
                set dynamic_t_refresh 5
            }
            set graph(t_refresh) $dynamic_t_refresh
            set max [expr {$max*2}]
            set increment [expr {$increment*2}]
            #incr min
            #incr max
            set graph(xscale) [list $min $max $increment]
            $graph(plot) xconfig -scale $graph(xscale)
            $graph(plot) rescale $graph(xscale) $graph(yscale)
        }
        lassign $graph(yscale) min max increment
        if { [lindex $graph(ys_nodes) end] > $max || [lindex $graph(ys_memory) end] > $max} {
            set max [expr {$max*2}]
            set increment [expr {$increment*2}]
            set graph(yscale) [list $min $max $increment]
            $graph(plot) yconfig -scale $graph(yscale)
            $graph(plot) rescale $graph(xscale) $graph(yscale)
        }
        $graph(plot) plotlist nodes $graph(xs) $graph(ys_nodes)
        $graph(plot) plotlist memory $graph(xs) $graph(ys_memory)
    }
}

proc ProgressInMeshing::Resize { w } {
    # To avoid redrawing the plot many times during resizing,
    # cancel the callback, until the last one is left.
    #
    variable _resizing
    if { [info exists _resizing] } {
        after cancel $_resizing
    }
    set _resizing [after 50 [list ProgressInMeshing::DoResize $w]]
}

proc ProgressInMeshing::DoResize { w } {
    variable _resizing
    variable graph
    if { $::GidPriv(ProgressInMeshing,More) } {
        #deletedata also does a $w.fplot.c delete data
        if { [catch { $graph(plot) deletedata } err] } {
            # WarnWinText "ProgressInMeshing::DoResize $err"
        }
        if { [winfo exists $w.fplot.c] } {
            $w.fplot.c delete all
            if { [catch { set graph(plot) [::Plotchart::createXYPlot $w.fplot.c $graph(xscale) $graph(yscale)] } err] } {
                #catch because Plotchat package 2.0.1 can raise error if canvas is deleted while evaluating Plotchart::createXYPlot
            } else {
                ProgressInMeshing::ConfigureGraph $graph(plot) $w.fplot.c
                if { [llength $graph(xs)] >= 2 } {
                    $graph(plot) plotlist nodes $graph(xs) $graph(ys_nodes)
                    $graph(plot) plotlist memory $graph(xs) $graph(ys_memory)
                }
            }
        }
    }
    unset -nocomplain _resizing
}

proc ProgressInMeshing::GetMemory { } {
    return [gid_cross_platform::get_process_info_memory [pid]]
}

proc ProgressInMeshing::MeshProgress { total_percent partial_percents_0 partial_percents_1 \
    partial_percents_2 partial_percents_3 n_nodes n_elems } {
    variable user_stop
    variable progress_var
    variable label
    variable busy
    variable graph
    variable window_topmost

    if { ![info exists busy] || $busy } {
        return 0
    }
    set busy 1
    set w .gid.wprogress
    if { [winfo exists $w] } {
        #set advance_bar_percent $total_percent ;#update advance bar
        foreach type {other line surface volume total} value [list $partial_percents_0 $partial_percents_1 $partial_percents_2 $partial_percents_3 $total_percent] {
            if { [info exists progress_var($type)] && [winfo exists $progress_var($type)] } {
                ProgressInMeshing::SetProgressBarValue $progress_var($type) $value
            }
        }
        $label(n_nodes) configure -text [ProgressInMeshing::AddThousandSeparator $n_nodes]
        $label(n_elems) configure -text [ProgressInMeshing::AddThousandSeparator $n_elems]
        # update labels
        update idletasks
        set t [clock seconds]
        if { [expr {$t-$graph(t_prev)}] >= $graph(t_refresh) } {
            set t_relative [expr {$t-$graph(t0)}]
            $label(time) configure -text $t_relative
            set memory_relative [expr {([ProgressInMeshing::GetMemory]-$graph(memory0))/1000000}]
            set knodes [expr $n_nodes/1000]
            #            ProgressInMeshing::AddGraphPoint $t_relative $n_nodes $memory_relative
            ProgressInMeshing::AddGraphPoint $t_relative $knodes $memory_relative
            set graph(t_prev) $t
        }

        update

        if { [winfo exists $w] && [wm state .gid]== "normal" } {
            #trick to show the advance bar on the top of the stack order
            if { $window_topmost } {
                # the meshing window on top is annoying always because is above all windows, not only gid's
                # wm attributes $w -topmost 1
                # meshing window isn't mapped yet, change stackorder after wm deiconify
                # wm stackorder $w isabove .gid
                # raise $w
                GidUtils::WindowAboveGid $w
                # to not grab the focus, you don't want this, otherwise
                # while the user is writting or press enter on another window
                # it may press enter in this window and will cancel the mesh process....
                # grab $w
            }
        }
    }
    set busy 0
    return $user_stop
}

proc ProgressInMeshing::DestroyWindow { W w } {
    variable user_stop
    if { $W == $w } {
        #unset -nocomplain user_stop
    }
}

# as in GidUtils::AddThousandsSeparator
# in english thousandSeparator is ','
proc ProgressInMeshing::AddThousandSeparator {num {sep ,}} {
    while {[regsub {^([-+]?\d+)(\d\d\d)} $num "\\1$sep\\2" num]} {}
    return $num
}
