
# procs for remote GiD browser
namespace eval gid_filesystem {
    variable cache_info_geo_file
}

if { ![info exists ::gid_start_mode] } {
    set ::gid_start_mode normal
}

if { $::gid_start_mode == "client" } {
    proc gid_filesystem::glob { args } {
        return [gid_server_eval [list gid_filesystem::glob {*}$args]]
    }

    proc gid_filesystem::cd { dir } {
        return [gid_server_eval [list gid_filesystem::cd $dir]]
    }

    proc gid_filesystem::pwd { } {
        return [gid_server_eval [list gid_filesystem::pwd]]
    }

    #do local options locally to not be so expensive
    proc gid_filesystem::file { option args } {
        if { [lsearch -dictionary -sorted [list extension join normalize rootname split tail] $option] !=  -1 } {
            set res [::file $option {*}$args]
        } else {
            set res [gid_server_eval [list gid_filesystem::file $option {*}$args]]
        }
        return $res
    }

    proc gid_filesystem::file_writable { dir } {
        return [gid_server_eval [list gid_filesystem::file_writable $dir]]
    }

    proc gid_filesystem::unzip { filename_zip } {
        return [gid_server_eval [list gid_filesystem::unzip $filename_zip]]
    }

    proc gid_filesystem::zip { filename } {
        return [gid_server_eval [list gid_filesystem::zip $filename]]
    }

    proc gid_filesystem::recycle_file { filename } {
        return [gid_server_eval [list gid_filesystem::recycle_file $filename]]
    }

    proc gid_filesystem::is_link { filename } {
        return [gid_server_eval [list gid_filesystem::is_link $filename]]
    }

    proc gid_filesystem::read_link { filename show_ui } {
        return [gid_server_eval [list gid_filesystem::read_link $filename $show_ui]]
    }

    proc gid_filesystem::read_info_geo_file { filename } {
        return [gid_server_eval [list gid_filesystem::read_info_geo_file $filename]]
    }

    proc gid_filesystem::is_gid_project { dir } {
        return [gid_server_eval [list gid_filesystem::is_gid_project $dir]]
    }

    proc gid_filesystem::read_file { filename {encoding ""} {binary 0} } {
        return [gid_server_eval [list gid_filesystem::read_file $filename $encoding $binary]]
    }

    proc gid_filesystem::is_gid_post { filename } {
        return [gid_server_eval [list gid_filesystem::is_gid_post $filename]]
    }

    proc gid_filesystem::read_gid_post { filename } {
        return [gid_server_eval [list gid_filesystem::read_gid_post $filename]]
    }

    proc gid_filesystem::get_folder_gid { } {
        return [gid_server_eval [list gid_filesystem::get_folder_gid]]
    }

    proc gid_filesystem::get_folder_standard { category } {
        return [gid_server_eval [list gid_filesystem::get_folder_standard $category]]
    }

    proc gid_filesystem::get_folder_user_settings { category } {
        return [gid_server_eval [list gid_filesystem::get_folder_user_settings $category]]
    }

    proc gid_filesystem::get_folder_nextcloud { category } {
        return [gid_server_eval [list gid_filesystem::get_folder_nextcloud $category]]
    }

    proc gid_filesystem::get_folder_appstream { category } {
        return [gid_server_eval [list gid_filesystem::get_folder_appstream $category]]
    }

    proc gid_filesystem::get_folder_special { name } {
        return [gid_server_eval [list gid_filesystem::get_folder_special $name]]
    }

} else {
    #"server" or "normal" (no client-server)
    proc gid_filesystem::glob { args } {
        return [::glob {*}$args]
    }

    proc gid_filesystem::cd { dir } {
        return [::cd $dir]
    }

    proc gid_filesystem::pwd { } {
        return [::pwd]
    }

    proc gid_filesystem::file { option args } {
        return [::file $option {*}$args]
    }

    # yes, it is twice similar, can use 
    # gid_filesystem::file writable $dir  -> file writable 
    # or
    # gid_filesystem::file_writable $dir  -> gid_cross_platform::file_writable
    # the second handle also the special case of a nextcloud disk unit that always return 0 in Windows at least

    proc gid_filesystem::file_writable { dir } {
        package require gid_cross_platform
        return [gid_cross_platform::file_writable $dir]
    }

    proc gid_filesystem::unzip { filename_zip } {
        package require gid_cross_platform
        set out_dir [file dirname $filename_zip] ;#extract in same folder as file
        set run_as_administrator 0
        return [gid_cross_platform::gid_unzip $filename_zip $out_dir $run_as_administrator]
    }

    proc gid_filesystem::zip { filename } {
        package require gid_cross_platform
        set filename_zip [file rootname $filename].zip
        set run_as_administrator 0
        return [gid_cross_platform::gid_zip $filename_zip $filename $run_as_administrator]
    }

    proc gid_filesystem::recycle_file { filename } {
        #return [trash $filename]
        package require gid_cross_platform
        return [gid_cross_platform::recycle_file $filename]
    }

    proc gid_filesystem::is_link { filename } {
        return [GidUtils::IsLink $filename]
    }

    proc gid_filesystem::read_link { filename show_ui } {
        return [GidUtils::ReadLink $filename $show_ui]
    }

    proc gid_filesystem::read_info_geo_file { filename_geo } {
        variable cache_info_geo_file
        set result ""
        set is_cloud_path [GidDataManager::isCloudPath $filename_geo]
        if { $is_cloud_path } {
            if { [info exists cache_info_geo_file($filename_geo)] } {
                set result $cache_info_geo_file($filename_geo)
            } else {
                set fgeo_full_name_nextcloud [GidDataManager::TrimLeftCloudPath $filename_geo]
                if { [catch { set first_part [GidDataManager::httpDownloadDataRange $fgeo_full_name_nextcloud bytes=0-1023] } msg] } {
                    set result ""
                } else {
                    set result [GidUtils::GetInfoGeoFileData $first_part]
                }
                set cache_info_geo_file($filename_geo) $result
            }
        } else {
            #too expensive to be done in cloud case, require download the full .geo file and can be big
            set result [GiD_GetInfoGeoFile $filename_geo]
        }
        return $result
    }

    #check also some expected GiD model file existence inside the folder, but use the filesystem and could be slow (network)
    proc gid_filesystem::is_gid_project { dir } {
        if { [file extension $dir] != ".gid" } {
            return 0
        } else {
            set model [file rootname [file tail $dir]]
            foreach ext {.prj .geo .msh} {
                if {[file exists [file join $dir ${model}${ext}]]} {
                    return 1
                }
            }
            return 0
        }
    }

    proc gid_filesystem::read_file { filename {encoding ""} {binary 0} } {
        return [GidUtils::ReadFile $filename $encoding $binary]
    }

    proc gid_filesystem::read_file_last_line { filename {encoding ""} {binary 0} } {
        return [GidUtils::ReadFileLastLine $filename $encoding $binary]
    }

    proc gid_filesystem::is_gid_post { filename } {
        return [GiDPost::IsGiDPost $filename]
    }

    proc gid_filesystem::read_gid_post { filename } {
        return [GiDPost::ReadPost $filename]
    }

    proc gid_filesystem::get_folder_gid { } {
        return $::GIDDEFAULT
    }

    #category: problemtypes scripts plugins Examples info themes Fonts resources
    proc gid_filesystem::get_folder_standard { category } {
        set lst_categories_in_resources [ list Fonts themes audio textures]
        set in_resources 0
        foreach item $lst_categories_in_resources {
            if { $category == $item} {
                set in_resources 1
            }
        }
        if { $in_resources} {
            return [file join $::GIDDEFAULT resources $category]
        } else {
            return [file join $::GIDDEFAULT $category]
        }
    }

    #category: problemtypes
    proc gid_filesystem::get_folder_user_settings { category } {
        return [file join [file dirname [GiD_GetUserSettingsFilename -ignore_alternative_configuration_file]] $category]
    }

    #category: problemtypes
    proc gid_filesystem::get_folder_nextcloud { category } {
        set folder ""
        set nextcloud_unit [GidDataManager::getConnectedPath]
        if { $nextcloud_unit != "" } {
            if { [string index $nextcloud_unit 1] == ":" } {
                append nextcloud_unit /
            }
            set folder [file join $nextcloud_unit gid [GiD_Info GiDVersion] $category]
        }
        return $folder
    }

    #category: problemtypes
    proc gid_filesystem::get_folder_appstream { category } {
        set folder ""
        set folder [file join {D:/PhotonUser/My files/Home Folder} gid [GiD_Info GiDVersion] $category]
        if { ![file exists $folder] } {
            set folder ""
        }
        return $folder
    }

    # specials folder in unix and macOs
    # name: home downloads documents desktop publicshare
    # pecials folder in windows
    # name: downloads documents desktop
    proc gid_filesystem::get_folder_special { name } {
        package require gid_cross_platform
        return [gid_cross_platform::get_folder_special $name]
    }

}
