
package require math::linearalgebra
package require math::interpolate

package provide gid_draw_opengl 1.1

namespace eval gid_draw_opengl {
    variable _opengl_draw_list_by_key ; # auxiliary array to store opengl draw list and allow re-use
    variable _opengl_draw_lists_to_draw {} ; # list to redraw the opengl draw list in this container
    variable rgb
    set rgb(red) {1.0 0.0 0.0}
    set rgb(green) {0.0 1.0 0.0}
    set rgb(blue) {0.0 0.0 1.0}
    set rgb(black) {0.0 0.0 0.0}
    set rgb(white) {1.0 1.0 1.0}
    set rgb(magenta) {1.0 0.0 1.0}
    set rgb(orange) {1.0 0.65 0.0}
    variable scale_draw_loads 1.0 ;#factor to scale visual representation
    variable texture_id_by_key ;#cache textures by name
}

# to draw the opengl draw list added to this container
proc gid_draw_opengl::draw_lists_draw { } {
    variable _opengl_draw_lists_to_draw
    foreach opengl_list $_opengl_draw_lists_to_draw {
        GiD_OpenGL draw -call $opengl_list
    }
}

proc gid_draw_opengl::draw_lists_append { opengl_lists } {
    variable _opengl_draw_lists_to_draw
    lappend _opengl_draw_lists_to_draw {*}$opengl_lists
}

proc gid_draw_opengl::draw_list_delete { opengl_lists } {
    variable _opengl_draw_lists_to_draw
    foreach opengl_list $opengl_lists {
        set idx [lsearch $_opengl_draw_lists_to_draw $opengl_list]
        if { $idx != -1 } {
            gid_draw_opengl::delete_opengl_list_drawing $opengl_list
            set _opengl_draw_lists_to_draw [lreplace $_opengl_draw_lists_to_draw $idx $idx]
        }
    }
}

proc gid_draw_opengl::draw_list_delete_all { } {
    variable _opengl_draw_lists_to_draw
    foreach opengl_list $_opengl_draw_lists_to_draw {
        gid_draw_opengl::delete_opengl_list_drawing $opengl_list
    }
    set _opengl_draw_lists_to_draw [list ]
}

#image must have width and height power or 2 (it seems that now this is not necessary)
proc gid_draw_opengl::create_texture_from_image { img } {
    set texture_ids [GiD_OpenGL draw -gentextures 1]
    set texture_id [lindex $texture_ids 0]
    GiD_OpenGL draw -bindtexture texture2d $texture_id
    GiD_OpenGL draw -texparameter texture2d texturewraps repeat
    GiD_OpenGL draw -texparameter texture2d texturewrapt repeat
    GiD_OpenGL draw -texparameter texture2d texturemagfilter nearest
    GiD_OpenGL draw -texparameter texture2d textureminfilter nearest
    GiD_OpenGL draw -teximage2d 0 0 $img
    return $texture_id
}

proc gid_draw_opengl::create_texture { filename } {
    set img [image create photo -file $filename]
    return [gid_draw_opengl::create_texture_from_image $img]
}

proc gid_draw_opengl::get_texture_id { key } {
    variable texture_id_by_key
    set texture_id ""
    if { [info exists texture_id_by_key($key)] } {
        set texture_id $texture_id_by_key($key)
    }
    return $texture_id
}

proc gid_draw_opengl::set_texture_id { key texture_id } {
    variable texture_id_by_key
    if { [info exists texture_id_by_key($key)] } {
        gid_draw_opengl::delete_texture $key
    }
    set texture_id_by_key($key) $texture_id
    return
}

proc gid_draw_opengl::delete_texture { key } {
    variable texture_id_by_key
    if { [info exists texture_id_by_key($key)] } {
        GiD_OpenGL draw -deletetextures [list $texture_id_by_key($key) 1]
        unset texture_id_by_key($key)
    }
}

proc gid_draw_opengl::delete_all_textures { } {
    variable texture_id_by_key
    foreach key [array names texture_id_by_key] {
        GiD_OpenGL draw -deletetextures [list $texture_id_by_key($key) 1]
    }
    array unset texture_id_by_key
}

#expected objarrays
#opengl_element_type: triangles, quads

proc gid_draw_opengl::draw_mesh_textures_by_texture_id { opengl_element_type vertices element_vertex_indices texcoords element_textures_indices texture_id } {
    GiD_OpenGL draw -enable texture2d
    GiD_OpenGL draw -texenv textureenv textureenvmode decal
    GiD_OpenGL draw -bindtexture texture2d $texture_id
    GiD_OpenGL draw -begin $opengl_element_type
    set n [objarray length $element_vertex_indices]
    for { set i 0 } { $i< $n } { incr i } {
        set vertex_index [objarray get $element_vertex_indices $i]
        set vertex [objarray range $vertices [expr $vertex_index*3] [expr $vertex_index*3+2]]
        set texture_index [objarray get $element_textures_indices $i]
        set texcoord [objarray range $texcoords [expr $texture_index*2] [expr $texture_index*2+1]]
        GiD_OpenGL draw -texcoord $texcoord -vertex $vertex
    }
    GiD_OpenGL draw -end
    GiD_OpenGL draw -flush
    GiD_OpenGL draw -disable texture2d
}

proc gid_draw_opengl::draw_mesh_textures { opengl_element_type vertices element_vertex_indices texcoords element_textures_indices filename_texture } {
    set texture_id [gid_draw_opengl::get_texture_id $filename_texture]
    if { $texture_id == "" } {
        set texture_id [gid_draw_opengl::create_texture $filename_texture]
        gid_draw_opengl::set_texture_id $filename_texture $texture_id
    }
    gid_draw_opengl::draw_mesh_textures_by_texture_id $opengl_element_type $vertices $element_vertex_indices $texcoords $element_textures_indices $texture_id
}

#expected objarrays
proc gid_draw_opengl::draw_mesh_colors_by_vertex { opengl_element_type vertices element_vertex_indices colors } {
    GiD_OpenGL draw -begin $opengl_element_type
    set n [objarray length $element_vertex_indices]
    for { set i 0 } { $i< $n } { incr i } {
        set vertex_index [objarray get $element_vertex_indices $i]
        set pos_ini [expr $vertex_index*3]
        set pos_end [expr $pos_ini+2]
        set vertex [objarray range $vertices $pos_ini $pos_end]
        set color [objarray range $colors $pos_ini $pos_end]
        GiD_OpenGL draw -color $color -vertex $vertex
    }
    GiD_OpenGL draw -end
    GiD_OpenGL draw -flush
    GiD_OpenGL draw -disable texture2d
}

#expected objarrays
proc gid_draw_opengl::draw_mesh_colors_by_elements_vertex { opengl_element_type vertices element_vertex_indices colors element_color_indices } {
    GiD_OpenGL draw -begin $opengl_element_type
    set n [objarray length $element_vertex_indices]
    for { set i 0 } { $i< $n } { incr i } {
        set vertex_index [objarray get $element_vertex_indices $i]
        set pos_ini [expr $vertex_index*3]
        set pos_end [expr $pos_ini+2]
        set vertex [objarray range $vertices $pos_ini $pos_end]
        set color_index [objarray get $element_color_indices $i]
        #4 components: r g b a
        set color [objarray range $colors [expr $color_index*4] [expr ($color_index+1)*4-1]]
        GiD_OpenGL draw -color $color -vertex $vertex
    }
    GiD_OpenGL draw -end
    GiD_OpenGL draw -flush
    GiD_OpenGL draw -disable texture2d
}

#transform_matrix could be "" for identity
proc gid_draw_opengl::create_opengl_list_drawing { drawing_procedure transform_matrix } {
    set list_id [GiD_OpenGL draw -genlists 1]
    GiD_OpenGL draw -newlist $list_id compile
    if { [llength $transform_matrix] } {
        GiD_OpenGL draw -pushmatrix -multmatrix $transform_matrix
    }
    {*}$drawing_procedure
    if { [llength $transform_matrix] } {
        GiD_OpenGL draw -popmatrix
    }
    GiD_OpenGL draw -endlist
    return $list_id
}

proc gid_draw_opengl::delete_opengl_list_drawing { list_id } {
    GiD_OpenGL draw -deletelists [list $list_id 1]
}

proc gid_draw_opengl::create_opengl_list_from_opengl_list { opengl_list transform_matrix } {
    set drawing_procedure [list GiD_OpenGL draw -call $opengl_list]
    return [gid_draw_opengl::create_opengl_list_drawing $drawing_procedure $transform_matrix]
}

#e.g. to draw a condition as a 3D shape defined in a GiD ASCII mesh file
#transform_matrix could be "" for the identity or a list with 16 values (representing a 4x4 opengl_matrix)
proc gid_draw_opengl::create_opengl_list_from_file_mesh { full_filename color_lines color_surfaces transform_matrix } {
    package require customLib
    set drawing_procedure [list gid_groups_conds::import_gid_mesh_as_openGL $full_filename $color_lines $color_surfaces]
    return [gid_draw_opengl::create_opengl_list_drawing  $drawing_procedure $transform_matrix]
}

proc gid_draw_opengl::create_opengl_list_from_file_mesh_oriented_z_direction { full_filename color_lines color_surfaces z_direction } {
    package require customLib
    lassign [gid_groups_conds::calc_xy_shell $z_direction] x_axis y_axis z_axis
    set transform_matrix [concat $x_axis 0 $y_axis 0 $z_axis 0 0 0 0 1]
    return [gid_draw_opengl::create_opengl_list_from_file_mesh $full_filename $color_lines $color_surfaces $transform_matrix]
}

#can return multiple lists, one for triangle and other for quadrilateral
proc gid_draw_opengl::create_opengl_lists_mesh_textured_from_background_image { } {
    set opengl_lists_local [list]
    set filename_tmp ""
    set filename_texture [GiD_BackgroundImage get filename]
    if { $filename_texture!="" } {
        if { ![file exists $filename_texture] } {
            set filename_tmp [gid_cross_platform::get_unused_tmp_filename gid_background_image .png]
            GiD_Project db save background_image $filename_tmp
            set filename_texture $filename_tmp
        }
    }
    # maybe must replace .bgi by the true format extension .png , .tiff,...
    # filename image must have width and height power or 2
    # otherwise must create another auxiliary resampled image with this restriction
    if { $filename_texture != "" } {
        lassign [GiD_BackgroundImage get location] ox oy ix iy jx jy
        #assume that image is aligned to XY axis, not rotated
        set x_min $ox
        set x_max $ix
        set y_min $oy
        set y_max $jy
        set d_x [expr $x_max-$x_min]
        set d_y [expr $y_max-$y_min]
        if { $d_x>0.0 && $d_y>0.0 } {
            set pre_post [string tolower [GetCurrentPrePostMode]]
            lassign [GiD_Info mesh -$pre_post nodes -array] node_ids node_coordinates_by_component
            lassign $node_coordinates_by_component node_coordinates_x node_coordinates_y node_coordinates_z
            set texcoords_u [objarray clone $node_coordinates_x]
            objarray incr $texcoords_u [expr -1.0*$x_min]
            objarray scale $texcoords_u [expr 1.0/$d_x]
            set texcoords_v [objarray clone $node_coordinates_y]
            objarray incr $texcoords_v [expr -1.0*$y_min]
            objarray scale $texcoords_v [expr 1.0/$d_y]
            set texcoords [objarray concat $texcoords_u $texcoords_v]
            objarray interleave $texcoords 2
            # move z an offset to try to not hide the results
            set offset_visual_z -0.01
            objarray incr $node_coordinates_z $offset_visual_z
            set node_coordinates [objarray concat $node_coordinates_x $node_coordinates_y $node_coordinates_z]
            objarray interleave $node_coordinates 3
            foreach element_type {triangle quadrilateral} {
                if { [GiD_Info Mesh -pre numelements $element_type] } {
                    set opengl_element_type triangles
                    if { $element_type == "quadrilateral" } {
                        set opengl_element_type quads
                    } else {
                        set opengl_element_type triangles
                    }
                    lassign [lindex [GiD_Info mesh -pre elements $element_type -array2] 0] element_type_out element_ids element_connectivities
                    # to renumbe ids in case that has numeration jumps, and in any case to be numbered by 0 instead 1
                    objarray renumber $element_connectivities $node_ids
                    # element_textures_indices are here like element_connectivities
                    set element_textures_indices $element_connectivities
                    set drawing_procedure [list gid_draw_opengl::draw_mesh_textures $opengl_element_type $node_coordinates $element_connectivities $texcoords $element_textures_indices $filename_texture]
                    set transform_matrix ""
                    lappend opengl_lists_local [gid_draw_opengl::create_opengl_list_drawing $drawing_procedure $transform_matrix]
                }
            }
        }
    }
    if { $filename_tmp != "" } {
        file delete -force $filename_tmp
    }
    return $opengl_lists_local
}

proc gid_draw_opengl::create_opengl_lists_from_background_image_mesh_rectangle { raster_z img } {
    set opengl_lists_local [list]
    set texture_id [gid_draw_opengl::get_texture_id $img]
    if { $texture_id == "" } {
        set texture_id [gid_draw_opengl::create_texture_from_image $img]
        gid_draw_opengl::set_texture_id $img $texture_id
    }

    # resampled arbitrary number of nodes on each direction of the drawing structured mesh
    # not necessarily equal to num_columns num_rows


    lassign $raster_z num_columns num_rows xllcenter yllcenter xcellsize ycellsize nodata_value values

    set x_n 256
    set y_n 256
    # do not use a texture mesh with more points than the z_raster data
    if { $x_n>$num_columns } {
        set x_n $num_columns
    }
    if { $y_n>$num_rows } {
        set y_n $num_rows
    }

    set num_nodes [expr $x_n*$y_n]

    #xmin-xmax, ymin-ymax are the limits of the cells, but the values are on the center of cells
    #there are num_columns cells on x direction, and then num_columns-1 raster values for the cell centers,
    #and then the xmin and xmax of the raster nodes is xmin=xmin_cells+0.5*xcellsize xmax=xmax_cells-0.5*xcellsize
    #set xmin [expr {$xllcenter+0.5*$xcellsize}]
    #set ymin [expr {$yllcenter+0.5*$ycellsize}]
    #set xmax [expr {$xllcenter+$num_columns*$xcellsize-0.5*$xcellsize}]
    #set ymax [expr {$yllcenter+$num_rows*$ycellsize-0.5*$ycellsize}]
    #
    # use same size as image, without half cell to each side
    set xmin $xllcenter
    set ymin $yllcenter
    set xmax [expr {$xllcenter+$num_columns*$xcellsize}]
    set ymax [expr {$yllcenter+$num_rows*$ycellsize}]

    set xs [GidUtils::GetRange $xmin $xmax $x_n]
    set ys [GidUtils::GetRange $ymin $ymax $y_n]
    set node_coordinates [objarray new_structured_coordinates [list $xs $ys]] ;# xy0xy0xy0...
    # node_coordinates has z 0.0, interpolate its value from $raster_z
    objarray deinterleave $node_coordinates 3 ;# xxxxxyyyyyyzzzz
    set xys [objarray range $node_coordinates 0 [expr $num_nodes*2-1]] ;# xxxxxyyyyyy
    objarray interleave $xys 2 ;# xyxyxy...
    set zs_interpolated [GiD_Raster interpolate $raster_z [list nodes $xys]]
    objarray set $node_coordinates [expr $num_nodes*2] $zs_interpolated
    objarray interleave $node_coordinates 3 ;# xxxxxyyyyyyzzzz

    set element_connectivities [objarray new_structured_connectivities [list $x_n $y_n] 0]

    set xs_relative [GidUtils::GetRange 0.0 1.0 $x_n]
    set ys_relative [GidUtils::GetRange 0.0 1.0 $y_n]
    set texcoords [objarray new_structured_coordinates [list $xs_relative $ys_relative]] ;# xy0xy0xy0...
    objarray deinterleave $texcoords 3 ;# xxxxxyyyyyyzzzz
    set texcoords [objarray range $texcoords 0 [expr $num_nodes*2-1]] ;# xxxxxyyyyyy
    objarray interleave $texcoords 2 ;# xxxxxyyyyy

    # element_textures_indices are here like element_connectivities
    set element_textures_indices $element_connectivities

    set opengl_element_type quads
    set drawing_procedure [list gid_draw_opengl::draw_mesh_textures_by_texture_id $opengl_element_type $node_coordinates $element_connectivities $texcoords $element_textures_indices $texture_id]
    set transform_matrix ""
    lappend opengl_lists_local [gid_draw_opengl::create_opengl_list_drawing $drawing_procedure $transform_matrix]
    return $opengl_lists_local
}

proc gid_draw_opengl::set_scale_draw_loads { v } {
    variable scale_draw_loads
    set scale_draw_loads $v
}

#n-arrows with upper line size 1x1 arrows pointing to -y
proc gid_draw_opengl::draw_distributed_load { } {
    set xold 0.0
    foreach x [GidUtils::GetRange 0.0 1.0 4] {
        GiD_OpenGL draw -begin linestrip -vertex [list $x 0 0] -vertex [list $x 1 0] -vertex [list $xold 1 0] -end
        GiD_OpenGL draw -begin trianglefan -vertex [list $x 0 0]
        foreach angle [GidUtils::GetRange 0.0 [expr {2.0*$MathUtils::Pi}] 9] {
            GiD_OpenGL draw -vertex [list [expr {$x+0.05*cos($angle)}] 0.1 [expr {0.05*sin($angle)}]]
        }
        GiD_OpenGL draw -end
        set xold $x
    }
}

proc gid_draw_opengl::create_opengl_list_distributed_load { } {
    return [gid_draw_opengl::create_opengl_list_drawing gid_draw_opengl::draw_distributed_load ""]
}

#arrow size 1 pointing to +x (arrow cone height 0.3, radius 0.05)
proc gid_draw_opengl::draw_point_load { } {
    #GiD_OpenGL draw -linewidth 2.0
    GiD_OpenGL draw -begin lines -vertex [list 0 0 0] -vertex [list 1 0 0] -end
    GiD_OpenGL draw -begin trianglefan -vertex [list 1 0 0]
    foreach angle [GidUtils::GetRange 0.0 $MathUtils::2PI 9] {
        GiD_OpenGL draw -vertex [list 0.7 [expr {0.05*cos($angle)}] [expr {0.05*sin($angle)}]]
    }
    GiD_OpenGL draw -end
    #GiD_OpenGL draw -linewidth 1.0
}

proc gid_draw_opengl::create_opengl_list_point_load { } {
    return [gid_draw_opengl::create_opengl_list_drawing gid_draw_opengl::draw_point_load ""]
}

#arc and double-arrow size 1 pointing to +x (arrow cone height 0.1, radius 0.05)
proc gid_draw_opengl::draw_point_momentum { } {
    GiD_OpenGL draw -begin lines -vertex [list 0 0 0] -vertex [list 1 0 0] -end
    foreach x_cone_start {0.9 0.7} x_cone_end {1 0.8} {
        GiD_OpenGL draw -begin trianglefan -vertex [list $x_cone_end 0 0]
        foreach angle [GidUtils::GetRange 0.0 [expr {2.0*$MathUtils::Pi}] 9] {
            GiD_OpenGL draw -vertex [list $x_cone_start [expr {0.05*cos($angle)}] [expr {0.05*sin($angle)}]]
        }
        GiD_OpenGL draw -end
    }
    GiD_OpenGL draw -begin linestrip
    foreach angle [GidUtils::GetRange 0.0 [expr {1.5*$MathUtils::Pi}] 13] {
        GiD_OpenGL draw -vertex [list 0 [expr {0.5*cos($angle)}] [expr {0.5*sin($angle)}]]
    }
    GiD_OpenGL draw -end
    GiD_OpenGL draw -begin trianglefan -vertex [list 0 0 -0.5]
    foreach angle [GidUtils::GetRange 0.0 [expr {2.0*$MathUtils::Pi}] 9] {
        GiD_OpenGL draw -vertex [list [expr {0.05*sin($angle)}] -0.1 [expr {-0.5+0.05*cos($angle)}] ]
    }
    GiD_OpenGL draw -end
}

proc gid_draw_opengl::create_opengl_list_point_momentum { } {
    return [gid_draw_opengl::create_opengl_list_drawing gid_draw_opengl::draw_point_momentum ""]
}

#arrow size 1 pointing to +x (arrow cone height 0.1, radius 0.05)
proc gid_draw_opengl::draw_beam { } {
    GiD_OpenGL draw -begin quads -vertex [list 0 0 -0.5] -vertex [list 1 0 -0.5] -vertex [list 1 0 0.5] -vertex [list 0 0 0.5] -end
    GiD_OpenGL draw -begin quads -vertex [list 0 -0.5 -0.5] -vertex [list 0 0.5 -0.5] -vertex [list 1 0.5 -0.5] -vertex [list 1 -0.5 -0.5] -end
    GiD_OpenGL draw -begin quads -vertex [list 0 -0.5 0.5] -vertex [list 0 0.5 0.5] -vertex [list 1 0.5 0.5] -vertex [list 1 -0.5 0.5] -end
}

proc gid_draw_opengl::create_opengl_list_beam { } {
    return [gid_draw_opengl::create_opengl_list_drawing gid_draw_opengl::draw_beam ""]
}

#origin: list {x y z} of the coordinates of the point
#force:  list {fx fy fz}
proc gid_draw_opengl::draw_symbol_point_load  { origin force } {
    variable _opengl_draw_list_by_key
    variable rgb
    variable scale_draw_loads
    if { [::math::linearalgebra::norm_two $force] < $MathUtils::Epsilon } {
        return 1
    }
    if { ![info exists _opengl_draw_list_by_key(point_load)] } {
        set _opengl_draw_list_by_key(point_load) [gid_draw_opengl::create_opengl_list_point_load]
    }
    #matrix to orient the reference picture (arrow for origin to 1 in x direction)
    set matrix_x_axis [math::linearalgebra::unitLengthVector $force]
    lassign [MathUtils::CalculateLocalAxisFromXAxis $matrix_x_axis] matrix_y_axis matrix_z_axis
    set scale [expr {1.0e1*$scale_draw_loads*[math::linearalgebra::norm_two $force]}]
    set transform_matrix [concat [math::linearalgebra::scale_vect $scale $matrix_x_axis] 0 [math::linearalgebra::scale_vect $scale $matrix_y_axis] 0 [math::linearalgebra::scale_vect $scale $matrix_z_axis] 0 0 0 0 1]
    GiD_OpenGL draw -pushmatrix -translate $origin -multmatrix $transform_matrix -color $rgb(green) -call $_opengl_draw_list_by_key(point_load) -popmatrix
    return 0
}

#origin: list {x y z} of the coordinates of the point
#momentum:  list {mx my mz}
proc gid_draw_opengl::draw_symbol_point_momentum  { origin momentum } {
    variable _opengl_draw_list_by_key
    variable rgb
    variable scale_draw_loads
    if { ![info exists _opengl_draw_list_by_key(point_momentum)] } {
        set _opengl_draw_list_by_key(point_momentum) [gid_draw_opengl::create_opengl_list_point_momentum]
    }
    #matrix to orient the reference picture (arrow for origin to 1 in x direction)
    set matrix_x_axis [math::linearalgebra::unitLengthVector $momentum]
    lassign [MathUtils::CalculateLocalAxisFromXAxis $matrix_x_axis] matrix_y_axis matrix_z_axis
    set scale [expr {1.0e1*$scale_draw_loads*[math::linearalgebra::norm_two $momentum]}]
    set transform_matrix [concat [math::linearalgebra::scale_vect $scale $matrix_x_axis] 0 [math::linearalgebra::scale_vect $scale $matrix_y_axis] 0 [math::linearalgebra::scale_vect $scale $matrix_z_axis] 0 0 0 0 1]
    GiD_OpenGL draw -pushmatrix -translate $origin -multmatrix $transform_matrix -color $rgb(magenta) -call $_opengl_draw_list_by_key(point_momentum) -popmatrix
    return 0
}

#coordinates_system: global or local
#relative_position: value from 0.0 to 1.0
proc gid_draw_opengl::draw_symbol_line_interior_point_load { origin end force coordinates_system relative_position } {
    variable _opengl_draw_list_by_key
    variable rgb
    variable scale_draw_loads
    if { ![info exists _opengl_draw_list_by_key(point_load)] } {
        set _opengl_draw_list_by_key(point_load) [gid_draw_opengl::create_opengl_list_point_load]
    }
    set v12 [math::linearalgebra::sub_vect $end $origin]
    if { $coordinates_system == "local" } {
        #calculate force in global axes
        set x_axis [math::linearalgebra::unitLengthVector $v12]
        lassign [MathUtils::CalculateLocalAxisFromXAxis $x_axis] y_axis z_axis
        set matrix [math::linearalgebra::mkMatrix 3 3]
        math::linearalgebra::setcol matrix 0 $x_axis
        math::linearalgebra::setcol matrix 1 $y_axis
        math::linearalgebra::setcol matrix 2 $z_axis
        set f [math::linearalgebra::matmul $matrix $force]
    } elseif { $coordinates_system == "global" } {
        set f $force
    } else {
        error "gid_draw_opengl::draw_symbol_line_interior_point_load. Unexpected coordinates_system=$coordinates_system"
    }
    #matrix to orient the reference picture (arrow for origin to 1 in x direction)
    set matrix_x_axis [math::linearalgebra::unitLengthVector $f]
    lassign [MathUtils::CalculateLocalAxisFromXAxis $matrix_x_axis] matrix_y_axis matrix_z_axis
    set scale [expr {1.0e1*$scale_draw_loads*[math::linearalgebra::norm_two $f]}]
    set transform_matrix [concat [math::linearalgebra::scale_vect $scale $matrix_x_axis] 0 [math::linearalgebra::scale_vect $scale $matrix_y_axis] 0 [math::linearalgebra::scale_vect $scale $matrix_z_axis] 0 0 0 0 1]
    set origin_t [math::linearalgebra::axpy_vect $relative_position $v12 $origin]
    GiD_OpenGL draw -pushmatrix -translate $origin_t -multmatrix $transform_matrix -color $rgb(green) -call $_opengl_draw_list_by_key(point_load) -popmatrix
    return 0
}

proc gid_draw_opengl::draw_symbol_line_uniform_load { origin end force coordinates_system } {
    variable _opengl_draw_list_by_key
    variable rgb
    variable scale_draw_loads
    if { ![info exists _opengl_draw_list_by_key(DistributedLoad)] } {
        set _opengl_draw_list_by_key(DistributedLoad) [gid_draw_opengl::create_opengl_list_distributed_load]
    }
    set v12 [math::linearalgebra::sub_vect $end $origin]
    if { $coordinates_system == "local" } {
        #calculate force in global axes
        set x_axis [math::linearalgebra::unitLengthVector $v12]
        lassign [MathUtils::CalculateLocalAxisFromXAxis $x_axis] y_axis z_axis
        set matrix [math::linearalgebra::mkMatrix 3 3]
        math::linearalgebra::setcol matrix 0 $x_axis
        math::linearalgebra::setcol matrix 1 $y_axis
        math::linearalgebra::setcol matrix 2 $z_axis
        set f [math::linearalgebra::matmul $matrix $force]
    } elseif { $coordinates_system == "global" } {
        set f $force
    } else {
        error "gid_draw_opengl::draw_symbol_line_uniform_load. Unexpected coordinates_system=$coordinates_system"
    }
    #matrix to orient the reference picture (1x1 square with arrows pointing to -y direction)
    #it is not an orthogonal matrix!!
    set matrix_x_axis [math::linearalgebra::unitLengthVector $v12]
    set scale [expr {1.0e1*$scale_draw_loads*[math::linearalgebra::norm_two $f]}]
    set matrix_y_axis [math::linearalgebra::scale_vect -1.0 [math::linearalgebra::unitLengthVector $f]]
    set normal [math::linearalgebra::crossproduct $matrix_x_axis $matrix_y_axis]
    if { [math::linearalgebra::norm_two $normal] > $MathUtils::Epsilon } {
        set matrix_z_axis [math::linearalgebra::unitLengthVector $normal]
    } else {
        #force parallel to line, square degenerate to zero area, z arbitrary but normal to the force and line
        set matrix_z_axis [lindex [MathUtils::CalculateLocalAxisFromXAxis $matrix_x_axis] 1]
    }
    set bar_length [math::linearalgebra::norm_two $v12]
    set transform_matrix [concat [math::linearalgebra::scale_vect $bar_length $matrix_x_axis] 0 [math::linearalgebra::scale_vect $scale $matrix_y_axis] 0 [math::linearalgebra::scale_vect $scale $matrix_z_axis] 0 0 0 0 1]
    GiD_OpenGL draw -pushmatrix -translate $origin -multmatrix $transform_matrix -color $rgb(green) -call $_opengl_draw_list_by_key(DistributedLoad) -popmatrix
    return 0
}

proc gid_draw_opengl::draw_symbol_line_trapezoidal_load { origin end force_0 force_1 coordinates_system relative_position_0 relative_position_1 } {
    variable _opengl_draw_list_by_key
    variable rgb
    variable scale_draw_loads
    if { ![info exists _opengl_draw_list_by_key(point_load)] } {
        set _opengl_draw_list_by_key(point_load) [gid_draw_opengl::create_opengl_list_point_load]
    }
    set v12 [math::linearalgebra::sub_vect $end $origin]
    if { $coordinates_system == "local" } {
        #calculate force in global axes
        set x_axis [math::linearalgebra::unitLengthVector $v12]
        lassign [MathUtils::CalculateLocalAxisFromXAxis $x_axis] y_axis z_axis
        set matrix [math::linearalgebra::mkMatrix 3 3]
        math::linearalgebra::setcol matrix 0 $x_axis
        math::linearalgebra::setcol matrix 1 $y_axis
        math::linearalgebra::setcol matrix 2 $z_axis
        set f0 [math::linearalgebra::matmul $matrix $force_0]
        set f1 [math::linearalgebra::matmul $matrix $force_1]
    } elseif { $coordinates_system == "global" } {
        set f0 $force_0
        set f1 $force_1
    } else {
        error "gid_draw_opengl::draw_symbol_line_trapezoidal_load. Unexpected coordinates_system=$coordinates_system"
    }
    set datax [list $relative_position_0 [lindex $force_0 0] $relative_position_1 [lindex $force_1 0]]
    set datay [list $relative_position_0 [lindex $force_0 1] $relative_position_1 [lindex $force_1 1]]
    set dataz [list $relative_position_0 [lindex $force_0 2] $relative_position_1 [lindex $force_1 2]]
    foreach t [GidUtils::GetRange $relative_position_0 $relative_position_1 4] {
        set fx [math::interpolate::interp-linear $datax $t]
        set fy [math::interpolate::interp-linear $datay $t]
        set fz [math::interpolate::interp-linear $dataz $t]
        set f [list $fx $fy $fz]
        #matrix to orient the reference picture (arrow for origin to 1 in x direction)
        set matrix_x_axis [math::linearalgebra::unitLengthVector $f]
        lassign [MathUtils::CalculateLocalAxisFromXAxis $matrix_x_axis] matrix_y_axis matrix_z_axis
        set scale [expr {1.0e1*$scale_draw_loads*[math::linearalgebra::norm_two $f]}]
        set transform_matrix [concat [math::linearalgebra::scale_vect $scale $matrix_x_axis] 0 [math::linearalgebra::scale_vect $scale $matrix_y_axis] 0 [math::linearalgebra::scale_vect $scale $matrix_z_axis] 0 0 0 0 1]
        set origin_t [math::linearalgebra::axpy_vect $t $v12 $origin]
        GiD_OpenGL draw -pushmatrix -translate $origin_t -multmatrix $transform_matrix -color $rgb(green) -call $_opengl_draw_list_by_key(point_load) -popmatrix
    }
    return 0
}


#rotated: 0 'normal position', 1 'rotated 90 degrees'
#section_width section_height in reference unit of lenght of the model
proc gid_draw_opengl::draw_symbol_section_properties { origin end rotated section_width section_height } {
    variable _opengl_draw_list_by_key
    variable rgb
    set v12 [math::linearalgebra::sub_vect $end $origin]
    set matrix_x_axis [math::linearalgebra::unitLengthVector $v12]
    lassign [MathUtils::CalculateLocalAxisFromXAxis $matrix_x_axis] matrix_y_axis matrix_z_axis

    set scale_y [expr {$section_width*1.0}]
    set scale_z [expr {$section_height*1.0}]
    set bar_length [math::linearalgebra::norm_two $v12]
    if { $rotated } {
        #normal orientation of the profile agaings gravity loads
        set transform_matrix [concat [math::linearalgebra::scale_vect $bar_length $matrix_x_axis] 0 [math::linearalgebra::scale_vect $scale_y $matrix_y_axis] 0 [math::linearalgebra::scale_vect $scale_z $matrix_z_axis] 0 0 0 0 1]
        set color $rgb(green)
    } else {
        #rotate 90 degrees swapping y-z axis
        set transform_matrix [concat [math::linearalgebra::scale_vect $bar_length $matrix_x_axis] 0 [math::linearalgebra::scale_vect $scale_z $matrix_z_axis] 0 [math::linearalgebra::scale_vect $scale_y $matrix_y_axis] 0 0 0 0 1]
        set color $rgb(orange)
    }
    GiD_OpenGL draw -pushmatrix -translate $origin -multmatrix $transform_matrix
    GiD_OpenGL draw -color $rgb(black)
    GiD_OpenGL draw -polygonmode frontandback line
    gid_draw_opengl::draw_beam
    GiD_OpenGL draw -polygonmode frontandback fill
    GiD_OpenGL draw -color $color
    gid_draw_opengl::draw_beam
    GiD_OpenGL draw -popmatrix
    return 0
}


# to represent the  thickess of surface
proc gid_draw_opengl::draw_thickness { thickness } {
    set z_max [expr $thickness*0.5]
    set z_min [expr -1.0*$z_max]
    GiD_OpenGL draw -color $rgb(black)
    GiD_OpenGL draw -polygonmode frontandback line
    GiD_OpenGL draw -begin quads -vertex [list 0 -0.5 $z_min] -vertex [list 0 0.5 $z_min] -vertex [list 1 0.5 $z_min] -vertex [list 1 -0.5 $z_min] -end
    GiD_OpenGL draw -begin quads -vertex [list 0 -0.5 $z_max] -vertex [list 0 0.5 $z_max] -vertex [list 1 0.5 $z_max] -vertex [list 1 -0.5 $z_max] -end
    GiD_OpenGL draw -color $rgb(green)
    GiD_OpenGL draw -polygonmode frontandback fill
    GiD_OpenGL draw -begin quads -vertex [list 0 -0.5 $z_min] -vertex [list 0 0.5 $z_min] -vertex [list 1 0.5 $z_min] -vertex [list 1 -0.5 $z_min] -end
    GiD_OpenGL draw -begin quads -vertex [list 0 -0.5 $z_max] -vertex [list 0 0.5 $z_max] -vertex [list 1 0.5 $z_max] -vertex [list 1 -0.5 $z_max] -end
}
